use strict;
use warnings;

use Test::More qw/no_plan/;
use FindBin qw($Bin);
use File::Spec;

BEGIN { use_ok('Bio::Tools::Run::QCons') }

my @Data = map { eval } <DATA>;

my $m = Bio::Tools::Run::QCons->new(
    file         => "$Bin/1.pdb",
    chains       => [ 'A', 'L' ],
    program_name => File::Spec->catfile($Bin, 'bin', 'Qcontacts-mock'),
);

test_results( $m );

local $@;

my $q = eval {
    Bio::Tools::Run::QCons->new(
        file => "$Bin/1.pdb",
        chains => ['A', 'L'],
    );
};

SKIP: {
    skip "You don't appear to have the 'Qcontacts' executable in your path", 3 if $@;
    test_results( $q );
};

sub test_results {
    my $q = shift;

    isa_ok( $q, 'Bio::Tools::Run::QCons' );
    is_deeply( $q->atom_contacts,     $Data[0], 'Parse by atom' );
    is_deeply( $q->residue_contacts,  $Data[1], 'Parse by res'  );
}

__DATA__
[{'area' => '0.400','type' => 'V','atom2' => {'number' => '461','res_name' => 'SER','res_number' => '59','name' => 'OG'},'atom1' => {'number' => '2226','res_name' => 'ASN','res_number' => '318','name' => 'CB'}},{'area' => '11.448','type' => 'V','atom2' => {'number' => '460','res_name' => 'SER','res_number' => '59','name' => 'CB'},'atom1' => {'number' => '2226','res_name' => 'ASN','res_number' => '318','name' => 'CB'}},{'area' => '5.014','type' => 'V','atom2' => {'number' => '460','res_name' => 'SER','res_number' => '59','name' => 'CB'},'atom1' => {'number' => '2228','res_name' => 'ASN','res_number' => '318','name' => 'ND2'}},{'area' => '3.172','type' => 'V','atom2' => {'number' => '457','res_name' => 'SER','res_number' => '59','name' => 'CA'},'atom1' => {'number' => '2228','res_name' => 'ASN','res_number' => '318','name' => 'ND2'}},{'area' => '3.522','type' => 'V','atom2' => {'number' => '403','res_name' => 'TYR','res_number' => '52','name' => 'OH'},'atom1' => {'number' => '2259','res_name' => 'GLU','res_number' => '322','name' => 'CG'}},{'area' => '0.156','type' => 'V','atom2' => {'number' => '437','res_name' => 'ARG','res_number' => '57','name' => 'O'},'atom1' => {'number' => '2259','res_name' => 'GLU','res_number' => '322','name' => 'CG'}},{'area' => '2.239','type' => 'V','atom2' => {'number' => '430','res_name' => 'ASN','res_number' => '56','name' => 'CB'},'atom1' => {'number' => '2261','res_name' => 'GLU','res_number' => '322','name' => 'OE1'}},{'area' => '0.037','type' => 'V','atom2' => {'number' => '427','res_name' => 'ASN','res_number' => '56','name' => 'CA'},'atom1' => {'number' => '2261','res_name' => 'GLU','res_number' => '322','name' => 'OE1'}},{'area' => '1.399','type' => 'V','atom2' => {'number' => '403','res_name' => 'TYR','res_number' => '52','name' => 'OH'},'atom1' => {'number' => '2261','res_name' => 'GLU','res_number' => '322','name' => 'OE1'}},{'area' => '1.786','type' => 'V','atom2' => {'number' => '438','res_name' => 'ARG','res_number' => '57','name' => 'CB'},'atom1' => {'number' => '2261','res_name' => 'GLU','res_number' => '322','name' => 'OE1'}},{'area' => '10.366','type' => 'V','atom2' => {'number' => '437','res_name' => 'ARG','res_number' => '57','name' => 'O'},'atom1' => {'number' => '2261','res_name' => 'GLU','res_number' => '322','name' => 'OE1'}},{'area' => '2.663','type' => 'V','atom2' => {'number' => '434','res_name' => 'ARG','res_number' => '57','name' => 'N'},'atom1' => {'number' => '2261','res_name' => 'GLU','res_number' => '322','name' => 'OE1'}},{'area' => '0.046','type' => 'V','atom2' => {'number' => '403','res_name' => 'TYR','res_number' => '52','name' => 'OH'},'atom1' => {'number' => '2265','res_name' => 'ILE','res_number' => '323','name' => 'C'}},{'area' => '9.330','type' => 'V','atom2' => {'number' => '403','res_name' => 'TYR','res_number' => '52','name' => 'OH'},'atom1' => {'number' => '2266','res_name' => 'ILE','res_number' => '323','name' => 'O'}},{'area' => '0.127','type' => 'V','atom2' => {'number' => '400','res_name' => 'TYR','res_number' => '52','name' => 'CE1'},'atom1' => {'number' => '2266','res_name' => 'ILE','res_number' => '323','name' => 'O'}},{'area' => '0.245','type' => 'V','atom2' => {'number' => '437','res_name' => 'ARG','res_number' => '57','name' => 'O'},'atom1' => {'number' => '2266','res_name' => 'ILE','res_number' => '323','name' => 'O'}},{'area' => '1.009','type' => 'V','atom2' => {'number' => '403','res_name' => 'TYR','res_number' => '52','name' => 'OH'},'atom1' => {'number' => '2267','res_name' => 'ILE','res_number' => '323','name' => 'CB'}},{'area' => '1.069','type' => 'V','atom2' => {'number' => '403','res_name' => 'TYR','res_number' => '52','name' => 'OH'},'atom1' => {'number' => '2269','res_name' => 'ILE','res_number' => '323','name' => 'CG2'}},{'area' => '0.250','type' => 'V','atom2' => {'number' => '402','res_name' => 'TYR','res_number' => '52','name' => 'CZ'},'atom1' => {'number' => '2269','res_name' => 'ILE','res_number' => '323','name' => 'CG2'}},{'area' => '1.719','type' => 'V','atom2' => {'number' => '401','res_name' => 'TYR','res_number' => '52','name' => 'CE2'},'atom1' => {'number' => '2269','res_name' => 'ILE','res_number' => '323','name' => 'CG2'}},{'area' => '0.181','type' => 'V','atom2' => {'number' => '453','res_name' => 'PHE','res_number' => '58','name' => 'CE1'},'atom1' => {'number' => '2272','res_name' => 'GLN','res_number' => '324','name' => 'CA'}},{'area' => '1.361','type' => 'V','atom2' => {'number' => '451','res_name' => 'PHE','res_number' => '58','name' => 'CD1'},'atom1' => {'number' => '2272','res_name' => 'GLN','res_number' => '324','name' => 'CA'}},{'area' => '0.101','type' => 'V','atom2' => {'number' => '400','res_name' => 'TYR','res_number' => '52','name' => 'CE1'},'atom1' => {'number' => '2272','res_name' => 'GLN','res_number' => '324','name' => 'CA'}},{'area' => '0.257','type' => 'V','atom2' => {'number' => '446','res_name' => 'PHE','res_number' => '58','name' => 'CA'},'atom1' => {'number' => '2272','res_name' => 'GLN','res_number' => '324','name' => 'CA'}},{'area' => '0.049','type' => 'V','atom2' => {'number' => '437','res_name' => 'ARG','res_number' => '57','name' => 'O'},'atom1' => {'number' => '2272','res_name' => 'GLN','res_number' => '324','name' => 'CA'}},{'area' => '2.935','type' => 'V','atom2' => {'number' => '453','res_name' => 'PHE','res_number' => '58','name' => 'CE1'},'atom1' => {'number' => '2273','res_name' => 'GLN','res_number' => '324','name' => 'C'}},{'area' => '0.760','type' => 'V','atom2' => {'number' => '451','res_name' => 'PHE','res_number' => '58','name' => 'CD1'},'atom1' => {'number' => '2273','res_name' => 'GLN','res_number' => '324','name' => 'C'}},{'area' => '2.690','type' => 'V','atom2' => {'number' => '453','res_name' => 'PHE','res_number' => '58','name' => 'CE1'},'atom1' => {'number' => '2274','res_name' => 'GLN','res_number' => '324','name' => 'O'}},{'area' => '6.058','type' => 'V','atom2' => {'number' => '453','res_name' => 'PHE','res_number' => '58','name' => 'CE1'},'atom1' => {'number' => '2275','res_name' => 'GLN','res_number' => '324','name' => 'CB'}},{'area' => '0.402','type' => 'V','atom2' => {'number' => '451','res_name' => 'PHE','res_number' => '58','name' => 'CD1'},'atom1' => {'number' => '2275','res_name' => 'GLN','res_number' => '324','name' => 'CB'}},{'area' => '6.597','type' => 'V','atom2' => {'number' => '455','res_name' => 'PHE','res_number' => '58','name' => 'CZ'},'atom1' => {'number' => '2275','res_name' => 'GLN','res_number' => '324','name' => 'CB'}},{'area' => '5.704','type' => 'V','atom2' => {'number' => '461','res_name' => 'SER','res_number' => '59','name' => 'OG'},'atom1' => {'number' => '2276','res_name' => 'GLN','res_number' => '324','name' => 'CG'}},{'area' => '2.108','type' => 'V','atom2' => {'number' => '460','res_name' => 'SER','res_number' => '59','name' => 'CB'},'atom1' => {'number' => '2276','res_name' => 'GLN','res_number' => '324','name' => 'CG'}},{'area' => '2.727','type' => 'V','atom2' => {'number' => '455','res_name' => 'PHE','res_number' => '58','name' => 'CZ'},'atom1' => {'number' => '2276','res_name' => 'GLN','res_number' => '324','name' => 'CG'}},{'area' => '0.054','type' => 'V','atom2' => {'number' => '454','res_name' => 'PHE','res_number' => '58','name' => 'CE2'},'atom1' => {'number' => '2276','res_name' => 'GLN','res_number' => '324','name' => 'CG'}},{'area' => '0.039','type' => 'V','atom2' => {'number' => '461','res_name' => 'SER','res_number' => '59','name' => 'OG'},'atom1' => {'number' => '2277','res_name' => 'GLN','res_number' => '324','name' => 'CD'}},{'area' => '5.219','type' => 'V','atom2' => {'number' => '460','res_name' => 'SER','res_number' => '59','name' => 'CB'},'atom1' => {'number' => '2277','res_name' => 'GLN','res_number' => '324','name' => 'CD'}},{'area' => '0.624','type' => 'V','atom2' => {'number' => '457','res_name' => 'SER','res_number' => '59','name' => 'CA'},'atom1' => {'number' => '2277','res_name' => 'GLN','res_number' => '324','name' => 'CD'}},{'area' => '1.310','type' => 'V','atom2' => {'number' => '456','res_name' => 'SER','res_number' => '59','name' => 'N'},'atom1' => {'number' => '2277','res_name' => 'GLN','res_number' => '324','name' => 'CD'}},{'area' => '0.228','type' => 'V','atom2' => {'number' => '451','res_name' => 'PHE','res_number' => '58','name' => 'CD1'},'atom1' => {'number' => '2277','res_name' => 'GLN','res_number' => '324','name' => 'CD'}},{'area' => '0.931','type' => 'V','atom2' => {'number' => '446','res_name' => 'PHE','res_number' => '58','name' => 'CA'},'atom1' => {'number' => '2277','res_name' => 'GLN','res_number' => '324','name' => 'CD'}},{'area' => '0.350','type' => 'V','atom2' => {'number' => '437','res_name' => 'ARG','res_number' => '57','name' => 'O'},'atom1' => {'number' => '2277','res_name' => 'GLN','res_number' => '324','name' => 'CD'}},{'area' => '1.793','type' => 'V','atom2' => {'number' => '460','res_name' => 'SER','res_number' => '59','name' => 'CB'},'atom1' => {'number' => '2278','res_name' => 'GLN','res_number' => '324','name' => 'NE2'}},{'area' => '2.055','type' => 'V','atom2' => {'number' => '457','res_name' => 'SER','res_number' => '59','name' => 'CA'},'atom1' => {'number' => '2278','res_name' => 'GLN','res_number' => '324','name' => 'NE2'}},{'area' => '0.717','type' => 'V','atom2' => {'number' => '456','res_name' => 'SER','res_number' => '59','name' => 'N'},'atom1' => {'number' => '2278','res_name' => 'GLN','res_number' => '324','name' => 'NE2'}},{'area' => '2.792','type' => 'V','atom2' => {'number' => '437','res_name' => 'ARG','res_number' => '57','name' => 'O'},'atom1' => {'number' => '2278','res_name' => 'GLN','res_number' => '324','name' => 'NE2'}},{'area' => '0.265','type' => 'V','atom2' => {'number' => '461','res_name' => 'SER','res_number' => '59','name' => 'OG'},'atom1' => {'number' => '2279','res_name' => 'GLN','res_number' => '324','name' => 'OE1'}},{'area' => '0.612','type' => 'V','atom2' => {'number' => '460','res_name' => 'SER','res_number' => '59','name' => 'CB'},'atom1' => {'number' => '2279','res_name' => 'GLN','res_number' => '324','name' => 'OE1'}},{'area' => '0.031','type' => 'V','atom2' => {'number' => '457','res_name' => 'SER','res_number' => '59','name' => 'CA'},'atom1' => {'number' => '2279','res_name' => 'GLN','res_number' => '324','name' => 'OE1'}},{'area' => '4.128','type' => 'V','atom2' => {'number' => '456','res_name' => 'SER','res_number' => '59','name' => 'N'},'atom1' => {'number' => '2279','res_name' => 'GLN','res_number' => '324','name' => 'OE1'}},{'area' => '0.195','type' => 'V','atom2' => {'number' => '453','res_name' => 'PHE','res_number' => '58','name' => 'CE1'},'atom1' => {'number' => '2279','res_name' => 'GLN','res_number' => '324','name' => 'OE1'}},{'area' => '0.121','type' => 'V','atom2' => {'number' => '452','res_name' => 'PHE','res_number' => '58','name' => 'CD2'},'atom1' => {'number' => '2279','res_name' => 'GLN','res_number' => '324','name' => 'OE1'}},{'area' => '1.344','type' => 'V','atom2' => {'number' => '451','res_name' => 'PHE','res_number' => '58','name' => 'CD1'},'atom1' => {'number' => '2279','res_name' => 'GLN','res_number' => '324','name' => 'OE1'}},{'area' => '0.172','type' => 'V','atom2' => {'number' => '455','res_name' => 'PHE','res_number' => '58','name' => 'CZ'},'atom1' => {'number' => '2279','res_name' => 'GLN','res_number' => '324','name' => 'OE1'}},{'area' => '0.402','type' => 'V','atom2' => {'number' => '454','res_name' => 'PHE','res_number' => '58','name' => 'CE2'},'atom1' => {'number' => '2279','res_name' => 'GLN','res_number' => '324','name' => 'OE1'}},{'area' => '0.156','type' => 'V','atom2' => {'number' => '450','res_name' => 'PHE','res_number' => '58','name' => 'CG'},'atom1' => {'number' => '2279','res_name' => 'GLN','res_number' => '324','name' => 'OE1'}},{'area' => '0.868','type' => 'V','atom2' => {'number' => '446','res_name' => 'PHE','res_number' => '58','name' => 'CA'},'atom1' => {'number' => '2279','res_name' => 'GLN','res_number' => '324','name' => 'OE1'}},{'area' => '2.282','type' => 'V','atom2' => {'number' => '451','res_name' => 'PHE','res_number' => '58','name' => 'CD1'},'atom1' => {'number' => '2280','res_name' => 'GLY','res_number' => '325','name' => 'N'}},{'area' => '0.620','type' => 'V','atom2' => {'number' => '403','res_name' => 'TYR','res_number' => '52','name' => 'OH'},'atom1' => {'number' => '2280','res_name' => 'GLY','res_number' => '325','name' => 'N'}},{'area' => '1.975','type' => 'V','atom2' => {'number' => '402','res_name' => 'TYR','res_number' => '52','name' => 'CZ'},'atom1' => {'number' => '2280','res_name' => 'GLY','res_number' => '325','name' => 'N'}},{'area' => '3.963','type' => 'V','atom2' => {'number' => '400','res_name' => 'TYR','res_number' => '52','name' => 'CE1'},'atom1' => {'number' => '2280','res_name' => 'GLY','res_number' => '325','name' => 'N'}},{'area' => '0.462','type' => 'V','atom2' => {'number' => '446','res_name' => 'PHE','res_number' => '58','name' => 'CA'},'atom1' => {'number' => '2280','res_name' => 'GLY','res_number' => '325','name' => 'N'}},{'area' => '1.285','type' => 'V','atom2' => {'number' => '451','res_name' => 'PHE','res_number' => '58','name' => 'CD1'},'atom1' => {'number' => '2281','res_name' => 'GLY','res_number' => '325','name' => 'CA'}},{'area' => '3.067','type' => 'V','atom2' => {'number' => '402','res_name' => 'TYR','res_number' => '52','name' => 'CZ'},'atom1' => {'number' => '2281','res_name' => 'GLY','res_number' => '325','name' => 'CA'}},{'area' => '5.527','type' => 'V','atom2' => {'number' => '401','res_name' => 'TYR','res_number' => '52','name' => 'CE2'},'atom1' => {'number' => '2281','res_name' => 'GLY','res_number' => '325','name' => 'CA'}},{'area' => '2.850','type' => 'V','atom2' => {'number' => '400','res_name' => 'TYR','res_number' => '52','name' => 'CE1'},'atom1' => {'number' => '2281','res_name' => 'GLY','res_number' => '325','name' => 'CA'}},{'area' => '2.058','type' => 'V','atom2' => {'number' => '399','res_name' => 'TYR','res_number' => '52','name' => 'CD2'},'atom1' => {'number' => '2281','res_name' => 'GLY','res_number' => '325','name' => 'CA'}},{'area' => '2.929','type' => 'V','atom2' => {'number' => '398','res_name' => 'TYR','res_number' => '52','name' => 'CD1'},'atom1' => {'number' => '2281','res_name' => 'GLY','res_number' => '325','name' => 'CA'}},{'area' => '1.190','type' => 'V','atom2' => {'number' => '397','res_name' => 'TYR','res_number' => '52','name' => 'CG'},'atom1' => {'number' => '2281','res_name' => 'GLY','res_number' => '325','name' => 'CA'}},{'area' => '0.650','type' => 'V','atom2' => {'number' => '451','res_name' => 'PHE','res_number' => '58','name' => 'CD1'},'atom1' => {'number' => '2282','res_name' => 'GLY','res_number' => '325','name' => 'C'}},{'area' => '1.296','type' => 'V','atom2' => {'number' => '398','res_name' => 'TYR','res_number' => '52','name' => 'CD1'},'atom1' => {'number' => '2282','res_name' => 'GLY','res_number' => '325','name' => 'C'}},{'area' => '0.619','type' => 'V','atom2' => {'number' => '397','res_name' => 'TYR','res_number' => '52','name' => 'CG'},'atom1' => {'number' => '2282','res_name' => 'GLY','res_number' => '325','name' => 'C'}},{'area' => '1.032','type' => 'V','atom2' => {'number' => '396','res_name' => 'TYR','res_number' => '52','name' => 'CB'},'atom1' => {'number' => '2282','res_name' => 'GLY','res_number' => '325','name' => 'C'}},{'area' => '2.991','type' => 'V','atom2' => {'number' => '375','res_name' => 'LEU','res_number' => '49','name' => 'CD2'},'atom1' => {'number' => '2282','res_name' => 'GLY','res_number' => '325','name' => 'C'}},{'area' => '4.358','type' => 'V','atom2' => {'number' => '453','res_name' => 'PHE','res_number' => '58','name' => 'CE1'},'atom1' => {'number' => '2283','res_name' => 'GLY','res_number' => '325','name' => 'O'}},{'area' => '1.173','type' => 'V','atom2' => {'number' => '451','res_name' => 'PHE','res_number' => '58','name' => 'CD1'},'atom1' => {'number' => '2283','res_name' => 'GLY','res_number' => '325','name' => 'O'}},{'area' => '3.025','type' => 'V','atom2' => {'number' => '375','res_name' => 'LEU','res_number' => '49','name' => 'CD2'},'atom1' => {'number' => '2283','res_name' => 'GLY','res_number' => '325','name' => 'O'}},{'area' => '0.088','type' => 'V','atom2' => {'number' => '401','res_name' => 'TYR','res_number' => '52','name' => 'CE2'},'atom1' => {'number' => '2284','res_name' => 'ASP','res_number' => '326','name' => 'N'}},{'area' => '4.472','type' => 'V','atom2' => {'number' => '399','res_name' => 'TYR','res_number' => '52','name' => 'CD2'},'atom1' => {'number' => '2284','res_name' => 'ASP','res_number' => '326','name' => 'N'}},{'area' => '0.513','type' => 'V','atom2' => {'number' => '397','res_name' => 'TYR','res_number' => '52','name' => 'CG'},'atom1' => {'number' => '2284','res_name' => 'ASP','res_number' => '326','name' => 'N'}},{'area' => '1.966','type' => 'V','atom2' => {'number' => '396','res_name' => 'TYR','res_number' => '52','name' => 'CB'},'atom1' => {'number' => '2284','res_name' => 'ASP','res_number' => '326','name' => 'N'}},{'area' => '6.630','type' => 'V','atom2' => {'number' => '396','res_name' => 'TYR','res_number' => '52','name' => 'CB'},'atom1' => {'number' => '2285','res_name' => 'ASP','res_number' => '326','name' => 'CA'}},{'area' => '3.694','type' => 'V','atom2' => {'number' => '375','res_name' => 'LEU','res_number' => '49','name' => 'CD2'},'atom1' => {'number' => '2285','res_name' => 'ASP','res_number' => '326','name' => 'CA'}},{'area' => '1.697','type' => 'V','atom2' => {'number' => '374','res_name' => 'LEU','res_number' => '49','name' => 'CD1'},'atom1' => {'number' => '2285','res_name' => 'ASP','res_number' => '326','name' => 'CA'}},{'area' => '0.879','type' => 'V','atom2' => {'number' => '396','res_name' => 'TYR','res_number' => '52','name' => 'CB'},'atom1' => {'number' => '2286','res_name' => 'ASP','res_number' => '326','name' => 'C'}},{'area' => '1.240','type' => 'V','atom2' => {'number' => '410','res_name' => 'LYS','res_number' => '53','name' => 'CD'},'atom1' => {'number' => '2287','res_name' => 'ASP','res_number' => '326','name' => 'O'}},{'area' => '6.696','type' => 'V','atom2' => {'number' => '409','res_name' => 'LYS','res_number' => '53','name' => 'CG'},'atom1' => {'number' => '2287','res_name' => 'ASP','res_number' => '326','name' => 'O'}},{'area' => '1.662','type' => 'V','atom2' => {'number' => '404','res_name' => 'LYS','res_number' => '53','name' => 'N'},'atom1' => {'number' => '2287','res_name' => 'ASP','res_number' => '326','name' => 'O'}},{'area' => '1.576','type' => 'V','atom2' => {'number' => '399','res_name' => 'TYR','res_number' => '52','name' => 'CD2'},'atom1' => {'number' => '2287','res_name' => 'ASP','res_number' => '326','name' => 'O'}},{'area' => '4.453','type' => 'V','atom2' => {'number' => '396','res_name' => 'TYR','res_number' => '52','name' => 'CB'},'atom1' => {'number' => '2287','res_name' => 'ASP','res_number' => '326','name' => 'O'}},{'area' => '0.615','type' => 'V','atom2' => {'number' => '453','res_name' => 'PHE','res_number' => '58','name' => 'CE1'},'atom1' => {'number' => '2288','res_name' => 'ASP','res_number' => '326','name' => 'CB'}},{'area' => '6.715','type' => 'V','atom2' => {'number' => '375','res_name' => 'LEU','res_number' => '49','name' => 'CD2'},'atom1' => {'number' => '2288','res_name' => 'ASP','res_number' => '326','name' => 'CB'}},{'area' => '3.672','type' => 'V','atom2' => {'number' => '375','res_name' => 'LEU','res_number' => '49','name' => 'CD2'},'atom1' => {'number' => '2300','res_name' => 'LYS','res_number' => '327','name' => 'NZ'}},{'area' => '7.407','type' => 'V','atom2' => {'number' => '374','res_name' => 'LEU','res_number' => '49','name' => 'CD1'},'atom1' => {'number' => '2300','res_name' => 'LYS','res_number' => '327','name' => 'NZ'}},{'area' => '1.427','type' => 'V','atom2' => {'number' => '372','res_name' => 'LEU','res_number' => '49','name' => 'CB'},'atom1' => {'number' => '2300','res_name' => 'LYS','res_number' => '327','name' => 'NZ'}},{'area' => '0.186','type' => 'V','atom2' => {'number' => '412','res_name' => 'LYS','res_number' => '53','name' => 'NZ'},'atom1' => {'number' => '2304','res_name' => 'SER','res_number' => '328','name' => 'O'}},{'area' => '0.216','type' => 'V','atom2' => {'number' => '411','res_name' => 'LYS','res_number' => '53','name' => 'CE'},'atom1' => {'number' => '2304','res_name' => 'SER','res_number' => '328','name' => 'O'}},{'area' => '10.801','type' => 'V','atom2' => {'number' => '410','res_name' => 'LYS','res_number' => '53','name' => 'CD'},'atom1' => {'number' => '2304','res_name' => 'SER','res_number' => '328','name' => 'O'}},{'area' => '0.148','type' => 'V','atom2' => {'number' => '409','res_name' => 'LYS','res_number' => '53','name' => 'CG'},'atom1' => {'number' => '2304','res_name' => 'SER','res_number' => '328','name' => 'O'}},{'area' => '0.089','type' => 'V','atom2' => {'number' => '399','res_name' => 'TYR','res_number' => '52','name' => 'CD2'},'atom1' => {'number' => '2304','res_name' => 'SER','res_number' => '328','name' => 'O'}},{'area' => '1.734','type' => 'V','atom2' => {'number' => '411','res_name' => 'LYS','res_number' => '53','name' => 'CE'},'atom1' => {'number' => '2308','res_name' => 'GLU','res_number' => '329','name' => 'CA'}},{'area' => '0.052','type' => 'V','atom2' => {'number' => '410','res_name' => 'LYS','res_number' => '53','name' => 'CD'},'atom1' => {'number' => '2308','res_name' => 'GLU','res_number' => '329','name' => 'CA'}},{'area' => '0.678','type' => 'V','atom2' => {'number' => '412','res_name' => 'LYS','res_number' => '53','name' => 'NZ'},'atom1' => {'number' => '2316','res_name' => 'MET','res_number' => '330','name' => 'N'}},{'area' => '1.630','type' => 'V','atom2' => {'number' => '411','res_name' => 'LYS','res_number' => '53','name' => 'CE'},'atom1' => {'number' => '2316','res_name' => 'MET','res_number' => '330','name' => 'N'}},{'area' => '0.729','type' => 'V','atom2' => {'number' => '412','res_name' => 'LYS','res_number' => '53','name' => 'NZ'},'atom1' => {'number' => '2320','res_name' => 'MET','res_number' => '330','name' => 'CB'}},{'area' => '15.809','type' => 'V','atom2' => {'number' => '412','res_name' => 'LYS','res_number' => '53','name' => 'NZ'},'atom1' => {'number' => '2321','res_name' => 'MET','res_number' => '330','name' => 'CG'}},{'area' => '0.075','type' => 'V','atom2' => {'number' => '410','res_name' => 'LYS','res_number' => '53','name' => 'CD'},'atom1' => {'number' => '2321','res_name' => 'MET','res_number' => '330','name' => 'CG'}},{'area' => '0.109','type' => 'V','atom2' => {'number' => '410','res_name' => 'LYS','res_number' => '53','name' => 'CD'},'atom1' => {'number' => '2322','res_name' => 'MET','res_number' => '330','name' => 'SD'}},{'area' => '0.124','type' => 'V','atom2' => {'number' => '431','res_name' => 'ASN','res_number' => '56','name' => 'CG'},'atom1' => {'number' => '2323','res_name' => 'MET','res_number' => '330','name' => 'CE'}},{'area' => '4.251','type' => 'V','atom2' => {'number' => '412','res_name' => 'LYS','res_number' => '53','name' => 'NZ'},'atom1' => {'number' => '2323','res_name' => 'MET','res_number' => '330','name' => 'CE'}},{'area' => '0.644','type' => 'V','atom2' => {'number' => '411','res_name' => 'LYS','res_number' => '53','name' => 'CE'},'atom1' => {'number' => '2323','res_name' => 'MET','res_number' => '330','name' => 'CE'}},{'area' => '5.101','type' => 'V','atom2' => {'number' => '410','res_name' => 'LYS','res_number' => '53','name' => 'CD'},'atom1' => {'number' => '2323','res_name' => 'MET','res_number' => '330','name' => 'CE'}},{'area' => '2.413','type' => 'V','atom2' => {'number' => '432','res_name' => 'ASN','res_number' => '56','name' => 'ND2'},'atom1' => {'number' => '2323','res_name' => 'MET','res_number' => '330','name' => 'CE'}},{'area' => '3.159','type' => 'V','atom2' => {'number' => '403','res_name' => 'TYR','res_number' => '52','name' => 'OH'},'atom1' => {'number' => '2323','res_name' => 'MET','res_number' => '330','name' => 'CE'}},{'area' => '7.384','type' => 'V','atom2' => {'number' => '401','res_name' => 'TYR','res_number' => '52','name' => 'CE2'},'atom1' => {'number' => '2323','res_name' => 'MET','res_number' => '330','name' => 'CE'}},{'area' => '1.427','type' => 'V','atom2' => {'number' => '372','res_name' => 'LEU','res_number' => '49','name' => 'CB'},'atom1' => {'number' => '2300','res_name' => 'LYS','res_number' => '327','name' => 'NZ'}},{'area' => '7.407','type' => 'V','atom2' => {'number' => '374','res_name' => 'LEU','res_number' => '49','name' => 'CD1'},'atom1' => {'number' => '2300','res_name' => 'LYS','res_number' => '327','name' => 'NZ'}},{'area' => '1.697','type' => 'V','atom2' => {'number' => '374','res_name' => 'LEU','res_number' => '49','name' => 'CD1'},'atom1' => {'number' => '2285','res_name' => 'ASP','res_number' => '326','name' => 'CA'}},{'area' => '2.991','type' => 'V','atom2' => {'number' => '375','res_name' => 'LEU','res_number' => '49','name' => 'CD2'},'atom1' => {'number' => '2282','res_name' => 'GLY','res_number' => '325','name' => 'C'}},{'area' => '3.672','type' => 'V','atom2' => {'number' => '375','res_name' => 'LEU','res_number' => '49','name' => 'CD2'},'atom1' => {'number' => '2300','res_name' => 'LYS','res_number' => '327','name' => 'NZ'}},{'area' => '6.715','type' => 'V','atom2' => {'number' => '375','res_name' => 'LEU','res_number' => '49','name' => 'CD2'},'atom1' => {'number' => '2288','res_name' => 'ASP','res_number' => '326','name' => 'CB'}},{'area' => '3.694','type' => 'V','atom2' => {'number' => '375','res_name' => 'LEU','res_number' => '49','name' => 'CD2'},'atom1' => {'number' => '2285','res_name' => 'ASP','res_number' => '326','name' => 'CA'}},{'area' => '3.025','type' => 'V','atom2' => {'number' => '375','res_name' => 'LEU','res_number' => '49','name' => 'CD2'},'atom1' => {'number' => '2283','res_name' => 'GLY','res_number' => '325','name' => 'O'}},{'area' => '1.966','type' => 'V','atom2' => {'number' => '396','res_name' => 'TYR','res_number' => '52','name' => 'CB'},'atom1' => {'number' => '2284','res_name' => 'ASP','res_number' => '326','name' => 'N'}},{'area' => '1.032','type' => 'V','atom2' => {'number' => '396','res_name' => 'TYR','res_number' => '52','name' => 'CB'},'atom1' => {'number' => '2282','res_name' => 'GLY','res_number' => '325','name' => 'C'}},{'area' => '4.453','type' => 'V','atom2' => {'number' => '396','res_name' => 'TYR','res_number' => '52','name' => 'CB'},'atom1' => {'number' => '2287','res_name' => 'ASP','res_number' => '326','name' => 'O'}},{'area' => '0.879','type' => 'V','atom2' => {'number' => '396','res_name' => 'TYR','res_number' => '52','name' => 'CB'},'atom1' => {'number' => '2286','res_name' => 'ASP','res_number' => '326','name' => 'C'}},{'area' => '6.630','type' => 'V','atom2' => {'number' => '396','res_name' => 'TYR','res_number' => '52','name' => 'CB'},'atom1' => {'number' => '2285','res_name' => 'ASP','res_number' => '326','name' => 'CA'}},{'area' => '0.513','type' => 'V','atom2' => {'number' => '397','res_name' => 'TYR','res_number' => '52','name' => 'CG'},'atom1' => {'number' => '2284','res_name' => 'ASP','res_number' => '326','name' => 'N'}},{'area' => '0.619','type' => 'V','atom2' => {'number' => '397','res_name' => 'TYR','res_number' => '52','name' => 'CG'},'atom1' => {'number' => '2282','res_name' => 'GLY','res_number' => '325','name' => 'C'}},{'area' => '1.190','type' => 'V','atom2' => {'number' => '397','res_name' => 'TYR','res_number' => '52','name' => 'CG'},'atom1' => {'number' => '2281','res_name' => 'GLY','res_number' => '325','name' => 'CA'}},{'area' => '1.296','type' => 'V','atom2' => {'number' => '398','res_name' => 'TYR','res_number' => '52','name' => 'CD1'},'atom1' => {'number' => '2282','res_name' => 'GLY','res_number' => '325','name' => 'C'}},{'area' => '2.929','type' => 'V','atom2' => {'number' => '398','res_name' => 'TYR','res_number' => '52','name' => 'CD1'},'atom1' => {'number' => '2281','res_name' => 'GLY','res_number' => '325','name' => 'CA'}},{'area' => '0.089','type' => 'V','atom2' => {'number' => '399','res_name' => 'TYR','res_number' => '52','name' => 'CD2'},'atom1' => {'number' => '2304','res_name' => 'SER','res_number' => '328','name' => 'O'}},{'area' => '4.472','type' => 'V','atom2' => {'number' => '399','res_name' => 'TYR','res_number' => '52','name' => 'CD2'},'atom1' => {'number' => '2284','res_name' => 'ASP','res_number' => '326','name' => 'N'}},{'area' => '2.058','type' => 'V','atom2' => {'number' => '399','res_name' => 'TYR','res_number' => '52','name' => 'CD2'},'atom1' => {'number' => '2281','res_name' => 'GLY','res_number' => '325','name' => 'CA'}},{'area' => '1.576','type' => 'V','atom2' => {'number' => '399','res_name' => 'TYR','res_number' => '52','name' => 'CD2'},'atom1' => {'number' => '2287','res_name' => 'ASP','res_number' => '326','name' => 'O'}},{'area' => '3.963','type' => 'V','atom2' => {'number' => '400','res_name' => 'TYR','res_number' => '52','name' => 'CE1'},'atom1' => {'number' => '2280','res_name' => 'GLY','res_number' => '325','name' => 'N'}},{'area' => '0.101','type' => 'V','atom2' => {'number' => '400','res_name' => 'TYR','res_number' => '52','name' => 'CE1'},'atom1' => {'number' => '2272','res_name' => 'GLN','res_number' => '324','name' => 'CA'}},{'area' => '0.127','type' => 'V','atom2' => {'number' => '400','res_name' => 'TYR','res_number' => '52','name' => 'CE1'},'atom1' => {'number' => '2266','res_name' => 'ILE','res_number' => '323','name' => 'O'}},{'area' => '2.850','type' => 'V','atom2' => {'number' => '400','res_name' => 'TYR','res_number' => '52','name' => 'CE1'},'atom1' => {'number' => '2281','res_name' => 'GLY','res_number' => '325','name' => 'CA'}},{'area' => '7.384','type' => 'V','atom2' => {'number' => '401','res_name' => 'TYR','res_number' => '52','name' => 'CE2'},'atom1' => {'number' => '2323','res_name' => 'MET','res_number' => '330','name' => 'CE'}},{'area' => '1.719','type' => 'V','atom2' => {'number' => '401','res_name' => 'TYR','res_number' => '52','name' => 'CE2'},'atom1' => {'number' => '2269','res_name' => 'ILE','res_number' => '323','name' => 'CG2'}},{'area' => '0.088','type' => 'V','atom2' => {'number' => '401','res_name' => 'TYR','res_number' => '52','name' => 'CE2'},'atom1' => {'number' => '2284','res_name' => 'ASP','res_number' => '326','name' => 'N'}},{'area' => '5.527','type' => 'V','atom2' => {'number' => '401','res_name' => 'TYR','res_number' => '52','name' => 'CE2'},'atom1' => {'number' => '2281','res_name' => 'GLY','res_number' => '325','name' => 'CA'}},{'area' => '1.975','type' => 'V','atom2' => {'number' => '402','res_name' => 'TYR','res_number' => '52','name' => 'CZ'},'atom1' => {'number' => '2280','res_name' => 'GLY','res_number' => '325','name' => 'N'}},{'area' => '0.250','type' => 'V','atom2' => {'number' => '402','res_name' => 'TYR','res_number' => '52','name' => 'CZ'},'atom1' => {'number' => '2269','res_name' => 'ILE','res_number' => '323','name' => 'CG2'}},{'area' => '3.067','type' => 'V','atom2' => {'number' => '402','res_name' => 'TYR','res_number' => '52','name' => 'CZ'},'atom1' => {'number' => '2281','res_name' => 'GLY','res_number' => '325','name' => 'CA'}},{'area' => '3.522','type' => 'V','atom2' => {'number' => '403','res_name' => 'TYR','res_number' => '52','name' => 'OH'},'atom1' => {'number' => '2259','res_name' => 'GLU','res_number' => '322','name' => 'CG'}},{'area' => '3.159','type' => 'V','atom2' => {'number' => '403','res_name' => 'TYR','res_number' => '52','name' => 'OH'},'atom1' => {'number' => '2323','res_name' => 'MET','res_number' => '330','name' => 'CE'}},{'area' => '0.620','type' => 'V','atom2' => {'number' => '403','res_name' => 'TYR','res_number' => '52','name' => 'OH'},'atom1' => {'number' => '2280','res_name' => 'GLY','res_number' => '325','name' => 'N'}},{'area' => '1.069','type' => 'V','atom2' => {'number' => '403','res_name' => 'TYR','res_number' => '52','name' => 'OH'},'atom1' => {'number' => '2269','res_name' => 'ILE','res_number' => '323','name' => 'CG2'}},{'area' => '1.009','type' => 'V','atom2' => {'number' => '403','res_name' => 'TYR','res_number' => '52','name' => 'OH'},'atom1' => {'number' => '2267','res_name' => 'ILE','res_number' => '323','name' => 'CB'}},{'area' => '9.330','type' => 'V','atom2' => {'number' => '403','res_name' => 'TYR','res_number' => '52','name' => 'OH'},'atom1' => {'number' => '2266','res_name' => 'ILE','res_number' => '323','name' => 'O'}},{'area' => '0.046','type' => 'V','atom2' => {'number' => '403','res_name' => 'TYR','res_number' => '52','name' => 'OH'},'atom1' => {'number' => '2265','res_name' => 'ILE','res_number' => '323','name' => 'C'}},{'area' => '1.399','type' => 'V','atom2' => {'number' => '403','res_name' => 'TYR','res_number' => '52','name' => 'OH'},'atom1' => {'number' => '2261','res_name' => 'GLU','res_number' => '322','name' => 'OE1'}},{'area' => '1.662','type' => 'V','atom2' => {'number' => '404','res_name' => 'LYS','res_number' => '53','name' => 'N'},'atom1' => {'number' => '2287','res_name' => 'ASP','res_number' => '326','name' => 'O'}},{'area' => '0.148','type' => 'V','atom2' => {'number' => '409','res_name' => 'LYS','res_number' => '53','name' => 'CG'},'atom1' => {'number' => '2304','res_name' => 'SER','res_number' => '328','name' => 'O'}},{'area' => '6.696','type' => 'V','atom2' => {'number' => '409','res_name' => 'LYS','res_number' => '53','name' => 'CG'},'atom1' => {'number' => '2287','res_name' => 'ASP','res_number' => '326','name' => 'O'}},{'area' => '5.101','type' => 'V','atom2' => {'number' => '410','res_name' => 'LYS','res_number' => '53','name' => 'CD'},'atom1' => {'number' => '2323','res_name' => 'MET','res_number' => '330','name' => 'CE'}},{'area' => '0.109','type' => 'V','atom2' => {'number' => '410','res_name' => 'LYS','res_number' => '53','name' => 'CD'},'atom1' => {'number' => '2322','res_name' => 'MET','res_number' => '330','name' => 'SD'}},{'area' => '0.075','type' => 'V','atom2' => {'number' => '410','res_name' => 'LYS','res_number' => '53','name' => 'CD'},'atom1' => {'number' => '2321','res_name' => 'MET','res_number' => '330','name' => 'CG'}},{'area' => '0.052','type' => 'V','atom2' => {'number' => '410','res_name' => 'LYS','res_number' => '53','name' => 'CD'},'atom1' => {'number' => '2308','res_name' => 'GLU','res_number' => '329','name' => 'CA'}},{'area' => '10.801','type' => 'V','atom2' => {'number' => '410','res_name' => 'LYS','res_number' => '53','name' => 'CD'},'atom1' => {'number' => '2304','res_name' => 'SER','res_number' => '328','name' => 'O'}},{'area' => '1.240','type' => 'V','atom2' => {'number' => '410','res_name' => 'LYS','res_number' => '53','name' => 'CD'},'atom1' => {'number' => '2287','res_name' => 'ASP','res_number' => '326','name' => 'O'}},{'area' => '0.644','type' => 'V','atom2' => {'number' => '411','res_name' => 'LYS','res_number' => '53','name' => 'CE'},'atom1' => {'number' => '2323','res_name' => 'MET','res_number' => '330','name' => 'CE'}},{'area' => '1.630','type' => 'V','atom2' => {'number' => '411','res_name' => 'LYS','res_number' => '53','name' => 'CE'},'atom1' => {'number' => '2316','res_name' => 'MET','res_number' => '330','name' => 'N'}},{'area' => '1.734','type' => 'V','atom2' => {'number' => '411','res_name' => 'LYS','res_number' => '53','name' => 'CE'},'atom1' => {'number' => '2308','res_name' => 'GLU','res_number' => '329','name' => 'CA'}},{'area' => '0.216','type' => 'V','atom2' => {'number' => '411','res_name' => 'LYS','res_number' => '53','name' => 'CE'},'atom1' => {'number' => '2304','res_name' => 'SER','res_number' => '328','name' => 'O'}},{'area' => '4.251','type' => 'V','atom2' => {'number' => '412','res_name' => 'LYS','res_number' => '53','name' => 'NZ'},'atom1' => {'number' => '2323','res_name' => 'MET','res_number' => '330','name' => 'CE'}},{'area' => '15.809','type' => 'V','atom2' => {'number' => '412','res_name' => 'LYS','res_number' => '53','name' => 'NZ'},'atom1' => {'number' => '2321','res_name' => 'MET','res_number' => '330','name' => 'CG'}},{'area' => '0.729','type' => 'V','atom2' => {'number' => '412','res_name' => 'LYS','res_number' => '53','name' => 'NZ'},'atom1' => {'number' => '2320','res_name' => 'MET','res_number' => '330','name' => 'CB'}},{'area' => '0.678','type' => 'V','atom2' => {'number' => '412','res_name' => 'LYS','res_number' => '53','name' => 'NZ'},'atom1' => {'number' => '2316','res_name' => 'MET','res_number' => '330','name' => 'N'}},{'area' => '0.186','type' => 'V','atom2' => {'number' => '412','res_name' => 'LYS','res_number' => '53','name' => 'NZ'},'atom1' => {'number' => '2304','res_name' => 'SER','res_number' => '328','name' => 'O'}},{'area' => '0.037','type' => 'V','atom2' => {'number' => '427','res_name' => 'ASN','res_number' => '56','name' => 'CA'},'atom1' => {'number' => '2261','res_name' => 'GLU','res_number' => '322','name' => 'OE1'}},{'area' => '2.239','type' => 'V','atom2' => {'number' => '430','res_name' => 'ASN','res_number' => '56','name' => 'CB'},'atom1' => {'number' => '2261','res_name' => 'GLU','res_number' => '322','name' => 'OE1'}},{'area' => '0.124','type' => 'V','atom2' => {'number' => '431','res_name' => 'ASN','res_number' => '56','name' => 'CG'},'atom1' => {'number' => '2323','res_name' => 'MET','res_number' => '330','name' => 'CE'}},{'area' => '2.413','type' => 'V','atom2' => {'number' => '432','res_name' => 'ASN','res_number' => '56','name' => 'ND2'},'atom1' => {'number' => '2323','res_name' => 'MET','res_number' => '330','name' => 'CE'}},{'area' => '2.663','type' => 'V','atom2' => {'number' => '434','res_name' => 'ARG','res_number' => '57','name' => 'N'},'atom1' => {'number' => '2261','res_name' => 'GLU','res_number' => '322','name' => 'OE1'}},{'area' => '0.156','type' => 'V','atom2' => {'number' => '437','res_name' => 'ARG','res_number' => '57','name' => 'O'},'atom1' => {'number' => '2259','res_name' => 'GLU','res_number' => '322','name' => 'CG'}},{'area' => '0.049','type' => 'V','atom2' => {'number' => '437','res_name' => 'ARG','res_number' => '57','name' => 'O'},'atom1' => {'number' => '2272','res_name' => 'GLN','res_number' => '324','name' => 'CA'}},{'area' => '0.245','type' => 'V','atom2' => {'number' => '437','res_name' => 'ARG','res_number' => '57','name' => 'O'},'atom1' => {'number' => '2266','res_name' => 'ILE','res_number' => '323','name' => 'O'}},{'area' => '2.792','type' => 'V','atom2' => {'number' => '437','res_name' => 'ARG','res_number' => '57','name' => 'O'},'atom1' => {'number' => '2278','res_name' => 'GLN','res_number' => '324','name' => 'NE2'}},{'area' => '0.350','type' => 'V','atom2' => {'number' => '437','res_name' => 'ARG','res_number' => '57','name' => 'O'},'atom1' => {'number' => '2277','res_name' => 'GLN','res_number' => '324','name' => 'CD'}},{'area' => '10.366','type' => 'V','atom2' => {'number' => '437','res_name' => 'ARG','res_number' => '57','name' => 'O'},'atom1' => {'number' => '2261','res_name' => 'GLU','res_number' => '322','name' => 'OE1'}},{'area' => '1.786','type' => 'V','atom2' => {'number' => '438','res_name' => 'ARG','res_number' => '57','name' => 'CB'},'atom1' => {'number' => '2261','res_name' => 'GLU','res_number' => '322','name' => 'OE1'}},{'area' => '0.462','type' => 'V','atom2' => {'number' => '446','res_name' => 'PHE','res_number' => '58','name' => 'CA'},'atom1' => {'number' => '2280','res_name' => 'GLY','res_number' => '325','name' => 'N'}},{'area' => '0.257','type' => 'V','atom2' => {'number' => '446','res_name' => 'PHE','res_number' => '58','name' => 'CA'},'atom1' => {'number' => '2272','res_name' => 'GLN','res_number' => '324','name' => 'CA'}},{'area' => '0.868','type' => 'V','atom2' => {'number' => '446','res_name' => 'PHE','res_number' => '58','name' => 'CA'},'atom1' => {'number' => '2279','res_name' => 'GLN','res_number' => '324','name' => 'OE1'}},{'area' => '0.931','type' => 'V','atom2' => {'number' => '446','res_name' => 'PHE','res_number' => '58','name' => 'CA'},'atom1' => {'number' => '2277','res_name' => 'GLN','res_number' => '324','name' => 'CD'}},{'area' => '0.156','type' => 'V','atom2' => {'number' => '450','res_name' => 'PHE','res_number' => '58','name' => 'CG'},'atom1' => {'number' => '2279','res_name' => 'GLN','res_number' => '324','name' => 'OE1'}},{'area' => '2.282','type' => 'V','atom2' => {'number' => '451','res_name' => 'PHE','res_number' => '58','name' => 'CD1'},'atom1' => {'number' => '2280','res_name' => 'GLY','res_number' => '325','name' => 'N'}},{'area' => '0.402','type' => 'V','atom2' => {'number' => '451','res_name' => 'PHE','res_number' => '58','name' => 'CD1'},'atom1' => {'number' => '2275','res_name' => 'GLN','res_number' => '324','name' => 'CB'}},{'area' => '0.760','type' => 'V','atom2' => {'number' => '451','res_name' => 'PHE','res_number' => '58','name' => 'CD1'},'atom1' => {'number' => '2273','res_name' => 'GLN','res_number' => '324','name' => 'C'}},{'area' => '1.361','type' => 'V','atom2' => {'number' => '451','res_name' => 'PHE','res_number' => '58','name' => 'CD1'},'atom1' => {'number' => '2272','res_name' => 'GLN','res_number' => '324','name' => 'CA'}},{'area' => '1.344','type' => 'V','atom2' => {'number' => '451','res_name' => 'PHE','res_number' => '58','name' => 'CD1'},'atom1' => {'number' => '2279','res_name' => 'GLN','res_number' => '324','name' => 'OE1'}},{'area' => '0.228','type' => 'V','atom2' => {'number' => '451','res_name' => 'PHE','res_number' => '58','name' => 'CD1'},'atom1' => {'number' => '2277','res_name' => 'GLN','res_number' => '324','name' => 'CD'}},{'area' => '0.650','type' => 'V','atom2' => {'number' => '451','res_name' => 'PHE','res_number' => '58','name' => 'CD1'},'atom1' => {'number' => '2282','res_name' => 'GLY','res_number' => '325','name' => 'C'}},{'area' => '1.285','type' => 'V','atom2' => {'number' => '451','res_name' => 'PHE','res_number' => '58','name' => 'CD1'},'atom1' => {'number' => '2281','res_name' => 'GLY','res_number' => '325','name' => 'CA'}},{'area' => '1.173','type' => 'V','atom2' => {'number' => '451','res_name' => 'PHE','res_number' => '58','name' => 'CD1'},'atom1' => {'number' => '2283','res_name' => 'GLY','res_number' => '325','name' => 'O'}},{'area' => '0.121','type' => 'V','atom2' => {'number' => '452','res_name' => 'PHE','res_number' => '58','name' => 'CD2'},'atom1' => {'number' => '2279','res_name' => 'GLN','res_number' => '324','name' => 'OE1'}},{'area' => '6.058','type' => 'V','atom2' => {'number' => '453','res_name' => 'PHE','res_number' => '58','name' => 'CE1'},'atom1' => {'number' => '2275','res_name' => 'GLN','res_number' => '324','name' => 'CB'}},{'area' => '2.690','type' => 'V','atom2' => {'number' => '453','res_name' => 'PHE','res_number' => '58','name' => 'CE1'},'atom1' => {'number' => '2274','res_name' => 'GLN','res_number' => '324','name' => 'O'}},{'area' => '2.935','type' => 'V','atom2' => {'number' => '453','res_name' => 'PHE','res_number' => '58','name' => 'CE1'},'atom1' => {'number' => '2273','res_name' => 'GLN','res_number' => '324','name' => 'C'}},{'area' => '0.181','type' => 'V','atom2' => {'number' => '453','res_name' => 'PHE','res_number' => '58','name' => 'CE1'},'atom1' => {'number' => '2272','res_name' => 'GLN','res_number' => '324','name' => 'CA'}},{'area' => '0.195','type' => 'V','atom2' => {'number' => '453','res_name' => 'PHE','res_number' => '58','name' => 'CE1'},'atom1' => {'number' => '2279','res_name' => 'GLN','res_number' => '324','name' => 'OE1'}},{'area' => '0.615','type' => 'V','atom2' => {'number' => '453','res_name' => 'PHE','res_number' => '58','name' => 'CE1'},'atom1' => {'number' => '2288','res_name' => 'ASP','res_number' => '326','name' => 'CB'}},{'area' => '4.358','type' => 'V','atom2' => {'number' => '453','res_name' => 'PHE','res_number' => '58','name' => 'CE1'},'atom1' => {'number' => '2283','res_name' => 'GLY','res_number' => '325','name' => 'O'}},{'area' => '0.402','type' => 'V','atom2' => {'number' => '454','res_name' => 'PHE','res_number' => '58','name' => 'CE2'},'atom1' => {'number' => '2279','res_name' => 'GLN','res_number' => '324','name' => 'OE1'}},{'area' => '0.054','type' => 'V','atom2' => {'number' => '454','res_name' => 'PHE','res_number' => '58','name' => 'CE2'},'atom1' => {'number' => '2276','res_name' => 'GLN','res_number' => '324','name' => 'CG'}},{'area' => '6.597','type' => 'V','atom2' => {'number' => '455','res_name' => 'PHE','res_number' => '58','name' => 'CZ'},'atom1' => {'number' => '2275','res_name' => 'GLN','res_number' => '324','name' => 'CB'}},{'area' => '0.172','type' => 'V','atom2' => {'number' => '455','res_name' => 'PHE','res_number' => '58','name' => 'CZ'},'atom1' => {'number' => '2279','res_name' => 'GLN','res_number' => '324','name' => 'OE1'}},{'area' => '2.727','type' => 'V','atom2' => {'number' => '455','res_name' => 'PHE','res_number' => '58','name' => 'CZ'},'atom1' => {'number' => '2276','res_name' => 'GLN','res_number' => '324','name' => 'CG'}},{'area' => '4.128','type' => 'V','atom2' => {'number' => '456','res_name' => 'SER','res_number' => '59','name' => 'N'},'atom1' => {'number' => '2279','res_name' => 'GLN','res_number' => '324','name' => 'OE1'}},{'area' => '0.717','type' => 'V','atom2' => {'number' => '456','res_name' => 'SER','res_number' => '59','name' => 'N'},'atom1' => {'number' => '2278','res_name' => 'GLN','res_number' => '324','name' => 'NE2'}},{'area' => '1.310','type' => 'V','atom2' => {'number' => '456','res_name' => 'SER','res_number' => '59','name' => 'N'},'atom1' => {'number' => '2277','res_name' => 'GLN','res_number' => '324','name' => 'CD'}},{'area' => '3.172','type' => 'V','atom2' => {'number' => '457','res_name' => 'SER','res_number' => '59','name' => 'CA'},'atom1' => {'number' => '2228','res_name' => 'ASN','res_number' => '318','name' => 'ND2'}},{'area' => '0.031','type' => 'V','atom2' => {'number' => '457','res_name' => 'SER','res_number' => '59','name' => 'CA'},'atom1' => {'number' => '2279','res_name' => 'GLN','res_number' => '324','name' => 'OE1'}},{'area' => '2.055','type' => 'V','atom2' => {'number' => '457','res_name' => 'SER','res_number' => '59','name' => 'CA'},'atom1' => {'number' => '2278','res_name' => 'GLN','res_number' => '324','name' => 'NE2'}},{'area' => '0.624','type' => 'V','atom2' => {'number' => '457','res_name' => 'SER','res_number' => '59','name' => 'CA'},'atom1' => {'number' => '2277','res_name' => 'GLN','res_number' => '324','name' => 'CD'}},{'area' => '5.014','type' => 'V','atom2' => {'number' => '460','res_name' => 'SER','res_number' => '59','name' => 'CB'},'atom1' => {'number' => '2228','res_name' => 'ASN','res_number' => '318','name' => 'ND2'}},{'area' => '0.612','type' => 'V','atom2' => {'number' => '460','res_name' => 'SER','res_number' => '59','name' => 'CB'},'atom1' => {'number' => '2279','res_name' => 'GLN','res_number' => '324','name' => 'OE1'}},{'area' => '1.793','type' => 'V','atom2' => {'number' => '460','res_name' => 'SER','res_number' => '59','name' => 'CB'},'atom1' => {'number' => '2278','res_name' => 'GLN','res_number' => '324','name' => 'NE2'}},{'area' => '5.219','type' => 'V','atom2' => {'number' => '460','res_name' => 'SER','res_number' => '59','name' => 'CB'},'atom1' => {'number' => '2277','res_name' => 'GLN','res_number' => '324','name' => 'CD'}},{'area' => '2.108','type' => 'V','atom2' => {'number' => '460','res_name' => 'SER','res_number' => '59','name' => 'CB'},'atom1' => {'number' => '2276','res_name' => 'GLN','res_number' => '324','name' => 'CG'}},{'area' => '11.448','type' => 'V','atom2' => {'number' => '460','res_name' => 'SER','res_number' => '59','name' => 'CB'},'atom1' => {'number' => '2226','res_name' => 'ASN','res_number' => '318','name' => 'CB'}},{'area' => '0.265','type' => 'V','atom2' => {'number' => '461','res_name' => 'SER','res_number' => '59','name' => 'OG'},'atom1' => {'number' => '2279','res_name' => 'GLN','res_number' => '324','name' => 'OE1'}},{'area' => '0.039','type' => 'V','atom2' => {'number' => '461','res_name' => 'SER','res_number' => '59','name' => 'OG'},'atom1' => {'number' => '2277','res_name' => 'GLN','res_number' => '324','name' => 'CD'}},{'area' => '5.704','type' => 'V','atom2' => {'number' => '461','res_name' => 'SER','res_number' => '59','name' => 'OG'},'atom1' => {'number' => '2276','res_name' => 'GLN','res_number' => '324','name' => 'CG'}},{'area' => '0.400','type' => 'V','atom2' => {'number' => '461','res_name' => 'SER','res_number' => '59','name' => 'OG'},'atom1' => {'number' => '2226','res_name' => 'ASN','res_number' => '318','name' => 'CB'}},{'Rno' => '2.484','area' => '0.810','angle' => '134.214','type' => 'H','atom2' => {'number' => '403','res_name' => 'TYR','res_number' => '52','name' => 'OH'},'atom1' => {'number' => '2266','res_name' => 'ILE','res_number' => '323','name' => 'O'}},{'Rno' => '2.546','area' => '0.233','angle' => '65.024','type' => 'H','atom2' => {'number' => '461','res_name' => 'SER','res_number' => '59','name' => 'OG'},'atom1' => {'number' => '2279','res_name' => 'GLN','res_number' => '324','name' => 'OE1'}},{'Rno' => '2.951','area' => '1.187','angle' => '86.470','type' => 'H','atom2' => {'number' => '403','res_name' => 'TYR','res_number' => '52','name' => 'OH'},'atom1' => {'number' => '2280','res_name' => 'GLY','res_number' => '325','name' => 'N'}}];
[{'area' => '20.033','res1' => {'number' => '318','name' => 'ASN'},'res2' => {'number' => '59','name' => 'SER'}},{'area' => '4.921','res1' => {'number' => '322','name' => 'GLU'},'res2' => {'number' => '52','name' => 'TYR'}},{'area' => '14.971','res1' => {'number' => '322','name' => 'GLU'},'res2' => {'number' => '57','name' => 'ARG'}},{'area' => '2.275','res1' => {'number' => '322','name' => 'GLU'},'res2' => {'number' => '56','name' => 'ASN'}},{'area' => '13.550','res1' => {'number' => '323','name' => 'ILE'},'res2' => {'number' => '52','name' => 'TYR'}},{'area' => '0.245','res1' => {'number' => '323','name' => 'ILE'},'res2' => {'number' => '57','name' => 'ARG'}},{'area' => '28.442','res1' => {'number' => '324','name' => 'GLN'},'res2' => {'number' => '58','name' => 'PHE'}},{'area' => '0.101','res1' => {'number' => '324','name' => 'GLN'},'res2' => {'number' => '52','name' => 'TYR'}},{'area' => '3.191','res1' => {'number' => '324','name' => 'GLN'},'res2' => {'number' => '57','name' => 'ARG'}},{'area' => '24.606','res1' => {'number' => '324','name' => 'GLN'},'res2' => {'number' => '59','name' => 'SER'}},{'area' => '10.209','res1' => {'number' => '325','name' => 'GLY'},'res2' => {'number' => '58','name' => 'PHE'}},{'area' => '27.126','res1' => {'number' => '325','name' => 'GLY'},'res2' => {'number' => '52','name' => 'TYR'}},{'area' => '6.016','res1' => {'number' => '325','name' => 'GLY'},'res2' => {'number' => '49','name' => 'LEU'}},{'area' => '20.577','res1' => {'number' => '326','name' => 'ASP'},'res2' => {'number' => '52','name' => 'TYR'}},{'area' => '12.106','res1' => {'number' => '326','name' => 'ASP'},'res2' => {'number' => '49','name' => 'LEU'}},{'area' => '9.598','res1' => {'number' => '326','name' => 'ASP'},'res2' => {'number' => '53','name' => 'LYS'}},{'area' => '0.615','res1' => {'number' => '326','name' => 'ASP'},'res2' => {'number' => '58','name' => 'PHE'}},{'area' => '12.506','res1' => {'number' => '327','name' => 'LYS'},'res2' => {'number' => '49','name' => 'LEU'}},{'area' => '11.351','res1' => {'number' => '328','name' => 'SER'},'res2' => {'number' => '53','name' => 'LYS'}},{'area' => '0.089','res1' => {'number' => '328','name' => 'SER'},'res2' => {'number' => '52','name' => 'TYR'}},{'area' => '1.786','res1' => {'number' => '329','name' => 'GLU'},'res2' => {'number' => '53','name' => 'LYS'}},{'area' => '29.027','res1' => {'number' => '330','name' => 'MET'},'res2' => {'number' => '53','name' => 'LYS'}},{'area' => '2.537','res1' => {'number' => '330','name' => 'MET'},'res2' => {'number' => '56','name' => 'ASN'}},{'area' => '10.543','res1' => {'number' => '330','name' => 'MET'},'res2' => {'number' => '52','name' => 'TYR'}},{'area' => '12.506','res1' => {'number' => '327','name' => 'LYS'},'res2' => {'number' => '49','name' => 'LEU'}},{'area' => '12.106','res1' => {'number' => '326','name' => 'ASP'},'res2' => {'number' => '49','name' => 'LEU'}},{'area' => '6.016','res1' => {'number' => '325','name' => 'GLY'},'res2' => {'number' => '49','name' => 'LEU'}},{'area' => '20.577','res1' => {'number' => '326','name' => 'ASP'},'res2' => {'number' => '52','name' => 'TYR'}},{'area' => '27.126','res1' => {'number' => '325','name' => 'GLY'},'res2' => {'number' => '52','name' => 'TYR'}},{'area' => '0.089','res1' => {'number' => '328','name' => 'SER'},'res2' => {'number' => '52','name' => 'TYR'}},{'area' => '0.101','res1' => {'number' => '324','name' => 'GLN'},'res2' => {'number' => '52','name' => 'TYR'}},{'area' => '13.550','res1' => {'number' => '323','name' => 'ILE'},'res2' => {'number' => '52','name' => 'TYR'}},{'area' => '10.543','res1' => {'number' => '330','name' => 'MET'},'res2' => {'number' => '52','name' => 'TYR'}},{'area' => '4.921','res1' => {'number' => '322','name' => 'GLU'},'res2' => {'number' => '52','name' => 'TYR'}},{'area' => '9.598','res1' => {'number' => '326','name' => 'ASP'},'res2' => {'number' => '53','name' => 'LYS'}},{'area' => '11.351','res1' => {'number' => '328','name' => 'SER'},'res2' => {'number' => '53','name' => 'LYS'}},{'area' => '29.027','res1' => {'number' => '330','name' => 'MET'},'res2' => {'number' => '53','name' => 'LYS'}},{'area' => '1.786','res1' => {'number' => '329','name' => 'GLU'},'res2' => {'number' => '53','name' => 'LYS'}},{'area' => '2.275','res1' => {'number' => '322','name' => 'GLU'},'res2' => {'number' => '56','name' => 'ASN'}},{'area' => '2.537','res1' => {'number' => '330','name' => 'MET'},'res2' => {'number' => '56','name' => 'ASN'}},{'area' => '14.971','res1' => {'number' => '322','name' => 'GLU'},'res2' => {'number' => '57','name' => 'ARG'}},{'area' => '3.191','res1' => {'number' => '324','name' => 'GLN'},'res2' => {'number' => '57','name' => 'ARG'}},{'area' => '0.245','res1' => {'number' => '323','name' => 'ILE'},'res2' => {'number' => '57','name' => 'ARG'}},{'area' => '10.209','res1' => {'number' => '325','name' => 'GLY'},'res2' => {'number' => '58','name' => 'PHE'}},{'area' => '28.442','res1' => {'number' => '324','name' => 'GLN'},'res2' => {'number' => '58','name' => 'PHE'}},{'area' => '0.615','res1' => {'number' => '326','name' => 'ASP'},'res2' => {'number' => '58','name' => 'PHE'}},{'area' => '24.606','res1' => {'number' => '324','name' => 'GLN'},'res2' => {'number' => '59','name' => 'SER'}},{'area' => '20.033','res1' => {'number' => '318','name' => 'ASN'},'res2' => {'number' => '59','name' => 'SER'}}];
