/*	$NetBSD: t_cgd_aes.c,v 1.7 2019/07/10 06:21:40 martin Exp $	*/
/*-
 * Copyright (c) 2016 The NetBSD Foundation, Inc.
 * Copyright (c) 2007 The Institute of Electrical and Electronics Engineers, Inc
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Alexander Nasonov.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE
 * COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/sysctl.h>

#include <atf-c.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <util.h>

#include <dev/cgdvar.h>

#include <rump/rump.h>
#include <rump/rump_syscalls.h>

#include "h_macros.h"

#define SECSIZE 512

struct testvec {
	unsigned int blkno;
	const uint8_t *ptxt;	/* PlainText  */
	const uint8_t *ctxt;	/* CipherText */
};

/*
 * 128 bits CBC key, NUL terminated.
 */
static const char aes_cbc_128_key[17] = {
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, /* 01234567 */
	0x38, 0x39, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, /* 89ABCDEF */
	0
};

/*
 * 192 bits CBC key, NUL terminated.
 */
static const char aes_cbc_192_key[25] = {
	0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, /* ABCDEFGH */
	0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, /* IJKLMNOP */
	0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, /* QRSTUVWX */
	0
};

/*
 * 256 bits CBC key, NUL terminated.
 */
static const char aes_cbc_256_key[33] = {
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, /* 01234567 */
	0x38, 0x39, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, /* 89ABCDEF */
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, /* 01234567 */
	0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, /* 89abcdef */
	0
};

static const uint8_t aes_cbc_ptxt[SECSIZE] =
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop"
	"                abcdefghijklmnop";

/*
 * IV method encblkno1, blkno 0.
 */
static const uint8_t aes_cbc_128_encblkno1_vec0_ctxt[SECSIZE] = {
	0x1e, 0x95, 0x12, 0x15, 0xf6, 0xe0, 0xa7, 0x56,
	0x95, 0xa0, 0xa7, 0x35, 0x77, 0xf4, 0xdd, 0xdc,
	0x37, 0xc0, 0x28, 0x20, 0x00, 0x79, 0xa0, 0x35,
	0xe0, 0x83, 0x23, 0x95, 0x4e, 0xea, 0x8d, 0xa2,
	0x11, 0xbf, 0x9a, 0xd5, 0x21, 0x1e, 0x15, 0xb9,
	0xd1, 0x2e, 0xd2, 0xd9, 0xa5, 0xcc, 0x26, 0x75,
	0xba, 0x3e, 0x30, 0x11, 0xb2, 0x40, 0xdd, 0x1d,
	0x07, 0x3b, 0xe6, 0x00, 0xa7, 0x31, 0x9e, 0x58,
	0x41, 0xf3, 0x02, 0xf5, 0xad, 0x35, 0x79, 0x9a,
	0x9e, 0x03, 0xc8, 0x7a, 0x9d, 0x1d, 0x58, 0x9f,
	0x98, 0x67, 0xe2, 0x49, 0x81, 0x0c, 0x23, 0x90,
	0xd8, 0xc6, 0xf0, 0xc5, 0x73, 0x46, 0xd5, 0x14,
	0x1d, 0x78, 0x24, 0x7c, 0x9f, 0x5c, 0x8c, 0xe6,
	0x5d, 0x85, 0x7a, 0x5f, 0x76, 0xcc, 0xd8, 0xe9,
	0x03, 0xff, 0xfd, 0xd4, 0x12, 0x3f, 0xcb, 0xb0,
	0xfe, 0xfd, 0x86, 0x00, 0x0c, 0xe3, 0xdd, 0xa6,
	0x89, 0x92, 0xfe, 0xc8, 0x07, 0x5a, 0x94, 0x55,
	0x75, 0xae, 0x68, 0x47, 0xba, 0x84, 0x75, 0x58,
	0x33, 0x30, 0x2c, 0x16, 0x5b, 0xe9, 0x3f, 0x2a,
	0x09, 0xf9, 0x69, 0x23, 0x77, 0xd7, 0x2b, 0x95,
	0x4b, 0x78, 0x59, 0xcc, 0xfa, 0xf5, 0x79, 0xd2,
	0x05, 0x87, 0x66, 0x57, 0x93, 0xbf, 0x05, 0x90,
	0x4d, 0x6d, 0xd2, 0x72, 0x92, 0x24, 0xec, 0x14,
	0xe7, 0xbf, 0x82, 0x57, 0xbb, 0x14, 0x51, 0xe6,
	0xce, 0x3f, 0xa1, 0xfc, 0x63, 0x75, 0xee, 0xde,
	0xf9, 0x31, 0xd3, 0xa0, 0x07, 0xcd, 0x4d, 0x8f,
	0x83, 0x7d, 0x65, 0xe1, 0xc6, 0x60, 0x9e, 0x5c,
	0x51, 0x76, 0xfa, 0x64, 0xdf, 0xdc, 0xaf, 0x38,
	0xee, 0xe9, 0x8f, 0x4b, 0xa0, 0x3a, 0x21, 0xdf,
	0x58, 0x3b, 0x73, 0xf5, 0x30, 0xbb, 0x29, 0xe0,
	0xff, 0x60, 0xf0, 0x05, 0x5e, 0x37, 0xbc, 0x78,
	0x95, 0x3f, 0xa8, 0xd4, 0xb4, 0x82, 0x0d, 0xe1,
	0x10, 0xe3, 0xa7, 0x61, 0x37, 0x58, 0x28, 0x14,
	0x22, 0x57, 0x32, 0x28, 0x80, 0x98, 0x3e, 0x5f,
	0x71, 0xcf, 0x34, 0xb8, 0x6d, 0x6b, 0xc0, 0x23,
	0xc1, 0x9e, 0x58, 0x4f, 0xd5, 0xa4, 0x14, 0x03,
	0x2a, 0xed, 0xc4, 0xa7, 0x77, 0x7c, 0x4f, 0x94,
	0x91, 0x1d, 0x47, 0x34, 0x82, 0xe8, 0x9d, 0x32,
	0x5c, 0xc7, 0x38, 0xe9, 0x92, 0xcd, 0x35, 0xfd,
	0x1c, 0xcc, 0x3c, 0x28, 0x75, 0x6f, 0xff, 0xd5,
	0xe8, 0xbf, 0x90, 0x92, 0x34, 0x13, 0x11, 0x89,
	0xe0, 0xa2, 0x25, 0xeb, 0x82, 0x63, 0x31, 0x80,
	0x50, 0x6c, 0x99, 0xaa, 0x97, 0x0e, 0x59, 0x45,
	0x64, 0xb8, 0x77, 0x78, 0x6b, 0x24, 0xac, 0xc0,
	0xc9, 0xa9, 0xbc, 0x13, 0xd1, 0x5e, 0x50, 0x9a,
	0x91, 0x1a, 0x08, 0xf7, 0xc5, 0x18, 0x9f, 0x87,
	0x97, 0x9c, 0x0a, 0x27, 0xf1, 0x66, 0xf8, 0x09,
	0x52, 0x09, 0x41, 0x07, 0xc1, 0xa1, 0x91, 0xa4,
	0x59, 0x09, 0x75, 0x57, 0x5b, 0x53, 0x79, 0x58,
	0xa2, 0x9e, 0x49, 0xa2, 0x5e, 0xf7, 0x28, 0x1c,
	0x43, 0xa6, 0xcb, 0x88, 0x46, 0x84, 0xc9, 0x7f,
	0x84, 0xdb, 0x45, 0x0c, 0xb3, 0x7f, 0x01, 0x40,
	0x71, 0x3e, 0x48, 0x12, 0x1f, 0xbc, 0x1e, 0xdf,
	0x41, 0x50, 0xb2, 0x11, 0x67, 0x83, 0x19, 0x04,
	0x0e, 0x21, 0xd5, 0xf2, 0x54, 0x99, 0xfb, 0x47,
	0xf2, 0x5e, 0x02, 0x4b, 0x61, 0x6d, 0xef, 0x78,
	0x29, 0xe4, 0x3a, 0x56, 0x14, 0x20, 0x6f, 0x70,
	0x82, 0xea, 0x5d, 0xbc, 0x48, 0x89, 0x34, 0x69,
	0xdb, 0x4a, 0x06, 0xa7, 0xd6, 0xc7, 0xb7, 0x06,
	0x8e, 0x64, 0x21, 0x3e, 0xa6, 0x32, 0x61, 0x59,
	0x03, 0xea, 0xc3, 0x71, 0xf0, 0x26, 0x02, 0xe0,
	0x71, 0x95, 0x38, 0x11, 0x32, 0xe6, 0x3b, 0x25,
	0x53, 0x14, 0x24, 0x34, 0xe8, 0x8c, 0xa8, 0xef,
	0x52, 0xfe, 0x06, 0x2c, 0x20, 0x88, 0x4f, 0xa6,
};

/*
 * IV method encblkno1, blkno 1.
 */
static const uint8_t aes_cbc_128_encblkno1_vec1_ctxt[SECSIZE] = {
	0x2f, 0x69, 0x3e, 0x95, 0x87, 0x91, 0x99, 0xd4,
	0xd9, 0x5d, 0xf2, 0x52, 0x32, 0x54, 0x2a, 0x80,
	0xa0, 0x77, 0x6e, 0x73, 0x15, 0xb4, 0xc9, 0x13,
	0x85, 0xed, 0x79, 0x9b, 0x84, 0x0a, 0x7e, 0xdb,
	0xee, 0x09, 0x78, 0x11, 0x28, 0xd5, 0x26, 0xec,
	0x1d, 0x52, 0xba, 0x33, 0x26, 0xeb, 0x91, 0xc6,
	0x4b, 0xf0, 0x38, 0xdf, 0x9f, 0x9d, 0x6c, 0xd8,
	0x49, 0x83, 0x88, 0xbe, 0x62, 0x2d, 0x5e, 0x88,
	0xc0, 0x35, 0xe4, 0xc3, 0xc9, 0x9f, 0x62, 0x59,
	0x16, 0xa7, 0x2e, 0xc0, 0xda, 0x3c, 0x2e, 0x10,
	0x53, 0xf0, 0x84, 0x27, 0x38, 0xd0, 0xf4, 0xb5,
	0x7c, 0x4a, 0x63, 0x04, 0x51, 0x22, 0xae, 0xf3,
	0xe7, 0x97, 0x53, 0xee, 0xe6, 0xaf, 0xc3, 0x49,
	0x3a, 0x5a, 0x74, 0x83, 0x18, 0xa3, 0x6b, 0xf3,
	0x6a, 0x3b, 0xe2, 0x1b, 0xd4, 0x64, 0x41, 0xdf,
	0xd1, 0xd2, 0xdd, 0x22, 0xa8, 0x66, 0xbd, 0x8e,
	0xc4, 0x9a, 0x6d, 0x15, 0x38, 0x5b, 0x50, 0x9a,
	0x65, 0x48, 0x97, 0xf1, 0x04, 0x85, 0x8b, 0x5c,
	0x44, 0x32, 0x15, 0xea, 0x28, 0x5f, 0x98, 0x53,
	0xb4, 0x80, 0xd0, 0x2c, 0x59, 0x04, 0x08, 0xaf,
	0xa4, 0xb7, 0x49, 0xd1, 0x98, 0x87, 0xb9, 0xb6,
	0x3d, 0x89, 0xd1, 0xbe, 0xf4, 0x89, 0xec, 0xf9,
	0x2d, 0xc7, 0xc6, 0xe9, 0xe6, 0xfa, 0x1e, 0x67,
	0x68, 0xe7, 0xb7, 0x91, 0x55, 0x77, 0xf3, 0x27,
	0x38, 0x23, 0xcf, 0x2e, 0x3e, 0x8b, 0xfd, 0xb3,
	0x90, 0xd8, 0x6b, 0x1e, 0x93, 0x8f, 0xb6, 0xc1,
	0x27, 0xc2, 0xb7, 0x76, 0x10, 0x69, 0xe8, 0x7f,
	0xfc, 0x03, 0x59, 0xa4, 0xd3, 0x7f, 0x2f, 0x03,
	0x1c, 0x21, 0x6d, 0x2e, 0xae, 0xba, 0xa2, 0x04,
	0x67, 0xe9, 0x33, 0xc9, 0x3a, 0x96, 0xb6, 0x7c,
	0xf6, 0x21, 0x6b, 0x34, 0x9a, 0x5b, 0xa0, 0x8b,
	0x51, 0xf0, 0xd4, 0x3a, 0xa3, 0xcb, 0x22, 0xfb,
	0x8a, 0x56, 0xab, 0x9a, 0x15, 0x75, 0x07, 0x87,
	0x32, 0xa7, 0x15, 0xc7, 0xd9, 0x40, 0x95, 0xe5,
	0xfb, 0xb0, 0xc5, 0xb1, 0x60, 0xf8, 0xcc, 0x8b,
	0x30, 0x20, 0xd9, 0x84, 0x6f, 0xa2, 0xcb, 0x72,
	0xf5, 0xa5, 0x2c, 0xa3, 0xc6, 0x1c, 0xd2, 0x74,
	0x01, 0x74, 0xdd, 0xb4, 0x68, 0x3b, 0x3b, 0x3e,
	0x4f, 0xb5, 0x67, 0x9a, 0x9c, 0x37, 0x3d, 0xbf,
	0xd3, 0xab, 0xd7, 0x70, 0x03, 0x28, 0x5c, 0x3b,
	0xb7, 0x08, 0x38, 0x3d, 0x69, 0xa9, 0xcb, 0x63,
	0x04, 0x95, 0x8a, 0x16, 0x4c, 0xff, 0x9f, 0x0c,
	0xe2, 0x51, 0x95, 0x44, 0x52, 0x3b, 0x59, 0x9d,
	0x0b, 0x77, 0xa0, 0x39, 0x40, 0xea, 0x33, 0x25,
	0xc8, 0xc5, 0x90, 0x47, 0x23, 0xe3, 0x03, 0x8c,
	0x6a, 0xe0, 0x4f, 0x76, 0xe7, 0x72, 0x82, 0xcc,
	0xb2, 0xfd, 0xfb, 0x82, 0x1a, 0x28, 0x30, 0x89,
	0x0e, 0x25, 0xa7, 0x63, 0x85, 0x2e, 0x9b, 0xa6,
	0x0b, 0xa0, 0xb5, 0x34, 0xa2, 0x2e, 0x7f, 0xd4,
	0xe5, 0xd6, 0x95, 0xe8, 0x09, 0x3d, 0x4d, 0xdf,
	0xd9, 0xc0, 0x63, 0x17, 0xa5, 0x9c, 0xf6, 0xa3,
	0x59, 0x17, 0xc0, 0xf8, 0xa2, 0x11, 0x14, 0x88,
	0xf0, 0x1e, 0x4a, 0x4b, 0x13, 0xf6, 0xd6, 0x09,
	0xac, 0xf8, 0x39, 0x5d, 0x4c, 0x68, 0x69, 0x99,
	0x08, 0xd4, 0xf5, 0x39, 0x6d, 0x78, 0xde, 0xb5,
	0x6f, 0x34, 0xc4, 0x28, 0x73, 0x6c, 0x29, 0xa1,
	0xef, 0xfe, 0xed, 0x56, 0xb2, 0x70, 0x7b, 0xd5,
	0x5b, 0xd1, 0x09, 0x6a, 0x9a, 0x59, 0xe9, 0x79,
	0xe9, 0xee, 0xa4, 0x03, 0xc1, 0x67, 0xce, 0x62,
	0xf6, 0x4f, 0x04, 0xa5, 0x04, 0x71, 0x13, 0xeb,
	0x3d, 0x0a, 0x65, 0x2f, 0x57, 0xb0, 0xc0, 0xa4,
	0xf2, 0x8d, 0x78, 0x90, 0xeb, 0xc9, 0x5e, 0x8b,
	0xd8, 0xfb, 0xbc, 0x74, 0x1a, 0x70, 0x94, 0x2c,
	0xeb, 0xf2, 0x5e, 0x6d, 0xbb, 0x96, 0x7a, 0x2c,
};

/*
 * IV method encblkno1, blkno 2.
 */
static const uint8_t aes_cbc_128_encblkno1_vec2_ctxt[SECSIZE] = {
	0xbc, 0x49, 0x35, 0x2c, 0xe3, 0x10, 0x12, 0x65,
	0x7a, 0xf4, 0xde, 0xd3, 0xf8, 0xe1, 0x49, 0x97,
	0x0a, 0x07, 0x93, 0x6c, 0xf8, 0x0e, 0xb7, 0xdf,
	0x53, 0xba, 0x1e, 0x8e, 0x14, 0xbd, 0xf6, 0x81,
	0xd6, 0xf6, 0x3d, 0xb2, 0xe7, 0x6a, 0x9d, 0x50,
	0x68, 0xc2, 0x75, 0x8e, 0xfb, 0x44, 0xfa, 0xc8,
	0x9f, 0x30, 0x15, 0xd5, 0xbe, 0xce, 0x04, 0xc1,
	0x99, 0xde, 0x3d, 0x2b, 0xc1, 0xc4, 0x8a, 0xb1,
	0xc5, 0x54, 0x47, 0x52, 0xf6, 0x38, 0x11, 0xcb,
	0x11, 0xf6, 0xb7, 0xbd, 0x4d, 0x24, 0xa1, 0xac,
	0x04, 0x17, 0x7e, 0x3c, 0xbc, 0x3b, 0xa0, 0x8d,
	0xfb, 0x22, 0x82, 0x56, 0xa2, 0xbe, 0xfe, 0xe7,
	0xde, 0xa4, 0xe9, 0xeb, 0xa0, 0x7a, 0x45, 0xc9,
	0x18, 0x0b, 0x14, 0xd5, 0xff, 0x4c, 0xe5, 0x86,
	0xac, 0xac, 0xaa, 0xb4, 0x70, 0x0c, 0x4a, 0x20,
	0xb6, 0xd8, 0x2d, 0xac, 0x09, 0xd8, 0xf6, 0x24,
	0xdf, 0xa5, 0x62, 0xef, 0x8c, 0x01, 0xa8, 0x1d,
	0x8f, 0x52, 0xee, 0xa6, 0x2f, 0x9b, 0x81, 0x18,
	0x0e, 0x6b, 0xa3, 0xe5, 0x67, 0xb3, 0xd5, 0x30,
	0xb1, 0x9f, 0x87, 0x05, 0xd0, 0x52, 0x62, 0x6f,
	0xb9, 0x3b, 0xbc, 0x0c, 0x0c, 0xcb, 0x73, 0x55,
	0x23, 0x83, 0x14, 0x78, 0x05, 0x5b, 0x05, 0xf5,
	0x3e, 0xa7, 0xf3, 0x4d, 0x0d, 0x34, 0x6f, 0xe0,
	0x58, 0x52, 0x0a, 0x82, 0xa7, 0x49, 0x8a, 0xd2,
	0x23, 0xb1, 0xc5, 0x0d, 0xa7, 0x0f, 0x56, 0xfc,
	0x7e, 0xf6, 0x19, 0x4b, 0xe7, 0x63, 0x72, 0x4c,
	0xb8, 0x5c, 0x80, 0x54, 0xf5, 0x1f, 0xb0, 0x29,
	0x40, 0x88, 0x75, 0x54, 0x42, 0xca, 0x2c, 0xc3,
	0xcf, 0xd7, 0xc1, 0xb2, 0xd6, 0x90, 0x70, 0x5e,
	0xf5, 0x58, 0x70, 0xe0, 0xff, 0x5a, 0xf5, 0xee,
	0x32, 0x4f, 0x61, 0x1c, 0xf6, 0xbf, 0xd5, 0x7c,
	0x73, 0xb9, 0x1d, 0x30, 0xc2, 0xfb, 0x2f, 0x9a,
	0xf7, 0x57, 0x2e, 0x87, 0x7d, 0xcb, 0xdd, 0x7e,
	0xda, 0xec, 0x47, 0x1a, 0x0e, 0x70, 0x2d, 0x6e,
	0x18, 0x2b, 0x89, 0xc1, 0x85, 0x58, 0x6d, 0x4b,
	0x45, 0x11, 0xcf, 0x82, 0x9f, 0x31, 0xd0, 0x42,
	0x11, 0xca, 0xa8, 0x52, 0x66, 0xf7, 0xf1, 0x1d,
	0x86, 0xe3, 0xb4, 0x41, 0xcb, 0x92, 0xb1, 0x9f,
	0x8d, 0x8e, 0x08, 0xe9, 0xc4, 0x66, 0xce, 0x9d,
	0xae, 0x91, 0xaf, 0xe6, 0xa6, 0x2e, 0x06, 0x3a,
	0xf5, 0x27, 0x48, 0xe4, 0x31, 0x0f, 0xc5, 0xdf,
	0x29, 0x56, 0xed, 0x62, 0xf3, 0xef, 0xca, 0xa6,
	0x58, 0xd1, 0x84, 0x99, 0xd3, 0x34, 0x67, 0x92,
	0x6a, 0xb2, 0xd1, 0xd1, 0x50, 0x1f, 0xe9, 0xd8,
	0x3c, 0xbe, 0x12, 0x97, 0x7c, 0x4f, 0xc0, 0xbe,
	0x91, 0x32, 0x15, 0xd5, 0xf2, 0x5e, 0xe6, 0x13,
	0x86, 0xfa, 0xc6, 0xde, 0xd8, 0xe1, 0x70, 0xb4,
	0xf7, 0x5b, 0x9f, 0x79, 0x55, 0x22, 0x7a, 0xe1,
	0x54, 0x40, 0x39, 0x11, 0xe1, 0x78, 0x01, 0x01,
	0xc0, 0x44, 0xeb, 0x92, 0xb9, 0x01, 0xdd, 0x56,
	0xb9, 0x7e, 0xa0, 0x50, 0x41, 0x58, 0xb2, 0x13,
	0x12, 0x44, 0xd2, 0x39, 0xf2, 0x76, 0x3c, 0x53,
	0x36, 0xfe, 0x17, 0x74, 0x91, 0x8d, 0xbe, 0xc5,
	0x40, 0xf7, 0xa2, 0xe9, 0x65, 0xd9, 0xdf, 0x80,
	0x7b, 0xd9, 0xc3, 0x1f, 0xb4, 0xfc, 0x9f, 0x8d,
	0x7a, 0x4e, 0x1e, 0x32, 0x6d, 0xb1, 0x82, 0x1e,
	0x0c, 0xb6, 0x0b, 0xe6, 0x15, 0x82, 0x5c, 0xcc,
	0xc8, 0x4a, 0x73, 0x56, 0x9d, 0x11, 0xfa, 0xcd,
	0x21, 0x95, 0x23, 0x71, 0xe8, 0xfe, 0x06, 0x43,
	0xf6, 0x17, 0x51, 0x28, 0x0d, 0xfb, 0x0a, 0x49,
	0x1b, 0x35, 0x1e, 0x4a, 0x38, 0x08, 0x6b, 0xcd,
	0x67, 0x21, 0x94, 0x09, 0x28, 0xca, 0x0a, 0x18,
	0xdf, 0x6c, 0x86, 0x47, 0x10, 0x29, 0x81, 0x9a,
	0x73, 0xba, 0x78, 0xbd, 0xc0, 0x61, 0xee, 0x23,
};

/*
 * IV method encblkno1, blkno 3.
 */
static const uint8_t aes_cbc_128_encblkno1_vec3_ctxt[SECSIZE] = {
	0x29, 0x9f, 0xb1, 0x0f, 0x7d, 0xb4, 0xd9, 0xbb,
	0xf9, 0x06, 0x60, 0xdc, 0xb9, 0xeb, 0x73, 0x6e,
	0xfe, 0xdb, 0x99, 0x29, 0xe8, 0x42, 0x46, 0xe7,
	0x49, 0xcf, 0x90, 0x2d, 0x08, 0xd7, 0xd5, 0xbf,
	0x0f, 0x4f, 0x66, 0xce, 0xcd, 0xb1, 0x8a, 0xc7,
	0x47, 0xc9, 0x8e, 0xe3, 0x9f, 0x80, 0x79, 0xc6,
	0xa8, 0xe5, 0x20, 0x66, 0x58, 0xde, 0xab, 0x87,
	0x5e, 0x7e, 0xcd, 0x55, 0x81, 0x09, 0x40, 0xd9,
	0x8b, 0x7e, 0xd3, 0xf9, 0x16, 0x55, 0x72, 0x7d,
	0xe8, 0x36, 0x76, 0x06, 0x77, 0x47, 0xa5, 0xdc,
	0x80, 0x33, 0x7d, 0x88, 0x5f, 0x56, 0x48, 0x0f,
	0x66, 0xb5, 0x91, 0x9d, 0xf8, 0xdb, 0x83, 0x0d,
	0xd4, 0xc6, 0x13, 0xfc, 0xd4, 0xe5, 0x34, 0x81,
	0x70, 0x4d, 0x96, 0x82, 0x5d, 0xb2, 0x36, 0x37,
	0xdf, 0xd2, 0x5e, 0x31, 0xf0, 0x9d, 0x6d, 0xb7,
	0xf9, 0x2d, 0xcb, 0x77, 0xb8, 0x59, 0xa0, 0xbb,
	0x4f, 0x8d, 0xa0, 0xd1, 0x49, 0x17, 0x93, 0x3c,
	0xf0, 0x4e, 0x72, 0xdd, 0x99, 0x9a, 0x87, 0xf1,
	0x01, 0x89, 0xdf, 0xef, 0xed, 0x04, 0x86, 0x3d,
	0x9b, 0xab, 0x6c, 0xa7, 0xef, 0x1b, 0xbb, 0x24,
	0xbc, 0x65, 0x01, 0x06, 0x12, 0x3f, 0x7e, 0x9f,
	0x83, 0xf3, 0xd4, 0x43, 0x18, 0x03, 0xa3, 0x07,
	0xbc, 0x85, 0xe8, 0xdb, 0x6c, 0x8f, 0xaf, 0x70,
	0x71, 0x5d, 0xbc, 0x6d, 0x14, 0x05, 0xdf, 0xce,
	0x9f, 0xe2, 0xa3, 0x51, 0x66, 0x92, 0x52, 0x19,
	0x98, 0xbd, 0xb2, 0x68, 0x79, 0xf4, 0x5d, 0x71,
	0xcb, 0xa0, 0x1b, 0x77, 0x34, 0x46, 0x69, 0x3c,
	0xa4, 0x0f, 0x72, 0xf5, 0x73, 0xf6, 0xa0, 0xe9,
	0x72, 0xef, 0xa1, 0xcc, 0x43, 0xfc, 0xb7, 0xf3,
	0x59, 0xeb, 0x40, 0x61, 0x02, 0x26, 0x9b, 0x71,
	0x57, 0x17, 0x36, 0xac, 0xc8, 0xd5, 0x9d, 0xcb,
	0x4d, 0x4f, 0xf7, 0xc1, 0x58, 0xce, 0xbf, 0x73,
	0xe7, 0xd0, 0x58, 0x0d, 0x08, 0x01, 0x8f, 0x68,
	0x1b, 0x3f, 0x3a, 0x7b, 0xdb, 0x9e, 0xa7, 0x33,
	0x59, 0x91, 0xa8, 0xe3, 0x58, 0x22, 0x3f, 0x97,
	0xe1, 0xdb, 0xd6, 0x99, 0x0e, 0xdd, 0x76, 0xcd,
	0x4d, 0x02, 0x28, 0x43, 0x8a, 0xdd, 0x10, 0xad,
	0x55, 0xe0, 0x62, 0xf7, 0x44, 0xbc, 0x3f, 0x99,
	0x3c, 0x09, 0x25, 0xfb, 0xfd, 0x1e, 0x4c, 0x45,
	0x0e, 0x6e, 0x75, 0x15, 0x48, 0x19, 0x08, 0xc3,
	0x2b, 0x81, 0x60, 0x5f, 0x19, 0x09, 0x74, 0x0a,
	0xbd, 0x0d, 0x8d, 0x94, 0x55, 0x04, 0x2b, 0x8e,
	0x0d, 0x10, 0x60, 0x64, 0x0d, 0x7f, 0x63, 0x2e,
	0x89, 0x0b, 0xfc, 0x1c, 0xbc, 0xf3, 0x66, 0xc5,
	0x80, 0x93, 0x3a, 0x74, 0x15, 0x11, 0xd5, 0xba,
	0xbc, 0x06, 0x3f, 0x85, 0xcc, 0x6c, 0xd3, 0xf2,
	0x74, 0xc6, 0x10, 0x15, 0x0a, 0x02, 0x66, 0xa4,
	0xa8, 0x93, 0x0b, 0x5c, 0xe7, 0x13, 0x96, 0x90,
	0xdd, 0xe3, 0x25, 0x22, 0x46, 0x7b, 0x49, 0xde,
	0x72, 0x55, 0xf3, 0x30, 0x62, 0x5f, 0x7a, 0x2a,
	0x37, 0x88, 0xea, 0x57, 0x99, 0x64, 0x50, 0x2d,
	0xd3, 0x6a, 0x09, 0x4b, 0xd6, 0x61, 0xf2, 0x22,
	0x53, 0x36, 0xf7, 0x42, 0x21, 0xde, 0xda, 0xcb,
	0x8b, 0x6f, 0xf3, 0x4e, 0x2c, 0x16, 0xfb, 0xfc,
	0x13, 0xa7, 0xd1, 0xd8, 0xfd, 0x16, 0x39, 0x20,
	0xe0, 0xb2, 0xb4, 0xd5, 0x40, 0x93, 0x30, 0xf3,
	0xab, 0x88, 0xe3, 0xfb, 0xbe, 0xb8, 0x02, 0x3a,
	0x78, 0x2d, 0x56, 0x4b, 0x92, 0x5b, 0x0a, 0x8d,
	0x18, 0xa4, 0x5b, 0x11, 0x60, 0x0b, 0x45, 0xad,
	0x0b, 0x64, 0x96, 0x7d, 0x84, 0xf2, 0x20, 0xa8,
	0x95, 0x78, 0xb3, 0xb5, 0x98, 0x1f, 0xa7, 0x3e,
	0x30, 0x77, 0x43, 0xd2, 0x8c, 0x20, 0xc5, 0x5e,
	0x76, 0xcd, 0x2c, 0x7c, 0xfa, 0x34, 0x36, 0xda,
	0x39, 0x00, 0x2e, 0x69, 0x4a, 0xb3, 0x0f, 0x6f,
};

const struct testvec aes_cbc_128_1_vectors[] = {
	{
		.blkno = 0,
		.ptxt = aes_cbc_ptxt,
		.ctxt = aes_cbc_128_encblkno1_vec0_ctxt,
	},
	{
		.blkno = 1,
		.ptxt = aes_cbc_ptxt,
		.ctxt = aes_cbc_128_encblkno1_vec1_ctxt,
	},
	{
		.blkno = 2,
		.ptxt = aes_cbc_ptxt,
		.ctxt = aes_cbc_128_encblkno1_vec2_ctxt,
	},
	{
		.blkno = 3,
		.ptxt = aes_cbc_ptxt,
		.ctxt = aes_cbc_128_encblkno1_vec3_ctxt,
	},
};

/*
 * IV method encblkno8, blkno 0.
 */
static const uint8_t aes_cbc_128_encblkno8_vec0_ctxt[SECSIZE] = {
	0xa6, 0x64, 0xef, 0x0f, 0xc4, 0x45, 0xcc, 0x5e,
	0xf8, 0x27, 0x42, 0x5e, 0xbd, 0x93, 0x99, 0xcd,
	0x79, 0xa8, 0x7d, 0x72, 0xc4, 0x02, 0x99, 0xa6,
	0xe4, 0x69, 0x57, 0x82, 0xdf, 0x32, 0x4e, 0x67,
	0x2a, 0xd9, 0x58, 0x8c, 0x9f, 0xfc, 0x4d, 0xcf,
	0x7b, 0xa4, 0xa1, 0xfa, 0xd9, 0x4d, 0xb5, 0x67,
	0x06, 0x4a, 0x9e, 0x6d, 0xe8, 0xaa, 0xdd, 0xae,
	0x8c, 0xda, 0xcf, 0x26, 0xd5, 0x94, 0x8d, 0x12,
	0xf8, 0xdd, 0x21, 0x4c, 0xcb, 0xc8, 0x5d, 0xd1,
	0x6a, 0x89, 0x37, 0xd0, 0x32, 0xe4, 0x87, 0xbc,
	0x5d, 0xef, 0xca, 0x38, 0xd5, 0x08, 0xfb, 0xcf,
	0xb7, 0x8d, 0x65, 0x52, 0x13, 0xea, 0x2d, 0x30,
	0xd3, 0x9a, 0xe8, 0x9c, 0x76, 0x25, 0x44, 0x2a,
	0xf1, 0xe1, 0xbb, 0xcd, 0xbc, 0x9c, 0xf5, 0xa3,
	0xfb, 0x23, 0x53, 0x95, 0x61, 0xea, 0x46, 0x97,
	0xf6, 0xbf, 0xdf, 0xf9, 0xb7, 0x94, 0x73, 0xa8,
	0xc1, 0xaa, 0x64, 0xfb, 0x66, 0x26, 0x0f, 0x4c,
	0xee, 0x3c, 0xb6, 0x3f, 0x13, 0x88, 0xaa, 0x7d,
	0x91, 0x01, 0x1a, 0x95, 0x3b, 0xb5, 0x7e, 0x1f,
	0xc1, 0x84, 0xa6, 0x83, 0x99, 0xe6, 0xaf, 0x21,
	0x33, 0xff, 0x2e, 0xc9, 0xfe, 0xf2, 0xa1, 0x37,
	0xed, 0x91, 0x73, 0x70, 0x4f, 0xb4, 0x69, 0x69,
	0x98, 0x1d, 0x6d, 0xd4, 0xa4, 0xac, 0x73, 0x61,
	0x04, 0xf5, 0x13, 0x50, 0x2a, 0xa9, 0xf7, 0x61,
	0x78, 0xf5, 0x87, 0x26, 0xc5, 0x4a, 0x30, 0xbb,
	0x94, 0x55, 0x51, 0xb4, 0xa0, 0x83, 0x30, 0xe6,
	0xf7, 0xc7, 0x89, 0x61, 0x73, 0xd9, 0xbd, 0xe1,
	0x4e, 0x14, 0x8a, 0x02, 0x3d, 0x7a, 0x58, 0x92,
	0x41, 0xe7, 0x90, 0x8d, 0xd7, 0x67, 0x62, 0xf8,
	0x99, 0xa7, 0x9d, 0x55, 0xec, 0x18, 0x6b, 0x42,
	0xaf, 0x27, 0x97, 0xe5, 0x51, 0xa9, 0x10, 0x27,
	0x5e, 0x3f, 0xac, 0xda, 0xd3, 0xb5, 0x2b, 0x43,
	0x2e, 0x10, 0xdc, 0xd9, 0xe2, 0x2f, 0x4f, 0xfe,
	0xf3, 0x0d, 0x06, 0x76, 0xf9, 0x25, 0xcd, 0x26,
	0xef, 0x8e, 0x9b, 0xc2, 0xb3, 0x20, 0x2b, 0x00,
	0xb6, 0xe6, 0x2e, 0xf7, 0x17, 0xc7, 0xa8, 0x3c,
	0x00, 0xfc, 0x78, 0x8d, 0x10, 0x38, 0xd1, 0x11,
	0x94, 0xed, 0xb4, 0x22, 0x13, 0xcb, 0x52, 0x0f,
	0x0f, 0xd7, 0x33, 0x3b, 0xbd, 0x01, 0x04, 0x56,
	0xfa, 0x2c, 0xaa, 0xaf, 0x2b, 0x93, 0xde, 0xf4,
	0x31, 0x36, 0x13, 0x71, 0xef, 0x7a, 0xf0, 0xae,
	0xbd, 0xa7, 0x4a, 0x57, 0xa5, 0xc5, 0xf3, 0x5c,
	0x08, 0x2b, 0xe7, 0x12, 0x42, 0x4b, 0x4b, 0x12,
	0x49, 0x3a, 0x2e, 0x26, 0x67, 0x67, 0xa1, 0xd5,
	0x59, 0xa6, 0x18, 0x96, 0x22, 0x2b, 0xeb, 0x56,
	0x1e, 0x0a, 0x08, 0x75, 0xb4, 0x2b, 0x5c, 0x0a,
	0x4e, 0x9d, 0x17, 0xd4, 0x0c, 0xfe, 0x46, 0x60,
	0x6d, 0xfa, 0xc0, 0xb9, 0x5e, 0x1f, 0x88, 0x0e,
	0x08, 0x2c, 0xf3, 0xb4, 0x3a, 0x15, 0xc5, 0xf9,
	0x5b, 0x85, 0x92, 0x94, 0xa8, 0x8f, 0x2c, 0x3a,
	0x7e, 0x22, 0x86, 0x88, 0x51, 0x03, 0xee, 0xf9,
	0x2e, 0x83, 0xce, 0x39, 0x0c, 0x76, 0x64, 0xe5,
	0x5a, 0x88, 0xef, 0xc5, 0x06, 0xb2, 0xe4, 0x13,
	0x82, 0xc9, 0xee, 0xba, 0x6d, 0x56, 0xa8, 0x87,
	0x51, 0x69, 0x3b, 0x86, 0x29, 0x8e, 0xe8, 0xb4,
	0x44, 0x42, 0x07, 0x5b, 0xff, 0x0e, 0x1e, 0x9f,
	0x42, 0xb1, 0xc8, 0x5f, 0xab, 0x3b, 0xc7, 0xba,
	0x75, 0x20, 0xe6, 0x9f, 0x93, 0xb5, 0xcf, 0x8f,
	0x7c, 0x1c, 0xf3, 0xdb, 0x6a, 0xf4, 0xde, 0x00,
	0xe9, 0xaf, 0xd5, 0xf4, 0x36, 0x98, 0x14, 0x2d,
	0x53, 0x20, 0x74, 0xab, 0x0c, 0xf6, 0xcd, 0x15,
	0x32, 0xa6, 0x01, 0x8d, 0x24, 0x1b, 0x4b, 0x1f,
	0xa3, 0xfc, 0x38, 0x82, 0x3a, 0xa1, 0xb5, 0x52,
	0x53, 0xc7, 0x2b, 0x30, 0x7c, 0x65, 0xb9, 0x7d,
};

/*
 * IV method encblkno8, blkno 1.
 */
static const uint8_t aes_cbc_128_encblkno8_vec1_ctxt[SECSIZE] = {
	0x63, 0x45, 0x16, 0x0c, 0xe4, 0x4f, 0x51, 0xde,
	0x74, 0xf8, 0x7b, 0xf5, 0x05, 0x17, 0x13, 0x1e,
	0xa5, 0x3d, 0x84, 0xfa, 0x35, 0x5a, 0x2d, 0x3c,
	0xb7, 0x61, 0x68, 0xff, 0xcd, 0x33, 0x1f, 0x0b,
	0x53, 0x79, 0xf2, 0x2f, 0xbc, 0x8d, 0xac, 0xb9,
	0xf9, 0xb7, 0x9c, 0x0a, 0x9d, 0xa1, 0x4d, 0x78,
	0x9e, 0x4e, 0xfa, 0xe8, 0xc8, 0x46, 0x4b, 0x99,
	0x91, 0x7e, 0x33, 0x6e, 0x18, 0x24, 0x01, 0xc3,
	0x9f, 0x8c, 0x43, 0xb5, 0x15, 0x7e, 0xdd, 0xf9,
	0x1b, 0xb0, 0xf2, 0xc3, 0x97, 0x1f, 0x7c, 0x3f,
	0x43, 0x4c, 0x9f, 0x93, 0x29, 0x83, 0x8f, 0xad,
	0xd1, 0x5e, 0x92, 0x1a, 0x17, 0xd1, 0xa0, 0x05,
	0x6e, 0x62, 0x59, 0x80, 0x50, 0x6d, 0xe3, 0x28,
	0x9a, 0x43, 0xdc, 0x81, 0x4f, 0x49, 0xc4, 0x98,
	0xcd, 0x6d, 0x28, 0xb4, 0x86, 0xe4, 0x83, 0x45,
	0xd4, 0x43, 0x52, 0x8a, 0xd6, 0xc8, 0x1c, 0x90,
	0xeb, 0xf0, 0xe7, 0x76, 0xb4, 0x43, 0x9b, 0x56,
	0x48, 0x73, 0xdd, 0x01, 0x50, 0x1c, 0x61, 0xfc,
	0x22, 0xac, 0xf4, 0x27, 0x94, 0x02, 0x54, 0xd3,
	0x7d, 0x25, 0xf6, 0x14, 0x29, 0xbb, 0x2b, 0x22,
	0xc8, 0xe8, 0x7f, 0xa1, 0xfe, 0x19, 0x79, 0x97,
	0xb6, 0xa6, 0xba, 0x5b, 0x89, 0xdf, 0x73, 0x6d,
	0x79, 0x38, 0x5b, 0xf8, 0x89, 0xa2, 0x95, 0x1d,
	0xda, 0x38, 0x17, 0x4b, 0x01, 0xf1, 0x7d, 0x0a,
	0xa2, 0x8f, 0x5a, 0x02, 0x51, 0xb0, 0x88, 0x10,
	0x16, 0xc8, 0x82, 0xb9, 0x06, 0x9f, 0x01, 0x94,
	0xf9, 0xe0, 0x2e, 0x86, 0x8a, 0xb1, 0x7f, 0x74,
	0x22, 0xce, 0xee, 0xa6, 0x66, 0xee, 0xe2, 0x1d,
	0x98, 0x1b, 0x46, 0x22, 0x7e, 0x89, 0x0c, 0xc4,
	0x91, 0xfb, 0xe4, 0xd7, 0x36, 0x2a, 0xa9, 0x53,
	0xe9, 0xaf, 0x6c, 0xc1, 0xdd, 0x69, 0x4f, 0xde,
	0xd8, 0xd0, 0x7f, 0xc9, 0xf4, 0x8f, 0x84, 0xfe,
	0x0f, 0x16, 0x36, 0x90, 0x09, 0xd6, 0x0f, 0xbc,
	0x85, 0xad, 0xe9, 0xc3, 0xa1, 0x8d, 0x14, 0x5c,
	0x40, 0x7d, 0x0f, 0x22, 0xfe, 0x5e, 0xaf, 0xd9,
	0x0f, 0xe5, 0x2e, 0xa6, 0x04, 0xda, 0x35, 0x90,
	0x7f, 0x9a, 0x1f, 0xb8, 0x34, 0x1c, 0xd0, 0xf5,
	0x5c, 0x29, 0xce, 0xbe, 0x57, 0xd8, 0x55, 0x15,
	0x2d, 0x4c, 0x3c, 0x16, 0x93, 0x96, 0x3c, 0xf3,
	0xa8, 0x2f, 0x09, 0xb3, 0x8b, 0xe3, 0xce, 0xf7,
	0x3e, 0x8e, 0xcf, 0x47, 0xe2, 0xf2, 0xad, 0x06,
	0x00, 0x9a, 0x3a, 0x55, 0xf5, 0x9e, 0xbf, 0x5a,
	0x2e, 0x5a, 0x6c, 0x2b, 0x8f, 0x33, 0x71, 0x2c,
	0x56, 0x43, 0xd1, 0x8b, 0xd2, 0x98, 0x14, 0xb7,
	0x5a, 0xdc, 0x8b, 0xbc, 0xfe, 0x50, 0x99, 0x84,
	0x48, 0x5f, 0xcd, 0xed, 0xce, 0x61, 0x6f, 0xa6,
	0x83, 0xa3, 0x34, 0xbe, 0xf2, 0x66, 0xf3, 0x09,
	0xf3, 0xd3, 0x97, 0xd4, 0xee, 0x66, 0x9a, 0x81,
	0x62, 0x84, 0x85, 0x7f, 0x79, 0x18, 0xd2, 0x82,
	0xd6, 0x96, 0x09, 0x61, 0x1e, 0x53, 0x97, 0x80,
	0x0a, 0x81, 0x4b, 0x93, 0xf0, 0x03, 0x65, 0x18,
	0x93, 0x5b, 0x60, 0x2f, 0xb5, 0xfe, 0x82, 0xaf,
	0x85, 0xb7, 0x79, 0x7c, 0xee, 0xad, 0xea, 0xfa,
	0x9b, 0xad, 0xca, 0x38, 0x21, 0x3d, 0x46, 0x8a,
	0x5b, 0xa7, 0x55, 0x3d, 0x88, 0x4a, 0x52, 0xdb,
	0xf2, 0x07, 0xed, 0xd6, 0x3c, 0x9f, 0x1b, 0x42,
	0xb4, 0x14, 0x12, 0xb7, 0x00, 0xfc, 0x6a, 0x79,
	0x61, 0x0b, 0x43, 0xaa, 0x44, 0x48, 0x30, 0x15,
	0x48, 0x76, 0x27, 0x32, 0x7a, 0x2e, 0x25, 0x6a,
	0x8c, 0x8c, 0x64, 0x67, 0x86, 0x54, 0x4a, 0x93,
	0x14, 0xfe, 0x81, 0xf5, 0xcf, 0x98, 0x92, 0xd3,
	0x92, 0xf5, 0x6a, 0x74, 0x28, 0x10, 0x6b, 0xd4,
	0x1d, 0x64, 0x7e, 0x05, 0x32, 0xba, 0xf7, 0x4c,
	0xe9, 0xa8, 0xa9, 0xc5, 0x35, 0x34, 0x26, 0x41,
};

/*
 * IV method encblkno8, blkno 2.
 */
static const uint8_t aes_cbc_128_encblkno8_vec2_ctxt[SECSIZE] = {
	0x64, 0x7b, 0x62, 0x7a, 0xa6, 0xa9, 0xb3, 0x47,
	0xbc, 0x03, 0x14, 0x3d, 0x9b, 0x56, 0xfc, 0x13,
	0x08, 0x32, 0x81, 0xe3, 0x57, 0x3c, 0x0d, 0xbb,
	0x85, 0x44, 0x47, 0x12, 0xc4, 0x80, 0x35, 0x37,
	0xe1, 0xb4, 0x3f, 0x35, 0x98, 0x7c, 0xb0, 0x3b,
	0x85, 0xab, 0x3d, 0x0b, 0xd3, 0x6f, 0xf9, 0x92,
	0x00, 0x6b, 0x18, 0xe7, 0x31, 0x8b, 0x77, 0x4c,
	0xd0, 0x7b, 0x1d, 0xfc, 0x95, 0xe6, 0x02, 0x01,
	0x9c, 0x17, 0x4d, 0x9b, 0x3a, 0x1d, 0x12, 0x23,
	0xd4, 0x24, 0xf8, 0x47, 0x5e, 0x2d, 0xfd, 0xc8,
	0x74, 0x28, 0xb4, 0x3a, 0x99, 0x6b, 0xcc, 0xba,
	0xe5, 0x51, 0x0b, 0xab, 0x4d, 0x63, 0xfc, 0x6d,
	0x2d, 0xd9, 0x2b, 0x4f, 0xa4, 0x26, 0xc7, 0x8d,
	0x9d, 0x12, 0x7f, 0xc7, 0x6b, 0x15, 0x8b, 0x4a,
	0x41, 0xf8, 0x50, 0x32, 0x76, 0x10, 0xca, 0x8e,
	0xfe, 0x08, 0x7d, 0x9a, 0xb5, 0x1a, 0xdb, 0x10,
	0xb3, 0x6a, 0x5f, 0xd9, 0x0a, 0x5f, 0x31, 0x19,
	0x3e, 0xa9, 0xa1, 0x72, 0x1f, 0x6c, 0x97, 0x20,
	0xd4, 0xab, 0xb8, 0xad, 0xf7, 0x70, 0x12, 0xd0,
	0x8f, 0x70, 0x24, 0x58, 0x2e, 0x99, 0xcd, 0xd4,
	0xf4, 0xcd, 0xef, 0x93, 0xfb, 0x4f, 0x9a, 0x40,
	0x46, 0x92, 0x6b, 0xd0, 0x25, 0x24, 0xec, 0x4d,
	0x4c, 0x42, 0x50, 0x61, 0xb6, 0x21, 0xa6, 0x2e,
	0xc1, 0x42, 0x9e, 0x86, 0x9f, 0x57, 0x2a, 0x2c,
	0x82, 0xbd, 0xc2, 0x25, 0xb6, 0x9f, 0x2d, 0x9f,
	0xba, 0xe0, 0xa6, 0x06, 0x04, 0x08, 0xc5, 0x1d,
	0x8c, 0x0f, 0xbf, 0x21, 0x85, 0x6d, 0x61, 0x4d,
	0x93, 0xc0, 0xa2, 0x8b, 0xca, 0x07, 0xd0, 0x88,
	0x74, 0xf9, 0x42, 0x92, 0xd5, 0x0d, 0x0c, 0x34,
	0xa6, 0xa5, 0x86, 0x51, 0xcf, 0x40, 0x36, 0x66,
	0x35, 0x9f, 0xa8, 0x95, 0x0b, 0xfb, 0x0c, 0xe8,
	0xdc, 0x12, 0x78, 0x4c, 0x52, 0xf4, 0xfc, 0x4a,
	0x77, 0xdd, 0x77, 0x34, 0xf7, 0x35, 0x94, 0x7a,
	0x31, 0x16, 0x86, 0x44, 0x50, 0x30, 0x1c, 0x6d,
	0x9f, 0x66, 0x49, 0xb5, 0xe6, 0x71, 0x00, 0x83,
	0xd1, 0xa0, 0x01, 0xff, 0xc3, 0x27, 0xaa, 0x9a,
	0xdb, 0xad, 0x24, 0xdb, 0xbd, 0xde, 0xfd, 0xa6,
	0xaa, 0x87, 0x98, 0x98, 0xde, 0x90, 0xd5, 0x40,
	0x20, 0x8f, 0xe9, 0xdd, 0xa8, 0xec, 0xd3, 0x18,
	0x20, 0x85, 0x5e, 0xd5, 0xe7, 0x50, 0x58, 0x15,
	0x69, 0x03, 0xa5, 0xe8, 0xa9, 0x7a, 0x0f, 0xd1,
	0x7d, 0x22, 0x8a, 0xe0, 0xc6, 0x17, 0x33, 0x00,
	0x57, 0xcb, 0xf6, 0x8d, 0xf0, 0xc1, 0x7b, 0xb5,
	0x96, 0x0f, 0x08, 0x84, 0x5b, 0x7e, 0xa6, 0x1e,
	0xd8, 0x5e, 0x0c, 0xca, 0x30, 0x4b, 0xe0, 0x87,
	0x2f, 0xbc, 0x07, 0x83, 0x35, 0x76, 0x36, 0x17,
	0xcf, 0xdb, 0x27, 0x53, 0x43, 0xf5, 0x07, 0xd0,
	0x91, 0x83, 0xa1, 0xaa, 0x8d, 0xdb, 0x00, 0x2b,
	0xd1, 0x88, 0xe5, 0x59, 0x47, 0x17, 0xf0, 0xe8,
	0xce, 0x3b, 0xa0, 0x73, 0x1f, 0x22, 0x9b, 0x1b,
	0x59, 0x02, 0xe6, 0xaf, 0x3f, 0xdd, 0xfe, 0xba,
	0xc3, 0x6b, 0xe5, 0x82, 0x02, 0x92, 0x0c, 0x5e,
	0x5a, 0x87, 0x88, 0x91, 0x00, 0xb5, 0x30, 0x37,
	0xf5, 0xc6, 0xdf, 0x0a, 0x7f, 0x03, 0x1c, 0x1f,
	0x20, 0xf1, 0xd4, 0x5f, 0x94, 0xc3, 0x6f, 0x21,
	0x5e, 0xf2, 0x77, 0x5a, 0x42, 0xfd, 0xd3, 0xc4,
	0x31, 0xaf, 0xd6, 0x6c, 0x6c, 0xde, 0x8c, 0x50,
	0x01, 0x8f, 0x57, 0x90, 0x88, 0x43, 0xf9, 0x44,
	0x09, 0x4d, 0x27, 0x58, 0x9f, 0xae, 0x50, 0x28,
	0x12, 0x47, 0x20, 0x79, 0x2b, 0xe4, 0x02, 0x97,
	0xcd, 0xab, 0x53, 0x28, 0x8f, 0x8f, 0xe3, 0x3b,
	0xd6, 0xc9, 0xc8, 0xff, 0xbf, 0x18, 0x3b, 0x75,
	0xdb, 0xcf, 0x07, 0x8c, 0xfe, 0x58, 0xee, 0x75,
	0x01, 0x98, 0x98, 0xe4, 0x60, 0xfe, 0xe6, 0x7f,
};

/*
 * IV method encblkno8, blkno 3.
 */
static const uint8_t aes_cbc_128_encblkno8_vec3_ctxt[SECSIZE] = {
	0x98, 0xae, 0x82, 0x1d, 0x76, 0x3a, 0xfe, 0x80,
	0x04, 0xa3, 0x43, 0xf0, 0x06, 0x45, 0x83, 0xb7,
	0xe2, 0xb5, 0x73, 0x46, 0x78, 0x01, 0x2f, 0xd6,
	0x0d, 0x49, 0x64, 0x4c, 0xeb, 0x8d, 0xdc, 0xa9,
	0xdc, 0xea, 0x22, 0x25, 0xd4, 0x8f, 0xba, 0x9f,
	0xd4, 0x7a, 0x3c, 0x9e, 0xd0, 0xd9, 0xcd, 0xa4,
	0x12, 0xdf, 0x8f, 0x50, 0x24, 0x18, 0xa2, 0x0b,
	0xd9, 0x7f, 0xda, 0x78, 0xd6, 0x11, 0xf3, 0x99,
	0xc4, 0xec, 0x95, 0xe2, 0x85, 0xe1, 0xa0, 0x0d,
	0x07, 0x22, 0x56, 0xaf, 0x2f, 0xf5, 0x7d, 0x63,
	0xf2, 0x90, 0x6c, 0x26, 0x4f, 0xa5, 0x47, 0xcd,
	0x66, 0x2d, 0x4c, 0x4d, 0x94, 0x6a, 0x3c, 0x98,
	0xe4, 0x5e, 0x3b, 0x42, 0x3a, 0x93, 0x02, 0xd0,
	0x90, 0xc7, 0xcd, 0x87, 0x0e, 0x84, 0x82, 0xf5,
	0x77, 0x7b, 0x29, 0xe4, 0xea, 0x5b, 0x60, 0x50,
	0xf7, 0x60, 0x8d, 0xf7, 0xd8, 0xd7, 0x7d, 0x99,
	0x8a, 0xdc, 0xe2, 0xb9, 0x40, 0xac, 0x4b, 0x9f,
	0x55, 0x30, 0xcb, 0x5a, 0x73, 0x64, 0xf2, 0xca,
	0x76, 0x88, 0xf7, 0x55, 0xb5, 0x33, 0xc0, 0x44,
	0xdf, 0x42, 0xee, 0xc9, 0xc5, 0x2a, 0x47, 0x18,
	0x8b, 0x74, 0xb9, 0x4f, 0x2c, 0xd8, 0x7a, 0xd1,
	0x12, 0x19, 0xf9, 0x21, 0x8d, 0x21, 0x7e, 0x2a,
	0xcf, 0xd5, 0xbb, 0x69, 0xaa, 0x20, 0x25, 0xe0,
	0xf5, 0x3b, 0x33, 0x77, 0x63, 0xb2, 0x05, 0x5c,
	0x10, 0x9c, 0x61, 0x48, 0xf5, 0xe6, 0x04, 0xd3,
	0xc8, 0xb4, 0xf6, 0xcf, 0x22, 0x1c, 0xf6, 0xbb,
	0x4b, 0xd7, 0x5d, 0x23, 0xfa, 0x0e, 0xc0, 0xac,
	0x27, 0x38, 0x95, 0xd0, 0xdd, 0x83, 0xad, 0x9e,
	0xcf, 0xde, 0x99, 0xe7, 0x04, 0xb7, 0x23, 0x9f,
	0x46, 0x91, 0xb8, 0xcb, 0x18, 0xd0, 0xc5, 0xf8,
	0xec, 0xfc, 0x33, 0xb7, 0xbe, 0x2d, 0xe9, 0x3a,
	0x7f, 0x83, 0x5e, 0x44, 0x0f, 0x12, 0x6d, 0x05,
	0xaa, 0xfb, 0x80, 0x7a, 0xf6, 0xdb, 0x25, 0xc6,
	0x51, 0xf3, 0x5d, 0xf3, 0xa9, 0xb8, 0x34, 0x88,
	0x88, 0x25, 0xd5, 0xa3, 0xe5, 0x8e, 0xb2, 0xc7,
	0xdc, 0xd5, 0x2e, 0x99, 0xb9, 0xc5, 0x1d, 0x91,
	0x49, 0x7b, 0xa3, 0x5e, 0x4b, 0xaf, 0x29, 0x7b,
	0x37, 0xb5, 0x39, 0x2c, 0xdf, 0x3b, 0xb1, 0xd8,
	0xba, 0x14, 0xc9, 0xd3, 0x6d, 0x67, 0x6a, 0x80,
	0x89, 0x6f, 0x11, 0xc8, 0xbc, 0xd6, 0xc7, 0xab,
	0x42, 0x1f, 0xf4, 0xa2, 0xc0, 0x9c, 0x2d, 0xca,
	0x5f, 0xe6, 0x65, 0xfa, 0x28, 0x49, 0x99, 0xa3,
	0x0b, 0x7b, 0x7d, 0x39, 0xaa, 0xa6, 0xd8, 0x0a,
	0xfd, 0xde, 0x31, 0x86, 0x15, 0x95, 0x1e, 0x5c,
	0x05, 0x4e, 0x3c, 0x18, 0xee, 0xa9, 0x56, 0x9c,
	0x3c, 0xc3, 0x67, 0x84, 0x57, 0x77, 0x8d, 0xff,
	0xea, 0x34, 0x3c, 0xf9, 0x58, 0xb8, 0xdc, 0x4e,
	0xa1, 0x92, 0x2d, 0x9a, 0x91, 0x61, 0x23, 0x6a,
	0xd9, 0xb7, 0x41, 0xc5, 0x0d, 0xb6, 0x57, 0x58,
	0x42, 0x39, 0x4a, 0x86, 0x7e, 0x9d, 0xeb, 0x7d,
	0xa8, 0x14, 0x1a, 0x5c, 0xa1, 0x54, 0x34, 0xb6,
	0xb6, 0xbc, 0x1f, 0xf5, 0xe2, 0xb5, 0xe4, 0xa8,
	0x42, 0xe3, 0x3d, 0x06, 0x6b, 0x50, 0xbb, 0xa1,
	0x6b, 0x63, 0xe5, 0x60, 0x28, 0x07, 0x49, 0x06,
	0x61, 0x0e, 0xa3, 0x6c, 0xc3, 0xc8, 0x3e, 0x5a,
	0x9c, 0xa5, 0xb3, 0x9b, 0x8d, 0x46, 0xb9, 0xf5,
	0x4a, 0x4d, 0xbe, 0xc0, 0xc1, 0x24, 0x92, 0x09,
	0x7c, 0x9a, 0x21, 0x2c, 0x08, 0x8a, 0x0d, 0xfc,
	0xff, 0xda, 0xdc, 0xf1, 0x45, 0x66, 0xf9, 0xcd,
	0x64, 0x7c, 0x2f, 0x0e, 0x95, 0x5e, 0xec, 0x92,
	0xd1, 0x03, 0x03, 0xa0, 0xcc, 0x73, 0x92, 0x15,
	0x74, 0x42, 0x54, 0x48, 0x77, 0xbe, 0x96, 0xfb,
	0x1f, 0x0c, 0x7a, 0x25, 0x67, 0x6b, 0x85, 0x71,
	0x06, 0x15, 0xd3, 0x11, 0xfe, 0xf7, 0xa9, 0xb1,
};

const struct testvec aes_cbc_128_8_vectors[] = {
	{
		.blkno = 0,
		.ptxt = aes_cbc_ptxt,
		.ctxt = aes_cbc_128_encblkno8_vec0_ctxt,
	},
	{
		.blkno = 1,
		.ptxt = aes_cbc_ptxt,
		.ctxt = aes_cbc_128_encblkno8_vec1_ctxt,
	},
	{
		.blkno = 2,
		.ptxt = aes_cbc_ptxt,
		.ctxt = aes_cbc_128_encblkno8_vec2_ctxt,
	},
	{
		.blkno = 3,
		.ptxt = aes_cbc_ptxt,
		.ctxt = aes_cbc_128_encblkno8_vec3_ctxt,
	},
};

/*
 * IV method encblkno1, blkno 0.
 */
static const uint8_t aes_cbc_192_encblkno1_vec0_ctxt[SECSIZE] = {
	0x7c, 0xc4, 0xec, 0x89, 0x7c, 0x13, 0xac, 0x99,
	0x49, 0xa9, 0x96, 0xe7, 0xb1, 0x1a, 0xd6, 0xb0,
	0xeb, 0x89, 0x27, 0x0f, 0x8b, 0x1b, 0xab, 0x8e,
	0x2c, 0xd4, 0x00, 0x66, 0x12, 0x3a, 0x9a, 0x03,
	0xc4, 0x49, 0xa4, 0xf0, 0xc1, 0x90, 0xf9, 0x38,
	0xb2, 0x5c, 0xa5, 0x0d, 0x1b, 0x60, 0x94, 0xf6,
	0x31, 0x4a, 0x72, 0xdb, 0xfc, 0xe1, 0x3c, 0xd6,
	0x9d, 0x03, 0x07, 0x45, 0xdb, 0xad, 0xdb, 0xb3,
	0x86, 0xfa, 0xce, 0x2c, 0xeb, 0xa2, 0xac, 0x05,
	0xd9, 0x52, 0xb8, 0xae, 0xa9, 0x91, 0x86, 0x4b,
	0xbb, 0xf8, 0x03, 0xb0, 0x6c, 0x40, 0xcc, 0xbf,
	0xa3, 0x76, 0x60, 0xf7, 0x29, 0x03, 0xe6, 0x44,
	0xcc, 0x2a, 0xe7, 0x74, 0x8e, 0x62, 0xfe, 0x99,
	0x6a, 0x6d, 0x04, 0x1b, 0xe7, 0xf7, 0x9f, 0x13,
	0xa7, 0x1d, 0x93, 0x0e, 0x8f, 0xe0, 0x77, 0x9b,
	0xe3, 0x91, 0x67, 0x12, 0x33, 0x12, 0x42, 0x55,
	0x28, 0x04, 0x2d, 0x01, 0x2b, 0xd2, 0xda, 0xbe,
	0x7c, 0x83, 0xf2, 0x87, 0x71, 0x67, 0xaf, 0x6b,
	0x50, 0x6c, 0x8c, 0x9f, 0x48, 0xee, 0x90, 0x0c,
	0x9a, 0x9e, 0x40, 0xa8, 0x13, 0x2f, 0x58, 0xfb,
	0xdc, 0xb1, 0xda, 0xff, 0x06, 0x9c, 0xeb, 0x5e,
	0x0f, 0xaf, 0xc0, 0x9a, 0x47, 0x88, 0x25, 0xfd,
	0x19, 0x5e, 0xd4, 0xe0, 0x7f, 0xe0, 0x71, 0x7a,
	0x60, 0x54, 0xe7, 0x0d, 0xfe, 0x11, 0x9d, 0x77,
	0xbd, 0x9b, 0xd0, 0xf8, 0x77, 0xe4, 0x5b, 0x88,
	0x90, 0x12, 0x29, 0x88, 0xb6, 0xd9, 0x1e, 0x6c,
	0xbf, 0xa4, 0x18, 0xe1, 0xe0, 0x5e, 0xed, 0x48,
	0x9b, 0x05, 0x13, 0x37, 0x0f, 0x41, 0x54, 0xc8,
	0xe4, 0x25, 0x0e, 0x82, 0x5f, 0x81, 0xba, 0x5d,
	0x79, 0x8f, 0x9c, 0x17, 0x4b, 0x59, 0xf4, 0x5d,
	0xd6, 0x83, 0xfd, 0x44, 0xd0, 0xe1, 0x89, 0x09,
	0xf9, 0xe2, 0xb6, 0x9c, 0x1c, 0xbd, 0x13, 0xaa,
	0xa0, 0x43, 0xaa, 0xaf, 0x6d, 0x65, 0x73, 0xba,
	0x3a, 0x55, 0x69, 0x51, 0xb9, 0x52, 0x09, 0xaa,
	0x9f, 0x91, 0x3c, 0x65, 0xe2, 0x81, 0xdb, 0xe8,
	0x5a, 0xe3, 0x74, 0x11, 0x7b, 0xec, 0x2f, 0x18,
	0x8d, 0x4c, 0x8f, 0xf2, 0x06, 0x3d, 0x22, 0xc6,
	0x43, 0xef, 0x42, 0x7d, 0xe1, 0xe7, 0xde, 0x4c,
	0x58, 0xad, 0x40, 0xbb, 0x8b, 0xce, 0x1f, 0x57,
	0x8e, 0x6a, 0x27, 0x43, 0x46, 0x7f, 0x94, 0xe5,
	0x45, 0x67, 0x12, 0xc8, 0x99, 0x85, 0x08, 0x2a,
	0x37, 0x40, 0x0b, 0xb5, 0xd9, 0xa3, 0xf7, 0xc8,
	0x87, 0xb1, 0xe6, 0x87, 0x2f, 0x86, 0xd8, 0x9c,
	0x7b, 0xec, 0xcf, 0xa4, 0xe5, 0xd5, 0x50, 0x3f,
	0xdf, 0xc9, 0xb7, 0x29, 0x97, 0xd6, 0x33, 0xba,
	0xf0, 0x72, 0xf0, 0x76, 0x12, 0xd3, 0x99, 0x4f,
	0x1b, 0x36, 0xda, 0xa1, 0x83, 0xfe, 0xf5, 0x94,
	0x9e, 0x61, 0x82, 0x62, 0xe0, 0x08, 0x3a, 0xbd,
	0xba, 0x8b, 0x3d, 0xd6, 0xbd, 0x16, 0x5f, 0xd7,
	0x1d, 0x6c, 0x0e, 0x92, 0x89, 0x8c, 0x38, 0x62,
	0x80, 0xee, 0x7e, 0x63, 0x82, 0x88, 0x0b, 0xbf,
	0xdd, 0x9f, 0xbc, 0xba, 0xa7, 0x5a, 0xc6, 0x0d,
	0x87, 0x59, 0xbf, 0x0a, 0x85, 0x06, 0xa3, 0xb4,
	0x66, 0x63, 0xda, 0x12, 0x29, 0x5f, 0x2e, 0x4d,
	0x60, 0xfd, 0x85, 0x76, 0xaf, 0xf7, 0x87, 0xed,
	0x1f, 0x46, 0xc2, 0xd6, 0x6c, 0x98, 0x6b, 0x4b,
	0x60, 0x04, 0xed, 0x89, 0x3b, 0x85, 0x6c, 0xe9,
	0x46, 0xd9, 0xfa, 0x35, 0x61, 0xe8, 0x0c, 0x84,
	0x1b, 0x93, 0xc0, 0xfe, 0x5d, 0x29, 0x14, 0xe1,
	0x1c, 0x66, 0x73, 0xc8, 0x0b, 0x98, 0xff, 0x1a,
	0x78, 0x2b, 0x6a, 0x93, 0x7a, 0x29, 0xd8, 0x7b,
	0xb1, 0x39, 0xf0, 0xad, 0x93, 0x4d, 0x2d, 0xab,
	0x67, 0x3c, 0xa4, 0xa1, 0x08, 0x36, 0x0b, 0xe9,
	0x77, 0xd0, 0xe3, 0x45, 0x7d, 0x99, 0x75, 0xc3,
};

/*
 * IV method encblkno1, blkno 1.
 */
static const uint8_t aes_cbc_192_encblkno1_vec1_ctxt[SECSIZE] = {
	0xe6, 0x41, 0x75, 0xd6, 0x80, 0xdf, 0x44, 0x37,
	0xa7, 0xa2, 0xb2, 0x29, 0x0d, 0xf0, 0x02, 0x78,
	0x92, 0xb2, 0x06, 0x5f, 0x86, 0xd3, 0x9c, 0xa3,
	0xd0, 0xc5, 0x08, 0x03, 0x6d, 0x41, 0x9d, 0x61,
	0xb4, 0xb9, 0xa1, 0x69, 0x6e, 0x3a, 0x78, 0xd7,
	0x04, 0x94, 0xf2, 0x53, 0xed, 0xd1, 0xf6, 0xd8,
	0x98, 0xe2, 0x49, 0x75, 0x15, 0x85, 0xe0, 0x78,
	0x5b, 0x28, 0x5e, 0xe6, 0xfa, 0x60, 0x3d, 0x4b,
	0x8c, 0xf1, 0x1a, 0xfd, 0x1f, 0xe8, 0xad, 0xb4,
	0xa1, 0xe7, 0xd3, 0x71, 0x16, 0xdf, 0xc6, 0x95,
	0xd4, 0x43, 0xaf, 0x92, 0xab, 0x74, 0x0f, 0x77,
	0x75, 0x4d, 0xd7, 0x13, 0x97, 0x18, 0xea, 0x43,
	0x92, 0x0d, 0x88, 0xc8, 0x41, 0xf7, 0x15, 0x34,
	0x0f, 0x63, 0xbf, 0x50, 0x18, 0xbe, 0x9d, 0x3b,
	0xfc, 0x17, 0x7d, 0x03, 0x39, 0xc2, 0x39, 0x28,
	0xb2, 0x23, 0x1c, 0x7f, 0x3f, 0x19, 0x6c, 0x2f,
	0x64, 0xbd, 0xc9, 0x7d, 0xbe, 0x98, 0xe0, 0x83,
	0xa4, 0x48, 0xfc, 0x89, 0xe7, 0xe0, 0x93, 0x93,
	0x7b, 0x15, 0x35, 0xaf, 0xf8, 0x00, 0x81, 0xcc,
	0x04, 0x80, 0x8b, 0x20, 0xc8, 0x6a, 0xb7, 0x5e,
	0x95, 0xce, 0x69, 0x50, 0x39, 0x88, 0x90, 0x41,
	0x3f, 0xa8, 0x62, 0x42, 0xf1, 0xa9, 0x56, 0xce,
	0x25, 0x53, 0x1d, 0x97, 0x5d, 0x3a, 0x4e, 0x6b,
	0x1f, 0xd6, 0xea, 0x20, 0x81, 0x6c, 0xe5, 0xa1,
	0x0d, 0x9a, 0xd9, 0x3c, 0xbb, 0xbc, 0xc1, 0x77,
	0xe2, 0xf4, 0x9c, 0x11, 0x3a, 0x2f, 0xd0, 0x77,
	0x10, 0xa6, 0x36, 0xd1, 0xbf, 0x3b, 0x50, 0x39,
	0x4b, 0x2c, 0x62, 0x06, 0x1a, 0xe4, 0x18, 0xc0,
	0x35, 0x7c, 0xc3, 0xd0, 0x22, 0xf8, 0xee, 0x19,
	0xa5, 0x3d, 0x69, 0xa9, 0x34, 0xe6, 0x29, 0xf9,
	0xf1, 0xff, 0x26, 0x7a, 0x66, 0x13, 0x1a, 0xa2,
	0xc6, 0xac, 0x84, 0xf6, 0x6b, 0x09, 0xbd, 0x32,
	0x6f, 0x26, 0x37, 0x7c, 0x7d, 0x74, 0xe4, 0xa0,
	0xeb, 0x85, 0x7a, 0xa1, 0x92, 0x19, 0x2e, 0x64,
	0x82, 0x7c, 0x89, 0x1b, 0x14, 0x92, 0xd1, 0xf4,
	0x1f, 0x29, 0x84, 0x04, 0x70, 0x09, 0x13, 0x4c,
	0x62, 0x9a, 0xb4, 0xf7, 0xc1, 0x7b, 0x83, 0xd1,
	0x2d, 0x1a, 0xbe, 0x83, 0x9b, 0x73, 0xba, 0x8d,
	0xbb, 0xb0, 0xf2, 0x5c, 0x72, 0x75, 0x01, 0x0b,
	0xa6, 0x43, 0x6b, 0x76, 0x56, 0x4e, 0x71, 0x1b,
	0xb2, 0x34, 0x1f, 0x70, 0x44, 0xe6, 0xfb, 0x67,
	0xd1, 0x4d, 0x63, 0xce, 0x17, 0x46, 0x9b, 0x11,
	0xda, 0x93, 0xf8, 0x03, 0x11, 0x8f, 0x90, 0xff,
	0x80, 0x85, 0x02, 0x1f, 0xb6, 0x6a, 0x28, 0x3f,
	0x01, 0xa8, 0x36, 0x2e, 0xc7, 0x42, 0xd4, 0x02,
	0x26, 0xea, 0xb5, 0x84, 0x6c, 0x9f, 0xa0, 0x4a,
	0x73, 0x49, 0xea, 0x91, 0x4d, 0x62, 0xf8, 0x23,
	0xe4, 0x3d, 0x91, 0xfb, 0x53, 0x2c, 0x8c, 0xa4,
	0xfe, 0x81, 0x05, 0x5d, 0x4b, 0x9a, 0x75, 0x29,
	0xf8, 0xbe, 0x3f, 0x05, 0xb4, 0x8f, 0xdc, 0xcc,
	0xfa, 0xcc, 0xd7, 0xb2, 0x06, 0x03, 0xd4, 0xf3,
	0x8e, 0x09, 0x09, 0x80, 0xf8, 0xc3, 0x3b, 0x66,
	0xe9, 0x9c, 0x5b, 0x16, 0xed, 0x2d, 0x35, 0x1c,
	0x99, 0x3b, 0x1f, 0x0e, 0x04, 0x30, 0x23, 0x3a,
	0x83, 0x0c, 0xec, 0x76, 0xf2, 0x5d, 0x13, 0x54,
	0x15, 0x62, 0x36, 0x26, 0x6b, 0x21, 0x62, 0xdd,
	0xb4, 0x1a, 0x57, 0x16, 0xfd, 0xa0, 0x9c, 0xfa,
	0x37, 0xb3, 0xda, 0xe0, 0x46, 0x91, 0xb3, 0x20,
	0xe7, 0xe2, 0xf3, 0x0e, 0x20, 0x3c, 0x98, 0x1b,
	0xe4, 0xc2, 0xd3, 0xa9, 0x97, 0xaf, 0x12, 0x69,
	0x23, 0x97, 0x62, 0x6e, 0xae, 0x54, 0x9c, 0x82,
	0x92, 0x50, 0x74, 0x07, 0x4a, 0xb1, 0xdc, 0xcf,
	0x53, 0x1d, 0xc8, 0x29, 0x1f, 0x6e, 0xf1, 0x13,
	0xec, 0xb6, 0x60, 0xb1, 0x4c, 0x9d, 0xd7, 0x77,
};

/*
 * IV method encblkno1, blkno 2.
 */
static const uint8_t aes_cbc_192_encblkno1_vec2_ctxt[SECSIZE] = {
	0x33, 0xfd, 0xfa, 0x44, 0x64, 0x75, 0x22, 0x7e,
	0xe3, 0xb3, 0xa0, 0x75, 0x99, 0x96, 0xc0, 0xec,
	0x56, 0x06, 0x7d, 0x19, 0x0b, 0x66, 0x89, 0xe0,
	0x69, 0x1d, 0x93, 0x91, 0xd7, 0x0f, 0xf8, 0xf5,
	0x5a, 0x39, 0x30, 0xad, 0x64, 0x42, 0x06, 0xa3,
	0xce, 0x3f, 0x67, 0xd6, 0x6e, 0xcd, 0x3b, 0xf5,
	0x03, 0x2b, 0x07, 0x83, 0x18, 0x1a, 0x4f, 0x4c,
	0xe7, 0x6b, 0xe8, 0xf9, 0x19, 0xa5, 0x23, 0x8f,
	0x46, 0x35, 0x13, 0x7b, 0x61, 0x05, 0xfc, 0x7d,
	0x17, 0x39, 0x03, 0xa8, 0xec, 0x7a, 0xd2, 0x5f,
	0x91, 0xa7, 0x26, 0x07, 0x9d, 0xd7, 0x0c, 0xd7,
	0xd4, 0x8e, 0x37, 0xf3, 0x1a, 0x3c, 0x04, 0x83,
	0x04, 0x71, 0x06, 0xa6, 0x5f, 0x82, 0xe0, 0x6d,
	0x87, 0x5c, 0x7c, 0x03, 0x25, 0x03, 0x4b, 0x24,
	0x07, 0x40, 0xad, 0xe4, 0x1d, 0x1d, 0xcb, 0x34,
	0xc2, 0x53, 0x1d, 0x13, 0xc5, 0x87, 0xab, 0xa7,
	0x95, 0x11, 0x8b, 0xbb, 0xf0, 0xc3, 0x00, 0xeb,
	0xe5, 0xb0, 0x9e, 0x88, 0x8b, 0xad, 0xca, 0xcb,
	0x17, 0xf8, 0x92, 0x4d, 0x00, 0xb0, 0x08, 0x74,
	0x08, 0xb9, 0x8b, 0x95, 0x96, 0xd9, 0x36, 0x35,
	0x31, 0x92, 0x89, 0xf6, 0x35, 0x33, 0xfb, 0x18,
	0x5b, 0x84, 0xa1, 0xfe, 0xe1, 0x62, 0x04, 0x6f,
	0x3c, 0xc1, 0xd2, 0xc2, 0x10, 0xd7, 0x97, 0xba,
	0x29, 0x7c, 0xe3, 0x85, 0xee, 0x59, 0x90, 0xaf,
	0x7f, 0x6f, 0x97, 0x97, 0xa2, 0x41, 0x18, 0x7f,
	0x2f, 0x06, 0x15, 0xb2, 0x46, 0x82, 0x49, 0x39,
	0xd0, 0xfb, 0xa8, 0x48, 0x44, 0x28, 0x58, 0xff,
	0xd8, 0xf2, 0x65, 0xf9, 0x4f, 0x2c, 0xbe, 0xec,
	0xb6, 0xdf, 0x27, 0x1a, 0xf2, 0x05, 0x15, 0x5e,
	0xe3, 0x2a, 0x98, 0x29, 0x92, 0x4a, 0x1b, 0x5d,
	0x5c, 0x2c, 0x70, 0xf6, 0x41, 0xd4, 0xbe, 0x64,
	0xa1, 0xd9, 0x79, 0xf1, 0x11, 0x16, 0xda, 0xa2,
	0xaf, 0xdd, 0x4d, 0xa8, 0xed, 0xec, 0xbe, 0x7d,
	0x49, 0x6c, 0x30, 0xf2, 0xf5, 0x36, 0x3c, 0xae,
	0x4b, 0xa7, 0x77, 0xa3, 0xca, 0x22, 0xa5, 0xe2,
	0x4d, 0x44, 0xcb, 0x36, 0xd5, 0x3f, 0x20, 0x13,
	0xb6, 0xfb, 0xcd, 0x79, 0xd7, 0x42, 0xf9, 0x75,
	0x09, 0x45, 0x28, 0x9e, 0xf2, 0xbd, 0x15, 0x57,
	0xf8, 0x4b, 0xc0, 0xd3, 0xb3, 0xb8, 0xde, 0x55,
	0x9e, 0x11, 0x67, 0xab, 0xc5, 0x5d, 0x58, 0xdb,
	0x4d, 0x20, 0x34, 0x77, 0x33, 0x9c, 0x46, 0x76,
	0x9b, 0x1e, 0x0e, 0x6b, 0x4e, 0xd9, 0x63, 0x68,
	0x78, 0x5e, 0x7c, 0x52, 0xa2, 0x64, 0xa9, 0xfc,
	0x21, 0x35, 0x17, 0x93, 0x18, 0x9e, 0x10, 0xcf,
	0x95, 0x6b, 0xf0, 0x55, 0x46, 0xc3, 0x4b, 0xfc,
	0x86, 0x8b, 0x0d, 0x3b, 0x5c, 0x30, 0xcc, 0xf1,
	0x4c, 0x43, 0xf0, 0xd6, 0xf6, 0x3b, 0x0b, 0x68,
	0x6f, 0x21, 0xd1, 0x61, 0xda, 0x35, 0x92, 0x94,
	0xa5, 0x5d, 0x47, 0x39, 0x96, 0x50, 0x5f, 0xbd,
	0x57, 0x22, 0xd2, 0x65, 0x73, 0x05, 0x8f, 0x2b,
	0xf2, 0x96, 0x53, 0x6b, 0x8e, 0xd3, 0x1e, 0xe7,
	0x92, 0xd4, 0xea, 0x41, 0xee, 0x92, 0x4d, 0x10,
	0x9f, 0x68, 0xd8, 0xe9, 0xac, 0x1f, 0x38, 0x0b,
	0x12, 0xa4, 0x1c, 0xb2, 0x63, 0x2b, 0x87, 0x07,
	0xb8, 0x1e, 0x02, 0x2b, 0x4d, 0xad, 0x99, 0xdf,
	0xe3, 0x98, 0x69, 0x29, 0x11, 0xe3, 0x77, 0x45,
	0x9a, 0xe9, 0x6c, 0x47, 0x4e, 0xc0, 0x85, 0x15,
	0x68, 0x58, 0x41, 0x37, 0xab, 0x96, 0x11, 0x94,
	0x9e, 0xbb, 0xa8, 0x5d, 0x51, 0x05, 0x93, 0xdd,
	0x2e, 0xb8, 0xdf, 0xcf, 0x83, 0xbc, 0xf6, 0x53,
	0x95, 0x93, 0x27, 0xda, 0xa5, 0x20, 0x1b, 0x7d,
	0x1d, 0xd9, 0x0c, 0xde, 0xe5, 0x3f, 0xc8, 0x60,
	0x16, 0x32, 0x95, 0x24, 0xa7, 0x2b, 0x74, 0xf1,
	0x67, 0xf9, 0xf2, 0x49, 0xda, 0x12, 0x97, 0xdd,
};

/*
 * IV method encblkno1, blkno 3.
 */
static const uint8_t aes_cbc_192_encblkno1_vec3_ctxt[SECSIZE] = {
	0xa5, 0x81, 0x86, 0x78, 0x4a, 0xd7, 0x5b, 0x83,
	0xcf, 0xbf, 0x7e, 0x3c, 0xd7, 0xcd, 0xaf, 0xfa,
	0x82, 0x18, 0xce, 0xbd, 0x8b, 0xe6, 0xd9, 0x39,
	0x22, 0x2d, 0x1e, 0x75, 0x65, 0xee, 0x61, 0xf2,
	0xc3, 0x8b, 0xf4, 0x40, 0x03, 0x73, 0x8a, 0x21,
	0x9f, 0xf3, 0xcc, 0x93, 0x08, 0x3d, 0xff, 0x8a,
	0xbc, 0x0f, 0x19, 0xa1, 0x9f, 0xc8, 0x73, 0xe8,
	0xa6, 0x14, 0x2e, 0x43, 0x19, 0x79, 0x61, 0x35,
	0x8d, 0x55, 0x06, 0xeb, 0x96, 0xe7, 0xf5, 0x4b,
	0x95, 0x5f, 0x9b, 0xb2, 0x18, 0x0d, 0x13, 0xc2,
	0x96, 0x79, 0x50, 0x78, 0x98, 0x50, 0x88, 0x2b,
	0xab, 0x05, 0x66, 0xa1, 0x3a, 0x25, 0x85, 0xe2,
	0xd0, 0xe2, 0xac, 0xb5, 0x26, 0xde, 0x95, 0x04,
	0x45, 0xe7, 0x22, 0x71, 0x02, 0xb3, 0x84, 0x4c,
	0xb5, 0xad, 0x64, 0x5c, 0x27, 0x5c, 0x71, 0xcd,
	0x0b, 0x62, 0x91, 0xd6, 0x84, 0x00, 0x62, 0x52,
	0x54, 0xbd, 0x22, 0xc8, 0x57, 0xa7, 0x41, 0xac,
	0xc7, 0xa8, 0x56, 0x6f, 0x1b, 0x7e, 0xce, 0x02,
	0x29, 0x3b, 0xc0, 0x5d, 0x8e, 0x11, 0xa9, 0x54,
	0xc2, 0xf2, 0xf0, 0x81, 0x6c, 0x9a, 0x24, 0x5b,
	0x81, 0x7d, 0xf3, 0x84, 0x93, 0xc6, 0x2a, 0xd4,
	0xd3, 0x1a, 0x2f, 0x97, 0x2e, 0x31, 0x8a, 0x62,
	0x43, 0xcb, 0xc7, 0x3d, 0x73, 0x8e, 0xd6, 0x86,
	0x17, 0x8f, 0x63, 0xd4, 0xb1, 0x50, 0x92, 0xce,
	0x90, 0x37, 0x91, 0xce, 0x37, 0x13, 0x8e, 0x61,
	0x21, 0xd8, 0x1a, 0xbf, 0x42, 0x65, 0x1d, 0x86,
	0x07, 0x04, 0x9b, 0xd1, 0xd3, 0x26, 0x6b, 0x7c,
	0xa1, 0x77, 0x54, 0x5b, 0x9f, 0x95, 0x62, 0x43,
	0xb3, 0x71, 0x1e, 0x4c, 0x32, 0xd1, 0x3e, 0xe8,
	0x60, 0x9c, 0x0c, 0x15, 0x55, 0xf0, 0x38, 0xb7,
	0x1e, 0x40, 0xe5, 0x26, 0x4e, 0x46, 0x49, 0x47,
	0x59, 0x3d, 0x49, 0x76, 0x28, 0xd3, 0xed, 0x03,
	0xdd, 0xf8, 0x1a, 0xf4, 0x1a, 0xfe, 0xe4, 0x03,
	0xb9, 0xa5, 0x8e, 0x7c, 0x91, 0x7a, 0xb2, 0x17,
	0x84, 0x97, 0x3f, 0x12, 0x68, 0xaa, 0xf5, 0x73,
	0xbc, 0x84, 0xdd, 0x03, 0x4a, 0xc4, 0xcd, 0xdb,
	0xb0, 0x8a, 0x3b, 0xac, 0xf1, 0xdd, 0x10, 0x20,
	0x69, 0xee, 0x94, 0xcd, 0x60, 0x3f, 0x01, 0xcf,
	0xf4, 0xff, 0xdb, 0x91, 0x8a, 0xf3, 0xb8, 0x44,
	0x62, 0xdc, 0xdc, 0xc8, 0x2b, 0xaf, 0x0d, 0x5e,
	0x1b, 0x58, 0x7f, 0x6b, 0x0d, 0xc4, 0xd4, 0x1c,
	0x89, 0x29, 0x60, 0x5d, 0xe9, 0x59, 0xbb, 0x19,
	0x03, 0x7c, 0x25, 0x63, 0xc6, 0x89, 0x6f, 0xe6,
	0xbe, 0xcd, 0xaa, 0xf2, 0xbf, 0x16, 0xcb, 0x47,
	0xc6, 0x74, 0xdd, 0x90, 0x41, 0x75, 0x7f, 0x26,
	0x7b, 0x5a, 0xb9, 0x11, 0xa0, 0xc7, 0x75, 0x60,
	0xc5, 0x54, 0x7d, 0xb0, 0xb4, 0xd0, 0x95, 0x01,
	0xff, 0x07, 0x49, 0x56, 0xfb, 0xec, 0xa9, 0x4c,
	0x68, 0x28, 0x41, 0x81, 0x80, 0x05, 0x88, 0x58,
	0xf5, 0xdc, 0x42, 0x99, 0xd8, 0xb7, 0x47, 0xd9,
	0xf7, 0x0e, 0x2c, 0x0f, 0x95, 0x04, 0xb3, 0xc8,
	0x8a, 0xe2, 0x21, 0x57, 0x8d, 0x64, 0x54, 0x40,
	0xf6, 0xd0, 0x3c, 0x97, 0xcf, 0x22, 0xce, 0xcd,
	0xbf, 0x05, 0x15, 0xaa, 0x89, 0xd9, 0x2b, 0x48,
	0xaf, 0x34, 0xe0, 0xf5, 0xe3, 0x58, 0x06, 0xd7,
	0x49, 0x00, 0x95, 0x3a, 0xb3, 0xc8, 0xcd, 0x2b,
	0x3e, 0xe8, 0x1b, 0x60, 0xe8, 0xea, 0xaf, 0x09,
	0xbb, 0xee, 0xce, 0xbc, 0xa0, 0x9b, 0x17, 0x90,
	0x42, 0x40, 0x18, 0x35, 0x2e, 0x17, 0xa0, 0x6e,
	0x43, 0xe7, 0xac, 0x89, 0x96, 0x3c, 0x16, 0xe0,
	0xdb, 0x09, 0x51, 0x4a, 0x45, 0x33, 0x63, 0xe9,
	0x4e, 0x3f, 0x32, 0x34, 0x36, 0x43, 0xd5, 0x0c,
	0x5a, 0x2e, 0x0e, 0x8b, 0x80, 0xb7, 0xf4, 0xe4,
	0x99, 0x9b, 0x05, 0xf5, 0xb2, 0xe4, 0x03, 0xe4,
};

const struct testvec aes_cbc_192_1_vectors[] = {
	{
		.blkno = 0,
		.ptxt = aes_cbc_ptxt,
		.ctxt = aes_cbc_192_encblkno1_vec0_ctxt,
	},
	{
		.blkno = 1,
		.ptxt = aes_cbc_ptxt,
		.ctxt = aes_cbc_192_encblkno1_vec1_ctxt,
	},
	{
		.blkno = 2,
		.ptxt = aes_cbc_ptxt,
		.ctxt = aes_cbc_192_encblkno1_vec2_ctxt,
	},
	{
		.blkno = 3,
		.ptxt = aes_cbc_ptxt,
		.ctxt = aes_cbc_192_encblkno1_vec3_ctxt,
	},
};

/*
 * IV method encblkno8, blkno 0.
 */
static const uint8_t aes_cbc_192_encblkno8_vec0_ctxt[SECSIZE] = {
	0x87, 0x9c, 0x05, 0xd6, 0x25, 0xb9, 0xe0, 0xbe,
	0x78, 0x21, 0x85, 0x81, 0x8e, 0x2f, 0x13, 0x5e,
	0xf2, 0x73, 0x11, 0xfa, 0x73, 0x77, 0x93, 0x5e,
	0x71, 0x16, 0x98, 0x68, 0x6f, 0xe8, 0x22, 0x34,
	0xf5, 0x11, 0xfa, 0x61, 0xe6, 0x1a, 0xfb, 0x42,
	0xa7, 0xa3, 0x2e, 0x0d, 0xc1, 0x9d, 0x7d, 0xd9,
	0xfb, 0xbb, 0xc5, 0x08, 0x9d, 0xc2, 0xab, 0x5c,
	0xdf, 0x9b, 0x3c, 0x1a, 0xbd, 0x66, 0x5a, 0x91,
	0x1c, 0x00, 0x59, 0x2e, 0x92, 0xe9, 0x23, 0xf6,
	0x90, 0x3d, 0x5b, 0x72, 0x76, 0x78, 0xd9, 0xa2,
	0x48, 0x33, 0x29, 0xe2, 0xfd, 0x77, 0x14, 0xda,
	0x01, 0x92, 0x63, 0xdd, 0x8c, 0x1c, 0x2e, 0xf7,
	0x61, 0xfb, 0xc5, 0x76, 0xec, 0x7f, 0xef, 0xdc,
	0xbe, 0x2d, 0x3b, 0x69, 0x30, 0xb9, 0x08, 0x00,
	0xe8, 0x37, 0x09, 0xaa, 0x2a, 0x02, 0x80, 0x11,
	0x91, 0x16, 0x94, 0x7d, 0xb5, 0xdc, 0x9f, 0xb3,
	0xb0, 0x26, 0x72, 0x85, 0x93, 0x85, 0x19, 0x08,
	0x97, 0xef, 0x97, 0x57, 0xa8, 0x76, 0x0e, 0x85,
	0xb1, 0x1d, 0x79, 0xe3, 0x7a, 0xe8, 0x06, 0x3b,
	0xc4, 0x00, 0xbd, 0xaa, 0xd9, 0x17, 0x81, 0x37,
	0x12, 0x86, 0x52, 0xea, 0x04, 0xb2, 0x11, 0x0f,
	0x5a, 0x08, 0x68, 0xcb, 0x48, 0xca, 0x2f, 0xda,
	0xa3, 0x0a, 0x60, 0x57, 0xc7, 0x80, 0x36, 0x60,
	0x05, 0xce, 0xd5, 0x43, 0xc9, 0xbc, 0x6c, 0xe6,
	0x63, 0x38, 0x2e, 0x81, 0x90, 0x34, 0x11, 0x2c,
	0x84, 0x0c, 0x62, 0x68, 0xde, 0x17, 0x57, 0x43,
	0x19, 0xa5, 0x92, 0x9d, 0x91, 0x2b, 0xa2, 0x95,
	0x7c, 0x20, 0x72, 0xaa, 0x83, 0x24, 0x54, 0x94,
	0x10, 0x80, 0xd4, 0x3f, 0x58, 0xb9, 0x7b, 0x74,
	0x68, 0xd5, 0xfb, 0x3e, 0xdd, 0xb4, 0xdf, 0x65,
	0x72, 0x88, 0x45, 0x8a, 0xd0, 0x93, 0x6e, 0x99,
	0x84, 0xad, 0x39, 0x73, 0x16, 0x88, 0xdc, 0x89,
	0x33, 0x34, 0xd7, 0xd8, 0x97, 0xfb, 0x90, 0xd2,
	0xc5, 0x8e, 0x94, 0xc4, 0xf1, 0xfe, 0xbe, 0x23,
	0xf1, 0x3a, 0x10, 0x1c, 0x42, 0x6b, 0xf5, 0xee,
	0xe4, 0x78, 0x8a, 0x7e, 0x13, 0x02, 0x25, 0xcb,
	0xd1, 0x61, 0x1f, 0xab, 0x45, 0x1f, 0x90, 0x88,
	0x0f, 0x6b, 0xff, 0x61, 0xba, 0xf3, 0xac, 0x8e,
	0x13, 0xc2, 0xfb, 0xca, 0x41, 0xed, 0xfe, 0x6c,
	0xcb, 0xdf, 0x97, 0x60, 0x29, 0x8a, 0x72, 0x8d,
	0x7d, 0xad, 0x6e, 0xe9, 0x7b, 0xc4, 0x92, 0x14,
	0x5e, 0x33, 0x27, 0xe2, 0xda, 0x2f, 0x95, 0x5f,
	0x40, 0x27, 0xeb, 0xdb, 0x0d, 0x1e, 0xc5, 0xd4,
	0x43, 0x50, 0x1a, 0x62, 0x82, 0xbe, 0x24, 0x7f,
	0xb7, 0x46, 0xa8, 0x70, 0x10, 0x33, 0xb6, 0x3f,
	0xbf, 0xa8, 0xa8, 0x85, 0xab, 0x1d, 0xb4, 0x3f,
	0x84, 0x06, 0x91, 0xd6, 0x18, 0x3d, 0xeb, 0x8b,
	0x3f, 0x9b, 0x37, 0x9e, 0x2e, 0xd2, 0xec, 0xe5,
	0x2d, 0xf0, 0x3f, 0x45, 0xd5, 0x9d, 0xb9, 0x28,
	0x89, 0xe4, 0x0c, 0xa9, 0x38, 0xca, 0x22, 0x56,
	0x53, 0xdf, 0x49, 0xba, 0x5d, 0x99, 0xd6, 0x4b,
	0x1d, 0x0d, 0x6d, 0xee, 0x7c, 0xf2, 0x6f, 0x50,
	0x04, 0xf1, 0xf8, 0x49, 0xd1, 0x2f, 0x50, 0x3e,
	0xf1, 0x08, 0x49, 0x17, 0x08, 0xd2, 0xac, 0x5d,
	0x58, 0xe7, 0x27, 0xe6, 0x59, 0x02, 0x9f, 0x1c,
	0x40, 0xff, 0x6c, 0x67, 0xae, 0x49, 0x1a, 0x2a,
	0xab, 0xd9, 0x63, 0x25, 0x2d, 0x9b, 0xd8, 0x1a,
	0x41, 0xa6, 0xea, 0x72, 0xfd, 0x56, 0xa1, 0x57,
	0x59, 0xdd, 0xf5, 0xa3, 0xb2, 0x2f, 0x64, 0xb1,
	0xc5, 0xfe, 0x8d, 0x9b, 0x93, 0xd1, 0x51, 0x77,
	0x13, 0x50, 0x74, 0x30, 0x28, 0xe4, 0x7a, 0x06,
	0x69, 0xd4, 0xa8, 0x0a, 0xae, 0x02, 0x4a, 0x61,
	0x24, 0xc2, 0xcd, 0xc8, 0xd3, 0x12, 0x2e, 0xac,
	0x9a, 0x0c, 0x24, 0x06, 0xb8, 0x1e, 0x3d, 0x29,
};

/*
 * IV method encblkno8, blkno 1.
 */
static const uint8_t aes_cbc_192_encblkno8_vec1_ctxt[SECSIZE] = {
	0x1e, 0x3b, 0x66, 0x76, 0xd9, 0x9e, 0xf7, 0x82,
	0x17, 0x76, 0x69, 0x4d, 0x64, 0x63, 0xf1, 0x01,
	0x81, 0x8a, 0xa4, 0x97, 0x05, 0x42, 0xdb, 0x8c,
	0x27, 0xc8, 0xfd, 0x08, 0x21, 0x17, 0x87, 0xa9,
	0x0c, 0x86, 0x2d, 0xda, 0x17, 0xd5, 0x5d, 0x67,
	0x12, 0x93, 0x8d, 0x34, 0x5a, 0xfc, 0x2a, 0x49,
	0x1a, 0x1a, 0x77, 0x20, 0xfb, 0x1d, 0x5d, 0xd8,
	0x99, 0xb0, 0x8f, 0x1c, 0x13, 0x4d, 0x28, 0x6d,
	0x2d, 0x79, 0xa9, 0x8e, 0x04, 0x0c, 0x5a, 0xd5,
	0x52, 0x09, 0x15, 0x4a, 0xfb, 0x7a, 0xf8, 0xdc,
	0x3b, 0x77, 0xaf, 0xe0, 0x80, 0x6b, 0xac, 0x5f,
	0xc0, 0x0f, 0x0f, 0x29, 0xf5, 0xcc, 0xbc, 0x85,
	0x77, 0xe7, 0x9f, 0x59, 0x23, 0x83, 0x67, 0x74,
	0x3b, 0x1c, 0x0f, 0x75, 0xd8, 0x58, 0xa2, 0xce,
	0x8c, 0x3a, 0x80, 0xd7, 0xff, 0xa1, 0x83, 0xa3,
	0xe0, 0xad, 0x18, 0x7a, 0xc5, 0x28, 0x28, 0x71,
	0x46, 0xb5, 0x13, 0x76, 0x4d, 0x67, 0x37, 0x38,
	0x3f, 0x9e, 0xa6, 0x8b, 0xc2, 0xaf, 0x83, 0x7d,
	0x8b, 0x82, 0xd0, 0xe2, 0xec, 0x13, 0xce, 0x2b,
	0x1e, 0x13, 0xe7, 0xb6, 0xfa, 0x9e, 0xa2, 0x32,
	0xb7, 0xdc, 0xe5, 0xb5, 0x35, 0xa3, 0xb4, 0x84,
	0x57, 0x05, 0x2d, 0x3e, 0xb0, 0x0a, 0x52, 0x61,
	0x00, 0xe4, 0x84, 0xab, 0xf4, 0x98, 0xe4, 0xe6,
	0xcd, 0xb1, 0xd4, 0x40, 0x31, 0x5f, 0x8f, 0x73,
	0x16, 0x6e, 0xc0, 0x3d, 0x07, 0x5d, 0x6b, 0x91,
	0x70, 0x71, 0x8a, 0x4b, 0xfe, 0xeb, 0xbe, 0x04,
	0x5d, 0x75, 0x0a, 0x74, 0x52, 0x1e, 0xd3, 0x94,
	0xc5, 0xcd, 0xc1, 0xd6, 0x12, 0x6a, 0x58, 0x52,
	0x6e, 0x45, 0x1f, 0x49, 0x09, 0x4c, 0x32, 0xf3,
	0x3d, 0x3d, 0x73, 0x15, 0xa3, 0xa5, 0x2f, 0xf2,
	0x02, 0x10, 0x1e, 0xaf, 0xf5, 0xb4, 0x78, 0x48,
	0x8a, 0x6c, 0x58, 0x71, 0x77, 0x91, 0x95, 0x57,
	0x79, 0xbf, 0x1f, 0x3e, 0xb3, 0xf8, 0xc4, 0x33,
	0x07, 0x5d, 0x96, 0x41, 0x76, 0xb1, 0xe1, 0xe0,
	0xa9, 0x97, 0x14, 0x99, 0x1d, 0xaa, 0x91, 0xbb,
	0xdf, 0x89, 0xf1, 0x0d, 0xd0, 0x52, 0xf9, 0xa7,
	0x4c, 0x82, 0xc0, 0xeb, 0xb7, 0xaf, 0x7b, 0x4b,
	0x5a, 0x2a, 0x7a, 0x4e, 0xb2, 0x69, 0x87, 0x28,
	0x84, 0xf7, 0x76, 0x56, 0xee, 0xf8, 0x37, 0x35,
	0xc9, 0xbc, 0x08, 0x8b, 0xfe, 0x1e, 0x54, 0xb3,
	0x01, 0xa7, 0x0f, 0x20, 0x70, 0xac, 0xa6, 0x6b,
	0x9f, 0x98, 0xfe, 0xdb, 0x3e, 0x4f, 0x9f, 0xfc,
	0x95, 0x37, 0xf4, 0x90, 0x61, 0x62, 0x60, 0xeb,
	0x7a, 0x4a, 0x56, 0xae, 0x49, 0xcc, 0x92, 0xff,
	0xd3, 0x06, 0xc6, 0x62, 0x4c, 0x05, 0x28, 0xa7,
	0x3f, 0xe9, 0xee, 0x70, 0x6f, 0xd2, 0x80, 0x41,
	0x4d, 0xa0, 0xbc, 0x00, 0xaf, 0x30, 0xe4, 0x34,
	0x61, 0xda, 0xb4, 0xff, 0x2a, 0x85, 0x8b, 0x1a,
	0xbf, 0xb5, 0xe4, 0x7f, 0x27, 0xee, 0xf3, 0x25,
	0xe6, 0x52, 0x2a, 0x83, 0xbe, 0xe4, 0x64, 0xc3,
	0x67, 0x0c, 0x9e, 0x0f, 0xba, 0xb4, 0x67, 0xd1,
	0x1b, 0x4a, 0xb0, 0xb2, 0xb4, 0xf2, 0x8a, 0x1b,
	0x21, 0x34, 0x3c, 0x97, 0x5a, 0xdb, 0x92, 0x8b,
	0x2d, 0xe9, 0x94, 0x4e, 0x11, 0xfb, 0xd4, 0x2e,
	0xc2, 0xed, 0xf9, 0x75, 0xfd, 0x1a, 0xef, 0x3b,
	0x98, 0x5d, 0xa9, 0x75, 0xd5, 0x14, 0x0a, 0xe3,
	0xda, 0x07, 0xa6, 0x20, 0x7b, 0x49, 0x47, 0x87,
	0xff, 0xf0, 0xe8, 0x7e, 0xcf, 0xc4, 0x2c, 0x02,
	0xdd, 0x53, 0xe9, 0x79, 0xc7, 0x6d, 0x16, 0x9f,
	0x2b, 0xd7, 0x1a, 0x36, 0x25, 0x5c, 0xba, 0x5c,
	0xdb, 0x44, 0x88, 0x99, 0x32, 0x2e, 0xb6, 0x3f,
	0xb4, 0xdd, 0x15, 0xeb, 0xec, 0x2a, 0x9e, 0xc5,
	0x37, 0x30, 0x2a, 0xd5, 0xc4, 0x2a, 0x9b, 0x40,
	0x97, 0x83, 0x94, 0xe7, 0x79, 0x79, 0x63, 0x4b,
};

/*
 * IV method encblkno8, blkno 2.
 */
static const uint8_t aes_cbc_192_encblkno8_vec2_ctxt[SECSIZE] = {
	0x34, 0x07, 0x20, 0x14, 0x64, 0x0b, 0xa2, 0x2c,
	0xed, 0xba, 0x46, 0x24, 0xa0, 0xe6, 0x99, 0x8a,
	0x20, 0x75, 0x5f, 0x9f, 0x2a, 0x10, 0xa6, 0x1c,
	0x52, 0x60, 0x18, 0x67, 0xd6, 0x0d, 0x90, 0x4e,
	0xbc, 0x25, 0x5f, 0x81, 0xb4, 0x10, 0xdb, 0xd9,
	0xaf, 0x36, 0x84, 0x5c, 0x20, 0x25, 0x25, 0xbf,
	0x0d, 0xfa, 0xc5, 0x75, 0x2b, 0xec, 0xf2, 0xa6,
	0x69, 0x5c, 0xfe, 0xee, 0x21, 0xd8, 0x87, 0xdf,
	0xe3, 0x83, 0xeb, 0xb3, 0x3f, 0x5b, 0xda, 0x37,
	0x11, 0x05, 0xf7, 0xd8, 0xe0, 0x94, 0x08, 0x2b,
	0x75, 0x6b, 0xf3, 0x40, 0x53, 0x85, 0xde, 0x7a,
	0x64, 0xb1, 0x0e, 0x5f, 0x01, 0xb5, 0xfb, 0x74,
	0x48, 0x9a, 0xd4, 0x41, 0x33, 0x70, 0x9b, 0x08,
	0x7e, 0x34, 0x60, 0xfc, 0xfa, 0xe6, 0x2c, 0xec,
	0x0e, 0xb7, 0x1a, 0xf1, 0x49, 0x48, 0x0c, 0xd4,
	0xd7, 0xbc, 0x60, 0x28, 0xdb, 0x57, 0xa4, 0x29,
	0x55, 0x2d, 0x92, 0xa6, 0xca, 0x9a, 0xaf, 0x4d,
	0x7f, 0xb8, 0x29, 0x9f, 0x50, 0x98, 0x21, 0x94,
	0x7a, 0x94, 0x44, 0x3d, 0xd1, 0xcf, 0xf4, 0x6f,
	0xad, 0xb4, 0x58, 0x66, 0x74, 0x01, 0x2c, 0x5b,
	0x8f, 0x1b, 0xa6, 0x09, 0xd0, 0x3f, 0x79, 0xc9,
	0x4f, 0x3b, 0x37, 0x0d, 0xb8, 0x07, 0xb0, 0x61,
	0xbc, 0x5a, 0x40, 0x3a, 0x10, 0x3c, 0x12, 0xe6,
	0x04, 0xc7, 0xd1, 0xe1, 0x18, 0x6f, 0xde, 0x72,
	0xf5, 0xcf, 0x24, 0x58, 0x76, 0xe1, 0xcd, 0x62,
	0x90, 0xc3, 0x16, 0xcc, 0x3f, 0xda, 0xd6, 0x6b,
	0x6a, 0xcc, 0x61, 0x76, 0xc1, 0xaf, 0xdc, 0x53,
	0xef, 0x06, 0x23, 0x22, 0x93, 0x11, 0x59, 0xf5,
	0x7f, 0x46, 0xac, 0xb8, 0x6c, 0x3b, 0x36, 0x69,
	0xc5, 0x14, 0x0a, 0x51, 0xa1, 0x5f, 0xb9, 0xc7,
	0x37, 0xe3, 0xd9, 0xaf, 0x8c, 0xe9, 0x49, 0xd4,
	0xf9, 0xf9, 0x5e, 0x1f, 0x5f, 0x7c, 0x07, 0xb5,
	0x1c, 0x9e, 0xbd, 0x10, 0x75, 0xc3, 0x93, 0x48,
	0xdc, 0x32, 0xe7, 0x55, 0x90, 0x48, 0x42, 0xc0,
	0x73, 0x20, 0x40, 0x17, 0xbb, 0x71, 0x30, 0xfe,
	0xd1, 0x84, 0xe9, 0x7d, 0x92, 0xd4, 0xff, 0xbe,
	0x3e, 0xd9, 0x41, 0xfb, 0x41, 0x43, 0x2b, 0x9f,
	0x04, 0x7b, 0xe7, 0x81, 0xbb, 0x2a, 0xd6, 0x7b,
	0x96, 0x72, 0x29, 0x30, 0x52, 0x5c, 0xea, 0xcc,
	0x4c, 0x77, 0xed, 0x5a, 0xd9, 0xab, 0x51, 0x90,
	0x21, 0x3b, 0x5b, 0x26, 0xeb, 0x14, 0xd5, 0xea,
	0x01, 0xb0, 0x7c, 0xbd, 0xa6, 0x3d, 0x7f, 0x42,
	0xd7, 0x7e, 0xf1, 0x6c, 0x71, 0x7d, 0xc0, 0x25,
	0x61, 0xe9, 0x66, 0xe1, 0xf2, 0x67, 0x99, 0xa1,
	0xe7, 0x3a, 0x6f, 0x88, 0x1e, 0x8b, 0x76, 0xed,
	0x50, 0x2c, 0x4e, 0xac, 0x73, 0xd7, 0xf2, 0x85,
	0x8f, 0xcc, 0xb1, 0x4f, 0x6c, 0x9a, 0xf7, 0x45,
	0x28, 0x4f, 0xfc, 0x3f, 0xf1, 0x80, 0xc3, 0xf3,
	0xce, 0x5e, 0xfc, 0x56, 0xd9, 0x45, 0xdd, 0x81,
	0xe3, 0x48, 0x22, 0xc9, 0xb8, 0x13, 0xc1, 0x48,
	0x6c, 0x95, 0x97, 0xc0, 0x91, 0x37, 0xf5, 0x8a,
	0x11, 0x3b, 0xab, 0xce, 0x7a, 0xb0, 0xb4, 0x4c,
	0xba, 0xc0, 0x91, 0x7f, 0x3c, 0x27, 0xe9, 0xc0,
	0x58, 0x92, 0x70, 0x67, 0xf4, 0x80, 0x40, 0x92,
	0x51, 0x80, 0x8e, 0x9d, 0x2d, 0x87, 0x89, 0x8e,
	0xe7, 0xd1, 0xb5, 0xc5, 0x4f, 0xd0, 0x86, 0x31,
	0x7f, 0x90, 0x77, 0x05, 0x35, 0xfe, 0xa7, 0xcb,
	0x9d, 0x94, 0xf3, 0xf8, 0xbb, 0x4f, 0xe1, 0xb3,
	0x48, 0x57, 0xbf, 0xd1, 0x77, 0xe8, 0x72, 0x31,
	0x4d, 0x2f, 0xe8, 0xa0, 0xf4, 0x7c, 0x25, 0x9c,
	0xcd, 0xa5, 0x7e, 0xd3, 0x30, 0xda, 0x46, 0xf5,
	0x48, 0x9e, 0x39, 0x34, 0x94, 0xd6, 0x24, 0x10,
	0xfc, 0x74, 0x2b, 0x6d, 0xcc, 0x00, 0x76, 0x3e,
	0x3b, 0x85, 0xfa, 0xef, 0x87, 0x70, 0x53, 0x4e,
};

/*
 * IV method encblkno8, blkno 3.
 */
static const uint8_t aes_cbc_192_encblkno8_vec3_ctxt[SECSIZE] = {
	0xfe, 0xad, 0xf3, 0x4a, 0x9d, 0x64, 0x4e, 0x5d,
	0xaf, 0xa8, 0x44, 0x7e, 0xc9, 0x75, 0xe8, 0xd0,
	0x87, 0x73, 0x66, 0x4c, 0x77, 0x00, 0xfb, 0x7b,
	0x04, 0xe7, 0xd8, 0x82, 0x75, 0xe3, 0xa5, 0xbc,
	0xf3, 0x80, 0xae, 0x7c, 0xc9, 0x75, 0x9a, 0xc1,
	0x73, 0x49, 0x69, 0xf6, 0xa0, 0x49, 0x6e, 0x77,
	0x5f, 0x9b, 0x95, 0x9b, 0x9f, 0x41, 0x54, 0x57,
	0x0e, 0x3c, 0xe5, 0x2c, 0xbb, 0xbf, 0xd5, 0x76,
	0xf6, 0xb6, 0x05, 0xaa, 0x20, 0x5b, 0xdb, 0xcb,
	0x81, 0xad, 0x0c, 0x8a, 0x68, 0x94, 0x7d, 0x88,
	0xdc, 0x15, 0x6c, 0x89, 0x97, 0x53, 0x30, 0x96,
	0x4a, 0x54, 0xf9, 0x88, 0x00, 0xf7, 0x3b, 0x99,
	0xfc, 0x82, 0xe3, 0x48, 0xd2, 0x16, 0x2b, 0xba,
	0xd4, 0xba, 0x24, 0xd0, 0xd1, 0xb0, 0x8e, 0xcd,
	0x77, 0xdc, 0x01, 0xdf, 0xb2, 0x20, 0xc5, 0xa7,
	0x48, 0x2a, 0xcf, 0x56, 0xc8, 0x63, 0x6e, 0xc9,
	0xa8, 0xa4, 0xc2, 0x9c, 0x66, 0x25, 0x50, 0x77,
	0x08, 0x51, 0x92, 0xce, 0x3c, 0xaf, 0xff, 0xee,
	0x3e, 0x6d, 0x61, 0x37, 0xcd, 0x85, 0x67, 0x9c,
	0xe0, 0x7e, 0xa6, 0x17, 0x7b, 0x5f, 0x6a, 0xe2,
	0x4e, 0x76, 0xca, 0x95, 0x88, 0xdf, 0xad, 0x78,
	0x91, 0xfa, 0x9e, 0x71, 0x3e, 0xfd, 0x10, 0x78,
	0x32, 0x2b, 0x75, 0xbc, 0x3a, 0x06, 0x55, 0x8b,
	0x9b, 0xfb, 0x9c, 0x4b, 0xa1, 0x7d, 0x35, 0x3d,
	0x63, 0x80, 0x30, 0x61, 0xe0, 0x2d, 0x8a, 0x28,
	0xb4, 0x2d, 0x48, 0x9d, 0x27, 0x1a, 0x28, 0x86,
	0xfc, 0xfa, 0x93, 0xcf, 0x3e, 0x9c, 0x41, 0xc8,
	0xc5, 0x5e, 0x88, 0x22, 0xb8, 0xaf, 0x1d, 0x92,
	0xc5, 0x91, 0x1b, 0x1e, 0x95, 0x62, 0xbb, 0x80,
	0x0c, 0xae, 0x2a, 0xb3, 0x55, 0x77, 0x86, 0x39,
	0xa6, 0xed, 0xc1, 0xd2, 0xc4, 0x95, 0x7e, 0xd4,
	0xbe, 0xf3, 0x1b, 0xbc, 0x5e, 0x92, 0x0d, 0x9c,
	0x38, 0xb1, 0xb9, 0xd3, 0xf6, 0x3f, 0x97, 0xf9,
	0x48, 0x08, 0x2b, 0xa6, 0x98, 0x50, 0xc9, 0x84,
	0xec, 0x54, 0xe0, 0x1a, 0x65, 0x76, 0xf2, 0xbe,
	0x62, 0xb9, 0x40, 0x3a, 0xb1, 0xef, 0xa0, 0x51,
	0xab, 0x3a, 0xfa, 0xaf, 0x33, 0x32, 0xa5, 0x0c,
	0xc7, 0x9a, 0x9c, 0x5c, 0xa7, 0x8e, 0xc6, 0x4e,
	0x61, 0xe3, 0x83, 0xa1, 0xd4, 0x2c, 0xb2, 0x2c,
	0x46, 0x5a, 0xbf, 0x96, 0xeb, 0xda, 0x45, 0x2d,
	0x25, 0x37, 0x69, 0x1a, 0x6b, 0xd6, 0xbc, 0xe1,
	0x28, 0x65, 0xf9, 0xfc, 0xa7, 0xda, 0xf8, 0x79,
	0x87, 0x18, 0x99, 0x01, 0x74, 0x5a, 0x42, 0x79,
	0x8e, 0xe4, 0x23, 0x1a, 0x6c, 0xda, 0x93, 0x0f,
	0x19, 0xf0, 0xff, 0x0e, 0x25, 0x45, 0x1e, 0xbb,
	0x17, 0xca, 0x87, 0x6a, 0x9e, 0xd0, 0xd3, 0xd5,
	0x22, 0x5f, 0xce, 0x92, 0xeb, 0x82, 0x8e, 0x3e,
	0x4e, 0x99, 0x44, 0xa2, 0x9e, 0x78, 0x53, 0x89,
	0x4e, 0x45, 0x51, 0x41, 0x28, 0x91, 0xdb, 0x7e,
	0x8f, 0xac, 0xc2, 0xee, 0x09, 0xcb, 0xed, 0x04,
	0x7b, 0x37, 0xa1, 0x1d, 0x9c, 0x90, 0x19, 0xb1,
	0xdd, 0xc3, 0x22, 0xc8, 0x70, 0x07, 0x26, 0xce,
	0x4a, 0xc4, 0xde, 0xee, 0x87, 0xf3, 0x62, 0x69,
	0xed, 0xb2, 0x2d, 0x10, 0xc4, 0xfa, 0x86, 0x2e,
	0xd1, 0xb8, 0x58, 0xa3, 0xa4, 0x0b, 0x30, 0x87,
	0x23, 0x62, 0xed, 0xf3, 0x7b, 0x80, 0x7e, 0x4f,
	0xc2, 0xb3, 0xe8, 0xba, 0x25, 0x3e, 0xd3, 0x12,
	0x7e, 0x27, 0xd5, 0x72, 0x3b, 0x02, 0xf4, 0xfd,
	0x2f, 0x8b, 0xc2, 0x5f, 0x44, 0x40, 0x31, 0x88,
	0x73, 0x81, 0xa3, 0xcc, 0xc4, 0x78, 0x2b, 0xfc,
	0x41, 0x2e, 0xb2, 0xd0, 0xb4, 0x00, 0x29, 0xc1,
	0x46, 0xdf, 0xc1, 0xbd, 0x15, 0x59, 0xa3, 0x6a,
	0xc8, 0x2f, 0x29, 0x28, 0x12, 0x9b, 0x1e, 0xea,
	0x4e, 0xa9, 0x80, 0xa1, 0xb8, 0x89, 0x21, 0x3b,
};

const struct testvec aes_cbc_192_8_vectors[] = {
	{
		.blkno = 0,
		.ptxt = aes_cbc_ptxt,
		.ctxt = aes_cbc_192_encblkno8_vec0_ctxt,
	},
	{
		.blkno = 1,
		.ptxt = aes_cbc_ptxt,
		.ctxt = aes_cbc_192_encblkno8_vec1_ctxt,
	},
	{
		.blkno = 2,
		.ptxt = aes_cbc_ptxt,
		.ctxt = aes_cbc_192_encblkno8_vec2_ctxt,
	},
	{
		.blkno = 3,
		.ptxt = aes_cbc_ptxt,
		.ctxt = aes_cbc_192_encblkno8_vec3_ctxt,
	},
};

/*
 * IV method encblkno1, blkno 0.
 */
static const uint8_t aes_cbc_256_encblkno1_vec0_ctxt[SECSIZE] = {
	0x1a, 0xa4, 0xe3, 0x09, 0x23, 0x2e, 0x91, 0x1b,
	0xa8, 0x3e, 0xda, 0x92, 0xb3, 0x22, 0xd2, 0xe8,
	0x8b, 0xed, 0x6c, 0xa7, 0x78, 0xe6, 0x32, 0x25,
	0xc4, 0x88, 0xd5, 0xb7, 0x6e, 0xef, 0xbf, 0x37,
	0x00, 0xd9, 0xb2, 0x55, 0x10, 0x4f, 0x7d, 0x84,
	0x3a, 0xae, 0xd2, 0xc6, 0x48, 0xdd, 0x3c, 0xd5,
	0x9b, 0xa7, 0xf8, 0xc2, 0xda, 0x6d, 0x14, 0xa2,
	0xdc, 0x54, 0x12, 0x8f, 0x1c, 0x22, 0x98, 0x6a,
	0xc0, 0x5f, 0x47, 0xa7, 0x78, 0xec, 0x79, 0x5d,
	0x04, 0xed, 0x5e, 0x20, 0x33, 0x53, 0x66, 0x40,
	0x83, 0x94, 0x5b, 0x34, 0x05, 0x25, 0x2e, 0x17,
	0xba, 0x23, 0x60, 0xb1, 0xd0, 0x27, 0xf0, 0x24,
	0xd2, 0x0b, 0xd3, 0xea, 0xa7, 0x13, 0x1e, 0xf9,
	0x56, 0xe1, 0xd4, 0xa2, 0x89, 0x5a, 0xaa, 0x42,
	0xa9, 0xd7, 0x85, 0x64, 0x9e, 0x44, 0x71, 0xa2,
	0xf9, 0xc3, 0xf4, 0x81, 0xbd, 0xa0, 0x40, 0xed,
	0x33, 0xeb, 0x09, 0x0f, 0x7f, 0x78, 0xe4, 0xd5,
	0x7b, 0x61, 0x42, 0xee, 0x65, 0x25, 0xcc, 0xba,
	0xc6, 0x99, 0x29, 0x25, 0x71, 0x9a, 0xf0, 0x0e,
	0x98, 0x3f, 0x12, 0xf2, 0xf9, 0x4d, 0x00, 0x3c,
	0xbe, 0x9f, 0x2b, 0x83, 0x1e, 0x5b, 0xab, 0x80,
	0x4c, 0x81, 0x82, 0x29, 0xbb, 0xeb, 0xc0, 0x89,
	0x07, 0x43, 0xdd, 0x69, 0xd3, 0x02, 0x6c, 0x1c,
	0x4b, 0xab, 0x44, 0x42, 0x6c, 0x25, 0xfc, 0xf5,
	0x73, 0xaa, 0x60, 0x48, 0xbc, 0xd2, 0x1c, 0x77,
	0x8b, 0x64, 0x3e, 0x5f, 0x24, 0xae, 0x14, 0x65,
	0xea, 0x18, 0xb1, 0xab, 0xbc, 0x3d, 0xa3, 0xb9,
	0xfc, 0xcc, 0x0f, 0x8d, 0x8e, 0x13, 0x0f, 0x4d,
	0x4e, 0xeb, 0x90, 0x9b, 0x1e, 0xbf, 0x2a, 0xc7,
	0xac, 0x5b, 0x11, 0xeb, 0x67, 0xf2, 0x9d, 0xef,
	0xf3, 0x66, 0x9e, 0x81, 0x9f, 0x24, 0x4d, 0xcd,
	0x4f, 0x31, 0xce, 0xc9, 0xa4, 0x2c, 0xd7, 0x58,
	0x7c, 0x2e, 0x88, 0xa2, 0xec, 0x4c, 0x02, 0x29,
	0x00, 0xbd, 0x14, 0x0f, 0xaa, 0xd8, 0xc3, 0x02,
	0x64, 0xdc, 0xa0, 0x15, 0xc8, 0xf6, 0x17, 0x8b,
	0x9c, 0xb3, 0xf2, 0x27, 0xc1, 0x3f, 0x60, 0x94,
	0x33, 0x10, 0x89, 0x49, 0x5f, 0xd2, 0x0e, 0xfe,
	0x9e, 0x99, 0x68, 0x95, 0xe4, 0x12, 0xfc, 0xe3,
	0x7f, 0xc4, 0xb1, 0x88, 0x4f, 0x66, 0xcd, 0x24,
	0x89, 0x09, 0xbb, 0x01, 0xf6, 0x9a, 0xe4, 0x41,
	0xee, 0x83, 0xd2, 0x28, 0xf5, 0x28, 0x49, 0x13,
	0x78, 0xfb, 0xb2, 0x0d, 0x5c, 0x97, 0xf4, 0x9c,
	0xe0, 0xdf, 0xef, 0x84, 0x36, 0x7d, 0xe5, 0x45,
	0xe0, 0xf8, 0xce, 0x82, 0x39, 0xc4, 0x54, 0x69,
	0xf1, 0x62, 0x7d, 0x1a, 0xf6, 0x6c, 0x20, 0x86,
	0x72, 0x4b, 0xf9, 0x3d, 0x87, 0x68, 0xec, 0x74,
	0x67, 0xee, 0xbd, 0xb8, 0xc6, 0x12, 0x91, 0x0f,
	0xf6, 0xd9, 0x4f, 0x34, 0x96, 0xa9, 0xe7, 0x52,
	0x7b, 0xe0, 0x08, 0x57, 0x0a, 0x8b, 0x09, 0xcb,
	0xd3, 0x3e, 0x4e, 0x64, 0xca, 0x38, 0x50, 0x07,
	0x0e, 0x7b, 0x95, 0x69, 0x1b, 0x82, 0xba, 0x50,
	0x93, 0x4f, 0x9a, 0x8e, 0x11, 0x9b, 0x64, 0xf5,
	0x6a, 0xd4, 0x81, 0xf0, 0x1f, 0xb8, 0x85, 0x90,
	0x9c, 0x79, 0xde, 0xcb, 0x50, 0xba, 0xa9, 0x56,
	0x66, 0xd1, 0x1e, 0x78, 0xa8, 0x6a, 0xd5, 0xa5,
	0x83, 0x73, 0xe2, 0x88, 0xf2, 0x04, 0x33, 0x61,
	0xdf, 0x89, 0xd5, 0x3d, 0x03, 0x4e, 0x94, 0xb0,
	0x0f, 0x8d, 0x4d, 0xb4, 0x09, 0xb2, 0xf1, 0xb0,
	0xe7, 0xfe, 0xb0, 0x18, 0xe2, 0xfc, 0x92, 0xeb,
	0x2d, 0x7d, 0x56, 0x29, 0xbd, 0x34, 0x20, 0x7c,
	0xb6, 0xe7, 0x7b, 0xd7, 0x95, 0xa5, 0x0d, 0x10,
	0xbc, 0x7d, 0x9d, 0xd9, 0xbe, 0xc7, 0x23, 0x44,
	0x37, 0xb3, 0x98, 0x36, 0x33, 0x1a, 0x11, 0xfe,
	0x41, 0xea, 0x59, 0x48, 0x75, 0x34, 0xf6, 0xc4,
};

/*
 * IV method encblkno1, blkno 1.
 */
static const uint8_t aes_cbc_256_encblkno1_vec1_ctxt[SECSIZE] = {
	0x58, 0xfc, 0x1e, 0x48, 0x66, 0x7d, 0x91, 0xc7,
	0x56, 0xa3, 0x41, 0x89, 0xe8, 0x1e, 0x02, 0x77,
	0x93, 0x38, 0x12, 0x99, 0x06, 0x0d, 0xf3, 0x6d,
	0x2a, 0x5d, 0x3d, 0x7b, 0x4e, 0x05, 0x4f, 0x8f,
	0xe3, 0x86, 0x76, 0xfe, 0x11, 0x9d, 0xde, 0xd4,
	0x83, 0xd9, 0x47, 0x8d, 0x51, 0xdf, 0x4a, 0x24,
	0x2d, 0x11, 0xf0, 0xbd, 0xde, 0x17, 0x7e, 0x52,
	0x20, 0xc7, 0x17, 0x88, 0x2e, 0xa4, 0xd5, 0xa0,
	0x1e, 0xbc, 0x61, 0x15, 0x1e, 0x52, 0xa1, 0x8b,
	0xe9, 0xe4, 0x1f, 0x81, 0x49, 0x64, 0x17, 0xd4,
	0xef, 0xb6, 0x40, 0x05, 0x2f, 0x36, 0xf7, 0x39,
	0x03, 0x05, 0x80, 0xff, 0xf2, 0x1a, 0x15, 0xf1,
	0xfc, 0xaf, 0x71, 0x51, 0x73, 0xc5, 0x9e, 0x2f,
	0xd1, 0x7a, 0x2d, 0xd7, 0xed, 0x90, 0x11, 0xd2,
	0x80, 0x49, 0x46, 0x9f, 0x13, 0xa7, 0x32, 0x33,
	0x24, 0x39, 0x59, 0xf1, 0xed, 0x64, 0x75, 0x61,
	0xc3, 0x14, 0x68, 0x48, 0xf7, 0xc7, 0xbd, 0xe0,
	0x21, 0x59, 0x91, 0x07, 0x70, 0x83, 0x8f, 0xfc,
	0x59, 0x72, 0xca, 0xdd, 0x60, 0xa0, 0xbb, 0xb1,
	0x2f, 0xb8, 0x98, 0x8d, 0xf2, 0x4d, 0x3a, 0x19,
	0xbc, 0x6b, 0x37, 0xad, 0xd2, 0xb5, 0x7d, 0x1c,
	0x4a, 0x7b, 0x58, 0x76, 0x2e, 0xf5, 0x6b, 0xaf,
	0x4c, 0x92, 0x00, 0x8a, 0xb4, 0xa3, 0x86, 0x66,
	0x07, 0xc7, 0xfc, 0x57, 0x3c, 0x73, 0xf4, 0x8b,
	0xef, 0xb6, 0xae, 0x01, 0xfb, 0x88, 0x13, 0x04,
	0xa8, 0xc7, 0xec, 0xc4, 0xe0, 0x67, 0x3a, 0xfb,
	0x67, 0xce, 0x83, 0x9b, 0x8e, 0x66, 0xff, 0xa6,
	0x17, 0x1b, 0x66, 0x27, 0xdf, 0x2a, 0xfe, 0xf3,
	0x9a, 0xba, 0x59, 0xce, 0x28, 0xd4, 0xd2, 0x40,
	0x78, 0xb6, 0xe9, 0x7d, 0x8b, 0xcc, 0x47, 0x5c,
	0xf6, 0x5d, 0xc2, 0x5d, 0xe0, 0xa7, 0x61, 0x8b,
	0xe6, 0x7d, 0x38, 0xb6, 0xea, 0xfb, 0x13, 0x31,
	0x33, 0x2a, 0xb5, 0x45, 0x7b, 0xf6, 0x9f, 0x29,
	0x06, 0x2d, 0xd8, 0xab, 0x36, 0x27, 0xe4, 0x6c,
	0xf1, 0xab, 0xcd, 0xb9, 0x08, 0x0f, 0x4b, 0x8f,
	0x22, 0xea, 0xe4, 0x5d, 0x22, 0x05, 0x2e, 0xd4,
	0xd7, 0xff, 0x58, 0x50, 0x38, 0x17, 0x6f, 0x80,
	0x61, 0x98, 0xdc, 0xd4, 0x9f, 0x8f, 0xeb, 0x13,
	0xd3, 0x86, 0xe9, 0xa9, 0xe7, 0x07, 0x6f, 0x4f,
	0x54, 0x9e, 0x37, 0x3d, 0xbc, 0x82, 0x5f, 0x4f,
	0xd5, 0x0c, 0x21, 0xaa, 0x91, 0xcb, 0x06, 0x9a,
	0xaf, 0x57, 0x14, 0xfb, 0x57, 0xd8, 0x63, 0x58,
	0x0a, 0x03, 0x12, 0x0e, 0xd3, 0x37, 0x0b, 0xbf,
	0x67, 0xb7, 0x9d, 0xb7, 0x6b, 0x38, 0xeb, 0x17,
	0xd8, 0xb9, 0x5a, 0x37, 0x9f, 0x98, 0xa6, 0xca,
	0x7e, 0x95, 0xa7, 0x27, 0xc4, 0xd3, 0x15, 0x00,
	0x7b, 0x5e, 0x05, 0xc0, 0xc1, 0xb0, 0xe0, 0x13,
	0x7d, 0x91, 0xed, 0x2b, 0x99, 0x74, 0x1c, 0x16,
	0x45, 0x55, 0x21, 0xbc, 0x7c, 0x52, 0x87, 0x0f,
	0xb9, 0xbf, 0x71, 0x7c, 0x3a, 0x81, 0x72, 0x97,
	0xf8, 0x86, 0x61, 0x20, 0x17, 0xd8, 0xc8, 0xe0,
	0xfc, 0x42, 0x0f, 0x5b, 0xd6, 0x7e, 0x99, 0x81,
	0x5c, 0x2e, 0x2e, 0x3e, 0xe8, 0xce, 0x12, 0xcf,
	0x2c, 0xe6, 0x7a, 0x00, 0x5d, 0x36, 0x00, 0x92,
	0x60, 0xc5, 0xc0, 0xfd, 0xe0, 0xa3, 0xb9, 0x3e,
	0x92, 0xf8, 0x8f, 0xe2, 0x0f, 0xe5, 0xb4, 0x6a,
	0xd6, 0x5b, 0xa4, 0x5d, 0xf9, 0xef, 0x7e, 0xae,
	0xdd, 0xd0, 0x5d, 0x40, 0xfe, 0xa7, 0xed, 0xda,
	0xa9, 0x48, 0x1d, 0x6f, 0xc2, 0xd3, 0x35, 0x65,
	0xd8, 0x67, 0xc2, 0x9d, 0xed, 0xf7, 0x9f, 0x7b,
	0x7c, 0xd4, 0x03, 0xe0, 0xa6, 0xf9, 0x3c, 0xd0,
	0x21, 0x98, 0x60, 0xa6, 0x59, 0x86, 0xbd, 0x40,
	0x17, 0x47, 0x82, 0xb9, 0xe1, 0x11, 0x8d, 0x4b,
	0xcd, 0x1f, 0x54, 0x96, 0x17, 0x42, 0x22, 0x44,
};

/*
 * IV method encblkno1, blkno 2.
 */
static const uint8_t aes_cbc_256_encblkno1_vec2_ctxt[SECSIZE] = {
	0x1d, 0x65, 0xb2, 0x4e, 0xfa, 0x3f, 0xdb, 0xab,
	0x34, 0x9d, 0x37, 0x03, 0x17, 0x44, 0xed, 0x5b,
	0xf7, 0x1b, 0x6b, 0xc0, 0x5c, 0xfe, 0x5b, 0xcd,
	0xf0, 0xaf, 0x26, 0x82, 0x97, 0x12, 0xb8, 0x4f,
	0x76, 0x3d, 0x07, 0xd8, 0x29, 0x56, 0x3c, 0xbd,
	0x0e, 0xac, 0xd1, 0x8f, 0x53, 0x1a, 0x8f, 0xcd,
	0x04, 0x5b, 0x49, 0xe0, 0xf0, 0xea, 0xc9, 0x8a,
	0x08, 0x3d, 0x1f, 0x2d, 0x8c, 0xec, 0xb8, 0xea,
	0xe9, 0x24, 0xd1, 0x93, 0xd7, 0x9a, 0x0f, 0xd7,
	0x0f, 0x6b, 0xa0, 0x08, 0x58, 0x81, 0x68, 0x2f,
	0xde, 0x36, 0xb5, 0x87, 0xd9, 0xcd, 0x82, 0x13,
	0x36, 0x16, 0x6a, 0x9a, 0x02, 0xca, 0xda, 0x6f,
	0x51, 0x87, 0x75, 0x47, 0x89, 0xa4, 0x10, 0x60,
	0xfb, 0x1a, 0x74, 0x55, 0x6d, 0x18, 0x8d, 0x42,
	0x74, 0x2d, 0x12, 0x56, 0xc0, 0xcd, 0xa2, 0x57,
	0x53, 0x31, 0x8c, 0x7a, 0x8b, 0xa8, 0x6d, 0x89,
	0x81, 0xaf, 0x9c, 0xd9, 0x56, 0xe6, 0xdc, 0xe7,
	0x84, 0x0f, 0x81, 0x56, 0x1a, 0xc8, 0x5d, 0xa3,
	0xe0, 0x93, 0xea, 0x62, 0x7d, 0xa4, 0x5a, 0x58,
	0x8f, 0x05, 0x85, 0x34, 0x0c, 0x74, 0x8e, 0xe7,
	0xb4, 0x47, 0x93, 0x61, 0xbf, 0x61, 0x0a, 0xa2,
	0x37, 0xcd, 0x82, 0x9d, 0x55, 0x9e, 0x32, 0x9e,
	0x30, 0xce, 0x61, 0x89, 0xed, 0x25, 0x9e, 0x7c,
	0x2a, 0xcd, 0x39, 0x45, 0x56, 0xbb, 0x1a, 0xe8,
	0xb0, 0x75, 0x8f, 0xa1, 0x59, 0x09, 0xf8, 0x7a,
	0xbd, 0x4f, 0x69, 0x8b, 0xe2, 0xf3, 0xbe, 0x9b,
	0xea, 0x5f, 0x2c, 0x1e, 0x84, 0x69, 0xb2, 0xfa,
	0xaf, 0x1d, 0xc8, 0xcf, 0x76, 0x91, 0xd0, 0x7a,
	0xc9, 0xd1, 0x3d, 0xa5, 0xae, 0xae, 0xd7, 0x23,
	0xbb, 0xb3, 0x5e, 0x8a, 0x10, 0xc6, 0xbe, 0xa6,
	0x79, 0x69, 0x40, 0x83, 0x81, 0xe6, 0xb1, 0xa3,
	0x7e, 0x57, 0x44, 0x66, 0xc9, 0x2e, 0x84, 0xdd,
	0x00, 0xb4, 0x93, 0xae, 0x8f, 0x23, 0x12, 0xd6,
	0x54, 0x56, 0xc3, 0x51, 0xe5, 0xf7, 0x69, 0x47,
	0x00, 0x97, 0x71, 0x29, 0xcb, 0xcf, 0xeb, 0xd9,
	0xaf, 0xc0, 0x2f, 0x5c, 0xd7, 0x3e, 0xe4, 0x07,
	0xc9, 0x65, 0x2e, 0x8c, 0xf4, 0x54, 0xce, 0x8b,
	0xc7, 0x0c, 0xb4, 0x74, 0x56, 0x79, 0xa6, 0x40,
	0x4a, 0x58, 0xfd, 0x3f, 0x7b, 0x4c, 0xe9, 0xdb,
	0x33, 0x85, 0x6f, 0xf7, 0x5a, 0x9f, 0x6f, 0xc4,
	0x60, 0xad, 0x1b, 0xe2, 0xf5, 0xeb, 0x42, 0x7d,
	0xa4, 0x43, 0x8d, 0x40, 0xfa, 0x53, 0xcc, 0xf0,
	0x5f, 0x90, 0x0d, 0x04, 0x51, 0xb1, 0x48, 0xc7,
	0x90, 0x65, 0xb2, 0xef, 0xca, 0xc5, 0x9a, 0xec,
	0xde, 0x84, 0x21, 0x22, 0xeb, 0x97, 0xdd, 0xa2,
	0x9d, 0x71, 0xb1, 0xe0, 0x86, 0x58, 0xc3, 0x57,
	0xd5, 0x5a, 0x6f, 0xdc, 0xe5, 0xcc, 0x64, 0xc7,
	0x80, 0x2a, 0xef, 0x90, 0x91, 0x96, 0xb4, 0xeb,
	0xda, 0x3b, 0x7b, 0xbc, 0x14, 0x60, 0x52, 0xe5,
	0xe5, 0xc8, 0x6a, 0x99, 0x46, 0x9d, 0x00, 0x77,
	0x3b, 0x60, 0x75, 0x04, 0x06, 0x4a, 0x5a, 0x64,
	0x6f, 0x2f, 0x58, 0x77, 0x27, 0x9a, 0xc5, 0x90,
	0x37, 0xa7, 0xf3, 0x89, 0x72, 0x47, 0x4e, 0x08,
	0xa5, 0x79, 0x11, 0x2f, 0x22, 0x5a, 0xbb, 0xcf,
	0x76, 0xb9, 0x28, 0xc8, 0xc4, 0x5a, 0xe5, 0x90,
	0xf7, 0x02, 0xe4, 0xf9, 0x0c, 0x4c, 0x9a, 0xb1,
	0xa7, 0x99, 0x34, 0xd4, 0x77, 0x66, 0xff, 0x3c,
	0x50, 0x9a, 0xff, 0x13, 0x49, 0xd6, 0x5a, 0xf6,
	0x17, 0x6f, 0xca, 0x1a, 0xef, 0x0a, 0x2d, 0xf1,
	0xdf, 0xd0, 0xa5, 0x6f, 0xa6, 0x22, 0x3c, 0x1f,
	0xcf, 0xe7, 0xec, 0x23, 0x39, 0x6e, 0xc0, 0x37,
	0x31, 0x84, 0xff, 0xe2, 0x5a, 0xd6, 0x88, 0x74,
	0x15, 0x15, 0x46, 0x21, 0x00, 0xe4, 0x13, 0x9a,
	0xfa, 0xb2, 0x49, 0x7e, 0x79, 0xfb, 0x8a, 0x2a,
};

/*
 * IV method encblkno1, blkno 3.
 */
static const uint8_t aes_cbc_256_encblkno1_vec3_ctxt[SECSIZE] = {
	0xc1, 0x4a, 0x3c, 0x90, 0xba, 0xd4, 0x9c, 0xe7,
	0xf2, 0x5b, 0x3a, 0xc4, 0xce, 0x4a, 0x26, 0x4a,
	0x9c, 0x3f, 0xe5, 0x7a, 0x15, 0xbb, 0xbd, 0x3e,
	0xc6, 0x49, 0x47, 0x04, 0x4e, 0x8b, 0x73, 0xa6,
	0x02, 0x3a, 0xc1, 0xa3, 0xfa, 0x1a, 0xd0, 0x03,
	0xf7, 0x26, 0x9f, 0xad, 0x06, 0x8f, 0x86, 0xdc,
	0xb8, 0x73, 0x87, 0xa2, 0x82, 0xc6, 0x80, 0xe1,
	0xac, 0x3d, 0x16, 0x4c, 0xc3, 0x7c, 0x86, 0x01,
	0xa5, 0x7a, 0x1c, 0x4b, 0x56, 0x68, 0xf6, 0x06,
	0x99, 0x32, 0x42, 0x40, 0xf1, 0xb7, 0x44, 0x4b,
	0xd1, 0xdb, 0xad, 0x4e, 0xa6, 0xc2, 0x5f, 0xee,
	0x21, 0x1d, 0x58, 0xc6, 0xd5, 0x02, 0xef, 0xb2,
	0x38, 0xef, 0x29, 0x25, 0xfd, 0x28, 0x8a, 0x5b,
	0x8b, 0x36, 0x1a, 0xd6, 0x68, 0xf8, 0x77, 0xed,
	0xba, 0xb3, 0xa5, 0x6f, 0x76, 0x5e, 0xb5, 0xd4,
	0xc3, 0xb8, 0xf9, 0x67, 0x7a, 0x18, 0x43, 0xb6,
	0x65, 0x07, 0x48, 0x1d, 0x56, 0x20, 0x11, 0xe1,
	0x62, 0x6b, 0x70, 0xc9, 0x18, 0xa9, 0xa7, 0x36,
	0xbf, 0x31, 0x74, 0xe3, 0x33, 0x02, 0x96, 0x7a,
	0xf5, 0xd9, 0x8d, 0x05, 0x2b, 0xfd, 0x85, 0x4f,
	0x03, 0x0f, 0xe1, 0xfb, 0x1a, 0x57, 0xaf, 0xdc,
	0xff, 0xff, 0x5a, 0x96, 0x27, 0xca, 0xf3, 0x0c,
	0xd8, 0x39, 0x3e, 0xbe, 0x41, 0x5a, 0x21, 0x95,
	0x66, 0xe0, 0x69, 0x14, 0x2b, 0x18, 0xf2, 0x9b,
	0xf4, 0x22, 0xdf, 0xa9, 0xe4, 0x7d, 0x32, 0x5d,
	0x98, 0xa0, 0xe0, 0xe1, 0xe1, 0xbe, 0xae, 0x58,
	0x63, 0xbe, 0x4b, 0x97, 0x83, 0xaa, 0x67, 0xd3,
	0x1a, 0x70, 0xca, 0x82, 0x98, 0x77, 0x74, 0x1a,
	0xf2, 0x3d, 0x6a, 0x7b, 0x8e, 0xc8, 0xca, 0x34,
	0x44, 0xb8, 0xc0, 0xd0, 0x77, 0x8c, 0x4a, 0x5c,
	0xae, 0xd3, 0x17, 0x7c, 0x12, 0x47, 0x3e, 0xe2,
	0x2e, 0x51, 0xe0, 0x52, 0x88, 0x8e, 0xe9, 0x68,
	0xb6, 0x13, 0xf8, 0x69, 0xc9, 0x4b, 0xdd, 0x91,
	0x27, 0xb0, 0x22, 0x0c, 0x7d, 0xad, 0xb0, 0x75,
	0xe8, 0x76, 0x34, 0xc2, 0xd9, 0xf3, 0x20, 0xf7,
	0x1d, 0x0f, 0x07, 0x61, 0xc2, 0xb8, 0x7d, 0x00,
	0xa6, 0x68, 0xad, 0xd4, 0x0b, 0xa4, 0xa0, 0x32,
	0x6d, 0xa5, 0xc0, 0x07, 0x65, 0xae, 0xda, 0x2e,
	0xd6, 0xb7, 0xd3, 0x99, 0x8b, 0x37, 0x08, 0x13,
	0x6a, 0x94, 0xe9, 0xe4, 0xea, 0x34, 0xee, 0x07,
	0xee, 0x77, 0xb1, 0x3f, 0x54, 0x05, 0xbe, 0x66,
	0x7f, 0xf2, 0x70, 0x34, 0x45, 0xa7, 0x4b, 0x27,
	0xef, 0xe6, 0x39, 0x2e, 0x13, 0x41, 0xdb, 0x2d,
	0x1f, 0x76, 0x11, 0x76, 0x33, 0xf3, 0x92, 0x33,
	0x69, 0x16, 0x34, 0x86, 0x23, 0xc5, 0xfa, 0xaf,
	0xff, 0xbf, 0xee, 0x84, 0x56, 0xf6, 0x1e, 0x54,
	0x37, 0x32, 0x79, 0x83, 0x45, 0x04, 0x6f, 0x0e,
	0x75, 0x75, 0xd9, 0xd6, 0x4a, 0x87, 0xfb, 0x3c,
	0xb1, 0xcf, 0x66, 0xab, 0xa4, 0xaa, 0xf6, 0x96,
	0xb0, 0xcd, 0xaf, 0xac, 0x2c, 0x6d, 0x72, 0xca,
	0x43, 0xef, 0xb7, 0xa0, 0x4c, 0x62, 0xba, 0x7e,
	0x59, 0x0b, 0xff, 0x90, 0x49, 0x63, 0xf6, 0x31,
	0x8b, 0x50, 0x20, 0x82, 0x7d, 0xf0, 0x2d, 0xe4,
	0x5b, 0xda, 0xdf, 0xb0, 0xfb, 0x07, 0x7b, 0xe3,
	0x5f, 0xac, 0xd8, 0xe5, 0xa0, 0x3e, 0xc5, 0x60,
	0x94, 0xbc, 0xf7, 0x7e, 0xdc, 0x18, 0x27, 0x20,
	0xb1, 0xdd, 0x51, 0x4a, 0xb2, 0xe0, 0xc0, 0xe7,
	0x5d, 0x0f, 0x88, 0xb2, 0xa0, 0x42, 0x73, 0xfb,
	0xc4, 0x24, 0xa7, 0x17, 0x8a, 0xc9, 0x6d, 0x34,
	0xe8, 0x7b, 0x51, 0x37, 0x32, 0x3f, 0xf8, 0x7e,
	0x92, 0xe4, 0x87, 0xd2, 0x89, 0x66, 0xb0, 0xf6,
	0xc2, 0xba, 0x2f, 0x42, 0x8f, 0x1d, 0x5d, 0x81,
	0xad, 0xfd, 0x00, 0xbc, 0xa9, 0x11, 0x96, 0xae,
	0x80, 0xf1, 0x27, 0xe0, 0xeb, 0x5b, 0x60, 0x39,
};

const struct testvec aes_cbc_256_1_vectors[] = {
	{
		.blkno = 0,
		.ptxt = aes_cbc_ptxt,
		.ctxt = aes_cbc_256_encblkno1_vec0_ctxt,
	},
	{
		.blkno = 1,
		.ptxt = aes_cbc_ptxt,
		.ctxt = aes_cbc_256_encblkno1_vec1_ctxt,
	},
	{
		.blkno = 2,
		.ptxt = aes_cbc_ptxt,
		.ctxt = aes_cbc_256_encblkno1_vec2_ctxt,
	},
	{
		.blkno = 3,
		.ptxt = aes_cbc_ptxt,
		.ctxt = aes_cbc_256_encblkno1_vec3_ctxt,
	},
};

/*
 * IV method encblkno8, blkno 0.
 */
static const uint8_t aes_cbc_256_encblkno8_vec0_ctxt[SECSIZE] = {
	0xe5, 0x55, 0xd9, 0xae, 0xc0, 0x66, 0x2d, 0x2f,
	0x11, 0xb1, 0x27, 0xd2, 0xb2, 0x73, 0xe4, 0x0a,
	0x85, 0xb5, 0x3c, 0x79, 0x78, 0xd6, 0x35, 0x3d,
	0x46, 0xac, 0xa3, 0x81, 0x55, 0x76, 0x86, 0xfc,
	0x37, 0xa0, 0x95, 0xc2, 0x30, 0xc9, 0x19, 0xc2,
	0x5f, 0xb0, 0x13, 0xa2, 0xa8, 0xe1, 0xc7, 0xb7,
	0x61, 0x54, 0xd8, 0xe6, 0xca, 0x94, 0x6f, 0x47,
	0x87, 0x33, 0x58, 0xd9, 0xd5, 0xd2, 0x95, 0x73,
	0x87, 0x9a, 0x31, 0xe5, 0x2e, 0x95, 0x83, 0x7d,
	0xdc, 0x0f, 0xc5, 0x2f, 0x14, 0xbc, 0x80, 0xac,
	0x47, 0xd6, 0xd8, 0x17, 0x9e, 0xf7, 0xff, 0x5b,
	0x85, 0x05, 0x91, 0xe0, 0x73, 0xd2, 0x5c, 0xa7,
	0x41, 0xf8, 0xcb, 0x3d, 0x38, 0x14, 0x28, 0x3e,
	0x89, 0x6f, 0xd4, 0xac, 0xb6, 0x11, 0x35, 0x67,
	0x7b, 0xef, 0x0d, 0xd8, 0x4d, 0xdd, 0x7e, 0x73,
	0xcd, 0x58, 0x0f, 0x5a, 0xcf, 0x42, 0xc5, 0x2f,
	0x61, 0x62, 0x13, 0xde, 0xcd, 0x2e, 0x22, 0xab,
	0xb0, 0x47, 0x5c, 0x1e, 0x5c, 0xc5, 0x49, 0xc6,
	0x3b, 0x0c, 0xe3, 0xb3, 0x59, 0xbf, 0xbf, 0x85,
	0xf6, 0x0a, 0x3d, 0x14, 0x74, 0x2a, 0xcd, 0xea,
	0x67, 0xd6, 0x80, 0x42, 0x3c, 0x6a, 0x92, 0x50,
	0x95, 0x73, 0xb5, 0x7a, 0xb2, 0xbc, 0x76, 0xe5,
	0x8f, 0xf3, 0x85, 0x5e, 0xcd, 0xf9, 0xb4, 0x9d,
	0xa8, 0x0a, 0xda, 0x95, 0x11, 0x2e, 0x22, 0x0c,
	0x2f, 0xb0, 0xbf, 0x92, 0x6b, 0x45, 0xec, 0x20,
	0xd2, 0x2b, 0x98, 0x3f, 0x4f, 0x97, 0xf2, 0xed,
	0xf7, 0x9b, 0x89, 0x4e, 0xd6, 0x59, 0xbb, 0x24,
	0x22, 0x44, 0x9f, 0x03, 0xb5, 0x42, 0xc8, 0x97,
	0xc7, 0xdb, 0x21, 0xfc, 0xcf, 0x33, 0xa1, 0xf1,
	0xc0, 0x1f, 0x28, 0x77, 0xee, 0xa5, 0x6a, 0x12,
	0xef, 0x8b, 0x48, 0xd1, 0xb3, 0xac, 0x65, 0x69,
	0x46, 0x04, 0x39, 0xb1, 0x9e, 0xfa, 0xab, 0x21,
	0x51, 0xa4, 0x33, 0xe9, 0x58, 0x5d, 0xf1, 0xc6,
	0x69, 0x44, 0x8c, 0x17, 0xf9, 0xaa, 0x96, 0xcb,
	0x40, 0xb4, 0x5c, 0x83, 0x76, 0x1e, 0x8a, 0x2b,
	0x5f, 0x6b, 0xc1, 0x73, 0xd4, 0x5f, 0x48, 0xa3,
	0x0e, 0x07, 0x69, 0x12, 0xc1, 0xbd, 0x13, 0xad,
	0xe2, 0xcf, 0x3d, 0x96, 0xd8, 0xaf, 0xed, 0xdc,
	0x4c, 0x72, 0xf6, 0xce, 0x15, 0x86, 0x88, 0x8c,
	0xbb, 0x60, 0xb3, 0xb9, 0xde, 0x42, 0x58, 0x6e,
	0xc4, 0x58, 0xac, 0x77, 0x8d, 0x35, 0x23, 0x5f,
	0xc3, 0xf9, 0x33, 0x46, 0x17, 0x80, 0x31, 0xfd,
	0xcd, 0x0a, 0x1e, 0x9b, 0xac, 0x42, 0xda, 0x70,
	0x54, 0x9a, 0xeb, 0x22, 0x27, 0x09, 0x0c, 0x6c,
	0x18, 0x1d, 0x1a, 0x5b, 0x86, 0x4d, 0x80, 0xca,
	0x4d, 0xda, 0x0e, 0x9a, 0x8e, 0x61, 0x04, 0x68,
	0x29, 0x08, 0x3b, 0xae, 0x14, 0x7d, 0x8e, 0x43,
	0x7a, 0xa7, 0x83, 0xcf, 0xb3, 0x95, 0xd3, 0x42,
	0x2d, 0x6b, 0xd8, 0x5c, 0x43, 0x31, 0x5b, 0x9c,
	0x18, 0x30, 0x0d, 0x61, 0x9c, 0xab, 0x29, 0x55,
	0xdd, 0x84, 0x24, 0x21, 0xec, 0x44, 0xad, 0xf3,
	0xb3, 0x70, 0x77, 0x2b, 0xfc, 0x3f, 0x99, 0xb8,
	0x50, 0x26, 0x8d, 0x96, 0xa2, 0x22, 0x99, 0x33,
	0x53, 0xa8, 0x5d, 0x84, 0x9b, 0x76, 0x26, 0x6e,
	0x75, 0x14, 0x7e, 0x63, 0xc6, 0x7a, 0x4f, 0x5c,
	0xfe, 0x4b, 0x80, 0xee, 0xb3, 0x32, 0x8d, 0x25,
	0x1c, 0x80, 0x7b, 0x3d, 0xd3, 0x84, 0x01, 0x1e,
	0x16, 0xa4, 0xca, 0x0d, 0x38, 0x40, 0x03, 0x6f,
	0x81, 0x8b, 0x5c, 0xad, 0x22, 0xfa, 0x6f, 0xeb,
	0x60, 0xa1, 0xcb, 0x2d, 0x97, 0xf8, 0xa6, 0x5e,
	0xbe, 0x93, 0xb7, 0xe6, 0x66, 0xbf, 0x9b, 0xd2,
	0x5c, 0x31, 0xcc, 0x70, 0x0c, 0xf1, 0xfb, 0x9f,
	0x09, 0x1b, 0xc2, 0x85, 0x2f, 0x22, 0x7c, 0x95,
	0x58, 0x09, 0xce, 0x9c, 0x7c, 0x50, 0xca, 0x89,
};

/*
 * IV method encblkno8, blkno 1.
 */
static const uint8_t aes_cbc_256_encblkno8_vec1_ctxt[SECSIZE] = {
	0x37, 0x4d, 0x25, 0xdb, 0x35, 0xe0, 0x8b, 0x82,
	0x5f, 0x77, 0xd2, 0x53, 0xd1, 0x1f, 0xf0, 0x91,
	0x5b, 0xd8, 0x93, 0x2f, 0xb2, 0x81, 0xbd, 0x75,
	0xf0, 0xd8, 0xff, 0x46, 0x8c, 0x9d, 0xf6, 0xe2,
	0x99, 0x1e, 0x76, 0x9d, 0x00, 0x3a, 0xe3, 0xcf,
	0x6d, 0x24, 0xa8, 0xe8, 0xb4, 0xa7, 0xa0, 0x87,
	0xa8, 0x43, 0x01, 0x21, 0x29, 0x70, 0x39, 0x2d,
	0x0b, 0x2a, 0xe9, 0x11, 0x57, 0x86, 0x13, 0xd4,
	0x1c, 0x1b, 0x59, 0x19, 0xc4, 0x7d, 0x2c, 0x94,
	0xc7, 0x01, 0xb8, 0x96, 0x01, 0xd2, 0x01, 0x17,
	0x97, 0x41, 0x68, 0xab, 0xba, 0x9c, 0xc8, 0xad,
	0x4e, 0xd7, 0xa8, 0x4c, 0x96, 0x3f, 0xf9, 0xfc,
	0x7e, 0xd7, 0x59, 0xe8, 0x25, 0x51, 0x4d, 0x1d,
	0x99, 0xfd, 0x0b, 0xe9, 0x88, 0x23, 0xd1, 0x4b,
	0x30, 0x6e, 0x18, 0x7c, 0xe3, 0x7a, 0x54, 0x2a,
	0x4f, 0x2a, 0x99, 0x8f, 0xaf, 0xd7, 0x5e, 0x25,
	0xfe, 0x9c, 0x47, 0x09, 0x63, 0x38, 0x0d, 0x5f,
	0xb3, 0x43, 0xa6, 0x66, 0x9b, 0xc5, 0x3d, 0x88,
	0x5e, 0xc7, 0x60, 0x99, 0x8e, 0xcb, 0x6a, 0x65,
	0x60, 0x92, 0x88, 0xe1, 0x2b, 0xfe, 0x83, 0x34,
	0x92, 0xa6, 0x6c, 0x22, 0x56, 0x5b, 0x75, 0x8a,
	0x93, 0xc3, 0x72, 0xca, 0xff, 0x59, 0x3b, 0xd8,
	0xa0, 0x80, 0x56, 0x98, 0x62, 0x8a, 0x70, 0xf2,
	0x5d, 0xd9, 0x40, 0x6b, 0xbf, 0x9f, 0x71, 0x8d,
	0x2e, 0x38, 0xe8, 0x06, 0x42, 0xa9, 0x95, 0x70,
	0x31, 0xd1, 0xe9, 0x6c, 0xab, 0xbb, 0xed, 0x25,
	0xe8, 0xca, 0xe8, 0xa4, 0x98, 0x82, 0xf5, 0xe3,
	0x11, 0x3c, 0xc4, 0xea, 0xea, 0x88, 0x56, 0x91,
	0xd6, 0x5d, 0xaa, 0xf7, 0xe9, 0x49, 0x2f, 0x42,
	0x5b, 0x76, 0xef, 0xed, 0x03, 0x9e, 0x5f, 0x4d,
	0x65, 0x25, 0xa5, 0xe8, 0x26, 0xba, 0x03, 0x4f,
	0x0e, 0x39, 0xd2, 0x53, 0x62, 0x98, 0x81, 0x9d,
	0x8e, 0xad, 0x50, 0x17, 0x9f, 0xcc, 0x34, 0x45,
	0x19, 0x5c, 0x1c, 0xd1, 0xbc, 0x71, 0x89, 0xaa,
	0x9a, 0x65, 0x55, 0x6f, 0x78, 0xd4, 0xd3, 0x5b,
	0x27, 0x8d, 0x94, 0x46, 0xd9, 0x95, 0xb3, 0x5f,
	0xc4, 0x35, 0x8d, 0xba, 0x1c, 0x40, 0x52, 0xd1,
	0x99, 0x27, 0x5d, 0x42, 0x28, 0xef, 0xcb, 0x9b,
	0x10, 0x7a, 0x19, 0xbf, 0x72, 0xa3, 0x4a, 0xb9,
	0x56, 0x83, 0x39, 0xa6, 0xb2, 0xcd, 0x48, 0x85,
	0xf9, 0xcc, 0x72, 0x88, 0xb3, 0x5a, 0x9b, 0x45,
	0xb2, 0xd3, 0x66, 0x2d, 0x24, 0x51, 0x68, 0x91,
	0x9d, 0x47, 0x6a, 0xb3, 0x9a, 0x60, 0xb3, 0xcd,
	0x6b, 0x43, 0x96, 0x21, 0xa0, 0x65, 0x43, 0xde,
	0x4f, 0x6e, 0xb5, 0x81, 0x50, 0x7e, 0xca, 0x4b,
	0xdb, 0x30, 0xf2, 0xcb, 0x28, 0x3b, 0x19, 0x6a,
	0x0a, 0xfa, 0x05, 0x5e, 0x61, 0xde, 0x76, 0x7e,
	0xdf, 0xd9, 0xa9, 0x1b, 0xd0, 0x8a, 0xb5, 0x04,
	0x51, 0xf5, 0x66, 0xa2, 0x32, 0x21, 0xb2, 0xa9,
	0x40, 0x78, 0x60, 0x9d, 0xdc, 0x45, 0xbe, 0xb4,
	0x3a, 0xba, 0xd1, 0xec, 0x31, 0x53, 0x24, 0x22,
	0x70, 0x99, 0xda, 0xc8, 0x17, 0x04, 0x87, 0x2c,
	0x89, 0x86, 0x24, 0xec, 0x52, 0x12, 0x6a, 0x51,
	0x1e, 0x2a, 0x5e, 0x96, 0xfb, 0xac, 0x95, 0x4a,
	0x1a, 0x06, 0x8f, 0xdf, 0xa7, 0x26, 0xeb, 0x6c,
	0x79, 0x4a, 0x77, 0xea, 0xb3, 0xb1, 0x3a, 0x19,
	0xe6, 0x5e, 0xe2, 0x26, 0x1b, 0x85, 0x3c, 0x9b,
	0x1d, 0x05, 0x1d, 0xbe, 0x5c, 0x25, 0x7f, 0x45,
	0x4c, 0x09, 0x4c, 0xc1, 0x47, 0xa5, 0x44, 0xfc,
	0x04, 0x2b, 0xad, 0x53, 0xac, 0x57, 0x22, 0x54,
	0x13, 0x7c, 0xc9, 0x96, 0x44, 0xda, 0x74, 0x95,
	0x6e, 0x8c, 0xe6, 0x6a, 0x05, 0x05, 0xf3, 0x8c,
	0x81, 0xaf, 0xbc, 0xb1, 0x91, 0xe7, 0xfd, 0x81,
	0x3f, 0x47, 0xc2, 0x6f, 0x0d, 0x62, 0xf6, 0x6e,
};

/*
 * IV method encblkno8, blkno 2.
 */
static const uint8_t aes_cbc_256_encblkno8_vec2_ctxt[SECSIZE] = {
	0x5a, 0x24, 0xfd, 0xee, 0x9a, 0x44, 0xfb, 0xac,
	0x3e, 0x46, 0x53, 0x95, 0x9e, 0xeb, 0x1f, 0xd9,
	0xfd, 0xc6, 0x4f, 0xae, 0x0b, 0xc8, 0xf2, 0xbd,
	0x77, 0x16, 0x7a, 0x2e, 0x8e, 0xec, 0x7a, 0x53,
	0xf4, 0xe0, 0x34, 0xba, 0x6e, 0xfa, 0xc4, 0x69,
	0xd7, 0x50, 0x13, 0x03, 0xfb, 0xb9, 0x66, 0x96,
	0xd4, 0x21, 0x67, 0xcc, 0x4c, 0x4d, 0x10, 0x2f,
	0x17, 0xeb, 0x41, 0xf4, 0x65, 0x80, 0x0b, 0x57,
	0x2d, 0xdf, 0xcf, 0x9f, 0xb9, 0xd8, 0x53, 0x36,
	0xbc, 0x1d, 0x9a, 0xe3, 0x17, 0xe7, 0x08, 0x23,
	0xb3, 0x60, 0xfe, 0xdf, 0x24, 0x06, 0xc5, 0x86,
	0x74, 0x89, 0xa3, 0xb2, 0xfc, 0x4a, 0x57, 0xf5,
	0xa6, 0x96, 0xfb, 0x56, 0xf0, 0xf4, 0xdc, 0xdc,
	0xb8, 0x53, 0x5f, 0xb2, 0xb0, 0x8d, 0x2d, 0x90,
	0x3d, 0x78, 0x4d, 0x42, 0x3a, 0x74, 0xa4, 0x8e,
	0x41, 0x7c, 0x2a, 0xff, 0xe4, 0x57, 0x1c, 0x9d,
	0x94, 0xc5, 0x5d, 0xd8, 0x8b, 0x88, 0x48, 0x15,
	0x16, 0x8a, 0xf3, 0x07, 0x3a, 0xee, 0x65, 0x24,
	0xbc, 0x7f, 0x58, 0xad, 0xed, 0xf2, 0xbd, 0x18,
	0x88, 0x1a, 0x80, 0x6e, 0xb7, 0x93, 0xe0, 0x45,
	0x04, 0xb0, 0xfc, 0xf9, 0x48, 0x76, 0xaf, 0xec,
	0x08, 0xca, 0x99, 0x64, 0x85, 0x98, 0x2c, 0xd8,
	0x85, 0x72, 0x32, 0xbe, 0x92, 0x18, 0xdd, 0xab,
	0x20, 0x8f, 0x8e, 0x11, 0xc6, 0x08, 0xf9, 0x8b,
	0xaf, 0x5f, 0xa9, 0xe5, 0x11, 0xc7, 0x45, 0x91,
	0x6e, 0x47, 0xaa, 0x0f, 0x4c, 0xf4, 0xc1, 0xb0,
	0x75, 0x4c, 0xba, 0x1d, 0xb3, 0x33, 0xf7, 0x47,
	0xbe, 0x94, 0x0b, 0x2e, 0xfa, 0x38, 0x5e, 0x5f,
	0x0a, 0xc2, 0x0c, 0x4e, 0x72, 0x29, 0x16, 0xc1,
	0x82, 0x70, 0xd4, 0xd3, 0x1b, 0x25, 0xbe, 0x0d,
	0x6b, 0x0a, 0x13, 0x9f, 0x4d, 0x3d, 0x7b, 0x10,
	0x9f, 0x93, 0x43, 0x50, 0xd1, 0x17, 0x08, 0x77,
	0x23, 0x58, 0x35, 0x41, 0x23, 0xf6, 0x9c, 0x6f,
	0x2e, 0x81, 0x6e, 0x75, 0x9b, 0x9f, 0x37, 0x4f,
	0xb7, 0xa1, 0xce, 0xde, 0x0c, 0x74, 0x99, 0x31,
	0x0e, 0x27, 0x42, 0x99, 0xdd, 0x93, 0x03, 0x6b,
	0x44, 0x22, 0xd4, 0xc8, 0x67, 0xb5, 0xb2, 0x4d,
	0x11, 0x2e, 0x80, 0x09, 0xa2, 0x5b, 0xcf, 0x0c,
	0xff, 0xfa, 0x51, 0xe5, 0x9b, 0xdd, 0x11, 0xa1,
	0x17, 0x04, 0x9e, 0xc8, 0xd8, 0x1d, 0xc1, 0x5c,
	0xc3, 0xde, 0x83, 0x77, 0xa3, 0xec, 0x59, 0x7e,
	0xfb, 0xe8, 0x45, 0xff, 0xc3, 0xb3, 0xd3, 0x9e,
	0x3e, 0xc4, 0x75, 0xca, 0xc1, 0x77, 0xee, 0x1a,
	0xdc, 0x58, 0xab, 0x27, 0xc1, 0xfe, 0x21, 0x26,
	0x9a, 0xe0, 0x15, 0xab, 0x35, 0x8d, 0xbc, 0x22,
	0x37, 0xbb, 0x4e, 0xab, 0x9d, 0xa2, 0xaf, 0xf9,
	0x45, 0x17, 0xb1, 0xb8, 0xd4, 0x52, 0x1e, 0x67,
	0xeb, 0xac, 0xe0, 0x87, 0x6c, 0xe4, 0x7a, 0x03,
	0x73, 0xe4, 0x43, 0xeb, 0x3b, 0x57, 0x3f, 0x56,
	0x4b, 0x6c, 0x26, 0x81, 0x27, 0xbf, 0x7e, 0x59,
	0xcd, 0xab, 0x67, 0x8c, 0x4b, 0x6f, 0xa5, 0x47,
	0x2c, 0x45, 0x28, 0x5a, 0x3d, 0x88, 0x53, 0xf9,
	0x25, 0xdf, 0x5d, 0xba, 0xf7, 0x18, 0xa7, 0x3d,
	0x79, 0xb4, 0x43, 0x59, 0x77, 0xf9, 0xd5, 0x5d,
	0x4f, 0x31, 0x56, 0x8c, 0x21, 0xb5, 0xc0, 0xa2,
	0xca, 0x04, 0x62, 0x2c, 0xc8, 0xa8, 0x11, 0x82,
	0x1b, 0xde, 0xad, 0x20, 0x5b, 0xd2, 0x63, 0xfb,
	0x6d, 0xba, 0xd4, 0xcc, 0xb4, 0x9d, 0xe8, 0xa8,
	0xd1, 0x06, 0x81, 0xf0, 0xb9, 0xd4, 0x90, 0x30,
	0xcd, 0x0a, 0xe8, 0xd2, 0x8c, 0x7a, 0xbf, 0xf6,
	0x0d, 0xa0, 0xae, 0x1b, 0x21, 0x18, 0x93, 0x18,
	0x71, 0xe1, 0xa0, 0x63, 0x5a, 0x9d, 0x4e, 0x6a,
	0x52, 0x90, 0xaf, 0xdb, 0x26, 0x1e, 0xa9, 0xa1,
	0xc7, 0xf9, 0xf8, 0xa7, 0x3f, 0x85, 0xa1, 0xa4,
};

/*
 * IV method encblkno8, blkno 3.
 */
static const uint8_t aes_cbc_256_encblkno8_vec3_ctxt[SECSIZE] = {
	0x83, 0x77, 0xd8, 0xa8, 0x6a, 0x36, 0x41, 0x72,
	0xb6, 0x03, 0x4e, 0x5e, 0x39, 0x36, 0xe3, 0xf5,
	0xd0, 0x1b, 0x0d, 0x97, 0x78, 0x46, 0xee, 0xfd,
	0x34, 0x34, 0x16, 0xa0, 0x44, 0xcf, 0x0b, 0xdc,
	0xfb, 0x82, 0x98, 0xa2, 0x79, 0xc2, 0xe7, 0x1c,
	0x43, 0x43, 0x4c, 0x7f, 0xe7, 0xa6, 0xe6, 0x10,
	0x9e, 0x65, 0xb2, 0x09, 0xc7, 0x5f, 0xaa, 0xb7,
	0xb8, 0xad, 0x83, 0xd5, 0x9e, 0xd1, 0xb2, 0xce,
	0x4b, 0xa4, 0x5d, 0xbc, 0xd6, 0xf6, 0x0a, 0xe7,
	0x1b, 0xe9, 0x86, 0xbc, 0x72, 0xcc, 0x6f, 0xcc,
	0xf2, 0xde, 0x08, 0x48, 0xa2, 0x20, 0x31, 0x6a,
	0xdd, 0xbe, 0xc5, 0x36, 0x55, 0xff, 0xce, 0xfa,
	0xdf, 0x60, 0x26, 0x77, 0x7f, 0xd0, 0xfa, 0xd7,
	0x76, 0x70, 0x14, 0x11, 0xbf, 0x57, 0xc2, 0xdd,
	0x5f, 0xd3, 0x50, 0x49, 0xf8, 0xd1, 0xa7, 0xe2,
	0x8b, 0x89, 0xa0, 0xbc, 0x44, 0x42, 0x45, 0x10,
	0xfe, 0x66, 0x3d, 0x56, 0x09, 0x21, 0x7c, 0x49,
	0x30, 0xde, 0xe2, 0x4b, 0x26, 0x65, 0x8a, 0xe4,
	0x79, 0x08, 0x3a, 0xca, 0x36, 0x4f, 0x97, 0x3c,
	0xe4, 0x6a, 0xc3, 0xdb, 0xce, 0xac, 0x78, 0x76,
	0x25, 0x81, 0x7a, 0x01, 0x7b, 0xd8, 0xf1, 0x00,
	0x8d, 0x2e, 0xb7, 0x98, 0x3c, 0x86, 0x20, 0xa3,
	0x4c, 0x24, 0x2a, 0x78, 0x3a, 0x8d, 0xeb, 0xcd,
	0xae, 0xe1, 0x32, 0xf8, 0x21, 0x37, 0x30, 0x27,
	0xe1, 0xf3, 0x14, 0x60, 0x96, 0x77, 0x37, 0x50,
	0xa2, 0x92, 0xae, 0xe5, 0xd8, 0xea, 0x1a, 0x7e,
	0xa3, 0xd1, 0x04, 0x17, 0x03, 0x51, 0x2f, 0x21,
	0xa7, 0x00, 0x23, 0xb3, 0x24, 0xd8, 0x7d, 0xb7,
	0x4c, 0x51, 0xb1, 0xaf, 0xb0, 0x64, 0xe4, 0x62,
	0x91, 0x4c, 0xd5, 0x4b, 0xe8, 0xfb, 0x95, 0x61,
	0xa4, 0x6f, 0xf8, 0xb8, 0xea, 0xa9, 0xb2, 0x10,
	0xd3, 0x96, 0xcb, 0x1c, 0xdc, 0x86, 0x43, 0x26,
	0x2d, 0x39, 0xc2, 0xa7, 0x69, 0xfa, 0x8f, 0x3a,
	0xe7, 0xe0, 0x27, 0xbe, 0xc2, 0xe8, 0xd5, 0x05,
	0xbe, 0x5a, 0x96, 0xdc, 0x86, 0xcd, 0x93, 0x75,
	0x1b, 0x61, 0x40, 0x8c, 0x60, 0x64, 0x79, 0x85,
	0x6c, 0xed, 0x39, 0x72, 0x26, 0x69, 0xba, 0xb2,
	0xff, 0xa8, 0x68, 0x29, 0x03, 0xf7, 0x26, 0xe7,
	0x0f, 0x53, 0x1b, 0x5b, 0x37, 0x21, 0x68, 0x70,
	0x1c, 0x39, 0x7f, 0x5b, 0x31, 0xca, 0xde, 0xed,
	0x33, 0x8d, 0xaf, 0xe6, 0x01, 0xd5, 0x72, 0x5f,
	0x46, 0x44, 0x34, 0x1b, 0x4c, 0xd7, 0x75, 0xf0,
	0x47, 0x16, 0x6c, 0xd6, 0x65, 0x3c, 0xd3, 0xc2,
	0xb1, 0x46, 0x7d, 0xd5, 0x5c, 0x48, 0x5b, 0x61,
	0x3e, 0x88, 0xff, 0x24, 0x5c, 0x7b, 0xf7, 0xa9,
	0x44, 0xcb, 0x3b, 0x3e, 0x3b, 0x93, 0x24, 0x46,
	0x7e, 0x34, 0x8d, 0xc4, 0x2b, 0xb7, 0x8e, 0x22,
	0x9e, 0x87, 0x62, 0xca, 0xbc, 0x10, 0x09, 0x4a,
	0x4b, 0x0b, 0xdb, 0x57, 0x9b, 0xa9, 0x3e, 0xa8,
	0x99, 0x59, 0xa0, 0x12, 0xf3, 0xa5, 0xe4, 0x91,
	0xbb, 0xb9, 0x05, 0x8d, 0xcf, 0xb9, 0xcb, 0x36,
	0x97, 0xb2, 0x6a, 0x31, 0x8f, 0xcb, 0xf8, 0x5a,
	0x2f, 0x9e, 0xa1, 0xf9, 0x7a, 0xf4, 0x10, 0x0e,
	0xe7, 0x7f, 0x4c, 0xcb, 0xe3, 0x83, 0x17, 0x39,
	0x34, 0xef, 0x49, 0x35, 0x68, 0x50, 0x80, 0xf9,
	0xcd, 0x3a, 0x10, 0xf6, 0x71, 0x1a, 0x94, 0xc3,
	0xec, 0xb9, 0x36, 0x84, 0x36, 0xe7, 0x3f, 0x6f,
	0x9b, 0xa9, 0x2b, 0x5c, 0x96, 0x49, 0x26, 0xda,
	0xb3, 0x08, 0x3d, 0x5e, 0x9e, 0x59, 0xdf, 0x0f,
	0xfc, 0xbe, 0xa8, 0x0b, 0xbc, 0xaa, 0x32, 0xf0,
	0xa5, 0x21, 0x50, 0x15, 0x7e, 0x46, 0xb9, 0x76,
	0x09, 0x4e, 0x4b, 0x6f, 0x9f, 0xc7, 0x8c, 0x6d,
	0x80, 0x37, 0xf9, 0xaa, 0xd1, 0x5f, 0x12, 0xb9,
	0xb3, 0x15, 0xe4, 0x96, 0xa1, 0x01, 0xd5, 0xa0,
};

const struct testvec aes_cbc_256_8_vectors[] = {
	{
		.blkno = 0,
		.ptxt = aes_cbc_ptxt,
		.ctxt = aes_cbc_256_encblkno8_vec0_ctxt,
	},
	{
		.blkno = 1,
		.ptxt = aes_cbc_ptxt,
		.ctxt = aes_cbc_256_encblkno8_vec1_ctxt,
	},
	{
		.blkno = 2,
		.ptxt = aes_cbc_ptxt,
		.ctxt = aes_cbc_256_encblkno8_vec2_ctxt,
	},
	{
		.blkno = 3,
		.ptxt = aes_cbc_ptxt,
		.ctxt = aes_cbc_256_encblkno8_vec3_ctxt,
	},
};

/*
 * 256 bits key from IEEE 1619/D16, NUL terminated.
 */
static const char aes_xts_256_key[33] = {
	0x27, 0x18, 0x28, 0x18, 0x28, 0x45, 0x90, 0x45,
	0x23, 0x53, 0x60, 0x28, 0x74, 0x71, 0x35, 0x26,
	0x31, 0x41, 0x59, 0x26, 0x53, 0x58, 0x97, 0x93,
	0x23, 0x84, 0x62, 0x64, 0x33, 0x83, 0x27, 0x95,
	0
};

/*
 * 512 bits key from IEEE 1619/D16, NUL terminated.
 */
static const char aes_xts_512_key[65] = {
	0x27, 0x18, 0x28, 0x18, 0x28, 0x45, 0x90, 0x45,
	0x23, 0x53, 0x60, 0x28, 0x74, 0x71, 0x35, 0x26,
	0x62, 0x49, 0x77, 0x57, 0x24, 0x70, 0x93, 0x69,
	0x99, 0x59, 0x57, 0x49, 0x66, 0x96, 0x76, 0x27,
	0x31, 0x41, 0x59, 0x26, 0x53, 0x58, 0x97, 0x93,
	0x23, 0x84, 0x62, 0x64, 0x33, 0x83, 0x27, 0x95,
	0x02, 0x88, 0x41, 0x97, 0x16, 0x93, 0x99, 0x37,
	0x51, 0x05, 0x82, 0x09, 0x74, 0x94, 0x45, 0x92,
	0
};

/*
 * Vector 4 from IEEE 1619/D16, blkno 0.
 */
static const uint8_t aes_xts_256_vec4_ptxt[SECSIZE] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
	0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
	0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
	0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
	0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
	0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
	0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
	0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
	0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
	0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
	0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
	0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
	0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
	0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
	0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
	0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
	0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
	0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
	0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
	0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
	0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
	0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
	0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
	0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
	0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
	0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
	0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
	0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
	0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
	0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
	0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
	0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
	0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
	0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
	0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
	0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
	0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
	0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
	0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
	0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
	0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
	0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
	0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
	0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
	0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
	0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
	0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
	0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
	0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
	0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
	0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
	0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
	0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
};

static const uint8_t aes_xts_256_vec4_ctxt[SECSIZE] = {
	0x27, 0xa7, 0x47, 0x9b, 0xef, 0xa1, 0xd4, 0x76,
	0x48, 0x9f, 0x30, 0x8c, 0xd4, 0xcf, 0xa6, 0xe2,
	0xa9, 0x6e, 0x4b, 0xbe, 0x32, 0x08, 0xff, 0x25,
	0x28, 0x7d, 0xd3, 0x81, 0x96, 0x16, 0xe8, 0x9c,
	0xc7, 0x8c, 0xf7, 0xf5, 0xe5, 0x43, 0x44, 0x5f,
	0x83, 0x33, 0xd8, 0xfa, 0x7f, 0x56, 0x00, 0x00,
	0x05, 0x27, 0x9f, 0xa5, 0xd8, 0xb5, 0xe4, 0xad,
	0x40, 0xe7, 0x36, 0xdd, 0xb4, 0xd3, 0x54, 0x12,
	0x32, 0x80, 0x63, 0xfd, 0x2a, 0xab, 0x53, 0xe5,
	0xea, 0x1e, 0x0a, 0x9f, 0x33, 0x25, 0x00, 0xa5,
	0xdf, 0x94, 0x87, 0xd0, 0x7a, 0x5c, 0x92, 0xcc,
	0x51, 0x2c, 0x88, 0x66, 0xc7, 0xe8, 0x60, 0xce,
	0x93, 0xfd, 0xf1, 0x66, 0xa2, 0x49, 0x12, 0xb4,
	0x22, 0x97, 0x61, 0x46, 0xae, 0x20, 0xce, 0x84,
	0x6b, 0xb7, 0xdc, 0x9b, 0xa9, 0x4a, 0x76, 0x7a,
	0xae, 0xf2, 0x0c, 0x0d, 0x61, 0xad, 0x02, 0x65,
	0x5e, 0xa9, 0x2d, 0xc4, 0xc4, 0xe4, 0x1a, 0x89,
	0x52, 0xc6, 0x51, 0xd3, 0x31, 0x74, 0xbe, 0x51,
	0xa1, 0x0c, 0x42, 0x11, 0x10, 0xe6, 0xd8, 0x15,
	0x88, 0xed, 0xe8, 0x21, 0x03, 0xa2, 0x52, 0xd8,
	0xa7, 0x50, 0xe8, 0x76, 0x8d, 0xef, 0xff, 0xed,
	0x91, 0x22, 0x81, 0x0a, 0xae, 0xb9, 0x9f, 0x91,
	0x72, 0xaf, 0x82, 0xb6, 0x04, 0xdc, 0x4b, 0x8e,
	0x51, 0xbc, 0xb0, 0x82, 0x35, 0xa6, 0xf4, 0x34,
	0x13, 0x32, 0xe4, 0xca, 0x60, 0x48, 0x2a, 0x4b,
	0xa1, 0xa0, 0x3b, 0x3e, 0x65, 0x00, 0x8f, 0xc5,
	0xda, 0x76, 0xb7, 0x0b, 0xf1, 0x69, 0x0d, 0xb4,
	0xea, 0xe2, 0x9c, 0x5f, 0x1b, 0xad, 0xd0, 0x3c,
	0x5c, 0xcf, 0x2a, 0x55, 0xd7, 0x05, 0xdd, 0xcd,
	0x86, 0xd4, 0x49, 0x51, 0x1c, 0xeb, 0x7e, 0xc3,
	0x0b, 0xf1, 0x2b, 0x1f, 0xa3, 0x5b, 0x91, 0x3f,
	0x9f, 0x74, 0x7a, 0x8a, 0xfd, 0x1b, 0x13, 0x0e,
	0x94, 0xbf, 0xf9, 0x4e, 0xff, 0xd0, 0x1a, 0x91,
	0x73, 0x5c, 0xa1, 0x72, 0x6a, 0xcd, 0x0b, 0x19,
	0x7c, 0x4e, 0x5b, 0x03, 0x39, 0x36, 0x97, 0xe1,
	0x26, 0x82, 0x6f, 0xb6, 0xbb, 0xde, 0x8e, 0xcc,
	0x1e, 0x08, 0x29, 0x85, 0x16, 0xe2, 0xc9, 0xed,
	0x03, 0xff, 0x3c, 0x1b, 0x78, 0x60, 0xf6, 0xde,
	0x76, 0xd4, 0xce, 0xcd, 0x94, 0xc8, 0x11, 0x98,
	0x55, 0xef, 0x52, 0x97, 0xca, 0x67, 0xe9, 0xf3,
	0xe7, 0xff, 0x72, 0xb1, 0xe9, 0x97, 0x85, 0xca,
	0x0a, 0x7e, 0x77, 0x20, 0xc5, 0xb3, 0x6d, 0xc6,
	0xd7, 0x2c, 0xac, 0x95, 0x74, 0xc8, 0xcb, 0xbc,
	0x2f, 0x80, 0x1e, 0x23, 0xe5, 0x6f, 0xd3, 0x44,
	0xb0, 0x7f, 0x22, 0x15, 0x4b, 0xeb, 0xa0, 0xf0,
	0x8c, 0xe8, 0x89, 0x1e, 0x64, 0x3e, 0xd9, 0x95,
	0xc9, 0x4d, 0x9a, 0x69, 0xc9, 0xf1, 0xb5, 0xf4,
	0x99, 0x02, 0x7a, 0x78, 0x57, 0x2a, 0xee, 0xbd,
	0x74, 0xd2, 0x0c, 0xc3, 0x98, 0x81, 0xc2, 0x13,
	0xee, 0x77, 0x0b, 0x10, 0x10, 0xe4, 0xbe, 0xa7,
	0x18, 0x84, 0x69, 0x77, 0xae, 0x11, 0x9f, 0x7a,
	0x02, 0x3a, 0xb5, 0x8c, 0xca, 0x0a, 0xd7, 0x52,
	0xaf, 0xe6, 0x56, 0xbb, 0x3c, 0x17, 0x25, 0x6a,
	0x9f, 0x6e, 0x9b, 0xf1, 0x9f, 0xdd, 0x5a, 0x38,
	0xfc, 0x82, 0xbb, 0xe8, 0x72, 0xc5, 0x53, 0x9e,
	0xdb, 0x60, 0x9e, 0xf4, 0xf7, 0x9c, 0x20, 0x3e,
	0xbb, 0x14, 0x0f, 0x2e, 0x58, 0x3c, 0xb2, 0xad,
	0x15, 0xb4, 0xaa, 0x5b, 0x65, 0x50, 0x16, 0xa8,
	0x44, 0x92, 0x77, 0xdb, 0xd4, 0x77, 0xef, 0x2c,
	0x8d, 0x6c, 0x01, 0x7d, 0xb7, 0x38, 0xb1, 0x8d,
	0xeb, 0x4a, 0x42, 0x7d, 0x19, 0x23, 0xce, 0x3f,
	0xf2, 0x62, 0x73, 0x57, 0x79, 0xa4, 0x18, 0xf2,
	0x0a, 0x28, 0x2d, 0xf9, 0x20, 0x14, 0x7b, 0xea,
	0xbe, 0x42, 0x1e, 0xe5, 0x31, 0x9d, 0x05, 0x68,
};

/*
 * Vector 5 from IEEE 1619/D16, blkno 1.
 */
static const uint8_t aes_xts_256_vec5_ptxt[SECSIZE] = {
	0x27, 0xa7, 0x47, 0x9b, 0xef, 0xa1, 0xd4, 0x76,
	0x48, 0x9f, 0x30, 0x8c, 0xd4, 0xcf, 0xa6, 0xe2,
	0xa9, 0x6e, 0x4b, 0xbe, 0x32, 0x08, 0xff, 0x25,
	0x28, 0x7d, 0xd3, 0x81, 0x96, 0x16, 0xe8, 0x9c,
	0xc7, 0x8c, 0xf7, 0xf5, 0xe5, 0x43, 0x44, 0x5f,
	0x83, 0x33, 0xd8, 0xfa, 0x7f, 0x56, 0x00, 0x00,
	0x05, 0x27, 0x9f, 0xa5, 0xd8, 0xb5, 0xe4, 0xad,
	0x40, 0xe7, 0x36, 0xdd, 0xb4, 0xd3, 0x54, 0x12,
	0x32, 0x80, 0x63, 0xfd, 0x2a, 0xab, 0x53, 0xe5,
	0xea, 0x1e, 0x0a, 0x9f, 0x33, 0x25, 0x00, 0xa5,
	0xdf, 0x94, 0x87, 0xd0, 0x7a, 0x5c, 0x92, 0xcc,
	0x51, 0x2c, 0x88, 0x66, 0xc7, 0xe8, 0x60, 0xce,
	0x93, 0xfd, 0xf1, 0x66, 0xa2, 0x49, 0x12, 0xb4,
	0x22, 0x97, 0x61, 0x46, 0xae, 0x20, 0xce, 0x84,
	0x6b, 0xb7, 0xdc, 0x9b, 0xa9, 0x4a, 0x76, 0x7a,
	0xae, 0xf2, 0x0c, 0x0d, 0x61, 0xad, 0x02, 0x65,
	0x5e, 0xa9, 0x2d, 0xc4, 0xc4, 0xe4, 0x1a, 0x89,
	0x52, 0xc6, 0x51, 0xd3, 0x31, 0x74, 0xbe, 0x51,
	0xa1, 0x0c, 0x42, 0x11, 0x10, 0xe6, 0xd8, 0x15,
	0x88, 0xed, 0xe8, 0x21, 0x03, 0xa2, 0x52, 0xd8,
	0xa7, 0x50, 0xe8, 0x76, 0x8d, 0xef, 0xff, 0xed,
	0x91, 0x22, 0x81, 0x0a, 0xae, 0xb9, 0x9f, 0x91,
	0x72, 0xaf, 0x82, 0xb6, 0x04, 0xdc, 0x4b, 0x8e,
	0x51, 0xbc, 0xb0, 0x82, 0x35, 0xa6, 0xf4, 0x34,
	0x13, 0x32, 0xe4, 0xca, 0x60, 0x48, 0x2a, 0x4b,
	0xa1, 0xa0, 0x3b, 0x3e, 0x65, 0x00, 0x8f, 0xc5,
	0xda, 0x76, 0xb7, 0x0b, 0xf1, 0x69, 0x0d, 0xb4,
	0xea, 0xe2, 0x9c, 0x5f, 0x1b, 0xad, 0xd0, 0x3c,
	0x5c, 0xcf, 0x2a, 0x55, 0xd7, 0x05, 0xdd, 0xcd,
	0x86, 0xd4, 0x49, 0x51, 0x1c, 0xeb, 0x7e, 0xc3,
	0x0b, 0xf1, 0x2b, 0x1f, 0xa3, 0x5b, 0x91, 0x3f,
	0x9f, 0x74, 0x7a, 0x8a, 0xfd, 0x1b, 0x13, 0x0e,
	0x94, 0xbf, 0xf9, 0x4e, 0xff, 0xd0, 0x1a, 0x91,
	0x73, 0x5c, 0xa1, 0x72, 0x6a, 0xcd, 0x0b, 0x19,
	0x7c, 0x4e, 0x5b, 0x03, 0x39, 0x36, 0x97, 0xe1,
	0x26, 0x82, 0x6f, 0xb6, 0xbb, 0xde, 0x8e, 0xcc,
	0x1e, 0x08, 0x29, 0x85, 0x16, 0xe2, 0xc9, 0xed,
	0x03, 0xff, 0x3c, 0x1b, 0x78, 0x60, 0xf6, 0xde,
	0x76, 0xd4, 0xce, 0xcd, 0x94, 0xc8, 0x11, 0x98,
	0x55, 0xef, 0x52, 0x97, 0xca, 0x67, 0xe9, 0xf3,
	0xe7, 0xff, 0x72, 0xb1, 0xe9, 0x97, 0x85, 0xca,
	0x0a, 0x7e, 0x77, 0x20, 0xc5, 0xb3, 0x6d, 0xc6,
	0xd7, 0x2c, 0xac, 0x95, 0x74, 0xc8, 0xcb, 0xbc,
	0x2f, 0x80, 0x1e, 0x23, 0xe5, 0x6f, 0xd3, 0x44,
	0xb0, 0x7f, 0x22, 0x15, 0x4b, 0xeb, 0xa0, 0xf0,
	0x8c, 0xe8, 0x89, 0x1e, 0x64, 0x3e, 0xd9, 0x95,
	0xc9, 0x4d, 0x9a, 0x69, 0xc9, 0xf1, 0xb5, 0xf4,
	0x99, 0x02, 0x7a, 0x78, 0x57, 0x2a, 0xee, 0xbd,
	0x74, 0xd2, 0x0c, 0xc3, 0x98, 0x81, 0xc2, 0x13,
	0xee, 0x77, 0x0b, 0x10, 0x10, 0xe4, 0xbe, 0xa7,
	0x18, 0x84, 0x69, 0x77, 0xae, 0x11, 0x9f, 0x7a,
	0x02, 0x3a, 0xb5, 0x8c, 0xca, 0x0a, 0xd7, 0x52,
	0xaf, 0xe6, 0x56, 0xbb, 0x3c, 0x17, 0x25, 0x6a,
	0x9f, 0x6e, 0x9b, 0xf1, 0x9f, 0xdd, 0x5a, 0x38,
	0xfc, 0x82, 0xbb, 0xe8, 0x72, 0xc5, 0x53, 0x9e,
	0xdb, 0x60, 0x9e, 0xf4, 0xf7, 0x9c, 0x20, 0x3e,
	0xbb, 0x14, 0x0f, 0x2e, 0x58, 0x3c, 0xb2, 0xad,
	0x15, 0xb4, 0xaa, 0x5b, 0x65, 0x50, 0x16, 0xa8,
	0x44, 0x92, 0x77, 0xdb, 0xd4, 0x77, 0xef, 0x2c,
	0x8d, 0x6c, 0x01, 0x7d, 0xb7, 0x38, 0xb1, 0x8d,
	0xeb, 0x4a, 0x42, 0x7d, 0x19, 0x23, 0xce, 0x3f,
	0xf2, 0x62, 0x73, 0x57, 0x79, 0xa4, 0x18, 0xf2,
	0x0a, 0x28, 0x2d, 0xf9, 0x20, 0x14, 0x7b, 0xea,
	0xbe, 0x42, 0x1e, 0xe5, 0x31, 0x9d, 0x05, 0x68,
};

static const uint8_t aes_xts_256_vec5_ctxt[SECSIZE] = {
	0x26, 0x4d, 0x3c, 0xa8, 0x51, 0x21, 0x94, 0xfe,
	0xc3, 0x12, 0xc8, 0xc9, 0x89, 0x1f, 0x27, 0x9f,
	0xef, 0xdd, 0x60, 0x8d, 0x0c, 0x02, 0x7b, 0x60,
	0x48, 0x3a, 0x3f, 0xa8, 0x11, 0xd6, 0x5e, 0xe5,
	0x9d, 0x52, 0xd9, 0xe4, 0x0e, 0xc5, 0x67, 0x2d,
	0x81, 0x53, 0x2b, 0x38, 0xb6, 0xb0, 0x89, 0xce,
	0x95, 0x1f, 0x0f, 0x9c, 0x35, 0x59, 0x0b, 0x8b,
	0x97, 0x8d, 0x17, 0x52, 0x13, 0xf3, 0x29, 0xbb,
	0x1c, 0x2f, 0xd3, 0x0f, 0x2f, 0x7f, 0x30, 0x49,
	0x2a, 0x61, 0xa5, 0x32, 0xa7, 0x9f, 0x51, 0xd3,
	0x6f, 0x5e, 0x31, 0xa7, 0xc9, 0xa1, 0x2c, 0x28,
	0x60, 0x82, 0xff, 0x7d, 0x23, 0x94, 0xd1, 0x8f,
	0x78, 0x3e, 0x1a, 0x8e, 0x72, 0xc7, 0x22, 0xca,
	0xaa, 0xa5, 0x2d, 0x8f, 0x06, 0x56, 0x57, 0xd2,
	0x63, 0x1f, 0xd2, 0x5b, 0xfd, 0x8e, 0x5b, 0xaa,
	0xd6, 0xe5, 0x27, 0xd7, 0x63, 0x51, 0x75, 0x01,
	0xc6, 0x8c, 0x5e, 0xdc, 0x3c, 0xdd, 0x55, 0x43,
	0x5c, 0x53, 0x2d, 0x71, 0x25, 0xc8, 0x61, 0x4d,
	0xee, 0xd9, 0xad, 0xaa, 0x3a, 0xca, 0xde, 0x58,
	0x88, 0xb8, 0x7b, 0xef, 0x64, 0x1c, 0x4c, 0x99,
	0x4c, 0x80, 0x91, 0xb5, 0xbc, 0xd3, 0x87, 0xf3,
	0x96, 0x3f, 0xb5, 0xbc, 0x37, 0xaa, 0x92, 0x2f,
	0xbf, 0xe3, 0xdf, 0x4e, 0x5b, 0x91, 0x5e, 0x6e,
	0xb5, 0x14, 0x71, 0x7b, 0xdd, 0x2a, 0x74, 0x07,
	0x9a, 0x50, 0x73, 0xf5, 0xc4, 0xbf, 0xd4, 0x6a,
	0xdf, 0x7d, 0x28, 0x2e, 0x7a, 0x39, 0x3a, 0x52,
	0x57, 0x9d, 0x11, 0xa0, 0x28, 0xda, 0x4d, 0x9c,
	0xd9, 0xc7, 0x71, 0x24, 0xf9, 0x64, 0x8e, 0xe3,
	0x83, 0xb1, 0xac, 0x76, 0x39, 0x30, 0xe7, 0x16,
	0x2a, 0x8d, 0x37, 0xf3, 0x50, 0xb2, 0xf7, 0x4b,
	0x84, 0x72, 0xcf, 0x09, 0x90, 0x20, 0x63, 0xc6,
	0xb3, 0x2e, 0x8c, 0x2d, 0x92, 0x90, 0xce, 0xfb,
	0xd7, 0x34, 0x6d, 0x1c, 0x77, 0x9a, 0x0d, 0xf5,
	0x0e, 0xdc, 0xde, 0x45, 0x31, 0xda, 0x07, 0xb0,
	0x99, 0xc6, 0x38, 0xe8, 0x3a, 0x75, 0x59, 0x44,
	0xdf, 0x2a, 0xef, 0x1a, 0xa3, 0x17, 0x52, 0xfd,
	0x32, 0x3d, 0xcb, 0x71, 0x0f, 0xb4, 0xbf, 0xbb,
	0x9d, 0x22, 0xb9, 0x25, 0xbc, 0x35, 0x77, 0xe1,
	0xb8, 0x94, 0x9e, 0x72, 0x9a, 0x90, 0xbb, 0xaf,
	0xea, 0xcf, 0x7f, 0x78, 0x79, 0xe7, 0xb1, 0x14,
	0x7e, 0x28, 0xba, 0x0b, 0xae, 0x94, 0x0d, 0xb7,
	0x95, 0xa6, 0x1b, 0x15, 0xec, 0xf4, 0xdf, 0x8d,
	0xb0, 0x7b, 0x82, 0x4b, 0xb0, 0x62, 0x80, 0x2c,
	0xc9, 0x8a, 0x95, 0x45, 0xbb, 0x2a, 0xae, 0xed,
	0x77, 0xcb, 0x3f, 0xc6, 0xdb, 0x15, 0xdc, 0xd7,
	0xd8, 0x0d, 0x7d, 0x5b, 0xc4, 0x06, 0xc4, 0x97,
	0x0a, 0x34, 0x78, 0xad, 0xa8, 0x89, 0x9b, 0x32,
	0x91, 0x98, 0xeb, 0x61, 0xc1, 0x93, 0xfb, 0x62,
	0x75, 0xaa, 0x8c, 0xa3, 0x40, 0x34, 0x4a, 0x75,
	0xa8, 0x62, 0xae, 0xbe, 0x92, 0xee, 0xe1, 0xce,
	0x03, 0x2f, 0xd9, 0x50, 0xb4, 0x7d, 0x77, 0x04,
	0xa3, 0x87, 0x69, 0x23, 0xb4, 0xad, 0x62, 0x84,
	0x4b, 0xf4, 0xa0, 0x9c, 0x4d, 0xbe, 0x8b, 0x43,
	0x97, 0x18, 0x4b, 0x74, 0x71, 0x36, 0x0c, 0x95,
	0x64, 0x88, 0x0a, 0xed, 0xdd, 0xb9, 0xba, 0xa4,
	0xaf, 0x2e, 0x75, 0x39, 0x4b, 0x08, 0xcd, 0x32,
	0xff, 0x47, 0x9c, 0x57, 0xa0, 0x7d, 0x3e, 0xab,
	0x5d, 0x54, 0xde, 0x5f, 0x97, 0x38, 0xb8, 0xd2,
	0x7f, 0x27, 0xa9, 0xf0, 0xab, 0x11, 0x79, 0x9d,
	0x7b, 0x7f, 0xfe, 0xfb, 0x27, 0x04, 0xc9, 0x5c,
	0x6a, 0xd1, 0x2c, 0x39, 0xf1, 0xe8, 0x67, 0xa4,
	0xb7, 0xb1, 0xd7, 0x81, 0x8a, 0x4b, 0x75, 0x3d,
	0xfd, 0x2a, 0x89, 0xcc, 0xb4, 0x5e, 0x00, 0x1a,
	0x03, 0xa8, 0x67, 0xb1, 0x87, 0xf2, 0x25, 0xdd,
};

/*
 * Vector 6 from IEEE 1619/D16, blkno 2.
 */
static const uint8_t aes_xts_256_vec6_ptxt[SECSIZE] = {
	0x26, 0x4d, 0x3c, 0xa8, 0x51, 0x21, 0x94, 0xfe,
	0xc3, 0x12, 0xc8, 0xc9, 0x89, 0x1f, 0x27, 0x9f,
	0xef, 0xdd, 0x60, 0x8d, 0x0c, 0x02, 0x7b, 0x60,
	0x48, 0x3a, 0x3f, 0xa8, 0x11, 0xd6, 0x5e, 0xe5,
	0x9d, 0x52, 0xd9, 0xe4, 0x0e, 0xc5, 0x67, 0x2d,
	0x81, 0x53, 0x2b, 0x38, 0xb6, 0xb0, 0x89, 0xce,
	0x95, 0x1f, 0x0f, 0x9c, 0x35, 0x59, 0x0b, 0x8b,
	0x97, 0x8d, 0x17, 0x52, 0x13, 0xf3, 0x29, 0xbb,
	0x1c, 0x2f, 0xd3, 0x0f, 0x2f, 0x7f, 0x30, 0x49,
	0x2a, 0x61, 0xa5, 0x32, 0xa7, 0x9f, 0x51, 0xd3,
	0x6f, 0x5e, 0x31, 0xa7, 0xc9, 0xa1, 0x2c, 0x28,
	0x60, 0x82, 0xff, 0x7d, 0x23, 0x94, 0xd1, 0x8f,
	0x78, 0x3e, 0x1a, 0x8e, 0x72, 0xc7, 0x22, 0xca,
	0xaa, 0xa5, 0x2d, 0x8f, 0x06, 0x56, 0x57, 0xd2,
	0x63, 0x1f, 0xd2, 0x5b, 0xfd, 0x8e, 0x5b, 0xaa,
	0xd6, 0xe5, 0x27, 0xd7, 0x63, 0x51, 0x75, 0x01,
	0xc6, 0x8c, 0x5e, 0xdc, 0x3c, 0xdd, 0x55, 0x43,
	0x5c, 0x53, 0x2d, 0x71, 0x25, 0xc8, 0x61, 0x4d,
	0xee, 0xd9, 0xad, 0xaa, 0x3a, 0xca, 0xde, 0x58,
	0x88, 0xb8, 0x7b, 0xef, 0x64, 0x1c, 0x4c, 0x99,
	0x4c, 0x80, 0x91, 0xb5, 0xbc, 0xd3, 0x87, 0xf3,
	0x96, 0x3f, 0xb5, 0xbc, 0x37, 0xaa, 0x92, 0x2f,
	0xbf, 0xe3, 0xdf, 0x4e, 0x5b, 0x91, 0x5e, 0x6e,
	0xb5, 0x14, 0x71, 0x7b, 0xdd, 0x2a, 0x74, 0x07,
	0x9a, 0x50, 0x73, 0xf5, 0xc4, 0xbf, 0xd4, 0x6a,
	0xdf, 0x7d, 0x28, 0x2e, 0x7a, 0x39, 0x3a, 0x52,
	0x57, 0x9d, 0x11, 0xa0, 0x28, 0xda, 0x4d, 0x9c,
	0xd9, 0xc7, 0x71, 0x24, 0xf9, 0x64, 0x8e, 0xe3,
	0x83, 0xb1, 0xac, 0x76, 0x39, 0x30, 0xe7, 0x16,
	0x2a, 0x8d, 0x37, 0xf3, 0x50, 0xb2, 0xf7, 0x4b,
	0x84, 0x72, 0xcf, 0x09, 0x90, 0x20, 0x63, 0xc6,
	0xb3, 0x2e, 0x8c, 0x2d, 0x92, 0x90, 0xce, 0xfb,
	0xd7, 0x34, 0x6d, 0x1c, 0x77, 0x9a, 0x0d, 0xf5,
	0x0e, 0xdc, 0xde, 0x45, 0x31, 0xda, 0x07, 0xb0,
	0x99, 0xc6, 0x38, 0xe8, 0x3a, 0x75, 0x59, 0x44,
	0xdf, 0x2a, 0xef, 0x1a, 0xa3, 0x17, 0x52, 0xfd,
	0x32, 0x3d, 0xcb, 0x71, 0x0f, 0xb4, 0xbf, 0xbb,
	0x9d, 0x22, 0xb9, 0x25, 0xbc, 0x35, 0x77, 0xe1,
	0xb8, 0x94, 0x9e, 0x72, 0x9a, 0x90, 0xbb, 0xaf,
	0xea, 0xcf, 0x7f, 0x78, 0x79, 0xe7, 0xb1, 0x14,
	0x7e, 0x28, 0xba, 0x0b, 0xae, 0x94, 0x0d, 0xb7,
	0x95, 0xa6, 0x1b, 0x15, 0xec, 0xf4, 0xdf, 0x8d,
	0xb0, 0x7b, 0x82, 0x4b, 0xb0, 0x62, 0x80, 0x2c,
	0xc9, 0x8a, 0x95, 0x45, 0xbb, 0x2a, 0xae, 0xed,
	0x77, 0xcb, 0x3f, 0xc6, 0xdb, 0x15, 0xdc, 0xd7,
	0xd8, 0x0d, 0x7d, 0x5b, 0xc4, 0x06, 0xc4, 0x97,
	0x0a, 0x34, 0x78, 0xad, 0xa8, 0x89, 0x9b, 0x32,
	0x91, 0x98, 0xeb, 0x61, 0xc1, 0x93, 0xfb, 0x62,
	0x75, 0xaa, 0x8c, 0xa3, 0x40, 0x34, 0x4a, 0x75,
	0xa8, 0x62, 0xae, 0xbe, 0x92, 0xee, 0xe1, 0xce,
	0x03, 0x2f, 0xd9, 0x50, 0xb4, 0x7d, 0x77, 0x04,
	0xa3, 0x87, 0x69, 0x23, 0xb4, 0xad, 0x62, 0x84,
	0x4b, 0xf4, 0xa0, 0x9c, 0x4d, 0xbe, 0x8b, 0x43,
	0x97, 0x18, 0x4b, 0x74, 0x71, 0x36, 0x0c, 0x95,
	0x64, 0x88, 0x0a, 0xed, 0xdd, 0xb9, 0xba, 0xa4,
	0xaf, 0x2e, 0x75, 0x39, 0x4b, 0x08, 0xcd, 0x32,
	0xff, 0x47, 0x9c, 0x57, 0xa0, 0x7d, 0x3e, 0xab,
	0x5d, 0x54, 0xde, 0x5f, 0x97, 0x38, 0xb8, 0xd2,
	0x7f, 0x27, 0xa9, 0xf0, 0xab, 0x11, 0x79, 0x9d,
	0x7b, 0x7f, 0xfe, 0xfb, 0x27, 0x04, 0xc9, 0x5c,
	0x6a, 0xd1, 0x2c, 0x39, 0xf1, 0xe8, 0x67, 0xa4,
	0xb7, 0xb1, 0xd7, 0x81, 0x8a, 0x4b, 0x75, 0x3d,
	0xfd, 0x2a, 0x89, 0xcc, 0xb4, 0x5e, 0x00, 0x1a,
	0x03, 0xa8, 0x67, 0xb1, 0x87, 0xf2, 0x25, 0xdd,
};

static const uint8_t aes_xts_256_vec6_ctxt[SECSIZE] = {
	0xfa, 0x76, 0x2a, 0x36, 0x80, 0xb7, 0x60, 0x07,
	0x92, 0x8e, 0xd4, 0xa4, 0xf4, 0x9a, 0x94, 0x56,
	0x03, 0x1b, 0x70, 0x47, 0x82, 0xe6, 0x5e, 0x16,
	0xce, 0xcb, 0x54, 0xed, 0x7d, 0x01, 0x7b, 0x5e,
	0x18, 0xab, 0xd6, 0x7b, 0x33, 0x8e, 0x81, 0x07,
	0x8f, 0x21, 0xed, 0xb7, 0x86, 0x8d, 0x90, 0x1e,
	0xbe, 0x9c, 0x73, 0x1a, 0x7c, 0x18, 0xb5, 0xe6,
	0xde, 0xc1, 0xd6, 0xa7, 0x2e, 0x07, 0x8a, 0xc9,
	0xa4, 0x26, 0x2f, 0x86, 0x0b, 0xee, 0xfa, 0x14,
	0xf4, 0xe8, 0x21, 0x01, 0x82, 0x72, 0xe4, 0x11,
	0xa9, 0x51, 0x50, 0x2b, 0x6e, 0x79, 0x06, 0x6e,
	0x84, 0x25, 0x2c, 0x33, 0x46, 0xf3, 0xaa, 0x62,
	0x34, 0x43, 0x51, 0xa2, 0x91, 0xd4, 0xbe, 0xdc,
	0x7a, 0x07, 0x61, 0x8b, 0xde, 0xa2, 0xaf, 0x63,
	0x14, 0x5c, 0xc7, 0xa4, 0xb8, 0xd4, 0x07, 0x06,
	0x91, 0xae, 0x89, 0x0c, 0xd6, 0x57, 0x33, 0xe7,
	0x94, 0x6e, 0x90, 0x21, 0xa1, 0xdf, 0xfc, 0x4c,
	0x59, 0xf1, 0x59, 0x42, 0x5e, 0xe6, 0xd5, 0x0c,
	0xa9, 0xb1, 0x35, 0xfa, 0x61, 0x62, 0xce, 0xa1,
	0x8a, 0x93, 0x98, 0x38, 0xdc, 0x00, 0x0f, 0xb3,
	0x86, 0xfa, 0xd0, 0x86, 0xac, 0xce, 0x5a, 0xc0,
	0x7c, 0xb2, 0xec, 0xe7, 0xfd, 0x58, 0x0b, 0x00,
	0xcf, 0xa5, 0xe9, 0x85, 0x89, 0x63, 0x1d, 0xc2,
	0x5e, 0x8e, 0x2a, 0x3d, 0xaf, 0x2f, 0xfd, 0xec,
	0x26, 0x53, 0x16, 0x59, 0x91, 0x2c, 0x9d, 0x8f,
	0x7a, 0x15, 0xe5, 0x86, 0x5e, 0xa8, 0xfb, 0x58,
	0x16, 0xd6, 0x20, 0x70, 0x52, 0xbd, 0x71, 0x28,
	0xcd, 0x74, 0x3c, 0x12, 0xc8, 0x11, 0x87, 0x91,
	0xa4, 0x73, 0x68, 0x11, 0x93, 0x5e, 0xb9, 0x82,
	0xa5, 0x32, 0x34, 0x9e, 0x31, 0xdd, 0x40, 0x1e,
	0x0b, 0x66, 0x0a, 0x56, 0x8c, 0xb1, 0xa4, 0x71,
	0x1f, 0x55, 0x2f, 0x55, 0xde, 0xd5, 0x9f, 0x1f,
	0x15, 0xbf, 0x71, 0x96, 0xb3, 0xca, 0x12, 0xa9,
	0x1e, 0x48, 0x8e, 0xf5, 0x9d, 0x64, 0xf3, 0xa0,
	0x2b, 0xf4, 0x52, 0x39, 0x49, 0x9a, 0xc6, 0x17,
	0x6a, 0xe3, 0x21, 0xc4, 0xa2, 0x11, 0xec, 0x54,
	0x53, 0x65, 0x97, 0x1c, 0x5d, 0x3f, 0x4f, 0x09,
	0xd4, 0xeb, 0x13, 0x9b, 0xfd, 0xf2, 0x07, 0x3d,
	0x33, 0x18, 0x0b, 0x21, 0x00, 0x2b, 0x65, 0xcc,
	0x98, 0x65, 0xe7, 0x6c, 0xb2, 0x4c, 0xd9, 0x2c,
	0x87, 0x4c, 0x24, 0xc1, 0x83, 0x50, 0x39, 0x9a,
	0x93, 0x6a, 0xb3, 0x63, 0x70, 0x79, 0x29, 0x5d,
	0x76, 0xc4, 0x17, 0x77, 0x6b, 0x94, 0xef, 0xce,
	0x3a, 0x0e, 0xf7, 0x20, 0x6b, 0x15, 0x11, 0x05,
	0x19, 0x65, 0x5c, 0x95, 0x6c, 0xbd, 0x8b, 0x24,
	0x89, 0x40, 0x5e, 0xe2, 0xb0, 0x9a, 0x6b, 0x6e,
	0xeb, 0xe0, 0xc5, 0x37, 0x90, 0xa1, 0x2a, 0x89,
	0x98, 0x37, 0x8b, 0x33, 0xa5, 0xb7, 0x11, 0x59,
	0x62, 0x5f, 0x4b, 0xa4, 0x9d, 0x2a, 0x2f, 0xdb,
	0xa5, 0x9f, 0xbf, 0x08, 0x97, 0xbc, 0x7a, 0xab,
	0xd8, 0xd7, 0x07, 0xdc, 0x14, 0x0a, 0x80, 0xf0,
	0xf3, 0x09, 0xf8, 0x35, 0xd3, 0xda, 0x54, 0xab,
	0x58, 0x4e, 0x50, 0x1d, 0xfa, 0x0e, 0xe9, 0x77,
	0xfe, 0xc5, 0x43, 0xf7, 0x41, 0x86, 0xa8, 0x02,
	0xb9, 0xa3, 0x7a, 0xdb, 0x3e, 0x82, 0x91, 0xec,
	0xa0, 0x4d, 0x66, 0x52, 0x0d, 0x22, 0x9e, 0x60,
	0x40, 0x1e, 0x72, 0x82, 0xbe, 0xf4, 0x86, 0xae,
	0x05, 0x9a, 0xa7, 0x06, 0x96, 0xe0, 0xe3, 0x05,
	0xd7, 0x77, 0x14, 0x0a, 0x7a, 0x88, 0x3e, 0xcd,
	0xcb, 0x69, 0xb9, 0xff, 0x93, 0x8e, 0x8a, 0x42,
	0x31, 0x86, 0x4c, 0x69, 0xca, 0x2c, 0x20, 0x43,
	0xbe, 0xd0, 0x07, 0xff, 0x3e, 0x60, 0x5e, 0x01,
	0x4b, 0xcf, 0x51, 0x81, 0x38, 0xdc, 0x3a, 0x25,
	0xc5, 0xe2, 0x36, 0x17, 0x1a, 0x2d, 0x01, 0xd6,
};

/*
 * Vector 7 from IEEE 1619/D16, blkno 0xfd.
 */
static const uint8_t aes_xts_256_vec7_ptxt[SECSIZE] = {
	0x8e, 0x41, 0xb7, 0x8c, 0x39, 0x0b, 0x5a, 0xf9,
	0xd7, 0x58, 0xbb, 0x21, 0x4a, 0x67, 0xe9, 0xf6,
	0xbf, 0x77, 0x27, 0xb0, 0x9a, 0xc6, 0x12, 0x40,
	0x84, 0xc3, 0x76, 0x11, 0x39, 0x8f, 0xa4, 0x5d,
	0xaa, 0xd9, 0x48, 0x68, 0x60, 0x0e, 0xd3, 0x91,
	0xfb, 0x1a, 0xcd, 0x48, 0x57, 0xa9, 0x5b, 0x46,
	0x6e, 0x62, 0xef, 0x9f, 0x4b, 0x37, 0x72, 0x44,
	0xd1, 0xc1, 0x52, 0xe7, 0xb3, 0x0d, 0x73, 0x1a,
	0xad, 0x30, 0xc7, 0x16, 0xd2, 0x14, 0xb7, 0x07,
	0xae, 0xd9, 0x9e, 0xb5, 0xb5, 0xe5, 0x80, 0xb3,
	0xe8, 0x87, 0xcf, 0x74, 0x97, 0x46, 0x56, 0x51,
	0xd4, 0xb6, 0x0e, 0x60, 0x42, 0x05, 0x1d, 0xa3,
	0x69, 0x3c, 0x3b, 0x78, 0xc1, 0x44, 0x89, 0x54,
	0x3b, 0xe8, 0xb6, 0xad, 0x0b, 0xa6, 0x29, 0x56,
	0x5b, 0xba, 0x20, 0x23, 0x13, 0xba, 0x7b, 0x0d,
	0x0c, 0x94, 0xa3, 0x25, 0x2b, 0x67, 0x6f, 0x46,
	0xcc, 0x02, 0xce, 0x0f, 0x8a, 0x7d, 0x34, 0xc0,
	0xed, 0x22, 0x91, 0x29, 0x67, 0x3c, 0x1f, 0x61,
	0xae, 0xd5, 0x79, 0xd0, 0x8a, 0x92, 0x03, 0xa2,
	0x5a, 0xac, 0x3a, 0x77, 0xe9, 0xdb, 0x60, 0x26,
	0x79, 0x96, 0xdb, 0x38, 0xdf, 0x63, 0x73, 0x56,
	0xd9, 0xdc, 0xd1, 0x63, 0x2e, 0x36, 0x99, 0x39,
	0xf2, 0xa2, 0x9d, 0x89, 0x34, 0x5c, 0x66, 0xe0,
	0x50, 0x66, 0xf1, 0xa3, 0x67, 0x7a, 0xef, 0x18,
	0xde, 0xa4, 0x11, 0x3f, 0xae, 0xb6, 0x29, 0xe4,
	0x67, 0x21, 0xa6, 0x6d, 0x0a, 0x7e, 0x78, 0x5d,
	0x3e, 0x29, 0xaf, 0x25, 0x94, 0xeb, 0x67, 0xdf,
	0xa9, 0x82, 0xaf, 0xfe, 0x0a, 0xac, 0x05, 0x8f,
	0x6e, 0x15, 0x86, 0x42, 0x69, 0xb1, 0x35, 0x41,
	0x82, 0x61, 0xfc, 0x3a, 0xfb, 0x08, 0x94, 0x72,
	0xcf, 0x68, 0xc4, 0x5d, 0xd7, 0xf2, 0x31, 0xc6,
	0x24, 0x9b, 0xa0, 0x25, 0x5e, 0x1e, 0x03, 0x38,
	0x33, 0xfc, 0x4d, 0x00, 0xa3, 0xfe, 0x02, 0x13,
	0x2d, 0x7b, 0xc3, 0x87, 0x36, 0x14, 0xb8, 0xae,
	0xe3, 0x42, 0x73, 0x58, 0x1e, 0xa0, 0x32, 0x5c,
	0x81, 0xf0, 0x27, 0x0a, 0xff, 0xa1, 0x36, 0x41,
	0xd0, 0x52, 0xd3, 0x6f, 0x07, 0x57, 0xd4, 0x84,
	0x01, 0x43, 0x54, 0xd0, 0x2d, 0x68, 0x83, 0xca,
	0x15, 0xc2, 0x4d, 0x8c, 0x39, 0x56, 0xb1, 0xbd,
	0x02, 0x7b, 0xcf, 0x41, 0xf1, 0x51, 0xfd, 0x80,
	0x23, 0xc5, 0x34, 0x0e, 0x56, 0x06, 0xf3, 0x7e,
	0x90, 0xfd, 0xb8, 0x7c, 0x86, 0xfb, 0x4f, 0xa6,
	0x34, 0xb3, 0x71, 0x8a, 0x30, 0xba, 0xce, 0x06,
	0xa6, 0x6e, 0xaf, 0x8f, 0x63, 0xc4, 0xaa, 0x3b,
	0x63, 0x78, 0x26, 0xa8, 0x7f, 0xe8, 0xcf, 0xa4,
	0x42, 0x82, 0xe9, 0x2c, 0xb1, 0x61, 0x5a, 0xf3,
	0xa2, 0x8e, 0x53, 0xbc, 0x74, 0xc7, 0xcb, 0xa1,
	0xa0, 0x97, 0x7b, 0xe9, 0x06, 0x5d, 0x0c, 0x1a,
	0x5d, 0xec, 0x6c, 0x54, 0xae, 0x38, 0xd3, 0x7f,
	0x37, 0xaa, 0x35, 0x28, 0x3e, 0x04, 0x8e, 0x55,
	0x30, 0xa8, 0x5c, 0x4e, 0x7a, 0x29, 0xd7, 0xb9,
	0x2e, 0xc0, 0xc3, 0x16, 0x9c, 0xdf, 0x2a, 0x80,
	0x5c, 0x76, 0x04, 0xbc, 0xe6, 0x00, 0x49, 0xb9,
	0xfb, 0x7b, 0x8e, 0xaa, 0xc1, 0x0f, 0x51, 0xae,
	0x23, 0x79, 0x4c, 0xeb, 0xa6, 0x8b, 0xb5, 0x81,
	0x12, 0xe2, 0x93, 0xb9, 0xb6, 0x92, 0xca, 0x72,
	0x1b, 0x37, 0xc6, 0x62, 0xf8, 0x57, 0x4e, 0xd4,
	0xdb, 0xa6, 0xf8, 0x8e, 0x17, 0x08, 0x81, 0xc8,
	0x2c, 0xdd, 0xc1, 0x03, 0x4a, 0x0c, 0xa7, 0xe2,
	0x84, 0xbf, 0x09, 0x62, 0xb6, 0xb2, 0x62, 0x92,
	0xd8, 0x36, 0xfa, 0x9f, 0x73, 0xc1, 0xac, 0x77,
	0x0e, 0xef, 0x0f, 0x2d, 0x3a, 0x1e, 0xaf, 0x61,
	0xd3, 0xe0, 0x35, 0x55, 0xfd, 0x42, 0x4e, 0xed,
	0xd6, 0x7e, 0x18, 0xa1, 0x80, 0x94, 0xf8, 0x88,
};

static const uint8_t aes_xts_256_vec7_ctxt[SECSIZE] = {
	0xd5, 0x5f, 0x68, 0x4f, 0x81, 0xf4, 0x42, 0x6e,
	0x9f, 0xde, 0x92, 0xa5, 0xff, 0x02, 0xdf, 0x2a,
	0xc8, 0x96, 0xaf, 0x63, 0x96, 0x28, 0x88, 0xa9,
	0x79, 0x10, 0xc1, 0x37, 0x9e, 0x20, 0xb0, 0xa3,
	0xb1, 0xdb, 0x61, 0x3f, 0xb7, 0xfe, 0x2e, 0x07,
	0x00, 0x43, 0x29, 0xea, 0x5c, 0x22, 0xbf, 0xd3,
	0x3e, 0x3d, 0xbe, 0x4c, 0xf5, 0x8c, 0xc6, 0x08,
	0xc2, 0xc2, 0x6c, 0x19, 0xa2, 0xe2, 0xfe, 0x22,
	0xf9, 0x87, 0x32, 0xc2, 0xb5, 0xcb, 0x84, 0x4c,
	0xc6, 0xc0, 0x70, 0x2d, 0x91, 0xe1, 0xd5, 0x0f,
	0xc4, 0x38, 0x2a, 0x7e, 0xba, 0x56, 0x35, 0xcd,
	0x60, 0x24, 0x32, 0xa2, 0x30, 0x6a, 0xc4, 0xce,
	0x82, 0xf8, 0xd7, 0x0c, 0x8d, 0x9b, 0xc1, 0x5f,
	0x91, 0x8f, 0xe7, 0x1e, 0x74, 0xc6, 0x22, 0xd5,
	0xcf, 0x71, 0x17, 0x8b, 0xf6, 0xe0, 0xb9, 0xcc,
	0x9f, 0x2b, 0x41, 0xdd, 0x8d, 0xbe, 0x44, 0x1c,
	0x41, 0xcd, 0x0c, 0x73, 0xa6, 0xdc, 0x47, 0xa3,
	0x48, 0xf6, 0x70, 0x2f, 0x9d, 0x0e, 0x9b, 0x1b,
	0x14, 0x31, 0xe9, 0x48, 0xe2, 0x99, 0xb9, 0xec,
	0x22, 0x72, 0xab, 0x2c, 0x5f, 0x0c, 0x7b, 0xe8,
	0x6a, 0xff, 0xa5, 0xde, 0xc8, 0x7a, 0x0b, 0xee,
	0x81, 0xd3, 0xd5, 0x00, 0x07, 0xed, 0xaa, 0x2b,
	0xcf, 0xcc, 0xb3, 0x56, 0x05, 0x15, 0x5f, 0xf3,
	0x6e, 0xd8, 0xed, 0xd4, 0xa4, 0x0d, 0xcd, 0x4b,
	0x24, 0x3a, 0xcd, 0x11, 0xb2, 0xb9, 0x87, 0xbd,
	0xbf, 0xaf, 0x91, 0xa7, 0xca, 0xc2, 0x7e, 0x9c,
	0x5a, 0xea, 0x52, 0x5e, 0xe5, 0x3d, 0xe7, 0xb2,
	0xd3, 0x33, 0x2c, 0x86, 0x44, 0x40, 0x2b, 0x82,
	0x3e, 0x94, 0xa7, 0xdb, 0x26, 0x27, 0x6d, 0x2d,
	0x23, 0xaa, 0x07, 0x18, 0x0f, 0x76, 0xb4, 0xfd,
	0x29, 0xb9, 0xc0, 0x82, 0x30, 0x99, 0xc9, 0xd6,
	0x2c, 0x51, 0x98, 0x80, 0xae, 0xe7, 0xe9, 0x69,
	0x76, 0x17, 0xc1, 0x49, 0x7d, 0x47, 0xbf, 0x3e,
	0x57, 0x19, 0x50, 0x31, 0x14, 0x21, 0xb6, 0xb7,
	0x34, 0xd3, 0x8b, 0x0d, 0xb9, 0x1e, 0xb8, 0x53,
	0x31, 0xb9, 0x1e, 0xa9, 0xf6, 0x15, 0x30, 0xf5,
	0x45, 0x12, 0xa5, 0xa5, 0x2a, 0x4b, 0xad, 0x58,
	0x9e, 0xb6, 0x97, 0x81, 0xd5, 0x37, 0xf2, 0x32,
	0x97, 0xbb, 0x45, 0x9b, 0xda, 0xd2, 0x94, 0x8a,
	0x29, 0xe1, 0x55, 0x0b, 0xf4, 0x78, 0x7e, 0x0b,
	0xe9, 0x5b, 0xb1, 0x73, 0xcf, 0x5f, 0xab, 0x17,
	0xda, 0xb7, 0xa1, 0x3a, 0x05, 0x2a, 0x63, 0x45,
	0x3d, 0x97, 0xcc, 0xec, 0x1a, 0x32, 0x19, 0x54,
	0x88, 0x6b, 0x7a, 0x12, 0x99, 0xfa, 0xae, 0xec,
	0xae, 0x35, 0xc6, 0xea, 0xac, 0xa7, 0x53, 0xb0,
	0x41, 0xb5, 0xe5, 0xf0, 0x93, 0xbf, 0x83, 0x39,
	0x7f, 0xd2, 0x1d, 0xd6, 0xb3, 0x01, 0x20, 0x66,
	0xfc, 0xc0, 0x58, 0xcc, 0x32, 0xc3, 0xb0, 0x9d,
	0x75, 0x62, 0xde, 0xe2, 0x95, 0x09, 0xb5, 0x83,
	0x93, 0x92, 0xc9, 0xff, 0x05, 0xf5, 0x1f, 0x31,
	0x66, 0xaa, 0xac, 0x4a, 0xc5, 0xf2, 0x38, 0x03,
	0x8a, 0x30, 0x45, 0xe6, 0xf7, 0x2e, 0x48, 0xef,
	0x0f, 0xe8, 0xbc, 0x67, 0x5e, 0x82, 0xc3, 0x18,
	0xa2, 0x68, 0xe4, 0x39, 0x70, 0x27, 0x1b, 0xf1,
	0x19, 0xb8, 0x1b, 0xf6, 0xa9, 0x82, 0x74, 0x65,
	0x54, 0xf8, 0x4e, 0x72, 0xb9, 0xf0, 0x02, 0x80,
	0xa3, 0x20, 0xa0, 0x81, 0x42, 0x92, 0x3c, 0x23,
	0xc8, 0x83, 0x42, 0x3f, 0xf9, 0x49, 0x82, 0x7f,
	0x29, 0xbb, 0xac, 0xdc, 0x1c, 0xcd, 0xb0, 0x49,
	0x38, 0xce, 0x60, 0x98, 0xc9, 0x5b, 0xa6, 0xb3,
	0x25, 0x28, 0xf4, 0xef, 0x78, 0xee, 0xd7, 0x78,
	0xb2, 0xe1, 0x22, 0xdd, 0xfd, 0x1c, 0xbd, 0xd1,
	0x1d, 0x1c, 0x0a, 0x67, 0x83, 0xe0, 0x11, 0xfc,
	0x53, 0x6d, 0x63, 0xd0, 0x53, 0x26, 0x06, 0x37,
};

/*
 * Vector 8 from IEEE 1619/D16, blkno 0xfe.
 */
static const uint8_t aes_xts_256_vec8_ptxt[SECSIZE] = {
	0xd5, 0x5f, 0x68, 0x4f, 0x81, 0xf4, 0x42, 0x6e,
	0x9f, 0xde, 0x92, 0xa5, 0xff, 0x02, 0xdf, 0x2a,
	0xc8, 0x96, 0xaf, 0x63, 0x96, 0x28, 0x88, 0xa9,
	0x79, 0x10, 0xc1, 0x37, 0x9e, 0x20, 0xb0, 0xa3,
	0xb1, 0xdb, 0x61, 0x3f, 0xb7, 0xfe, 0x2e, 0x07,
	0x00, 0x43, 0x29, 0xea, 0x5c, 0x22, 0xbf, 0xd3,
	0x3e, 0x3d, 0xbe, 0x4c, 0xf5, 0x8c, 0xc6, 0x08,
	0xc2, 0xc2, 0x6c, 0x19, 0xa2, 0xe2, 0xfe, 0x22,
	0xf9, 0x87, 0x32, 0xc2, 0xb5, 0xcb, 0x84, 0x4c,
	0xc6, 0xc0, 0x70, 0x2d, 0x91, 0xe1, 0xd5, 0x0f,
	0xc4, 0x38, 0x2a, 0x7e, 0xba, 0x56, 0x35, 0xcd,
	0x60, 0x24, 0x32, 0xa2, 0x30, 0x6a, 0xc4, 0xce,
	0x82, 0xf8, 0xd7, 0x0c, 0x8d, 0x9b, 0xc1, 0x5f,
	0x91, 0x8f, 0xe7, 0x1e, 0x74, 0xc6, 0x22, 0xd5,
	0xcf, 0x71, 0x17, 0x8b, 0xf6, 0xe0, 0xb9, 0xcc,
	0x9f, 0x2b, 0x41, 0xdd, 0x8d, 0xbe, 0x44, 0x1c,
	0x41, 0xcd, 0x0c, 0x73, 0xa6, 0xdc, 0x47, 0xa3,
	0x48, 0xf6, 0x70, 0x2f, 0x9d, 0x0e, 0x9b, 0x1b,
	0x14, 0x31, 0xe9, 0x48, 0xe2, 0x99, 0xb9, 0xec,
	0x22, 0x72, 0xab, 0x2c, 0x5f, 0x0c, 0x7b, 0xe8,
	0x6a, 0xff, 0xa5, 0xde, 0xc8, 0x7a, 0x0b, 0xee,
	0x81, 0xd3, 0xd5, 0x00, 0x07, 0xed, 0xaa, 0x2b,
	0xcf, 0xcc, 0xb3, 0x56, 0x05, 0x15, 0x5f, 0xf3,
	0x6e, 0xd8, 0xed, 0xd4, 0xa4, 0x0d, 0xcd, 0x4b,
	0x24, 0x3a, 0xcd, 0x11, 0xb2, 0xb9, 0x87, 0xbd,
	0xbf, 0xaf, 0x91, 0xa7, 0xca, 0xc2, 0x7e, 0x9c,
	0x5a, 0xea, 0x52, 0x5e, 0xe5, 0x3d, 0xe7, 0xb2,
	0xd3, 0x33, 0x2c, 0x86, 0x44, 0x40, 0x2b, 0x82,
	0x3e, 0x94, 0xa7, 0xdb, 0x26, 0x27, 0x6d, 0x2d,
	0x23, 0xaa, 0x07, 0x18, 0x0f, 0x76, 0xb4, 0xfd,
	0x29, 0xb9, 0xc0, 0x82, 0x30, 0x99, 0xc9, 0xd6,
	0x2c, 0x51, 0x98, 0x80, 0xae, 0xe7, 0xe9, 0x69,
	0x76, 0x17, 0xc1, 0x49, 0x7d, 0x47, 0xbf, 0x3e,
	0x57, 0x19, 0x50, 0x31, 0x14, 0x21, 0xb6, 0xb7,
	0x34, 0xd3, 0x8b, 0x0d, 0xb9, 0x1e, 0xb8, 0x53,
	0x31, 0xb9, 0x1e, 0xa9, 0xf6, 0x15, 0x30, 0xf5,
	0x45, 0x12, 0xa5, 0xa5, 0x2a, 0x4b, 0xad, 0x58,
	0x9e, 0xb6, 0x97, 0x81, 0xd5, 0x37, 0xf2, 0x32,
	0x97, 0xbb, 0x45, 0x9b, 0xda, 0xd2, 0x94, 0x8a,
	0x29, 0xe1, 0x55, 0x0b, 0xf4, 0x78, 0x7e, 0x0b,
	0xe9, 0x5b, 0xb1, 0x73, 0xcf, 0x5f, 0xab, 0x17,
	0xda, 0xb7, 0xa1, 0x3a, 0x05, 0x2a, 0x63, 0x45,
	0x3d, 0x97, 0xcc, 0xec, 0x1a, 0x32, 0x19, 0x54,
	0x88, 0x6b, 0x7a, 0x12, 0x99, 0xfa, 0xae, 0xec,
	0xae, 0x35, 0xc6, 0xea, 0xac, 0xa7, 0x53, 0xb0,
	0x41, 0xb5, 0xe5, 0xf0, 0x93, 0xbf, 0x83, 0x39,
	0x7f, 0xd2, 0x1d, 0xd6, 0xb3, 0x01, 0x20, 0x66,
	0xfc, 0xc0, 0x58, 0xcc, 0x32, 0xc3, 0xb0, 0x9d,
	0x75, 0x62, 0xde, 0xe2, 0x95, 0x09, 0xb5, 0x83,
	0x93, 0x92, 0xc9, 0xff, 0x05, 0xf5, 0x1f, 0x31,
	0x66, 0xaa, 0xac, 0x4a, 0xc5, 0xf2, 0x38, 0x03,
	0x8a, 0x30, 0x45, 0xe6, 0xf7, 0x2e, 0x48, 0xef,
	0x0f, 0xe8, 0xbc, 0x67, 0x5e, 0x82, 0xc3, 0x18,
	0xa2, 0x68, 0xe4, 0x39, 0x70, 0x27, 0x1b, 0xf1,
	0x19, 0xb8, 0x1b, 0xf6, 0xa9, 0x82, 0x74, 0x65,
	0x54, 0xf8, 0x4e, 0x72, 0xb9, 0xf0, 0x02, 0x80,
	0xa3, 0x20, 0xa0, 0x81, 0x42, 0x92, 0x3c, 0x23,
	0xc8, 0x83, 0x42, 0x3f, 0xf9, 0x49, 0x82, 0x7f,
	0x29, 0xbb, 0xac, 0xdc, 0x1c, 0xcd, 0xb0, 0x49,
	0x38, 0xce, 0x60, 0x98, 0xc9, 0x5b, 0xa6, 0xb3,
	0x25, 0x28, 0xf4, 0xef, 0x78, 0xee, 0xd7, 0x78,
	0xb2, 0xe1, 0x22, 0xdd, 0xfd, 0x1c, 0xbd, 0xd1,
	0x1d, 0x1c, 0x0a, 0x67, 0x83, 0xe0, 0x11, 0xfc,
	0x53, 0x6d, 0x63, 0xd0, 0x53, 0x26, 0x06, 0x37,
};

static const uint8_t aes_xts_256_vec8_ctxt[SECSIZE] = {
	0x72, 0xef, 0xc1, 0xeb, 0xfe, 0x1e, 0xe2, 0x59,
	0x75, 0xa6, 0xeb, 0x3a, 0xa8, 0x58, 0x9d, 0xda,
	0x2b, 0x26, 0x1f, 0x1c, 0x85, 0xbd, 0xab, 0x44,
	0x2a, 0x9e, 0x5b, 0x2d, 0xd1, 0xd7, 0xc3, 0x95,
	0x7a, 0x16, 0xfc, 0x08, 0xe5, 0x26, 0xd4, 0xb1,
	0x22, 0x3f, 0x1b, 0x12, 0x32, 0xa1, 0x1a, 0xf2,
	0x74, 0xc3, 0xd7, 0x0d, 0xac, 0x57, 0xf8, 0x3e,
	0x09, 0x83, 0xc4, 0x98, 0xf1, 0xa6, 0xf1, 0xae,
	0xcb, 0x02, 0x1c, 0x3e, 0x70, 0x08, 0x5a, 0x1e,
	0x52, 0x7f, 0x1c, 0xe4, 0x1e, 0xe5, 0x91, 0x1a,
	0x82, 0x02, 0x01, 0x61, 0x52, 0x9c, 0xd8, 0x27,
	0x73, 0x76, 0x2d, 0xaf, 0x54, 0x59, 0xde, 0x94,
	0xa0, 0xa8, 0x2a, 0xda, 0xe7, 0xe1, 0x70, 0x3c,
	0x80, 0x85, 0x43, 0xc2, 0x9e, 0xd6, 0xfb, 0x32,
	0xd9, 0xe0, 0x04, 0x32, 0x7c, 0x13, 0x55, 0x18,
	0x0c, 0x99, 0x5a, 0x07, 0x74, 0x14, 0x93, 0xa0,
	0x9c, 0x21, 0xba, 0x01, 0xa3, 0x87, 0x88, 0x2d,
	0xa4, 0xf6, 0x25, 0x34, 0xb8, 0x7b, 0xb1, 0x5d,
	0x60, 0xd1, 0x97, 0x20, 0x1c, 0x0f, 0xd3, 0xbf,
	0x30, 0xc1, 0x50, 0x0a, 0x3e, 0xcf, 0xec, 0xdd,
	0x66, 0xd8, 0x72, 0x1f, 0x90, 0xbc, 0xc4, 0xc1,
	0x7e, 0xe9, 0x25, 0xc6, 0x1b, 0x0a, 0x03, 0x72,
	0x7a, 0x9c, 0x0d, 0x5f, 0x5c, 0xa4, 0x62, 0xfb,
	0xfa, 0x0a, 0xf1, 0xc2, 0x51, 0x3a, 0x9d, 0x9d,
	0x4b, 0x53, 0x45, 0xbd, 0x27, 0xa5, 0xf6, 0xe6,
	0x53, 0xf7, 0x51, 0x69, 0x3e, 0x6b, 0x6a, 0x2b,
	0x8e, 0xad, 0x57, 0xd5, 0x11, 0xe0, 0x0e, 0x58,
	0xc4, 0x5b, 0x7b, 0x8d, 0x00, 0x5a, 0xf7, 0x92,
	0x88, 0xf5, 0xc7, 0xc2, 0x2f, 0xd4, 0xf1, 0xbf,
	0x7a, 0x89, 0x8b, 0x03, 0xa5, 0x63, 0x4c, 0x6a,
	0x1a, 0xe3, 0xf9, 0xfa, 0xe5, 0xde, 0x4f, 0x29,
	0x6a, 0x28, 0x96, 0xb2, 0x3e, 0x7e, 0xd4, 0x3e,
	0xd1, 0x4f, 0xa5, 0xa2, 0x80, 0x3f, 0x4d, 0x28,
	0xf0, 0xd3, 0xff, 0xcf, 0x24, 0x75, 0x76, 0x77,
	0xae, 0xbd, 0xb4, 0x7b, 0xb3, 0x88, 0x37, 0x87,
	0x08, 0x94, 0x8a, 0x8d, 0x41, 0x26, 0xed, 0x18,
	0x39, 0xe0, 0xda, 0x29, 0xa5, 0x37, 0xa8, 0xc1,
	0x98, 0xb3, 0xc6, 0x6a, 0xb0, 0x07, 0x12, 0xdd,
	0x26, 0x16, 0x74, 0xbf, 0x45, 0xa7, 0x3d, 0x67,
	0xf7, 0x69, 0x14, 0xf8, 0x30, 0xca, 0x01, 0x4b,
	0x65, 0x59, 0x6f, 0x27, 0xe4, 0xcf, 0x62, 0xde,
	0x66, 0x12, 0x5a, 0x55, 0x66, 0xdf, 0x99, 0x75,
	0x15, 0x56, 0x28, 0xb4, 0x00, 0xfb, 0xfb, 0x3a,
	0x29, 0x04, 0x0e, 0xd5, 0x0f, 0xaf, 0xfd, 0xbb,
	0x18, 0xae, 0xce, 0x7c, 0x5c, 0x44, 0x69, 0x32,
	0x60, 0xaa, 0xb3, 0x86, 0xc0, 0xa3, 0x7b, 0x11,
	0xb1, 0x14, 0xf1, 0xc4, 0x15, 0xae, 0xbb, 0x65,
	0x3b, 0xe4, 0x68, 0x17, 0x94, 0x28, 0xd4, 0x3a,
	0x4d, 0x8b, 0xc3, 0xec, 0x38, 0x81, 0x3e, 0xca,
	0x30, 0xa1, 0x3c, 0xf1, 0xbb, 0x18, 0xd5, 0x24,
	0xf1, 0x99, 0x2d, 0x44, 0xd8, 0xb1, 0xa4, 0x2e,
	0xa3, 0x0b, 0x22, 0xe6, 0xc9, 0x5b, 0x19, 0x9d,
	0x8d, 0x18, 0x2f, 0x88, 0x40, 0xb0, 0x9d, 0x05,
	0x95, 0x85, 0xc3, 0x1a, 0xd6, 0x91, 0xfa, 0x06,
	0x19, 0xff, 0x03, 0x8a, 0xca, 0x2c, 0x39, 0xa9,
	0x43, 0x42, 0x11, 0x57, 0x36, 0x17, 0x17, 0xc4,
	0x9d, 0x32, 0x20, 0x28, 0xa7, 0x46, 0x48, 0x11,
	0x3b, 0xd8, 0xc9, 0xd7, 0xec, 0x77, 0xcf, 0x3c,
	0x89, 0xc1, 0xec, 0x87, 0x18, 0xce, 0xff, 0x85,
	0x16, 0xd9, 0x6b, 0x34, 0xc3, 0xc6, 0x14, 0xf1,
	0x06, 0x99, 0xc9, 0xab, 0xc4, 0xed, 0x04, 0x11,
	0x50, 0x62, 0x23, 0xbe, 0xa1, 0x6a, 0xf3, 0x5c,
	0x88, 0x3a, 0xcc, 0xdb, 0xe1, 0x10, 0x4e, 0xef,
	0x0c, 0xfd, 0xb5, 0x4e, 0x12, 0xfb, 0x23, 0x0a,
};

/*
 * Vector 9 from IEEE 1619/D16, blkno 0xff.
 */
static const uint8_t aes_xts_256_vec9_ptxt[SECSIZE] = {
	0x72, 0xef, 0xc1, 0xeb, 0xfe, 0x1e, 0xe2, 0x59,
	0x75, 0xa6, 0xeb, 0x3a, 0xa8, 0x58, 0x9d, 0xda,
	0x2b, 0x26, 0x1f, 0x1c, 0x85, 0xbd, 0xab, 0x44,
	0x2a, 0x9e, 0x5b, 0x2d, 0xd1, 0xd7, 0xc3, 0x95,
	0x7a, 0x16, 0xfc, 0x08, 0xe5, 0x26, 0xd4, 0xb1,
	0x22, 0x3f, 0x1b, 0x12, 0x32, 0xa1, 0x1a, 0xf2,
	0x74, 0xc3, 0xd7, 0x0d, 0xac, 0x57, 0xf8, 0x3e,
	0x09, 0x83, 0xc4, 0x98, 0xf1, 0xa6, 0xf1, 0xae,
	0xcb, 0x02, 0x1c, 0x3e, 0x70, 0x08, 0x5a, 0x1e,
	0x52, 0x7f, 0x1c, 0xe4, 0x1e, 0xe5, 0x91, 0x1a,
	0x82, 0x02, 0x01, 0x61, 0x52, 0x9c, 0xd8, 0x27,
	0x73, 0x76, 0x2d, 0xaf, 0x54, 0x59, 0xde, 0x94,
	0xa0, 0xa8, 0x2a, 0xda, 0xe7, 0xe1, 0x70, 0x3c,
	0x80, 0x85, 0x43, 0xc2, 0x9e, 0xd6, 0xfb, 0x32,
	0xd9, 0xe0, 0x04, 0x32, 0x7c, 0x13, 0x55, 0x18,
	0x0c, 0x99, 0x5a, 0x07, 0x74, 0x14, 0x93, 0xa0,
	0x9c, 0x21, 0xba, 0x01, 0xa3, 0x87, 0x88, 0x2d,
	0xa4, 0xf6, 0x25, 0x34, 0xb8, 0x7b, 0xb1, 0x5d,
	0x60, 0xd1, 0x97, 0x20, 0x1c, 0x0f, 0xd3, 0xbf,
	0x30, 0xc1, 0x50, 0x0a, 0x3e, 0xcf, 0xec, 0xdd,
	0x66, 0xd8, 0x72, 0x1f, 0x90, 0xbc, 0xc4, 0xc1,
	0x7e, 0xe9, 0x25, 0xc6, 0x1b, 0x0a, 0x03, 0x72,
	0x7a, 0x9c, 0x0d, 0x5f, 0x5c, 0xa4, 0x62, 0xfb,
	0xfa, 0x0a, 0xf1, 0xc2, 0x51, 0x3a, 0x9d, 0x9d,
	0x4b, 0x53, 0x45, 0xbd, 0x27, 0xa5, 0xf6, 0xe6,
	0x53, 0xf7, 0x51, 0x69, 0x3e, 0x6b, 0x6a, 0x2b,
	0x8e, 0xad, 0x57, 0xd5, 0x11, 0xe0, 0x0e, 0x58,
	0xc4, 0x5b, 0x7b, 0x8d, 0x00, 0x5a, 0xf7, 0x92,
	0x88, 0xf5, 0xc7, 0xc2, 0x2f, 0xd4, 0xf1, 0xbf,
	0x7a, 0x89, 0x8b, 0x03, 0xa5, 0x63, 0x4c, 0x6a,
	0x1a, 0xe3, 0xf9, 0xfa, 0xe5, 0xde, 0x4f, 0x29,
	0x6a, 0x28, 0x96, 0xb2, 0x3e, 0x7e, 0xd4, 0x3e,
	0xd1, 0x4f, 0xa5, 0xa2, 0x80, 0x3f, 0x4d, 0x28,
	0xf0, 0xd3, 0xff, 0xcf, 0x24, 0x75, 0x76, 0x77,
	0xae, 0xbd, 0xb4, 0x7b, 0xb3, 0x88, 0x37, 0x87,
	0x08, 0x94, 0x8a, 0x8d, 0x41, 0x26, 0xed, 0x18,
	0x39, 0xe0, 0xda, 0x29, 0xa5, 0x37, 0xa8, 0xc1,
	0x98, 0xb3, 0xc6, 0x6a, 0xb0, 0x07, 0x12, 0xdd,
	0x26, 0x16, 0x74, 0xbf, 0x45, 0xa7, 0x3d, 0x67,
	0xf7, 0x69, 0x14, 0xf8, 0x30, 0xca, 0x01, 0x4b,
	0x65, 0x59, 0x6f, 0x27, 0xe4, 0xcf, 0x62, 0xde,
	0x66, 0x12, 0x5a, 0x55, 0x66, 0xdf, 0x99, 0x75,
	0x15, 0x56, 0x28, 0xb4, 0x00, 0xfb, 0xfb, 0x3a,
	0x29, 0x04, 0x0e, 0xd5, 0x0f, 0xaf, 0xfd, 0xbb,
	0x18, 0xae, 0xce, 0x7c, 0x5c, 0x44, 0x69, 0x32,
	0x60, 0xaa, 0xb3, 0x86, 0xc0, 0xa3, 0x7b, 0x11,
	0xb1, 0x14, 0xf1, 0xc4, 0x15, 0xae, 0xbb, 0x65,
	0x3b, 0xe4, 0x68, 0x17, 0x94, 0x28, 0xd4, 0x3a,
	0x4d, 0x8b, 0xc3, 0xec, 0x38, 0x81, 0x3e, 0xca,
	0x30, 0xa1, 0x3c, 0xf1, 0xbb, 0x18, 0xd5, 0x24,
	0xf1, 0x99, 0x2d, 0x44, 0xd8, 0xb1, 0xa4, 0x2e,
	0xa3, 0x0b, 0x22, 0xe6, 0xc9, 0x5b, 0x19, 0x9d,
	0x8d, 0x18, 0x2f, 0x88, 0x40, 0xb0, 0x9d, 0x05,
	0x95, 0x85, 0xc3, 0x1a, 0xd6, 0x91, 0xfa, 0x06,
	0x19, 0xff, 0x03, 0x8a, 0xca, 0x2c, 0x39, 0xa9,
	0x43, 0x42, 0x11, 0x57, 0x36, 0x17, 0x17, 0xc4,
	0x9d, 0x32, 0x20, 0x28, 0xa7, 0x46, 0x48, 0x11,
	0x3b, 0xd8, 0xc9, 0xd7, 0xec, 0x77, 0xcf, 0x3c,
	0x89, 0xc1, 0xec, 0x87, 0x18, 0xce, 0xff, 0x85,
	0x16, 0xd9, 0x6b, 0x34, 0xc3, 0xc6, 0x14, 0xf1,
	0x06, 0x99, 0xc9, 0xab, 0xc4, 0xed, 0x04, 0x11,
	0x50, 0x62, 0x23, 0xbe, 0xa1, 0x6a, 0xf3, 0x5c,
	0x88, 0x3a, 0xcc, 0xdb, 0xe1, 0x10, 0x4e, 0xef,
	0x0c, 0xfd, 0xb5, 0x4e, 0x12, 0xfb, 0x23, 0x0a,
};

static const uint8_t aes_xts_256_vec9_ctxt[SECSIZE] = {
	0x32, 0x60, 0xae, 0x8d, 0xad, 0x1f, 0x4a, 0x32,
	0xc5, 0xca, 0xfe, 0x3a, 0xb0, 0xeb, 0x95, 0x54,
	0x9d, 0x46, 0x1a, 0x67, 0xce, 0xb9, 0xe5, 0xaa,
	0x2d, 0x3a, 0xfb, 0x62, 0xde, 0xce, 0x05, 0x53,
	0x19, 0x3b, 0xa5, 0x0c, 0x75, 0xbe, 0x25, 0x1e,
	0x08, 0xd1, 0xd0, 0x8f, 0x10, 0x88, 0x57, 0x6c,
	0x7e, 0xfd, 0xfa, 0xaf, 0x3f, 0x45, 0x95, 0x59,
	0x57, 0x1e, 0x12, 0x51, 0x17, 0x53, 0xb0, 0x7a,
	0xf0, 0x73, 0xf3, 0x5d, 0xa0, 0x6a, 0xf0, 0xce,
	0x0b, 0xbf, 0x6b, 0x8f, 0x5c, 0xcc, 0x5c, 0xea,
	0x50, 0x0e, 0xc1, 0xb2, 0x11, 0xbd, 0x51, 0xf6,
	0x3b, 0x60, 0x6b, 0xf6, 0x52, 0x87, 0x96, 0xca,
	0x12, 0x17, 0x3b, 0xa3, 0x9b, 0x89, 0x35, 0xee,
	0x44, 0xcc, 0xce, 0x64, 0x6f, 0x90, 0xa4, 0x5b,
	0xf9, 0xcc, 0xc5, 0x67, 0xf0, 0xac, 0xe1, 0x3d,
	0xc2, 0xd5, 0x3e, 0xbe, 0xed, 0xc8, 0x1f, 0x58,
	0xb2, 0xe4, 0x11, 0x79, 0xdd, 0xdf, 0x0d, 0x5a,
	0x5c, 0x42, 0xf5, 0xd8, 0x50, 0x6c, 0x1a, 0x5d,
	0x2f, 0x8f, 0x59, 0xf3, 0xea, 0x87, 0x3c, 0xbc,
	0xd0, 0xee, 0xc1, 0x9a, 0xcb, 0xf3, 0x25, 0x42,
	0x3b, 0xd3, 0xdc, 0xb8, 0xc2, 0xb1, 0xbf, 0x1d,
	0x1e, 0xae, 0xd0, 0xeb, 0xa7, 0xf0, 0x69, 0x8e,
	0x43, 0x14, 0xfb, 0xeb, 0x2f, 0x15, 0x66, 0xd1,
	0xb9, 0x25, 0x30, 0x08, 0xcb, 0xcc, 0xf4, 0x5a,
	0x2b, 0x0d, 0x9c, 0x5c, 0x9c, 0x21, 0x47, 0x4f,
	0x40, 0x76, 0xe0, 0x2b, 0xe2, 0x60, 0x50, 0xb9,
	0x9d, 0xee, 0x4f, 0xd6, 0x8a, 0x4c, 0xf8, 0x90,
	0xe4, 0x96, 0xe4, 0xfc, 0xae, 0x7b, 0x70, 0xf9,
	0x4e, 0xa5, 0xa9, 0x06, 0x2d, 0xa0, 0xda, 0xeb,
	0xa1, 0x99, 0x3d, 0x2c, 0xcd, 0x1d, 0xd3, 0xc2,
	0x44, 0xb8, 0x42, 0x88, 0x01, 0x49, 0x5a, 0x58,
	0xb2, 0x16, 0x54, 0x7e, 0x7e, 0x84, 0x7c, 0x46,
	0xd1, 0xd7, 0x56, 0x37, 0x7b, 0x62, 0x42, 0xd2,
	0xe5, 0xfb, 0x83, 0xbf, 0x75, 0x2b, 0x54, 0xe0,
	0xdf, 0x71, 0xe8, 0x89, 0xf3, 0xa2, 0xbb, 0x0f,
	0x4c, 0x10, 0x80, 0x5b, 0xf3, 0xc5, 0x90, 0x37,
	0x6e, 0x3c, 0x24, 0xe2, 0x2f, 0xf5, 0x7f, 0x7f,
	0xa9, 0x65, 0x57, 0x73, 0x75, 0x32, 0x5c, 0xea,
	0x5d, 0x92, 0x0d, 0xb9, 0x4b, 0x9c, 0x33, 0x6b,
	0x45, 0x5f, 0x6e, 0x89, 0x4c, 0x01, 0x86, 0x6f,
	0xe9, 0xfb, 0xb8, 0xc8, 0xd3, 0xf7, 0x0a, 0x29,
	0x57, 0x28, 0x5f, 0x6d, 0xfb, 0x5d, 0xcd, 0x8c,
	0xbf, 0x54, 0x78, 0x2f, 0x8f, 0xe7, 0x76, 0x6d,
	0x47, 0x23, 0x81, 0x99, 0x13, 0xac, 0x77, 0x34,
	0x21, 0xe3, 0xa3, 0x10, 0x95, 0x86, 0x6b, 0xad,
	0x22, 0xc8, 0x6a, 0x60, 0x36, 0xb2, 0x51, 0x8b,
	0x20, 0x59, 0xb4, 0x22, 0x9d, 0x18, 0xc8, 0xc2,
	0xcc, 0xbd, 0xf9, 0x06, 0xc6, 0xcc, 0x6e, 0x82,
	0x46, 0x4e, 0xe5, 0x7b, 0xdd, 0xb0, 0xbe, 0xbc,
	0xb1, 0xdc, 0x64, 0x53, 0x25, 0xbf, 0xb3, 0xe6,
	0x65, 0xef, 0x72, 0x51, 0x08, 0x2c, 0x88, 0xeb,
	0xb1, 0xcf, 0x20, 0x3b, 0xd7, 0x79, 0xfd, 0xd3,
	0x86, 0x75, 0x71, 0x3c, 0x8d, 0xaa, 0xdd, 0x17,
	0xe1, 0xca, 0xbe, 0xe4, 0x32, 0xb0, 0x97, 0x87,
	0xb6, 0xdd, 0xf3, 0x30, 0x4e, 0x38, 0xb7, 0x31,
	0xb4, 0x5d, 0xf5, 0xdf, 0x51, 0xb7, 0x8f, 0xcf,
	0xb3, 0xd3, 0x24, 0x66, 0x02, 0x8d, 0x0b, 0xa3,
	0x65, 0x55, 0xe7, 0xe1, 0x1a, 0xb0, 0xee, 0x06,
	0x66, 0x06, 0x1d, 0x16, 0x45, 0xd9, 0x62, 0x44,
	0x4b, 0xc4, 0x7a, 0x38, 0x18, 0x89, 0x30, 0xa8,
	0x4b, 0x4d, 0x56, 0x13, 0x95, 0xc7, 0x3c, 0x08,
	0x70, 0x21, 0x92, 0x7c, 0xa6, 0x38, 0xb7, 0xaf,
	0xc8, 0xa8, 0x67, 0x9c, 0xcb, 0x84, 0xc2, 0x65,
	0x55, 0x44, 0x0e, 0xc7, 0xf1, 0x04, 0x45, 0xcd,
};

const struct testvec aes_xts_256_vectors[] = {
	{
		.blkno = 0,
		.ptxt = aes_xts_256_vec4_ptxt,
		.ctxt = aes_xts_256_vec4_ctxt,
	},
	{
		.blkno = 1,
		.ptxt = aes_xts_256_vec5_ptxt,
		.ctxt = aes_xts_256_vec5_ctxt,
	},
	{
		.blkno = 2,
		.ptxt = aes_xts_256_vec6_ptxt,
		.ctxt = aes_xts_256_vec6_ctxt,
	},
	{
		.blkno = 0xfd,
		.ptxt = aes_xts_256_vec7_ptxt,
		.ctxt = aes_xts_256_vec7_ctxt,
	},
	{
		.blkno = 0xfe,
		.ptxt = aes_xts_256_vec8_ptxt,
		.ctxt = aes_xts_256_vec8_ctxt,
	},
	{
		.blkno = 0xff,
		.ptxt = aes_xts_256_vec9_ptxt,
		.ctxt = aes_xts_256_vec9_ctxt,
	},
};

/*
 * Vector 10 from IEEE 1619/D16, blkno 0xff.
 */
static const uint8_t aes_xts_512_vec10_ptxt[SECSIZE] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
	0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
	0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
	0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
	0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
	0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
	0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
	0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
	0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
	0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
	0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
	0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
	0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
	0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
	0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
	0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
	0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
	0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
	0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
	0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
	0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
	0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
	0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
	0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
	0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
	0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
	0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
	0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
	0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
	0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
	0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
	0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
	0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
	0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
	0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
	0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
	0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
	0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
	0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
	0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
	0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
	0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
	0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
	0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
	0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
	0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
	0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
	0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
	0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
	0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
	0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
	0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
	0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
};

static const uint8_t aes_xts_512_vec10_ctxt[SECSIZE] = {
	0x1c, 0x3b, 0x3a, 0x10, 0x2f, 0x77, 0x03, 0x86,
	0xe4, 0x83, 0x6c, 0x99, 0xe3, 0x70, 0xcf, 0x9b,
	0xea, 0x00, 0x80, 0x3f, 0x5e, 0x48, 0x23, 0x57,
	0xa4, 0xae, 0x12, 0xd4, 0x14, 0xa3, 0xe6, 0x3b,
	0x5d, 0x31, 0xe2, 0x76, 0xf8, 0xfe, 0x4a, 0x8d,
	0x66, 0xb3, 0x17, 0xf9, 0xac, 0x68, 0x3f, 0x44,
	0x68, 0x0a, 0x86, 0xac, 0x35, 0xad, 0xfc, 0x33,
	0x45, 0xbe, 0xfe, 0xcb, 0x4b, 0xb1, 0x88, 0xfd,
	0x57, 0x76, 0x92, 0x6c, 0x49, 0xa3, 0x09, 0x5e,
	0xb1, 0x08, 0xfd, 0x10, 0x98, 0xba, 0xec, 0x70,
	0xaa, 0xa6, 0x69, 0x99, 0xa7, 0x2a, 0x82, 0xf2,
	0x7d, 0x84, 0x8b, 0x21, 0xd4, 0xa7, 0x41, 0xb0,
	0xc5, 0xcd, 0x4d, 0x5f, 0xff, 0x9d, 0xac, 0x89,
	0xae, 0xba, 0x12, 0x29, 0x61, 0xd0, 0x3a, 0x75,
	0x71, 0x23, 0xe9, 0x87, 0x0f, 0x8a, 0xcf, 0x10,
	0x00, 0x02, 0x08, 0x87, 0x89, 0x14, 0x29, 0xca,
	0x2a, 0x3e, 0x7a, 0x7d, 0x7d, 0xf7, 0xb1, 0x03,
	0x55, 0x16, 0x5c, 0x8b, 0x9a, 0x6d, 0x0a, 0x7d,
	0xe8, 0xb0, 0x62, 0xc4, 0x50, 0x0d, 0xc4, 0xcd,
	0x12, 0x0c, 0x0f, 0x74, 0x18, 0xda, 0xe3, 0xd0,
	0xb5, 0x78, 0x1c, 0x34, 0x80, 0x3f, 0xa7, 0x54,
	0x21, 0xc7, 0x90, 0xdf, 0xe1, 0xde, 0x18, 0x34,
	0xf2, 0x80, 0xd7, 0x66, 0x7b, 0x32, 0x7f, 0x6c,
	0x8c, 0xd7, 0x55, 0x7e, 0x12, 0xac, 0x3a, 0x0f,
	0x93, 0xec, 0x05, 0xc5, 0x2e, 0x04, 0x93, 0xef,
	0x31, 0xa1, 0x2d, 0x3d, 0x92, 0x60, 0xf7, 0x9a,
	0x28, 0x9d, 0x6a, 0x37, 0x9b, 0xc7, 0x0c, 0x50,
	0x84, 0x14, 0x73, 0xd1, 0xa8, 0xcc, 0x81, 0xec,
	0x58, 0x3e, 0x96, 0x45, 0xe0, 0x7b, 0x8d, 0x96,
	0x70, 0x65, 0x5b, 0xa5, 0xbb, 0xcf, 0xec, 0xc6,
	0xdc, 0x39, 0x66, 0x38, 0x0a, 0xd8, 0xfe, 0xcb,
	0x17, 0xb6, 0xba, 0x02, 0x46, 0x9a, 0x02, 0x0a,
	0x84, 0xe1, 0x8e, 0x8f, 0x84, 0x25, 0x20, 0x70,
	0xc1, 0x3e, 0x9f, 0x1f, 0x28, 0x9b, 0xe5, 0x4f,
	0xbc, 0x48, 0x14, 0x57, 0x77, 0x8f, 0x61, 0x60,
	0x15, 0xe1, 0x32, 0x7a, 0x02, 0xb1, 0x40, 0xf1,
	0x50, 0x5e, 0xb3, 0x09, 0x32, 0x6d, 0x68, 0x37,
	0x8f, 0x83, 0x74, 0x59, 0x5c, 0x84, 0x9d, 0x84,
	0xf4, 0xc3, 0x33, 0xec, 0x44, 0x23, 0x88, 0x51,
	0x43, 0xcb, 0x47, 0xbd, 0x71, 0xc5, 0xed, 0xae,
	0x9b, 0xe6, 0x9a, 0x2f, 0xfe, 0xce, 0xb1, 0xbe,
	0xc9, 0xde, 0x24, 0x4f, 0xbe, 0x15, 0x99, 0x2b,
	0x11, 0xb7, 0x7c, 0x04, 0x0f, 0x12, 0xbd, 0x8f,
	0x6a, 0x97, 0x5a, 0x44, 0xa0, 0xf9, 0x0c, 0x29,
	0xa9, 0xab, 0xc3, 0xd4, 0xd8, 0x93, 0x92, 0x72,
	0x84, 0xc5, 0x87, 0x54, 0xcc, 0xe2, 0x94, 0x52,
	0x9f, 0x86, 0x14, 0xdc, 0xd2, 0xab, 0xa9, 0x91,
	0x92, 0x5f, 0xed, 0xc4, 0xae, 0x74, 0xff, 0xac,
	0x6e, 0x33, 0x3b, 0x93, 0xeb, 0x4a, 0xff, 0x04,
	0x79, 0xda, 0x9a, 0x41, 0x0e, 0x44, 0x50, 0xe0,
	0xdd, 0x7a, 0xe4, 0xc6, 0xe2, 0x91, 0x09, 0x00,
	0x57, 0x5d, 0xa4, 0x01, 0xfc, 0x07, 0x05, 0x9f,
	0x64, 0x5e, 0x8b, 0x7e, 0x9b, 0xfd, 0xef, 0x33,
	0x94, 0x30, 0x54, 0xff, 0x84, 0x01, 0x14, 0x93,
	0xc2, 0x7b, 0x34, 0x29, 0xea, 0xed, 0xb4, 0xed,
	0x53, 0x76, 0x44, 0x1a, 0x77, 0xed, 0x43, 0x85,
	0x1a, 0xd7, 0x7f, 0x16, 0xf5, 0x41, 0xdf, 0xd2,
	0x69, 0xd5, 0x0d, 0x6a, 0x5f, 0x14, 0xfb, 0x0a,
	0xab, 0x1c, 0xbb, 0x4c, 0x15, 0x50, 0xbe, 0x97,
	0xf7, 0xab, 0x40, 0x66, 0x19, 0x3c, 0x4c, 0xaa,
	0x77, 0x3d, 0xad, 0x38, 0x01, 0x4b, 0xd2, 0x09,
	0x2f, 0xa7, 0x55, 0xc8, 0x24, 0xbb, 0x5e, 0x54,
	0xc4, 0xf3, 0x6f, 0xfd, 0xa9, 0xfc, 0xea, 0x70,
	0xb9, 0xc6, 0xe6, 0x93, 0xe1, 0x48, 0xc1, 0x51,
};

/*
 * Vector 11 from IEEE 1619/D16, blkno 0xffff.
 */
static const uint8_t aes_xts_512_vec11_ptxt[SECSIZE] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
	0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
	0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
	0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
	0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
	0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
	0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
	0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
	0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
	0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
	0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
	0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
	0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
	0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
	0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
	0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
	0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
	0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
	0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
	0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
	0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
	0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
	0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
	0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
	0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
	0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
	0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
	0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
	0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
	0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
	0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
	0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
	0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
	0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
	0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
	0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
	0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
	0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
	0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
	0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
	0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
	0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
	0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
	0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
	0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
	0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
	0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
	0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
	0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
	0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
	0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
	0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
	0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
	0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
	0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
};

static const uint8_t aes_xts_512_vec11_ctxt[SECSIZE] = {
	0x77, 0xa3, 0x12, 0x51, 0x61, 0x8a, 0x15, 0xe6,
	0xb9, 0x2d, 0x1d, 0x66, 0xdf, 0xfe, 0x7b, 0x50,
	0xb5, 0x0b, 0xad, 0x55, 0x23, 0x05, 0xba, 0x02,
	0x17, 0xa6, 0x10, 0x68, 0x8e, 0xff, 0x7e, 0x11,
	0xe1, 0xd0, 0x22, 0x54, 0x38, 0xe0, 0x93, 0x24,
	0x2d, 0x6d, 0xb2, 0x74, 0xfd, 0xe8, 0x01, 0xd4,
	0xca, 0xe0, 0x6f, 0x20, 0x92, 0xc7, 0x28, 0xb2,
	0x47, 0x85, 0x59, 0xdf, 0x58, 0xe8, 0x37, 0xc2,
	0x46, 0x9e, 0xe4, 0xa4, 0xfa, 0x79, 0x4e, 0x4b,
	0xbc, 0x7f, 0x39, 0xbc, 0x02, 0x6e, 0x3c, 0xb7,
	0x2c, 0x33, 0xb0, 0x88, 0x8f, 0x25, 0xb4, 0xac,
	0xf5, 0x6a, 0x2a, 0x98, 0x04, 0xf1, 0xce, 0x6d,
	0x3d, 0x6e, 0x1d, 0xc6, 0xca, 0x18, 0x1d, 0x4b,
	0x54, 0x61, 0x79, 0xd5, 0x55, 0x44, 0xaa, 0x77,
	0x60, 0xc4, 0x0d, 0x06, 0x74, 0x15, 0x39, 0xc7,
	0xe3, 0xcd, 0x9d, 0x2f, 0x66, 0x50, 0xb2, 0x01,
	0x3f, 0xd0, 0xee, 0xb8, 0xc2, 0xb8, 0xe3, 0xd8,
	0xd2, 0x40, 0xcc, 0xae, 0x2d, 0x4c, 0x98, 0x32,
	0x0a, 0x74, 0x42, 0xe1, 0xc8, 0xd7, 0x5a, 0x42,
	0xd6, 0xe6, 0xcf, 0xa4, 0xc2, 0xec, 0xa1, 0x79,
	0x8d, 0x15, 0x8c, 0x7a, 0xec, 0xdf, 0x82, 0x49,
	0x0f, 0x24, 0xbb, 0x9b, 0x38, 0xe1, 0x08, 0xbc,
	0xda, 0x12, 0xc3, 0xfa, 0xf9, 0xa2, 0x11, 0x41,
	0xc3, 0x61, 0x3b, 0x58, 0x36, 0x7f, 0x92, 0x2a,
	0xaa, 0x26, 0xcd, 0x22, 0xf2, 0x3d, 0x70, 0x8d,
	0xae, 0x69, 0x9a, 0xd7, 0xcb, 0x40, 0xa8, 0xad,
	0x0b, 0x6e, 0x27, 0x84, 0x97, 0x3d, 0xcb, 0x60,
	0x56, 0x84, 0xc0, 0x8b, 0x8d, 0x69, 0x98, 0xc6,
	0x9a, 0xac, 0x04, 0x99, 0x21, 0x87, 0x1e, 0xbb,
	0x65, 0x30, 0x1a, 0x46, 0x19, 0xca, 0x80, 0xec,
	0xb4, 0x85, 0xa3, 0x1d, 0x74, 0x42, 0x23, 0xce,
	0x8d, 0xdc, 0x23, 0x94, 0x82, 0x8d, 0x6a, 0x80,
	0x47, 0x0c, 0x09, 0x2f, 0x5b, 0xa4, 0x13, 0xc3,
	0x37, 0x8f, 0xa6, 0x05, 0x42, 0x55, 0xc6, 0xf9,
	0xdf, 0x44, 0x95, 0x86, 0x2b, 0xbb, 0x32, 0x87,
	0x68, 0x1f, 0x93, 0x1b, 0x68, 0x7c, 0x88, 0x8a,
	0xbf, 0x84, 0x4d, 0xfc, 0x8f, 0xc2, 0x83, 0x31,
	0xe5, 0x79, 0x92, 0x8c, 0xd1, 0x2b, 0xd2, 0x39,
	0x0a, 0xe1, 0x23, 0xcf, 0x03, 0x81, 0x8d, 0x14,
	0xde, 0xdd, 0xe5, 0xc0, 0xc2, 0x4c, 0x8a, 0xb0,
	0x18, 0xbf, 0xca, 0x75, 0xca, 0x09, 0x6f, 0x2d,
	0x53, 0x1f, 0x3d, 0x16, 0x19, 0xe7, 0x85, 0xf1,
	0xad, 0xa4, 0x37, 0xca, 0xb9, 0x2e, 0x98, 0x05,
	0x58, 0xb3, 0xdc, 0xe1, 0x47, 0x4a, 0xfb, 0x75,
	0xbf, 0xed, 0xbf, 0x8f, 0xf5, 0x4c, 0xb2, 0x61,
	0x8e, 0x02, 0x44, 0xc9, 0xac, 0x0d, 0x3c, 0x66,
	0xfb, 0x51, 0x59, 0x8c, 0xd2, 0xdb, 0x11, 0xf9,
	0xbe, 0x39, 0x79, 0x1a, 0xbe, 0x44, 0x7c, 0x63,
	0x09, 0x4f, 0x7c, 0x45, 0x3b, 0x7f, 0xf8, 0x7c,
	0xb5, 0xbb, 0x36, 0xb7, 0xc7, 0x9e, 0xfb, 0x08,
	0x72, 0xd1, 0x70, 0x58, 0xb8, 0x3b, 0x15, 0xab,
	0x08, 0x66, 0xad, 0x8a, 0x58, 0x65, 0x6c, 0x5a,
	0x7e, 0x20, 0xdb, 0xdf, 0x30, 0x8b, 0x24, 0x61,
	0xd9, 0x7c, 0x0e, 0xc0, 0x02, 0x4a, 0x27, 0x15,
	0x05, 0x52, 0x49, 0xcf, 0x3b, 0x47, 0x8d, 0xdd,
	0x47, 0x40, 0xde, 0x65, 0x4f, 0x75, 0xca, 0x68,
	0x6e, 0x0d, 0x73, 0x45, 0xc6, 0x9e, 0xd5, 0x0c,
	0xdc, 0x2a, 0x8b, 0x33, 0x2b, 0x1f, 0x88, 0x24,
	0x10, 0x8a, 0xc9, 0x37, 0xeb, 0x05, 0x05, 0x85,
	0x60, 0x8e, 0xe7, 0x34, 0x09, 0x7f, 0xc0, 0x90,
	0x54, 0xfb, 0xff, 0x89, 0xee, 0xae, 0xea, 0x79,
	0x1f, 0x4a, 0x7a, 0xb1, 0xf9, 0x86, 0x82, 0x94,
	0xa4, 0xf9, 0xe2, 0x7b, 0x42, 0xaf, 0x81, 0x00,
	0xcb, 0x9d, 0x59, 0xce, 0xf9, 0x64, 0x58, 0x03,
};

const struct testvec aes_xts_512_vectors[] = {
	{
		.blkno = 0xff,
		.ptxt = aes_xts_512_vec10_ptxt,
		.ctxt = aes_xts_512_vec10_ctxt,
	},
	{
		.blkno = 0xffff,
		.ptxt = aes_xts_512_vec11_ptxt,
		.ctxt = aes_xts_512_vec11_ctxt,
	},
};

static int
open_disk(const char *devpath, const char *imgpath, size_t size)
{
	int fd;

	fd = open(imgpath, O_CREAT | O_RDWR | O_TRUNC, 0600);
	if (fd < 0)
		return -1;

	if (ftruncate(fd, size) < 0)
		goto fail;

	if (rump_pub_etfs_register_withsize(devpath,
	    imgpath, RUMP_ETFS_BLK, 0, size) < 0) {
		goto fail;
	}

	unlink(imgpath);
	return fd;
fail:
	close(fd);
	unlink(imgpath);
	return -1;
}

static int
open_cgd(int devno)
{
	char devpath[32];

	sprintf(devpath, "/dev/rcgd%d%c", devno, getrawpartition() + 'a');

	return rump_sys_open(devpath, O_RDWR, 0);
}

static int
configure_cgd(int fd, const char *dkpath, const char *alg,
    const char *ivmethod, const char *key, size_t keylen)
{
	struct cgd_ioctl ci;

	memset(&ci, 0, sizeof(ci));
	ci.ci_disk = dkpath;
	ci.ci_alg = alg;
	ci.ci_ivmethod = ivmethod;
	ci.ci_keylen = 8 * keylen - 8; /* Exclude the NUL terminator. */
	ci.ci_key = key;
	ci.ci_blocksize = 128;

	return rump_sys_ioctl(fd, CGDIOCSET, &ci);
}

static int
unconfigure_cgd(int fd)
{
	struct cgd_ioctl ci;

	return rump_sys_ioctl(fd, CGDIOCCLR, &ci);
}

static int
write_testvec(int cgdfd, const struct testvec *tv)
{
	ssize_t written;

	if (rump_sys_lseek(cgdfd, tv->blkno * SECSIZE, SEEK_SET) < 0)
		return -1;

	written = rump_sys_write(cgdfd, tv->ptxt, SECSIZE);
	if (written < 0)
		return -1;
	if (written != SECSIZE) {
		errno = EDOM; /* Something distinct. */
		return -1;
	}

	return 0;
}

static int
read_testvec(int cgdfd, const struct testvec *tv)
{
	char *buf;
	int res = -1;

	buf = malloc(SECSIZE);
	if (buf == NULL)
		return -1;

	if (rump_sys_lseek(cgdfd, tv->blkno * SECSIZE, SEEK_SET) < 0)
		goto fail;

	if (rump_sys_read(cgdfd, buf, SECSIZE) != SECSIZE)
		goto fail;

	res = memcmp(buf, tv->ptxt, SECSIZE);
fail:
	free(buf);
	return res;
}

static int
check_testvec(int dkfd, const struct testvec *tv)
{
	char *buf;
	int res = -1;

	buf = malloc(SECSIZE);
	if (buf == NULL)
		return -1;

	if (lseek(dkfd, tv->blkno * SECSIZE, SEEK_SET) < 0)
		goto fail;

	if (read(dkfd, buf, SECSIZE) != SECSIZE)
		goto fail;

	res = memcmp(buf, tv->ctxt, SECSIZE);
fail:
	free(buf);
	return res;
}

ATF_TC(cgd_aes_cbc_128_encblkno1);
ATF_TC_HEAD(cgd_aes_cbc_128_encblkno1, tc)
{
	atf_tc_set_md_var(tc, "descr",
	    "Test aes-cbc with 128 bits key, ivmethod encblkno1");
}

ATF_TC_BODY(cgd_aes_cbc_128_encblkno1, tc)
{
	const char imgpath[] = "aes-cbc-128-encblkno1.img";
	const char *dkpath = "/dev/dk";
	const size_t dksize = 4 * SECSIZE; /* Last blkno is 3. */
	int dkfd, cgdfd;

	rump_init();

	RL(dkfd = open_disk(dkpath, imgpath, dksize));

	RL(cgdfd = open_cgd(0));
	RL(configure_cgd(cgdfd, dkpath, "aes-cbc", "encblkno1",
	    aes_cbc_128_key, sizeof(aes_cbc_128_key)));

	CHECK_LIBC(write_testvec(cgdfd, &aes_cbc_128_1_vectors[0]), -1);
	CHECK_LIBC(write_testvec(cgdfd, &aes_cbc_128_1_vectors[1]), -1);
	CHECK_LIBC(write_testvec(cgdfd, &aes_cbc_128_1_vectors[2]), -1);
	CHECK_LIBC(write_testvec(cgdfd, &aes_cbc_128_1_vectors[3]), -1);

	RL(unconfigure_cgd(cgdfd));
	RL(configure_cgd(cgdfd, dkpath, "aes-cbc", "encblkno1",
	    aes_cbc_128_key, sizeof(aes_cbc_128_key)));

	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_cbc_128_1_vectors[0]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_cbc_128_1_vectors[1]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_cbc_128_1_vectors[2]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_cbc_128_1_vectors[3]), 0);

	RL(unconfigure_cgd(cgdfd));
	RL(rump_sys_close(cgdfd));

	ATF_CHECK_EQ(check_testvec(dkfd, &aes_cbc_128_1_vectors[0]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &aes_cbc_128_1_vectors[1]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &aes_cbc_128_1_vectors[2]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &aes_cbc_128_1_vectors[3]), 0);

	RL(close(dkfd));
}

ATF_TC(cgd_aes_cbc_128_encblkno8);
ATF_TC_HEAD(cgd_aes_cbc_128_encblkno8, tc)
{
	atf_tc_set_md_var(tc, "descr",
	    "Test aes-cbc with 128 bits key, ivmethod encblkno8");
}

ATF_TC_BODY(cgd_aes_cbc_128_encblkno8, tc)
{
	const char imgpath[] = "aes-cbc-128-encblkno8.img";
	const char *dkpath = "/dev/dk";
	const size_t dksize = 4 * SECSIZE; /* Last blkno is 3. */
	int dkfd, cgdfd;

	rump_init();

	RL(dkfd = open_disk(dkpath, imgpath, dksize));

	RL(cgdfd = open_cgd(0));
	RL(configure_cgd(cgdfd, dkpath, "aes-cbc", "encblkno8",
	    aes_cbc_128_key, sizeof(aes_cbc_128_key)));

	CHECK_LIBC(write_testvec(cgdfd, &aes_cbc_128_8_vectors[0]), -1);
	CHECK_LIBC(write_testvec(cgdfd, &aes_cbc_128_8_vectors[1]), -1);
	CHECK_LIBC(write_testvec(cgdfd, &aes_cbc_128_8_vectors[2]), -1);
	CHECK_LIBC(write_testvec(cgdfd, &aes_cbc_128_8_vectors[3]), -1);

	RL(unconfigure_cgd(cgdfd));
	RL(configure_cgd(cgdfd, dkpath, "aes-cbc", "encblkno8",
	    aes_cbc_128_key, sizeof(aes_cbc_128_key)));

	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_cbc_128_8_vectors[0]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_cbc_128_8_vectors[1]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_cbc_128_8_vectors[2]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_cbc_128_8_vectors[3]), 0);

	RL(unconfigure_cgd(cgdfd));
	RL(rump_sys_close(cgdfd));

	ATF_CHECK_EQ(check_testvec(dkfd, &aes_cbc_128_8_vectors[0]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &aes_cbc_128_8_vectors[1]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &aes_cbc_128_8_vectors[2]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &aes_cbc_128_8_vectors[3]), 0);

	RL(close(dkfd));
}

ATF_TC(cgd_aes_cbc_192_encblkno1);
ATF_TC_HEAD(cgd_aes_cbc_192_encblkno1, tc)
{
	atf_tc_set_md_var(tc, "descr",
	    "Test aes-cbc with 192 bits key, ivmethod encblkno1");
}

ATF_TC_BODY(cgd_aes_cbc_192_encblkno1, tc)
{
	const char imgpath[] = "aes-cbc-192-encblkno1.img";
	const char *dkpath = "/dev/dk";
	const size_t dksize = 4 * SECSIZE; /* Last blkno is 3. */
	int dkfd, cgdfd;

	rump_init();

	RL(dkfd = open_disk(dkpath, imgpath, dksize));

	RL(cgdfd = open_cgd(0));
	RL(configure_cgd(cgdfd, dkpath, "aes-cbc", "encblkno1",
	    aes_cbc_192_key, sizeof(aes_cbc_192_key)));

	CHECK_LIBC(write_testvec(cgdfd, &aes_cbc_192_1_vectors[0]), -1);
	CHECK_LIBC(write_testvec(cgdfd, &aes_cbc_192_1_vectors[1]), -1);
	CHECK_LIBC(write_testvec(cgdfd, &aes_cbc_192_1_vectors[2]), -1);
	CHECK_LIBC(write_testvec(cgdfd, &aes_cbc_192_1_vectors[3]), -1);

	RL(unconfigure_cgd(cgdfd));
	RL(configure_cgd(cgdfd, dkpath, "aes-cbc", "encblkno1",
	    aes_cbc_192_key, sizeof(aes_cbc_192_key)));

	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_cbc_192_1_vectors[0]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_cbc_192_1_vectors[1]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_cbc_192_1_vectors[2]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_cbc_192_1_vectors[3]), 0);

	RL(unconfigure_cgd(cgdfd));
	RL(rump_sys_close(cgdfd));

	ATF_CHECK_EQ(check_testvec(dkfd, &aes_cbc_192_1_vectors[0]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &aes_cbc_192_1_vectors[1]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &aes_cbc_192_1_vectors[2]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &aes_cbc_192_1_vectors[3]), 0);

	RL(close(dkfd));
}

ATF_TC(cgd_aes_cbc_192_encblkno8);
ATF_TC_HEAD(cgd_aes_cbc_192_encblkno8, tc)
{
	atf_tc_set_md_var(tc, "descr",
	    "Test aes-cbc with 192 bits key, ivmethod encblkno8");
}

ATF_TC_BODY(cgd_aes_cbc_192_encblkno8, tc)
{
	const char imgpath[] = "aes-cbc-192-encblkno8.img";
	const char *dkpath = "/dev/dk";
	const size_t dksize = 4 * SECSIZE; /* Last blkno is 3. */
	int dkfd, cgdfd;

	rump_init();

	RL(dkfd = open_disk(dkpath, imgpath, dksize));

	RL(cgdfd = open_cgd(0));
	RL(configure_cgd(cgdfd, dkpath, "aes-cbc", "encblkno8",
	    aes_cbc_192_key, sizeof(aes_cbc_192_key)));

	CHECK_LIBC(write_testvec(cgdfd, &aes_cbc_192_8_vectors[0]), -1);
	CHECK_LIBC(write_testvec(cgdfd, &aes_cbc_192_8_vectors[1]), -1);
	CHECK_LIBC(write_testvec(cgdfd, &aes_cbc_192_8_vectors[2]), -1);
	CHECK_LIBC(write_testvec(cgdfd, &aes_cbc_192_8_vectors[3]), -1);

	RL(unconfigure_cgd(cgdfd));
	RL(configure_cgd(cgdfd, dkpath, "aes-cbc", "encblkno8",
	    aes_cbc_192_key, sizeof(aes_cbc_192_key)));

	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_cbc_192_8_vectors[0]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_cbc_192_8_vectors[1]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_cbc_192_8_vectors[2]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_cbc_192_8_vectors[3]), 0);

	RL(unconfigure_cgd(cgdfd));
	RL(rump_sys_close(cgdfd));

	ATF_CHECK_EQ(check_testvec(dkfd, &aes_cbc_192_8_vectors[0]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &aes_cbc_192_8_vectors[1]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &aes_cbc_192_8_vectors[2]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &aes_cbc_192_8_vectors[3]), 0);

	RL(close(dkfd));
}

ATF_TC(cgd_aes_cbc_256_encblkno1);
ATF_TC_HEAD(cgd_aes_cbc_256_encblkno1, tc)
{
	atf_tc_set_md_var(tc, "descr",
	    "Test aes-cbc with 256 bits key, ivmethod encblkno1");
}

ATF_TC_BODY(cgd_aes_cbc_256_encblkno1, tc)
{
	const char imgpath[] = "aes-cbc-256-encblkno1.img";
	const char *dkpath = "/dev/dk";
	const size_t dksize = 4 * SECSIZE; /* Last blkno is 3. */
	int dkfd, cgdfd;

	rump_init();

	RL(dkfd = open_disk(dkpath, imgpath, dksize));

	RL(cgdfd = open_cgd(0));
	RL(configure_cgd(cgdfd, dkpath, "aes-cbc", "encblkno1",
	    aes_cbc_256_key, sizeof(aes_cbc_256_key)));

	CHECK_LIBC(write_testvec(cgdfd, &aes_cbc_256_1_vectors[0]), -1);
	CHECK_LIBC(write_testvec(cgdfd, &aes_cbc_256_1_vectors[1]), -1);
	CHECK_LIBC(write_testvec(cgdfd, &aes_cbc_256_1_vectors[2]), -1);
	CHECK_LIBC(write_testvec(cgdfd, &aes_cbc_256_1_vectors[3]), -1);

	RL(unconfigure_cgd(cgdfd));
	RL(configure_cgd(cgdfd, dkpath, "aes-cbc", "encblkno1",
	    aes_cbc_256_key, sizeof(aes_cbc_256_key)));

	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_cbc_256_1_vectors[0]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_cbc_256_1_vectors[1]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_cbc_256_1_vectors[2]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_cbc_256_1_vectors[3]), 0);

	RL(unconfigure_cgd(cgdfd));
	RL(rump_sys_close(cgdfd));

	ATF_CHECK_EQ(check_testvec(dkfd, &aes_cbc_256_1_vectors[0]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &aes_cbc_256_1_vectors[1]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &aes_cbc_256_1_vectors[2]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &aes_cbc_256_1_vectors[3]), 0);

	RL(close(dkfd));
}

ATF_TC(cgd_aes_cbc_256_encblkno8);
ATF_TC_HEAD(cgd_aes_cbc_256_encblkno8, tc)
{
	atf_tc_set_md_var(tc, "descr",
	    "Test aes-cbc with 256 bits key, ivmethod encblkno8");
}

ATF_TC_BODY(cgd_aes_cbc_256_encblkno8, tc)
{
	const char imgpath[] = "aes-cbc-256-encblkno8.img";
	const char *dkpath = "/dev/dk";
	const size_t dksize = 4 * SECSIZE; /* Last blkno is 3. */
	int dkfd, cgdfd;

	rump_init();

	RL(dkfd = open_disk(dkpath, imgpath, dksize));

	RL(cgdfd = open_cgd(0));
	RL(configure_cgd(cgdfd, dkpath, "aes-cbc", "encblkno8",
	    aes_cbc_256_key, sizeof(aes_cbc_256_key)));

	CHECK_LIBC(write_testvec(cgdfd, &aes_cbc_256_8_vectors[0]), -1);
	CHECK_LIBC(write_testvec(cgdfd, &aes_cbc_256_8_vectors[1]), -1);
	CHECK_LIBC(write_testvec(cgdfd, &aes_cbc_256_8_vectors[2]), -1);
	CHECK_LIBC(write_testvec(cgdfd, &aes_cbc_256_8_vectors[3]), -1);

	RL(unconfigure_cgd(cgdfd));
	RL(configure_cgd(cgdfd, dkpath, "aes-cbc", "encblkno8",
	    aes_cbc_256_key, sizeof(aes_cbc_256_key)));

	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_cbc_256_8_vectors[0]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_cbc_256_8_vectors[1]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_cbc_256_8_vectors[2]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_cbc_256_8_vectors[3]), 0);

	RL(unconfigure_cgd(cgdfd));
	RL(rump_sys_close(cgdfd));

	ATF_CHECK_EQ(check_testvec(dkfd, &aes_cbc_256_8_vectors[0]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &aes_cbc_256_8_vectors[1]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &aes_cbc_256_8_vectors[2]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &aes_cbc_256_8_vectors[3]), 0);

	RL(close(dkfd));
}

ATF_TC(cgd_aes_xts_256);
ATF_TC_HEAD(cgd_aes_xts_256, tc)
{
	atf_tc_set_md_var(tc, "descr", "Test aes-xts with 256 bits key");
}

ATF_TC_BODY(cgd_aes_xts_256, tc)
{
	const char imgpath[] = "aes-xts-256.img";
	const char *dkpath = "/dev/dk";
	const size_t dksize = 256 * SECSIZE; /* Last blkno is 0xff. */
	int dkfd, cgdfd;

	rump_init();

	RL(dkfd = open_disk(dkpath, imgpath, dksize));

	RL(cgdfd = open_cgd(0));
	RL(configure_cgd(cgdfd, dkpath, "aes-xts", "encblkno1",
	    aes_xts_256_key, sizeof(aes_xts_256_key)));

	CHECK_LIBC(write_testvec(cgdfd, &aes_xts_256_vectors[0]), -1);
	CHECK_LIBC(write_testvec(cgdfd, &aes_xts_256_vectors[1]), -1);
	CHECK_LIBC(write_testvec(cgdfd, &aes_xts_256_vectors[2]), -1);
	CHECK_LIBC(write_testvec(cgdfd, &aes_xts_256_vectors[3]), -1);
	CHECK_LIBC(write_testvec(cgdfd, &aes_xts_256_vectors[4]), -1);
	CHECK_LIBC(write_testvec(cgdfd, &aes_xts_256_vectors[5]), -1);

	RL(unconfigure_cgd(cgdfd));
	RL(configure_cgd(cgdfd, dkpath, "aes-xts", "encblkno1",
	    aes_xts_256_key, sizeof(aes_xts_256_key)));

	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_xts_256_vectors[0]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_xts_256_vectors[1]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_xts_256_vectors[2]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_xts_256_vectors[3]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_xts_256_vectors[4]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_xts_256_vectors[5]), 0);

	RL(unconfigure_cgd(cgdfd));
	RL(rump_sys_close(cgdfd));

	ATF_CHECK_EQ(check_testvec(dkfd, &aes_xts_256_vectors[0]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &aes_xts_256_vectors[1]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &aes_xts_256_vectors[2]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &aes_xts_256_vectors[3]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &aes_xts_256_vectors[4]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &aes_xts_256_vectors[5]), 0);

	RL(close(dkfd));
}

ATF_TC(cgd_aes_xts_512);
ATF_TC_HEAD(cgd_aes_xts_512, tc)
{
	atf_tc_set_md_var(tc, "descr", "Test aes-xts with 512 bits key");
}

ATF_TC_BODY(cgd_aes_xts_512, tc)
{
	const char imgpath[] = "aes-xts-512.img";
	const char *dkpath = "/dev/dk";
	const size_t dksize = 65536 * SECSIZE; /* Last blkno is 0xffff. */
	int dkfd, cgdfd;

	rump_init();

	dkfd = open_disk(dkpath, imgpath, dksize);
	if (dkfd == -1 && errno == ENOSPC) {
		atf_tc_skip("not enough space");
	} else {
		ATF_CHECK_MSG(dkfd != -1, "open_disk: %s", strerror(errno));
	}

	RL(cgdfd = open_cgd(0));
	RL(configure_cgd(cgdfd, dkpath, "aes-xts", "encblkno1",
	    aes_xts_512_key, sizeof(aes_xts_512_key)));

	CHECK_LIBC(write_testvec(cgdfd, &aes_xts_512_vectors[0]), -1);
	CHECK_LIBC(write_testvec(cgdfd, &aes_xts_512_vectors[1]), -1);

	RL(unconfigure_cgd(cgdfd));
	RL(configure_cgd(cgdfd, dkpath, "aes-xts", "encblkno1",
	    aes_xts_512_key, sizeof(aes_xts_512_key)));

	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_xts_512_vectors[0]), 0);
	ATF_CHECK_EQ(read_testvec(cgdfd, &aes_xts_512_vectors[1]), 0);

	RL(unconfigure_cgd(cgdfd));
	RL(rump_sys_close(cgdfd));

	ATF_CHECK_EQ(check_testvec(dkfd, &aes_xts_512_vectors[0]), 0);
	ATF_CHECK_EQ(check_testvec(dkfd, &aes_xts_512_vectors[1]), 0);

	RL(close(dkfd));
}

ATF_TP_ADD_TCS(tp)
{

	ATF_TP_ADD_TC(tp, cgd_aes_cbc_128_encblkno1);
	ATF_TP_ADD_TC(tp, cgd_aes_cbc_128_encblkno8);
	ATF_TP_ADD_TC(tp, cgd_aes_cbc_192_encblkno1);
	ATF_TP_ADD_TC(tp, cgd_aes_cbc_192_encblkno8);
	ATF_TP_ADD_TC(tp, cgd_aes_cbc_256_encblkno1);
	ATF_TP_ADD_TC(tp, cgd_aes_cbc_256_encblkno8);
	ATF_TP_ADD_TC(tp, cgd_aes_xts_256);
	ATF_TP_ADD_TC(tp, cgd_aes_xts_512);

	return atf_no_error();
}
