/***************************************************************************
    Description          : KPuzzle - A KDE Jigsaw Puzzle Game
    Version              : 0.2
    Copyright            : (C) 2000-2001 by Michael Wand
    EMail                : mwand@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "main.h"

#include <qpixmap.h>
#include <qpaintdevice.h>
#include <qpainter.h>
#include <qcolor.h>
#include <qsize.h>
#include <qwmatrix.h>
#include <qbitmap.h>
//#include <qmessagebox.h>
//#include <qfiledialog.h>
#include <qdialog.h>
#include <qfile.h>
#include <qdatastream.h>
#include <qdir.h>

#include <kapp.h>
#include <kfiledialog.h>
#include <kmessagebox.h>

#include "kpuzzleapp.h"

#include "kpuzzle.h"
#include "kpuzzleview.h"
#include "gamewidget.h"
#include "piecewidget.h"
#include "picview.h"
#include "gamedialog.h"
#include "savedialog.h"
#include "highscore.h"
#include "piece.h"
#include "prefs.h"

#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <stdlib.h>

// Why to change a score
#define SC_PIECE_RIGHT 1
#define SC_PIECE_WRONG 2
#define SC_COUNT_TIME 3
#define SC_BONUS 4

bool ScoreData::save(QDataStream*) const
{
        return true;
}

bool ScoreData::load(QDataStream*)
{
        return true;
}

bool GameData::save(QDataStream* f) const
{
//         QString base(_filename.right(_filename.length() - (_filename.findRev('/') + 1)));
//         QString dir(_filename.left(_filename.findRev('/') + 1));
//         debug("Filename=%s,Base=%s,Dir=%s",(const char*) _filename,(const char*) base,(const char*) dir);
//         *f << base;
//         *f << dir;
        *f << _filename;
        *f << (Q_INT8) _savePixmap;
        *f << _originalPixmapSize;
        *f << _annotation;

        *f << (Q_INT8) _gameType;
        *f << (Q_INT8) _difficulty;
        *f << (Q_INT8) _scale;
        *f << (Q_INT8) _useMask;
        *f << (Q_INT8) _dialog_pieceSize;
        *f << _displace;
        *f << _pieceSize;
        *f << _piecesCount;
//        *f << _mainPixmap;
        debug("Filename=%s",(const char*) _filename);
        debug("GameData::save(): _gameType = %d, _difficulty = %d, _scale = %d, _useMask = %d, _dialog_pieceSize = %d",
              _gameType,_difficulty,_scale,_useMask,_dialog_pieceSize);
        debug("   _displace = %d, _pieceSize = %d;%d, _piecesCount = %d;%d",_displace,_pieceSize.width(),_pieceSize.height(),
              _piecesCount.width(),_piecesCount.height());
        return true;
}

bool GameData::load(QDataStream* f)
{
//         QString base;
//         QString dir;
//         *f >> base;
//         *f >> dir;
//         _filename = base + dir;

        *f >> _filename;
        *f >> (Q_INT8&) _savePixmap;
        *f >> _originalPixmapSize;
        *f >> _annotation;

        *f >> (Q_INT8&) _gameType;
        *f >> (Q_INT8&) _difficulty;
        *f >> (Q_INT8&) _scale;
        *f >> (Q_INT8&) _useMask;
        *f >> (Q_INT8&) _dialog_pieceSize;
        *f >> _displace;
        *f >> _pieceSize;
        *f >> _piecesCount;

        debug("Filename=%s",(const char*) _filename);
        debug("GameData::load(): _gameType = %d, _difficulty = %d, _scale = %d, _useMask = %d, _dialog_pieceSize = %d",
              _gameType,_difficulty,_scale,_useMask,_dialog_pieceSize);
        debug("   _displace = %d, _pieceSize = %d;%d, _piecesCount = %d;%d",_displace,_pieceSize.width(),_pieceSize.height(),
              _piecesCount.width(),_piecesCount.height());
//        debug("   _displace = %d, _pieceSize = %d, _piecesCount = %d",_displace,_pieceSize,_piecesCount);
//        *f >> _mainPixmap;

//         debug(QDir::homeDirPath());
//         // Try to find the pixmap in the same directory
//         QPixmap pm;
//         if (!(QFile::exists(dir + base) && pm.load(dir + base))) // pixmap cannot be loaded from the original place
//                 if (!(QFile::exists(PUZZLE_DIR + "levels/" + base) && pm.load(PUZZLE_DIR + "levels/" + base))) // ...from standard place
//                         if (!(QFile::exists(QDir::homeDirPath() + base) && pm.load(QDir::homeDirPath() + base))) {
//                                 QString dialTitle;
//                                 dialTitle.sprintf("Select picture file with size %d, %d",ctrlPixmapSize.width(),ctrlPixmapSize.height());
//                                 QString newFileName =
//                                         KFileDialog::getOpenFileName(PUZZLE_DIR + "levels/","*.bmp",0,dialTitle);
//                                 if (!(QFile::exists(newFileName) && pm.load(newFileName))) {
//                                         debug("error finding pixmap");
//                                         return;
//                                 }
//                         }
//         // Pixmap is now loaded
//         if (pm.size() != ctrlPixmapSize) {
//                 debug("error in pixmap size");
//                 return;
//         }

        return true;
}

bool CurrentData::save(QDataStream* f) const
{
        Q_INT32 currentPieceNr=-1; // this is saved to make it possible to restore the current item
        int i; // iterator

        *f << _score;
        *f << _faults;
        *f << _rounds;
        *f << _elapsed;
        *f << (Q_INT8)_dirty;
        Q_INT32 cnt = _pieceList->count();
        *f << cnt;
        // _running need not be saved, as it is always true

        QListIterator<CPiece> savingIterator(*_pieceList);
        ASSERT(savingIterator.count() > 0);
        int t_c = 0;
        for(i = 0,savingIterator.toFirst();savingIterator.current() != NULL;++i,++savingIterator) {
                savingIterator.current()->save(f);
                if (_currentPieceData == savingIterator.current()) currentPieceNr = i;
                t_c++;
        }
        ASSERT(currentPieceNr != -1);

        *f << currentPieceNr;

        debug("CurrentData::save: _score = %d, _faults = %d, _rounds = %d, _elapsed = %d, _dirty = %d",
              _score,_faults,_rounds,_elapsed,_dirty);
        debug("   _pieceList->count() = %d, currentPieceNr = %d, temp_count = %d",cnt,currentPieceNr,t_c);

        return true;
}

bool CurrentData::load(QDataStream* f)
{
        Q_INT32 c;
        Q_INT32 currentPieceNr; // see save

        *f >> _score;
        *f >> _faults;
        *f >> _rounds;
        *f >> _elapsed;
        *f >> (Q_INT8) _dirty;
        *f >> c; // Number of pieces
        _pieceList = new QList<CPiece>;
        for(int i = 1;i <= c;i++) {
                CPiece* p = new CPiece(_owner,f);
                _pieceList->append(p);
//                if (p->isCurrent()) _currentPieceData = p;
        }
        *f >> currentPieceNr;

        debug("CurrentData::load: _score = %d, _faults = %d, _rounds = %d, _elapsed = %d, _dirty = %d",
              _score,_faults,_rounds,_elapsed,_dirty);
        debug("   _pieceList->count() = %d, currentPieceNr = %d",c,currentPieceNr);


//        debug("currentPieceNr = %d in CurrentData::load",currentPieceNr);

//        _currentPieceData = _pieceList->at(currentPieceNr);

        _pieceListIterator = new QListIterator<CPiece>(*_pieceList);
        _pieceListIterator->toFirst();
        *_pieceListIterator += currentPieceNr;
        _currentPieceData = _pieceListIterator->current();

//         *f >> currentPieceNr;
//         debug("cd::load: it.count = %d, currentPieceNr = %d",_pieceListIterator->count(),currentPieceNr);
//         _pieceListIterator->toFirst();
//         (*_pieceListIterator) += currentPieceNr;
//         ASSERT(_pieceListIterator->current()->isCurrent());
        return true;
}

KPuzzleApp::KPuzzleApp(KPuzzleView* view)
{
	_mainWidget = view;

	_status = GS_NOT_STARTED;

        //        enableMenuItems(MI_NONE);

	connect(this,SIGNAL(sigTerm(int)),this,SLOT(slotTerm(int)));
	//	connect(this,SIGNAL(sigUpdate()),_mainWidget,SLOT(update()));
	connect(&_timer,SIGNAL(timeout()),this,SLOT(slotTimer()));

        _scoreData = new ScoreData;
        scoreData()->_owner = this;
        _gameData = new GameData;
        gameData()->_owner = this;
        _mainPixmap = NULL;
        _currentData = new CurrentData;
        currentData()->_owner = this;

        blackPixmap = new QPixmap;
}

bool KPuzzleApp::initialize()
{
	_picview = mainWidget()->picview();
	_fullview = mainWidget()->fullview();
        maskCache->setAutoDelete(true);
        maskCache->clear();
        return true;
}

bool KPuzzleApp::initializeNew()
{
        if (!initialize()) return false;

        gameData()->_savePixmap = false;

        _mainPixmap = new QPixmap;

	gameFilename() = QString::null;

	_mainPixmap->load(gameData()->_filename);
	if (_mainPixmap->isNull()) {
                delete _mainPixmap;
		return false;
	}

// 	QPainter pt;
// 	pt.begin(_gamePixmap);
// 	pt.setBackgroundColor(QColor(0,0,0));
// 	pt.eraseRect(QRect(QPoint(0,0),pixmapSize()));
// 	pt.end();

	calcPiecesCount();
        if (scalePixmap()) {
//                gameData()->_piecesCount += QSize(1,1);
                gameData()->_piecesCount.setWidth(pixmapSize().width() / pieceSize().width());
                gameData()->_piecesCount.setHeight(pixmapSize().height() / pieceSize().height());
        }

        _gamePixmap = new QPixmap;
        _gamePixmap->resize(pixmapSize());
        _gamePixmap->fill((QColor)black);

	//	CPiece::_maskCache = new QCache<QBitmap>;

	ASSERT(piecesCount().width() >= 3);
	ASSERT(piecesCount().height() >= 3);

// Moved to calcPiecesCount
// 	gameData()->_pieceSize = QSize(pixmapSize().width() / piecesCount().width(),
//                            pixmapSize().height() / piecesCount().height());

        debug("Resizing blackPixmap to %d---%d",pieceSizeDisp().width(),pieceSizeDisp().height());
        blackPixmap->resize(pieceSizeDisp());
        blackPixmap->fill(QColor(black));

        currentData()->_pieceList = new QList<CPiece>;
	currentData()->_pieceList->setAutoDelete(true);
	minglePieces();

        currentData()->_pieceListIterator = new QListIterator<CPiece>(*pieceList());

	currentData()->_currentPieceData = currentData()->_pieceListIterator->toFirst();
	currentData()->_currentPiecePixmap = new QPixmap(pieceSizeDisp());
	getPiecePixmap();

// 	_gameStatus = (char*) malloc(piecesTotal());
// 	memset(_gameStatus,0,piecesTotal());

	_picview->updatePixmap(gamePixmap(),gamePixmap()->rect());
	_fullview->updatePixmap(mainPixmap(),mainPixmap()->rect());

	setScore(0);

	currentData()->_elapsed = 0;
        currentData()->_dirty = false;

	initGameData();

	mainWidget()->updateButtons(false,false,
		pixmapSize().width() > mainWidget()->gameWidget()->width(),
		pixmapSize().height() > mainWidget()->gameWidget()->height());
	mainWidget()->gameWidget()->initialize(pixmapSize());

	char timeStr[6];
	snprintf(timeStr,5,"%i:%i",scoreData()->_maxTime / 60,scoreData()->_maxTime % 60);
	mainWidget()->time()->display(timeStr);

        startTimer();
	setStatus(GS_RUNNING);
        updateAll();
        mainWidget()->update();

        currentData()->_running = true;

        currentPieceData()->showPiece();

        return true;

}

bool KPuzzleApp::initializeLoad()
{
        if (!initialize()) return false;

//	gameFilename() = QString::null;

// 	_mainPixmap->load(gameData()->_filename);
// 	if (_mainPixmap->isNull()) {
// 		return false;
// 	}

        if (!gameData()->_savePixmap) scalePixmap();
//                        gameData()->_piecesCount += QSize(1,1); //NO!!!

        _gamePixmap = new QPixmap(pixmapSize());
        _gamePixmap->fill((QColor)black);

        debug("Resizing blackPixmap to %d---%d",pieceSizeDisp().width(),pieceSizeDisp().height());
        blackPixmap->resize(pieceSizeDisp());
        blackPixmap->fill(QColor(black));

	currentData()->_currentPiecePixmap = new QPixmap(pieceSizeDisp());
	getPiecePixmap();

	// First difficult thing: the game pixmap
        QPixmap piecePxm(currentPiecePixmap()->size());

#define CUR (drawingIterator.current())

        QListIterator<CPiece> drawingIterator(*pieceList());
        for (drawingIterator.toFirst();drawingIterator.current() != NULL;++drawingIterator)
                if (CUR->hasBeenSet()) {
                        getPiecePixmap(CUR,&piecePxm);
                        QPoint tpos(CUR->pos().x() * pieceSize().width(),CUR->pos().y() * pieceSize().height());
                        if (maskedPieces()) tpos -= QPoint(displace(),displace());
                        bitBlt(gamePixmap(),tpos,&piecePxm,QRect(QPoint(0,0),pieceSizeDisp()));
                }

#undef CUR

        _picview->updatePixmap(gamePixmap(),gamePixmap()->rect());
	_fullview->updatePixmap(mainPixmap(),mainPixmap()->rect());

	setScore(currentData()->_score);

	initGameData();

	mainWidget()->updateButtons(false,false,
		pixmapSize().width() > mainWidget()->gameWidget()->width(),
		pixmapSize().height() > mainWidget()->gameWidget()->height());
	mainWidget()->gameWidget()->initialize(pixmapSize());

	int rest = scoreData()->_maxTime - currentData()->_elapsed;
	QString s;
	s.sprintf("%i:%2i",rest / 60,rest % 60);
	mainWidget()->time()->display(s);
// 	char timeStr[6];
// 	snprintf(timeStr,5,"%i:%i",scoreData()->_maxTime / 60,scoreData()->_maxTime % 60);
// 	mainWidget()->time()->display(timeStr);

	startTimer();
	setStatus(GS_RUNNING);
        updateAll();
        mainWidget()->update();

        currentData()->_running = true;

        currentPieceData()->showPiece();

        return true;

}

void KPuzzleApp::done()
{
	_picview = NULL;
	_fullview = NULL;

	ASSERT(currentData()->_pieceList != NULL);
        delete currentData()->_pieceListIterator;
	while (!currentData()->_pieceList->isEmpty()) currentData()->_pieceList->removeFirst(); // deleted automatically
	delete currentData()->_pieceList;

//	if (_maskCache) delete _maskCache;

	if (currentData()->_currentPiecePixmap) delete currentData()->_currentPiecePixmap;

	delete _mainPixmap;
        _mainPixmap = NULL;

	delete _gamePixmap;
        _gamePixmap = NULL;

//	free(_gameStatus);
}

KPuzzleApp::~KPuzzleApp()
{
	if (status() != GS_NOT_STARTED) done();

        delete _scoreData;
        delete _gameData;
        delete _currentData;

        delete blackPixmap;
}

void KPuzzleApp::slotNewGame()
{
        bool gameIsPaused = paused();
        pause(true);
	if (status() != GS_NOT_STARTED)
		if (KMessageBox::warningContinueCancel(0,i18n("A game is already running. Start a new one and stop the current?"),
                                                       i18n("Stop this game?"),i18n("Continue")) == KMessageBox::Cancel) {
                        pause(gameIsPaused);
                        return;
                }
	QString fname = KFileDialog::getOpenFileName(LEVELS_DIR,"*.bmp",mainWidget(),i18n("Open Image"));
     	if (fname.isNull()) {
                pause(gameIsPaused);
                return;
        }
        CGameDialog gd(mainWidget());
	if (gd.exec() != QDialog::Accepted) {
                pause(gameIsPaused);
                return;
        }

        gameData()->_filename = fname;

	if (status() != GS_NOT_STARTED) emit sigTerm(TM_FORCE);

 	gameData()->_dialog_pieceSize = gd.pieceSize();
	gameData()->_gameType = gd.gameType();
	gameData()->_difficulty = gd.difficulty();
	gameData()->_scale = true;
	gameData()->_useMask = gd.useMask();

//         gameData()->_dialog_pieceSize = 3;
//         gameData()->_gameType = STD_GAME;
//         gameData()->_difficulty = 1;
//         gameData()->_scale = true;
//         gameData()->_useMask = true;

//        abort();

        mainWidget()->createPlayground(gameData()->_gameType == UNBEARABLE_GAME);
	initializeNew(); // calls initialize()
}

void KPuzzleApp::slotOpenGame()
{
        bool gameIsPaused = paused();
        pause(true);
	if (status() != GS_NOT_STARTED)
		if (KMessageBox::warningContinueCancel(0,i18n("A game is already running. Stop it and open a saved game?"),
                                                       i18n("Stop this game?"),i18n("Continue")) == KMessageBox::Cancel) {
                        pause(gameIsPaused);
                        return;
                }
	gameFilename() = KFileDialog::getOpenFileName(QDir::homeDirPath(),"*.pzz",mainWidget(),i18n("Open Game"));
	if (gameFilename().isNull()) {
                pause(gameIsPaused);
                return;
        }
        if (gameFilename().findRev(".") <= gameFilename().findRev("/")) // There is no dot after the last slash
                gameFilename() += ".pzz";
        if (!QFile::exists(gameFilename())) {
                KMessageBox::sorry(0,i18n("The file you specified does not exist"),i18n("File does not exist"));
                pause(gameIsPaused);
                return;
        }
        QFile file(gameFilename());
        if (!file.open(IO_ReadOnly)) {
                KMessageBox::sorry(0,i18n("The file you specified could not be opened for reading. Check the file's permissions."),
                                   i18n("File could not be opened"));
                pause(gameIsPaused);
                return;
        }

        QDataStream stream(&file);
//        stream.setPrintableData(true);
        if (!load(&stream)) return;
	mainWidget()->createPlayground(gameData()->_gameType == UNBEARABLE_GAME);
        initializeLoad();
}

void KPuzzleApp::slotSaveGame()
{
        bool gameIsPaused = paused();
        pause(true);
        if (gameFilename().isNull()) {
                gameFilename() = KFileDialog::getSaveFileName(QDir::homeDirPath(),"*.pzz",mainWidget(),i18n("Save Game"));
                if (gameFilename().isNull()) {
                        pause(gameIsPaused);
                        return;
                }

        }

//        bool addExt = (gameFilename().right(4) != QString(".pzz"));
        CSaveDialog sd(mainWidget(),gameData()->_savePixmap);
        if (sd.exec() == QDialog::Rejected) {
                pause(gameIsPaused);
                return;
        }

//        if (sd.addExt()) gameFilename() += (gameFilename.right(1) == "." ? "pzz" : ".pzz");
        gameData()->_savePixmap = sd.savePixmap();
        gameData()->_annotation = sd.annotation();

        ASSERT(!gameFilename().isNull());
        if (gameFilename().findRev(".") <= gameFilename().findRev("/")) // There is no dot behind the last slash
                gameFilename() += ".pzz";
        debug((const char*) gameFilename());
        QFile file(gameFilename());
        if (!file.open(IO_WriteOnly)) {
                KMessageBox::sorry(0,i18n("The file you specified could not be opened for writing. Check the file's"
                                          "permissions."),i18n("File could not be opened"));
                pause(gameIsPaused);
                return;
        }

        QDataStream stream(&file);
//        stream.setPrintableData(true);
        save(&stream);
        pause(gameIsPaused);
}

void KPuzzleApp::slotSaveGameAs()
{
        bool gameIsPaused = paused();
        pause(true);
        QString fname = KFileDialog::getSaveFileName(QDir::homeDirPath(),"*.pzz",mainWidget(),"Save Game");
        if (fname.isNull()) {
                debug("saving cancelled");
                pause(gameIsPaused);
                return;
        }

        CSaveDialog sd(mainWidget(),gameData()->_savePixmap);
        if (sd.exec() == QDialog::Rejected) {
                pause(gameIsPaused);
                return;
        }

//        if (sd.addExt()) gameFilename() += (gameFilename.right(1) == "." ? "pzz" : ".pzz");
        gameData()->_savePixmap = sd.savePixmap();
        gameData()->_annotation = sd.annotation();

        debug("fname = %s",(const char*) fname);
        gameFilename() = fname;
        if (gameFilename().findRev(".") <= gameFilename().findRev("/")) // There is no dot behind the last slash
                gameFilename() += ".pzz";
        QFile file(gameFilename());
        if (!file.open(IO_WriteOnly)) {
                KMessageBox::sorry(0,i18n("The file you specified could not be opened for writing. Check the file's"
                                          "permissions."),i18n("File could not be opened"));
                pause(gameIsPaused);
                return;
        }

        QDataStream stream(&file);
//        stream.setPrintableData(true);
        save(&stream);
        pause(gameIsPaused);
}

void KPuzzleApp::slotHighscores()
{
        bool gameIsPaused = paused();
        pause(true);
 	CHighscoreDialog hs(mainWidget()->parentWidget(),0);
	hs.initializeDialog();
	hs.exec();
        pause(gameIsPaused);
}

void KPuzzleApp::slotShowLarge()
{
	if (status() != GS_RUNNING) return;
	pause(true);
	mainWidget()->setLargePxm(new QPixmap(getLargePixmap(mainWidget()->size())));
	setStatus(GS_SHOW_LARGE);
	//	mainWidget()->update();
	emit sigUpdate();
}

void KPuzzleApp::slotShowMainPixmap()
{
	if (status() != GS_RUNNING) return;
	pause(true);
	mainWidget()->setLargePxm(new QPixmap(getLargeMainPixmap(mainWidget()->size())));
	setStatus(GS_SHOW_LARGE);
	//	mainWidget()->update();
	emit sigUpdate();
}

void KPuzzleApp::slotPause()
{
// 	switch (status()) {
// 	case GS_RUNNING:
		pause(true);
// 		break;
// 	case GS_PAUSED:
// 		pause(false);
// 		break;
// 	}
}

void KPuzzleApp::slotPauseToggle(bool on)
{
        fatal("slotPauseToggle called");
	if ((status() == GS_PAUSED && on) || (status() == GS_RUNNING && !on)) return;
	slotPause();
}

void KPuzzleApp::initGameData()
{
	int tt = piecesTotal();
	int r; // score for placing a piece correctly

	scoreData()->_maxFaults = -1;
	scoreData()->_maxPieceTime = -1;
	scoreData()->_maxRounds = -1;

	switch (gameData()->_difficulty) {
	case 0: // easy
		r = 50;
		scoreData()->_maxTime = tt * 60;
		scoreData()->_score1Time = tt * 10;
		scoreData()->_score2Time = tt * 20;
		scoreData()->_score3Time = tt * 30;
		scoreData()->_timeScore1 = tt * r / 10;
		scoreData()->_timeScore2 = tt * r / 15;
		scoreData()->_timeScore3 = tt * r / 20;
		scoreData()->_rightPieceScore = r;
		scoreData()->_wrongPieceScore = 0;

		switch (gameType()) {
		case STD_GAME:
			scoreData()->_bonus = 0;
			break;
		case PIECE_TIME_GAME:
			scoreData()->_bonus = tt * r / 15;
			scoreData()->_maxPieceTime = 60;
			break;
		case PIECE_FAULTS_GAME:
			scoreData()->_bonus = tt * r / 20;
			scoreData()->_maxFaults = 10;
			break;
		case FAULTS_SUM_GAME:
			scoreData()->_bonus = tt * r / 20;
			scoreData()->_maxFaults = 9 * tt;
			currentData()->_faults = 0;
			break;
		case UNBEARABLE_GAME:
			scoreData()->_bonus = tt * r / 10;
			scoreData()->_maxPieceTime = 15;
			scoreData()->_maxRounds = 5;
			currentData()->_rounds = 0;
			break;
		}
		break;

	case 1: // medium
		r = 70;
		scoreData()->_maxTime = tt * 50;
		scoreData()->_score1Time = tt * 15;
		scoreData()->_score2Time = tt * 25;
		scoreData()->_score3Time = tt * 35;
		scoreData()->_timeScore1 = tt * r / 10;
		scoreData()->_timeScore2 = tt * r / 15;
		scoreData()->_timeScore3 = tt * r / 20;
		scoreData()->_rightPieceScore = r;
		scoreData()->_wrongPieceScore = 10;

		switch (gameType()) {
		case STD_GAME:
			scoreData()->_bonus = tt * r / 20;
			break;
		case PIECE_TIME_GAME:
			scoreData()->_bonus = tt * r / 12;
			scoreData()->_maxPieceTime = 50;
			break;
		case PIECE_FAULTS_GAME:
			scoreData()->_bonus = tt * r / 16;
			scoreData()->_maxFaults = 8;
			break;
		case FAULTS_SUM_GAME:
			scoreData()->_bonus = tt * r / 16;
			scoreData()->_maxFaults = 7 * tt;
			currentData()->_faults = 0;
			break;
		case UNBEARABLE_GAME:
			scoreData()->_bonus = tt * r / 6;
			scoreData()->_maxPieceTime = 12;
			scoreData()->_maxRounds = 4;
			currentData()->_rounds = 0;
			break;
		}
		break;

	case 2: // hard
		r = 90;
		scoreData()->_maxTime = tt * 40;
		scoreData()->_score1Time = tt * 20;
		scoreData()->_score2Time = tt * 30;
		scoreData()->_score3Time = tt * 40;
		scoreData()->_timeScore1 = tt * r / 5;
		scoreData()->_timeScore2 = tt * r / 10;
		scoreData()->_timeScore3 = tt * r / 15;
		scoreData()->_rightPieceScore = r;
		scoreData()->_wrongPieceScore = 20;

		switch (gameType()) {
		case STD_GAME:
			scoreData()->_bonus = tt * r / 15;
			break;
		case PIECE_TIME_GAME:
			scoreData()->_bonus = tt * r / 10;
			scoreData()->_maxPieceTime = 40;
			break;
		case PIECE_FAULTS_GAME:
			scoreData()->_bonus = tt * r / 12;
			scoreData()->_maxFaults = 6;
			break;
		case FAULTS_SUM_GAME:
			scoreData()->_bonus = tt * r / 12;
			scoreData()->_maxFaults = 5 * tt;
			currentData()->_faults = 0;
			break;
		case UNBEARABLE_GAME:
			scoreData()->_bonus = tt * r / 3;
			scoreData()->_maxPieceTime = 10;
			scoreData()->_maxRounds = 3;
			currentData()->_rounds = 0;
			break;
		}
		break;
	}

}

void KPuzzleApp::calcPiecesCount()
{
 	int tcx;
 	int tcy;
	switch(gameData()->_dialog_pieceSize) {
	case 0: tcx = 88; tcy = 66; gameData()->_displace = maskedPieces() ? 19 : 0; break;
	case 1: tcx = 100; tcy = 75; gameData()->_displace = maskedPieces() ? 21 : 0; break;
	case 2: tcx = 112; tcy = 84; gameData()->_displace = maskedPieces() ? 24 : 0; break;
	case 3: tcx = 132; tcy = 99; gameData()->_displace = maskedPieces() ? 28 : 0; break;
	case 4: tcx = 140; tcy = 105; gameData()->_displace = maskedPieces() ? 30 : 0; break;
	}
// 	if (pixmapSize().width() % tcx) { //TODO
// 		QWMatrix m;
// 		float scaleVal = (float) ((pixmapSize().width() / tcx + 1) *
//                         tcx) / pixmapSize().width();
// 		m.scale(scaleVal,scaleVal);
// 		QPixmap* pxm = new QPixmap(_mainPixmap->xForm(m));
// 		delete _mainPixmap;
// 		_mainPixmap = pxm;

// 		pxm = new QPixmap(_gamePixmap->xForm(m));
// 		delete _gamePixmap;
// 		_gamePixmap = pxm;
// 	}

	gameData()->_piecesCount = QSize(pixmapSize().width() / tcx,pixmapSize().height() / tcy);
        gameData()->_pieceSize = QSize(tcx,tcy);
}

bool KPuzzleApp::scalePixmap()
{
        gameData()->_originalPixmapSize = pixmapSize();
	if (pixmapSize().width() % pieceSize().width()) {
		QWMatrix m;
		float scaleVal = (float) ((pixmapSize().width() / pieceSize().width() + 1) *
                        pieceSize().width()) / pixmapSize().width();
		m.scale(scaleVal,scaleVal);
		QPixmap* pxm = new QPixmap(_mainPixmap->xForm(m));
		delete _mainPixmap;
		_mainPixmap = pxm;

                return true;
	}
        return false;
}

//bool KPuzzleApp::setNextPiece(bool currentInvalid,bool forceUpdate)
bool KPuzzleApp::setNextPiece()
{
// 	int newNr = _currentPieceNr;
// 	int maxNr = availablePiecesCount() - 1;
// 	int startNr = newNr;

// 	if (maxNr == -1) {
// 		emit sigTerm(TM_WON);
// 		return false;
// 	}
// 	if (maxNr == 0) {
// 		if (currentInvalid) {
// 			if (!pieceAvailable(0)) {
// 				emit sigTerm(TM_LOST);
// 				return false;
// 			}
// 			_currentPieceNr = 0;
// 			getPiecePixmap();
// 			return true;
// 		}
// 		if (forceUpdate) {
// 			emit sigTerm(TM_LOST);
// 			return false;
// 		}
// 		return false; // No change

// 	}

// 	if (!currentInvalid) newNr++;
// 	if (newNr > maxNr) newNr = 0;
// 	if (startNr > maxNr) startNr = 0;

// 	while (!pieceAvailable(newNr)) {
// 		newNr++;
// 		if (newNr > maxNr) newNr = 0;
// 		if (newNr == startNr && (forceUpdate || currentInvalid)) {
// 			emit sigTerm(TM_LOST);
// 			return false;
// 		}
// 	}


// 	_currentPieceNr = newNr;
// 	getPiecePixmap();
//  	return true; // redraw

	CPiece* mark = currentPieceData();
        mark->hidePiece();

	do {
		if (++(*pieceListIterator()) == NULL) {
			pieceListIterator()->toFirst();
			if (gameType() == UNBEARABLE_GAME) {
				currentData()->_rounds++;
				if (currentData()->_rounds > scoreData()->_maxRounds) {
					emit sigTerm(TM_LOST);
					return false;
				}
			}
		}
		if (pieceListIterator()->current()->showPiece()) {
			currentData()->_currentPieceData = pieceListIterator()->current();
			getPiecePixmap();
			updateAll();
			return true;
		}
	} while (pieceListIterator()->current() != mark);

	if (currentData()->_dirty) {
		emit sigTerm(TM_LOST);
		return false;
	} else {
		emit sigTerm(TM_WON);
		return false;
	}
}

bool KPuzzleApp::setPrevPiece()
{
//  	int newNr = _currentPieceNr;
//  	int maxNr = availablePiecesCount() - 1;

// 	if (maxNr == -1) {
// 		emit sigTerm(TM_WON);
// 		return false;
// 	}
// 	if (maxNr == 0) return false; // No change

// 	do {
// 		newNr--;
// 		if (newNr < 0) newNr = maxNr;
// 		if (newNr == _currentPieceNr) {
// 			emit sigTerm(TM_LOST);
// 			return false;
// 		}
// 	} while (!pieceAvailable(newNr));

// 	_currentPieceNr = newNr;
// 	getPiecePixmap();
//  	return true; // redraw
	CPiece* mark = currentPieceData();
        mark->hidePiece();

	do {
		if (--(*pieceListIterator()) == NULL) pieceListIterator()->toLast();
		if (pieceListIterator()->current()->showPiece()) {
			currentData()->_currentPieceData = pieceListIterator()->current();
			getPiecePixmap();
			updateAll();
			return true;
		}
	} while (pieceListIterator()->current() != mark);

	if (currentData()->_dirty) {
		emit sigTerm(TM_LOST);
		return false;
	} else {
		emit sigTerm(TM_WON);
		return false;
	}
}

bool KPuzzleApp::setPiece(QPoint pos)
{
	if (currentPieceData()->setPiece(pos)) {
		if (!currentData()->_running) // Game has been terminated by winning
			//			return;
			fatal("setPiece: Game has been terminated: 1");
		changeScore(SC_PIECE_RIGHT);
		QPoint tpos(pos.x() * pieceSize().width(),pos.y() * pieceSize().height());
		if (maskedPieces()) {
			tpos -= QPoint(displace(),displace());
			bitBlt(gamePixmap(),tpos,currentPiecePixmap(),QRect(QPoint(0,0),pieceSizeDisp()));
		} else {
			bitBlt(gamePixmap(),tpos,currentPiecePixmap(),QRect(QPoint(0,0),pieceSize()));
		}
		_picview->addPiece(currentPiecePixmap(),currentPieceData()->pos());
		setNextPiece();
                return true;
	} else {
		if (!currentData()->_running) // Game has been terminated by losing

			fatal("setPiece: Game has been terminated: 2");
		//			return;
		changeScore(SC_PIECE_WRONG);
		if (gameType() == FAULTS_SUM_GAME) {
			currentData()->_faults++;
			if (currentData()->_faults > scoreData()->_maxFaults) {
				emit sigTerm(TM_LOST);
				return false;
			}
		}
		if (!currentPieceData()->mayKeepPiece()) setNextPiece();
                return false;
		//		updateAll();
	}
}

void KPuzzleApp::hidePiece()
{
	switch(gameType()) {
	case PIECE_TIME_GAME:
		currentData()->_dirty = true;
		break;
	case UNBEARABLE_GAME:
		break;
	default:
		fatal("error in KPuzzleApp::hidePiece");
	}
	setNextPiece();
}

void KPuzzleApp::minglePieces()
{
	ASSERT(pieceList()->isEmpty());
	for (int j = 0;j < piecesCount().height();j++)
		for (int i = 0;i < piecesCount().width();i++) {
			CPiece* p = new CPiece(this,QPoint(i,j));;
// 			ts->pos = QPoint(i,j);
			p->turn() = rand() % 4;
// 			ts->time = 0;
// 			ts->faults = 0;
			currentData()->_pieceList->insert(rand() %
				(j * piecesCount().width() + i + 1),p);
		}
}

//void KPuzzleApp::getPiece(QPixmap* dst,int nr)
void KPuzzleApp::getPiecePixmap(CPiece* pc,QPixmap* pxm) const
{
        if (pc) pc->getPixmap((QPixmap*) (pxm ? pxm : currentPiecePixmap()));
        else currentPieceData()->getPixmap((QPixmap*) (pxm ? pxm : currentPiecePixmap()));
}

QPixmap* KPuzzleApp::currentPieceTurned() const
{
        QPixmap* ret = new QPixmap;
        currentPieceData()->getTurnedPixmap(ret);
        return ret;
}

QPixmap KPuzzleApp::getLargePixmap(QSize size) const
{
	float scaleX = (float) size.width() / gamePixmap()->width();
	// With this factor, the width of
	// the scaled picture is equal to the width of the window
	float scaleY = (float) size.height() / gamePixmap()->height();
	// same with height

	QWMatrix m;
	m.scale(min(scaleX,scaleY),min(scaleX,scaleY));
	QPixmap pxm = gamePixmap()->xForm(m);
	return pxm;
}

QPixmap KPuzzleApp::getLargeMainPixmap(QSize size) const
{
	float scaleX = (float) size.width() / gamePixmap()->width();
	// With this factor, the width of
	// the scaled picture is equal to the width of the window
	float scaleY = (float) size.height() / gamePixmap()->height();
	// same with height

	QWMatrix m;
	m.scale(min(scaleX,scaleY),min(scaleX,scaleY));
	QPixmap pxm = mainPixmap()->xForm(m);
	return pxm;
}

void KPuzzleApp::updateAll()
{
	mainWidget()->updateAll();
}

void KPuzzleApp::changeScore(char reason)
{
	int add = 0;
	switch(reason) {
	case SC_PIECE_RIGHT: setScore(score() + scoreData()->_rightPieceScore);
		break;
	case SC_PIECE_WRONG:
		if (score() > 0)
			setScore(score() > scoreData()->_wrongPieceScore ?
				 score() - scoreData()->_wrongPieceScore : 1);
		break;
	case SC_COUNT_TIME:
		if (elapsed() < scoreData()->_score3Time)
			add = scoreData()->_timeScore3;
		if (elapsed() < scoreData()->_score2Time)
			add = scoreData()->_timeScore2;
		if (elapsed() < scoreData()->_score1Time)
			add = scoreData()->_timeScore1;
		setScore(score() + add);
		break;
	case SC_BONUS: setScore(score() + scoreData()->_bonus);
		break;
	}
	//LATER: TODO
}

void KPuzzleApp::showHighscoreDialog() const
{
	CHighscoreDialog hs(mainWidget()->parentWidget(),score());
	hs.initializeDialog();
	hs.exec();


}

bool KPuzzleApp::save(QDataStream* f) const
{
//        *f << (Q_INT8) gameData()->_savePixmap;

//        scoreData()->save(f);
        if (!gameData()->save(f)) return false;
        if (gameData()->_savePixmap) {
                *f << *mainPixmap();
                debug("Pixmap saved");
        } else {
///                *f << gameData()->_originalPixmapSize;
                debug("SIZE saved in gameData");
        }
        if (!currentData()->save(f)) return false;
        return true;
}

bool KPuzzleApp::load(QDataStream* f)
{
        GameData* newGD = new GameData;
	newGD->_owner = this;
        QPixmap* newMainPixmap;
        if (!newGD->load(f)) { // This should load the filename of the main pixmap
                //        delete _mainPixmap;
                delete newGD;
                return false;
        }

        newMainPixmap = new QPixmap;
        QSize gpSize; // for assertion that we have got the right pixmap

        if (newGD->_savePixmap) {
                *f >> *newMainPixmap;
                debug("Pixmap loaded");
        } else {
//                *f >> gpSize;
                gpSize = newGD->_originalPixmapSize;
                debug("loading siize");
                ASSERT(!(newGD->_filename.isNull()));

                QString base(newGD->_filename.right(newGD->_filename.length() - (newGD->_filename.findRev('/') + 1)));
                QString dir(newGD->_filename.left(newGD->_filename.findRev('/') + 1));

                if (!(QFile::exists(dir + base) && newMainPixmap->load(dir + base))) // pixmap cannot be loaded from the original place
                        if (!(QFile::exists(LEVELS_DIR + base) && newMainPixmap->load(LEVELS_DIR + base))) // ...from standard place
                                if (!(QFile::exists(QDir::homeDirPath() + base) && newMainPixmap->load(QDir::homeDirPath() + base))) {
                                        // from home directory
//                                         QString sorryMsg(i18n("Sorry, the picture for the puzzle game you selected could not be found."
//                                                               "Please select a picture you want to use instead."
//                                                               "Make sure it has the size of");
//                                         sorryMsg + QString::number(gpSize.width()) + QString(" x ") + QString::number(gpSize.height())
//                                                 + QString("pixels.");

                                         QString sorryMsg(i18n("Sorry, the picture for the puzzle game you selected could not be found.\n"
                                                               "Please select a picture you want to use instead.\n"
                                                               "Make sure it has the size of %1 x %2 pixels."));
                                         sorryMsg = sorryMsg.arg(gpSize.width()).arg(gpSize.height());


                                        KMessageBox::sorry(mainWidget(),sorryMsg);
                                        while (true) {
                                                QString newFileName = KFileDialog::getOpenFileName(LEVELS_DIR,
                                                                                                   "*.bmp",0,i18n("Choose picture"));
                                                if (newFileName.isNull()) { // user hit cancel
                                                        delete newGD;
                                                        delete newMainPixmap;
                                                        return false;
                                                }
                                                if (!(QFile::exists(newFileName) && newMainPixmap->load(newFileName))) {
                                                        debug("error finding pixmap");
                                                        KMessageBox::sorry(mainWidget(),
                                                                           i18n("Your picture could not be loaded. Try again."));
                                                } else {
                                                        if (newMainPixmap->size() != gpSize) {
                                                                debug("pixmap size error: _mainPxm->size = %d,%d, gpSize = %d,%d",
                                                                      newMainPixmap->width(),newMainPixmap->height(),
                                                                      gpSize.width(),gpSize.height());
                                                                KMessageBox::sorry(mainWidget(),
                                                                                   i18n("Your picture has the wrong size. Try again."));
                                                        } else { // Everything OK
                                                                break;
                                                        }
                                                }
                                        }
                                }
                        }
        CurrentData* newCD = new CurrentData;
	newCD->_owner = this;
        if (!newCD->load(f)) {
                delete newGD;
                delete newMainPixmap;
                return false;
        }

        debug("Now replacing all datafields");
        if (_mainPixmap) delete _mainPixmap;
        delete _gameData;
        delete _currentData;
        _mainPixmap = newMainPixmap;
        _gameData = newGD;
        _currentData = newCD;

        return true;
}

void KPuzzleApp::setStatus(int s)
{
	int enable;
	_status = s;
	mainWidget()->hideStatusMsg();
	switch(s) {
	case GS_NOT_STARTED:
		enable = MI_NONE;

		done();

		break;

	case GS_RUNNING:
		enable = MI_STOP_GAME | MI_PAUSE | MI_SHOW_LARGE | MI_SAVE_GAME;

		mainWidget()->freeGamePixmap(GP_BIGLOGO);
		mainWidget()->freeGamePixmap(GP_PAUSE);

		break;

	case GS_WINNING:
		enable = MI_STOP_GAME;

		mainWidget()->showStatusMsg(i18n("Press mouse key to continue"));
		mainWidget()->showAll(false);
		mainWidget()->useGamePixmap(GP_WIN);
		mainWidget()->setLargePxm(NULL);

		break;

	case GS_SHOW_LARGE:
		enable = MI_STOP_GAME | MI_SAVE_GAME;

		mainWidget()->showStatusMsg(i18n("Press mouse key to continue"));

		break;

	case GS_SHOW_LARGE_WINNING:
		enable = MI_STOP_GAME;

		mainWidget()->showStatusMsg(i18n("Press mouse key to continue"));
		mainWidget()->setLargePxm(new QPixmap(getLargePixmap(mainWidget()->size())));
		mainWidget()->freeGamePixmap(GP_WIN);

		break;

	case GS_LOSING:
		enable = MI_STOP_GAME;

		mainWidget()->showStatusMsg(i18n("Press mouse key to continue"));
		mainWidget()->showAll(false);
		mainWidget()->useGamePixmap(GP_LOSE);
		mainWidget()->setLargePxm(NULL);

		break;

	case GS_SHOW_LARGE_LOSING:
		enable = MI_STOP_GAME;

		mainWidget()->showStatusMsg(i18n("Press mouse key to continue"));
		mainWidget()->setLargePxm(new QPixmap(getLargePixmap(mainWidget()->size())));
		mainWidget()->freeGamePixmap(GP_LOSE);

		break;

	case GS_SHOW_HIGHSCORES:
		enable = MI_NONE;

		showHighscoreDialog();
		mainWidget()->destroyPlayground();
		setStatus(GS_NOT_STARTED);
		mainWidget()->update();

		break;

	case GS_PAUSED:
		enable = MI_STOP_GAME | MI_SAVE_GAME;

		mainWidget()->showStatusMsg(i18n("Press mouse key to continue"));
		mainWidget()->useGamePixmap(GP_PAUSE);

		break;
	}
        //	((KPuzzle*) (mainWidget()->parentWidget()))->enableMenuItems(enable);
        enableMenuItems(enable);
}

void KPuzzleApp::pause(bool p)
{
        if (status() == GS_NOT_STARTED) return;
        if (p) {
		if (status() == GS_PAUSED) return;
		stopTimer();
                currentPieceData()->stopTimer();
		setStatus(GS_PAUSED);
		mainWidget()->showAll(false);
		mainWidget()->update();
	} else {
		if (status() == GS_RUNNING) return;
		startTimer();
                currentPieceData()->startTimer();
		setStatus(GS_RUNNING);
		mainWidget()->showAll(true);
		mainWidget()->update();
	}

}

void KPuzzleApp::slotTimer()
{
	currentData()->_elapsed++;
	int rest = scoreData()->_maxTime - currentData()->_elapsed;
	QString s;
	s.sprintf("%i:%2i",rest / 60,rest % 60);
	mainWidget()->time()->display(s);
	// Something has to be done, dependent on the game type
// 	PieceState* current = _pieceList->at(_currentPieceNr);
// 	switch (_gameType) {
// 	case PIECE_TIME_GAME:
// 		current->time++;
// 		if (current->time > scoreData().maxPieceTime) {
// 			if (availablePiecesCount() == 1) {
// 				emit sigTerm(TM_LOST);
// 				return;
// 			}
// 			if (setNextPiece(false,true)) {
// 				mainWidget()->pieceWidget()->update();
// 				mainWidget()->gameWidget()->update();
// 			}
// 		}
// 		break;
// 	case UNBEARABLE_GAME:
// 		current->time++;
// 		if (current->time > scoreData().maxPieceTime) {
// 			current->time = 0;
// 			if (setNextPiece(false,true)) {
// 				mainWidget()->pieceWidget()->update();
// 				mainWidget()->gameWidget()->update(); //TODO
// 			}
// 			if (_currentPieceNr == 0) _rounds++;
// 			if (_rounds > scoreData().maxRounds) {
// 				emit sigTerm(TM_LOST);
// 				return;
// 			}
// 		}
// 		break;
// 	}

}

void KPuzzleApp::slotTerm(int reason)
{
	stopTimer();
	switch (reason) {
	case TM_WON:
		changeScore(SC_COUNT_TIME);
		changeScore(SC_BONUS);
		setStatus(GS_WINNING);
		mainWidget()->update();
		break;
	case TM_LOST:
		setStatus(GS_LOSING);
		mainWidget()->update();
		break;
	case TM_QUIT:
		setStatus(GS_LOSING);
		mainWidget()->update();
		break;

	// If reason is TM_FORCE, _status should be GS_NOT_STARTED when leaving, i.e.
	// the game should be in its initial state
	case TM_FORCE:
		switch (status()) {
		case GS_RUNNING:
		case GS_PAUSED:
		case GS_SHOW_LARGE:
		case GS_SHOW_LARGE_WINNING:
		case GS_SHOW_LARGE_LOSING:
			mainWidget()->destroyPlayground();
//			setStatus(GS_NOT_STARTED);
			break;
		case GS_WINNING:
		case GS_LOSING:
			// Nothing more currently
			break;
		}

		setStatus(GS_NOT_STARTED);
		mainWidget()->update();
		return;
	}

}

void KPuzzleApp::enableMenuItems(int enable)
{
        debug("enableMenuItems called!!!");
        actSave()->setEnabled(enable & MI_SAVE_GAME);
        actSaveAs()->setEnabled(enable & MI_SAVE_GAME);

        actTerminate()->setEnabled(enable & MI_STOP_GAME);
        actShowLarge()->setEnabled(enable & MI_SHOW_LARGE);
        actShowMainPixmap()->setEnabled((enable & MI_SHOW_LARGE) &&
                                        kapp->sessionConfig()->readBoolEntry("ShowMainPixmap",SV_SHOW_MAIN_PIXMAP));
        actPause()->setEnabled(enable & MI_SHOW_LARGE);
}
