#include <stdio.h> //for debugging

#include <math.h>

#include <qdialog.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qtabdialog.h>

#include "kps2dparametric.moc"
#include "kps2doptions.h"
#include "kpsaxis.h"
#include "kpscoordframe.h"
#include "kpview.h"
#include "kporder.h"
#include "kputil.h"

// Number of points to draw in one reDraw().  Ideally, we would
//  stop not after a certain number of points but after a certain number
//  of milliseconds.
const int NPOINTS = 750;
const double  maxpointsize = 1.0;

KPS2DParametric::KPS2DParametric (KPView *_view, const KPCoords &_coords,
			    KPMatrix *_matrix, 
			    double _x, double _y,
			    double _w, double _h, int _z) 
  : KPS2D (_view, _coords, _matrix, _x, _y, _w, _h, _z)
{
  addMatrix (_matrix);
  propwid=0;
  plots.setAutoDelete (true);
}


KPS2DParametric::KPS2DParametric (KPView *_view, const KPCoords &_coords,
				  double _x, double _y,
				  double _w, double _h, int _z) :
  KPS2D (_view, _coords, _x, _y, _w, _h, _z)
{}

int
KPS2DParametric::addMatrix (KPMatrix *_matrix)
{
  KPS2DParametricOptions *pr = new KPS2DParametricOptions;
  plots.append (pr);
  pr->matrix = _matrix;
  pr->pname.sprintf ("Series %d", plots.count());
  unsigned int i = plots.findRef (pr);

  printf ("adding series %d\n", plots.count());
  return i;
}

void
KPS2DParametric::reDraw (QPainter *painter)
{

  QPainter *p;
  p = painter;


  int i, j, col;

  printf ("REDRAW\n");
  if (matrix())
    {
      printf ("REDRAW-YES\n");
      if (needRedraw())  //fresh start
	{
	  rdcol=0;
	  rdpoints= 0;
	  rdi=rdj=0;
	  rdflag = FALSE;
	}
      
      
      //Draw the plot.
      double lastx, lasty;
      lastx = rdlastx;
      lasty = rdlasty;
      
      double x,y;
      
      
      i=rdi;
      j=rdj;
      for (col=rdcol; (unsigned)col<plots.count(); col++)
	{
	  int nr = matrix(col)->nRows();
	  int nc = matrix(col)->nCols();
	  
	  double *m = matrix(col)->matrix();
	  
	  unsigned int pw= (int)
	    kpmax (pointSize(col) * view()->width(), 1.);
	  unsigned int ph=pw;
	  int pwo2, pho2;
	  pwo2 = (int) kpmax (pw/2., 1.);
	  pho2 = (int) kpmax (ph/2., 1.);
	  
	  p->setPen (color(col));
	  switch (pointStyle(col))
	    {
	    case FilledBoxes:
	    case FilledCircles:
	    case FilledTriangles:
	      p->setBrush (color(col));
	      break;
	    case Points:
	    case OpenBoxes:
	    case OpenCircles:
	    case OpenTriangles:
	      p->setBrush (NoBrush);
	      break;
	    }               
	  for (; i<nr; i++, j+=nc)
	    {
	      x = m [j];
	      y = m [j+1];
	      
	      if (x>=coords()->cxmin() &&
		  x<=coords()->cxmax() &&
		  y>=coords()->cymin() &&
		  y<=coords()->cymax())
		{
		  int pxc = coords()->xc_to_o (x);
		  int pyc = coords()->yc_to_o (y);
		  int px = pxc-pwo2;
		  int py = pyc-pho2;
		  
		  switch (pointStyle(col))
		    {
		    case Points:
		      p->drawPoint (pxc, pyc);
		      break;
		case OpenBoxes:
		    case FilledBoxes:
		      p->drawRect (px, py, pw, ph);
		      break;
		    case FilledCircles:
		    case OpenCircles:
		      if (pw==1 && ph==1)
			pw=ph=2;
		      p->drawEllipse (px, py, pw, ph);
		      break;
		    case OpenTriangles:
		    case FilledTriangles:
		      {
			QPointArray qpa (4);
			qpa.setPoint (1, px, py+ph);
			qpa.setPoint (2, pxc, py);
			qpa.setPoint (3, px + pw, py+ph);
			qpa.setPoint (4, px, py+ph);
			p->drawPolygon (qpa);
		      }             
		      break;
		    }
		  if (connected(col)) //connect the dots?
		    {
		      double ppx = px+pwo2,
			ppy = py + pho2;
		      
		      if (rdflag)
			{
			  p->moveTo (lastx, lasty);
			  p->lineTo (ppx, ppy);
			}
		      else
			rdflag=TRUE;
		      
		      lastx=ppx;
		      lasty=ppy;
		    }
		  
		}
	      rdpoints++;
	      if (rdpoints>NPOINTS)
		break;
	    }
	  if (rdpoints>NPOINTS)
	    break;  
	  
	  i=j=0;
	  rdflag = FALSE;
	}
      if ((unsigned)col<plots.count())
	{
	  rdpoints = 0;
	  rdcol=col;
	  rdi=i;
	  rdj=j;
	  rdlastx = lastx;
	  rdlasty = lasty;
	}
      else
	doneRedrawing();
    }
}

void
KPS2DParametric::setSeriesName (QString n, unsigned int matrixnum)
{
  bneedredraw=true;
  plots.at(matrixnum)->pname = n; 
}


bool
KPS2DParametric::setPointSize (double _size, unsigned int matrixnum)
{
  if (_size > maxpointsize)
    return FALSE;

  plots.at(matrixnum)->pointsize = _size;
  plots.at(matrixnum)->pointsizeo2 = _size/2;

  bneedredraw = TRUE;
 
  return TRUE;
}               

KPS2DParametric::~KPS2DParametric ()
{}


KPCoords
KPS2DParametric::autoCoords (const KPMatrix *_matrix,
			     const KPCoords *_coords)
{
  
  printf ("KPS2DParametric::autocoords\n");
  //Make the axes fit the plot and start and end on good
  // numbers.
  double cminx=1e300, cmaxx=-1e300;
  double cminy=1e300, cmaxy=-1e300;

  cminx = _matrix->min(0);
  cmaxx = _matrix->max(0);
  cminy = _matrix->min(1);
  cmaxy = _matrix->max(1);


  if (_coords)
    {
      cminx = kpmin (cminx, _coords->cxmin());
      cmaxx = kpmax (cmaxx, _coords->cxmax());
      cminy = kpmin (cminy, _coords->cymin());
      cmaxy = kpmax (cmaxy, _coords->cymax());
    }	    

  double w = .05*(cmaxx - cminx);
  double h = .05*(cmaxy - cminy);
  cminx -= w;
  cmaxx += w;
  cminy -= h;
  cmaxy += h;
  
	    
  w = (cmaxx - cminx);
  h = (cmaxy - cminy);
  double cm;
  
  int p = (int) log10 (w);
  double dp = pow(10,p);
  do
    {
      cm = floor (cminx/dp)*dp;
      dp/=10;
    } while (fabs (cm-cminx)>.1*w);
  cminx=cm;
  
  dp = pow(10,p);
  do
    {
      cm = ceil (cmaxx/dp)*dp;
      dp/=10;
    } while (fabs (cm-cmaxx)>.1*w);
  cmaxx=cm;
  
  p = (int) log10 (h);
  dp = pow(10,p);
  do
    {
      cm = floor (cminy/dp)*dp;
      dp/=10;
    } while (fabs (cm-cminy)>.1*h);
  cminy=cm;
  
  dp = pow(10,p);
  do
    {
      cm = ceil (cmaxy/dp)*dp;
      dp/=10;
    } while (fabs (cm-cmaxy)>.1*h);
  cmaxy = cm;
  
  
  
  const double closetozero = .20;
  if (cminx < closetozero*w && cminx>0)
    cminx = 0;
  if (cminy < closetozero*h && cminy>0)
    cminy = 0;
  
  if (-cmaxx < closetozero*w && cmaxx<0)
    cmaxx = 0;
  if (-cmaxy < closetozero*h && cmaxx<0)
    cmaxy = 0;
  
  
  printf ("COORDS %f %f %f %f\n",	  cminx, cmaxx, cminy, cmaxy);

  return  KPCoords (cminx, cmaxx, cminy, cmaxy,
		    cminx, cminy);

}



void
KPS2DParametric::installDefaultRMBMenu (void)
{
  rmbmenu = new QPopupMenu;
  rmbmenu->insertItem ("&Properties", this, SLOT(slotProperties()));
  if (xAxis())
    xAxis()->installDefaultRMBMenu();
  if (yAxis())
    yAxis()->installDefaultRMBMenu();
  if (coordFrame())
    coordFrame()->installDefaultRMBMenu();
}


QWidget *
KPS2DParametric::propWidget (unsigned int matrixnum, 
			     QWidget *parent, const char *name)
{
  return  new KPS2DParametricSeriesProp (this, matrixnum, parent, name);
}


void
KPS2DParametric::slotProperties ()
{
  QTabDialog d (0, 0, true);

  unsigned int i;
  KPS2DParametricSeriesProp **p =
    new KPS2DParametricSeriesProp *[plots.count()];

  //  QString qs;
  for (i=0;i<plots.count();i++)
    {
      p [i] = (KPS2DParametricSeriesProp *)propWidget (i, &d);
      d.addTab (p[i], plots.at(i)->pname);
    }

  d.setOkButton();
  d.setCancelButton();
  if (d.exec())
    {
      for (i=0;i<plots.count();i++)
	{
	  setPointSize (p[i]->pointSize(), i);
	  setPointStyle (p[i]->pointStyle(), i);
	  setConnected (p[i]->connected(), i);
	  setColor (p[i]->color(), i);
	  delete p[i];
	}
      update();
    }

  //  delete propwid;
  //  propwid=0;
}

void
KPS2DParametric::setOptions (const KPS2DParametricOptions &op,
			     unsigned int series)
{
  KPS2DParametricOptions *opp;

  opp = new KPS2DParametricOptions (op);
  plots.remove (series);
  plots.insert (series, opp);

  bneedredraw=true;
}

int
KPS2DParametric::whatami() const
{
  return RTTI_KPS2DParametric;
}
  
//////////



KPS2DParametricSeriesProp::KPS2DParametricSeriesProp 
(KPS2DParametric *kp2p, unsigned int matrixnum, QWidget *parent, const char *name, WFlags f) :
QWidget (parent, name, f)
{
  QGridLayout *layout = new QGridLayout (this, 4, 2, 10);
  
  cbcon = new QCheckBox ("Co&nnect markers", this);
  cbcon->setChecked (kp2p->connected(matrixnum));
  cbcon->setMinimumSize (cbcon->sizeHint());
  layout->addMultiCellWidget (cbcon, 0, 0, 0, 1);
  
  lesize = new KLineEdit (this);
  QString qs;
  qs.setNum (100.*kp2p->pointSize(matrixnum));
  lesize->setText (qs);
  lesize->setMinimumSize (lesize->sizeHint());
  //  lesize->setMaximumSize (lesize->sizeHint());
  layout->addWidget (lesize, 1, 1);
  
  cbstyle = new QComboBox (false, this);
  int i;
  for (i=0;i<npointstyles;i++)
    cbstyle->insertItem (dpointstyle[i].name);
  cbstyle->setCurrentItem (kp2p->pointStyle(matrixnum));
  cbstyle->setMinimumSize (cbstyle->sizeHint());
  //  cbstyle->setMaximumHeight (cbstyle->sizeHint().height());
  connect (cbstyle, SIGNAL (activated(int)),
	   this, SLOT (slotCombo(int)));
  slotCombo (cbstyle->currentItem());
  layout->addWidget (cbstyle, 2, 1);

  colb = new KColorButton (kp2p->color (matrixnum), this);
  colb->setMinimumSize (colb->sizeHint());
  layout->addWidget (colb, 3,1);

  QLabel *label = new QLabel(lesize, "&Marker size (% of plot):", this);
  label->setMinimumSize (label->sizeHint());
  layout->addWidget (label, 1, 0);
  label = new QLabel (cbstyle, "Marker &style:", this);
  label->setMinimumSize (label->sizeHint());
  layout->addWidget (label, 2, 0);
  label = new QLabel (colb, "Marker/line &color:", this);
  label->setMinimumSize (label->sizeHint());
  layout->addWidget (label, 3, 0);
    
  
  layout->activate();
}

void
KPS2DParametricSeriesProp::slotCombo (int id)
{
  if (id==0)
    lesize->setEnabled (false);
  else
    lesize->setEnabled (true);
}

