/***************************************************************************
                          kcontainerimpl.cpp  -  description
                             -------------------
    begin                : Mon Oct 16 2000
    copyright            : (C) 2000 by Sergio Moretti
    email                : sermore@libero.it
    revision             : $Revision: 1.4 $
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <kdebug.h>
#include <klocale.h>
#include "kfactoryimpl.h"
#include "kcontainerimpl.h"

const char KContainerImpl::DOCID[] = "Container";
const char KContainerImpl::MIMETYPE[][50] = { "application/" DOC "-container",
					    "text/x-" DOC "-container" };
const char KContainerImpl::DOCTYPE[] = "<!DOCTYPE Container >";


KContainerImpl::KContainerImpl(int type) : KObjectImpl(type) {
  _items.setAutoDelete(false);
}

KContainerImpl::~KContainerImpl() {
  //kdDebug(D_INI) << name() << ": destroy" << endl;
  _items.setAutoDelete(true);
  clear();
}

void KContainerImpl::clear() {
  kdDebug(D_INI) << name() << ": clear" << endl;
  _items.setAutoDelete(true);
  _items.clear();
  _items.setAutoDelete(false);
}

KObjectImpl * KContainerImpl::findGlobal(int type) {
  KObjectImpl * glb = _glbDict[type];
  kdError(glb == 0, D_INI) << name() << ": unknown global " << type << endl;
  return glb;
}

KObjectImpl * KContainerImpl::globalNew(int type) {
  kdDebug(D_INI) << name() << ": global item new " << type << endl;
  // create and insert new global instance
  QString typeStr = KFactoryImpl::factory()->typeStr(type);
  KObjectImpl * obj = dynamic_cast<KObjectImpl*>(KFactoryImpl::factory()->createObject(type));
  QDomElement e = domDoc().createElement(obj->docId());
  domGlobals().appendChild(e);
  e.setAttribute("Id", 0);
  e.setAttribute("Type", type);
  e.setAttribute("Name", QString("G%1-%2").arg(id()).arg(typeStr));
  if (dynamic_cast<KContainerImpl*>(obj) != 0)
    e.setAttribute("ItemType", "0");
  obj->load(this, e);
  _glbDict.insert(type, obj);
  return obj;
}

KObjectImpl * KContainerImpl::globalLoad(const QDomElement &e) {
  kdDebug(D_INI) << name() << ": global item load " << e.attribute("Name") << endl;
  // load and insert global instance
  int type = e.attribute("Type").toInt();
  KObjectImpl * obj = dynamic_cast<KObjectImpl*>(KFactoryImpl::factory()->createObject(type));
  obj->load(this, e);
  _glbDict.insert(type, obj);
  return obj;
}

void KContainerImpl::itemActivate(bool on) {
  _itemActiveCount += (on ? 1 : -1);
  if (!isRoot())
    container()->itemActivate(on);
}

bool KContainerImpl::itemAdd(KObjectImpl *item) {
  kdDebug(D_INI) << name() << ": item add " << item->name() << endl;
  if (_items.findRef(item) != -1)
    throw Error(name(), QString("item %1 duplicate").arg(item->name()));
  if (itemFindId(item->id()) != 0)
    throw Error(name(), QString("id %1 already present").arg(item->id()));
  if (!isItemInsertable(item))
    return false;
  _items.append(item);
  itemChanged(item, MOD_ADD, PRP_ALL);
  return true;
}

void KContainerImpl::itemChanged(KObjectImpl *item, ModType type, 
				 PropMode prop) {
  if (!isRoot())
    container()->itemChanged(item, type, prop);
  if (prop == PRP_ALL)
    setModified(MOD_STATE, PRP_THIS, true);

}

KObjectImpl * KContainerImpl::itemCopy(const QDomElement &e) {
  kdDebug(D_INI) << name() << ": item copy " << e.attribute("Name") << endl;
  QDomElement i = e.cloneNode().toElement();
  domItems().appendChild(i);
  int objId = assignId();
  int type = e.attribute("Type").toInt();
  QString name = 
    QString("%1-%2").arg(KFactoryImpl::factory()->typeStr(type)).arg(objId);
  i.setAttribute("Id", objId);
  i.setAttribute("Name", name);
  return itemLoad(i);
}

KObjectImpl * KContainerImpl::itemCreate(int type) {
  kdDebug(D_INI) << name() << ": item create type " << type << endl;
  kdFatal(findGlobal(type) == 0, D_INI) << name() << ": unknown global " << type << endl;
  kdFatal(type & _itemType != _itemType, D_INI) << name() << " WRONG ITEM TYPE" 
						<< type << endl;
  // create new object without initializing it
  KObjectImpl *obj = KFactoryImpl::factory()->createObject(type);
  return obj;
} 

KObjectImpl * KContainerImpl::itemFindId(int id) {
  for (KObjectImpl *i = itemFirst(); i != 0; i = itemNext())
    if (i->id() == id)
      return i;
  return 0;
}

KObjectImpl* KContainerImpl::itemLoad(const QDomElement &e) {
  kdDebug(D_INI) << name() << ": item load " << e.attribute("Name") << endl;
  int type = e.attribute("Type").toInt();
  KObjectImpl *obj = itemCreate(type);
  obj->load(this, e);
  if(!itemAdd(obj)) {
    delete obj;
    return 0;
  }
  return obj;
}

bool KContainerImpl::itemMove(KObjectImpl *item, KContainerImpl *to) {
  kdDebug(D_INI) << name() << ": item move " << item->name() 
		 << " to " << to->name() << endl;
  if (to->itemCopy(item->dom()) != 0) {
    itemRemove(item);
    return true;
  }
  return false;
}

KObjectImpl* KContainerImpl::itemNew(int type) {
  kdDebug(D_INI) << name() << ": item new" << endl;
  QString docId = KFactoryImpl::factory()->docId(type);
  int objId = assignId();
  QString name = 
    QString("%1-%2").arg(KFactoryImpl::factory()->typeStr(type)).arg(objId);
  QDomElement e = domDoc().createElement(docId);
  domItems().appendChild(e);
  e.setAttribute("Id", objId);
  e.setAttribute("Name", name);
  e.setAttribute("Type", type);
  return itemLoad(e);
}

void KContainerImpl::itemRemove(KObjectImpl *item) {
  kdDebug(D_INI) << name() << ": item remove " << item->name() << endl;
  if (_items.findRef(item) == -1)
    throw Error(name(), QString("%1 not in container").arg(item->name()));
  if (isModified(MOD_ALL)) {
    kdWarning(D_INI) << name() << ": removing modified " << item->name() << endl;
    item->setModified(MOD_ALL, PRP_NONE, false);
  }
  _items.remove(item);
  domItems().removeChild(item->dom());
  itemChanged(item, MOD_REMOVE, PRP_ALL);
  delete item;
}

void KContainerImpl::loadData() {
  KObjectImpl::loadData();
  kdDebug(D_INI) << name() << ": container LoadData" << endl;
  if (!dom().hasAttribute("ItemType")) {
    throw DomError(name(), "loadData: ItemType not found", dom());
    //kdFatal(D_INI) << name() << ": ItemType not found in:\n" << dom().ownerDocument().toString() << endl;
  }
  // data
  if (!isRoot())
    _doc = container()->domDoc();
  _itemType = dom().attribute("ItemType", "0").toInt();
  _itemActiveCount = 0;
  // global elements
  _domGlobals = dom().namedItem("Globals").toElement();
  if (_domGlobals.isNull()) {
    _domGlobals = domDoc().createElement("Globals");
    dom().appendChild(_domGlobals);
    IntList lst = KFactoryImpl::factory()->typeList(_itemType);
    for(IntList::Iterator i = lst.begin(); i != lst.end(); i++)
      globalNew(*i);
  } else {
    QDomNode n = domGlobals().firstChild();
    while (!n.isNull()) {
      QDomElement item = n.toElement();
      if (item.isNull())
	throw DomError(name(), i18n("node unknown"), n);
      if (KFactoryImpl::factory()->prototype(item.attribute("Type").toInt()) != 0)
	globalLoad(item);
      else
	kdWarning(D_RUN) << name() << " skip unknown item " << item.attribute("Name") << endl;
      n = n.nextSibling();
    }
    // check to see if all globals are loaded
    IntList lst = KFactoryImpl::factory()->typeList(_itemType);
    for(IntList::Iterator i = lst.begin(); i != lst.end(); i++)
      if (findGlobal(*i) == 0) {
	kdWarning(D_RUN) << name() << " add Global " << KFactoryImpl::factory()->typeStr(*i) << endl;
	globalNew(*i);
      }
    setModified(MOD_STATE, PRP_NONE);
  }
  // child elements
  _domItems = dom().namedItem("Items").toElement();
  if (_domItems.isNull()) {
    _domItems = domDoc().createElement("Items");
    dom().appendChild(_domItems);
  } else {
    QDomNode n = domItems().firstChild();
    while (!n.isNull()) {
      QDomElement item = n.toElement();
      if (item.isNull())
	throw DomError(name(), i18n("node unknown"), n);
      if (KFactoryImpl::factory()->prototype(item.attribute("Type").toInt()) != 0)
	itemLoad(item);
      else
	kdWarning(D_RUN) << name() << " skip unknown item " << item.attribute("Name") << endl;
      n = n.nextSibling();
    }
  }
}

void KContainerImpl::runPeriodically() {
  KObjectImpl::runPeriodically();
  for(KObjectImpl *obj = itemFirst(); obj != 0; obj = itemNext())
    obj->runPeriodically();
}

void KContainerImpl::updateState() {
  KObjectImpl::updateState();
  // update globals
  for (ObjDictIterator i(_glbDict); i.current() != 0; ++i)
    i.current()->update();
  // update childs
  for (KObjectImpl *obj = itemFirst(); obj != 0; obj = itemNext())
    obj->update();
}

#include "kcontainerimpl.moc"
