/***************************************************************************
                          kmago.cpp  -  description
                             -------------------
    begin                : Mon Sep 18 2000
    copyright            : (C) 2000 by Sergio Moretti
    email                : sermore@libero.it
    revision             : $Revision: 1.18 $
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qdir.h>
#include <qprinter.h>
#include <qpainter.h>
#include <qclipboard.h>
#include <kurl.h>
#include <qfileinfo.h>
#include <kiconloader.h>
#include <kfiledialog.h>
#include <kdebug.h>
#include <kmenubar.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kstdaction.h>
#include <qdragobject.h>
#include <qtooltip.h>

// application specific includes
#include "utils.h"
#include "ktmanager.h"
#include "ktransfer.h"
#include "ktmanagerdrag.h"
#include "ktransferdrag.h"
#include "kdlgopentransfer.h"
#include "kdlgopenmanager.h"
#include "kdlgapp.h"
#include "kdlgtransferstatus.h"
#include "klogviewer.h"
#include "kdlgmanager.h"
#include "kdlgtransfer.h"
#include "ksystrayview.h"
#include "kmago.h"
#include "kmagoview.h"
#include "kmagodoc.h"


const char KMagoApp::IDS_STATUS_DEFAULT[] = "Ready.";


KMagoApp::KMagoApp() {
  static int appCounter = 0;
  setName(QString("App-%1").arg(++appCounter));

  _timer = new QTimer(this);

  initActions();
  initMenuBar();
  initToolBar();
  initStatusBar();
  initDocument();
  initView();
  initSysTray();

  connect(kapp->clipboard(), SIGNAL(dataChanged()), 
	  this, SLOT(slotClipboardChange()));

  // init state of actions
  editCut->setEnabled(false);
  editCopy->setEnabled(false);
  editPaste->setEnabled(false);
  viewToolBarMain->setChecked(true);
  viewToolBarTrn->setChecked(true);
  viewManagers->setChecked(true);
  viewSplitOrient->setChecked(false);
  viewSysTray->setChecked(false);
  transferRemove->setEnabled(false);
  transferConf->setEnabled(false);
  transferLog->setEnabled(false);
  transferStatus->setEnabled(false);
  transferPause->setEnabled(false);
  transferResume->setEnabled(false);
  transferCheck->setEnabled(false);
  transferAutoResume->setEnabled(false);
  transferAutoCheck->setEnabled(false);
  slotStatusMsg(i18n(IDS_STATUS_DEFAULT));

  //_currentManager = 0;
  //_currentTransfer = 0;
  _transferCount = 0;
  _tmpDoc = false;
  _savedState = -1; // all bits set

  loadState();
}

KMagoApp::~KMagoApp() {
  delete _timer;
}

void KMagoApp::initActions() {
  fileNewWindow = new KAction(i18n("New Window"),
			      QIconSet(kapp->miniIcon()), 0,
			      this, SLOT(slotFileNewWindow()),
			      this);
  fileNewWindow->setToolTip(i18n("Opens a new application window"));
  fileNew = KStdAction::openNew(this, SLOT(slotFileNew()), this);
  //fileNew->setToolTip(i18n("Creates a new document"));
  fileOpen = KStdAction::open(this, SLOT(slotFileOpen()), this);
  //fileOpen->setToolTip(i18n("Opens an existing document"));
  fileOpenRecent = 
    KStdAction::openRecent(this, SLOT(slotFileOpenRecent(const KURL&)), 
			   this);
  //fileOpenRecent->setToolTip(i18n("Opens a recently used file"));
  fileClose = KStdAction::close(this, SLOT(slotFileClose()), 
				this);
  //fileQuit->setToolTip(i18n("Closes the actual document"));
  fileSaveAs = KStdAction::saveAs(this, SLOT(slotFileSaveAs()), 
				  this);
  //fileSaveAs->setToolTip(i18n("Saves the actual document as..."));
  fileQuit = KStdAction::quit(this, SLOT(slotFileQuit()), 
			      this);
  //fileQuit->setToolTip(i18n("Quits the application"));
  fileSelectMngr = new KListAction(i18n("Select Manager"),
				   0, this, SLOT(slotFileSelectMngr(int)),
				   this);
  fileAddMngr = new KAction(i18n("Add Manager..."),
			    0, this, SLOT(slotFileAddMngr()),
			    this);
  managerRemoveMngr = new KAction(i18n("Remove Manager"),
			       0, this, SLOT(slotManagerRemoveMngr()),
			       this);

  editCopy = KStdAction::copy(this, SLOT(slotEditCopy()), this);
  //editCopy->setToolTip(i18n("Copies the selected section to the clipboard"));
  editCut = KStdAction::cut(this, SLOT(slotEditCut()), this);
  //editCut->setToolTip(i18n("Cuts the selected section and puts it to the clipboard"));
  editPaste = KStdAction::paste(this, SLOT(slotEditPaste()), this);
  //editPaste->setToolTip(i18n("Pastes the clipboard contents to actual position"));

  managerAdd = new KAction(i18n("Add Transfer..."),
			   QIconSet(BarIcon("queue")), 
			   0, this, SLOT(slotManagerAdd()),
			   this);
  managerAdd->setToolTip(i18n("Add a transfer"));
  managerAddRecent = 
    new KRecentFilesAction(i18n("Add Recent Transfer"),
			   0, this, SLOT(slotManagerAddRecent(const KURL&)),
			   this);
  managerAddRecent->setToolTip(i18n("Add a recent transfer"));
  managerAutoDownload = 
    new KToggleAction(i18n("Auto Download"),
		      QIconSet(BarIcon("run")), 0,
		      this, SLOT(slotManagerAutoDownload()),
		      this);
  managerAutoDownload->setToolTip(i18n("Automatic Downloading"));
  managerConf = new KAction(i18n("Configure"),
			    0, this, SLOT(slotManagerConfigure()),
			    this);
  managerConf->setToolTip(i18n("Open the Manager Configuration Dialog"));
  managerPaste = new KAction(i18n("Paste Transfer..."),
			     0, this, SLOT(slotManagerPaste()),
			     this);
  managerPaste->setToolTip(i18n("Add a transfer from clipboard"));
  managerPauseAll = new KAction(i18n("Pause All"),
				QIconSet(BarIcon("stop")),
				0, this, SLOT(slotManagerPauseAll()),
				this);
  managerPauseAll->setToolTip(i18n("Pause all transfers"));
  
  transferAutoCheck = 
    new KToggleAction(i18n("Auto Check Resume"),
		      QIconSet(BarIcon("wizard")),
		      0, this, SLOT(slotTransferAutoCheck()),
		      this);
  transferAutoCheck->setToolTip(i18n("Enable/disable automatic check resuming"));
  transferAutoResume = 
    new KToggleAction(i18n("Auto Resume"),
		      QIconSet(BarIcon("redo")),
		      0, this, SLOT(slotTransferAutoResume()),
		      this);
  transferAutoResume->setToolTip(i18n("Enable/disable automatic resuming"));
  transferCheck = new KAction(i18n("Check Resume"),
			      0, this, SLOT(slotTransferCheck()),
			      this);
  transferCheck->setToolTip(i18n("Check for transfer resume capability"));
  transferConf = new KAction(i18n("Configure"),
			     QIconSet(BarIcon("document")),
			     0, this, SLOT(slotTransferConfigure()),
			     this);
  transferConf->setToolTip(i18n("Open the transfer configure dialog"));
  transferPause = new KAction(i18n("Pause"),
			      QIconSet(BarIcon("player_pause")),
			      0, this, SLOT(slotTransferPause()),
			      this);
  transferPause->setToolTip(i18n("Pause the current transfer"));
  transferRemove = new KAction(i18n("Remove"),
			       QIconSet(BarIcon("trashcan_empty")),
			       0, this, SLOT(slotTransferRemove()),
			       this);
  transferRemove->setToolTip(i18n("Remove the current transfer"));
  transferResume = new KAction(i18n("Resume"),
			       QIconSet(BarIcon("1rightarrow")),
			       0, this, SLOT(slotTransferResume()),
			       this);
  transferResume->setToolTip(i18n("Start/resume the current transfer"));
  transferStatus = new KAction(i18n("Status"),
			       0, this, SLOT(slotTransferStatus()),
			       this);
  transferStatus->setToolTip(i18n("View status of the current transfer"));
  transferLog = new KAction(i18n("View Log"),
			    0, this, SLOT(slotTransferLog()),
			    this);
  transferLog->setToolTip(i18n("View log of the current transfer process"));
  
  viewRefresh = KStdAction::redisplay(this, SLOT(slotViewRefresh()),
				      this);
  //viewRefresh->setToolTip(i18n("Refresh views"));
  viewStatusBar = KStdAction::showStatusbar(this, 
					    SLOT(slotViewStatusBar()), 
					    this);
  //viewStatusBar->setToolTip(i18n("Enables/disables the statusbar"));
  viewToolBarMain = 
    KStdAction::showToolbar(this, SLOT(slotViewToolBarMain()), 
			    this);
  //viewToolBarMain->setToolTip(i18n("Enables/disables the standard toolbar"));
  viewToolBarTrn = new KToggleAction(i18n("Show Transfers Toolbar"),
				     0, this, SLOT(slotViewToolBarTrn()),
				     this);
  viewToolBarTrn->setToolTip(i18n("Enables/disables the transfers toolbar"));
  viewPrefs = KStdAction::preferences(this, SLOT(slotViewPreferences()), 
				      this);
  //viewPrefs->setToolTip(i18n("Open the Preferences Dialog"));
  viewManagers = new KToggleAction(i18n("Show Managers List"), 
				   0, this, SLOT(slotViewManagers()),
				   this);
  viewSplitOrient = new KToggleAction(i18n("Vertical Orientation"),
				      0, this, SLOT(slotViewSplitOrient()),
				      this);
  viewSysTray = new KToggleAction(i18n("System Tray"),
				  0, this, SLOT(slotViewSysTray()),
				  this);
  
  separator = new KActionSeparator(this, "separator");
  kdFatal(separator == 0) << "???? NULL" << endl;
}

void KMagoApp::initMenuBar() {
  
  //KAction *separator = this->action("separator");
  kdFatal(separator == 0) << "SEPARATOR NULL" << endl;
  
  QPopupMenu *menu = new QPopupMenu();
  fileNewWindow->plug(menu);
  separator->plug(menu);
  fileNew->plug(menu);
  fileOpen->plug(menu);
  fileOpenRecent->plug(menu);
  fileClose->plug(menu);
  separator->plug(menu);
  fileSaveAs->plug(menu);
  separator->plug(menu);
  fileSelectMngr->plug(menu);
  fileAddMngr->plug(menu);
  separator->plug(menu);
  fileQuit->plug(menu);
  menuBar()->insertItem(i18n("&File"), menu);

  menu = new QPopupMenu();
  editCut->plug(menu);
  editCopy->plug(menu);
  editPaste->plug(menu);
  menuBar()->insertItem(i18n("&Edit"), menu);

  menu = new QPopupMenu();
  viewToolBarMain->plug(menu);
  viewToolBarTrn->plug(menu);
  viewStatusBar->plug(menu);
  viewManagers->plug(menu);
  viewSplitOrient->plug(menu);
  viewSysTray->plug(menu);
  separator->plug(menu);
  viewRefresh->plug(menu);
  separator->plug(menu);
  viewPrefs->plug(menu);
  menuBar()->insertItem(i18n("&View"), menu);
  
  _managerMenu = menu = new QPopupMenu();
  managerAdd->plug(menu);
  managerPaste->plug(menu);
  managerAddRecent->plug(menu);
  separator->plug(menu);
  managerPauseAll->plug(menu);
  managerAutoDownload->plug(menu);
  separator->plug(menu);
  managerConf->plug(menu);
  separator->plug(menu);
  managerRemoveMngr->plug(menu);
  menuBar()->insertItem(i18n("&Manager"), menu);
  
  _transferMenu = menu = new QPopupMenu();
  transferResume->plug(menu);
  transferPause->plug(menu);
  separator->plug(menu);
  transferRemove->plug(menu);
  separator->plug(menu);
  transferCheck->plug(menu);
  separator->plug(menu);
  transferAutoCheck->plug(menu);
  transferAutoResume->plug(menu);
  separator->plug(menu);
  transferStatus->plug(menu);
  transferLog->plug(menu);
  separator->plug(menu);
  transferConf->plug(menu);
  menuBar()->insertItem(i18n("&Transfer"), menu);

  menuBar()->insertSeparator();
  menuBar()->insertItem(i18n("&Help"), helpMenu("KMago " VERSION "\n\n(c) 2000 by\nSergio Moretti\nsermore@libero.it"));
}

void KMagoApp::initToolBar() {

  //KAction *separator = this->action("separator");
  kdFatal(separator == 0) << "SEPARATOR NULL" << endl;

  KToolBar *toolbar = toolBar("main");

  fileNew->plug(toolbar);
  fileOpen->plug(toolbar);

  separator->plug(toolbar);
  editCut->plug(toolbar);
  editCopy->plug(toolbar);
  editPaste->plug(toolbar);

  toolbar = toolBar("transfer");

  managerAdd->plug(toolbar);

  separator->plug(toolbar);
  managerAutoDownload->plug(toolbar);
  managerPauseAll->plug(toolbar);

  separator->plug(toolbar);
  transferResume->plug(toolbar);
  transferPause->plug(toolbar);

  separator->plug(toolbar);
  transferRemove->plug(toolbar);

  separator->plug(toolbar);
  transferAutoCheck->plug(toolbar);
  transferAutoResume->plug(toolbar);
}

void KMagoApp::initStatusBar() {
  statusBar()->insertFixedItem("XXXXXXXXXXXXXXXXXXXXXXXXXXXXX", STATUS_MSG);
  statusBar()->changeItem(IDS_STATUS_DEFAULT, STATUS_MSG);
  statusBar()->insertFixedItem("000/000", STATUS_COUNT);
  statusBar()->insertFixedItem("00000/00000", STATUS_BAND);
  statusBar()->insertFixedItem("000/000", STATUS_GLB_COUNT);
  statusBar()->insertFixedItem("00000/00000", STATUS_GLB_BAND);
  //statusBar()->setMaximumWidth(size().width());
}

void KMagoApp::initDocument() {

  doc = new KMagoDoc(this, QString("%1-Doc").arg(name()));
  //doc->newDocument();
  //FIXME!! inizializzazione in readProperties se restore from session
  connect(_timer, SIGNAL(timeout()), doc,  SLOT(slotUpdate()));
}

void KMagoApp::initView() {
  view = new KMagoView(this, QString("%1-View").arg(name()));
  //doc->addView(view);
  setCentralWidget(view);
  //setView(view);
  //setCaption(doc->title());
}

void KMagoApp::initSysTray() {
  sysTray = new KSysTrayView(this, QString("%1-TView").arg(name()));
}

QSize KMagoApp::sizeHint() const {
  return QSize(500, 280);
}

/*
void KMagoApp::resizeEvent ( QResizeEvent * ) {
  statusBar()->setMaximumWidth(size().width());
}
*/

bool KMagoApp::newDocument() {
  kdDebug(D_VIE) << name() << ": document new" << endl;
  closeDocument();
  bool check = doc->openNew();
  kdFatal(check == false, D_INI) << "CAN'T OPEN NEW DOCUMENT" << endl;
  //setCaption(getDocument()->title() + " - " + currentManager()->title());
  _tmpDoc = true;
  resetView();
  return true;
}

bool KMagoApp::openDocument(const QString & fileToOpen) {
  kdDebug(D_VIE) << name() << ": document open " << fileToOpen << endl;
  closeDocument();
  if (!doc->open(fileToOpen)) {
    KMessageBox::sorry(this, i18n("unable to open " + fileToOpen),
		       i18n("Error opening file"), false);
    return newDocument();
  }
  //setCaption(getDocument()->title() + " - " + currentManager()->title());
  fileOpenRecent->addURL(fileToOpen);
  _tmpDoc = false;
  _lastUsedDocument = fileToOpen;
  resetView();
  return true;
}

bool KMagoApp::openLastUsedDocument() {
  kdDebug(D_VIE) << name() << ": last used document open " << _lastUsedDocument << endl;
  if (_lastUsedDocument.isEmpty())
    return newDocument();
  return openDocument(_lastUsedDocument);
}

bool KMagoApp::saveAsDocument(const QString &newFile) {
  kdDebug(D_VIE) << name() << ": document save as " << newFile << endl;
  if (!doc->saveAs(newFile))
    return false;
  //setCaption(getDocument()->title() + " - " + currentManager()->title());
  fileOpenRecent->addURL(newFile);
  _lastUsedDocument = newFile;
  resetView();
  return true;
}

void KMagoApp::closeDocument() {
  kdDebug(D_VIE) << name() << ": document close" << endl;
  if (getDocument()->isClosed())
    return; // document already closed;
  QString oldfile = doc->manager().fileName();
  getDocument()->close();
  if (_tmpDoc)
    QDir::current().remove(oldfile);
}

void KMagoApp::saveConfig(KConfig *config) {
  kdDebug(D_INI) << name() << ": save config" << endl;
  config->writeEntry("System Tray", viewSysTray->isChecked());
  config->writeEntry("QSplitter Sizes", view->sizes());
  config->writeEntry("Split Vertical Orientation", viewSplitOrient->isChecked());
  config->writeEntry("View Managers", viewManagers->isChecked());
  config->writeEntry("Open Flags", _cfg_openFlags);
  config->writeEntry("Update Time", _cfg_updateTime);
  config->writeEntry("Expert Mode", _cfg_expertMode);
  config->writeEntry("Start Mode", _cfg_startMode);
  config->writeEntry("Load Last Doc", _cfg_loadLastDoc);
  if (!_lastUsedDocument.isEmpty())
    config->writeEntry("Last Used Document", _lastUsedDocument);
  fileOpenRecent->saveEntries(config, "Recent Files");
  managerAddRecent->saveEntries(config, "Recent Transfers");
}

void KMagoApp::readConfig(KConfig* config) {
  viewSysTray->setChecked(config->readBoolEntry("System Tray"));
  slotViewSysTray();
  viewSplitOrient->setChecked(config->readBoolEntry("Split Vertical Orientation"));
  slotViewSplitOrient();
  view->setSizes(config->readIntListEntry("QSplitter Sizes"));
  viewManagers->setChecked(config->readBoolEntry("View Managers"));
  slotViewManagers();

  _cfg_openFlags = config->readNumEntry("Open Flags", 0);
  _cfg_expertMode = config->readBoolEntry("Expert Mode", false);
  _cfg_updateTime = config->readNumEntry("Update Time", 5);
  _cfg_startMode = config->readNumEntry("Start Mode", START_PROMPT);
  _cfg_loadLastDoc = config->readBoolEntry("Load Last Doc", true);
  _timer->start(_cfg_updateTime*1000);
  _lastUsedDocument = config->readEntry("Last Used Document", "");
  fileOpenRecent->loadEntries(config, "Recent Files");
  managerAddRecent->loadEntries(config, "Recent Transfers");
#ifdef KACTION_ERR
  sysTray->managerAddRecent->loadEntries(config, "Recent Transfers");
#endif
}

/** save state, called from session manager */
void KMagoApp::saveProperties(KConfig *config) {
  kdDebug(D_INI) << name() << ": save properties" << endl;
  saveConfig(config);
}

/** read state, and open document, called from session manager */
void KMagoApp::readProperties(KConfig* config) {
  kdDebug(D_INI) << name() << ": read properties" << endl;
  readConfig(config);
  // document initialization
  openLastUsedDocument();
}

/** load initial state, called by init */
void KMagoApp::loadState() {
  KConfig * config = kapp->config();
  QString grp = config->group();
  config->setGroup("Configuration");
  readConfig(config);
  config->setGroup(grp);
}  

/** save state before closing, called by queryExit */
void KMagoApp::saveState() {
  KConfig * config = kapp->config();
  QString grp = config->group();
  config->setGroup("Configuration");
  saveConfig(config);
  config->setGroup(grp);
}

bool KMagoApp::queryClose() {
  if (getDocument()->manager().itemActiveCount() > 0)
    if (getExpertMode() 
	|| KMessageBox::questionYesNo(this,
				      i18n("Active Transfers exist, confirm exit application?"),
				      i18n("Exit"),
				      QString::null, QString::null, 
				      false) == KMessageBox::Yes) {
      return true;
    } else 
      return false;
  return true;
}

bool KMagoApp::queryExit() {
  saveState();
  //closeDocument();
  return true;
}

bool KMagoApp::stateChange(int p, bool val) {
  int bitv = val << p;
  if (_savedState ^ bitv) {
    _savedState = (_savedState & ~(1 << p)) | bitv;
    return true;
  }
  return false;
}

/** add a new transfer */
void KMagoApp::addTransfer(const KURL &u) {
  KTManager mngr = currentManager();
  int type = mngr.getDefaultType();
  KURL rmt = u;
  KURL lcl;
  lcl.setPath(QFileInfo(mngr.getDownloadDir(), rmt.fileName()).absFilePath());
  if ((rmt.path().isEmpty() || !(getOpenFlags() & FLG_NO_PROMPT))
      && !KDlgOpenTransfer::getOpenTransfer(mngr, rmt, lcl, type, 
					    getOpenFlags())
      )
    return;
  KTransfer glb = mngr.findGlobal(type);
  if (glb.isNull()) {
    KMessageBox::sorry(this,
		       i18n("Transfer Type not acceptable"),
		       i18n("transfer type not acceptable"), false);
    return;
  }
  if (!glb.isProtocolSupported(rmt.protocol())) {
    KMessageBox::sorry(this,
		       i18n("Protocol " + rmt.protocol() + " not supported"),
		       i18n("protocol not supported"), false);
    return;
  }
  if (lcl.path().isEmpty() || QFileInfo(lcl.path()).isDir()) {
    KMessageBox::sorry(this,
		       i18n("File " + lcl.path() + " not acceptable"),
		       i18n("File not acceptable"), false);
    return;
  }
  // check for a transfer with these file or url
  int chk = getDocument()->isDuplicate(rmt, lcl);
  if (chk)
    switch (chk) {
    case 1:
      KMessageBox::sorry(this,
			 i18n("File " + lcl.path() + " in use"),
			 i18n("File in use"), false);
      return;
    case 2:
      KMessageBox::sorry(this,
			 i18n("Url " + rmt.url() + " in use"),
			 i18n("Url in use"), false);
      return;
    }
  // checking for file recover
  QString file, tmpfile;
  file = lcl.path();
  tmpfile = glb.getTempFile(rmt, lcl).path();
  if (QDir::current().exists(file)) {
    if (QDir::current().exists(tmpfile) && file != tmpfile) {
      // either tmpfile and file exists
      switch (promptFilesToRecover(file, tmpfile)) {
      case KMessageBox::Ok: // recover file
	QDir::current().remove(tmpfile);
	QDir::current().rename(file, tmpfile);
	break;
      case KMessageBox::Yes: // recover tmpfile
	QDir::current().remove(file);
	break;
      case KMessageBox::No: // no recover
	QDir::current().remove(file);
	QDir::current().remove(tmpfile);
	break;
      case KMessageBox::Cancel:
	return;
      default: // cancel operation
	kdFatal(D_INI) << "PROMPT ERROR" << endl;
      } // end switch
    } else { // tmpfile not exists
      // recover file
      switch (promptFileToRecover(file)) {
      case KMessageBox::Yes: // recover file
	QDir::current().rename(file, tmpfile);
	break;
      case KMessageBox::No: // don't recover
	QDir::current().remove(file);
	break;
      case KMessageBox::Cancel: // cancel operation
	return;
      default:
	kdFatal(D_INI) << "PROMPT ERROR" << endl;
      }
    } // end else
  } else { // file don't exists
    if (QDir::current().exists(tmpfile)) {
      // recover tmpfile
      switch (promptFileToRecover(tmpfile)) {
      case KMessageBox::Yes: // recover file
	break;
      case KMessageBox::No: // don't recover
	QDir::current().remove(tmpfile);
	break;
      case KMessageBox::Cancel:
	return;
      default: // cancel operation
	kdFatal(D_INI) << "PROMPT ERROR" << endl;
      }
    }
  } // end else
  // evaluting start mode
  int start = getStartMode();
  if (start == START_PROMPT) {
    switch (promptStartMode(i18n("Select a start mode"),
			    i18n("Transfer:\n" + rmt.url() + "\nto " + file))
	    ) {
    case KMessageBox::Yes:
      start = START_NOW;
      break;
    case KMessageBox::No:
      start = START_NO;
      break;
    case KMessageBox::Cancel:
      return;
    default:
      kdFatal(D_INI) << "PROMPT ERROR" << endl;
    }
  }
  // add transfer	
  managerAddRecent->addURL(rmt);
#ifdef KACTION_ERR
  sysTray->managerAddRecent->addURL(rmt);
#endif
  KTransfer t = mngr.itemNew(type, rmt, lcl);
  if (start == START_NOW)
    t.start();
}

void KMagoApp::setUpdateTime(int u) {
  if (_cfg_updateTime != u) {
    _cfg_updateTime = u;
    _timer->start(u*1000);
  }
}

void KMagoApp::closeTransferWindows(KTransfer t) {
  QWidget *wnd = 0;
  if ((wnd = _logWnds[t.id()]) != 0) {
    wnd->close();
    _logWnds.remove(t.id());
  }
  if ((wnd = _statusWnds[t.id()]) != 0) {
    wnd->close();
    _statusWnds.remove(t.id());
  }
}

//////////////////////////////

/** set the current manager */

void KMagoApp::setCurrentManager(KTManager m) {
  if (m == currentManager())
    return;
  _currentManager = m;
  setCaption(getDocument()->title() + " - " + m.title());
  sysTray->contextMenu()->changeTitle(sysTray->contextMenu()->idAt(0), getDocument()->title() + " - " + m.title());
  //sysTray->contextMenu()->changeTitle(1, m->title() + " - KMago");
  kdDebug(D_RUN) << name() << ": current manager " << m.id() << endl;
  // update current transfer
  setCurrentTransfer(KObject::null().toTransfer());
  // update main view
  bool cond = !m.isNull();
  if (stateChange(7, cond)) {
    managerAdd->setEnabled(cond);
    managerAddRecent->setEnabled(cond);
    //editCopy->setEnabled(cond);
    //editCut->setEnabled(cond);
#ifdef KACTION_ERR
    sysTray->managerAddRecent->setEnabled(cond);
#endif
    managerConf->setEnabled(cond);
    managerPaste->setEnabled(cond);
  }
  slotDocTManagerStateChange(m);
  // update view
  view->selectManager(m);
}

/** set the current transfer */
void KMagoApp::setCurrentTransfer(KTransfer t) {
  if (t == currentTransfer())
    return;
  _currentTransfer = t;
  kdDebug(D_RUN) << name() << ": current transfer " << (t.isNull() ? QString("NULL") : t.name()) << endl;
  // update main view
  bool cond = !t.isNull();
  if (stateChange(0, cond)) {
    transferRemove->setEnabled(cond);
    transferConf->setEnabled(cond);
    transferLog->setEnabled(cond);
    transferStatus->setEnabled(cond);
    editCopy->setEnabled(cond);
    editCut->setEnabled(cond);
    if (cond) {
      slotStatusMsg(t.remote().prettyURL() + " -> " + t.local().fileName());
      slotDocTransferStateChange(t);
    } else {
      transferPause->setEnabled(cond);
      transferResume->setEnabled(cond);
      transferCheck->setEnabled(cond);
      transferAutoResume->setEnabled(cond);
      transferAutoCheck->setEnabled(cond);
      slotStatusMsg(i18n(IDS_STATUS_DEFAULT));
    }
  }
  // update view
  view->selectTransfer(t);
}

/** update the view */
void KMagoApp::resetView() {
  _transferCount = 0;
  QStringList lst;
  for(KTManager m = (getDocument()->manager().itemFirst()).toTManager(); 
      !m.isNull(); 
      m = (getDocument()->manager().itemNext()).toTManager()) {
    _transferCount += m.itemCount();
    lst += m.title();
  }
  fileSelectMngr->setItems(lst);
#ifdef KACTION_ERR
  sysTray->fileSelectMngr->setItems(lst);
#endif
  _currentManager.setNull(); // set current manager but don't emit any signal
  slotDocMManagerChange();
  view->reset();
  setCurrentManager(getDocument()->manager().itemFirst().toTManager());
}

/////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATION
/////////////////////////////////////////////////////////////////////

// slot menu FILE_*

void KMagoApp::slotFileNewWindow() {
  KMagoApp *new_window= new KMagoApp();
  new_window->newDocument();
  new_window->show();
}

void KMagoApp::slotFileNew() {
  newDocument();
}

void KMagoApp::slotFileOpen() {
  QString fileToOpen = 
    KFileDialog::getOpenFileName(QDir::currentDirPath(),
				 i18n("*|All files"), 
				 this, i18n("Open File..."));
  QDir::setCurrent(QFileInfo(fileToOpen).dirPath());
  if(fileToOpen.isEmpty())
    return;
  if (!openDocument(fileToOpen))
    KMessageBox::sorry(this, 
		       i18n("Error in opening file " + fileToOpen + ", operation aborted"),
		       i18n("Error in opening file"), false);
}

void KMagoApp::slotFileOpenRecent(const KURL &url) {
  if(url.path().isEmpty())
    return;
  QFileInfo file(url.directory(), url.fileName());
  if (!openDocument(file.absFilePath()))
    KMessageBox::sorry(this, 
		       i18n("Error in opening file " + file.absFilePath() + ", operation aborted"),
		       i18n("Error in opening file"), false);
}

void KMagoApp::slotFileSaveAs() {
  QString newFile = KFileDialog::getSaveFileName(QDir::currentDirPath(),
						 QString::null, 
						 this, i18n("Save as..."));
  if(newFile.isEmpty())
    return;
  if (!saveAsDocument(newFile))
    KMessageBox::sorry(this, 
		       i18n("Error in saving file " + newFile + ", operation aborted"),
		       i18n("Error in saving file"), false);
  QDir::setCurrent(QFileInfo(newFile).dirPath());
}

void KMagoApp::slotFileClose() {
  close();
}

void KMagoApp::slotFileQuit() {
  // close the first window, the list makes the next one the first again.
  // This ensures that queryClose() is called on each window to ask for closing
  KMainWindow* w;
  if(memberList) {
    for(w=memberList->first(); w!=0; w=memberList->first()) {
      // only close the window if the closeEvent is accepted. If the user presses Cancel on the saveModified() dialog,
      // the window and the application stay open.
      if(!w->close())
	break;
    }
  }
}

void KMagoApp::slotFileSelectMngr(int i) {
  //int i = fileSelectMngr->currentItem();
  setCurrentManager(getDocument()->manager().itemAt(i).toTManager());
}

void KMagoApp::slotFileAddMngr() {
  // dialog box manager
  KDlgOpenManager *dlg = new KDlgOpenManager(this, "KDlgOpenManager");
  QString title = dlg->exec();
  delete dlg;
  if (title.isEmpty())
    return;
  getDocument()->manager().itemNew(title);
}

// slot menu MANAGER_*

void KMagoApp::slotManagerRemoveMngr() {
  if (getExpertMode()
      || KMessageBox::questionYesNo(this,
				    i18n("Remove the highlighted Manager?"),
				    i18n("Remove Manager"),
				    QString::null, QString::null, false
				    ) == KMessageBox::Yes)
    getDocument()->manager().itemRemove(currentManager());
}

void KMagoApp::slotManagerAdd() {
  addTransfer("");
}

void KMagoApp::slotManagerPaste() {
  QStringList uris;
  QString text;
  if (QUriDrag::canDecode(kapp->clipboard()->data())) {
    if (!QUriDrag::decodeToUnicodeUris(kapp->clipboard()->data(), uris)) {
      kdWarning(D_RUN) << name() << ": PASTE URI: DECODE FAIL" << endl;
      return;
    }
    //if (event->source() != this && event->action() == QDropEvent::Move)
    kdDebug(D_RUN) << name() << ": URIS=" << uris.join(",") << endl;
    for (QStringList::Iterator i = uris.begin(); i != uris.end(); ++i)
      addTransfer(*i);
  } else if (QTextDrag::canDecode(kapp->clipboard()->data())) {
    if (!QTextDrag::decode(kapp->clipboard()->data(), text)) {
      kdWarning(D_RUN) << name() << ": PASTE TEXT: DECODE FAIL" << endl;
      return;
    }
    kdDebug(D_RUN) << name() << ": TEXT=" << text << endl;
    addTransfer(text);
  }
  //addTransfer(kapp->clipboard()->text());
}

void KMagoApp::slotManagerAddRecent(const KURL &url) {
  addTransfer(url);
}

/** stop all transfers */
void KMagoApp::slotManagerPauseAll(){
  if (getExpertMode() ||
      KMessageBox::questionYesNo(this,
				 i18n("Pause All Transfers?"),
				 i18n("Pause all transfers"),
				 QString::null, QString::null, 
				 false) == KMessageBox::Yes)
    currentManager().killAll();
}

/** enable/disable auto-downloading mode */
void KMagoApp::slotManagerAutoDownload() {
  bool state = managerAutoDownload->isChecked();
  if (currentManager().getAutoDownload() != state) {
    currentManager().setAutoDownload(state);
    currentManager().setModified(MOD_CONF, PRP_ALL);
    //slotSetCurrentTransfer(doc->currentTransfer());
  }
}

/** open the Transfer list (document) dialog */
void KMagoApp::slotManagerConfigure() {
  KDlgManager *dlg = new KDlgManager(currentManager(), this);
  dlg->exec();
  delete dlg;
}

// slot menu TRANSFER_*

void KMagoApp::slotTransferResume() {
  if (getExpertMode()
      || !currentTransfer().isComplete()
      || KMessageBox::questionYesNo(this,
				    i18n("Starting transfer will overwrite current content, continue?"),
				    i18n("Start Transfer"),
				    QString::null, QString::null, 
				    false) == KMessageBox::Yes)
    currentTransfer().start();
}

void KMagoApp::slotTransferPause() {
  currentTransfer().kill();
}	

void KMagoApp::slotTransferRemove() {
  if (getExpertMode()
      || KMessageBox::questionYesNo(this,
				    i18n("Remove the highlighted transfer?"),
				    i18n("Remove Transfer"),
				    QString::null, QString::null, false
				    ) == KMessageBox::Yes)
    currentManager().itemRemove(currentTransfer());
}

/** check for transfer resume capability */
void KMagoApp::slotTransferCheck(){
  currentTransfer().checkResumeCapability();
}

/** enable/disable auto-resuming mode */
void KMagoApp::slotTransferAutoResume() {
  bool state = transferAutoResume->isChecked();
  KTransfer t = currentTransfer();
  kdFatal(t.isNull(), D_VIE) << "CHANGE AUTORESUME WHEN CURRENT NULL" << endl;
  if (t.getAutoResume() != state) {
    t.setAutoResume(state);
    t.setModified(MOD_CONF, PRP_ALL);
  }
}

/** enable/disable auto-check mode */
void KMagoApp::slotTransferAutoCheck() {
  bool state = transferAutoCheck->isChecked();
  KTransfer t = currentTransfer();
  kdFatal(t.isNull(), D_VIE) << "CHANGE AUTOCHECK WHEN CURRENT NULL" << endl;
  if (t.getCheckResume() != state) {
    t.setCheckResume(state);
    t.setModified(MOD_CONF, PRP_ALL);
  }
}
	
/** open the current transfer options dialog */
void KMagoApp::slotTransferConfigure() {
  KDlgTransfer *dlg = new KDlgTransfer(currentTransfer(), this);
  dlg->exec();
  delete dlg;
}

/** open the transfer status dialog */
void KMagoApp::slotTransferStatus() {
  QWidget *status;
  int id = currentTransfer().id();
  if ((status = _statusWnds[id]) != 0)
    status->raise();
  else {
    status = new KDlgTransferStatus(currentTransfer(), this);
    connect(status, SIGNAL(sigClose(int)), 
	    this, SLOT(slotWndStatusClosed(int)));
    _statusWnds.insert(id, status);
    status->show();
  }
}

/** open dialog to view the transfer cmd output */
void KMagoApp::slotTransferLog() {
  QWidget *viewer;
  int id = currentTransfer().id();
  if ((viewer = _logWnds[id]) != 0)
    viewer->raise();
  else {
    viewer = new KLogViewer(currentTransfer(), this);
    connect(viewer, SIGNAL(sigClose(int)), 
	    this, SLOT(slotWndLogClosed(int)));
    _logWnds.insert(id, viewer);
    viewer->show();
  }
}

// slot menu EDIT_*

void KMagoApp::slotEditCut() {
  KTransfer t = currentTransfer();
  KTManager m = currentManager();
  if (t.isNull() && m.itemCount() > 1 && m.itemActiveCount() == 0) {
    // cut manager
    //kapp->clipboard()->setData(new KTManagerDrag(m, this));
    //getDocument()->manager().itemRemove(m);
  } else if (!t.isNull() && !t.isRunning()) {
    // cut transfer
    //kapp->clipboard()->setData(new KTransferDrag(t, this));
    //m.itemRemove(t);
    kapp->clipboard()->setData(new QTextDrag(t.remote().url(), this));
  } 
}

void KMagoApp::slotEditCopy() {
  KTransfer t = currentTransfer();
  KTManager m = currentManager();
  if (t.isNull() && m.itemCount() > 1 && m.itemActiveCount() == 0) {
    // copy manager
    //kapp->clipboard()->setData(new KTManagerDrag(m, this));
    //FIXME!! copy action in setCurrentManager
  } else if (!t.isNull() && !t.isRunning()) {
    // copy transfer
    //kapp->clipboard()->setData(new KTransferDrag(t, this));
    kapp->clipboard()->setData(new QTextDrag(t.remote().url(), this));
  } 
}

void KMagoApp::slotEditPaste() {
  /*
  if (KTManagerDrag::canDecode(kapp->clipboard()->data())) {
    QDomDocument doc(KTManager::docId());
    if (!KTManagerDrag::decode(kapp->clipboard()->data(), doc)) {
      kdWarning(D_RUN) << name() << ":PASTE TMANAGER : DECODE FAIL" << endl;
      return;
    }
    getDocument()->manager().itemCopy(doc.documentElement());
  } else if (KTransferDrag::canDecode(kapp->clipboard()->data())) {
    QDomDocument doc(KTransfer::docId());
    if (!KTransferDrag::decode(kapp->clipboard()->data(), doc)) {
      kdWarning(D_RUN) << name() << ":PASTE TRANSFER : DECODE FAIL" << endl;
      return;
    }
    currentManager().itemCopy(doc.documentElement());
  } else {
    slotManagerPaste();
  }
  */
  slotManagerPaste();
}

// slot menu VIEW_*

void KMagoApp::slotViewToolBarMain() {
  bool state = viewToolBarMain->isChecked();
  if (state)
    toolBar("main")->show();
  else
    toolBar("main")->hide();
}

void KMagoApp::slotViewToolBarTrn() {
  bool state = viewToolBarTrn->isChecked();
  if (state)
    toolBar("transfer")->show();
  else
    toolBar("transfer")->hide();
}

void KMagoApp::slotViewStatusBar() {
  bool state = viewStatusBar->isChecked();
  if (state)
    statusBar()->show();
  else
    statusBar()->hide();
}

/** open the application options dialog */
void KMagoApp::slotViewPreferences(){
  KDlgApp *dlg = new KDlgApp(this, "KDlgApp");
  dlg->exec();
  delete dlg;
}

void KMagoApp::slotViewManagers() {
  view->showManagers(viewManagers->isChecked());
}

void KMagoApp::slotViewSplitOrient() {
  view->setOrientation(viewSplitOrient->isChecked());
}

void KMagoApp::slotViewSysTray() {
  if (viewSysTray->isChecked())
    sysTray->show();
  else
    sysTray->hide();
}

/** request to refresh the views emitting a signalReset*/
void KMagoApp::slotViewRefresh(){
  resetView();
}

////////////////////////////

void KMagoApp::slotStatusMsg(const QString &text) {
  statusBar()->changeItem(text, STATUS_MSG);
}

void KMagoApp::slotStatusHelpMsg(const QString &text) {
  statusBar()->message(text, 2000);
}

/** called when clipboard data change */
void KMagoApp::slotClipboardChange(){
  QString s = kapp->clipboard()->text();
  editPaste->setEnabled(!s.isEmpty());
}

////////////////////////////////

/** update view when a transfer is added */
void KMagoApp::slotDocTransferAdd(KTransfer t) {
  kdDebug(D_VIE) << name() << ": add transfer " << t.id() << endl;
  _transferCount++;
  // update main view
  //slotDocChange();
  // update view
  if (currentManager() == t.container().toTManager())
    view->addTransfer(t);
}

/** update view when a transfer is removed */
void KMagoApp::slotDocTransferRemove(KTransfer t) {
  kdDebug(D_VIE) << name() << ": remove transfer " << t.id() << endl;
  // close windows associated with transfer
  closeTransferWindows(t);
  if (currentTransfer() == t)
    setCurrentTransfer(currentManager().itemFirst().toTransfer());
  _transferCount--;
  //update main view
  //slotDocChange();
  // update view
  if (currentManager() == t.container().toTManager())
    view->removeTransfer(t);
}

/** update view when a manager is added */
void KMagoApp::slotDocTManagerAdd(KTManager m) {
  kdDebug(D_RUN) << name() << ": add manager " << m.id() << endl;
  // update main viev
  _transferCount += m.itemCount();
  QStringList lst = fileSelectMngr->items();
  lst += m.title();
  //fileSelectMngr->clear();
  fileSelectMngr->setItems(lst);
#ifdef KACTION_ERR
  sysTray->fileSelectMngr->setItems(lst);
#endif
  //slotDocChange();
  // update view
  view->addManager(m);
}

/** update view when a manager is removed */
void KMagoApp::slotDocTManagerRemove(KTManager m) {
  kdDebug(D_RUN) << name() << ": remove manager " << m.id() << endl;
  //update main view
  _transferCount -= m.itemCount();
  QStringList lst;
  for (KTManager i = getDocument()->manager().itemFirst().toTManager(); 
       !i.isNull(); 
       i = getDocument()->manager().itemNext().toTManager())
    lst += i.title();
  //fileSelectMngr->clear();
  fileSelectMngr->setItems(lst);
#ifdef KACTION_ERR
  sysTray->fileSelectMngr->setItems(lst);
#endif
  if (currentManager() == m)
    setCurrentManager(getDocument()->manager().itemFirst().toTManager());
  //slotDocChange();
  // update view
  view->removeManager(m);
}

/** update the view when the current transfer state change */
void KMagoApp::slotDocTransferStateChange(KTransfer t) {
  // update view
  if (currentManager() == t.container().toTManager())
    view->updateTransfer(t);
  // update main view
  if (currentTransfer() != t)
    return;
  KTManager mngr = currentManager();
  kdFatal(t.isNull()) << "slotDocTransferStateChange(null)" << endl;
  bool cond = (!mngr.getAutoDownload() && t.isRunning());
  if (stateChange(2, cond)) {
    transferPause->setEnabled(cond);
  }
  cond = (!mngr.getAutoDownload() && !t.isRunning());
  if (stateChange(3, cond)) {
    transferResume->setEnabled(cond);
    transferCheck->setEnabled(cond);
  }
  cond = !t.isGlobal();
  if (stateChange(4, cond)) {
    transferAutoResume->setEnabled(cond);
    transferAutoCheck->setEnabled(cond);
  }
  slotDocTransferConfChange(t);
}

/** update the view when the current transfer config change */
void KMagoApp::slotDocTransferConfChange(KTransfer t) {
  // update view
  // update main view
  if (currentTransfer() != t)
    return;
  kdFatal(t.isNull()) << "slotTransferConfChange(null)" << endl;
  bool cond = t.getAutoResume() && t.isResumable();
  if (stateChange(5, cond)) {
    transferAutoResume->setChecked(cond);
  }
  cond = t.getCheckResume();
  if (stateChange(6, cond)) {
    transferAutoCheck->setChecked(cond);
  }
}

/** update the view when the current transfer read data */
void KMagoApp::slotDocTransferReadData(KTransfer t) {
  //update view
  if (currentManager() == t.container().toTManager())
    view->updateTransferText(t);
}

/** update visualization of manager properties */
void KMagoApp::slotDocTManagerStateChange(KTManager m) {
  // update view
  // update main view
  if (currentManager() != m)
    return;
  kdFatal(m.isNull()) << "slotDocTManagerStateChange(null)" << endl;
  bool cond = !m.isNull() && m.getAutoDownload();
  if (stateChange(8, cond)) {
    managerAutoDownload->setChecked(cond);
    if (!currentTransfer().isNull())
      slotDocTransferStateChange(currentTransfer());
  }
  cond = !m.isNull() && m.itemActiveCount() > 0;
  if (stateChange(9, cond))
    managerPauseAll->setEnabled(cond);
  cond = !m.isNull() && getDocument()->manager().itemCount() > 1;
  if (stateChange(10, cond))
    managerRemoveMngr->setEnabled(cond);
  if (!m.isNull()) {
    statusBar()->changeItem(QString("%1/%2").arg(m.itemActiveCount(), 3).arg(m.itemCount(), 3), STATUS_COUNT);
    statusBar()->changeItem(QString("%1/%2").arg(unit(m.bandwidth()), 6).arg(unit(m.mediumBandwidth()), 6), STATUS_BAND);
  } else {
    statusBar()->changeItem("-/-", STATUS_COUNT);
    statusBar()->changeItem("-/-", STATUS_BAND);
  }
}

void KMagoApp::slotDocMManagerChange() {
  // update main view
  KMManager m = getDocument()->manager();
  statusBar()->changeItem(QString("%1/%2").arg(m.itemActiveCount(), 3).arg(transferCount(), 3), STATUS_GLB_COUNT);
  statusBar()->changeItem(QString("%1/%2").arg(unit(m.bandwidth()), 6).arg(unit(m.mediumBandwidth()), 6), STATUS_GLB_BAND);
  // update systray tooltip
  QToolTip::remove(sysTray);
  QToolTip::add(sysTray, QString("%1 - %2/%3 - %4").arg(PACKAGE).arg(m.itemActiveCount()).arg(transferCount()).arg(unit(m.mediumBandwidth())));

}

void KMagoApp::slotWndStatusClosed(int id) {
  _statusWnds.remove(id);
}

void KMagoApp::slotWndLogClosed(int id) {
  _logWnds.remove(id);
}

#include "kmago.moc"
