/**
 * downloader.cpp
 *
 * Copyright (C)  2004  Zack Rusin <zack@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307  USA
 */

#include "downloader.h"
#include "hostdata.h"
#include "kernel.h"

#include <kio/netaccess.h>
#include <kio/job.h>
#include <kio/jobclasses.h>
#include <kio/scheduler.h>
#include <kfileitem.h>

using KIO::MultiGetJob;
using KIO::SimpleJob;
using KIO::UDSEntry;
using KIO::UDSEntryListConstIterator;

#include <klocale.h>
#include <kglobal.h>
#include <kstandarddirs.h>
#include <kurl.h>
#include <kdebug.h>

#include <qfile.h>
#include <qcstring.h>

#include <cstring>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

namespace KCfgCreator
{

Downloader::Downloader( QObject* parent, const char* name )
    : QObject( parent, name )
{
    m_pendingData.setAutoDelete( true );
    m_jobs.setAutoDelete( true );

    KIO::Scheduler::connect( SIGNAL(slaveConnected(KIO::Slave*)),
                             this, SLOT(slotSlaveConnected()) );
    KIO::Scheduler::connect( SIGNAL(slaveError(KIO::Slave*,int,const QString&)),
                             this, SLOT(slotSlaveError()) );
}

void
Downloader::get( const QString& host, const QString& user,
                 const QString& pass )
{
    kdDebug()<<"Get host "<<host<<endl;

    if ( host.isEmpty() ) {
        getLocal();
    } else {
        getFish( host, user, pass );
    }
}

void
Downloader::getLocal()
{
    QStringList cfgPaths( KGlobal::dirs()->resourceDirs( "kcfg" ) );
    QStringList paths( KGlobal::dirs()->resourceDirs( "config" ) );

    kdDebug()<<"XXXXXXXXXXXXX KCFG   Path = "<<cfgPaths<<endl;
    kdDebug()<<"XXXXXXXXXXXXX Config Path = "<<paths<<endl;

    KURL url( "file:/" );
    HostData *host = new HostData( url );

    host->addNumberOfWaitingUrls( cfgPaths.size() + paths.size() );
    for ( QStringList::Iterator it = cfgPaths.begin(); it != cfgPaths.end(); ++it ) {
        KURL url;
        url.setPath( *it );
        host->addKCfgPath( url );
        createListJob( url, host );
    }
    for ( QStringList::Iterator it = paths.begin(); it != paths.end(); ++it ) {
        KURL url;
        url.setPath( *it );
        host->addKConfigPath( url );
        createListJob( url, host );
    }
}

void Downloader::getFish( const QString& hostName, const QString& user,
                          const QString& pass )
{
    static QString kconfig_query( "--path config" );
    static QString kcfg_query( "--path kcfg" );

    KURL url = createURL( "fish", hostName, user, pass );
    QString kdeconfig = KIO::NetAccess::fish_execute( url, "which kde-config", 0 );
    //we want just the first line of the output
    kdeconfig = kdeconfig.section( '\n', 0, 0);

    QString query = QString( "%1 %2" ).arg( kdeconfig ).arg( kconfig_query );
    QString kcpaths   = KIO::NetAccess::fish_execute( url, query, 0 );
    kcpaths = kcpaths.section( '\n', 0, 0 );

    query = QString( "%1 %2" ).arg( kdeconfig ).arg( kcfg_query );
    QString kcfgpaths = KIO::NetAccess::fish_execute( url, kdeconfig + " --path kcfg", 0 );
    kcfgpaths = kcfgpaths.section( '\n', 0, 0 );

    kdDebug()<<"KC Paths on  " << url.prettyURL() << " are "<<kcpaths<<endl;
    kdDebug()<<"KCFG Paths on "<< url.prettyURL() << " are "<<kcfgpaths<<endl;

    QStringList cfgPaths = QStringList::split( ":", kcfgpaths );
    QStringList paths    = QStringList::split( ":", kcpaths );

    HostData *host = new HostData( url );
    host->addNumberOfWaitingUrls( cfgPaths.size() + paths.size() );
    for ( QStringList::Iterator it = cfgPaths.begin(); it != cfgPaths.end(); ++it ) {
        KURL lurl( url );
        lurl.setPath( *it );
        host->addKCfgPath( lurl );
        createListJob( lurl, host );
    }
    for ( QStringList::Iterator it = paths.begin(); it != paths.end(); ++it ) {
        KURL lurl( url );
        lurl.setPath( *it );
        host->addKConfigPath( lurl );
        createListJob( lurl, host );
    }
}

KURL
Downloader::createURL( const QString& protocol, const QString& host,
                       const QString& user, const QString& pass )
{
    KURL url;

    url.setProtocol( protocol );
    url.setHost( host );

    if ( !user.isEmpty() ) {
        url.setUser( user );
        if ( !pass.isEmpty() ) url.setPass( pass );
    }

    return url;
}

void
Downloader::createListJob( const KURL& url, HostData *hdata )
{
    //kdDebug()<<"List job with url = "<<url.prettyURL()<<endl;
    SimpleJob* job = KIO::listDir( url, false );
    //FIXME: add error handling

    //kdDebug()<<"inserting " << job << " with " << hdata <<endl;
    m_pendingData.insert( job, hdata );
    connect( hdata, SIGNAL(done(HostData*)),
             SLOT(slotDone(HostData*)) );
    connect( job, SIGNAL(entries( KIO::Job*, const KIO::UDSEntryList&)),
             SLOT(slotEntries(KIO::Job*, const KIO::UDSEntryList&)) );
    connect( job, SIGNAL(result(KIO::Job *)),
             SLOT(slotListResult(KIO::Job *)) );
}

void
Downloader::slotEntries( KIO::Job* job, const KIO::UDSEntryList& list )
{
    SimpleJob *sjob = static_cast<SimpleJob*>( job );
    KURL baseUrl( sjob->url() );

    UDSEntryListConstIterator it = list.begin();

    HostData *data = m_pendingData.take( job );
    if ( !data ) {
        kernel->emitLogMsg( "Downloader",
                            i18n("URL from %1 came without data!").arg( baseUrl.prettyURL() )
            );
        return;
    }

    MultiGetJob* mjob = 0;

    for (; it != list.end(); ++it) {
        KFileItem fitem( (*it), baseUrl, false, true );

        if ( fitem.isFile() ) {
            //kdDebug() << "Get " << url.prettyURL() <<  endl;
            if ( !mjob ) {
                mjob = KIO::multi_get( m_id, fitem.url(), KIO::MetaData() );
                connect( mjob, SIGNAL(data(long, const QByteArray&)),
                         SLOT(slotData(long, const QByteArray&)) );
                connect( mjob, SIGNAL(result(long)),
                         SLOT(slotResult(long)) );
            } else {
                mjob->get( m_id, fitem.url(), KIO::MetaData() );
            }
            m_jobs.insert( m_id, new FileData( data, fitem.url(),
                                               data->isKCfgPath( baseUrl ) ) );
            ++m_id;
        }
    }

    data->decrementNumberOfWaitingUrls();
}

void
Downloader::slotSlaveConnected()
{
    kdDebug()<<"Slave connected"<<endl;
}

void
Downloader::slotSlaveError()
{
    kdDebug()<<"Slave error"<<endl;
}

void
Downloader::slotData( long id, const QByteArray& data )
{
    if ( data.size() == 0 ) {
        FileData* fdata = m_jobs.take( id );
        fdata->data += '\0';
        fdata->finished();
    } else {
        FileData* fdata = m_jobs[ id ];
        //unsigned offset = fdata->data.size();
        //fdata->data.resize( offset + data.size() );
        //std::memcpy( fdata->data.data() + offset, data.data(), data.size() - 1 );
        QCString cdata( data, data.size() + 1);
        fdata->data.append( cdata );
    }
}

void
Downloader::slotResult( long id )
{
    FileData* data = m_jobs.take( id );
    if ( data ) {
        kdDebug()<<"Funky thing happened : job[" << data->url().path()
                 <<"](" << data->data.size() << ") finished before ending data stream"<<endl;
        data->finished();
    }
}

void
Downloader::slotListResult( KIO::Job* job )
{
    if ( job->error() ) {
        HostData *data = m_pendingData.take( job );
        if ( data )
            data->decrementNumberOfWaitingUrls();
    }
}

void
Downloader::slotDone( HostData* hdata )
{
    if ( !hdata->waitingForUrls() && !hdata->isDone() ) {
        kdDebug()<<"Host data : "<< hdata->waitingForUrls()
                 <<", unfinished " << hdata->m_unfinishedCount
                 <<", kcfg = "<<hdata->kcfgFiles().count()
                 <<", kconfig = "<< hdata->kconfigFiles().count()
                 <<endl;
        emit done( hdata );
        hdata->setDone( true );
    }
}

}

#include "downloader.moc"
