/***************************************************************************
                          kaspabase.cpp  -  description
                             -------------------
    begin                : Tue Sep 7 1999
    copyright            : (C) 1999 by Jan Mueller
    email                : janmueller7@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "kaspabase.h"
#include <assert.h>
#include <time.h>
#include <string.h>
#include <stdlib.h>

#define Inherited Sql

#define TYPEAUTHOR 0
#define TYPEPUBL   1
#define TYPEPART   2
#define TYPENOTE   3
#define TYPEFILE   4


////////////////////////////////////////////////////////
//// 	Field

 	Field::Field():changed(false) {}
  Field::~Field() {}
  void Field::setChanged() { changed=true; }
  void Field::setUnchanged() { changed=false; }
  bool Field::hasChanged() const { return changed; }

////////////////////////////////////////////////////////
//// DataField

  DataField::DataField(DataField& T):Field() { set(T.get(), T.length()); }

  DataField::DataField():Field(), d(0), len(0) { }

  DataField::~DataField() { del(); }

  DataField& DataField::operator=(const DataField& T) {
		if(!(this==&T)) {
			Field::operator=(T);
			set(T.get(), T.length());
		}
		return *this;
	}

  void DataField::empty() { del(); d=0; }

  long DataField::length() const { return len; }

  const unsigned char *DataField::get() const { return d; }

  void DataField::set(const unsigned char *buf, long l) {
    del();
    len=l;
    if(!len) { d=0; return; }
    d=new unsigned char[len];
    memcpy(d, buf, len);
  }	

  void DataField::take(unsigned char *buf, long l) {
    len=l;
    d=buf;
  }	

  void DataField::del() { delete d; d=0; }


////////////////////////////////////////////////////////
//// TextField
  TextField::TextField(TextField& T): Field() { set(T.get()); }

  TextField::TextField():Field(), t(0) { empty(); }

  TextField::~TextField() { del(); }

	TextField& TextField::operator=(TextField &T) {
		if(!(this==&T)) {
			Field::operator=(T);
			set(T.get());
		}
		return *this;
	}

  void TextField::empty() { del(); t=new char[1]; *t=0; }

  const char *TextField::get() const { return t; }

	char *TextField::getSql() const { return esc4sql(t); }

	void TextField::getSql(Str &s) const { char *tmp=esc4sql(t); s+=tmp; delete tmp; }

  void TextField::set(const char *text) {
    del();
    if(!text) { t=0; return; }
    t=new char[strlen(text)+1];
    strcpy(t, text);
  }	

	void TextField::append(const char *text) {
		char *s=new char[strlen(t)+strlen(text)+1];
		strcpy(s, t); strcat(s, text);
		del();
		set(s); delete s;
	}

  void TextField::take(char *text) { t=text; }		

  void TextField::del() { if(t) delete t; t=0; }

////////////////////////////////////////////////////////////////////////////
//// IntField

  IntField::IntField(IntField& I):Field(), i(I.get()) {};
  IntField::IntField(): Field(), i(0) {}
  void IntField::empty() { i=0; }
  int IntField::get() const { return i; }
  void IntField::get(Str& s) const {
		s+=i;
	}	
  void IntField::set(int j) { i=j; }
  void IntField::set(const char *s) { i=strtol(s,0,0); }

////////////////////////////////////////////////////////////////////////////
//// BoolField
//// BoolField::BoolField(BoolField& I):Field(), i(I.get()) {};

  BoolField::BoolField(): Field(), i(false) {}

//	BoolField& BoolField::operator=(BoolField& f) {  Field::operator=(t); i=f.get(); };

  void BoolField::empty() { i=false; }

  void BoolField::get(Str &s) const { if(i) s+="t"; else s+="f"; }

  bool BoolField::get() const { return i; }

  void BoolField::set(bool j) { i=j; }

  void BoolField::set(const char *s) { if(!strcmp(s, "t")) i=true; else i=false; }


////////////////////////////////////////////////////////////////////////////
//// OidField

//  OidField::OidField(OidField& O): Field(), o(O.get()) {}

  OidField::OidField(): Field(), o(InvalidOid) {}

  void OidField::empty() { o=InvalidOid; }

  Oid OidField::get() const { return o; }

  void OidField::get(Str& s) const { s+=o; }

  void OidField::set(Oid i) { o=i; }

  void OidField::set(const char *s) { o=str2oid(s); }

////////////////////////////////////////////////////////////////////////////
//// KaspaBase

KaspaBase::KaspaBase(Sql *conn): Inherited(*conn) {
}

KaspaBase::KaspaBase(const char *dbname): Inherited(dbname) {
}

KaspaBase::~KaspaBase() throw() {
}

LockTableItem *KaspaBase::locktable=0L;

Oid KaspaBase::getNo(Str tab, Oid o) {
  exec("select no from "+tab+" where oid="+oid2str(o));
  if(tuples())
    return str2oid(getValue(0, "no"));
  else
    return InvalidOid;
}

/*******************************************************
*              Author
*******************************************************/
Oid KaspaBase::newAuthor() {
  Oid ret;
  try {
    exec("begin");
    Str s("insert into author (created, modified) values ('");
		s+=getTime(); s+="','";
		s+=getTime(); s+="')";
    exec(s);
    Oid o=oidStatus();
    exec("commit");
    ret=getNo(Str("author"), o);
  } catch (...) {
    exec("rollback");
    throw;
  }
  return ret;
}

void KaspaBase::deleteAuthor(Oid author) {
	if(isLocked(author)) throw BaseDeleteLock("Can't delete records - one is locked!");
  try {
    exec("begin");
    Str where;
    where="where author_no="; 
    where+=author;
    deletePublAuthor(where);
    deletePartAuthor(where);
    where="where no="; where+=author;
    deleteTuples("author", where);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
}

AuthorRec *KaspaBase::getAuthor(const char *where) {
  AuthorRec *last=0L;
  try {
    exec("begin");
    Str s("select no, lastname, firstname, pseudonym, born, died, country, memo, modified, created from author ");
    s+=where;
    exec(s);

    int i;	
    for(i=0; i<tuples(); i++) {
      AuthorRec *rec= new AuthorRec;
      rec->id.set(getValue(i, "no"));
      rec->lastname.set(getValue(i, "lastname"));
      rec->firstname.set(getValue(i, "firstname"));
      rec->born.set(getValue(i, "born"));
      rec->died.set(getValue(i, "died"));
      rec->country.set(getValue(i, "country"));
			rec->modified.set(getValue(i, "modified"));
			rec->created.set(getValue(i, "created"));
      rec->memo.set(getValue(i, "memo"));
      rec->pseudonym.set(getValue(i, "pseudonym"));
      rec->next=last;
      last=rec;
    }
    exec("commit");
  } catch(...) {
    delete last;
    last=0L;
    exec("rollback");
    throw;
  }
  return last;
}

void KaspaBase::updateAuthor(AuthorRec *author, const char *where, bool block) {
  try {
    if(!where || !strlen(where)) throw KaspaErr("No Where-Condition!");

    if(author->lastname.hasChanged()+author->firstname.hasChanged()+author->pseudonym.hasChanged()+
       author->country.hasChanged()+author->born.hasChanged()+author->died.hasChanged()+
       author->memo.hasChanged()==0) return;

    exec("begin", block);

    Str s=Str("update author set ");
    if(author->lastname.hasChanged())  { s+="lastname ='"; author->lastname.getSql(s);  s+="', ";}
    if(author->firstname.hasChanged()) { s+="firstname='"; author->firstname.getSql(s); s+="', ";}
    if(author->country.hasChanged())   { s+="country='";   author->country.getSql(s);   s+="', ";}
    if(author->born.hasChanged()) 		 { s+="born='";      author->born.getSql(s);      s+="', ";}
    if(author->died.hasChanged()) 		 { s+="died='"; 		 author->died.getSql(s); 		  s+="', ";}
    if(author->pseudonym.hasChanged()) { s+="pseudonym='"; author->pseudonym.getSql(s); s+="', ";}
    if(author->memo.hasChanged()) 		 { s+="memo='";      author->memo.getSql(s);      s+="', ";}

		s+="modified='"; s+=getTime(); s+="', ";

    if(s[s.length()-2]==',') s[s.length()-2]=' ';
    s+=" "; s+=where;
    exec(s, block);
    author->unchanged();
    exec("commit", block);
  } catch(...) {
    exec("rollback", block);
    throw;
  }
}

void KaspaBase::movePublAuthorUp(Oid author,Oid publ) {
  try {
    exec("begin", true);

    Str s=Str("select author_no from publication_author where publication_no=");
    s+=publ;
    s+=" order by author_order";
    exec(s, true);
    int i;
    Str q;
    for(i=0; i<tuples(); i++) {
      Oid o=str2oid(getValue(i, "author_no"));
      if(o==author) {
        if(i==0) return;
        Oid prev=str2oid(getValue(i-1, "author_no"));

        q+="update publication_author set author_order=";
        q+=i-1;
        q+=" where author_no=";
        q+=author;
        q+=" and publication_no=";
        q+=publ;
        q+=";";

        q+="update publication_author set author_order=";
        q+=i;
        q+=" where author_no=";
        q+=prev;
        q+=" and publication_no=";
        q+=publ;
        q+=";";
      }
    }
    if(q.length()) exec(q);
  } catch(...) {
    exec("rollback", true);
    throw;
  }
}

void KaspaBase::movePublAuthorDown(Oid author,Oid publ) {
  try {
    exec("begin", true);

    Str s=Str("select author_no from publication_author where publication_no=");
    s+=publ;
    s+=" order by author_order";
    exec(s, true);
    int i;
    Str q;
    for(i=0; i<tuples(); i++) {
      Oid o=str2oid(getValue(i, "author_no"));
      if(o==author) {
        if(int(i)==tuples()-1) return;
        Oid next=str2oid(getValue(i+1, "author_no"));

        q+="update publication_author set author_order=";
        q+=i+1;
        q+=" where author_no=";
        q+=author;
        q+=" and publication_no=";
        q+=publ;
        q+=";";

        q+="update publication_author set author_order=";
        q+=i;
        q+=" where author_no=";
        q+=next;
        q+=" and publication_no=";
        q+=publ;
        q+=";";
      }
    }
    if(q.length()) exec(q);
  } catch(...) {
    exec("rollback", true);
    throw;
  }
}


void KaspaBase::movePartAuthorUp(Oid author,Oid publ) {
  try {
    exec("begin", true);

    Str s=Str("select author_no from part_author where part_no=");
    s+=publ;
    s+=" order by author_order";
    exec(s, true);
    int i;
    Str q;
    for(i=0; i<tuples(); i++) {
      Oid o=str2oid(getValue(i, "author_no"));
      if(o==author) {
        if(i==0) return;
        Oid prev=str2oid(getValue(i-1, "author_no"));

        q+="update part_author set author_order=";
        q+=i-1;
        q+=" where author_no=";
        q+=author;
        q+=" and part_no=";
        q+=publ;
        q+=";";

        q+="update part_author set author_order=";
        q+=i;
        q+=" where author_no=";
        q+=prev;
        q+=" and part_no=";
        q+=publ;
        q+=";";
      }
    }
    if(q.length()) exec(q);
  } catch(...) {
    exec("rollback", true);
    throw;
  }
}

void KaspaBase::movePartAuthorDown(Oid author,Oid publ) {
  try {
    exec("begin", true);

    Str s=Str("select author_no from part_author where part_no=");
    s+=publ;
    s+=" order by author_order";
    exec(s, true);
    int i;
    Str q;
    for(i=0; i<tuples(); i++) {
      Oid o=str2oid(getValue(i, "author_no"));
      if(o==author) {
        if(int(i)==tuples()-1) return;
        Oid next=str2oid(getValue(i+1, "author_no"));

        q+="update part_author set author_order=";
        q+=i+1;
        q+=" where author_no=";
        q+=author;
        q+=" and part_no=";
        q+=publ;
        q+=";";

        q+="update part_author set author_order=";
        q+=i;
        q+=" where author_no=";
        q+=next;
        q+=" and part_no=";
        q+=publ;
        q+=";";
      }
    }
    if(q.length()) exec(q);
  } catch(...) {
    exec("rollback", true);
    throw;
  }
}


/*********************************************************************
 * Publications
 **********************************************************************/
Oid KaspaBase::newPubl() {
  Oid ret;
  try {
    exec("begin");
    Str s("insert into publication (created, modified) values ('");
		s+=getTime();
		s+="','";
		s+=getTime();
		s+="')";
    exec(s);
    ret=getNo(Str("publication"), oidStatus());
    exec("commit");
  } catch (...) {
    exec("rollback");
    throw;
  }
  return ret;
}

void KaspaBase::deletePubl(Oid o) {
		if(isLocked(o)) throw BaseDeleteLock("Can't delete records - one is locked!");
    Str s("where part.publication_no=");
    s+=o;
    deletePart(s);
  try {
    exec("begin");
    Str s=Str("where publication_no="); s+=o;
    deletePublAuthor(s);
		s="where no="; s+=o;
    deleteTuples("publication", s);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
}

void KaspaBase::deletePubl(const char *where) {
		// Won't kill the whole table...
		if(!where || strlen(where)==0) abort();
		if(isLocked("publication", where)) throw BaseDeleteLock("Can't delete records - one is locked!");

    Str s("where publication_author.publication_no in (select no from publication ");
		s+=where;
		s+=")";
    deletePublAuthor(s);

    s="where part.publication_no in (select no from publication ";
		s+=where;
		s+=")";
    deletePart(s);
  try {
    exec("begin");
    deleteTuples("publication", where);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
}

PublRec *KaspaBase::getPubl(const char *where) {
  PublRec *last=0L;
  try {
    exec("begin");
    Str s("select * from publication ");
    s+=where;
    exec(s);

    int i;	
    for(i=0; i<tuples(); i++) {
      PublRec *rec= new PublRec;
      rec->title.set(getValue(i, "title"));
      rec->subtitle.set(getValue(i, "subtitle"));
      rec->edition.set(getValue(i, "edition"));
      rec->editor.set(getValue(i, "editor"));
      rec->howpublished.set(getValue(i, "howpublished"));
      rec->organization.set(getValue(i, "organization"));
      rec->publisher.set(getValue(i, "publisher_no"));
      rec->year.set(getValue(i, "year"));
      rec->pages.set(getValue(i, "pages"));
      rec->translator.set(getValue(i, "translator"));
      rec->volume.set(getValue(i, "volume"));
      rec->number.set(getValue(i, "number"));
      rec->month.set(getValue(i, "month"));
      rec->series.set(getValue(i, "series"));
      rec->type.set(getValue(i, "type"));
      rec->key.set(getValue(i, "key"));
      rec->orgtitle.set(getValue(i, "orgtitle"));
      rec->orgpublisher.set(getValue(i, "orgpublisher_no"));
      rec->orgyear.set(getValue(i, "orgyear"));
      rec->isbn_issn.set(getValue(i, "isbn_issn"));
      rec->hidingplace.set(getValue(i, "hidingplace"));
      rec->bibtex.set(getValue(i, "bibtex"));
      rec->createbibtex.set(getValue(i, "createbibtex"));
      rec->entrytype.set(getValue(i, "entrytype"));
      rec->modified.set(getValue(i, "modified"));
      rec->created.set(getValue(i, "created"));
      rec->memo.set(getValue(i, "memo"));
      rec->id.set(getValue(i, "no"));
      rec->next=last;
      last=rec;
    }
    exec("commit");
  } catch(...) {
    delete last;
    last=0L;
    exec("rollback");
    throw;
  }
  return last;
}


void KaspaBase::updatePubl(PublRec *publ, const char *where, bool block) {
  try {
   if(!where || !strlen(where)) throw KaspaErr("No Where-Condition!");
    if(publ->title.hasChanged()  + publ->subtitle.hasChanged()     + publ->edition.hasChanged()+
	 publ->editor.hasChanged()     + publ->howpublished.hasChanged() + publ->organization.hasChanged()+
	 publ->publisher.hasChanged()  + publ->year.hasChanged()         + publ->pages.hasChanged()+
	 publ->translator.hasChanged() + publ->volume.hasChanged()       + publ->number.hasChanged()+
	 publ->month.hasChanged()      + publ->series.hasChanged()       + publ->type.hasChanged()+
	 publ->key.hasChanged()        + publ->orgtitle.hasChanged()     + publ->orgpublisher.hasChanged()+
	 publ->orgyear.hasChanged()    + publ->isbn_issn.hasChanged()    + publ->hidingplace.hasChanged()+
	 publ->memo.hasChanged()       + publ->entrytype.hasChanged()    + publ->bibtex.hasChanged()+
	 publ->createbibtex.hasChanged()== 0 ) return;

    exec("begin", block);
    Str s=Str("update publication set ");
    if(publ->title.hasChanged())        {s+="title='";          publ->title.getSql(s);								 s+="', ";}
    if(publ->subtitle.hasChanged())     {s+="subtitle='"; 		  publ->subtitle.getSql(s);							 s+="', ";}
    if(publ->edition.hasChanged())      {s+="edition=";  			  publ->edition.get(s);	  							 s+=", ";}
    if(publ->editor.hasChanged())       {s+="editor='";   		  publ->editor.getSql(s);  							 s+="', ";}
    if(publ->howpublished.hasChanged()) {s+="howpublished='";   publ->howpublished.getSql(s); 				 s+="', ";}
    if(publ->organization.hasChanged()) {s+="organization='";   publ->organization.getSql(s);					 s+="', ";}
    if(publ->publisher.hasChanged())    {s+="publisher_no=";    publ->publisher.get(s); 	 						 s+=", ";}
    if(publ->year.hasChanged())         {s+="year='"; 				  publ->year.getSql(s); 								 s+="', ";}
    if(publ->pages.hasChanged())        {s+="pages='"; 				  publ->pages.getSql(s); 								 s+="', ";}
    if(publ->translator.hasChanged())   {s+="translator='"; 	  publ->translator.getSql(s); 					 s+="', ";}
    if(publ->volume.hasChanged())       {s+="volume='"; 			  publ->volume.getSql(s); 							 s+="', ";}
    if(publ->number.hasChanged())       {s+="number='"; 			  publ->number.getSql(s); 							 s+="', ";}
    if(publ->month.hasChanged())        {s+="month='"; 				  publ->month.getSql(s); 								 s+="', ";}
    if(publ->series.hasChanged())       {s+="series='";			    publ->series.getSql(s); 							 s+="', ";}
    if(publ->type.hasChanged())         {s+="type='"; 				  publ->type.getSql(s); 								 s+="', ";}
    if(publ->key.hasChanged())          {s+="key='"; 					  publ->key.getSql(s); 									 s+="', ";}
    if(publ->orgtitle.hasChanged())     {s+="orgtitle='"; 		  publ->orgtitle.getSql(s); 						 s+="', ";}
    if(publ->orgpublisher.hasChanged()) {s+="orgpublisher_no="; publ->orgpublisher.get(s); 						 s+=", ";}
    if(publ->orgyear.hasChanged())      {s+="orgyear='"; 			  publ->orgyear.getSql(s); 							 s+="', ";}
    if(publ->isbn_issn.hasChanged())    {s+="isbn_issn='"; 		  publ->isbn_issn.getSql(s); 						 s+="', ";}
    if(publ->hidingplace.hasChanged())  {s+="hidingplace='"; 	  publ->hidingplace.getSql(s); 					 s+="', ";}
    if(publ->memo.hasChanged())         {s+="memo='";           publ->memo.getSql(s);                  s+="', ";}
    if(publ->bibtex.hasChanged())       {s+="bibtex='"; 			  s+=publ->bibtex.getSql();                s+="', ";}
    if(publ->createbibtex.hasChanged()) {s+="createbibtex='";   publ->createbibtex.get(s);  					 s+="', ";}
    if(publ->entrytype.hasChanged() )   {s+="entrytype="; 		  publ->entrytype.get(s); 							 s+=", ";};
		s+="modified='"; s+=getTime(); s+="', ";
    if(s[s.length()-2]==',') s[s.length()-2]=' ';
    s+=" "; s+=where;
    exec(s, block);
    publ->unchanged();
    exec("commit", block);
  } catch(...) {
    exec("rollback", block);
    throw;
  }
}


/*************************************************************************
 *  Note
 **************************************************************************/
Oid KaspaBase::newNote(Oid o) {
  Oid ret;
  try {
    exec("begin");
		Str s;
		if(o==InvalidOid)
			s="insert into note (type, publication_no, created, modified) values (0,";
		else
			s="insert into note (type, publication_no, created, modified) values (1,";
		s+=o;
		s+=",'";
		s+=getTime();
		s+="','";
		s+=getTime();
		s+="')";
    exec(s);
    ret=getNo(Str("note"), oidStatus());
    exec("commit");
  } catch (...) {
    exec("rollback");
    throw;
  }
  return ret;
}

void KaspaBase::deleteNote(const char *where) {
	// Won't kill the whole table...
	if(!where || strlen(where)==0) abort();
	if(isLocked("note", where)) throw BaseDeleteLock("Can't delete records - one is locked!");
  try {
    exec("begin");
    deleteTuples("note", where);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
}

NoteRec *KaspaBase::getNote(const char *where) {
  NoteRec *last=0L;
  try {
    exec("begin");
    Str s("select no, title, memo, type, publication_no, modified, created from note ");
    s+=where;
    exec(s);

    int i;	
    for(i=0; i<tuples(); i++) {
      NoteRec *rec= new NoteRec;
      rec->id.set(getValue(i, "no"));
      rec->title.set(getValue(i, "title"));
      rec->publno.set(getValue(i, "publication_no"));
      rec->type.set(getValue(i, "type"));
      rec->created.set(getValue(i, "created"));
      rec->modified.set(getValue(i, "modified"));
      rec->memo.set(getValue(i, "memo"));
      rec->next=last;
      last=rec;
    }
    exec("commit");
  } catch(...) {
    delete last;
    last=0L;
    exec("rollback");
    throw;
  }
  return last;
}

void KaspaBase::updateNote(NoteRec *note, const char *where, bool block) {
  try {
    if(!where || !strlen(where)) throw KaspaErr("No Where-Condition!");
    if(note->title.hasChanged()+note->memo.hasChanged()+note->type.hasChanged()+
       note->publno.hasChanged()==0) return;

    exec("begin", block);  		

    Str s=Str("update note set ");
    if(note->title.hasChanged())  { s+="title ='"; note->title.getSql(s); s+="', ";}
    if(note->type.hasChanged())   { s+="type=";    note->type.get(s); s+=", ";}
    if(note->publno.hasChanged()) { s+="publication_no=";  note->publno.get(s); s+=", ";}
    if(note->memo.hasChanged())   { s+="memo='"; note->memo.getSql(s); s+="', "; }
		s+="modified='"; s+=getTime(); s+="', ";
    if(s[s.length()-2]==',') s[s.length()-2]=' ';
    s+=" "; s+=where;
    exec(s, block);
    note->unchanged();
    exec("commit", block);
  } catch(...) {
    exec("rollback", block);
    throw;
  }
}


void KaspaBase::moveNote(Oid id, Oid parent, Oid sibling) {
  try {
    Str s=Str("update note set ");
    s+="parent="; oid2str(parent, &s);
    s+=", sibling="; oid2str(sibling, &s);
    s+=" where no=";
    oid2str(id, &s);
    exec(s, true);
    exec("commit", true);
  } catch(...) {
    exec("rollback", true);
    throw;
  }

}


/*************************************************************************
 *  Part
 **************************************************************************/
Oid KaspaBase::newPart(Oid publ) {
  Oid ret;
  try {
    exec("begin");
    Str s=Str("insert into part (publication_no, created, modified) values (");
    s+=publ;
    s+=",'";
		s+=getTime();
		s+="','";
		s+=getTime();
		s+="')";
    exec(s);
    ret=getNo(Str("part"), oidStatus());
    exec("commit");
  } catch (...) {
    exec("rollback");
    throw;
  }
  return ret;
}

void KaspaBase::deletePart(const char *where) {
		// Won't kill the whole table...
		if(!where || strlen(where)==0) abort();
		Str s;
		if(isLocked("part", where)) throw BaseDeleteLock("Can't delete records - one is locked!");

    s="where part_author.part_no in (select no from part ";
		s+=where;
		s+=")";
    deletePartAuthor(s);

		if(strlen(where)>5) {
	    s="where partdata.part_no=part.no and ";
			s+=(where+5);
		} else {
//			debug("Kill all Parts?!?!?! - Abort");
			abort();
		}
    deletePartData(s);	
  try {
    exec("begin");
    deleteTuples("part", where);
    exec("commit");
  } catch(...) {
  	exec("rollback");
	throw;
  }
}

PartRec *KaspaBase::getPart(const char *where) {
  PartRec *last=0L;
  try {
    exec("begin");
    Str s("select no, publication_no, partno, title, " \
					"pages, key, intro_no, bibtex, memo, language, "\
					"createbibtex, created, modified from part ");
    s+=where;
    exec(s);

    int i;	
    for(i=0; i<tuples(); i++) {
      PartRec *rec= new PartRec;
      rec->id.set(getValue(i, "no"));
      rec->partno.set(getValue(i, "partno"));
      rec->publno.set(getValue(i, "publication_no"));
      rec->title.set(getValue(i, "title"));
      rec->pages.set(getValue(i, "pages"));
      rec->key.set(getValue(i, "key"));
      rec->intro.set(getValue(i, "intro_no"));
      rec->bibtex.set(getValue(i, "bibtex"));
      rec->createbibtex.set(getValue(i, "createbibtex"));
      rec->memo.set(getValue(i, "memo"));
      rec->modified.set(getValue(i, "modified"));
      rec->created.set(getValue(i, "created"));
      rec->language.set(getValue(i, "language"));
      rec->next=last;
      last=rec;
    }
    exec("commit");
  } catch(...) {
    delete last;
    last=0L;
    exec("rollback");
    throw;
  }
  return last;
}

void KaspaBase::updatePart(PartRec *part, const char *where, bool block) {
  try {
    if(!where || !strlen(where)) throw KaspaErr("No Where-Condition!");
    if(part->title.hasChanged()+part->key.hasChanged()+part->partno.hasChanged()+
       part->pages.hasChanged()+part->intro.hasChanged()+
       part->memo.hasChanged()+part->language.hasChanged()+
			 part->bibtex.hasChanged()+part->createbibtex.hasChanged()==0) return;

    exec("begin", block);			
    Str s=Str("update part set ");
    if(part->title.hasChanged()) { s+="title ='"; part->title.getSql(s); s+="', ";}
    if(part->language.hasChanged()) { s+="language ='"; part->language.getSql(s); s+="', ";}
    if(part->key.hasChanged()) { s+="key='"; part->key.getSql(s); s+="', ";}
    if(part->partno.hasChanged()) { s+="partno="; part->partno.get(s); s+=", ";}
	  if(part->publno.hasChanged()) { s+="publication_no="; part->publno.get(s); s+=", ";}
    if(part->pages.hasChanged()) { s+="pages='"; part->pages.getSql(s); s+="', ";}
    if(part->intro.hasChanged()) { s+="intro_no="; part->intro.get(s); s+=", ";}
    if(part->memo.hasChanged()) { s+="memo='"; part->memo.getSql(s); s+="', ";}
    if(part->bibtex.hasChanged()) { s+="bibtex='"; s+=part->bibtex.getSql(); s+="', ";}
    if(part->createbibtex.hasChanged()) { s+="createbibtex='"; part->createbibtex.get(s); s+="', ";}
		s+="modified='"; s+=getTime(); s+="', ";
    if(s[s.length()-2]==',') s[s.length()-2]=' ';
    s+=" "; s+=where;
    exec(s, block);
    part->unchanged();
    exec("commit", block);
  } catch(...) {
    exec("rollback", block);
    throw;
  }
}


/*************************************************************************
 *  PartData
 **************************************************************************/
Oid KaspaBase::newPartData(Oid part) {
  Oid ret;
  try {
    exec("begin");
		Str s=Str("insert into partdata (part_no, created, modified) values (");
		s+=part;
		s+=",'";
		s+=getTime();
 		s+="','";
		s+=getTime();
		s+="')";
    exec(s);
    ret=getNo(Str("partdata"), oidStatus());
    exec("commit");
  } catch (...) {
    exec("rollback");
    throw;
  }
  return ret;
}

void KaspaBase::deletePartData(const char *where) {
	// Won't kill the whole table...
	if(!where || strlen(where)==0) abort();
	if(isLocked("partdata", where)) throw BaseDeleteLock("Can't delete records - one is locked!");
  try {
    exec("begin");
    deleteTuples("partdata", where);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
}

PartDataRec *KaspaBase::getPartData(const char *where) {
  PartDataRec *last=0L;
  try {
    exec("begin");
    Str s("select partdata.no, partdata.file, partdata.astext, partdata.filename, partdata.type, "\
          "partdata.part_no, part.memo from partdata, part ");
    if(where) { s+=where; s+=" and partdata.part_no=part.no"; }
    else s+="where partdata.part_no=part.no";
    exec(s);

    int i;	
    for(i=0; i<tuples(); i++) {
      PartDataRec *rec= new PartDataRec;
      rec->id.set(getValue(i, "no"));
      rec->file.set(getValue(i, "file"));
      rec->filename.set(getValue(i, "filename"));
      rec->text.set(getValue(i, "astext"));
      rec->type.set(getValue(i, "type"));
      rec->partno.set(getValue(i, "part_no"));
			rec->memo.set(getValue(i, "memo"));
      rec->next=last;
      last=rec;
    }
    exec("commit");
  } catch(...) {
    delete last;
    last=0L;
    exec("rollback");
    throw;
  }
  return last;
}


void KaspaBase::updatePartData(PartDataRec *partdata, const char *where, bool block) {
  try {
    if(!where || !strlen(where)) throw KaspaErr("No Where-Condition!");
    if(!(partdata->file.hasChanged()+partdata->filename.hasChanged()+partdata->type.hasChanged()+
       partdata->text.hasChanged()+partdata->partno.hasChanged()==0)) {
      exec("begin", block);
      deleteLo("partdata", "file", where);
      exec("commit");
      exec("begin");
      Str s=Str("update partdata set ");
      if(partdata->filename.hasChanged()) { s+="filename='"; partdata->filename.getSql(s); s+="', ";}
      if(partdata->type.hasChanged())     { s+="type='"; partdata->type.getSql(s); s+="', ";}
      if(partdata->partno.hasChanged())   { s+="part_no="; partdata->partno.get(s); s+=", ";}
      if(partdata->file.hasChanged())   { s+="file='"; partdata->file.getSql(s); s+="', ";}
      if(partdata->text.hasChanged())   { s+="astext='"; partdata->text.getSql(s); s+="', ";}
	  	s+="modified='"; s+=getTime(); s+="', ";
      if(s[s.length()-2]==',') s[s.length()-2]=' ';
      s+=" "; s+=where;
      exec(s, block);
      exec("commit", block);
    }
  } catch(...) {
    exec("rollback", block);
    throw;
  }
  try {
    if(!(partdata->memo.hasChanged()==0)) {
      exec("begin");
      Str s=Str("update part set ");
      if(partdata->memo.hasChanged()) { s+="memo='"; partdata->memo.getSql(s); s+="', ";}
  		s+="modified='"; s+=getTime(); s+="', ";
      if(s[s.length()-2]==',') s[s.length()-2]=' ';
      s+=" where part.no=";
      partdata->partno.get(s);
      exec(s, block);
      exec("commit", block);
    }
  } catch(...) {
    exec("rollback", block);
    throw;
  }

  partdata->unchanged();
}


/*************************************************************************
 *  Publisher
 **************************************************************************/
Oid KaspaBase::newPublisher() {
  Oid ret;
  try {
    exec("begin");
    Str s("insert into publisher (created, modified) values ('");
		s+=getTime();
		s+="','";
		s+=getTime();
		s+="')";
    exec(s);
    ret=getNo(Str("publisher"), oidStatus());
    exec("commit");
  } catch (...) {
    exec("rollback");
    throw;
  }
  return ret;
}

void KaspaBase::deletePublisher(const char *where) {
	// Won't kill the whole table...
	if(!where || strlen(where)==0) abort();
	if(isLocked("publisher", where)) throw BaseDeleteLock("Can't delete records - one is locked!");
  try {
    exec("begin");
    deleteTuples("publisher", where);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
}

PublisherRec *KaspaBase::getPublisher(const char *where) {
  PublisherRec *last=0L;
  try {
    exec("begin");
    Str s("select no, name, city, serie, modified, created from publisher ");
    s+=where;
    exec(s);

    int i;	
    for(i=0; i<tuples(); i++) {
      PublisherRec *rec= new PublisherRec;
      rec->id.set(getValue(i, "no"));
      rec->name.set(getValue(i, "name"));
      rec->city.set(getValue(i, "city"));
      rec->serie.set(getValue(i, "serie"));
      rec->modified.set(getValue(i, "modified"));
      rec->created.set(getValue(i, "created"));
      rec->next=last;
      last=rec;
    }
    exec("commit");
  } catch(...) {
    delete last;
    last=0L;
    exec("rollback");
    throw;
  }
  return last;
}


void KaspaBase::updatePublisher(PublisherRec *publisher, const char *where, bool block) {
  try {
    if(!where || !strlen(where)) throw KaspaErr("No Where-Condition!");

    if(publisher->name.hasChanged()+publisher->city.hasChanged()+publisher->serie.hasChanged()==0) return;

    exec("begin", block);

    Str s=Str("update publisher set ");
    if(publisher->name.hasChanged()) { s+="name='"; publisher->name.getSql(s); s+="', ";}
    if(publisher->city.hasChanged()) { s+="city='"; publisher->city.getSql(s); s+="', ";}
    if(publisher->serie.hasChanged()) { s+="serie='"; publisher->serie.getSql(s); s+="', "; }
		s+="modified='"; s+=getTime(); s+="', ";
    if(s[s.length()-2]==',') s[s.length()-2]=' ';
    s+=" "; s+=where;
    exec(s, block);
    publisher->unchanged();
    exec("commit", block);
  } catch(...) {
    exec("rollback", block);
    throw;
  }
}


/*************************************************************************
 *
 **************************************************************************/
void KaspaBase::newPublAuthor(Oid publ, Oid author) {
  try {
    exec("begin");
    Str s=Str("select count(*) as c from publication_author where author_no=");
    oid2str(author, &s); s+=" and publication_no="; oid2str(publ, &s);
    exec(s);
    if(strcmp(getValue(0, "c"), "0")) return;


    s="select count(*) as c from publication_author where publication_no=";
    oid2str(publ, &s);
    exec(s);
    Str c;
    c=getValue(0, "c");

    s="insert into publication_author (publication_no, author_no, author_order, created, modified) values (";
    oid2str(publ, &s);
    s+=",";
		oid2str(author, &s);
    s+=",";
		s+=c;
    s+=",'";
		s+=getTime();
		s+="','";
		s+=getTime();
		s+="')";
    exec(s);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
}	

void KaspaBase::renumber_publauthors(Oid publ) {
  try {
    exec("begin");
    Str s=Str("select author_no from publication_author where publication_no=");
    oid2str(publ, &s);
    s+=" order by author_order";
    exec(s);
    Str t;
    for(int i=0; i<tuples(); i++) {
      t+="update publication_author set author_order=";
      t+=i;
      t+=" where author_no=";
      t+=getValue(i, "author_no");
      t+=";";
    }
    exec(t);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
}

void KaspaBase::renumber_partauthors(Oid publ) {
  try {
    exec("begin");
    Str s=Str("select author_no from part_author where part_no=");
    oid2str(publ, &s);
    s+=" order by author_order";
    exec(s);
    Str t;
    for(int i=0; i<tuples(); i++) {
      t+="update part_author set author_order=";
      t+=i;
      t+=" where author_no=";
      t+=getValue(i, "author_no");
      t+=";";
    }
    exec(t);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
}

void KaspaBase::deletePublAuthor(const char *where) {
	// Won't kill the whole table...
	if(!where || strlen(where)==0) abort();
  Str s("select author_no, publication_no from publication_author "); s+=where;
  exec(s);
  Oid publ[tuples()];
  Oid author[tuples()];
  for(int i=0; i<tuples(); i++) {
     publ[i]=str2oid(getValue(i, "publication_no"));
     author[i]=str2oid(getValue(i, "author_no"));
  }
  for(int i=0; i<tuples(); i++) {
     deletePublAuthor(publ[i], author[i]);
  }
}

void KaspaBase::deletePublAuthor(Oid publ, Oid author) {
  Str where;
  where="where publication_no="; where+=publ; where+=" and author_no="; where+=author;
	if(isLocked("publication_author", where)) throw BaseDeleteLock("Can't delete records - one is locked!");
  try {
    exec("begin");
    deleteTuples("publication_author", where);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
  renumber_publauthors(publ);
}

void KaspaBase::newPartAuthor(Oid part, Oid author) {
  try {
    exec("begin");

    Str s=Str("select count(*) as c from part_author where author_no=");
    oid2str(author, &s);  s+=" and part_no="; oid2str(part, &s);
    exec(s);
    if(strcmp(getValue(0, "c"), "0")) return;

    s="select count(*) as c from part_author where part_no=";
    oid2str(part, &s);
    exec(s);
    Str c;
    c=getValue(0, "c");

    s="insert into part_author (part_no, author_no, author_order, created, modified) values (";
    s+=part;
    s+=",";
    s+=author;
		s+=",";
    s+=c;
		s+=",'";
		s+=getTime();
		s+="','";
		s+=getTime();
		s+="')";
    exec(s);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
}

/*
void KaspaBase::deletePartAuthor(const char *where) {
	// Won't kill the whole table...
	if(!where || strlen(where)==0) abort();
	if(isLocked("part_author", where)) throw BaseDeleteLock("Can't delete records - one is locked!");
  try {
    exec("begin");
    deleteTuples("part_author", where);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
}
*/

void KaspaBase::deletePartAuthor(const char *where) {
	// Won't kill the whole table...
	if(!where || strlen(where)==0) abort();
  Str s("select author_no, part_no from part_author "); s+=where;
  exec(s);
  Oid part[tuples()];
  Oid author[tuples()];
  for(int i=0; i<tuples(); i++) {
     part[i]=str2oid(getValue(i, "part_no"));
     author[i]=str2oid(getValue(i, "author_no"));
  }
  for(int i=0; i<tuples(); i++) {
     deletePartAuthor(part[i], author[i]);
  }
}

void KaspaBase::deletePartAuthor(Oid part, Oid author) {
  Str where;
  where="where part_no="; where+=part; where+=" and author_no="; where+=author;
	if(isLocked("part_author", where)) throw BaseDeleteLock("Can't delete records - one is locked!");
  try {
    exec("begin");
    deleteTuples("part_author", where);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
  renumber_partauthors(part);
}


void KaspaBase::newLink(Oid obj1, Oid obj2, const char *table) {
  try {
    exec("begin");
    Str s=Str("insert into link (obj_1, obj_2, tbl, created, modified) values (");
    s+=obj1;
    s+=",";
    s+=obj2;
    s+=",'";
    s+=table;
    s+="','";
		s+=getTime();
		s+="','";
		s+=getTime();
		s+="')";
    exec(s);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
}

void KaspaBase::deleteLink(const char *where) {
	// Won't kill the whole table...
	if(!where || strlen(where)==0) abort();
	if(isLocked("link", where)) throw BaseDeleteLock("Can't delete records - one is locked!");
  try {
    exec("begin");
    deleteTuples("link", where);
    exec("commit");
  } catch(...) {
    exec("rollback");
    throw;
  }
}



//*******************************************************
void KaspaBase::buildGetLinkClause(const char *table, Oid obj, Str& s) {
  s=",link where (";
  s+=mystrdup(table);
  s+=".no=link.obj_1 and link.obj_2=";
  s+=obj;
  s+=") or (";
  s+=table;
  s+=".no=link.obj_2 and link.obj_1=";
  s+=obj;
  s+=")";
}

Title *KaspaBase::getLinks(Oid obj) {
  Title *rec=0L;
  try {
    Str s;
    buildGetLinkClause("author", obj, s);
    rec=getAuthorNames(s);

    buildGetLinkClause("publication", obj, s);
    if(rec) rec->append(getPublTitles(s));
		else rec=getPublTitles(s);

    buildGetLinkClause("part", obj, s);
    if(rec) rec->append(getPartTitles(s));
		else rec=getPartTitles(s);

    buildGetLinkClause("partdata", obj, s);
    if(rec) rec->append(getPartDataNames(s));
		else rec=getPartDataNames(s);

    buildGetLinkClause("note", obj, s);
    if(rec) rec->append(getNoteTitles(s));
		else rec=getNoteTitles(s);
  } catch(...) { throw; }

  return rec;
}
//******************************************************


struct oid_chain {
	Oid o;
	oid_chain *next;
	~oid_chain() { delete next; };
};

void KaspaBase::deleteLo(const char *table, const char *field, const char *where) {
  Str s("select ");
  s+=field;
  s+=" from ";
  s+=table;
  s+=" ";
  s+=where;
  exec(s);
  int i;
	oid_chain *last=0L;
  for(i=0; i<tuples(); i++){
    oid_chain *o_id=new oid_chain;
		o_id->o=str2oid(getValue(i, field));
		o_id->next=last;
		last=o_id;
	}
	for(oid_chain *oc=last; oc; oc=oc->next) {
//  printf("\noc->o=%d", oc->o);
    if(oc->o!=InvalidOid)
      lUnlink(oc->o);
  }
	delete last;
} 	

bool KaspaBase::isLocked(const char *table, const char *where) {
	Str s("select oid from ");
	s+=table;
	s+=" ";
	s+=where;
	exec(s, false);
	bool locked=false;
	for(int i=0; i<tuples(); i++) {
		Oid o=str2oid(getValue(i, "oid"));
		locked=isLocked(o);
	}
  return locked;
}

bool KaspaBase::isLocked(Oid o) {

	LockTableItem *i;
	for(i=locktable; i; i=i->next)
		if(i->id==o) return true;
	return false;

/*	if(!lockObj(o))	return true; // old lock mechanism, needs user_locks
	else {
		unlockObj(o);
		return false;
	} */
}

void KaspaBase::deleteTuples(const char *table, const char *where) {
	// Won't delete the whole table...
	if(!where || strlen(where)==0) abort();

  Str s("delete from ");
  s+=table;
  s+=" ";
  s+=where;
  exec(s, false);
} 	


bool KaspaBase::lockObj(Oid o) {
	if(isLocked(o)) return false;
	LockTableItem *lock=new LockTableItem;
	lock->id=o;
	lock->next=locktable;
	locktable=lock;
	return true;

/*  int ret=0;             // old lock mechanism, needs user_locks
  exec("begin");
  Str s="select user_write_lock_oid(";
  s+=o;
  s+=") AS locked";
  exec(s);
  if(tuples()==1)
    ret=strcmp(getValue(0, "locked"), "0");
  exec("commit");
  return ret; */
}


bool KaspaBase::unlockObj(Oid o) {
	LockTableItem *prev=0L;
	for(LockTableItem *lock=locktable; lock; prev=lock, lock=lock->next)
		if(lock->id==o) {
			if(prev) prev->next=lock->next;
			else locktable=lock->next;
			delete lock;
			return true;
		}
	return false;


/*  int ret=0;       // old lock mechanism, needs user_locks
  exec("begin");
  Str s("select user_write_unlock_oid(");
  s+=o;
  s+=") AS unlocked";
  exec(s);
  if(tuples()==1)
    ret=strcmp(getValue(0, "unlocked"), "0");
  exec("commit");
  return ret; */
}

AuthorName *KaspaBase::getAuthorNames(const char *where) {
  AuthorName *last=0;
  try {
    exec("begin");
    Str s("select no, firstname, lastname from author ");
    s+=where;
    exec(s);
    for(int i=0; i<tuples(); i++) {
      AuthorName *rec=new AuthorName;
      rec->firstname.set(getValue(i, "firstname"));
      rec->lastname.set(getValue(i, "lastname"));
      rec->title.set(getValue(i, "firstname"));
      rec->title.append(" ");
      rec->title.append(getValue(i, "lastname"));
      rec->table.set("author");
      rec->o.set(getValue(i, "no"));
      rec->next=last;
      last=rec;
    }
    exec("commit");
  } catch(...) {
    delete last;
    last=0L;
    exec("rollback");
    throw;
  }
  return last;	
}

PublTitle *KaspaBase::getPublTitles(const char *where) {
  PublTitle *last=0;
  try {
    exec("begin");
    Str s("select no, year, title, entrytype from publication ");
    s+=where;
    exec(s);
    for(int i=0; i<tuples(); i++) {
      PublTitle *rec=new PublTitle;
      rec->title.set(getValue(i, "title"));
      rec->year.set(getValue(i, "year"));
      rec->table.set("publication");
      rec->o.set(getValue(i, "no"));
			rec->entrytype.set(getValue(i, "entrytype"));
      rec->next=last;
      last=rec;
    }
    exec("commit");
  } catch(...) {
    delete last;
    last=0L;
    exec("rollback");
    throw;
  }
  return last;	
}

PartTitle *KaspaBase::getPartTitles(const char *where, bool fullpath) {
  PartTitle *last=0;
  try {
    exec("begin");
    Str s;
    if (fullpath)
  	  s="select part.no, part.title as t2, publication.title as t1 " \
        "from part join publication on (part.publication_no = publication.no) ";
    else
  	  s="select no, title as t2 from part ";
    s+=where;
    exec(s);
    for(int i=0; i<tuples(); i++) {
      PartTitle *rec=new PartTitle;
      Str s;
      if(fullpath) {
        s=getValue(i, "t1");
        s+=" : ";
      }
      rec->title.set(s+getValue(i, "t2"));
      rec->table.set("part");
      rec->o.set(getValue(i, "no"));
      rec->next=last;
      last=rec;
    }
    exec("commit");
  } catch(...) {
    delete last;
    last=0L;
    exec("rollback");
    throw;
  }
  return last;	
}

PartDataName *KaspaBase::getPartDataNames(const char *where, bool fullpath) {
  PartDataName *last=0;
  try {
		exec("begin");
    Str s;
    if (fullpath)
  	  s="select partdata.no, partdata.filename, part.title as t2, publication.title as t1 " \
        "from partdata join part on (part.no=partdata.part_no) " \
        "join publication on (part.publication_no = publication.no) ";
    else
  	  s="select no, filename from partdata ";

		s+=where;
		exec(s);
		for(int i=0; i<tuples(); i++) {
			PartDataName *rec=new PartDataName;
      Str s;
      if(fullpath) {
        s=getValue(i, "t1"); s+=" : "; s+=getValue(i, "t2"); s+=" : ";
      }
      rec->title.set(s+getValue(i, "filename"));
      rec->filename.set(getValue(i, "filename"));
			rec->table.set("partdata");
			rec->o.set(getValue(i, "no"));
			rec->next=last;
			last=rec;
		}
		exec("commit");
	} catch(...) {
		delete last;
    last=0L;
    exec("rollback");
    throw;
  }
  return last;
}


NoteTitle *KaspaBase::getNoteTitles(const char *where) {
  NoteTitle *last=0;
  try {
    exec("begin");
    Str s("select no, parent, title, publication_no from note ");
    s+=where;
    exec(s);
    for(int i=0; i<tuples(); i++) {
      NoteTitle *rec=new NoteTitle;
      rec->title.set(getValue(i, "title"));
      rec->table.set("note");
      rec->publno.set(getValue(i, "publication_no"));
      rec->o.set(getValue(i, "no"));
      rec->parent.set(getValue(i, "parent"));
      rec->next=last;
      last=rec;
    }
    exec("commit");
  } catch(...) {
    delete last;
    last=0L;
    exec("rollback");
    throw;
  }
  return last;	
}

PublisherName *KaspaBase::getPublisherNames(const char *where) {
  PublisherName *last=0;
  try {
    exec("begin");
    Str s("select no, name, city, serie from publisher ");
    s+=where;
    exec(s);
    for(int i=0; i<tuples(); i++) {
    	if(strcmp(getValue(i, "serie"), ""))
    		continue;
      PublisherName *rec=new PublisherName;
/*
			rec->title.set(getValue(i, "serie"));
			if(!strlen(rec->title.get()))
*/			
	    rec->title.set(getValue(i, "name"));
      rec->address.set(getValue(i, "city"));
      rec->table.set("publisher");
      rec->o.set(getValue(i, "no"));
      rec->next=last;
      last=rec;
    }
    exec("commit");
  } catch(...) {
    delete last;
    last=0L;
    exec("rollback");
    throw;
  }
  return last;	
}


JournalName *KaspaBase::getJournalNames(const char *where) {
  JournalName *last=0;
  try {
    exec("begin");
    Str s("select no, name, city, serie from publisher ");
    s+=where;
    exec(s);
    for(int i=0; i<tuples(); i++) {
    	if(!strcmp(getValue(i, "serie"), ""))
    		continue;
      JournalName *rec=new JournalName;
			rec->title.set(getValue(i, "serie"));
/*			if(!strlen(rec->title.get()))
	      rec->title.set(getValue(i, "name"));
*/	
      rec->address.set(getValue(i, "city"));
      rec->table.set("publisher");
      rec->o.set(getValue(i, "no"));
      rec->next=last;
      last=rec;
    }
    exec("commit");
  } catch(...) {
    delete last;
    last=0L;
    exec("rollback");
    throw;
  }
  return last;	
}


/*
void KaspaBase::search(const char *where, AuthorName **an, PartTitle **pt,
									PublTitle **pbt, NoteTitle **nt, PartDataName **pdt) {
	try {
		exec("drop table tmpsearch");
	} catch (KaspaErr) {};
	Str s("select distinct tab, type into table tmpsearch from word ");
	s+=where;
	exec(s);

	s="where author.oid = tmpsearch.tab";
	*an=getAuthorNames(s);
	s="where publication.oid = tmpsearch.tab";
	*pbt=getPublTitles(s);
	s="where note.oid = tmpsearch.tab";
	*nt=getNoteTitles(s);

	try {
	  exec("begin");
  	s = "select part.oid, part.title as pt, publication.title as pb";
		s+= " from part, publication where part.oid=tmpsearch.tab and";
		s+= " part.publicationno=publication.oid order by pb";
	  exec(s);

  	for(int i=tuples()-1; i>=0; i--) {
      PartTitle *rec=new PartTitle;
			Str s=Str(getValue(i, "pb")); s+=" : ";
			s+=getValue(i, "pt");
			rec->title.set(s);
      rec->table.set("part");
      rec->o.set(getValue(i, "oid"));
      rec->next=*pt;
      *pt=rec;
    }
    exec("commit");
  } catch(...) {
    delete *pt;
    *pt=0L;
    exec("rollback");
    throw;
  }


	try {
	  exec("begin");
  	s = "select partdata.oid, partdata.filename as pd, part.title as pt, publication.title as pb";
		s+= " from part, partdata, publication where partdata.oid=tmpsearch.tab and";
		s+= " partdata.partno=part.oid and part.publicationno=publication.oid order by pb";
	  exec(s);

  	for(int i=tuples()-1; i>=0; i--) {
      PartDataName *rec=new PartDataName;
			Str s=Str(getValue(i, Str("pb"))); s+=" : ";
			s+=getValue(i, "pt"); s+=" : ";
			s+=getValue(i, "pd");
			rec->title.set(s);
      rec->table.set("partdata");
      rec->o.set(getValue(i, "oid"));
      rec->next=*pdt;
      *pdt=rec;
    }
    exec("commit");
  } catch(...) {
    delete *pdt;
    *pdt=0L;
    exec("rollback");
    throw;
  }

//	*pdt=getPartDataNames(s);

	exec("drop table tmpsearch");
}
*/

/**  */
int KaspaBase::bibTexIndex(Oid author, const char *year, Oid publ){
	exec("begin");
  Str s("(select no from publication where publication.year='");
	s+=year;
	s+="' and publication.no=publication_author.publication_no and publication_author.author_no=";
	s+=author;
	s+=" union select no from publication where publication.year='";
	s+=year;
	s+="' and publication.no=part.publication_no and part.no=part_author.part_no ";
	s+="and part_author.author_no=";
	s+=author;
	s+=")";
// order by publication.title";
	
	int j=0;
	exec(s);
	if(tuples()>1) j++;
	for(int i=0; i<tuples(); i++)
		if(publ==Oid(strtol(getValue(i, "no"), NULL, 10))) j=i;
	exec("end");
	return j;
}

void KaspaBase::lo2buf(Oid i, DataField& f) {
  long len=0L;
  char *buf=lo2buf(i, &len);
  f.take((unsigned char*)buf, len);
}



char *KaspaBase::lo2str(Oid o) {
  long i=0L;
  char *s=lo2buf(o, &i);
  if(!s) {
    s=new char[1];
    *s=0;
  }
  return s;
}

Oid KaspaBase::str2lo(const char *s) {
  return buf2lo(s, strlen(s));
}

Oid KaspaBase::buf2lo(const char *buf, long len) {
  if(!buf)
    return InvalidOid;
  Oid j=InvalidOid;
  j=lCreate(INV_READ | INV_WRITE);
  int lfd=lOpen(j, INV_WRITE);

  if(lfd>=0) {
    lSeek(lfd, 0, SEEK_SET);
    lWrite(lfd, buf, len);
    lClose(lfd);
  }
  return j;
}

char *KaspaBase::lo2buf(Oid i, long *len) {
  int lfd=0;
  char *buf=0L;

  if(i==InvalidOid) return 0L;
  lfd=lOpen(i, INV_READ);
  lSeek(lfd, 0, SEEK_END);
  *len=lTell(lfd);
  lSeek(lfd, 0, SEEK_SET);
  buf=new char[*len+1];
  assert(buf);
  buf[*len]=0;
  lRead(lfd, buf, *len);
  lClose(lfd);
  return buf;
}

const char *KaspaBase::getTime() {
	static char buffer[100];
  time_t curtime;
  struct tm *gtime;
  curtime = time (NULL);
  gtime = gmtime (&curtime);
	strftime(buffer, 100, "%Y-%m-%d %H:%M:%S", gtime);
	return buffer;
}
