/*------------------------------------------------------------------------*/
/*                                                                        */
/*  File:       profile.cpp                                               */
/*                                                                        */
/*  Contains:   Functions getProfileString(), getProfileInt(),            */
/*              writeProfileString()                                      */
/*  Purpose:    MS-Windows Style .ini File Interface for C++              */
/*  Author:     XuYiFeng, China                                           */
/*  Date:       25.04.1995                                                */
/*------------------------------------------------------------------------*/
/*
 * Re-Written, re-designed and some functions inserted
 * changed some on the concept to work with gcc and C++
 *
 * Rajko Albrecht, Germany
 */

#ifdef WIN32
#pragma warning ( disable : 4786 )
#endif

#ifndef rcsid
static const char rcsid[]="$Id: profile.cpp,v 2.8 2003/02/04 00:08:53 ral Exp $";
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string>
#include <ctype.h>
#include <fstream>
#include <sstream>
#include <iostream>

#ifndef WIN32
#include <sstream>
#include <unistd.h>
#include <sys/stat.h>
#else
#include <io.h>
#include <iostream>
#include <istream>
#endif

#include "kmysqladmin/helpers/helpers.h"
#include "profile.h"
#include "kmysqladmin/helpers/smart_pointer.h"

using namespace std;

//--------------------------------------------------------------------------
// removeLastSpace: shorts a string if there are specialchars at end
//--------------------------------------------------------------------------
void removeLastSpace(char *buf)
{
    if (!buf)
	return;
    int i = strlen(buf);
    int j = buf[i-1];
    while (isspace(j)) {
	buf[i-1] = '\0';
	i = strlen(buf);
	j = buf[i];
    }
}

//--------------------------------------------------------------------------
// titlePos: get a section title position & length in a string.
//--------------------------------------------------------------------------

static char *titlePos( char *buf, int *len ) 
{
    char *p = buf, *q;

    while( *p && isspace(*p) ) p++;
    if( *p != '[' )
        return 0;

    q = p+1;
    while( *q && *q != ']' ) q++;
    if( *q != ']' )
        return 0;
    if( len )
        *len = int(q - p - 1);
    return p+1;
}

//-------------------------------
// check if the line is a comment
//-------------------------------
static int isComment(char*bufPtr)
{
    if (!bufPtr)
	return 0;
    return (bufPtr[0] == ';');
}

//--------------------------------------------------------------------------
// isTitleLine: check if a string is a section title line
//--------------------------------------------------------------------------

static int isTitleLine( char *bufPtr ) 
{
    if (isComment(bufPtr))        // of course a comment is not a titleline
	return 0;
    return titlePos( bufPtr, 0 ) != 0;
}

//--------------------------------------------------------------------------
// containTitle: check if a string contains a section a title
//--------------------------------------------------------------------------

static int containTitle( char *buf, const char *section ) 
{
    char *p;
    int len;

    p = titlePos( buf, &len );
    if( p ) {
#ifdef WIN32
	if( strlen( section ) == (unsigned)len && strnicmp( section, p, len ) == 0 )
#else
	    if( strlen( section ) == (unsigned)len && strncasecmp( section, p, len ) == 0 )	 
#endif	   
		return true;
    }
    return false;
}

//--------------------------------------------------------------------------
// gotoSection: move file position to start line of a section
//--------------------------------------------------------------------------

static int gotoSection( ifstream &is, const char *section ) 
{
    char line[257];
    memset(line,'\0',257);
    while( is.getline( line, 256 ) )
	if( containTitle( line, section ) )
	    return true;
    return false;
}

//--------------------------------------------------------------------------
// textPos: get content's position of an entry
//--------------------------------------------------------------------------

static char *textPos( char *buf, const char *entry ) 
{
    if( buf[0] == ';' ) // it is comment line
        return 0;

    char *p = strchr( buf, '=' );
    if( !p )
        return 0;

    int len = int(p - buf);
#ifdef WIN32   
    if( strlen(entry) == (unsigned)len && strnicmp( buf, entry, len ) == 0 )
#else     
	if( strlen(entry) == (unsigned)len && strncasecmp( buf, entry, len ) == 0 )
#endif       
	    return p+1;

    return 0;
}

//--------------------------------------------------------------------------
// stripQuotationChar: strip a pair of quotation chars in a string
//--------------------------------------------------------------------------

static void stripQuotationChar( char *buf ) {

    char *p;
    char *q;

    p = buf;
    while( *p && isspace(*p) ) p++;
    if( !(*p == '\"' || *p == '\'') )
	return;
    q = p+strlen(p);
    while( *q != *p && q > p ) q--;
    if( q == p )
        return;
    int len = int(q - p - 1);
    memmove( buf, p+1, len );
    buf[len] = 0;
}

//--------------------------------------------------------------------------
// readEntry: read contents of entry
//--------------------------------------------------------------------------

static int readEntry( ifstream &is, const char *entry, char *buf,
                      int bufSize, bool strip )
{

    char lineBuf[256];
    char *p, *cur;
    int  len;

    cur  = buf;
    *cur = '\0';
    len  = -1;
    while( is.getline( lineBuf, 256 ) ) {
        if( isTitleLine( lineBuf ) )       // section is ended
            break;

        p = textPos( lineBuf, entry );     // not equal this entry
        if( p == 0 )
            continue;

        if( strip )
            stripQuotationChar( p );

        len = strlen(p);
        if( bufSize-1 < len )
            len = bufSize-1;

        strncpy( cur, p, len );
        cur[len] = 0;
        break;
    }

    return len;
}

//--------------------------------------------------------------------------
// getProfileString:
//--------------------------------------------------------------------------

int CProfile::getProfileString( const char *section,const char *entry,
				const std::string&defaultString,std::string& targetbuffer)const
{
    char buffer[1024];
    if (m_FileName.size()==0) {
        targetbuffer = defaultString;
        return targetbuffer.size();
    }
#ifndef WIN32
    ifstream input(m_FileName.c_str(),ios::in);
#else
    ifstream input(m_FileName.c_str(),ios::in );
#endif
    int len = -1;
    if( input ) {
	if( gotoSection( input, section ) )
	    len = readEntry(input, entry, buffer, 1023, true);
    }
    
    if( len < 0 ) {             //can not read entry, use default string
	targetbuffer = defaultString;
    } else {
	targetbuffer = buffer;
    }
    return targetbuffer.size();
}

//--------------------------------------------------------------------------
// getProfileInt:
//--------------------------------------------------------------------------

int CProfile::getProfileInt( const char *section,
			     const char *entry,int defaultInt)const
{
    char iBuf[34];   //"34" is max space "itoa" requires under 32 bit C++
#ifndef WIN32
    stringstream outputbuffer(iBuf);
#endif
#ifdef WIN32
    itoa( defaultInt, iBuf, 10 );
#else
    outputbuffer << defaultInt << ends;
#endif
    std::string defaults;
    defaults = iBuf;
    std::string target;
    getProfileString( section, entry, defaults, target);
    return atoi( target.c_str() );
}

static void writeEntry(ofstream &os, const char *entry, const char *astring)
{
    os << entry << '=' << astring << endl;
}

//--------------------------------------------------------------------------
// writeProfileString:
//--------------------------------------------------------------------------

bool CProfile::writeProfileString( const char *section,
				   const char *entry,
				   const char *astring) 
{
    char buf  [256];
    //char*path;
    std::string t_filename;
    int  titleFound, entryFound;
#ifndef WIN32
    int _iMode = umask(m_UseMode);
#endif
    bool retval = true;
    if (m_FileName.size()==0)
        return false;
#ifndef WIN32
    t_filename = m_FileName+".tmp";
    ofstream os(t_filename.c_str());
#else
    path = _mktemp(t_name);
    if (!path) {
        printf("Error getting pathname for temp file\n");
        return false;
    }
    ofstream os(path,ios::out);
#endif
    ifstream is( m_FileName.c_str(), ios::in  );

    if( !os || entry == 0 )     //maybe cannot create file or invalid entry
        return false;
    titleFound = false;
    entryFound = false;
    while( is.getline(buf, 256) ) {
        os << buf << endl;
        if( containTitle(buf, section) ) {
            titleFound = true;
            break;
        }
    }
    if( !titleFound ) {         // add section
        os << '[' << section << ']' << endl;
        writeEntry( os, entry, astring );
    } else {
	while( is.getline(buf, 256) ) {
            if( isTitleLine( buf ) )     // section ended, but still not
                break;                   // found the entry
            if( textPos( buf, entry ) ) {// entry found, so rewrite it
                entryFound = true;
                break;
            }
	    os << buf << endl;
        }
        writeEntry( os, entry, astring );
        if( is.gcount() > 0 && !entryFound )
            os << buf << endl;
        while( is.getline(buf, 256) )  // copy left lines
            os << buf << endl;
    }
    is.close();
    os.close();
    unlink( m_FileName.c_str() );
#ifdef WIN32
    rename( path, m_FileName.c_str() );
#else
    if (move_file(t_filename.c_str(),m_FileName.c_str()) == -1) {
	retval = false;
    }
    umask(_iMode);
#endif
    return retval;
}

bool CProfile::writeProfileInt( const char *section,const char *entry,int Int)
{
    int _iMode = umask(m_UseMode);
    bool retval = true;

    ostringstream outputbuffer;
    outputbuffer << Int << ends;

    if ( (unsigned)writeProfileString( section, entry, outputbuffer.str().c_str()) != outputbuffer.str().size()) 
	retval = false;
    umask(_iMode);
    return retval;
}

int CProfile::getProfileSection(  const char *section,
				  stringIlist&target)
{
    int iSize = 0;
    char iBuf[256];
    memset(iBuf,'\0',256);
    target.clear();
    if (m_FileName.size() == 0 || !section)
	return -1;

#ifndef WIN32
    ifstream i(m_FileName.c_str(), ios::in);
#else
    ifstream i(m_FileName.c_str(), ios::in);
#endif
    if (!i)
	return -1;
    if (!gotoSection(i,section))
	return 0;
    while ( !i.eof()) {
	i.getline(iBuf,255);
	if (!i)
	    break;
	if (isTitleLine(iBuf))
	    break;
	removeLastSpace(iBuf);
	iSize = strlen(iBuf);
	if (iSize && !isComment(iBuf)) {
	    target.push_back(iBuf);
	}
	memset(iBuf,0,256);
    }
    return target.size();
}

CProfile::CProfile(const char*_FileName)
    :ref_count()
{
    FileName(_FileName);
}

void CProfile::FileName(const char* name,int n_Mode)
{
    if (!name)
	m_FileName = "";
    else
	m_FileName = name;
    m_UseMode = n_Mode;
}

CProfile::~CProfile()
{
}

eProfile::eProfile(const char*_FileName,const char*_Path)
    :CProfile(_FileName)
{
    if (_Path)
	m_path = _Path;
    else
	m_path = "";
}

eProfile::~eProfile()
{    
}

bool eProfile::SetValue(const std::string&name,const std::string&value)
{
    return writeProfileString(m_path.c_str(),name.c_str(),value.c_str());
}

bool eProfile::SetValue(const std::string&name, const int value)
{
    return writeProfileInt(m_path.c_str(),name.c_str(),value);
}

bool eProfile::SetValue(const std::string&name, const bool value)
{
    return SetValue(name,value?(int)1:(int)0);
}

std::string eProfile::ReadValue(const std::string&name, const std::string&def)const
{
    string target;

    getProfileString(m_path.c_str(),name.c_str(),def,target);
    return target;
}

int eProfile::ReadValue(const std::string&name, const int def)const
{
    return getProfileInt(m_path.c_str(),name.c_str(),def);
}

bool eProfile::ReadValue(const std::string&name, const bool def)const
{
    return (getProfileInt(m_path.c_str(),name.c_str(),def?1:0)!=0?true:false);
}
