/***************************************************************************
                          account.cpp  -  description
                             -------------------
    begin                : Sat Aug 10 2002
    copyright            : (C) 2002 by Richard Garand
    email                : richard@garandnet.net
    $Id: account.cpp,v 1.19 2002/09/13 02:25:16 richard Exp $
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qtextstream.h>
 
#include "account.h"
#include "kbudgetdoc.h"
#include "kbundoaction.h"

Account::Account()
  : initialBalance(0)
{

}

Account::Account(KBudgetDoc* doc, int id, types type, QTextStream &stream, int streamVersion)
{
  m_doc = doc;
  m_id = id;
  stream >> initialBalance;
  m_type = (types)type;
  m_name = stream.readLine().stripWhiteSpace();
  dump();

  if ( streamVersion == 2 ) {
    int budgetValues, year, month;
    float value;
    stream >> budgetValues;
    for ( int i = 0; i < budgetValues; i++ ) {
      stream >> year >> month >> value;
      setBudgeted(QDate(year, month, 1), value);
    }
  }
}

Account::Account(KBudgetDoc* d, int i, types t, QString n, float ib) {
  m_type = t;
  m_name = n;
  initialBalance = ib;
  m_doc = d;
  m_id = i;
  dump();
}

Account::Account(const Account& from) {
  m_type = from.m_type;
  m_name = from.m_name;
  months = from.months;
  initialBalance = from.initialBalance;
  m_doc = from.m_doc;
  m_id = from.m_id;
}

Account::~Account() {

}

void Account::serialize(QTextStream &stream)
{
  stream << QString("Account %1 %2 %3 %4\n").arg(m_id).arg((int)m_type).arg(initialBalance).arg(m_name);
  if ( months.size() ) {
    QMap<int,year_t>::const_iterator year = months.begin();
    year_t::const_iterator month = (*year).begin();
    QString budgetValues;
    int count = 0;
    while ( year != months.end() ) {
      while ( month != (*year).end() ) {
        if ( (*month).budgeted ) {
          budgetValues += QString(" %1 %2 %3").arg(year.key()).arg(month.key()).arg((*month).budgeted);
          count++;
        }
        month++;
      }
      year++;
    }
    stream << count << budgetValues << "\n";
  } else
    stream << "0\n";
}

int Account::id() const
{
  return m_id;
}

void Account::setType(types tp)
{
  m_type = tp;
}

Account::types Account::type() const
{
  return m_type;
}

void Account::setName(QString n)
{
  if ( m_doc->canAddUndo() )
    m_doc->addUndo(new KBAccountUndo(m_doc, KBAccountUndo::UA_RENAME, m_id));
  m_name = n;
  m_doc->setModified(true);
  m_doc->signalAcctChanged(m_id);
}

QString Account::name() const
{
	return m_name;
}

void Account::addTransaction(int id)
{
  transactionUpdate(id, true);
}

void Account::removeTransaction(int id)
{
  transactionUpdate(id, false);
}

QValueList<int> Account::getTransactions(int year, int month) const
{
  QValueList<int> list;

  if ( !months.size() )
    return list;

  QMap<int,year_t>::const_iterator yr;// = months.begin()
  year_t::const_iterator mn;
  for ( yr = months.begin(); yr != months.end(); yr++ ) {
    if ( year && yr.key() != year )
      continue;
    for ( mn = (*yr).begin(); mn != (*yr).end(); mn++ ) {
      if ( month && mn.key() != month )
        continue;
      list += (*mn).transactions;
    }
  }

  return list;
}

QValueList<int> Account::getTransactions(QDate start, QDate end) const
{
  return QValueList<int>();
}

float Account::getBalance() const
{
  if ( months.size() )
    return (*--(*--months.end()).end()).endBalance;
  else
    return initialBalance;
}

float Account::startingBalance() const
{
  return initialBalance;
}

void Account::setStartingBalance(float balance)
{
  if ( m_doc->canAddUndo() )
    m_doc->addUndo(new KBAccountUndo(m_doc, KBAccountUndo::UA_STBALANCE, m_id));

  float diff = balance - initialBalance;
  initialBalance = balance;
  QMap<int,year_t>::iterator yr = months.begin();
  year_t::iterator month = (*yr).begin();
  add(yr, month, diff, 0.0f);
  m_doc->setModified(true);
  m_doc->signalAcctChanged(m_id);
}

float Account::change() const
{
  if ( m_type == AC_INCOME )
    return initialBalance - getBalance();
  else
    return getBalance() - initialBalance;
}

float Account::startingBudget(QDate date) const
{
  if ( months.size() )
    if ( date.isNull() )
      return (*(*months.begin()).begin()).startBudget;
    else if ( months.find(date.year()) != months.end() && \
      months[date.year()].find(date.month()) != months[date.year()].end() )
      return months[date.year()][date.month()].startBudget;
    else
      return 0;
  else
    return 0;
}

float Account::budget(QDate month) const
{
  if ( months.size() )
    if ( month.isNull() )
      return (*--(*--months.end()).end()).endBudget;
    else if ( months.find(month.year()) != months.end() && \
      months[month.year()].find(month.month()) != months[month.year()].end() )
      return months[month.year()][month.month()].endBudget;
    else
      return 0;
  else
    return 0;
}

float Account::budgeted(QDate month) const
{
  if ( months.size() )
    if ( month.isNull() )
      return (*--(*--months.end()).end()).budgeted;
    else if ( months.find(month.year()) != months.end() && \
      months[month.year()].find(month.month()) != months[month.year()].end() )
      return months[month.year()][month.month()].budgeted;
    else
      return 0;
  else
    return 0;
}

void Account::setBudgeted(QDate month, float value)
{
  QMap<int,year_t>::iterator yearIt;
  year_t::iterator monthIt;
  createMonth(month.year(), month.month(), yearIt, monthIt);

  float addBudget = value - (*monthIt).budgeted;
  (*monthIt).budgeted = value;
  (*monthIt).endBudget += addBudget;
  
  monthIt++;
  add(yearIt, monthIt, 0, addBudget);

  m_doc->setModified(true);
}

Account Account::clone(QDate fromDate, QDate toDate) const
{
  Account newAcct = *this;
  newAcct.months.clear();
  
  QMap<int,year_t>::const_iterator year = months.begin();
  year_t::const_iterator month = (*year).begin();
  bool firstMonth = true;

  while ( year != months.end() ) {
    while ( month != (*year).end() ) {
      if ( isInRange(fromDate, toDate, QDate(year.key(), month.key(), 1)) ) {
        if ( firstMonth ) {
          newAcct.initialBalance = (*month).startBalance;
          firstMonth = false;
        }
        newAcct.months[year.key()][month.key()] = *month;
      }
      month++;
    }
    year++;
  }

  // no months were added to the clone, so we need to set the initial balance
  const Month* m = monthBefore(fromDate);
  if ( firstMonth )
    if ( m ) {
      newAcct.initialBalance = m->endBalance;
      newAcct.months[fromDate.year()][fromDate.month()].startBudget = m->endBudget;
      newAcct.months[fromDate.year()][fromDate.month()].endBudget = m->endBudget;
      newAcct.months[fromDate.year()][fromDate.month()].startBalance = newAcct.initialBalance;
      newAcct.months[fromDate.year()][fromDate.month()].endBalance = newAcct.initialBalance;
    } else
      newAcct.initialBalance = initialBalance;

  return newAcct;
}

const Account::Month* Account::monthBefore(QDate date) const
{
  QMap<int,year_t>::const_iterator year = months.begin();
  year_t::const_iterator month = (*year).begin();
  year_t::const_iterator prevmonth;
  if ( months.size() ) {
    QDate firstMonth(months.begin().key(), (*months.begin()).begin().key(), 1);
    if ( firstMonth < date ) {
      year = months.begin();
      while ( year != months.end() ) {
        month = (*year).begin();
        while ( month != (*year).end() ) {
          if ( year.key() > date.year() || (year.key() == date.year() && month.key() > date.month()) )
            return &*prevmonth;
          else {
            prevmonth = month;
            month++;
          } // end of test
        } // end of month iterator
        year++;
      } // end of year iterator
      return &*--month;
    } // end of have previous month
  } // end of have months
  return 0;
}

bool Account::isInRange(QDate from, QDate to, QDate test) const
{
  if ( to.isValid() ) {
    if ( test.year() < from.year() || test.year() > to.year() )
      return false;
    if ( from.month() && test.year() == from.year() && test.month() < from.month() )
      return false;
    if ( to.month() && test.year() == to.year() && test.month() > to.month() )
      return false;
    return true;
  }
  if ( test.year() != from.year() )
    return false;
  if ( from.month() && test.month() != from.month() )
    return false;
  return true;
}

void Account::createMonth(int year, int month, QMap<int,year_t>::iterator &yearIt, year_t::iterator &monthIt)
{
  // if the year doesn't exist, create it and the month
  if ( months.find(year) == months.end() ) {
    months[year][month].startBalance = initialBalance;
    months[year][month].endBalance = initialBalance;
    // if the new year isn't the first in the list...
    yearIt = months.find(year);
    monthIt = (*yearIt).find(month);
    if ( yearIt != months.begin() ) {
      yearIt--;
      monthIt = (*yearIt).end();
      months[year][month].startBalance = (*monthIt).endBalance;
      months[year][month].endBalance = (*monthIt).endBalance;
      months[year][month].startBudget = (*monthIt).endBudget;
      months[year][month].endBudget = (*monthIt).endBudget;
    }
  } else if ( months[year].find(month) == months[year].end() ) {
    // the year exists but the month doesn't
    months[year][month].startBalance = 0;
    yearIt = months.find(year);
    monthIt = (*yearIt).find(month);
    if ( monthIt != (*yearIt).begin() )
      monthIt--;
    else { // assuming there are no empty years
      yearIt--;
      monthIt = --(*yearIt).end();
    }
    months[year][month].startBalance = (*monthIt).endBalance;
    months[year][month].endBalance = (*monthIt).endBalance;
    months[year][month].startBudget = (*monthIt).endBudget;
    months[year][month].endBudget = (*monthIt).endBudget;
  }

  yearIt = months.find(year);
  monthIt = (*yearIt).find(month);
}

void Account::dump() const
{
//  printf( "Account %i: %0.2f to %0.2f\n", m_id, initialBalance, getBalance());
}

void Account::transactionUpdate(int id, bool adding)
{
  Transaction &tr = m_doc->getTransaction(id);
  QDate date = tr.date();

  QMap<int,year_t>::iterator year;
  year_t::iterator month;
  //if ( adding )
    createMonth(date.year(), date.month(), year, month);
  Month& mnth = months[date.year()][date.month()];

  float addBalance = tr.value();
  // if it's from, we want to subtract, not add
  addBalance *= m_id == tr.to() ? 1 : m_id == tr.from() ? -1 : 0;
  // if we're removing, negate the value
  if ( !adding )
    addBalance *= -1;
  mnth.endBalance += addBalance;

  float budgetChange = addBalance;
  budgetChange *= m_type == AC_EXPENSE ? -1 : 1;
  mnth.endBudget += budgetChange;

  if ( adding )
    mnth.transactions.push_back(id);
  else {
    mnth.transactions.remove(id);
    // clean up the map
    // TODO: re-enable this and fix it (see ~/documents/bug.kbudget for procedure)
/*    if ( !mnth.transactions.size() ) {
      months[date.year()].remove(date.month());
      if ( months[date.year()].keys().size() == 0 )
        months.remove(date.year());
    }*/
  }

  month++;
  add(year, month, addBalance, budgetChange);

  m_doc->signalAcctChanged(m_id);
}

void Account::add(QMap<int,year_t>::iterator year, year_t::iterator month, float balance, float budget)
{
  while ( year != months.end() ) {
    while ( month != (*year).end() ) {
      (*month).startBalance += balance;
      (*month).endBalance += balance;
      (*month).startBudget += budget;
      (*month).endBudget += budget;
      month++;
    }
    year++;
    if ( year != months.end() )
      month = (*year).begin();
  }
}
