/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

// KDE includes
#include <klistview.h>
#include <kpushbutton.h>

// Qt includes
#include <qlabel.h>

// command includes
#include "selectlistitemsdialog.h"

bool SelectListItemsDialog::select(QWidget* parent, const QString& title, const QString& prompt, bool allowMulti, bool allowEmpty, const QStringList& items, QStringList& result)
{
  SelectListItemsDialog dialog(parent, title, prompt, allowMulti, allowEmpty, items);
  if (dialog.exec()) {
    result = dialog.selectedItems();
    return true;
  }
  return false;
}

SelectListItemsDialog::SelectListItemsDialog(QWidget* parent, const QString& title, const QString& prompt, bool allowMulti, bool allowEmpty, const QStringList& items)
  : SelectListItemsDialogBase(parent), m_allowMulti(allowMulti), m_allowEmpty(allowEmpty)
{
  setCaption(title);
  m_promptLabel->setText(prompt);
  m_listView->setSortColumn(-1);

  if (!m_allowEmpty)
    m_okButton->setEnabled(false);

  if (!m_allowMulti) {
    connect(m_listView, SIGNAL(selectionChanged()), this, SLOT(slotSelectionChanged()));
    connect(m_listView, SIGNAL(doubleClicked(QListViewItem*)), this, SLOT(accept()));

    m_selectAllButton->hide();
    m_invertSelectionButton->hide();
  } else {
    connect(m_listView, SIGNAL(clicked(QListViewItem*)), this, SLOT(slotSelectionChanged()));
  }

  for (QStringList::ConstIterator i = items.begin(); i != items.end(); ++i) {
    if (allowMulti) {
      new QCheckListItem(m_listView, m_listView->lastItem(), *i, QCheckListItem::CheckBox);
    } else {
      new KListViewItem(m_listView, m_listView->lastItem(), *i);
    }
  }
}

SelectListItemsDialog::~SelectListItemsDialog()
{
}

void SelectListItemsDialog::slotSelectionChanged()
{
  if (m_allowMulti) {
    m_okButton->setEnabled(m_allowEmpty || !selectedItems().isEmpty());
  } else {
    QListViewItem* item = m_listView->selectedItem();
    m_okButton->setEnabled(m_allowEmpty || item != 0);
  }
}

QStringList SelectListItemsDialog::selectedItems()
{
  QStringList result;

  if (m_allowMulti) {
    QListViewItemIterator it(m_listView);
    while (it.current()) {
      QCheckListItem* item = static_cast<QCheckListItem*>(it.current());
      if (item->isOn())
        result << item->text(0);
      ++it;
    }
  } else {
    QListViewItem* item = m_listView->selectedItem();
    if (item)
      result << item->text(0);
  }

  return result;
}

void SelectListItemsDialog::selectAll()
{
  QListViewItemIterator it(m_listView);
  while (it.current()) {
    QCheckListItem* item = static_cast<QCheckListItem*>(it.current());
    item->setOn(true);
    ++it;
  }
}

void SelectListItemsDialog::invertSelection()
{
  QListViewItemIterator it(m_listView);
  while (it.current()) {
    QCheckListItem* item = static_cast<QCheckListItem*>(it.current());
    item->setOn(!item->isOn());
    ++it;
  }
}

#include "selectlistitemsdialog.moc"
