/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

// KDE includes
#include <klocale.h>
#include <klistview.h>
#include <kpushbutton.h>
#include <kprocess.h>

// Qt includes
#include <qlineedit.h>
#include <qradiobutton.h>
#include <qgroupbox.h>
#include <qcombobox.h>
#include <qwidgetstack.h>

// WorKflow includes
#include "commanddescription.h"
#include "commandmanager.h"
#include "category.h"

// designer includes
#include "commanddialog.h"
#include "categorylistitem.h"
#include "dialoghelper.h"

using namespace WorKflow;

bool CommandDialog::editCommand(QWidget* parent, CommandDescription* desc)
{
  CommandDialog dialog(parent, desc);
  return dialog.exec();
}

CommandDescription* CommandDialog::newCommand(QWidget* parent, LibraryDescription* library)
{
  CommandDescription* desc = new CommandDescription(library);
  CommandDialog dialog(parent, desc);
  if (dialog.exec())
    return dialog.m_description;
  else
    delete desc;

  return 0;
}

// initialization
void CommandDialog::init()
{
  initGeneralTab();
  initKeywordsTab();
  initGuiTab();
}

void CommandDialog::initGeneralTab()
{
  m_idEdit->setText(m_description->id());
  m_nameEdit->setText(m_description->name());
  m_descriptionEdit->setText(m_description->description());
  m_iconEdit->setText(m_description->iconName());

  // init categories
  QStringList categories = m_description->categories();
  CommandManager::CategoryList allCategories = CommandManager::self()->categories();
  for (CommandManager::CategoryList::ConstIterator i = allCategories.begin(); i != allCategories.end(); ++i) {
    CategoryListItem* item = new CategoryListItem(m_categoriesListView, *i);
    if (categories.contains((*i)->id()))
      item->setOn(true);
  }
}

void CommandDialog::initKeywordsTab()
{
  QStringList keywords = m_description->keywords();

  for (QStringList::ConstIterator i = keywords.begin(); i != keywords.end(); ++i) {
    new QListViewItem(m_keywordsListView, *i);
  }
}

void CommandDialog::initGuiTab()
{
  QString uiFile = m_description->uiFile();
  if (!uiFile.isEmpty()) {
    m_uiFileRadio->setChecked(true);
    m_uiFileEdit->setText(uiFile);
  }
}

void CommandDialog::accept()
{
  m_description->setId(m_idEdit->text());
  m_description->setName(m_nameEdit->text());
  m_description->setDescription(m_descriptionEdit->text());
  m_description->setIconName(m_iconEdit->text());

  m_description->clearCategories();
  m_description->clearKeywords();

  QListViewItemIterator cat(m_categoriesListView);
  while (cat.current()) {
    CategoryListItem* item = static_cast<CategoryListItem*>(cat.current());
    if (item->isOn()) {
      m_description->addCategory(item->category()->id());
    }
    ++cat;
  }

  QListViewItemIterator key(m_keywordsListView);
  while (key.current()) {
    m_description->addKeyword(key.current()->text(0));
    ++key;
  }

  QDialog::accept();
}

CommandDialog::CommandDialog(QWidget* parent, CommandDescription* desc)
  : CommandDialogBase(parent), m_description(desc)
{
  init();
}

CommandDialog::~CommandDialog()
{
}

// Keyword tab
void CommandDialog::addKeyword()
{
  QListViewItem* item = new QListViewItem(m_keywordsListView, i18n("New Keyword"));
  m_keywordsListView->setSelected(item, true);
}

void CommandDialog::removeKeyword()
{
  DialogHelper::removeItem(m_keywordsListView);
}

void CommandDialog::slotKeywordSelectionChanged()
{
  QListViewItem* item = m_keywordsListView->selectedItem();
  m_removeKeywordButton->setEnabled(item != 0);
  m_editKeywordGroupBox->setEnabled(item != 0);

  if (item) {
    m_keywordEdit->setText(item->text(0));
    m_keywordEdit->selectAll();
    m_keywordEdit->setFocus();
  }
}

void CommandDialog::updateKeyword()
{
  QListViewItem* item = m_keywordsListView->selectedItem();
  if (item) {
    item->setText(0, m_keywordEdit->text());
  }
}

// GUI Tab
void CommandDialog::editUiFile()
{
  KProcess proc;
  proc << "designer";
  QString uiFile = m_uiFileEdit->text();
  if (!uiFile.isEmpty())
    proc << uiFile;

  proc.start();
  proc.detach();
}

#include "commanddialog.moc"
