/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

// KDE includes
#include <klocale.h>
#include <kdebug.h>

// Qt includes
#include <qhbox.h>
#include <qvbox.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qpopupmenu.h>
#include <qtooltip.h>

// WorKflow includes
#include "commandwidget.h"
#include "commandcounter.h"
#include "commandport.h"
#include "closewidget.h"
#include "collapsewidget.h"
#include "view.h"
#include "command.h"
#include "parameter.h"
#include "result.h"

using namespace WorKflow;

CommandWidget::CommandWidget(QWidget *parent, Command* command)
  : QWidget(parent), m_command(command), m_collapse(0), m_selected(false)
{
  kdDebug() << k_funcinfo << endl;

  QGridLayout* mainLayout = new QGridLayout(this, 3, 5, 0, -4);

  QColor col = paletteBackgroundColor();
  setPaletteBackgroundColor(Qt::lightGray);

  if (command->widget() || command->isContainer()) {
    m_mainWidget = new QFrame(this);
    m_mainWidget->setFrameShape(QFrame::StyledPanel);
    m_mainWidget->setFrameShadow(QFrame::Sunken);
    m_mainWidget->setPaletteBackgroundColor(col);

    QGridLayout* widgetLayout = new QGridLayout(m_mainWidget, 1, 1, 5, 0);

    int layoutRow = 0;
    if (command->widget()) {
      command->widget()->reparent(m_mainWidget, 0, QPoint(), false);
      widgetLayout->addWidget(command->widget(), layoutRow++, 0);
    }

    if (command->isContainer()) {
      kdDebug() << "command is container!" << endl;
      m_containerView = new View(m_mainWidget, m_command->childDocument());
      m_containerView->setMinimumHeight(500);
      widgetLayout->addWidget(m_containerView, layoutRow++, 0);
    }
    mainLayout->addMultiCellWidget(m_mainWidget, 1, 1, 0, 4);
  }

  m_counter = new CommandCounter(this);
  m_counter->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed, 0, 0));
  mainLayout->addWidget(m_counter, 0, 0);

  if (command->widget() || command->isContainer()) {
    m_collapse = new CollapseWidget(this);
    m_collapse->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed, 0, 0));
    mainLayout->addWidget(m_collapse, 0, 1);

    connect(m_collapse, SIGNAL(toggled(bool)), m_mainWidget, SLOT(setHidden(bool)));
    connect(m_collapse, SIGNAL(toggled(bool)), this, SIGNAL(minimumSizeChanged()));
  }

  m_title = new QLineEdit(this);
  m_title->setText(command->title());
  m_title->setSizePolicy(QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed, 0, 0));
  m_title->setPaletteBackgroundColor(Qt::lightGray);
  m_title->setFrame(false);
  m_title->setPaletteForegroundColor(Qt::white);

  QFont f = m_title->font();
  f.setBold(true);
  m_title->setFont(f);
  mainLayout->addWidget(m_title, 0, 2);

  m_inPort = new CommandPort(this, CommandPort::InputPort);
  m_inPort->setCommand(command);

  mainLayout->addWidget(m_inPort, 0, 3);

  m_close = new CloseWidget(this);
  m_close->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed, 0, 0));
  mainLayout->addWidget(m_close, 0, 4);

  m_outPort = new CommandPort(this, CommandPort::OutputPort);
  m_outPort->setCommand(command);
  mainLayout->addWidget(m_outPort, 2, 3);

  connect(command, SIGNAL(sizeChanged()), this, SIGNAL(minimumSizeChanged()));
  connect(command, SIGNAL(guiUpdateNeeded()), this, SLOT(updateGUI()));

  connect(m_counter, SIGNAL(popup(QPoint)), this, SLOT(showMenu(QPoint)));
  connect(m_close, SIGNAL(clicked()), this, SLOT(remove()));

  connect(m_title, SIGNAL(textChanged(const QString&)), command, SLOT(setTitle(const QString&)));
  connect(m_title, SIGNAL(textChanged(const QString&)), this, SLOT(updateToolTip()));
  connect(m_inPort, SIGNAL(selected(const QString&)), command, SLOT(setInput(const QString&)));
  connect(m_outPort, SIGNAL(selected(const QString&)), command, SLOT(setOutput(const QString&)));
}

CommandWidget::~CommandWidget()
{
  if (m_command->widget()) {
    m_command->widget()->reparent(0, QPoint(), false);
  }
}

void CommandWidget::slotPortChanged()
{
  emit minimumSizeChanged();
}

void CommandWidget::setOrder(int row)
{
  m_counter->setOrder(row);
}

int CommandWidget::row()
{
  return m_counter->row();
}

void CommandWidget::setSelected(bool selected)
{
  m_selected = selected;

  m_counter->setSelected(selected);
  m_close->setSelected(selected);

  emit selectionChanged(this);
}

bool CommandWidget::isSelected()
{
  return m_selected;
}

void CommandWidget::showMenu(QPoint pos)
{
  QPopupMenu menu(this);

  if (m_collapse) {
    if (m_collapse->isCollapsed())
      menu.insertItem(i18n("Uncollapse"), this, SLOT(toggleCollapse()));
    else
      menu.insertItem(i18n("Collapse"), this, SLOT(toggleCollapse()));
    menu.insertSeparator();
  }

  menu.insertItem(i18n("Move up"), this, SLOT(moveUp()));
  menu.insertItem(i18n("Move down"), this, SLOT(moveDown()));
  menu.insertSeparator();
  menu.insertItem(i18n("Rename"), this, SLOT(rename()));
  menu.insertItem(i18n("Delete"), this, SLOT(remove()));
  menu.exec(pos);
}

void CommandWidget::rename()
{
  m_title->selectAll();
  m_title->setFocus();
}

void CommandWidget::remove()
{
  emit closed(m_command);
}

void CommandWidget::moveUp()
{
}

void CommandWidget::moveDown()
{
}

void CommandWidget::toggleCollapse()
{
  if (m_collapse)
    m_collapse->setCollapsed(!m_collapse->isCollapsed());
}

void CommandWidget::updateGUI()
{
  // TODO: during type-checking, this is called way too often

  m_title->setText(m_command->title());
  updateToolTip();

  if (m_command->input()) {
    m_inPort->updateText(m_command->input());
    m_inPort->setProblem(m_command->input()->problem());
  }
  if (m_command->output()) {
    m_outPort->updateText(m_command->output());
    m_outPort->setProblem(m_command->output()->problem());
  }
}

void CommandWidget::collapse()
{
  if (m_collapse)
    m_collapse->setCollapsed(true);
}

void CommandWidget::uncollapse()
{
  if (m_collapse)
    m_collapse->setCollapsed(false);
}

void CommandWidget::updateToolTip()
{
  // set title tooltip if title was changed
  if (m_command->title() == m_command->originalTitle())
    QToolTip::remove(m_title);
  else
    QToolTip::add(m_title, i18n("Original title: <b>%1</b>").arg(m_command->originalTitle()));
}

#include "commandwidget.moc"
