/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

// Qt includes
#include <qpainter.h>
#include <qlayout.h>
#include <qdragobject.h>
#include <qtimer.h>
#include <qclipboard.h>
#include <qlabel.h>

// KDE includes
#include <kapplication.h>
#include <kurl.h>
#include <kdebug.h>
#include <klocale.h>

// WorKflow includes
#include "view.h"
#include "view_p.h"
#include "document.h"
#include "command.h"
#include "commandwidget.h"
#include "commandmanager.h"
#include "commandfactory.h"
#include "commanddescription.h"
#include "commanddrag.h"
#include "progressdialog.h"

// STL includes
#include <algorithm>
  using std::min;
  using std::max;
  using std::swap;

using namespace WorKflow;

const int xMargin = 16;
const int ySpacing = 20;

// View::Cursor implementation
View::Cursor::Cursor(View::Private* view)
  : m_insertPosition(Top), m_view(view)
{

}

View::Cursor::Cursor(View::Private* view, const CommandCell& cell, InsertionPosition pos)
  : m_view(view), m_insertCell(cell), m_insertPosition(Top)
{

}

void View::Cursor::normalize()
{
  if (m_insertPosition == Bottom) {
    m_insertPosition = Top;
    m_view->findCommandCell(row() + 1, &m_insertCell);
  }
}

bool View::Cursor::isPriorTo(int row)
{
  return m_insertPosition == Top &&
           m_insertCell.row() <= row;
}

bool View::Cursor::isPriorTo(const View::Cursor& cursor)
{
  return isPriorTo(cursor.row());
}

bool View::Cursor::isEqualTo(int row)
{
  return m_insertPosition == Top &&
           row == m_insertCell.row();
}

bool View::Cursor::isEqualTo(const View::Cursor& cursor)
{
  return isEqualTo(cursor.row());
}

void View::Cursor::setPosition(QPoint pos)
{
  // special case: empty document
  if (m_view->cellAreas.count() == 1) {
    m_insertCell = m_view->cellAreas.first();
    m_insertPosition = Top;
    return;
  }

  CommandCell cell;
  if (m_view->findCommandCell(pos, &cell)) {
    // here we know that pos is in the rect

    QRect rect = cell.rect();
    int halfHeight = rect.height() / 2;
    if (pos.y() < rect.top() + halfHeight) {
      // top
      m_insertPosition = Top;
    } else {
      // bottom
      m_insertPosition = Bottom;
    }
    m_insertCell = CommandCell(cell.rect(), cell.row());
  } else {
    // append at the end
    m_insertCell = m_view->cellAreas.last();
    m_insertPosition = Top;
  }

  normalize();
  // Here, the cursor position in the rect is top.
}

void View::Cursor::move(int deltarow)
{
  int destRow = row() + deltarow;

  CommandCell cell;

  // clamp to [0, numRows]
  destRow = min(destRow, m_view->document->numRows());
  destRow = max(destRow, 0);

  if (m_view->findCommandCell(destRow, &cell)) {
    m_insertCell = cell;
  } else {
    // we're in trouble here.
    kdDebug() << k_funcinfo << "attempted to move the cursor to nonexistent cell: (" << destRow << ")." << endl;
  }
}

void View::Cursor::moveTo(int row)
{
  if (!m_view->findCommandCell(row, &m_insertCell)) {
    goToStart();
  }
}

void View::Cursor::goToStart()
{
  m_view->findCommandCell(0, &m_insertCell);
  m_insertPosition = Top;
}

void View::Cursor::goToEnd()
{
  m_view->findCommandCell(m_view->document->numRows(), &m_insertCell);
  m_insertPosition = Top;
}

// View::Selection implementation
View::Selection::Selection(const Cursor& start, const Cursor& end)
  : m_start(start), m_end(end)
{
  if (m_end.isPriorTo(m_start)) {
    swap(m_start, m_end);
  }

  m_startRow = m_start.row();
  m_endRow = m_end.row();
}

// View::Private implementation
CommandWidget* View::Private::commandWidget(Command* cmd)
{
  if (commandMap.contains(cmd))
    return commandMap[cmd];
  return 0;
}

CommandWidget* View::Private::commandWidget(int row)
{
  Command* cmd = document->commandAt(row);
  return commandWidget(cmd);
}

bool View::Private::findCommandCell(QPoint position, CommandCell* cell)
{
  if (!cell)
    return false;

  for (AreaList::ConstIterator i = cellAreas.begin(); i != cellAreas.end(); ++i) {
    if ((*i).rect().contains(position)) {
      *cell = *i;
      return true;
    }
  }

  return false;
}

bool View::Private::findCommandCell(int row, CommandCell* cell)
{
  if (!cell)
    return false;

  for (AreaList::ConstIterator i = cellAreas.begin(); i != cellAreas.end(); ++i) {
    if ((*i).row() == row) {
      *cell = *i;
      return true;
    }
  }

  return false;
}

void View::Private::startCursorTimer()
{
  // TODO: KDE's cursor blinking speed
  cursorTimer.start(500);
  showCursor = true;
}

void View::Private::stopCursorTimer()
{
  cursorTimer.stop();
  showCursor = false;
}

View::Selection View::Private::selection()
{
  return View::Selection(startCursor, cursor);
}

void View::Private::startMove(bool select)
{
  if (select && !showSelection) {
    startCursor = cursor;
  } else if (!select) {
    p->clearSelection();
  }
}

void View::Private::endMove(bool select)
{
  if (select) {
    p->calculateSelection();
  }

  startCursorTimer();

  const QRect& cell = cursor.rect();
  p->ensureVisible(cell.left() + xMargin / 2, cell.top() + ySpacing / 2);

  p->updateContents();
}

void View::Private::slotOnURL(const QString& url)
{
  emit p->signalChangeStatusbar(url);
}

void View::Private::slotSetTitle(const QString& title)
{
  emit p->signalChangeCaption(title);
}

void View::Private::slotCommandClosed(Command* cmd)
{
  // The QTimer mechanism is used to avoid internal Qt crashes,
  // because you can't delete a signal sender from the called slot :)
  commandToDelete = cmd;
  QTimer::singleShot(0, this, SLOT(deleteCommand()));
}

void View::Private::deleteCommand()
{
  document->removeCommand(commandToDelete->row());
}

void View::Private::slotCommandRemoved(Command* cmd)
{
  CommandWidget* cmdWidget = commandWidget(cmd);
  commandMap.remove(cmd);

  // prevent deletion of widget
  if (cmd->widget())
    cmd->widget()->reparent(0, QPoint());

  delete cmdWidget;

  p->calculateLayout();
}

void View::Private::slotCommandSizeChanged()
{
  kdDebug() << k_funcinfo << endl;
  // schedule layout recalculation
  QTimer::singleShot(0, p, SLOT(calculateLayout()));
}

void View::Private::slotCursorTimer()
{
  showCursor = !showCursor;
  p->updateContents();
}

void View::Private::slotSelectionChanged(CommandWidget* widget)
{
  
}

// View implementation
View::View(QWidget *parent)
  : QScrollView(parent)
//       DCOPObject("WorKflowIface")
{
  init();
  d->document = new Document(this);
  d->ownsDocument = true;
  initDocument();
  initDragHint();
}

View::View(QWidget* parent, Document* doc)
  : QScrollView(parent)
{
  init();
  d->document = doc;
  d->ownsDocument = false;
  initDocument();

  for (int i = 0; i != doc->numRows(); ++i) {
    newCommand(doc->commandAt(i));
  }
}

View::~View()
{
  if (d->ownsDocument)
    delete d->document;
  delete d;
}

void View::init()
{
  d = new Private(this);
  d->showCursor = false;
  d->showSelection = false;
  d->dragHint = 0;

  viewport()->setAcceptDrops(true);
  viewport()->setPaletteBackgroundColor(Qt::white);
}

void View::initDocument()
{
  connect(d->document, SIGNAL(orderChanged()), this, SLOT(calculateLayout()));
  connect(d->document, SIGNAL(commandAdded(Command*)), this, SLOT(newCommand(Command*)));
  connect(d->document, SIGNAL(commandRemoved(Command*)), d, SLOT(slotCommandRemoved(Command*)));
  connect(&d->cursorTimer, SIGNAL(timeout()), d, SLOT(slotCursorTimer()));
}

void View::initDragHint()
{
  QGridLayout* layout = new QGridLayout(viewport(), 0, 0);
  d->dragHint = new QLabel(i18n("Drag commands here"), viewport());
  layout->addItem(new QSpacerItem(0, 0, QSizePolicy::Fixed, QSizePolicy::Expanding), 0, 1);
  layout->addItem(new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Fixed), 1, 0);
  layout->addWidget(d->dragHint, 1, 1);
  layout->addItem(new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Fixed), 1, 2);
  layout->addItem(new QSpacerItem(0, 0, QSizePolicy::Fixed, QSizePolicy::Expanding), 2, 1);
}

QString View::currentURL()
{
}

void View::openURL(QString url)
{
    openURL(KURL(url));
}

void View::openURL(const KURL& url)
{
}

void View::calculateLayout()
{
  d->cellAreas.clear();

  int rows = d->document->numRows();
  int y = ySpacing;
  int minCellWidth = visibleWidth() - 2 * xMargin;
  int maxRowWidth = 0;

//   QWidget* tabWidget = this;

  // calculate layout
  for (int i = 0; i != rows; i++) {
    Command* cmd = d->document->commandAt(i);
    cmd->setRow(i);

    CommandWidget* widget = d->commandWidget(cmd);
    moveChild(widget, xMargin, y);
//     setTabOrder(tabWidget, widget);
//     tabWidget = widget;

    QSize minimumSize = widget->minimumSizeHint();
    int cellHeight = minimumSize.height();
    int cellWidth = max(minCellWidth, minimumSize.width());
    maxRowWidth = max(maxRowWidth, cellWidth);

    widget->resize(cellWidth, cellHeight);

    d->cellAreas.append(CommandCell(QRect(0,
                                          y - ySpacing,
                                          cellWidth + 2 * xMargin,
                                          cellHeight + 2 * ySpacing), i));

    y += cellHeight + ySpacing;
  }

  // Append an extra row to ease cursor handling
  d->cellAreas.append(CommandCell(QRect(xMargin,
                                        y - ySpacing,
                                        visibleWidth() - 2 * xMargin,
                                        ySpacing), rows));

  resizeContents(maxRowWidth, y);

  CommandCell cell;
  if (!d->findCommandCell(d->cursor.row(), &cell)) {
    d->cursor.goToStart();
  } else {
    d->cursor.moveTo(d->cursor.row());
  }

  updateContents();
}

void View::newCommand(Command* cmd)
{
  if (d->dragHint) {
    delete d->dragHint;
    d->dragHint = 0;
    enableClipper(true);

    // reinit viewport
    setFocusPolicy(StrongFocus);
    viewport()->setFocusProxy(this);

    viewport()->setAcceptDrops(true);
  }

  CommandWidget* w = new CommandWidget(viewport(), cmd);
  addChild(w);
  w->show();
  d->commandMap[cmd] = w;

  connect(cmd, SIGNAL(orderChanged(int)), w, SLOT(setOrder(int)));
  connect(w, SIGNAL(closed(Command*)), d, SLOT(slotCommandClosed(Command*)));
  connect(w, SIGNAL(minimumSizeChanged()), d, SLOT(slotCommandSizeChanged()));

  QTimer::singleShot(0, this, SLOT(calculateLayout()));
}

void View::appendCommand(const QString& commandId)
{
  if (hasSelection()) {
    removeSelection();
  }

  CommandFactoryBase* factory = CommandManager::self()->commandDescription(commandId)->factory();

  int row = d->cursor.row();

  d->document->insertCommand(row, factory->create(d->document));
  d->cursor.moveTo(d->cursor.row() + 1);
}

void View::viewportResizeEvent(QResizeEvent*)
{
  calculateLayout();
}

void View::setCursor(QPoint position)
{
  d->showCursor = true;
  d->cursor.setPosition(position);
}

void View::contentsDragEnterEvent(QDragEnterEvent* e)
{
  d->acceptingCommands.clear();
  if (e->provides("application/x-workflowcommand")) {
    e->accept(true);
    return;
  } else {
    int n = 0;
    const char* fmt = e->format(n);
    while (fmt) {
      d->acceptingCommands = CommandManager::self()->queryForDropType(QString::fromLatin1(fmt));
      if (!d->acceptingCommands.isEmpty()) {
        e->accept(true);
        return;
      }
      fmt = e->format(++n);
    }
  }

  e->accept(false);
}

void View::contentsDragMoveEvent(QDragMoveEvent* e)
{
  if (!e->provides("application/x-workflowcommand") && d->acceptingCommands.isEmpty()) {
    e->accept(false);
    return;
  }

  setCursor(e->pos());
  updateContents();
  e->accept(true);
}

void View::contentsDragLeaveEvent(QDragLeaveEvent*)
{
  d->showCursor = false;
  updateContents();
}

void View::contentsDropEvent(QDropEvent* e)
{
  CommandFactoryBase* factory = 0;

  if (e->provides("application/x-workflowcommand")) {
    CommandDrag::Info dragInfo;

    if (CommandDrag::decode(e, dragInfo)) {
      if (dragInfo.kind() == CommandDrag::Info::NewCommand) {
        factory = CommandManager::self()->commandDescription(dragInfo.commandId())->factory();
      } else {
  //       Command* cmd = d->document->commandAt(row, column);
  //       d->document->moveCommand(dragInfo.row(), dragInfo.column(), row, column, betweenRows);
      }
    }
  } else if (!d->acceptingCommands.isEmpty()) {
    // TODO: give user choice
    factory = d->acceptingCommands.first()->factory();
  } else {
    e->accept(false);
    return;
  }

  int row = d->cursor.row();

  if (factory) {
    Command* cmd = factory->create(d->document);
    d->document->insertCommand(row, cmd);

    if (!d->acceptingCommands.isEmpty()) {
      if (!cmd->drop(e))
        kdWarning() << "command " << cmd->title() << " claimed to accept drop, but didn't" << endl;
    }
  }

  d->acceptingCommands.clear();

  e->accept(true);
  d->showCursor = false;
  updateContents();
}

void View::contentsMousePressEvent(QMouseEvent* e)
{
  if (e->button() == Qt::LeftButton) {
    if (!hasSelection() && e->state() & Qt::ShiftButton) {
      d->startCursor = d->cursor;
      calculateSelection();
    }

    setCursor(e->pos());
//     d->startCursorTimer();

    if (!(e->state() & Qt::ShiftButton)) {
      d->startCursor = d->cursor;
      clearSelection();
    }

    updateContents();
  }
}

void View::contentsMouseMoveEvent(QMouseEvent* e)
{
  if (e->state() & Qt::LeftButton) {
    setCursor(e->pos());
    calculateSelection();

    updateContents();
  }
}

void View::contentsMouseReleaseEvent(QMouseEvent* e)
{
  if (e->button() == Qt::LeftButton) {
    calculateSelection();
    updateContents();
  }
}

void View::drawContents(QPainter* p, int clipx, int clipy, int clipw, int cliph)
{
  if (d->dragHint)
    return;

  if (hasSelection()) {
    p->setPen(Qt::blue);
    p->setBrush(Qt::blue);

    for (AreaList::ConstIterator i = d->cellAreas.begin(); i != d->cellAreas.end(); ++i) {
      CommandWidget* w = d->commandWidget((*i).row());
      if (w && w->isSelected()) {
        p->drawRect((*i).rect());
      }
    }
  }

  if (d->showCursor) {
    int x, y, w, h;
    QRect rect = d->cursor.rect();

    if (d->cursor.insertPosition() == Cursor::Top) {
      x = contentsX() + xMargin;
      y = rect.top() + ySpacing / 2;
      w = visibleWidth() - 2 * xMargin;
      h = 2;
    } else {
      kdDebug() << k_funcinfo << " unnormalized cursor!" << endl;
    }

    p->setPen(Qt::black);
    p->drawRect(x, y, w, h);
  }

  // Debug paintings
/*  p->setBrush(Qt::NoBrush);
  p->setPen(Qt::red);
  for (AreaList::ConstIterator i = d->cellAreas.begin(); i != d->cellAreas.end(); ++i) {
    p->drawRect((*i).rect());
  }

  p->setPen(Qt::green);
  p->drawRect(d->cursor.rect());*/
}

void View::keyPressEvent(QKeyEvent* e)
{
  switch (e->key()) {
    case Key_Up:
      moveCursor(-1, e->state() & Qt::ShiftButton);
      break;
    case Key_Down:
      moveCursor(1, e->state() & Qt::ShiftButton);
      break;
    case Key_Home:
      goToStart(e->state() & Qt::ShiftButton);
      break;
    case Key_End:
      goToEnd(e->state() & Qt::ShiftButton);
      break;
    case Key_Delete:
      deletePressed();
      break;
    case Key_Backspace:
      backspacePressed();
      break;
    default:
      e->ignore();
      return;
  }

  e->accept();
}

void View::focusInEvent(QFocusEvent*)
{
  d->startCursorTimer();
  updateContents();
}

void View::focusOutEvent(QFocusEvent*)
{
  d->stopCursorTimer();
  updateContents();
}

void View::moveCursor(int deltarow, bool select)
{
  d->startMove(select);

  d->cursor.move(deltarow);

  d->endMove(select);
}

void View::goToStart(bool select)
{
  d->startMove(select);

  d->cursor.goToStart();

  d->endMove(select);
}

void View::goToEnd(bool select)
{
  d->startMove(select);

  d->cursor.goToEnd();

  d->endMove(select);
}

void View::clearSelection()
{
  for (CommandMap::Iterator i = d->commandMap.begin(); i != d->commandMap.end(); ++i) {
    (*i)->setSelected(false);
  }

  d->showSelection = false;
  emit selectionChanged();
  updateContents();
}

void View::selectAll()
{
  for (CommandMap::Iterator i = d->commandMap.begin(); i != d->commandMap.end(); ++i) {
    (*i)->setSelected(true);
  }

  d->startCursor = Cursor(d, d->cellAreas.first(), Cursor::Top);
  d->cursor = Cursor(d, d->cellAreas.last(), Cursor::Top);
  d->showSelection = true;
  emit selectionChanged();
  updateContents();
}

void View::calculateSelection()
{
  bool inside = false;
  View::Cursor start = d->startCursor;
  View::Cursor end = d->cursor;

  if (end.isPriorTo(start)) {
    swap(start, end);
  }

  for (AreaList::ConstIterator i = d->cellAreas.begin(); i != d->cellAreas.end(); ++i) {
    if (!inside && start.isEqualTo((*i).row())) {
      inside = true;
    }

    if (inside && end.isEqualTo((*i).row())) {
      inside = false;
    }

    CommandWidget* w = d->commandWidget((*i).row());

    if (w) {
      w->setSelected(inside);
    }
  }

  if (start.isEqualTo(end)) {
    d->showSelection = false;
  } else {
    d->showSelection = true;
  }
  emit selectionChanged();
}

void View::deletePressed()
{
  if (hasSelection()) {
    removeSelection();
  } else {
    if (d->cursor.insertPosition() == Cursor::Top)
      return;

    int row = d->cursor.row();

    Command* cmd = d->document->commandAt(row);
    if (cmd) {
      d->document->removeCommand(row);
    }
  }
}

void View::backspacePressed()
{
  if (hasSelection()) {
    removeSelection();
  } else {
    if (d->cursor.insertPosition() == Cursor::Top)
      return;

    int row = d->cursor.row();
  }
}

void View::removeSelection()
{
  Selection sel = d->selection();

  d->cursor = sel.start();
  d->showSelection = false;
  emit selectionChanged();
  updateContents();

  d->document->removeRange(sel.startRow(), sel.endRow());
}

bool View::hasSelection()
{
  return d->showSelection;
}

void View::copy()
{
  if (hasSelection()) {
    Selection sel = d->selection();
    d->document->copy(sel.startRow(), sel.endRow());
  }
}

void View::cut()
{
  if (hasSelection()) {
    Selection sel = d->selection();
    d->document->cut(sel.startRow(), sel.endRow());
    clearSelection();
  }
}

void View::paste()
{
  if (!KApplication::clipboard()->data(QClipboard::Clipboard))
    return;

  if (hasSelection()) {
    removeSelection();
  }

  d->document->paste(d->cursor.row());
}

Document* View::document()
{
  return d->document;
}

void View::collapseAll()
{
  for (Document::Iterator i = document()->begin(); i != document()->end(); ++i) {
    CommandWidget* w = d->commandWidget(*i);
    w->collapse();
  }
}

void View::uncollapseAll()
{
  for (Document::Iterator i = document()->begin(); i != document()->end(); ++i) {
    CommandWidget* w = d->commandWidget(*i);
    w->uncollapse();
  }
}

void View::execute()
{
  ProgressDialog dialog(this);

  connect(d->document, SIGNAL(progress(double, double, const QString&, const QString&)),
          &dialog, SLOT(showProgress(double, double, const QString&, const QString&)));
  connect(&dialog, SIGNAL(cancelled()), d->document, SLOT(abort()));

  dialog.show();

  d->document->execute();
}

#include "view.moc"
#include "view_p.moc"
