/*
    This file is part of KOrganizer.
    Copyright (c) 2002, Klaralvdalens Datakonsult AB <info@klaralvdalens-datakonsult.se>

    Author: Steffen Hansen <steffen@klaralvdalens-datakonsult.se>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <qptrlist.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qpushbutton.h>
#include <qcombobox.h>
#include <qcheckbox.h>

#include <klocale.h>
#include <kdebug.h>

#ifndef KORG_NOKABC
#include <kabc/addresseedialog.h>
#include <kabc/stdaddressbook.h>
#endif

#include <libkcal/incidence.h>

#include "kolineedit.h"

#include "kotodoeditordetails.h"

KOTodoEditorDetails::KOTodoEditorDetails( int spacing, QWidget* parent, const char* name )
  : KOTodoEditorDetails_base( parent, name ), mAssignee(0)
{
  KOTodoEditorDetails_baseLayout->setSpacing( spacing );
  assigneeBoxLayout->setSpacing( spacing );

  nameLE->setEmailLineEdit( emailLE );

  statusCombo->insertStringList(KCal::Attendee::statusList());
  connect(statusCombo,SIGNAL(activated(int)),SLOT(updateAttendeeItem()));

  connect( nameLE, SIGNAL( textChanged(const QString&) ), 
	   this, SLOT( updateAttendeeItem() ) );
  connect( emailLE, SIGNAL( textChanged(const QString&) ), 
	   this, SLOT( updateAttendeeItem() ) );

  connect( adrbkPB, SIGNAL( clicked() ),
	   this, SLOT( openAddressBook() ) );

  connect( rsvpCB, SIGNAL( clicked() ),
	   this, SLOT( updateAttendeeItem() ) );
}

void KOTodoEditorDetails::updateAttendeeItem()
{
  QString name = nameLE->text();
  QString email = emailLE->text();

  if( mAssignee == 0 ) delete mAssignee;
  if( !email.isEmpty() ) mAssignee = new KCal::Attendee( name, email, rsvpCB->isChecked(),
							 KCal::Attendee::PartStat( statusCombo->currentItem()));
}

void KOTodoEditorDetails::openAddressBook()
{
#ifndef KORG_NOKABC
  // Refresh the addressbook in case someone added new entries
  KABC::StdAddressBook::self()->load();

  // ... and run the dialog
  KABC::Addressee a = KABC::AddresseeDialog::getAddressee(this);
  if (!a.isEmpty()) {
    nameLE->setText( a.realName() );
    emailLE->setText( a.preferredEmail() );
    if( mAssignee ) delete mAssignee;
    mAssignee = new KCal::Attendee( a.realName(), a.preferredEmail(),true,
				    KCal::Attendee::NeedsAction,KCal::Attendee::ReqParticipant,a.uid());
  }
#endif
}


void KOTodoEditorDetails::readEvent(KCal::Incidence *incidence, bool)
{
  organizerLabel->setText(i18n("Organizer: %1").arg(incidence->organizer()));
  QPtrList<KCal::Attendee> tmpAList = incidence->attendees();
  KCal::Attendee *a = tmpAList.first();
  if( a && a->role() == KCal::Attendee::ReqParticipant ) {
    if( mAssignee ) delete mAssignee;
    mAssignee = new KCal::Attendee(*a);
    nameLE->setText( a->name() );
    emailLE->setText( a->email() );
    rsvpCB->setChecked( a->RSVP() );
    statusCombo->setCurrentText( a->statusStr() );
  }
}


void KOTodoEditorDetails::writeEvent(KCal::Incidence *incidence)
{
  incidence->clearAttendees();
  if( mAssignee ) {
    incidence->addAttendee(new KCal::Attendee(*mAssignee));
  }
}

#if 0
void KOTodoEditorDetails::cancelAttendeeEvent(KCal::Incidence *incidence)
{
  incidence->clearAttendees();
  Attendee * att;
  for (att=mdelAttendees.first();att;att=mdelAttendees.next()) {
    incidence->addAttendee(new Attendee(*att));
  }
  mdelAttendees.clear();
}
#endif

bool KOTodoEditorDetails::validateInput()
{
  return true;
}
