/*
 *   cddbd - CD Database Protocol Server
 *
 *   Copyright (C) 1996  Steve Scherf
 *   Email: steve@moonsoft.com
 *   Moondog Software Productions - makers of fine public domain software.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifndef LINT
static char *_inet_c_ident_ = "@(#)$Id: inet.c,v 1.5 1996/12/22 01:48:54 steve Exp $";
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <ctype.h>
#include <errno.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <dirent.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <time.h>
#include <sys/time.h>
#include <stdio.h>
#include "patchlevel.h"
#include "access.h"
#include "list.h"
#include "cddbd.h"

/* Structure definitions. */

/* Values for si_flags. */
#define SF_NOADDR	0x000000001
#define SF_NOHOST	0x000000002

/* Values for si_match. */
#define SI_MATCH_NONE		0
#define SI_MATCH_DEFAULT	1
#define SI_MATCH_DOMAIN		2
#define SI_MATCH_NET		3
#define SI_MATCH_HOST		4

/* Connected host socket info. */
typedef struct sockinfo {
	int si_flags;
	int si_match;
	int si_domain_len;
	char si_hname[CDDBHOSTNAMELEN + 1];
	struct in_addr si_haddr;
} sinfo_t;


/* Prototypes. */

int comp_domain(char *, char *);
int comp_ip(char *, char *);
int domain_len(char *);
int isipaddr(char *);
int sock_getc(void);
int sock_open(char *, int, int, char *);
int sock_read(char *, int, int);
int sock_serv_read(char *, int, int, char *);
int sock_wait(void);
int sock_write(char *, int);

void cddbd_bcopy(char *, char *, int);
void chldhand(int);
void sock_close(void);


/* Global variables. */

int sock_i;
int sock_e;
int sock_fd = -1;

FILE *sock_fp;

sinfo_t sinfo;

char cddbphost[CDDBHOSTNAMELEN + 1];

/* Default services table. */
cservent_t servtab[] = {
	"cddbp", "tcp", 888,
	"smtp",  "tcp", 25,
	{ 0 }
};


void
get_rmt_hostname(int fd, char *rh)
{
	char *h;
	int len;
	struct hostent *hp;
	struct sockaddr_in name;

	len = sizeof(name);

	/* Get the peer name. If this call fails, the fd is a tty or file. */
	if(getpeername(fd, (struct sockaddr *)&name, &len) < 0) {
		sinfo.si_flags |= SF_NOADDR;
		hp = gethostbyname(host);
	}
	else {
		sinfo.si_haddr = name.sin_addr;
		hp = gethostbyaddr((char *)&name.sin_addr,
		    sizeof(name.sin_addr), AF_INET);
	}

	if(hp == 0) {
		sinfo.si_flags |= SF_NOHOST;
		if(sinfo.si_flags & SF_NOADDR)
			h = "localhost";
		else
			h = inet_ntoa(name.sin_addr);
	}
	else
		h = (char *)hp->h_name;

	strncpy(sinfo.si_hname, h, CDDBHOSTNAMELEN);
	sinfo.si_hname[CDDBHOSTNAMELEN] = '\0';
	strncpy(rh, h, CDDBHOSTNAMELEN);
	rh[CDDBHOSTNAMELEN] = '\0';
}


int
match_host(char *ahost)
{
	char *p;
	char **ap;
	char net[CDDBBUFSIZ];
	unsigned long in;
	struct netent *np;
	struct hostent *hp;
	struct in_addr neta;

	/* Try and match the default definition. */
	if(!strcmp(ahost, "default")) {
		if(sinfo.si_match <= SI_MATCH_DEFAULT) {
			sinfo.si_match = SI_MATCH_DEFAULT;
			return 1;
		}

		return 0;
	}

	/* Try and match the domain. */
	p = ahost + 2;

	/* Check even if we already have a domain match. */
	if(sinfo.si_match <= SI_MATCH_DOMAIN && !strncmp("*.", ahost, 2) &&
	    (sinfo.si_domain_len < domain_len(p)) && !(sinfo.si_match &
	    SF_NOHOST) && !isipaddr(sinfo.si_hname)) {
		hp = gethostbyname(sinfo.si_hname);

		if(hp == 0) {
			/* Compare the primary host name with the domain. */
			if(!comp_domain((char *)sinfo.si_hname, p)) {
				sinfo.si_domain_len = domain_len(p);
				sinfo.si_match = SI_MATCH_DOMAIN;
				return 1;
			}
		}
		else {
			/* Compare the primary host name with the domain. */
			if(!comp_domain((char *)hp->h_name, p)) {
				sinfo.si_domain_len = domain_len(p);
				sinfo.si_match = SI_MATCH_DOMAIN;
				return 1;
			}

			/* Compare the secondary host names with the domain. */
			for(ap = hp->h_aliases; *ap; ap++)
				if(!comp_domain(*ap, p)) {
					sinfo.si_domain_len = domain_len(p);
					sinfo.si_match = SI_MATCH_DOMAIN;
					return 1;
				}
		}
	}

	/* If we know this is a domain search, return. */
	if(!strncmp("*.", ahost, 2))
		return 0;

	/* Try and match the host. */
	if(sinfo.si_match < SI_MATCH_HOST && !(sinfo.si_flags & SF_NOHOST)) {
		if(isipaddr(ahost)) {
			/* Try and match the host (IP format). */
			if(!(sinfo.si_flags & SF_NOADDR) &&
			    !strcmp(inet_ntoa(sinfo.si_haddr), ahost)) {
				sinfo.si_match = SI_MATCH_HOST;
				return 1;
			}

			in = inet_addr(ahost);
			hp = gethostbyaddr((char *)&in, sizeof(in),
			    AF_INET);
		}
		else
			hp = gethostbyname(ahost);

		if(hp != 0) {
			/* Check the primary host name. */
			if(!strcmp(sinfo.si_hname, hp->h_name)) {
				sinfo.si_match = SI_MATCH_HOST;
				return 1;
			}

			/* Check the secondary host names. */
			for(ap = hp->h_aliases; *ap; ap++)
				if(!strcmp(sinfo.si_hname, *ap)) {
					sinfo.si_match = SI_MATCH_HOST;
					return 1;
				}
		}
	}

	/* Try to match the network. */
	if(sinfo.si_match < SI_MATCH_NET && !(sinfo.si_flags & SF_NOADDR)) {
		/* Try and match the network (IP format). */
		if(!comp_ip(inet_ntoa(sinfo.si_haddr), ahost)) {
			sinfo.si_match = SI_MATCH_NET;
			return 1;
		}

		/* Try to match the net name. */
		np = getnetbyname(ahost);

		if(np != 0) {
			neta.s_addr = htonl(np->n_net);
			strcpy(net, inet_ntoa(neta));

			if(!comp_ip(inet_ntoa(sinfo.si_haddr), net)) {
				sinfo.si_match = SI_MATCH_NET;
				return 1;
			}
		}
	}

	return 0;
}


int
comp_ip(char *ip, char *domain)
{
	int len;
	char *p;
	char tdom[CDDBBUFSIZ];
	
	if(!isipaddr(ip) || !isipaddr(domain))
		return 1;

	len = strlen(domain);
	if(len >= sizeof(tdom))
		return 1;

	strcpy(tdom, domain);

	p = &tdom[len];
	do {
		p--;

		while(p != tdom && *p != '.') {
			if(*p != '0')
				return(strncmp(ip, tdom, strlen(tdom)));
			p--;
		}

		*p = '\0';
	} while(p != tdom);

	return 1;
}


int
comp_domain(char *host, char *domain)
{
	int len;

	len = strlen(host) - strlen(domain);
	if(len <= 0)
		return 1;

	host = &host[len];
	if(*(host - 1) != '.')
		return 1;

	return(strcmp(host, domain));
}


int
domain_len(char *domain)
{
	int len;

	while(*domain == '.')
		domain++;

	if(*domain == '\0')
		return 0;

	for(len = 1; *domain != '\0'; domain++)
		if(*domain == '.')
			len++;

	return len;
}


int
isipaddr(char *ip)
{
	int i;

	for(i = 0; i < 4 && *ip != '\0'; i++) {
		while(*ip != '\0' && isdigit(*ip))
			ip++;

		if(*ip != '\0') {
			if(*ip == '.' && i < 3)
				ip++;
			else
				break;
		}
	}

	if(i == 4 && *ip == '\0')
		return 1;

	return 0;
}


void
cddbd_stand(int port)
{
	int f;
	int s;
	int sock;
	int opt;
	int len;
	struct sockaddr_in sin;
	struct sockaddr_in from;

	strcpy(rhost, "none");

	sin.sin_port = get_serv_port(port, SERV_CDDBP);

	if((s = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		perror("socket");
		exit(QUIT_ERR);
	}

	opt = 1;
	if(setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (char *)&opt, sizeof(opt)))
		fprintf(stderr,
		    "Warning: setsockopt(SO_REUSEADDR) failed (%d).", errno);

	sin.sin_family = AF_INET;
	sin.sin_addr.s_addr = htonl(INADDR_ANY);

	if(bind(s, (struct sockaddr *)&sin, sizeof(sin))) {
		perror("bind");
		exit(QUIT_ERR);
	}

	if(listen(s, 5)) {
		perror("listen");
		exit(QUIT_ERR);
	}

	signal(SIGCHLD, chldhand);

	f = fork();
	if(f < 0) {
		perror("fork");
		exit(QUIT_ERR);
	}

	if(f > 0)
		exit(QUIT_OK);

	if(setsid() == -1)
		fprintf(stderr, "Warning: Can't set process group.\n");

	if(verbose)
		printf("Running standalone at port %d.", port);

	len = sizeof(from);

	for(;;) {
		sock = accept(s, (struct sockaddr *)&from, &len);

		if(sock < 0) {
			if(errno == EINTR)
				continue;

			perror("accept");
			quit(QUIT_ERR);
		}

		f = fork();

		if(f < 0) {
			perror("fork");
			quit(QUIT_ERR);
		}

		/* We're the child - crank. */
		if(f == 0) {
			curpid = getpid();
			dup2(sock, 0);
			close(sock);
			return;
		}

		close(sock);
	}
}


int
cddbp_open(char *site, int port)
{
	char buf[CDDBBUFSIZ];
	char errstr[CDDBBUFSIZ];

	if(verbose)
		cddbd_log(LOG_INFO, "Opening CDDBP server: %s", site);

	if(!sock_open(site, SERV_CDDBP, port, errstr)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to open %s CDDBP server: %s.", site, errstr);
		return 0;
	}

	if(!sock_read(buf, sizeof(buf), 1)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to init %s CDDBP server: connection closed.", site);
		sock_close();
		return 0;
	}

	/* Good codes start with 2. */
	if(buf[0] != '2') {
		strip_crlf(buf);
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to init %s CDDBP server: unexpected server "
		    "response: %s.", site, buf);

		sock_close();
		return 0;
	}

	cddbd_snprintf(buf, sizeof(buf), "cddb hello %s %s cddbd %sPL%d\n",
	    user, host, VERSION, PATCHLEVEL);

	if(!sock_write(buf, 1)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to handshake with %s CDDBP server (%s).",
		    site, errno);
		sock_close();
		return 0;
	}

	if(!sock_serv_read(buf, sizeof(buf), 200, errstr)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to open %s CDDBP server: %s", site, errstr);
		sock_close();
		return 0;
	}

	strncpy(cddbphost, site, sizeof(cddbphost));
	cddbphost[sizeof(cddbphost) - 1] = '\0';

	return 1;
}


int
cddbp_log_stats(void)
{
	char buf[CDDBBUFSIZ];
	char errstr[CDDBBUFSIZ];

	if(!sock_write("log day\n", 1)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to send log command to %s CDDBP server (%s)",
		    cddbphost, errno);
		return 0;
	}

	if(!sock_serv_read(buf, sizeof(buf), 210, errstr)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to get log stats from %s CDDBP server: %s",
		    cddbphost, errstr);
		return 0;
	}

	while(sock_read(buf, sizeof(buf), 0)) {
		strip_crlf(buf);

		if(!strcmp(buf, "."))
			return 1;

		printf("%s\n", buf);
	}

	cddbd_log(LOG_ERR | LOG_NET,
	    "Failed to get log stats from %s CDDBP server: "
	    "connection closed (%d)", cddbphost, errno);

	return 0;
}


int
cddbp_transmit(db_t *db, char *category, unsigned int discid)
{
	char buf[CDDBBUFSIZ];
	char errstr[CDDBBUFSIZ];

	cddbd_snprintf(buf, sizeof(buf), "cddb write %s %08x\n",
	    category, discid);

	if(!sock_write(buf, 1)) {
		cddbd_log(LOG_ERR | LOG_XMIT,
		    "Failed to send write command to %s CDDBP server (%s).",
		    cddbphost, errno);
		return 0;
	}

	if(!sock_serv_read(buf, sizeof(buf), 320, errstr)) {
		cddbd_log(LOG_ERR | LOG_XMIT,
		    "Failed to send write command to %s CDDBP server: %s.",
		    cddbphost, errstr);
		return 0;
	}

	if(!db_write(sock_fp, db)) {
		cddbd_log(LOG_ERR | LOG_XMIT,
		    "Failed to write DB entry to %s CDDBP server (%d).",
		    cddbphost, errno);
		return 0;
	}

	fflush(sock_fp);

	if(!sock_write(".\n", 1)) {
		cddbd_log(LOG_ERR | LOG_XMIT,
		    "Failed to write DB entry to %s CDDBP server (%d).",
		    cddbphost, errno);
		return 0;
	}

	buf[0] = '\0';

	if(!sock_serv_read(buf, sizeof(buf), 200, errstr)) {
		cddbd_log(LOG_ERR | LOG_XMIT,
		    "Failed to write DB entry to %s CDDBP server: %s.",
		    cddbphost, errstr);

		/* Was the DB entry bad, or was there a server error? */
		if(!strncmp(buf, "501", 3))
			return -1;
		else
			return 0;
	}

	return 1;
}


int
cddbp_update(void)
{
	char buf[CDDBBUFSIZ];
	char errstr[CDDBBUFSIZ];

	if(!sock_write("update\n", 1)) {
		cddbd_log(LOG_ERR | LOG_XMIT,
		    "Failed to initiate update on %s CDDBP server (%d).",
		    cddbphost, errno);
		return 0;
	}

	if(!sock_serv_read(buf, sizeof(buf), 200, errstr)) {
		cddbd_log(LOG_ERR | LOG_XMIT,
		    "Failed to initiate update on %s CDDBP server: %s.",
		    cddbphost, errstr);
		return 0;
	}

	return 1;
}


void
cddbp_close(void)
{
	(void)sock_write("quit\n", 1);
	sock_close();
}


int
smtp_open()
{
	int code;
	char buf[CDDBBUFSIZ];
	char errstr[CDDBBUFSIZ];

	if(verbose)
		cddbd_log(LOG_INFO, "Opening SMTP server: %s", smtphost);

	if(!sock_open(smtphost, SERV_SMTP, 0, errstr)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to open %s SMTP server: %s", smtphost, errstr);
		return 0;
	}

	if(!sock_serv_read(buf, sizeof(buf), 220, errstr)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to init %s SMTP server: %s", smtphost, errstr);
		sock_close();
		return 0;
	}

	if(!sock_write("helo\n", 1)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to handshake with %s SMTP server (%s)",
		    smtphost, errno);

		sock_close();
		return 0;
	}

	if(!sock_serv_read(buf, sizeof(buf), 0, errstr)) {
		if(sscanf(buf, "%d", &code) != 1) {
			cddbd_log(LOG_ERR | LOG_NET,
			    "Failed to handshake with %s SMTP "
			    "server: %s", smtphost, errstr);

			sock_close();
			return 0;
		}

		if(code == 250)
			return 1;
	}

	/* Try the helo with an argument to see if that helps. */
	cddbd_snprintf(buf, sizeof(buf), "helo %s\n", host);

	if(!sock_write(buf, 1)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to handshake with %s SMTP server (%s)",
		    smtphost, errno);

		sock_close();
		return 0;
	}

	if(!sock_serv_read(buf, sizeof(buf), 0, errstr)) {
		if(sscanf(buf, "%d", &code) != 1 || code != 250) {
			cddbd_log(LOG_ERR | LOG_NET,
			    "Failed to handshake with %s SMTP "
			    "server: %s", smtphost, errstr);

			sock_close();
			return 0;
		}
	}

	return 1;
}


void
smtp_close(void)
{
	(void)sock_write("quit\n", 1);
	sock_close();
}


int
smtp_transmit(FILE *fp, char *subject, char *rcpt, char *hto, int enc, int len)
{
	char buf[CDDBBUFSIZ];
	char errstr[CDDBBUFSIZ];

	cddbd_snprintf(buf, sizeof(buf), "mail from: %s\n", admin_email);

	if(!sock_write(buf, 1)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to specify return addr to %s SMTP server (%d)",
		    smtphost, errno);
		return 0;
	}

	if(!sock_serv_read(buf, sizeof(buf), 250, errstr)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to handshake with %s SMTP server: %s",
		    smtphost, errstr);
		return 0;
	}

	cddbd_snprintf(buf, sizeof(buf), "rcpt to: %s\n", rcpt);

	if(!sock_write(buf, 1)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to specify recipient to %s SMTP server (%d)",
		    smtphost, errno);
		return 0;
	}

	if(!sock_serv_read(buf, sizeof(buf), 250, errstr)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to handshake with %s SMTP server: %s",
		    smtphost, errstr);
		return 0;
	}

	strcpy(buf, "data\n");

	if(!sock_write(buf, 1)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to send data to %s SMTP server (%d)",
		    smtphost, errno);
		return 0;
	}

	if(!sock_serv_read(buf, sizeof(buf), 354, errstr)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed \"data\" command on %s SMTP server (%d)",
		    smtphost, errno);
		return 0;
	}

	cddbd_snprintf(buf, sizeof(buf), "%s: %s\n", rpath, admin_email);

	if(!sock_write(buf, 0)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to write email header to %s SMTP server (%d)",
		    smtphost, errno);
		return 0;
	}

	cddbd_snprintf(buf, sizeof(buf), "%s: CDDBD v%sPL%d\n",
	    xsender, VERSION, PATCHLEVEL);

	if(!sock_write(buf, 0)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to write email header to %s SMTP server (%d)",
		    smtphost, errno);
		return 0;
	}

	cddbd_snprintf(buf, sizeof(buf), "%s: %s\n", to, hto);

	if(!sock_write(buf, 0)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to write email header to %s SMTP server (%d)",
		    smtphost, errno);
		return 0;
	}

	cddbd_snprintf(buf, sizeof(buf), "%s: %s\n", subj, subject);

	if(!sock_write(buf, 0)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to write email header to %s SMTP server (%d)",
		    smtphost, errno);
		return 0;
	}

	cddbd_snprintf(buf, sizeof(buf), "%s: %s\n", mime_ver, "1.0");

	if(!sock_write(buf, 0)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to write email header to %s SMTP server (%d)",
		    smtphost, errno);
		return 0;
	}

	cddbd_snprintf(buf, sizeof(buf), "%s: text/plain; charset=%s\n",
	    content_type, char_types[CC_ISO_8859_1]);

	if(!sock_write(buf, 0)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to write email header to %s SMTP server (%d)",
		    smtphost, errno);
		return 0;
	}

	cddbd_snprintf(buf, sizeof(buf), "%s: %s\n", content_encoding,
	    encoding_types[enc].en_type);

	if(!sock_write(buf, 0)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to write email header to %s SMTP server (%d)",
		    smtphost, errno);
		return 0;
	}

	/* If we have a length, and we're not going to change it, print it. */
	if(len > 0 && encoding_types[enc].en_mapfunc == 0) {
		cddbd_snprintf(buf, sizeof(buf), "%s: %d\n", content_len, len);

		if(!sock_write(buf, 0)) {
			cddbd_log(LOG_ERR | LOG_NET,
			    "Failed to write email header to %s SMTP "
			    "server (%d)", smtphost, errno);
			return 0;
		}
	}

	/* Separate the header from the body. */
	if(!sock_write("\n", 0)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to write email header to %s SMTP server (%d)",
		    smtphost, errno);
		return 0;
	}

	while(fgets(buf, sizeof(buf), fp) != NULL) {
		/* Remap the charset if needed. */
		if(encoding_types[enc].en_mapfunc != 0)
			(encoding_types[enc].en_mapfunc)(buf);

		if(!sock_write(buf, 0)) {
			cddbd_log(LOG_ERR | LOG_NET,
			    "Failed to write mail to %s SMTP server (%d)",
			    smtphost, errno);
			return 0;
		}
	}

	if(!sock_write(".\n", 0)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to send data %s SMTP server (%d)",
		    smtphost, errno);
		return 0;
	}

	if(!sock_serv_read(buf, sizeof(buf), 250, errstr)) {
		cddbd_log(LOG_ERR | LOG_NET,
		    "Failed to send data to %s SMTP server (%d)",
		    smtphost, errno);
		return 0;
	}

	return 1;
}


int
sock_open(char *shost, int ind, int rport, char *errstr)
{
	int port;
	struct in_addr ad;
	struct hostent *hp;
	struct sockaddr_in sin;

	if(sock_fd != -1)
		return 0;

	port = get_serv_port(rport, ind);

	if((hp = gethostbyname(shost)) != 0) {
		cddbd_bcopy(hp->h_addr, (char *)&sin.sin_addr, hp->h_length);
	}
	else {
		if(isipaddr(shost) && (ad.s_addr = inet_addr(shost)) != -1) {
			cddbd_bcopy((char *)&ad.s_addr, (char *)&sin.sin_addr,
			    sizeof(ad.s_addr));
		}
		else {
			strcpy(errstr, "unknown host");
			return 0;
		}
	}

	sin.sin_family = AF_INET;
	sin.sin_port = (short)port;

	if((sock_fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		cddbd_snprintf(errstr, CDDBBUFSIZ, "can't allocate socket (%d)",
		     errno);
		return 0;
	}

	if(connect(sock_fd, (struct sockaddr *)&sin, sizeof(sin)) == -1) {
		cddbd_snprintf(errstr, CDDBBUFSIZ, "connect failed (%d)",errno);
		close(sock_fd);
		sock_fd = -1;

		return 0;
	}

	if((sock_fp = fdopen(sock_fd, "r+")) == NULL) {
		cddbd_snprintf(errstr, CDDBBUFSIZ,
		    "failed to allocate FILE pointer (%d)", errno);
		close(sock_fd);
		return 0;
	}

	return 1;
}


void
sock_close(void)
{
	if(sock_fd == -1)
		return;

	shutdown(sock_fd, 2);
	fclose(sock_fp);

	sock_i = 0;
	sock_e = 0;
	sock_fd = -1;
}


int
sock_serv_read(char *buf, int size, int expect, char *errstr)
{
	int code;
	char dash;

	while(sock_read(buf, size, 1)) {
		if((sscanf(buf, "%d%c", &code, &dash) != 2) ||
		    (code && code != expect)) {
			/* Don't overflow. Assume CDDBBUFSIZ error string. */
			strcpy(errstr, "unexpected server response: ");
			strip_crlf(buf);
			strncat(errstr, buf, (CDDBBUFSIZ - strlen(errstr)));
			errstr[CDDBBUFSIZ - 1] = '\0';

			return 0;
		}

		if(dash != '-')
			return 1;
	}

	cddbd_snprintf(errstr, CDDBBUFSIZ, "socket read error (%d)", errno);
	return 0;
}


int
sock_write(char *buf, int flag)
{
	int cc;
	int cnt;
	char *p;
	char *pbuf;

	p = buf;
	cnt = strlen(buf);

	while(cnt > 0) {
		cc = write(sock_fd, buf, cnt);
		if(cc < 0)
			return 0;

		buf += cc;
		cnt -= cc;
	}

	if(verbose && flag) {
		pbuf = strdup(p);

		if(pbuf == NULL)
			cddbd_log(LOG_ERR, "Can't malloc pbuf.");
		else {
			strip_crlf(pbuf);
			cddbd_log(LOG_INFO, "-> %s", pbuf);
			free(pbuf);
		}
	}

	return 1;
}


int
sock_read(char *buf, int cnt, int flag)
{
	int c;
	char *p;
	char *pbuf;

	p = buf;

	while((c = sock_getc()) >= 0) {
		*buf = (char)c;

		buf++;
		cnt--;

		if(c == '\n' || cnt == 1) {
			*buf = '\0';

			if(verbose && flag) {
				pbuf = strdup(p);

				if(pbuf == NULL)
					cddbd_log(LOG_ERR, "Can't malloc buf.");
				else {
					strip_crlf(pbuf);
					cddbd_log(LOG_INFO, "<- %s", pbuf);
					free(pbuf);
				}
			}

			return 1;
		}
	}

	return 0;
}


int
sock_getc(void)
{
	static char buf[CDDBBUFSIZ];

	if(sock_i == sock_e) {
		if(!sock_wait())
			return -1;

		sock_e = read(sock_fd, buf, sizeof(buf));
		if(sock_e < 0) {
			sock_e = sock_i;
			return 0;
		}

		sock_i = 1;
	}
	else
		sock_i++;

	return((int)buf[sock_i - 1]);
}


int
sock_wait(void)
{
	int n;
	fd_set readfds;
	struct timeval tv;

	if(xmit_time == 0)
		return 1;

	do {
		FD_ZERO(&readfds);
		FD_SET(sock_fd, &readfds);

		tv.tv_sec = xmit_time;
		tv.tv_usec = 0;

#ifdef __hpux
		n = select((sock_fd + 1), (int *)&readfds, (int *)NULL,
		    (int *)NULL, &tv);
#else
		n = select((sock_fd + 1), &readfds, (fd_set *)NULL,
		    (fd_set *)NULL, &tv);
#endif

		if(n > 0)
			return 1;

	} while(n < 0 && errno == EINTR);

	if(n == 0) {
		errno = ETIMEDOUT;
		cddbd_log(LOG_ERR | LOG_XMIT, "Read timeout after %d seconds.",
		    xmit_time);
	}

	return 0;
}


void
cddbd_bcopy(char *from, char *to, int size)
{
	int i;

	for(i = 0; i < size; i++)
		to[i] = from[i];
}


short
get_serv_port(int port, int serv)
{
	short sport;
	struct servent *sp;

	if(port == 0) {
		if((sp = getservbyname(servtab[serv].s_name,
		    servtab[serv].s_proto)) == 0)
			sport = htons((short)servtab[serv].s_port);
		else
			sport = sp->s_port;
	}
	else
		sport = htons((short)port);

	return sport;
}


/* ARGSUSED */

void
chldhand(int sig)
{
	while(waitpid(-1, 0, WNOHANG) != -1)
		continue;

	signal(SIGCHLD, chldhand);
}
