/*
 *   cddbd - CD Database Protocol Server
 *
 *   Copyright (C) 1996  Steve Scherf
 *   Email: steve@moonsoft.com
 *   Moondog Software Productions - makers of fine public domain software.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifndef LINT
static char *_sites_c_ident_ = "@(#)$Id: xmit.c,v 1.1 1996/12/13 23:35:22 steve Exp $";
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <fcntl.h>
#include <stdio.h>
#include <errno.h>
#include <dirent.h>
#include <time.h>
#include "list.h"
#include "cddbd.h"


/* Preprocessor definitions. */

#define PROT_CDDBP	0
#define PROT_SMTP	1
#define PROT_INFO	2
#define PROT_LOG	3


/* Structures. */

typedef struct tfile {
	int tf_dir;
	unsigned int tf_discid;
} tfile_t;


/* Prototypes. */

void cddbd_catchup(char *, char *);
void cddbd_do_catchup(char *, char *);
void cddbd_do_remote_log(char *, int);
void cddbd_do_transmit(site_t *);
void cddbd_remote_log(char *);
void cddbd_transmit(char *);
void cddbd_write_history(char *, unsigned int);
void tfree(void *);

int cddbd_clean_history(void);
int cddbd_do_merge_ulist(char *, int);
int cddbd_oc_history(char *, int);
int cddbd_open_history(char *);
int cddbd_open_ulist(void);
int cddbd_read_history(char *, unsigned int *);
int parse_coord(char *, coord_t *);
int tfile_comp(void *, void *);


/* Global variables. */

FILE *site_fp = NULL;
FILE *ulist_fp = NULL;

char *histstr = "Start %s\n";

int hist_opened;
int site_line;

lhead_t *hist;


site_t *
getsitenam(char *site, int type)
{
	site_t *sp;

	setsiteent();

	while((sp = getsiteent(type)) != NULL)
		if(!cddbd_strcasecmp(site, sp->st_name))
			return sp;

	return NULL;
}


site_t *
getsiteent(int type)
{
	int i;
	int proto;
	arg_t args;
	char buf[CDDBBUFSIZ];
	static struct site site;

	if(site_fp == NULL) {
		setsiteent();

		if(site_fp == NULL)
			return NULL;
	}

	while(fgets(args.buf, sizeof(args.buf), site_fp) != NULL) {
		site_line++;

		/* Skip comments and blanks. */
		if(args.buf[0] == '#' || is_blank(args.buf, 0))
			continue;

		cddbd_parse_args(&args, 0);

		if(args.nargs < 2) {
			cddbd_log(LOG_ERR, "Syntax error in %s on line %d.",
			    sitefile, site_line);

			continue;
		}

		strncpy(site.st_name, args.arg[0], sizeof(site.st_name));
		site.st_name[sizeof(site.st_name) - 1] = '\0';
		site.st_port = 0;

		if(!cddbd_strcasecmp(args.arg[1], "cddbp"))
			proto = PROT_CDDBP;
		else if(!cddbd_strcasecmp(args.arg[1], "smtp"))
			proto = PROT_SMTP;
		else if(!cddbd_strcasecmp(args.arg[1], "log"))
			proto = PROT_LOG;
		else if(!cddbd_strcasecmp(args.arg[1], "info"))
			proto = PROT_INFO;
		else {
			cddbd_log(LOG_ERR,
			    "Unknown protocol \"%s\" in %s on line %d.",
			    args.arg[1], sitefile, site_line);

			continue;
		}

		switch(type) {
		case SITE_XMIT:
			if(proto == PROT_CDDBP)
				site.st_proto = ST_PROTO_CDDBP;
			else if(proto == PROT_SMTP)
				site.st_proto = ST_PROTO_SMTP;
			else
				continue;

			break;

		case SITE_LOG:
			if(proto == PROT_LOG)
				site.st_proto = ST_PROTO_CDDBP;
			else
				continue;

			break;

		case SITE_INFO:
			if(proto == PROT_INFO)
				site.st_proto = ST_PROTO_NONE;
			else
				continue;

			break;

		default:
			cddbd_log(LOG_ERR,
			    "Internal error: unknown site type: %s.", type);
			continue;
		}
			

		switch(site.st_proto) {
		case ST_PROTO_CDDBP:
			if(args.nargs > 3) {
				cddbd_log(LOG_ERR,
				    "Syntax error in %s on line %d.",
				    sitefile, site_line);
				continue;
			}

			if(args.nargs == 3) {
				site.st_port = atoi(args.arg[2]);

				if(site.st_port <= 0) {
					cddbd_log(LOG_ERR,
					    "Invalid port in %s on line %d.",
					    sitefile, site_line);

					continue;
				}
			}
			else
				site.st_port = 0;

			break;

		case ST_PROTO_SMTP:
			if(args.nargs != 3) {
				cddbd_log(LOG_ERR,
				    "Syntax error in %s on line %d.",
				    sitefile, site_line);
				continue;
			}

			site.st_proto = ST_PROTO_SMTP;

			strncpy(site.st_user, args.arg[2],
			    sizeof(site.st_user));

			break;

		case ST_PROTO_NONE:
			if(args.nargs < 6) {
				cddbd_log(LOG_ERR,
				    "Syntax error in %s on line %d.",
				    sitefile, site_line);
				continue;
			}

			if(cddbd_strcasecmp(args.arg[2], "-")) {
				site.st_port = atoi(args.arg[2]);

				if(site.st_port <= 0) {
					cddbd_log(LOG_ERR,
					    "Invalid port in %s on line %d.",
					    sitefile, site_line);

					continue;
				}
			}
			else
				site.st_port = ntohs(get_serv_port(0,
				    SERV_CDDBP));

			strncpy(buf, args.arg[3], (sizeof(buf) - 1));
			buf[strlen(buf)] = '\0';

			if(!parse_coord(buf, &site.st_lat)) {
				cddbd_log(LOG_ERR,
				    "Malformed coordinate in %s on line %d.",
				    sitefile, site_line);
				continue;
			}

			strncpy(buf, args.arg[4], (sizeof(buf) - 1));
			buf[strlen(buf)] = '\0';

			if(!parse_coord(buf, &site.st_long)) {
				cddbd_log(LOG_ERR,
				    "Malformed coordinate in %s on line %d.",
				    sitefile, site_line);
				continue;
			}

			site.st_desc[0] = '\0';

			for(i = 5; i < args.nargs; i++) {
				if(strlen(site.st_desc) + strlen(args.arg[i])
				    + 2 > sizeof(site.st_desc))
					break;

				strcat(site.st_desc, args.arg[i]);

				if(i != (args.nargs - 1))
					strcat(site.st_desc, " ");
			}

			break;

		default:
			cddbd_log(LOG_ERR,
			    "Internal error: impossible default case.");
			break;
		}

		return(&site);
	}

	return NULL;
}


void
endsiteent(void)
{
	if(site_fp != NULL) {
		fclose(site_fp);
		site_fp = NULL;
	}
}


void
setsiteent(void)
{
	if(site_fp == NULL) {
		if(sitefile[0] == '\0')
			return;
		if((site_fp = fopen(sitefile, "r")) == NULL)
			return;
	}
	else
		rewind(site_fp);

	site_line = 0;
}


void
tfree(void *p)
{
	free(p);
}


int
tfile_comp(void *t1, void *t2)
{
	int x;

	x = ((tfile_t *)t1)->tf_dir - ((tfile_t *)t2)->tf_dir;

	if(x != 0)
		return x;

	if(((tfile_t *)t1)->tf_discid > ((tfile_t *)t2)->tf_discid)
		return 1;

	if(((tfile_t *)t1)->tf_discid < ((tfile_t *)t2)->tf_discid)
		return -1;

	return 0;
}


void
cddbd_rmt_op(int op, char *site, char *when)
{
	int type;

	if(op == RMT_OP_LOG)
		type = SITE_LOG;
	else
		type = SITE_XMIT;

	if(strcmp(site, "all") && (getsitenam(site, type) == NULL)) {
		cddbd_log(LOG_ERR,
		    "Site \"%s\" is not defined in %s.", site, sitefile);
		return;
	}

	(void)cddbd_clean_history();
	endsiteent();

	switch(op) {
	case RMT_OP_TRANSMIT:
		cddbd_transmit(site);
		break;

	case RMT_OP_LOG:
		cddbd_remote_log(site);
		break;

	case RMT_OP_CATCHUP:
		cddbd_catchup(site, when);
		break;

	default:
		cddbd_log(LOG_ERR,
		    "Internal server error: unknown remote op: %d.", op);
		quit(QUIT_ERR);
	}
}


void
cddbd_transmit(char *site)
{
	int f;
	int xcnt;
	site_t *sp;

	if(strcmp(site, "all")) {
		if((sp = getsitenam(site, SITE_XMIT)) != NULL)
			cddbd_do_transmit(sp);
	}
	else {
		setsiteent();

		for(xcnt = 0;;) {
			if((sp = getsiteent(SITE_XMIT)) == NULL)
				break;

			if(max_xmits > 1) {
				if(xcnt >= max_xmits) {
					while(wait(0) < 0 && errno == EINTR)
						continue;
					xcnt--;
				}

				f = cddbd_fork();

				if(f < 0) {
					cddbd_log(LOG_ERR | LOG_XMIT,
					    "Can't fork child for xmit (%d).",
					    errno);

					return;
				}

				/* The child does the transmit. */
				if(f == 0) {
					cddbd_do_transmit(sp);
					quit(QUIT_OK);
				}

				xcnt++;
			}
			else
				cddbd_do_transmit(sp);
		}

		endsiteent();

		while(xcnt > 0) {
			while(wait(0) < 0 && errno == EINTR)
				continue;
			xcnt--;
		}
	}
}


void
cddbd_do_transmit(site_t *sp)
{
	int bps;
	int xmit;
	int fail;
	int tcnt;
	int files;
	int encoding;
	db_t *db;
	time_t xtime;
	FILE *fp;
	lhead_t *li;
	struct stat sbuf;
	unsigned int discid;
	char dir[CDDBBUFSIZ];
	char file[CDDBBUFSIZ];
	char subj[CDDBBUFSIZ];
	char errstr[CDDBBUFSIZ];
	char xmit_email[CDDBBUFSIZ];

	if(!cddbd_open_history(sp->st_name)) {
		cddbd_log(LOG_ERR | LOG_XMIT,
		    "Failed to open history file %s.", histfile);
		return;
	}

	files = list_count(hist);

	if(files > 0) {
		cddbd_log(LOG_INFO, "Beginning file transmission to %s.",
		    sp->st_name);
	}
	else {
		if(verbose)
			cddbd_log(LOG_INFO, "Remote host %s up to date.",
			    sp->st_name);
		return;
	}

	switch(sp->st_proto) {
	case ST_PROTO_CDDBP:
		if(!cddbp_open(sp->st_name, sp->st_port)) {
			cddbd_log(LOG_ERR | LOG_XMIT,
			    "Failed to open %s for file transmission.",
			    sp->st_name);

			if(!cddbd_close_history()) {
				cddbd_log(LOG_ERR | LOG_XMIT,
				    "Failed to update the history for: %s.",
				    sp->st_name);
			}

			return;
		}

		break;

	case ST_PROTO_SMTP:
		if(!smtp_open()) {
			cddbd_log(LOG_ERR | LOG_XMIT,
			    "Failed to open SMTP for file transmission to %s.",
			    sp->st_name);

			if(!cddbd_close_history()) {
				cddbd_log(LOG_ERR | LOG_XMIT,
				    "Failed to update the history for: %s.",
				    sp->st_name);
			}

			return;
		}

		cddbd_snprintf(xmit_email, sizeof(xmit_email), "%s@%s",
		    sp->st_user, sp->st_name);

		break;

	default:
		cddbd_log(LOG_ERR | LOG_XMIT,
		    "Internal error: unknown transmit protocol (%d).",
		    sp->st_proto);

		if(!cddbd_close_history()) {
			cddbd_log(LOG_ERR | LOG_XMIT,
			    "Failed to update the history for: %s.",
			    sp->st_name);
		}

		quit(QUIT_ERR);
	}

	li = list_init(0, 0, 0, 0);
	if(li == 0) {
		cddbd_log(LOG_ERR | LOG_XMIT, "Can't malloc linked list.");

		if(!cddbd_close_history()) {
			cddbd_log(LOG_ERR | LOG_XMIT,
			    "Failed to update the history for: %s.",
			    sp->st_name);
		}

		quit(QUIT_ERR);
	}

	xmit = 0;
	fail = 0;
	tcnt = 0;
	xtime = time(0);

	while(cddbd_read_history(dir, &discid)) {
		cddbd_snprintf(file, sizeof(file), "%s/%s/%08x", cddbdir,
		    dir, discid);

		if(stat(file, &sbuf) != 0) {
			cddbd_log(LOG_ERR | LOG_XMIT,
			    "Can't stat DB file: %s (%d).", file, errno);
			continue;
		}

		/* Don't transmit the same file twice. */
		if(list_find(li, (void *)(int)sbuf.st_ino) != 0) {
			files--;
			continue;
		}

		if(list_add_cur(li, (void *)(int)sbuf.st_ino) == 0) {
			cddbd_log(LOG_ERR, "Can't malloc linked list entry.");

			if(!cddbd_close_history()) {
				cddbd_log(LOG_ERR | LOG_XMIT,
				    "Failed to update the history for: %s.",
				    sp->st_name);
			}

			quit(QUIT_ERR);
		}

		if((fp = fopen(file, "r")) == NULL) {
			cddbd_log(LOG_ERR | LOG_XMIT,
			    "Failed to open DB file %s for transmission.",file);
			continue;
		}

		db = db_read(fp, errstr, 0);

		if(db == 0) {
			switch(db_errno) {
			case DE_INVALID:
				cddbd_log(LOG_ERR | LOG_XMIT,
				    "Invalid DB file: %s: %s.",
				    file, errstr);

				break;

			case DE_FILE:
			default:
				cddbd_log(LOG_ERR | LOG_XMIT,
				    "Can't read DB file: %s: %s (%d).",
				    file, errstr, errno);

				break;
			}

			fclose(fp);
			continue;
		}

		switch(sp->st_proto) {
		case ST_PROTO_CDDBP:
			switch(cddbp_transmit(db, dir, discid)) {
			case -1:
				/* DB file was corrupt. */
				break;

			case 0:
				/* Something wrong with server. */
				fail++;
				break;

			default:
				/* File transferred. */
				xmit++;
				tcnt += sbuf.st_size;
				break;
			}

			break;

		case ST_PROTO_SMTP:
			rewind(fp);

			cddbd_snprintf(subj, sizeof(subj), "cddb %s %08x",
			    dir, discid);

			if(db->db_flags & DB_REMAPPABLE)
				encoding = CE_QUOTED_PRINT;
			else
				encoding = CE_7BIT;

			if(!smtp_transmit(fp, subj, xmit_email, xmit_email,
			    encoding, 0))
				fail++;
			else {
				xmit++;
				tcnt += sbuf.st_size;
			}

			break;

		default:
			cddbd_log(LOG_ERR | LOG_XMIT,
			    "Internal error: unknown transmit protocol (%d).",
			    sp->st_proto);

			if(!cddbd_close_history()) {
				cddbd_log(LOG_ERR | LOG_XMIT,
				    "Failed to update the history for: %s.",
				    sp->st_name);
			}

			quit(QUIT_ERR);
		}

		fclose(fp);
		db_free(db);

		if(fail) {
			cddbd_log(LOG_ERR | LOG_XMIT,
			    "Failed to transmit DB file: %s/%08x.",
			    dir, discid);

			/* Put the entry back in the history list. */
			cddbd_write_history(dir, discid);

			break;
		}
	}

	list_free(li);

	xtime = time(0) - xtime;
	if(xtime == 0)
		xtime = 1;

	if(fail) {
		cddbd_log(LOG_ERR | LOG_XMIT, "Aborting transmit to %s.",
		    sp->st_name);
	}
	else
		cddbd_log(LOG_INFO, "Completed transmit to %s.", sp->st_name);

	bps = tcnt / xtime;

	cddbd_log(LOG_INFO,
	    "Transmitted %d of %d files, %d bytes in %d sec (%d.%dK/sec).",
	    xmit, files, tcnt, xtime, (int)(bps / 1024.0),
	    (int)((bps % 1024) / 1024.0 * 10));

	switch(sp->st_proto) {
	case ST_PROTO_CDDBP:
		if(!fail)
			(void)cddbp_update();
		cddbp_close();
		break;

	case ST_PROTO_SMTP:
		smtp_close();
		break;

	default:
		cddbd_log(LOG_ERR | LOG_XMIT,
		    "Internal error: unknown transmit protocol (%d).",
		    sp->st_proto);

		if(!cddbd_close_history()) {
			cddbd_log(LOG_ERR | LOG_XMIT,
			    "Failed to update the history for: %s.",
			    sp->st_name);
		}

		quit(QUIT_ERR);
	}

	if(!cddbd_close_history()) {
		cddbd_log(LOG_ERR | LOG_XMIT,
		    "Failed to update the history for: %s.", sp->st_name);
	}
}


void
cddbd_remote_log(char *site)
{
	int first;
	site_t *sp;

	if(strcmp(site, "all")) {
		if((sp = getsitenam(site, SITE_LOG)) != NULL &&
		    sp->st_proto == ST_PROTO_CDDBP)
			cddbd_do_remote_log(site, sp->st_port);

		return;
	}

	for(first = 1;; first = 0) {
		if((sp = getsiteent(SITE_LOG)) == NULL)
			break;

		if(sp->st_proto != ST_PROTO_CDDBP)
			continue;

		if(!first)
			printf("\n\n");

		cddbd_do_remote_log(sp->st_name, sp->st_port);
	}

	endsiteent();
}


void
cddbd_do_remote_log(char *site, int port)
{
	printf("Log statistics for %s:\n\n", site);

	if(!cddbp_open(site, port)) {
		printf("Unknown host %s or host unavailable.\n", site);
		cddbd_log(LOG_ERR | LOG_XMIT,
		    "Failed to open %s for log acquisition.", site);

		return;
	}

	if(!cddbp_log_stats())
		printf("Unable to get log stats.\n");

	cddbp_close();
}


void
cddbd_catchup(char *site, char *when)
{
	site_t *sp;
	site_t sitebuf;

	if(strcmp(site, "all")) {
		cddbd_do_catchup(site, when);
		return;
	}

	/* Catching up uses getsiteent, so be careful here. */
	setsiteent();

	if((sp = getsiteent(SITE_XMIT)) == NULL)
		return;

	sitebuf = *sp;

	for(;;) {
		sitebuf = *sp;
		cddbd_do_catchup(sitebuf.st_name, when);

		if(getsitenam(sitebuf.st_name, SITE_XMIT) == NULL)
			break;
		if((sp = getsiteent(SITE_XMIT)) == NULL)
			break;
	}

	endsiteent();
}


void
cddbd_do_catchup(char *site, char *when)
{
	int i;
	time_t t;
	DIR *dirp;
	struct stat sbuf;
	struct dirent *dp;
	char buf[CDDBBUFSIZ];
	char cdir[CDDBBUFSIZ];
	char file[CDDBBUFSIZ];

	/* Don't scan the database if they want to catch everything up. */
	if(!strcmp("now", when)) {
		t = time(0);
		strcpy(buf, make_time2(localtime(&t))); 

		cddbd_log(LOG_INFO,
		    "Resetting transmit history for %s to %s.", site, buf);

		if(!cddbd_open_ulist()) {
			cddbd_log(LOG_ERR, "Failed to update history for %s.",
			    site);
			quit(QUIT_ERR);
		}
	}
	else {
		strcpy(buf, make_time2(date_to_tm(when))); 

		cddbd_log(LOG_INFO,
		    "Resetting transmit history for %s to %s.", site, buf);

		for(i = 0; categlist[i] != 0; i++) {
			cddbd_snprintf(cdir, sizeof(cdir), "%s/%s", cddbdir,
			    categlist[i]);

			if((dirp = opendir(cdir)) == NULL)
				continue;

			while((dp = readdir(dirp)) != NULL) {
				cddbd_snprintf(buf, sizeof(buf), "%s/%s", cdir,
				    dp->d_name);

				/* Make sure this is a database file. */
				if(strlen(dp->d_name) != CDDBDISCIDLEN) {
					if(!is_parent_dir(dp->d_name)) {
						cddbd_log(LOG_INFO,
						    "Non-CDDB file: %s", file);
					}

					continue;
				}

				if(stat(buf, &sbuf)) {
					cddbd_log(LOG_ERR,
					    "Can't stat DB file: %s (%d).",
					    file, errno);

					continue;
				}

				/* If the file is too old, continue. */
				cvt_time(sbuf.st_mtime, buf);
				if(strcmp(when, buf) > 0)
					continue;

				if(!cddbd_write_ulist(categlist[i],
				    dp->d_name)) {
					cddbd_log(LOG_ERR,
					    "Failed to update history for %s.",
					    site);
					quit(QUIT_ERR);
				}
			}

			closedir(dirp);
		}
	}

	if(!cddbd_merge_ulist(site, 0)) {
		cddbd_log(LOG_ERR, "Failed to update history for %s.", site);
		quit(QUIT_ERR);
	}
}


int
cddbd_oc_history(char *name, int oflag)
{
	int dir;
	int insite;
	char *p;
	FILE *fp;
	FILE *tfp;
	link_t *lp;
	tfile_t *tf;
	unsigned int discid;
	char buf[CDDBBUFSIZ];
	char buf2[CDDBBUFSIZ];

	if(histfile[0] == '\0') {
		cddbd_log(LOG_ERR, "No history file defined.");
		return 0;
	}

	/* Don't open if we already have, or close if we're not open. */
	if(oflag && hist_opened)
		return 0;

	if(!oflag && !hist_opened)
		return 0;

	(void)cddbd_lock(lock_hist, 1);

	close(open(histfile, O_WRONLY | O_CREAT, file_mode));

	(void)cddbd_fix_file(histfile, file_mode, uid, gid);

	if((fp = fopen(histfile, "r")) == NULL) {
		cddbd_log(LOG_ERR, "Can't open history file: %s.", histfile);
		cddbd_unlock(lock_hist);
		return 0;
	}

	cddbd_snprintf(thistfile, CDDBBUFSIZ, "%s_tmp", histfile);

	if((tfp = fopen(thistfile, "w")) == NULL) {
		cddbd_log(LOG_ERR, "Can't open tmp history file: %s.",
		    thistfile);

		fclose(fp);
		cddbd_unlock(lock_hist);

		return 0;
	}

	(void)cddbd_fix_file(thistfile, file_mode, uid, gid);

	if(oflag) {
		p = strdup(name);
		if(p == NULL) {
			cddbd_log(LOG_ERR | LOG_XMIT,
			    "Can't malloc list string.");
			quit(QUIT_ERR);
		}

		hist = list_init(p, tfile_comp, tfree, tfree);
		if(hist == 0) {
			cddbd_log(LOG_ERR | LOG_XMIT,
			    "Can't malloc linked list.");
			quit(QUIT_ERR);
		}
	}

	insite = 0;

	while(fgets(buf, sizeof(buf), fp) != NULL) {
		if(sscanf(buf, histstr, buf2) == 1) {
			if(!strcmp(name, buf2)) {
				insite = 2;
				continue;
			}
			else {
				if(fputs(buf, tfp) == EOF) {
					cddbd_log(LOG_ERR | LOG_XMIT,
					    "Can't write tmp history file: %s.",
					    thistfile);
					quit(QUIT_ERR);
				}

				insite = 1;
				continue;
			}
		}
		else

		if(sscanf(buf, "%[^/]/%08x", buf2, &discid) != 2)
			continue;

		dir = categ_index(buf2);
		if(dir < 0)
			continue;

		switch(insite) {
		case 0:
		default:
			break;

		case 1:
			if(fprintf(tfp, "%s/%08x\n", buf2, discid) == EOF) {
				cddbd_log(LOG_ERR | LOG_XMIT,
				    "Can't write tmp history file: %s.",
				    thistfile);
				quit(QUIT_ERR);
			}

			break;

		case 2:
			cddbd_write_history(buf2, discid);

			break;
		}
	}

	if(!oflag && list_count(hist) > 0) {
		if(fprintf(tfp, histstr, (char *)hist->lh_data) == EOF) {
			cddbd_log(LOG_ERR | LOG_XMIT,
			    "Can't write tmp history file: %s.", thistfile);
			quit(QUIT_ERR);
		}

		for(list_rewind(hist), list_forw(hist); !list_rewound(hist);
		    list_forw(hist)) {
			lp = list_cur(hist);
			tf = (tfile_t *)lp->l_data;

			if(fprintf(tfp, "%s/%08x\n", categlist[tf->tf_dir],
			    tf->tf_discid) == EOF) {
				cddbd_log(LOG_ERR | LOG_XMIT,
				    "Can't write tmp history file: %s.",
				    thistfile);
				quit(QUIT_ERR);
			}
		}
	}

	fclose(fp);
	fclose(tfp);

	if(unlink(histfile) != 0 && errno != ENOENT) {
		cddbd_log(LOG_ERR | LOG_XMIT,
		    "Can't unlink %s (%d).", histfile, errno);
		quit(QUIT_ERR);
	}

	if(cddbd_link(thistfile, histfile) != 0) {
		cddbd_log(LOG_ERR | LOG_XMIT,
		    "Can't link %s to %s (%d).", thistfile, histfile, errno);
		quit(QUIT_ERR);
	}

	if(oflag) {
		hist_opened = 1;
		list_rewind(hist);
	}
	else {
		hist_opened = 0;
		list_free(hist);
	}

	if(unlink(thistfile) != 0) {
		cddbd_log(LOG_ERR | LOG_XMIT,
		    "Warning: can't unlink %s (%d).", thistfile, errno);
	}

	cddbd_unlock(lock_hist);

	return 1;
}


int
cddbd_read_history(char *dir, unsigned int *discid)
{
	link_t *lp;
	tfile_t *tf;

	if(list_count(hist) == 0)
		return 0;

	lp = list_first(hist);
	tf = (tfile_t *)lp->l_data;
	*discid = tf->tf_discid;
	strcpy(dir, categlist[tf->tf_dir]);

	list_delete(hist, lp);

	return 1;
}


void
cddbd_write_history(char *dir, unsigned int discid)
{
	tfile_t *tf;

	if((tf = (tfile_t *)malloc(sizeof(tfile_t))) == NULL) {
		cddbd_log(LOG_ERR | LOG_XMIT,
		    "Can't malloc transmit list entry.");
		quit(QUIT_ERR);
	}

	tf->tf_dir = categ_index(dir);
	tf->tf_discid = discid;

	if(list_find(hist, (void *)tf) != 0) {
		free(tf);
	}
	else if(list_add_back(hist, (void *)tf) == 0) {
		cddbd_log(LOG_ERR | LOG_XMIT,
		    "Can't malloc linked list entry.");
		quit(QUIT_ERR);
	}
}


int
cddbd_close_history(void)
{
	if(!hist_opened)
		return 0;

	return(cddbd_oc_history((char *)hist->lh_data, 0));
}


int
cddbd_open_history(char *site)
{
	return(cddbd_oc_history(site, 1));
}


int
cddbd_count_history(char *site)
{
	FILE *fp;
	int cnt;
	int insite;
	char buf[CDDBBUFSIZ];
	char buf2[CDDBBUFSIZ];

	if(histfile[0] == '\0') {
		cddbd_log(LOG_ERR, "No history file defined.");
		return -1;
	}

	(void)cddbd_lock(lock_hist, 1);

	if((fp = fopen(histfile, "r")) == NULL) {
		cddbd_log(LOG_ERR,
		    "Can't open history file: %s.", histfile);
		return -1;
	}

	cnt = 0;
	insite = 0;

	while(fgets(buf, sizeof(buf), fp) != NULL) {
		/* If we found a nonexistent site, clear it. */
		if(sscanf(buf, histstr, buf2) == 1) {
			if(!strcmp(site, buf2))
				insite = 1;
			else
				insite = 0;

			continue;
		}

		if(insite)
			cnt++;
	}

	fclose(fp);
	cddbd_unlock(lock_hist);

	return cnt;
}


int
cddbd_clean_history(void)
{
	int ret;
	FILE *fp;
	char buf[CDDBBUFSIZ];
	char buf2[CDDBBUFSIZ];

	if(histfile[0] == '\0') {
		cddbd_log(LOG_ERR, "No history file defined.");
		return 0;
	}

	(void)cddbd_lock(lock_hist, 1);

	ret = 1;

	for(fp = NULL; fp == NULL;) {
		if((fp = fopen(histfile, "r")) == NULL) {
			cddbd_log(LOG_ERR,
			    "Can't open history file: %s.", histfile);

			ret = 0;
			break;
		}

		/* Scan the file for nonexistent sites. */
		while(fgets(buf, sizeof(buf), fp) != NULL) {
			/* If we found a nonexistent site, clear it. */
			if(sscanf(buf, histstr, buf2) == 1 &&
			    getsitenam(buf2, SITE_XMIT) == NULL) {
				fclose(fp);
				fp = NULL;

				/* This clears the history for the site. */
				if(cddbd_open_history(buf2)) {
					hist_opened = 0;
					list_free(hist);
				}

				break;
			}
		}
	}

	endsiteent();
	cddbd_unlock(lock_hist);

	return ret;
}


int
cddbd_merge_ulist(char *site, int merge)
{
	int ret;
	site_t *sp;

	if(ulist_fp == NULL)
		return 1;

	if(strcmp(site, "all")) {
		if((sp = getsitenam(site, SITE_XMIT)) != NULL)
			ret = cddbd_do_merge_ulist(sp->st_name, merge);
	}
	else {
		ret = 1;

		setsiteent();

		for(;;) {
			if((sp = getsiteent(SITE_XMIT)) == NULL)
				break;

			if(!cddbd_do_merge_ulist(sp->st_name, merge))
				ret = 0;
		}
	}


	fclose(ulist_fp);
	unlink(uhistfile);
	ulist_fp = NULL;
	endsiteent();

	return ret;
}


int
cddbd_do_merge_ulist(char *site, int merge)
{
	unsigned int discid;
	char buf[CDDBBUFSIZ];
	char dir[CDDBBUFSIZ];

	if(!cddbd_open_history(site))
		return 0;

	/* Clear the history and reopen the file if we're erasing first. */
	if(!merge) {
		hist_opened = 0;
		list_free(hist);

		if(!cddbd_open_history(site))
			return 0;
	}

	rewind(ulist_fp);

	while(fgets(buf, sizeof(buf), ulist_fp) != NULL)
		if(sscanf(buf, "%s%08x", dir, &discid) == 2)
			cddbd_write_history(dir, discid);

	if(!cddbd_close_history()) {
		cddbd_log(LOG_ERR | LOG_UPDATE,
		    "Failed to update history for %s.", site);
		return 0;
	}

	return 1;
}
	

int
cddbd_open_ulist(void)
{
	if(ulist_fp == NULL) {
		cddbd_snprintf(uhistfile, CDDBBUFSIZ, "%s.%05d",
		    histfile, curpid);

		if((ulist_fp = fopen(uhistfile, "w+")) == NULL) {
			cddbd_log(LOG_ERR | LOG_UPDATE,
			    "Can't open update-history file: %s.", uhistfile);
			return 0;
		}
	}

	return 1;
}


int
cddbd_write_ulist(char *dir, char *disc)
{
	if(!cddbd_open_ulist())
		return 0;

	if(fprintf(ulist_fp, "%s %s\n", dir, disc) == EOF) {
		cddbd_log(LOG_ERR | LOG_UPDATE,
		    "Can't write update-history file: %s.", uhistfile);
		return 0;
	}

	return 1;
}


int
parse_coord(char *buf, coord_t *cp)
{
	if(sscanf(buf, "%c%d.%d\n", &cp->co_compass, &cp->co_degrees,
	    &cp->co_minutes) != 3)
		return 0;

	if(cp->co_compass == 'N' || cp->co_compass == 'S') {
		if(cp->co_degrees > 90)
			return 0;
	}
	else if(cp->co_compass == 'W' || cp->co_compass == 'E') {
		if(cp->co_degrees > 180)
			return 0;
	}
	else
		return 0;

	if(cp->co_degrees < 0 || cp->co_minutes > 59 || cp->co_minutes < 0)
		return 0;

	return 1;
}


void
copy_coord(char *buf, coord_t *cp)
{
	cddbd_snprintf(buf, CDDBBUFSIZ, "%c%03d.%02d", cp->co_compass,
	    cp->co_degrees, cp->co_minutes);
}
