/* -*-gnucash-c-*-
   gnucash.c -- gnucash startup function.
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.
                                                                   
   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
                                                                   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   (C) 1998 Rob Browning <rlb@cs.utexas.edu>
*/

#define _GNU_SOURCE
#include <stdio.h>
#include <assert.h>

#include <guile/gh.h>

#include "gnucash.h"

static int guile_is_initialized = 0;
static int gargc;
static char **gargv;

static const char *
reverse_scan_argv_for_string_arg(const char *target,
                                 const int argc,
                                 char *argv[])
{
  const char *result = NULL;
  
  int i;
  for(i = argc - 1; (!result) && (i > 0); i--)
  {
    if(strcmp(target, argv[i]) == 0)
    {
      assert((i + 1) < argc); /* FIXME: This needs to cause a usage failure */
      result = argv[i + 1];
    }
  }
  return result;
}

static int
running_as_shell(const int argc, const char **argv) {
  
  char *cmd_name = rindex(argv[0], '/');
  int result = 0;
  
  if(cmd_name != NULL) {
    /* Skip the '/' */
    cmd_name++;
  } else {
    cmd_name = argv[0];
  }
  return(strcmp(cmd_name, "gnucash-shell") == 0);
}

/* Because swig doesn't declare the prototype anywhere */
void gnucash_swig_init();

static void gnucash_main_helper(int argc, char *argv[])
{  
  const char *startup_dir = NULL;
  char *startup_file = NULL;
  guile_is_initialized = 1;

  gnucash_swig_init();

  /* This has to be done from C because it's the one arg we *have* to
     have (if present) before we start up guile.  It's used to find
     the guile code.  The last --startup-dir on the command line
     wins. */

  startup_dir = reverse_scan_argv_for_string_arg("--startup-dir", argc, argv);
  
  if(!startup_dir)
  {
    startup_dir = GNC_STARTUP_DIR;
  }

  asprintf(&startup_file, "%s/init.scm", startup_dir);
  assert(startup_file);
  
  fprintf(stderr, "gnucash: startup-file is %s\n", startup_file);
  
  /* We can't just use gh_eval_str here because --startup-dir might have
     double quotes or other funny chars in it.  */
  
  gh_define("gnc:_startup-dir-default_", gh_str02scm(startup_dir));
  
  if(running_as_shell(gargc, gargv)) {
    gh_repl(argc, argv);
  } else {
    if(gh_eval_file(startup_file) == SCM_BOOL_F) {
      fprintf(stderr, "gnucash: (load %s) failed\n", startup_file);
      exit(1);
    }
  }
  exit(0);
}

void
gnc_shutdown(const int exit_status) {
  int handled_shutdown = 0;

  if(guile_is_initialized) {
    /* This should be gh_lookup, but that appears to be broken */
    SCM scm_shutdown = gh_eval_str("gnc:shutdown");

    if(gh_procedure_p(scm_shutdown)) {
      SCM scm_exit_code = gh_long2scm(exit_status);

      gh_call1(scm_shutdown, scm_exit_code);
      /* this will never execute */
      handled_shutdown = 1;
    }
  } else {
    _gnc_shutdown_(exit_status);
    exit(exit_status);
  }
}

void
_gnc_shutdown_(const int exit_status) {
  /* Put any C level routines that must be called at shutdown here
     gnc:shutdown will call this eventually...
  */

}

int
main(int argc, char *argv[])
{
  gargc = argc;
  gargv = argv;
  
  gh_enter(argc, argv, gnucash_main_helper);
  /* This will never return. */
  
  assert(0);  /* Just to be anal */
  return 0;   /* Just to shut gcc up. */
}
