/*******************************************************************\
 * RegWindow.c -- the register window for xacc (X-Accountant)       *
 * Copyright (C) 1997 Robin D. Clark                                *
 * Copyright (C) 1997, 1998 Linas Vepstas                           *
 *                                                                  *
 * This program is free software; you can redistribute it and/or    *
 * modify it under the terms of the GNU General Public License as   *
 * published by the Free Software Foundation; either version 2 of   *
 * the License, or (at your option) any later version.              *
 *                                                                  *
 * This program is distributed in the hope that it will be useful,  *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of   *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the    *
 * GNU General Public License for more details.                     *
 *                                                                  *
 * You should have received a copy of the GNU General Public License*
 * along with this program; if not, write to the Free Software      *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.        *
 *                                                                  *
 *   Author: Rob Clark                                              *
 * Internet: rclark@cs.hmc.edu                                      *
 *  Address: 609 8th Street                                         *
 *           Huntington Beach, CA 92648-4632                        *
\********************************************************************/


#include <Xm/Xm.h>
#include <Xm/DialogS.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/LabelGP.h>
#include <Xm/PanedW.h>
#include <Xm/PushB.h>
#include <Xm/RowColumn.h>

#include <Xm/Text.h>
#include <Xm/ToggleB.h>

#include "config.h"

#include "Account.h"
#include "AdjBWindow.h"
#include "BuildMenu.h"
#include "datecell.h"
#include "FileBox.h"
#include "Group.h"
#include "LedgerUtils.h"
#include "MainWindow.h"
#include "top-level.h"
#include "messages.h"
#include "MultiLedger.h"
#include "RecnWindow.h"
#include "RegWindow.h"
#include "SplitLedger.h"
#include "table-html.h"
#include "Transaction.h"
#include "util.h"
#include "xtutil.h"

/* The names of X11 resources for the different types of register
 * windows.  Used to set color resources, etc.
 * Must match the enums in register.h */
String accRes[] ={
  "regbank",
  "regcash",
  "regasset",
  "regcredit",
  "regliability",
  "regliability"
  "regincome",
  "regexpense",
  "regequity",
  "regstock",
  "ledgeneral",
  "ledincome",
  "ledportfolio"
};

/** STRUCTS *********************************************************/
/* The RegWindow struct contains info needed by an instance of an open 
 * register.  Any state info for the regWindow goes here. */

struct _RegWindow {
  xaccLedgerDisplay * ledger;
  DateCell *early_date_handler;
  DateCell *late_date_handler;
  int query_date;

  /* display widgets */
  Widget   dialog;
  Widget   reg;               /* The matrix widget...                    */
  Widget   balance;           /* The balance text field                  */
  Widget   date;              /* The date text field                     */
  Widget   record;            /* the record transaction button           */

};


/** GLOBALS *********************************************************/
extern Widget  toplevel;

/** PROTOTYPES ******************************************************/
RegWindow * regWindowLedger( Widget parent, xaccLedgerDisplay *);
static void regRefresh (xaccLedgerDisplay *ledger);
static void regDestroy (xaccLedgerDisplay *ledger);

static void closeRegWindow( Widget mw, XtPointer cd, XtPointer cb );
static void startRecnCB( Widget mw, XtPointer cd, XtPointer cb );
static void startAdjBCB( Widget mw, XtPointer cd, XtPointer cb );

static void recordCB( Widget mw, XtPointer cd, XtPointer cb );
static void deleteCB( Widget mw, XtPointer cd, XtPointer cb );
static void cancelCB( Widget mw, XtPointer cd, XtPointer cb );

static void reportCB( Widget mw, XtPointer cd, XtPointer cb );
static void webCB   ( Widget mw, XtPointer cd, XtPointer cb );

static void singleCB   ( Widget mw, XtPointer cd, XtPointer cb );
static void doubleCB   ( Widget mw, XtPointer cd, XtPointer cb );
static void multiCB    ( Widget mw, XtPointer cd, XtPointer cb );
static void dynosingCB ( Widget mw, XtPointer cd, XtPointer cb );
static void dynodoubCB ( Widget mw, XtPointer cd, XtPointer cb );

static void byDateCB ( Widget mw, XtPointer cd, XtPointer cb );
static void byNumCB  ( Widget mw, XtPointer cd, XtPointer cb );
static void byAmtCB  ( Widget mw, XtPointer cd, XtPointer cb );
static void byMemoCB ( Widget mw, XtPointer cd, XtPointer cb );
static void byDescCB ( Widget mw, XtPointer cd, XtPointer cb );

static void dateChangeCB( Widget mw, XtPointer cd, XtPointer cb );

/********************************************************************\
 * regWindowSimple                                                  *
 *   opens up a register window for Account account                 *
 *                                                                  *
 * Args:   parent  - the parent of this window                      *
 *         acc     - the account associated with this register      *
 * Return: regData - the register window instance                   *
\********************************************************************/
RegWindow *
regWindowSimple( Widget parent, Account *acc )
  {
  xaccLedgerDisplay * ledger;
  RegWindow *regData;

  ledger = xaccLedgerDisplaySimple (acc);
  regData = regWindowLedger (parent, ledger);

  return regData;
  }

/********************************************************************\
 * regWindowAccGroup                                                *
 *   opens up a register window for a group of Accounts             *
 *                                                                  *
 * Args:   parent  - the parent of this window                      *
 *         acc     - the account associated with this register      *
 * Return: regData - the register window instance                   *
\********************************************************************/
RegWindow *
regWindowAccGroup( Widget parent, Account *acc )
  {
  xaccLedgerDisplay * ledger;
  RegWindow *regData;

  ledger = xaccLedgerDisplayAccGroup (acc);
  regData = regWindowLedger (parent, ledger);

  return regData;
  }

/********************************************************************\
 * regWindowLedger                                                  *
 *   opens up a ledger window for the account list                  *
 *                                                                  *
 * Args:   parent   - the parent of this window                     *
 *         lead_acc - the account associated with this register     *
 *                     (may be null)                                *
 *         acc_list - the list of accounts to display in register   *
 *                     (may be null)                                *
 * Return: regData  - the register window instance                  *
\********************************************************************/
RegWindow *
regWindowLedger (Widget parent, xaccLedgerDisplay *ledger)
  {
  RegWindow   *regData = NULL;
  Widget menubar, pane, buttonform, frame, reg, widget;
  Dimension buttonform_height;
  int    position=0;
  char *windowname;
  char buf [BUFSIZE];

  /******************************************************************\
   * Set up the menubar menu-items.                                 *
   * Menu structures must be initialized before any code is         *
   * executed.  Some compilers insist on this, although gcc is      *
   * freindly about this.  Note that some of the activityMenu       *
   * values are changed below. Be careful with which row is which.  *
  \******************************************************************/
  MenuItem reportMenu[] = {
    { "To File",            &xmPushButtonWidgetClass, 'S', NULL, NULL, True,
      reportCB, (XtPointer)0,  (MenuItem *)NULL, 0 },
    { "To WWW",             &xmPushButtonWidgetClass, 'W', NULL, NULL, True,
      webCB,    (XtPointer)0,  (MenuItem *)NULL, 0 },
/*
    { SIMPLE_E_STR,         &xmPushButtonWidgetClass, 'S', NULL, NULL, True,
      NULL, (XtPointer)0,  (MenuItem *)NULL, 0 },
*/
    { NULL,                 NULL,                      0,  NULL, NULL, False,
      NULL, (XtPointer)0,  (MenuItem *)NULL, 0 },
  };
  
  MenuItem activityMenu[] = {
    { TRANSFER_E_STR,       &xmPushButtonWidgetClass, 'T', NULL, NULL, True,
      accountMenubarCB, (XtPointer)AMB_TRNS,   (MenuItem *)NULL, 0 },
    { "",                   &xmSeparatorWidgetClass,    0, NULL, NULL, True,
      NULL,              NULL,                 (MenuItem *)NULL, 0 },
    { RECONCILE_E_STR,      &xmPushButtonWidgetClass, 'C', NULL, NULL, True,
      startRecnCB,       NULL,                 (MenuItem *)NULL, 0 },
    { ADJ_BALN_E_STR,       &xmPushButtonWidgetClass, 'A', NULL, NULL, True,
      startAdjBCB,       NULL,                 (MenuItem *)NULL, 0 },
    { "Print...",         &xmPushButtonWidgetClass,   'R', NULL, NULL, True,
      NULL,              (XtPointer)0,         (MenuItem *)NULL, 0 },
/*
    { REPORT_E_STR,         &xmPushButtonWidgetClass, 'R', NULL, NULL, False,
      NULL,              (XtPointer)0,         (MenuItem *)NULL, 0 },
*/
    { "",                   &xmSeparatorWidgetClass,    0, NULL, NULL, True,
      NULL,              NULL,                 (MenuItem *)NULL, 0 },
    { DEL_TRANS_STR,        &xmPushButtonWidgetClass, 'D', NULL, NULL, True,
      deleteCB,          NULL,                 (MenuItem *)NULL, 0 },
    { "",                   &xmSeparatorWidgetClass,    0, NULL, NULL, True,
      NULL,              NULL,                 (MenuItem *)NULL, 0 },
    { CLOSE_WIN_STR,        &xmPushButtonWidgetClass, 'Q', NULL, NULL, True,
      destroyShellCB,    NULL,                 (MenuItem *)NULL, 0 },
    { NULL,                 NULL,                      0,  NULL, NULL, False,
      NULL,              (XtPointer)0,         (MenuItem *)NULL, 0 },
  };

  
  MenuItem helpMenu[] = {
    { ABOUT_E_STR,          &xmPushButtonWidgetClass, 'A', NULL, NULL, True,
      helpMenubarCB, (XtPointer)HMB_ABOUT, (MenuItem *)NULL, 0 },
    { HELP_E_STR,           &xmPushButtonWidgetClass, 'H', NULL, NULL, True,
      helpMenubarCB, (XtPointer)HMB_REGWIN,(MenuItem *)NULL, 0 },
    { "",                   &xmSeparatorWidgetClass,    0, NULL, NULL, True,
      NULL,          NULL,                 (MenuItem *)NULL, 0 },
    { LICENSE_E_STR,        &xmPushButtonWidgetClass, 'L', NULL, NULL, True,
      helpMenubarCB, (XtPointer)HMB_LIC,   (MenuItem *)NULL, 0 },
    { NULL,                 NULL,                      0,  NULL, NULL, False,
      NULL,          (XtPointer)0,         (MenuItem *)NULL, 0 },
  };
  

  regData = (RegWindow *) (ledger->gui_hook);
  if (regData) return (regData);

  regData = (RegWindow *) malloc (sizeof (RegWindow));

  ledger->gui_hook = (void *) regData;
  ledger->redraw = regRefresh;
  ledger->destroy = regDestroy;
  regData->ledger = ledger;

  regData->query_date = 1;
  regData->early_date_handler = xaccMallocDateCell();
  regData->late_date_handler = xaccMallocDateCell();

  /******************************************************************\
   * Start creating the Motif Widgets ...                           *
  \******************************************************************/

  /* pick a window name */
  if (ledger->leader) {
    char * acc_name = xaccAccountGetName (ledger->leader);
    switch (ledger->type) {
       case GENERAL_LEDGER:
       case INCOME_LEDGER:
         sprintf (buf, "%s General Ledger", acc_name);
         break;
       case PORTFOLIO:
         sprintf (buf, "%s Portfolio", acc_name);
         break;
       default:
         sprintf (buf, "%s Register", acc_name);
         break;
    }
    windowname = buf;
  } else {
    windowname = "General Ledger";
  }

  setBusyCursor( parent );

  regData->dialog =
    XtVaCreatePopupShell( "dialog", 
                          xmDialogShellWidgetClass, parent,
                          XmNdeleteResponse,   XmDESTROY,
                          XmNtitle,            windowname,
                          /*
                           * Let the window find it's own size, 
                           * based on the size of the fonts.
                           * XmNwidth,            395,
                           * XmNheight,           400,
                           * XmNminWidth,         495,
                           * XmNmaxWidth,         495,
                           * XmNminHeight,        500,
                           */
                          /* XmNresizable,        False, */
                          /* XmNallowShellResize, False, */
                          XmNtransient,        FALSE,  /* allow window to be repositioned */
                          XmNkeyboardFocusPolicy, XmEXPLICIT,  /* enable tab groups */
                          NULL );
  
  XtAddCallback( regData->dialog, XmNdestroyCallback, 
                 closeRegWindow, (XtPointer)regData );
  
  /* Create a PanedWindow Manager for the dialog box... the paned 
   * window is the parent of the two forms which comprise the two
   * areas of the dialog box */
  /* Important Note: the paned window MUST have traversal enabled,
   * otherwise the matrix cells will only get focus when the pointer
   * is in the cell, which basically defeats the whole idea of a tab 
   * group.  Put is another way: it is REALLY annoying to have to
   * put the mouse in the cell being edited. */
  pane = XtVaCreateWidget( "pane", 
                           xmPanedWindowWidgetClass, regData->dialog,
                           XmNsashWidth,     1,
                           XmNsashHeight,    1,
                           XmNseparatorOn,   False,
                           XmNtraversalOn,   True,
                           XmNmarginHeight,  1,
                           XmNmarginWidth,   1,
                           XmNallowResize,   True,
                           XmNpaneMaximum,   200,
                           XmNpaneMinimum,   800,
                           NULL );
  
  /******************************************************************\
   * Setup the menubar at the top of the window                     *
  \******************************************************************/

  /* Be careful not to scramble the order of the rows.  */
  reportMenu[0].callback_data=(XtPointer)regData;
  reportMenu[1].callback_data=(XtPointer)regData;

  activityMenu[2].callback_data=(XtPointer)regData;
  activityMenu[3].callback_data=(XtPointer)regData;
  activityMenu[6].callback_data=(XtPointer)regData;
  activityMenu[8].callback_data=(XtPointer)(regData->dialog);  /* destroy callback */

  activityMenu[4].subitems = reportMenu;

  /* can't adjust the balance on a ledger window */
  if (1 < ledger->numAcc) {
    activityMenu[2].sensitive = False;
    activityMenu[3].sensitive = False;
  }

  menubar = XmCreateMenuBar( pane, "menubar", NULL, 0 );  
  
  BuildMenu( menubar, XmMENU_PULLDOWN, ACTIVITIES_STR, 'A',
             False, 0, activityMenu );
  BuildMenu( menubar, XmMENU_PULLDOWN, HELP_STR,       'H', 
             False, 0, helpMenu );
  
  XtManageChild( menubar );
  
  frame = XtVaCreateWidget( "reg", 
                            xmFrameWidgetClass, pane,
                            NULL );
  
  /******************************************************************\
   * The main register window itself                                *
  \******************************************************************/

  /* The CreateTable will do the actual gui init, returning a widget */
  reg = xaccCreateTable (ledger->ledger->table, frame, 
        accRes[(ledger->type & REG_TYPE_MASK)]);
  regData->reg     = reg;

  /* be sure to initialize the gui elements associated with the cursor */
  xaccCreateCursor (ledger->ledger->table,  ledger->ledger->single_cursor);
  xaccCreateCursor (ledger->ledger->table,  ledger->ledger->double_cursor);
  xaccCreateCursor (ledger->ledger->table,  ledger->ledger->trans_cursor);
  xaccCreateCursor (ledger->ledger->table,  ledger->ledger->split_cursor);
    
  /* complete GUI initialization */
  {
    AccountGroup *grp;
    Account * base_acc;
    grp = xaccGetAccountRoot (ledger->leader);
    base_acc = ledger->leader;

    if (!grp) {
      grp = xaccGetAccountRoot (ledger->displayed_accounts[0]);
      base_acc = ledger->displayed_accounts[0];
    }
    xaccLoadXferCell (ledger->ledger->xfrmCell, grp, base_acc);
    xaccLoadXferCell (ledger->ledger->mxfrmCell, grp, base_acc);
    /* xaccLoadXferCell (ledger->ledger->xtoCell, grp);  */
  }

  XtManageChild (reg);
  XtManageChild (frame);
  XtManageChild (pane);


  /******************************************************************\
   * The button area... also contains balance fields                *
  \******************************************************************/
  
  buttonform = XtVaCreateWidget( "form", 
				 xmFormWidgetClass, pane,
				 XmNfractionBase,   22,
				 XmNresizable,      False,
                                 XmNtraversalOn,    True,
                                 XmNnavigationType, XmSTICKY_TAB_GROUP,
				 NULL );

  position = 0;                    /* puts the buttons in the right place */

  /* traverse to the buttons, when leaving the table */
  xaccNextTabGroup (ledger->ledger->table, buttonform); 
  
  /* The "Record" button */
  widget = XtVaCreateManagedWidget( RECORD_STR,
				    xmPushButtonWidgetClass, buttonform,
				    XmNtopAttachment,      XmATTACH_FORM,
				    XmNbottomAttachment,   XmATTACH_FORM,
				    XmNleftAttachment,     XmATTACH_POSITION,
				    XmNleftPosition,       position,
				    XmNrightAttachment,    XmATTACH_POSITION,
				    XmNrightPosition,      position+2,
				    XmNshowAsDefault,      True,
                                    XmNnavigationType,     XmTAB_GROUP, 
				    NULL );
  
  XtAddCallback( widget, XmNactivateCallback, 
		 recordCB, (XtPointer)regData );
  regData->record = widget;

  
  /* The "Cancel" button */
  position +=2;
  widget = XtVaCreateManagedWidget( CANCEL_STR, 
				    xmPushButtonWidgetClass, buttonform,
				    XmNtopAttachment,      XmATTACH_FORM,
				    XmNbottomAttachment,   XmATTACH_FORM,
				    XmNleftAttachment,     XmATTACH_POSITION,
				    XmNleftPosition,       position,
				    XmNrightAttachment,    XmATTACH_POSITION,
				    XmNrightPosition,      position+2,
				    XmNshowAsDefault,      True,
                                    XmNnavigationType,     XmEXCLUSIVE_TAB_GROUP, 
				    NULL );
  
  XtAddCallback( widget, XmNactivateCallback, 
                 cancelCB, (XtPointer)regData );
  
  /* the "close" button */
  position +=2;
  widget = XtVaCreateManagedWidget( CLOSE_STR, 
				    xmPushButtonWidgetClass, buttonform,
				    XmNtopAttachment,      XmATTACH_FORM,
				    XmNbottomAttachment,   XmATTACH_FORM,
				    XmNleftAttachment,     XmATTACH_POSITION,
				    XmNleftPosition,       position,
				    XmNrightAttachment,    XmATTACH_POSITION,
				    XmNrightPosition,      position+2,
				    XmNshowAsDefault,      True,
                                    XmNnavigationType,     XmEXCLUSIVE_TAB_GROUP, 
				    NULL );
  
  XtAddCallback( widget, XmNactivateCallback, 
                 destroyShellCB, (XtPointer)(regData->dialog) );
  
  /* Fix button area of the buttonform to its current size, and not let 
   * it resize. */
  XtVaGetValues( widget, XmNheight, &buttonform_height, NULL );
  XtVaSetValues( buttonform, XmNpaneMaximum, buttonform_height,
                             XmNpaneMinimum, buttonform_height,
                             NULL );
    
  position += 2;

  {
  MenuItem ledger_style_list[] = {

    { "Single Line", &xmPushButtonWidgetClass, 'S', NULL, NULL, True,
      singleCB, (XtPointer)regData,   (MenuItem *)NULL, 0 },
    { "Double Line", &xmPushButtonWidgetClass, 'D', NULL, NULL, True,
      doubleCB, (XtPointer)regData,   (MenuItem *)NULL, 0 },
    { "Multi Line",  &xmPushButtonWidgetClass, 'M', NULL, NULL, True,
      multiCB,  (XtPointer)regData,   (MenuItem *)NULL, 0 },
    { "Auto Single",   &xmPushButtonWidgetClass, 'A', NULL, NULL, True,
      dynosingCB, (XtPointer)regData, (MenuItem *)NULL, 0 },
    { "Auto Double",   &xmPushButtonWidgetClass, 'E', NULL, NULL, True,
      dynodoubCB, (XtPointer)regData, (MenuItem *)NULL, 0 },
    { NULL,                  NULL,                      0,  NULL, NULL, False,
      NULL, (XtPointer)0,  (MenuItem *)NULL, 0 },
  };
  
  widget = BuildMenu (buttonform, XmMENU_OPTION, 
                    "", 'F', False, 0, ledger_style_list);
  XtVaSetValues (widget,
				    XmNtopAttachment,      XmATTACH_FORM,
				    XmNbottomAttachment,   XmATTACH_FORM,
				    XmNleftAttachment,     XmATTACH_POSITION,
				    XmNleftPosition,       position,
				    XmNrightAttachment,    XmATTACH_POSITION,
				    XmNrightPosition,      position+4,
				    XmNshowAsDefault,      True,
                                    XmNshadowThickness, 0,
                                    NULL );

  XtManageChild(widget);
  }

  position += 4;
  {
  MenuItem ledger_style_list[] = {

    { "By Date",   &xmPushButtonWidgetClass, 'D', NULL, NULL, True,
      byDateCB, (XtPointer)regData,   (MenuItem *)NULL, 0 },
    { "By Number", &xmPushButtonWidgetClass, 'N', NULL, NULL, True,
      byNumCB, (XtPointer)regData,   (MenuItem *)NULL, 0 },
    { "By Amount",  &xmPushButtonWidgetClass, 'A', NULL, NULL, True,
      byAmtCB,  (XtPointer)regData,   (MenuItem *)NULL, 0 },
    { "By Memo",   &xmPushButtonWidgetClass, 'M', NULL, NULL, True,
      byMemoCB, (XtPointer)regData, (MenuItem *)NULL, 0 },
    { "By Desc",   &xmPushButtonWidgetClass, 'E', NULL, NULL, True,
      byDescCB, (XtPointer)regData, (MenuItem *)NULL, 0 },
    { NULL,                  NULL,                      0,  NULL, NULL, False,
      NULL, (XtPointer)0,  (MenuItem *)NULL, 0 },
  };
  
  widget = BuildMenu (buttonform, XmMENU_OPTION, 
                    "", 'S', False, 0, ledger_style_list);
  XtVaSetValues (widget,
				    XmNtopAttachment,      XmATTACH_FORM,
				    XmNbottomAttachment,   XmATTACH_FORM,
				    XmNleftAttachment,     XmATTACH_POSITION,
				    XmNleftPosition,       position,
				    XmNrightAttachment,    XmATTACH_POSITION,
				    XmNrightPosition,      position+4,
				    XmNshowAsDefault,      True,
                                    XmNshadowThickness, 0,
                                    NULL );

  XtManageChild(widget);
  }
  position += 4;
  
  /* ------------------------------------------------------------------- */
#ifdef NOT_ENOUGH_ROOM_FOR_THIS
  /* The date field labels: */ 
  widget = XtVaCreateManagedWidget( FROM_C_STR,
				    xmLabelGadgetClass,    buttonform,
				    XmNtopAttachment,      XmATTACH_FORM,
				    XmNleftAttachment,     XmATTACH_POSITION,
				    XmNleftPosition,       position,
				    XmNrightAttachment,    XmATTACH_POSITION,
				    XmNrightPosition,      position+1,
				    NULL );
  widget = XtVaCreateManagedWidget( TO_C_STR,
				    xmLabelGadgetClass,    buttonform,
				    XmNtopAttachment,      XmATTACH_WIDGET,
				    XmNtopWidget,          widget,
				    XmNbottomAttachment,   XmATTACH_FORM,
				    XmNleftAttachment,     XmATTACH_POSITION,
				    XmNleftPosition,       position,
				    XmNrightAttachment,    XmATTACH_POSITION,
				    XmNrightPosition,      position+1,
				    NULL );
  position++;
#endif /* NOT_ENOUGH_ROOM_FOR_THIS */
  
  /* and the date fields: */
  widget = XtVaCreateManagedWidget( "date",
				    xmTextWidgetClass,     buttonform,
				    XmNeditable,           True,
				    XmNverifyBell,         False,
				    XmNeditMode,           XmMULTI_LINE_EDIT,
				    XmNmarginHeight,       0,
				    XmNmarginWidth,        1,
				    XmNtopAttachment,      XmATTACH_FORM,
				    XmNbottomAttachment,   XmATTACH_FORM,
				    XmNleftAttachment,     XmATTACH_POSITION,
				    XmNleftPosition,       position,
				    XmNrightAttachment,    XmATTACH_POSITION,
				    XmNrightPosition,      position+3,
                                    XmNnavigationType,     XmNONE,  /* don't tab here! */
				    NULL );

  XtAddCallback( widget, XmNmodifyVerifyCallback, 
		 dateChangeCB, (XtPointer)regData );
  regData->date = widget;
  position +=3;
  
  /* ------------------------------------------------------------------- */
  /* The balance field labels: */ 
  widget = XtVaCreateManagedWidget( BALN_C_STR,
				    xmLabelGadgetClass,    buttonform,
				    XmNtopAttachment,      XmATTACH_FORM,
				    XmNleftAttachment,     XmATTACH_POSITION,
				    XmNleftPosition,       position,
				    XmNrightAttachment,    XmATTACH_POSITION,
				    XmNrightPosition,      position+2,
				    NULL );
  widget = XtVaCreateManagedWidget( CLEARED_C_STR,
				    xmLabelGadgetClass,    buttonform,
				    XmNtopAttachment,      XmATTACH_WIDGET,
				    XmNtopWidget,          widget,
				    XmNbottomAttachment,   XmATTACH_FORM,
				    XmNleftAttachment,     XmATTACH_POSITION,
				    XmNleftPosition,       position,
				    XmNrightAttachment,    XmATTACH_POSITION,
				    XmNrightPosition,      position+2,
				    NULL );
  position +=2;
  
  /* and the balance fields: */
  widget = XtVaCreateManagedWidget( "balance",
				    xmTextWidgetClass,     buttonform,
				    XmNeditable,           False,
				    XmNeditMode,           XmMULTI_LINE_EDIT,
				    XmNcursorPositionVisible, False,
				    XmNmarginHeight,       0,
				    XmNmarginWidth,        1,
				    XmNtopAttachment,      XmATTACH_FORM,
				    XmNbottomAttachment,   XmATTACH_FORM,
				    XmNleftAttachment,     XmATTACH_POSITION,
				    XmNleftPosition,       position,
				    XmNrightAttachment,    XmATTACH_POSITION,
				    XmNrightPosition,      position+3,
                                    XmNnavigationType,     XmNONE,  /* don't tab here! */
                                    XmNtraversalOn,        FALSE,   /* please no tab */
				    NULL );
  regData->balance = widget;
  
  XtManageChild(buttonform);
  
  /******************************************************************/
  XtManageChild(pane);
  
  ledger->dirty = 1;
  xaccLedgerDisplayRefresh (ledger);
  
  XtPopup( regData->dialog, XtGrabNone );
  
  unsetBusyCursor( parent );
  
  return regData;
}

/********************************************************************\
 * refresh only the indicated register window                       *
\********************************************************************/

static void 
regRefresh (xaccLedgerDisplay *ledger)
{
  RegWindow *regData = (RegWindow *) (ledger->gui_hook);
  xaccLedgerDisplay *ld = regData->ledger;
  char buf [BUFSIZE];
  char * amt;

  amt = xaccPrintAmount (ledger->balance, PRTSYM);
  strcpy (buf, amt);
  strcat (buf, "\n");
  amt = xaccPrintAmount (ledger->clearedBalance, PRTSYM);
  strcat (buf, amt);
  XmTextSetString( regData->balance, buf );

  if (regData->query_date) {
    time_t early, late;

    early = xaccQueryGetEarliestDateFound (ld->query);
    late = xaccQueryGetLatestDateFound (ld->query);
    xaccSetDateCellValueSecs (regData->early_date_handler, early);
    xaccSetDateCellValueSecs (regData->late_date_handler, late);

    strcpy (buf, regData->early_date_handler->cell.value);
    strcat (buf,"\n"); 
    strcat (buf, regData->late_date_handler->cell.value);
    XmTextSetString( regData->date, buf );
  }
}

/********************************************************************\
 * regDestroy()
 * It is enought to call just XtDestroy Widget.  Any allocated
 * memory will be freed by the close callbacks.
\********************************************************************/

static void
regDestroy (xaccLedgerDisplay *ledger)
{
   RegWindow *regData = (RegWindow *) (ledger->gui_hook);

   if (regData) XtDestroyWidget(regData->dialog);
}

/********************************************************************\
 * closeRegWindow                                                   *
 *   frees memory allocated for an regWindow, and other cleanup     *
 *   stuff                                                          *
 *                                                                  *
 * Args:   mw - the widget that called us                           *
 *         cd - regData - the data struct for this register         *
 *         cb -                                                     *
 * Return: none                                                     *
\********************************************************************/
static void 
closeRegWindow( Widget mw, XtPointer cd, XtPointer cb )
{
  RegWindow *regData = (RegWindow *)cd;

  xaccLedgerDisplayClose (regData->ledger);

  free(regData);
  DEBUG("closed RegWindow\n");
}

/********************************************************************\
 * startAdjBCB -- open up the adjust balance window... called       *
 *   from the menubar.                                              *
 *                                                                  *
 * Args:   mw - the widget that called us                           *
 *         cd - regData - the data struct for this register         *
 *         cb -                                                     *
 * Return: none                                                     *
\********************************************************************/
static void 
startAdjBCB( Widget mw, XtPointer cd, XtPointer cb )
{
  RegWindow *regData = (RegWindow *)cd;
  xaccLedgerDisplay *ledger = regData->ledger;
  Account *acc;
  
  /* Must have number of accounts be one.  If not one,
   * then this callback should never have been called,
   * since the menu entry is supposed to be greyed out.
   */
  if (ledger->leader) {
    acc = ledger->leader;
  } else {
    if (1 != ledger->numAcc) return;
    acc = ledger->displayed_accounts[0];
  }
  adjBWindow( toplevel, acc );
}

/********************************************************************\
 * startRecnCB -- open up the reconcile window... called from       *
 *   menubar.                                                       *
 *                                                                  *
 * Args:   mw - the widget that called us                           *
 *         cd - regData - the data struct for this register         *
 *         cb -                                                     *
 * Return: none                                                     *
\********************************************************************/
static void 
startRecnCB( Widget mw, XtPointer cd, XtPointer cb )
{
  RegWindow *regData = (RegWindow *)cd;
  xaccLedgerDisplay *ledger = regData->ledger;
  Account *acc;
  
  /* Must have number of accounts be one.  If not one,
   * then this callback should never have been called,
   * since the menu entry is supposed to be greyed out.
   */
  if (ledger->leader) {
    acc = ledger->leader;
  } else {
    if (1 != ledger->numAcc) return;
    acc = ledger->displayed_accounts[0];
  }
  recnWindow( toplevel, acc );
}

/********************************************************************\
 * recordCB                                                         *
 *                                                                  *
 * Args:   mw - the widget that called us                           *
 *         cd - regData - the data struct for this register         *
 *         cb -                                                     *
 * Return: none                                                     *
\********************************************************************/
static void
recordCB( Widget mw, XtPointer cd, XtPointer cb )
{
  RegWindow *regData = (RegWindow *)cd;
  xaccLedgerDisplay *ld = regData->ledger;
  SplitRegister *ledger = ld->ledger;
  Transaction *trans=NULL;
  
  xaccSRSaveRegEntry (ledger);

  /* Get the currently open transaction, commit it ..
   * and then repaint everything.  
   */
  trans = (Transaction *) (ledger->user_huck);
  if (trans) {
     xaccTransCommitEdit (trans);
     ledger->user_huck = NULL;
  }
  xaccSRRedrawRegEntry (ledger);
  xaccTransDisplayRefresh (trans);
  refreshMainWindow ();
}

/********************************************************************\
 * deleteCB                                                         *
 *                                                                  *
 * Args:   mw - the widget that called us                           *
 *         cd - regData - the data struct for this register         *
 *         cb -                                                     *
 * Return: none                                                     *
\********************************************************************/

static void
deleteCB( Widget mw, XtPointer cd, XtPointer cb )
{
  RegWindow *regData = (RegWindow *)cd;
  Split * split;
  Transaction *trans;
  char buf[BUFSIZE];
  int i, num_splits;
  Account **affected_accounts;
  
  /* get the current split based on cursor position */
  split = xaccSRGetCurrentSplit (regData->ledger->ledger);
  if (NULL == split ) return;

  /* ask for user confirmation before performing 
   * permanent damage */
  trans = xaccSplitGetParent (split);
  sprintf (buf, TRANS_DEL_MSG, xaccSplitGetMemo (split), xaccTransGetDescription (trans));
  if (!verifyBox (toplevel, buf)) return;

  /* make a copy of all of the accounts that will be  
   * affected by this deletion, so that we can update
   * thier register windows after the deletion.
   */
  num_splits = xaccTransCountSplits (trans);
  affected_accounts = (Account **) malloc ((num_splits+1) * sizeof (Account *));
  for (i=0; i<num_splits; i++) {
    split = xaccTransGetSplit (trans, i);
    affected_accounts[i] = xaccSplitGetAccount (split);
  }
  affected_accounts[num_splits] = NULL;

  xaccSplitDestroy (split);
  xaccAccListDisplayRefresh (affected_accounts);
  free (affected_accounts);
  refreshMainWindow ();
}

/********************************************************************\
 * cancelCB                                                         *
 *                                                                  *
 * Args:   mw - the widget that called us                           *
 *         cd - regData - the data struct for this register         *
 *         cb -                                                     *
 * Return: none                                                     *
\********************************************************************/

static void
cancelCB( Widget mw, XtPointer cd, XtPointer cb )
{
   RegWindow *regData = (RegWindow *)cd;
   xaccLedgerDisplay *ld = regData->ledger;
   SplitRegister *ledger = ld->ledger;
   Transaction *trans=NULL;
   Split * split;
   int i, num_splits=0, more_splits=0;
   Account **affected_accounts=NULL;

#ifdef CANCEL_ONLY_ONE_SPLIT_NOT_ENTIRE_TRANS
  /* when cancelling edits, reload the cursor from the transaction */
  split = xaccSRGetCurrentSplit (ledger);
  xaccSRLoadRegEntry (ledger, split);
  xaccRefreshTableGUI (ledger->table);
#endif /* CANCEL_ONLY_ONE_SPLIT_NOT_ENTIRE_TRANS */

#define CANCEL_THE_WHOLE_ENCHILADA
#ifdef CANCEL_THE_WHOLE_ENCHILADA
  /* Get the currently open transaction, rollback the edits on it,
   * and then repaint everything.  To repaint everything, 
   * make a note of all of the accounts that will be  
   * affected by this rollback.  Geez, there must be some easier
   * way of doing redraw notification ... 
   */
  trans = (Transaction *) (ledger->user_huck);
  if (trans) {
     num_splits = xaccTransCountSplits (trans);
     affected_accounts = (Account **) malloc ((num_splits+1) * sizeof (Account *));
     for (i=0; i<num_splits; i++) {
       split = xaccTransGetSplit (trans, i);
       affected_accounts[i] = xaccSplitGetAccount (split);
     }
     affected_accounts[num_splits] = NULL;
   
     /* rollback ... */
     xaccTransRollbackEdit (trans);
     
     /* and do some more redraw, for the new set of accounts .. */
     more_splits = xaccTransCountSplits (trans);
     affected_accounts = (Account **) realloc (affected_accounts, 
                              (more_splits+num_splits+1) * sizeof (Account *));
     for (i=0; i<more_splits; i++) {
       split = xaccTransGetSplit (trans, i);
       affected_accounts[i+num_splits] = xaccSplitGetAccount (split);
     }
     affected_accounts[num_splits+more_splits] = NULL;
   
     xaccAccListDisplayRefresh (affected_accounts);
     free (affected_accounts);
   
     ledger->user_huck = NULL;
  }
#endif /* CANCEL_THE_WHOLE_ENCHILADA */

  refreshMainWindow ();
}

/********************************************************************\
\********************************************************************/

static void
reportCB( Widget mw, XtPointer cd, XtPointer cb )
{
  RegWindow *regData = (RegWindow *)cd;
  char *outfile = fileBox( toplevel, OPEN_STR, "*.html" );  

  if (!outfile) return;
  xaccTablePrintHTML (regData->ledger->ledger->table, outfile);
}

static void
webCB( Widget mw, XtPointer cd, XtPointer cb )
{
  RegWindow *regData = (RegWindow *)cd;
  /* hack alert -- make the port number configureable */    
  xaccTableWebServeHTML (regData->ledger->ledger->table, 1080);
}

/********************************************************************\
\********************************************************************/

static void
styleCB( Widget mw, XtPointer cd, XtPointer cb, int what)
{
  RegWindow *regData = (RegWindow *)cd;
  xaccLedgerDisplay *ld = regData->ledger;
  SplitRegister *reg = ld->ledger;
  int typo = reg->type;

  typo &= ~REG_STYLE_MASK;
  switch (what) {
    case 1:
      typo |=  REG_SINGLE_LINE;
      break;
    case 2:
      typo |=  REG_DOUBLE_LINE;
      break;
    case 3: 
      typo |=  REG_MULTI_LINE;
      break;
    case 4:
      typo |=  REG_SINGLE_DYNAMIC;
      break;
    case 5:
      typo |=  REG_DOUBLE_DYNAMIC;
      break;
    default:
      return;
  }
  xaccConfigSplitRegister (reg, typo);
  ld->dirty = 1;
  xaccLedgerDisplayRefresh (ld);
}

static void singleCB( Widget mw, XtPointer cd, XtPointer cb )
{ 
  styleCB (mw, cd, cb, 1);
}
static void doubleCB( Widget mw, XtPointer cd, XtPointer cb )
{
  styleCB (mw, cd, cb, 2);
}
static void multiCB( Widget mw, XtPointer cd, XtPointer cb )
{
  styleCB (mw, cd, cb, 3);
}
static void dynosingCB( Widget mw, XtPointer cd, XtPointer cb )
{
  styleCB (mw, cd, cb, 4);
}
static void dynodoubCB( Widget mw, XtPointer cd, XtPointer cb )
{
  styleCB (mw, cd, cb, 5);
}

/********************************************************************\
\********************************************************************/

static void
sortCB( Widget mw, XtPointer cd, XtPointer cb, int what)
{
  RegWindow *regData = (RegWindow *)cd;
  xaccLedgerDisplay *ld = regData->ledger;

  xaccQuerySetMaxSplits (ld->query, MAX_QUERY_SPLITS);
  switch (what) {
    case 1:
      xaccQuerySetSortOrder (ld->query, BY_DATE, BY_NUM, BY_AMOUNT);
      break;
    case 2:
      xaccQuerySetSortOrder (ld->query, BY_NUM, BY_DATE, BY_AMOUNT);
      break;
    case 3: 
      xaccQuerySetSortOrder (ld->query, BY_AMOUNT, BY_DATE, BY_NUM);
      break;
    case 4:
      xaccQuerySetSortOrder (ld->query, BY_MEMO, BY_DATE, BY_NUM);
      break;
    case 5:
      xaccQuerySetSortOrder (ld->query, BY_DESC, BY_DATE, BY_NUM);
      break;
    default:
      return;
  }

  ld->dirty = 1;
  xaccLedgerDisplayRefresh (ld);
}

static void byDateCB( Widget mw, XtPointer cd, XtPointer cb )
{
  sortCB (mw, cd, cb, 1);
}

static void byNumCB( Widget mw, XtPointer cd, XtPointer cb )
{
  sortCB (mw, cd, cb, 2);
}

static void byAmtCB( Widget mw, XtPointer cd, XtPointer cb )
{
  sortCB (mw, cd, cb, 3);
}

static void byMemoCB( Widget mw, XtPointer cd, XtPointer cb )
{
  sortCB (mw, cd, cb, 4);
}

static void byDescCB( Widget mw, XtPointer cd, XtPointer cb )
{
  sortCB (mw, cd, cb, 5);
}

/********************************************************************\
\********************************************************************/

static void
dateChangeCB( Widget mw, XtPointer cd, XtPointer cb)
{
  RegWindow *regData = (RegWindow *)cd;
  xaccLedgerDisplay *ld = regData->ledger;
  XmTextVerifyCallbackStruct *tvcbs = (XmTextVerifyCallbackStruct *) cb;
  char *early_str, *late_str;
  char *oldval, *change, *newval, *retval, *retvaltoo;
  int early_len, adj_len, len;
  char buf [BUFSIZE];
  DateCell *handler;
  time_t early_time, late_time;

  tvcbs->doit = True;
  change = tvcbs->text->ptr;

  /* freaking text widget forces complex freakin logic on you. bummer. */
  early_str = regData->early_date_handler->cell.value;
  late_str = regData->late_date_handler->cell.value;

  /* see if there was no change */
  strcpy (buf, early_str); strcat (buf,"\n"); strcat (buf, late_str);
  if (change) {
     if (!strcmp (buf, change)) return;
  }
  regData->query_date = 0;

  /* determine if the change is in the early or the late string. */
  early_len = strlen (early_str);
  if (early_len > tvcbs->startPos) {
     oldval = early_str;
     adj_len = 0;
     handler = regData->early_date_handler;
  } else 
  if (early_len < tvcbs->startPos) {
     oldval = late_str;
     adj_len = early_len +1;
     handler = regData->late_date_handler;
  } else {
     /* attempt is made to edit out the newline ... */
     tvcbs->doit = False;
     return;
  }

  len = 1;
  len += strlen (oldval);
  if (change) len += strlen (change);
  newval = (char *) malloc (len);

  /* text may be inserted, or deleted, or replaced ... */
  /* compute the modified string (newval==string after edit) */
  newval[0] = 0;
  strncat (newval, oldval, (tvcbs->startPos)-adj_len);
  if (change) strcat (newval, change);
  strcat (newval, &oldval[((tvcbs->endPos)-adj_len)]);

  /* let the cell handler convert it into a valid date */
  retval = (char *) 
     ((*handler->cell.modify_verify) (&(handler->cell), oldval, change, newval));
  retvaltoo = (char *) ((*handler->cell.leave_cell) (&(handler->cell), retval));
  if (retval != newval) free (retval);
  tvcbs->doit = False;

  /* print it out ... post it into the display */
  if (early_len > tvcbs->startPos) {
     strcpy (buf, retvaltoo); strcat (buf,"\n"); strcat (buf, late_str);
  } else {
     strcpy (buf, early_str); strcat (buf,"\n"); strcat (buf, retvaltoo);
  }
  XmTextSetString( regData->date, buf );
  XtVaSetValues (regData->date, XmNcursorPosition, tvcbs->startPos, NULL);
  free (newval);
  free (retvaltoo);

  /* OK, now use these new values to control the displayed splits */
  early_time = mktime (&(regData->early_date_handler->date));
  late_time  = mktime (&(regData-> late_date_handler->date));

  xaccQuerySetDateRange (ld->query, early_time, late_time);

  /* don't let max-number of splits to display to interfere with the date 
   * fiddling; so unclamp it (ok, clamp at 1000 which should be acceptable!?!)
   */
  xaccQuerySetMaxSplits (ld->query, MAX_QUERY_SPLITS_UNCLAMP);
  
  /* refresh the display */
  ld->dirty = 1;
  xaccLedgerDisplayRefresh (ld);
}

/************************** END OF FILE *************************/
