/* Copyright 1994 GROUPE BULL -- See license conditions in file COPYRIGHT */

/* $Id: agenda.c,v 1.1.1.1 1996/04/16 15:12:03 leon Exp $ */

#include <stdio.h>
#include <string.h>
#ifndef Mips
#include <malloc.h>
#endif /* ! Mips */
#include "pstructs.h"
#include "agenda.h"
#include "cm.h"
#include "misc.h"


Agenda
AgendaCreate(char *user)
{
    Agenda ag;
    int i;

    ag = Create(AgendaRec);
    ag->user = Strdup(user);
    for(i = 0; i < MAXCAL; i++) {
	ag->calendars[i] = NULL;
    }
    ag->numCals = 0;
    return ag;
}





int
AgendaOpenCalendar(Agenda ag, char *name)
{
    int i;
    char filename[256];
    
    for(i = 0; i < MAXCAL; i++) {
	if(NULL == ag->calendars[i]) {	    
	    sprintf(filename, "%s/.%s/%s", getenv("HOME"), APP_NAME, name);
	    ag->calendars[i] = ClOpen(filename, name);
	    if(NULL != ag->calendars[i])
		ag->numCals++;
	    return (NULL != ag->calendars[i]);
	}
    }
    fprintf(stderr, "sorry: not enough slots in this agenda.\n");
    return 0;
}



AppointmentId
AgendaRegisterAppointment(Agenda ag,
			  char *calName, 
			  Appointment ap,
			  CalendarId *cid)
{
    int i;
    
    for(i = 0; i < MAXCAL; i++) {
	if(NULL == ag->calendars[i]) continue;
	if(!strcmp(ag->calendars[i]->name, calName)) {
	    *cid = i;
	    return
		ClAddAppointment(ag->calendars[i], ap);
	}
    }
    return 0;
}


/* closure->p1 : the user proc, it is != NULL
   closure->p2 : the calendar slot
   closure->p3 : the user closure
   */
void
CallUserIterator(Calendar cal,
		 AppointmentId id, 
		 void *closure)
{
    P3 *ptrs = (P3 *)closure;
    AgendaIteratorProc proc = (AgendaIteratorProc)ptrs->p1;
    int cid = (int)ptrs->p2;

    (*proc)(id, cid, ptrs->p3);

}



void
AgendaEnumerateAppointments(Agenda ag, 
			    Date date, 
			    AgendaIteratorProc proc, 
			    void *closure)
{
    int i;
    P3 ptrs;
    
    if(NULL == proc) return;
    ptrs.p1 = (void *)proc;
    ptrs.p3 = (void *)closure;
    for(i = 0; i < MAXCAL; i++) {
	if((ag->calendars[i] != NULL) &&(ag->calendars[i]->selected)){
	    ptrs.p2 = (void *)i;
	    ClEnumerateAppointments(ag->calendars[i], date, 
				    CallUserIterator, &ptrs);
	}
    }
}



void
AgendaEnumerateAnyAppointments(Agenda ag, 
			    AgendaIteratorProc proc, 
			    void *closure)
{
    int i;
    P3 ptrs;
    
    if(NULL == proc) return;
    ptrs.p1 = (void *)proc;
    ptrs.p3 = (void *)closure;
    for(i = 0; i < MAXCAL; i++) {
	if((ag->calendars[i] != NULL) &&(ag->calendars[i]->selected)){
	    ptrs.p2 = (void *)i;
	    ClEnumerateAnyAppointments(ag->calendars[i], 
				       CallUserIterator, &ptrs);
	}
    }
}



void
AgendaSave(Agenda ag)
{
    int i;
    char filename[256];

    for(i = 0; i < MAXCAL; i++) {
	if(ag->calendars[i] != NULL) {
	    sprintf(filename, "%s/.%s/%s", getenv("HOME"), APP_NAME, ag->calendars[i]->name); 
	    ClSave(ag->calendars[i], filename);
	}
    }
}







int
AgendaFindDayAppointment(Agenda ag, Date date)
{
    int i;

    for(i = 0; i < MAXCAL; i++) {
	if((ag->calendars[i] != NULL) && (ag->calendars[i]->selected)){
	    if(ClFindDayAppointment(ag->calendars[i], date)) {
		return 1;
	    }
	}
    }
    return 0;
}




void
AgendaRemoveAppointment(Agenda ag, CalendarId cal, AppointmentId id)
{
    ClRemoveAppointment(ag->calendars[cal], id);
}




void
AgendaSetCalendarStatus(Agenda ag, char *name, Bool status)
{
    int i;

    for(i = 0; i < MAXCAL; i++) {
	if(ag->calendars[i] != NULL) {
	    if(!strcmp(ag->calendars[i]->name, name)) {
		ag->calendars[i]->selected = status;
		return;
	    }
	}
    }
}


char *
AgendaGetCalendarName(Agenda ag, CalendarId cal)
{
    return ag->calendars[cal]->name;
}



static void
ProcessMessage(CmGroup group, char *header, char *msg, void *closure)
{
    Calendar cal = (Calendar)closure;
    Appointment ap;
    AppointmentRec apr;

    if(!strncmp(header, "ADD", 3)) {
	/* should not appear */
#ifdef Debug
	fprintf(stderr, "received ADD message: [%s] %s\n", header, msg);
#endif /* Debug */
	AppointmentSscan(msg, &apr);
	if(!ClFindAppointment(cal, apr.user, apr.timestamp)) {
	    ap = AppointmentCreate();
	    *ap = apr;
	    ClRegisterAppointment(cal, ap);
	}
    }
    if(!strncmp(header, "DEL", 3)) {
#ifdef Debug
	fprintf(stderr, "received DEL message: [%s] %s\n", header, msg);
#endif /* Debug */
	
	AppointmentSscan(msg, &apr);
	/* strings allocated here are lost */
	if((ap=ClFindAppointment(cal, apr.user, apr.timestamp)))
	    ClDestroyAppointment(cal, ap);
    }
}



int
AgendaUpdate(Agenda ag)
{
    int i;
    int res = 0;

    for(i = 0; i < MAXCAL; i++) {
	if(ag->calendars[i] != NULL) {
	    res |= CmScanArrivals(ag->calendars[i]->group, 
				   ProcessMessage, ag->calendars[i]);
	}
    }
    return res;
}




Calendar
AgendaGetCalendar(Agenda ag, char *name)
{
    int i;
    for(i = 0; i < MAXCAL; i++) {
	if(ag->calendars[i] != NULL) {
	    if(!strcmp(ag->calendars[i]->name, name))
		return ag->calendars[i];
	}
    }
    return NULL;
}



void
AgendaEnumerateCalendars(Agenda ag, EnumerateCalendarsProc proc, void *closure)
{
    int i;
    for(i = 0; i < MAXCAL; i++) {
	if(ag->calendars[i] != NULL) {
	    (*proc)(ag->calendars[i], closure);
	}
    }
}
