/*****************************************************************************

	smterm.c

	Environment:    Unix R40V3/Solaris2/Linux.

	Revision history:	@(#)smterm.c	1.1     97/06/23

	DESCRIPTION: Smterm. Experimental pseudo tty terminal.
	   Create a dumb terminal running in an XmText window.
	   Create a ListTree widget to navigate the child (shell)
	   through out the file system.
	   Perform actions on selected items (files) as defined
	   in the resource file (see fall_back_res in smterm.h).

	CAVEATS: The Auto/NoAuto button has to be toggled manually
	   if selections are made and if the shell is running its own
	   child. This is to prevent the application to keep issuing
	   shell commands when selections are made.

	SEE ALSO: ListTree - Copyright (c) 1995 Robert W. McMullen.

        COPYRIGHT NOTICE:
        Permission to use,  copy,  modify,  and  distribute  this
        software  and  its    documentation   is  hereby  granted
        without fee, provided that  the  above  copyright  notice
        appear  in all copies and that both that copyright notice
        and  this  permission   notice   appear   in   supporting
        documentation.  The   author  makes  no   representations
        about   the   suitability   of   this  software  for  any
        purpose.  It  is  provided  "as  is"  without  express or
        implied warranty.

        THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD  TO  THIS
        SOFTWARE,    INCLUDING    ALL   IMPLIED   WARRANTIES   OF
        MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL THE AUTHOR
        BE  LIABLE  FOR  ANY  SPECIAL,  INDIRECT OR CONSEQUENTIAL
        DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS  OF
        USE, DATA OR PROFITS, WHETHER IN AN ACTION  OF  CONTRACT,
        NEGLIGENCE  OR  OTHER  TORTIOUS   ACTION,   ARISING   OUT
        OF   OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
        SOFTWARE.

******************************************************************************/
/******************************************************************************/
#pragma ident "@(#)smterm.c      1.1		97/06/23"

#include "mdb.h"
#include "smterm.h"

#ifdef TTYMON

/*
 * The Main Action Area:
 *      LabelString:            Callback:       cbValue:        Sens:   Help:
 */
static ActionAreaItem action_items[] = {
	{ "makeButton",		(XtCP)main_cb,	MAKE,		0,	NULL },
	{ "editButton",		(XtCP)main_cb,	EDIT,		0,	NULL },
	{ "printButton",	(XtCP)main_cb,	PRINT,		0,	NULL },
	{ "srcsButton",		(XtCP)main_cb,	SRCS,		0,	NULL },
	{ "noAutoButton",	(XtCP)main_cb,	AUTO,		0,	NULL },
	{ "newButton",		(XtCP)main_cb,	NEW,		0,	NULL },
	{ "cancelButton",	(XtCP)main_cb,	QUIT,		0,	NULL },
};

static Widget	Cmd_w;
static char	Cmd[200];
static char	*TempFile;
static char	*StartDir;
static char	*LsCmd;
static Boolean	Autols = False;
static Boolean	MatchDots = True;
static Boolean	Auto = True;
static CFILE 	*Cfd;


/*
 * Start of smterm.
 */
/*ARGSUSED*/
void main( int argc, char *argv[] )
{

	Widget paned_w;
	Widget tree_w, paned1_w;
	Atom wm_close;
	char cmd[200];
	char *ptr;
	XmTextScanType sType[4];
	int i;

#if defined(SVR4) || defined(linux)
	(void)setlocale( LC_ALL, "" );
#endif

	*Cmd = *cmd = '\0';

	for ( i = 0; i < argc; i++ ) {
		/*
		 * Save args for the "new" action.
		 */
		(void)strcat( Cmd, argv[i] );
		(void)strcat( Cmd, " " );
	}

	for ( i = 1; i < argc; i++ ) {

		/*
		 * Xoptions first, then collect
		 * args for the child process.
		 */
		if ( (*argv[i] == '-') && (*cmd == '\0') ) {
			i++;
			continue;
		}

		(void)strcat( cmd, argv[i] );
		(void)strcat( cmd, " " );
	}

	if ( ! *cmd ) {
		if ( (ptr = getenv( "SHELL" )) == NULL ) {
			(void)fprintf( stderr,
			       "usage: %s -xoptions shell args\n", argv[0] );
			exit(1);
		}
		(void)strcpy( cmd, ptr );
	}

	StartDir = getcwd( NULL, PATH_MAX+1 );
	(void)strcat( StartDir, "/" );


	Top = XtVaAppInitialize( &App, Class, NULL, 0,
		&argc, argv, fall_back_res,
		XmNdeleteResponse, XmDO_NOTHING,
		NULL);

	XtAppAddActions( App, actions, XtNumber(actions) );

	wm_close = XmInternAtom( XtDisplay(Top), "WM_DELETE_WINDOW", False );
	XmAddWMProtocolCallback( Top, wm_close, (XtCP)main_cb,(XtPointer)QUIT );

	paned_w = XmCreatePanedWindow( Top, "panedWin", NULL, 0 );

	paned1_w = XmCreatePanedWindow( paned_w, "panedWin1", NULL, 0 );
	XtManageChild(paned1_w);

	Cmd_w = XmCreateScrolledText( paned1_w, "cmdText", NULL, 0 );

	sType[0] = XmSELECT_POSITION;
	sType[1] = XmSELECT_WORD;
	sType[2] = XmSELECT_LINE;
	XtVaSetValues( Cmd_w,
			XmNselectionArray, sType,
			XmNselectionArrayCount, 3,
			NULL );

	XtOverrideTranslations( Cmd_w,
			XtParseTranslationTable(extraTranslations) );

	XtManageChild(Cmd_w);

	if ( (Cfd = attachChild( Top, cmd, NULL, True, Cmd_w )) == NULL ) {
		perror( "attachChild" );
		exit(1);
	}

	TempFile = GetTempFile(1);

	if ( (ptr = strchr( cmd, ' ' )) != NULL ) *ptr = '\0';

	XtVaSetValues( Top, XtNtitle, cmd,  NULL );

	tree_w = XmCreateScrolledListTree( paned_w, "listTree", NULL, 0 );

	XtManageChild(tree_w);

	/*
	 * Run ls each time we open up a
	 * new directory through listTree.
	 */
	if ( (ptr = GetRes( "*autoLs" )) != NULL ) {
		if ( !strcmp( "rue", ++ptr ) )
			Autols = True;
	}
	if ( (LsCmd = GetRes( "*lsCmd" )) == NULL )
		LsCmd = "ls -xF";

	/*
	 * Show hidden files.
	 */
	if ( (ptr = GetRes( "*matchDots" )) != NULL ) {
		if ( !strcmp( "rue", ++ptr ) )
			MatchDots = False;
	}

	/*
	 * Set up the root tree.
	 */
	if ( scanDirs( tree_w, "/", NULL, True ) != -1 ) {
		XtAddCallback( tree_w,
			XtNactivateCallback, (XtCP)activate_cb,
			(XtPointer)NULL );
	}

	/*
	 * These actions are meaningful 
	 * only if the command is a shell,
	 * That is, sh, tcsh etc.
	 */
	(void)CreateActionArea( paned_w, action_items,
		XtNumber(action_items), NULL, NULL, NULL, False );

	XtManageChild(paned_w);

	main_cb( NULL, INIT, NULL );

	XtRealizeWidget(Top);
	XtAppMainLoop(App);
}


/*
 * Flush any pending data
 * on cfd and move  to a
 * new line at end.
 */
static void newLine(void)
{
	XmTextPosition pos;

	(void)fflush( Cfd->fd );

	Cfd->echo = False;
	pos = XmTextGetLastPosition( Cfd->w );
	XmTextInsert( Cfd->w, pos, " \n" );
	XmTextSetInsertionPosition( Cfd->w, pos+1 );
	Cfd->currInsert = pos+1;
	Cfd->echo = True;
}


/*
 * Called internally fromm X-lib.
 */
/*ARGSUSED*/
static XtInputCallbackProc
doFileAction( char *sel, int *fid, XtInputId *inputId )
{

	struct stat sbuf;
	char dbuff[PATH_MAX];
	char fileActionX[80];
	char pp[100];
	char *ptr, *ext, *fact, *cmd;
	Boolean bg;
	int i;

	(void)fstat( *fid, &sbuf );
	if ( !sbuf.st_size )
		return(0);

	/*
	 * Read the temp file with the
	 * shell's current working directory.
	 */
	(void)read( *fid, dbuff, sizeof(dbuff) );
	XtRemoveInput(*inputId);
	(void)close(*fid);

	if ( (ptr = strchr( dbuff, '\n' )) != NULL )
		*ptr = '\0';
	if ( (ptr = strrchr( sel, '*' )) != NULL )	/* ls -xF */
		*ptr = '\0';
	if ( (ptr = strrchr( sel, ':' )) != NULL )	/* file xx */
		*ptr = '\0';

	/*
	 * Cat so that we get a qualified
	 * path to the file item clicked on.
	 */
	(void)strcat( dbuff, "/" );
	(void)strcat( dbuff, sel );

	/*
	 * Stat the the selected file.
	 */
	if ( stat( dbuff, &sbuf ) < 0 )
		return(0);

	if ( ((sbuf.st_mode & S_IEXEC) == S_IXUSR) &&
				((sbuf.st_mode & S_IFMT ) != S_IFDIR) ) {
		/*
		 * Execute program.
		 */
		newLine();
		fprintfChild( Cfd, "%s\n", sel );
		XmTextClearSelection( Cmd_w, CurrentTime );
		XtFree( sel );
		return(0);
	}

	if ( (sbuf.st_mode & S_IFMT) == S_IFDIR ) {

		/*
		 * Change directory.
		 */
		newLine();
		fprintfChild( Cfd, "cd \"%s\"\n", sel );
		XmTextClearSelection( Cmd_w, CurrentTime );
		XtFree( sel );

		if ( Autols == True )
			fprintfChild( Cfd, "%s\n", LsCmd );
		return(0);
	}

	/*
	 * Run actions based on file extension.
	 */
	if ( (ext = strrchr( sel, '.' )) == NULL )
		return(0);

	/*
	 * Get the resource file's
	 * defined file acions.
	 */
	i = 0; *pp = '\0'; bg = False;

	while( (fact = GetRes( "*fileAction-%d", i++ )) != NULL ) {

		(void)strcpy( fileActionX, fact );

		if ( (cmd = strchr( fileActionX, ' ' )) == NULL )
			continue;

		*cmd++ = '\0';
		if ( ! *cmd )
			continue;

		if ( !strcmp( fileActionX, ext ) ) {

			newLine();

			if ( cmd[strlen(cmd)-1] == '&' ) {
				/*
				 * Run in the background.
				 */
				bg = True;
				cmd[strlen(cmd)-1] = '\0';
			} else
				bg = False;

			if ( (ptr = strchr( cmd, '|' )) != NULL ) {
				/*
				 * Command is piped.
				 */
				(void)strcpy( pp, ptr );
				*ptr = '\0';
			}

			if ( bg == True ) {
				fprintfChild( Cfd, "%s %s %s &\n",
						cmd, sel, pp );
			} else {
				fprintfChild( Cfd, "%s %s %s\n",
						cmd, sel, pp );
			}

			XmTextClearSelection( Cmd_w, CurrentTime );
			XtFree( sel );
			break;
		}
	}

	return(0);
}


/*
 * Perform action on selected file.
 */
/*ARGSUSED*/
static void initFileAction( char *sel )
{
	int fd;

	/*
	 * Since we may not be in the
	 * same working directory as 
	 * the child task (the shell),
	 * we have to ask by issuing
	 * a `pwd' shell command.
	 */

	if ( (fd = open( TempFile, O_CREAT | O_TRUNC, 0600 )) < 0 )
		return;

	/*
	 * Let the X-library monitor
	 * the completion of the `pwd'
	 * output from the shell.
	 */
	(void)XtAppAddInput( GetAppContext(), fd, (XtPointer)XtInputReadMask,
		(XtInputCallbackProc)doFileAction, (XtPointer)sel );

	/*
	 * Do not show the  shell's two line
	 * response to this. You will find
	 * in in the shell's history though.
	 */
	Cfd->noResponse = 2;

	fprintfChild( Cfd, "pwd > %s\n", TempFile );
}


/*
 * Main Button Callbacks.
 */
/*ARGSUSED*/
void main_cb( Widget w, int client_data, XmPushButtonCallbackStruct *cbs )
{

	static char *selected;
	static Boolean ppend;
	char dbuff[200];

	xpmsg_close();

	if ( (client_data == QUIT) || (client_data == ABORT) ) {
		detachChild(Cfd);
		RemoveTempFiles();
		exit(client_data);
	}

	if ( XmTextGetLastPosition( Cmd_w ) > 40000 ) {
		/*
		 * Max history.
		 */
		Cfd->recur = True;
		XmTextReplace( Cmd_w, 0, 30000, ""  );
		Cfd->recur = False;
		Cfd->currInsert -= 30000;
	}

	if ( (client_data == SELECT) &&  (Auto == True) ) {
		XmTextPosition l, r;

		XmTextGetSelectionPosition( Cmd_w, &l, &r );

		if ( r > XmTextGetLastPosition(Cmd_w)-2 ) {
			XmTextClearSelection( Cmd_w, CurrentTime );
			selected = NULL;
			return;
		}

		if ( (selected = XmTextGetSelection(Cmd_w)) != NULL ) {
			Boolean multi;

			multi = False;

			for ( l = 0; l < strlen(selected); l++ ) {
				if ( selected[l] == '\n' )
					selected[l] = ' ';

				if ( selected[l] == ' ' )
					multi = True;
			}

			if ( multi == False ) {
				/*
				 * Check for to rapid clicking.
				 */
				if ( ! Cfd->noResponse )
					(void)initFileAction( selected );
			}

		} else
			XmTextClearSelection( Cmd_w, CurrentTime );

		return;
	}

	/*
	 * Built in Button Actions.
	 */

	if ( client_data == MAKE ) {
		char *sel;

		sel = selected == NULL ? "" : selected;

		fprintfChild( Cfd, "make %s\n", sel );
		if ( selected != NULL ) {
			XtFree(selected);
			selected = NULL;
			XmTextClearSelection( Cmd_w, CurrentTime );
		}
		return;
	}

	if ( client_data == EDIT ) {
		char *ed;
		char *sel;

		sel = selected == NULL ? "" : selected;

		if ( (ed = getenv( "XEDITOR" )) == NULL ) {
			errno = 0;
			(void)xpmsg( Top, "error: No XEDITOR in environment" );
			return;
		}

		fprintfChild( Cfd, "%s %s &\n", ed, sel );

		if ( selected != NULL ) {
			XtFree(selected);
			selected = NULL;
			XmTextClearSelection( Cmd_w, CurrentTime );
		}
		return;
	}

	if ( client_data == SRCS ) {
		fprintfChild( Cfd, "%s *.C *.c *.h\n", LsCmd );
		return;
	}

	if ( client_data == PRINT ) {

		char *lp;
		char *mpp;

		if ( (lp = GetPrinter()) == NULL ) {
			errno = 0;
			(void)xpmsg( Top, "error: no default printer" );
			return;
		}

		if ( selected == NULL ) {
			errno = 0;
			(void)xpmsg( Top, "error: nothing selected" );
			return;
		}

		if ( strchr( selected, ' ' ) != NULL ) {
			errno = 0;
			(void)xpmsg( Top, "error: single select only" );
			return;
		}

		if ( (mpp = GetRes( MPPAPER )) == NULL )
			mpp = MPA4;

		(void)sprintf( dbuff, "mp -s %s", selected );
		PrintCommand( Top, MPPRINT, lp, selected,
				dbuff, mpp, 1, &ppend );

		XtFree(selected);
		selected = NULL;
		XmTextClearSelection( Cmd_w, CurrentTime );
		return;
	}

	if ( client_data == NEW ) {

		(void)chdir( StartDir );

		if ( fork() == (pid_t)0 ) {
			execl( "/bin/sh", "sh", "-c", Cmd, (char *)0 );
			_exit(1);
		}
	}

	if ( client_data == AUTO ) {
		/*
		 * Toggle automod. This is
		 * typically done when the
		 * shell has its own child
		 * running on top of the 
		 * tty session.
		 */
		char *ptr;
		XmString str;

		if ( Auto == True ) {
			if ((ptr = GetRes("*autoButton.labelString")) == NULL)
				return;
			str = XmStringCreateLtoR(ptr, XmSTRING_DEFAULT_CHARSET);
			XtVaSetValues( w, XmNlabelString,  str, NULL );
			XmStringFree(str);
			Auto = False;
		} else {
			if ((ptr = GetRes("*noAutoButton.labelString")) == NULL)
				return;
			str = XmStringCreateLtoR(ptr, XmSTRING_DEFAULT_CHARSET);
			XtVaSetValues( w, XmNlabelString,  str, NULL );
			XmStringFree(str);
			Auto = True;
		}
		return;

	}

	if ( client_data == INIT ) {
		selected = NULL;
	}
}


/* Extra actions primarily to handle
 * history browsing in shells, gdb etc.
 */
/*ARGSUSED*/
XtActionProc
keyUp( Widget w, XEvent *event, String *args, Cardinal *num_args)
{
	fprintfChild( Cfd, "%c%cA", 0x1b, 0x5b );
	return(0);
}

/*ARGSUSED*/
XtActionProc
keyDown( Widget w, XEvent *event, String *args, Cardinal *num_args)
{
	fprintfChild( Cfd, "%c%cB", 0x1b, 0x5b );
	return(0);
}

/*ARGSUSED*/
XtActionProc
keyBackSpace( Widget w, XEvent *event, String *args, Cardinal *num_args)
{
	fprintfChild( Cfd, "%c", ERASE );
	return(0);
}

/*ARGSUSED*/
XtActionProc
keyLeft( Widget w, XEvent *event, String *args, Cardinal *num_args)
{
	fprintfChild( Cfd, "%c%cD", 0x1b, 0x5b );
	return(0);
}

/*ARGSUSED*/
XtActionProc
keyRight( Widget w, XEvent *event, String *args, Cardinal *num_args)
{
	fprintfChild( Cfd, "%c%cC", 0x1b, 0x5b );
	return(0);
}

/*ARGSUSED*/
XtActionProc
keyEsacpe( Widget w, XEvent *event, String *args, Cardinal *num_args)
{
	fprintfChild( Cfd, "%c", 0x1b );
	return(0);
}

/*ARGSUSED*/
XtActionProc
btn1Up( Widget w, XEvent *event, String *args, Cardinal *num_args)
{

	if ( Auto == False ) {
		XmTextSetInsertionPosition( Cmd_w, Cfd->currInsert );
		return(0);
	}

	if ( Cfd->currInsert == XmTextGetLastPosition(Cmd_w) )
		main_cb( w, SELECT, NULL );
	else {
		/*
		 * The current line is being edited. Ignore.
		 */
		XmTextClearSelection( Cmd_w, CurrentTime );
	}

	XmTextSetInsertionPosition( Cmd_w, Cfd->currInsert );

	return(0);
}

/*ARGSUSED*/
XtActionProc
keyNop( Widget w, XEvent *event, String *args, Cardinal *num_args)
{
	return(0);
}


/*
 * ListTree callback.
 */
/*ARGSUSED*/
static void activate_cb( Widget w, XtPointer client_data, XtPointer cbs )
{
	ListTreeActivateStruct *ret = cbs;
	int i;
	char dir[PATH_MAX];
	char ndir[PATH_MAX];
	Boolean curdir = False;

	xpmsg_close();

	(void)strncpy( dir, ret->item->text, ret->item->length );
	dir[ret->item->length] = '\0';

	/*
	 * A click on the Dot child will
	 * perform a ls (if true) only.
	 */
	if ( (*dir == '.') && (ret->item->length == 1) )
		curdir = True;

	if ( ret->open == False ) {
		if ( curdir == False ) {
			/*
			 * A directory is about to be
			 * closed. Delete all children
			 * and restore the Dot child
			 * with the true direcory name.
			 */
			(void)ListTreeDeleteChildren( w, ret->item );
			(void)ListTreeAdd( w, ret->item, dir );
			return;
		} else if ( Autols == False )
			return;
	}

	bzero( ndir, sizeof(ndir) );
	*ndir = '/';

	/*
	 * Collect to a qualified path.
	 */
	for( i = 0; i < (curdir == True ? ret->count-1 : ret->count); i++ ) {
		(void)strncat( ndir, ret->path[i]->text, ret->path[i]->length );
		(void)strcat( ndir, "/" );
	}

	if (  curdir == False ) {
		/*
		 * Rename the single dir
		 * child to a Dot child.
		 */
		(void)ListTreeRenameItem( w,
			ListTreeFindChildName( w, ret->item, dir ), "."  );
	}

	if ( Auto == True ) {
		if ( chdir( ndir ) < 0 ) {
			(void)xpmsg( Top, "error: %s", ndir );
			return;
		}

		fprintfChild( Cfd, "cd \"%s\"\n", ndir );

		if ( Autols == True )
			fprintfChild( Cfd, "%s\n", LsCmd );
	}

	/*
	 * Add subdirs to a
	 * just opened parent.
	 */
	if (  curdir == False )
		(void)scanDirs( w, ndir, ret->item, False );
}



/*
 * Used by qsort.
 */
/*ARGSUSED*/
static int compar( char **s1, char **s2 )
{
        return( strcmp( *s1, *s2 ) );
}


/*
 * Scan a directory and add
 * childs to the listTree.
 */
/*ARGSUSED*/
static int scanDirs( Widget w, String root, ListTreeItem *lvl, Boolean init )
{
	ListTreeItem *lp, *lc;
	String *dirs, ptr;
	char path[PATH_MAX];
	char dbuff[100];
	static char *next;
	unsigned int i, ndir, nalloc;
	Boolean first = True;

	ndir = nalloc = 0;
	dirs = NULL;
	if ( lvl == NULL )
		next = StartDir;

	/*
	 * Avoiding arg 5 = False, since it seems to trigger a
	 * bug on Solaris, motif 1.2.3 when used in combination
	 * with XmFILE_DIRECTORY.
	 */
	_XmOSGetDirEntries( root, "*", XmFILE_DIRECTORY, MatchDots, True,
		&dirs, &ndir, &nalloc);

	(void)qsort( dirs, ndir, sizeof(dirs), (sortProc)compar );

	ListTreeRefreshOff(w);

	for( i = 0; i < ndir; i++ ) {

		if ( (ptr = strrchr( dirs[i], '/' )) != NULL )
			dirs[i] = ++ptr;

		if ( !strcmp( dirs[i] , "." ) || !strcmp( dirs[i], ".." ) )
			continue;

		lc = ListTreeAdd(w, lp = ListTreeAdd(w, lvl, dirs[i]), dirs[i]);


		if ( init == True ) {
			char ch;
			/*
			 * Recursively add and open items up to `pwd'.
			 */

			if ( ! next[1] ) {
				init = False;
				continue;
			}

			if ( first == True ) {
				(void)strcpy( dbuff, &next[1] );
				*strchr( dbuff, '/' ) = '\0';
			}

			if ( !strcmp( dirs[i], dbuff ) ) {

				(void)ListTreeRenameItem( w, lc, "."  );

				if ( first == True )
					next += strlen(dbuff)+1;
				first = False;

				ch = next[1]; next[1] = '\0';
				(void)strcpy( path, StartDir );
				next[1] = ch;

				lp->open = True;
				(void)scanDirs( w, path, lp, init );
			}
		}
	}

	ListTreeRefreshOn(w);

	if ( (!i) && (lvl == NULL) ) {
		(void)ListTreeAdd( w, NULL, "ERROR" );
		return(-1);
	}

	return(0);
}

#endif
