/******************************************************************************

	ttymon.c

	Environment:    Unix R40V3/Solaris2/Linux.

	Revision history:	@(#)ttymon.c	1.9     97/06/23

	DESCRIPTION: Part of the Mdb Application.
	Open  psuedo TTYs for stdin stdout and stderr,
	and associate a child application to them.
	The code here is ment to be reentrant so that
	an arbitrary number of childs can run simultaneously.
	NOTE: This code has been tested on Solaris 2.x and Linux.

        COPYRIGHT NOTICE:
        Permission to use,  copy,  modify,  and  distribute  this
        software  and  its    documentation   is  hereby  granted
        without fee, provided that  the  above  copyright  notice
        appear  in all copies and that both that copyright notice
        and  this  permission   notice   appear   in   supporting
        documentation.  The   author  makes  no   representations
        about   the   suitability   of   this  software  for  any
        purpose.  It  is  provided  "as  is"  without  express or
        implied warranty.

        THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD  TO  THIS
        SOFTWARE,    INCLUDING    ALL   IMPLIED   WARRANTIES   OF
        MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL THE AUTHOR
        BE  LIABLE  FOR  ANY  SPECIAL,  INDIRECT OR CONSEQUENTIAL
        DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS  OF
        USE, DATA OR PROFITS, WHETHER IN AN ACTION  OF  CONTRACT,
        NEGLIGENCE  OR  OTHER  TORTIOUS   ACTION,   ARISING   OUT
        OF   OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
        SOFTWARE.

******************************************************************************/
/******************************************************************************/
#pragma ident "@(#)ttymon.c      1.9		97/06/23"

#if ! defined(TTYMON)
static char not_def[] = "not_def";
#else
#include "mdb.h"
#include <wait.h>
#include <termios.h>
#if defined(SVR4)
#include <stropts.h>
#endif
#if defined(linux)
#include <sys/ioctl.h>
#endif

#define INTR	'\003'

static char	Lines[40];
static char	Columns[40];


/*
 * Open the master TTY, and bind a
 * slave TTY name to it.
 */
/*ARGSUSED*/
static int initSlaveTTY( CFILE *cfd )
{


#if defined(SVR4)

	char *pty;

	if ( (cfd->master = open( "/dev/ptmx", O_RDWR )) < 0 ) {
		(void)xpmsg( cfd->top, "error: open ptmx" );
		return(-1);
	}

	if ( (pty = (char *)ptsname(cfd->master)) == NULL ) {
		(void)xpmsg( cfd->top, "error: ptsname" );
		(void)close(cfd->master);
		return(-1);
	}
	(void)strcpy( cfd->slave_tty, pty );

	if ( grantpt(cfd->master) < 0 ) {
		(void)xpmsg( cfd->top, "error: grantpt" );
		(void)close(cfd->master);
		return(-1);
	}

	if ( unlockpt(cfd->master) < 0 ) {
		(void)xpmsg( cfd->top, "error: unlockpt" );
		(void)close(cfd->master);
		return(-1);
	}

	if ( access( cfd->slave_tty, R_OK | W_OK ) ) {
		(void)xpmsg( cfd->top, "error: access" );
		(void)close(cfd->master);
		return(-1);
	}

	(void)ioctl( cfd->master, TIOCFLUSH, (char *)0 );

	return(0);
#endif

#if defined(linux)

	static char *ptyx = { "pqrstuvwxyzPQRST" };
	char pty[80];
	char tty[80];
	int i;
	
	cfd->master = -1;

	while ( *ptyx ) {

		for( i = 0; i < 16; i++ ) {

			(void)sprintf( pty, "/dev/pty%c%x", *ptyx, i );
			(void)sprintf( tty, "/dev/tty%c%x", *ptyx, i );

			if ( (cfd->master = open( pty, O_RDWR )) < 0 )
				continue;

			if ( access( tty, R_OK | W_OK ) ) {
				(void)close( cfd->master );
				cfd->master = -1;
				continue;
			}
			break;
		}

		if ( cfd->master >= 0 )
			break;
		ptyx++;
	}

	if ( cfd->master  < 0 )
		return(-1);

	(void)strcpy( cfd->slave_tty, tty );

	return(0);

#endif
}


/*
 * Associate the master file descriptor with
 * an io stream. Set unbuffered modes.
 */
/*ARGSUSED*/
static int initParentTTY( CFILE *cfd )
{
	int flags;

	/*
	 * Set the child file descriptor to nonblocking mode
	 */
	if ( (flags = fcntl( cfd->master, F_GETFL, 0 )) == -1 ) {
		(void)xpmsg( cfd->top, "error: fcntl" );
		return(-1);
	}

	if ( fcntl( cfd->master, F_SETFL, flags | O_NONBLOCK ) == -1 ) {
		(void)xpmsg( cfd->top, "error: fcntl" );
		return(-1);
	}
	
	/*
	 * Open file pointer with read/write access to child process
	 */
	if ( ( cfd->fd = (FILE *)fdopen( cfd->master, "r+" )) == NULL ) {
		(void)xpmsg( cfd->top, "error: fdopen" );
		return(-1);
	}

	/*
	 * Set unbuffered mode
	 */
	setbuf( cfd->fd, NULL );

	return(0);
}


/*
 * This code runs on behalf of the child, so we can't use any of
 * the parents X-stuff here, i.e, we use perror etc. for output.
 * - Open slave tty and dup2 us.
 */
/*ARGSUSED*/
static int initChildTTY( CFILE *cfd )
{

	struct termios attr;

#ifdef	TIOCGWINSZ
	struct winsize wz;
#endif

	/*
	 * Close master.
	 */
	(void)close( cfd->master );

	cfd->child = getpid();

	/*
	 * Make child a process leader:
	 * Set the process group of the pty
	 * and of us to our process id.
	 */
	if ( (int)setsid() < 0 ) {
		perror( "setsid" );
		return(-1);
	}

	/*
	 * Open and set slave
	 * so that we become
	 * the controlling tty.
	 */
	if ( (cfd->slave = open( cfd->slave_tty, O_RDWR )) < 0 ) {
		perror( "open" );
		return(-1);
	}

#if defined(SVR4)

	if ( ioctl( cfd->slave, I_PUSH, "ptem" ) )
		perror( "ioctl ptem" );

	if ( ioctl( cfd->slave, I_PUSH, "ldterm" ) )
		perror( "ioctl ldterm" );

	if ( ioctl( cfd->slave, I_PUSH, "ttcompat" ) )
		perror( "ioctl ttcompat");

#endif

	if ( tcgetattr( cfd->slave, &attr ) < 0 )
		perror( "tcgetattr" );
	else {
		attr.c_oflag &= ~OPOST;
		attr.c_oflag &= ~ONLCR;
		if ( cfd->lecho == False )
			attr.c_lflag &= ~ECHO;
		attr.c_cc[VINTR] = INTR;
		attr.c_cc[VERASE] = ERASE;
		attr.c_lflag |= ISIG;
		if ( tcsetattr( cfd->slave, TCSANOW, &attr ) < 0 )
			perror( "tcsetattr" );
	}

#ifdef TIOCGWINSZ
	/*
	 * Set default window size.
	 */
	if ( cfd->emode == XmMULTI_LINE_EDIT ) {
		wz.ws_row = cfd->rows;
		wz.ws_col = cfd->cols;
		wz.ws_xpixel = cfd->x;
		wz.ws_ypixel = cfd->y;

		(void)ioctl( cfd->slave, TIOCSWINSZ, &wz );
	}
#endif /* TIOCGWINSZ */

	/*
	 * Redirect stdin, stdout, stderr of child to pty
	 */
	if ( dup2( cfd->slave, STDIN_FILENO ) < 0 ) {
		perror( "dup2 stdin" );
		(void)close(cfd->slave);
		return(-1);
	}
	if ( dup2( cfd->slave, STDOUT_FILENO ) < 0 ) {
		perror( "dup2 stdout" );
		(void)close(cfd->slave);
		return(-1);
	}
	if ( dup2( cfd->slave, STDERR_FILENO ) < 0 ) {
		perror( "dup2 stderr" );
		(void)close(cfd->slave);
		return(-1);
	}

	if ( cfd->slave > STDERR_FILENO )
		(void)close(cfd->slave);

	/*
	 * Unbuffer output data from child
	 */
	(void)fcntl( STDOUT_FILENO, F_SETFL, O_APPEND );
	(void)setbuf( stdout, NULL );

	return(0);
}


/*
 * Check existent of child.
 */
/*ARGSUSED*/
static Boolean childIsAlive( CFILE *cfd )
{
	Boolean alive = cfd->alive;
	int status = -1;

	if ( alive == False )
		return(False);

	/*
	 * May be truly dead or just <defunct>
	 */
	if ( (waitpid( cfd->child, &status, WNOHANG) && WIFEXITED(status)) )
		alive = False;

	if ( kill( cfd->child, 0 ) < 0 )
		alive = False;

	if ( (cfd->alive = alive) == False )
		(void)xpmsg( cfd->top,
			     "error: %s (%d)", cfd->program, cfd->child );

	return(cfd->alive);
}


/*
 * Read data from the child process.
 * Then write it to the associated
 * XmText widget.
 * Called from the X-library.
 */
/*ARGSUSED*/
static XtInputCallbackProc
outRdy( CFILE *cfd, int *fid, XtInputId *inputId )
{
	char dbuff[BUFSIZ];
	char *ptr;

	if ( childIsAlive(cfd) == False )
		return(0);

	if ( fgets( dbuff, sizeof(dbuff), cfd->fd ) != NULL ) {

		if ( cfd->noResponse > 0 ) {
			cfd->noResponse --; 
			return(0);
		}

		if ( (*cfd->prompt == '\0') ||
			(strncmp(dbuff, cfd->prompt, strlen(cfd->prompt)-1)) ) {
			/*
			 * Turn off echo in case we are
			 * writing to a window that is used
			 * for both in and out, i.e no echo.
			 * If not, we may get stucked in a
			 * loopback through fprintfChild()
			 * or fputsChild() since we may be
			 * `XmNmodifyVerifyCallback 'activated.
			 */
			cfd->echo = False;
			if ( cfd->emode == XmSINGLE_LINE_EDIT ) {
				if ( (ptr = strchr( dbuff, '\n' )) != NULL )
					*ptr = '\0';
				if ( *dbuff )
					XmTextSetString( cfd->w, dbuff );
			} else {
				/*
				 * Append last and then set cursor
				 * position as returned from the
				 * modify callback.
				 */
				XmTextInsert( cfd->w, 1000000, dbuff );
				XmTextSetInsertionPosition( cfd->w,
							cfd->currInsert );
			}
			cfd->echo = True;
		}
	}
	return(0);
}


/*
 * Initialize I.O.
 * Add the file descriptors to be
 * multiplexed by the X-windows system.
 */
static void initIO( CFILE *cfd )
{

	cfd->in = XtAppAddInput( GetAppContext(),
		fileno( cfd->fd ),
		(XtPointer)XtInputReadMask,
		(XtInputCallbackProc)outRdy,
		(XtPointer)cfd );
}


/*
 * Process in/out data from child
 * when we are running both in and
 * out from the same text widget.
 */
static void fputsChild( Widget w, CFILE *cfd, XmTextVerifyCallbackStruct *cbs )
{
	Widget top;
	XmTextPosition lineStart, lastPos;
	Position x, y;
	size_t i, indx, bse;
	char *ptr;
	char line[BUFSIZ*2];

	if ( cfd == NULL ) {
		cbs->doit = False;
		return;
	}

	cbs->doit = True;

	if ( cfd->recur == True )
		return;

	if ( cfd->echo == False ) {

		/*
		 * The child is sending characters.
		 */

		top = w;
   		cbs->doit = False;
		ptr = cbs->text->ptr;
		bse = 0;
		lastPos = XmTextGetLastPosition(w);

		/*
		 * Do character processing and the
		 * `dumb' terminal's line editing.
		 */

		/*
		 * Get the current line position.
		 */
		(void)XmTextPosToXY( w, cbs->currInsert, &x, &y );
		lineStart = XmTextXYToPos( w, 0, y );

		if ( (lastPos-lineStart) + cbs->text->length > sizeof(line) )
			return;

		bzero( line, sizeof(line) );

		/*
		 * Get a copy of the current line.
		 */
   		(void)XmTextGetSubstring( w,
			lineStart, lastPos-lineStart,
			sizeof(line), line );

		indx = cfd->currInsert - lineStart;
		

		for ( i = 0; i < cbs->text->length; i++ ) {

			if ( ptr[i] == '\b' ) {
				indx--;
				cfd->currInsert--;
				bse++;
				continue;
			}

			if ( ptr[i] == 007 ) {
				while (XtIsTopLevelShell(top) == False)
					top = XtParent(top);
				XBell( XtDisplay(top), 0 );
				continue;
			}

			bse = 0;

			if ( (ptr[i] == '\r') && (ptr[i+1] == '\n') )
				continue;

			if ( (ptr[i] == '\r') && (ptr[i+1] == '\r') )
				continue;

			if ( ptr[i] == '\r' ) {
				indx = 0;
				cfd->currInsert = lineStart;
				continue;
			}

			if ( (ptr[i] == '\n') || (ptr[i] == '\t') ) {
				line[indx++] = ptr[i];
				cfd->currInsert++;
				continue;
			}

#if ! defined(linux)
			/*
			 * Full locale support required.
			 */
			if ( isprint(ptr[i]) == False )
				continue;
#endif

			line[indx++] = ptr[i];
			cfd->currInsert++;
		}

		if ( (bse > 0) && (cbs->text->length >1) ) {
			/*
			 * remove any trailing white spaces.
			 */
			indx = strlen(line)-1;
			while( bse-- ) {
				if ( line[indx] == ' ' )
					line[indx--] = '\0';
				else
					break;
			}
		}

		cfd->recur = True;
		XmTextReplace( w, lineStart, lastPos, line );
		cfd->recur = False;
		return;
	}

	if ( cbs->text->length > 0 ) {
		/*
		 * Send the user input
		 * with no echo since
		 * the child will do so.
		 */
		cbs->doit = False;

		if ( cbs->text->length == 1 )
			(void)putc( *cbs->text->ptr, cfd->fd );
		else
			(void)fwrite( cbs->text->ptr,
			      sizeof(char), cbs->text->length, cfd->fd );

	} else if ( cbs->startPos == cbs->endPos -1 ) {
		/*
		 * Moving backwards.
		 * Send Erase.
		 */
		(void)putc( ERASE, cfd->fd );
	}
}


/*
 * As fprintf() but here we pass
 * the CFILE struct instead of
 * the stream file descriptor.
 */
/*VARARGS*/
void fprintfChild( CFILE *cfd, ... )
{
	va_list ap;
	char *fmt;

	if ( (cfd == NULL) || (childIsAlive(cfd) == False) )
		return;

	va_start(ap, cfd);
	fmt = va_arg( ap, char * );
	(void)vfprintf( cfd->fd, fmt, ap );
	va_end(ap);

	return;
}


/*
 * As fclose.
 */
/*ARGSUSED*/
void detachChild( CFILE *cfd )
{
	/*
	 * Terminate child gracefylly, and
	 * then the hard way. Remove it
	 * from the X-windos system's
	 * i.o multiplexor and finally
	 * close the i.o stream.
	 */
	if ( cfd != NULL ) {
		if ( cfd->alive == True ) {
			(void)fprintfChild( cfd, "\nexit\nquit\n" );
			sleep(1);
		}
		(void)kill( cfd->child, SIGHUP );
		XtRemoveInput(cfd->in);
		(void)fclose(cfd->fd);
		XtFree( (char *)cfd );
		cfd = NULL;
	}
}


/*
 * Handle resize events.
 */
#ifdef	TIOCGWINSZ
/*ARGSUSED*/
static void resize_cb( Widget w, CFILE *cfd, XmAnyCallbackStruct *cbs )

{
	short nrows, ncols;
	struct winsize wz;
	int slave;

	/*
	 * Handle shell resize events.
	 */
	XtVaGetValues( w,
			XmNrows,	&nrows,
			XmNcolumns,	&ncols,
			XmNwidth,	&cfd->x,
			XmNheight,	&cfd->y,
			NULL );

	if ( (nrows != cfd->rows ) || (ncols != cfd->cols) ) {

		XmTextSetInsertionPosition( cfd->w,
			cfd->currInsert = XmTextGetLastPosition(cfd->w) );
		/*
		 * This will force a SIGWINCH signal to
		 * tty group. Some childs such as tcsh
		 * and bash will honor this signal and
		 * set LINES and COLUMNS accordingly.
		 */
		cfd->cols = ncols; cfd->rows = nrows;
		wz.ws_row = cfd->rows;
		wz.ws_col = cfd->cols;
		wz.ws_xpixel = cfd->x;
		wz.ws_ypixel = cfd->y;

		if ( (slave = open( cfd->slave_tty, O_RDWR )) < 0 )
			return;

		(void)ioctl( slave, TIOCSWINSZ, &wz );
		(void)close(slave);
	}
}
#endif


/*
 * Attach a child process to the application.
 * Synopsis:
 * CFILE
 * *attachChild( Widget top, char *cmd, char *prompt, Boolean echo, Widget w );
 * Where:
 * 	top	The top shell using this routine.
 * 	cmd	The system command to run.
 * 	prompt	(Optional) The command's prompt that we will ignore.
 * 	echo	Child does not echo characters if True.
 * 	w	The XmText widget to use as the output window.
 * 	        If w is MULTI_LINE_EDIT and EDITABLE, then
 * 	        that window will be used for both in and out.
 * 	        The caller must NOT define XmNmodifyVerifyCallback
 * 	        nor the XmNfocusCallback when this mode is activated.
 * Return:
 * 	NULL	On failure.
 * 	cfd	A pointer to the child data structure.
 */
/*ARGSUSED*/
CFILE *
attachChild( Widget top, char *cmd, char *prompt, Boolean echo, Widget w )
{

	CFILE *cfd;
	Boolean editable;

	cfd = (CFILE *)XtMalloc( sizeof(CFILE) );
	cfd->w = w;
	cfd->top = top;
	cfd->echo = False;
	cfd->lecho = echo;
	cfd->alive = True;
	cfd->recur = False;
	cfd->noResponse = 0;
	cfd->currInsert = 0;
	cfd->parent = getpid();
	(void)strcpy( cfd->program, cmd );
	(void)strcpy( cfd->prompt, prompt == NULL ? "" : prompt );

	XtVaGetValues( w,
			XmNeditMode,	&cfd->emode,
			XmNeditable,	&editable,
			XmNrows,	&cfd->rows,
			XmNcolumns,	&cfd->cols,
			XmNwidth,	&cfd->x,
			XmNheight,	&cfd->y,
			NULL );

	if ( initSlaveTTY(cfd) ) {
		XtFree( (char *)cfd );
		return(NULL);
	}

	if ( (cfd->child = fork()) == 0 ) {
		if ( initChildTTY(cfd) ) {
			(void)fprintf( stderr, "initChildTTY failed\n" );
			_exit(1);
		}

		(void)putenv( "TERM=dumb" );
		(void)putenv( "TERMTYPE=dumb" );

		if ((editable == True) && (cfd->emode == XmMULTI_LINE_EDIT)) {
			(void)sprintf( Columns, "COLUMNS=%d", cfd->cols );
			(void)putenv( Columns );
			(void)sprintf( Lines, "LINES=%d", cfd->rows );
			(void)putenv( Lines );
		}

		execl( "/bin/sh", "sh", "-c", cmd, (char *)0 );
		/*
		 * This message will appear on the text Widget(w).
		 */
		(void)fprintf( stderr, "Could not execl %s\n", cmd );
		_exit(1);
    	}

	if ( cfd->child == -1 ) {
		(void)xpmsg( top, "cannot fork for %s", cmd );
		XtFree( (char *)cfd );
		return(NULL);
	}

	if ( initParentTTY(cfd) ) {
		XtFree( (char *)cfd );
		return(NULL);
	}

	if ( (editable == True) && (cfd->emode == XmMULTI_LINE_EDIT) ) {
		/*
		 * Input and output is from the same widget.
		 */
		XtAddCallback( w, XmNmodifyVerifyCallback,
			(XtCP)fputsChild, (XtPointer)cfd );
		XtVaSetValues( w, XmNverifyBell, False, NULL );
#ifdef	TIOCGWINSZ
		XtAddCallback( w, XmNfocusCallback,
			(XtCP)resize_cb, (XtPointer)cfd );
#endif
	}

	/*
	 * child is now set up and running
	 */
	initIO( cfd );

	return(cfd);
}
#endif /* TTYMON */
