/*
 * main.c
 *	Initialization and entry to mgv.
 *
 * Copyright (C) 1996  Eric A. Howe
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Authors:	Matthew D. Francey
 *		Eric A. Howe (mu@echo-on.net)
 */
/*
 * application initialization
 */
#include	<wlib/rcs.h>
MU_ID("$Mu: mgv/main.c,v 1.155 $")

#include	<stdlib.h>
#include	<signal.h>
#include	<unistd.h>
#include	<X11/Intrinsic.h>
#include	<X11/StringDefs.h>
#include	<X11/Shell.h>

#include	<wlib/wlib.h>
#include	<wlib/build.h>
#include	<wlib/icons.h>
#include	<wlib/WlAppShell.h>
#include	<mine/mgv.h>
#include	<mine/help.h>
#include	<mine/strdefs.h>
#include	<mine/mgv_cicon.xpm>
#include	<mine/mgv_micon.xbm>
#include	<mine/mgv_mask.xbm>

/*
 * some resources that only main needs
 */
typedef struct {
	Boolean	want_help;
	Boolean	want_long_help;
	Boolean	want_version;
} HVRES;
#define	OFFSET(f)	XtOffsetOf(HVRES, f)
static XtResource hvres[] = {
	{
		"wantHelp", "WantHelp",
		XtRBoolean, sizeof(Boolean), OFFSET(want_help),
		XtRImmediate, (XtPointer)False
	}, {
		"wantVersion", "WantVersion",
		XtRBoolean, sizeof(Boolean), OFFSET(want_version),
		XtRImmediate, (XtPointer)False
	}, {
		"wantLongHelp", "WantLongHelp",
		XtRBoolean, sizeof(Boolean), OFFSET(want_long_help),
		XtRImmediate, (XtPointer)False
	}
};
#undef	OFFSET
 
static XrmOptionDescRec options[] = {
	/*
	 * MGv resources
	 */
	{"-aliaseps",       "*antialiasEPS",        XrmoptionNoArg,  "True"},
	{"-autoheight",     "*autoHeight",          XrmoptionNoArg,  "True"},
	{"-autowidth",      "*autoWidth",           XrmoptionNoArg,  "True"},
	{"-confirmexit",    "*confirmExit",         XrmoptionNoArg,  "True"},
	{"-decompress",     "*decompress",          XrmoptionSepArg, NULL},
	{"-help",           "*wantHelp",            XrmoptionNoArg,  "True"},
	{"-killkids",       "*killChildrenAtExit",  XrmoptionNoArg,  "True"},
	{"-labels",         "*showLabels",          XrmoptionNoArg,  "True"},
	{"-log",            "*hideLog",             XrmoptionNoArg,  "False"},
	{"-longhelp",       "*wantLongHelp",        XrmoptionNoArg,  "True"},
	{"-magicresistance","*magicResistance",     XrmoptionSepArg, NULL},
	{"-magicscroll",    "*magicScrolling",      XrmoptionNoArg,  "True"},
	{"-magstep",        "*magstep",             XrmoptionSepArg, NULL},
	{"-media",          "*defaultPageMedia",    XrmoptionSepArg, NULL},
	{"-menubar",        "*hideMenuBar",         XrmoptionNoArg,  "False"},
	{"-noaliaseps",     "*antialiasEPS",        XrmoptionNoArg,  "False"},
	{"-noautoheight",   "*autoHeight",          XrmoptionNoArg,  "False"},
	{"-noautowidth",    "*autoWidth",           XrmoptionNoArg,  "False"},
	{"-noconfirmexit",  "*confirmExit",         XrmoptionNoArg,  "False"},
	{"-nokillkids",     "*killChildrenAtExit",  XrmoptionNoArg,  "False"},
	{"-nolabels",       "*showLabels",          XrmoptionNoArg,  "False"},
	{"-nolog",          "*hideLog",             XrmoptionNoArg,  "True"},
	{"-nomagicscroll",  "*magicScrolling",      XrmoptionNoArg,  "True"},
	{"-nomenubar",      "*hideMenuBar",         XrmoptionNoArg,  "True"},
	{"-nopagelist",     "*hidePageList",        XrmoptionNoArg,  "True"},
	{"-nopgnums",       "*showPageNumbers",     XrmoptionNoArg,  "False"},
	{"-nosmartsizing",  "*smartSizing",         XrmoptionNoArg,  "False"},
	{"-nosmooth",       "*smoothScrolling",     XrmoptionNoArg,  "False"},
	{"-nostatus",       "*hideStatusLine",      XrmoptionNoArg,  "True"},
	{"-pagelist",       "*hidePageList",        XrmoptionNoArg,  "False"},
	{"-paper",          "*defaultPageMedia",    XrmoptionSepArg, NULL},
	{"-pgnums",         "*showPageNumbers",     XrmoptionNoArg,  "True"},
	{"-printcommand",   "*printCommand",        XrmoptionSepArg, NULL},
	{"-printer",        "*printer",             XrmoptionSepArg, NULL},
	{"-reticulefg",     "*reticuleForeground",  XrmoptionSepArg, NULL},
	{"-reticulewidth",  "*reticuleLineWidth",   XrmoptionSepArg, NULL},
	{"-scrollpercent",  "*scrollPercentage",    XrmoptionSepArg, NULL},
	{"-smartsizing",    "*smartSizing",         XrmoptionNoArg,  "True"},
	{"-smooth",         "*smoothScrolling",     XrmoptionNoArg,  "True"},
	{"-smoothness",     "*smoothness",          XrmoptionSepArg, NULL},
	{"-status",         "*hideStatusLine",      XrmoptionNoArg,  "False"},
	{"-temp",           "*tempDir",             XrmoptionSepArg, NULL},
	{"-version",        "*wantVersion",         XrmoptionNoArg,  "True"},
	{"-wmheight",       "*wmHeight",            XrmoptionSepArg, NULL},
	{"-wmwidth",        "*wmWidth",             XrmoptionSepArg, NULL},
	{"-zoomfg",         "*reticuleForeground",  XrmoptionSepArg, NULL},
	{"-zoomwidth",      "*reticuleLineWidth",   XrmoptionSepArg, NULL},

	/*
	 * wlib resources
	 */
#	include	<wlib/xrmoptions.h>

	/*
	 * WlAppShell things.
	 */
#	include	<wlib/WlAppShell-xrm.h>

	/*
	 * Ghostview resources
	 */
#	define	SGV(x)	"*Ghostview." #x
	{"-antialias",	SGV(antialias),		XrmoptionNoArg,  "True"},
	{"-arguments",	SGV(arguments),		XrmoptionSepArg, NULL},
	{"-color",	SGV(palette),		XrmoptionNoArg,  "Color"},
	{"-dpi",	SGV(Resolution),	XrmoptionSepArg, NULL},
	{"-grayscale",	SGV(palette),		XrmoptionNoArg,  "Grayscale"},
	{"-greyscale",	SGV(palette),		XrmoptionNoArg,  "Grayscale"},
	{"-gslocale",	SGV(gsLocale),		XrmoptionSepArg, NULL},
	{"-interpreter",SGV(interpreter),	XrmoptionSepArg, NULL},
	{"-landscape",	SGV(orientation),	XrmoptionNoArg,  "landscape"},
	{"-monochrome",	SGV(palette),		XrmoptionNoArg,  "Monochrome"},
	{"-noantialias",SGV(antialias),		XrmoptionNoArg,  "False"},
	{"-noquiet",	SGV(quiet),		XrmoptionNoArg,  "False"},
	{"-nosafer",	SGV(safer),		XrmoptionNoArg,  "False"},
	{"-portrait",	SGV(orientation),	XrmoptionNoArg,  "portrait"},
	{"-quiet",	SGV(quiet),		XrmoptionNoArg,  "True"},
	{"-resolution",	SGV(Resolution),	XrmoptionSepArg, NULL},
	{"-safer",	SGV(safer),		XrmoptionNoArg,  "True"},
	{"-seascape",	SGV(orientation),	XrmoptionNoArg,  "seascape"},
	{"-upsidedown",	SGV(orientation),	XrmoptionNoArg,  "upsidedown"},
	{"-xdpi",	SGV(xdpi),		XrmoptionSepArg, NULL},
	{"-ydpi",	SGV(ydpi),		XrmoptionSepArg, NULL},
#	undef SGV
};

static int
helpem_long(char *me, int ret)
{
	int	i;

	printf("%s [options] [files...]\n", me);
	for(i = 0; bld_switch_longhelp[i] != NULL; ++i)
		printf("%s\n", bld_switch_longhelp[i]);
	return ret;
}

#define	LINELEN	75
static int
helpem(char *me, int ret)
{
	int	i, len, slen;

	printf("%s [options] [files...]", me);
	for(i = 0, len = LINELEN + 100; bld_switch_help[i] != NULL; ++i) {
		slen = strlen(bld_switch_help[i]);
		if(len + slen + 1 > LINELEN) {
			printf("\n\t");
			len = 8;
		}
		printf("%s ", bld_switch_help[i]);
		len += slen + 1;
	}
	printf("\n");
	return ret;
}

static int
version(char *me)
{
	char	buf[2048];

	printf("%s version information:\n%s\n"
		"available from http://www.echo-on.net/~mu\n",
		me, wl_version(&buf[0]));
	return EXIT_SUCCESS;
}

/*
 * POSIX doesn't say that fileno() is safe in a signal handler
 * so we grab the file descriptor of stderr (POSIX doesn't appear
 * to force it be 2 even though we all think it is).
 *
 * Okay, okay, we're only using this in a signal handler for SIGSEGV
 * and friends so who cares about safety?  I do, if I'm going to screw up,
 * I'd like to do it properly.
 */
static	int	fd_stderr;

/*
 * some people don't have SIGBUS but I don't want to put in more
 * than one ifdef to handle it
 */
#if !defined(SIGBUS)
#	define	SIGBUS	0
#endif

static void flaming_death(int);
#define	DEATH(sig)	{sig, flaming_death, #sig, sizeof(#sig)}
/*
 * the name field, and hence the n_name field, is only needed if
 * the handler is set to flaming_death
 */
static struct {
	int	sig;
	void	(*handler)(int);
	char	*name;
	int	n_name;
} sigs[] = {
	DEATH(SIGABRT),
	DEATH(SIGBUS),
	DEATH(SIGFPE),
	DEATH(SIGILL),
	DEATH(SIGSEGV),
	DEATH(SIGTRAP),

	{SIGCHLD, SIG_DFL},
	{SIGPIPE, SIG_DFL},
	{SIGTERM, SIG_DFL},
};
#define	N_SIGS	(int)(sizeof(sigs)/sizeof(sigs[0]))

/*
 * just because I screwed up doesn't mean I have to be
 * messy about it!
 */
static char swansong[] =
	"\tSomething in mgv broke!\n"
	"\tYou found a bug in mgv--I'm trying to die gracefully--if you\n"
	"\tcan reproduce the bug, please send a bug report to mu@echo-on.net\n";
static void
flaming_death(int sig)
{
	int	i;

	for(i = 0; i < N_SIGS; ++i)
		if(sigs[i].sig == sig)
			break;
	if(i != N_SIGS)
		write(fd_stderr, (void *)sigs[i].name, sigs[i].n_name);
	write(fd_stderr, (void *)swansong, sizeof(swansong));

	/*
	 * hey man, I'm not goin' alone!
	 */
	kill(-getpgrp(), SIGTERM);
	exit(1);
}

static void
sig_init(void)
{
	struct sigaction sa;
	int	i;

	/*
	 * We want to lead our process group so that flaming_death
	 * will do the right thing.  All of this may seem a little
	 * excessive just to deal with SIGSEGV and SIGABRT but I'm
	 * sick of writting little awk one-liners to clean up the
	 * left over gs processes when I find a bug in mgv.
	 *
	 * Besides, I thought it was really cool when I trapped
	 * my first seg fault and died gracefully.
	 *
	 * I suppose I could keep a list of all the child processes
	 * that have been made (and maintain this list) but that is
	 * just a big pain in the ass; using process groups is just
	 * too cute and convenient a trick to not use it.
	 */
	setpgid(getpid(), getpid());

	fd_stderr = fileno(stderr);
	memset((void *)&sa, '\0', sizeof(sa));
	sigemptyset(&sa.sa_mask);
	for(i = 0; i < N_SIGS; ++i) {
		if(sigs[i].sig == 0)
			continue;
		sa.sa_handler = sigs[i].handler;
		sigaction(sigs[i].sig, &sa, NULL);
	}
}

static WL_ICONS icons = {
	cicon,
	(char *)micon_bits, micon_width, micon_height,
	(char *)mask_bits,  mask_width,  mask_height
};

int
main(int argc, char **argv)
{
	MGV		*m;
	char		*me;
	HVRES		hv;
	Widget		top;
	Display		*dpy;
	int		virgin;

	/*
	 * prepare for the worst
	 */
	sig_init();

	if((me = strrchr(argv[0], '/')) == NULL)
		me = argv[0];
	else
		++me;

	/*
	 * Crank Xt into life.
	 *
	 * We don't use the Xt convenience functions since we
	 * want several top level shells.
	 */
	dpy = wl_init(&argc, argv, "MGv", &options[0], XtNumber(options),
							&bld_fallbacks[0]);
	if(dpy == NULL) {
		fprintf(stderr, "mgv: could not initialize!\n");
		return EXIT_FAILURE;
	}
	wl_addclass("Ghostview", ghostviewWidgetClass, NULL, FALSE);
	wl_icons_init(&icons);

	/*
	 * Here we go...
	 */
	++argv;
	virgin = TRUE;
	do {
		top = XtAppCreateShell(NULL, "MGv", wlAppShellWidgetClass,
								dpy, NULL, 0);

		/*
		 * We only care about -help and -version on the first pass.
		 */
		if(virgin) {
			memset((void *)&hv, '\0', sizeof(hv));
			XtGetApplicationResources(top, (XtPointer)&hv, hvres,
						XtNumber(hvres), NULL, 0);
			if(hv.want_help)
				exit(helpem(me, EXIT_SUCCESS));
			if(hv.want_long_help)
				exit(helpem_long(me, EXIT_SUCCESS));
			if(hv.want_version)
				exit(version(me));
			virgin = FALSE;
		}

		if((m = mgv_create(top, *argv)) == NULL) {
			fprintf(stderr, "%s: could not create top level window"
					"for %s.\n", me,
					*argv == NULL ? "(nil)" : *argv);
			XtDestroyWidget(top);
			top = NULL;
			continue;
		}
		mgv_show(m, 0);
	} while(*argv != NULL && *++argv != NULL);

	if(top == NULL) {
		fprintf(stderr, "%s: could not create any top level windows\n"
							"\texitting.\n", me);
		exit(EXIT_FAILURE);
	}

	XtAppMainLoop(XtWidgetToApplicationContext(top));

	/*
	 * Sacrifice for the angry compiler gods.
	 */
	return 0;
}
