/*-----------------------------------------------------------------------------
  Module FmFw.c                                                             

  (c) Simon Marlow 1991
  (c) Albert Graef 1994
  (c) Oliver Mai 1995
                                                                           
  functions & data for creating a file window, and various functions        
  related to file windows                                                   
-----------------------------------------------------------------------------*/

#include <pwd.h>
#include <time.h>
#include <string.h>

#ifdef SVR4
#ifndef USE_STATVFS
#define USE_STATVFS
#endif
#endif	/* SVR4 */

#ifdef USE_STATVFS
#include <sys/statvfs.h>
#else
#include <sys/vfs.h>
#endif	/* USE_STATVFS */

#ifdef _AIX
#include <sys/resource.h>
#endif

#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/StringDefs.h>
#include <Xm/ScrolledW.h>
#include <Xm/RowColumn.h>
#include <Xm/Form.h>
#include <Xm/LabelG.h>
#include <Xm/Protocols.h>
#include <Xm/TextF.h>
#include <Xm/Frame.h>
#include <Xm/PushBG.h>
#include <Xm/ToggleBG.h>
#include <Xm/SeparatoG.h>
#include <Xm/DragDrop.h>
#include <Xm/AtomMgr.h>

#include "Fm.h"

#define MAXCFGLINELEN 1024
#define FW_WIDTH 400
#define FW_HEIGHT 300
#define TEXT_PADDING 10

/*-----------------------------------------------------------------------------
  PUBLIC DATA                                       
-----------------------------------------------------------------------------*/

FileWindowList file_windows = NULL;

int n_types;
TypeList types;
Boolean bitmaps_read = False;

CopyInfoRec copyInfo;

/*-----------------------------------------------------------------------------
  Translation tables
-----------------------------------------------------------------------------*/

static XtTranslations form_translations;

static char form_translations_s[] = "\
#override <Btn2Down>	: startFileDrag()\n\
          <Btn3Down>	: filePopup()\n";

static char file_translations_s[] = "#override\
          Ctrl<Btn1Down> : fileToggleCb()\n\
          Shift<Btn1Down>: selectBlockCb()\n\
          <Btn1Down>	 : fileSelectCb()\n\
	  <Btn2Down>	 : startFileDrag()\n\
          <Btn1Up>(2)	 : fileOpenCb()\n";

/*-----------------------------------------------------------------------------
  Action table
-----------------------------------------------------------------------------*/

/*
static void dummy(Widget w, XtPointer client_data, XtPointer call_data)
{
}
*/

static XtActionsRec file_actions[] = {
    { "startFileDrag",	startFileDrag },
    { "filePopup",	filePopup },
    { "fileSelectCb",	fileSelectCb },
    { "fileToggleCb",	fileToggleCb },
    { "selectBlockCb",	selectBlockCb },
    { "fileOpenCb",	fileOpenCb },
};

/*-----------------------------------------------------------------------------
  Widget Argument Lists
-----------------------------------------------------------------------------*/

static Arg menu_bar_args[] = {
    { XmNleftAttachment, XmATTACH_FORM },
    { XmNtopAttachment, XmATTACH_FORM },
    { XmNrightAttachment, XmATTACH_FORM }
};

static Arg icon_form_args[] = {
    { XmNbackground, 0 },
    { XmNwidth, 0 },
    { XmNtranslations, 0 },
    { XmNborderWidth, 1 },
    { XmNborderColor, 0 },
    { XmNforeground, 0 }
};

static Arg text_form_args[] = {
    { XmNbackground, 0 },
    { XmNtranslations, 0 },
    { XmNborderWidth, 1 },
    { XmNborderColor, 0 },
    { XmNforeground, 0 }
};

static Arg icon_toggle_args[] = {
    { XmNlabelType, XmPIXMAP },
    { XmNtopAttachment, XmATTACH_FORM },
    { XmNleftAttachment, XmATTACH_FORM },
    { XmNrightAttachment, XmATTACH_FORM },
    { XmNlabelPixmap, 0 },
    { XmNheight, 0 },
    { XmNfontList, 0 }  /* Seems to be necessary to avoid warnings; Motif bug? */
};

static Arg drop_form_args[] = {
    { XmNimportTargets, 0 },
    { XmNnumImportTargets, 2 },
    { XmNdropSiteOperations, XmDROP_COPY | XmDROP_MOVE | XmDROP_LINK },
    { XmNdropSiteType, XmDROP_SITE_COMPOSITE },
    { XmNdropProc, (XtArgVal) handleFileWinDrop }
};

static Arg drop_file_args[] = {
    { XmNimportTargets, 0 },
    { XmNnumImportTargets, 3 },
    { XmNdropSiteOperations, XmDROP_COPY | XmDROP_MOVE | XmDROP_LINK },
    { XmNdropProc, (XtArgVal) handleFileDrop }
};

static Arg icon_label_args[] = {
    { XmNleftAttachment, XmATTACH_FORM },
    { XmNrightAttachment, XmATTACH_FORM },
    { XmNbottomAttachment, XmATTACH_FORM },
    { XmNtopAttachment, XmATTACH_WIDGET },
    { XmNtopWidget, 0 },
    { XmNlabelString, 0 },
    { XmNfontList, 0 }};

/*-----------------------------------------------------------------------------
  STATIC DATA 
-----------------------------------------------------------------------------*/

static MenuItemRec file_pdwn_menu[] = {
    { "New...", &xmPushButtonGadgetClass, 0, "Ctrl<Key>N", "Ctrl+N", newFileCb, NULL, NULL, NULL },
    { "", &xmSeparatorGadgetClass, 0, NULL, NULL, NULL, NULL, NULL, NULL },
    { "Copy...", &xmPushButtonGadgetClass, 0, "Ctrl<Key>C", "Ctrl+C", fileCopyCb, NULL, NULL, NULL },
    { "Move...", &xmPushButtonGadgetClass, 0, "Ctrl<Key>M", "Ctrl+M", fileMoveCb, NULL, NULL, NULL },
    { "Link...", &xmPushButtonGadgetClass, 0, "Ctrl<Key>L", "Ctrl+L", fileLinkCb, NULL, NULL, NULL },
    { "", &xmSeparatorGadgetClass, 0, NULL, NULL, NULL, NULL, NULL, NULL },
    { "Delete", &xmPushButtonGadgetClass, 0, "Ctrl<Key>D", "Ctrl+D", fileDeleteCb, NULL, NULL, NULL },
    { "", &xmSeparatorGadgetClass, 0, NULL, NULL, NULL, NULL, NULL, NULL },
    { "Select...", &xmPushButtonGadgetClass, 0, NULL, NULL, selectCb, NULL, NULL, NULL },
    { "Select all", &xmPushButtonGadgetClass, 0, "Ctrl<Key>A", "Ctrl+A", selectAllCb, NULL, NULL, NULL },
    { "Deselect all", &xmPushButtonGadgetClass, 0, NULL, NULL, deselectCb, NULL, NULL, NULL },
    { "", &xmSeparatorGadgetClass, 0, NULL, NULL, NULL, NULL, NULL, NULL },
    { "Close window", &xmPushButtonGadgetClass, 0, "Ctrl<Key>W", "Ctrl+W", closeFileCB, NULL, NULL, NULL },
    { "Exit", &xmPushButtonGadgetClass, 0, "Ctrl<Key>Q", "Ctrl+Q", fileExitCb, NULL, NULL },
    { NULL, NULL }
};

static MenuItemRec dir_pdwn_menu[] = {
    { "Make directory...", &xmPushButtonGadgetClass, 0, NULL, NULL, mkDirCb, NULL, NULL, NULL },
    { "Filesystem info...", &xmPushButtonGadgetClass, 0, NULL, NULL, fsInfoCb, NULL, NULL, NULL },
    { "", &xmSeparatorGadgetClass, 0, NULL, NULL, NULL, NULL, NULL, NULL },
    { "Reread files", &xmPushButtonGadgetClass, 0, "Ctrl<Key>R", "Ctrl+R", fileRefreshCb, NULL, NULL, NULL },
    { "Filter files...", &xmPushButtonGadgetClass, 0, NULL, NULL, dirFilterCb, NULL, NULL, NULL },
    { "", &xmSeparatorGadgetClass, 0, NULL, NULL, NULL, NULL, NULL, NULL },
    { "Go home", &xmPushButtonGadgetClass, 0, NULL, NULL, goHomeCb, NULL, NULL, NULL },
    { "Go up", &xmPushButtonGadgetClass, 0, NULL, NULL, goUpCb, NULL, NULL, NULL },
    { "", &xmSeparatorGadgetClass, 0, NULL, NULL, NULL, NULL, NULL, NULL },
    { "Enter command...", &xmPushButtonGadgetClass, 0, "Ctrl<Key>E", "Ctrl+E", commandCb, NULL, NULL, NULL },
    { NULL, NULL }
};

static MenuItemRec wdw_pdwn_menu[] = {
    { "New file window", &xmPushButtonGadgetClass, 0, NULL, NULL, newFileWinCb, NULL, NULL, NULL },
    { "New application window", &xmPushButtonGadgetClass, 0, NULL, NULL, newAppWinCb, NULL, NULL, NULL },
    { "Mount table window", &xmPushButtonGadgetClass, 0, NULL, NULL, mountTableCb, NULL, NULL, NULL },
    { "Start xterm", &xmPushButtonGadgetClass, 0, "Ctrl<Key>X", "Ctrl+X", xtermCb, NULL, NULL, NULL },
    { "", &xmSeparatorGadgetClass, 0, NULL, NULL, NULL, NULL, NULL, NULL },
    { "Save windows", &xmPushButtonGadgetClass, 0, NULL, NULL, saveStartupCb, NULL, NULL, NULL },
    { NULL, NULL }
};

static MenuItemRec opt_pdwn_menu[] = {
    { "Icon view", &xmToggleButtonGadgetClass, 0, NULL, NULL, viewTypeCb, NULL, NULL, NULL },
    { "Text view", &xmToggleButtonGadgetClass, 0, NULL, NULL, viewTypeCb, NULL, NULL, NULL },
    { "", &xmSeparatorGadgetClass, 0, NULL, NULL, NULL, NULL, NULL, NULL },
    { "Sort by name", &xmToggleButtonGadgetClass, 0, NULL, NULL, sortTypeCb, NULL, NULL, NULL },
    { "Sort by size", &xmToggleButtonGadgetClass, 0, NULL, NULL, sortTypeCb, NULL, NULL, NULL },
    { "Sort by date", &xmToggleButtonGadgetClass, 0, NULL, NULL, sortTypeCb, NULL, NULL, NULL },
    { "", &xmSeparatorGadgetClass, 0, NULL, NULL, NULL, NULL, NULL, NULL },
    { "Show hidden files", &xmToggleButtonGadgetClass, 0, NULL, NULL, showHiddenCb, NULL, NULL, NULL },
    { "Show directories", &xmToggleButtonGadgetClass, 0, NULL, NULL, showDirsCb, NULL, NULL, NULL },
    { "Directories first", &xmToggleButtonGadgetClass, 0, NULL, NULL, dirsFirstCb, NULL, NULL, NULL },
    { NULL, NULL }
};

static MenuItemRec file_popup_menu[] = {
    { "Edit", &xmPushButtonGadgetClass, 0, NULL, NULL, fileEditCb, NULL, NULL, NULL },
    { "View", &xmPushButtonGadgetClass, 0, NULL, NULL, fileViewCb, NULL, NULL, NULL },
    { "Perform action...", &xmPushButtonGadgetClass, 0, NULL, NULL, fileActionCb, NULL, NULL, NULL },
    { "", &xmSeparatorGadgetClass, 0, NULL, NULL, NULL, NULL, NULL, NULL },
    { "Rename...", &xmPushButtonGadgetClass, 0, NULL, NULL, fileRenameCb, NULL, NULL, NULL },
    { "Copy...", &xmPushButtonGadgetClass, 0, NULL, NULL, fileCopyCb, NULL, NULL, NULL },
    { "Move...", &xmPushButtonGadgetClass, 0, NULL, NULL, fileMoveCb, NULL, NULL, NULL },
    { "Link...", &xmPushButtonGadgetClass, 0, NULL, NULL, fileLinkCb, NULL, NULL, NULL },
    { "Delete", &xmPushButtonGadgetClass, 0, NULL, NULL, fileDeleteCb, NULL, NULL, NULL },
    { "", &xmSeparatorGadgetClass, 0, NULL, NULL, NULL, NULL, NULL, NULL },
    { "Properties...", &xmPushButtonGadgetClass, 0, NULL, NULL, filePropsCb, NULL, NULL, NULL },
    { NULL, NULL }
};

static MenuItemRec exec_popup_menu[] = {
    { "Execute...", &xmPushButtonGadgetClass, 0, NULL, NULL, fileExecCb, NULL, NULL, NULL },
    { "", &xmSeparatorGadgetClass, 0, NULL, NULL, NULL, NULL, NULL, NULL },
    { "Edit", &xmPushButtonGadgetClass, 0, NULL, NULL, fileEditCb, NULL, NULL, NULL },
    { "View", &xmPushButtonGadgetClass, 0, NULL, NULL, fileViewCb, NULL, NULL, NULL },
    { "Perform action...", &xmPushButtonGadgetClass, 0, NULL, NULL, fileActionCb, NULL, NULL, NULL },
    { "", &xmSeparatorGadgetClass, 0, NULL, NULL, NULL, NULL, NULL, NULL },
    { "Rename...", &xmPushButtonGadgetClass, 0, NULL, NULL, fileRenameCb, NULL, NULL, NULL },
    { "Copy...", &xmPushButtonGadgetClass, 0, NULL, NULL, fileCopyCb, NULL, NULL, NULL },
    { "Move...", &xmPushButtonGadgetClass, 0, NULL, NULL, fileMoveCb, NULL, NULL, NULL },
    { "Link...", &xmPushButtonGadgetClass, 0, NULL, NULL, fileLinkCb, NULL, NULL, NULL },
    { "Delete", &xmPushButtonGadgetClass, 0, NULL, NULL, fileDeleteCb, NULL, NULL, NULL },
    { "", &xmSeparatorGadgetClass, 0, NULL, NULL, NULL, NULL, NULL, NULL },
    { "Properties...", &xmPushButtonGadgetClass, 0, NULL, NULL, filePropsCb, NULL, NULL, NULL },
    { NULL, NULL }
};

static MenuItemRec dir_popup_menu[] = {
    { "Open", &xmPushButtonGadgetClass, 0, NULL, NULL, dirOpenCb, NULL, NULL, NULL },
    { "", &xmSeparatorGadgetClass, 0, NULL, NULL, NULL, NULL, NULL, NULL },
    { "Rename...", &xmPushButtonGadgetClass, 0, NULL, NULL, fileRenameCb, NULL, NULL, NULL },
    { "Copy...", &xmPushButtonGadgetClass, 0, NULL, NULL, fileCopyCb, NULL, NULL, NULL },
    { "Move...", &xmPushButtonGadgetClass, 0, NULL, NULL, fileMoveCb, NULL, NULL, NULL },
    { "Link...", &xmPushButtonGadgetClass, 0, NULL, NULL, fileLinkCb, NULL, NULL, NULL },
    { "Delete", &xmPushButtonGadgetClass, 0, NULL, NULL, fileDeleteCb, NULL, NULL, NULL },
    { "", &xmSeparatorGadgetClass, 0, NULL, NULL, NULL, NULL, NULL, NULL },
    { "Properties...", &xmPushButtonGadgetClass, 0, NULL, NULL, filePropsCb, NULL, NULL, NULL },
    { NULL, NULL }
};

/*-----------------------------------------------------------------------------
  PRIVATE FUNCTIONS
-----------------------------------------------------------------------------*/

static int longestName(FileWindowRec *fw, XmFontList fontlist);
static void dropInit(FileRec *file);

static int longestName(FileWindowRec *fw, XmFontList fontlist)
{
 int i,l, longest = 0;
 XmString name;

 for (i=0; i<fw->n_files; i++)
 {
     name = XmStringCreateLocalized(fw->files[i]->name);
     if ((l = XmStringWidth(fontlist, name)) > longest)
	 longest = l;
     XmStringFree(name);
 }
 return longest;
}

static void dropInit(FileRec *file)
{
 file->drop_site = False;
 if (S_ISDIR(file->stats.st_mode))
 {
     if (permission(&file->stats, P_WRITE))  file->drop_site = True;
 }
 else if (permission(&file->stats, P_EXECUTE))
     file->drop_site = True;
 else if (file->type && file->type->drop_action[0])
     file->drop_site = True;

 if (file->drop_site)
     XmDropSiteRegister(file->form, drop_file_args, XtNumber(drop_file_args));
}

/*-----------------------------------------------------------------------------
  PUBLIC FUNCTIONS
-----------------------------------------------------------------------------*/

void initSelections(SelectionRec *sel)
{
 sel->file = (FileSpec *) XtMalloc((sel->n_alloc = 10) * sizeof(FileRec));
 if (!sel->file)  sel->n_alloc = 0;
 sel->n_sel = 0;
 sel->n_bytes = 0;
}

/*---------------------------------------------------------------------------*/

int getSelNr(FileWindowRec *fw, int item)
{
 int i;

 for (i=0; i < fw->selected.n_sel; i++)
 {
     if (fw->selected.file[i].nr == item)  return i;
 }
 return -1;
}

/*---------------------------------------------------------------------------*/

void unselectAll(FileWindowRec *fw)
{
 SelectionRec *sel = &fw->selected;
 int i;

 for (i=0; i < sel->n_sel; i++)
 {
     XTFREE(sel->file[i].name);
     if (fw->readable)
	 XtVaSetValues(fw->files[sel->file[i].nr]->form, XmNborderColor, winInfo.background, NULL);
 }
 sel->n_sel = 0;
 sel->n_bytes = 0;
}

/*---------------------------------------------------------------------------*/

void updateSelections(FileWindowRec *fw)
{
 SelectionRec *oldselp = &fw->selected, newsel;
 int i, j;

 newsel.n_sel = 0;  newsel.n_bytes = 0;
 newsel.file = (FileSpec *) XtMalloc((newsel.n_alloc = oldselp->n_alloc) * sizeof(FileSpec));
 if (newsel.file)
 {
     for (i=0; i < fw->n_files && oldselp->n_sel; i++)
     {
	 for (j=0; j < oldselp->n_sel; j++)
	 {
	     if (!strcmp(fw->files[i]->name, oldselp->file[j].name))
	     {
		 newsel.file[newsel.n_sel].nr = i;
		 newsel.file[newsel.n_sel++].name = oldselp->file[j].name;
		 newsel.n_bytes += fw->files[i]->stats.st_size;
		 memcpy(&oldselp->file[j], &oldselp->file[--oldselp->n_sel], sizeof(FileSpec));
	     }
	 }
     }
 }
 for (j=0; j < oldselp->n_sel; j++)  XTFREE(oldselp->file[j].name);
 XTFREE(oldselp->file);
 *oldselp = newsel;
}

/*---------------------------------------------------------------------------*/

SelFileNamesRec *getSelFiles(FileWindowRec *fw)
{
 SelFileNamesRec *fnames;
 FileRec *file;
 int i;

 if (!fw || !fw->selected.n_sel)  return NULL;
 fnames =  (SelFileNamesRec *) XtMalloc(sizeof(SelFileNamesRec));
 fnames->n_sel = fw->selected.n_sel;
 fnames->first = 0;
 fnames->names = (String *) XtMalloc(fnames->n_sel * sizeof(String));
 for (i=0; i < fnames->n_sel; i++)
     fnames->names[i] = XtNewString(fw->selected.file[i].name);
 fnames->directory = XtNewString(fw->directory);
 if (fnames->n_sel == 1)
 {
     file = fw->files[fw->selected.file[0].nr];
     fnames->icon = file->icon;
 }
 else  fnames->icon = icons[FILES_BM];
 fnames->target = NULL;
 fnames->conf_ovwr = resources.confirm_overwrite;
 fnames->update = False;
 fnames->shell = fw->shell;
 fnames->from_remote = False;
 fnames->op = 0;
 return fnames;
}

/*---------------------------------------------------------------------------*/

void freeSelFiles(SelFileNamesRec *fnames)
{
 int i;

 if (!fnames)  return;
 for (i=0; i < fnames->n_sel; i++)  XTFREE(fnames->names[i]);
 XTFREE(fnames->names);
 XTFREE(fnames->directory);
 XTFREE(fnames->target);
 XTFREE(fnames);
}

/*---------------------------------------------------------------------------*/

static int parseType(FILE *fp, char **pattern,
#ifdef MAGIC_HEADERS
	char **magic_type,
#endif
        char **icon, char **push_action, char **drop_action)
{
  static char s[MAXCFGLINELEN];
  int l;

 start:
  if (feof(fp)||!fgets(s, MAXCFGLINELEN, fp))
    return 0;
  l = strlen(s);
  if (s[l-1] == '\n')
    s[--l] = '\0';
  if (!l || *s == '#')
    goto start;
  if (!(*pattern = split(s, ':')))
    return -1;
#ifdef MAGIC_HEADERS
  if (**pattern == '<') {
    char *ptr;
    ptr = *pattern + 1;
    while(*ptr && (*ptr != '>' || ptr[-1] == '\\'))
	ptr++;
    if(*ptr != '>')
	return -1;
    *ptr = '\0';
    *magic_type = *pattern + 1;
    *pattern = ptr + 1;
  }
  else
    *magic_type = NULL;
#endif
  if (!(*icon = split(NULL, ':')))
    return -1;
  if (!(*push_action = split(NULL, ':')))
    return -1;
  if (!(*drop_action = split(NULL, ':')))
    return -1;
  return l;
}

/*---------------------------------------------------------------------------*/

static void readFileBitmaps(Widget shell)
{
  int i;

  for (i=0; i<n_types; i++)
    if (!types[i].icon[0])
      types[i].icon_pm = icons[FILE_BM];
    else if ((types[i].icon_pm = readIcon(shell, types[i].icon)).bm == None) {
#ifdef MAGIC_HEADERS
      fprintf(stderr, "%s: can't read icon for type %s%s%s%s%s%s\n", progname,
	      types[i].magic_type?"<":"",
	      types[i].magic_type?types[i].magic_type:"",
	      types[i].magic_type?">":"",
	      types[i].dir<0?"*":"", types[i].pattern,
	      types[i].dir>0?"*":"");
#else
      fprintf(stderr, "%s: can't read icon for type %s%s%s\n", progname,
	      types[i].dir<0?"*":"", types[i].pattern,
	      types[i].dir>0?"*":"");
      types[i].icon_pm = icons[FILE_BM];
#endif
    }
}

/*---------------------------------------------------------------------------*/

static void readFileTypes(Widget shell, String path)
{
  FILE *fp;
  char *pattern, *icon, *push_action, *drop_action;
#ifdef MAGIC_HEADERS
  char *magic_type;
#endif
  char s[MAXCFGSTRINGLEN];
  int i, l, p;
  
  n_types = 0;
  types = NULL;
  
  if (!(fp = fopen(path, "r"))) return;

  for (i=0; (p = parseType(fp, &pattern,
#ifdef MAGIC_HEADERS
		&magic_type,
#endif
		&icon, &push_action, &drop_action)) > 0; i++)
  {
      types = (TypeList) XTREALLOC(types, (i+1) * sizeof(TypeRec));
      l = strlen(pattern);
      if (pattern[0] == '*')
      {
	  types[i].dir = -1;
#ifdef MAGIC_HEADERS
	  strparse(s, pattern+1, "\\:<>");
#else
	  strparse(s, pattern+1, "\\:");
#endif
      }
      else if (pattern[l-1] == '*')
      {
	  types[i].dir = 1;
	  pattern[l-1] = '\0';
#ifdef MAGIC_HEADERS
	  strparse(s, pattern, "\\:<>");
#else
	  strparse(s, pattern, "\\:");
#endif
      }
      else
      {
	  types[i].dir = 0;
#ifdef MAGIC_HEADERS
	  strparse(s, pattern, "\\:<>");
#else
	  strparse(s, pattern, "\\:");
#endif
      }
      types[i].len = strlen(s);
      types[i].pattern = XtNewString(s);
#ifdef MAGIC_HEADERS
      if(magic_type)
	  types[i].magic_type = XtNewString(strparse(s, magic_type, "\\:"));
      else
	  types[i].magic_type = NULL;
#endif
      types[i].icon = XtNewString(strparse(s, icon, "\\:"));
      types[i].push_action = XtNewString(strparse(s, push_action, "\\:"));
      types[i].drop_action = XtNewString(strparse(s, drop_action, "\\:"));
  }

  if (p == -1)
    error(shell, "Error in configuration file", NULL);

  n_types = i;
  
  if (fclose(fp))
    sysError(shell, "Error reading configuration file:");

/*  readFileBitmaps(topShell);*/
}

/*---------------------------------------------------------------------------*/

#ifdef MAGIC_HEADERS
static TypeRec *fileType(char *name, char *magic_type)
#else
static TypeRec *fileType(char *name)
#endif
{
  int i, l = strlen(name);

  for (i = 0; i < n_types; i++) {
#ifdef MAGIC_HEADERS
    if (types[i].magic_type) {
      if(strcmp(types[i].magic_type, magic_type))
        continue;
      else if (!strcmp(types[i].pattern, "")) /* Empty pattern. */
        return types+i;
    }
#endif
    switch (types[i].dir) {
    case 0:
      if (!strcmp(name, types[i].pattern))
	return types+i;
      break;
    case 1:
      if (!strncmp(types[i].pattern, name, types[i].len))
	return types+i;
      break;
    case -1:
      if (l >= types[i].len && !strncmp(types[i].pattern, name+l-types[i].len, types[i].len))
	return types+i;
      break;
    }
  }
  return NULL;
}

/*-----------------------------------------------------------------------------
  PUBLIC FUNCTIONS
-----------------------------------------------------------------------------*/

/* initialise the file Windows module */
void initFileWindows(void)
{
 Atom *targets = (Atom *) XtMalloc(3 * sizeof(Atom));

 targets[0] = dragAtoms.filelist;
 targets[1] = dragAtoms.file;
 targets[2] = dragAtoms.string;
 drop_form_args[0].value = drop_file_args[0].value = (XtArgVal) targets;
 XtAppAddActions(app_context, file_actions, XtNumber(file_actions));

 form_translations = XtParseTranslationTable(form_translations_s);
 icon_form_args[0].value = text_form_args[0].value = winInfo.background;
 icon_form_args[2].value = text_form_args[1].value = (XtArgVal) XtParseTranslationTable(file_translations_s);
 icon_toggle_args[5].value = (XtArgVal) resources.file_icon_height;
 icon_toggle_args[6].value = (XtArgVal) resources.icon_font;
 icon_label_args[6].value = (XtArgVal) resources.icon_font;
 readFileTypes(topShell, resources.cfg_file);
#ifdef MAGIC_HEADERS
  magic_parse_file(resources.magic_file);
#endif

 if (!cpy.copies && !(cpy.copies = (CopyRec *) XtMalloc((cpy.n_alc = 100) * sizeof(CopyRec))))  cpy.n_alc = 0;
 copyInfo.dialog = NULL; 
}

/*---------------------------------------------------------------------------*/
/* Create a file Window at the specified path, in the specified format */

FileWindowRec *createFileWindow(String path, FileWindowRec *parent, FILE *fin, GeomRec *geom)
{
 FileWindowRec *fw;
 Widget actionBar, statusform, button;
 static char dir[MAXPATHLEN+12];
 XmString sample = XmStringCreateLocalized("8");
 Dimension height;
 MenuItemRec filePopupMenu[XtNumber(file_popup_menu)];
 MenuItemRec dirPopupMenu[XtNumber(dir_popup_menu)];
 MenuItemRec execPopupMenu[XtNumber(exec_popup_menu)];
 MenuItemRec formPopupMenu[XtNumber(file_pdwn_menu)];
 MenuItemRec filePulldownMenu[XtNumber(file_pdwn_menu)];
 MenuItemRec dirPulldownMenu[XtNumber(dir_pdwn_menu)];
 MenuItemRec windowPulldownMenu[XtNumber(wdw_pdwn_menu)];
 MenuItemRec optPulldownMenu[XtNumber(opt_pdwn_menu)];
 int i;

  /* put at front of linked list */
 fw = (FileWindowRec *) XtMalloc(sizeof(FileWindowRec));
 fw->next = file_windows;
 if (fin)
 {
     if (readFileWindow(fw, geom, fin))
     {
	 sysError(getAnyShell(), "Error reading startup file:");
	 XTFREE(fw);
	 return NULL;
     }
 }
 else  strcpy(fw->directory, path);

 if (!(fw->getwd = absolutePath(fw->directory)))
 {
     XTFREE(fw);
     sysError(getAnyShell(), "Can't open folder:");
     return NULL;
 }

 if ((fw->dev = findDev(fw->getwd)) != -1)
     if (mountDev(fw->dev, False))
     {
	 XTFREE(fw->getwd);
	 XTFREE(fw);
	 error(getAnyShell(), "Cannot mount device on", mntable.devices[fw->dev].def_mpoint);
	 return NULL;
     }

 strcpy(dir, "moxfm - ");
 if (chdir(fw->directory) || !getwd(&dir[strlen(dir)]))
 {
    chdir(user.home);
    XTFREE(fw->getwd);
    XTFREE(fw);
    sysError(getAnyShell(), "Can't open folder:");
    return NULL;
 }
 chdir(user.home);

 /* set up defaults */
 if (!fin)
 {
     geom->iconic = False;
     if (parent)
     {
	 fw->display_type = parent->display_type;
	 fw->sort_type = parent->sort_type;
	 fw->show_hidden = parent->show_hidden;
	 fw->dirs_first = parent->dirs_first;
     }
     else
     {
	 fw->display_type = resources.initial_display_type;
	 fw->sort_type = resources.default_sort_type;
	 fw->show_hidden = resources.show_hidden;
	 fw->dirs_first = resources.dirs_first;
     }
     fw->show_dirs = True;
 /* KMR */ /* AG removed inherited do_filter attribute */
     fw->do_filter = False;
     fw->dirFilter[0] = 0;
 }
 fw->files = NULL;
 fw->n_files = 0;
 fw->n_bytes = 0;
 fw->update = CHECK_DIR;
 fw->iconBoxCreated = False;
 fw->drag_source = False;
 fw->iconic = True;
 initSelections(&fw->selected);

 fw->shell = XtVaAppCreateShell("file window", "Moxfm", topLevelShellWidgetClass, dpy, XmNiconPixmap, icons[ICON_BM].bm, XmNiconMask, icons[ICON_BM].mask, XmNtitle, dir, XmNiconName, dir, XmNdeleteResponse, XmDO_NOTHING, XmNiconic, geom->iconic, NULL);
 file_windows = fw;

 if (fin)
     XtVaSetValues(fw->shell, XmNwidth, geom->width, XmNheight, geom->height, NULL);
 else if (!winInfo.fileXPos && !winInfo.fileYPos)
     XtVaSetValues(fw->shell, XmNgeometry, resources.init_file_geometry, NULL);
 else
 {
     Dimension xPos, yPos;

     xPos = winInfo.fileXPos;
     yPos = winInfo.fileYPos;
     if (xPos < 75)  xPos += winInfo.rootWidth - winInfo.fileWidth;
     if (yPos < 50)  yPos += winInfo.rootHeight - winInfo.fileHeight;
     xPos -= 75;
     yPos -= 50;
     XtVaSetValues(fw->shell, XmNgeometry, resources.file_geometry, XmNx, xPos, XmNy, yPos, NULL);
 }

 fw->cont = XtVaCreateManagedWidget("container", xmFormWidgetClass, fw->shell, XmNbackground, winInfo.background, NULL);

 fw->menu_bar = XmCreateMenuBar(fw->cont, "menubar", menu_bar_args, XtNumber(menu_bar_args));

 actionBar = XtVaCreateManagedWidget("actionbar", xmFormWidgetClass, fw->cont, XmNbackground, winInfo.background, XmNtopAttachment, XmATTACH_WIDGET, XmNtopWidget, fw->menu_bar, XmNleftAttachment, XmATTACH_FORM, XmNrightAttachment, XmATTACH_FORM, NULL);

 button = XtVaCreateManagedWidget("reread button", xmPushButtonGadgetClass, actionBar, XmNlabelType, XmPIXMAP, XmNlabelPixmap, icons[GLASSES_BM].bm, XmNleftAttachment, XmATTACH_FORM, XmNtopAttachment, XmATTACH_FORM, XmNbottomAttachment, XmATTACH_FORM, NULL);
 XtAddCallback(button, XmNactivateCallback, fileRefreshCb, (XtPointer) fw);

 button = XtVaCreateManagedWidget("home button", xmPushButtonGadgetClass, actionBar, XmNlabelType, XmPIXMAP, XmNlabelPixmap, icons[HOME_BM].bm, XmNleftAttachment, XmATTACH_WIDGET, XmNleftWidget, button, XmNtopAttachment, XmATTACH_FORM, XmNbottomAttachment, XmATTACH_FORM, NULL);
 XtAddCallback(button, XmNactivateCallback, goHomeCb, (XtPointer) fw);

 button = XtVaCreateManagedWidget("filewin button", xmPushButtonGadgetClass, actionBar, XmNlabelType, XmPIXMAP, XmNlabelPixmap, icons[NEWFILEWIN_BM].bm, XmNleftAttachment, XmATTACH_WIDGET, XmNleftWidget, button, XmNtopAttachment, XmATTACH_FORM, XmNbottomAttachment, XmATTACH_FORM, NULL);
 XtAddCallback(button, XmNactivateCallback, newFileWinCb, (XtPointer) fw);

 button = XtVaCreateManagedWidget("appwin button", xmPushButtonGadgetClass, actionBar, XmNlabelType, XmPIXMAP, XmNlabelPixmap, icons[NEWAPPWIN_BM].bm, XmNleftAttachment, XmATTACH_WIDGET, XmNleftWidget, button, XmNtopAttachment, XmATTACH_FORM, XmNbottomAttachment, XmATTACH_FORM, NULL);
 XtAddCallback(button, XmNactivateCallback, newAppWinCb, NULL);

 button = XtVaCreateManagedWidget("goup button", xmPushButtonGadgetClass, actionBar, XmNlabelType, XmPIXMAP, XmNlabelPixmap, icons[GOUP_BM].bm, XmNleftAttachment, XmATTACH_WIDGET, XmNleftWidget, button, XmNtopAttachment, XmATTACH_FORM, XmNbottomAttachment, XmATTACH_FORM, NULL);
 XtAddCallback(button, XmNactivateCallback, goUpCb, (XtPointer) fw);

 fw->dirfield = XtVaCreateManagedWidget("dirfield", xmTextFieldWidgetClass, actionBar, XmNfontList, (XtArgVal) resources.tty_font, XmNleftAttachment, XmATTACH_WIDGET, XmNleftWidget, button, XmNtopAttachment, XmATTACH_FORM, XmNrightAttachment, XmATTACH_FORM, XmNrightOffset, (XtPointer) (icons[BACKARR_BM].width + 12), XmNbottomAttachment, XmATTACH_FORM, NULL);
 XtAddCallback(fw->dirfield, XmNactivateCallback, changeDirCb, fw);
 XmTextFieldSetString(fw->dirfield, fw->directory);

 button = XtVaCreateManagedWidget("clear", xmPushButtonGadgetClass, actionBar, XmNlabelType, XmPIXMAP, XmNlabelPixmap, icons[BACKARR_BM].bm, XmNleftAttachment, XmATTACH_WIDGET, XmNleftWidget, fw->dirfield, XmNtopAttachment, XmATTACH_FORM, XmNrightAttachment, XmATTACH_FORM, XmNbottomAttachment, XmATTACH_FORM, NULL);
 XtAddCallback(button, XmNactivateCallback, clrDirFieldCb, fw);

 height = XmStringHeight(resources.status_font, sample) + 7;
 XmStringFree(sample);

 fw->form = XtVaCreateManagedWidget("form", xmScrolledWindowWidgetClass, fw->cont,  XmNscrollingPolicy, XmAUTOMATIC, XmNbackground, (XtArgVal) winInfo.background, XmNspacing, 0, XmNtranslations, form_translations, XmNheight, (XtArgVal) height, XmNbackground, winInfo.background, XmNtopAttachment, XmATTACH_WIDGET, XmNtopWidget, actionBar, XmNleftAttachment, XmATTACH_FORM, XmNrightAttachment, XmATTACH_FORM, XmNbottomAttachment, XmATTACH_FORM, XmNbottomOffset, height, NULL);

 XmDropSiteRegister(fw->form, drop_form_args, XtNumber(drop_form_args));

 statusform = XtVaCreateManagedWidget("statusform", xmFormWidgetClass, fw->cont, XmNbackground, (XtArgVal) winInfo.background, XmNtopAttachment, XmATTACH_WIDGET, XmNtopWidget, fw->form, XmNleftAttachment, XmATTACH_FORM, XmNleftOffset, 10, XmNrightAttachment, XmATTACH_FORM, XmNbottomAttachment, XmATTACH_FORM, NULL);

 fw->status = XtVaCreateManagedWidget("Status", xmLabelGadgetClass, statusform, XmNfontList, (XtArgVal) resources.status_font, XmNleftAttachment, XmATTACH_FORM, XmNtopAttachment, XmATTACH_FORM, XmNrightAttachment, XmATTACH_FORM, XmNbottomAttachment, XmATTACH_FORM, XmNalignment, XmALIGNMENT_BEGINNING, NULL);

 for (i=0; i<XtNumber(file_pdwn_menu); i++)
 {
     filePulldownMenu[i] = formPopupMenu[i] = file_pdwn_menu[i];
     filePulldownMenu[i].callback_data = formPopupMenu[i].callback_data = fw;
     formPopupMenu[i].accelerator = formPopupMenu[i].accel_text = NULL;
 }
 BuildMenu(fw->menu_bar, XmMENU_PULLDOWN, "File", 'F', False, filePulldownMenu);

 for (i=0; i<XtNumber(dir_pdwn_menu); i++)
 {
     dirPulldownMenu[i] = dir_pdwn_menu[i];
     dirPulldownMenu[i].callback_data = fw;
 }
 BuildMenu(fw->menu_bar, XmMENU_PULLDOWN, "Directory", 'D', False, dirPulldownMenu);

 for (i=0; i<XtNumber(wdw_pdwn_menu); i++)
 {
     windowPulldownMenu[i] = wdw_pdwn_menu[i];
     windowPulldownMenu[i].callback_data = fw;
 }
 BuildMenu(fw->menu_bar, XmMENU_PULLDOWN, "Windows", 'W', False, windowPulldownMenu);
 if (!mntable.n_dev)
     XtVaSetValues(windowPulldownMenu[2].object, XmNsensitive, False, NULL);

 for (i=0; i<XtNumber(opt_pdwn_menu); i++)
 {
     optPulldownMenu[i] = opt_pdwn_menu[i];
     optPulldownMenu[i].callback_data = fw;
 }
 BuildMenu(fw->menu_bar, XmMENU_PULLDOWN, "Options", 'O', False, optPulldownMenu);
 XmToggleButtonGadgetSetState(fw->icon_view_button = optPulldownMenu[0].object, (fw->display_type == Icons), False);
 XmToggleButtonGadgetSetState(fw->text_view_button = optPulldownMenu[1].object, (fw->display_type == Text), False);
 for (i=0; i<2; i++)
     XtVaSetValues(optPulldownMenu[i].object, XmNindicatorType, XmONE_OF_MANY, NULL);
 XmToggleButtonGadgetSetState(fw->sort_name_button = optPulldownMenu[3].object, (fw->sort_type == SortByName), False);
 XmToggleButtonGadgetSetState(fw->sort_size_button = optPulldownMenu[4].object, (fw->sort_type == SortBySize), False);
 XmToggleButtonGadgetSetState(fw->sort_date_button= optPulldownMenu[5].object, (fw->sort_type == SortByMTime), False);
 for (i=3; i<7; i++)
     XtVaSetValues(optPulldownMenu[i].object, XmNindicatorType, XmONE_OF_MANY, NULL);
 XmToggleButtonGadgetSetState(optPulldownMenu[7].object, fw->show_hidden, False);
 XmToggleButtonGadgetSetState(optPulldownMenu[8].object, fw->show_dirs, False);
 XmToggleButtonGadgetSetState(fw->dirs_first_button = optPulldownMenu[9].object, fw->dirs_first, False);
 if (!fw->show_dirs)
     XtVaSetValues(fw->dirs_first_button, XmNsensitive, False, NULL);

 XtManageChild(fw->menu_bar);
 fw->formPopup = BuildMenu(fw->form, XmMENU_POPUP, "SelFile actions", 0, False, formPopupMenu);

 for (i=0; i<XtNumber(file_popup_menu); i++)
 {
     filePopupMenu[i] = file_popup_menu[i];
     filePopupMenu[i].callback_data = fw;
 }
 fw->filePopup = BuildMenu(fw->form, XmMENU_POPUP, "File actions", 0, False, filePopupMenu);
/* fw->formPopup = fw->filePopup;*/
 fw->var_file_items[0] = filePulldownMenu[2].object;
 fw->var_file_items[1] = filePulldownMenu[3].object;
 fw->var_file_items[2] = filePulldownMenu[4].object;
 fw->var_file_items[3] = filePulldownMenu[6].object;
 fw->var_file_items[4] = formPopupMenu[2].object;
 fw->var_file_items[5] = formPopupMenu[3].object;
 fw->var_file_items[6] = formPopupMenu[4].object;
 fw->var_file_items[7] = formPopupMenu[6].object;
 for (i=0; i<XtNumber(dir_popup_menu); i++)
 {
     dirPopupMenu[i] = dir_popup_menu[i];
     dirPopupMenu[i].callback_data = fw;
 }
 fw->dirPopup = BuildMenu(fw->form, XmMENU_POPUP, "Directory actions", 0, False, dirPopupMenu);
 for (i=0; i<XtNumber(exec_popup_menu); i++)
 {
     execPopupMenu[i] = exec_popup_menu[i];
     execPopupMenu[i].callback_data = fw;
 }
 fw->execPopup = BuildMenu(fw->form, XmMENU_POPUP, "Executable actions", 0, False, execPopupMenu);

 if (!bitmaps_read)
 {
     readFileBitmaps(topShell);
     bitmaps_read = True;
 }

 return fw;
}

/*----------------------------------------------------------------------------*/

int saveFileWindow(FileWindowRec *fw, FILE *fout)
{
 size_t size;

 size = strlen(fw->directory);
 if (1 != fwrite(&size, sizeof(size_t), 1, fout))
     return 1;
 if (size != fwrite(fw->directory, sizeof(char), size, fout))
     return 1;
 if (1 != fwrite(&fw->display_type, sizeof(DisplayType), 1, fout))
     return 1;
 if (1 != fwrite(&fw->show_dirs, sizeof(Boolean), 1, fout))
     return 1;
 if (1 != fwrite(&fw->dirs_first, sizeof(Boolean), 1, fout))
     return 1;
 if (1 != fwrite(&fw->show_hidden, sizeof(Boolean), 1, fout))
     return 1;
 if (1 != fwrite(&fw->sort_type, sizeof(SortType), 1, fout))
     return 1;
 if (1 != fwrite(&fw->do_filter, sizeof(Boolean), 1, fout))
     return 1;
 if (fw->do_filter)
 {
     size = strlen(fw->dirFilter);
     if (1 != fwrite(&size, sizeof(size_t), 1, fout))
	 return 1;
     if (size != fwrite(fw->dirFilter, sizeof(char), size, fout))
	 return 1;
 }
 return writeGeometry(fw->shell, fw->iconic, fout);
}

/*----------------------------------------------------------------------------*/

int readFileWindow(FileWindowRec *fw, GeomRec *geom, FILE *fin)
{
 size_t size;

 if (1 != fread(&size, sizeof(size_t), 1, fin))
     return 1;
 if (size != fread(fw->directory, sizeof(char), size, fin))
     return 1;
 fw->directory[size] = 0;
 if (1 != fread(&fw->display_type, sizeof(DisplayType), 1, fin))
     return 1;
 if (1 != fread(&fw->show_dirs, sizeof(Boolean), 1, fin))
     return 1;
 if (1 != fread(&fw->dirs_first, sizeof(Boolean), 1, fin))
     return 1;
 if (1 != fread(&fw->show_hidden, sizeof(Boolean), 1, fin))
     return 1;
 if (1 != fread(&fw->sort_type, sizeof(SortType), 1, fin))
     return 1;
 if (1 != fread(&fw->do_filter, sizeof(Boolean), 1, fin))
     return 1;
 if (fw->do_filter)
 {
     if (1 != fread(&size, sizeof(size_t), 1, fin))
	 return 1;
     if (size != fread(fw->dirFilter, sizeof(char), size, fin))
	 return 1;
     fw->dirFilter[size] = 0;
 }
 return readGeometry(geom, fin);
}

/*----------------------------------------------------------------------------*/

void getFileIcon(FileRec *file)
{
 IconRec icon;

#ifdef MAGIC_HEADERS
 /* determine file type first, to allow special items like directories to
      have custom icons */
 file->type = fileType(file->name, file->magic_type);
 if (file->type)
     icon = file->type->icon_pm;
 else
     icon.bm = None;
#else
 file->type = NULL;
 icon.bm = None;
#endif

 if (icon.bm == None)
 {
     /* Symbolic link to non-existent file */
     if (S_ISLNK(file->stats.st_mode))
	 icon = icons[BLACKHOLE_BM];
     else if (S_ISDIR(file->stats.st_mode))
     {
	 if (file->sym_link)
	     icon = icons[DIRLNK_BM];
	 else if (!strcmp(file->name, ".."))
	     icon = icons[UPDIR_BM];
	 else  icon = icons[DIR_BM];
     }
     else if (file->stats.st_mode & (S_IXUSR | S_IXGRP | S_IXOTH))
     {
	 if (file->sym_link)  icon = icons[EXECLNK_BM];
	 else  icon = icons[EXEC_BM];
     }
     else
     {
#ifdef MAGIC_HEADERS
	 /* Already got file->type. */
#else
	 file->type = fileType(file->name);
#endif
	 if (file->type && file->type->icon_pm.bm != None)
	     icon = file->type->icon_pm;
	 else if (file->sym_link)  icon = icons[SYMLNK_BM];
	 else  icon = icons[FILE_BM];
     }
 }
 file->icon = icon;
}

/*----------------------------------------------------------------------------*/

void createIconDisplay(FileWindowRec *fw)
{
 Widget toggle;
 Dimension buttonwidth;
 FileRec *file;
 XmString label;
 int i;

 buttonwidth = setFileGeom(fw);
 for (i=0; i < fw->n_files; i++)
 {
     file = fw->files[i];

     if (getSelNr(fw, i) != -1)
	 icon_form_args[4].value = resources.select_color;
     else
	 icon_form_args[4].value = winInfo.background;
     icon_form_args[1].value = buttonwidth;

     if (file->sym_link)
	 icon_form_args[5].value = resources.linkname_color;
     else
	 icon_form_args[5].value = resources.filename_color;

     file->form = XtCreateManagedWidget(file->name, xmFormWidgetClass, fw->icon_box, icon_form_args, XtNumber(icon_form_args));

     getFileIcon(file);
     icon_toggle_args[4].value = (XtArgVal) file->icon.bm;
     dropInit(file);

     toggle = XtCreateManagedWidget("icon", xmLabelGadgetClass, file->form, icon_toggle_args, XtNumber(icon_toggle_args));

     icon_label_args[4].value = (XtArgVal) toggle;
     label = XmStringCreateLocalized(file->name);
     icon_label_args[5].value = (XtArgVal) label;

     XtCreateManagedWidget("label", xmLabelGadgetClass, file->form, icon_label_args, XtNumber(icon_label_args) );
     XmStringFree(label);
 }
}

/*----------------------------------------------------------------------------*/

void createTextDisplay(FileWindowRec *fw)
{
 FileRec *file;
 Widget label;
 struct passwd *pw;
 Dimension m_width, name_w, size_w, perm_w, own_w = 0, date_w, l;
 XmString *owners = NULL;
 XmString text;
 char name[FILENAME_MAX], s[11];
 int i;

 XtVaSetValues(fw->icon_box, XmNorientation, XmVERTICAL, XmNnumColumns, 1, NULL);
 text = XmStringCreateLocalized("m");
 m_width = XmStringWidth(resources.icon_font, text);
 XmStringFree(text);
 name_w = longestName(fw, resources.filename_font) + 2*m_width;
 size_w = m_width * 7;
 perm_w = m_width * 9;
 date_w = m_width * 20;
  
 if (resources.show_owner)
 {
     owners = (XmString *) XtMalloc(fw->n_files * sizeof(XmString));
     own_w = 0;
     for (i=0; i<fw->n_files; i++)
     { 
      /* bug fixed by hkarhune@hydra.helsinki.fi - Thanks */
	 if ((pw = getpwuid(fw->files[i]->stats.st_uid)) == NULL)
	 {
	     char tmp[11];
 	
	     sprintf(tmp, "%lu", (unsigned long) fw->files[i]->stats.st_uid);
	     owners[i] = XmStringCreateLocalized(tmp);
	 }
	 else
	     owners[i] = XmStringCreateLocalized(pw->pw_name);
	 l = XmStringWidth(resources.icon_font, owners[i]);
	 if (l > own_w)	own_w = l;
     }
     own_w += 2*m_width;
 }

 for (i=0; i < fw->n_files; i++)
 {
     file = fw->files[i];

     getFileIcon(file);

     if (getSelNr(fw, i) != -1)
	 text_form_args[3].value = resources.select_color;
     else
	 text_form_args[3].value = winInfo.background;

     if (S_ISDIR(file->stats.st_mode))
	 text_form_args[4].value = resources.textdir_color;
     else if (file->sym_link)
	 text_form_args[4].value = resources.linkname_color;
     else
	 text_form_args[4].value = resources.filename_color;

     file->form = XtCreateManagedWidget("file_form", xmFormWidgetClass, fw->icon_box, text_form_args, XtNumber(text_form_args));

     dropInit(file);

     if (S_ISDIR(file->stats.st_mode))
     {
	 sprintf(name, "[%s]", file->name);
	 text = XmStringCreateLocalized(name);
     }
     else
	 text = XmStringCreateLocalized(file->name);
     label = XtVaCreateManagedWidget("file_name", xmLabelGadgetClass, file->form, XmNlabelString, text, XmNfontList, resources.filename_font, XmNwidth, name_w, XmNalignment, XmALIGNMENT_BEGINNING, XmNleftAttachment, XmATTACH_FORM, XmNtopAttachment, XmATTACH_FORM, XmNbottomAttachment, XmATTACH_FORM, NULL);
     XmStringFree(text);

     if (resources.show_length)
     {
	 sprintf(s, "%lu", (unsigned long) file->stats.st_size);
	 text = XmStringCreateLocalized(s);
	 label = XtVaCreateManagedWidget("file_size", xmLabelGadgetClass, file->form, XmNlabelString, text, XmNfontList, resources.icon_font, XmNwidth, size_w, XmNalignment, XmALIGNMENT_END, XmNleftAttachment, XmATTACH_WIDGET, XmNleftWidget, label, XmNtopAttachment, XmATTACH_FORM, XmNbottomAttachment, XmATTACH_FORM, NULL);
	 XmStringFree(text);
     }

     if (resources.show_owner)
     {
	 label = XtVaCreateManagedWidget("file_owner", xmLabelGadgetClass, file->form, XmNlabelString, owners[i], XmNfontList, resources.icon_font, XmNwidth, own_w, XmNalignment, XmALIGNMENT_CENTER, XmNleftAttachment, XmATTACH_WIDGET, XmNleftWidget, label, XmNtopAttachment, XmATTACH_FORM, XmNbottomAttachment, XmATTACH_FORM, NULL);
	 XmStringFree(owners[i]);
     }

     if (resources.show_perms)
     {
	 makePermissionsString(s, file->stats.st_mode, file->sym_link);
	 text = XmStringCreateLocalized(s);
	 label = XtVaCreateManagedWidget("file_perms", xmLabelGadgetClass, file->form, XmNlabelString, text, XmNfontList, resources.icon_font, XmNwidth, perm_w, XmNalignment, XmALIGNMENT_CENTER, XmNleftAttachment, XmATTACH_WIDGET, XmNleftWidget, label, XmNtopAttachment, XmATTACH_FORM, XmNbottomAttachment, XmATTACH_FORM, NULL);
	 XmStringFree(text);
     }

     if (resources.show_date)
     {
	 text =
	   XmStringCreateLocalized(ctime(&file->stats.st_mtime));
	 label = XtVaCreateManagedWidget("file_date", xmLabelGadgetClass, file->form, XmNlabelString, text, XmNfontList, resources.icon_font, XmNwidth, date_w, XmNalignment, XmALIGNMENT_BEGINNING, XmNleftAttachment, XmATTACH_WIDGET, XmNleftWidget, label, XmNtopAttachment, XmATTACH_FORM, XmNrightAttachment, XmATTACH_FORM, XmNbottomAttachment, XmATTACH_FORM, NULL);
	 XmStringFree(text);
     }
 }
 if (resources.show_owner)  XTFREE(owners);
}

/*----------------------------------------------------------------------------*/

Boolean chFileDir(FileWindowRec *fw, String path)
{
 char dir[MAXPATHLEN+12], *cwd;
 int d;

 if (!(cwd = absolutePath(path)))
  {
     sysError(fw->shell, "Can't open folder:");
     return False;
 }

 d = findDev(cwd);
 if (d != fw->dev)
 {
     if (mountDev(d, False))
     {
	 XTFREE(cwd);
	 error(fw->shell, "Cannot mount device on", mntable.devices[d].def_mpoint);
	 return False;
     }
     umountDev(fw->dev, False);
     fw->dev = d;
 }

 strcpy(dir, "moxfm - ");
 if (chdir(path) || !getwd(&dir[strlen(dir)]))
 {
     chdir(user.home);
     XTFREE(cwd);
     sysError(fw->shell, "Can't open folder:");
     return False;
 }
 chdir(user.home);
 fw->getwd = cwd;

 zzz();
 strcpy(fw->directory, path);
 unselectAll(fw);
 if (fw->readable)  freeFileList(fw->files, fw->n_files);
 readFiles(fw);
 showFileDisplay(fw);
 XmTextFieldSetString(fw->dirfield, path);
 XtVaSetValues(fw->shell, XmNtitle, dir, XmNiconName, dir, NULL);
 wakeUp();
 return True;
}

/*----------------------------------------------------------------------------*/

Widget newFileWindow(String path, FileWindowRec *parent, FILE *fin)
{
 FileWindowRec *fw;
 GeomRec geom;

 if (!(fw = createFileWindow(path, parent, fin, &geom)))
     return None;
 readFiles(fw);
 XtAddEventHandler(fw->shell, StructureNotifyMask, True, (XtEventHandler) fileIconifyHandler, fw);
/* resizeFile() is called on first exposure and on resize events: */
 XtAddEventHandler(fw->form, StructureNotifyMask, True, (XtEventHandler) resizeFile, fw);
 XmAddWMProtocolCallback(fw->shell, wm_delete_window, closeFileCB, fw);
 XtRealizeWidget(fw->shell);
 if (fin)
     XtVaSetValues(fw->shell, XmNx, geom.x, XmNy, geom.y, NULL);
 else
     XtVaGetValues(fw->shell, XmNx, &winInfo.fileXPos, XmNy, &winInfo.fileYPos, NULL);
 XSetIconName(XtDisplay(fw->shell), XtWindow(fw->shell), fw->directory);
 return fw->shell;
}

/*---------------------------------------------------------------------------*/

/* Main procedure to create the display*/
void readFiles(FileWindowRec *fw)
{
 if ((fw->readable = readDirectory(fw)))
 {
     filterDirectory(fw, fw->show_dirs ? All : Files);
     sortDirectory(fw->files, fw->n_files, fw->sort_type, fw->dirs_first);
 }
}

/*---------------------------------------------------------------------------*/

Boolean showFileDisplay(FileWindowRec *fw)
{
 if (fw->iconBoxCreated)  XtDestroyWidget(fw->icon_box);

 fw->icon_box = XtVaCreateWidget("icon box", xmRowColumnWidgetClass, fw->form, XmNpacking, XmPACK_COLUMN, XmNorientation, XmHORIZONTAL, XmNbackground, winInfo.background, NULL);
 fw->iconBoxCreated = True;

 if (!fw->readable)
     XtVaCreateManagedWidget("Directory is unreadable", xmLabelGadgetClass, fw->icon_box, XmNfontList, resources.label_font, NULL);
 else
 {
     chdir(fw->directory);
     switch (fw->display_type)
     {
	 case Icons:	createIconDisplay(fw); break;
	 case Text:	createTextDisplay(fw);
     }
     chdir(user.home);
 }
 updateStatus(fw);
 XtManageChild(fw->icon_box);
 fw->update = CHECK_DIR;
 return True;
}

/*---------------------------------------------------------------------------*/

/* Update the display in the viewport */
void updateFileDisplay(FileWindowRec *fw, int reread)
{
 struct stat cur;
 FileList oldfiles = fw->files;
 int on_files = fw->n_files, i;
 Boolean reshow = False;

 if (reread == RESHOW || fw->update == RESHOW || stat(fw->directory, &cur) || cur.st_ctime > fw->stats.st_ctime || cur.st_mtime > fw->stats.st_mtime)
 {
     readFiles(fw);
     reshow = True;
 }
 else if (reread == CHECK_FILES || fw->update == CHECK_FILES)
 {
     readFiles(fw);
     if (fw->n_files != on_files)  /* Should never happen, but anyway ... */
	 reshow = True;
     else
     {
	 for (i=0; i<on_files; i++)
	 {
	     if (!S_ISDIR(fw->files[i]->stats.st_mode) &&
		 (fw->files[i]->stats.st_ctime > oldfiles[i]->stats.st_ctime || fw->files[i]->stats.st_mtime > oldfiles[i]->stats.st_mtime))
		 reshow = True;
	 }
     }
 }
 if (reshow && !fw->drag_source)
 {
     zzz();
     if (fw->readable)  freeFileList(oldfiles, on_files);
     updateSelections(fw);
     showFileDisplay(fw);
     wakeUp();
 }
 else if (reread || fw->update != CHECK_DIR || reshow)
 {
     if (fw->readable)  freeFileList(fw->files, fw->n_files);
     fw->files = oldfiles;
     fw->n_files = on_files;
     if (reshow)  fw->update = RESHOW;
     else  fw->update = CHECK_DIR;
 }
}

/*---------------------------------------------------------------------------*/

Dimension setFileGeom(FileWindowRec *fw)
{
 Dimension width, itemwidth, scrollbarwidth;
 Widget vsb;
 int ncols, nrows;

 XtVaGetValues(fw->form, XmNwidth, &width, XmNverticalScrollBar, &vsb, NULL);
 XtVaGetValues(vsb, XmNwidth, &scrollbarwidth, NULL);
 XtVaSetValues(fw->dirfield, XmNwidth, width-150, NULL);

 width -= scrollbarwidth + 20;
 itemwidth = longestName(fw, resources.icon_font);
 if (itemwidth < resources.file_icon_width)
     itemwidth = resources.file_icon_width;
 ncols = width / (itemwidth + 4);
 if (!ncols)  ncols = 1;
 nrows = (fw->n_files-1)/ncols + 1;
 XtVaSetValues(fw->icon_box, XmNnumColumns, nrows, NULL);
 return itemwidth;
}

/*---------------------------------------------------------------------------*/

void resizeFile(Widget form, XtPointer client_data, XEvent *event)
{
 showFileDisplay((FileWindowRec *) client_data);
}

/*---------------------------------------------------------------------------*/

void fileIconifyHandler(Widget shell, XtPointer client_data, XEvent *event)
{
 FileWindowRec *fw = (FileWindowRec *) client_data;

 if (event->type == MapNotify)
     fw->iconic = False;
 else if (event->type == UnmapNotify)
     fw->iconic = True;
}

/*----------------------------------------------------------------------------
  Intelligent update - only update the windows needed.
  Use markForUpdate() to explicitly mark a directory for update.
  Call intUpdate() to execute all the actions.
-----------------------------------------------------------------------------*/

void markForUpdate(String path, int reread)
{
  FileWindowRec *fw;
  char dir[MAXPATHLEN];

  if (!chdir(path) && getwd(dir))
  {
      for (fw = file_windows; fw; fw = fw->next)
	  if (!strcmp(fw->getwd, dir))
	      fw->update = reread;
  }
  chdir(user.home);
}

/*---------------------------------------------------------------------------*/

void intUpdate(int reread)
{
 FileWindowRec *fw;

 for (fw = file_windows; fw; fw = fw->next)
     updateFileDisplay(fw, reread);
}

/*---------------------------------------------------------------------------*/

void updateStatus(FileWindowRec *fw)
{
 char s[1024];
 XmString info;
 int i;
 long bsize, ksize, free, free_blocks;
 int res;
#ifdef USE_STATVFS
 struct statvfs stat;
#else
 struct statfs stat;
#endif

 if (fw->do_filter)
     sprintf(s, "[%s]: ", fw->dirFilter);
 else
     s[0] = 0;

 sprintf(s+strlen(s),
	 "%ld byte%s in %d item%s",
	 fw->n_bytes, fw->n_bytes == 1 ? "" : "s",
	 fw->n_files, fw->n_files == 1 ? "" : "s");
 if (fw->selected.n_sel)
 {
     for (i=0; i<NVARITEMS; i++)
	 fillIn(fw->var_file_items[i]);
     sprintf(s+strlen(s), ", %ld byte%s in %d selected item%s",
	     fw->selected.n_bytes, fw->selected.n_bytes == 1 ? "" : "s",
	     fw->selected.n_sel, fw->selected.n_sel == 1 ? "" : "s");
 }
 else
 {
     for (i=0; i<NVARITEMS; i++)
	 grayOut(fw->var_file_items[i]);
 }

#ifdef USE_STATVFS
 res = statvfs(fw->directory, &stat);
#else
 res = statfs(fw->directory, &stat);
#endif

 if (!res)
 {
     if (geteuid())
	 free_blocks = stat.f_bavail;
     else
	 free_blocks = stat.f_bfree;

     if ((bsize = stat.f_bsize / 1024))
	 free = free_blocks * bsize;
     else
     {
	 ksize = 1024 / stat.f_bsize;
	 free = free_blocks / ksize;
     }
     sprintf(&s[strlen(s)], ", %ld KBytes available", free);
 }

 info = XmStringCreateLocalized(s);
 XtVaSetValues(fw->status, XmNlabelString, (XtArgVal) info, NULL);
 XmStringFree(info);
}

/*---------------------------------------------------------------------------*/

void closeFileProc(FileWindowRec *fw)
{
 FileWindowRec *fwi;
 int i;

 XtDestroyWidget(fw->shell);
 umountDev(fw->dev, False);
 if (fw == file_windows)  file_windows = fw->next;
 else
 {
     for (fwi=file_windows; (fwi); fwi=fwi->next)
     {
	 if (fwi->next == fw)
	 {
	     fwi->next = fw->next;
	     break;
	 }
     }
 }
 for (i=0; i < fw->selected.n_sel; i++)
     XTFREE(fw->selected.file[i].name);
 XTFREE(fw->selected.file);
 for (i=0; i < fw->n_files; i++)  XTFREE(fw->files[i]);
 XTFREE(fw->files);
 XTFREE(fw->getwd);
 XTFREE(fw);
}

/*---------------------------------------------------------------------------*/

void fileSelect(FileWindowRec *fw, int item)
{
 SelectionRec *sel = &fw->selected;

 if (!(sel->n_alloc))
     sel->file = (FileSpec *) XtMalloc((sel->n_alloc = 10) * sizeof(FileRec));
 if (sel->file)
 {
     unselectAll(fw);
     sel->n_sel = 1;
     sel->file[0].nr = item;
     sel->file[0].name = XtNewString(fw->files[item]->name);
     sel->n_bytes = fw->files[item]->stats.st_size;
     XtVaSetValues(fw->files[item]->form, XmNborderColor, resources.select_color, NULL);
 }
 else
 {
     sel->n_sel = 0;
     sel->n_bytes = 0;
 }
 updateStatus(fw);
}

/*---------------------------------------------------------------------------*/

void fileToggle(FileWindowRec *fw, int item)
{
 SelectionRec *sel = &fw->selected;
 int i, nr;

 if ((nr = getSelNr(fw, item)) != -1)
 {
     XTFREE(sel->file[nr].name);
     memcpy(&sel->file[nr], &sel->file[--(sel->n_sel)], sizeof(FileSpec));
     sel->n_bytes -= fw->files[item]->stats.st_size;
     XtVaSetValues(fw->files[item]->form, XmNborderColor, winInfo.background, NULL);
 }
 else
 {
     if ((i = sel->n_sel++) >= sel->n_alloc)
	 sel->file = (FileSpec *) XTREALLOC(sel->file, (sel->n_alloc += 10) * sizeof(FileSpec));
     sel->file[i].nr = item;
     sel->file[i].name = XtNewString(fw->files[item]->name);
     sel->n_bytes += fw->files[item]->stats.st_size;
     XtVaSetValues(fw->files[item]->form, XmNborderColor, resources.select_color, NULL);
 }
 updateStatus(fw);
}

/*---------------------------------------------------------------------------*/

void selectAll(FileWindowRec *fw)
{
 SelectionRec *sel = &fw->selected;
 FileRec *file;
 int i, nr = 0;

 if (fw->n_files > sel->n_alloc)
     sel->file = (FileSpec *) XTREALLOC(sel->file, (sel->n_alloc = fw->n_files) * sizeof(FileSpec));
 sel->n_bytes = 0;

 for (i=0; i < fw->n_files; i++)
 {
     file = fw->files[i];
     if (strcmp(file->name, ".."))
     {
	 sel->file[nr].name = XtNewString(file->name);
	 sel->file[nr++].nr = i;
	 sel->n_bytes += file->stats.st_size;
	 XtVaSetValues(file->form, XmNborderColor, resources.select_color, NULL);
     }
     else
	 XtVaSetValues(file->form, XmNborderColor, winInfo.background, NULL);
 }
 sel->n_sel = nr;
}
