////////////////////////////////////////////////////////////////////////////////
//
//	Class : akComponent
//
//	Author : Andy Jefferson
//
//	Description :
//	 	The akComponent class is the base class for all
//		ak components. It provides for storage of the base
//		widget and widget name, as well as manage/unmanage
//		facilities, methods for widget destruction, as well
//		as resource specification.
//		All actual ak Component classes are derived from the
//		akComponent class and the first two parameters required
//		by their constructors are the parent widget, and the
//		widget 'name'. The parent widget is used as a parent for
//		the ak Component being created, and the 'name' being
//		the widget name (for resources etc).
//		By default the component is unmanaged.
//		The "Bubble Help" functions provide a small popup window displaying
//		a help message just below the component if the pointer is held
//		over the component for > a time period (default = 1.5 sec).
//
//	Public Methods :
//		(Constructor) requires :-
//			name		Name used in resource specifications
//		busy()		Sets cursor to stopwatch.
//		ready()		Sets cursor back to normal.
//		manage()	Manages the component
//		unmanage()	Unmanages the component
//		activate()	Makes the component sensitive
//		deactivate()	Makes the component insensitive
//		installHelp()	Adds bubble-help to the component
//		removeHelp()	Removes bubble-help from the component
//
//	Limitations :
//
////////////////////////////////////////////////////////////////////////////////
// This example code is ADAPTED from the book:
//
// Object-Oriented Programming with C++ and OSF/Motif, 2nd Edition
//     by
// Douglas Young
// Prentice Hall, 1995
// ISBN 0-13-20925507
//
// Copyright 1995 by Prentice Hall
// All Rights Reserved
//
// Permission to use, copy, modify, and distribute this software for 
// any purpose except publication and without fee is hereby granted, provided 
// that the above copyright notice appear in all copies of the software.
////////////////////////////////////////////////////////////////////////////////
#ifndef _akComponent_h
#define _akComponent_h
#ifdef __cplusplus

#include <std/string.h>
#include <bool.h>
#include <Xm/Xm.h>

class akComponent
{
	public:

		virtual ~akComponent(void);

		const Widget baseWidget(void) {return (_w);}

		// Manage/Unmanage

		virtual void manage(void);
		virtual void unmanage(void);

		// Activate/Deactivate

		virtual void activate(void);
		virtual void deactivate(void);

		// Busy/Ready

		virtual void busy(void);
		virtual void ready(void);

		// Bubble-Help

		virtual	bool		installHelp(string helpText);
		virtual	void		removeHelp(void);
		virtual	void		setHelpTime(double secs);

		// Class name

		virtual const char *const className(void)
			{return ("akComponent");}

	protected:

		// Data members

		char		*_name;
		Widget		_w;

		Cursor		_busyCursor;

		// Protect constructor to prevent direct instantiation

		akComponent(const char *name);

		void installDestroyHandler(void); // Hook for derived classes

		// Called by widgetDestroyedCallback if basewidget is killed

		virtual void widgetDestroyed(void); 

		// Loads component's default resources into database

		void setDefaultResources(const Widget,const char **);

		// Retrieve resources for this class from the resource manager

		void getResources(const XtResourceList resources ,int no_res);

	private:

		// Bubble help callbacks/data

		static	void	enterCB(Widget,XtPointer,XEvent *,Boolean *);
		static	void	leaveCB(Widget,XtPointer,XEvent *,Boolean *);
		static	void	enterTimeout(XtPointer,XtIntervalId);

		XtIntervalId	_id;
		string		_helpText;
		double		_helpTime;
		Widget		_helpWidget;
		Widget		_helpLabel;

		// Interface between XmNdestroyCallback and this class

		static void widgetDestroyedCallback(Widget,XtPointer,XtPointer);
};

#endif
#endif
