////////////////////////////////////////////////////////////////////////////////
// This example code is ADAPTED from the book:
//
// Object-Oriented Programming with C++ and OSF/Motif, 2nd Edition
//    by
// Douglas Young
// Prentice Hall, 1995
// ISBN 0-13-20925507
//
// Copyright 1995 by Prentice Hall
// All Rights Reserved
//
// Permission to use, copy, modify, and distribute this software for 
// any purpose except publication and without fee is hereby granted, provided 
// that the above copyright notice appear in all copies of the software.
////////////////////////////////////////////////////////////////////////////////
#include "akComponent.h"
#include <X11/cursorfont.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <Xm/Xm.h>
#include <Xm/Label.h>
#include <Xm/MwmUtil.h>
#include <cassert>
#include <string>
#include <cstdio>
#include <iostream>
////////////////////////////////////////////////////////////////////////////////
//
//	Constructor
//
////////////////////////////////////////////////////////////////////////////////
akComponent::akComponent(const char *name)
	: _helpTime(1.5),
	  _helpWidget((Widget)0),
	  _helpLabel((Widget)0),
	  _id((XtIntervalId)0),
	  _busyCursor(0)
{
	_w = (Widget)0;
	if (name)
	  {
	   _name = new char[strlen(name)+1];
	   strcpy(_name,name);
	  }
	else
	  _name = (char *)0;
}
////////////////////////////////////////////////////////////////////////////////
//
//	Destructor
//
////////////////////////////////////////////////////////////////////////////////
akComponent::~akComponent()
{
//	Make sure the widget hasn't already been destroyed
  
	if (_w) 
	  {
           // Remove destroy callback so Xt can't call the callback
           // with a pointer to an object that has already been freed
  
  	   XtRemoveCallback(_w, 
  			    XmNdestroyCallback,
  			    &akComponent::widgetDestroyedCallback,
  			    (XtPointer)this);	
  	  }

	if (_w)
	  XtDestroyWidget(_w);
	if (_name)
	  delete _name;
	if (_id != (XtIntervalId)0)
	  XtRemoveTimeOut(_id);
	if (_busyCursor != 0)
	  XFreeCursor(XtDisplay(_w),_busyCursor);
}
////////////////////////////////////////////////////////////////////////////////
//
//	Widget Destroyed
//
////////////////////////////////////////////////////////////////////////////////
void akComponent::widgetDestroyedCallback(Widget, 
                                           XtPointer clientData, 
                                           XtPointer)
{
	akComponent *obj = (akComponent *)clientData;	

	obj->widgetDestroyed();
}
////////////////////////////////////////////////////////////////////////////////
//
//	Widget Destroyed
//
////////////////////////////////////////////////////////////////////////////////
void akComponent::widgetDestroyed()
{
	_w = (Widget)0;
}
////////////////////////////////////////////////////////////////////////////////
//
//	Destruction Mechanism installation
//
////////////////////////////////////////////////////////////////////////////////
void akComponent::installDestroyHandler()
{
	assert(_w);
	XtAddCallback(_w,XmNdestroyCallback,
			&akComponent::widgetDestroyedCallback, 
			(XtPointer)this);
}
////////////////////////////////////////////////////////////////////////////////
//
//	Manage
//
////////////////////////////////////////////////////////////////////////////////
void akComponent::manage()
{
	assert(_w);
	assert(XtHasCallbacks(_w,XmNdestroyCallback) == XtCallbackHasSome);
	XtManageChild(_w);
}
////////////////////////////////////////////////////////////////////////////////
//
//	Unmanage
//
////////////////////////////////////////////////////////////////////////////////
void akComponent::unmanage(void)
{
	assert(_w);
	XtUnmanageChild(_w);
}
////////////////////////////////////////////////////////////////////////////////
//
//	Activate
//
////////////////////////////////////////////////////////////////////////////////
void akComponent::activate()
{
	assert(_w);
	XtSetSensitive(_w,TRUE);
}
////////////////////////////////////////////////////////////////////////////////
//
//	Deactivate
//
////////////////////////////////////////////////////////////////////////////////
void akComponent::deactivate()
{
	assert(_w);
	XtSetSensitive(_w,FALSE);
}
////////////////////////////////////////////////////////////////////////////////
//
//	Busy
//
////////////////////////////////////////////////////////////////////////////////
void akComponent::busy()
{
	Display		*display;
	Colormap	cmap;
	XColor		fgd_color,bgd_color,ignore;

	assert(_w);
	display = XtDisplay(_w);

	if (display != (Display *)0 && XtWindow(_w) != (Window)0)
	  {
//	   Allocate the cursor if not already allocated

	   if (!_busyCursor)
	     {
	      _busyCursor = XCreateFontCursor(display,XC_watch);
	      cmap = DefaultColormap(display,DefaultScreen(display));
	      XAllocNamedColor(display,cmap,"Blue",&fgd_color,&ignore);
	      XAllocNamedColor(display,cmap,"Yellow",&bgd_color,&ignore);
	      XRecolorCursor(display,_busyCursor,&fgd_color,&bgd_color);
	     }

	   XDefineCursor(display,XtWindow(_w),_busyCursor);
	  }
	XmUpdateDisplay(_w);
}
////////////////////////////////////////////////////////////////////////////////
//
//	Ready
//
////////////////////////////////////////////////////////////////////////////////
void akComponent::ready()
{
	Display *display;

	assert(_w);
	display = XtDisplay(_w);

	if (display != (Display *)0 && XtWindow(_w) != (Window)0)
	  XUndefineCursor(display,XtWindow(_w));
	XmUpdateDisplay(_w);
}
////////////////////////////////////////////////////////////////////////////////
//
//	get resources
//
////////////////////////////////////////////////////////////////////////////////
void akComponent::getResources(const XtResourceList resources,
                                int numResources )
{
	assert(_w);
	assert(resources);

// 	Retrieve the requested resources relative to the 
// 	parent of this object's base widget
  
	XtGetSubresources(XtParent(_w),(XtPointer)this, 
 			  _name,className(),
 			  resources,numResources,
 			  NULL,0);
}
////////////////////////////////////////////////////////////////////////////////
//
//	set resources
//
////////////////////////////////////////////////////////////////////////////////
void akComponent::setDefaultResources(const Widget w, 
                                       const char **resourceSpec)
{
	int		i;	
	Display 	*dpy=XtDisplay(w);  // Retrieve the display pointer
	XrmDatabase 	rdb = NULL;         // A resource database
 
	// Create an empty resource database
  
	rdb = XrmGetStringDatabase("");
  
// 	Add the Component resources, prepending the name of the component
  
	i = 0;
	while(resourceSpec[i])
	  {
	   char buf[1000];
  
	   sprintf(buf,"*%s%s",_name,resourceSpec[i++]);
	   XrmPutLineResource(&rdb,buf);
	  }

// 	Merge them into the Xt database, with lowest precedence

	if (rdb)
	  {
	   XrmDatabase db = XtDatabase(dpy);
	   XrmCombineDatabase(rdb,&db,FALSE); 
	  }
}
////////////////////////////////////////////////////////////////////////////////
//
//	Install bubble help
//
////////////////////////////////////////////////////////////////////////////////
bool	akComponent::installHelp(string help)
{
	bool	already_installed = false;

	if (!_w)
	  return false;

//	Save help text

	if (_helpText.size() > 0)
	   already_installed = true;
	_helpText = help;

	if (already_installed)
	  return true;

//	Add event handlers for help

	XtAddEventHandler(_w,EnterWindowMask,FALSE,
			  &akComponent::enterCB,(XtPointer)this);
	XtAddEventHandler(_w,LeaveWindowMask,FALSE,
			  &akComponent::leaveCB,(XtPointer)this);
	XtAddEventHandler(_w,ButtonPressMask,FALSE,
			  &akComponent::leaveCB,(XtPointer)this);

	return true;
}
////////////////////////////////////////////////////////////////////////////////
//
//	Remove bubble help
//
////////////////////////////////////////////////////////////////////////////////
void	akComponent::removeHelp()
{
//	Check if already installed

	if (_helpText.size() == 0)
	  return;

//	Remove event handlers

	XtRemoveEventHandler(_w,EnterWindowMask,FALSE,
			     &akComponent::enterCB,(XtPointer)this);
	XtRemoveEventHandler(_w,LeaveWindowMask,FALSE,
			     &akComponent::leaveCB,(XtPointer)this);
	XtRemoveEventHandler(_w,ButtonPressMask,FALSE,
			     &akComponent::leaveCB,(XtPointer)this);

//	Remove any existing timeout

	if (_id)
	  XtRemoveTimeOut(_id);
	_id = (XtIntervalId)0;

//	Popdown the help window (if visible)

	if (_helpWidget)
	  {
	   if (XtIsRealized(_helpWidget))
	     XtPopdown(_helpWidget);
	  }

//	Remove help text

	_helpText.erase();
}
////////////////////////////////////////////////////////////////////////////////
//
//	Set help time interval
//
////////////////////////////////////////////////////////////////////////////////
void	akComponent::setHelpTime(double secs)
{
	if (secs > 0)
	  _helpTime = secs;
}
////////////////////////////////////////////////////////////////////////////////
//
//	Enter Window Callback
//
////////////////////////////////////////////////////////////////////////////////
void	akComponent::enterCB(Widget,XtPointer clientData,XEvent *,Boolean *)
{
	akComponent	*obj = (akComponent *)clientData;

//	Set up a timeout to be called in _helpTime secs
//	Need to check for duplicates for when menu popped above widget, and
//	ends up sending 2 Window Enter events !!

	if (!obj->_id)
	  obj->_id =
	    XtAppAddTimeOut(XtWidgetToApplicationContext(obj->_w),
			    (int)obj->_helpTime*1000,
			    (XtTimerCallbackProc)&akComponent::enterTimeout,
			    (XtPointer)obj);
}
////////////////////////////////////////////////////////////////////////////////
//
//	Enter Window Timeout
//
////////////////////////////////////////////////////////////////////////////////
void	akComponent::enterTimeout(XtPointer clientData,XtIntervalId)
{
	akComponent	*obj = (akComponent *)clientData;
	Position	xloc,yloc;
	Position	xpos,ypos;
	Dimension	height;
	int		status;
	Pixel		fgd,bgd;
	XColor		the_color,ignore;
	Colormap	cmap;
	Display		*display;
	Arg		args[10];
	Dimension	screenHt,labelHt;

	obj->_id = (XtIntervalId)0;

//	Create the help widget if not yet created

	if (!obj->_helpWidget)
	  {
	   XtSetArg(args[0],XmNallowShellResize,TRUE);
	   obj->_helpWidget =
		XtCreatePopupShell("BubbleHelp",
				   transientShellWidgetClass,
				   obj->_w,args,1);

	   // Set background to be yellow, foreground to be blue

	   display = XtDisplay(obj->_w);
	   cmap = DefaultColormap(display,DefaultScreen(display));
	   status =
	     XAllocNamedColor(display,cmap,"Yellow",&the_color,&ignore);
	   if (!status)
	     XtVaGetValues(obj->_w,XmNbackground,&bgd,NULL);
	   else
	     bgd = the_color.pixel;
	   status =
	     XAllocNamedColor(display,cmap,"Blue",&the_color,&ignore);
	   if (!status)
	     XtVaGetValues(obj->_w,XmNforeground,&fgd,NULL);
	   else
	     fgd = the_color.pixel;

	   // Remove any decorations also

	   XtVaSetValues(obj->_helpWidget,
			 XmNmwmDecorations,MWM_DECOR_ALL | MWM_DECOR_BORDER |
				MWM_DECOR_RESIZEH | MWM_DECOR_TITLE |
				MWM_DECOR_MENU | MWM_DECOR_MINIMIZE |
				MWM_DECOR_MAXIMIZE,
			 NULL);

	   obj->_helpLabel =
		XtVaCreateManagedWidget("HelpText",
					xmLabelWidgetClass,
					obj->_helpWidget,
					XmNforeground,fgd,
					XmNbackground,bgd,
					NULL);
	  }
	XmString	label_string;
	label_string = XmStringCreateLtoR((char *)obj->_helpText.c_str(),XmFONTLIST_DEFAULT_TAG);
	XtVaSetValues(obj->_helpLabel,XmNlabelString,label_string,NULL);
	XmStringFree(label_string);

//	Set position of help window (either just below, else just above)

	XtVaGetValues(obj->_w,XmNheight,&height,NULL);
	XtTranslateCoords(obj->_w,0,0,&xloc,&yloc);
	xpos = xloc;
	screenHt = HeightOfScreen(XtScreen(obj->_w));
	XtVaGetValues(obj->_helpLabel,XmNheight,&labelHt,NULL);

	ypos = yloc + height + 4;
	if (ypos+labelHt > screenHt)
	  ypos = yloc - labelHt - 4;

	XtVaSetValues(obj->_helpWidget,XmNx,xpos,XmNy,ypos,NULL);

//	Popup the help window

	XtPopup(obj->_helpWidget,XtGrabNone);
}
////////////////////////////////////////////////////////////////////////////////
//
//	Leave Window Callback
//
////////////////////////////////////////////////////////////////////////////////
void	akComponent::leaveCB(Widget,XtPointer clientData,XEvent *,Boolean *)
{
	akComponent	*obj = (akComponent *)clientData;

//	Popdown the help window

	if (obj->_helpWidget)
	  {
	   if (XtIsRealized(obj->_helpWidget))
	     XtPopdown(obj->_helpWidget);
	  }

//	Remove any existing timeout

	if (obj->_id)
	  XtRemoveTimeOut(obj->_id);
	obj->_id = (XtIntervalId)0;
}
