////////////////////////////////////////////////////////////////////////////////
//
//	Class : mxMailBox
//
//	Author : Andy Jefferson
//
//	Description :
//		mxMailBox is a representation of the data in a mailbox.
//		It consists of an array (vector) of messages (mxMsgSummary).
//		Additionally, the mailbox has a filename, is of a particular
//		type (INBOX/OUTBOX/FOLDER) and the messages can be sorted in
//		several ways (by DATE/SUBJECT/AUTHOR/SIZE).
//
//	Public Methods :
//		(Constructor) requires :
//			filename	The filename that the mailbox reflects
//			type		The mailbox type.
//
//		read()		Reads the mailbox to get the store up-to-date.
//		write()		Write the store back to the file (overwriting
//				its existing contents). Any messages tagged
//				for deletion are deleted at this point.
//		sortByAuthor()	Sorts the array of messages by author.
//		sortBySubject()	Sorts the array of messages by subject.
//		sortByDate()	Sorts the array of messages by date (default).
//		sortBySize()	Sorts the array of messages by message size.
//		insert()	Insert a message into the box.
//		remove()	Remove a message from the box.
//		modify()	Update a message in the box.
//		noOfMsgsLatest()	Returns the no of msgs in THE FILE.
//		noOfMsgsLatestRead()	Returns the no of msgs in the FILE that
//					are 'read'.
//		noOfMsgs()		Returns the no of current msgs
//		noOfMsgsRead()		Returns the no of current msgs that
//					are 'read'.
//		filename()	Returns the mailbox filename.
//		type()		Returns the mailbox type.
//		sortMethod()	Returns the mailbox sort method.
//		messageText()	Returns the text for the specified message.
//		message()	Returns the specified mxMsgSummary object.
//
//	Limitations :
//
////////////////////////////////////////////////////////////////////////////////
#ifndef mxMailBox_h
#define mxMailBox_h
#ifdef __cplusplus

#include <vector>
#include <algorithm>
#include <string>
#include <time.h>
#include "mxMsgSummary.h"

typedef enum
{
	INBOX,OUTBOX,FOLDER
} mxMailBoxType;

typedef enum
{
	SORT_BY_DATE,SORT_BY_SUBJECT,SORT_BY_AUTHOR,SORT_BY_SIZE
} mxMsgSummarySortMethod;

class mxMailBox
{
	public:

		// Constructor and destructor

		mxMailBox();
		mxMailBox(string filename,mxMailBoxType type);
		mxMailBox& operator=(const mxMailBox&);
		~mxMailBox();

		// Access functions

		int			noOfMsgs() const;
		int			noOfMsgsRead() const;

		int			noOfMsgsLatest() const;
		int			noOfMsgsLatestRead() const;

		const string&		filename() const;
		mxMailBoxType		type() const;
		mxMsgSummarySortMethod	sortMethod() const;
		string			messageText(int msg_no) const;
		const mxMsgSummary&	message(int msg_no) const;

		// Mail box update functions

		void		read();
		void		write();

		void		sortByDate();
		void		sortByAuthor();
		void		sortBySubject();
		void		sortBySize();

		void		insert(const mxMsgSummary& msg);
		void		remove(int msg_no);
		void		modify(int msg_no,const mxMsgSummary& msg);

		friend ostream& operator<< (ostream& s,mxMailBox& box);

	private:

		// Data members

		time_t			_readTime;
		string			_filename;
		mxMailBoxType		_type;
		mxMsgSummarySortMethod	_sortMethod;
		vector<mxMsgSummary>	_messages;

		// Functions used internally in finding start of a message

		static	bool	AtStartOfMailMessage(char *);
		static	int	LengthOfNextPart(char *);
		static	int	GetWordFromString(char *,int,char *,int,int *);
};

inline
const mxMsgSummary&	mxMailBox::message(int msg_no) const
{
	if (msg_no < 0 || msg_no >= _messages.size())
	  return _messages[0];
	else
	  return _messages[msg_no];
}
inline
int		mxMailBox::noOfMsgs() const
	{return _messages.size();}
inline
int		mxMailBox::noOfMsgsRead() const
{
	int	no_read=0;
	for (int i=0;i<_messages.size();i++)
	  if (_messages[i].status())
	    no_read++;
	return no_read;
}
inline
const string&	mxMailBox::filename() const
	{return _filename;}
inline
mxMailBoxType	mxMailBox::type() const
	{return _type;}
inline
mxMsgSummarySortMethod	mxMailBox::sortMethod() const
	{return _sortMethod;}
inline
void	mxMailBox::insert(const mxMsgSummary& msg)
	{_messages.push_back(msg);}
inline
void	mxMailBox::remove(int msg_no)
{
	if (msg_no < 0 || msg_no >= _messages.size())
	  return;
	else
	  _messages.erase(_messages.begin()+msg_no);
}
inline
void	mxMailBox::modify(int msg_no,const mxMsgSummary& msg)
{
	if (msg_no < 0 || msg_no >= _messages.size())
	  return;
	else
	  *(_messages.begin()+msg_no) = msg;
}

#endif
#endif
