/*
# X-BASED VECTOR GLOBE PROJECTION
#
#	vgp.c
#
###
#
#  Copyright (c) 1994	Frederick (Rick) A Niles, niles@axp745.gsfc.nasa.gov
#
#			All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

#include "vgp.h"
#include <X11/StringDefs.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#include <Xm/Xm.h>
#include <Xm/DrawingA.h>
#include <Xm/PushB.h>
#include <Xm/Form.h>
#include <Xm/Text.h>

/*
** Global Variables
*/
static int scale_change=1;
static int bBusyAnimating;
static int screenClear;

int main(int argc, char *argv[])
{
  static image_data ImageData;
  Widget toplevel, canvas, layout, slider,
                start, full, text;
  Arg wargs[10];
  short n;

  get_input(argc,argv,&ImageData);

  toplevel = XtInitialize(argv[0], "vgp", NULL, 0,(Cardinal *) &argc,argv);
  

  layout = XtCreateManagedWidget("layout",
				 xmFormWidgetClass,
				 toplevel, NULL, 0);
  n = 0;
  XtSetArg(wargs[n], XmNwidth, 400); n++;
  XtSetArg(wargs[n], XmNheight, 400); n++;

  canvas = XtCreateManagedWidget("canvas",
				 xmDrawingAreaWidgetClass,
				 layout, wargs, n);
  slider = (Widget) make_slider(layout, &ImageData, "Speed",
				(void *)  slider_moved);

  start = XtCreateManagedWidget("Start",
				 xmPushButtonWidgetClass,
				 layout, NULL, 0);
  full = XtCreateManagedWidget("Full Size",
				 xmPushButtonWidgetClass,
				 layout, NULL, 0);
  text = XtCreateManagedWidget("Text",
				 xmTextWidgetClass,
				 layout, NULL, 0);
  
  n = 0;
  XtSetArg(wargs[n], XmNbottomAttachment, XmATTACH_FORM); n++;
  XtSetArg(wargs[n], XmNleftAttachment,   XmATTACH_FORM); n++;
  XtSetArg(wargs[n], XmNrightAttachment,  XmATTACH_FORM); n++;
  XtSetValues(full, wargs, n); 
  n = 0;
  XtSetArg(wargs[n], XmNbottomAttachment,   XmATTACH_WIDGET); n++;
  XtSetArg(wargs[n], XmNbottomWidget,       full); n++;
  XtSetArg(wargs[n], XmNleftAttachment,  XmATTACH_FORM); n++;
  XtSetArg(wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
  XtSetValues(start, wargs, n); 
  n = 0;
  XtSetArg(wargs[n], XmNbottomAttachment,   XmATTACH_WIDGET); n++;
  XtSetArg(wargs[n], XmNbottomWidget,       start); n++;
  XtSetArg(wargs[n], XmNleftAttachment,  XmATTACH_FORM); n++;
  XtSetArg(wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
  XtSetValues(slider, wargs, n);
  n = 0;
  XtSetArg(wargs[n], XmNtopAttachment,   XmATTACH_FORM); n++;
  XtSetArg(wargs[n], XmNleftAttachment,  XmATTACH_FORM); n++;
  XtSetArg(wargs[n], XmNrightAttachment, XmATTACH_FORM); n++;
  XtSetArg(wargs[n], XmNbottomAttachment,XmATTACH_WIDGET); n++;
  XtSetArg(wargs[n], XmNbottomWidget,    slider); n++;
  XtSetValues(canvas, wargs, n);


  init_data(canvas, &ImageData);
  XtAddCallback(canvas, XmNexposeCallback,(XtCallbackProc) redisplay,
		&ImageData);
  XtAddCallback(canvas, XmNresizeCallback,(XtCallbackProc) resize, &ImageData);
  XtAddCallback(start,  XmNactivateCallback,(XtCallbackProc) restart_anim,
		&ImageData);
  XtAddCallback(full,   XmNactivateCallback,(XtCallbackProc) full_size,
		&ImageData);
  XtAddEventHandler(text, KeyPressMask, FALSE,
                    (XtPointer) keybd_event,          &ImageData);
  XtAddEventHandler(canvas, ButtonPressMask, FALSE,
                    (XtPointer) start_rubber_band,    &ImageData);
  XtAddEventHandler(canvas, Button1MotionMask, FALSE,
                    (XtPointer) track_button1_motion, &ImageData);
  XtAddEventHandler(canvas, Button2MotionMask, FALSE,
                    (XtPointer) track_rubber_band,    &ImageData);
  XtAddEventHandler(canvas, Button3MotionMask, FALSE,
                    (XtPointer) track_pan_motion,     &ImageData);
  XtAddEventHandler(canvas, ButtonReleaseMask, FALSE,
		    (XtPointer) end_rubber_band,      &ImageData);
  XtRealizeWidget(toplevel);
  /*
   * Establish a passive grab, for any button press.
   * Force the sprite to stay within the canvas window, and
   * change the sprite to a cross_hair.
   */
   XGrabButton(XtDisplay(canvas), AnyButton, AnyModifier, 
               XtWindow(canvas), TRUE, 
               ButtonPressMask | ButtonMotionMask | 
               ButtonReleaseMask,
               GrabModeAsync, GrabModeAsync,
               XtWindow(canvas), 
               XCreateFontCursor(XtDisplay(canvas),
                                 XC_crosshair));


  scale_change = 1;
  XtRealizeWidget(toplevel);
  full_size(full, &ImageData);
  resize(canvas, &ImageData, NULL);
  XtMainLoop();
  return 0;
}

/**************** init_data *************/

void init_data(Widget w, image_data_ptr data)
{
  set_window_size(w, data);
  /*
   * Find out how many colors we have to work with, and  
   * create a default, writable, graphics context.
   */
  data->ncolors = XDisplayCells(XtDisplay(w), 
                                XDefaultScreen(XtDisplay(w)));

  data->gc = XCreateGC(XtDisplay(w),
                       DefaultRootWindow(XtDisplay(w)),
                       (unsigned long) NULL,NULL); 
  /*
   *  Initialize the pixmap to NULL.
   */
  data->pix = (Pixmap) NULL;
  data->ana_c = 0;
  data->drawing = w;
  data->ana_delay = INIT_SPEED;
  data->iid = (XtIntervalId) NULL;
  data->rotation[0] = 2.;
  data->rotation[1] = -0.7;
  data->rotation[2] = -2.86;
  create_rot_matrix(data);
  data->start_T[0][0] = 1;
  data->start_T[0][1] = 0;
  data->start_T[0][2] = 0;
  data->start_T[1][0] = 0;
  data->start_T[1][1] = 1;
  data->start_T[1][2] = 0;
  data->start_T[2][0] = 0;
  data->start_T[2][1] = 0;
  data->start_T[2][2] = 1;
  create3DSphere(data);

}

/****************************************/
/********* Display Section **************/
/****************************************/


/********************************/
void resize(Widget w, image_data_ptr data, caddr_t call_data)
{
  set_window_size(w, data);
  scale_change = 1;
  create_image(w, data);
}

/****************************************/
void full_size (Widget w, image_data_ptr data)
{
 double radius;

 radius = (double) (min(data->width,data->height)) / 2.0;
 data->zoom   =  radius;
 data->pan.x  =  radius;
 data->pan.y  =  radius;

 scale_change = 1;
 create_image(data->drawing, data);
}
/****************************************/
void create_image (Widget w, image_data_ptr data)
{
  int num;
  int i;
  point_block planes, vector;
  point_block_ptr blocks[2];

  planes.points = (XPoint *) calloc((size_t) 3*(INC+1), sizeof(XPoint));
  vector.points = (XPoint *) calloc((size_t) data->vec_c, sizeof(XPoint));
  blocks[0] = &planes;
  blocks[1] = &vector;
  for(i = 0; i < 2; i++)
      if(blocks[i] == NULL)
	{
	  printf("Could not allocate memory...\n");
	  exit(1);
	}
  planes.npoints = 0;
  num = INC;
  TD_to_screen(data->T, data->zoom, data->pan, &data->xyplane[0][0],
	       &num, &(planes.points[planes.npoints]), 0);
  planes.npoints += num;
  planes.points[planes.npoints].x = 0;
  planes.points[planes.npoints++].y = 0;
  num = INC;
  TD_to_screen(data->T,data->zoom,data->pan,&(data->yzplane[0][0]),&num,
	       &(planes.points[planes.npoints]),0);
  planes.npoints += num;
  planes.points[planes.npoints].x = 0;
  planes.points[planes.npoints++].y = 0;
  num = INC;
  TD_to_screen(data->T,data->zoom,data->pan,&(data->xzplane[0][0]),&num,
	       &(planes.points[planes.npoints]),0);
  planes.npoints += num;
  
  
  vector.npoints = 0;
  num = data->vec_c;
  TD_to_screen(data->T,data->zoom,data->pan,&(data->vector[0][0]),&num,
	       &(vector.points[vector.npoints]),0);
  vector.npoints += num;

  if(scale_change)
    scale_label(data);
/*******end Calculations ******/
  
  /*
   * Clear the window.
   */
  if(XtIsRealized(w))
    {
      XSetForeground(XtDisplay(w), data->gc,
		     BlackPixelOfScreen(XtScreen(w)));
      XClearArea(XtDisplay(w), XtWindow(w), 0, 0, 0, 0, TRUE);
      draw_to(w, data, XtWindow(w), blocks, 2);
    }
  clear_pixmap(w, data);
  screenClear = 1;
  draw_to(w, data, data->pix, blocks, 2);
  free(planes.points);
  free(vector.points);
}
/******************************/
void draw_to(Widget w, image_data_ptr data, Drawable draw,
	     point_block_ptr *block, int nblocks)
{
  int i, num;
  XPoint axes_sc[6];
  double paxes[3][3] = IDEN_MAT;
  char names[1][3] = {'X','Y','Z'};

  num = 3;
  TD_to_screen(data->T,data->zoom,data->pan,&(paxes[0][0]),&num,axes_sc,1);
  for(i = 0; i < 3; i++)
    {
      /*  XDrawLine (XtDisplay(w), draw, data->gc,
	  (int) data->pan.x,(int) data->pan.y, axes_sc[i].x,axes_sc[i].y);*/
      XDrawString(XtDisplay(w), draw, data->gc,axes_sc[i].x,axes_sc[i].y,
		  &(names[0][i]), 1);
    }
  XDrawString(XtDisplay(w), draw, data->gc,
	      1,10,data->label, strlen(data->label));

  XDrawArc(XtDisplay(w), draw, data->gc, (int) data->pan.x- (int) data->zoom,
	   (int) data->pan.y-(int) data->zoom,
	   (int) (2.* data->zoom),
	   (int) (2.* data->zoom),
	   0, 360*64);

  for (i = 0; i < nblocks; i++)
    {
      /*** Change color here. ****/
      draw_block_of_lines(w, data, draw, block[i]->points, block[i]->npoints);
    }
}
/******************************/
void scale_label(image_data_ptr data)
{
  static int  nchars, min_dim;
  XPoint top_left, bottom_right;
  double vec1[3], vec2[3], vec3[3], ang, english;

  scale_change = 0;
  min_dim = min(data->height, data->width);
  
  if(min_dim == data->height)
    {
      sprintf(data->label,"From top to bottom:");
      nchars = strlen(data->label);
      top_left.x = min_dim / 2;
      top_left.y = 1;
      bottom_right.x = min_dim /2;
      bottom_right.y = min_dim;
    }
      else
	{
	  sprintf(data->label,"From left to right:");
	  nchars = strlen(data->label);
	  top_left.x = 1;
	  top_left.y = min_dim / 2;
	  bottom_right.x = min_dim;
	  bottom_right.y = min_dim / 2;
	}
  screen_to_3D(top_left.x, top_left.y,
	       data->zoom, data->pan, data->T, vec1);
  screen_to_3D(bottom_right.x, bottom_right.y,
	       data->zoom, data->pan, data->T, vec2);
  norm_vec(vec1);
  norm_vec(vec2);
  ang = acos(vdot(vec1,vec2));
  if(ang < 1.0)
    {
      vcross(vec1,vec2,vec3);
      ang = asin( sqrt(vdot(vec3,vec3)) );
    }
  sprintf(&(data->label[nchars])," %le radians,",ang );
  nchars = strlen(data->label);
  english = ang * 180 / PI;
  if( english > 1.)
    {
      sprintf(&(data->label[nchars])," %lf degrees",english );
      nchars = strlen(data->label);
    } 
  else
    {
      english *= 60.;
      if(english > 1.)
	  sprintf(&(data->label[nchars])," %lf arc-minutes",english );
      else
	{
	  english *= 60.;
	  sprintf(&(data->label[nchars]),"%lf arc-seconds",english );
	}
    }        
}



/****************************************/
/********** Slider Section **************/
/****************************************/

/*******************************************/
Widget make_slider(Widget parent,image_data_ptr data,
		   char *name, void *callback ) 
/*  void (*callback)();*/
{
  Widget  w;
  int     n;
  Arg     wargs[10];
  XmString title;
  /*
   * Create an XmScale widget.
   */
  title = XmStringCreate("Animation Speed", XmSTRING_DEFAULT_CHARSET);
  n = 0;
  XtSetArg(wargs[n], XmNminimum, 100); n++;
  XtSetArg(wargs[n], XmNmaximum, 300); n++;
  XtSetArg(wargs[n], XmNorientation, XmHORIZONTAL); n++;
  XtSetArg(wargs[n], XmNtitleString, title); n++;
  XtSetArg(wargs[n], XmNvalue, 200); n++;

  w = XtCreateManagedWidget(name, xmScaleWidgetClass, 
                            parent, wargs, n);
  /*
   * Add callbacks to be invoked when the slider moves.
   */
  XtAddCallback(w, XmNvalueChangedCallback,
		(XtCallbackProc) callback,data);
  XtAddCallback(w, XmNdragCallback,
		(XtCallbackProc) callback, data);

  return (w);
}

/**********************************/
void slider_moved(Widget w, image_data_ptr data, 
		  XmScaleCallbackStruct *call_data)
{
  data->ana_delay = (int) pow(10.0,(4.0 - ((double) call_data->value)/100.0)); 
  if(data->ana_delay < 1)
    {
      printf("prob\n");
      data->ana_delay = 1;
    }
}

/****************************************/
/******** Rubber Band Section ***********/
/****************************************/

/***************************************/
XtPointer keybd_event(Widget w, image_data_ptr data, XEvent *event)
{
  printf("here: %u \n",event->xkey.keycode);
}
/***************************************/
XtPointer start_rubber_band(Widget w, image_data_ptr data, XEvent *event)
{
  int i,j;

  while(bBusyAnimating) { }

  data->last.x  =  data->start.x = event->xbutton.x;
  data->last.y  =  data->start.y = event->xbutton.y;
  
  if(event->xbutton.button == 1)
    {
      for(i = 0;i < 3; i++)
	for(j = 0;j < 3; j++)
	  data->start_T[i][j] = data->T[i][j];
    }

  if(event->xbutton.button == 2)
    {
      if(event->xbutton.state)
	full_size(w,data);
      else
	  xor_mode(data, 1);
    }

}
/***************************************/
XtPointer track_button1_motion(Widget w, image_data_ptr data, XEvent *event)
{
  /*
   * Update the endpoints.
   */
  data->last.x  =  event->xbutton.x;
  data->last.y  =  event->xbutton.y;

  while(bBusyAnimating) { }

  move_sphere(data);
  create_image(w, data);
}
/***************************************/
XtPointer track_rubber_band(Widget w, image_data_ptr data, XEvent *event)
{ 
  /*
   * Draw once to clear the previous line.
   */
  draw_rectangle(w,data);
  /*
   * Update the endpoints.
   */
  data->last.x  =  event->xbutton.x;
  data->last.y  =  event->xbutton.y;
  /*
   * Draw the new rectangle.
   */
  draw_rectangle(w,data);
 }
/***************************************/
XtPointer track_pan_motion(Widget w, image_data_ptr data, XEvent *event)
{
  /*
   * Update the endpoints.
   */
  data->pan.x += (event->xbutton.x - data->last.x);
  data->pan.y += (event->xbutton.y - data->last.y);
  data->last.x  =  event->xbutton.x;
  data->last.y  =  event->xbutton.y;

  create_image(w, data);
}
/******************************************/
XtPointer end_rubber_band(w, data, event)
   Widget            w;
   image_data *data;
   XEvent           *event;
{
  XGCValues values;
  int max_dis,screen_sq;
  double ratio;

  while(bBusyAnimating) { }
  if(event->xbutton.button == 1)
    {
      data->last.x  =  event->xbutton.x;
      data->last.y  =  event->xbutton.y;
      move_sphere(data);
      resize(w, data, NULL);
    }
  if(event->xbutton.button == 2)
    {
  /*
   * Draw once to clear the previous rectangle.
   */
      draw_rectangle(w, data);
  /*
   * Return GC to Normal Mode.
   */
      values.function = GXcopy;
      values.line_style = LineSolid;
      XChangeGC(XtDisplay(w), data->gc,
		GCFunction | GCLineStyle, &values);
      max_dis =
      max((data->last.y - data->start.y),
	  (data->last.x - data->start.x));
      if(max_dis > 0) /** down and to the right square **/
	{
	  screen_sq = min(data->width, data->height);
	  ratio = (double) screen_sq / (double) max_dis;
	  data->pan.x = (data->pan.x - data->start.x) * ratio;
	  data->pan.y = (data->pan.y - data->start.y) * ratio;
	  data->zoom *=  ratio;
	  scale_change = 1;
	  create_image(data->drawing, data);
	}
    }
}
  
/****************************************/
/********* Animation Section ************/
/****************************************/

/******************************************/
void restart_anim(Widget w,image_data_ptr data, XmAnyCallbackStruct *call_data)
{
  create_image(data->drawing, data);
  if(data->vec_c > 0)
    {
      if(data->ana_c > 0)
	XtRemoveTimeOut(data->iid);
      data->ana_c = 0;
      do_animation(data, NULL);
    }
}

/***************************************/
XtTimerCallbackProc do_animation(data, id)
XtIntervalId *id;
image_data *data;
{
  
  int num;
  static int oldnum;
  XPoint vec;
  static XPoint oldvec;
  XGCValues values;
  static char label[80];

  bBusyAnimating = 1;
     
  num = 1;
  TD_to_screen(data->T,data->zoom,data->pan,
	       &(data->vector[data->ana_c][0]),&num,
	       &vec,0);
  if(vec.x == 0 && vec.y ==0 )
    num = 0;
  xor_mode(data, 0);

    /**** Clear Last Vector *********/

  if(data->ana_c > 0 && oldnum  && !screenClear)
    {
      if(XtIsRealized(data->drawing))
	{
	  XDrawLine (XtDisplay(data->drawing), XtWindow(data->drawing),
		     data->gc, (int) data->pan.x,(int) data->pan.y,
		     oldvec.x,oldvec.y);
	  XDrawString(XtDisplay(data->drawing), XtWindow(data->drawing),
		      data->gc, 1, 20, label, strlen(label));

	}
      XDrawLine (XtDisplay(data->drawing), data->pix, data->gc,
		 (int) data->pan.x,(int) data->pan.y,
		 oldvec.x,oldvec.y);
      XDrawString(XtDisplay(data->drawing), data->pix, data->gc,
		  1, 20, label, strlen(label));
    }
  if(num)
    {
      
      sprintf(label,"Current sample number = %3d",data->ana_c);
      if(XtIsRealized(data->drawing))
	{
	  XDrawLine (XtDisplay(data->drawing), XtWindow(data->drawing),
		     data->gc,
		     (int) data->pan.x,(int) data->pan.y,
		     vec.x,vec.y);
	  XDrawString(XtDisplay(data->drawing), XtWindow(data->drawing),
		      data->gc, 1, 20, label, strlen(label));

	}
      XDrawLine (XtDisplay(data->drawing), data->pix, data->gc,
		 (int) data->pan.x,(int) data->pan.y,
		 vec.x,vec.y);
      XDrawString(XtDisplay(data->drawing), data->pix, data->gc,
		  1, 20, label, strlen(label));

      screenClear = 0;
      oldvec = vec; 
    }
  oldnum = num;
  data->ana_c++;
  values.function = GXcopy;
  XChangeGC(XtDisplay(data->drawing), data->gc,
	    GCFunction, &values);

  if(data->ana_c < data->vec_c)
    data->iid = XtAddTimeOut(data->ana_delay,
			     (XtTimerCallbackProc) do_animation, data);
    
  bBusyAnimating = 0;
}

/****************************************/
/****************************************/
/****************************************/
/*
** get input
*/

FILE *get_input(int argc,char *argv[], image_data *data)
{
  FILE *fp;

  int filearg = 1;
  long i=0;

  if(argc > 1)
    {
      if((fp=fopen(argv[filearg],"r"))==NULL) 
	{
	  printf("cannot open data file\n");
	  exit(1);
	}
      i = 0;
      while(fscanf(fp,"%lf %lf %lf \n",
		   &(data->vector[i][0]),
		   &(data->vector[i][1]),
		   &(data->vector[i][2])
		   ) == 3 && i < MAXSIZE)
	{
	  norm_vec(&(data->vector[i][0]));
	  i++;
	}    
      printf("  There were %d rows of data read...\n",i);
    }

    data->vec_c = i;

    return fp;
}
