/* Map Widget - a map drawing window for Xt/Motif
   Copyright (C) 1998 Andrew Skypeck
  
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "MapP.h"
#include "MapBasemap.h"
#include "MapDraw.h"
#include "MapProject.h"

#if USE_SHM
#include <sys/ipc.h>
#include <sys/shm.h>
#include <X11/extensions/XShm.h>
#endif

/* #include <sys/socket.h> */
#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#if USE_GUILE
#include <guile/gh.h>
#endif

/* resource default values so we get a nice map without external resources */
static float default_center_lon      = 0.0;  /* Greenwich meridian */
static float default_major_grid_size = 10.0;
static float default_minor_grid_size = 2.0;

/* these are the map resources, we also inherit those of Core and Primative */
static XtResource resources[] = {
#define offset(field) XtOffsetOf(MapRec, field)
  {
    XmNprojection,
    XmCProjection,
    XmRMapProjection,
    sizeof(XmProjection),
    offset(map.projection),
    XtRImmediate,
    (XtPointer) XmMILLER_PROJECTION
  },
  {
    XmNcenterLon,
    XmCCenterLon,
    XmRFloat,
    sizeof(XmRFloat),
    offset(map.center_lon),
    XmRFloat,
    (XtPointer) &default_center_lon,
  },
  {
    XmNmajorGridSize,
    XmCMajorGridSize,
    XmRFloat,
    sizeof(XmRFloat),
    offset(map.major_grid_size),
    XmRFloat,
    (XtPointer) &default_major_grid_size
  },
  {
    XmNminorGridSize,
    XmCMinorGridSize,
    XmRFloat,
    sizeof(XmRFloat),
    offset(map.minor_grid_size),
    XmRFloat,
    (XtPointer) &default_minor_grid_size
  },
  {
    XmNdrawCoastline,
    XmCDrawCoastline,
    XmRBoolean,
    sizeof(Boolean),
    offset(map.draw_coastline),
    XtRImmediate,
    (XtPointer) False
  },
  {
    XmNdrawLakes,
    XmCDrawLakes,
    XmRBoolean,
    sizeof(Boolean),
    offset(map.draw_lakes),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmNdrawRivers,
    XmCDrawRivers,
    XmRBoolean,
    sizeof(Boolean),
    offset(map.draw_rivers),
    XtRImmediate,
    (XtPointer) False
  },
  {
    XmNdrawBoundaries,
    XmCDrawBoundaries,
    XmRBoolean,
    sizeof(Boolean),
    offset(map.draw_boundaries),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmNdrawGrid,
    XmCDrawGrid,
    XmRBoolean,
    sizeof(Boolean),
    offset(map.draw_grid),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmNoceanColor,
    XmCOceanColor,
    XmRPixel,
    sizeof(Pixel),
    offset(map.ocean_color),
    XtRImmediate,
    (XtPointer) XtDefaultForeground
  },
  {
    XmNlandColor,
    XmCLandColor,
    XmRPixel,
    sizeof(Pixel),
    offset(map.land_color),
    XtRImmediate,
    (XtPointer) XtDefaultForeground
  },
  {
    XmNcoastlineColor,
    XmCCoastlineColor,
    XmRPixel,
    sizeof(Pixel),
    offset(map.coastline_color),
    XtRImmediate,
    (XtPointer) XtDefaultForeground
  },
  {
    XmNlakeColor,
    XmCLakeColor,
    XmRPixel,
    sizeof(Pixel),
    offset(map.lake_color),
    XtRImmediate,
    (XtPointer) XtDefaultForeground
  },
  {
    XmNriverColor,
    XmCRiverColor,
    XmRPixel,
    sizeof(Pixel),
    offset(map.river_color),
    XtRImmediate,
    (XtPointer) XtDefaultForeground
  },
  {
    XmNboundaryColor,
    XmCBoundaryColor,
    XmRPixel,
    sizeof(Pixel),
    offset(map.boundary_color),
    XtRImmediate,
    (XtPointer) XtDefaultForeground
  },
  {
    XmNgridColor,
    XmCGridColor,
    XmRPixel,
    sizeof(Pixel),
    offset(map.grid_color),
    XtRImmediate,
    (XtPointer) XtDefaultForeground
  },
  {
    XmNzoomBoxColor,
    XmCZoomBoxColor,
    XmRPixel,
    sizeof(Pixel),
    offset(map.zoom_box_color),
    XtRImmediate,
    (XtPointer) XtDefaultForeground
  },
  {
    XmNmapDir,
    XmCMapDir,
    XmRString,
    sizeof(String),
    offset(map.map_dir),
    XmRString,

/* define on compile line to be in datadir */
#ifndef MAPDATADIR
#define MAPDATADIR "./mapdata"
#endif

    MAPDATADIR
  },
  {
    XmNcoastlinesFile,
    XmCCoastlinesFile,
    XmRString,
    sizeof(String),
    offset(map.coastlines_file),
    XmRString,
    "coastlines.map"
  },
  {
    XmNlakesFile,
    XmCLakesFile,
    XmRString,
    sizeof(String),
    offset(map.lakes_file),
    XmRString,
    "lakes.map"
  },
  {
    XmNriversFile,
    XmCRiversFile,
    XmRString,
    sizeof(String),
    offset(map.rivers_file),
    XmRString,
    "rivers.map"
  },
  {
    XmNBoundariesFile,
    XmCBoundariesFile,
    XmRString,
    sizeof(String),
    offset(map.boundaries_file),
    XmRString,
    "boundaries.map"
  },
  {
    XmNfontList,
    XmCFontList,
    XmRFontList,
    sizeof(XmFontList),
    offset(map.font_list),
    XmRString,
    "fixed"
  },
  {
    XmNuseShm,
    XmCUseShm,
    XmRBoolean,
    sizeof(Boolean),
    offset(map.use_shm),
    XtRImmediate,
    (XtPointer) False
  },
  {
    XmNbeforeResetCallback,
    XtCCallback,
    XtRCallback,
    sizeof(XtPointer),
    offset(map.before_reset_callbacks),
    XtRCallback,
    NULL
  },
  {
    XmNafterResetCallback,
    XtCCallback,
    XtRCallback,
    sizeof(XtPointer),
    offset(map.after_reset_callbacks),
    XtRCallback,
    NULL
  },
#undef offset
};

/* Widget Methods */
static void ClassInitialize(void);
static void Initialize(Widget treq, Widget tnew, ArgList args,
		       Cardinal *num_args);
static void Destroy(Widget w);
static void Resize(Widget w);
static void Redisplay(Widget w, XEvent *event, Region region);
static Boolean SetValues(Widget current, Widget request, Widget wnew,
			 ArgList args, Cardinal *num_args);
static XtGeometryResult QueryGeometry(Widget w, XtWidgetGeometry *proposed,
				      XtWidgetGeometry *answer);

/* string to projection converter for XmNmapProjection resource */
Boolean CvtStringToProjection(Display *display, XrmValuePtr args, Cardinal
			      num_args, XrmValuePtr from, XrmValuePtr to,
			      XtPointer *converter_data);

/* Internal declarations */
static void MapCheckWidgetRec(MapWidget mw);

/* Default widget actions */
static void ZoomStart(Widget w, XEvent *event, String *params,
		      Cardinal *numParams);

static void ZoomMotion(Widget w, XEvent *event, String *params,
		       Cardinal *numParams);

static void ZoomEnd(Widget w, XEvent *event, String *params,
			  Cardinal *numParams);

static void ChangeCursor(Widget w, XEvent *event, String *params,
			 Cardinal *numParams);

static char default_translations[] = "\
  <Btn1Down>:           zoom-start()    \n\
  Button1 <Btn1Motion>: zoom-motion()   \n\
  <Btn1Up>:             zoom-end()      \n\
  <EnterWindow>:        change-cursor()";

static XtActionsRec actions[] = {
  {"zoom-start",     ZoomStart},
  {"zoom-motion",    ZoomMotion},
  {"zoom-end",       ZoomEnd},
  {"change-cursor",  ChangeCursor},
};

MapClassRec mapClassRec = {
  {
    /* core_class fields */
    /* superclass	  	 */ (WidgetClass) &coreClassRec,
    /* class_name	  	 */ "Map",
    /* widget_size	  	 */ sizeof(MapRec),
    /* class_initialize   	 */ ClassInitialize,
    /* class_part_initialize	 */ NULL,
    /* class_inited       	 */ False,
    /* initialize	  	 */ Initialize,
    /* initialize_hook		 */ NULL,
    /* realize		  	 */ XtInheritRealize,
    /* actions		  	 */ actions,
    /* num_actions	  	 */ XtNumber(actions),
    /* resources	  	 */ resources,
    /* num_resources	  	 */ XtNumber(resources),
    /* xrm_class	  	 */ NULLQUARK,
    /* compress_motion	  	 */ True,
    /* compress_exposure  	 */ XtExposeCompressMultiple,
    /* compress_enterleave	 */ True,
    /* visible_interest	  	 */ False,
    /* destroy		  	 */ Destroy,
    /* resize		  	 */ Resize,
    /* expose		  	 */ Redisplay,
    /* set_values	  	 */ SetValues,
    /* set_values_hook		 */ NULL,
    /* set_values_almost	 */ XtInheritSetValuesAlmost,
    /* get_values_hook		 */ NULL,
    /* accept_focus	 	 */ NULL,
    /* version			 */ XtVersion,
    /* callback_private   	 */ NULL,
    /* tm_table		   	 */ default_translations,
    /* query_geometry		 */ QueryGeometry,
    /* display_accelerator       */ XtInheritDisplayAccelerator,
    /* extension                 */ NULL
  },
  {
    /* dummy_field               */ 0,
  },
};

WidgetClass xmMapWidgetClass = (WidgetClass) &mapClassRec;

/* define a new resource representation type for projections */
static void
ClassInitialize(void)
{
  XtSetTypeConverter(XmRString,
		     XmRMapProjection,
		     (XtTypeConverter) CvtStringToProjection,
		     (XtConvertArgList) NULL, 0,
		     XtCacheNone,
		     NULL);
}

/* called to check resources values and draw the map for the first time */
static void
Initialize(Widget treq, Widget tnew, ArgList args,
		       Cardinal *num_args)
{
  MapWidget wnew = (MapWidget) tnew;
  Display *display = XtDisplay(wnew);

#ifdef USE_GUILE
  int InitGuile(MapWidget w);
#endif

  /* If use_shm is set, test to if the server is on the same machine and
     is capable of handling shared memory */
#if USE_SHM
  /* rewrite this, use gethostname instead */
  if (wnew->map.use_shm)
  {
    /* Check if the shared memory is available on the display */
    wnew->map.use_shm = XShmQueryExtension(display);

    if (!wnew->map.use_shm)
	XtWarning("Map: Shared memory not available\n");
  }
#endif /* USE_SHM */

  /* Initialize private resources */
  wnew->map.refresh = True;
  wnew->map.copy_gc = NULL;
  wnew->map.xor_gc = NULL;
  wnew->map.ext_gc = XDefaultGC(display, XDefaultScreen(display));
  wnew->map.zoom_center_x = 0.0;
  wnew->map.zoom_center_y = 0.0;
  wnew->map.zoom_factor = 1.0;
  wnew->map.map_pixmap = XmUNSPECIFIED_PIXMAP;
  wnew->map.font = NULL;
  wnew->map.basemap_image = NULL;

  /* Create the font structure */
  MapSetFont(wnew);

  /* check all of the resources to make sure they're legal */
  MapCheckWidgetRec(wnew);

#ifdef USE_GUILE
  InitGuile(wnew);
#endif

  /* this does all of the pixmap creation and drawing */
  Resize((Widget) wnew);
}

static void
Destroy(Widget w)
{
  MapWidget mw = (MapWidget) w;
  Display *dpy = XtDisplay(mw);

  /* free allocated structures in this widget */
  if (mw->map.map_pixmap != XmUNSPECIFIED_PIXMAP)
    XFreePixmap(dpy, mw->map.map_pixmap);

#ifdef USE_SHM
  if (mw->map.use_shm)
    XmMapDestroyShm((Widget) mw);
#endif /* USE_SHM */

  else if (mw->map.basemap_image != NULL)
    XDestroyImage(mw->map.basemap_image);

  if (mw->map.copy_gc != NULL)
    XFreeGC(dpy, mw->map.copy_gc);

  if (mw->map.xor_gc != NULL)
    XFreeGC(dpy, mw->map.xor_gc);

  if (mw->map.ext_gc != NULL)
    XFreeGC(dpy, mw->map.xor_gc);
}

static void
Resize(Widget w)
{
  MapWidget mw = (MapWidget) w;

  /* Create a new pixmap */
  MapCreatePixmap(mw);

  /* We need to create a new GC as we have a new pixmap */
  MapCreateGC(mw);

  /* Create a shared memory image if we need to */
  if (mw->map.use_shm)
    MapCreateShmImage(mw);

  /* Draw the basemap and freeze it */
  XmMapReset(w);
}

/* this repairs window damage on expose events */
static void
Redisplay(Widget w, XEvent *event, Region region)
{
  MapWidget mw = (MapWidget) w;
  register int x, y;
  unsigned int width, height;

  if (!XtIsRealized((Widget) mw))
    return;

  if (event)
  {				/* called from btn-event or expose */
    XExposeEvent* exposeEvent = (XExposeEvent*) event;

    x = exposeEvent->x;
    y = exposeEvent->y; 
    width = exposeEvent->width;
    height = exposeEvent->height;
  } 
  else
  {				/* called because complete redraw */
    x = 0;
    y = 0; 
    width = mw->core.width;
    height = mw->core.height;
  }

  XCopyArea(XtDisplay(mw), mw->map.map_pixmap, XtWindow(mw),
	    mw->map.copy_gc, x, y, width, height, x, y);
}

/* called whenever a resource value is changed, whenever any of the
   resources are changed we will have to redraw */
static Boolean
SetValues(Widget current, Widget request, Widget wnew, ArgList args,
	  Cardinal *num_args)
{
  MapWidget curmw = (MapWidget) current;
  MapWidget newmw = (MapWidget) wnew;
  Boolean do_redisplay = False;

  /* test that the new widget does not have any bad resource settings */
  MapCheckWidgetRec(newmw);

  /* Shared memory flag can only be set during initialization */
  if (curmw->map.use_shm != newmw->map.use_shm)
    {
      XtWarning("Map: can only set shared memory flag during initialization");
      newmw->map.use_shm = curmw->map.use_shm;
    }

  /* test for changed stuff that forces a redraw of everything */
  if (curmw->core.background_pixel	!= newmw->core.background_pixel
      || curmw->map.projection		!= newmw->map.projection
      || curmw->map.center_lon          != newmw->map.center_lon
      || curmw->map.draw_coastline	!= newmw->map.draw_coastline
      || curmw->map.draw_lakes		!= newmw->map.draw_lakes
      || curmw->map.draw_rivers		!= newmw->map.draw_rivers
      || curmw->map.draw_boundaries	!= newmw->map.draw_boundaries
      || curmw->map.draw_grid		!= newmw->map.draw_grid
      || curmw->map.ocean_color		!= newmw->map.ocean_color
      || curmw->map.land_color	        != newmw->map.land_color
      || curmw->map.coastline_color	!= newmw->map.coastline_color
      || curmw->map.lake_color		!= newmw->map.lake_color
      || curmw->map.river_color		!= newmw->map.river_color
      || curmw->map.boundary_color	!= newmw->map.boundary_color
      || curmw->map.grid_color		!= newmw->map.grid_color
      || curmw->map.zoom_box_color      != newmw->map.zoom_box_color
      || curmw->map.map_dir		!= newmw->map.map_dir
      || curmw->map.coastlines_file	!= newmw->map.coastlines_file
      || curmw->map.lakes_file		!= newmw->map.lakes_file
      || curmw->map.rivers_file		!= newmw->map.rivers_file
      || curmw->map.boundaries_file	!= newmw->map.boundaries_file)
    {
      MapCheckWidgetRec(newmw);

      XmMapReset(wnew);
    }

  /* Change the font if need to */
  if (curmw->map.font_list != newmw->map.font_list)
    {
      MapSetFont(newmw);
      MapCreateGC(newmw);
    }

  return do_redisplay;
}

static XtGeometryResult
QueryGeometry(Widget w, XtWidgetGeometry *proposed, XtWidgetGeometry *answer)
{
  MapWidget cw = (MapWidget) w;
  answer->request_mode = CWWidth | CWHeight;

  /* initial width and height */
  answer->width = cw->core.width;
  answer->height = cw->core.height;

  if (((proposed->request_mode & (CWWidth | CWHeight)) == (CWWidth | CWHeight))
      && proposed->width == answer->width
      && proposed->height == answer->height)
  {
    return XtGeometryYes;
  }
  else if (answer->width == cw->core.width
	   && answer->height == cw->core.height)
  {
    return XtGeometryNo;
  }
  else
    return XtGeometryAlmost;
}

static void
MapCheckWidgetRec(MapWidget mw)
{
  char s[80];
  Screen *screen = XtScreen((Widget) mw);

  if ((mw->map.major_grid_size < mw->map.minor_grid_size)
      || (mw->map.major_grid_size < 0 && mw->map.major_grid_size > 120)
      || (mw->map.minor_grid_size < 0 && mw->map.minor_grid_size > 10))
  {
    sprintf(s, "Map: invalid majorGridSize or minorGridSize, "
	    "using defaults %6.2f, %6.2f\n",
	    default_major_grid_size, default_minor_grid_size);
    XtWarning(s);
    mw->map.major_grid_size = default_major_grid_size;
    mw->map.minor_grid_size = default_minor_grid_size;
  }

  /* Use default width and height if ones are not defines */
  if ((mw->core.width < 20) || (mw->core.width > XWidthOfScreen(screen) - 10))
  {
    mw->core.width = 700;
  }

  if ((mw->core.height < 20)
      || (mw->core.height > XHeightOfScreen(screen) - 10))
  {
    mw->core.height = 500;
  }
}

static void
DrawZoomRectangle(MapWidget mw)
{
  if (mw->map.zoom_region.x1 != mw->map.zoom_region.x2
      && mw->map.zoom_region.y1 != mw->map.zoom_region.y2)
  {
    XRectangle rect;

    rect.x = (mw->map.zoom_region.x1 < mw->map.zoom_region.x2)
	? mw->map.zoom_region.x1 : mw->map.zoom_region.x2;

    rect.y = (mw->map.zoom_region.y1 < mw->map.zoom_region.y2)
	? mw->map.zoom_region.y1 : mw->map.zoom_region.y2;

    rect.width = fabs(mw->map.zoom_region.x2 - mw->map.zoom_region.x1);
    rect.height = fabs(mw->map.zoom_region.y2 - mw->map.zoom_region.y1);

    XDrawRectangle(XtDisplay(mw), XtWindow(mw), mw->map.xor_gc,
		   rect.x, rect.y, rect.width, rect.height);
  }
}

static void
ZoomStart(Widget w, XEvent *event, String *params,
			    Cardinal *num_params)
{
  MapWidget mw = (MapWidget) w;
  XButtonEvent *buttonEvent = (XButtonEvent*) event;
  XtGCMask mask;
  XGCValues values;

  mask = GCForeground | GCLineWidth | GCLineStyle;
  /* values.foreground = mw->map.land_color; */
  values.foreground = mw->map.zoom_box_color;
  values.background = mw->map.ocean_color;
  values.line_width = 2;
  values.line_style = LineSolid;
  MapChangeGC(XtDisplay(mw), mw->map.xor_gc, mask, &values);

  mw->map.zoom_region.x1 = mw->map.zoom_region.x2 = buttonEvent->x;
  mw->map.zoom_region.y1 = mw->map.zoom_region.y2 = buttonEvent->y;
}

static void
ZoomMotion(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
  MapWidget mw = (MapWidget) w;
  XMotionEvent *motion_event = (XMotionEvent *) event;

  if (motion_event->x >= 0 && (motion_event->x <= (int) mw->core.width)
      && motion_event->y >= 0 && (motion_event->y <= (int) mw->core.height))
  {
    double x2, y2;

    DrawZoomRectangle(mw);

    x2 = motion_event->x;
    y2 = mw->map.zoom_region.y1
      + copysign((x2 - mw->map.zoom_region.x1)
		 / (double) mw->core.width * (double) mw->core.height,
		 motion_event->y - mw->map.zoom_region.y1);

    /* Change the x position if the y does not appear on the screen */
    if (y2 < 0.0)
    {
      y2 = 0.0;
      x2 = mw->map.zoom_region.x1
	+ copysign((y2 - mw->map.zoom_region.y1) 
		   / (double) mw->core.height * (double) mw->core.width,
		   motion_event->x - mw->map.zoom_region.x1);
    }
    else if (y2 > (double) mw->core.height)
    {
      y2 = mw->core.height;
      x2 = mw->map.zoom_region.x1
	+ copysign((y2 - mw->map.zoom_region.y1)
		   / (double) mw->core.height * (double) mw->core.width,
		   motion_event->x - mw->map.zoom_region.x1);
    }

    mw->map.zoom_region.x2 = x2;
    mw->map.zoom_region.y2 = y2;

    DrawZoomRectangle(mw);
  }
}

static void
ZoomEnd(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
  MapWidget mw = (MapWidget) w;
  double c;

  /* Undraw the zoom rectangle */
  DrawZoomRectangle(mw);

  /* make sure the zoom region has a decent size.  It doesn't make
     much sense to zoom on a tiny zoom box! This will prevent
     inadvertent zooms when pulling down menus over the map */
  if ((mw->map.zoom_region.x2 - mw->map.zoom_region.x1) > 20)
    {
      /* Calculate the new zoom center longitude in radians */
      c = mw->map.zoom_region.x1
	+ (mw->map.zoom_region.x2 - mw->map.zoom_region.x1) / 2.0;

      mw->map.zoom_center_x += c / (double) mw->core.width * 2.0 * M_PI
	/ mw->map.zoom_factor - M_PI / mw->map.zoom_factor;

      /* Calculate the new zoom center latitude in radians */
      c = mw->map.zoom_region.y1
	+ (mw->map.zoom_region.y2 - mw->map.zoom_region.y1) / 2.0;

      mw->map.zoom_center_y += -c / (double) mw->core.height * M_PI
	/ mw->map.zoom_factor + M_PI_2 / mw->map.zoom_factor;

      /* Calculate the new zoom magnification factor */
      mw->map.zoom_factor *= (double) mw->core.width
	/ fabs(mw->map.zoom_region.x2 - mw->map.zoom_region.x1);
    }

  /* Redraw the map */
  XmMapReset(w);
}

#include <X11/cursorfont.h>

static void
ChangeCursor(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
  Boolean first_call = True;

  if (first_call)
    {
      Display *display = XtDisplay(w);
      Cursor cursor = XCreateFontCursor(display, XC_crosshair);
  
      XDefineCursor(display, XtWindow(w), cursor);

      first_call = False;
    }
}

void
XmMapRefresh(Widget w)
{
  if (XtIsRealized(w))
  {
    MapWidget mw = (MapWidget) w;

    XCopyArea(XtDisplay(mw), mw->map.map_pixmap, XtWindow(mw), mw->map.copy_gc,
	      0, 0, mw->core.width, mw->core.height, 0, 0);
  }
}

void
XmMapRefreshOff(Widget w)
{
  ((MapWidget) w)->map.refresh = False;
}

void
XmMapRefreshOn(Widget w)
{
  ((MapWidget) w)->map.refresh = True;
  XmMapRefresh(w);
}

void
XmMapReset(Widget w)
{
  MapWidget mw = (MapWidget) w;

  /* Call the before reset callbacks */
  if (mw->map.before_reset_callbacks != NULL)
    XtCallCallbacks(w, XmNbeforeResetCallback, NULL);

  /* Draw a new basemap */
  MapDrawBasemap(mw);

  /* Set the basemap image to the new basemap */
  XmMapFreeze(w);

  /* Call any after reset callbacks */
  if (mw->map.after_reset_callbacks != NULL)
    XtCallCallbacks(w, XmNafterResetCallback, NULL);
}

void
XmMapFreeze(Widget w)
{
  MapSetBasemapImage((MapWidget) w);
}

void
XmMapClear(Widget w)
{
  MapGetBasemapImage((MapWidget) w);
}

void
XmMapUnzoom(Widget w)
{
  MapWidget mw = (MapWidget) w;

  /* Reset the map zoom items back to the center with 1.0 magnification */
  mw->map.zoom_center_x = 0.0;
  mw->map.zoom_center_y = 0.0;
  mw->map.zoom_factor = 1.0;

  /* Draw the new basemap, freeze it, and call any redraw callbacks */
  XmMapReset(w);
}

void
XmMapDrawPoints(Widget w, XmMapGPoint *gpts, Cardinal npts)

{
  MapWidget mw = (MapWidget) w;

  MapDrawPoints(mw, mw->map.ext_gc, gpts, npts);
}

void
XmMapDrawStrings(Widget w, XmMapGPoint *gpts, String *s, Cardinal npts)
{
  MapWidget mw = (MapWidget) w;

  MapDrawStrings(mw, mw->map.ext_gc, gpts, s, npts);
}

void
XmMapDrawMarkers(Widget w, XmMapGPoint *gpts, Cardinal npts,
		 XmMarkerType marker_type, Cardinal marker_size)
{
  MapWidget mw = (MapWidget) w;

  MapDrawMarkers(mw, mw->map.ext_gc, gpts, npts, marker_type, marker_size);
}

void
XmMapDrawLines(Widget w, XmMapGPoint *gpts, Cardinal npts)
{
  MapWidget mw = (MapWidget) w;

  MapDrawLines((MapWidget) w, mw->map.ext_gc, gpts, npts);
}

void
XmMapFillPolygon(Widget w, XmMapGPoint *gpts, Cardinal npts)
{
  MapWidget mw = (MapWidget) w;

  MapFillPolygon((MapWidget) w, mw->map.ext_gc, gpts, npts);
}

Boolean
XmMapGPointToXPoint(Widget w, const XmMapGPoint* gpt, XPoint* xp)
{
  return MapGPointToXPoint((MapWidget) w, gpt, xp);
}

Boolean
XmMapXPointToGPoint(Widget w, const XPoint* xp, XmMapGPoint* gpt)
{
  return MapXPointToGPoint((MapWidget) w, xp, gpt);
}

void
XmMapDestroyShm(Widget w)
{
  MapWidget mw = (MapWidget) w;

#ifdef USE_SHM
  if (mw->map.use_shm)
    {
      if (!XShmDetach(XtDisplay(mw), &mw->map.shm_info))
	fputs("XShmDetach failed\n", stderr);

      if (mw->map.basemap_image != NULL)
	XDestroyImage(mw->map.basemap_image);

      if (shmdt(mw->map.shm_info.shmaddr) == -1)
	perror("shmdt");

      if (shmctl(mw->map.shm_info.shmid, IPC_RMID, NULL) == -1)
	perror("shmctl");
    }
#else
  XtWarning("Map: shared memory extension not configured");
#endif /* USE_SHM */
}

#ifdef USE_GUILE
SCM
_EvalHandler(void *data, SCM tag, SCM throw_args)
{
  puts("_EvalHandler Called");

  return (SCM) 0;
}
#endif

void
XmMapEvalGuileFile(Widget w, String fname)
{
#ifdef USE_GUILE
  gh_eval_file_with_catch(fname, (scm_catch_handler_t) _EvalHandler);
#else
  XtWarning("Map: XmMapEvalGuileFile() is not configured to work with Guile");
#endif
}

void
XmMapEvalFile(Widget w, String fname)
{
  MapEvalFile(w, fname);
}

GC
XmMapGC(Widget w)
{
  MapWidget mw = (MapWidget) w;

  return mw->map.ext_gc;
}
