 /*
 * MapGuile.c - Map Widget Guile Scheme interface
 */

#ifdef USE_GUILE

#include <assert.h>
#include <unistd.h>
#include <string.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

#include "Map.h"
#include <guile/gh.h>

static Widget map;

#define SCM2FLOAT(_s) ((float) gh_scm2double(_s))
#define SCM2INT(_s) ((int) gh_scm2long(_s))

/* convert a scheme object (either string or integer) to an XFont */
Font
SCMToFont(SCM svalue)
{
  XrmValue from, to;
  String s;
  int n;
  Font font = 0;

  if (gh_string_p(svalue))
    {
      s = gh_scm2newstr(svalue, &n);
      from.addr = s;
      from.size = n;
      to.addr = NULL;		/* initialize or core dump! */
      to.size = 0;
      if (!XtConvertAndStore(map, XtRString, &from, XtRFont, &to))
	{
	  XtWarning("map/guile interface: String to Font conversion error");
	}
      else
	font = *(Font *) to.addr;
    }
  else
    {
      XtWarning("map/guile interface: Font specifier not a string");
    }

  XtFree(s);
  return font;
}

/* convert a scm object (color name string) to an X Pixel index */
unsigned long
SCMToPixel(SCM svalue)
{
  unsigned long pixel = 0;

  /* if the color is passed as a string convert it to a color */
  if (gh_string_p(svalue))
    {
      XrmValue from, to;
      String s;
      int n;

      s = gh_scm2newstr(svalue, &n);
      from.addr = s;
      from.size = n;
      to.addr = NULL;
      to.size = 0;
      if (!XtConvertAndStore(map, XtRString, &from, XtRPixel, &to))
	{
	  XtWarning("map/guile interface: String to Pixel conversion error");
	}
      else
	pixel =  *(unsigned long *) to.addr;

      XtFree(s);
    }
  else if (gh_exact_p(svalue))
    pixel =  gh_scm2long(svalue);
  else
    {
      XtWarning("map/guile interface: String to Pixel conversion error");
    }

  return pixel;
}

Pixmap
SCMToStipple(SCM ls)
{
  Display *display = XtDisplay(map);
  Pixmap stipple = (Pixmap) NULL;
  int width, height, len, i;
  char *bits;
  SCM sval;

  if (gh_exact_p(sval = gh_car(ls)))
    width = SCM2INT(sval);
  else
    {
      XtWarning("map/guile interface: Stipple width conversion error");

      return stipple;
    }

  if (gh_exact_p(sval = gh_cadr(ls)))
    height = SCM2INT(sval);
  else
    {
      XtWarning("map/guile interface: Stipple height conversion error");

      return stipple;
    }

  ls = gh_caddr(ls);
  len = gh_list_length(ls);

  bits = (char *) XtMalloc(len * sizeof(char));
  for (i = 0; i < len; i++)
    {
      bits[i] = SCM2INT(gh_car(ls));
      ls = gh_cdr(ls);
    }

  stipple = \
    XCreateBitmapFromData(display, RootWindow(display, DefaultScreen(display)),
			  bits, width, height);

  return stipple;
}

static void
GetGCValuesFromSCMArgs(SCM args, XGCValues *xgcv, unsigned long *mask)
{
  SCM sbit, svalue;
  unsigned long bit, len;

  *mask = 0;
  (void) memset((char *) xgcv, 0, sizeof(XGCValues));

  len = gh_list_length(args);

  if (len == 0) return;

  while (len)
    {
      sbit = gh_car(args);
      svalue = gh_cadr(args);
      args = gh_cddr(args);

      if (!gh_exact_p(sbit))
	{
	  char es[80];
	  sprintf(es,
		  "map/guile interface: "
		  "non integer GC flag ignored");
	  XtWarning(es);
	  continue;
	}

      bit = gh_scm2ulong(sbit);
      *mask |= bit;
      switch (bit)
	{
	case GCFunction:  /* int */
	  xgcv->function = SCM2INT(svalue);
	  break;

	case GCPlaneMask: /* unsigned long */
	  xgcv->plane_mask = gh_scm2ulong(svalue);
	  break;

	case GCForeground: /* long */
	  xgcv->foreground = SCMToPixel(svalue);
	  break;

	case GCBackground: /* long */
	  xgcv->background = SCMToPixel(svalue);
	  break;

	case GCLineWidth: /* int */
	  xgcv->line_width = SCM2INT(svalue);
	  break;

	case GCLineStyle: /* int */
	  xgcv->line_style = SCM2INT(svalue);
	  break;

	case GCFillStyle: /* int */
	  xgcv->fill_style = SCM2INT(svalue);
	  break;

	case GCStipple:
	  xgcv->stipple = SCMToStipple(svalue);
	  break;

	case GCFont:
	  xgcv->font = SCMToFont(svalue);
	  break;

	default:
	  XtWarning("map/guile interface: invalid GC flag ignored");
	  break;
	}

      len -= 2;
    }
}

static SCM
c_MapClear()
{
  XmMapClear(map);

  return (SCM) 0;
}

static SCM
c_MapFreeze()
{
  XmMapFreeze(map);

  return (SCM) 0;
}

static SCM
c_MapChangeGC(SCM args)
{
  XGCValues xgcvalues;
  unsigned long mask = 0; 

  gh_defer_ints();
  GetGCValuesFromSCMArgs(args, &xgcvalues, &mask);
  XChangeGC(XtDisplay(map), XmMapGC(map), mask, &xgcvalues);
  gh_allow_ints();

  return (SCM) 0;
}

/* draw a vector of lists of length 2 */
static SCM
c_MapDrawPoints(SCM pv)
{
  XmMapGPoint *gp;
  int i, n;
  
  n = (int) gh_vector_length(pv);

  /* prevent GC while we use dynamic vars */
  gh_defer_ints();
  gp = (XmMapGPoint *) XtMalloc(n * sizeof(XmMapGPoint));
  for (i = 0; i < n; i++)
    {
      SCM ls;

      ls = gh_vref(pv, gh_long2scm((long) i));
      gp[i].lat = SCM2FLOAT(gh_car(ls));
      gp[i].lon = SCM2FLOAT(gh_cadr(ls));
    }
  XmMapDrawPoints(map, gp, n);
  XtFree((char *) gp);
  gh_allow_ints();

  return (SCM) 0;
}

static SCM
c_MapDrawMarkers(SCM stype, SCM swidth, SCM ls)
{
  XmMarkerType marker_type = (XmMarkerType) SCM2INT(stype);
  int marker_size = SCM2INT(swidth);
  XmMapGPoint *gp;
  unsigned long n = gh_list_length(ls);
  int i;

  /* prevent GC while we use dynamic vars */
  gh_defer_ints();
  gp = (XmMapGPoint *) XtMalloc(n * sizeof(XmMapGPoint));

  for (i = 0; i < n; i++)
    {
      SCM lls;

      lls = gh_car(ls);
      gp[i].lat = SCM2FLOAT(gh_car(lls));
      gp[i].lon = SCM2FLOAT(gh_cadr(lls));
      ls = gh_cdr(ls);
    }

  XmMapDrawMarkers(map, gp, n, marker_type, marker_size);

  XtFree((char *) gp);
  gh_allow_ints();

  return (SCM) 0;
}

static SCM
c_MapDrawStrings(SCM ls)
{
  XmMapGPoint *gp;
  String *s;
  unsigned long n = gh_list_length(ls);
  int i, m;
  
  /* prevent GC while we use dynamic vars */
  gh_defer_ints();
  gp = (XmMapGPoint *) XtMalloc(n * sizeof(XmMapGPoint));
  s = (String *) XtMalloc(n * sizeof(String));

  for (i = 0; i < n; i++)
    {
      SCM lls;

      lls = gh_car(ls);
      gp[i].lat = SCM2FLOAT(gh_car(lls));
      gp[i].lon = SCM2FLOAT(gh_cadr(lls));
      s[i] = gh_scm2newstr(gh_caddr(lls), &m);
      ls = gh_cdr(ls);
    }

  XmMapDrawStrings(map, gp, s, n);

  XtFree((char *) gp);
  for (i = 0; i < n; i++)
    XtFree((char *) s[i]);
  XtFree((char *) s);
  gh_allow_ints();

  return (SCM) 0;
}

/* draw a vector of lists of length 2 */
static SCM
c_MapDrawLines(SCM ls)
{
  XmMapGPoint *gp;
  unsigned long n = gh_list_length(ls);
  int i;

  if (n == 0) return (SCM) 0;

  /* prevent GC while we use dynamic vars */
  gh_defer_ints();
  gp = (XmMapGPoint *) XtMalloc(n * sizeof(XmMapGPoint));
  for (i = 0; i < n; i++)
    {
      SCM lls;

      lls = gh_car(ls);
      gp[i].lat = SCM2FLOAT(gh_car(lls));
      gp[i].lon = SCM2FLOAT(gh_cadr(lls));
      ls = gh_cdr(ls);
    }

  XmMapDrawLines(map, gp, n);
  XtFree((char *) gp);
  gh_allow_ints();

  return (SCM) 0;
}

static SCM
c_MapFillPolygon(SCM ls)
{
  XmMapGPoint *gp;
  unsigned long n = gh_list_length(ls);
  int i;

  if (n == 0) return (SCM) 0;

  /* prevent GC while we use dynamic vars */
  gh_defer_ints();
  gp = (XmMapGPoint *) XtMalloc(n * sizeof(XmMapGPoint));
  for (i = 0; i < n; i++)
    {
      SCM lls;

      lls = gh_car(ls);
      gp[i].lat = SCM2FLOAT(gh_car(lls));
      gp[i].lon = SCM2FLOAT(gh_cadr(lls));
      ls = gh_cdr(ls);
    }

  XmMapFillPolygon(map, gp, n);
  XtFree((char *) gp);
  gh_allow_ints();

  return (SCM) 0;
}

static SCM
c_sleep(SCM s_sec)
{
  if (gh_exact_p(s_sec))
    sleep((unsigned) gh_scm2long(s_sec));

  return (SCM) 0;
}

static SCM
c_MapSync()
{
  XSync(XtDisplay(map), False);

  return (SCM) 0;
}


/* identify the map widget interface to the guile interpreter */
int
InitGuile(Widget w)
{
  if (XmIsMap(w))
    {
      char *gc_defs = \
	"(define gc:function	       (expt 2 0))
         (define gc:plane-mask	       (expt 2 1))
         (define gc:foreground	       (expt 2 2))
         (define gc:background	       (expt 2 3))
         (define gc:line-width	       (expt 2 4))
         (define gc:line-style	       (expt 2 5))
         (define gc:cap-style	       (expt 2 6))
         (define gc:join-style	       (expt 2 7))
         (define gc:fill-style	       (expt 2 8))
         (define gc:fill-rule	       (expt 2 9)) 
         (define gc:tile               (expt 2 10))
         (define gc:stipple	       (expt 2 11))
         (define gc:tile-stip-x-origin (expt 2 12))
         (define gc:tile-stip-y-origin (expt 2 13))
         (define gc:font 	       (expt 2 14))
         (define gc:subwindow-mode     (expt 2 15))
         (define gc:graphics-exposures (expt 2 16))
         (define gc:clip-x-origin      (expt 2 17))
         (define gc:clip-y-origin      (expt 2 18))
         (define gc:clip-mask	       (expt 2 19))
         (define gc:dash-offset	       (expt 2 20))
         (define gc:dash-list	       (expt 2 21))
         (define gc:arc-mode	       (expt 2 22))

         (define gx-clear	       #x0)
         (define gx-and		       #x1)
         (define gx-and-reverse	       #x2)
         (define gx-copy	       #x3)
         (define gx-and-inverted       #x4)
         (define gx-noop               #x5)
         (define gx-xor		       #x6)
         (define gx-or		       #x7)
         (define gx-nor		       #x8)
         (define gx-equiv              #x9)
         (define gx-invert             #xa)	
         (define gx-or-reverse         #xb)
         (define gx-copy-inverted      #xc)
         (define gx-or-inverted	       #xd)
         (define gx-nand               #xe)
         (define gx-set		       #xf)

         (define line-solid            0)
         (define line-on-off-dash      1)
         (define line-double-dash      2)

         (define cap-not-last	       0)
         (define cap-butt	       1)
         (define cap-round             2)
         (define cap-projecting	       3)

         (define join-miter            0)
         (define join-round	       1)
         (define join-bevel            2)

         (define fill-solid	       0)
         (define fill-tiled	       1)
         (define fill-stippled	       2)
         (define fill-opaque-stippled  3)
         (define gray-stipple          (list 4 2 (list #x08 #x02)))

         (define marker-circle         0)
         (define marker-square         1)
         (define marker-triangle       2)";

      gh_eval_str(gc_defs);

      /* install the new scheme functions and set the target widget */
      gh_new_procedure("map-clear", c_MapClear, 0, 0, 0);
      gh_new_procedure("map-freeze", c_MapFreeze, 0, 0, 0);
      gh_new_procedure("map-change-gc", c_MapChangeGC, 1, 0, 0);
      gh_new_procedure("map-draw-points", c_MapDrawPoints, 1, 0, 0);
      gh_new_procedure("map-draw-markers", c_MapDrawMarkers, 3, 0, 0);
      gh_new_procedure("map-draw-strings", c_MapDrawStrings, 1, 0, 0);
      gh_new_procedure("map-draw-lines", c_MapDrawLines, 1, 0, 0);
      gh_new_procedure("map-fill-polygon", c_MapFillPolygon, 1, 0, 0);
      gh_new_procedure("map-sync", c_MapSync, 0, 0, 0);
      gh_new_procedure("sleep", c_sleep, 1, 0, 0);
      map = w;

      return 1;
    }
  else
    return 0;
}

#endif /* USE_GUILE */

