/* canvas.c -- routines for xa's canvas, including image display */

/* $Id: canvas.c,v 1.14 1996/05/04 19:56:04 richards Exp $ */

/*
 * $Log: canvas.c,v $
 * Revision 1.14  1996/05/04 19:56:04  richards
 * fixed behaviour for -copy n -visualload
 *
 * Revision 1.13  1996/04/19 22:03:01  richards
 * replaced CanvasCursor() (global) with make_cursor() (local)
 * added -oneatatime support
 * fixed colormap problems with -visualload -private
 * added Busy() global (calls make_cursor())
 * fixed centering so it works like expected...
 * added POINTER_ON_IMAGE() macro
 *
 * Revision 1.12  1996/04/05 16:29:46  richards
 * trivial change
 *
 * Revision 1.11  1996/03/16 16:55:52  richards
 * renamed xrastool to xa
 * fixed indenting
 * added CanvasCursor() global to restore cursor after BUSY events
 *
 * Revision 1.10  1996/03/14 21:55:53  richards
 * minor changes
 *
 * Revision 1.9  1996/03/08 20:49:37  richards
 * fixed up update_cursor_info(), still needs work
 *
 * Revision 1.8  1996/02/21 20:44:34  richards
 * implemented keyboard cursor control ('h','j','k','l')
 * note update_cursor_info() contains a hack to make color mapping work
 *
 * Revision 1.7  1996/02/08 01:24:29  richards
 * added cursor mask so cursor now visible on white backgrounds
 * replaced XView cursor handling with xlib calls so the mask can be used
 * added AllowManualSizing() to fix problems with toggling scrollbars and
 *    the footer when NO_RESIZING is in effect
 * fixed problems with images smaller than MIN_IMAGE_WIDTH/HEIGHT
 * removed INNER_CANVAS_BORDER (now hardwired as ScrollMargin += 1)
 * still having problems with auto moving, depending on the window manager
 *
 * Revision 1.6  1996/01/30 19:48:15  richards
 * removed #include <xview/notify.h>
 * replaced TmpColors[] with ScaledColors[] in subpanel.c
 * incorporated color macros
 * removed redundant CURSOR_BACKGROUND_COLOR
 * replaced InstallCms() with StoreColors() in color.c
 * added WaitingForImages flag to suppress canvas repainting
 * replaced Xv_singlecolor with COLORCELL_T
 * live cursor now shows actual pixel value
 *
 * Revision 1.5  1996/01/27 22:29:45  richards
 * changed some comment formats and indenting
 * made live cursor code into a separate routine (update_cursor_info())
 * if image changes in live cursor mode, new pixel value is displayed
 *
 * Revision 1.4  1996/01/25 15:23:44  richards
 * replaced TmpXImage global with ximage local
 * can now load Pixmaps (server) OR XImages (client) -- user's choice
 * enforced MIN_IMAGE_{WIDTH,HEIGHT}
 *
 * Revision 1.3  1996/01/21 21:31:03  richards
 * prototyping, NewCmap functionality
 *
 * Revision 1.2  1996/01/18 18:13:00  richards
 * fix exposure event handling
 *
 * Revision 1.1  1996/01/18 16:56:55  richards
 * Initial revision
 *
 */

#include "xa.h"
#include <X11/Xutil.h> /* for XSizeHints */
#include <xview/scrollbar.h>

/* Useful macro */

#define POINTER_ON_IMAGE(x,y)\
  ((x) >= 0 && (x) < Image[Current]->w && (y) >= 0 && (y) < Image[Current]->h)

/* Global variables */

Frame CanvasFrame;
Xv_window CanvasWindow;
int ScrollMargin;
Window CanvasXID;
Server_image Backdrop[NUM_BACKDROPS];
GC BackdropGC;

/* Local variables */

static Canvas canvas;         /* Canvas handle */
static int image_x, image_y;  /* Pos'n of top left of image in frame */
static XImage *ximage = NULL; /* Temporary XImage for LIVE_CURSOR */

/* Bitmaps */

#include "bitmaps/canvas_icon.xbm"
#include "bitmaps/cursor.xbm"
#include "bitmaps/cursor_mask.xbm"
#include "bitmaps/backdrops.xbm"

/* Local functions */

static Notify_value canvas_frame_quit(Notify_client client,
				      Destroy_status status);

static void canvas_event_proc(Xv_window window, Event *event, Notify_arg arg);
static void repaint_canvas(void);
static void set_canvas_size_and_pos(void);
static void make_cursor(void);
static void move_cursor(int dx, int dy);
static void update_cursor_info(void);

/*** END OF PREAMBLE ***/

void MakeCanvas(void)
{
  /* Constructs canvas, icon, scrollbars, cursor and a few other things... */

  Server_image image;
  Icon icon;
  XGCValues values;

  /* Create icon */

  image = (Server_image)
    xv_create(XV_NULL, SERVER_IMAGE,
	      SERVER_IMAGE_X_BITS, canvas_icon_bits,
	      XV_WIDTH, canvas_icon_width,
	      XV_HEIGHT, canvas_icon_height,
	      NULL);

  icon = (Icon)
    xv_create(XV_NULL, ICON,
	      ICON_IMAGE, image,
	      WIN_FOREGROUND_COLOR, 1,
	      WIN_BACKGROUND_COLOR, 0,
	      NULL);

  /* Create canvas frame */

  CanvasFrame = (Frame)
    xv_create(XV_NULL, FRAME,
	      XV_VISUAL, MainVisual,
	      XV_X, DFLT_CANVAS_FRAME_XV_X,
	      XV_Y, DFLT_CANVAS_FRAME_XV_Y,
	      FRAME_LABEL, "xa: Canvas",
	      FRAME_SHOW_FOOTER, TRUE, /* to propagate colors */
	      FRAME_ICON, icon,
	      NULL);

  /* Interpose destroy function */

  notify_interpose_destroy_func(CanvasFrame, canvas_frame_quit);

  /* Create canvas */

  canvas = (Canvas)
    xv_create(CanvasFrame, CANVAS,
	      CANVAS_AUTO_EXPAND, FALSE,
	      CANVAS_AUTO_SHRINK, FALSE,
	      CANVAS_REPAINT_PROC, repaint_canvas,
	      NULL);

  /* Get window ID */

  CanvasWindow = canvas_paint_window(canvas);

  /* Intercept key presses and mouse movements */

  (void) xv_set(CanvasWindow,
		WIN_CONSUME_EVENTS, WIN_NO_EVENTS, WIN_ASCII_EVENTS, LOC_MOVE,
		  LOC_WINEXIT, LOC_DRAG,
		WIN_MOUSE_BUTTONS, NULL,
		WIN_EVENT_PROC, canvas_event_proc,
		NULL);

  /* Also intercept exposure events (required for some window managers) */

  (void) xv_set(CanvasWindow,
		WIN_CONSUME_X_EVENT_MASK, ExposureMask,
		NULL); /* Is there an XView way to do this? */

  /* Construct scollbars if desired initially */

  SetScrollbars(!(Options & NO_SCROLLING));

  /* Get canvas XID */

  CanvasXID = (Window) xv_get(CanvasWindow, XV_XID);

  /* Construct canvas cursor */

  make_cursor();

  /* Prepare GC for backdrops (stipple set in SetBackdrop()) */

  values.foreground = BlackPixel(MainDisplay, MainScreen);
  values.background = WhitePixel(MainDisplay, MainScreen);
  values.fill_style = FillOpaqueStippled;

  BackdropGC = XCreateGC(MainDisplay, CanvasXID,
			 GCForeground | GCBackground | GCFillStyle, &values);
}

void SetScrollbars(Bool toggle_on)
{
  /* Creates (toggle_on) or destroys (!toggle_on) scrollbars */

  static Bool no_scrollbars = TRUE; /* To counter repeated requests */
  static Scrollbar vbar, hbar; /* Handles need only be local */

  if (no_scrollbars && toggle_on) { /* Create scrollbars */

    vbar = (Scrollbar)
      xv_create(canvas, SCROLLBAR,
		SCROLLBAR_DIRECTION, SCROLLBAR_VERTICAL,
		NULL);

    /* Determine scrollbar thickness */

    ScrollMargin = (int) xv_get(vbar, XV_WIDTH);

    hbar = (Scrollbar)
      xv_create(canvas, SCROLLBAR,
		SCROLLBAR_DIRECTION, SCROLLBAR_HORIZONTAL,
		NULL);

    if (ScrollMargin != (int) xv_get(hbar, XV_HEIGHT))
      Warning("Unequal scrollbar thicknesses.");

    no_scrollbars = FALSE;

    if (Options & NO_RESIZING)	/* Force the window to grow */
      AllowManualSizing(TRUE);

    set_canvas_size_and_pos();

    if (Options & NO_RESIZING)
      AllowManualSizing(FALSE);
  }
  else if (!no_scrollbars && !toggle_on) { /* Destroy scrollbars */
    (void) xv_destroy_safe(hbar);
    (void) xv_destroy_safe(vbar);
    ScrollMargin = 0;
    no_scrollbars = TRUE;
    if (Options & NO_RESIZING) /* Force the window to shrink */
      AllowManualSizing(TRUE);
    set_canvas_size_and_pos();
    if (Options & NO_RESIZING)
      AllowManualSizing(FALSE);
  }

  ScrollMargin += 1; /* Why? Good question */
}

void ShowNewImage(void)
{
  /* Displays (or redisplays) current image */

  static int first_call = TRUE; /* For color management and XV_SHOW */

  IMAGE_T *image;

  /* Abort if image hasn't been loaded yet */

  if (UNDEF_IMAGE)
    return;

  /* Otherwise get image */

  image = Image[Current];

  /* If loading images one at a time, get image data now */

  if (OneAtATime && !image->mem.ximage) {
    int result;
    char *filename;

    filename = (char *) malloc(strlen(image->filename) + 1);
    (void) strcpy(filename, image->filename); /* Local copy */
    if (!Cycling)
      Busy(TRUE);
    result = LoadData(1, &filename, Current);
    if (!Cycling)
      Busy(FALSE);
    free((void *) filename);
    if (result != 1) {
      Warning("ShowNewImage(): Error occurred during load.");
      return;
    }
  }

  /* If sizing is allowed or forced, set canvas size (also reposition) */

  if (!(Options & NO_RESIZING) || ForceFirstResize)
    set_canvas_size_and_pos();

  /* If colors are locked and first call, initialize locked colors */

  if (first_call && (Options & LOCK_COLORS)) {
    if (OneCmap || NewCmap) { /* Currently unique cmap must be 1st image */
      NumLockedColors = Image[0]->num_colors;
      LockedColors = Image[0]->colors;
    }
    else {
      NumLockedColors = image->num_colors;
      LockedColors = image->colors;
    }
  }

  /*
   * If color scaling has been disabled, it may be necessary to store
   * new colors here, unless colors are locked. Also ScaledColors needs
   * to be loaded with the current colors for LIVE_CURSOR to work (this
   * is done automatically if color scaling is activated).
   *
   */

  if (Options & NO_SCALING) {
    if (first_call || !(Options & LOCK_COLORS))
      StoreColors(image->colors, image->num_colors);
    if (Options & LIVE_CURSOR)
      if (Options & LOCK_COLORS)
	CopyColors(NumLockedColors, LockedColors, ScaledColors);
      else
	CopyColors(image->num_colors, image->colors, ScaledColors);
  }
  else
    ApplyColorScaling(FALSE);

  /*
   * If honouring a -visualload -private request, must manually check
   * for input focus and install (or uninstall) private colormap. This
   * is necessary because notifier blocks until entire load is complete.
   *
   */

  if (LoadingDelayedImages && OWN_COLORMAP) {
    int x, y;
    Rect *r;

    r = (Rect *) xv_get(canvas, WIN_MOUSE_XY);

    x = r->r_left - 1 - image_x;
    y = r->r_top - 1 - image_y;

    if (NumColorsToCopy > 0 || POINTER_ON_IMAGE(x, y))
      XInstallColormap(MainDisplay, XaColormap);
    else
      XUninstallColormap(MainDisplay, XaColormap);
  }

  /* On first call, show the canvas frame (with footer if applicable) */

  if (first_call) {
    (void) xv_set(CanvasFrame,
		  XV_SHOW, TRUE,
		  FRAME_SHOW_FOOTER, (Options & LIVE_CURSOR),
		  NULL);
    if (Options & LIVE_CURSOR)
      (void) xv_set(CanvasFrame, FRAME_LEFT_FOOTER, "(no cursor)", NULL);
  }

  /* Repaint the canvas now */

  repaint_canvas();

  /* Update label items on the panels, etc. if allowed */

  if (!((LoadingDelayedImages || Cycling) && (Options & NO_UPDATES)))
    UpdateLabels();

  /*
   * Need an XImage of the current pixmap for LIVE_CURSOR to work.
   * Note that any existing global XImage should be deleted, otherwise
   * memory use will grow at an uncomfortable rate...
   *
   */

  if (Options & LIVE_CURSOR) {
    if (UsePixmaps) {
      if (ximage)
	(void) XDestroyImage(ximage);
      ximage = XGetImage(MainDisplay, image->mem.pixmap, 0, 0,
			 image->w, image->h, AllPlanes, ZPixmap);
    }
    else
      ximage = image->mem.ximage;
    update_cursor_info();
  }

  first_call = FALSE;
}

void MakeBackdropImages(void)
{
  /* Makes server images of the available backdrops */

  int i;

  for (i = 0; i < NUM_BACKDROPS; i++)
    Backdrop[i] = (Server_image)
      xv_create(XV_NULL, SERVER_IMAGE,
		SERVER_IMAGE_X_BITS, backdrop_bits[i],
		XV_WIDTH, BACKDROP_WIDTH,
		XV_HEIGHT, BACKDROP_HEIGHT,
		NULL);
}

void PaintBackground(void)
{
  /*
   * Installs current backdrop on any exposed space behind current
   * image. The background is painted in up to four rectangular
   * regions, depending on the relation between the position of the
   * top-left corner of the image (image_x,image_y), the image size,
   * and the canvas frame size.
   *
   */

  int cw, ch, iw, ih, x, y, w, h; /* Working variables */

  /* Get width and height of canvas frame */

  cw = (int) xv_get(CanvasFrame, XV_WIDTH) - ScrollMargin;
  ch = (int) xv_get(CanvasFrame, XV_HEIGHT) - ScrollMargin;

  /* If image hasn't been loaded yet, paint entire canvas and return */

  if (UNDEF_IMAGE) {
    XFillRectangle(MainDisplay, CanvasXID, BackdropGC, 0, 0, cw, ch);
    return;
  }

  /* Otherwise get image dimensions */

  iw = Image[Current]->w;
  ih = Image[Current]->h;

  /*
   * Return if image is larger than canvas, since image can only be
   * centred or at top left of the frame.
   *
   */

  if (iw >= cw && ih >= ch)
    return;

  /* Paint left- and/or right-most rectangles if frame is wider than image */

  if (iw < cw) {
    if (image_x > 0) {
      x = 0;
      y = 0;
      w = image_x;
      h = ch;
      XFillRectangle(MainDisplay, CanvasXID, BackdropGC, x, y, w, h);
    }
    if (image_x + iw < cw) {
      x = image_x + iw;
      y = 0;
      w = cw - x;
      h = ch;
      XFillRectangle(MainDisplay, CanvasXID, BackdropGC, x, y, w, h);
    }
  }

  /* Paint top- and/or bottom-most rectangles if frame is taller than image */

  if (ih < ch) {
    if (image_y > 0) {
      x = image_x;
      y = 0;
      w = iw;
      h = image_y;
      XFillRectangle(MainDisplay, CanvasXID, BackdropGC, x, y, w, h);
    }
    if (image_y + ih < ch) {
      x = image_x;
      y = image_y + ih;
      w = iw;
      h = ch - y;
      XFillRectangle(MainDisplay, CanvasXID, BackdropGC, x, y, w, h);
    }
  }
}

void AllowManualSizing(Bool flag)
{
  if (flag) {

    /*
     * For some reason, setting FRAME_MIN_SIZE to (0,0) or (1,1) and
     * FRAME_MAX_SIZE to (ScreenWidth,ScreenHeight) doesn't work if
     * they were set prior to mapping the canvas frame to the screen.
     * Instead, xlib calls are required to re-enable manual sizing.
     * Mysteriously, BOTH PMinSize and PBaseSize must be specified.
     * No explicit values are set for these below since XAllocSizeHints()
     * automatically zeroes these values.
     *
     */

    XSizeHints *hints = XAllocSizeHints();
    Window xid = (Window) xv_get(CanvasFrame, XV_XID);

    hints->flags = PMinSize | PMaxSize | PBaseSize;
    hints->max_width = ScreenWidth;
    hints->max_height = ScreenHeight;
    XSetWMNormalHints(MainDisplay, xid, hints);
    XFree((caddr_t) hints);
  }
  else {
    int w = (int) xv_get(CanvasFrame, XV_WIDTH);
    int h = (int) xv_get(CanvasFrame, XV_HEIGHT);

    (void) xv_set(CanvasFrame, /* Disable manual resizing by mouse */
		  FRAME_MIN_SIZE, w, h,
		  FRAME_MAX_SIZE, w, h,
		  NULL);
  }
}

void Busy(Bool flag)
{
  /* Toggles busy state of main xa frames */

  xv_set(BaseFrame, FRAME_BUSY, flag, NULL);
  xv_set(SubFrame, FRAME_BUSY, flag, NULL);
  xv_set(CanvasFrame, FRAME_BUSY, flag, NULL);

  /* Restore canvas cursor */

  if (flag == FALSE)
    make_cursor();
}

static Notify_value canvas_frame_quit(Notify_client client,
				      Destroy_status status)
{
  /* Intercepts and ignores destroy requests unless NukeCanvas is set */

  if (status == DESTROY_CHECKING && !NukeCanvas) {
    notify_veto_destroy(client);
    return NOTIFY_DONE;
  }

  return notify_next_destroy_func(client, status);
}

static void canvas_event_proc(Xv_window window, Event *event, Notify_arg arg)
/*ARGSUSED*/
{
  /* Intercepts keypresses and mouse movements */

  if (event_is_ascii(event) && event_is_up(event)) {

    /*
     * A key was pressed and released (or there was an auto-repeat event).
     * Cursor movement is handled by routines in canvas.c; other keys are
     * interpreted in mainpanel.c.
     *
     */

    switch (event_action(event)) {
    case CURSOR_LEFT:
      move_cursor(-1, 0);
      break;
    case CURSOR_DOWN:
      move_cursor( 0,-1);
      break;
    case CURSOR_UP:
      move_cursor( 0, 1);
      break;
    case CURSOR_RIGHT:
      move_cursor( 1, 0);
      break;
    default:
      KeyPressed(event_action(event)); /* In mainpanel.c */
    }
  }
  else if (Options & LIVE_CURSOR) {

    /*
     * The mouse/pointer/cursor has moved. If LIVE_CURSOR option is set,
     * display the RGB values at the new pixel location, if valid.
     *
     */

    if (event_action(event) == LOC_MOVE || event_action(event) == LOC_DRAG)
      update_cursor_info();
    else if (event_action(event) == LOC_WINEXIT && !UNDEF_IMAGE)
      (void) xv_set(CanvasFrame, FRAME_LEFT_FOOTER, "(no cursor)", NULL);
  }
}

static void repaint_canvas(void)
{
  /* Repaints canvas with current background and image */

  IMAGE_T *image;

  if (UNDEF_IMAGE)
    return;

  image = Image[Current];

  /* Assume image origin coincides with frame origin */

  image_x = image_y = 0;

  /* Paint background if frame size *could* exceed image size */

  if (Sizing == FIXED || image->w < MIN_IMAGE_WIDTH ||
      image->h < MIN_IMAGE_HEIGHT) {
    if ((Options & CENTERING)) {

      /*
       * Determine origin for centering case. Note image_x and/or image_y
       * may be negative if image size exceeds frame size.
       *
       */

      image_x = ((image->w > MIN_IMAGE_WIDTH || Sizing == FIXED ?
		  FixedWidth : MIN_IMAGE_WIDTH) - image->w) / 2;
      image_y = ((image->h > MIN_IMAGE_HEIGHT || Sizing == FIXED ?
		  FixedHeight : MIN_IMAGE_HEIGHT) - image->h) / 2;
    }
    if (!(Cycling && (Options & NO_BACKDROPS)))
      PaintBackground();
  }

  /* Copy server pixmap or client XImage to canvas drawable */

  if (UsePixmaps)
    (void) XCopyArea(MainDisplay, image->mem.pixmap, CanvasXID, MainGC, 0, 0,
		     image->w, image->h, image_x, image_y);
  else
    (void) XPutImage(MainDisplay, CanvasXID, MainGC, image->mem.ximage, 0, 0,
		     image_x, image_y, image->w, image->h);
}

static void set_canvas_size_and_pos(void)
{
  /* Sets frame size and moves frame if part of it lies off-screen */

  int w = 0, h = 0; /* (initialized here to keep gcc happy) */

  /* Set fixed width and/or height to image size if forced */

  if (FixedWidth == FORCE_SIZE) {
    FixedWidth = MIN(Image[Current]->w, MAX_IMAGE_WIDTH);
    FixedWidth = MAX(FixedWidth, MIN_IMAGE_WIDTH);
  }
  if (FixedHeight == FORCE_SIZE) {
    FixedHeight = MIN(Image[Current]->h, MAX_IMAGE_HEIGHT);
    FixedHeight = MAX(FixedHeight, MIN_IMAGE_HEIGHT);
  }

  /* Determine new nominal width and height */

  switch (Sizing) {
  case AUTO:
    if (UNDEF_IMAGE) {
      w = FixedWidth;
      h = FixedHeight;
    }
    else {
      w = MIN(FixedWidth, Image[Current]->w);
      w = MAX(w, MIN_IMAGE_WIDTH);
      h = MIN(FixedHeight, Image[Current]->h);
      h = MAX(h, MIN_IMAGE_HEIGHT);
    }
    break;
  case FULL:
    if (UNDEF_IMAGE)
      return;
    w = MIN(Image[Current]->w, MAX_IMAGE_WIDTH);
    w = MAX(w, MIN_IMAGE_WIDTH);
    h = MIN(Image[Current]->h, MAX_IMAGE_HEIGHT);
    h = MAX(h, MIN_IMAGE_HEIGHT);
    break;
  case FIXED:
    w = FixedWidth;
    h = FixedHeight;
    break;
  default:
    Error("Undefined sizing option.");
  }

  /* Set the canvas to at least include the whole image */

  if (!UNDEF_IMAGE)
    (void) xv_set(canvas,
		  CANVAS_WIDTH, MAX(w, Image[Current]->w),
		  CANVAS_HEIGHT, MAX(h, Image[Current]->h),
		  NULL);

  /* Add scrollbar thickness */

  w += ScrollMargin;
  h += ScrollMargin;

  /* Set new canvas window size */

  (void) xv_set(canvas,
		XV_WIDTH, w,
		XV_HEIGHT, h,
		NULL);

  /* Fit the frame to the canvas */

  window_fit(CanvasFrame);

  /* If resizing was forced, may now need to disable manual resizing */

  if (ForceFirstResize && (Options & NO_RESIZING))
    (void) xv_set(CanvasFrame,
		  FRAME_MIN_SIZE, w, h,
		  FRAME_MAX_SIZE, w, h,
		  NULL);

  /* Reposition frame if part is off-screen, if allowed */

  if (!(Options & NO_MOVING) || ForceFirstResize) {
    int x, y;

    /* Get current frame position */

    x = (int) xv_get(CanvasFrame, XV_X);
    y = (int) xv_get(CanvasFrame, XV_Y);

    /* Take decorations into account */

    x -= FRAME_BORDER;
    y -= (FRAME_BORDER + HEADER_MARGIN);
    w += 2 * FRAME_BORDER;
    h += HEADER_MARGIN + 2 * FRAME_BORDER;

    /* Add the footer width if LIVE_CURSOR */

    if (Options & LIVE_CURSOR)
      h += FOOTER_MARGIN;

    /* Set new position (assuming wm handles request correctly...) */

    if (x < 0)
      (void) xv_set(CanvasFrame, XV_X, 0, NULL);
    else if (x + w > ScreenWidth)
      (void) xv_set(CanvasFrame, XV_X, ScreenWidth - 1 - w, NULL);

    if (y < 0)
      (void) xv_set(CanvasFrame, XV_Y, 0, NULL);
    else if (y + h > ScreenHeight)
      (void) xv_set(CanvasFrame, XV_Y, ScreenHeight - 1 - h, NULL);

    ForceFirstResize = FALSE;
  }
}

static void make_cursor(void)
{
  /* Constructs the canvas cursor if needed and assigns it to canvas window */

  static Cursor cursor = (Cursor) 0;

  if (!cursor) {

    Pixmap src, msk;
    XColor fg, bg;

    /* Construct cursor using xlib calls to ensure a visible image */

    fg.red = fg.green = fg.blue = 0xffff; /* White */
    bg.red = bg.green = bg.blue = 0x0000; /* Black */

    src = XCreatePixmapFromBitmapData(MainDisplay, CanvasXID, cursor_bits,
				    cursor_width, cursor_height, 1L, 0L, 1);
    msk = XCreatePixmapFromBitmapData(MainDisplay, CanvasXID,
				      cursor_mask_bits, cursor_mask_width,
				      cursor_mask_height, 1L, 0L, 1);

    if (!src || !msk)
      Error("Unable to create cursor pixmaps.");

    cursor = XCreatePixmapCursor(MainDisplay, src, msk, &fg, &bg,
				 cursor_x_hot, cursor_y_hot);

    XFreePixmap(MainDisplay, msk);
    XFreePixmap(MainDisplay, src);
  }

  /* Assign the cursor */

  XDefineCursor(MainDisplay, CanvasXID, cursor);
}

static void move_cursor(int dx, int dy)
{
  /* Incrementally adjusts cursor position */

  Rect *r;
  int x, y;

  r = (Rect *) xv_get(canvas, WIN_MOUSE_XY);

  x = r->r_left;
  y = r->r_top;

  x += dx;
  y -= dy; /* y-coordinate works in opposite sense */

  /* Note: cursor coordinates are unit offset, not zero offset */

  x = MIN(MAX(x - image_x, 1), MIN(Image[Current]->w, FixedWidth));
  y = MIN(MAX(y - image_y, 1), MIN(Image[Current]->h, FixedHeight));

  (void) xv_set(canvas, WIN_MOUSE_XY, image_x + x, image_y + y, NULL);

  if (Options & LIVE_CURSOR)
    update_cursor_info();
}

static void update_cursor_info(void)
{
  static char msg[MAX_STR_LEN];

  int x, y;
  Rect *r;

  if (UNDEF_IMAGE)
    return;

  /* Get cursor position relative to image origin */

  r = (Rect *) xv_get(canvas, WIN_MOUSE_XY);

  x = r->r_left - 1 - image_x;
  y = r->r_top - 1 - image_y;

  /*
   * Cursor may be off image if this routine was called by ShowNewImage()
   * and image dimensions changed. If so, generate appropriate message.
   * Otherwise, get color info.
   *
   */

  if (!POINTER_ON_IMAGE(x, y))
    (void) sprintf(msg, "(no cursor)");
  else {
    int i;
    COLOR_T c;
    XColor cell;
    COLORCELL_T *orig_colors;

    /* Get pixel value */

    c = (COLOR_T) XGetPixel(ximage, x, y);

    /* Get original colors */

    if (OneCmap || NewCmap)
      orig_colors = LockedColors;
    else
      orig_colors = Image[Current]->colors;

    /* Show mapping from original colors to current colors */

    for (i = 0; i < MAX_NUM_COLORS; i++)
      if (c == AllocPixels[i])
	break;

    cell.pixel = c;
    XQueryColor(MainDisplay, XaColormap, &cell);

    (void) sprintf(msg, "(%i,%i) = %i (%i,%i,%i) --> (%i,%i,%i)", x, y, c,
		   orig_colors[i].r, orig_colors[i].g, orig_colors[i].b,
		   cell.red >> 8, cell.green >> 8, cell.blue >> 8);
  }

  /* Display info in footer */

  (void) xv_set(CanvasFrame, FRAME_LEFT_FOOTER, msg, NULL);
}

/*** canvas.c ***/
