/* color.c -- routines for handling xa colors */

/* $Id: color.c,v 1.14 1996/05/04 19:57:08 richards Exp $ */

/*
 * $Log: color.c,v $
 * Revision 1.14  1996/05/04 19:57:08  richards
 * minor changes
 *
 * Revision 1.13  1996/04/19 22:12:40  richards
 * moved PRIVATE_COLORMAP and OWN_COLORMAP macros to xa.h
 * replaced FastQuant() and SlowQuant() with a single function Quant()
 * other minor changes
 *
 * Revision 1.12  1996/04/05 16:30:07  richards
 * complete rewrite of pair_compress()
 * fixed a bug in hist_compress()
 *
 * Revision 1.11  1996/03/16 16:56:32  richards
 * renamed xrastool to xa
 * fixed indenting
 * fixed -hist and -pair verbose output
 *
 * Revision 1.10  1996/03/14 21:56:19  richards
 * implemented -fastquant
 * other minor changes
 *
 * Revision 1.9  1996/03/08 20:50:28  richards
 * extensive changes: rewrote hist_compress(), removed UnionColors(),
 *    changed some variable and function names, etc.
 *
 * Revision 1.8  1996/02/21 20:46:42  richards
 * implemented -copy option (new colormap model)
 * added Reduce24() [hack!]
 * added too_many_colors() and fast_quant() (not very fast...)
 *
 * Revision 1.7  1996/02/08 01:31:59  richards
 * implemented Verbose flag
 * fixed bug with "-fast -private" (image data overwritten with zeroes)
 *
 * Revision 1.6  1996/01/30 19:52:52  richards
 * incorporated color macros
 * eliminated XView colormap segment (Cf. StoreColors())
 * replaced global NumReservedColors with local num_shared_colors
 * added local colormap and private pixels array
 * replaced num_free_colors() with get_free_colors() (doesn't deallocate)
 * incorporated COLOR_T (unsigned char) and COLORCELL_T (RGB structure)
 * replaced MakeCms() with InitColors()
 * for -private option, only the canvas frame gets the colormap
 * updated CompressColors(), hist_compress(), & pair_compress() to use pixels[]
 * UnionColors() is BROKEN in this revision
 * added basename() call to shorten filename output
 * replaced (char *) casts with (void *) casts in free() calls
 * use BlackPixel() instead of color 0
 *
 * Revision 1.5  1996/01/28 15:38:10  richards
 * renamed from cms.c to color.c in preparation for new changes
 * changed some comment formats
 *
 * Revision 1.4  1996/01/25 15:27:16  richards
 * malloc.h now included in xrastool.h
 *
 * Revision 1.3  1996/01/21 21:31:57  richards
 * prototyping, NewCmap functionality
 *
 * Revision 1.2  1996/01/18 18:13:30  richards
 * smarter color allocation
 * added routine to get best common colors
 *
 */

#include "xa.h"

#ifdef SUN
#include <memory.h> /* for memcpy() */
#endif

/* Useful flags */

#define COLOR_OK    -1
#define COLOR_FIXED -2
#define COLOR_UNDEF -3

#define MAX_COLOR_DIST 195075 /* (3 * 255**2) */

/* Special type for color histogram (Cf. hist_compress()) */

typedef struct {
  COLOR_T i; /* pixel */
  u_long  n; /* no. occurences in image */
} HIST_T;

/* Special type for distance table (Cf. pair_compress()) */

typedef struct {
  int index, best;
  u_long dist;
} DIST_TBL_T;

/* Global variables */

int NumAllocPixels;
u_long AllocPixels[MAX_NUM_COLORS];
Colormap XaColormap;

/* Local functions */

static int alloc_colors(Display *dpy, u_long *pix, u_int ncols, Colormap cmap);
static void hist_compress(IMAGE_T *image, COLOR_T *new_index);
static int hist_compare(const void *h1, const void *h2);
static void pair_compress(IMAGE_T *image, COLOR_T *new_index);
static Bool too_many_colors(IMAGE_T *image, COLOR_T *rgb);
static void hash_to_colors(u_long *hash_table, IMAGE_T *image, int n);

/*** END OF PREAMBLE ***/

void InitColors(void)
{
  /*
   * Either creates and installs a new read/write-only colormap (if a private
   * colormap is requested or if one or more cells are to be copied from the
   * standard colormap) or allocates as many read/write cells as are available
   * in the standard colormap.
   *
   */

  int num_shared_pixels;

  /* Create and assign a new colormap if appropriate */

  if (OWN_COLORMAP) {

    /* Create a new virtual colormap with all cells allocated read/write */

    XaColormap = XCreateColormap(MainDisplay,
				 RootWindow(MainDisplay, MainScreen),
				 MainVisual, AllocAll);

    /*
     * If private, assign new colormap to canvas frame only, otherwise
     * assign it to the base frame and sub frame as well.
     *
     */

    (void) XSetWindowColormap(MainDisplay,
			      (Window) xv_get(CanvasFrame, XV_XID),
			      XaColormap);

    if (!PRIVATE_COLORMAP) {
      (void) XSetWindowColormap(MainDisplay,
				(Window) xv_get(BaseFrame, XV_XID),
				XaColormap);
      (void) XSetWindowColormap(MainDisplay,
				(Window) xv_get(SubFrame, XV_XID),
				XaColormap);
    }      
  }

  /* Colorcell manipulation may be required if this is not a private map */

  if (PRIVATE_COLORMAP) {
    num_shared_pixels = 0;
    NumAllocPixels = MAX_NUM_COLORS;
  }
  else {
    Colormap std_cmap;

    /* Get the standard colormap, currently installed */

    std_cmap = DefaultColormap(MainDisplay, MainScreen);

    /* Allocate all available read/write pixels in standard colormap */

    NumAllocPixels =
      alloc_colors(MainDisplay, AllocPixels, MAX_NUM_COLORS, std_cmap);

    num_shared_pixels = MAX_NUM_COLORS - NumAllocPixels;

    if (Verbose)
      (void) printf("%i read-only color%s found.", num_shared_pixels,
		    PLURAL(num_shared_pixels));

    /* Copy cells from the standard colormap if requested */

    if (NumColorsToCopy > 0) {
      int i, j, k;
      XColor *xcolors;

      /*
       * First free up the read/write cells in the standard colormap so
       * other applications can use them.
       *
       */

      XFreeColors(MainDisplay, std_cmap, AllocPixels, NumAllocPixels, 0L);

      num_shared_pixels = MIN(num_shared_pixels, NumColorsToCopy);

      if (Verbose)
	(void) printf("..copying %i.", num_shared_pixels);

      /* Allocate temporary storage for colors to copy */

      xcolors = (XColor *) malloc(num_shared_pixels * sizeof(XColor));

      /* Find the first num_shared_pixels read-only cells */

      for (i = j = 0; i < num_shared_pixels && j < MAX_NUM_COLORS; j++) {
	for (k = 0; k < NumAllocPixels; k++)
	  if (AllocPixels[k] == j)
	    break;
	if (k < NumAllocPixels)
	  continue;
	xcolors[i++].pixel = j;
      }

      /* May now have more colors available */

      NumAllocPixels = MAX_NUM_COLORS - num_shared_pixels;

      /* Get the colors in the read-only cells */

      XQueryColors(MainDisplay, std_cmap, xcolors, num_shared_pixels);

      /* Store them in the new colormap */

      XStoreColors(MainDisplay, XaColormap, xcolors, num_shared_pixels);

      /* Free storage */

      free((void *) xcolors);
    } /* if copying colors */
    else
      XaColormap = std_cmap; /* Otherwise just use default colormap */

    if (Verbose) {
      (void) printf("\n");
      if (NumColorsToCopy == 0)
	if (NumAllocPixels == 0)
	  Warning("No colors available!\n");
	else if (NumAllocPixels < 10)
	  Warning("Not very many colors available!\n");
    }
  } /* if !PRIVATE_COLORMAP */

  /* Finally, assign correct pixel mapping if this is a new colormap */

  if (OWN_COLORMAP) {
    int i;

    for (i = 0; i < NumAllocPixels; i++)
      AllocPixels[i] = num_shared_pixels + i;
  }
}

void CompressColors(IMAGE_T *image)
{
  /* Compresses image colormap according to current option */

  static Bool first_call = TRUE;
  static COLOR_T new_index[MAX_NUM_COLORS];

  int n;

  /* Apply compression choice if first call or not unique colormap */

  if (first_call || !OneCmap) {
    if (Verbose)
      (void) printf("   [%i color%s found", image->num_colors,
		    PLURAL(image->num_colors));
    first_call = FALSE;
    switch (CompressOption) {
    case NO_COMPRESS:   /* Private colormap */
      if (Verbose)
	(void) printf(" -- using private colormap]\n");
      return;
    case PAIR_COMPRESS:	/* Merge most-like colors (fast) */
      pair_compress(image, new_index);
      break;
    case HIST_COMPRESS: /* Also remove least-used colors (slow) */
      hist_compress(image, new_index);
      break;
    default:
      Error("Invalid colormap compression option.");
    }
  }

  /* Adjust image data to conform to new colormap */

  if (CompressOption != NO_COMPRESS)
    for (n = image->l - 1; n >= 0; n--)
      image->data[n] = new_index[image->data[n]];
}

void Reduce24to8(IMAGE_T *image, COLOR_T *rgb)
{
  /* Quantizes colors in rgb to image->data (assumed allocated) */

  if (too_many_colors(image, rgb))
    Quant(&image, 1, rgb);
}

void StoreColors(COLORCELL_T colors[], int num_colors)
{
  /* Stores new colors in private (read/write) portion of colormap */

  int i;
  XColor *xcolors;

  xcolors = (XColor *) malloc(num_colors * sizeof(XColor));

  for (i = 0; i < num_colors; i++) {
    xcolors[i].pixel = AllocPixels[i];
    xcolors[i].red   = colors[i].r << 8;
    xcolors[i].green = colors[i].g << 8;
    xcolors[i].blue  = colors[i].b << 8;
    xcolors[i].flags = DoRed | DoGreen | DoBlue;
  }

  XStoreColors(MainDisplay, XaColormap, xcolors, num_colors);

  free((void *) xcolors);
}

void CopyColors(int n, COLORCELL_T *src, COLORCELL_T *dst)
{
  /* Copies n colors from src array to dst array */

  int i;

  for (i = 0; i < n; i++)
    COPY_COLOR(src[i], dst[i]);
}

static int alloc_colors(Display *dpy, u_long *pix, u_int ncols, Colormap cmap)
{
  /*
   * Allocates all the non-shared color cells in the given colormap and
   * returns the corresponding colormap pixels in the array pix. The return
   * value of this function is the number of cells successfully allocated.
   * To get all the private cells, it is necessary to make repeated calls to
   * XAllocColorCells(). Based on xGetCells() by David Robinson, March 1992.
   *
   */

  int ngot, ntry;

  /* Try to allocate all colors initially */

  ngot = 0;
  ntry = ncols;

  /* Allocate in decreasing powers of two as required */

  while (ngot < ncols && ntry) {
    while (XAllocColorCells(dpy, cmap, FALSE, NULL, 0, &pix[ngot], ntry))
      ngot += ntry;
    if (ngot < ncols)
      ntry >>= 1;
  }

  /* Return the number of cells allocated */

  return ngot;
}

static void hist_compress(IMAGE_T *image, COLOR_T *new_index)
{
  /*
   * Compresses image colors by constructing a histogram of color usage
   * and choosing the most popular colors for half the colormap and the
   * most distinct colors for the other half.
   *
   * NOTE: this routine cannot currently identify repeated colors.
   *
   */

  int i, j, k, orig_num_colors, w, h, l, best, *status;
  u_long d, max_dist;
  HIST_T *hist;
  COLORCELL_T *palette;

  if (Verbose)
    (void) printf(" -- hist compress]");

  orig_num_colors = image->num_colors;

  /* Just map image pixels if colors will fit in allowed space */

  if (orig_num_colors <= NumAllocPixels) {
    for (i = 0; i < orig_num_colors; i++)
      new_index[i] = AllocPixels[i];
    if (Verbose)
      (void) printf(" (not req'd)\n");
    return;
  }

  /* Get some memory */

  if ((status = (int *) malloc(orig_num_colors * sizeof(int))) == NULL)
    Error("Out of memory.");

  if ((hist = (HIST_T *) malloc(orig_num_colors * sizeof(HIST_T))) == NULL)
    Error("Out of memory.");

  if ((palette = (COLORCELL_T *) malloc(orig_num_colors *
					sizeof(COLORCELL_T))) == NULL)
    Error("Out of memory.");

  /* Get a copy of the image colors */

  (void) memcpy(palette, image->colors, orig_num_colors * sizeof(COLORCELL_T));

  w = image->w;
  h = image->h;
  l = image->l;

  /* Initialize */

  for (i = 0; i < orig_num_colors; i++) {
    status[i] = COLOR_UNDEF;
    hist[i].i = i;
    hist[i].n = 0;
  }

  /* Construct usage histogram */

  for (i = 0; i < l; i++)
    ++hist[image->data[i]].n;

  /* Sort colors by usage */

  qsort(hist, orig_num_colors, sizeof(HIST_T), hist_compare);

  /*
   * Choose about 50% by popularity, 50% by distinctiveness. Note however
   * that the first color will always be the most popular, and the next 9
   * the most distinct. Inspired by SortColormap() in xvcolor.c from the
   * xv 3.10a distribution, Copyright (C) 1994 by John Bradley.
   *
   */

  for (i = j = 0; i < orig_num_colors; i++) {
    if (i > 0 && (i & 1 || i < 10)) {
      COLORCELL_T c = palette[new_index[best]];
      max_dist = 0;
      for (k = 0; k < orig_num_colors; k++)
	if (status[k] == COLOR_UNDEF) {
	  d = COLOR_DIST(c, palette[k]);
	  if (d > max_dist) {
	    max_dist = d;
	    best = k;
	  }
	}
      if (max_dist == 0)
	Error("Unable to find most unique color.");
    }
    else
      do {
	best = hist[j++].i;
      } while (status[best] == COLOR_OK);
    new_index[best] = i;
    status[best] = COLOR_OK;
    COPY_COLOR(palette[best], image->colors[i]);
  }

  /* Map the pixels */

  for (i = 0; i < orig_num_colors; i++) {
    if (new_index[i] < NumAllocPixels)
      new_index[i] = AllocPixels[new_index[i]];
    else {
      max_dist = MAX_COLOR_DIST;
      for (k = 0; k < orig_num_colors; k++) {
	if (new_index[k] < NumAllocPixels)
	  d = COLOR_DIST(image->colors[new_index[i]],
			 image->colors[new_index[k]]);
	if (d < max_dist) {
	  max_dist = d;
	  best = new_index[k];
	}
      }
      if (max_dist == MAX_COLOR_DIST)
	Error("Unable to find closest color.");
      new_index[i] = AllocPixels[best];
    }
  }

  image->num_colors = NumAllocPixels;

  /* Free storage */

  free((void *) palette);
  free((void *) hist);
  free((void *) status);

  if (Verbose)
    (void) printf("\n");
}

static int hist_compare(const void *h1, const void *h2)
{
  /*
   * Comparison function for color histogram entries. Returns 1 if entry h1
   * is LESS popular than entry h2; -1 if more popular; otherwise 0.
   *
   */

  return (((HIST_T *) h1)->n < ((HIST_T *) h2)->n ?  1 :
	  ((HIST_T *) h1)->n > ((HIST_T *) h2)->n ? -1 : 0);
}

#define COPY_TBL(tbl1,tbl2) {\
   (tbl2).index = (tbl1).index;\
   (tbl2).best  = (tbl1).best;\
   (tbl2).dist  = (tbl1).dist;\
}

static void pair_compress(IMAGE_T *image, COLOR_T *new_index)
{
  /*
   * Compresses image colormap by joining colors that are most alike.
   * Note that no reference is made to the image data itself, so this
   * form of compression is fast but not necessarily the best.
   *
   */

  static int dist_tbl_cmp(const void *td1, const void *tb2);

  int i, j, num_colors, *status, num_to_replace, tmp, best;
  u_long min_dist, dist;
  COLORCELL_T *colors;
  DIST_TBL_T *dist_tbl, *top, tmp_tbl;

  /* Get color info */

  num_colors = image->num_colors;
  colors = image->colors;

  if (Verbose)
    (void) printf(" -- pair compress]");

  /* Initialize */

  if ((status = (int *) malloc(num_colors * sizeof(int))) == NULL)
    Error("Out of memory.");

  for (i = 0; i < num_colors; i++)
    status[i] = COLOR_OK;

  /* Find replacements for reserved colors if required */

  if ((num_to_replace = num_colors - NumAllocPixels) < 0) {
    for (i = 0; i < num_colors; i++)
      new_index[i] = AllocPixels[i];
    if (Verbose)
      (void) printf(" (not req'd)\n");
    return;
  }

  /* Construct "distance" table */

  if ((dist_tbl = (DIST_TBL_T *) malloc(num_colors * sizeof(DIST_TBL_T))) ==
      NULL)
    Error("Out of memory.");

  for (i = 0; i < num_colors; i++) {
    best = COLOR_UNDEF;
    min_dist = MAX_COLOR_DIST + 1;
    for (j = 0; j < num_colors; j++) {
      if (j != i && (dist = COLOR_DIST(colors[i], colors[j])) < min_dist) {
	best = j;
	min_dist = dist;
      }
    }
    if (best == COLOR_UNDEF)
      Error("This shouldn't happen.");
    dist_tbl[i].index = i;
    dist_tbl[i].best = best;
    dist_tbl[i].dist = min_dist;
  }

  /* Sort the table in order of increasing distance */

  qsort(dist_tbl, num_colors, sizeof(DIST_TBL_T), dist_tbl_cmp);

  /* Merge colors pair-wise */

  top = dist_tbl;

  for (i = 0; i < num_to_replace; i++) {

    /*
     * Compute average of current color and its closest match and assign
     * result to current color.
     *
     */

    colors[top->index].r = (colors[top->index].r + colors[top->best].r) / 2;
    colors[top->index].g = (colors[top->index].g + colors[top->best].g) / 2;
    colors[top->index].b = (colors[top->index].b + colors[top->best].b) / 2;

    /* Close color will be replaced by current color */

    status[top->best] = top->index;

    /* All other colors with this close color will also be replaced */

    for (j = 0; j < num_colors; j++)
      if (status[j] == top->best)
	status[j] = top->index;

    /*
     * Recompute distances of all colors that had top->index or top->best as
     * closest match, and find new closest match to new current color.
     * 
     * Note: for best speed, it is assumed that average color (new current
     * color) is still best match to those colors that had top->index or
     * top->best as closest match.
     *
     */

    best = COLOR_UNDEF;
    min_dist = MAX_COLOR_DIST + 1;
    for (j = 1; j < num_colors; j++)
      if (dist_tbl[j].index == top->best)
	tmp = j;
      else {
	dist = COLOR_DIST(colors[top->index], colors[dist_tbl[j].index]);
	if (dist_tbl[j].best == top->index || dist_tbl[j].best == top->best) {
	  dist_tbl[j].best = top->index;
	  dist_tbl[j].dist = dist;
	}
	if (dist < min_dist) {
	  best = dist_tbl[j].index;
	  min_dist = dist;
	}
      }
    if (tmp == num_colors)
      Error("Unable to find matching entry.");
    if (best == COLOR_UNDEF)
      Error("This shouldn't happen.");
    top->best = best;
    top->dist = min_dist;

    /* Delete top->best from distance table */

    j = tmp; tmp = COLOR_UNDEF;
    for (; j < num_colors - i - 1; j++) {
      COPY_TBL(dist_tbl[j + 1], dist_tbl[j]);
      if (tmp == COLOR_UNDEF && dist_tbl[j].dist >= top->dist)
	tmp = j - 1;
    }

    if (tmp == COLOR_UNDEF && dist_tbl[j].dist >= top->dist)
      tmp = j - 1;
    else
      tmp = j; /* i.e. put at end of table */

    /* Move current color in distance table to maintain sort order */

    COPY_TBL(*top, tmp_tbl);
    for (j = 0; j < tmp; j++)
      COPY_TBL(dist_tbl[j + 1], dist_tbl[j]);
    COPY_TBL(tmp_tbl, dist_tbl[tmp]);
  }

  /* Construct new_index and save new color count */

  for (i = j = 0; i < num_colors; i++)
    if (status[i] == COLOR_OK) {
      new_index[i] = AllocPixels[j];
      if (i > j)
	COPY_COLOR(colors[i], colors[j]);
      ++j;
    }

  for (i = 0; i < num_colors; i++)
    if (status[i] != COLOR_OK)
      new_index[i] = new_index[status[i]];

  image->num_colors = num_colors - num_to_replace;

  free((void *) dist_tbl);
  free((void *) status);

  if (Verbose)
    (void) printf("\n");
}

#undef COPY_TBL

static int dist_tbl_cmp(const void *dt1, const void *dt2)
{
  return (((DIST_TBL_T *) dt1)->dist < ((DIST_TBL_T *) dt2)->dist ? -1 :
	  ((DIST_TBL_T *) dt1)->dist > ((DIST_TBL_T *) dt2)->dist ?  1 : 0);
}

static Bool too_many_colors(IMAGE_T *image, COLOR_T *rgb)
{
  /*
   * Returns true if there are more than MAX_NUM_COLORS unique colors
   * in given RGB array. If not, the image is converted from 24 bits to
   * 8 bits and a representative colormap is constructed.
   *
   * NOTE: 8 bit storage (image->data) must be allocated in advance.
   *
   * This routine is based on quick_check() in xv24to8.c from the xv 3.10a
   * distribution, Copyright (C) 1994 by John Bradley.
   *
   */

  int i, l, n, lo, hi, mid;
  u_long hash_table[MAX_NUM_COLORS], hash;
  COLOR_T *ptr;

  l = image->l;

  n = 0;

  for (i = 0, ptr = rgb; i < l; i++) {
    lo = 0;
    hi = n - 1;
    /* Following assumes 8 bit (u_char) color components */
    hash  = (*ptr++ << 16);
    hash += (*ptr++ << 8);
    hash +=  *ptr++;
    while (lo <= hi) {
      mid = (lo + hi) / 2;
      if (hash < hash_table[mid])
	hi = mid - 1;
      else if (hash > hash_table[mid])
	lo = mid + 1;
      else
	break;
    }
    if (hi < lo) {
      if (n == MAX_NUM_COLORS) { /* i.e. have at least 257 distinct colors */
	hash_to_colors(hash_table, image, MAX_NUM_COLORS);
	return TRUE;
      }
      if (n)
	memcpy(&hash_table[lo + 1], &hash_table[lo], (n - lo) * sizeof(long));
      hash_table[lo] = hash;
      ++n;
    }
  }

  if (Verbose)
    (void) printf(" %i unique color%s", n, PLURAL(n));

  /* Construct 8 bit image and colormap */

  for (i = 0, ptr = rgb; i < l; i++) {
    lo = 0;
    hi = n - 1;
    hash  = (*ptr++ << 16);
    hash += (*ptr++ << 8);
    hash +=  *ptr++;
    while (lo <= hi) {
      mid = (lo + hi) / 2;
      if (hash < hash_table[mid])
	hi = mid - 1;
      else if (hash > hash_table[mid])
	lo = mid + 1;
      else
	break;
    }
    if (hi < lo)
      Error("This can't happen.");
    image->data[i] = mid; /* recall num_colors = no. pixels at this stage */
  }

  hash_to_colors(hash_table, image, n);

  return FALSE;
}

static void hash_to_colors(u_long *hash_table, IMAGE_T *image, int n)
{
  /* Converts a hash table to a colormap */

  int i;

  if (image->colors != NULL)
    free((void *) image->colors);

  if (!(image->colors = (COLORCELL_T *) malloc(n * sizeof(COLORCELL_T))))
    Error("Out of memeory.");

  image->num_colors = n;

  for (i = 0; i < n; i++) {
    image->colors[i].r = hash_table[i] >> 16;
    image->colors[i].g = (hash_table[i] >> 8) & 0xff;
    image->colors[i].b = hash_table[i] & 0xff;
  }
}

/*** color.c ***/
