/* xagif.c -- GIF reader based on giftopnm.c from the Netpbm toolkit */

/* $Id: gif.c,v 1.4 1996/04/19 22:14:08 richards Exp $ */

/*
 * $Log: gif.c,v $
 * Revision 1.4  1996/04/19 22:14:08  richards
 * added save_data and save_colormap arguments to ReadGIF()
 * other minor changes
 *
 * Revision 1.3  1996/03/16 17:06:54  richards
 * previously xagif.c
 * renamed xrastool to xa
 * fixed indenting
 *
 * Revision 1.2  1996/03/14 22:13:14  richards
 * minor changes to comments
 *
 * Revision 1.1  1996/03/14 18:07:31  richards
 * Initial revision
 *
 */

#include "xa.h"

#ifdef SUN
#include <memory.h> /* For memcpy() */
#include <unistd.h> /* For SEEK_CUR */
#endif

/* Uncomment the following for verbose output of GIF extension info */

/*#define SHOW_EXTENSION*/

/* Handy macro */

#define READ_ERROR(msg) {\
  if (Verbose)\
    (void) printf("%s\n", msg);\
  return -1;\
}

/*** END OF PREAMBLE ***/

/* +-------------------------------------------------------------------+ */
/* | Copyright 1990, 1991, 1993, David Koblas.  (koblas@netcom.com)    | */
/* |   Permission to use, copy, modify, and distribute this software   | */
/* |   and its documentation for any purpose and without fee is hereby | */
/* |   granted, provided that the above copyright notice appear in all | */
/* |   copies and that both that copyright notice and this permission  | */
/* |   notice appear in supporting documentation.  This software is    | */
/* |   provided "as is" without express or implied warranty.           | */
/* +-------------------------------------------------------------------+ */

#define MAXCOLORMAPSIZE 256

#define MAX_LWZ_BITS 12

#define INTERLACE     0x40
#define LOCALCOLORMAP 0x80

#define BitSet(byte, bit) (((byte) & (bit)) == (bit))

#define ReadOK(file,buffer,len) (fread(buffer, len, 1, file) != 0)

#define LM_to_uint(a,b) (((b)<<8)|(a))

static struct {
  unsigned int Width;
  unsigned int Height;
  COLORCELL_T  ColorMap[MAXCOLORMAPSIZE];
  unsigned int BitPixel; /* number of colors */
  unsigned int ColorResolution;
  unsigned int Background;
  unsigned int AspectRatio;
} GifScreen;

static struct {
  int transparent;
  int delayTime;
  int inputFlag;
  int disposal;
} Gif89 = { -1, -1, -1, 0 }; /* unused... */

static int ReadColorMap(FILE *fd, int number, COLORCELL_T *buffer);
static int DoExtension(FILE *fd, int label);
static int GetDataBlock(FILE *fd, unsigned char *buf);
static int GetCode(FILE *fd, int code_size, int flag);
static int LWZReadByte(FILE *fd, int flag, int input_code_size);
static int ReadImage(FILE *fd, int len, int height, IMAGE_T *image,
		     int interlace, int ignore);

int ReadGIF(FILE *fd, IMAGE_T *image, Bool save_data, Bool save_colormap)
{
  /* RETURN VALUES: -1 on error, 0 if images remain, 1 if last image read */

  unsigned char buf[16];
  unsigned char c;
  COLORCELL_T   localColorMap[MAXCOLORMAPSIZE];
  int           useGlobalColormap;
  int           bitPixel;

  /* If file pointer is at top of file, read header info */

  if (ftell(fd) == 0) {
    char version[4];

    if (! ReadOK(fd,buf,6))
      READ_ERROR("Error reading magic number.");

    if (strncmp((char *)buf,"GIF",3) != 0)
      READ_ERROR("Not a GIF file.");

    strncpy(version, (char *)buf + 3, 3);
    version[3] = '\0';

    if ((strcmp(version, "87a") != 0) && (strcmp(version, "89a") != 0))
      READ_ERROR("Bad version number, not '87a' or '89a'.");

    if (! ReadOK(fd,buf,7))
      READ_ERROR("Failed to read screen descriptor.");

    GifScreen.Width           = LM_to_uint(buf[0],buf[1]);
    GifScreen.Height          = LM_to_uint(buf[2],buf[3]);
    GifScreen.BitPixel        = 2<<(buf[4]&0x07);
    GifScreen.ColorResolution = (((buf[4]&0x70)>>3)+1); /* not used */
    GifScreen.Background      = buf[5]; /* not used */
    GifScreen.AspectRatio     = buf[6];

    if (BitSet(buf[4], LOCALCOLORMAP)) { /* Global Colormap */
      if (ReadColorMap(fd,GifScreen.BitPixel,GifScreen.ColorMap) != 0)
	READ_ERROR("Error reading global colormap.");
    }

    if (GifScreen.AspectRatio != 0 && GifScreen.AspectRatio != 49)
      (void) fprintf(stderr, "ReadGIF() warning: non-square pixels.\n");

  } /* if at top of file */

  /* Read the next image */

  for (;;) {

    if (! ReadOK(fd,&c,1))
      READ_ERROR("EOF / read error on image data.");

    if (c == ';') /* GIF terminator */
      return 1;

    if (c == '!') { /* Extension */
      if (! ReadOK(fd,&c,1))
	READ_ERROR("OF / read error on extention function code.");
      (void) DoExtension(fd, c);
      continue;
    }

    if (c != ',') { /* Not a valid start character */
      READ_ERROR("Bogus character -- ignoring.");
      continue;
    }

    if (! ReadOK(fd,buf,9))
      READ_ERROR("Couldn't read left/top/width/height.");

    useGlobalColormap = ! BitSet(buf[8], LOCALCOLORMAP);

    if (! useGlobalColormap) {
      bitPixel = 1<<((buf[8]&0x07)+1);
      if (ReadColorMap(fd, bitPixel, localColorMap) != 0)
	READ_ERROR("Error reading local colormap.");
    }

    if (ReadImage(fd, LM_to_uint(buf[4],buf[5]), LM_to_uint(buf[6],buf[7]), 
		  image, BitSet(buf[8], INTERLACE), (image == NULL) ||
		  !save_data) != 0)
      READ_ERROR("Error reading image.");

    if (image && save_data) {
      if (image->colors)
	free((void *) image->colors);
      if (save_colormap) {
	image->num_colors =
	  (useGlobalColormap ? GifScreen.BitPixel : bitPixel);
	if (!(image->colors = (COLORCELL_T *) malloc(image->num_colors *
						     sizeof(COLORCELL_T))))
	  READ_ERROR("Out of memory.");
	memcpy(image->colors,
	       (useGlobalColormap ? GifScreen.ColorMap : localColorMap),
	       image->num_colors * sizeof(COLORCELL_T));
      }
      else {
	image->num_colors = 0;
	image->colors = NULL;
      }
      image->compressed = FALSE;
      image->w = GifScreen.Width;
      image->h = GifScreen.Height;
      image->l = image->w * image->h;
      image->d = 8;
    }

    /* Check if this was the last image */

    if (! ReadOK(fd,&c,1))
      READ_ERROR("EOF / read error on image data.");

    /* Reposition file pointer */

    (void) fseek(fd, -1L, SEEK_CUR);

    return (c == ';' ? 1 : 0);
  }
}

static int ReadColorMap(fd,number,buffer)
     FILE        *fd;
     int         number;
     COLORCELL_T *buffer;
{
  int i;

  for (i = 0; i < number; ++i)
    if (! ReadOK(fd, &buffer[i], sizeof(COLORCELL_T)))
      READ_ERROR("Bad colormap.");

  return 0;
}

static int DoExtension(fd, label)
     FILE *fd;
     int  label;
{
  static char buf[256];
  char        *str;

  switch (label) {
  case 0x01:			/* Plain Text Extension */
    str = "Plain Text Extension";
    break;
  case 0xff:			/* Application Extension */
    str = "Application Extension";
    break;
  case 0xfe:			/* Comment Extension */
    str = "Comment Extension";
    while (GetDataBlock(fd, (unsigned char*) buf) != 0)
#ifdef SHOW_EXTENSION
      if (Verbose)
	(void) fprintf(stderr, "GIF comment: %s", buf)
#endif
	  ;
    return 0;
  case 0xf9:			/* Graphic Control Extension -- not used */
    str = "Graphic Control Extension";
    (void) GetDataBlock(fd, (unsigned char*) buf);
    Gif89.disposal  = (buf[0] >> 2) & 0x7;
    Gif89.inputFlag = (buf[0] >> 1) & 0x1;
    Gif89.delayTime = LM_to_uint(buf[1],buf[2]);
    if ((buf[0] & 0x1) != 0)
      Gif89.transparent = buf[3];
    while (GetDataBlock(fd, (unsigned char*) buf) != 0)
      ;
    return 0;
  default:
    str = buf;
    sprintf(buf, "UNKNOWN (0x%02x)", label);
  }

#ifdef SHOW_EXTENSION
  if (Verbose)
    (void) fprintf(stderr, "Got a '%s' extension", str);
#endif

  while (GetDataBlock(fd, (unsigned char*) buf) != 0)
    ;

  return 0;
}

int ZeroDataBlock = FALSE;

static int GetDataBlock(fd, buf)
     FILE          *fd;
     unsigned char *buf;
{
  unsigned char count;

  if (! ReadOK(fd,&count,1))
    READ_ERROR("Error in getting DataBlock size.");

  ZeroDataBlock = count == 0;

  if ((count != 0) && (! ReadOK(fd, buf, count)))
    READ_ERROR("Error in reading DataBlock.");

  return count;
}

static int GetCode(fd, code_size, flag)
     FILE *fd;
     int  code_size;
     int  flag;
{
  static unsigned char buf[280];
  static int           curbit, lastbit, done, last_byte;
  int                  i, j, ret;
  unsigned char        count;

  if (flag) {
    curbit = 0;
    lastbit = 0;
    done = FALSE;
    return 0;
  }

  if ( (curbit+code_size) >= lastbit) {
    if (done) {
      if (curbit >= lastbit)
	READ_ERROR("Ran off the end of my bits.");
      return -1;
    }
    buf[0] = buf[last_byte-2];
    buf[1] = buf[last_byte-1];

    if ((count = GetDataBlock(fd, &buf[2])) == 0)
      done = TRUE;

    last_byte = 2 + count;
    curbit = (curbit - lastbit) + 16;
    lastbit = (2+count)*8 ;
  }

  ret = 0;
  for (i = curbit, j = 0; j < code_size; ++i, ++j)
    ret |= ((buf[ i / 8 ] & (1 << (i % 8))) != 0) << j;

  curbit += code_size;

  return ret;
}

static int LWZReadByte(fd, flag, input_code_size)
     FILE *fd;
     int  flag;
     int  input_code_size;
{
  static int   fresh = FALSE;
  int          code, incode;
  static int   code_size, set_code_size;
  static int   max_code, max_code_size;
  static int   firstcode, oldcode;
  static int   clear_code, end_code;
  static int   table[2][(1<< MAX_LWZ_BITS)];
  static int   stack[(1<<(MAX_LWZ_BITS))*2], *sp;
  register int i;

  if (flag) {
    set_code_size = input_code_size;
    code_size = set_code_size+1;
    clear_code = 1 << set_code_size ;
    end_code = clear_code + 1;
    max_code_size = 2*clear_code;
    max_code = clear_code+2;

    (void) GetCode(fd, 0, TRUE);
               
    fresh = TRUE;

    for (i = 0; i < clear_code; ++i) {
      table[0][i] = 0;
      table[1][i] = i;
    }
    for (; i < (1<<MAX_LWZ_BITS); ++i)
      table[0][i] = table[1][0] = 0;

    sp = stack;

    return 0;
  } else if (fresh) {
    fresh = FALSE;
    do {
      firstcode = oldcode = GetCode(fd, code_size, FALSE);
    } while (firstcode == clear_code);
    return firstcode;
  }

  if (sp > stack)
    return *--sp;

  while ((code = GetCode(fd, code_size, FALSE)) >= 0) {
    if (code == clear_code) {
      for (i = 0; i < clear_code; ++i) {
	table[0][i] = 0;
	table[1][i] = i;
      }
      for (; i < (1<<MAX_LWZ_BITS); ++i)
	table[0][i] = table[1][i] = 0;
      code_size = set_code_size+1;
      max_code_size = 2*clear_code;
      max_code = clear_code+2;
      sp = stack;
      firstcode = oldcode = GetCode(fd, code_size, FALSE);
      return firstcode;
    } else if (code == end_code) {
      int           count;
      unsigned char buf[260];

      if (ZeroDataBlock)
	return -2;

      while ((count = GetDataBlock(fd, buf)) > 0)
	;

      if (count != 0 && Verbose)
	(void) fprintf(stderr,
		       "Missing EOD in data stream (common occurence).");
      return -2;
    }

    incode = code;

    if (code >= max_code) {
      *sp++ = firstcode;
      code = oldcode;
    }

    while (code >= clear_code) {
      *sp++ = table[1][code];
      if (code == table[0][code])
	READ_ERROR("Circular table entry BIG ERROR.");
      code = table[0][code];
    }

    *sp++ = firstcode = table[1][code];

    if ((code = max_code) <(1<<MAX_LWZ_BITS)) {
      table[0][code] = oldcode;
      table[1][code] = firstcode;
      ++max_code;
      if ((max_code >= max_code_size) &&
	  (max_code_size < (1<<MAX_LWZ_BITS))) {
	max_code_size *= 2;
	++code_size;
      }
    }

    oldcode = incode;

    if (sp > stack)
      return *--sp;
  }
  return code;
}

static int ReadImage(fd, len, height, image, interlace, ignore)
     FILE    *fd;
     int     len, height;
     IMAGE_T *image;
     int     interlace, ignore;
{
  unsigned char c;      
  int           v;
  int           xpos = 0, ypos = 0, pass = 0;
  DATA_T        *ptr;

  /*
   **  Initialize the Compression routines
   */
  if (! ReadOK(fd,&c,1))
    READ_ERROR("EOF / read error on image data.");

  if (LWZReadByte(fd, TRUE, c) < 0)
    READ_ERROR("Error reading image.");

  /*
   **  If this is an "uninteresting picture" ignore it.
   */
  if (ignore) {
    while (LWZReadByte(fd, FALSE, c) >= 0)
      ;
    return 0;
  }

  if (!(image->data = (DATA_T *) malloc(len * height * sizeof(DATA_T))))
    READ_ERROR("Out of memory.");

  if (Verbose && interlace)
    (void) printf("interlaced ");

  ptr = image->data;

  while ((v = LWZReadByte(fd,FALSE,c)) >= 0 ) {
    *ptr = v;

    ++xpos;
    if (xpos == len) {
      xpos = 0;
      if (interlace) {
	switch (pass) {
	case 0:
	case 1:
	  ypos += 8; break;
	case 2:
	  ypos += 4; break;
	case 3:
	  ypos += 2; break;
	}

	if (ypos >= height) {
	  ++pass;
	  switch (pass) {
	  case 1:
	    ypos = 4; break;
	  case 2:
	    ypos = 2; break;
	  case 3:
	    ypos = 1; break;
	  default:
	    goto fini;
	  }
	}
      } else
	++ypos;
      if (ypos >= height)
	break;
      ptr = &image->data[ypos * len];
    }
    else
      ++ptr;
  }

 fini:
  if (LWZReadByte(fd,FALSE,c)>=0 && Verbose)
    (void) fprintf(stderr, "Too much input data, ignoring extra...\n");

  return 0;
}

/*** xagif.c ***/
