/* loadimages.c -- routines for loading images into xa */

/* $Id: loadimages.c,v 1.13 1996/05/04 19:58:24 richards Exp $ */

/*
 * $Log: loadimages.c,v $
 * Revision 1.13  1996/05/04 19:58:24  richards
 * fixed bug in -newcmap behaviour
 * fixed bug in pgm and ppm load routines (credit: Yann Ricquebourg)
 *
 * Revision 1.12  1996/04/19 22:18:16  richards
 * added -oneatatime support (extensive changes)
 * single-image and multi-image files now treated very differently
 * other minor changes
 *
 * Revision 1.11  1996/04/05 16:31:31  richards
 * images now transferred to server as they are read in (unless -newcmap)
 * global DataToImage() -> local data_to_image()
 * reading of Sun rasterfile headers now allows for byte swapping
 * cleaned up pgm reading
 * load_mpeg() now allows for byte swapping in colormap
 *
 * Revision 1.10  1996/03/16 17:07:17  richards
 * renamed xrastool to xa
 * fixed indenting
 *
 * Revision 1.9  1996/03/14 22:04:33  richards
 * fixed #include's
 * added GIF and JPEG support
 * added TotalImageMem global to keep track of image memory usage
 * renamed some of the functions
 * added MULTI_IMAGE macro
 * implemented -fastquant option
 * cleaned up verbose output
 * make_graymap() now scales colormap if < 256 gray values
 * other miscellaneous changes
 *
 * Revision 1.8  1996/03/08 20:55:25  richards
 * extensive changes: added support for compressed files as well as
 *    MPEG movies (still needs work), changed arguments to many
 *    functions to improve -visualload handling, added basename() in
 *    case it's needed, introduced FRAME_MARKER for multi-image files,
 *    added 24 bit color support (ppm, MPEG), etc.
 *
 * Revision 1.7  1996/02/08 01:33:40  richards
 * implemented Verbose flag
 * introduced useful READ_ERROR macro
 * added basename() truncation (Alpha only)
 * more consistent stream error handling
 * replaced some inefficient fgetc()'s with fread()'s
 * replaced \n with \x0A in parse_pnm() (safer, but fails on SGI)
 * added check for existing colors in make_graymap()
 *
 * Revision 1.6  1996/01/30 20:04:19  richards
 * replaced time() with stat() to avoid unsynced clocks (Cf. GetLastChange())
 * replaced (char *) casts with (void *) casts in free() calls
 * added WaitingForImages flag to skip loading image data initially
 * added basename() call to shorten filename output
 * incorporated COLOR_T (unsigned char) and COLORCELL_T (RGB structure)
 * fixed bug in parse_pnm() caused by different assumptions about char type
 * added code in load_raw_file() to prompt user for w/h if file length changes
 * added RawQuery flag to force w/h prompts in load_raw_file()
 * incorporated color macros
 * miscellaneous minor changes
 *
 * Revision 1.5  1996/01/27 22:36:30  richards
 * changed some comment formats and indenting
 * introduced DATA_T (unsigned char) for image data
 * removed redundant free((char *) data) which causes bus error on Alpha
 * replaced all getc()'s with fgetc()'s for correct behaviour
 *
 * Revision 1.4  1996/01/25 15:31:43  richards
 * malloc.h now included in xrastool.h
 * added get_file_type() to check image file magic numbers
 * added load_pgm_file() and parse_pnm() for PGM support
 * added load_raw_file() for RAW (gray bytes) support
 * added make_graymap() to generate grayscale colormap
 * removed dealloc_image(): instead, hold onto image data as long as possible
 * image structure now initialized to minimize access errors
 * image access time stored in image structure (for rescan() function)
 * can now use Pixmaps OR XImages for storing image data
 *
 * Revision 1.3  1996/01/21 21:32:26  richards
 * prototyping, NewCmap functionality
 *
 * Revision 1.2  1996/01/18 18:15:18  richards
 * updates for common color code in cms.c
 *
 * Revision 1.1  1996/01/18 16:56:55  richards
 * Initial revision
 *
 */

#include "xa.h"
#include <sys/types.h> /* for stat() */
#include <sys/stat.h>  /* ditto */
#include <ctype.h>     /* for isspace() */
#include <X11/Xutil.h> /* For XDestroyImage() */

#ifdef SUN
#include <memory.h> /* for memcpy() */
#include <unistd.h> /* for SEEK_CUR and SEEK_END */
#endif

#ifdef SGI
#include <unistd.h> /* for unlink() */
#endif

/* Image-specific includes */

#include "rasterfile.h"

#ifdef JPEG
#include <jpeglib.h>
#include <setjmp.h>
#endif

#ifdef MPEG
#include <mpeg.h>
#endif

/* Number of bytes to read to determine file type */

#define MAX_NUM_MAGIC_BYTES 6

/* Supported file types */

#define FT_UNK 0 /* unknown */
#define FT_COM 1 /* compressed */
#define FT_RAS 2 /* Sun rasterfile */
#define FT_PBM 3 /* portable bitmap */
#define FT_PGM 4 /* portable graymap */
#define FT_PPM 5 /* portable pixmap */
#define FT_RAW 6 /* raw bytes */
#define FT_GIF 7 /* graphics interchange format */
#define FT_JPG 8 /* JPEG (JFIF) */
#define FT_MPG 9 /* MPEG */

#define MULTI_IMAGE(type) ((type) == FT_GIF || (type) == FT_MPG)

/* Other useful macros */

#define MAX_COLOR 255

#define BITS_PER_BYTE (MainVisual->bits_per_rgb)

#define ALL_FRAMES (frame == 0)

/* Global variables */

u_long TotalImageMem = 0; /* Total image memory allocation in bytes */
XImage **OldXImagePtr = NULL; /* For OneAtATime functionality */

/* Local variables */

static char old_filename[MAXPATHLEN] = "";
static u_long old_file_size;
static Bool save_colormap;
static FILE *fp;
static int frame; /* For multi-image files */

/* Local functions */

static IMAGE_T *init_image(char *filename, int index);
static int get_file_type(char *filename);
static int uncompress(char *cfile, char **ufile);
static void data_to_image(int index);

static int load_ras(IMAGE_T *image);
static int load_pgm(IMAGE_T *image);
static int load_ppm(IMAGE_T *image);
static int load_raw(IMAGE_T *image);
static int load_gif(char *filename, int index);

#ifdef JPEG
static int load_jpg(IMAGE_T *image);
#endif

#ifdef MPEG
static int load_mpg(char *filename, int index);
#endif

static int read_sun_long(int *l);
static int read_rle_data(int l, int w, int odd_byte, DATA_T *data);
static int parse_pnm(void);
static int make_graymap(IMAGE_T *image, int num_gray);

/*** END OF PREAMBLE ***/

/* Some systems don't have basename(); if not, define it here (global) */

#ifndef HAVE_BASENAME
char *basename(char *path)
{
  char *ptr;

  if (!path)
    return (char *) NULL;

  ptr = strrchr(path, '/');

  if (ptr)
    return (ptr + 1);
  else
    return path;
}
#endif

int LoadData(int num_files, char *filename[], int index)
{
  /*
   * Loads images from given files starting at Image[index]. Note that
   * some files (viz. MPEGs) may contain more than one image.
   *
   */

  int i, n, new_index, file_type;
  char *tmp_name = NULL;

  /* Loop through requested filenames */

  for (new_index = index, i = 0; i < num_files; i++) {

    if (new_index >= MAX_NUM_IMAGES) {
      Warning("Image limit reached.");
      break;
    }

    if (MAX_IMAGE_MEM > 0 && TotalImageMem >= MAX_IMAGE_MEM * 1000000) {
      Warning("Image memory limit reached.");
      break;
    }

    if (Verbose && !Preprocessing)
      (void) printf("\"%s\": ", basename(filename[i]));

    /* Determine whether to retain this image's colormap */

    save_colormap = (!OneCmap || new_index == 0);

    /* Get file type, if possible (opens stream) */

    if ((file_type = get_file_type(filename[i])) == ERROR)
      continue;

    /* If the file is compressed, uncompress it and try again */

    if (file_type == FT_COM) {
      (void) fclose(fp);
      if (tmp_name) {
	(void) unlink(tmp_name);
	free((void *) tmp_name);
      }
      if (uncompress(filename[i], &tmp_name) == ERROR)
	continue;
      if ((file_type = get_file_type(tmp_name)) == ERROR)
	continue;

      if (file_type == FT_COM) {
	(void) fclose(fp);
	Warning("Unable to uncompress file? Skipping...");
	continue;
      }

      if (MULTI_IMAGE(file_type)) {
	(void) fclose(fp);
	Warning("Can't handle compressed multi-image file types...");
	continue;
      }
    }

    if (file_type == FT_UNK) {
      (void) fclose(fp);
      if (Preprocessing)
	(void) printf("\"%s\": ", filename[i]);
      Warning("Unknown or unsupported image type -- skipping...");
      continue;
    }

    /* Call appropriate loader (treat possible multi-image files seperately) */

    n = 0;

    if (MULTI_IMAGE(file_type)) {
      switch (file_type) {
      case FT_GIF: /* GIF */
	n = load_gif(filename[i], new_index);
	break;
      case FT_MPG: /* MPEG */
#ifdef MPEG
	n = load_mpg(filename[i], new_index);
#endif
	break;
      default:
	Error("This shouldn't happen!");
      }
      if (Verbose) {
	if (ALL_FRAMES) {
	  if (n > 1)
	    (void) printf("Multi-image file: %i frames read.\n", n);
	  else if (n == 0)
	    (void) printf("Multi-image file: No frames found!\n");
	}
	else if (n == 0)
	  (void) printf("Multi-image file: Requested frame not found!\n");
      }
    } /* if (MULTI_IMAGE(file_type)) */
    else {
      IMAGE_T *image;

      if (!(image = init_image(filename[i], new_index)))
	(void) printf("Unable to initialize image.\n");
      else if (Preprocessing)
	n = 1; /* Assume success */
      else {
	switch (file_type) {
	case FT_RAS: /* rasterfile */
	  n = load_ras(image);
	  break;
	case FT_PGM: /* pgm file */
	  n = load_pgm(image);
	  break;
	case FT_PPM: /* ppm file */
	  n = load_ppm(image);
	  break;
	case FT_RAW: /* raw file */
	  n = load_raw(image);
	  break;
	case FT_JPG: /* JPEG */
#ifdef JPEG
	  n = load_jpg(image);
#endif
	  break;
	default:
	  Error("This shouldn't happen!");
	}
      } /* if (!Preprocessing) */

      if (n == 1) {

	/* If a new image was loaded, adjust image counter */

	if (new_index == NumImages)
	  ++NumImages;

	/*
	 * Store data as client XImage or server Pixmap and show image now
	 * if visual load requested. If this is the first call and a new
	 * colormap has been requested, data transfer to XImage or Pixmap is
	 * delayed until all the images have been read in.
	 *
	 */

	if (!Preprocessing) {
	  image->last_change = GetLastChange(filename[i]);
	  TotalImageMem += image->l;
	  if (!NewCmap)
	    data_to_image(new_index);
	  if (LoadingDelayedImages) {
	    ShowNewImage();
	    ++Current;
	  }
	}

      } /* if (n == 1) */

    } /* if !(MULTI_IMAGE(file_type)) */

    /* Close input stream */

    (void) fclose(fp);

    /* Point to next element in Image array */

    new_index += n;
  }

  /* If a temporary file exists, remove it */

  if (tmp_name) {
    (void) unlink(tmp_name);
    free((void *) tmp_name);
  }

  /* If a new colormap was requested, compute it now */

  if (NewCmap) {
    Quant(Image, NumImages, NULL);
    for (i = index; i < new_index; i++)
      data_to_image(i);
    NewCmap = FALSE;
  }

  /* Return number of files successfully loaded */

  return new_index - index;
}

time_t GetLastChange(char *filename)
{
  /*
   * Uses stat() system call to return time of last change to file.
   * NOTE: this method can be defeated by file system cacheing.
   *
   */

  char *ptr;
  time_t t;
  struct stat buffer;

  /* Truncate filename at frame marker if present */

  if ((ptr = strrchr(filename, FRAME_MARKER)))
    *ptr = '\0';

  /* Get time of last change to file if available */

  t = (stat(filename, &buffer) ? 0 : buffer.st_ctime);

  /* Restore frame marker if necessary */

  if (ptr)
    *ptr = FRAME_MARKER;

  return t;
}

static IMAGE_T *init_image(char *filename, int index)
{
  /*
   * Either allocates space for a new image or prepares to overwrite
   * an existing image, depending on the value of "index".
   *
   */

  IMAGE_T *image;

  /*
   * If image already exists, hold onto it as long as possible, otherwise
   * attempt to allocate space for new image structure.
   *
   * NOTE: in the following comparison, use '<=' instead of '<' to reuse
   * space that may have been allocated but not filled when a read error
   * occurred.
   *
   */

  if (index <= NumImages && ((image = Image[index]) != NULL)) {
    if (filename == image->filename) /* Safety check */
      Error("init_image(): filename and image->filename identical pointers!");
    (void) strncpy(old_filename, image->filename, MAXPATHLEN);
    old_filename[MAXPATHLEN - 1] = '\0'; /* Ensure termination */
    if ((image->filename = (char *) realloc((void *) image->filename,
					    strlen(filename) + 1)) == NULL) {
      Warning("Out of memory?");
      return NULL;
    }
    if (!OneAtATime) /* otherwise done in data_to_image() */
      TotalImageMem -= image->l;
  }
  else {
    if ((image = (IMAGE_T *) malloc(sizeof(IMAGE_T))) == NULL ||
	(image->filename = (char *) malloc(strlen(filename) + 1)) == NULL) {
      Warning("Out of memory.");
      return NULL;
    }
    image->last_change = 0;
    image->num_colors = 0;
    image->colors = NULL;
    image->compressed = FALSE;
    image->w = image->h = image->l = image->d = 0;
    image->data = NULL;
    if (UsePixmaps)
      image->mem.pixmap = (Pixmap) 0;
    else
      image->mem.ximage = NULL;
    Image[index] = image;
    *old_filename = '\0';
    old_file_size = 0;
  }

  /* Store filename */

  (void) strcpy(image->filename, filename);

  return image;
}

static int get_file_type(char *filename)
{
  /* Returns file type, if known */

  byte magic[MAX_NUM_MAGIC_BYTES];

  (void) memset(magic, 0, MAX_NUM_MAGIC_BYTES);

  if ((fp = fopen(filename, "r")) == NULL) {

    /* If filename contains a frame marker, user may want a single frame */

    char *ptr;

    if ((ptr = strrchr(filename, FRAME_MARKER)))
      *ptr = '\0'; /* Truncate at frame marker */

    /* Try again */

    if ((fp = fopen(filename, "r")) == NULL) {
      (void) fprintf(stderr, "Unable to open \"%s\".\n", filename);
      return ERROR;
    }
    else {

      /*
       * Check to see if this is a GIF or MPEG file, otherwise continue with
       * truncated filename.
       *
       */

      (void) fread(magic, 1, MAX_NUM_MAGIC_BYTES, fp);

      rewind(fp);

      frame = atoi(ptr + 1);

      if (frame <= 0) {
	(void) fprintf(stderr, "Invalid frame specification.\n");
	return FT_UNK;
      }

      if (strncmp((char *) magic, "GIF87a", (size_t) 6) == 0 ||
	  strncmp((char *) magic, "GIF89a", (size_t) 6) == 0) {
	return FT_GIF;
      }

      if (magic[0] == 0x00 && (magic[1] & 0x7f) == 0x00 &&
	  magic[2] == 0x01 && (magic[3] & 0x7f) == 0x33) {
#ifdef MPEG
	return FT_MPG;
#else
	return FT_UNK;
#endif
      }
    }
  }
  else
    (void) fread(magic, 1, MAX_NUM_MAGIC_BYTES, fp);

  /* No frame numbers associated with image */

  frame = 0;

  /*
   * In some cases it's easier to read the magic number again when loading
   * the image data, so rewind (note this is ESSENTIAL for MPEGs).
   *
   */

  rewind(fp);

  /* Determine file type */

  if (magic[0] == 0x1f && (magic[1] == 0x9d || magic[1] == 0x8b))
    return FT_COM; /* compressed file */

  if (magic[0] == 0x59 && (magic[1] & 0x7f) == 0x26 &&
      magic[2] == 0x6a && (magic[3] & 0x7f) == 0x15)
    return FT_RAS; /* ras file */

  if (magic[0] == 'P') /* pnm file */
    switch (magic[1]) {
    case '1':
    case '4':
      return FT_UNK; /* FT_PBM not currently supported */
    case '2':
    case '5':
      return FT_PGM;
    case '3':
    case '6':
      return FT_PPM;
    }

  if (strncmp((char *) magic, "GIF87a", (size_t) 6) == 0 ||
      strncmp((char *) magic, "GIF89a", (size_t) 6) == 0)
    return FT_GIF; /* GIF */

  if (magic[0] == 0xff && magic[1] == 0xd8 && magic[2] == 0xff)
#ifdef JPEG
    return FT_JPG; /* JPEG */
#else
    return FT_UNK;
#endif

  if (magic[0] == 0x00 && (magic[1] & 0x7f) == 0x00 &&
      magic[2] == 0x01 && (magic[3] & 0x7f) == 0x33)
#ifdef MPEG
    return FT_MPG; /* MPEG */
#else
    return FT_UNK;
#endif

  /* All other cases exhausted...assume raw file if allowed */

  if (Raw)
    return FT_RAW;

  /* Otherwise the file type is unknown */

  return FT_UNK;
}

#define MKTEMP_XMASK "XXXXXX" /* Mask used by mktemp() */

static int uncompress(char *cfile, char **ufile)
{
  /*
   * Uncompresses "cfile" into a temporary file "*ufile". It is the
   * responsibility of the calling function to remove the temporary
   * file and deallocate the space reserved for the filename.
   *
   */

  size_t lf, lc;
  char *command;

  /* Determine storage required for temporary file name and command string */

  lf = strlen(cfile) + strlen(MKTEMP_XMASK) + 1; /* Extra '1' for \0 char */

  lc = strlen(ZCAT_FORMAT) + strlen(cfile) + lf + 3; /* 3 = '<' + '>' + '\0' */

  /* Construct unique filename */

  if ((*ufile = (char *) malloc(lf)) == NULL) {
    Warning("Out of memory.");
    return ERROR;
  }

  (void) sprintf(*ufile, "%s%s", cfile, MKTEMP_XMASK);

  if (!mktemp(*ufile)) {
    Warning("Unable to construct unique filename.");
    free((void *) *ufile);
    return ERROR;
  }

  /* Construct command string */

  if ((command = (char *) malloc(lc)) == NULL) {
    Warning("Out of memory.");
    free((void *) *ufile);
    return ERROR;
  }

  (void) sprintf(command, "%s<%s>%s", ZCAT_FORMAT, cfile, *ufile);

  /*
   * Execute command, but don't deallocate *ufile on failure, in case the
   * temporary file was created (in which case the calling function should
   * delete it).
   *
   */

  if (system(command)) {
    Warning("Unable to uncompress file.");
    free((void *) command);
    return ERROR;
  }

  free((void *) command);

  return OK;
}

#undef MKTEMP_XMASK

static void data_to_image(int index)
{
  /*
   * Compresses image colormap if necessary and transfers image data
   * to XImage (client resident) or Pixmap (server resident).
   *
   */

  IMAGE_T *image;
  XImage *ximage;

  image = Image[index];

  if (!image || !image->data)
    Error("Undefined image and/or data structure.");

  /* Compress colors (24 bit images are already done) */

  if (!image->compressed)
    CompressColors(image);

  /* Create client-resident ximage */

  ximage = XCreateImage(MainDisplay, MainVisual, image->d, ZPixmap, 0,
			(char *) image->data, image->w, image->h,
			BITS_PER_BYTE, 0);

  /* Move to server-resident pixmap if requested, otherwise store it  */

  if (UsePixmaps) {
    if (image->mem.pixmap)
      XFreePixmap(MainDisplay, image->mem.pixmap);
    image->mem.pixmap = XCreatePixmap(MainDisplay, CanvasXID, image->w,
				      image->h, image->d);
    XPutImage(MainDisplay, image->mem.pixmap, MainGC, ximage, 0, 0, 0, 0,
	      image->w, image->h);
    (void) XDestroyImage(ximage); /* Also deallocates image->data */
  }
  else {

    /*
     * If only one image is to be stored in memory at a time (XImages
     * only), deallocate previous image if it exists then assign new
     * image.
     *
     */

    if (OneAtATime) {
      if (OldXImagePtr) {
	TotalImageMem -= (*OldXImagePtr)->width * (*OldXImagePtr)->height;
	(void) XDestroyImage(*OldXImagePtr);
	*OldXImagePtr = NULL;
      }
      image->mem.ximage = ximage;
      OldXImagePtr = &image->mem.ximage;
    }
    else {
      if (image->mem.ximage)
	(void) XDestroyImage(image->mem.ximage);
      image->mem.ximage = ximage; /* Image data in "image->data" */
    }
  }

  image->data = NULL; /* No further direct access of "image->data" allowed */
}

#define READ_ERROR(msg) {\
  if (Verbose)\
    (void) printf("%s\n", msg);\
  if (image) {\
    if (*old_filename)\
      (void) strcpy(image->filename, old_filename);\
    TotalImageMem += old_file_size;\
  }\
  return 0;\
}

static int load_ras(IMAGE_T *image)
{
  /* Loads Sun rasterfile info into image and data */

  int i, w, h, d, num_colors, odd_byte, ras_length, num_pixels;
  struct rasterfile header;

  /* Read header, taking byte ordering into account */

  (void) read_sun_long(&header.ras_magic);
  (void) read_sun_long(&header.ras_width);
  (void) read_sun_long(&header.ras_height);
  (void) read_sun_long(&header.ras_depth);
  (void) read_sun_long(&header.ras_length);
  (void) read_sun_long(&header.ras_type);
  (void) read_sun_long(&header.ras_maptype);
  if (read_sun_long(&header.ras_maplength) != 0)
    READ_ERROR("Error reading ras header.");

  if (header.ras_magic != RAS_MAGIC)
    READ_ERROR("Expected rasterfile...?");

  /* Determine type */

  switch (header.ras_type) {
  case RT_OLD:
    if (Verbose)
      (void) printf("Old ras ");
    break;
  case RT_STANDARD:
    if (Verbose)
      (void) printf("Std ras ");
    break;
  case RT_BYTE_ENCODED:
    if (Verbose)
      (void) printf("RLE ras ");
    break;
  default:
    READ_ERROR("Unknown ras encoding.");
  }

  /* Read colormap */

  switch (header.ras_maptype) {
  case RMT_NONE:
    READ_ERROR("B&W ras images not currently supported.");
  case RMT_EQUAL_RGB:
    if (Verbose)
      (void) printf("RGB ");
    break;
  case RMT_RAW:
    READ_ERROR("Raw colors in ras files not supported.");
  default:
    READ_ERROR("Unknown ras colormap type.");
  }

  num_colors = (header.ras_maptype == RMT_NONE ? 2 : header.ras_maplength / 3);

  if (num_colors <= 0)
    READ_ERROR("No ras colors found!");

  if (Verbose)
    (void) printf("%i color%s ", num_colors, PLURAL(num_colors));

  if (image->colors)
    free((void *) image->colors);

  if (save_colormap) {
    image->num_colors = num_colors;
    if ((image->colors = (COLORCELL_T *) malloc(num_colors *
						sizeof(COLORCELL_T))) == NULL)
      READ_ERROR("Out of memory.");
  }
  else {
    image->num_colors = 0;
    image->colors = NULL;
  }

  image->compressed = FALSE;

  if (header.ras_maptype == RMT_EQUAL_RGB) {
    COLOR_T *r, *g, *b;

    r = (COLOR_T *) malloc(num_colors * sizeof(COLOR_T));
    g = (COLOR_T *) malloc(num_colors * sizeof(COLOR_T));
    b = (COLOR_T *) malloc(num_colors * sizeof(COLOR_T));

    (void) fread((void *) r, sizeof(COLOR_T), num_colors, fp);
    (void) fread((void *) g, sizeof(COLOR_T), num_colors, fp);
    (void) fread((void *) b, sizeof(COLOR_T), num_colors, fp);

    if (save_colormap)
      for (i = 0; i < num_colors; i++) {
	image->colors[i].r = r[i];
	image->colors[i].g = g[i];
	image->colors[i].b = b[i];
      }

    free((void *) b);
    free((void *) g);
    free((void *) r);
  }

  /* Read image data (note image lines rounded to a multiple of 16 bits...) */

  w = header.ras_width;
  h = header.ras_height;
  d = header.ras_depth;

  if (Verbose)
    (void) printf("%ix%i.", w, h);

  odd_byte = w % 2;

  ras_length = (w + odd_byte) * h * d / BITS_PER_BYTE;

  num_pixels = w * h * d / BITS_PER_BYTE;

  if (header.ras_length != ras_length && header.ras_type != RT_OLD &&
      header.ras_type != RT_BYTE_ENCODED)
    READ_ERROR("Inconsistent eas data length.");

  if (ras_length <= 0)
    READ_ERROR("No ras data found!");

  if ((image->data = (DATA_T *) malloc(num_pixels * sizeof(DATA_T))) == NULL)
    READ_ERROR("Out of memory.");

  if (header.ras_type == RT_BYTE_ENCODED) { /* RLE raster */
    if (read_rle_data(ras_length, w, odd_byte, image->data) == ERROR)
      READ_ERROR("Corrupt RLE file.");
  }
  else /* Standard raster */
    for (i = 0; i < h; i++) {
      if (fread((void *) &(image->data[i * w]), sizeof(DATA_T), w, fp) != w)
	READ_ERROR("Error reading rasterfile data.");
      if (odd_byte)
	(void) fgetc(fp);
    }

  image->w = w;
  image->h = h;
  image->l = w * h;
  image->d = d;

  if (Verbose)
    (void) printf("\n");

  return 1; /* One file successfully read */
}

#define MAX_NUM_PNM_CHARS_PER_LINE 70

#define PARSE_PNM()\
  if (parse_pnm() == EOF)\
    READ_ERROR("Unexpected end of file.")

static int load_pgm(IMAGE_T *image)
{
  int i, w, h, l, max_gray;
  char line[MAX_NUM_PNM_CHARS_PER_LINE];
  Bool byte_encoded;

  /* Read magic number */

  (void) fscanf(fp, "%s", line);

  if (line[0] != 'P' || (line[1] != '2' && line[1] != '5') ||
      strlen(line) != 2)
    READ_ERROR("Expected pgm file...?");

  /* Set flag if this is the compressed format */

  byte_encoded = (line[1] == '5');

  if (Verbose)
    if (byte_encoded)
      (void) printf("Raw pgm ");
    else
      (void) printf("Std pgm ");

  PARSE_PNM();

  /* Get width, height, and maximum gray value */

  (void) fscanf(fp, "%d", &w);
  PARSE_PNM();
  (void) fscanf(fp, "%d", &h);
  PARSE_PNM();

  /*
   * In byte_encoded format, only ONE whitespace character (typically a
   * newline) is allowed after the maxval.
   *
   */

  if (byte_encoded)
    (void) fscanf(fp, "%d%*c", &max_gray);
  else {
    (void) fscanf(fp, "%d", &max_gray);
    PARSE_PNM();
  }

  if (Verbose)
    (void) printf("(max val %i) %ix%i.", max_gray, w, h);

  /* Generate graymap if needed */

  if (save_colormap) {
    if (make_graymap(image, max_gray < MAX_COLOR ? max_gray + 1 :
		     MAX_NUM_COLORS) == ERROR)
      READ_ERROR("Out of memory.");
  }
  else {
    image->num_colors = 0;
    image->colors = NULL;
    image->compressed = FALSE;
  }

  /* Read data (should not contain comments) */

  l = w * h;

  if ((image->data = (DATA_T *) malloc(l * sizeof(DATA_T))) == NULL)
    READ_ERROR("Out of memory.");

  if (byte_encoded) { /* Compressed pgm */
    if ((i = fread((void *) image->data, sizeof(DATA_T), l, fp)) < l) {
      if (Verbose)
	(void) printf("--unexpected EOF-- ");
      l = i;
    }
  }
  else { /* Standard pgm */
    int g;
    DATA_T *ptr = image->data;

    for (i = 0; i < l; i++, ptr++)
      if (fscanf(fp, "%d", &g) != 1) {
	if (Verbose)
	  (void) printf("--unexpected EOF-- ");
	l = i;
	break;
      }
      else {
	if (max_gray > MAX_COLOR)
	  *ptr = (MAX_COLOR * g) / max_gray; /* Scale down */
        else
	  *ptr = g;
      }
  }

  image->w = w;
  image->h = h;
  image->l = l;
  image->d = 8;

  if (Verbose)
    (void) printf("\n");

  return 1;
}

static int load_ppm(IMAGE_T *image)
{
  int i, w, h, l, n, max_color;
  char line[MAX_NUM_PNM_CHARS_PER_LINE];
  Bool byte_encoded;
  COLOR_T *rgb; /* For temporary storage of 24 bit image */

  /* Read magic number */

  (void) fscanf(fp, "%s", line);

  if (line[0] != 'P' || (line[1] != '3' && line[1] != '6') ||
      strlen(line) != 2)
    READ_ERROR("Expected ppm file...?");

  /* Set flag if this is the compressed format */

  byte_encoded = (line[1] == '6');

  if (Verbose)
    if (byte_encoded)
      (void) printf("Raw ppm ");
    else
      (void) printf("Std ppm ");

  PARSE_PNM();

  /* Get width, height, and maximum color value */

  (void) fscanf(fp, "%d", &w);
  PARSE_PNM();
  (void) fscanf(fp, "%d", &h);
  PARSE_PNM();

  if (byte_encoded)
    (void) fscanf(fp, "%d%*c", &max_color);
  else {
    (void) fscanf(fp, "%d", &max_color);
    PARSE_PNM();
  }

  if (Verbose)
    (void) printf("(max val %i) %ix%i ", max_color, w, h);

  /* Allocate space for 8 bit and 24 bit images */

  l = w * h;

  if ((image->data = (DATA_T *) malloc(l * sizeof(DATA_T))) == NULL)
    READ_ERROR("Out of memory.");

  n = 3 * l;

  if ((rgb = (COLOR_T *) malloc(n * sizeof(COLOR_T))) == NULL)
    READ_ERROR("Out of memory.");

  /* Read RGB triplets */

  if (byte_encoded) { /* Compressed ppm */
    if ((i = fread((void *) rgb, sizeof(COLOR_T), n, fp)) < n) {
      if (Verbose)
	(void) printf("--unexpected EOF-- ");
      l = (n = i) / 3;
    }
  }
  else { /* Standard ppm */
    int c;
    COLOR_T *ptr = rgb;

    for (i = 0; i < n; i++, ptr++)
      if (fscanf(fp, "%d", &c) != 1) {
	if (Verbose)
	  (void) printf("--unexpected EOF-- ");
	break;
      }
      else {
	if (max_color > MAX_COLOR)
	  *ptr = (MAX_COLOR * c) / max_color;
	else
	  *ptr = c;
      }
  }

  if (max_color < MAX_COLOR) {
    COLOR_T scale[MAX_NUM_COLORS];

    for (i = 0; i < MAX_NUM_COLORS; i++)
      scale[i] = (i * MAX_COLOR) / max_color;

    for (i = 0; i < n; i++)
      rgb[i] *= scale[rgb[i]];
  }

  image->w = w;
  image->h = h;
  image->l = l;

  if (image->colors)
    free((void *) image->colors);

  /* Convert to 8 bit image */

  if (Verbose)
    (void) printf("[24->8]");

  Reduce24to8(image, rgb);

  free((void *) rgb);

  /* Dump color info if not saving colormap */

  if (!save_colormap) {
    image->num_colors = 0;
    free((void *) image->colors);
    image->colors = NULL;
    image->compressed = FALSE;
  }

  image->d = 8;

  if (Verbose)
    (void) printf(".\n");

  return 1;
}

#undef PARSE_PNM
#undef MAX_NUM_PNM_CHARS_PER_LINE 70

static int load_raw(IMAGE_T *image)
{
  int i, l;

  if (Verbose)
    (void) printf("Raw ");

  /* Generate graymap if needed */

  if (save_colormap) {
    if (make_graymap(image, MAX_NUM_COLORS) == ERROR)
      READ_ERROR("Out of memory.");
  }
  else {
    image->num_colors = 0;
    image->colors = NULL;
    image->compressed = FALSE;
  }

  /* Get length of file in bytes */

  (void) fseek(fp, 0, SEEK_END);
  l = ftell(fp);
  rewind(fp);

  /* Reset width and height if user requested file-by-file querying */

  if (RawQuery)
    RawWidth = RawHeight = 0; /* Overrides -rw/-rh */

  /*
   * If the width and/or height have been supplied, check that the values
   * are consistent with the file length, otherwise query the user for the
   * correct values. These will be the defaults for subsequent raw files,
   * unless the RawQuery flag is set.
   *
   */

  do {
    if (!Verbose)
      (void) printf("Raw file \"%s\":\n", image->filename);
    else if (!RawWidth || !RawHeight)
      (void) printf("\n");

    while (RawWidth <= 0) {
      (void) printf("==> w? ");
      (void) scanf("%d", &RawWidth);
    }

    while (RawHeight <= 0) {
      (void) printf("==> h? ");
      (void) scanf("%d", &RawHeight);
    }

    if (l != RawWidth * RawHeight) {
      (void) printf("Expected %i bytes (%ix%i) but found %i.\n",
		    RawWidth * RawHeight, RawWidth, RawHeight, l);
      (void) printf("Please supply correct width (w) and height (h):");
      RawWidth = RawHeight = 0;
    }

  } while (!RawWidth || !RawHeight);

  if (Verbose)
    (void) printf("%ix%i.", RawWidth, RawHeight);

  /* Read bytes */

  if ((image->data = (DATA_T *) malloc(l * sizeof(DATA_T))) == NULL)
    READ_ERROR("Out of memory.");

  if ((i = fread((void *) image->data, sizeof(DATA_T), l, fp)) < l && Verbose)
    (void) printf("--unexpected EOF-- ");

  image->w = RawWidth;
  image->h = RawHeight;
  image->l = l;
  image->d = 8;

  if (Verbose)
    (void) printf("\n");

  return 1;
}

#define MULTI_IMAGE_READ_ERROR(msg) {\
  if (Verbose)\
    (void) printf("%s\n", msg);\
  if (image) {\
    if (*old_filename)\
      (void) strcpy(image->filename, old_filename);\
    TotalImageMem += old_file_size;\
  }\
  (void) fclose(mfp);\
  mfp = NULL;\
  return n;\
}

static int load_gif(char *filename, int index)
{
  /* Static memory for OneAtATime support */

  static FILE *mfp = NULL;
  static char prev_filename[MAXPATHLEN];
  static time_t prev_last_change = 0;
  static int prev_frame = 0;

  int n, this_frame, result;
  char image_name[MAXPATHLEN];
  IMAGE_T *image;
  time_t last_change;
  Bool save_data, more_frames, keep_going;

  last_change = GetLastChange(filename);

  /*
   * Unless the file is already open and the requested frame is located
   * further down the stream, need to (re)open dedicated input stream.
   *
   */

  if (!mfp || strcmp(filename, prev_filename) ||
      last_change != prev_last_change || frame <= prev_frame) {

    if (mfp)
      (void) fclose(mfp);

    prev_frame = 0;

    if ((mfp = fopen(filename, "r")) == NULL)
      READ_ERROR("No longer able to read GIF file?!");

    (void) strcpy(prev_filename, filename);
    prev_last_change = last_change;
  }

  /* Load the frames */

  n = 0; this_frame = prev_frame;

  do {
    ++this_frame;
    save_data = (LoadingDelayedImages || !OneAtATime ||
		 (!Preprocessing && (!ALL_FRAMES || n == 0)));
    if (ALL_FRAMES || this_frame == frame) {
      (void) sprintf(image_name, "%s%c%i", filename, FRAME_MARKER, this_frame);
      if (!(image = init_image(image_name, index)))
	MULTI_IMAGE_READ_ERROR("GIF read error: Unable to initialize image.");
      if (Verbose && save_data) {
	if (n > 0)
	  (void) printf("\"%s\": ", basename(image_name));
	(void) printf("GIF ");
      }
    }
    else
      image = NULL;
    if ((result = ReadGIF(mfp, image, save_data, save_data && save_colormap))
	== -1)
      MULTI_IMAGE_READ_ERROR("Error during GIF read.");
    more_frames = (result == 0);
    keep_going = (more_frames && (ALL_FRAMES || this_frame < frame));
    if (image) {
      if (save_data) {
	if (save_colormap && (image->num_colors == 0 || image->colors == NULL))
	  MULTI_IMAGE_READ_ERROR("Error: No GIF color information.");
	if (image->data == NULL)
	  MULTI_IMAGE_READ_ERROR("Error: No GIF data.");
      }
      if (Verbose && save_data) {
	if (save_colormap)	 
	  (void) printf("%i color%s ", image->num_colors,
			PLURAL(image->num_colors));
	(void) printf("%ix%i.\n", image->w, image->h);
      }
      if (index == NumImages)
	if (++NumImages == MAX_NUM_IMAGES) {
	  if (Verbose)
	    (void) printf("Image limit reached.\n");
	  keep_going = FALSE;
	}
      if (save_data) {
	image->last_change = last_change;
	TotalImageMem += image->l;
	if (!NewCmap)
	  data_to_image(index);
	if (LoadingDelayedImages) {
	  ShowNewImage();
	  ++Current;
	}
      }
      ++n; ++index;
    } /* if (image) */
  } while (keep_going);

  prev_frame = this_frame;

  /* If GIF consists of only one image, remove frame designator */

  if (ALL_FRAMES && n == 1)
    *(strrchr(image->filename, FRAME_MARKER)) = '\0';
  else if (Verbose && Preprocessing)
    (void) printf("GIF ");

  /* All done */

  if (!more_frames) {
    (void) fclose(mfp);
    mfp = NULL;
  }

  return n;
}

#ifdef JPEG

#define JPEG_ERROR(msg) {\
  jpeg_destroy_decompress(&cinfo);\
  READ_ERROR(msg);\
}

struct my_error_mgr {
  struct jpeg_error_mgr pub; /* "public" fields */
  jmp_buf setjmp_buffer;     /* for return to caller */
};

typedef struct my_error_mgr *my_error_ptr;

static void my_error_exit(j_common_ptr cinfo)
{
  /* cinfo->err really points to a my_error_mgr struct, so coerce pointer */

  my_error_ptr myerr = (my_error_ptr) cinfo->err;

  /* Always display the message */

  (*cinfo->err->output_message) (cinfo);

  /* Return control to the setjmp point */

  longjmp(myerr->setjmp_buffer, 1);
}

static int load_jpg(IMAGE_T *image)
{
  int l, skip;
  COLOR_T *rgb, *ptr;

  struct jpeg_decompress_struct cinfo;
  struct my_error_mgr jerr;

  if (Verbose)
    (void) printf("JPEG ");

  /* Allocate JPEG decompression object */

  cinfo.err = jpeg_std_error((struct jpeg_error_mgr *) &jerr);
  jerr.pub.error_exit = my_error_exit; /* override error_exit */

  /* Establish the setjmp return context for my_error_exit to use */

  if (setjmp(jerr.setjmp_buffer))

    /* If we get here, the JPEG code has signaled an error */

    JPEG_ERROR("Error during JPEG read.");

  /* Initialize the JPEG decompression object */

  jpeg_create_decompress(&cinfo);

  /* Specify data source */

  jpeg_stdio_src(&cinfo, fp);

  /* Read file parameters */

  (void) jpeg_read_header(&cinfo, TRUE);

  /* Start decompressor */

  (void) jpeg_start_decompress(&cinfo);

  /* Allocate space for 8 bit and 24 bit images */

  l = cinfo.image_width * cinfo.image_height;

  if ((image->data = (DATA_T *) malloc(l * sizeof(DATA_T))) == NULL)
    JPEG_ERROR("Out of memory.");

  switch (cinfo.out_color_space) {
  case JCS_GRAYSCALE:
    if (Verbose)
      (void) printf("grayscale ");
    rgb = NULL;
    break;
  case JCS_RGB:
    if (Verbose)
      (void) printf("RGB ");
    if ((rgb = (COLOR_T *) malloc(3 * l * sizeof(COLOR_T))) == NULL)
      JPEG_ERROR("Out of memory.");
    break;
  default:
    JPEG_ERROR("Unsupported JPEG output color space.");
  }

  if (Verbose)
    (void) printf("%ix%i", cinfo.image_width, cinfo.image_height);

  /* Read the image */

  ptr = (rgb ? rgb : image->data);
  skip = (rgb ? 3 * cinfo.output_width : cinfo.output_width);

  while (cinfo.output_scanline < cinfo.output_height) {
    (void) jpeg_read_scanlines(&cinfo, (JSAMPARRAY) &ptr, 1);
    ptr += skip;
  }

  /* Finish decompression */

  (void) jpeg_finish_decompress(&cinfo);

  /* Release JPEG decompression object */

  jpeg_destroy_decompress(&cinfo);

  image->w = cinfo.output_width;
  image->h = cinfo.output_height;
  image->l = l;

  if (image->colors)
    free((void *) image->colors);

  if (rgb) {
    if (Verbose)
      (void) printf(" [24->8]");
    Reduce24to8(image, rgb);
    free((void *) rgb);
    if (!save_colormap) {
      image->num_colors = 0;
      free((void *) image->colors);
      image->colors = NULL;
      image->compressed = FALSE;
    }
  }
  else {
    if (save_colormap) {
      if (make_graymap(image, MAX_NUM_COLORS) == ERROR)
	READ_ERROR("Out of memory.");
    }
    else {
      image->num_colors = 0;
      image->colors = NULL;
      image->compressed = FALSE;
    }
  }

  image->d = 8;

  if (Verbose)
    (void) printf(".\n");

  return 1;
}

#undef JPEG_ERROR

#endif /* JPEG */

#ifdef MPEG

#define MPEG_ERROR(msg) {\
  CloseMPEG(&mpeg);\
  MULTI_IMAGE_READ_ERROR(msg);\
}

#define SHOW_FRAME_COUNT (verbose && ALL_FRAMES && !LoadingDelayedImages)

static int load_mpg(char *filename, int index)
{
  /* Static memory for OneAtATime functionality */

  static FILE *mfp = NULL;
  static char prev_filename[MAXPATHLEN];
  static time_t prev_last_change = 0;
  static int prev_frame = 0, num_colors;
  static COLORCELL_T colors[MAX_NUM_COLORS];
  static ImageDesc mpeg;

  int n, this_frame;
  char image_name[MAXPATHLEN];
  IMAGE_T *image;
  DATA_T *data = NULL;
  time_t last_change;
  Bool verbose, more_frames, keep_going, save_data;

  if (Preprocessing)
    (void) printf("\"%s\": ", filename);

  last_change = GetLastChange(filename);

  /* Disable verbose output from CompressColors() during MPEG load */

  verbose = Verbose;
  Verbose = FALSE;

  /* Open the MPEG file if needed */

  if (!mfp || strcmp(filename, prev_filename) ||
      last_change != prev_last_change || frame <= prev_frame) {

    int i;

    if (mfp) {
      CloseMPEG(&mpeg);
      (void) fclose(mfp);
    }

    prev_frame = 0;

    if ((mfp = fopen(filename, "r")) == NULL)
      READ_ERROR("No longer able to read MPEG file?!");

    if (!OpenMPEG(mfp, &mpeg))
      READ_ERROR("Problem opening MPEG.");

    /* Get the colormap and store it if needed */

    if ((num_colors = mpeg.ColormapSize) <= 0)
      MPEG_ERROR("Corrupt MPEG? (no colors found!)");

    if (save_colormap) {
      if (num_colors > MAX_NUM_COLORS)
	MPEG_ERROR("Too many MPEG colors!");

      for (i = 0; i < num_colors; i++) {
	colors[i].r = ((mpeg.Colormap[i].red   & 0xff00) >> 8) |
	               (mpeg.Colormap[i].red   & 0xff);
	colors[i].g = ((mpeg.Colormap[i].green & 0xff00) >> 8) |
	               (mpeg.Colormap[i].green & 0xff);
	colors[i].b = ((mpeg.Colormap[i].blue  & 0xff00) >> 8) |
                       (mpeg.Colormap[i].blue  & 0xff);
      }
    }

    (void) strcpy(prev_filename, filename);
    prev_last_change = last_change;
  }

  if (verbose)
    (void) printf("MPEG %i color%s %ix%i", num_colors, PLURAL(num_colors),
		  mpeg.Width, mpeg.Height);

  if (SHOW_FRAME_COUNT)
    (void) printf(" f    ");

  n = 0; this_frame = prev_frame;

  do {
    ++this_frame;
    if (ALL_FRAMES || this_frame == frame) {
      (void) sprintf(image_name, "%s%c%i", filename, FRAME_MARKER, this_frame);
      if (!(image = init_image(image_name, index)))
	MULTI_IMAGE_READ_ERROR("MPEG read error: Unable to initialize image.");
    }
    else
      image = NULL;
    if (!data && !(data = (DATA_T *) malloc(mpeg.Size * sizeof(DATA_T))))
      MPEG_ERROR("Out of memory.");
    more_frames = GetMPEGFrame(&mpeg, data);
    keep_going = more_frames && (ALL_FRAMES || this_frame < frame);
    if (image) {
      if (SHOW_FRAME_COUNT)
	(void) printf("\b\b\b%03i", n + 1);
      save_data = (LoadingDelayedImages || !OneAtATime ||
		   (!Preprocessing && (!ALL_FRAMES || n == 0)));
      if (save_data) {
	if (image->colors)
	  free((void *) image->colors);
	if (save_colormap) {
	  image->num_colors = num_colors;
	  if (!(image->colors = (COLORCELL_T *)
		malloc(num_colors * sizeof(COLORCELL_T))))
	    MPEG_ERROR("Out of memory.");
	  memcpy(image->colors, colors, num_colors * sizeof(COLORCELL_T));
	}
	else {
	  image->num_colors = 0;
	  image->colors = NULL;
	}
	image->compressed = FALSE; /* Unfortunately must assume this */
	image->w = mpeg.Width;
	image->h = mpeg.Height;
	image->l = mpeg.Size;
	image->d = mpeg.Depth;
	image->data = data;
      } /* if (save_data) */
      else
	free((void *) data);
      data = NULL;
      if (index == NumImages)
	if (++NumImages == MAX_NUM_IMAGES) {
	  if (verbose)
	    (void) printf(" Image limit reached");
	  keep_going = FALSE;
	}
      if (save_data) {
	image->last_change = last_change;
	TotalImageMem += image->l;
	if (!NewCmap)
	  data_to_image(index);
	if (LoadingDelayedImages) {
	  ShowNewImage();
	  ++Current;
	}
      }
      ++n; ++index;
    } /* if (image) */
  } while (keep_going);

  prev_frame = this_frame;

  /* If MPEG consists of only one image, remove frame designator */

  if (ALL_FRAMES && n == 1)
    *(strrchr(image->filename, FRAME_MARKER)) = '\0';

  /* All done */

  if (!more_frames) {
    CloseMPEG(&mpeg);
    (void) fclose(mfp);
    mfp = NULL;
  }

  if (data)
    free((void *) data);

  if (SHOW_FRAME_COUNT)
    (void) printf("\n");
  else if (verbose)
    (void) printf(".\n");

  Verbose = verbose;

  return n;
}

#undef SHOW_FRAME_COUNT

#undef MPEG_ERROR

#endif /* MPEG */

#undef MULTI_IMAGE_READ_ERROR

#undef READ_ERROR

static int read_sun_long(int *l)
{
  /*
   * Taken by permission from read_sun_long() in xvsunras.c from the
   * xv 3.10a distribution, Copyright (C) 1994 by John Bradley.
   *
   */

  int c0, c1, c2, c3;

  c0 = fgetc(fp);
  c1 = fgetc(fp);
  c2 = fgetc(fp);
  c3 = fgetc(fp);

  *l = (((u_long) c0 & 0xff) << 24) |
       (((u_long) c1 & 0xff) << 16) |
       (((u_long) c2 & 0xff) <<  8) |
       (((u_long) c3 & 0xff));

  if (ferror(fp) || feof(fp))
    return EOF;

  return 0;
}

#define RLE_ESC 128 /* Escape character for RLE encoding */

/* Useful macro */

#define PUT(c) {++i; if (!odd_byte || i % mw) data[p++] = (DATA_T) c;}

static int read_rle_data(int l, int w, int odd_byte, DATA_T *data)
{
  /* Reads RLE data on fp */

  int i, p, mw, c, nc;

  i = p = 0;

  mw = w + 1; /* For use in PUT(c) */

  /* It's a bit complicated... */

  while (i < l) {
    if ((c = fgetc(fp)) == EOF)
      break;
    if (c == RLE_ESC) {
      if ((nc = fgetc(fp)) == EOF)
	break;
      if (nc == 0)
	PUT(RLE_ESC) /* Note: no semicolon */
      else if (nc == 1) {
	if ((c = fgetc(fp)) != RLE_ESC)
	  return ERROR;
	PUT(RLE_ESC)
	PUT(RLE_ESC)
      }
      else {
	if ((c = fgetc(fp)) == EOF)
	  break;
	for (; nc >= 0; nc--)
	  PUT(c)
      }
    }
    else
      PUT(c)
  }

  return OK;
}

#undef PUT
#undef RLE_ESC

static int parse_pnm(void)
{
  int c;
  DATA_T cc;

  /* Absorb whitespace */

  while (isspace(c = fgetc(fp)));

  /* Absorb comments */

  while (c == '#') {
    (void) fscanf(fp, "%*[^\012]\n%c", &cc); /* Octal 012 = x0A = '\n' */
    c = cc;
  }

  /* Backup one character */

  (void) fseek(fp, -1L, SEEK_CUR);

  /* Return last character read */

  return c;
}

static int make_graymap(IMAGE_T *image, int num_gray)
{
  /* Constructs grayscale colormap, scaled to "num_gray" gray values */

  int i;

  image->num_colors = num_gray;

  /*
   * It would be nice to use realloc() here (and in other similar situations)
   * but the implementation varies between architectures. In particular,
   * some versions do not accept a NULL pointer as an argument.
   *
   */

  if (image->colors)
    free((void *) image->colors);

  if ((image->colors = (COLORCELL_T *)
       malloc(num_gray * sizeof(COLORCELL_T))) == NULL)
    return ERROR;

  if (num_gray > 1 && num_gray < MAX_NUM_COLORS)
    for (i = 0; i < num_gray; i++) /* No semi-colon before "else" */
      SET_GRAY(image->colors[i], i * MAX_COLOR / (num_gray - 1))
  else
    for (i = 0; i < num_gray; i++)
      SET_GRAY(image->colors[i], i);

  image->compressed = FALSE;

  return OK;
}

/*** loadimages.c ***/
