#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <sys/audio.h>

#include "xbiff-proto.h"

#ifdef __alpha
typedef int u_32;
#else
typedef long u_32;
#endif

typedef struct ahdr {
  u_32 magic;
  u_32 hdr_size;
  u_32 data_size;
  u_32 encoding;
  u_32 sample_rate;
  u_32 channels;
} ahdr;

#define MAGIC_LITTLE	((u_32 )0x2e736e64)
#define MAGIC_BIG	((u_32 )0x646e732e)

#define FORMAT_ULAW	1
#define FORMAT_LINEAR16	3
#define FORMAT_ALAW	27

extern int errno;
extern char *sys_errlist[];

static int debug = 0;

int
setupAudio(audio, data, name)
int audio;
int data;
const char *name;
{
  ahdr dataHdr;
  size_t bytes;
  int format;
  const char *formatStr;
  int dest;
  struct audio_limits alim;

  bytes = read(data, &dataHdr, sizeof(dataHdr));
  if (bytes <= 0) {
    fprintf(stderr, "Couldn't read audio file header from %s: %s\n",
	    name, sys_errlist[errno]);
    return(0);
  }

  if (dataHdr.magic != MAGIC_LITTLE) {
    fprintf(stderr, "%s is a big-endian audio file!\n", name);
    return(0);
  }

  if (dataHdr.encoding == FORMAT_ULAW) {
    format = AUDIO_FORMAT_ULAW;
    formatStr = "u-law";
  } else if (dataHdr.encoding == FORMAT_LINEAR16) {
    format = AUDIO_FORMAT_LINEAR16BIT;
    formatStr = "linear16";
  } else if (dataHdr.encoding == FORMAT_ALAW) {
    format = AUDIO_FORMAT_ALAW;
    formatStr = "a-law";
  } else {
    format = -1;
    formatStr = "UNKNOWN";
  }

  if (debug) {
    printf("%s", (dataHdr.magic == MAGIC_BIG ? "big" :
		  (dataHdr.magic == MAGIC_LITTLE ? "little" :
		   "unknown")));
    printf(" %s", formatStr);
    printf(" %d Hz", dataHdr.sample_rate);
    printf(" %s", (dataHdr.channels == 1 ? "mono" :
		   (dataHdr.channels == 2 ? "stereo" :
		    (dataHdr.channels == 4 ? "quad" :
		     "unknown"))));
/*  printf(" H(%ld)", dataHdr.hdr_size); */
/*  printf(" D(%ld)", dataHdr.data_size); */
    printf("\n");
  }

  dest = AUDIO_OUT_SPEAKER;
  if (ioctl(audio, AUDIO_SET_OUTPUT, dest)) {
    perror("Can't set output device");
    return(0);
  }

  if (format != -1 && ioctl(audio, AUDIO_SET_DATA_FORMAT, format)) {
    perror("Can't set audio format");
    return(0);
  }

  if (dataHdr.sample_rate == 8012)
    dataHdr.sample_rate = 8000;
  if (ioctl(audio, AUDIO_SET_SAMPLE_RATE, dataHdr.sample_rate)) {
    if (errno == EINVAL)
      fprintf(stderr, "Bad sample rate %d from %s\n",
	      dataHdr.sample_rate, name);
    else
      fprintf(stderr, "Can't set sample rate %d from %s: %s\n",
	      dataHdr.sample_rate, name, sys_errlist[errno]);
    return(0);
  }

  if (ioctl(audio, AUDIO_SET_CHANNELS, dataHdr.channels)) {
    fprintf(stderr, "Can't set %d channels from %s: %s\n",
	    dataHdr.channels, name, sys_errlist[errno]);
    return(0);
  }

  if (ioctl(audio, AUDIO_SET_GAINS, AUDIO_MAX_GAIN)) {
    perror("Can't set gain");
    return(0);
  }

  if (ioctl(audio, AUDIO_GET_LIMITS, &alim)) {
    perror("Can't get limits");
    return(0);
  }

  /* max out transmit buffer size */
  if (ioctl(audio, AUDIO_SET_TXBUFSIZE, alim.max_transmit_buffer_size)) {
    perror("Can't set xmit buffer size");
    return(0);
  }

  if (lseek(data, dataHdr.hdr_size, SEEK_SET) == -1) {
    perror("Can't seek to end of audio header");
    return(0);
  }

  return(dataHdr.data_size);
}

int
HPplay(filename)
const char *filename;
{
  int audio, data;
  const char *fullname;
  char *buf;
  int buflen;
  size_t bytes;

  /* make sure we can open the audio device first */
  audio = open("/dev/audio", O_WRONLY | O_NDELAY, 0);
  if (audio < 0) {
    if (debug)
      perror("Can't open audio device");
    return(1);
  }

  /* try to find the sound file */
  fullname = findSoundFile(filename);
  if (fullname == NULL) {
    fprintf(stderr, "Can't find audio file \"%s\".\n", filename);
    return(1);
  }

  /* make sure we can open the sound file */
  data = open(fullname, O_RDONLY, 0);
  if (data < 0) {
    fprintf(stderr, "Can't open audio file \"%s\"\n", fullname);
    return(1);
  }

  buflen = setupAudio(audio, data, filename);
  if (!buflen) {
    close(audio);
    close(data);
    return(1);
  }

  buf = (char *)malloc(buflen+1);
  if (!buf) {
    perror("buffer allocation");
    return(1);
  }

  while ((bytes = read(data, buf, buflen)) > 0) {
    write(audio, buf, bytes);
    if (debug)
      printf("%d bytes\n", bytes);
  }

  free(buf);
  close(audio);
  close(data);
  return(0);
}

#ifdef HPPLAY_MAIN

/* needed by findSound() */
char *sound_path = NULL;

int
main(argc, argv)
int argc;
char *argv[];
{
  int i;

  for (i = 1; i < argc; i++) {
    HPplay(argv[i]);
    sleep(2);
  }

  exit(0);
}
#endif /* HPPLAY_MAIN */
