/*-------------------------------------------------------------------------*/
/* functions.c 								   */
/*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*/
/*      written by Tim Edwards, 8/13/93    				   */
/*-------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <math.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

/*-------------------------------------------------------------------------*/
/* Local includes							   */
/*-------------------------------------------------------------------------*/

#include "colordefs.h"
#include "xcircuit.h"

/*-------------------------------------------------------------------------*/
/* External Variable definitions					   */
/*-------------------------------------------------------------------------*/

extern Display *dpy;
extern Window  win;
extern Pixmap STIPPLE[8];
extern short eventmode;
extern Clientdata areastruct;
extern int *appcolors;

/*------------------------------------------------------------------------*/
/* Declarations of functions defined externally to functions.c		  */
/*------------------------------------------------------------------------*/

extern short ULength();
extern void UTransformPoints();

void UTransformbyCTM();

/*-----------------------*/
/* Arithmetic functions  */
/*-----------------------*/

#define RADFAC 0.0174532925199   /* pi / 180 */

/*------------------------------------------------------------------------*/
/* A sine and cosine lookup table for fast axis rotation 		  */
/*------------------------------------------------------------------------*/

float sinevals[] = {0, 0.0872, 0.1736, 0.2588, 0.342, 0.4226, 0.5, 0.5736,
	0.6427, 0.7071, 0.7660, 0.8192, 0.8660, 0.9063, 0.9397, 0.9659,
	0.9848, 0.9962, 1,
	-0.0872, -0.1736, -0.2588, -0.342, -0.4226, -0.5, -0.5736, -0.6427,
	-0.7071, -0.7660, -0.8192, -0.866, -0.9063, -0.9397, -0.9659,
	-0.9848, -0.9962, -1};

float *cosine[RSTEPS + 1] = {
     &sinevals[18], &sinevals[18], &sinevals[17], &sinevals[16],
     &sinevals[15], &sinevals[14], &sinevals[13], &sinevals[12], &sinevals[11],
     &sinevals[10], &sinevals[9], &sinevals[8], &sinevals[7], &sinevals[6],
     &sinevals[5], &sinevals[4], &sinevals[3], &sinevals[2], &sinevals[1],
     &sinevals[0], &sinevals[19], &sinevals[20], &sinevals[21], &sinevals[22],
     &sinevals[23], &sinevals[24], &sinevals[25], &sinevals[26], &sinevals[27],
     &sinevals[28], &sinevals[29], &sinevals[30], &sinevals[31], &sinevals[32],
     &sinevals[33], &sinevals[34], &sinevals[35], &sinevals[36], &sinevals[35],
     &sinevals[34], &sinevals[33], &sinevals[32], &sinevals[31], &sinevals[30],
     &sinevals[29], &sinevals[28], &sinevals[27], &sinevals[26], &sinevals[25],
     &sinevals[24], &sinevals[23], &sinevals[22], &sinevals[21], &sinevals[20],
     &sinevals[19], &sinevals[0], &sinevals[1], &sinevals[2], &sinevals[3],
     &sinevals[4], &sinevals[5], &sinevals[6], &sinevals[7], &sinevals[8],
     &sinevals[9], &sinevals[10], &sinevals[11], &sinevals[12], &sinevals[13],
     &sinevals[14], &sinevals[15], &sinevals[16], &sinevals[17]};
float *sine[RSTEPS + 1] =  {
     &sinevals[0], &sinevals[0], &sinevals[1], &sinevals[2],
     &sinevals[3], &sinevals[4], &sinevals[5], &sinevals[6], &sinevals[7],
     &sinevals[8], &sinevals[9], &sinevals[10], &sinevals[11], &sinevals[12],
     &sinevals[13], &sinevals[14], &sinevals[15], &sinevals[16], &sinevals[17],
     &sinevals[18], &sinevals[17], &sinevals[16], &sinevals[15], &sinevals[14],
     &sinevals[13], &sinevals[12], &sinevals[11], &sinevals[10], &sinevals[9],
     &sinevals[8], &sinevals[7], &sinevals[6], &sinevals[5], &sinevals[4],
     &sinevals[3], &sinevals[2], &sinevals[1], &sinevals[0], &sinevals[19],
     &sinevals[20], &sinevals[21], &sinevals[22], &sinevals[23], &sinevals[24],
     &sinevals[25], &sinevals[26], &sinevals[27], &sinevals[28], &sinevals[29],
     &sinevals[30], &sinevals[31], &sinevals[32], &sinevals[33], &sinevals[34],
     &sinevals[35], &sinevals[36], &sinevals[35], &sinevals[34], &sinevals[33],
     &sinevals[32], &sinevals[31], &sinevals[30], &sinevals[29], &sinevals[28],
     &sinevals[27], &sinevals[26], &sinevals[25], &sinevals[24], &sinevals[23],
     &sinevals[22], &sinevals[21], &sinevals[20], &sinevals[19]};

/*------------------------------------------------------------------------*/
/* find the squared length of a wire (or distance between two points in   */
/* user space).								  */
/*------------------------------------------------------------------------*/

long sqwirelen(userpt1, userpt2)
  XPoint *userpt1, *userpt2;
{
  long xdist, ydist;

  xdist = (long)userpt2->x - (long)userpt1->x;
  ydist = (long)userpt2->y - (long)userpt1->y;
  return (xdist * xdist + ydist * ydist);
}

/*------------------------------------------------------------------------*/
/* floating-point version of the above					  */
/*------------------------------------------------------------------------*/

float fsqwirelen(userpt1, userpt2)
  XfPoint *userpt1, *userpt2;
{
  float xdist, ydist;

  xdist = userpt2->x - userpt1->x;
  ydist = userpt2->y - userpt1->y;
  return (xdist * xdist + ydist * ydist);
}

/*------------------------------------------------------------------------*/
/* Find absolute distance between two points in user space		  */
/*------------------------------------------------------------------------*/

int wirelength(userpt1, userpt2)
  XPoint *userpt1, *userpt2;
{
  unsigned long xdist, ydist;

  xdist = (long)(userpt2->x) - (long)(userpt1->x);
  ydist = (long)(userpt2->y) - (long)(userpt1->y);
  return (int)sqrt((double)(xdist * xdist + ydist * ydist));
}

/*------------------------------------------------------------------------*/
/* Find the closest (squared) distance from a point to a line		  */
/*------------------------------------------------------------------------*/

long finddist(linept1, linept2, userpt)
  XPoint *linept1, *linept2, *userpt;
{
   long a, b, c, frac;
   float protod;

   c = sqwirelen(linept1, linept2);
   a = sqwirelen(linept1, userpt);
   b = sqwirelen(linept2, userpt);
   frac = a - b;
   if (frac >= c) return b;	  /* "=" is important if c = 0 ! */
   else if (-frac >= c) return a;
   else {
      protod = (float)(c + a - b);
      return (a - (long)((protod * protod) / (float)(c << 2)));
   }
}

/*------------------------------------------------------------------------*/
/* Calculate points for an arc						  */
/*------------------------------------------------------------------------*/

void calcarc(thearc)
   arcptr thearc;
{
   short idx, thidx;
   int sarc;
   float theta;

   /* assume that angle2 > angle1 always: must be guaranteed by other routines */

   sarc = (int)(thearc->angle2 - thearc->angle1) * RSTEPS;
   thearc->number = (sarc / 360) + 1;
   if (sarc % 360 != 0) thearc->number++;
	   
   thidx = ((int)(thearc->angle1 * RSTEPS) / 360) + 2;
   if (thidx < 1) thidx += RSTEPS;
   else if (thidx > RSTEPS) thidx -= RSTEPS;

   theta = thearc->angle1 * RADFAC;
   thearc->points[0].x = (float)thearc->position.x + 
	   fabs((float)thearc->radius) * cos(theta);
   thearc->points[0].y = (float)thearc->position.y +
	   (float)thearc->yaxis * sin(theta);

   for (idx = 1; idx < thearc->number - 1; idx++) {
      thearc->points[idx].x = (float)thearc->position.x + 
	   fabs((float)thearc->radius) * (*cosine[thidx]);
      thearc->points[idx].y = (float)thearc->position.y +
	   (float)thearc->yaxis * (*sine[thidx]);
      thidx++;
      if (thidx == RSTEPS + 1) thidx = 1;
   }

   theta = thearc->angle2 * RADFAC;
   thearc->points[thearc->number - 1].x = (float)thearc->position.x + 
	   fabs((float)thearc->radius) * cos(theta);
   thearc->points[thearc->number - 1].y = (float)thearc->position.y +
	   (float)thearc->yaxis * sin(theta);

   if (thearc->radius < 0) reversefpoints(thearc->points, thearc->number);
}

/*------------------------------------------------------------------------*/
/* Create a Bezier curve approximation from control points		  */
/* (using PostScript formula for Bezier cubic curve)			  */
/*------------------------------------------------------------------------*/

float par[INTSEGS];
float parsq[INTSEGS];
float parcb[INTSEGS];

void initsplines()
{
   float t;
   short idx;

   for (idx = 0; idx < INTSEGS; idx++) {
      t = (float)(idx + 1) / (INTSEGS + 1);
      par[idx] = t;
      parsq[idx] = t * t;
      parcb[idx] = parsq[idx] * t;
   }
}

/*------------------------------------------------------------------------*/
/* Compute spline coefficients						  */
/*------------------------------------------------------------------------*/

void computecoeffs(thespline, ax, bx, cx, ay, by, cy)
  splineptr thespline;
  float *ax, *bx, *cx, *ay, *by, *cy;
{
   *cx = 3.0 * (float)(thespline->ctrl[1].x - thespline->ctrl[0].x);
   *bx = 3.0 * (float)(thespline->ctrl[2].x - thespline->ctrl[1].x) - *cx;
   *ax = (float)(thespline->ctrl[3].x - thespline->ctrl[0].x) - *cx - *bx;

   *cy = 3.0 * (float)(thespline->ctrl[1].y - thespline->ctrl[0].y);
   *by = 3.0 * (float)(thespline->ctrl[2].y - thespline->ctrl[1].y) - *cy;
   *ay = (float)(thespline->ctrl[3].y - thespline->ctrl[0].y) - *cy - *by;   
}

/*------------------------------------------------------------------------*/

void calcspline(thespline)
   splineptr	thespline;
{
   float ax, bx, cx, ay, by, cy;
   short idx;

   computecoeffs(thespline, &ax, &bx, &cx, &ay, &by, &cy);
   for (idx = 0; idx < INTSEGS; idx++) {
      thespline->points[idx].x = ax * parcb[idx] + bx * parsq[idx] +
	 cx * par[idx] + (float)thespline->ctrl[0].x;
      thespline->points[idx].y = ay * parcb[idx] + by * parsq[idx] +
	 cy * par[idx] + (float)thespline->ctrl[0].y;
   }
}

/*------------------------------------------------------------------------*/
/* Find the (x,y) position and tangent rotation of a point on a spline    */
/*------------------------------------------------------------------------*/

void findsplinepos(thespline, t, retpoint, retrot)
  splineptr    thespline;
  float t;
  XPoint *retpoint;
  int *retrot;
{
   float ax, bx, cx, ay, by, cy;
   float tsq = t * t;
   float tcb = tsq * t;
   double dxdt, dydt;

   computecoeffs(thespline, &ax, &bx, &cx, &ay, &by, &cy);
   retpoint->x = (short)(ax * tcb + bx * tsq + cx * t + (float)thespline->ctrl[0].x);
   retpoint->y = (short)(ay * tcb + by * tsq + cy * t + (float)thespline->ctrl[0].y);

   if (retrot != NULL) {
      dxdt = (double)(3 * ax * tsq + 2 * bx * t + cx);
      dydt = (double)(3 * ay * tsq + 2 * by * t + cy);
      *retrot = (int)(INVRFAC * atan2(dxdt, dydt));  /* reversed y, x */
      if (*retrot < 0) *retrot += 360;
   }
}

/*------------------------------------------------------------------------*/
/* floating-point version of the above					  */
/*------------------------------------------------------------------------*/

void ffindsplinepos(thespline, t, retpoint)
  splineptr    thespline;
  float t;
  XfPoint *retpoint;
{
   float ax, bx, cx, ay, by, cy;
   float tsq = t * t;
   float tcb = tsq * t;

   computecoeffs(thespline, &ax, &bx, &cx, &ay, &by, &cy);
   retpoint->x = ax * tcb + bx * tsq + cx * t + (float)thespline->ctrl[0].x;
   retpoint->y = ay * tcb + by * tsq + cy * t + (float)thespline->ctrl[0].y;
}

/*------------------------------------------------------------------------*/
/* Find the closest distance between a point and a spline and return the  */
/* fractional distance along the spline of this point.			  */
/*------------------------------------------------------------------------*/

float findsplinemin(thespline, upoint)
  splineptr     thespline;
  XPoint  	*upoint;
{
   XfPoint 	*spt, flpt, newspt;
   float	minval = 1000000, tval, hval, ndist;
   short	j, ival;

   flpt.x = (float)(upoint->x);
   flpt.y = (float)(upoint->y);

   /* get estimate from precalculated spline points */

   for (spt = thespline->points; spt < thespline->points + INTSEGS;
	spt++) {
      ndist = fsqwirelen(spt, &flpt);
      if (ndist < minval) {
	 minval = ndist;
	 ival = (short)(spt - thespline->points);
      }
   }
   tval = (float)(ival + 1) / (INTSEGS + 1);
   hval = 0.5 / (INTSEGS + 1);

   /* short fixed iterative loop to converge on minimum t */

   for (j = 0; j < 5; j++) {
      tval += hval;
      ffindsplinepos(thespline, tval, &newspt);
      ndist = fsqwirelen(&newspt, &flpt);
      if (ndist < minval) minval = ndist;
      else {
         tval -= hval * 2;
         ffindsplinepos(thespline, tval, &newspt);
         ndist = fsqwirelen(&newspt, &flpt);
         if (ndist < minval) minval = ndist;
	 else tval += hval;
      }
      hval /= 2;
   }

   if (tval < 0.1) {
      if ((float)sqwirelen(&(thespline->ctrl[0]), upoint) < minval) tval = 0;
   }
   else if (tval > 0.9) {
      if ((float)sqwirelen(&(thespline->ctrl[3]), upoint) < minval) tval = 1;
   }
   return tval;
}

/*----------------------------------------------------------------------------*/
/* Find closest point of a polygon to the cursor			      */
/*----------------------------------------------------------------------------*/

short closepoint(curpoly, cursloc)
  polyptr curpoly;
  XPoint *cursloc;
{
   short curdist, mindist;
   XPoint *curpt, *savept; 

   curpt = savept = curpoly->points;
   mindist = wirelength(curpt, cursloc);
   while (++curpt < curpoly->points + curpoly->number) {
      curdist = wirelength(curpt, cursloc);
      if (curdist < mindist) {
         mindist = curdist;
         savept = curpt;
      }
   }
   return (short)(savept - curpoly->points);
}

/*----------------------------------------------------------------------------*/
/* Coordinate system transformations 					      */
/*----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------*/
/*  Check screen bounds:  minimum, maximum scale and translation is determined	*/
/*  by values which fit in an X11 type XPoint (short int).  If the window	*/
/*  extremes exceed type short when mapped to user space, or if the page 	*/
/*  bounds exceed type short when mapped to X11 window space, return error.	*/ 
/*------------------------------------------------------------------------------*/

short checkbounds()
{
   XPoint testpt;
   long lval;

   /* check window-to-user space */

   lval = 2 * (long)((float) (areastruct.width) / (*areastruct.vscale)) +
	(long)areastruct.lowerleft->x;
   if (lval != (long)((short)lval)) return -1;
   lval = 2 * (long)((float) (areastruct.height) / (*areastruct.vscale)) +
	(long)areastruct.lowerleft->y;
   if (lval != (long)((short)lval)) return -1;

   /* check user-to-window space */

   lval = (long)((float)(objectdata->lowerleft.x - areastruct.lowerleft->x) *
	(*areastruct.vscale));
   if (lval != (long)((short)lval)) return -1;
   lval = (long)areastruct.height - (long)((float)(objectdata->lowerleft.y -
	areastruct.lowerleft->y) * (*areastruct.vscale)); 
   if (lval != (long)((short)lval)) return -1;
   UTransformbyCTM(DCTM, &(objectdata->lowerleft), &testpt, 1);

   lval = (long)((float)(objectdata->lowerleft.x + objectdata->width -
	areastruct.lowerleft->x) * (*areastruct.vscale));
   if (lval != (long)((short)lval)) return -1;
   lval = (long)areastruct.height - (long)((float)(objectdata->lowerleft.y +
	objectdata->height - areastruct.lowerleft->y) * (*areastruct.vscale)); 
   if (lval != (long)((short)lval)) return -1;

   return 0;
}

/*------------------------------------------------------------------------*/
/* Transform X-window coordinate to xcircuit coordinate system		  */
/*------------------------------------------------------------------------*/

void window_to_user(xw, yw, upt)
  short 	xw, yw;
  XPoint	*upt;
{
  upt->x = (short)((float) xw / (*areastruct.vscale)) + areastruct.lowerleft->x;
  upt->y = (short)((float)(areastruct.height - yw) / (*areastruct.vscale)) + 
	areastruct.lowerleft->y;
}

/*------------------------------------------------------------------------*/
/* Transform xcircuit coordinate back to X-window coordinate system       */
/*------------------------------------------------------------------------*/

void user_to_window(upt, wpt)
  XPoint  upt;
  XPoint  *wpt;
{
  wpt->x = (short)((float)(upt.x - areastruct.lowerleft->x) * (*areastruct.vscale));
  wpt->y = areastruct.height - (short)((float)(upt.y - areastruct.lowerleft->y)
	* (*areastruct.vscale)); 
}

/*------------------------------------------------------------------------*/
/* Transformations in the object hierarchy				  */
/*------------------------------------------------------------------------*/

float UTopScale()
{
   Matrix *ctm = DCTM;
   return (float)(sqrt((double)(ctm->a * ctm->a + ctm->d * ctm->d)));
}

short UTopTransScale(length)
  float		length;
{
   /* 0.5 gives nearent int but unfortunately gives odd results. . .*/
  
   return (short)(length * UTopScale() + 0.45);
}

/*------------------------------------------------------------------------*/
/* Translate a point to the nearest snap-to grid point			  */
/*------------------------------------------------------------------------*/
/* user coordinates to user coordinates version 			  */

void u2u_snap(uvalue)
  XPoint	*uvalue;
{
   short tmpx, tmpy;

   if (areastruct.snapto) {
      tmpx = uvalue->x / areastruct.snapspace;
      if (abs(uvalue->x) % areastruct.snapspace > areastruct.snapspace / 2) 
	 tmpx += sign(uvalue->x);
      tmpy = uvalue->y / areastruct.snapspace;
      if (abs(uvalue->y) % areastruct.snapspace > areastruct.snapspace / 2) 
	 tmpy += sign(uvalue->y);
      uvalue->x = tmpx * areastruct.snapspace;
      uvalue->y = tmpy * areastruct.snapspace;
   }
}

/*------------------------------------------------------------------------*/
/* window coordinates to user coordinates version 			  */

void snap(valuex, valuey, returnpt)
  int 		valuex, valuey;
  XPoint 	*returnpt;
{
   window_to_user(valuex, valuey, returnpt);  
   u2u_snap(returnpt);
}

/*------------------------------------------------------------------------*/
/* Transform object coordinates through scale, translation, and rotation  */
/* This routine attempts to match the PostScript definition of trans-     */
/*    formation matrices.						  */
/*------------------------------------------------------------------------*/

/*------------------------------------------------------------------------*/
/* Current transformation matrix manipulation routines			  */
/*------------------------------------------------------------------------*/

void UResetCTM(ctm)
  Matrix *ctm;
{
   ctm->a = ctm->e = 1;
   ctm->b = ctm->d = 0;
   ctm->c = ctm->f = 0;  /* 0.5 for nearest-int real->int conversion? */
}

/*------------------------------------------------------------------------*/

void UCopyCTM(fctm, tctm)
   Matrix *fctm, *tctm;
{
   tctm->a = fctm->a;
   tctm->b = fctm->b;
   tctm->c = fctm->c;
   tctm->d = fctm->d;
   tctm->e = fctm->e;
   tctm->f = fctm->f;
}

/*-------------------------------------------------------------------------*/
/* Multiply CTM by current screen position and scale to get transformation */
/* matrix from a user point to the X11 window				   */
/*-------------------------------------------------------------------------*/

void UMakeWCTM(ctm)
  Matrix *ctm;
{
   ctm->a *= (*areastruct.vscale);
   ctm->b *= (*areastruct.vscale);
   ctm->c = (ctm->c - (float)areastruct.lowerleft->x) * (*areastruct.vscale);
   
   ctm->d *= -(*areastruct.vscale);
   ctm->e *= -(*areastruct.vscale);
   ctm->f = (float)areastruct.height + ((float)areastruct.lowerleft->y - ctm->f) *
	    (*areastruct.vscale);
}

/*------------------------------------------------------------------------*/

void UMultCTM(ctm, position, scale, rotate)
  Matrix *ctm;
  XPoint position;
  short  rotate; 
  float  scale;
{
   float tmpa, tmpb, tmpd, tmpe, ascale;
   float mata, matb, matc;

   if (rotate < 0) {  /* implies flipped object */
      rotate = -rotate;
      scale = -scale;
   }
   ascale = abs(scale);
   
   tmpa =  scale * (*cosine[rotate]);
   tmpb = ascale * (*sine[rotate]);
   tmpd = -scale * (*sine[rotate]);
   tmpe = ascale * (*cosine[rotate]);
   
   mata = ctm->a * tmpa + ctm->d * tmpd;
   matb = ctm->b * tmpa + ctm->e * tmpb;
   matc = ctm->c * tmpa + ctm->f * tmpb + position.x;

   ctm->d = ctm->d * tmpe + ctm->a * tmpd;
   ctm->e = ctm->e * tmpe + ctm->b * tmpd;
   ctm->f = ctm->f * tmpe + ctm->c * tmpd + position.y; 

   ctm->a = mata;
   ctm->b = matb;
   ctm->c = matc;
}

/*------------------------------------------------------------------------*/
/* Slanting function x' = x + beta * y, y' = y				  */
/*------------------------------------------------------------------------*/

void USlantCTM(ctm, beta)
  Matrix *ctm;
  float beta;
{
   ctm->b += ctm->a * beta;
   ctm->e += ctm->d * beta;
}

/*------------------------------------------------------------------------*/

void UPreMultCTM(ctm, position, scale, rotate)
  Matrix *ctm;
  XPoint position;
  short rotate;
  float scale;
{
   float tmpa, tmpb, tmpd, tmpe, ascale;
   float mata, matd;

   if (rotate < 0) {  /* implies flipped object */
      rotate = -rotate;
      scale = -scale;
   }
   ascale = abs(scale);
   
   tmpa =  scale * (*cosine[rotate]);
   tmpb = ascale * (*sine[rotate]);
   tmpd = -scale * (*sine[rotate]);
   tmpe = ascale * (*cosine[rotate]);
   
   ctm->c += ctm->a * position.x + ctm->b * position.y;
   ctm->f += ctm->d * position.x + ctm->e * position.y;

   mata = ctm->a * tmpa + ctm->b * tmpd;
   ctm->b = ctm->a * tmpb + ctm->b * tmpe;

   matd = ctm->d * tmpa + ctm->e * tmpd;
   ctm->e = ctm->d * tmpb + ctm->e * tmpe;

   ctm->a = mata;
   ctm->d = matd;
}

/*------------------------------------------------------------------------*/

void UTransformbyCTM(ctm, ipoints, points, number)
  Matrix *ctm;
  XPoint *ipoints, *points;
  short  number;
{
   pointlist current, ptptr = points;

   for (current = ipoints; current < ipoints + number; current++, ptptr++) {
      ptptr->x = (short)(ctm->a * (float)current->x + ctm->b * (float)current->y
		 + ctm->c);
      ptptr->y = (short)(ctm->d * (float)current->x + ctm->e * (float)current->y
		 + ctm->f);
   }
}
  
/*------------------------------------------------------------------------*/
/* (same as above routine but using type (float) for point values;  this  */
/* is for calculation of Bezier curve internal points.			  */
/*------------------------------------------------------------------------*/

void UfTransformbyCTM(ctm, fpoints, points, number)
  Matrix *ctm;
  XfPoint *fpoints;
  XPoint *points;
  short number;
{
   fpointlist current;
   pointlist new = points;
   short tmpx;

   for (current = fpoints; current < fpoints + number; current++, new++) {
      new->x = (short)(ctm->a * current->x + ctm->b * current->y + ctm->c);
      new->y = (short)(ctm->d * current->x + ctm->e * current->y + ctm->f);
   }
}

/*------------------------------------------------------------------------*/

void UPushCTM()
{
   areastruct.MatStackDepth++;
   areastruct.MatStack = (Matrix *)realloc(areastruct.MatStack,
		areastruct.MatStackDepth * sizeof(Matrix));
   UCopyCTM(DCTM2, DCTM);
}

/*------------------------------------------------------------------------*/
/* UPopCTM()   defined as macro in xcircuit.h				  */
/*------------------------------------------------------------------------*/

void UTransformPoints(points, newpoints, number, atpt, scale, rotate)
  XPoint       *points, *newpoints, atpt;
  short        number, rotate;
  float        scale;
{
   Matrix LCTM;
 
   UResetCTM(&LCTM);
   UMultCTM(&LCTM, atpt, scale, rotate);
   UTransformbyCTM(&LCTM, points, newpoints, number);
}

/*------------------------------------------------------------------------*/
/* Translate wire coords to force wire to horizontal or vertical position */
/*------------------------------------------------------------------------*/

void manhattanize(pospt, newwire)
  XPoint	*pospt;
  polyptr	newwire;
{
   short deltax, deltay;
   XPoint *tpoint = newwire->points + newwire->number - 2;

   deltax = abs(tpoint->x - pospt->x);
   deltay = abs(tpoint->y - pospt->y);
   if (deltay > deltax) pospt->x = tpoint->x;
   else pospt->y = tpoint->y;
}

/*-------------------------------------------------------------------------*/
/* Bounding box calculation routines					   */
/*-------------------------------------------------------------------------*/

void bboxcalc(testval, lowerval, upperval)
  short	testval, *lowerval, *upperval;
{
   if (testval < *lowerval) *lowerval = testval;
   if (testval > *upperval) *upperval = testval;
}

/*------------------------------------------------------------------------*/

calcextents(bboxgen, llx, lly, urx, ury)
  genericptr *bboxgen;
  short *llx, *lly, *urx, *ury;
{
   switch ((*bboxgen)->type) {
      case(POLYGON): {
         pointlist bboxpts;
         for (bboxpts = TOPOLY(bboxgen)->points; bboxpts < TOPOLY(bboxgen)->points
		 + TOPOLY(bboxgen)->number; bboxpts++) {
	    bboxcalc(bboxpts->x, llx, urx);
	    bboxcalc(bboxpts->y, lly, ury);
         }
         } break;

      case(SPLINE): {
         fpointlist bboxpts;
         bboxcalc(TOSPLINE(bboxgen)->ctrl[0].x, llx, urx);
         bboxcalc(TOSPLINE(bboxgen)->ctrl[0].y, lly, ury);
         bboxcalc(TOSPLINE(bboxgen)->ctrl[3].x, llx, urx);
         bboxcalc(TOSPLINE(bboxgen)->ctrl[3].y, lly, ury);
         for (bboxpts = TOSPLINE(bboxgen)->points; bboxpts < 
		 TOSPLINE(bboxgen)->points + INTSEGS; bboxpts++) {
	    bboxcalc((short)(bboxpts->x), llx, urx);
	    bboxcalc((short)(bboxpts->y), lly, ury);
         }
         } break;

      case (ARC): {
         fpointlist bboxpts;
         for (bboxpts = TOARC(bboxgen)->points; bboxpts < TOARC(bboxgen)->points +
	         TOARC(bboxgen)->number; bboxpts++) {
            bboxcalc((short)(bboxpts->x), llx, urx);
	    bboxcalc((short)(bboxpts->y), lly, ury);
         }
         } break;
   }
}

/*-------------------------------------------------------------------------*/

void calcbbox(localdata)
  objinstptr	localdata;
{
   /* set starting bounds as maximum bounds of screen */

   XPoint points[4], npoints[4];
   short i, j;
   genericptr *bboxgen;

   short llx, lly, urx, ury;

   llx = lly = 32767;
   urx = ury = -32768;

   /* now, for each screen element, compute the extents and revise bounding
      box points, if necessary. */

   for (bboxgen = localdata->thisobject->plist; bboxgen < localdata->
	thisobject->plist + localdata->thisobject->parts; bboxgen++) {
      switch((*bboxgen)->type) {

	 case(OBJECT):
            points[0].x = points[1].x = TOOBJINST(bboxgen)->thisobject->lowerleft.x;
            points[1].y = points[2].y = TOOBJINST(bboxgen)->thisobject->lowerleft.y + 
	       TOOBJINST(bboxgen)->thisobject->height;
            points[2].x = points[3].x = TOOBJINST(bboxgen)->thisobject->lowerleft.x + 
	       TOOBJINST(bboxgen)->thisobject->width;
            points[0].y = points[3].y = TOOBJINST(bboxgen)->thisobject->lowerleft.y;
            UTransformPoints(points, npoints, 4, TOOBJINST(bboxgen)->position,
		 TOOBJINST(bboxgen)->scale, TOOBJINST(bboxgen)->rotation); 
            for (j = 0; j < 4; j++) {
               bboxcalc(npoints[j].x, &llx, &urx);
               bboxcalc(npoints[j].y, &lly, &ury);
            }
	    break;

	 case(LABEL): {
            float tmpwidth = ULength(TOLABEL(bboxgen)->string, 0.0, 0);
            points[0].x = points[1].x = (TOLABEL(bboxgen)->justify & NOTLEFT ? 
	       (TOLABEL(bboxgen)->justify & RIGHT ? -tmpwidth : -tmpwidth / 2) : 0);
            points[0].y = points[3].y = (TOLABEL(bboxgen)->justify & NOTBOTTOM ?
	       (TOLABEL(bboxgen)->justify & TOP ? -TEXTHEIGHT : -HALFHEIGHT) : 0);
            points[1].y = points[2].y = points[0].y + TEXTHEIGHT;
            points[2].x = points[3].x = points[0].x + tmpwidth;

            UTransformPoints(points, npoints, 4, TOLABEL(bboxgen)->position,
		TOLABEL(bboxgen)->scale, TOLABEL(bboxgen)->rotation); 
            for (j = 0; j < 4; j++) {
               bboxcalc(npoints[j].x, &llx, &urx);
               bboxcalc(npoints[j].y, &lly, &ury);
            }
            } break;

	 case(PATH): {
	    genericptr *pathc;
	    for (pathc = TOPATH(bboxgen)->plist; pathc < TOPATH(bboxgen)->plist
		  + TOPATH(bboxgen)->parts; pathc++)
	       calcextents(pathc, &llx, &lly, &urx, &ury);
	    } break;

	 default:
	    calcextents(bboxgen, &llx, &lly, &urx, &ury);
      }
   }
   
   /* default action if there are no elements */

   if (localdata->thisobject->parts == 0) {
      llx = areastruct.lowerleft->x;
      lly = areastruct.lowerleft->y;
      urx = llx + areastruct.width / (*areastruct.vscale);
      ury = lly + areastruct.height / (*areastruct.vscale);
   }

   /* set the new bounding box */

   localdata->thisobject->lowerleft.x = llx;
   localdata->thisobject->lowerleft.y = lly;
   localdata->thisobject->width = urx - llx;
   localdata->thisobject->height = ury - lly;

   /* avoid width or height of zero---use of TEXTHEIGHT is arbitrary. . . */

   if (localdata->thisobject->width == 0) {
      localdata->thisobject->width = TEXTHEIGHT;
      localdata->thisobject->lowerleft.x -= HALFHEIGHT;
   }
   if (localdata->thisobject->height == 0) {
      localdata->thisobject->height = TEXTHEIGHT;
      localdata->thisobject->lowerleft.y -= HALFHEIGHT;
   }
}

/*--------------------------------------------------------*/
/* Modify areastruct on basis of bounding box calculation */
/*--------------------------------------------------------*/

void zoomview(w, mode, nulldata)
  Widget 	 w;
  unsigned short mode;
  caddr_t	 nulldata;
{
   calcbbox(areastruct.topobject);

   if (eventmode == NORMAL_MODE || eventmode == POPUP_MODE ||
	 eventmode == COPY2_MODE || eventmode == PUSH_MODE ||
	 eventmode == PRESS_MODE || eventmode == CATALOG_MODE) {

      if ((mode == 1) || (areastruct.vscale !=
	       &(areastruct.viewscale[areastruct.page]))) {
         float fitwidth = (float)areastruct.width / (float)(objectdata->width + 
	       2 * areastruct.gridspace);
         float fitheight = (float)areastruct.height / (float)(objectdata->height +
	       2 * areastruct.gridspace);

         if (fitwidth < fitheight) 
            (*areastruct.vscale) = min(MINAUTOSCALE, fitwidth);
         else 
            (*areastruct.vscale) = min(MINAUTOSCALE, fitheight);

         areastruct.lowerleft->x = objectdata->lowerleft.x - (areastruct.width
	       / (*areastruct.vscale) - objectdata->width) / 2;
         areastruct.lowerleft->y = objectdata->lowerleft.y - (areastruct.height
	       / (*areastruct.vscale) - objectdata->height) / 2;
      }
      UResetCTM(DCTM);
      UMakeWCTM(DCTM);

      drawarea(w, NULL, NULL);
      drawhbar(areastruct.scrollbarh, NULL, Number(1));
      drawvbar(areastruct.scrollbarv, NULL, Number(1));
   }
}

/*-------------------------------------------------------------------------*/
/* Basic X Graphics Routines in the User coordinate system		   */
/*-------------------------------------------------------------------------*/

void UDrawSimpleLine(localdata, pt1, pt2)
  objinstptr	localdata;
  XPoint	*pt1, *pt2;
{
   XPoint newpt1, newpt2;

   UTransformbyCTM(DCTM, pt1, &newpt1, 1);
   UTransformbyCTM(DCTM, pt2, &newpt2, 1);

   XDrawLine(dpy, win, areastruct.gc, newpt1.x, newpt1.y, newpt2.x, newpt2.y); 
} 

/*-------------------------------------------------------------------------*/

void UDrawLine(localdata, pt1, pt2)
  objinstptr	localdata;
  XPoint	*pt1, *pt2;
{
   short tmpwidth = UTopTransScale(areastruct.wirewidth[areastruct.page]);

   XSetLineAttributes(dpy, areastruct.gc, tmpwidth >= 2.0 ? (int)tmpwidth : 
	0, LineSolid, CapRound, JoinBevel);
   UDrawSimpleLine(localdata, pt1, pt2); 
} 

/*-------------------------------------------------------------------------*/
/* Draw line for editing text (position of cursor in string is given by    */
/*   areastruct.origin.(x,y)						   */
/*-------------------------------------------------------------------------*/

void UDrawTLine(curlabel)
   labelptr	curlabel;
{
   XPoint  upt, points[2]; /* top and bottom of text cursor line */
   short   xdist, tmplength;

   XSetFunction(dpy, areastruct.gc, GXxor);
   XSetForeground(dpy, areastruct.gc, AUXCOLOR ^ BACKGROUND);

   tmplength = ULength(curlabel->string, 0.0, 0);
   xdist = ULength(curlabel->string, 0.0, 1);

   points[0].x = (curlabel->justify & NOTLEFT ?
        (curlabel->justify & RIGHT ? -tmplength : -tmplength >> 1) : 0)
	+ xdist;
   points[0].y = (curlabel->justify & NOTBOTTOM ?
        (curlabel->justify & TOP ? -TEXTHEIGHT : -HALFHEIGHT) : 0) - 3;
   points[1].x = points[0].x;
   points[1].y = points[0].y + TEXTHEIGHT + 6;

   /* correct for position, rotation and scale of text */

   UPushCTM();
   UPreMultCTM(DCTM, curlabel->position, curlabel->scale, curlabel->rotation);
   UDrawLine(areastruct.topobject, &points[0], &points[1]);
   UPopCTM();

   /* Add "X" at string origin */

   user_to_window(curlabel->position, &upt);
   XSetLineAttributes(dpy, areastruct.gc, 0, LineSolid, CapButt, JoinMiter);
   XDrawLine(dpy, win, areastruct.gc, upt.x - 3, upt.y - 3, upt.x + 3, upt.y + 3);
   XDrawLine(dpy, win, areastruct.gc, upt.x + 3, upt.y - 3, upt.x - 3, upt.y + 3);

}

/*-------------------------------------------------------------------------*/
void UDrawXLine(opt, cpt)
  XPoint	opt, cpt;
{
   XPoint upt, vpt;

   XSetForeground(dpy, areastruct.gc, AUXCOLOR ^ BACKGROUND);
   XSetFunction(dpy, areastruct.gc, GXxor);

   user_to_window(cpt, &upt);
   user_to_window(opt, &vpt);

   XSetLineAttributes(dpy, areastruct.gc, 0, LineOnOffDash, CapButt, JoinMiter);
   XDrawLine(dpy, win, areastruct.gc, vpt.x, vpt.y, upt.x, upt.y);

   XSetLineAttributes(dpy, areastruct.gc, 0, LineSolid, CapButt, JoinMiter);
   XDrawLine(dpy, win, areastruct.gc, upt.x - 3, upt.y - 3, upt.x + 3, upt.y + 3);
   XDrawLine(dpy, win, areastruct.gc, upt.x + 3, upt.y - 3, upt.x - 3, upt.y + 3);

   XSetFunction(dpy, areastruct.gc, areastruct.gctype);
   XSetForeground(dpy, areastruct.gc, areastruct.gccolor);
}

/*-------------------------------------------------------------------------*/
void UDrawLines(localdata, points, number)
  objinstptr	localdata;
  XPoint	*points;
  short		number;
{
   XPoint newpoints[MAXSEGS];
   UTransformbyCTM(DCTM, points, newpoints, number);
   XDrawLines(dpy, win, areastruct.gc, newpoints, number, CoordModeOrigin);
}

/*-------------------------------------------------------------------------*/
void UDrawBox(origin, corner)
  XPoint	origin, corner;
{
   XPoint	worig, wcorn;

   user_to_window(origin, &worig);
   user_to_window(corner, &wcorn);

   XSetFunction(dpy, areastruct.gc, GXxor);
   XSetForeground(dpy, areastruct.gc, AUXCOLOR ^ BACKGROUND);
   XSetLineAttributes(dpy, areastruct.gc, 0, LineSolid, CapRound, JoinBevel);

   XDrawLine(dpy, win, areastruct.gc, worig.x, worig.y, worig.x, wcorn.y);
   XDrawLine(dpy, win, areastruct.gc, worig.x, wcorn.y, wcorn.x, wcorn.y);
   XDrawLine(dpy, win, areastruct.gc, wcorn.x, wcorn.y, wcorn.x, worig.y);
   XDrawLine(dpy, win, areastruct.gc, wcorn.x, worig.y, worig.x, worig.y);
}

/*-------------------------------------------------------------------------*/
/* Fill and/or draw a border around the stroking path			   */
/*-------------------------------------------------------------------------*/

void strokepath(pathlist, number, style, width)
  XPoint *pathlist;
  short  number;
  short  style;
  float  width;
{
   char         solidpart;
   char         dashstring[3];
   short        tmpwidth, i;

   tmpwidth = max(1, UTopTransScale(areastruct.wirewidth[areastruct.page] *
	width));

   if (style & FILLED || (!(style & FILLED) && style & OPAQUE)) {
      if ((style & FILLSOLID) == FILLSOLID)
         XSetFillStyle(dpy, areastruct.gc, FillSolid);
      else if (!(style & FILLED)) {
         XSetFillStyle(dpy, areastruct.gc, FillOpaqueStippled); 
	 XSetStipple(dpy, areastruct.gc, STIPPLE[7]);
      }
      else {
	 if (style & OPAQUE)
            XSetFillStyle(dpy, areastruct.gc, FillOpaqueStippled);
	 else
            XSetFillStyle(dpy, areastruct.gc, FillStippled);
         XSetStipple(dpy, areastruct.gc, STIPPLE[ (style &
                FILLSOLID) >> 5] );
      }
      XFillPolygon(dpy, win, areastruct.gc, pathlist, number, Nonconvex,
		CoordModeOrigin);
      /* return to original state */
      XSetFillStyle(dpy, areastruct.gc, FillSolid);
   }
   if (!(style & NOBORDER)) {
      /* set up dots or dashes */
      if (style & DASHED) solidpart = (char)(4 * tmpwidth);
      else if (style & DOTTED) solidpart = (char)tmpwidth;
      sprintf(dashstring, "%c%c", solidpart, (char)(4 * tmpwidth));
      if (style & (DASHED | DOTTED)) {
         XSetDashes(dpy, areastruct.gc, 0, dashstring, 2);
         XSetLineAttributes(dpy, areastruct.gc, tmpwidth >= 2.0 ?
            (int)tmpwidth : 0, LineOnOffDash, CapButt, JoinBevel);
      }
      else
         XSetLineAttributes(dpy, areastruct.gc, tmpwidth >= 2.0 ?
            (int)tmpwidth : 0, LineSolid, CapRound, JoinBevel);

      /* draw the spline and close off if so specified */
      XDrawLines(dpy, win, areastruct.gc, pathlist, number, CoordModeOrigin);
      if (!(style & UNCLOSED))
         XDrawLine(dpy, win, areastruct.gc, pathlist[0].x, pathlist[0].y,
		pathlist[number - 1].x, pathlist[number - 1].y);
   }
}

/*-------------------------------------------------------------------------*/

void makesplinepath(thespline, pathlist)
  splineptr      thespline;
  XPoint	*pathlist;
{
   XPoint *tmpptr = pathlist;

   UTransformbyCTM(DCTM, &(thespline->ctrl[0]), tmpptr, 1);
   UfTransformbyCTM(DCTM, thespline->points, ++tmpptr, INTSEGS);
   UTransformbyCTM(DCTM, &(thespline->ctrl[3]), tmpptr + INTSEGS, 1);
}

/*-------------------------------------------------------------------------*/

void UDrawSpline(localdata, thespline)
  objinstptr     localdata;
  splineptr      thespline;
{
   XPoint       tmppoints[SPLINESEGS];

   makesplinepath(thespline, tmppoints);
   strokepath(tmppoints, SPLINESEGS, thespline->style, thespline->width);
}

/*-------------------------------------------------------------------------*/

void UDrawEditSpline(localdata, thespline)
  objinstptr     localdata;
  splineptr      thespline;
{
   UDrawSpline(localdata, thespline);
   UDrawXLine(thespline->ctrl[0], thespline->ctrl[1]);  
   UDrawXLine(thespline->ctrl[3], thespline->ctrl[2]);
}

/*-------------------------------------------------------------------------*/

void UDrawPolygon(localdata, thepoly)
  objinstptr     localdata;
  polyptr	 thepoly;
{
   XPoint *tmppoints = (pointlist) malloc(thepoly->number * sizeof(XPoint));

   UTransformbyCTM(DCTM, thepoly->points, tmppoints, thepoly->number);
   strokepath(tmppoints, thepoly->number, thepoly->style, thepoly->width);
   free(tmppoints);
}

/*-------------------------------------------------------------------------*/

void UDrawArc(localdata, thearc)
  objinstptr	localdata;
  arcptr	thearc;
{
   XPoint  tmppoints[RSTEPS + 2];

   UfTransformbyCTM(DCTM, thearc->points, tmppoints, thearc->number);
   strokepath(tmppoints, thearc->number, thearc->style, thearc->width);
}

/*-------------------------------------------------------------------------*/

void UDrawPath(localdata, thepath)
  objinstptr	localdata;
  pathptr	thepath;
{
   XPoint	*tmppoints = (pointlist) malloc(sizeof(XPoint));
   genericptr	*genpath;
   polyptr	thepoly;
   splineptr	thespline;
   arcptr	thearc;
   int		pathsegs = 0, curseg = 0;
   
   for (genpath = thepath->plist; genpath < thepath->plist + thepath->parts;
	  genpath++) {
      switch((*genpath)->type) {
	 case POLYGON:
	    thepoly = TOPOLY(genpath);
	    pathsegs += thepoly->number;
	    tmppoints = (pointlist) realloc(tmppoints, pathsegs * sizeof(XPoint));
   	    UTransformbyCTM(DCTM, thepoly->points, tmppoints + curseg, thepoly->number);
	    curseg = pathsegs;
	    break;
	 case SPLINE:
	    thespline = TOSPLINE(genpath);
	    pathsegs += SPLINESEGS;
	    tmppoints = (pointlist) realloc(tmppoints, pathsegs * sizeof(XPoint));
   	    makesplinepath(thespline, tmppoints + curseg);
	    curseg = pathsegs;
	    break;
	 case ARC:
	    thearc = TOARC(genpath);
	    pathsegs += thearc->number;
	    tmppoints = (pointlist) realloc(tmppoints, pathsegs * sizeof(XPoint));
   	    UfTransformbyCTM(DCTM, thearc->points, tmppoints + curseg, thearc->number);
	    curseg = pathsegs;
	    break;
      }
   } 
   strokepath(tmppoints, pathsegs, thepath->style, thepath->width);
   free(tmppoints);
}

/*-------------------------------------------------------------------------*/

void UDrawObject(localdata, theobject, level, passcolor)
  objinstptr	localdata;
  objectptr	theobject;
  short		level;		/* level of recursion */
  int		passcolor;	/* inherited color value passed to object */
{
   genericptr	*areagen;
   short	*selectobj;
   short	tmpwidth;
   short	defaultcolor = passcolor;
   short	curcolor = passcolor;
   XPoint 	bboxin, bboxout[2];
   Boolean	xm, ym;

   /* add self to hierarchy list */

   areastruct.hlevels++;
   areastruct.hierarchy = (objinstptr *) realloc (areastruct.hierarchy,
      (areastruct.hlevels + 1) * sizeof(objinstptr));
   *(areastruct.hierarchy + areastruct.hlevels) = localdata;

   UPushCTM();
   UPreMultCTM(DCTM, localdata->position, localdata->scale,
	 localdata->rotation);

   /* do a quick test for intersection with the display window */

   UTransformbyCTM(DCTM, &(theobject->lowerleft), &(bboxout[0]), 1);
   bboxin.x = theobject->lowerleft.x + theobject->width;
   bboxin.y = theobject->lowerleft.y + theobject->height; 
   UTransformbyCTM(DCTM, &bboxin, &(bboxout[1]), 1);

   xm = (bboxout[0].x < bboxout[1].x) ? 0 : 1;  
   ym = (bboxout[0].y < bboxout[1].y) ? 0 : 1;  

   if (bboxout[xm].x < areastruct.width && bboxout[ym].y < areastruct.height &&
       bboxout[!xm].x > 0 && bboxout[!ym].y > 0) {       

     /* draw all of the elements */
   
     tmpwidth = UTopTransScale(areastruct.wirewidth[areastruct.page]);
     XSetLineAttributes(dpy, areastruct.gc, tmpwidth >= 2.0 ? (int)tmpwidth :
		0, LineSolid, CapRound, JoinBevel);

     for (areagen = theobject->plist; areagen < theobject->plist +
           theobject->parts; areagen++) {

       if (defaultcolor != DOFORALL)
	  if ((*areagen)->color != curcolor) {
	     if ((*areagen)->color == DEFAULTCOLOR) {
		curcolor = defaultcolor;
	        XSetForeground(dpy, areastruct.gc, defaultcolor);
	     }
	     else {
		curcolor = (*areagen)->color;
	        XSetForeground(dpy, areastruct.gc, curcolor);
	     }
	  }

       switch((*areagen)->type) {
	  case(POLYGON):
             UDrawPolygon(localdata, TOPOLY(areagen));
	     break;
   
	  case(SPLINE):
             UDrawSpline(localdata, TOSPLINE(areagen));
	     break;
   
	  case(ARC):
             UDrawArc(localdata, TOARC(areagen));
	     break;

	  case(PATH):
	     UDrawPath(localdata, TOPATH(areagen));
	     break;
   
          case(OBJECT):
             UDrawObject(TOOBJINST(areagen), TOOBJINST(areagen)->thisobject,
			level + 1, curcolor);
	     break;
   
  	  case(LABEL): 
             UDrawString(localdata, TOLABEL(areagen));
	     break;
       }
     }

     /* restore the color passed to the object, if different from current color */

     if ((defaultcolor != DOFORALL) && (passcolor != curcolor))
	XTopSetForeground(passcolor);
   }

   /* pop the hierarchy stack */

   areastruct.hlevels--;
   UPopCTM();
}

/*-------------------------------------------------------------------------*/
