/*-------------------------------------------------------------------------*/
/* xcircuit.c --- An X-windows program for drawing circuit diagrams	   */
/*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------*/
/*      Uses the Xw widget set 						   */
/*      written by Tim Edwards, 8/13/93    				   */
/*-------------------------------------------------------------------------*/

#include <stdio.h>
#include <strings.h>
#include <sys/types.h>
#ifdef linux
#include <dirent.h>
#include <linux/dirent.h>
#include <linux/unistd.h>
#define direct dirent
#else
#ifdef SVR4_DIRENT
#include <dirent.h>
#include <unistd.h>
#define direct dirent
#else
#include <stdlib.h>
#include <sys/dir.h>
#endif
#endif
#include <sys/stat.h>
#include <errno.h>
#include <values.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>

#include "Xw/Xw.h"
#include "Xw/Form.h"
#include "Xw/WorkSpace.h"
#include "Xw/PButton.h"
#include "Xw/SText.h"
#include "Xw/Cascade.h"
#include "Xw/PopupMgr.h"
#include "Xw/MenuBtn.h"
#include "Xw/BBoard.h"
#include "Xw/TextEdit.h"

/*-------------------------------------------------------------------------*/
/* Local includes							   */
/*-------------------------------------------------------------------------*/

#include "cursors.h"
#include "colordefs.h"
#include "xcircuit.h"
#include "menudefs.h"
#include "menudep.h"

/*-------------------------------------------------------------------------*/
/* Local defines							   */
/*-------------------------------------------------------------------------*/

#define ROWHEIGHT    (areastruct.textstruct->ascent + \
		      areastruct.textstruct->descent + 6)
#define LISTHEIGHT   200
#define STIPPLES     8

/*-------------------------------------------------------------------------*/
/* Global Variable definitions						   */
/*-------------------------------------------------------------------------*/

char	_STR2[250];  /* Specifically for text returned from the popup prompt */
char	_STR[150];          /* Generic multipurpose string */
Widget	message1, message2, message3, top;
Display *dpy;	            /* Works well to make this globally accessible */
Window  win;
short	popups;	            /* total number of popup widgets on the screen */
Pixmap  STIPPLE[STIPPLES];  /* Polygon fill-style stipple patterns */
Pixmap  flistpix = NULL;    /* For file-selection widget */
short   flstart, flfiles, flcurrent;
int	flcurwidth;
GC	hgc = NULL, sgc = NULL;
char    **filenames = NULL;
char	*cwdname = NULL;

static char STIPDATA[STIPPLES][4] = {
   "\000\004\000\001",
   "\000\005\000\012",
   "\001\012\005\010",
   "\005\012\005\012",
   "\016\005\012\007",
   "\017\012\017\005",
   "\017\012\017\016",
   "\000\000\000\000"
};

Cursor	appcursors[NUM_CURSORS];
ApplicationData appdata;
Clientdata areastruct;
object	areaobject;
int *appcolors;
int number_colors;
colorindex *colorlist;

XtIntervalId printtime_id;
short help_up;
short beeper;
float version;

/* if position of Symbol-Oblique changes, revise files.c #define */

short fontcount;
char **fonts;
char *prefonts[FONTS] = {
 "Times-Roman", "Times-Italic",      "Times-Bold",     "Times-BoldItalic",
 "Helvetica",   "Helvetica-Oblique", "Helvetica-Bold", "Helvetica-BoldOblique",
 "Courier",     "Courier-Oblique",   "Courier-Bold",   "Courier-BoldOblique",
 "Symbol",      "Symbol-Oblique"
};

extern short eventmode, textpos, pushes;
#ifdef DOUBLEBUFFER
extern Pixmap dbuf;
#endif

/* Bad hack for problems with the DECstation. . . don't know why */
#ifdef UniqueContextProblem
#undef XUniqueContext
XContext XUniqueContext()
{
   return XrmUniqueQuark();
}
#endif
/* End of bad hack. . . */

/*------------------------------------------------------------------------*/
/* Declarations of functions defined externally to xcircuit.c		  */
/*------------------------------------------------------------------------*/

extern void selectbutton(), releasebutton(), keyhandler(), resizearea(),
	loadlibrary(), Wprintf(), setfloat(), setscalex(), setscaley(),
	calcbbox(), setint(), setfilename(), setlabel(), undrawtext(),
	setpagelabel();
extern void redrawtext(), setpmode(), setorient(), getgeneric(), setdscale(),
	panhbar(), panvbar(), drawhbar(), drawvbar(), printhelp(),
	W2printf(), setfile(), initsplines(), setpagesize();
extern void startloadfile(), importfile(), loadblib(), loadulib(), loadlgf();
extern float getpsscale();
extern int xc_alloccolor();
void renamepage(), W1printf(), updatetext(), updatename();
void showlscroll(), listfiles(), newfilelist(), makecursors();

/*-------------------------------------------------------------------------*/
/* Initial Resource Management						   */
/*-------------------------------------------------------------------------*/

static XtResource resources[] = {

  /* Color scheme 1 */

  { XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, fg), XtRString, "Black"},
  { XtNbackground, XtCBackground, XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, bg), XtRString, "White"},
  { "gridcolor", "GridColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, gridpix), XtRString, "Gray95"},
  { "snapcolor", "SnapColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, snappix), XtRString, "Orange"},
  { "selectcolor", "SelectColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, selectpix), XtRString, "Gold3"},
  { "querycolor", "QueryColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, querypix), XtRString, "Turquoise"},
  { "axescolor", "AxesColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, axespix), XtRString, "Antique White"},
  { "offbuttoncolor", "OffButtonColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, buttonpix), XtRString, "Gray70"},
  { "auxiliarycolor", "AuxiliaryColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, auxpix), XtRString, "Green3"},
  { "barcolor", "BarColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, barpix), XtRString, "Tan"},

  /* Color scheme 2 */

  { "foreground2", XtCForeground, XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, fg2), XtRString, "White"},
  { "background2", XtCBackground, XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, bg2), XtRString, "DarkSlateGray"},
  { "gridcolor2", "GridColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, gridpix2), XtRString, "Gray40"},
  { "snapcolor2", "SnapColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, snappix2), XtRString, "Red"},
  { "selectcolor2", "SelectColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, selectpix2), XtRString, "Gold"},
  { "querycolor2", "QueryColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, querypix2), XtRString, "Turquoise"},
  { "axescolor2", "AxesColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, axespix2), XtRString, "NavajoWhite4"},
  { "offbuttoncolor2", "OffButtonColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, buttonpix2), XtRString, "Gray70"},
  { "auxiliarycolor2", "AuxiliaryColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, auxpix2), XtRString, "Green"},
  { "barcolor2", "BarColor", XtRPixel, sizeof(Pixel),
      XtOffset(ApplicationDataPtr, barpix2), XtRString, "Tan"},

  /* Other XDefaults-set properties */

  { XtNfont, XtCFont, XtRFontStruct, sizeof(XFontStruct *),
      XtOffset(ApplicationDataPtr, xcfont), XtRString, "Fixed"},
  { XtNwidth, XtCWidth, XtRInt, sizeof(int),
      XtOffset(ApplicationDataPtr, width), XtRString, "800"},
  { XtNheight, XtCHeight, XtRInt, sizeof(int),
      XtOffset(ApplicationDataPtr, height), XtRString, "600"}
};

/*-------------------------------------------------------------------------*/
/* Update the list of colors in the colormap				   */
/*-------------------------------------------------------------------------*/

addtocolorlist(button, cvalue)
  Widget button;
  int    cvalue;
{
   Colormap cmap = DefaultColormap (dpy, DefaultScreen(dpy));

   number_colors++;
   colorlist = (colorindex *)realloc(colorlist, number_colors *
		sizeof(colorindex));
   colorlist[number_colors - 1].cbutton = button;
   colorlist[number_colors - 1].color.pixel = cvalue; 

   /* Get and store the RGB values of the new color */
   XQueryColor(dpy, cmap, &(colorlist[number_colors - 1].color));
}

/*-------------------------------------------------------------------------*/
/* Add a new color button to the color menu				   */
/*-------------------------------------------------------------------------*/

void addnewcolorentry(ccolor)	/* called if new color button needs to be made */
   int ccolor;
{
   Widget colormenu, newbutton;
   Arg wargs[2];
   int i, n = 0;

   /* check to see if entry is already in the color list */

   for (i = 0; i < number_colors; i++)
      if (colorlist[i].color.pixel == ccolor) break;

   /* make new entry in the menu */

   if (i == number_colors) {
      colormenu = XtNameToWidget(top, ColorsPath);
      XtnSetArg(XtNlabelType, XwRECT);
      XtnSetArg(XtNrectColor, ccolor);

      newbutton = XtCreateWidget("NewColor", XwmenubuttonWidgetClass,
         colormenu, wargs, n);
      XtAddCallback (newbutton, XtNselect, setcolor, NULL);
      XtManageChild(newbutton);

      addtocolorlist(newbutton, ccolor);
   }
}

/*-------------------------------------------------------------------------*/
/* This recursive function looks down the button menu hierarchy and        */
/*   creates the necessary buttons and submenus.			   */
/*   Menu entries are marked if the corresponding "size" entry in the	   */
/*   menu structure is > 0.						   */
/*-------------------------------------------------------------------------*/

void makesubmenu(menuname, attachname, buttonmenu, arraysize, manager)
char *menuname, *attachname;
menuptr buttonmenu;
int arraysize;
Widget manager;
{
   short i, n;
   int cval;
   WidgetList newbuttons;
   Widget popupshell, cascade;
   Arg	wargs[6];
   menuptr p;
   char popupname[30];

   newbuttons = (WidgetList) XtMalloc (arraysize * sizeof (Widget));
   sprintf(popupname, "popup%s", menuname);
   popupshell = XtCreatePopupShell (popupname, transientShellWidgetClass,
	manager,  NULL, 0);

   XtSetArg (wargs[0], XtNattachTo, attachname);
   cascade = XtCreateManagedWidget (menuname, XwcascadeWidgetClass,
	popupshell, wargs, 1);
   
   for (p = buttonmenu, i = 0; p < buttonmenu + arraysize; p++, i++) {
      n = 0;
      if (p->size > 0 && p->submenu == NULL) { /* This denotes a marked entry */
	 XtnSetArg(XtNsetMark, True);
      }
      XtnSetArg(XtNfont, areastruct.textstruct);
      if (p->name[0] == '_') {  /* Color button */
	 cval = xc_alloccolor(p->name + 1);
	 XtnSetArg(XtNlabelType, XwRECT);
	 XtnSetArg(XtNrectColor, cval);
      }
      else if (p->name[0] == ':') {  /* Stipple button */
	 XtnSetArg(XtNlabelType, XwRECT);
         if (((int)(p->passeddata) == (FILLED | FILLSOLID))) {
	    XtnSetArg(XtNrectColor, BlackPixel(dpy,DefaultScreen(dpy)));
	 }
	 else {
	    XtnSetArg(XtNrectStipple, STIPPLE[((int)(p->passeddata) &
		  FILLSOLID) >> 5]);
	 }
      }
	 
      newbuttons[i] = XtCreateWidget(p->name, XwmenubuttonWidgetClass,
	   cascade, wargs, n);
      if (p->submenu != NULL)
	 makesubmenu(p->name, p->name, p->submenu, p->size, manager);
      else 
	 XtAddCallback (newbuttons[i], XtNselect, p->func, p->passeddata);

      /* For color buttons, maintain a list of Widgets and color values */

      if (p->name[0] == '_') addtocolorlist(newbuttons[i], cval);
   }
   XtManageChildren (newbuttons, arraysize);
}

/*-------------------------------------------------------------------------*/
/* Hierarchical Menu Creator						   */
/*   This function creates the top level of buttons which are arranged     */
/*   across the top starting at the left edge.  For each button 	   */
/*   that has a submenu, a Popup manager is created, and then menu	   */
/*   panes are attached to the manager in a hierarchical fashion.	   */
/*   Note: Returns widget for last button on top level			   */
/*-------------------------------------------------------------------------*/

Widget createmenus (topbuttons, maxbuttons, form)
menustruct topbuttons[];
short maxbuttons;
Widget form;
{
   int i, maxmgrs = 0, n = 0, j = 0;
   WidgetList buttonw, mgr_shell, menu_mgr;
   Arg	wargs[5];

   buttonw = (WidgetList) XtMalloc(maxbuttons * sizeof(Widget));

   for (i = 0; i < maxbuttons; i++) 
      if (topbuttons[i].submenu != NULL) maxmgrs++;

   mgr_shell = (WidgetList) XtMalloc(maxmgrs * sizeof(Widget));
   menu_mgr = (WidgetList) XtMalloc(maxmgrs * sizeof(Widget));

   for (i = 0; i < maxbuttons; i++) {
      XtnSetArg(XtNheight, ROWHEIGHT);
      XtnSetArg(XtNlabel, topbuttons[i].name);
      XtnSetArg(XtNfont, areastruct.textstruct);
      if (i > 0) {
	 XtnSetArg(XtNxRefWidget, buttonw[i - 1]);
	 XtnSetArg(XtNxAddWidth, True);
      }
      buttonw[i] = XtCreateManagedWidget(topbuttons[i].name, 
	 XwmenuButtonWidgetClass, form, wargs, n); n = 0;

      if(topbuttons[i].submenu == NULL) 
	 XtAddCallback(buttonw[i], XtNselect, topbuttons[i].func, NULL);
      else {
         mgr_shell[j] = XtCreatePopupShell("mgr_shell", shellWidgetClass,
	    buttonw[i], NULL, 0);
         menu_mgr[j] = XtCreateManagedWidget("menu_mgr", XwpopupmgrWidgetClass,
	    mgr_shell[j], NULL, 0);
	 makesubmenu(topbuttons[i].name, "menu_mgr", topbuttons[i].submenu, 
	    topbuttons[i].size, menu_mgr[j]);
	 j++;
      }
   }
   return buttonw[i - 1];
}

/*---------------*/
/* Quit xcircuit */
/*---------------*/

void quit()
{
   /* free up CTM Matrix Stack */
   free(areastruct.MatStack);

   exit(0);
}

/*--------------*/
/* Null routine */
/*--------------*/

void DoNothing()
{
   /* Nothing here! */
}

/*-------------------------------------------------------------------------*/
/* Popup dialog box routines						   */
/*-------------------------------------------------------------------------*/
/* Propogate any key event from the dialog box into the textedit widget    */
/*-------------------------------------------------------------------------*/

void propevent(w, editwidget, event)
  Widget w, editwidget;
  XEvent *event;
{
   Window ewin = XtWindow(editwidget);

   event->xany.window = ewin;
   XSendEvent(dpy, ewin, False, KeyPressMask, event);
}

/*-------------------------------------------------------------------------*/
/* Destroy an interactive text-editing popup box 			   */
/*-------------------------------------------------------------------------*/

void destroypopup(button, callstruct, calldata)
  Widget button;
  popupstruct *callstruct;
  caddr_t calldata;
{
   Arg	wargs[1];

   if (eventmode == NORMAL_MODE) eventmode = POPUP_MODE;

   if(XtNameToWidget(callstruct->popup, "help2") != NULL) help_up = False;

   if (callstruct->buttonptr->button != NULL) {  

      /* return the button to its normal state */

      XtSetArg(wargs[0], XtNforeground, callstruct->buttonptr->foreground);
      XtSetValues(callstruct->buttonptr->button, wargs, 1);
   
      XtAddCallback(callstruct->buttonptr->button, XtNselect, 
	  callstruct->buttonptr->buttoncall, callstruct->buttonptr->dataptr);
   }

   XtDestroyWidget(callstruct->popup);
   popups--;

   /* free the allocated structure space */

   free(callstruct->buttonptr);
   free(callstruct);
}

/*-------------------------------------------------------------------------*/
/* Pull text from the popup prompt buffer into a global string variable    */
/*-------------------------------------------------------------------------*/

void gettext(button, callstruct, calldata)
  Widget 	button;
  popupstruct	*callstruct;
  caddr_t 	calldata;
{
   void *function = callstruct->setvalue;

   sprintf(_STR2, "%.249s", XwTextCopyBuffer(callstruct->textw)); 

   /* functions which use the file selector should look for directory name */
   /* in string rather than a file name, and take appropriate action.	   */

   if (callstruct->fileprompt == True) {
      int slen = strlen(_STR2);
      if (_STR2[slen - 1] == '/' || (opendir(_STR2) != NULL)) {
	 short n;
         if (_STR2[slen - 1] != '/') strcat(_STR2, "/");
	 cwdname = (char *)realloc(cwdname, (slen + 2) * sizeof(char));
	 strcpy(cwdname, _STR2);
	 newfilelist(XtNameToWidget(XtParent(button), "Filelist"),
	    callstruct->textw);
	 return;
      }
   }

   /* call the function which sets the variable according to type */
   /* This is in format (function)(calling-widget, ptr-to-data)   */

   (*(callstruct->setvalue))(callstruct->buttonptr->button,
	 callstruct->buttonptr->dataptr);

   if (callstruct->fileprompt == True)
      newfilelist(XtNameToWidget(XtParent(button), "Filelist"), callstruct->textw);
   destroypopup(button, callstruct, calldata);
}

/*-------------------------------------------------------------------------*/
/* Grab text from the "output properties" window			   */
/*-------------------------------------------------------------------------*/

void getproptext(button, callstruct, calldata)
  Widget 	button;
  propstruct	*callstruct;
  caddr_t 	calldata;
{
   /* areastruct.filename[areastruct.page] can be realloc'd by the call */
   /* to *(callstruct->setvalue), so callstruct->dataptr may no longer  */
   /* be pointing to the data.						*/

   short fileyes = (callstruct->setvalue == setfilename);

   sprintf(_STR2, XwTextCopyBuffer(callstruct->textw)); 
   (*(callstruct->setvalue))(button, callstruct->dataptr);

   /* special stuff for filename changes */

   if (fileyes) {
      char blabel[50];
      Widget wrbutton;
      struct stat statbuf;
      Arg wargs[1];

      /* get updated file information */

      if (strstr(areastruct.filename[areastruct.page], ".") == NULL)
         sprintf(blabel, "%s.ps", areastruct.filename[areastruct.page]);
      else sprintf(blabel, "%s", areastruct.filename[areastruct.page]);
      if (stat(blabel, &statbuf) == 0) {
         sprintf(blabel, " Overwrite File ");
         if (beeper) XBell(dpy, 100);
         Wprintf("    Warning:  File exists");
      }
      else {
         sprintf(blabel, " Write File ");
         if (errno == ENOTDIR)
            Wprintf("Error:  Incorrect pathname");
         else if (errno == EACCES)
            Wprintf("Error:  Path not readable");
	 Wprintf("  ");
      }

      wrbutton = XtNameToWidget(XtParent(button), "Write File");
      XtSetArg(wargs[0], XtNlabel, blabel);
      XtSetValues(wrbutton, wargs, 1);
   }

   /* objectdata->name is not malloc'd, so is not changed by call to */
   /* *(callstruct->setvalue).					   */

   else if (callstruct->dataptr == objectdata->name) {
      printname(objectdata);
      renamepage(areastruct.page);
   }
}

/*-------------------------------------------------------------------------*/
/* Update scale, width, and height in response to change of one of them	   */
/*-------------------------------------------------------------------------*/

void updatetext(button, callstruct, calldata)
  Widget	button;
  WidgetList	callstruct;
  caddr_t	calldata;
{
   float oscale, psscale;
   char edit[3][50];
   short i, n, posit;
   char  *pdptr;
   Arg	 wargs[2];

   oscale = areastruct.outscale[areastruct.page];
   psscale = getpsscale(oscale, areastruct.page);

   sprintf(edit[0], "%6.5f", oscale);
   sprintf(edit[1], "%6.5f", (objectdata->width * psscale) / 72);
   sprintf(edit[2], "%6.5f", (objectdata->height * psscale) / 72);

   for (i = 0; i < 3; i++) {
      n = 0;
      XtnSetArg(XtNstring, edit[i]);
      pdptr = strchr(edit[i], '.');
      posit = (pdptr != NULL) ? (short)(pdptr - edit[i]) : strlen(edit[i]);
      XtnSetArg(XtNinsertPosition, posit);
      XtSetValues(callstruct[i + 2], wargs, n);
   }
}

/*-------------------------------------------------------------------------*/
/* Update the object name in response to a change in filename		   */
/*-------------------------------------------------------------------------*/

void updatename(button, callstruct, calldata)
  Widget        button;
  WidgetList    callstruct;
  caddr_t       calldata;
{
   short i, n, posit;
   char  *rootptr;
   Arg   wargs[2]; 
      
   if (strstr(objectdata->name, "Page ") != NULL || strstr(objectdata->name,
	"Page_") != NULL || objectdata->name[0] == '\0') {

      rootptr = strrchr(areastruct.filename[areastruct.page], '/');
      if (rootptr == NULL) rootptr = areastruct.filename[areastruct.page];
      else rootptr++;

      sprintf(objectdata->name, "%.79s", rootptr);
  
      n = 0;
      posit = strlen(objectdata->name);
      XtnSetArg(XtNstring, objectdata->name);
      XtnSetArg(XtNinsertPosition, posit);
      XtSetValues(callstruct[1], wargs, n);
      printname(objectdata);
      renamepage(areastruct.page);
   }
}

/*-------------------------------------------------------------------------*/
/* Compare two filenames (for use by qsort())				   */
/*-------------------------------------------------------------------------*/

int fcompare(a, b)
  char **a, **b;
{
   return(strcmp(*a, *b));
}

/*-------------------------------------------------------------------------*/
/* Routines for drawing a box around the currently selected file	   */
/*-------------------------------------------------------------------------*/

void dragfilebox(w, calldata, event)
  Widget w;
  caddr_t calldata;
  XMotionEvent *event;
{
   short fileno;
   int charheight = ROWHEIGHT - 6;
   int twidth;
   Window lwin = XtWindow(w);

   fileno = (event->y + (charheight / 2) - 10) / charheight + flstart;
   if (fileno < 0) fileno = 0;
   else if (fileno >= flfiles) fileno = flfiles - 1;
  
   if (fileno == flcurrent) return;

   if (flcurrent >= 0) 		/* erase previous box */
      XDrawRectangle(dpy, lwin, areastruct.gc, 5, -4 + charheight * (flcurrent
	    - flstart), flcurwidth + 10, charheight);

   twidth = XTextWidth(areastruct.textstruct, filenames[fileno],
	    strlen(filenames[fileno]));
   XDrawRectangle(dpy, lwin, areastruct.gc, 5, -4 + charheight * (fileno
	    - flstart), twidth + 10, charheight);

   flcurrent = fileno;
   flcurwidth = twidth;
}

/*-------------------------------------------------------------------------*/

void startfiletrack(w, calldata, event)
  Widget w;
  caddr_t calldata;
  XCrossingEvent *event;
{
   XtAddEventHandler(w, PointerMotionMask, False, dragfilebox, NULL);

   XSetFunction(dpy, areastruct.gc, GXcopy);
   XSetForeground(dpy, areastruct.gc, AUXCOLOR);

   /* draw initial box */

   flcurrent = -1;
   dragfilebox(w, calldata, (XMotionEvent *)event);

   XSetFunction(dpy, areastruct.gc, GXxor);
   XSetForeground(dpy, areastruct.gc, AUXCOLOR ^ BACKGROUND);
}

/*-------------------------------------------------------------------------*/

void endfiletrack(w, callstruct, event)
  Widget w;
  caddr_t callstruct;
  XCrossingEvent *event;
{
   Window lwin = XtWindow(w);
   int charheight = ROWHEIGHT - 6;

   XDrawRectangle(dpy, lwin, areastruct.gc, 5, -4 + charheight * (flcurrent
	    - flstart), flcurwidth + 10, charheight);

   XtRemoveEventHandler(w, Button1MotionMask | Button2MotionMask,
	False, dragfilebox, NULL);

   /* Restore graphics state values */
   XSetForeground(dpy, areastruct.gc, areastruct.gccolor);
   XSetFunction(dpy, areastruct.gc, areastruct.gctype);
}

/*-------------------------------------------------------------------------*/
/* Make a list of the files in the list widget window			   */
/*-------------------------------------------------------------------------*/

#define INITDIRS 10

void listfiles(w, callstruct, calldata)
  Widget w;
  caddr_t callstruct;
  caddr_t calldata;
{
   XGCValues	values;
   Arg wargs[2];
   DIR *cwd;
   Window lwin = XtWindow(w);
   short allocd = INITDIRS;
   short n = 0;
   struct direct *dp;
   struct stat statbuf;
   int pixheight;
   int charheight = ROWHEIGHT - 6;
   Dimension textwidth, textheight;
   XFontStruct *tmpstruct;

   if (sgc == NULL) {
      tmpstruct = XLoadQueryFont(dpy, "-*-helvetica-medium-r-normal--14-*");
      values.foreground = FOREGROUND;
      values.font = tmpstruct->fid;
      values.function = GXcopy;
      sgc = XCreateGC(dpy, lwin, GCForeground | GCFont | GCFunction, &values);
   }

   XtnSetArg(XtNwidth, &textwidth);
   XtnSetArg(XtNheight, &textheight);
   XtGetValues(w, wargs, n);

   if (!flistpix) {

      /* get list of files */

      if (filenames == NULL) 
         filenames = (char **) malloc (INITDIRS * sizeof(char *));
      flfiles = 0;
      if (cwdname == NULL) {
	 cwdname = (char *) malloc (sizeof(char));
	 cwdname[0] = '\0';
      }
      if (cwdname[0] == '\0')
         cwd = opendir(".");
      else
         cwd = opendir(cwdname);

      if (cwd == NULL) {
         XSetForeground(dpy, sgc, BACKGROUND);
         XFillRectangle(dpy, lwin, sgc, 0, 0, textwidth, textheight);
         XSetForeground(dpy, sgc, AUXCOLOR);
         XDrawString(dpy, lwin, sgc, 10, textheight / 2,
	    "(Invalid Directory)", 19);
	 return;
      }
      else {
         while ((dp = readdir(cwd)) != NULL) {
	    /* don't put current directory in list */
	    if (!strcmp(dp->d_name, ".")) continue;
	    filenames[flfiles] = (char *) malloc ((strlen(dp->d_name) + 2) *
		 sizeof(char));
	    strcpy(filenames[flfiles++], dp->d_name);
	    if (flfiles == allocd) {
	       allocd += INITDIRS;
	       filenames = (char **) realloc (filenames, allocd * sizeof(char *));
	    }
         }
      }
      closedir(cwd);
      qsort((void *)filenames, (size_t)flfiles, sizeof(char *), fcompare);

      pixheight = flfiles * charheight + 20;

      flistpix = XCreatePixmap(dpy, win, textwidth, pixheight,
	   DefaultDepthOfScreen(XtScreen(w)));

      XSetForeground(dpy, sgc, BACKGROUND);
      XFillRectangle(dpy, flistpix, sgc, 0, 0, textwidth, pixheight);
      XSetForeground(dpy, sgc, FOREGROUND);
      for (n = 0; n < flfiles; n++) {
	 sprintf(_STR2, "%s%s", cwdname, filenames[n]); 
	 stat(_STR2, &statbuf);
	 if ((statbuf.st_mode & S_IFDIR) != 0) {  /* is a directory */
	    XSetForeground(dpy, sgc, SELECTCOLOR);
	    strcat(filenames[n], "/");
	 }
         XDrawString(dpy, flistpix, sgc, 10, 10 + n * charheight,
	    filenames[n], strlen(filenames[n]));
	 if ((statbuf.st_mode & S_IFDIR) != 0)
	    XSetForeground(dpy, sgc, FOREGROUND);
      }
   }
   XSetForeground(dpy, sgc, BACKGROUND);
   XFillRectangle(dpy, lwin, sgc, 0, 0, textwidth, textheight);
   XCopyArea(dpy, flistpix, lwin, sgc, 0, flstart * charheight,
	textwidth, textheight, 0, 0);
}

void newfilelist(w, textw)
   Widget w, textw;
{
   short n;

   for (n = 0; n < flfiles; n++)
      free(filenames[n]);
   free(filenames);
   if (flistpix != NULL) XFreePixmap(dpy, flistpix);
   filenames = flistpix = NULL;
   flstart = 0;
   listfiles(w, NULL, NULL);
   showlscroll(XtNameToWidget(XtParent(w), "LScroll"), NULL, NULL);
   XwTextClearBuffer(textw);
   XwTextInsert(textw, cwdname);
#ifdef HAVE_XWTEXTRESIZE
   XwTextResize(textw);
#endif
}

void fileselect(w, textw, event)
  Widget w;
  Widget textw;
  XButtonEvent *event;
{
   Arg wargs[2];
   Window lwin = XtWindow(w);
   int pixheight;
   Dimension textwidth, textheight;
   int charheight = ROWHEIGHT - 6;
   short fileno, n = 0;
   DIR *cwd;
   char *tbuf;

   flcurrent = -1;

   XtnSetArg(XtNwidth, &textwidth);
   XtnSetArg(XtNheight, &textheight);
   XtGetValues(w, wargs, n);

   /* third mouse button cancels selection and reverts buffer to cwd name */

   if (event->button == Button3){
      newfilelist(w, textw);
      return;
   }

   fileno = (event->y + (charheight / 2) - 10) / charheight + flstart;
   if (fileno < 0) fileno = 0;
   else if (fileno >= flfiles) fileno = flfiles - 1;
   
   /* check if this file is a directory or not */

   if (strchr(filenames[fileno], '/') == NULL)	{

      /* highlight the entry. . . */

      XSetForeground(dpy, sgc, AUXCOLOR);
      XDrawString(dpy, flistpix, sgc, 10, 10 + fileno * charheight,
   	   filenames[fileno], strlen(filenames[fileno]));
      XCopyArea(dpy, flistpix, lwin, sgc, 0, flstart * charheight,
	   textwidth, textheight, 0, 0);

      /* . . .and append it to the text field */

      tbuf = (char *)malloc((XwTextGetLastPos(textw)
	     + strlen(filenames[fileno]) + 5) * sizeof(char));
      sprintf(tbuf, "%s", XwTextCopyBuffer(textw)); 
      if (tbuf[0] != '\0') {
         if (tbuf[strlen(tbuf) - 1] != '/') strcat(tbuf, ",");
      }
      else {
	 if (cwdname != NULL) {
	    if (cwdname[0] != '\0') {
	       tbuf = (char *)realloc(tbuf, (strlen(cwdname) +
			strlen(filenames[fileno]) + 5) * sizeof(char));
	       strcpy(tbuf, cwdname);
	    }
 	 }
      }
      strcat(tbuf, filenames[fileno]);
      XwTextClearBuffer(textw);
      XwTextInsert(textw, tbuf);
#ifdef HAVE_XWTEXTRESIZE
      XwTextResize(textw);
#endif
      free(tbuf);
   }
   else {  /* move to new directory */

      if (!strcmp(filenames[fileno], "../")) {
         char *cptr, *sptr = cwdname;
	 if (!strcmp(cwdname, "/")) return;	/* no way up from root dir. */
	 while(strstr(sptr, "../") != NULL) sptr += 3;
	 if ((cptr = strrchr(sptr, '/')) != NULL) {
	    *cptr = '\0';
	    if ((cptr = strrchr(sptr, '/')) != NULL) *(cptr + 1) = '\0';
	    else *sptr = '\0';
         }
	 else {
      	    cwdname = (char *)realloc(cwdname, (strlen(cwdname) +
	        4) * sizeof(char));
            strcat(cwdname, "../");
	 }
      }
      else {
	 cwdname = (char *)realloc(cwdname, (strlen(cwdname) +
	        strlen(filenames[fileno]) + 1) * sizeof(char));
         strcat(cwdname, filenames[fileno]);
      }
      newfilelist(w, textw);
   }
}

void showlscroll(w, callstruct, calldata)
  Widget w;
  caddr_t callstruct;
  caddr_t calldata;
{
   Arg wargs[2];
   Window swin = XtWindow(w);
   Dimension swidth, sheight;
   int pstart, pheight, finscr;
   short n = 0;


   XtnSetArg(XtNwidth, &swidth);
   XtnSetArg(XtNheight, &sheight);
   XtGetValues(w, wargs, n);

   XClearWindow(dpy, swin);

   if (flfiles > 0) {	/* no files, no bar */

      finscr = LISTHEIGHT / ROWHEIGHT;
      if (finscr > flfiles) finscr = flfiles;

      pstart = flstart * (sheight / flfiles);
      pheight = finscr * (sheight / flfiles);


      XSetForeground(dpy, sgc, BARCOLOR);
      XFillRectangle(dpy, swin, sgc, 0, pstart, swidth, pheight);
   }
   flcurrent = -1;
}

void draglscroll(w, filew, event)
  Widget w;
  Widget filew;
  XButtonEvent *event;
{
   Arg wargs[2];
   Window swin = XtWindow(w);
   Dimension swidth, sheight;
   int phheight, finscr, flsave = flstart;
   short n = 0;

   XtnSetArg(XtNwidth, &swidth);
   XtnSetArg(XtNheight, &sheight);
   XtGetValues(w, wargs, n);

   finscr = LISTHEIGHT / ROWHEIGHT;
   if (finscr > flfiles) finscr = flfiles;

   /* center scrollbar on pointer vertical position */   

   phheight = finscr * (sheight / flfiles) / 2;
   flstart = (event->y > phheight) ? (event->y - phheight) / (sheight / flfiles) : 0;
   if (flstart >= flfiles) flstart = flfiles - 1;

   if (flstart != flsave) {
      showlscroll(w, NULL, NULL);
      listfiles(filew, NULL, NULL); 
   }
}

/*-------------------------------------------------------------------------*/
/* Create a popup window with "OK" and "Cancel" buttons,		   */
/* and text and label fields.						   */
/*-------------------------------------------------------------------------*/

void popupprompt(button, request, current, function, datastruct, fileprompt)
Widget  button;
char *request, *current;
void (*function)();
buttonsave *datastruct;
Boolean fileprompt;
{
    Arg         wargs[8];
    Widget      popup, dialog, okbutton, cancelbutton, entertext, tmpwidget;
    Widget	staticarea;
    XWMHints	*wmhints;	/* for proper input focus */
    Position    xpos, ypos;
    short	n = 0;
    Dimension	height, width, areawidth, areaheight, bwidth, owidth;
    static char defaultTranslations[] = "<Key>Return:	execute()";
    XGCValues	promptvalues;
    popupstruct	*okaystruct;

    height = (current == NULL) ? ROWHEIGHT * 4 : ROWHEIGHT * 5;
    if (fileprompt) height += LISTHEIGHT;

    width = XTextWidth(areastruct.textstruct, request, strlen(request)) + 20;
    bwidth = XTextWidth(areastruct.textstruct, "Cancel", strlen("Cancel")) + 50;
    owidth = XTextWidth(areastruct.textstruct, "Okay", strlen("Okay")) + 50;
    if (width < 400) width = 400;

    XtnSetArg(XtNwidth, &areawidth);
    XtnSetArg(XtNheight, &areaheight);
    XtGetValues(areastruct.area, wargs, n); n = 0;
    XtTranslateCoords(areastruct.area, (Position) (areawidth / 2 - width 
	/ 2 + popups * 20), (Position) (areaheight / 2 - height / 2 +
	popups * 20), &xpos, &ypos);
    XtnSetArg(XtNx, xpos);
    XtnSetArg(XtNy, ypos);
    popup = XtCreatePopupShell("prompt", transientShellWidgetClass,
        button, wargs, n); n = 0;
    popups++;

    XtnSetArg(XtNlayout, XwIGNORE);
    XtnSetArg(XtNwidth, width);
    XtnSetArg(XtNheight, height);
    dialog = XtCreateManagedWidget("dialog", XwbulletinWidgetClass,
        popup, wargs, n); n = 0;

    XtnSetArg(XtNx, 20);
    XtnSetArg(XtNy, ROWHEIGHT - 10 + (fileprompt ? LISTHEIGHT : 0));
    XtnSetArg(XtNstring, request);
    XtnSetArg(XtNborderWidth, 0);
    XtnSetArg(XtNgravity, WestGravity);
    staticarea = XtCreateManagedWidget("static", XwstaticTextWidgetClass,
	dialog, wargs, n); n = 0;

    XtnSetArg(XtNx, 20);
    XtnSetArg(XtNy, height - ROWHEIGHT - 10);
    XtnSetArg(XtNwidth, owidth); 
    XtnSetArg(XtNfont, areastruct.textstruct);
    okbutton = XtCreateManagedWidget("Okay", XwmenuButtonWidgetClass, 
	dialog, wargs, n); n = 0;

    okaystruct = (popupstruct *) malloc(sizeof(popupstruct));
    okaystruct->buttonptr = datastruct;
    okaystruct->popup = popup;
    okaystruct->fileprompt = fileprompt;

    if (current != NULL) {   /* A Text Edit widget is required */
       char		*pdptr;
       short		posit;

       XtnSetArg(XtNx, width - bwidth - 20);
       XtnSetArg(XtNy, height - ROWHEIGHT - 10);
       XtnSetArg(XtNwidth, bwidth);
       XtnSetArg(XtNfont, areastruct.textstruct);
       cancelbutton = XtCreateManagedWidget("Cancel", XwmenuButtonWidgetClass, 
	   dialog, wargs, n); n = 0; 

       XtnSetArg(XtNx, 20);
       XtnSetArg(XtNy, ROWHEIGHT + 10 + (fileprompt ? LISTHEIGHT : 0));
       XtnSetArg(XtNheight, ROWHEIGHT + 5);
       XtnSetArg(XtNstring, current);
       pdptr = strchr(current, '.');
       posit = (pdptr != NULL) ? (short)(pdptr - current) : strlen(current);
       XtnSetArg(XtNinsertPosition, posit);
       XtnSetArg(XtNgrow, XwGrowHorizontal);
       entertext = XtCreateManagedWidget("Edit", XwtextEditWidgetClass,
	   dialog, wargs, n); n = 0; 

       okaystruct->textw = entertext;
       okaystruct->setvalue = function;

       XtAddCallback(okbutton, XtNselect, gettext, okaystruct);
       XtAddEventHandler(dialog, KeyPressMask, False, propevent, entertext);
       XtAddEventHandler(staticarea, KeyPressMask, False, propevent, entertext);
       XtOverrideTranslations(entertext, XtParseTranslationTable
	  (defaultTranslations));
       XtAddCallback(entertext, XtNexecute, gettext, okaystruct);
       XtAddCallback(cancelbutton, XtNselect, destroypopup, okaystruct);

       /* Generate file prompting widget */

       if (fileprompt) {
          Widget 		listarea, lscroll;

	  XtnSetArg(XtNx, 20);
          XtnSetArg(XtNy, ROWHEIGHT - 10);
          XtnSetArg(XtNwidth, width - SBARSIZE - 40);
	  XtnSetArg(XtNheight, LISTHEIGHT - ROWHEIGHT);
          XtnSetArg(XtNfont, areastruct.textstruct);

	  listarea = XtCreateManagedWidget("Filelist", XwworkSpaceWidgetClass,
		dialog, wargs, n); n = 0;
	  XtAddCallback(listarea, XtNexpose, listfiles, NULL);
	  XtAddEventHandler(listarea, ButtonPressMask, False, fileselect, entertext);
	  XtAddEventHandler(listarea, EnterWindowMask, False, startfiletrack, NULL);
	  XtAddEventHandler(listarea, LeaveWindowMask, False, endfiletrack, NULL);
	  flstart = 0;

	  XtnSetArg(XtNx, width - SBARSIZE - 20);
          XtnSetArg(XtNy, ROWHEIGHT - 10);
          XtnSetArg(XtNwidth, SBARSIZE);
	  XtnSetArg(XtNheight, LISTHEIGHT - ROWHEIGHT);
          XtnSetArg(XtNfont, areastruct.textstruct);

	  lscroll = XtCreateManagedWidget("LScroll", XwworkSpaceWidgetClass,
		dialog, wargs, n); n = 0;
	  XtAddCallback(lscroll, XtNexpose, showlscroll, NULL);
          XtAddEventHandler(lscroll, Button1MotionMask | Button2MotionMask,
		False, draglscroll, listarea);
       }
    }
    else XtAddCallback(okbutton, XtNselect, destroypopup, okaystruct);

    XtPopup(popup, XtGrabNone);

    /* set the input focus for the window */

    wmhints = XAllocWMHints();
    wmhints->flags |= InputHint;
    wmhints->input = True;
    XSetWMHints(dpy, XtWindow(popup), wmhints);
    XFree(wmhints);

    if (current != NULL) XDefineCursor(dpy, XtWindow(entertext), 
	TEXTPTR);
}

/*-------------------------------------------------------------------------*/
/* Create a popup window for property changes				   */
/*-------------------------------------------------------------------------*/

#define MAXPROPS 7
propstruct okstruct[MAXPROPS], fpokstruct;

void dooutput(button, clientdata, calldata)
  Widget        button;
  caddr_t       clientdata, calldata;
{
   buttonsave  *savebutton;
   Arg         wargs[8];
   Widget      popup, dialog, okbutton, tmpwidget, titlearea, wrbutton;
   Widget      fpentertext, fpokay;
   WidgetList  staticarea, entertext, okays;
   XWMHints    *wmhints;	/* for proper input focus */
   Position    xpos, ypos;
   short       n = 0;
   Dimension   height, width, areawidth, areaheight, bwidth, owidth, wwidth;
   XGCValues   promptvalues;
   char	       *pdptr;
   short       posit, i;
   popupstruct *donestruct;
   void	(*function[MAXPROPS])();
   void	(*update[MAXPROPS])();
   char	statics[MAXPROPS][50], edit[MAXPROPS][75], request[150];
   char fpedit[75], outname[75];
   void	*data[MAXPROPS], *objpointer;
   float oscale = areastruct.outscale[areastruct.page];
   float psscale = getpsscale(oscale, areastruct.page);
   struct stat statbuf;
   static char defaultTranslations[] = "<Key>Return:	execute()";

   if (pushes != 0) {
      Wprintf("Can only save a top-level page!");
      return;
   }
   savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   getgeneric(savebutton, button, dooutput, NULL); 

   sprintf(request, "PostScript output properties (Page %d):", 
	areastruct.page + 1);
   sprintf(statics[0], "Filename:");
   sprintf(statics[1], "Page label:");
   sprintf(statics[2], "Scale:");
   sprintf(statics[3], "X Size (in):");
   sprintf(statics[4], "Y Size (in):");
   sprintf(statics[5], "Orientation:");
   sprintf(statics[6], "Mode:");
   calcbbox(areastruct.topobject);
   sprintf(edit[0], "%s", areastruct.filename[areastruct.page]);
   sprintf(edit[1], "%s", objectdata->name);
   sprintf(edit[2], "%6.5f", oscale);
   sprintf(edit[3], "%6.5f", (objectdata->width * psscale) / 72.0);
   sprintf(edit[4], "%6.5f", (objectdata->height * psscale) / 72.0);
   sprintf(edit[5], "%s", (areastruct.orient[areastruct.page] == 0)
	? "Portrait" : "Landscape");
   sprintf(edit[6], "%s", (areastruct.pmode[areastruct.page] == 0)
	? "Embedded (EPS)" : "Full page (PS)");
   function[0] = setfilename;
   function[1] = setpagelabel;
   function[2] = setfloat;
   function[3] = setscalex;
   function[4] = setscaley;
   function[5] = setorient;
   function[6] = setpmode;
   update[0] = updatename;
   update[1] = update[5] = update[6] = NULL;
   update[2] = updatetext;
   update[3] = updatetext;
   update[4] = updatetext;
   data[0] = &(areastruct.filename[areastruct.page]);
   data[1] = objectdata->name;
   data[2] = data[3] = data[4] = &(areastruct.outscale[areastruct.page]);
   data[5] = &(areastruct.orient[areastruct.page]);
   data[6] = &(areastruct.pmode[areastruct.page]);

   entertext = (WidgetList) XtMalloc (7 * sizeof (Widget));
   staticarea = (WidgetList) XtMalloc (7 * sizeof (Widget));
   okays = (WidgetList) XtMalloc (6 * sizeof (Widget));

   /* get file information */

   if (strstr(edit[0], ".") == NULL)
      sprintf(outname, "%s.ps", edit[0]);  
   else sprintf(outname, "%s", edit[0]);
   if (stat(outname, &statbuf) == 0) {
      sprintf(outname, "Overwrite File");
      Wprintf("  Warning:  File exists");
   }
   else {
      sprintf(outname, "Write File");
      if (errno == ENOTDIR)
	 Wprintf("Error:  Incorrect pathname");
      else if (errno == EACCES)
	 Wprintf("Error:  Path not readable");
      else
         Wprintf("  ");
   }

   height = ROWHEIGHT * 17;  /* 3 + (2 * MAXPROPS) */
   width = XTextWidth(areastruct.textstruct, request, strlen(request)) + 20;
   bwidth = XTextWidth(areastruct.textstruct, "Close", strlen("Close")) + 50;
   owidth = XTextWidth(areastruct.textstruct, "Okay", strlen("Okay")) + 50;
   wwidth = XTextWidth(areastruct.textstruct, outname, strlen(outname)) + 80;
   if (width < 500) width = 500;

   XtnSetArg(XtNwidth, &areawidth);
   XtnSetArg(XtNheight, &areaheight);
   XtGetValues(areastruct.area, wargs, n); n = 0;
   XtTranslateCoords(areastruct.area, (Position) (areawidth / 2 - width 
	/ 2 + popups * 20), (Position) (areaheight / 2 - height / 2 +
	popups * 20), &xpos, &ypos);
   XtnSetArg(XtNx, xpos);
   XtnSetArg(XtNy, ypos);
   popup = XtCreatePopupShell("prompt", transientShellWidgetClass,
        areastruct.area, wargs, n); n = 0;
   popups++;

   XtnSetArg(XtNlayout, XwIGNORE);
   XtnSetArg(XtNwidth, width);
   XtnSetArg(XtNheight, height);
   dialog = XtCreateManagedWidget("dialog", XwbulletinWidgetClass,
        popup, wargs, n); n = 0;

   XtnSetArg(XtNx, 20);
   XtnSetArg(XtNy, ROWHEIGHT - 10);
   XtnSetArg(XtNstring, request);
   XtnSetArg(XtNborderWidth, 0);
   XtnSetArg(XtNgravity, WestGravity);
   XtnSetArg(XtNbackground, BARCOLOR);
   titlearea = XtCreateManagedWidget("title", XwstaticTextWidgetClass,
	dialog, wargs, n); n = 0;

   XtnSetArg(XtNx, 20);
   XtnSetArg(XtNy, height - ROWHEIGHT - 10);
   XtnSetArg(XtNwidth, owidth); 
   XtnSetArg(XtNfont, areastruct.textstruct);
   okbutton = XtCreateManagedWidget("Close", XwmenuButtonWidgetClass, 
	dialog, wargs, n); n = 0;

   XtnSetArg(XtNx, width - wwidth - 20);
   XtnSetArg(XtNy, height - ROWHEIGHT - 10);
   XtnSetArg(XtNwidth, wwidth);
   XtnSetArg(XtNfont, areastruct.textstruct);
   XtnSetArg(XtNlabel, outname);
   wrbutton = XtCreateManagedWidget("Write File", XwmenuButtonWidgetClass,
	dialog, wargs, n); n = 0;

   for (i = 0; i < MAXPROPS; i++) {
      XtnSetArg(XtNx, 20);
      XtnSetArg(XtNy, ROWHEIGHT + 15 + (i * 2 * ROWHEIGHT));
      XtnSetArg(XtNstring, statics[i]);
      XtnSetArg(XtNborderWidth, 0);
      XtnSetArg(XtNgravity, WestGravity);
      staticarea[i] = XtCreateManagedWidget("static", XwstaticTextWidgetClass,
	   dialog, wargs, n); n = 0;

      XtnSetArg(XtNx, 150);
      XtnSetArg(XtNy, ROWHEIGHT + 10 + (i * 2 * ROWHEIGHT));
      if (i < 5) {
         XtnSetArg(XtNheight, ROWHEIGHT + 5);
         XtnSetArg(XtNstring, edit[i]);
         XtnSetArg(XtNwidth, width - owidth - 190);
         pdptr = strchr(edit[i], '.');
         posit = (pdptr != NULL) ? (short)(pdptr - edit[i]) : strlen(edit[i]);
         XtnSetArg(XtNinsertPosition, posit);
         XtnSetArg(XtNgrow, XwGrowHorizontal);
         entertext[i] = XtCreateManagedWidget("Edit", XwtextEditWidgetClass,
	     dialog, wargs, n); n = 0; 

         XtnSetArg(XtNx, width - owidth - 20);
         XtnSetArg(XtNy, ROWHEIGHT + 10 + (i * 2 * ROWHEIGHT));
         XtnSetArg(XtNwidth, owidth); 
         XtnSetArg(XtNfont, areastruct.textstruct);
         okays[i] = XtCreateManagedWidget("Okay", XwmenuButtonWidgetClass, 
	    dialog, wargs, n); n = 0;

         okstruct[i].textw = entertext[i];
         okstruct[i].setvalue = function[i];
         okstruct[i].dataptr = data[i];

         XtAddCallback(okays[i], XtNselect, getproptext, &okstruct[i]);
	 if (update[i] != NULL)
            XtAddCallback(okays[i], XtNselect, update[i], entertext);
         XtOverrideTranslations(entertext[i], XtParseTranslationTable
              (defaultTranslations));
         XtAddCallback(entertext[i], XtNexecute, getproptext, &okstruct[i]);
         if (update[i] != NULL) XtAddCallback(entertext[i], XtNexecute, update[i],
	      entertext);

      }
      else {
	 XtnSetArg(XtNlabel, edit[i]);
         XtnSetArg(XtNfont, areastruct.textstruct);
         entertext[i] = XtCreateManagedWidget("Toggle", XwpushButtonWidgetClass,
	    dialog, wargs, n); n = 0;
	 XtAddCallback(entertext[i], XtNselect, function[i], data[i]);
      }
   }

   /* If full-page pmode is chosen, there is an additional text structure.
      Make this text structure always but allow it to be managed and
      unmanaged as necessary. */

   sprintf(fpedit, "%3.2f x %3.2f in", (float)areastruct.pagesize[areastruct.page].x
	/ 72.0, (float)areastruct.pagesize[areastruct.page].y / 72.0);
   XtnSetArg(XtNx, 260);
   XtnSetArg(XtNy, ROWHEIGHT + 10 + (12 * ROWHEIGHT));
   XtnSetArg(XtNheight, ROWHEIGHT + 5);
   XtnSetArg(XtNstring, fpedit);
   XtnSetArg(XtNwidth, width - owidth - 300);
   pdptr = strchr(fpedit, '.');
   posit = (pdptr != NULL) ? (short)(pdptr - fpedit) : strlen(fpedit);
   XtnSetArg(XtNinsertPosition, posit);
   XtnSetArg(XtNgrow, XwGrowHorizontal);
   fpentertext = XtCreateWidget("fpedit", XwtextEditWidgetClass,
       dialog, wargs, n); n = 0;

   XtnSetArg(XtNx, width - owidth - 20);
   XtnSetArg(XtNy, ROWHEIGHT + 10 + (12 * ROWHEIGHT));
   XtnSetArg(XtNwidth, owidth);
   XtnSetArg(XtNfont, areastruct.textstruct);
   XtnSetArg(XtNlabel, "Okay");
   fpokay = XtCreateWidget("fpokay", XwmenuButtonWidgetClass,
      dialog, wargs, n); n = 0;

   fpokstruct.textw = fpentertext;
   fpokstruct.setvalue = setpagesize;
   fpokstruct.dataptr = &(areastruct.pagesize[areastruct.page]);

   XtAddCallback(fpokay, XtNselect, getproptext, &fpokstruct);
   XtOverrideTranslations(fpentertext, XtParseTranslationTable
        (defaultTranslations));
   XtAddCallback(fpentertext, XtNexecute, getproptext, &fpokstruct);

   if (areastruct.pmode[areastruct.page] == 1) {
      XtManageChild(fpentertext);
      XtManageChild(fpokay);
   }

   /* end of pagesize extra Widget definitions */

   donestruct = (popupstruct *) malloc(sizeof(popupstruct));
   donestruct->popup = popup;
   donestruct->buttonptr = savebutton;
   XtAddCallback(okbutton, XtNselect, destroypopup, donestruct);

   /* Send setfile() the widget entertext[0] in case because user sometimes
      forgets to type "okay" but buffer contains the expected filename */

   XtAddCallback(wrbutton, XtNselect, setfile, entertext[0]);

   /* Begin Popup */

   XtPopup(popup, XtGrabNone);

   /* set the input focus for the window */

   wmhints = XAllocWMHints();
   wmhints->flags |= InputHint;
   wmhints->input = True;
   XSetWMHints(dpy, XtWindow(popup), wmhints);
   XFree(wmhints);
   for (i = 0; i < 5; i++)
      XDefineCursor(dpy, XtWindow(entertext[i]), TEXTPTR);
}

/*----------------------------------------------------------------*/
/* Menu subroutines for changing variables from the menus	  */
/*----------------------------------------------------------------*/

/* Generic routine for use by all other data handling routines */

void getgeneric(saveptr, button, getfunction, dataptr)
  buttonsave *saveptr;
  Widget button;
  void  (*getfunction)();
  void	*dataptr;
{
   char buffer[200];
   Arg  wargs[1];

   saveptr->button = button;
   saveptr->buttoncall = getfunction;
   saveptr->dataptr = dataptr;

   XtSetArg(wargs[0], XtNforeground, &saveptr->foreground);
   XtGetValues(button, wargs, 1);
   XtSetArg(wargs[0], XtNforeground, OFFBUTTONCOLOR);
   XtSetValues(button, wargs, 1);
   XtRemoveAllCallbacks(button, XtNselect);
}

/* setint, getint are for grid and snap spacing sizes;	*/
/* include routines to parse fractions			*/

void setint(w, dataptr)
  Widget w;
  short *dataptr;
{
   short oldvalue = *dataptr;
   float oscale, iscale = (float)areastruct.drawingscale[areastruct.page].y /
        (float)areastruct.drawingscale[areastruct.page].x;
   float fval;

   /* For now, assume that the value is in the current display style. */
   /* Might be nice in the future to make it accept any input. . .    */

   switch (areastruct.coordstyle[areastruct.page]) {
      case CM:
         oscale = areastruct.outscale[areastruct.page] * CMSCALE;
	 if (sscanf(_STR2, "%f", fval) == 0) {
	    *dataptr = oldvalue;
	    Wprintf("Illegal value");
	 }
	 else *dataptr = (int)(fval * 72.0 / (iscale * oscale) + 0.5);
	 break;
      case DEC_INCH: case FRAC_INCH: {
	 short parts;
	 char *sptr;
	 int f2, f3;

         oscale = areastruct.outscale[areastruct.page] * INCHSCALE;
	 for (sptr = _STR2; *sptr != '\0'; sptr++)
	    if (*sptr == '/') *sptr = ' ';
	 parts = sscanf(_STR2, "%f %d %d", &fval, &f2, &f3);
	 if ((parts == 0) || (parts != 1 && (fval != (float)((int)fval)))) {
	    *dataptr = oldvalue;
	    Wprintf("Illegal value");
	    break;
	 }
	 if (parts == 2) fval /= (float)f2;
	 else if (parts == 3) fval += ((float)f2 / (float)f3);
	 *dataptr = (int)(fval * 72.0 / (iscale * oscale) + 0.5);
	 } break;
   }
   if (oldvalue != *dataptr) drawarea(NULL, objectdata, dataptr);
}

void measurestr(value, buffer)
  float         value;
  char          *buffer;
{  
   float oscale, iscale = (float)areastruct.drawingscale[areastruct.page].y /
        (float)areastruct.drawingscale[areastruct.page].x;
   switch (areastruct.coordstyle[areastruct.page]) {
      case CM:
         oscale = areastruct.outscale[areastruct.page] * CMSCALE;
         sprintf(buffer, "%5.3f cm", value * iscale * oscale / 72.0);
         break; 
      case DEC_INCH:
         oscale = areastruct.outscale[areastruct.page] * INCHSCALE;
         sprintf(buffer, "%5.3f in", value * iscale * oscale / 72.0);
         break;
      case FRAC_INCH:
         oscale = areastruct.outscale[areastruct.page] * INCHSCALE;
         fraccalc(((value * iscale * oscale) / 72.0), buffer);
         strcat(buffer, " in");
         break;
   }
}        

void getint(button, intptr, calldata)
  Widget 	button;
  short 	*intptr;
  caddr_t 	calldata;
{
   char buffer[50];
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));

   getgeneric(savebutton, button, getint, intptr);
   measurestr((float)(*intptr), buffer);
   popupprompt(button, "Enter value:", buffer, setint, savebutton, False);
}

/* floating-point value routines */

void setfloat(w, dataptr)
  Widget w;
  float *dataptr;
{
   float oldvalue = *dataptr;
   int res = sscanf(_STR2, "%f", dataptr);

   if (res == 0 || *dataptr <= 0) {
      *dataptr = oldvalue;
      Wprintf("Illegal value");
   }
   if (oldvalue != *dataptr) drawarea(NULL, objectdata, dataptr);
}

void setwidth(w, dataptr) /* set wire width --- internal value is 2x */
  Widget w;
  float *dataptr;
{
   float oldvalue = *dataptr;
   if (sscanf(_STR2, "%f", dataptr) == 0) {
      *dataptr = oldvalue;
      Wprintf("Illegal value");
      return;
   }
   (*dataptr) *= 2.0;
   if (oldvalue != *dataptr) drawarea(NULL, objectdata, dataptr);
}

void settsize(w, settext)
  Widget w;
  labelptr settext;
{
   float tmpres;
   short *osel;
   labelptr nstext;
   int res = sscanf(_STR2, "%f", &tmpres);

   if (res == 0 || tmpres <= 0) {  /* can't interpret value or bad value */
      Wprintf("Illegal value");
      return;
   }

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      if (strlen(settext->string) > 0) {
	 undrawtext(settext);
         settext->scale = tmpres;
	 redrawtext(settext);
      }
      else (settext->scale = tmpres);
   }
   else if (areastruct.selects > 0) {
      for (osel = areastruct.selectlist; osel < areastruct.selectlist +
	     areastruct.selects; osel++) {
	 if (SELECTTYPE(osel) == LABEL) {
	    nstext = SELTOLABEL(osel);
	    undrawtext(nstext);
            nstext->scale = tmpres;
	    redrawtext(nstext);
	 }
      }
      objectdeselect();
   }
}

void setscaley(w, dataptr)
  Widget w;
  float *dataptr;
{
   float oldvalue = *dataptr;
   int res = sscanf(_STR2, "%f", dataptr);

   if (res == 0 || *dataptr <= 0) {
      *dataptr = oldvalue;
      Wprintf("Illegal value");
   };
   *dataptr = (*dataptr * 72) / objectdata->height;
   *dataptr /= getpsscale(1.0, areastruct.page);
}

void setscalex(w, dataptr)
  Widget w;
  float *dataptr;
{
   float oldvalue = *dataptr;
   int res = sscanf(_STR2, "%f", dataptr);

   if (res == 0 || *dataptr <= 0) {
      *dataptr = oldvalue;
      Wprintf("Illegal value");
   }
   *dataptr = (*dataptr * 72) / objectdata->width;
   *dataptr /= getpsscale(1.0, areastruct.page);
}

void setorient(w, dataptr)
  Widget w;
  short *dataptr;
{
   Arg wargs[1];

   if (*dataptr == 0) {
      *dataptr = 90;
      XtSetArg(wargs[0], XtNlabel, "Landscape");
   }
   else {
      *dataptr = 0;
      XtSetArg(wargs[0], XtNlabel, "Portrait");
   }
   XtSetValues(w, wargs, 1);
}

void setpmode(w, dataptr)
  Widget w;
  short *dataptr;
{
   Arg wargs[1];
   Widget pwidg = XtParent(w);

   if (*dataptr == 0) {
      *dataptr = 1;
      XtSetArg(wargs[0], XtNlabel, "Full Page");

      XtManageChild(XtNameToWidget(pwidg, "fpedit"));
      XtManageChild(XtNameToWidget(pwidg, "fpokay"));
   }
   else {
      *dataptr = 0;
      XtSetArg(wargs[0], XtNlabel, "Embedded");

      XtUnmanageChild(XtNameToWidget(pwidg, "fpedit"));
      XtUnmanageChild(XtNameToWidget(pwidg, "fpokay"));
   }
   XtSetValues(w, wargs, 1);
}

void setpagesize(w, dataptr)
  Widget w;
  XPoint *dataptr;
{
   float px, py;
   char units[10], *expos, fpedit[75];
   Arg wargs[1];

   strcpy(units, "in");

   if (sscanf(_STR2, "%f %*c %f %s", &px, &py, units) < 4) {
      if (sscanf(_STR2, "%f %*c %f", &px, &py) < 3) {
	 if ((expos = strchr(_STR2, 'x')) == NULL) {
            Wprintf("Illegal Form for page size.");
	    return;
	 }
	 else {
	    *expos = '\0';
	    if (sscanf(_STR2, "%f", &px) == 0 ||
		  sscanf(expos + 1, "%f %s", &py, units) == 0) {
               Wprintf("Illegal Form for page size.");
	       return;
	    }
	 }
      }
   }

   dataptr->x = (short)(px * 72.0);
   dataptr->y = (short)(py * 72.0);

   if (!strcmp(units, "cm")) {
      sprintf(fpedit, "%3.2f x %3.2f cm", (float)areastruct.pagesize[areastruct.page].x
	  / 72.0, (float)areastruct.pagesize[areastruct.page].y / 72.0);
      dataptr->x /= 2.54;
      dataptr->y /= 2.54;
   }
   else {
      sprintf(fpedit, "%3.2f x %3.2f in", (float)areastruct.pagesize[areastruct.page].x
	  / 72.0, (float)areastruct.pagesize[areastruct.page].y / 72.0);
   }
   XtSetArg(wargs[0], XtNstring, fpedit);
   XtSetValues(XtNameToWidget(XtParent(w), "fpedit"), wargs, 1);
}

void getdscale(button, nulldata, calldata)
  Widget	button;
  caddr_t	nulldata, calldata;
{
   char buffer[50];
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   XPoint *ptptr = &(areastruct.drawingscale[areastruct.page]);

   getgeneric(savebutton, button, getdscale, ptptr);
   sprintf(buffer, "%d:%d", ptptr->x, ptptr->y);
   popupprompt(button, "Enter Scale:", buffer, setdscale, savebutton, False);
}

void gettsize(button, nulldata, calldata)
  Widget        button;
  caddr_t       nulldata, calldata;
{
   char buffer[50];
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   labelptr settext = NULL;
   float      *floatptr = &areastruct.textscale;
   short *osel;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      settext = *((labelptr *)CONDLASTPART(TEXT3_MODE));
      floatptr = &(settext->scale);
   }
   else if (areastruct.selects > 0) {
      for (osel = areastruct.selectlist; osel < areastruct.selectlist +
		areastruct.selects; osel++)
	 if (SELECTTYPE(osel) == LABEL) {
            settext = SELTOLABEL(osel);
            floatptr = &(settext->scale);
	    break;
	 }
   }
   sprintf(buffer, "%4.2f", *floatptr);
   if (settext != NULL) {
      getgeneric(savebutton, button, gettsize, settext);
      popupprompt(button, "Enter label scale:", buffer, settsize, savebutton, False);
   }
   else {
      getgeneric(savebutton, button, gettsize, floatptr);
      popupprompt(button,
	    "Enter default text scale:", buffer, setfloat, savebutton, False);
   }
}

void setosize(w, dataptr)
  Widget w;
  objinstptr dataptr;
{
   float tmpres;
   short *osel;
   objinstptr nsobj;
   int res = sscanf(_STR2, "%f", &tmpres);

   if (res == 0 || tmpres <= 0) {
      Wprintf("Illegal value");
      return;
   }
   for (osel = areastruct.selectlist; osel < areastruct.selectlist +
	     areastruct.selects; osel++) {
      if (SELECTTYPE(osel) == OBJECT) {
	 nsobj = SELTOOBJINST(osel);
         nsobj->scale = tmpres;
      }
   }
   objectdeselect();
   drawarea(NULL, objectdata, dataptr);
}

void getosize(button, nullptr, calldata)
  Widget        button;
  void		*nullptr;
  caddr_t       calldata;
{
   char buffer[50];
   float flval;
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   short *osel = areastruct.selectlist;
   short selects = 0;
   objinstptr setobj = NULL;

   for (; osel < areastruct.selectlist + areastruct.selects; osel++)
      if (SELECTTYPE(osel) == OBJECT) {
	 setobj = SELTOOBJINST(osel);
	 selects++;
	 break;
      }
   if (setobj == NULL) {
      Wprintf("No objects were selected for scaling.");
      return;
   }
   flval = setobj->scale;
   getgeneric(savebutton, button, getosize, setobj);
   sprintf(buffer, "%4.2f", flval);
   popupprompt(button, "Enter object scale:", buffer, setosize, savebutton, False);
}

void getwirewidth(button, floatptr, calldata)
  Widget	button;
  float		*floatptr;
  caddr_t	calldata;
{
   char buffer[50];
   float *widthptr;
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));

   widthptr = &(areastruct.wirewidth[areastruct.page]);
   getgeneric(savebutton, button, getwirewidth, widthptr);
   sprintf(buffer, "%4.2f", *widthptr / 2.0);
   popupprompt(button, "Enter new global wire width:", buffer, setwidth,
	savebutton, False);
}

void setwwidth(w, dataptr)
  Widget w;
  void *dataptr;
{
   float     tmpres;
   short     *osel;
   arcptr    nsarc;
   polyptr   nspoly;
   splineptr nsspline;
   pathptr   nspath;

   if (sscanf(_STR2, "%f", &tmpres) == 0) {
      Wprintf("Illegal value");
      return;
   }
   for (osel = areastruct.selectlist; osel < areastruct.selectlist +
	     areastruct.selects; osel++) {
      if (SELECTTYPE(osel) == ARC) {
	 nsarc = SELTOARC(osel);
         nsarc->width = tmpres;
      }
      else if (SELECTTYPE(osel) == POLYGON) {
	 nspoly = SELTOPOLY(osel);
         nspoly->width = tmpres;
      }
      else if (SELECTTYPE(osel) == SPLINE) {
	 nsspline = SELTOSPLINE(osel);
         nsspline->width = tmpres;
      }
      else if (SELECTTYPE(osel) == PATH) {
	 nspath = SELTOPATH(osel);
         nspath->width = tmpres;
      }
   }
   objectdeselect();
   drawarea(NULL, objectdata, dataptr);
}

void getwwidth(button, nullptr, calldata)
  Widget        button;
  void		*nullptr;
  caddr_t       calldata;
{
   char buffer[50];
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));
   short *osel = areastruct.selectlist;
   genericptr setel;
   float flval;

   for (; osel < areastruct.selectlist + areastruct.selects; osel++) {
      setel = *(objectdata->plist + (*osel));
      if (setel->type == ARC) {
	 flval = ((arcptr)setel)->width;
	 break;
      }
      else if (setel->type == POLYGON) {
	 flval = ((polyptr)setel)->width;
	 break;
      }
      else if (setel->type == SPLINE) {
	 flval = ((splineptr)setel)->width;
	 break;
      }
      else if (setel->type == PATH) {
	 flval = ((pathptr)setel)->width;
	 break;
      }
   }
   if (osel == areastruct.selectlist + areastruct.selects) {
      Wprintf("No arcs, polygons, splines, or paths were selected.");
      return;
   }
   getgeneric(savebutton, button, getwwidth, setel);
   sprintf(buffer, "%4.2f", flval);
   popupprompt(button, "Enter new line width:", buffer, setwwidth, savebutton, False);
}

void getfloat(button, floatptr, calldata)
  Widget	button;
  float		*floatptr;
  caddr_t	calldata;
{
   char buffer[50];
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));

   getgeneric(savebutton, button, getfloat, floatptr);
   sprintf(buffer, "%4.2f", *floatptr);
   popupprompt(button, "Enter value:", buffer, setfloat, savebutton, False);
}

/* string routines */

void setfilename(w, dataptr)
  Widget w;
  char	**dataptr;
{
   short cpage, depend = 0;
   objectptr checkpage;

   if (!strcmp(*dataptr, _STR2)) return;   /* no change in string */

   /* Check for dependencies on this string from other pages.        */
   /* If there are no other pointers to this string, we can free it. */

   for (cpage = 0; cpage < areastruct.pages; cpage++) {
      checkpage = *(areastruct.pagelist + cpage);
      if ((checkpage != NULL) && (cpage != areastruct.page))  
	 if (areastruct.filename[cpage] == areastruct.filename[areastruct.page])  
	    depend = 1;
   }

   /* check if the new filename is an existing filename */

   for (cpage = 0; cpage < areastruct.pages; cpage++) {
      checkpage = *(areastruct.pagelist + cpage);
      if ((checkpage != NULL) && (cpage != areastruct.page)) {
	 if (!strcmp(_STR2, areastruct.filename[cpage])) {
	    if (!depend) free(areastruct.filename[areastruct.page]);
	    areastruct.filename[areastruct.page] = areastruct.filename[cpage];
	    break;
	 }
      }
   }
   if (cpage == areastruct.pages) {
      if (!depend) free(areastruct.filename[areastruct.page]);
      areastruct.filename[areastruct.page] = (char *) malloc((strlen(_STR2)
		+ 1) * sizeof(char));
      strcpy(areastruct.filename[areastruct.page], _STR2);
   }
}

void setpagelabel(w, dataptr)
  Widget w;
  char	*dataptr;
{
   short cpage, i;
   objectptr checkpage;

   /* whitespace and non-printing characters not allowed */

   for (i = 0; i < strlen(_STR2); i++) {
      if ((!isprint(_STR2[i])) || (isspace(_STR2[i]))) {
         _STR2[i] = '_';
         Wprintf("Replaced illegal whitespace in name with underscore");
      }
   }
   if (!strcmp(dataptr, _STR2)) return; /* no change in string */
   if (strlen(_STR2) == 0)
      sprintf(objectdata->name, "Page %d", areastruct.page + 1);
   else
      sprintf(objectdata->name, "%.79s", _STR2);
   printname(objectdata);
   renamepage(areastruct.page);
}

void setlabel(w, dataptr)
  Widget w;
  char  *dataptr;
{
   char oldvalue[150];
   labelptr *whatlabel;

   sprintf(oldvalue, dataptr);
   for (whatlabel = (labelptr *)objectdata->plist; whatlabel <
	(labelptr *)objectdata->plist + objectdata->parts; whatlabel++)
      if ((*whatlabel)->type == LABEL)
         if ((*whatlabel)->string == (uchar *)dataptr) break;
   dataptr = (char *) realloc(dataptr, (strlen(_STR2) + 1) * sizeof(char));
   (*whatlabel)->string = (uchar *)dataptr;
   sprintf(dataptr, _STR2);
   if (strcmp(oldvalue, dataptr)) drawarea(NULL, objectdata, dataptr);
}

void setnewfont(w, dataptr)
  Widget w;
  char	*dataptr;
{
   Arg	wargs[1];
   int  n = 0;
   Widget newbutton, cascade;

   if (w != NULL)
      cascade = XtParent(w);
   else
      cascade = XtNameToWidget(top, FontsPath);

   free(dataptr);
   fonts[fontcount] = (char *)malloc((strlen(_STR2) + 1) * sizeof(char));
   sprintf(fonts[fontcount], _STR2);

   XtnSetArg(XtNfont, areastruct.textstruct);
   newbutton = XtCreateWidget(fonts[fontcount], XwmenubuttonWidgetClass,
	 cascade, wargs, n);

   XtAddCallback (newbutton, XtNselect, setfont, Number(fontcount));
   XtManageChild(newbutton);
   fontcount++;
}

/*--------------------------------------------------------------------*/
/* Toggle one of a set of menu items, only one of which can be active */
/*--------------------------------------------------------------------*/

void toggleexcl(widget, menu, menusize)
  Widget	widget;
  menuptr	menu;
  int		menusize;
{
   Arg          args[1];
   Widget       parent = XtParent(widget);
   Widget       sibling;
   menuptr	mitem, mtest;
   short	i;

   /* find the menu item which corresponds to the widget which was pushed */

   for (mtest = menu; mtest < menu + menusize; mtest++) {
      sibling = XtNameToWidget(parent, mtest->name);
      if (sibling == widget) break;
   }

   /* remove checkmark from other widgets in the list */

   if (menu == Fonts) {     		/* special action for font list */
      for (i = FONTS; i < fontcount; i++) {
	 sibling = XtNameToWidget(parent, fonts[i]);
	 if (sibling != widget) {
	    XtSetArg(args[0], XtNsetMark, False);
	    XtSetValues(sibling, args, 1);
	 }
      }
      mtest = &Fonts[3];   /* so that mtest->func has correct value below */
   }
   else if (mtest == menu + menusize) return;  /* something went wrong? */

   for (mitem = menu; mitem < menu + menusize; mitem++) {
      sibling = XtNameToWidget(parent, mitem->name);
      if (mitem->func == mtest->func) {
         XtSetArg(args[0], XtNsetMark, False);
         XtSetValues(sibling, args, 1);
      }
   }

   /* Now set the currently pushed widget */

   XtSetArg(args[0], XtNsetMark, True);
   XtSetValues(widget, args, 1);
}

/*--------------------*/
/* Toggle a menu item */
/*--------------------*/

void toggle(w, boolvalue, calldata)
  Widget w;
  Boolean	*boolvalue;
  caddr_t	calldata;
{
   Arg	wargs[1];

   *boolvalue = !(*boolvalue);
   XtSetArg(wargs[0], XtNsetMark, *boolvalue);
   XtSetValues(w, wargs, 1);
   drawarea(w, objectdata, boolvalue);
}

/*---------------------------------------------------------------*/
/* Some Xcircuit routines using toggle and toggleexcl, 		 */
/* put here because they reference the menu structures directly. */
/*---------------------------------------------------------------*/

void setcolorscheme(boolvalue)
  Boolean	boolvalue;
{
   if (boolvalue) {
      AUXCOLOR = appdata.auxpix;
      BARCOLOR = appdata.barpix;
      OFFBUTTONCOLOR = appdata.buttonpix;
      SELECTCOLOR = appdata.selectpix;
      QUERYCOLOR = appdata.querypix;
      GRIDCOLOR = appdata.gridpix;
      SNAPCOLOR = appdata.snappix;
      AXESCOLOR = appdata.axespix;
      BACKGROUND = appdata.bg;
      FOREGROUND = appdata.fg;
   }
   else {
      AUXCOLOR = appdata.auxpix2;
      BARCOLOR = appdata.barpix2;
      OFFBUTTONCOLOR = appdata.buttonpix2;
      SELECTCOLOR = appdata.selectpix2;
      QUERYCOLOR = appdata.querypix2;
      GRIDCOLOR = appdata.gridpix2;
      SNAPCOLOR = appdata.snappix2;
      AXESCOLOR = appdata.axespix2;
      BACKGROUND = appdata.bg2;
      FOREGROUND = appdata.fg2;
   }

   makecursors();
}

void inversecolor(w, boolvalue, calldata)
  Widget w;
  Boolean	*boolvalue;
  caddr_t	calldata;
{
   /* change color scheme */

   setcolorscheme(*boolvalue);
   toggle(w, boolvalue, calldata);
   if (eventmode == NORMAL_MODE) XDefineCursor (dpy, win, CROSS);
}

void setgridtype(string)
   char *string;
{
   Widget button, bparent = XtNameToWidget(top, GridStylesPath);

   if (!strcmp(string, "inchscale")) {
      button = XtNameToWidget(bparent, "Fractional Inches");
      getgridtype(button, FRAC_INCH, NULL);
   }
   else if (!strcmp(string, "cmscale")) {
      button = XtNameToWidget(bparent, "Centimeters");
      getgridtype(button, CM, NULL);
   }
}

void togglegrid(type)
  unsigned short type;
{
   Widget button, bparent = XtNameToWidget(top, GridStylesPath);

   if (type == CM) button = XtNameToWidget(bparent, "Centimeters");
   else if (type == FRAC_INCH) button = XtNameToWidget(bparent, "Fractional Inches");
   else if (type == DEC_INCH) button = XtNameToWidget(bparent, "Decimal Inches");
   else button = XtNameToWidget(bparent, "Coordinates");
   toggleexcl(button, GridStyles, XtNumber(GridStyles));
   W1printf(" ");
}

void getgridtype(button, value, calldata)
  Widget button;
  unsigned short value;
  caddr_t calldata;
{
   short oldtype = areastruct.coordstyle[areastruct.page];
   float scalefac = getpsscale(1.0, areastruct.page);

   toggleexcl(button, GridStyles, XtNumber(GridStyles));
   areastruct.coordstyle[areastruct.page] = (short)value;

   /* only reset gridspace if going cm->in or vice versa */
   switch(value) {
      case FRAC_INCH: case DEC_INCH:
	 if (oldtype == CM) {
	    areastruct.gridspace = 32;	/* grid size 32 = 1/6 inch  */
	    areastruct.snapspace = 16;  /* (that's 192 to the inch) */
	    scalefac /= INCHSCALE;
            areastruct.outscale[areastruct.page] *= scalefac;
	 }
	 break;
      case CM:
	 if (oldtype != CM) {
	    areastruct.gridspace = 40;  /* grid size 40 = 1/2 cm */
	    areastruct.snapspace = 20;
	    scalefac /= CMSCALE;
            areastruct.outscale[areastruct.page] *= scalefac;
	 }
	 break;
   }
   if (oldtype != areastruct.coordstyle[areastruct.page]) {
      drawarea(NULL, objectdata, NULL);
      W1printf(" ");
   }
}

void setdscale(w, dataptr)
  Widget w;
  XPoint *dataptr;
{
   char *sptr;

   if ((sptr = strchr(_STR2, ':')) == NULL)
      Wprintf("Use ratio X:Y");
   else {
      *sptr = '\0';
      sscanf(_STR2, "%hd", &(dataptr->x));
      sscanf(sptr + 1, "%hd", &(dataptr->y));
      sprintf(_STR2, "New scale is %hd:%hd", dataptr->x, dataptr->y);
      Wprintf(_STR2);
      W1printf(" ");
   }
}

/*----------------------------------------------*/
/* Make new page; goto page			*/
/*----------------------------------------------*/

void newpagemenu(w, value, nulldata)  /* wrapper for events.c:newpage() */
  Widget        w;
  unsigned short value;
  caddr_t       nulldata;
{
   if (eventmode == CATALOG_MODE) {
      eventmode = NORMAL_MODE;
      catreturn();
   }
   newpage((short)value);
}

void makepagebutton()	/* called by newpage if new button needs to be made */
{
   Widget pagemenu, newbutton;
   Arg wargs[1];
   char pagestring[10];

   /* make new entry in the menu */

   pagemenu = XtNameToWidget(top, PagesPath);
   XtSetArg(wargs[0], XtNfont, areastruct.textstruct);
   sprintf(pagestring, "Page %d", areastruct.pages);
   newbutton = XtCreateWidget(pagestring, XwmenubuttonWidgetClass,
         pagemenu, wargs, 1);
   XtAddCallback (newbutton, XtNselect, newpagemenu, Number(areastruct.pages - 1));
   XtManageChild(newbutton);
}

void renamepage(pagenumber)
   short pagenumber;
{
   Arg wargs[1];
   Widget parent = XtNameToWidget(top, PagesPath);
   Widget button;
   char bname[10];

   sprintf(bname, "Page %d", pagenumber + 1);
   button = XtNameToWidget(parent, bname);

   XtSetArg(wargs[0], XtNlabel, objectdata->name);
   XtSetValues(button, wargs, 1);
}

/*----------------------------------------------*/
/* Set Poly and Arc line styles and fill styles */
/*----------------------------------------------*/

#define BORDERS  (NOBORDER | DOTTED | DASHED)
#define ALLFILLS (FILLSOLID | FILLED)

/*----------------------------------------------*/
/* Set the menu marks according to style value  */
/*----------------------------------------------*/

void setstylemarks(w, value, boolval)
  Widget w;
  short value;
  Boolean boolval;
{
   Arg	wargs[1];

   /* if widget is unknown, find button for each part */

   if (value & UNCLOSED) {
      XtSetArg(wargs[0], XtNsetMark, boolval);
      XtSetValues(w, wargs, 1);
   }
   if (value & OPAQUE) {
      XtSetArg(wargs[0], XtNsetMark, boolval);
      XtSetValues(w, wargs, 1);
   }
   if (value & BORDERS || value & ALLFILLS || value == 0) {
      toggleexcl(w, BorderStyles, XtNumber(BorderStyles));
      toggleexcl(w, Stipples, XtNumber(Stipples));
   }
}

/*----------------------------------------------*/

void setcolormark(colorval)
  short colorval;
{
   Widget w = NULL;
   short i;


   for (i = 0; i < number_colors; i++)
      if (colorlist[i].color.pixel == colorval) {
         w = colorlist[i].cbutton;
	 break;
      }

   if (colorval == DEFAULTCOLOR)
      w = XtNameToWidget(top, ColorsInheritColorPath);

   if (w != NULL) toggleexcl(w, Colors, XtNumber(Colors));
}

/*----------------------------------------------*/

void setallstylemarks(styleval)
  short styleval;
{
   Widget w;

   w = XtNameToWidget(top, BorderStylesClosedPath);
   setstylemarks(w, UNCLOSED, !(styleval & UNCLOSED));
   w = XtNameToWidget(top, StipplesOpaquePath);
   setstylemarks(w, OPAQUE, (styleval & OPAQUE) > 0);

   if (styleval & NOBORDER)
      w = XtNameToWidget(top, BorderStylesUnborderedPath);
   else if (styleval & DASHED)
      w = XtNameToWidget(top, BorderStylesDashedPath);
   else if (styleval & DOTTED)
      w = XtNameToWidget(top, BorderStylesDottedPath);
   else
      w = XtNameToWidget(top, BorderStylesSolidPath);
   setstylemarks(w, BORDERS, 0);

   if (!(styleval & FILLED))
      w = XtNameToWidget(top, StipplesWhitePath);
   else {
      styleval &= FILLSOLID;
      styleval /= STIP0;
      switch(styleval) {
	 case 0: w = XtNameToWidget(top, StipplesGray87Path); break;
	 case 1: w = XtNameToWidget(top, StipplesGray75Path); break;
	 case 2: w = XtNameToWidget(top, StipplesGray62Path); break;
	 case 3: w = XtNameToWidget(top, StipplesGray50Path); break;
	 case 4: w = XtNameToWidget(top, StipplesGray37Path); break;
	 case 5: w = XtNameToWidget(top, StipplesGray25Path); break;
	 case 6: w = XtNameToWidget(top, StipplesGray12Path); break;
	 case 7: w = XtNameToWidget(top, StipplesBlackPath);  break;
      }
   }
   setstylemarks(w, ALLFILLS, 0);
}

/*----------------------------------------------*/

void getfill(w, value, nulldata)
  Widget        w;
  unsigned short value;
  caddr_t       nulldata;
{
   Boolean boolval, selected = False;
   short *sstyle;

   if (areastruct.selects > 0) 
      for (sstyle = areastruct.selectlist; sstyle < areastruct.selectlist
	   + areastruct.selects; sstyle++) {
	 short stype = SELECTTYPE(sstyle);
	 if (stype == ARC || stype == POLYGON || stype == SPLINE || stype == PATH) {
	    short *estyle;
	    switch (stype) {
	       case ARC:
	          estyle = &((SELTOARC(sstyle))->style);
		  break;
	       case SPLINE:
	          estyle = &((SELTOSPLINE(sstyle))->style);
		  break;
	       case POLYGON:
	          estyle = &((SELTOPOLY(sstyle))->style);
		  break;
	       case PATH:
	          estyle = &((SELTOPATH(sstyle))->style);
		  break;
	    }

	    XSetFunction(dpy, areastruct.gc, GXcopy);
	    XSetForeground(dpy, areastruct.gc, BACKGROUND);
	    easydraw(*sstyle, DOFORALL);
	    if (value & UNCLOSED) {
	       boolval = (*estyle & UNCLOSED) ? True : False;
	       *estyle &= ~UNCLOSED;
	       *estyle |= (boolval ? NORMAL : UNCLOSED);
	    }
	    else if (value & OPAQUE) {
	       boolval = (*estyle & OPAQUE) ? True : False;
	       *estyle &= ~OPAQUE;
	       *estyle |= (boolval ? NORMAL : OPAQUE);
	    }
	    else {
	       if (value & BORDERS || value == NORMAL) *estyle &= ~BORDERS;
	       else *estyle &= ~ALLFILLS;
	       if (value != FILLSOLID) *estyle |= value;
	       if ((*estyle & NOBORDER) && !(*estyle & FILLED)) {
		  Wprintf("Must have either a border or filler");
		  *estyle &= ~NOBORDER;
		  value &= ~NOBORDER;
	       }
	    }
	    XSetFunction(dpy, areastruct.gc, GXxor);
	    XSetForeground(dpy, areastruct.gc, SELECTCOLOR ^ BACKGROUND);
	    easydraw(*sstyle, DOFORALL);
	    selected = True;
	 }
      }
   if (!selected) {
      if (value & UNCLOSED) {
         boolval = (areastruct.style & UNCLOSED) ? True : False;
	 areastruct.style &= ~UNCLOSED;
	 areastruct.style |= (boolval ? NORMAL : UNCLOSED);
      }
      else if (value & OPAQUE) {
         boolval = (areastruct.style & OPAQUE) ? True : False;
	 areastruct.style &= ~OPAQUE;
	 areastruct.style |= (boolval ? NORMAL : OPAQUE);
      }
      else {
	 short savestyle = areastruct.style;

         if (value & BORDERS || value == NORMAL)
	    areastruct.style &= ~BORDERS;
         else areastruct.style &= ~ALLFILLS;
         if (value != FILLSOLID) areastruct.style |= value;

	 if ((areastruct.style & NOBORDER) && !(areastruct.style & FILLED)) {
	    Wprintf("Must have either a border or filler");
	    areastruct.style = savestyle;
	    return;
	 }
      }
   }
   if (w)
      setstylemarks(w, (short)value, boolval);
   else
      setallstylemarks(areastruct.style);

   objectdeselect();
}

/*----------------------------------------------*/

void getset(w, value, nulldata)
  Widget        w;
  unsigned short value;
  caddr_t       nulldata;
{
   getfill(w, value, nulldata);
}

/*----------------------------------------------*/

void getline(w, value, nulldata)
  Widget        w;
  unsigned short value;
  caddr_t       nulldata;
{
   getfill(w, value, nulldata);
}

/*----------------------------------------------*/

void getopaque(w, value, nulldata)
  Widget	w;
  unsigned short value;
  caddr_t       nulldata;
{
   getfill(w, value, nulldata);
}
   
/*-------------------------*/
/* Color handling routines */
/*-------------------------*/

void setcolor(w, value, nulldata)
  Widget        w; 
  unsigned short value;
  caddr_t       nulldata;
{
   short *scolor, *ecolor;
   int cindex;
   Arg wargs[1];
   Boolean selected = False;

   /* Get the color index value from the widget itself */

   if (value == 1)
      cindex = -1;
   else {
      XtSetArg(wargs[0], XtNrectColor, &cindex);
      XtGetValues(w, wargs, 1);
   }
   

   if (areastruct.selects > 0)
      for (scolor = areastruct.selectlist; scolor < areastruct.selectlist
	   + areastruct.selects; scolor++) {
	 ecolor = &(SELTOCOLOR(scolor));

	 *ecolor = cindex;
	 selected = True;
      }

   if (!selected) {
      areastruct.color = cindex;      
   }

   toggleexcl(w, Colors, XtNumber(Colors));
   objectdeselect();
}

/*--------------------------------------------------------------*/

void setnewcolor(w, nullptr)
  Widget  w;
  caddr_t *nullptr;
{
   int ccolor, red, green, blue;
   char *ppos;

   if ((ppos = strchr(_STR2, '#')) != 0) {
      if (strlen(ppos + 1) == 6) {
	 sscanf(ppos + 1, "%2x%2x%2x", &red, &green, &blue);
	 red *= 256;
	 green *= 256;
	 blue *= 256;
      }
      else if (sscanf(ppos + 1, "%4x%4x%4x", &red, &green, &blue) != 3) {
	 Wprintf("Bad color entry.  Use #rrggbb");
	 return;
      }
      rgb_alloccolor(red, green, blue);
   }
   else
      ccolor = xc_alloccolor(_STR2);

   addnewcolorentry(ccolor);
}

/*--------------------------------------------------------------*/

void addnewcolor(w, value, nulldata)
  Widget        w;
  unsigned short value;
  caddr_t       nulldata;
{
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));

   getgeneric(savebutton, w, addnewcolor, NULL);
   popupprompt(w, "Enter color name or #rgb:", "\0", setnewcolor,
	savebutton, False);
}

/*-----------------*/
/* Set label style */
/*-----------------*/

void setfontmarks(fvalue, jvalue)
  short fvalue, jvalue;
{
   Widget w;

   if (fvalue >= FONTS) {
      w = XtNameToWidget(top, FontStylesNormalPath);
      toggleexcl(w, FontStyles, XtNumber(FontStyles));
      w = XtNameToWidget(XtNameToWidget(top, FontsPath), fonts[fvalue]);
      toggleexcl(w, Fonts, XtNumber(Fonts));
   }
   else if (fvalue >= 0) {
      switch(fvalue % 4) {
         case 0: w = XtNameToWidget(top, FontStylesNormalPath); break;
         case 1: w = XtNameToWidget(top, FontStylesItalicPath); break;
         case 2: w = XtNameToWidget(top, FontStylesBoldPath); break;
         case 3: w = XtNameToWidget(top, FontStylesBoldItalicPath); break;
      }
      toggleexcl(w, FontStyles, XtNumber(FontStyles));

      switch(fvalue / 4) {
         case 0: w = XtNameToWidget(top, FontsTimesRomanPath); break;
         case 1: w = XtNameToWidget(top, FontsHelveticaPath); break;
         case 2: w = XtNameToWidget(top, FontsCourierPath); break;
         case 3: w = XtNameToWidget(top, FontsSymbolPath); break;
      }
      toggleexcl(w, Fonts, XtNumber(Fonts));
   }
   if (jvalue >= 0) {
      switch(jvalue & (NOTLEFT | RIGHT)) {
         case NORMAL: w = XtNameToWidget(top, JustifsLeftJustifiedPath); break;
         case NOTLEFT: w = XtNameToWidget(top, JustifsCenterJustifiedPath); break;
         case RIGHT|NOTLEFT: w = XtNameToWidget(top, JustifsRightJustifiedPath); break;
      }
      toggleexcl(w, Justifs, XtNumber(Justifs));

      switch(jvalue & (NOTBOTTOM | TOP)) {
         case NORMAL: w = XtNameToWidget(top, JustifsBottomJustifiedPath); break;
         case NOTBOTTOM: w = XtNameToWidget(top, JustifsMiddleJustifiedPath); break;
         case TOP|NOTBOTTOM: w = XtNameToWidget(top, JustifsTopJustifiedPath); break;
      }
      toggleexcl(w, Justifs, XtNumber(Justifs));
   }
}

void setdefaultfontmarks()
{
   setfontmarks(areastruct.psfont, areastruct.justify);
}

/*------------------------------------------*/

void setjust(w, value, settext, mode)
  Widget        w;
  unsigned short value;
  labelptr settext;
  short mode;
{
   short newjust;

   if (settext != NULL) {

      if (mode == 1)
         newjust = (settext->justify & (TOP | NOTBOTTOM)) | value;
      else
         newjust = (settext->justify & (RIGHT|NOTLEFT)) | value;
      undrawtext(settext);
      settext->justify = newjust;
      redrawtext(settext);
   }
   else {
      if (mode == 1)
         newjust = (areastruct.justify & (TOP | NOTBOTTOM)) | value;
      else
         newjust = (areastruct.justify & (RIGHT|NOTLEFT)) | value;
      areastruct.justify = newjust;
   }
   if (w != NULL) toggleexcl(w, Justifs, XtNumber(Justifs));
}

void setvjust(w, value, nulldata)
  Widget        w;
  unsigned short value;
  caddr_t       nulldata;
{
   short *fselect;
   labelptr settext;
   short labelcount = 0;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      settext = *((labelptr *)CONDLASTPART(TEXT3_MODE));
      setjust(w, value, settext, 2);
   }
   else {
      for (fselect = areastruct.selectlist; fselect < areastruct.selectlist +
            areastruct.selects; fselect++) {
         if (SELECTTYPE(fselect) == LABEL) {
            labelcount++;
            settext = SELTOLABEL(fselect);
            setjust(NULL, value, settext, 2);
         }
      }
      if (labelcount == 0) setjust(w, value, NULL, 0);
      else objectdeselect();
   }
}

void sethjust(w, value, nulldata)
  Widget        w;
  unsigned short value;
  caddr_t       nulldata;
{
   short *fselect;
   labelptr settext;
   short labelcount = 0;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      settext = *((labelptr *)CONDLASTPART(TEXT3_MODE));
      setjust(w, value, settext, 1);
   }
   else {
      for (fselect = areastruct.selectlist; fselect < areastruct.selectlist +
            areastruct.selects; fselect++) {
         if (SELECTTYPE(fselect) == LABEL) {
            labelcount++;
            settext = SELTOLABEL(fselect);
            setjust(NULL, value, settext, 1);
         }
      }
      if (labelcount == 0) setjust(w, value, NULL, 0);
      else objectdeselect();
   }
}

/*---------------------------*/
/* Set polygon editing style */
/*---------------------------*/

void boxedit(w, value, nulldata)
  Widget        w;
  unsigned short value;
  caddr_t       nulldata;
{
   if (w == NULL) {
      Widget parent = XtNameToWidget(top, BoxEditStylesPath);
      switch (value) {
         case RHOMBOIDX: w = XtNameToWidget(parent, "Rhomboid X");
		 break;
	 case RHOMBOIDY: w = XtNameToWidget(parent, "Rhomboid Y");
		 break;
	 case NORMAL: w = XtNameToWidget(parent, "Normal");
		 break;
      }
   }

   if (areastruct.boxedit == value) return;

   toggleexcl(w, BoxEditStyles, XtNumber(BoxEditStyles));
   areastruct.boxedit = value;
}

/*-------------------------*/
/* Set font name and style */
/*-------------------------*/

void addnewfont(w, value, nulldata)
  Widget	w;
  unsigned short value;
  caddr_t	nulldata;
{
   buttonsave *savebutton = (buttonsave *)malloc(sizeof(buttonsave));

   if (fontcount == (255 - FONT_START))
      Wprintf("Cannot add any more fonts.  Sorry.");
   else {
      fonts = (char **) realloc (fonts, (fontcount + 1) * sizeof(char *));
      fonts[fontcount] = (char *) malloc (sizeof(char));
      fonts[fontcount][0] = '\0';
      getgeneric(savebutton, w, addnewfont, fonts[fontcount]);
      popupprompt(w, "Enter font name:", fonts[fontcount], setnewfont, savebutton,
		False);
   }
}

/*-------------------------------------------------*/

void setfontval(w, value, settext)
  Widget w;
  unsigned short value;
  labelptr settext;
{
   short style, newfont;
   short i, tc;

   if (settext != NULL) {

      /* if last byte was a font designator, use it */

      if (textpos > 0) {
	 if ((short)*(settext->string + textpos - 2) == TEXT_ESC
		 && (tc = (short)*(settext->string + textpos - 1) - FONT_START) >= 0) {
	    style = (tc < FONTS) ? tc % 4 : 0;
	    if (value == 12 && style > 1) {	/* no symbol-bold styles */
	       Wprintf("Font not available");
	       return;
	    }
	    undrawtext(settext);
	    *(settext->string + textpos - 1) = (value + style) + FONT_START;
	    redrawtext(settext);
	    if (w != NULL) {
	       charreport(settext);
   	       toggleexcl(w, Fonts, XtNumber(Fonts));
	    }
	    return;
         }
      }
      if (textpos < strlen(settext->string)) {
	 if ((short)*(settext->string + textpos - 1) == TEXT_ESC
		 && (tc = (short)*(settext->string + textpos) - FONT_START) >= 0) {
	    style = (tc < FONTS) ? tc % 4 : 0;
	    if (value == 12 && style > 1) {
	       Wprintf("Font not available");
	       return;
	    }
	    undrawtext(settext);
	    *(settext->string + textpos) = (value + style) + FONT_START;
	    redrawtext(settext);
	    if (w != NULL) {
	       charreport(settext);
   	       toggleexcl(w, Fonts, XtNumber(Fonts));
	    }
	    return;
	 }
      }
      /* otherwise, look for the last style used in the string */
      for (i = textpos - 2; i >= 0; i--)
	 if ((short)*(settext->string + i - 1) == TEXT_ESC
		&& (tc = (short)*(settext->string + i) - FONT_START) >= 0) {
	    style = (tc < FONTS) ? tc % 4 : 0;
	    if (value == 12 && style > 1) {
	       Wprintf("Font not available");
	       return;
	    }
	    break;
	 }
   }
   else style = (areastruct.psfont < FONTS) ? areastruct.psfont % 4 : 0;

   if (value == 12 && style > 1) {
      Wprintf("Font not available");
      return;
   }
   newfont = value + style;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      sprintf(_STR, "Font is now %s", fonts[newfont]);
      labeltext((unsigned short)newfont + FONT_START, 2);
   }
   else {
      sprintf(_STR, "Default font is now %s", fonts[newfont]);
      areastruct.psfont = newfont;
   }

   if (w != NULL) toggleexcl(w, Fonts, XtNumber(Fonts));

   Wprintf(_STR);
}

/*-------------------------------------------------*/

void setfont(w, value, nulldata)
  Widget	w;
  unsigned short value;
  caddr_t	nulldata;
{
   short *fselect;
   labelptr settext;
   short labelcount = 0;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      settext = *((labelptr *)CONDLASTPART(TEXT3_MODE));
      setfontval(w, value, settext);
   }
   else {
      textpos = 1;
      for (fselect = areastruct.selectlist; fselect < areastruct.selectlist +
	    areastruct.selects; fselect++) {
         if (SELECTTYPE(fselect) == LABEL) {
	    labelcount++;
	    settext = SELTOLABEL(fselect);
            setfontval(NULL, value, settext);
         }
      }
      if (labelcount == 0) setfontval(w, value, NULL);
      else objectdeselect();
   }
}

/*-------------------------------------------------*/

void setfontstyle(w, value, settext)
  Widget w;
  unsigned short value;
  labelptr settext;
{
   short mfont, newfont;
   short i, tc;

   if (settext != NULL) {

      /* if last byte was a font designator, use it */

      if (textpos > 0) {
         if ((short)*(settext->string + textpos - 2) == TEXT_ESC
              && (tc = (short)*(settext->string + textpos - 1) - FONT_START) >= 0) {
	    if (tc >= FONTS) {
	       Wprintf("Cannot set style for non-predefined fonts");
	       return;
	    }
            mfont = tc & 0xfc;
	    if (mfont == 12 && value > 1) {
	       Wprintf("Font not available");
	       return;
	    }
	    undrawtext(settext);
            *(settext->string + textpos - 1) = (mfont + value) + FONT_START;
	    redrawtext(settext);
	    if (w != NULL) {
	       charreport(settext);
	       toggleexcl(w, FontStyles, XtNumber(FontStyles));
	    }
            return;
         }
      }
      if (textpos < strlen(settext->string)) {
	 if ((short)*(settext->string + textpos - 1) == TEXT_ESC
	       && (tc = (short)*(settext->string + textpos) - FONT_START) >= 0) {
	    if (tc >= FONTS) {
	       Wprintf("Cannot set style for non-predefined fonts");
	       return;
	    }
	    mfont = tc & 0xfc;
	    if (mfont == 12 && value > 1) {
	       Wprintf("Font not available");
	       return;
	    }
	    undrawtext(settext);
	    *(settext->string + textpos) = (mfont + value) + FONT_START;
	    redrawtext(settext);
	    if (w != NULL) {
	       charreport(settext);
	       toggleexcl(w, FontStyles, XtNumber(FontStyles));
	    }
	    return;
	 }
      }
      /* otherwise, look for the last font used in the string */
      for (i = textpos - 2; i >= 0; i--) {
	 if (((tc = (short)*(settext->string + i - 1)) == TEXT_ESC)
	       && ((tc = (short)*(settext->string + i) - FONT_START)) >= 0) {
            mfont = (tc < FONTS) ? tc & 0xfc : tc;
	    if (mfont == 12 && value > 1) {
	       Wprintf("Font not available");
	       return;
	    }
            break;
	 }
      }
   }
   else
      mfont = (areastruct.psfont < FONTS) ? areastruct.psfont & 0xfc :
		areastruct.psfont;

   newfont = mfont + value;
   if (mfont == 12 && value > 1) {
      Wprintf("Font not available");
      return;
   }
   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      labeltext((unsigned short)newfont + FONT_START, 2);
      sprintf(_STR, "Font is now %s", fonts[newfont]);
   }
   else {
      areastruct.psfont = newfont;
      sprintf(_STR, "Default font is now %s", fonts[newfont]);
   }
   Wprintf(_STR);

   if (w != NULL) toggleexcl(w, FontStyles, XtNumber(FontStyles));
}

/*-------------------------------------------------*/

void fontstyle(w, value, nulldata)
  Widget	w;
  unsigned short value;
  caddr_t	nulldata;
{
   short *fselect;
   labelptr settext;
   short labelcount = 0;

   if (eventmode == TEXT2_MODE || eventmode == TEXT3_MODE) {
      settext = *((labelptr *)CONDLASTPART(TEXT3_MODE));
      setfontstyle(w, value, settext);
   }
   else {
      textpos = 1;
      for (fselect = areastruct.selectlist; fselect < areastruct.selectlist +
	    areastruct.selects; fselect++) {
         if (SELECTTYPE(fselect) == LABEL) {
	    labelcount++;
	    settext = SELTOLABEL(fselect);
            setfontstyle(NULL, value, settext);
         }
      }
      if (labelcount == 0) setfontstyle(w, value, NULL);
      else objectdeselect();
   }
}

/*-------------------------------------------------*/
/* Set default font from string name		   */
/*-------------------------------------------------*/

void topsetfont(name)
   char *name;
{
   int i;
   Widget buttonw = NULL, fontmenu = XtNameToWidget(top, FontsPath);
   Widget stylemenu = XtNameToWidget(top, FontStylesPath);
   short style;

   for (i = 0; i < fontcount; i++)
      if (!strcmp(name, fonts[i])) break;

   setfontval(NULL, (unsigned short) i, NULL);
   setfontstyle(NULL, (unsigned short) i, NULL);
   setdefaultfontmarks();
}

/*-------------------------------------------------*/
/* Subscript and superscript font styles	   */
/*-------------------------------------------------*/

void subscript(w, value, nulldata)
  Widget	w;
  unsigned short value;
  caddr_t	nulldata;
{
   if (eventmode != TEXT2_MODE && eventmode != TEXT3_MODE) return;
   labeltext(value, 2);
}

/*-----------------------------------------*/
/* Print help list                         */
/*-----------------------------------------*/

void starthelp(button, clientdata, nulldata)
  Widget button;
  caddr_t *clientdata;
  caddr_t nulldata;
{
    Arg		wargs[8];
    Widget	popup, cancelbutton, hspace, help2;
    short 	n = 0;
    popupstruct *okaystruct;
    buttonsave  *savebutton = (buttonsave *)malloc(sizeof(buttonsave));

    /*  If called from main window, redirect call to the proper button */

    if (button == areastruct.area) button = XtNameToWidget(top, OptionMenuHelpPath);

    getgeneric(savebutton, button, starthelp, NULL);

    popup = XtCreatePopupShell("help", transientShellWidgetClass,
	button, wargs, 0);
    popups++;
    help_up = True;

    XtnSetArg(XtNlayout, XwIGNORE);
    help2 = XtCreateManagedWidget("help2", XwbulletinWidgetClass,
	popup, wargs, n); n = 0;

    XtnSetArg(XtNborderWidth, 0);
    hspace = XtCreateManagedWidget("HSpace", XwworkSpaceWidgetClass,
	help2, wargs, n); n = 0;

    XtnSetArg(XtNx, 40);
    XtnSetArg(XtNfont, areastruct.textstruct);
    cancelbutton = XtCreateManagedWidget("Done", XwmenuButtonWidgetClass,
	help2, wargs, n); n = 0;

    okaystruct = (popupstruct *) malloc(sizeof(popupstruct));
    okaystruct->buttonptr = savebutton;
    okaystruct->popup = popup;

    XtPopup(popup, XtGrabNone);

    XtAddCallback(cancelbutton, XtNselect, destroypopup, okaystruct); 
    XtAddCallback(hspace, XtNexpose, printhelp, popup);
}

#define NENTRIES 44

void printhelp(hspace, popup, calldata)
  Widget hspace, popup;
  caddr_t *calldata;
{
   Window hwin = XtWindow(hspace);

   static char *helptitle = "Short Command Summary:";
   static char *helptext[NENTRIES * 2] = {
    "Z",        "Zoom in 3/2",
    "z",        "Zoom out 3/2",
    "p",        "Pan to center",
    "<arrows>", "Half-page pan U,D,R,L",
    "+/-",      "Change snap-to spacing",
    "l",        "Go To Library",
    ">",        "Push object",
    "<",        "Pop object",
    "<space>",  "Refresh screen",
    "0--9",     "Go To Page",
    " ",	" ",
    "<Button1>","Polygon (fast click)",
    "b",        "Box",
    "a",        "Arc",
    "t",        "Text",
    "s",        "Spline",
    "m",        "Make object",
    "j",        "Join (make path)",
    "J",        "Un-join",
    " ",	" ",
    "d",        "Delete",
    "u",        "Undelete",
    "<Button1>","Move (when held down)",
    "<Button2>","Select",
    "x",        "Deselect",
    "c",        "Copy",
    "e",        "Edit",
    " ",	" ",
    "R",        "Rotate counterclockwise",
    "r",        "Rotate clockwise",
    "O",        "5 degree ccw rotate",
    "o",        "5 degree cw rotate",
    "f/F",      "Flip object horiz/vert",
    "S",        "Snap to grid",
    "A",	"Attach to",
    "| / : / _","Dashed/Dotted/Solid",
    " ",	" ",
    "<Keypad>", "Text Justification",
    "<KP +/->", "Super/Subscript",
    " ",	" ",
    "E",	"Library name edit",
    "M",	"Library move object",
    "D",	"Library object delete",
    "C",	"Library copy object"};

    Arg		wargs[8];
    Widget	bboard = XtNameToWidget(popup, "help2");
    Widget	button = XtNameToWidget(bboard, "Done");
    Dimension	bwidth, theight, mtmp, mwidth, mwidth2, mheight;
    Dimension   areawidth;
    Position	xpos, ypos;
    XEvent	discard;
    XGCValues	values;
    short i, n = 0;
    int t1, t2, dum;
    unsigned int xmax, ymax, posx, posy;
    XCharStruct csdum;
    XFontStruct *tmpstruct;
    popupstruct *okaystruct;
    buttonsave	*savebutton = (buttonsave *)malloc(sizeof(buttonsave));

    tmpstruct = XLoadQueryFont(dpy, "-*-helvetica-medium-r-normal--*");

    mwidth = mwidth2 = mheight = 0;

    for (i = 0; i < NENTRIES; i += 2) {
       mtmp = XTextWidth(tmpstruct, helptext[i], strlen(helptext[i]));
       if (mtmp > mwidth) mwidth = mtmp;
       mtmp = XTextWidth(tmpstruct, helptext[i+1], strlen(helptext[i+1]));
       if (mtmp > mwidth2) mwidth2 = mtmp;
    }
    for (i = 0; i < NENTRIES; i ++) {
       XTextExtents(tmpstruct, helptext[i], strlen(helptext[i]), &dum, &t1,
	   &t2, &csdum);
       mtmp = t1 + t2 + 5;
       if (mtmp > mheight) mheight = mtmp;
    }
    XTextExtents(tmpstruct, helptitle, strlen(helptitle), &dum, &t1,
	&t2, &csdum);
    t1 += t2;
    mwidth2 += mwidth + 15;
    mtmp = mheight;
    mheight = mheight * NENTRIES + 15 + t1;

    /* set help window to side of xcircuit window, but not outside of   */
    /* display borders							*/

    xmax = DisplayWidth(dpy, DefaultScreen(dpy));
    ymax = DisplayHeight(dpy, DefaultScreen(dpy));

    XtnSetArg(XtNwidth, &areawidth);
    XtGetValues(areastruct.area, wargs, n); n = 0;
    XtTranslateCoords(areastruct.area, (Position) (areawidth + 10), -50,
	&xpos, &ypos);
    if (xpos + mwidth2 > xmax) xpos = xmax - mwidth2 - 4;
    if (ypos + mheight > ymax) ypos = ymax - mheight - 4;

    XtnSetArg(XtNx, xpos);
    XtnSetArg(XtNy, ypos);
    XtnSetArg(XtNwidth, mwidth2);
    XtnSetArg(XtNheight, mheight + t1 + 15);
    XtSetValues(popup, wargs, n); n = 0;

    XtnSetArg(XtNwidth, mwidth2);
    XtnSetArg(XtNheight, mheight + t1 + 15);
    XtSetValues(bboard, wargs, n); n = 0;

    XtnSetArg(XtNwidth, mwidth2);
    XtnSetArg(XtNheight, mheight);
    XtSetValues(hspace, wargs, n); n = 0;

    XtnSetArg(XtNy, mheight);
    XtSetValues(button, wargs, n); n = 0;

    if (hgc == NULL) {
       values.foreground = FOREGROUND;
       values.font = tmpstruct->fid;
       hgc = XCreateGC(dpy, hwin, GCForeground | GCFont, &values);
    }

    XDrawString(dpy, hwin, hgc, (mwidth2 - XTextWidth(tmpstruct, helptitle,
	strlen(helptitle))) >> 1, t1 + 2, helptitle, strlen(helptitle));
    for (i = 0; i < NENTRIES; i++) {
	XDrawString(dpy, hwin, hgc, 7, (i + 1) * mtmp + 15, helptext[i << 1],
		strlen(helptext[i << 1]));
	XDrawString(dpy, hwin, hgc, 7 + mwidth, (i + 1) * mtmp + 15,
		helptext[1 + (i << 1)], strlen(helptext[1 + (i << 1)]));
    }
    XSetForeground(dpy, hgc, AUXCOLOR);
    XDrawLine(dpy, hwin, hgc, 0, t1 + 7, mwidth2, t1 + 7);

    /* flush out multiple expose events */

    while (XCheckWindowEvent(dpy, hwin, ExposureMask, &discard) == True);
}

/*-------------------------------------------------*/
/* Print a string to the message widget. 	   */ 
/* Note: Widget message must be a global variable. */
/* For formatted strings, format first into _STR   */
/*-------------------------------------------------*/

void clrmessage(id, clientdata)
  XtIntervalId	id;
  caddr_t	clientdata;
{
   Wprintf(" ");
}

void W1printf(string)
char *string;
{
   Arg	wargs[1];

   XtSetArg(wargs[0], XtNstring, string);
   XtSetValues(message1, wargs, 1);
   XtSetArg(wargs[0], XtNheight, ROWHEIGHT);
   XtSetValues(message1, wargs, 1);
}   

void W2printf(string)
char *string;
{
   Arg	wargs[1];

   XtSetArg(wargs[0], XtNstring, string);
   XtSetValues(message2, wargs, 1);
   XtSetArg(wargs[0], XtNheight, ROWHEIGHT);
   XtSetValues(message2, wargs, 1);
}   

void Wprintf(string)
char *string;
{
   Arg	wargs[1];

   if (printtime_id != 0) {
      XtRemoveTimeOut(printtime_id);
      printtime_id = 0;
   }
   XtSetArg(wargs[0], XtNstring, string);
   XtSetValues(message3, wargs, 1);
   XtSetArg(wargs[0], XtNheight, ROWHEIGHT);
   XtSetValues(message3, wargs, 1);

   printtime_id = XtAddTimeOut(10000, clrmessage, NULL); /* 10 second timeout */
}   

/*------------------------------------------------------------------------------*/
/* Find the nearest color in the colormap to cvexact and return its pixel value */
/*------------------------------------------------------------------------------*/

int findnearcolor(cvexact)
   XColor *cvexact;
{
   /* find the nearest-matching color in the colormap */
 
   int i, ncolors = DisplayCells(dpy, DefaultScreen(dpy));
   XColor *cmcolors;
   long rdist, bdist, gdist;
   unsigned long dist, mindist;
   int minidx;
   Colormap cmap = DefaultColormap (dpy, DefaultScreen(dpy));

   cmcolors = (XColor *)malloc(ncolors * sizeof(XColor));

   for (i = 0; i < ncolors; i++) {
      cmcolors[i].pixel = i;
      cmcolors[i].flags = DoRed | DoGreen | DoBlue;
   }
   XQueryColors(dpy, cmap, cmcolors, ncolors); 

   mindist = MAXLONG;
   for (i = 0; i < ncolors; i++) {
      rdist = (cmcolors[i].red - cvexact->red);
      bdist = (cmcolors[i].blue - cvexact->blue);
      gdist = (cmcolors[i].green - cvexact->green);
      dist = rdist * rdist + bdist * bdist + gdist * gdist;
      if (dist < mindist) {
	 mindist = dist;
	 minidx = i;
      }
   }
   free(cmcolors);
   return minidx;
}

/*-------------------------------------------------------------------------*/
/* Hack on the resource database Color allocation			   */
/*									   */
/* This overrides the built-in routine.  The difference is that if a	   */
/* color cell cannot be allocated (colormap is full), then the color	   */
/* map is searched for the closest RGB value to the named color.	   */
/*									   */
/* This code depends on Display *dpy being set:  Do not install the	   */
/* converter until after XtInitialize().				   */
/*-------------------------------------------------------------------------*/

void CvtStringToPixel(args, nargs, fromVal, toVal)
   XrmValuePtr args, fromVal, toVal;
   int	*nargs;
{
   static XColor cvcolor;
   XColor cvexact;
   Colormap cmap;

   if (dpy == NULL) return;
   cmap = DefaultColormap (dpy, DefaultScreen(dpy));

   if (*nargs != 0)
      XtWarning("String to Pixel conversion needs no arguments");

   /* Color defaults to black if name is not found */

   if (XAllocNamedColor(dpy, cmap, (char *)fromVal->addr, &cvcolor, &cvexact)
	 == 0) {
      if (XLookupColor(dpy, cmap, (char *)fromVal->addr, &cvexact, &cvcolor) == 0)
	 cvcolor.pixel = BlackPixel(dpy, DefaultScreen(dpy));
      else
         cvcolor.pixel = findnearcolor(&cvexact);
   }

   toVal->size = sizeof(unsigned long);
   toVal->addr = (caddr_t) &(cvcolor.pixel);
}

/*-------------------------------------------------------------------------*/
/* Allocate new color using the CvtStringToPixel routine		   */
/*-------------------------------------------------------------------------*/

int xc_alloccolor(name)
  char *name;
{
   XrmValue fromC, toC;
   int zval = 0, i, pixval;

   fromC.size = strlen(name);
   fromC.addr = name;

   CvtStringToPixel(NULL, &zval, &fromC, &toC);
   pixval = (int)(*((unsigned long *)toC.addr));

   return pixval;
}

/*-------------------------------------------------------------------------*/
/* Allocate new color from RGB values (e.g., from PS file "scb" command)   */
/*-------------------------------------------------------------------------*/

int rgb_alloccolor(red, green, blue)
  int red, green, blue;
{
   XColor newcolor;
   int i, pixval = -1;
   Colormap cmap = DefaultColormap (dpy, DefaultScreen(dpy));

   /* first check if color within RGB roundoff error exists in the list */

   for (i = 0; i < number_colors; i++) {
      if (abs(colorlist[i].color.red - red) < 200 &&
	     abs(colorlist[i].color.green - green) < 200 &&
	     abs(colorlist[i].color.blue - blue) < 200) {
	 pixval = colorlist[i].color.pixel;
	 break;
      }
   }

   /* if color is not already in list, try to allocate it; if allocation */
   /* fails, grab the closest match in the colormap.			 */

   if (pixval < 0) {
      newcolor.red = red;
      newcolor.green = green;
      newcolor.blue = blue;
      newcolor.flags = DoRed | DoGreen | DoBlue;
      if (XAllocColor(dpy, cmap, &newcolor) == 0)
         pixval = findnearcolor(&newcolor);
      else
	 pixval = newcolor.pixel;
   }
   return pixval;
}

/*-------------------------------------------------------------------------*/
/* Make the cursors from the cursor bit data				   */
/*-------------------------------------------------------------------------*/

void makecursors()
{
   Colormap cmap = DefaultColormap (dpy, DefaultScreen(dpy));
   XColor fgcolor, bgcolor;

   bgcolor.pixel = BACKGROUND;
   fgcolor.pixel = FOREGROUND;
   XQueryColor(dpy, cmap, &fgcolor);
   XQueryColor(dpy, cmap, &bgcolor);

   ARROW = XCreatePixmapCursor(dpy, XCreateBitmapFromData(dpy, win, arrow_bits,
	arrow_width, arrow_height), XCreateBitmapFromData(dpy, win, arrowmask_bits,
	arrowmask_width, arrowmask_height), &fgcolor, &bgcolor, arrow_x_hot,
	arrow_y_hot);
   CROSS = XCreatePixmapCursor(dpy, XCreateBitmapFromData(dpy, win, cross_bits,
	cross_width, cross_height), XCreateBitmapFromData(dpy, win, crossmask_bits,
	crossmask_width, crossmask_height), &fgcolor, &bgcolor, cross_x_hot,
	cross_y_hot);
   SCISSORS = XCreatePixmapCursor(dpy, XCreateBitmapFromData(dpy, win, scissors_bits,
	scissors_width, scissors_height), XCreateBitmapFromData(dpy, win,
	scissorsmask_bits, scissorsmask_width, scissorsmask_height), &fgcolor, 
	&bgcolor, scissors_x_hot, scissors_y_hot);
   EDCURSOR = XCreatePixmapCursor(dpy, XCreateBitmapFromData(dpy, win, exx_bits,
	exx_width, exx_height), XCreateBitmapFromData(dpy, win, exxmask_bits,
	exxmask_width, exxmask_height), &fgcolor, &bgcolor, exx_x_hot, exx_y_hot);
   COPYCURSOR = XCreatePixmapCursor(dpy, XCreateBitmapFromData(dpy, win, copy_bits,
	copy_width, copy_height), XCreateBitmapFromData(dpy, win, copymask_bits,
	copymask_width, copymask_height), &fgcolor, &bgcolor, copy_x_hot, copy_y_hot);
   QUESTION = XCreatePixmapCursor(dpy, XCreateBitmapFromData(dpy, win, question_bits,
	question_width, question_height), XCreateBitmapFromData(dpy, win,
	questionmask_bits, questionmask_width, questionmask_height),
	&fgcolor, &bgcolor, question_x_hot, question_y_hot);
   CIRCLE = XCreatePixmapCursor(dpy, XCreateBitmapFromData(dpy, win, circle_bits,
	circle_width, circle_height), XCreateBitmapFromData(dpy, win, circlemask_bits,
	circlemask_width, circlemask_height), &fgcolor, &bgcolor, circle_x_hot,
	circle_y_hot);
   TEXTPTR = XCreateFontCursor(dpy, XC_xterm);
   XRecolorCursor(dpy, TEXTPTR, &fgcolor, &bgcolor);
}

/*-------------------------------------------------------------------------*/
/* Main program 							   */
/*-------------------------------------------------------------------------*/

#define entries(c)	(sizeof c / sizeof (menustruct))

main(argc, argv) 
  int argc; 
  char *argv[]; 
{ 
   Widget form, lastbutton, corner;
   XColor color, ignore;
   Colormap cmap;
   XGCValues    values;
   XWMHints	*wmhints;	/* for proper input focus */
   Arg		wargs[12];
   short i, j, n = 0, maxbuttons, maxcolors;
   objectptr	*page;

   /*---------------------------*/
   /* initialize user variables */
   /*---------------------------*/

   areastruct.psfont = 4;
   areastruct.justify = 0;
   areastruct.page = 0;
   areastruct.pages = PAGES;

   areastruct.filename = (char **)malloc(PAGES * sizeof(char *));
   areastruct.wirewidth = (float *)malloc(PAGES * sizeof(float));
   areastruct.outscale = (float *)malloc(PAGES * sizeof(float));
   areastruct.viewscale = (float *)malloc(PAGES * sizeof(float));
   areastruct.orient = (short *)malloc(PAGES * sizeof(short));
   areastruct.pmode = (short *)malloc(PAGES * sizeof(short));
   areastruct.drawingscale = (XPoint *)malloc(PAGES * sizeof(XPoint));
   areastruct.coordstyle = (short *)malloc(PAGES * sizeof(short));
   areastruct.pcorner = (XPoint *)malloc(PAGES * sizeof(XPoint));
   areastruct.pagesize = (XPoint *)malloc(PAGES * sizeof(XPoint));

   areastruct.wirewidth[0] = 2.0;
   areastruct.outscale[0] = 1.0;
   areastruct.viewscale[0] = 0.5;
   areastruct.pmode[0] = 0;
   areastruct.orient[0] = 0;
   areastruct.drawingscale[0].x = areastruct.drawingscale[0].y = 1;
   areastruct.coordstyle[0] = FRAC_INCH;
   areastruct.pagesize[0].x = 612;
   areastruct.pagesize[0].y = 792;

   areastruct.vscale = &areastruct.viewscale[0];
   areastruct.lowerleft = &areastruct.pcorner[0];  /* values set below */
   areastruct.textscale = 1.0;
   areastruct.style = UNCLOSED;
   areastruct.invert = False;
   areastruct.axeson = True;
   areastruct.snapspace = 16;
   areastruct.snapto = True;
   areastruct.gridon = True;
   areastruct.gridspace = 32;
   areastruct.selects = 0;
   areastruct.selectlist = NULL;
   areastruct.manhatn = False;
   areastruct.boxedit = MANHATTAN;
   areastruct.builtins = 0;
   areastruct.userobjs = 0;
   areastruct.library = (objectptr *) malloc(sizeof(objectptr));
   areastruct.userlib = (objectptr *) malloc(sizeof(objectptr));
   areastruct.editstack = (objectptr) malloc(sizeof(object));
   areastruct.tempvscale = 0.5;
   areastruct.hlevels = 0;
   areastruct.hierarchy = (objinstptr *) malloc(sizeof(objinstptr));
   areastruct.deletes = 0;
   areastruct.delbuffer = (objectptr *) malloc(DELBUFSIZE * sizeof(objectptr));
   areastruct.pagelist = (objectptr *) malloc(PAGES * sizeof(objectptr));
   for (page = areastruct.pagelist; page < areastruct.pagelist + PAGES; page++)
      *page = NULL;

   /* Set up the predefined fonts */

   fonts = (char **) malloc(FONTS * sizeof(char *));
   for (i = 0; i < FONTS; i++)
      fonts[i] = prefonts[i];
   fontcount = FONTS;

   initmem(areastruct.editstack);
   for (i = 0; i < LIBS; i++) {
      areastruct.libtop[i] = (objectptr) malloc(sizeof(object));
      initmem(areastruct.libtop[i]);
   }
   strcpy(areastruct.libtop[1]->name, "Built-in Library");
   strcpy(areastruct.libtop[0]->name, "User Library");

   areastruct.topobject = (objinstptr) malloc(sizeof(objinst));
   *(areastruct.pagelist) = &areaobject;
   areastruct.topobject->thisobject = &areaobject;
   areastruct.topobject->scale = 1.0;
   areastruct.topobject->rotation = 1;
   areastruct.topobject->position.x = 0;
   areastruct.topobject->position.y = 0;
   *areastruct.hierarchy = areastruct.topobject;

   /* note: objectdata is defined as areastruct.topobject->thisobject */
   sprintf(objectdata->name, "Page 1");
   areastruct.filename[0] = (char *)malloc((strlen(objectdata->name) + 1)
	* sizeof(char));
   strcpy(areastruct.filename[0], objectdata->name);
   objectdata->width = 0;
   objectdata->height = 0;
   objectdata->lowerleft.x = 0;
   objectdata->lowerleft.y = 0;
#ifdef SCHEMA
   objectdata->schematic = NULL;
#endif
   initmem(objectdata);

   popups = 0;        /* no popup windows yet */
   pushes = 0;        /* at the top of the hierarchy */
   beeper = 1;        /* Ring bell on certain warnings or errors */

   sscanf(VERSION, "%f", &version);  /* version of xcircuit */

   initsplines();	/* create lookup table of spline parameters */

   /*-----------------------------*/
   /* Create the widget hierarchy */
   /*-----------------------------*/

   top = XtInitialize(argv[0], "XCircuit", NULL, 0, &argc, argv);
   dpy = XtDisplay(top);

   /*-------------------------*/
   /* Create stipple patterns */
   /*-------------------------*/

   win = DefaultRootWindow(dpy);
   for (i = 0; i < STIPPLES; i++)
      STIPPLE[i] = XCreateBitmapFromData(dpy, win, STIPDATA[i], 4, 4);

   /*----------------------------------------*/
   /* Allocate space for the basic color map */
   /*----------------------------------------*/

   number_colors = 0;
   colorlist = (colorindex *)malloc(sizeof(colorindex));
   appcolors = (int *) malloc(NUMBER_OF_COLORS * sizeof(int));

   /*-------------------------------*/
   /* Get the application resources */
   /*-------------------------------*/

   XtAddConverter(XtRString, XtRPixel, CvtStringToPixel, NULL, 0);
   XtGetApplicationResources(top, &appdata, resources, XtNumber(resources), 
	NULL, 0);
   areastruct.textstruct = appdata.xcfont;

   XtnSetArg(XtNwidth, appdata.width);
   XtnSetArg(XtNheight, appdata.height);
   XtnSetArg(XtNforeground, appdata.fg);
   XtnSetArg(XtNbackground, appdata.bg);
   XtSetValues(top, wargs, n); n = 0;
   form = XtCreateManagedWidget("Form", XwformWidgetClass, top, NULL, 0);

   /* Set up the buttons and Graphics drawing area */

   maxbuttons = entries(TopButtons);
   lastbutton = createmenus(TopButtons, maxbuttons, form);

   XtnSetArg(XtNxRefWidget, lastbutton);
   XtnSetArg(XtNyRefWidget, form);
   XtnSetArg(XtNxAddWidth, True);
   XtnSetArg(XtNxAttachRight, True);
   XtnSetArg(XtNheight, ROWHEIGHT);
   sprintf(_STR, "   Welcome to Xcircuit Version %s", VERSION);
   XtnSetArg(XtNstring, _STR);
   XtnSetArg(XtNxResizable, True);
   XtnSetArg(XtNgravity, WestGravity);
   XtnSetArg(XtNfont, appdata.xcfont);
   XtnSetArg(XtNwrap, False);
   XtnSetArg(XtNstrip, False);
   message1 = XtCreateManagedWidget("Message1", XwstaticTextWidgetClass,
	form, wargs, n); n = 0;

   /*------------------------*/
   /* add scrollbar widget   */
   /*------------------------*/

   XtnSetArg(XtNyRefWidget, lastbutton);
   XtnSetArg(XtNyAddHeight, True);
   XtnSetArg(XtNxResizable, False);
   XtnSetArg(XtNwidth, SBARSIZE);
   XtnSetArg(XtNyResizable, True);
   XtnSetArg(XtNborderWidth, 1);
   areastruct.scrollbarv = XtCreateManagedWidget("SBV", XwworkSpaceWidgetClass, 
	form, wargs, n); n = 0;
   
   /*-----------------------------*/
   /* For the sake of beauty. . . */
   /*-----------------------------*/

   XtnSetArg(XtNyRefWidget, areastruct.scrollbarv);
   XtnSetArg(XtNyAddHeight, True);
   XtnSetArg(XtNyAttachBottom, True);
   XtnSetArg(XtNyAttachOffset, ROWHEIGHT + 4);
   XtnSetArg(XtNheight, SBARSIZE);
   XtnSetArg(XtNwidth, SBARSIZE);
   XtnSetArg(XtNborderWidth, 1);
   corner = XtCreateManagedWidget("corner", widgetClass,
        form, wargs, n); n = 0;

   /*-------------------------*/
   /* The main drawing window */
   /*-------------------------*/

   XtnSetArg(XtNyRefWidget, lastbutton);
   XtnSetArg(XtNyAddHeight, True);
   XtnSetArg(XtNyAttachBottom, True);
   XtnSetArg(XtNyAttachOffset, ROWHEIGHT + SBARSIZE + 3);
   XtnSetArg(XtNxOffset, SBARSIZE + 1);
   XtnSetArg(XtNxAttachRight, True);
   XtnSetArg(XtNxResizable, True);
   XtnSetArg(XtNyResizable, True);
   areastruct.area = XtCreateManagedWidget("Area", XwworkSpaceWidgetClass, 
	form, wargs, n); n = 0;

   /*-------------------------*/
   /* And the other scrollbar */
   /*-------------------------*/

   XtnSetArg(XtNyRefWidget, areastruct.area);
   XtnSetArg(XtNyAddHeight, True);
   XtnSetArg(XtNxOffset, SBARSIZE + 1);
   XtnSetArg(XtNxAttachRight, True);
   XtnSetArg(XtNyResizable, False);
   XtnSetArg(XtNheight, SBARSIZE);
   XtnSetArg(XtNxResizable, True);
   XtnSetArg(XtNborderWidth, 1);
   areastruct.scrollbarh = XtCreateManagedWidget("SBH", XwworkSpaceWidgetClass, 
	form, wargs, n); n = 0;

   /*------------------------------------------------*/
   /* Supplementary message widgets go at the bottom */
   /*------------------------------------------------*/

   XtnSetArg(XtNyRefWidget, areastruct.scrollbarh);
   XtnSetArg(XtNyAddHeight, True);
   XtnSetArg(XtNyOffset, -2);
   XtnSetArg(XtNheight, ROWHEIGHT);
   XtnSetArg(XtNstring, "Editing: Page 1");
   XtnSetArg(XtNxResizable, False);
   XtnSetArg(XtNgravity, WestGravity);
   XtnSetArg(XtNfont, appdata.xcfont);
   XtnSetArg(XtNwrap, False);
   message2 = XtCreateManagedWidget("Message2", XwstaticTextWidgetClass,
	form, wargs, n); n = 0;

   XtnSetArg(XtNyRefWidget, areastruct.scrollbarh);
   XtnSetArg(XtNyAddHeight, True);
   XtnSetArg(XtNyOffset, -2);
   XtnSetArg(XtNxAttachRight, True);
   XtnSetArg(XtNxRefWidget, message2);
   XtnSetArg(XtNxAddWidth, True);
   XtnSetArg(XtNheight, ROWHEIGHT);
   XtnSetArg(XtNstring, "Don't Panic");
   XtnSetArg(XtNxResizable, True);
   XtnSetArg(XtNgravity, WestGravity);
   XtnSetArg(XtNfont, appdata.xcfont);
   XtnSetArg(XtNwrap, False);
   message3 = XtCreateManagedWidget("Message3", XwstaticTextWidgetClass,
	form, wargs, n); n = 0;

   /* Setup callback routines for the area widget */
   /* Use Button1Press event to add the callback which tracks motion;  this */
   /*   will reduce the number of calls serviced during normal operation */ 

   XtAddEventHandler(areastruct.area, ButtonPressMask, False, selectbutton, 
	NULL);
   XtAddEventHandler(areastruct.area, ButtonReleaseMask, False, releasebutton, 
	NULL);
   XtAddEventHandler(areastruct.area, KeyPressMask, False, keyhandler, 
	NULL);

   XtAddCallback(areastruct.area, XtNexpose, drawarea, NULL);
   XtAddCallback(areastruct.area, XtNresize, resizearea, NULL);

   /* Setup callback routines for the scrollbar widgets */

   XtAddEventHandler(areastruct.scrollbarh, ButtonPressMask, False, panhbar, 
	NULL);
   XtAddEventHandler(areastruct.scrollbarv, ButtonPressMask, False, panvbar, 
	NULL);

   XtAddCallback(areastruct.scrollbarh, XtNexpose, drawhbar, NULL);
   XtAddCallback(areastruct.scrollbarv, XtNexpose, drawvbar, NULL);
   XtAddCallback(areastruct.scrollbarh, XtNresize, drawhbar, NULL);
   XtAddCallback(areastruct.scrollbarv, XtNresize, drawvbar, NULL);

   /*--------------------*/
   /* Realize the Widget */
   /*--------------------*/

   XtRealizeWidget(top);

   /* Define basic display variables */
   /* Redefine win to be just the drawing area window */

   win = XtWindow(areastruct.area);

   /*--------------------------------------------------*/
   /* Setup the (simple) colormap and make the cursors */
   /*--------------------------------------------------*/

   cmap = DefaultColormap (dpy, DefaultScreen(dpy));
   setcolorscheme(True);
   areastruct.color = DEFAULTCOLOR;

   /*-----------------------------*/
   /* Create the Graphics Context */
   /*-----------------------------*/

   values.foreground = BlackPixel(dpy, DefaultScreen(dpy));
   values.background = WhitePixel(dpy, DefaultScreen(dpy));
   values.font = areastruct.textstruct->fid;

   areastruct.gc = XCreateGC(dpy, win, GCForeground | GCBackground | GCFont, &values);

   /* Setup a passive grab on the area widget for mouse button 1 */

   XGrabButton(dpy, Button1, AnyModifier, win, True, ButtonPressMask |
	ButtonMotionMask | ButtonReleaseMask, GrabModeAsync, GrabModeAsync,
	win, ARROW);
   XGrabButton(dpy, Button2, AnyModifier, win, True, ButtonPressMask |
	ButtonMotionMask | ButtonReleaseMask, GrabModeAsync, GrabModeAsync,
	win, CIRCLE);


   /* set the area widget width and height, center userspace (0,0) on screen */

   XtSetArg(wargs[0], XtNwidth, &areastruct.width);
   XtSetArg(wargs[1], XtNheight, &areastruct.height);
   XtGetValues(areastruct.area, wargs, 2);
   areastruct.lowerleft->x = -areastruct.width;
   areastruct.lowerleft->y = -areastruct.height;

   areastruct.MatStackDepth = 1; /* Top transformation matrix */
   areastruct.MatStack = (Matrix *)malloc(sizeof(Matrix)); /* original allocation */
   UResetCTM(DCTM);
   UMakeWCTM(DCTM);  		 /* Find TM for window */

   /* set the input focus for the window */

   wmhints = XAllocWMHints();
   wmhints->flags |= InputHint;
   wmhints->input = True;
   XSetWMHints(dpy, XtWindow(top), wmhints);
   XFree(wmhints);

#ifdef DOUBLEBUFFER
   if (dbuf == NULL)
      dbuf = XCreatePixmap(dpy, win, areastruct.width, areastruct.height,
	      DefaultDepthOfScreen(XtScreen(areastruct.area)));
#endif

   /*----------------------------------------------------------*/
   /* Check home directory for initial settings	& other loads; */
   /* Load the (default) built-in set of objects 	       */
   /*----------------------------------------------------------*/

   loadrcfile();

   /*-----------------------------------------------------*/
   /* Set the cursor as a crosshair for the area widget.  */
   /*-----------------------------------------------------*/

   XDefineCursor (dpy, win, CROSS);

   /*-------------------------------------------------*/
   /* Parse the command line for initial file to load */
   /*-------------------------------------------------*/

   if (argc == 2) {
      sprintf(_STR2, argv[1]);
      startloadfile();
   }

   /*----------------------*/
   /* Start the event loop */
   /*----------------------*/

   XtMainLoop();
}

/*-------------------------------------------------------------------------*/
