#include <X11/StringDefs.h>
#include <X11/Intrinsic.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Toggle.h>

#include <string.h>
#if !STDC_HEADERS && HAVE_MEMORY_H
# include <memory.h>
#endif /* not STDC_HEADERS and HAVE_MEMORY_H */

#include <YShell.h>
#include <mymalloc.h>
#include <except.h>
#include <myxlib.h>

#include "connect.h"
#include "events.h"
#include "games.h"
#include "gointer.h"
#include "players.h"
#include "tell.h"
#include "utils.h"
#include "xgospel.h"

#ifdef    HAVE_NO_MEMCHR_PROTO
extern void *memchr(const void *s, int c, size_t n);
#endif /* HAVE_NO_MEMCHR_PROTO */

extern Tell   **PlayerTellAddress(Player *player);

struct _Tell {
    char         *FileName;
    Player       *Player;
    Widget        TopWidget, Info, Input, MessageBeep, BugBeep, ErrorBeep;
    Widget        MessageRaise, BugRaise, ErrorRaise;
    Widget        Overwrite;
};

static MyContext Context;
static XrmQuark  TellQuark;

static void SendTell();
static XtActionsRec actionTable[] = {
    { "tell",        SendTell   },
};

static void ChangeTellFilename(Widget w,
                               XtPointer clientdata, XtPointer calldata)
{
    char **Filename;
    Widget Root;

    Filename = (char **) clientdata;
    Root = ChangeFilename(toplevel, "tellFilename",
                          "Enter name of tell file",
                          Filename, "filename", *Filename, NULL);
    MyDependsOn(Root, w);
}

static void SaveTell(Widget w, XtPointer clientdata, XtPointer calldata)
{
    Boolean                   Overwrite;
    static char const * const ErrorType = "Tell save error";

    Tell *tell;

    tell = (Tell *) clientdata;
    if (tell->Info) {
        w = tell->Overwrite;
        if (w) XtVaGetValues(w, XtNstate, (XtArgVal) &Overwrite, NULL);
        else   Overwrite = False;

        SaveWrite(tell->FileName, Overwrite,
                  tell->ErrorBeep, tell->ErrorRaise, ErrorType,
                  SaveTextFun, (XtPointer) tell->Info);
    } else {
        IfBell(tell->ErrorBeep);
        IfRaise(tell->ErrorRaise, tell->ErrorRaise);
        PopMessage(ErrorType, "Messages were ignored so there is "
                   "nothing to be saved");
    }
}

void InitTell(Widget Toplevel)
{
    XtAppAddActions(XtWidgetToApplicationContext(Toplevel),
                    actionTable, XtNumber(actionTable));
    Context = YShellContext(Toplevel);
    TellQuark = XrmPermStringToQuark("tell");
}

static void CallDestroyTell(Widget w, XtPointer clientdata, XtPointer calldata)
{
    Tell *tell, **Ref;
    Player *player;

    player = (Player *) clientdata;
    Ref    =  PlayerTellAddress(player);
    tell   = *Ref;

    if (tell->Input) MyDeleteContext(Context, tell->Input, TellQuark);
    myfree(tell->FileName);
    myfree(tell);
   *Ref = NULL;
}

static void SendBug(Widget w, XtPointer clientdata, XtPointer calldata)
{
    const Player *player;

    player = (const Player *) clientdata;
    SendCommand(NULL, NULL, "bug %s", PlayerToName(player));
}

void SetTellDescription(Tell *tell)
{
    char *Name;

    SetPlayerTitles(tell->TopWidget, tell->Player);
    Name = StringToFilename(appdata.TellFilename,
                            (int) 'T', "tell",
                            (int) 't', "_",
                            (int) 'N', PlayerToName(tell->Player),
                            (int) 'n', PlayerToStrength(tell->Player),
                            0);
    myfree(tell->FileName);
    tell->FileName = Name;
}

void TellMessage(Tell *tell, const char *Message)
{
    if (tell->Info) BatchAddText(tell->Info, ".....     : %s\n", Message);
    IfBell(tell->MessageBeep);
    IfRaise(tell->MessageRaise, tell->MessageRaise);
}

/* Failure behaviour is incorrect -Ton */
Tell *FindTell(const char *Name)
{
    Tell       *tell, **Ref;
    Widget      Root, Collect, Input, Info, Buttons, MessageBeep, MessageRaise;
    Widget      Bug, File, Save, BugBeep, BugRaise, ErrorBeep, ErrorRaise;
    Widget      Overwrite, Stats;
    Player     *player;

    player = NameToPlayer(Name);
    Ref    = PlayerTellAddress(player);
    if (*Ref) return *Ref;

    *Ref = tell = mynew(Tell);
    tell->FileName = NULL;
    tell->Input    = 0;
    tell->Player   = player;

    WITH_HANDLING {
        Root = MyVaCreateManagedWidget("tell", toplevel, NULL);
        XtAddCallback(Root, XtNdestroyCallback, CallDestroyTell,
                      (XtPointer) player);
        tell->TopWidget = Root;
        SetTellDescription(tell);
 
        Collect      = XtNameToWidget(Root, "*collect");
        Buttons      = XtNameToWidget(Root, "*buttons");
        Bug          = XtNameToWidget(Root, "*bug");
        Stats        = XtNameToWidget(Root, "*getStats");
        Info         = XtNameToWidget(Root, "*info");
        Input        = XtNameToWidget(Root, "*input");
        MessageBeep  = XtNameToWidget(Root, "*beep");
        MessageRaise = XtNameToWidget(Root, "*raise");
        BugBeep      = XtNameToWidget(Root, "*bugBeep");
        BugRaise     = XtNameToWidget(Root, "*bugRaise");
        ErrorBeep    = XtNameToWidget(Root, "*errorBeep");
        ErrorRaise   = XtNameToWidget(Root, "*errorRaise");
        File         = XtNameToWidget(Root, "*file");
        Save         = XtNameToWidget(Root, "*save");
        Overwrite    = XtNameToWidget(Root, "*overwrite");

        if (Input)
            MySaveContext(Context, Input, TellQuark, (XtPointer) player);

        tell->Info         = Info;
        tell->Input        = Input;
        tell->MessageBeep  = MessageBeep;
        tell->MessageRaise = MessageRaise;
        tell->BugBeep      = BugBeep;
        tell->BugRaise     = BugRaise;
        tell->ErrorBeep    = ErrorBeep;
        tell->ErrorRaise   = ErrorRaise;
        tell->Overwrite    = Overwrite;

        if (Bug) XtAddCallback(Bug, XtNcallback, SendBug, (XtPointer) player);
        if (Stats) XtAddCallback(Stats,XtNcallback, GetStats,
                                 (XtPointer) PlayerToName(player));
        if (Save) XtAddCallback(Save, XtNcallback, SaveTell, (XtPointer) tell);
        if (File) XtAddCallback(File, XtNcallback, ChangeTellFilename,
                                (XtPointer) &tell->FileName);

        if (Collect && Input) XtSetKeyboardFocus(Collect, Input);
        MyRealizeWidget(Root);
    } ON_EXCEPTION {
        CallDestroyTell(0, (XtPointer) player, NULL);
    } END_HANDLING;

    return tell;
}

void GetTell(Widget w, XtPointer clientdata, XtPointer calldata)
{
    FindTell((char *) clientdata);
}

static void RealSendTell(const char *Text, int Length, Widget ShowInfo,
                         const char *Name, const char *MyName,
                         int UseSay, const Player *Opponent)
{
    if (UseSay) UserSendCommand(NULL, NULL, "say %.*s", Length, Text);
    else UserSendCommand(NULL, NULL, "tell %s %.*s", Name, Length, Text);
    if (Opponent) AddMyGameComment("%10s says: %.*s", MyName, Length, Text);
    if (ShowInfo) BatchAddText(ShowInfo, "%10s: %.*s\n", MyName, Length, Text);
}

static void SendTell(Widget w, XEvent *event, String *string, Cardinal *n)
{
    XtPointer     Data;
    Tell         *tell;
    const Player *player, *Opponent;
    String        Buffer;
    const char   *Name, *MyName, *From, *To, *End;
    int           UseSay;
    Widget        ShowInfo;

    if (MyFindContext(Context, w, TellQuark, &Data))
        WidgetWarning(w, "tell() could not get context");
    else {
        player = (const Player *) Data;
        Name   = PlayerToName(player);
        MyName = PlayerToName(Me);
        tell = *PlayerTellAddress((Player *) player);
        UseSay = SayP(player);
        Opponent = MyOpponent(player);
        if (player != Me) ShowInfo = tell->Info;
        else              ShowInfo = 0;
        XtVaGetValues(w, XtNstring, &Buffer, NULL);
        End = strchr(Buffer, 0);
        for (From = Buffer;
             (To = memchr((char *)From, '\n', End-From)) != NULL;
             From = To+1)
            RealSendTell(From, To-From,
                         ShowInfo, Name, MyName, UseSay, Opponent);
        RealSendTell(From, End-From,
                     ShowInfo, Name, MyName, UseSay, Opponent);
        XtVaSetValues(w, XtNstring, "", NULL);
        return;
    }
}

void NoTell(void)
{
    const char *Name, *Message;
    Tell       *tell;
    Player     *player;

    Name = StripArgsCommand(NULL, "tell");
    if (Name && *Name) {
        Message = strchr(Name, ' ');
        if (Message) *(char *) Message++ = 0;
        else Message = "";
        player = PlayerFromName(Name);
        if (!GotNoTell(player, Message)) {
            tell = FindTell(Name);
            if (tell->Info)
                BatchAddText(tell->Info,
                             ".....     : Tell not sent, player is gone\n");
            IfBell(tell->ErrorBeep);
            IfRaise(tell->ErrorBeep, tell->ErrorBeep);
        }
    } else Warning("Recipient of tell message is gone\n");
}

void ReceivedTell(const char *Name, const char *Message)
{
    Tell   *tell;
    Player *player;

    player = NameToPlayer(Name);
    if (!GotTell(player, Message)) {
        tell = FindTell(Name);
        if (tell->Info) BatchAddText(tell->Info, "%10s: %s\n", Name, Message);
        IfBell(tell->MessageBeep);
        IfRaise(tell->MessageRaise, tell->MessageRaise);
    }
}

void Bugging(const char *Name)
{
    Tell    *tell;

    tell = FindTell(Name);
    if (tell->Info) BatchAddText(tell->Info, ".....     : You are bugged\n");
    IfBell(tell->BugBeep);
    IfRaise(tell->BugRaise, tell->BugRaise);
}
