 #include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/AsciiText.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

#include <mymalloc.h>
#include <except.h>
#include <myxlib.h>

#include "GoBoard.h"
#include "SmeBell.h"

#include "analyze.h"
#include "broadcast.h"
#include "connect.h"
#include "events.h"
#include "games.h"
#include "gointer.h"
#include "gospel.h"
#include "messages.h"
#include "observe.h"    
#include "players.h"
#include "reviews.h"
#include "tell.h"
#include "utils.h"
#include "version.h"
#include "xgospel.h"

#define DEBUGFUN              1
#define DEBUGPENDING          2
#define DEBUGABORTONEXCEPTION 4
#define DEBUGROUNDTRIP        8
#define DEBUGBISON        0x100 
#define DEBUGFLEX         0x200 

/*
#define VERSIONMESSAGE                                                       \
"\nThe latest version of the program should be available for anonymous ftp " \
"on cc1.kuleuven.ac.be in the directory ANONYMOU.200 as the file "           \
"XGOSPEL.TARZ. Just get the file by name, you don't want to do a dir in "    \
"this place (in case you ever wondered why people invented directories, "    \
"DO try dir :-) ). Don't forget to set the transfermode to binary (cc1 is "  \
"an EBCDIC system). Once you have the file, rename it to xgospel.tar.Z and " \
"proceed as usual.\n"                                                        \
"\nIf you don't want this message anymore, set the X resource "              \
"xgospel*version: %s\n"
*/

#define VERSIONMESSAGE                                                       \
"\nThe latest version of the program should be available for anonymous ftp " \
"on bsdserver.ucsf.edu in the directory Go/clients as the file "             \
"xgospel???.tar.Z . Don't forget to set the transfermode to binary.\n"       \
"\nIf you don't want this message anymore, set the X resource "              \
"xgospel*version: %s\n"

/* #define SITE "bsdserver.ucsf.edu" */
#define SITE    "hellspark.wharton.upenn.edu"

/* We measure the roundtrip time (to the X server) every RETRIP seconds */
#define RETRIP    5
#define NEWOUTPUT 3
/*****************************************************************************/

static Widget Info, Input, Stdout, Localtime, Universaltime, Servertime;
static Widget ErrorBeep, ErrorRaise, OutputBeep, OutputRaise, Overwrite, Main;
static char  *MainFileName;
static String DateFormat;
static unsigned long TripTime, LastTrip, RoundTrip, LastRoundTrip;
static unsigned long OutputTime;
static Atom   TripAtom;
static int    DebugRoundTrip;
char         *SgfDateFormat;
static void   IgsCommand(), IgsUserCommand(), IgsSendCommand(), TripMessage();
static XtActionsRec actionTable[] = {
    { "igscommand",  IgsCommand     },
    { "usercommand", IgsUserCommand },
    { "sendcommand", IgsSendCommand },
    { "tripmessage", TripMessage },
};

AppData         appdata;
XtAppContext	app_context;

FILE               *DebugFile;
int                 DebugFun, DebugPending;
Widget              toplevel;
static int	    global_argc;
static char       **global_argv;
char               *MyPassword, *MyName, *UserId;
int                 SetServerTime, Entered;
Boolean             WantStdout;
struct tm           LocalTime, UniversalTime, ServerTime;

/*****************************************************************************/

extern String fallback_resources[];
extern int    RealQuit;
extern int    IgsYYdebug, IgsYY_flex_debug;

extern void   IgsRestartParse(void);

void IgsPrompt(void)
{
    Output("> ");
}

static void DebugFunCommand(const char *args)
{
    char *ptr;
    int   value;

    while (*args == ' ') args++;
    value = strtol(args, &ptr, 0);
    if (ptr == args) Outputf("DebugFun remains 0x%02x\n", appdata.DebugFun);
    else {
        Outputf("DebugFun changed from 0x%02x to 0x%02x\n",
                appdata.DebugFun, value);
        appdata.DebugFun = value;
        DebugFun         = (value & DEBUGFUN)       != 0;
        DebugPending     = (value & DEBUGPENDING)   != 0;
        AbortOnException = (value & DEBUGABORTONEXCEPTION) ? -1 : 0;
        DebugRoundTrip   = (value & DEBUGROUNDTRIP) != 0;
        IgsYYdebug       = (value & DEBUGBISON)     != 0;
        IgsYY_flex_debug = (value & DEBUGFLEX)      != 0;
    }
}

static void ReplayRateCommand(const char *args)
{
    char *ptr;
    int   value;

    while (*args == ' ') args++;
    value = strtol(args, &ptr, 0);
    if (ptr == args)
        Outputf("Replay rate remains %d\n", appdata.ReplayTimeout);
    else {
        Outputf("Replay rate changed from %d to %d\n",
                appdata.ReplayTimeout, value);
        appdata.ReplayTimeout = value;
    }
}

static void IgsCommand(Widget w, XEvent *event, String *string, Cardinal *n)
{
    String          Buffer;
    int             Prompt;

    XtVaGetValues(w, XtNstring, &Buffer, NULL);
/*  BatchAppendText(Info, Buffer, strlen(Buffer), 0); */
    Prompt = 1;
    while (isspace(*Buffer)) Buffer++;
    if      (!strncmp(Buffer, "dumpgames"  , 9)) DumpGames(Buffer+9);
    else if (!strncmp(Buffer, "dumpplayers",11)) DumpPlayers(Buffer+11);
    else if (!strncmp(Buffer, "debugfun"   , 8)) DebugFunCommand(Buffer+8);
    else if (!strncmp(Buffer, "replayrate" ,10)) ReplayRateCommand(Buffer+10);
    else if (!strncmp(Buffer, "xgospeltell",11))
        if (Buffer[11]) TellXgospelUsers(NULL, Buffer+12);
        else TellXgospelUsers(NULL, "");
    else {
        Prompt = 0;
        if (!strncmp(Buffer, "review", 6)) ReviewListWanted(NULL, 2);
        else UserCommand(NULL, "%s", Buffer);
    }
    if (Prompt) IgsPrompt();
    XtVaSetValues(Input, XtNstring, "", NULL);
}

static void IgsUserCommand(Widget w, XEvent *event,
                           String *string, Cardinal *n)
{
    if (*n == 1) UserCommand(NULL, "%s", string[0]);
    else WidgetWarning(w, "usercommand() expects exactly 1 argument");
}

static void IgsSendCommand(Widget w, XEvent *event,
                           String *string, Cardinal *n)
{
    if (*n == 1) SendCommand(NULL, NULL, "%s", string[0]);
    else WidgetWarning(w, "sendcommand() expects exactly 1 argument");
}

int Output(const char *Text)
{
    int Length;

    if (Entered && !OutputTime) {
        OutputTime = NEWOUTPUT;
        IfBell(OutputBeep);
        IfRaise(OutputRaise, OutputRaise);
    }

    Length = strlen(Text);
    BatchAppendText(Info, Text, Length, 0);
    return Length;
}

#ifndef HAVE_NO_STDARG_H
int Outputf(const char *Comment, ...)
#else  /* HAVE_NO_STDARG_H */
int Outputf(va_alist)
va_dcl
#endif /* HAVE_NO_STDARG_H */
{
    char    Text[2048];
    va_list args;

    if (Entered && !OutputTime) {
        OutputTime = NEWOUTPUT;
        IfBell(OutputBeep);
        IfRaise(OutputRaise, OutputRaise);
    }

#ifndef HAVE_NO_STDARG_H
    va_start(args, Comment);
#else  /* HAVE_NO_STDARG_H */
    const char *Comment;

    va_start(args);
    Comment = va_arg(args, const char *);
#endif /* HAVE_NO_STDARG_H */
    vsprintf(Text, Comment, args);
    va_end(args);
    return Output(Text);
}

static void TestBoard(Widget w, XtPointer clientdata, XtPointer calldata)
{
    OpenAnalyze(0, "", 0, appdata.AnalyzeSize, 0);
}

/* Very simpleminded writability test. God help you if you are root */
static Boolean TryToWrite(String Name)
{
    DebugFile = fopen(Name, "w");
    if (DebugFile) return True;
    else           return False;
}

/*****************************************************************************/

static time_t        BaseTime;
static unsigned long CurTime;

static const char *DateString(struct tm *date)
{
    static char Buffer[80];
    char       *ptr;
    int         Length;

    if (DateFormat) {
        if (strftime(Buffer, sizeof(Buffer), DateFormat, date))
            return Buffer;
        Warning("strftime(.., %d, %s, ..) overflows the buffer\n",
                    sizeof(Buffer), DateFormat);
        myfree(DateFormat);
        DateFormat = NULL;
    }
    
    /* This code can also be used if your libraries do not have strftime */
    ptr = asctime(date);
    Length = strlen(ptr)-1;
    memcpy(Buffer, ptr, Length);
    Buffer[Length] = 0;
    return Buffer;
}

/* Returns 0 on success, 1 on failure */
static int SendTripMessage(Widget w, long Message)
{
    Display *dpy;
    Window   win;
    XEvent   myEvent;

    dpy = XtDisplay(w);
    win = XtWindow(w);
    myEvent.xclient.type         = ClientMessage;
    myEvent.xclient.display      = dpy;
    myEvent.xclient.window       = win;
    myEvent.xclient.message_type = TripAtom;
    myEvent.xclient.format       = 32;
    myEvent.xclient.data.l[0]    = Message;
    return !XSendEvent(dpy, win, False, NoEventMask, &myEvent);
}

#define MIX        100
#define RESOLUTION 10000
static void TripMessage(Widget w, XEvent *event, String *string, Cardinal *n)
{
    LastTrip  = event->xclient.data.l[0];
    LastRoundTrip = CurTime-LastTrip;
    RoundTrip = ((MIX-1)*RoundTrip+RESOLUTION*LastRoundTrip)/MIX;
    if (DebugRoundTrip) {
        printf("LastRoundTrip = %3lu, RoundTrip = %4lu.%02lu\n",
               LastRoundTrip, RoundTrip/RESOLUTION,
               (100*RoundTrip)/RESOLUTION-100*(RoundTrip/RESOLUTION));
        fflush(stdout);
    }
}

static void HeartBeat(XtPointer closure, XtIntervalId *id)
{
    unsigned long Diff;
    time_t        NewTime;

    if (++UniversalTime.tm_sec < 60) {
        Diff = 1;
        if (++LocalTime.tm_sec >= 60) mktime(&LocalTime);
    } else {
        time(&NewTime);
        LocalTime     = *localtime(&NewTime);
        UniversalTime = *gmtime(   &NewTime);
#ifndef HAVE_NO_DIFFTIME
        Diff = (unsigned long) difftime(NewTime, BaseTime);
#else                           /* HAVE_NO_DIFFTIME */
        Diff = (unsigned long) (NewTime - BaseTime);
#endif                          /* HAVE_NO_DIFFTIME */
        Diff -= CurTime;
    }
    if (Diff) {
        GamesTime(Diff);
        ReviewsTime(Diff);
        PlayersTime(Diff);
        ConnectTime(Diff);
        if (Localtime)     XtVaSetValues(Localtime, XtNstring,
                                         (XtArgVal) DateString(&LocalTime),
                                         NULL);
        if (Universaltime) XtVaSetValues(Universaltime, XtNstring,
                                         (XtArgVal) DateString(&UniversalTime),
                                         NULL);
        if (SetServerTime) {
            if (++ServerTime.tm_sec >= 60) mktime(&ServerTime);
            if (Servertime) XtVaSetValues(Servertime, XtNstring,
                                          (XtArgVal) DateString(&ServerTime),
                                          NULL);
        }
    }
    CurTime += Diff;
    if (CurTime >= TripTime) {
        TripTime = CurTime + RETRIP;
        SendTripMessage(Main, (long) CurTime);
    }
    if (OutputTime)
        if (OutputTime > Diff) OutputTime -= Diff;
        else OutputTime = 0;
    XtAppAddTimeOut((XtAppContext) closure, 1000, HeartBeat, closure);
}

static void InitHeartBeat(XtAppContext appcontext)
{
    TripAtom = GetAtom(Main, "XGOSPEL_TRIP_PROBE");
    time(&BaseTime);
    LocalTime     = *localtime(&BaseTime);
    UniversalTime = *gmtime(   &BaseTime);
    CurTime       = 0;
    SetServerTime = 0;
    TripTime      = LastTrip = OutputTime = 0;
    RoundTrip     = 0;
    XtAppAddTimeOut(appcontext, 1000, HeartBeat, (XtPointer) appcontext);
}

/*****************************************************************************/

#define MAXSIZE 25

static void ConvertSize(XtPointer Closure)
{
    char **Result, *ptr;
    int    Size;

    Result = (char **) Closure;
    Size = strtol(*Result, &ptr, 0);
    if (*ptr || ptr == *Result)
        PopMessage("Analyze size error", "%.40s is not an integer", *Result);
    else if (1 >= Size || Size > MAXSIZE)
        PopMessage("Analyze size error", "Size must be between 2 and %d",
                   MAXSIZE);
    else appdata.AnalyzeSize = Size;
    myfree(*Result);
    *Result = NULL;
}

static void ChangeAnalyzeSize(Widget w,
                              XtPointer clientdata, XtPointer calldata)
{
    Widget        Root;
    char          size[80];
    static char  *Result;
    XtVarArgsList Args;

    Result = NULL;
    sprintf(size, "%d", appdata.AnalyzeSize);
    Args = XtVaCreateArgsList(NULL, "size", size, NULL);
    Root = AskString(toplevel, ConvertSize, &Result,
                     "Enter size of analyze board",
                     "analyzeSize", &Result, Args, NULL);
    XtFree(Args);
    MyDependsOn(Root, w);
}

typedef struct {
    AppDataPtr appData;
    char      *analyzeSize;
    char      *replayTimeout;
    char      *gamesTimeout;
    char      *whoTimeout;
    char      *reviewsTimeout;
    char      *serverTimeout;
    char      *playersTimeout;
    char      *quitTimeout;
    char      *sortPTimeout;
    char      *sortGTimeout;
    char      *tersePlay;
} SettingsType;

static void ConvertPositive(const char *Name, int *Target, XtPointer Closure)
{
    char **Result, *ptr;
    int    Value;

    Result = (char **) Closure;
    if (*Result) {
        Value = strtol(*Result, &ptr, 0);
        if (*ptr || ptr == *Result)
            PopMessage(Name, "%.40s is not an integer", *Result);
        else if (Value < 1) PopMessage(Name, "Value must be greater than 0");
        else *Target = Value;
        myfree(*Result);
        *Result = NULL;
    } else PopMessage(Name, "no value has been entered");
}

static void ConvertNatural(const char *Name, int *Target, XtPointer Closure)
{
    char **Result, *ptr;
    int    Value;

    Result = (char **) Closure;
    if (*Result) {
        Value = strtol(*Result, &ptr, 0);
        if (*ptr || ptr == *Result)
            PopMessage(Name, "%.40s is not an integer", *Result);
        else if (Value < 0)
            PopMessage(Name, "Value must be greater or equal than 0");
        else *Target = Value;
        myfree(*Result);
        *Result = NULL;
    } else PopMessage(Name, "no value has been entered");
}

static int ConvertBoolean(const char *Name, Boolean *Target,
                           XtPointer Closure)
{
    int      rc;
    char   **Result;
    XrmValue src, dst;

    rc = 0;
    Result = (char **) Closure;
    if (*Result) {
        src.size = strlen(*Result)+1;
        src.addr = (XPointer) *Result;
        dst.size = 0;
        dst.addr = NULL;

        if (XtConvertAndStore(toplevel,
                              XtRString, &src, XtRBoolean, &dst) != False) {
            rc = *Target != *(Boolean *) dst.addr;
            *Target = *(Boolean *) dst.addr;
        } else PopMessage(Name, "%.40s is not a boolean", *Result);
        myfree(*Result);
        *Result = NULL;
    } else PopMessage(Name, "no value has been entered");
    return rc;
}

static void ConvertSettings(XtPointer Closure)
{
    SettingsType *Settings;

    Settings = (SettingsType *) Closure;
    ConvertSize((XtPointer) &Settings->analyzeSize);
    ConvertPositive("Replay rate error", &Settings->appData->ReplayTimeout,
                    (XtPointer) &Settings->replayTimeout);
    ConvertPositive("Who rate error", &Settings->appData->WhoTimeout,
                    (XtPointer) &Settings->whoTimeout);
    ConvertPositive("Games rate error", &Settings->appData->GamesTimeout,
                    (XtPointer) &Settings->gamesTimeout);
    ConvertPositive("Reviews rate error", &Settings->appData->ReviewsTimeout,
                    (XtPointer) &Settings->reviewsTimeout);
    ConvertPositive("Server resend rate error",
                    &Settings->appData->ServerTimeout,
                    (XtPointer) &Settings->serverTimeout);
    ConvertNatural ("Player window update rate error",
                    &Settings->appData->PlayerUpdateTimeout,
                    (XtPointer) &Settings->playersTimeout);
    ConvertPositive("quit timeout error", &Settings->appData->QuitTimeout,
                    (XtPointer) &Settings->quitTimeout);
    if (ConvertBoolean ("Player sorting error",
                        &Settings->appData->SortPlayersByStrength,
                        (XtPointer) &Settings->sortPTimeout))
        PlayersResort();
    if (ConvertBoolean("Games sorting error",
                       &Settings->appData->SortGamesByStrength,
                       (XtPointer) &Settings->sortGTimeout))
        GamesResort();
    if (ConvertBoolean("Terse play error",
                       &Settings->appData->TersePlay,
                       (XtPointer) &Settings->tersePlay))
        /* Send games/who if was requested --Ton */ ;
}

static void ChangeSettings(Widget w, XtPointer clientdata, XtPointer calldata)
{
    AppDataPtr    appData;
    Widget        Root;
    char          size[40], replayTimeout[40], whoTimeout[40], quitTimeout[40];
    char          gamesTimeout[40], reviewsTimeout[40], serverTimeout[40];
    char          playersTimeout[40];
    char         *sortPTimeout, *sortGTimeout, *tersePlay;
    XtVarArgsList ArgsAna, ArgsReplay, ArgsWho, ArgsGames, ArgsReviews;
    XtVarArgsList ArgsServer, ArgsPlayers, ArgsQuit, ArgsSortP, ArgsSortG;
    XtVarArgsList ArgsTerse;
    /* Allocate is better as soon as we have more contexts --Ton */
    static SettingsType  Settings;

    appData = (AppDataPtr) clientdata;
    Settings.appData        = appData;
    Settings.analyzeSize    = NULL;
    Settings.replayTimeout  = NULL;
    Settings.whoTimeout     = NULL;
    Settings.gamesTimeout   = NULL;
    Settings.reviewsTimeout = NULL;
    Settings.serverTimeout  = NULL;
    Settings.playersTimeout = NULL;
    Settings.quitTimeout    = NULL;
    Settings.sortPTimeout   = NULL;
    Settings.sortGTimeout   = NULL;
    Settings.tersePlay      = NULL;

    sprintf(size,            "%d", appData->AnalyzeSize);
    ArgsAna     = XtVaCreateArgsList(NULL, "size",    size, NULL);
    sprintf(replayTimeout,  "%d", appData->ReplayTimeout);
    ArgsReplay  = XtVaCreateArgsList(NULL, "timeout", replayTimeout, NULL);
    sprintf(whoTimeout,     "%d", appData->WhoTimeout);
    ArgsWho     = XtVaCreateArgsList(NULL, "timeout", whoTimeout, NULL);
    sprintf(gamesTimeout,   "%d", appData->GamesTimeout);
    ArgsGames   = XtVaCreateArgsList(NULL, "timeout", gamesTimeout, NULL);
    sprintf(reviewsTimeout, "%d", appData->ReviewsTimeout);
    ArgsReviews = XtVaCreateArgsList(NULL, "timeout", reviewsTimeout, NULL);
    sprintf(serverTimeout,  "%d", appData->ServerTimeout);
    ArgsServer  = XtVaCreateArgsList(NULL, "timeout", serverTimeout, NULL);
    sprintf(playersTimeout, "%d", appData->PlayerUpdateTimeout);
    ArgsPlayers = XtVaCreateArgsList(NULL, "timeout", playersTimeout, NULL);
    sprintf(quitTimeout,    "%d", appData->QuitTimeout);
    ArgsQuit    = XtVaCreateArgsList(NULL, "timeout", quitTimeout, NULL);
    sortPTimeout = appData->SortPlayersByStrength != False ? "True" : "False";
    ArgsSortP    = XtVaCreateArgsList(NULL, "sort", sortPTimeout, NULL);
    sortGTimeout = appData->SortGamesByStrength != False ? "True" : "False";
    ArgsSortG    = XtVaCreateArgsList(NULL, "sort", sortGTimeout, NULL);
    tersePlay    = appData->TersePlay != False ? "True" : "False";
    ArgsTerse    = XtVaCreateArgsList(NULL, "boolean", tersePlay, NULL);

    Root = AskString(toplevel, ConvertSettings, &Settings,
                     "Enter new settings",
                     "analyzeSize",    &Settings.analyzeSize,    ArgsAna,
                     "replayTimeout",  &Settings.replayTimeout,  ArgsReplay,
                     "whoTimeout",     &Settings.whoTimeout,     ArgsWho,
                     "gamesTimeout",   &Settings.gamesTimeout,   ArgsGames,
                     "playersTimeout", &Settings.playersTimeout, ArgsPlayers,
                     "reviewsTimeout", &Settings.reviewsTimeout, ArgsReviews,
                     "serverTimeout",  &Settings.serverTimeout,  ArgsServer,
                     "quitTimeout",    &Settings.quitTimeout,    ArgsQuit,
                     "sortPlayersTimeout", &Settings.sortPTimeout, ArgsSortP,
                     "sortGamesTimeout", &Settings.sortGTimeout, ArgsSortG,
                     "tersePlay",      &Settings.tersePlay,      ArgsTerse,
                     NULL);

    XtFree(ArgsAna);
    XtFree(ArgsReplay);
    XtFree(ArgsWho);
    XtFree(ArgsGames);
    XtFree(ArgsServer);
    XtFree(ArgsPlayers);
    XtFree(ArgsQuit);
    XtFree(ArgsSortP);
    XtFree(ArgsSortG);
    XtFree(ArgsTerse);
    MyDependsOn(Root, w);
}

static void ChangeMainFilename(Widget w,
                               XtPointer clientdata, XtPointer calldata)
{
    char **Filename;
    Widget Root;

    Filename = (char **) clientdata;
    Root = ChangeFilename(toplevel, "mainFilename",
                          "Enter name of main file",
                          Filename, "filename", *Filename, NULL);
    MyDependsOn(Root, w);
}

static void SaveMain(Widget w, XtPointer clientdata, XtPointer calldata)
{
    Boolean                   overwrite;
    static char const * const ErrorType = "Main save error";

    if (Info) {
        if (Overwrite) CallReadToggle(Overwrite, (XtPointer) &overwrite, NULL);
        else           overwrite = False;

        SaveWrite(MainFileName, overwrite,
                  ErrorBeep, ErrorRaise, ErrorType,
                  SaveTextFun, (XtPointer) Info);
    } else {
        IfBell(ErrorBeep);
        IfRaise(ErrorRaise, ErrorRaise);
        PopMessage(ErrorType, "Main messages were ignored so there is "
                   "nothing to be saved");
    }
}

static char *MainTitle(const char *Pattern, XtPointer Closure)
{
    return StringToFilename(Pattern,
                            (int) 'N', (char *) Closure,
                            (int) 'V', appdata.Version,
                            0);
}

static XrmOptionDescRec options[] = {
    { "-fg",		"*foreground",	XrmoptionSepArg,	NULL},
    { "-foreground",	"*foreground",	XrmoptionSepArg,	NULL},
    { "-user",          "*user",        XrmoptionSepArg,        NULL},
    { "-password",      "*password",    XrmoptionSepArg,        NULL},
    { "-host",          "*site",        XrmoptionSepArg,        NULL},
    { "-site",          "*site",        XrmoptionSepArg,        NULL},
    { "-port",          "*port",        XrmoptionSepArg,        NULL},
    { "-debugfile",     "*debugFile",   XrmoptionSepArg,        NULL},
    { "-debugfun",      "*debugFun",    XrmoptionSepArg,        NULL},
    { "-debug",		"*debug",	XrmoptionNoArg,		"True"},
};

static const char *UseMessages[] = {
    "    -user       name         Userid on IGS",
    "    -password   secret       Password on IGS",
    "    -host       site         Connect to site",
    "    -site       site         Connect to site",
    "    -port       port         Connect to port",
    "    -debugfile  name         File to log interaction with IGS",
    "    -debugfun   int          Set program debugging flags",
    "    -debug                   Turn on internal widget debugging",
};

#define XtNuser                  "user"
#define XtCUser                  "User"
#define XtNpassword              "password"
#define XtCPassword              "Password"
#define XtNdebugFun              "debugFun"
#define XtCDebugFun              "DebugFun"
#define XtNdebugFile             "debugFile"
#define XtCDebugFile             "DebugFile"
#define XtNkibitzSize            "kibitzSize"
#define XtCKibitzSize            "KibitzSize"
#define XtNtellsize              "tellsize"
#define XtCTellsize              "Tellsize"
#define XtNbroadcastSize         "broadcastSize"
#define XtCBroadcastSize         "BroadcastSize"
#define XtNyellsize              "yellsize"
#define XtCYellsize              "Yellsize"
#define XtCKibitzSize            "KibitzSize"
#define XtNmaintainer            "maintainer"
#define XtCMaintainer            "Maintainer"
#define XtNversion               "version"
#define XtCVersion               "Version"
#define XtNdateFormat            "dateFormat"
#define XtNsgfDateFormat         "sgfDateFormat"
#define XtCDateFormat            "DateFormat"
#define XtNreplayTimeout         "replayTimeout"
#define XtNgamesTimeout          "gamesTimeout"
#define XtNreviewsTimeout        "reviewsTimeout"
#define XtNserverTimeout         "serverTimeout"
#define XtNwhoUpdateTimeout      "whoUpdateTimeout"
#define XtNwhoTimeout            "whoTimeout"
#define XtNquitTimeout           "quitTimeout"
#define XtNreconnectTimeout      "reconnectTimeout"
#define XtCTimeout               "Timeout"
#define XtNgamesScale            "gamesScale"
#define XtNplayersScale          "playersScale"
#define XTCScale                 "Scale"
#define XtNscrollUnit            "scrollUnit"
#define XtCScrollUnit            "ScrollUnit"
#define XtNsortPlayersByStrength "sortPlayersByStrength"
#define XtNsortGamesByStrength   "sortGamesByStrength"
#define XtCSortByStrength        "SortByStrength"
#define XtNsite                  "site"
#define XtCSite                  "Site"
#define XtNport                  "port"
#define XtCPort                  "Port"
#define XtNdirectory             "directory"
#define XtCDirectory             "Directory"
#define XtNanalyzeFilename       "analyzeFilename"
#define XtNsgfFilename           "sgfFilename"
#define XtNpsFilename            "psFilename"
#define XtNkibitzFilename        "kibitzFilename"
#define XtNbroadcastFilename     "broadcastFilename"
#define XtNyellFilename          "yellFilename"
#define XtNigsMessageFilename    "igsMessageFilename"
#define XtNtellFilename          "tellFilename"
#define XtNmessageFilename       "messageFilename"
#define XtNeventsFilename        "eventsFilename"
#define XtNgamesFilename         "gamesFilename"
#define XtNplayersFilename       "playersFilename"
#define XtNmainFilename          "mainFilename"
#define XtCFileFilename          "FileFilename"
#define XtNfriends               "friends"
#define XtCFriends               "Friends"
#define XtNplayersToWidget       "playersToWidget"
#define XtCPlayersToWidget       "PlayersToWidget"
#define XtNanalyzeSize           "analyzeSize"
#define XtCAnalyzeSize           "AnalyzeSize"
#define XtNversionMessage        "versionMessage"
#define XtCVersionMessage        "VersionMessage"
#define XtNtersePlay             "tersePlay"
#define XtCTersePlay             "TersePlay"

#define offset(field) XtOffset(AppDataPtr, field)

static XtResource resources[] = {
    { XtNuser, XtCUser, XtRString, sizeof(String),
      offset(User), XtRString, NULL }, 
    { XtNpassword, XtCPassword, XtRString, sizeof(String),
      offset(Password), XtRString, NULL }, 
    { XtNdebugFile, XtCDebugFile, XtRString, sizeof(String),
      offset(DebugFile), XtRString, NULL }, 
    { XtNdebugFun, XtCDebugFun, XtRInt, sizeof(int),
      offset(DebugFun), XtRString, "0" }, 
    { XtNtellsize, XtCTellsize, XtRInt, sizeof(int),
      offset(TellSize), XtRString, "128" },
    { XtNkibitzSize, XtCKibitzSize, XtRInt, sizeof(int),
      offset(KibitzSize), XtRString, "180" },
    { XtNmaintainer, XtCMaintainer, XtRString, sizeof(String),
      offset(Maintainer), XtRString, "AshaiRey" },
    { XtNversion, XtCVersion, XtRString, sizeof(String),
      offset(Version), XtRString, VERSION },
    { XtNversionMessage, XtCVersionMessage, XtRString, sizeof(String),
      offset(VersionMessage), XtRString, VERSIONMESSAGE },
    { XtNdateFormat, XtCDateFormat, XtRString, sizeof(String),
      offset(DateFormat), XtRString, NULL },
    { XtNsgfDateFormat, XtCDateFormat, XtRString, sizeof(String),
      offset(SgfDateFormat), XtRString, "%Y/%m/%d %H:%M:%S UT" },
    { XtNbroadcastSize, XtCBroadcastSize, XtRInt, sizeof(int),
      offset(BroadcastSize), XtRString, "128" },
    { XtNyellsize, XtCYellsize, XtRInt, sizeof(int),
      offset(YellSize), XtRString, "128" },
    { XtNsortPlayersByStrength, XtCSortByStrength, XtRBoolean, sizeof(Boolean),
      offset(SortPlayersByStrength), XtRString, "True" },    
    { XtNsortGamesByStrength, XtCSortByStrength, XtRBoolean, sizeof(Boolean),
      offset(SortGamesByStrength), XtRString, "True" },
    { XtNtersePlay, XtCTersePlay, XtRBoolean, sizeof(Boolean),
      offset(TersePlay), XtRString, "True" },
    { XtNplayersScale, XTCScale, XtRInt, sizeof(int),
      offset(PlayersScale), XtRString, "20" },
    { XtNgamesScale, XTCScale, XtRInt, sizeof(int),
      offset(GamesScale), XtRString, "5" },
    { XtNscrollUnit, XtCScrollUnit, XtRInt, sizeof(int),
      offset(ScrollUnit), XtRString, "20" },
    { XtNserverTimeout, XtCTimeout, XtRInt, sizeof(int),
      offset(ServerTimeout), XtRString, "120" },
    { XtNwhoTimeout, XtCTimeout, XtRInt, sizeof(int),
      offset(WhoTimeout), XtRString, "300" },
    { XtNwhoUpdateTimeout, XtCTimeout, XtRInt, sizeof(int),
      offset(PlayerUpdateTimeout), XtRString, "20" },
    { XtNreplayTimeout, XtCTimeout, XtRInt, sizeof(int),
      offset(ReplayTimeout), XtRString, "1" },
    { XtNgamesTimeout, XtCTimeout, XtRInt, sizeof(int),
      offset(GamesTimeout), XtRString, "300" },
    { XtNreviewsTimeout, XtCTimeout, XtRInt, sizeof(int),
      offset(ReviewsTimeout), XtRString, "1000" },
    { XtNquitTimeout, XtCTimeout, XtRInt, sizeof(int),
      offset(QuitTimeout), XtRString, "20" },
    { XtNreconnectTimeout, XtCTimeout, XtRInt, sizeof(int),
      offset(ReconnectTimeout), XtRString, "15" },
    { XtNanalyzeSize, XtCAnalyzeSize, XtRInt, sizeof(int),
      offset(AnalyzeSize), XtRString, "19" },
    { XtNdirectory, XtCDirectory, XtRString, sizeof(String),
      offset(Directory), XtRString, "" },
    { XtNanalyzeFilename, XtCFileFilename, XtRString, sizeof(String),
      offset(AnalyzeFilename), XtRString, "%D%N%l%L%U%B%U%b%U%V%U%w%U%W%t%T" },
    { XtNsgfFilename, XtCFileFilename, XtRString, sizeof(String),
      offset(SgfFilename), XtRString, "%D%B%U%b%U%V%U%w%U%W%t%T" },
    { XtNpsFilename, XtCFileFilename, XtRString, sizeof(String),
      offset(PsFilename),  XtRString, "%D%B%U%b%U%V%U%w%U%W%t%T" },
    { XtNkibitzFilename, XtCFileFilename, XtRString, sizeof(String),
      offset(KibitzFilename), XtRString, "%D%B%U%b%U%V%U%w%U%W%t%T" },
    { XtNbroadcastFilename, XtCFileFilename, XtRString, sizeof(String),
      offset(BroadcastFilename), XtRString, "%D%T" },
    { XtNyellFilename, XtCFileFilename, XtRString, sizeof(String),
      offset(YellFilename), XtRString, "%D%T" },
    { XtNigsMessageFilename, XtCFileFilename, XtRString, sizeof(String),
      offset(IgsMessageFilename), XtRString, "%D%T" },
    { XtNeventsFilename, XtCFileFilename, XtRString, sizeof(String),
      offset(EventsFilename), XtRString, "%D%T" },
    { XtNtellFilename, XtCFileFilename, XtRString, sizeof(String),
      offset(TellFilename), XtRString, "%D%N%t%n.%T" },
    { XtNmessageFilename, XtCFileFilename, XtRString, sizeof(String),
      offset(MessageFilename), XtRString, "%D%N%U%n.%T" },
    { XtNgamesFilename, XtCFileFilename, XtRString, sizeof(String),
      offset(GamesFilename), XtRString, "%D%T" },
    { XtNplayersFilename, XtCFileFilename, XtRString, sizeof(String),
      offset(PlayersFilename), XtRString, "%D%T" },
    { XtNmainFilename, XtCFileFilename, XtRString, sizeof(String),
      offset(MainFilename), XtRString, "%D%T" },
    { XtNplayersToWidget, XtCPlayersToWidget, XtRStringPairList,
      sizeof(StringPairListPtr), offset(PlayerToWidget), XtRString, NULL},
    { XtNfriends, XtCFriends, XtRStringList,
      sizeof(StringListPtr), offset(Friends), XtRString, NULL},
    { XtNsite, XtCSite, XtRString, sizeof(String),
      offset(Site), XtRString, SITE },
    { XtNport, XtCPort, XtRInt, sizeof(int),
      offset(Port), XtRString, "6969" },
};
#undef offset

int main(int argc, char **argv)
{
    int    i;
    char  *ptr, Name[200];
    Widget Buttons, Collect, AnalyzeButton, Save, File;
    Widget analyzeSize, programSettings, ResTree;
    Window GroupWindow;
    time_t Now;
    Arg    args[3];

    ExceptionProgram = argv[0];
    global_argc      = argc;
    global_argv      = argv;

    i = 0;
    NameClassArg(args,
                 "Board", boardWidgetClass,
                 "SmeBell", smeBellObjectClass,
                 NULL); i++;
    toplevel = MyAppInitialize(&app_context, "XGospel",
                               options, XtNumber(options),
                               &argc, argv, fallback_resources, args, i,
                               UseMessages, XtNumber(UseMessages),
                               NULL);

    Main    = XtNameToWidget(toplevel, XtName(toplevel));
    if (!Main) Raise1(FatalException, "Could not find real toplevel");

    XtAppAddActions(app_context, actionTable, XtNumber(actionTable));
    XtGetApplicationResources(toplevel, &appdata, resources, 
                              XtNumber(resources), NULL, 0);

    appdata.WantStdout = False;
    DebugFun         = (appdata.DebugFun & DEBUGFUN)       != 0;
    DebugPending     = (appdata.DebugFun & DEBUGPENDING)   != 0;
    AbortOnException = (appdata.DebugFun & DEBUGABORTONEXCEPTION) ? -1 : 0;
    DebugRoundTrip   = (appdata.DebugFun & DEBUGROUNDTRIP) != 0;
    IgsYYdebug       = (appdata.DebugFun & DEBUGBISON)     != 0;
    IgsYY_flex_debug = (appdata.DebugFun & DEBUGFLEX)      != 0;
    SgfDateFormat = DateFormat   = MyPassword = MyName = NULL;
    sprintf(Name, "%.*s@%.*s",
            (int) sizeof(Name)/2-1, GetUserId(),
            (int) sizeof(Name)/2-1, GetHostname());
    UserId = mystrdup(Name);
    
    if (appdata.DateFormat)
        DateFormat = mystrdup(appdata.DateFormat);
    if (appdata.SgfDateFormat)
        SgfDateFormat = mystrdup(appdata.SgfDateFormat);
    if (appdata.Password && *appdata.Password) {
        MyPassword = mystrdup(appdata.Password);
        ptr = strchr(MyPassword, ' ');
        if (ptr) *ptr = 0;
    }
    if (appdata.User && *appdata.User) {
        MyName = mystrdup(appdata.User);
        ptr = strchr(MyName, ' ');
        if (ptr) *ptr = 0;
    }
    if (appdata.ReplayTimeout <= 0) appdata.ReplayTimeout = 1;
    SetWidgetTitles(Main, MainTitle, (XtPointer) XtName(toplevel));

    DebugFile = NULL;
    if (appdata.DebugFile) {
        SubstitutionRec FileSubs[5];
        String          fn;

        i = 0;
        FileSubs[i].match = 'N';
        FileSubs[i].substitution = MyName ? MyName : "NoName";
        i++;
        FileSubs[i].match = 'P';
        FileSubs[i].substitution = MyPassword ? MyPassword : "NoPassword";
        i++;
        fn = XtFindFile(appdata.DebugFile, FileSubs, i, TryToWrite);
        if (fn) XtFree(fn);
    }

    time(&Now);
    Buttons = MyNameToWidget(Main, "main");
    if (!Buttons) Raise1(FatalException, "Could not find main widget");

    Collect         = XtNameToWidget(Buttons, "*collect");
    GamesButton     = XtNameToWidget(Buttons, "*gamesButton");
    PlayersButton   = XtNameToWidget(Buttons, "*playersButton");
    ReviewsButton   = XtNameToWidget(Buttons, "*reviewsButton");
    ServerButton    = XtNameToWidget(Buttons, "*messageButton");
    BroadcastButton = XtNameToWidget(Buttons, "*broadcastButton");
    YellButton      = XtNameToWidget(Buttons, "*yellButton");
    AnalyzeButton   = XtNameToWidget(Buttons, "*analyzeButton");
    EventsButton    = XtNameToWidget(Buttons, "*eventsButton");
    Info            = XtNameToWidget(Buttons, "*info");
    Input           = XtNameToWidget(Buttons, "*input");
    OutputBeep      = XtNameToWidget(Buttons, "*beep");
    OutputRaise     = XtNameToWidget(Buttons, "*raise");
    ErrorBeep       = XtNameToWidget(Buttons, "*errorBeep");
    ErrorRaise      = XtNameToWidget(Buttons, "*errorRaise");
    Overwrite       = XtNameToWidget(Buttons, "*overwrite");
    Save = XtNameToWidget(Buttons, "*save");
    if (Save) XtAddCallback(Save, XtNcallback, SaveMain, NULL);
    File = XtNameToWidget(Buttons, "*file");
    if (File) XtAddCallback(File, XtNcallback, ChangeMainFilename,
                            (XtPointer) &MainFileName);
    Stdout = XtNameToWidget(Buttons, "*stdout");
    if (Stdout) {
        CallReadToggle(Stdout, &appdata.WantStdout, NULL);
        XtAddCallback(Stdout, XtNcallback, CallReadToggle,
                      &appdata.WantStdout);
    }
    analyzeSize = XtNameToWidget(Buttons, "*analyzeSize");
    if (analyzeSize)
        XtAddCallback(analyzeSize, XtNcallback, ChangeAnalyzeSize, NULL);
    programSettings = XtNameToWidget(Buttons, "*programSettings");
    if (programSettings) XtAddCallback(programSettings, XtNcallback,
                                       ChangeSettings, (XtPointer) &appdata);
    MainFileName = StringToFilename(appdata.MainFilename,
                                    (int) 'T', "IGSsession",
                                    (int) 't', "_",
                                    0);
    XtAddCallback(Buttons, XtNdestroyCallback, CallFree, MainFileName);

    Localtime     = XtNameToWidget(Buttons, "*localTime");
    if (Localtime)     AddText(Localtime, DateString(localtime(&Now)));
    Universaltime = XtNameToWidget(Buttons, "*universalTime");
    if (Universaltime) AddText(Universaltime, DateString(gmtime(&Now)));
    Servertime    = XtNameToWidget(Buttons, "*serverTime");
    if (Servertime)    AddText(Servertime, "--- --- -- --:--:-- ----");

    XtSetKeyboardFocus(Collect, Input);
    MyRealizeWidget(Main);
    GroupWindow = XtWindow(Main);
    ResTree = XtNameToWidget(toplevel, "resourceTree");
    if (ResTree) 
        XtVaSetValues(ResTree, XtNwindowGroup, (XtArgVal) GroupWindow, NULL);
    
/*
    XtAppAddInput(app_context, 0,
                  (XtPointer) XtInputReadMask, UserThroughPut, NULL);
*/
    InitGospel();
    InitTextBatch();
    InitConnect(toplevel);
    InitEvents(toplevel);
    InitUtils(toplevel);
    InitPlayers(toplevel);
    InitGames(toplevel);
    InitReviews(toplevel);
    InitBroadcast(toplevel);
    InitYell(toplevel);
    InitMessages(toplevel);
    InitObserve(toplevel);
    InitTell(toplevel);

    InitHeartBeat(app_context);
    if (AnalyzeButton) XtAddCallback(AnalyzeButton,XtNcallback,TestBoard,NULL);

    WITH_UNWIND {
        Conn = Connect(appdata.Site, appdata.Port);
        RealQuit = 0;
        Entered  = 0;
        do {
            IgsYYparse();
            IgsRestartParse();
        } while (!RealQuit);
    } ON_UNWIND {
        CleanYell();
        CleanEvents();
        CleanReviews();
        CleanConnect();
        CleanTextBatch();
        CleanGospel();
        if (DebugFile) {
            fflush(DebugFile);
            fclose(DebugFile);
        }
        if (appdata.DebugFile != False) fflush(stdout);
        XtDestroyWidget(toplevel);
        myfree(SgfDateFormat);
        myfree(DateFormat);
        myfree(UserId);
        myfree(MyName);
        myfree(MyPassword);
        /*      XtDestroyApplicationContext(app_context); */
    } END_UNWIND;
    mallocstats();
    return 0;
}
