/* xtea - distribute beverages and other resources over the network
 *
 * Copyright (c) 1994 Henning Spruth (spruth@regent.e-technik.tu-muenchen.de)
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  No representations are made about the suitability of this
 * software for any purpose.  It is provided "as is" without express or 
 * implied warranty.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <netinet/tcp.h>
#include <netinet/in.h>
#include <signal.h>
#include <ctype.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Form.h>

#include <X11/xpm.h>

#include "xtea_icon.xpm"

#include "config.h"
#include "version.h"
#include "sockio.h"
#include "message.h"
#include "alloc.h"
#include "userlist.h"
#include "xresources.h"
#include "inform.h"
#include "client.h"

/* the socket we listen on */
int mastersocket;

/* toplevel widget */
Widget toplevel;

/* various pixmaps for icons */
Pixmap c_pixmap[MAX_RESOURCES];
Pixmap p_pixmap[MAX_RESOURCES];
Pixmap default_pixmap;

Pixel bg,fg;

/* program flags */
int verbose;
int have_connection;

/* the producer and consumer flags */
int pflags,cflags, group;

int path_count;
char **path_component;
char path_work[255];
char my_uname[80];

/* the names of the commodities to be managed */
extern char *xtea_resources[];

/* the number of different commodities */
int restypes;

static XtIntervalId heartbeat;

/* X resource data */
appresStruct appres;
XtAppContext app_context;

/* the message identification names */
extern char *msgtype[];

/* the "real" x resources that consist of basic_xresources and the
   commodity-dependent resources like icons and names */
static XtResource *xresources;

/* X resources for static data like colors and widget labels */
static XtResource basic_xresources[] = 
{
{"selectbgcolor", "SelectBgColor", XtRPixel, sizeof(Pixel),
   XtOffset(appresPtr, selectbgcolor), XtRString, (caddr_t) "yellow"},
{"selectfgcolor", "SelectFgColor", XtRPixel, sizeof(Pixel),
   XtOffset(appresPtr, selectfgcolor), XtRString, (caddr_t) "black"},
{"orderbgcolor", "OrderBgColor", XtRPixel, sizeof(Pixel),
   XtOffset(appresPtr, orderbgcolor), XtRString, (caddr_t) "green"},
{"orderfgcolor", "OrderFgColor", XtRPixel, sizeof(Pixel),
   XtOffset(appresPtr, orderfgcolor), XtRString, (caddr_t) "black"},
{"makeoffer", "MakeOffer", XtRString, sizeof(String),
   XtOffset(appresPtr, makeoffer), XtRString, 
   (caddr_t) "Offering resource type %s to:"},
{"gotoffer", "GotOffer", XtRString, sizeof(String),
   XtOffset(appresPtr, gotoffer), XtRString, 
   (caddr_t) "units of %s offered\nby %s"},
{"infolabel", "InfoLabel", XtRString, sizeof(String),
   XtOffset(appresPtr, infolabel), XtRString, 
   (caddr_t) "Info:"},
{"confirmok", "ConfirmOK", XtRString, sizeof(String),
   XtOffset(appresPtr, confirmmsg[0]), XtRString, 
   (caddr_t) "%s has reserved %s for you"},
{"confirmempty", "ConfirmEmpty", XtRString, sizeof(String),
   XtOffset(appresPtr, confirmmsg[1]), XtRString, 
   (caddr_t) "Sorry - %s is out of %s"},
{"confirmbad", "ConfirmBad", XtRString, sizeof(String),
   XtOffset(appresPtr, confirmmsg[2]), XtRString, 
   (caddr_t) "Not selected - %s is out of %s"},
{"iconpath", "IconPath", XtRString, sizeof(String),
   XtOffset(appresPtr, iconpath), XtRString, (caddr_t) ICONPATH},
{"beep","Beep",XtRBoolean, sizeof(Boolean),
   XtOffset(appresPtr, beep), XtRImmediate, (caddr_t) True},
{"eventprogram","EventProgram", XtRString, sizeof(String),
   XtOffset(appresPtr, eventprogram), XtRString, (caddr_t) NULL},
{"startup_event","Event", XtRString, sizeof(String),
   XtOffset(appresPtr, startup_event), XtRString, (caddr_t) NULL},
{"default_event","Event", XtRString, sizeof(String),
   XtOffset(appresPtr, default_event), XtRString, (caddr_t) NULL}
};


/* ***************************** send_startup() ******************************
 *
 * Send a startup message to the server and wait for an acknowldegement.
 *
*/
void send_startup(int pflags, int cflags, int group)
{
  int sock;
  char buf[256];
  int uid;
  struct passwd *pw;
  int n;
  struct sockaddr_in sockdata;
  char *s;

  uid=getuid();
  pw=getpwuid(uid);
  strcpy(my_uname,pw->pw_name);

  if(start_servermessage(STARTUP,&sock)) return;
  put_string(sock,my_uname);
  if(gethostname(buf,255)<0)
  { perror("gethostname");
    fatal_error(1);
  }
  put_string(sock,buf);
  for(n=0;;n++) if(pw->pw_gecos[n]==0 || pw->pw_gecos[n]==',') break;
  if(n>0)
  {
    strncpy(buf,pw->pw_gecos,n);
    buf[n]=0;
  }
  else
    /* If there is no full user name in the GECOS field, use uname */
    strcpy(buf,my_uname);
  if(verbose) printf("my_uname= <%s> len=%d\n",buf,strlen(buf));
  put_string(sock,buf);
  n=sizeof(sockdata) /*256*/;
  if(getsockname(mastersocket,(struct sockaddr *) &sockdata,&n))
  {
    perror("getsockname");
    fatal_error(1);
  }
  sprintf(buf,"%d",htons(sockdata.sin_port));
  put_string(sock,buf);
  sprintf(buf,"%d",pflags);
  put_string(sock,buf); 
  sprintf(buf,"%d",cflags);
  put_string(sock,buf); 
  sprintf(buf,"%d",group);
  put_string(sock,buf); 
  put_string(sock,"END");
  close(sock);
  if(verbose) 
    printf("Sent STARTUP to server, port %d %d\n",sockdata.sin_port, 
	   htons(sockdata.sin_port));

  sock=accept(mastersocket,(struct sockaddr *) &sockdata, &n);
  get_string(sock,&s);
  if(strcmp(s,"ACK"))
  {
    fprintf(stderr,"ACK expcected from server, got %s\n",s);
    free(s);
    end_of_message(sock);
    return;
  }
  free(s);
  end_of_message(sock);
}



/* ****************************** send_alive() ******************************
 * 
 * Callback function for the heartbeat interval timer. Send a startup message
 * to the client to indicate that we are alive and restart the timer.
 *
*/
static void send_alive(int dummy, void *dummy2)
{
  if(verbose)
    printf("Sending startup message to server to indicate we're alive\n");
  send_startup(pflags,cflags,group);
  heartbeat = XtAppAddTimeOut(app_context, HEARTRATE, 
			      (XtTimerCallbackProc) send_alive, 
			      (XtPointer) cflags);
}


/* ************************ button_callback() *******************************
 *
 * The user wants to offer some commodity to others. Send an OFFER message to
 * to the server, it will respond by a USERLIST message.
 *
*/
void button_callback(Widget w, int resource, void *dummy)
{
  int n,sock;
  char buf[10];

  n=0;
  start_servermessage(OFFER,&sock);
  put_string(sock,my_uname);
  sprintf(buf,"%d",resource);
  put_string(sock,buf);
  put_string(sock,"END");
  close(sock);
}


#define MAX_ARGV 30
void announce_event(char *event_type)
{
  char buf[255];
  char *argv[MAX_ARGV];
  int status,i,done,mode;
  char *s;

  /* if handler or event is not specified, do nothing */
  if(appres.eventprogram==NULL || event_type==NULL) return;

  /* fork event handler process, e.g. to play a sound */
  if((status=fork())==0)
  { 
    /* this is the son: start handler */

    /* create command line */
    sprintf(buf,appres.eventprogram,event_type);

    /* dissect command line into argv[] array */
    i=0;
    done=0;
    s=buf;
    do
    {
      while(*s==' ') s++;	/* skip leading blanks */
      argv[i++]=s;		/* store beginning of word */
      mode=(*s=='"');
      if(i>=MAX_ARGV-1) 
      {
	fprintf(stderr,"Too many arguments to event program\n");
	exit(1);
      }
      s++;
      while(*s && (mode==1 || *s!=' ') && 
      	(mode==0 || *s!='"')) s++;	/* skip work body */
      if(mode==1 && *s=='"') s++;
      if(*s==0) done=1;		/* end of string? */
      *s++=0;			/* mark end of word and skip char */
    } while(!done);
    argv[i]=NULL;		/* mark end of arguments */
/*  debugging:
    for(i=0;argv[i];i++) printf("argv[%d]=%s\n",i,argv[i]);
*/
    /* goodbye xtea, hello event program! */
    execvp(argv[0],&argv[0]);
  }

  /* this is the father: continue */
  if(status<0) perror("fork");
}


/* **************************** socket_callback() **************************
 *
 * Something happens on our 'mastersocket'. Receive the message type and
 * do the appropriate action.
 *
*/
void socket_callback()
{
  int msgsock;
  char *s;
  int type;
  struct sockaddr_in sa;
  int salen;
  salen=sizeof(sa);
  msgsock=accept(mastersocket,(struct sockaddr *) &sa,&salen);
  if(msgsock<0)
  {
    perror("accept");
    return;
  }
  if(!get_string(msgsock,&s))
  {
    if((type=parse_msgtype(s))>=0)
    {
      if(verbose) printf("Got message type %d = %s\n",type,msgtype[type]);
      switch(type)
      {
      case USERLIST:
	popup_userlist(msgsock);
	break;
      case INFORM:
	process_inform(msgsock);
	break;
      case ORDER:
	process_order(msgsock);
	break;
      case CONFIRM:
	popup_confirm(msgsock);
	break;
      default:
	while(!get_string(msgsock,&s) && parse_msgtype(s)!=END)
	{
	  puts(s);
	  free(s);
	} 
	break;
      }
    }
    else if(verbose) printf("Unknown message %s\n",s);
    free(s);
  }
  end_of_message(msgsock);
}

/* ******************************** parse_flags() **************************
 *
 * Parse the command-line flags specifying resources to be taken or given
 * and put it into the 'boolean vector' *flag.
 *
*/
static int parse_flags(char *string, int *flag)
{
  char *s0,*s,*t;
  int i;

  s0=s=myalloc(strlen(string)+1);
  strcpy(s,string);
  for(t=s; *t; t++) (*t)=tolower(*t);

  *flag=0;
  t=s;
  while(s)
  {
    while(*t  && *t!=',') t++;
    if(*t) 
    { *t=0;
      t++;
    }
    else t=NULL;
    if(strcmp(s,"all")==0)
    {
      (*flag) |= (1<<restypes)-1;
    }
    else
    {
      for(i=0;i<restypes;i++)
	if(strcmp(s,xtea_resources[i])==0) break;
      if(i>=restypes) return 1;
      (*flag) |= 1 << i;
    }
    s=t;
  }
  free(s0);
  return 0;
}


/* ****************************** parse_iconpath() *************************
 *
 * Parse the search path for xpm and bitmap files and put the components into
 * a linked list.
 *
*/
static void parse_iconpath()
{
  int i;
  char *s,*t;
  
  path_count=0;
  strcpy(path_work,appres.iconpath);

  /* count # of elements in path */
  path_count=0;
  for(s=path_work; *s; s++) if(*s==':') path_count++;
  if(path_work[0]) path_count++; else return;

  path_component = myalloc(path_count * sizeof(char *));
  
  i=0;
  s=t=path_work;
  for(s=path_work; *s; s++)
    if(*s==':')
    {
      path_component[i++]=t;
      *s=0;
      t=s+1;
    }
  path_component[i]=t;
  if(verbose) for(i=0;i<path_count;i++) printf("path %d: %s\n",
					     i,path_component[i]);
}
  
/* ****************************** get_pixmap() ******************************
 *
 * Load a pixmap. Depending on the file suffix, try a XPM file or a bitmap
 * file. Try all different search paths.
 *
*/
static void get_pixmap(Widget toplevel, Pixmap *p, char *name)
{
  int i,l;
  int status;
  int dummy;
  char fn[255];
  
  if(name)
  {
    l=strlen(name);
    if(l>4 && strcmp(&name[l-4],".xpm")==0)
    {
      for(i=0;i<path_count;i++)
      {
	strcpy(fn,path_component[i]);
	strcat(fn,"/");
	strcat(fn,name);
	status=XpmReadFileToPixmap(XtDisplay(toplevel), 
				   RootWindowOfScreen(XtScreen(toplevel)),
				   fn,p,NULL,NULL);
	if(status==0) break;
      }
    }
    else
    {
      for(i=0;i<path_count;i++)
      {
	strcpy(fn,path_component[i]);
	strcat(fn,"/");
	strcat(fn,name);
	status=XReadBitmapFile(XtDisplay(toplevel),
			       RootWindowOfScreen(XtScreen(toplevel)),
			       name, (unsigned int *) &dummy, 
			       (unsigned int *) &dummy, p, &dummy, &dummy);
	if(status==0) break;
      }
    }
    if(status)
    {
      fprintf(stderr,"Error %d reading pixmap file %s - using builtin default\n",
	      status,name);
      *p = default_pixmap;
    }
    else if(verbose) printf("Loaded %s\n",fn);
  }
  else *p = default_pixmap;
}


/* ********************************* usage() ********************************
 *
 * Display an usage message and abort.
 *
*/
static void usage(char *progname)
{
  int i;
  fprintf(stderr,"Usage: %s [-s] [-v] [ -c resourcelist ] [ -p resourcelist ]\n",
	  progname);
  fprintf(stderr,"       resourcelist = list of ");
  for(i=0;i<restypes;i++)
    fprintf(stderr,i==0?"%s":", %s",xtea_resources[i]);
  fprintf(stderr,"\n");
  fprintf(stderr,"At least one -c or -p list must be given\n");
  fatal_error(1);
}


/* ***************************** goodbye_and_exit() *************************
 *
 * We are about to quit. If we have an connection to xtead, send a GOODBYE
 * message first.
 *
*/
static void goodbye_and_exit(int exitcode)
{
  int sock;
  if(have_connection)
  {
    if(!start_servermessage(GOODBYE,&sock))
    {
      put_string(sock,my_uname);
      put_string(sock,"END");
      close(sock);
    }
  }
  exit(exitcode);
}


/* **************************** catchint() *****************************
 *
 * The signal handler: inform xtead and exit.
 *
*/
static void catchint()
{
  signal(SIGINT,  SIG_DFL);
  signal(SIGQUIT, SIG_DFL);
  signal(SIGPIPE, SIG_DFL);
  signal(SIGTERM, SIG_DFL);
  printf("xtea got signal - informing daemon and aborting\n");
  goodbye_and_exit(3);
}

static void sigchld_handler()
{
  int status;
  wait(&status);
}

/* ********************************** fatal_error() ************************
 *
 * Something real bad happened.
 *
*/
void fatal_error(int code)
{
  fprintf(stderr,"Fatal error - aborting\n");
  goodbye_and_exit(code);
}


/* ***************************** create_xresources() ***********************
 *
 * Build X resources for the name, provider icon, and consumer icon of all
 * commodities and combine them with the static resources in
 * basic_xresources.
 *
*/
#define RESCOM 4	/* # of resources specified per commodity */
static int create_xresources()
{
  int i,index,offset;
  char *s,*t;
  static char class[]="Name";
  static char can[]="Can";
  static char cup[]="Cup";
  static char event[]="Event";
  
  offset= sizeof(basic_xresources)/sizeof(XtResource);

  /* allocate memory for the static X resources in basic_xresources and
     the commodities defined in configure.c */
  xresources=myalloc((offset+RESCOM*restypes)*sizeof(XtResource));
  /* copy static part into xresources */
  memcpy(xresources,basic_xresources,sizeof(basic_xresources));

  for(i=0;i<restypes;i++)
  {
    /* 1st resource: name of commodity, e.g. 'teaname: tea' */
    index=offset+i*RESCOM;
    s=myalloc(strlen(xtea_resources[i])+5);
    strcpy(s,xtea_resources[i]);
    strcat(s,"name");
    xresources[index].resource_name=s;
    xresources[index].resource_class=class;
    xresources[index].resource_type=XtRString;
    xresources[index].resource_size=sizeof(String);
    xresources[index].resource_offset=XtOffset(appresPtr,resourcename[i]);
    xresources[index].default_type = XtRString;
    xresources[index].default_addr = xtea_resources[i];
  
    /* 2nd resource: icon for source of commodity, e.g. 
       'teasource: teacan.xpm' */
    index++;
    s=myalloc(strlen(xtea_resources[i])+7);
    strcpy(s,xtea_resources[i]);
    strcat(s,"source");
    t=myalloc(strlen(xtea_resources[i])+8);
    strcpy(t,xtea_resources[i]);
    strcat(t,"can.xpm");
    xresources[index].resource_name=s;
    xresources[index].resource_class=can;
    xresources[index].resource_type=XtRString;
    xresources[index].resource_size=sizeof(String);
    xresources[index].resource_offset=XtOffset(appresPtr,p_icon[i]);
    xresources[index].default_type = XtRString;
    xresources[index].default_addr = t;
  
    /* 3d resource: icon for sink of commodity, e.g.
       'teasink: teacup.xpm' */
    index++;
    s=myalloc(strlen(xtea_resources[i])+5);
    strcpy(s,xtea_resources[i]);
    strcat(s,"sink");
    t=myalloc(strlen(xtea_resources[i])+8);
    strcpy(t,xtea_resources[i]);
    strcat(t,"cup.xpm");
    xresources[index].resource_name=s;
    xresources[index].resource_class=cup;
    xresources[index].resource_type=XtRString;
    xresources[index].resource_size=sizeof(String);
    xresources[index].resource_offset=XtOffset(appresPtr,c_icon[i]);
    xresources[index].default_type = XtRString;
    xresources[index].default_addr = t;
    
    /* 4th resource: parameter passed to the event program, e.g.
       'tea_event: teaready.au' (default set to NULL) */
    index++;
    s=myalloc(strlen(xtea_resources[i])+7);
    strcpy(s,xtea_resources[i]);
    strcat(s,"_event");
    xresources[index].resource_name=s;
    xresources[index].resource_class=event;
    xresources[index].resource_type=XtRString;
    xresources[index].resource_size=sizeof(String);
    xresources[index].resource_offset=XtOffset(appresPtr,event_arg[i]);
    xresources[index].default_type = XtRString;
    xresources[index].default_addr = NULL;
  }
  
  return offset + RESCOM*restypes;
}

static int get_usergroup()
{
  static char group_cmd[]=GROUP_CMD;
  int g;

  if(verbose) printf("Determining user group with '%s'\n",
  	group_cmd);
  if(strlen(group_cmd)>0)
    /* the return status of the pipe is returned in the upper 8 bits */
    g=system(group_cmd)>>8;
  else
    g=0;
  if(verbose) printf("Group set to %d\n",g);
  return g;
}

int main(int argc, char **argv)
{

  Arg arg[10];
  int i,n;
  int c;
  int group_override,errflg,first,no_signal;

  Widget form,command;
  extern int optind;
  extern char *optarg;

  /* first, compute the actual # of resources */
  restypes=resource_count();

  /* no connection to server yet. Don't try to inform him if something goes
     wrong */
  have_connection=0;

  if(getuid()==0)
  {
    printf("Sorry - user root must not use %s\n", argv[0]);
    exit(1);
  }
  
  init_inform();

  toplevel=XtVaAppInitialize(&app_context,"XTea",NULL,0,&argc,argv,NULL,NULL);

  /* parse command line arguments */
  verbose=0;
  group_override=0;
  cflags=pflags=0;
  errflg=0;
  no_signal=0;
  while((c=getopt(argc,argv,"svG:c:p:"))!=EOF)
  {
    switch(c)
    {
    case 's':
      no_signal=1;
      break;
    case 'v':
      verbose=1;
      break;
    case 'G':
      if(group_override) errflg++;
      if(sscanf(optarg,"%d",&group)!=1) errflg++;
      group_override=1;
      break;
    case 'p':
      if(pflags || parse_flags(optarg,&pflags)) errflg++;
      break;
    case 'c':
      if(cflags || parse_flags(optarg,&cflags)) errflg++;
      break;
    case '?':
      errflg++;
      break;
    }
  }
  if(errflg || (optind!=argc) || ((cflags|pflags)==0)) usage(argv[0]);

  if(group_override)
  {
    if(verbose) printf("Group set to %d via override\n",group);
  }
  else
    group=get_usergroup();
  

  if(verbose) printf("xtea client version %s\n",XTEA_VERSION);

  if(!no_signal)
  {
    signal(SIGINT, catchint);
    signal(SIGQUIT, catchint);
    signal(SIGTERM, catchint);
  }
  signal(SIGCHLD, sigchld_handler);

  /* create X resources from static and commodity dependent ones */
  n=create_xresources();

  XtGetApplicationResources(toplevel, (XtPointer) &appres, 
     xresources, n, NULL, 0);

  /* create default pixmap */
  XpmCreatePixmapFromData(XtDisplay(toplevel),
			  RootWindowOfScreen(XtScreen(toplevel)),
			  xtea_icon_xpm,&default_pixmap,NULL,NULL);

  /* decompose icon path */
  parse_iconpath();


  /* load the required pixmaps */
  for(i=0;i<restypes;i++)
  {
    /* producer pixmap */
    if(pflags & 1<<i)
      get_pixmap(toplevel, &(p_pixmap[i]),appres.p_icon[i]);
    /* consumer pixmap */
    if((pflags|cflags) & 1<<i)
	get_pixmap(toplevel, &(c_pixmap[i]),appres.c_icon[i]);
  }

  n=0;
  form=XtCreateManagedWidget("main",formWidgetClass,toplevel,arg,n);

  first=1;
  /* if we produce something, put up an icon for every produced commodity */
  if(pflags)
  {
    for(i=0;i<restypes;i++)
      if(pflags & 1<<i)
      {
	n=0;
	if(!first) {XtSetArg(arg[n], XtNfromHoriz, command); n++;}
	XtSetArg(arg[n],XtNbitmap,p_pixmap[i]); n++;
	command=XtCreateManagedWidget("resbutton",commandWidgetClass,form,
				      arg,n);
	XtAddCallback(command,XtNcallback,
		      (XtCallbackProc) button_callback,
		      (XtPointer) i);
	first=0;
      }
  }
  else
  /* we are only a consumer, create a default icon so the user has something
     to look at */
  {
    n=0;
    XtSetArg(arg[n],XtNbitmap,default_pixmap); n++;
    command=XtCreateManagedWidget("resbutton",commandWidgetClass,form,
				      arg,n);
  }

   
  /* initialize the master socket */
  if(init_listen(0,&mastersocket)) fatal_error(1);

  /* notify the server that we're here */
  send_startup(pflags,cflags,group);

  /* create X events for socket activity */
  XtAppAddInput(app_context,mastersocket,(XtPointer) XtInputReadMask,
		(XtInputCallbackProc) socket_callback,NULL);

  /* connection exists, inform server if we have to exit */
  have_connection=1;

  XtRealizeWidget(toplevel);

  /* get default bg and fg colors */
  n=0;
  XtSetArg(arg[n],XtNforeground,&fg); n++;
  XtSetArg(arg[n],XtNbackground,&bg); n++;
  XtGetValues(command,arg,n);
  
  /* prepare to send 1st alive message */
  heartbeat = XtAppAddTimeOut(app_context, HEARTRATE, 
			      (XtTimerCallbackProc) send_alive, 
			      (XtPointer) NULL);

  /* if configured, announce the startup event, e.g. play a tune */
  announce_event(appres.startup_event);

  /* here we go! */
  XtAppMainLoop(app_context);

  return 0;

}
