/*****************************************************************************
                                jpegtopnm
******************************************************************************
  This program is part of the Netpbm package.

  This program converts from the JFIF format, which is based on JPEG, to
  the fundamental ppm or pgm format (depending on whether the JFIF 
  image is gray scale or color).

  This program is by Bryan Henderson on 2000.03.20, but is derived
  with permission from the program djpeg, which is in the Independent
  Jpeg Group's JPEG library package.  Under the terms of that permission,
  redistribution of this software is restricted as described in the 
  file README.JPEG.

  Copyright (C) 1991-1998, Thomas G. Lane.

  TODO:

    See if additional decompressor options effects signficant speedup.
    grayscale output of color image, downscaling, color quantization, and
    dithering are possibilities.  Djpeg's man page says they make it faster.
    
    Do wide pnms.

*****************************************************************************/

#include <ctype.h>		/* to declare isprint() */
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <jpeglib.h>
#include "shhopt.h"
#include "pnm.h"

#define EXIT_WARNING 2  /* Goes with EXIT_FAILURE, EXIT_SUCCESS in stdlib.h */

static const char * progname;	/* program name for error messages */

struct cmdline_info {
    /* All the information the user supplied in the command line,
       in a form easy for the program to use.
    */
    char *input_filename;
    int verbose;
    int nosmooth;
    J_DCT_METHOD dct_method;
    int grayscale;
    long int max_memory_to_use;
    unsigned int trace_level;
} cmdline;



static void 
interpret_maxmemory (const char * const maxmemory, 
                     long int * const max_memory_to_use_p) { 
/*----------------------------------------------------------------------------
   Interpret the "maxmemory" command line option.
-----------------------------------------------------------------------------*/
    long int lval;
    char ch;
    
    if (maxmemory == NULL) {
        *max_memory_to_use_p = -1;  /* unspecified */
    } else if (sscanf(maxmemory, "%ld%c", &lval, &ch) < 1) {
        pm_error("Invalid value for --maxmemory option: '%s'.", maxmemory);
    } else {
        if (ch == 'm' || ch == 'M') lval *= 1000L;
        *max_memory_to_use_p = lval * 1000L;
    }
}



static void
parse_command_line(const int argc, char ** argv,
                   struct cmdline_info *cmdline_p) {
/*----------------------------------------------------------------------------
   Note that many of the strings that this function returns in the
   *cmdline_p structure are actually in the supplied argv array.  And
   sometimes, one of these strings is actually just a suffix of an entry
   in argv!

   On the other hand, unlike other option processing functions, we do
   not change argv at all.
-----------------------------------------------------------------------------*/

    int i;  /* local loop variable */

    char *dctval;
    char *maxmemory;

    optStruct *option_def = malloc(100*sizeof(optStruct));
    /* Instructions to OptParseOptions on how to parse our options.
     */
    unsigned int option_def_index;

    int argc_parse;       /* argc, except we modify it as we parse */
    char ** const argv_parse = malloc(argc*sizeof(char *));
    /* argv, except we modify it as we parse */

    /* Create the OptStruct structure describing our options */
    #define OPTENTRY(shortvalue,longvalue,typevalue,outputvalue,flagvalue) {\
      option_def[option_def_index].shortName = (shortvalue); \
      option_def[option_def_index].longName = (longvalue); \
      option_def[option_def_index].type = (typevalue); \
      option_def[option_def_index].arg = (outputvalue); \
      option_def[option_def_index].flags = (flagvalue); \
      option_def_index++; \
      }
    option_def_index = 0;   /* incremented by OPTENTRY */
    OPTENTRY(0, "verbose",     OPT_FLAG,   &cmdline_p->verbose,        0);
    OPTENTRY(0, "dct",         OPT_STRING, &dctval,                    0);
    OPTENTRY(0, "maxmemory",   OPT_STRING, &maxmemory,                 0); 
    OPTENTRY(0, "nosmooth",    OPT_FLAG,   &cmdline_p->nosmooth,       0);
    OPTENTRY(0, "tracelevel",  OPT_UINT,   &cmdline_p->trace_level,    0);
    option_def[option_def_index].type = OPT_END; 

    /* Set the defaults */
    cmdline_p->verbose = FALSE;
    cmdline_p->nosmooth = FALSE;
    dctval = NULL;
    maxmemory = NULL;
    cmdline_p->trace_level = 0;

  /* Make private copy of arguments for optParseOptions to corrupt */
    argc_parse = argc;
    for (i=0; i < argc; i++) argv_parse[i] = argv[i];

    optParseOptions(&argc_parse, argv_parse, option_def, 0);
    /* Uses and sets argc_parse, argv_parse and all of *cmdline_p. */

    if (argc_parse - 1 == 0)
        cmdline_p->input_filename = NULL;  /* he wants stdin */
    else if (argc_parse - 1 == 1)
        cmdline_p->input_filename = strdup(argv_parse[1]);
    else 
        pm_error("Too many arguments.  The only argument accepted\n"
                 "is the input file specificaton");

    if (dctval == NULL)
        cmdline_p->dct_method = JDCT_DEFAULT;
    else {
        if (strcmp(dctval, "int") == 0)
            cmdline_p->dct_method = JDCT_ISLOW;
        else if (strcmp(dctval, "fast") == 0)
            cmdline_p->dct_method = JDCT_IFAST;
        else if (strcmp(dctval, "float") == 0)
            cmdline_p->dct_method = JDCT_FLOAT;
        else pm_error("Invalid value for the --dct option: '%s'.", dctval);
    }

    interpret_maxmemory(maxmemory, &cmdline_p->max_memory_to_use);

    free(argv_parse);
}


/*
 * Marker processor for COM and interesting APPn markers.
 * This replaces the library's built-in processor, which just skips the marker.
 * We want to print out the marker as text, to the extent possible.
 * Note this code relies on a non-suspending data source.
 */

static unsigned int
jpeg_getc (j_decompress_ptr cinfo)
/* Read next byte */
{
  struct jpeg_source_mgr * datasrc = cinfo->src;

  if (datasrc->bytes_in_buffer == 0) {
      if (! (*datasrc->fill_input_buffer) (cinfo)) 
          pm_error("Can't suspend here.");
  }
  datasrc->bytes_in_buffer--;
  return GETJOCTET(*datasrc->next_input_byte++);
}


static boolean
print_text_marker (j_decompress_ptr cinfo)
{
  const boolean traceit = (cinfo->err->trace_level >= 1);
  INT32 length;
  unsigned int ch;
  unsigned int lastch = 0;

  length = jpeg_getc(cinfo) << 8;
  length += jpeg_getc(cinfo);
  length -= 2;			/* discount the length word itself */

  if (traceit) {
    if (cinfo->unread_marker == JPEG_COM)
      fprintf(stderr, "Comment, length %ld:\n", (long) length);
    else			/* assume it is an APPn otherwise */
      fprintf(stderr, "APP%d, length %ld:\n",
	      cinfo->unread_marker - JPEG_APP0, (long) length);
  }

  while (--length >= 0) {
    ch = jpeg_getc(cinfo);
    if (traceit) {
      /* Emit the character in a readable form.
       * Nonprintables are converted to \nnn form,
       * while \ is converted to \\.
       * Newlines in CR, CR/LF, or LF form will be printed as one newline.
       */
      if (ch == '\r') {
	fprintf(stderr, "\n");
      } else if (ch == '\n') {
	if (lastch != '\r')
	  fprintf(stderr, "\n");
      } else if (ch == '\\') {
	fprintf(stderr, "\\\\");
      } else if (isprint(ch)) {
	putc(ch, stderr);
      } else {
	fprintf(stderr, "\\%03o", ch);
      }
      lastch = ch;
    }
  }

  if (traceit)
    fprintf(stderr, "\n");

  return TRUE;
}





/*
 * For 12-bit JPEG data, we either downscale the values to 8 bits
 * (to write standard byte-per-sample PPM/PGM files), or output
 * nonstandard word-per-sample PPM/PGM files.  Downscaling is done
 * if PPM_NORAWWORD is defined (this can be done in the Makefile
 * or in jconfig.h).
 * (When the core library supports data precision reduction, a cleaner
 * implementation will be to ask for that instead.)
 */

/* SAMPLE_SQUEEZE is a macro that takes a JSAMPLE (sample read from a JPEG
   file) and generates a 'pixval' (sample in the pnm output buffer), even
   though the JPEG file may have a wider sample than the pnm format allows.
   */

#if BITS_IN_JSAMPLE == 8
#define PUTPPMSAMPLE(ptr,v)  *ptr++ = (char) (v)
#define SAMPLE_SQUEEZE(x) (x)
#define BYTESPERSAMPLE 1
#define PPM_MAXVAL 255
#else
#ifdef PPM_NORAWWORD
#define PUTPPMSAMPLE(ptr,v)  *ptr++ = (char) ((v) >> (BITS_IN_JSAMPLE-8))
#define SAMPLE_SQUEEZE(x) ((x) >> (BITS_IN_JSAMPLE-8))
#define BYTESPERSAMPLE 1
#define PPM_MAXVAL 255
#else
/* The word-per-sample format always puts the LSB first. */
#define PUTPPMSAMPLE(ptr,v)			\
	{ register int val_ = v;		\
	  *ptr++ = (char) (val_ & 0xFF);	\
	  *ptr++ = (char) ((val_ >> 8) & 0xFF);	\
	}
#ifdef BIGGRAYS
#define SAMPLE_SQUEEZE(x) (x)
#else
#define SAMPLE_SQUEEZE(x) ((x) >> (BITS_IN_JSAMPLE-8))
#endif
#define BYTESPERSAMPLE 2
#define PPM_MAXVAL ((1<<BITS_IN_JSAMPLE)-1)
#endif
#endif


typedef struct rgb {int r; int g; int b;} rgb_type;


static rgb_type *
read_rgb(JSAMPLE *ptr, J_COLOR_SPACE color_space) {
/*----------------------------------------------------------------------------
  Return the RGB triple corresponding to the color of the JPEG pixel at
  'ptr', which is in color space 'color_space';
-----------------------------------------------------------------------------*/
    static rgb_type rgb;  /* Our return value */

    switch (color_space) {
    case JCS_RGB: {
        rgb.r = GETJSAMPLE(*(ptr+0));
        rgb.g = GETJSAMPLE(*(ptr+1)); 
        rgb.b = GETJSAMPLE(*(ptr+2)); 
    }
        break;
    case JCS_CMYK: {
        const int c = GETJSAMPLE(*(ptr+0));
        const int m = GETJSAMPLE(*(ptr+1));
        const int y = GETJSAMPLE(*(ptr+2));
        const int k = GETJSAMPLE(*(ptr+3));

        rgb.r = ((MAXJSAMPLE-k)*(MAXJSAMPLE-c))/MAXJSAMPLE;
        rgb.g = ((MAXJSAMPLE-k)*(MAXJSAMPLE-y))/MAXJSAMPLE;
        rgb.b = ((MAXJSAMPLE-k)*(MAXJSAMPLE-m))/MAXJSAMPLE;
    }
        break;
    default:
        pm_error("Internal error: unknown color space %d passed to "
                 "read_rgb().", (int) color_space);
    }
    return(&rgb);
}



/* pnmbuffer is declared global because it is really local to 
   copy_pixel_row().  But it would be impractical to allocate and free
   the storage with every call to copy_pixel_row() and improper to pass
   the pointer as input to copy_pixel_row(), since it isn't logically 
   a parameter of the operation.
   */
xel *pnmbuffer;      /* Output buffer.  Input to pnm_writepnmrow() */

static void
copy_pixel_row(const JSAMPROW jpegbuffer, const int width, 
               const unsigned int samples_per_pixel, 
               const J_COLOR_SPACE color_space,
               FILE * const output_file, const int output_type) {
  JSAMPLE *ptr;
  unsigned int output_cursor;     /* Cursor into output buffer 'pnmbuffer' */

  ptr = jpegbuffer;  /* Start at beginning of input row */

  for (output_cursor = 0; output_cursor < width; output_cursor++) {
      xel current_pixel;
      if (samples_per_pixel >= 3) {
          const rgb_type * const rgb_p = read_rgb(ptr, color_space);
          PPM_ASSIGN(current_pixel, 
                     SAMPLE_SQUEEZE(rgb_p->r),
                     SAMPLE_SQUEEZE(rgb_p->g),
                     SAMPLE_SQUEEZE(rgb_p->b)
                     );
      } else {
          PNM_ASSIGN1(current_pixel, SAMPLE_SQUEEZE(GETJSAMPLE(*ptr)));
      }
      ptr += samples_per_pixel;  /* move to next pixel of input */
      pnmbuffer[output_cursor] = current_pixel;
  }
  pnm_writepnmrow(output_file, pnmbuffer, width,
                  PPM_MAXVAL, output_type, FALSE);
}



static void
set_color_spaces(const J_COLOR_SPACE jpeg_color_space,
                 int * const output_type_p,
                 J_COLOR_SPACE * const out_color_space_p) {
/*----------------------------------------------------------------------------
   Decide what type of output (PPM or PGM) we shall generate and what 
   color space we must request from the JPEG decompressor, based on the
   color space of the input JPEG image.

   Write to stderr a message telling which type we picked.

   Exit the process with EXIT_FAILURE completion code and a message to
   stderr if the input color space is beyond our capability.
-----------------------------------------------------------------------------*/
    /* Note that the JPEG decompressor is not capable of translating
       CMYK or YCCK to RGB, but can translate YCCK to CMYK.
    */

    switch (jpeg_color_space) {
    case JCS_UNKNOWN:
        pm_error("Input JPEG image has 'unknown' color space "
                 "(JCS_UNKNOWN).\n"
                 "We cannot interpret this image.");
        break;
    case JCS_GRAYSCALE:
        *output_type_p = PGM_TYPE;
        *out_color_space_p = JCS_GRAYSCALE;
        break;
    case JCS_RGB:
        *output_type_p = PPM_TYPE;
        *out_color_space_p = JCS_RGB;
        break;
    case JCS_YCbCr:
        *output_type_p = PPM_TYPE;
        *out_color_space_p = JCS_RGB;
        break;
    case JCS_CMYK:
        *output_type_p = PPM_TYPE;
        *out_color_space_p = JCS_CMYK;
        break;
    case JCS_YCCK:
        *output_type_p = PPM_TYPE;
        *out_color_space_p = JCS_CMYK;
        break;
    }
    pm_message("WRITING %s FILE", 
               *output_type_p == PPM_TYPE ? "PPM" : "PGM");
}



void
main(int argc, char **argv) {
  struct jpeg_decompress_struct cinfo;
  struct jpeg_error_mgr jerr;
  FILE * input_file;
  FILE * output_file;
  int rc;
  int output_type;
    /* The type of output file, PGM or PPM.  Value is either PPM_TYPE
       or PGM_TYPE, which conveniently also pass as format values
       PPM_FORMAT and PGM_FORMAT.
    */
  JSAMPROW jpegbuffer;  /* Input buffer.  Filled by jpeg_scanlines() */


  pnm_init(&argc, argv);

  progname = argv[0];

  parse_command_line(argc, argv, &cmdline);

  /* Initialize the JPEG decompression object with default error handling. */
  cinfo.err = jpeg_std_error(&jerr);
  jpeg_create_decompress(&cinfo);
  if (cmdline.trace_level == 0 && cmdline.verbose) cinfo.err->trace_level = 1;
  else cinfo.err->trace_level = cmdline.trace_level;

  /* Insert custom marker processor for COM and APP12.
   * APP12 is used by some digital camera makers for textual info,
   * so we provide the ability to display it as text.
   * If you like, additional APPn marker types can be selected for display,
   * but don't try to override APP0 or APP14 this way (see libjpeg.doc).
   */
  jpeg_set_marker_processor(&cinfo, JPEG_COM, print_text_marker);
  jpeg_set_marker_processor(&cinfo, JPEG_APP0+12, print_text_marker);

  /* Open the input file */
  if (cmdline.input_filename == NULL) 
      input_file = stdin;
  else {
      if ((input_file = fopen(cmdline.input_filename, "r")) == NULL) 
          pm_error("Can't open %s.  Errno=%s(%d).", 
                   cmdline.input_filename, strerror(errno), errno);
      free(cmdline.input_filename);
  }

  output_file = stdout;

  /* Specify data source for decompression */
  jpeg_stdio_src(&cinfo, input_file);

  /* Read file header, set default decompression parameters */
  (void) jpeg_read_header(&cinfo, TRUE);

  cinfo.dct_method = cmdline.dct_method;
  if (cmdline.max_memory_to_use != -1)
      cinfo.mem->max_memory_to_use = cmdline.max_memory_to_use;
  if (cmdline.nosmooth)
      cinfo.do_fancy_upsampling = FALSE;

  set_color_spaces(cinfo.jpeg_color_space, &output_type, 
                   &cinfo.out_color_space);

  if (cmdline.verbose) 
      pm_message("Output file will have format %c%c "
                 "with max sample value of %d.", 
                 (char) (output_type/256), (char) (output_type % 256),
                 PPM_MAXVAL);
  
  /* Calculate output image dimensions so we can allocate space */
  jpeg_calc_output_dimensions(&cinfo);

  jpegbuffer = ((*cinfo.mem->alloc_sarray)
                ((j_common_ptr) &cinfo, JPOOL_IMAGE,
                 cinfo.output_width * cinfo.output_components, (JDIMENSION) 1)
               )[0];

  /* Start decompressor */
  jpeg_start_decompress(&cinfo);

  /* Write pnm output header */
  pnm_writepnminit(output_file, cinfo.output_width, cinfo.output_height,
                   PPM_MAXVAL, output_type, FALSE);

  pnmbuffer = pnm_allocrow(cinfo.output_width);

  /* Process data */
  while (cinfo.output_scanline < cinfo.output_height) {
    jpeg_read_scanlines(&cinfo, &jpegbuffer, 1);
    copy_pixel_row(jpegbuffer, cinfo.output_width, cinfo.out_color_components,
                   cinfo.out_color_space, output_file, output_type);
  }

  pnm_freerow(pnmbuffer);

  rc = fclose(output_file);
  if (rc == EOF) 
      pm_error("Error writing output file.  Errno = %s (%d).",
               strerror(errno), errno);

  /* Finish decompression and release decompressor memory. */
  (void) jpeg_finish_decompress(&cinfo);
  jpeg_destroy_decompress(&cinfo);

  /* Close files, if we opened them */
  if (input_file != stdin) fclose(input_file);

  /* All done. */
  exit(jerr.num_warnings > 0 ? EXIT_WARNING : EXIT_SUCCESS);
}
