/* Copyright 1994 GROUPE BULL -- See license conditions in file COPYRIGHT */
/* $Id: Knvas.c,v 1.10.2.2 96/03/27 15:37:36 leon Exp $ */

#include <math.h>		/* required for atof !!!! */
#include "misc.h"
#ifdef DO_MOTIF_WIDGET
#include <Xm/ScrolledWP.h>
#include <Xm/ScrollBar.h>
#endif /* DO_MOTIF_WIDGET */
#include "transformer.h"
#include "draw.h"
#include <X11/cursorfont.h>
#include "TagP.h"
#include "knoP.h"
#include "region.h"

#define ClearWidget(w)\
    if(XtIsRealized(w)) {\
        XClearArea(XtDisplay(w), XtWindow(w), 0, 0, -1, -1, True);}


#define KnMax(a, b) (((a)>(b))?(a):(b))
/* Initial Pixmap Size 500x500 = 250Ko*/
#define IPSIZE 500


/********    Static Function Declarations - start   ********/

static void ClassInitialize(WidgetClass wc);
static void Initialize(Widget w, Widget neww, ArgList args, Cardinal *n);
static void UpdateScrollBars(KnvasWidget kw);
static void Resize(Widget w);
static Boolean SetValues(Widget cw, Widget rw, Widget nw,
			 ArgList args, Cardinal *num_args);

static void VScrollCB(Widget w, XtPointer client, XtPointer call);
static void HScrollCB(Widget w, XtPointer client, XtPointer call);
static void CreateBackPixmap(Widget w);
static void Destroy(Widget w);

/********    Static Function Declarations - end   ********/



/* if this function is called, that mean that no kno has focus.
 * so unselect every selected kno.
 * returns: 
 */
static void
KnSelectProc(Widget w, XEvent *event, String *params, Cardinal *num)
{ 
    KnvasWidget kw = (KnvasWidget) w;
    Cardinal i;
#define THIS kw->knvas
    for(i = 0; i < THIS.sobjs->size; i++){
	KnUnselect(w, (KnO)THIS.sobjs->list[i]);
    }
    KlDecRef(THIS.sobjs);
    THIS.sobjs = KlListNMake(0);
    KlIncRef(THIS.sobjs);
    KnvasUpdateDisplay(w);
#undef THIS
}


/* Knvas's action proc to Zoom / Unzomm the knvas view
   
 * This action just calls the KnvasZoom function, giving the two action
 * parameters as sx and sy zoom factors. If parameter(s) are omitted, returns
 * without zooming.
 
 * returns: */
static void
KnvasZoomProc(Widget w,
	       XEvent *event,
	       String *params,
	       Cardinal *num)
{
    float sx, sy;
    if(*num != 2) {
	XtAppWarningMsg(XtWidgetToApplicationContext(w),
			invalidParameterCount,
			actionParameters,
			KnvasWarning,
			"Action 'KnvasZoom' requires 2 parameters",
			NULL, NULL);
	return;
    }
    sx = atof(params[0]);
    sy = atof(params[1]);
    KnvasZoom(w, sx, sy);
}



 void
KnvasFlipH(Widget w, XEvent *event, String *params, Cardinal *num)
{ 
    KnvasWidget kw = (KnvasWidget) w;
#define THIS kw->knvas   
    if(! THIS.t)
	THIS.t = TransformerMake();
    TransformerTranslate(THIS.t,
			 (-(float)kw->core.width) / 2.0,
			 (-(float)kw->core.height) / 2.0);
    TransformerRotate(THIS.t, 180);
    TransformerTranslate(THIS.t,
			 ((float)kw->core.width) / 2.0,
			 ((float)kw->core.height) / 2.0);
    if(XtIsRealized(w)) {
	XClearArea(XtDisplay(w), XtWindow(w), 0, 0, 0, 0, True);
    }
    if(TransformerIdentity(THIS.t)) {
	XtFree((void *)THIS.t);
	THIS.t = NULL;
    }
#undef THIS
}


static void
KnvasDoRotateView(Widget w, XEvent *event, String *params, Cardinal *num)
{ 
    KnvasWidget kw = (KnvasWidget) w;
#define THIS kw->knvas   
    if(! THIS.t)
	THIS.t = TransformerMake();
    TransformerRotate(THIS.t, 45);
    if(XtIsRealized(w)) {
	XClearArea(XtDisplay(w), XtWindow(w), 0, 0, 0, 0, True);
    }
    if(TransformerIdentity(THIS.t)) {
	XtFree((void *)THIS.t);
	THIS.t = NULL;
    }
#undef THIS
}




static char defaultTranslations[] =
"";


static XtActionsRec actions[] = {
    /* Knvas actions */
    {"KnvasSelect", KnSelectProc},
    {"KnvasZoom", KnvasZoomProc},
    {"KnvasRotateView", KnvasDoRotateView},
    {"KnvasFlipH", KnvasFlipH},
    {"KnvasDrawStart", KnvasDrawStartProc},
    {"KnvasDrawDrag", KnvasDrawDragProc},
    {"KnvasDrawDrop", KnvasDrawDropProc},
};


static XtResource 
resources[] = {
#define offset(field) XtOffset(KnvasWidget, knvas.field)
  {
      XtNinteractorCallback, XtCCallback,
      XtRCallback, sizeof(XtCallbackList),
      offset(interactorCallback), XtRPointer, NULL
  },
  {
      XtNknvasCallback, XtCCallback,
      XtRCallback, sizeof(XtCallbackList),
      offset(knvasCallback), XtRPointer, NULL
  },
  {
      XtNpixmapBuffering, XtCPixmapBuffering,
      XtRBoolean, sizeof(Boolean),
      offset(pixmapBuffering), XtRImmediate, (caddr_t)False
  },
  {
      XtNautoUpdate, XtCAutoUpdate,
      XtRBoolean, sizeof(Boolean),
      offset(autoUpdate), XtRImmediate, (caddr_t)False
  },
  /* These are private resources that should not be accessed from C.
   * Undocumented. 	- alpha - jml - 4/24/95.
   */
  {
      "grab", "Grab",
      "KlO", sizeof(KlO),
      offset(grab), XtRImmediate, (caddr_t)NULL
  },
  {
      "target", "Target",
      "KlO", sizeof(KlO),
      offset(target), XtRImmediate, (caddr_t)NULL
  },
  {
      "defaultTag", "DefaultTag",
      XtRWidget, sizeof(Widget),
      offset(defaultTag), XtRImmediate, (caddr_t)NULL
  },
  {
      "layers", "Layers",
      "KlO", sizeof(KlO),
      offset(layers), XtRImmediate, (caddr_t)NULL
  },
  {
      "selection", "Selection",
      "KlO", sizeof(KlO),
      offset(sobjs), XtRImmediate, (caddr_t)NULL
  },
#undef offset
};






/* dispatching requires two steps:
 * first we set an event handler on the knvas widget, that has to search for 
 * the kno target (the once that must rceive the event)
 * then, we copy translations from the kno to the widget, removing the 
 * event handler and we call XtDispatchEvent to call the translation manager
 * returns: 
 */
static void
DispatchEvent(Widget w, XtPointer client, XEvent *event, Boolean *cont)
{
    KnvasWidget kw = (KnvasWidget) w;
#define THIS kw->knvas
    KnPosition rx, ry;
    XtTMRec tm;

    THIS.in_dispatch = True;
    
    /* save widget translations */
    tm = w->core.tm;

    if(THIS.grab) {
	THIS.target = THIS.grab;
	/* there are target, grabbing events, dispatch */
	if(THIS.target->interactor) {
	    /* interactor may be NULL if this comes from a global interactor */
	    w->core.tm = THIS.target->interactor->tm;
	}
	_XtTranslateEvent(w, event); 
	if(THIS.target->interactor) {
	    THIS.target->interactor->tm = w->core.tm;
	}
	*cont = False;
    }
    else {
	int p;
	KnO layer;
	if(THIS.t) {
	    TransformerInvTransform(THIS.t, event->xbutton.x, 
				    event->xbutton.y, &rx, &ry);
	}
	else {
	    rx = event->xbutton.x;
	    ry = event->xbutton.y;	    
	}
	THIS.dispatched = False;
	/* start seeking from top layer */
	for(p = THIS.layers->size-1; p >= 0; p--) {
	    layer = (KnO)THIS.layers->list[p];
	    /* the dispatched flag must be set to true by any interactor
               function */
	    if(layer->flags & KnSENSITIVE_FLAG) {
		KlSend_dispatch(layer, w, event, (Int)rx, (Int)ry);
		/* here, the target field has been set by the dispatch
		   method */
		if(THIS.dispatched) {
		    *cont = False;
		    goto terminate;
		}
	    }
	}
	/* the event has not been dispatched, so we'll call the translation
           manager with the widget translations. Set the target to ther real
           target = the top kno that contains the point */
	for(p = THIS.layers->size-1; p >= 0; p--) {
	    layer = (KnO)THIS.layers->list[p];
	    /* the dispatched flag must be set to true by any interactor
               function */
	    if((THIS.target = (KnO)KlSend_contains(layer, w, (Int)rx,(Int)ry)))
		goto terminate;
	}
    } 
  terminate:
    w->core.tm = tm;
    THIS.in_dispatch = False;

#undef THIS
}


static void
ClassInitialize(WidgetClass wc)
{
    KnInitialize(True);
    /* register interactor's converters */
    KnInteractorRegisterConverters();
}

static void 
Initialize(Widget w, Widget neww, ArgList args, Cardinal *n)
{    
    KnvasWidget kw = (KnvasWidget) neww;
#define THIS kw->knvas

    THIS.maskgc1 = None;
    THIS.maskgc = None;
    THIS.pixgc = None;
    THIS.ghostgc = None;
    THIS.selectgc = None;
    
    THIS.grab = THIS.target = NULL;
    THIS.gt = NULL;
    
    THIS.layers = KlListNMake(0);
    KlIncRef(THIS.layers);

    THIS.delete = KlListNMake(0);
    KlIncRef(THIS.delete);

    THIS.clear_tags = KlListNMake(0);
    KlIncRef(THIS.clear_tags);

    THIS.sobjs = KlListNMake(0);
    KlIncRef(THIS.sobjs);

    THIS.tags = KlListNMake(0);
    KlIncRef(THIS.tags);

    THIS.interactors = KlListNMake(0);
    KlIncRef(THIS.interactors);

    THIS.defaultTag = KnCreateTag(neww, "default");

    /* With Motif 1.2, button motion events for Btn2 & btn3 are not received if
       we just select ButtonMotion. So we add Button2MotionMask &
       Button3MotionMask - jml - 9/26/95. */
    THIS.smask = KeyPressMask | ButtonPressMask | ButtonReleaseMask |
	KeyReleaseMask | ButtonPressMask | ButtonMotionMask |
	Button2MotionMask | Button3MotionMask ;

    /* create an empty region to be cleared */
    THIS.clear = KnCreateRegion();

    /* at the beginning, we have no transformer */
    THIS.t = NULL;

    /* back pixmap is created when needed */
    THIS.pixmap = None;

    /* scroll fields */
    if(0 == kw->core.width) kw->core.width = 500;
    if(0 == kw->core.height) kw->core.height = 500;
    THIS.xmin = 0;
    THIS.xmax = kw->core.width;
    THIS.ymin = 0;
    THIS.ymax = kw->core.height;
    THIS.hval = THIS.vval = 0;
    THIS.zoomx = THIS.zoomy = 1;

    THIS.in_dispatch = False;
    
#ifdef DO_MOTIF_WIDGET
    /* for a Motif widget, enable an APPLICATION_DEFINED scroll mode */
    THIS.sw = NULL; THIS.hsb = NULL; THIS.vsb = NULL;
    if((XmIsScrolledWindow(kw->core.parent)) &&
	((XmScrolledWindowWidget)(kw->core.parent))->swindow.ScrollPolicy ==
       XmAPPLICATION_DEFINED) {
	Arg args[6];
	Cardinal n;
	THIS.sw = (XmScrolledWindowWidget)kw->core.parent;
	/* Build the vertical scrollbar */
	n = 0;
	XtSetArg(args[n], XmNorientation, (XtArgVal)XmVERTICAL); n++;
	XtSetArg(args[n], XmNminimum, (XtArgVal)THIS.ymin); n++;
	XtSetArg(args[n], XmNmaximum, (XtArgVal)THIS.ymax); n++;
	XtSetArg(args[n], XmNsliderSize, (XtArgVal)kw->core.height); n++;
	THIS.vsb = (XmScrollBarWidget)
	    XmCreateScrollBar((Widget)THIS.sw,
			      "vScrollBar",
			      args, n);
	XtAddCallback((Widget)THIS.vsb, XmNincrementCallback,
		      VScrollCB, kw);
	XtAddCallback((Widget)THIS.vsb, XmNdecrementCallback,
		      VScrollCB, kw);
	XtAddCallback((Widget)THIS.vsb, XmNpageIncrementCallback,
		      VScrollCB, kw);
	XtAddCallback((Widget)THIS.vsb, XmNpageDecrementCallback,
		      VScrollCB, kw);
	XtAddCallback((Widget)THIS.vsb, XmNdragCallback,
		      VScrollCB, kw);
	XtAddCallback((Widget)THIS.vsb, XmNtoTopCallback,
		      VScrollCB, kw);
	XtAddCallback((Widget)THIS.vsb, XmNtoBottomCallback,
		      VScrollCB, kw);
	XtManageChild((Widget)THIS.vsb);
	/* Build the horizontal scrollbar */
	n = 0;
	XtSetArg(args[n], XmNorientation, (XtArgVal)XmHORIZONTAL); n++;
	XtSetArg(args[n], XmNminimum, (XtArgVal)THIS.xmin); n++;
	XtSetArg(args[n], XmNmaximum, (XtArgVal)THIS.xmax); n++;
	XtSetArg(args[n], XmNsliderSize, (XtArgVal)kw->core.width); n++;
	THIS.hsb = (XmScrollBarWidget)
	    XmCreateScrollBar((Widget)THIS.sw,
			      "hScrollBar",
			      args, n);
	XtAddCallback((Widget)THIS.hsb, XmNincrementCallback,
		      HScrollCB, kw);
	XtAddCallback((Widget)THIS.hsb, XmNdecrementCallback,
		      HScrollCB, kw);
	XtAddCallback((Widget)THIS.hsb, XmNpageIncrementCallback,
		      HScrollCB, kw);
	XtAddCallback((Widget)THIS.hsb, XmNpageDecrementCallback,
		      HScrollCB, kw);
	XtAddCallback((Widget)THIS.hsb, XmNdragCallback,
		      HScrollCB, kw);
	XtAddCallback((Widget)THIS.hsb, XmNtoTopCallback,
		      HScrollCB, kw);
	XtAddCallback((Widget)THIS.hsb, XmNtoBottomCallback,
		      HScrollCB, kw);
	XtManageChild((Widget)THIS.hsb);
    }
#endif /* DO_MOTIF_WIDGET */
#undef THIS
}


static Boolean
SetValues(Widget cw, Widget rw, Widget nw,
	  ArgList args, Cardinal *num_args)
{
    KnvasWidget ck = (KnvasWidget)cw;
    KnvasWidget nk = (KnvasWidget)nw;
    Display *dpy = XtDisplay(nw);

    if(nk->knvas.pixmapBuffering && XtIsRealized(nw) &&
       (None == nk->knvas.pixmap)) {
	/* if back buffer is required and not yet built, create it */
	CreateBackPixmap(nw);
    }else {
	/* if we just created the back pixmap, the background color is the
           right one. SO we do this only if we do not create the back pixmap */
	if(nk->knvas.pixmapBuffering &&
	   (nk->core.background_pixel != ck->core.background_pixel)) {
	    /* setting background, clear pixmap with new color */
	    XSetForeground(dpy, nk->knvas.pixgc, nw->core.background_pixel);
	    XFillRectangle(dpy, nk->knvas.pixmap, nk->knvas.pixgc, 0, 0, 
			   nk->knvas.psize.width, nk->knvas.psize.height);
	}
    }
    /* This is to be able to change the target of an interaction. */
    if(nk->knvas.grab != ck->knvas.grab) {
	/* the grab object do not need to be IncRefed, cos it is added on the
           list and so won't be GCed. grab will be resetted with the list. */
	KlListAppend(nk->knvas.sobjs, nk->knvas.grab);
    }
    return False;
}    


static void 
RealizeProc(Widget w, 
	    XtValueMask *value_mask, 
	    XSetWindowAttributes *attributes)
{
    KnvasWidget kw = (KnvasWidget)w;
    Display *dpy = XtDisplay(w);
#define THIS kw->knvas
    Cardinal i;
    
    XtCreateWindow(w, InputOutput, CopyFromParent, *value_mask, attributes);

    /* there is a bug somewhere that is solved like this, but you should better
       find what is exactly hapenning: some objects are cleared while the
       widget is not yet realized */
    KnDestroyRegion(THIS.clear);
    THIS.clear = KnCreateRegion();
    /* if the widget uses pixmap buffering, create the pixmap */
    if(THIS.pixmapBuffering) {
	CreateBackPixmap(w);
    }
    {
	/* create various GCs - maskgc1 will be created on need in knstring */
	XGCValues vals;
        Pixel bg;
	vals.line_style = LineDoubleDash;
	XtVaGetValues(w, XtNbackground, &bg, NULL);
	vals.background = bg ^ WhitePixelOfScreen(DefaultScreenOfDisplay(dpy));
	vals.foreground = bg ^ BlackPixelOfScreen(DefaultScreenOfDisplay(dpy));
	vals.function = GXxor;
	THIS.ghostgc = XCreateGC(dpy, XtWindow(w),
			    GCForeground|GCBackground|GCLineStyle|GCFunction,
			    &vals);
	THIS.selectgc = XCreateGC(dpy, XtWindow(w),
				  GCForeground, &vals);

	vals.background = 0xFFFFFFFFL;
	vals.foreground = 0L;
	vals.function = GXand;
	THIS.maskgc =
	    XCreateGC(dpy, XtWindow(w),
		    GCFunction | GCBackground | GCForeground, &vals);

	
    }
    /* build the widget's stipple */
#ifdef LOOK3D
#define gray_width 16
#define gray_height 16
    {
	static unsigned char gray_bits[] = {
	    0x55, 0x55, 0xaa, 0xaa, 0x55, 0x55,
	    0xaa, 0xaa, 0x55, 0x55, 0xaa, 0xaa,
	    0x55, 0x55, 0xaa, 0xaa, 0x55, 0x55,
	    0xaa, 0xaa, 0x55, 0x55, 0xaa, 0xaa,
	    0x55, 0x55, 0xaa, 0xaa, 0x55, 0x55, 0xaa, 0xaa};
	THIS.stipple = 
	    XCreatePixmapFromBitmapData(dpy, XtWindow(w),
					gray_bits,
				    gray_width, gray_height,
				    BlackPixel(dpy, 0),
				    WhitePixel(dpy, 0), 1);
    }
				    
#endif /* LOOK3D */
    /* realize all tags */
    for(i = 0; i < THIS.tags->size; i++) {
	KnTagRealize((KnTag)THIS.tags->list[i]);
    }
    for(i = 0; i < THIS.interactors->size; i++) {
	KnInteractorRealize((KnInteractor)THIS.interactors->list[i], w);
    }
    XtAddEventHandler(w, THIS.smask, False, DispatchEvent, 0);

    /* build interaction cursors */
    THIS.cursors[KnCENTER_CURSOR] =
	XCreateFontCursor(dpy, XC_fleur);
    THIS.cursors[KnTOP_LEFT_CURSOR] =
	XCreateFontCursor(dpy, XC_top_left_corner);
    THIS.cursors[KnTOP_CURSOR] =
	XCreateFontCursor(dpy, XC_top_side);
    THIS.cursors[KnTOP_RIGHT_CURSOR] =
	XCreateFontCursor(dpy, XC_top_right_corner);
    THIS.cursors[KnBOTTOM_LEFT_CURSOR] =
	XCreateFontCursor(dpy, XC_bottom_left_corner);
    THIS.cursors[KnBOTTOM_CURSOR] =
	XCreateFontCursor(dpy, XC_bottom_side);
    THIS.cursors[KnBOTTOM_RIGHT_CURSOR] =
	XCreateFontCursor(dpy, XC_bottom_right_corner);
    THIS.cursors[KnLEFT_CURSOR] =
	XCreateFontCursor(dpy, XC_left_side);
    THIS.cursors[KnRIGHT_CURSOR] =
	XCreateFontCursor(dpy, XC_right_side);
    THIS.cursors[KnERROR_CURSOR] =
	XCreateFontCursor(dpy, XC_fleur);

    /* This core dumps when integrated with klm */
/*    KlZrtGc(0); */

    THIS.xmax = w->core.width;
    THIS.ymax = w->core.height;
#ifdef APPLICATION_DEFINED_SCROLL_MODE   
    UpdateScrollBars(kw);
#endif /* APPLICATION_DEFINED_SCROLL_MODE */
#undef THIS
} 

static void
DrawOnExpose(Widget w, XEvent *event, KnRegion r);

static void
ExposeProc(Widget w, XEvent *event, Region r)
{
    XRectangle box;
    KnRegion region;

    region = KnCreateRegion();
    XClipBox(r, &box);
    KnUnionRectWithRegion(&box, NULL, region);
    DrawOnExpose(w, event, region);
    KnDestroyRegion(region);
}

void
KnvasSetClipRegion(Widget w, KnRegion r)
{
    KnvasWidget kw = (KnvasWidget)w;
#define THIS kw->knvas
    Cardinal i;
    TagObject tag;
    Display *dpy = XtDisplay(w);
    for(i = 0; i < THIS.tags->size; i++) {
	tag = (TagObject)THIS.tags->list[i];
	tag->tag.clipped = True;
	tag->tag.region = r;
	KnSetRegion(dpy, tag->tag.gc, r);
    }
    KnSetRegion(dpy, THIS.selectgc, r);
#undef THIS
}


static void
DrawOnExpose(Widget w, XEvent *event, KnRegion r)
{
    KnvasWidget kw = (KnvasWidget)w;
#define THIS kw->knvas
    Cardinal i;
    Transformer t = NULL;
    KnO layer;
    KnDrawingContextStruct cs;
    TagObject tag;
    Boolean need_free = False;
	    
    cs.w = w;
    cs.dpy = XtDisplay(w);
    cs.win = XtWindow(w);
    cs.clip = r;

    /* for an expose, we must clip to the exposed region */
    for(i = 0; i < THIS.tags->size; i++) {
	tag = (TagObject)THIS.tags->list[i];
	tag->tag.clipped = True;
	tag->tag.region = r;
	KnSetRegion(cs.dpy, tag->tag.gc, r);
    }
    KnSetRegion(cs.dpy, THIS.selectgc, r);

    for(i = 0; i < THIS.layers->size; i++) {
	layer = (KnO)THIS.layers->list[i];
	t = THIS.t;
	need_free = False;
	if(t) {
	    if(layer->t) {
		t = TransformerMakeCopy(t);
		TransformerPremultiply(t, layer->t);
		if(TransformerIdentity(t)) {
		    XtFree((void *)t);
		    t = NULL;
		}
		else
		    need_free = True;
	    }
	}
	else {
	    t = layer->t;
	}
	KlSend_draw(layer, &cs, t);
	if(need_free) {
	    XtFree((void *)t);
	    need_free = False;
	}
    }
    for(i = 0; i < THIS.tags->size; i++) {
	tag = (TagObject)THIS.tags->list[i];
	XSetClipMask(cs.dpy, tag->tag.gc, None);
	tag->tag.clipped = False;
    }
#undef THIS
}




static void
Redraw(Widget w, XEvent *event, KnRegion r)
{
    KnvasWidget kw = (KnvasWidget)w;
#define THIS kw->knvas
    Cardinal i;
    XRectangle box;
    KnDrawingContextStruct cs;

    

    for(i = 0; i < THIS.tags->size; i++) {
	((TagObject)THIS.tags->list[i])->tag.clipped = False;
	XSetClipMask(XtDisplay(w), ((TagObject)THIS.tags->list[i])->tag.gc, None);
    }
    KnClipBox(r, &box);
    /* enlarge pixmap if needed */
    if((THIS.psize.width < box.width) || 
       (THIS.psize.height < box.height)) {
	box.width = KnMax(box.width, THIS.psize.width);
	box.height = KnMax(box.height, THIS.psize.height);
	XFreePixmap(XtDisplay(w), THIS.pixmap);
	THIS.pixmap = XCreatePixmap(XtDisplay(w), w->core.window,
				    box.width, box.height,
				    w->core.depth);
	THIS.psize = box;
    }
    /* this stands for clearing the pixmap */
    XFillRectangle(XtDisplay(w), THIS.pixmap, THIS.pixgc, 0, 0, 
		   box.width, box.height);
    if(!THIS.t)
	THIS.t = TransformerMake();
    TransformerTranslate(THIS.t, -box.x, -box.y);
    KnSetRegion(XtDisplay(w), THIS.pixgc, r);
    /* translate region, so that we can test intersect with objs */
    KnOffsetRegion(r, -box.x, -box.y);
    cs.w = w;
    cs.dpy = XtDisplay(w);
    cs.win = THIS.pixmap;
    cs.clip = r;
    for(i = 0; i < THIS.layers->size; i++) {
	Transformer tf;   
	Boolean need_free = False;
	KnO child;
	child = (KnO)THIS.layers->list[i];
	tf = THIS.t;
	if(tf) {
	    if(child->t) {
		tf = TransformerMakeCopy(tf);
		TransformerPremultiply(tf, child->t);
		if(TransformerIdentity(tf)) {
		    XtFree((void *)tf);
		    tf = NULL;
		}
		else
		    need_free = True;
	    }
	}
	else {
	    tf = child->t;
	}
	KlSend_draw(child, &cs, tf);
	if(need_free) {
	    XtFree((void *)tf);
	    need_free = False;
	}
    }
    XCopyArea(XtDisplay(w), THIS.pixmap, w->core.window,
	      THIS.pixgc, 0, 0, box.width, box.height,
	      box.x, box.y);
    /* keep pixgc unclipped */
    XSetClipMask(XtDisplay(w), THIS.pixgc, None);
    /* restore local transformer */
    TransformerTranslate(THIS.t, box.x, box.y);
    if(TransformerIdentity(THIS.t)) {
	XtFree((void *)THIS.t);
	THIS.t = NULL;
    }
    
#undef THIS
}


static void
CreateBackPixmap(Widget w)
{
    KnvasWidget kw = (KnvasWidget) w;
#define THIS kw->knvas
    Display *dpy = XtDisplay(w);
    XGCValues values;
    XtGCMask value_mask;
    
    THIS.pixmap = XCreatePixmap(dpy, w->core.window,
				IPSIZE, IPSIZE,
				w->core.depth);
    THIS.pixgc = XCreateGC(dpy, THIS.pixmap, 0, 0);
    XSetForeground(dpy, THIS.pixgc, w->core.background_pixel);
    XFillRectangle(dpy, THIS.pixmap, THIS.pixgc, 0, 0, 
		   IPSIZE, IPSIZE);
    THIS.psize.x = THIS.psize.y = 0;
    THIS.psize.width = IPSIZE;
    THIS.psize.height = IPSIZE;
#undef THIS    
}    

#ifdef DO_ATHENA_WIDGET
static CompositeClassExtensionRec acceptObjectsExtension = {
    NULL,			/* next extension */
    NULLQUARK,			/* record_type */
    1L,
    0,
    True,
    False
};
#endif 


KnvasClassRec knvasClassRec = {
  { /* core fields */
#ifdef DO_MOTIF_WIDGET
      /* superclass		*/	(WidgetClass) &xmManagerClassRec,
#else /* DO_MOTIF_WIDGET */
    /* superclass		*/	(WidgetClass) &compositeClassRec,
#endif /* DO_MOTIF_WIDGET */
    /* class_name		*/	"Knvas",
    /* widget_size		*/	sizeof(KnvasRec),
    /* class_initialize		*/	NULL,
    /* class_part_initialize	*/	ClassInitialize,
    /* class_inited		*/	FALSE,
    /* initialize		*/	Initialize,
    /* initialize_hook		*/	NULL,
    /* realize			*/	RealizeProc,
    /* actions			*/	actions,
    /* num_actions		*/	XtNumber(actions),
    /* resources		*/	resources,
    /* num_resources		*/	XtNumber(resources),
    /* xrm_class		*/	NULLQUARK,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	XtExposeCompressMaximal,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	Destroy,
    /* resize			*/	Resize,
    /* expose			*/	ExposeProc,
    /* set_values		*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	XtInheritTranslations,
    /* query_geometry		*/	XtInheritQueryGeometry,
    /* display_accelerator	*/	XtInheritDisplayAccelerator,
    /* extension		*/	NULL
  },
  { /* composite fields */
    /* geometry_manager     */  XtInheritGeometryManager,
    /* change_managed       */  XtInheritChangeManaged,
    /* insert_child	    */  XtInheritInsertChild,
    /* delete_child	    */  XtInheritDeleteChild,
#ifndef DO_ATHENA_WIDGET				
    /* extension	    */  NULL
#else
    /* extension	    */  &acceptObjectsExtension
#endif
    },
#ifdef DO_MOTIF_WIDGET
    {
    /* constraint_class fields */
	/* resources			*/ NULL,
	/* num_resources		*/ 0,
	/* constraint_size		*/ 0,
	/* initialize			*/ NULL,
	/* destroy			*/ NULL,
	/* set_values			*/ NULL,
	/* extension			*/ NULL
    },
    {
    /* manager_class fields */
	/* translations			*/  XtInheritTranslations,
	/* syn_resources		*/  NULL,
	/* num_syn_resources		*/  0,
	/* syn_constraint_resources	*/  NULL,
	/* num_syn_constraint_resources */  0,
	/* parent_process		*/  XmInheritParentProcess,
	/* extension			*/  NULL
    },
#endif /* DO_MOTIF_WIDGET */
  { /* knvas fields */
    /* empty			*/	0
  }
};

WidgetClass knvasWidgetClass = (WidgetClass)&knvasClassRec;


/*
 *
 * general purpose knvas functions
 *
 */

void
KnvasClearArea(Widget w, XPoint *points, Cardinal n, XRectangle *extra)
{
    
    KnvasWidget kw = (KnvasWidget)w;
#define THIS kw->knvas
    XPoint *top;
    XPoint ul, br;
    Cardinal i;
    XRectangle clear;

    if(! XtIsRealized(w)) return;
    if(THIS.t) {
	top = (XPoint *)XtMalloc(sizeof(XPoint) * n);
	TransformerTransformPointList(THIS.t, points, n, top);
    }
    else {
	top = points;
    }
	
    ul = top[0];
    br = top[0];
    for(i = 1; i < n ; i++) {
	if(top[i].x < ul.x)
	   ul.x = top[i].x;
	else if(top[i].x > br.x)
	   br.x = top[i].x;
	if(top[i].y < ul.y)
	   ul.y = top[i].y;
	else if(top[i].y > br.y)
	   br.y = top[i].y;
    }

    clear.x = ul.x;
    clear.y = ul.y;
    clear.width = br.x - clear.x;
    clear.height = br.y - clear.y;
    if(extra) {
	clear.x -= extra->x;
	clear.y -= extra->y;
	clear.width += extra->width;
	clear.height += extra->height;
    }
#ifdef APPLICATION_DEFINED_SCROLL_MODE    
    if((clear.x / THIS.zoomx) < THIS.xmin) {
	THIS.xmin = clear.x / THIS.zoomx;
	update = True;
    }
    if((clear.y / THIS.zoomy) < THIS.ymin) {
	THIS.ymin = clear.y / THIS.zoomy;
	update = True;
    }
    if( (int) ((clear.x+(int)clear.width) / THIS.zoomx) > THIS.xmax) {
	THIS.xmax =
	    (int)((float)(clear.x+(int)clear.width) / (float)THIS.zoomx);
	update = True;
    }
    if(((float)(clear.y+(int)clear.height) / THIS.zoomy) > (float)THIS.ymax) {
	THIS.ymax = (float)(clear.y+(int)clear.height) / (float)THIS.zoomy;
	update = True;
    }
    if(update) {
	UpdateScrollBars(kw);
    }

#endif /* APPLICATION_DEFINED_SCROLL_MODE */
    
    KnUnionRectWithRegion(&clear, THIS.clear, THIS.clear);
    if(XtIsRealized(w) && THIS.autoUpdate && ! THIS.in_dispatch) {
	KnvasUpdateDisplay(w);
    }
    if(top != points) {
	XtFree((void *)top);
    }
#undef THIS
}



Transformer
KnvasGetTransformer(Widget w)
{
    KnvasWidget kw = (KnvasWidget)w;
#define THIS kw->knvas
    return THIS.t;
#undef THIS
}




void
KnvasZoom(Widget w,
	  float sx,
	  float sy)
{ 
    KnvasWidget kw = (KnvasWidget) w;
#define THIS kw->knvas
    if(! THIS.t)
	THIS.t = TransformerMake();
    TransformerScale(THIS.t , sx, sy);
    if(XtIsRealized(w)) {
	XClearArea(XtDisplay(w), XtWindow(w), 0, 0, 0, 0, True);
    }
    if(TransformerIdentity(THIS.t)) {
	XtFree((void *)THIS.t);
	THIS.t = NULL;
    }
#ifdef APPLICATION_DEFINED_SCROLL_MODE    
    THIS.zoomx *= sx;
    THIS.zoomy *= sy;
    UpdateScrollBars(kw);
#endif /* APPLICATION_DEFINED_SCROLL_MODE */
#undef THIS
}


#ifdef LOOK3D
Pixmap
KnvasGetStipple(Widget w)
{
    return ((KnvasWidget)w)->knvas.stipple;
}
#endif /* LOOK3D */


KnTag 
KnvasRegisterTag(Widget w, KnTag kntag)
{
    TagObject tag = (TagObject)kntag;
    KnvasWidget kw = (KnvasWidget)w;
#define THIS kw->knvas
    

    KlListAppendNonKlO(THIS.tags, tag);
    if(XtIsRealized(w)) {
	KnTagRealize((KnTag)tag);
    }
    tag->tag.id = THIS.tags->size-1;
    return (KnTag)tag;
#undef THIS
}



KnInteractor
KnvasRegisterInteractor(Widget w, KnInteractor interactor)
{
    KnvasWidget kw = (KnvasWidget)w;
#define THIS kw->knvas

    KlListAppend(THIS.interactors, interactor);

    return (KnInteractor)interactor;
#undef THIS
}


void
KnvasDeleteObject(Widget w, KnO kno)
{
    KnvasWidget kw = (KnvasWidget)w;
#define THIS kw->knvas
    KlListAppend(THIS.delete, kno);
    if(XtIsRealized(w) && THIS.autoUpdate && ! THIS.in_dispatch) {
	KnvasUpdateDisplay(w);
    }
#undef THIS
}


void
KnvasClearTaggedObjects(Widget w, KnTag kntag)
{
    KnvasWidget kw = (KnvasWidget)w;
#define THIS kw->knvas
    /* Here, we just check if the tag is already registred for clear. If not,
       we add it in the clear_tags list */
    int i;
    KnTagId id;
    TagObject rtag, tag = (TagObject)kntag;
    id = tag->tag.id;
    for(i = 0; i < THIS.clear_tags->size; i++) {
	rtag = (TagObject)THIS.clear_tags->list[i];
	if(id == rtag->tag.id) {
	    /* tag is already registered for clear */
	    return;
	}
    }
    KlListAppendNonKlO(THIS.clear_tags, tag);
    if(XtIsRealized(w) && THIS.autoUpdate && ! THIS.in_dispatch) {
	KnvasUpdateDisplay(w);
    }
#undef THIS
}




static void
ClearTaggedObjectsInGroup(KnGroup g, Widget knvas)
{
    KnvasWidget kw = (KnvasWidget)knvas;
#define THIS kw->knvas
    int i, t;
    KnO child;
    KnTagId ctid;
    TagObject tag;
    for(i = 0; i < g->children->size; i++) {
	child = (KnO)g->children->list[i];
	if(child && KnIsMapped(child)) {
	    if( KlIsOfType(child, KnGroupClass) ){
		ClearTaggedObjectsInGroup((KnGroup)child, knvas);
	    }
	    else {
		ctid = child->tid;
		for(t = 0; t < THIS.clear_tags->size; t++) {
		    tag = (TagObject)THIS.clear_tags->list[t];
		    if(tag->tag.id == ctid)
			KlSend_clear(child, knvas);
		}
	    }
	}
	
    }
#undef THIS
}    



static void
ClearTaggedObjects(Widget w)
{
    KnvasWidget kw = (KnvasWidget)w;
#define THIS kw->knvas
    /* we browse the whole objects hierarchy to know which objects are usiong
       the registered tags and we clear them */
    int i;

    for(i = 0; i <THIS.layers->size; i++) {
       ClearTaggedObjectsInGroup((KnGroup)THIS.layers->list[i], w);
    }

    Free(THIS.clear_tags->list);
    THIS.clear_tags->list = 0;
    THIS.clear_tags->size = 0;
    
#undef THIS
}    


static void
KnvasRemoveKnOs(Widget w)
{
    KnvasWidget kw = (KnvasWidget)w;
#define THIS kw->knvas
    Cardinal n, m;
    KnO kno;

    for(n = 0; n < THIS.delete->size; n++) {
	kno = (KnO)(THIS.delete->list[n]);
	/* remove it from the selection list if needed */
	for(m = 0; m < THIS.sobjs->size; m++) {
	    if((KnO)THIS.sobjs->list[m] == kno) {
		KlListDeletePos(THIS.sobjs, m);
		break;
	    }
	}
	KlSend_clear(kno, w);
	KlSend_destroy(kno);
	/* would the object be freed by the klone - GC ??? */

    }
    KlDecRef(THIS.delete);
    THIS.delete = KlListNMake(0);
    KlIncRef(THIS.delete);
#undef THIS
}


void
KnvasUpdateDisplay(Widget w)
{
    KnvasWidget kw = (KnvasWidget)w;
#define THIS kw->knvas
    Boolean autoUpdate;

    if(!XtIsRealized(w)) return; /* nothing to do */
    
    autoUpdate = THIS.autoUpdate;
    THIS.autoUpdate = False;	/* disable recursive call to update */
    /* call the expose proc */
    
    if((THIS.clear_tags) && (THIS.clear_tags->size)) {
	ClearTaggedObjects(w);
    }
    if((THIS.delete) && (THIS.delete->size > 0))
	KnvasRemoveKnOs(w);
    if(!(KnEmptyRegion(THIS.clear))) {
	/* add 1 extra pixel to deal with float->int covnersion errors */
	THIS.clear->x1 -= 1;
	THIS.clear->x2 += 1;
	THIS.clear->y1 -= 1;
	THIS.clear->y2 += 1;
	if(THIS.pixmapBuffering) {
	    Redraw(w, 0, THIS.clear);
	}
	else {
	    XRectangle box;
	    KnRegion clip;
	    KnClipBox(THIS.clear, &box);
	    clip = KnCreateRegion();
	    KnUnionRectWithRegion(&box, clip, clip);
	    XClearArea(XtDisplay(w), XtWindow(w), box.x, box.y, 
		       box.width, box.height, False);
	    DrawOnExpose(w, 0, clip);
	    KnDestroyRegion(clip);
	}
	KnDestroyRegion(THIS.clear);
	THIS.clear = KnCreateRegion();
    };
    THIS.autoUpdate = autoUpdate; /* restore previous state */
#undef THIS
}


void
KnvasUpdateLayers(Widget w)
{
    KnvasWidget kw = (KnvasWidget)w;
#define THIS kw->knvas
    Cardinal l, c;
    KnLayer layer;
    for(l = 0; l < THIS.layers->size; l++) {
	layer = (KnLayer)THIS.layers->list[l];
	for(c = 0; c < layer->parents->size; c++) {
	    KnvasUpdateDisplay((Widget)layer->parents->list[c]);
	}
    }
    
#undef THIS
}


void
KnvasAddLayer(Widget knvas, KnLayer layer)
{
    KnvasWidget kw = (KnvasWidget)knvas;
#define THIS kw->knvas
    KlListAppend(THIS.layers, layer);
#undef THIS
    
}


void
KnvasRemoveLayer(Widget knvas, KnLayer layer)
{
    KnvasWidget kw = (KnvasWidget)knvas;
#define THIS kw->knvas
    /* here, we must put the layer in the kill list, so that it will be keep
       until every object is cleared */
    KnvasDeleteObject(knvas, (KnO)layer);
    KlListDeleteObj(THIS.layers, layer);
#undef THIS
    
}



KnTag
KnvasGetDefaultTag(Widget w)
{
    KnvasWidget kw = (KnvasWidget)w;
    return kw->knvas.defaultTag;
}

void
KnvasSetDefaultTag(Widget w, KnTag tag)
{
    KnvasWidget kw = (KnvasWidget)w;
    kw->knvas.defaultTag = tag;
    /* do not incref tags */
}


KnLayer
KnvasGetDefaultLayer(Widget w)
{
    KnvasWidget kw = (KnvasWidget)w;
    return (KnLayer)kw->knvas.layers->list[0];
}



Widget
KnCreateKnvas(Widget parent, char *name, Arg *args, Cardinal numargs)
{
    return XtCreateWidget(name, knvasWidgetClass,parent, args, numargs);
}



void
KnvasAddInteractor(Widget w, KnInteractor intr)
{
    XtOverrideTranslations(w, intr->tm.translations);
}




void
KnvasShareLayer(Widget parent,	
		KnLayer layer)
{
    /* HACK NOT TO CALL KlIncRef ON A WIDGET  - start - */
    /* we use a KlList to store the widget list, so we need to hack not to call
       KlIncRef on the parent widget: we add the layer (which is a KlO) into
       the list and physically replace the element with the widget; then we
       call DecRef on the layer, because IncRef has been called by the
       KlListAppend func */
    KlListAppend(layer->parents, layer);
    layer->parents->list[layer->parents->size - 1] = (KlO)parent;
    KlDecRef(layer);
    /* HACK NOT TO CALL KlIncRef ON A WIDGET  - end - */
    KnvasAddLayer(parent, layer);
}




void
#ifdef _NO_PROTO
KnvasTranslate(w, dx, dy)
    Widget w;
    int dx;
    int dy;
#else /* _NO_PROTO */
KnvasTranslate(Widget w,
	       int dx,
	       int dy)
#endif /* _NO_PROTO */
{ 
    KnvasWidget kw = (KnvasWidget) w;
#define THIS kw->knvas
    if( ! THIS.t) {
	THIS.t = TransformerMake();
    }
    TransformerTranslate(THIS.t, dx, dy);
#undef THIS
}





#ifdef DO_MOTIF_WIDGET

static void 
#ifdef _NO_PROTO
VScrollCB(w, client, call)
    Widget w;
    XtPointer client;
    XtPointer call;
#else /* _NO_PROTO */
VScrollCB(
    Widget w,
    XtPointer client,
    XtPointer call)
#endif /* _NO_PROTO */
{
KnvasWidget kw = (KnvasWidget)client;    
#define THIS kw->knvas    
    XmScrollBarCallbackStruct *cs = (XmScrollBarCallbackStruct *)call;
    if(cs->value != THIS.vval) {
	int dy;
	dy = THIS.vval - cs->value;
	KnvasTranslate((Widget)kw, 0, dy);
	THIS.vval = cs->value / THIS.zoomy;
	ClearWidget((Widget)kw);
    }
#undef THIS
}    



static void 
#ifdef _NO_PROTO
HScrollCB(w, client, call)
    Widget w;
    XtPointer client;
    XtPointer call;
#else /* _NO_PROTO */
HScrollCB(
    Widget w,
    XtPointer client,
    XtPointer call)
#endif /* _NO_PROTO */
{
KnvasWidget kw = (KnvasWidget)client;    
#define THIS kw->knvas    
    XmScrollBarCallbackStruct *cs = (XmScrollBarCallbackStruct *)call;
    if(cs->value != THIS.hval) {
	KnvasTranslate((Widget)kw, THIS.hval - cs->value, 0);
	THIS.hval = cs->value;
	ClearWidget((Widget)kw);
    }
#undef THIS
}    



static void 
#ifdef _NO_PROTO
UpdateScrollBars(kw)
    KnvasWidget kw;
#else /* _NO_PROTO */
UpdateScrollBars(KnvasWidget kw)
#endif /* _NO_PROTO */
{
#define THIS kw->knvas
    Arg args[4];
    Cardinal n;
    int size;
    
    if(THIS.hsb) {
	/*  ***************           HORIZONTAL        ***************  */
	int xmax = (float)THIS.xmax * THIS.zoomx;
	int xmin = (float)THIS.xmin * THIS.zoomx;
	int hval = (float)THIS.hval * THIS.zoomx;
	n = 0;
	if(xmax < (int)kw->core.width) {
	    xmax = (int)kw->core.width;
	}
	size = kw->core.width;
	if(size > xmax -  hval) {
	    hval = xmax - size;
	}
	XtSetArg(args[n], XmNminimum, xmin); n++;
	XtSetArg(args[n], XmNmaximum, xmax);n++;
	XtSetArg(args[n], XmNsliderSize, size);n++;
	XtSetArg(args[n], XmNvalue, hval);n++;
	XtSetValues((Widget)THIS.hsb, args, n);
    }
    if(THIS.vsb) {
	/*  ***************           VERTICAL        ***************  */
	int ymax = (float)THIS.ymax * THIS.zoomy;
	int ymin = (float)THIS.ymin * THIS.zoomy;
	int vval = (float)THIS.vval * THIS.zoomy;
	n = 0;
	if(ymax < (int)kw->core.height) {
	    ymax = kw->core.height;
	}
	size = kw->core.height;
	if(size > ymax -  vval) {
	    vval = ymax -  size;
	}
	XtSetArg(args[n], XmNminimum, ymin); n++;
	XtSetArg(args[n], XmNmaximum, ymax);n++;
	XtSetArg(args[n], XmNsliderSize, size);n++;
	XtSetArg(args[n], XmNvalue, vval);n++;
	XtSetValues((Widget)THIS.vsb, args, n);
    }
#undef THIS    
}
#endif /* DO_MOTIF_WIDGET */




static void
#ifdef _NO_PROTO
Resize(w)
    Widget w;
#else /* _NO_PROTO */ void 
Resize(Widget w)
#endif /* _NO_PROTO */
{
#ifdef APPLICATION_DEFINED_SCROLL_MODE    
    UpdateScrollBars((KnvasWidget)w);    
#endif /* APPLICATION_DEFINED_SCROLL_MODE */
}






static void
Destroy(Widget w)
{
    KnvasWidget kw = (KnvasWidget) w;
#define THIS kw->knvas
    int i;

    KlDecRef(THIS.delete);
    
    KlDecRef(THIS.layers);

    KlDecRef(THIS.sobjs);

    KlDecRef(THIS.interactors);

    KnDestroyRegion(THIS.clear);

    if(None != THIS.pixmap) {
	XFreePixmap(XtDisplay(w), THIS.pixmap);
    }
    if(NULL != THIS.t) {
	XtFree((char*)THIS.t);
    }

    
    if(None != THIS.pixgc) {
	XFreeGC(XtDisplay(w), THIS.pixgc);
    }
    if(None != THIS.ghostgc) {
	XFreeGC(XtDisplay(w), THIS.ghostgc);
    }
    if(None != THIS.selectgc) {
	XFreeGC(XtDisplay(w), THIS.selectgc);
    }
    
#ifdef LOOK3D
    if(None != THIS.stipple) {
	XFreePixmap(XtDisplay(w), THIS.stipple);
    }
#endif /* LOOK3D */

    for(i = 0; i < THIS.tags->size; i++) {
	XtDestroyWidget((KnTag)THIS.tags->list[i]);
    }

    
    THIS.clear_tags->size = 0;	/* prevent DecRef on elements */
    KlDecRef(THIS.clear_tags);
    THIS.tags->size = 0;	/* prevent DecRef on elements */
    KlDecRef(THIS.tags);

    
    XFreeCursor(XtDisplay(w), THIS.cursors[KnCENTER_CURSOR]);
    XFreeCursor(XtDisplay(w), THIS.cursors[KnTOP_LEFT_CURSOR]);
    XFreeCursor(XtDisplay(w), THIS.cursors[KnTOP_CURSOR]);
    XFreeCursor(XtDisplay(w), THIS.cursors[KnTOP_RIGHT_CURSOR]);
    XFreeCursor(XtDisplay(w), THIS.cursors[KnBOTTOM_LEFT_CURSOR]);
    XFreeCursor(XtDisplay(w), THIS.cursors[KnBOTTOM_CURSOR]);
    XFreeCursor(XtDisplay(w), THIS.cursors[KnBOTTOM_RIGHT_CURSOR]);
    XFreeCursor(XtDisplay(w), THIS.cursors[KnLEFT_CURSOR]);
    XFreeCursor(XtDisplay(w), THIS.cursors[KnRIGHT_CURSOR]);
    XFreeCursor(XtDisplay(w), THIS.cursors[KnERROR_CURSOR]);


#undef THIS
}




KlList
KnvasGetSelection(Widget w)
{
    KnvasWidget kw = (KnvasWidget)w;
    return kw->knvas.sobjs;
}
