static char rcsid[]="$Id: hash.c,v 1.3 95/07/28 15:22:55 leon Exp $";
/* Copyright 1989-93 GROUPE BULL -- See license conditions in file COPYRIGHT */

/***************************************************************************
 * khash.c
 *
 * This is a simple generic package for hash tables. According to the
 * mode of the hash table, it will acccept (or not) items with the same
 * key.
 *
 * static:
 *
 * externs:
 * KnHashtableMake  : make a new hashtable
 * KnHashtableAdd   : add the given entry in hashtable
 * KnHashtableFind
 * KnHashtableFindNext 
 * KnHashtableMap:
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <X11/Intrinsic.h>	/* for XtNewString */

#include "hash.h"


/*
 * A few std cmp/hash function sets
 */

unsigned long KnHashCString (char *cstr) {
    unsigned long ret = 0;
    unsigned long ctr = 0;

    while (*(char*)cstr) {
        ret ^= *(char*)cstr++ << ctr;
        ctr = (ctr + 1) % sizeof (void *);
    }
    return (ret) ;
}

int OsCmpCString (char *s1, char *s2) {
    return (strcmp (s1, s2)) ;
}


/***************************************************************************
 * KnHashtableMake: make a new hashtable.
 *
 * RETURN: KnHashtable, exit on error
 ***************************************************************************/

KnHashtable KnHashtableMake (size)
    unsigned int  size ;
{
    KnHashtable hashtable ;

    if ( ( ! ( hashtable = (KnHashtable) malloc (sizeof (KnHashtableRec))))
	|| ( ! (hashtable->slots = (KnHashslot *) malloc (sizeof (KnHashslot *) * size))))
	
    {
	perror ("KnHashtableMake: malloc ") ;
	exit (1) ;
    }

    memset ((char *) hashtable->slots, 0, sizeof (KnHashslot *) * size) ;
    hashtable->size         = size ;
    hashtable->item_count   = 0 ;
    hashtable->hash_fct     = KnHashCString ;
    hashtable->cmp_fct      = OsCmpCString ;
    hashtable->mode         = 0 ;
    hashtable->current_slot = (KnHashslot) 0 ;

    return (hashtable) ;
}

/***************************************************************************
 * KnHashtableAdd: add the given item to hashtable
 * KnHashtableRemove: remove given item from hashtable.
 *
 * Slightly depends on the hashtable mode for Add. (Overide)
 * Removes compare item's adresses, not items key.
 *
 * RETURN: a pointer to the item (might != arg, because already in)
 ***************************************************************************/

void *KnHashtableAdd (hashtable, item, value)
    register KnHashtable hashtable ;
    char *item;
    void               *value ;

{
    register KnHashslot slot ;
    KnHashslot          *position ;

/*
 * Get the correct slot, check for present :
 */
    position = &(hashtable->slots[(hashtable->hash_fct)(item) % hashtable->size]) ;
    slot     = *position ;
    if ( hashtable->mode & HASH_OVERRIDE )
    {
	while ( slot )
	{
	    if ( ! (hashtable->cmp_fct)(slot->item, item) )
		return (slot->item) ;
	    slot = slot->next ;
	}
    }
/*
 * We must add it :
 */
    if ( ! (slot = (KnHashslot) malloc (sizeof (KnHashslotRec))) )
    {
	perror ("KnHashtableAdd, malloc ") ;
	exit (1) ;
    }

    (hashtable->item_count)++ ;
    slot->item = XtNewString(item) ;
    slot->value = value;
    slot->next = *position ;
    *position  = slot ;
    return (slot) ;
}

void *KnHashtableRemove (hashtable, item, value)
    KnHashtable hashtable ;
    char *item;
    void      *value ;

{
    register KnHashslot slot, *pslot ;
    unsigned long     hashval ;

    hashval = (hashtable->hash_fct) (item) % hashtable->size ;
    slot  = hashtable->slots[hashval] ;
    pslot = &(hashtable->slots[hashval]) ;
    while ( slot )
    {
	if ( ! strcmp(slot->item, item) )
	{
	    (hashtable->item_count)-- ;
	    *pslot = slot->next ;
	    free(slot->item);
	    free (slot) ;
	    return (item) ;
	}
	pslot = &(slot->next) ;
	slot  = slot->next ;
    }

    return ((void *) 0) ;
}

/***************************************************************************
 * KnHashtableFind: retrieve the given entry
 *
 * If the hashtable is not in override mode, set its internal lookup
 * pointer to the found item, so that FindNext can proceed.
 *
 * RETURN: the entry in hashtable
 ***************************************************************************/

void *KnHashtableFind (hashtable, item)
    register KnHashtable hashtable ;
    char               *item ;

{
    register KnHashslot slot ;

    slot = hashtable->slots[(hashtable->hash_fct)(item) % hashtable->size] ;
    while ( slot )
    {
	if ( ! (hashtable->cmp_fct)(slot->item, item) )
	{
	    if ( ! (hashtable->mode & HASH_OVERRIDE) )
		hashtable->current_slot = slot ;
	    return (slot->value) ;
	}
	slot = slot->next ;
    }
    
    return ((void *) 0) ;
}
