/* Copyright 1994 GROUPE BULL -- See license conditions in file COPYRIGHT */
/* $Id: pcache.c,v 1.8 95/09/25 14:55:30 leon Exp $ */
#include <stdio.h>
#include <math.h>

#define XtStrlen(s)                   ((s) ? strlen(s) : 0)
#define Abs(x)                        (((x) > 0) ? (x) : -(x))
#define Min(x, y)                     (((x) < (y)) ? (x) : (y))
#define Max(x, y)                     (((x) > (y)) ? (x) : (y))
#define Roundint(x)                   floor((x) + 0.5)
#define LOCAL_SEARCH_PATH "./%N%S:./%T/%N%S"

#define DefaultCloseness 40000

/* an application global pixmap cache */
#include "pcache.h"

#include "koala.c"

 XImage *
ScalePixmapImage(Screen *screen, XImage *src, double scale_x, double scale_y);
 XImage *
ScaleMaskImage(Screen *screen, XImage *src, double scale_x, double scale_y);

/*
  The cache is devided in 3 levels

  1 - file               the file is loaded
  2 - screen / colors    the XImages are built
  3 - size               pixmap and mask are built in the server
*/

typedef struct _PCacheFileStruct {
    String name;
    XpmImage *xpmimage;
    XpmInfo *xpminfos;
    struct _PCacheImageStruct *images;
    struct _PCacheFileStruct *next;
} PCacheFileStruct, *PCacheFile;



typedef struct _PCacheImageStruct {
    XImage *image_pixmap;	/* pixmap image pixmap */
    XImage *image_mask;		/* pixmap image mask */
    Screen *screen;
    XpmColorSymbol *symcolors;
    Cardinal nsymcolors;
    struct _PCachePixmapStruct *pixmaps;
    struct _PCacheImageStruct *next;
} PCacheImageStruct, *PCacheImage;





/* cache data */
static PCacheFile PC = NULL;




char *
PCacheLocatePixmapFile(Display *dpy, char *name)
{
    String res;
    
    /*
    ** step 1: search local dirs
    */
    res = XtResolvePathname(dpy, "bitmaps", name, ".xpm", 
			    LOCAL_SEARCH_PATH, 0, 0, 0);
    if(res) return res;

    /*
    ** step 2: search XFILESEARCHPATH
    */
    res = XtResolvePathname(dpy, "bitmaps", name, ".xpm", 0, 0, 0, 0);
    if(res) return res;

    /*
    ** step 3: search HOME
    */
    {
	char path[255];
	sprintf(path, "%s/%%T/%%N%%S", getenv("HOME"));
	res = XtResolvePathname(dpy, "bitmaps", name, ".xpm", 
				path, 0, 0, NULL);
	if(res) return res;
    }

    /*
    ** step 4: search XUSERFILESEARCHPATH
    */
    {
	String path = (char *)getenv("XUSERFILESEARCHPATH");
	res = XtResolvePathname(dpy, "bitmaps", name, ".xpm", 
				path, 0, 0, 0);
	if(res) return res;
    }
    XtAppWarning(XtDisplayToApplicationContext(dpy),
		 "PCacheLocatePixmapFile: unable to locate pixmap.");
    return False;
}




static void
PCacheCreateInstanceUnscaled(PCacheFile pcf, PCacheImage pci,
			     PCachePixmap pcp, PCacheAttributes values)
{
    Display *dpy = DisplayOfScreen(pci->screen);
#ifdef DebugPCache
    fprintf(stderr, "PCacheCreateInstanceUncaled: building %s - (%d,%d)\n",
	    pcf->name, values->width, values->height);
#endif
    /* create pixmap */
    pcp->pixmap = 
	XCreatePixmap(dpy,
		      RootWindowOfScreen(pci->screen),
		      values->width,
		      values->height,
		      pci->image_pixmap->depth);
    XPutImage(dpy,
	      pcp->pixmap,
	      DefaultGCOfScreen(pci->screen),
	      pci->image_pixmap,
	      0, 0,
	      0, 0,
	      values->width,
	      values->height);
    /* create mask */
    if(pci->image_mask) {
	GC gc;
	pcp->mask = 
	    XCreatePixmap(dpy,
			  RootWindowOfScreen(pci->screen),
			  values->width,
			  values->height,
			  pci->image_mask->depth);
	gc = XCreateGC(dpy,
		       pcp->mask,
		       0, NULL);
	XPutImage(dpy,
		  pcp->mask,
		  gc,
		  pci->image_mask,
		  0, 0,
		  0, 0,
		  values->width,
		  values->height);
	XFreeGC(dpy, gc);
    }
    else {
	pcp->mask = None;
    }
	      
}



static void
PCacheCreateInstanceScaled(PCacheFile pcf, PCacheImage pci,
			   PCachePixmap pcp, PCacheAttributes values,
			   double scale_x, double scale_y)
{
    XImage *tmpip,		/* tmp image pixmap */
	*tmpim;			/* tmp image mask */
    Display *dpy = DisplayOfScreen(pci->screen);

#ifdef DebugPCache
    fprintf(stderr, "PCacheCreateInstanceScaled: scaling %s - (%d,%d)\n",
	    pcf->name, values->width, values->height);
#endif
    tmpim = NULL;
    tmpip =
	ScalePixmapImage(pci->screen, pci->image_pixmap, scale_x, scale_y);
    /* create pixmap */
    pcp->pixmap = 
	XCreatePixmap(dpy,
		      RootWindowOfScreen(pci->screen),
		      values->width,
		      values->height,
		      tmpip->depth);
    XPutImage(dpy,
	      pcp->pixmap,
	      DefaultGCOfScreen(pci->screen),
	      tmpip,
	      0, 0,
	      0, 0,
	      values->width,
	      values->height);
    /* create mask */
    if(pci->image_mask) {
	GC gc;
	tmpim = ScaleMaskImage(pci->screen, pci->image_mask, scale_x, scale_y);
	pcp->mask = 
	    XCreatePixmap(dpy,
			  RootWindowOfScreen(pci->screen),
			  values->width,
			  values->height,
			  tmpim->depth);
	gc = XCreateGC(dpy,
		       pcp->mask,
		       0, NULL);
	XPutImage(dpy,
		  pcp->mask,
		  gc,
		  tmpim,
		  0, 0,
		  0, 0,
		  values->width,
		  values->height);
	XFreeGC(dpy, gc);
    }
    else {
	pcp->mask = None;
    }
    if(tmpip)
	XDestroyImage(tmpip);
    if(tmpim)
	XDestroyImage(tmpim);
}



static void
PCachePixmapBuild(PCacheFile pcf, PCacheImage pci, PCachePixmap pcp,
		  PCacheAttributes values)
{
    if((values->width == values->o_width) &&
       (values->height == values->o_height)) {
	PCacheCreateInstanceUnscaled(pcf, pci, pcp, values);
    }
    else {
	double scale_x, scale_y;
	scale_x = (double)values->width / (double)pci->image_pixmap->width;
	scale_y = (double)values->height / (double)pci->image_pixmap->height;
	PCacheCreateInstanceScaled(pcf, pci, pcp, values, scale_x, scale_y);
    }
    pcp->width = values->width;
    pcp->height = values->height;
    
}

static Boolean
PCacheImageBuild(PCacheFile pcf, PCacheImage pci, PCacheAttributes values)
{
    XpmAttributes att;
    int status;
#ifdef DebugPCache
    fprintf(stderr, "PCacheImageBuild: building new image for %s\n", pcf->name);
#endif
    att.valuemask = 0;
    if(values->valuemask & PCACHE_SYM_COLORS) {
	att.valuemask |= XpmColorSymbols;
	att.colorsymbols = values->symcolors;
	att.numsymbols = values->nsymcolors;
    }
    att.valuemask |= XpmExtensions|XpmCloseness;
    att.closeness = DefaultCloseness;
    status =
	XpmCreateImageFromXpmImage(DisplayOfScreen(pci->screen),
				   pcf->xpmimage,
				   &pci->image_pixmap,
				   &pci->image_mask,
				   &att);
    if(status != XpmSuccess) {
	fprintf(stderr, "PCacheImageBuild %s\n", XpmGetErrorString(status));
	return False;
    }
    return True;
}


static Boolean
PCacheFileBuild(PCacheFile pcf, Screen *screen, PCacheAttributes values)
{
    int status;
    char *filename;
    
    filename = PCacheLocatePixmapFile(DisplayOfScreen(screen),
				      pcf->name);
    if(! filename) return False;
    
#ifdef DebugPCache
    fprintf(stderr, "PCacheFileBuild: loading [%s]\n", filename);
#endif
    pcf->xpmimage = XtNew(XpmImage);
    pcf->xpminfos = XtNew(XpmInfo);
    pcf->xpminfos->valuemask = XpmExtensions;
    status = XpmReadFileToXpmImage(filename,
				   pcf->xpmimage,
				   pcf->xpminfos);
    XtFree(filename);		/* free before possible return */
    if(status != XpmSuccess) {
	fprintf(stderr, "PCacheReadFile %s\n", XpmGetErrorString(status));
	return False;
    }

    return True;
}

PCachePixmap
PCacheImageGetPixmap(PCacheFile pcf, PCacheImage pci, PCacheAttributes values)
{
    PCachePixmap pcp;
    pcp = pci->pixmaps;

    if(values) {
	values->nextensions = pcf->xpminfos->nextensions;
	values->extensions = pcf->xpminfos->extensions;
    }

    /* if a dimension is not required, use the default dimension */
    if( ! (values->valuemask & PCACHE_WIDTH))
	values->width = values->o_width;
    if( ! (values->valuemask & PCACHE_HEIGHT))
	values->height = values->o_height;

    while(pcp) {
	if((pcp->width == values->width) &&
	   (pcp->height == values->height)) {
	    /* pixmap is already in the cache */
	    return pcp;
	}
	else {
	    pcp = pcp->next;
	}
    }
    pcp = XtNew(PCachePixmapStruct);
    pcp->pixmap = None;
    pcp->mask = None;
    pcp->next = pci->pixmaps;
    pci->pixmaps = pcp;
    PCachePixmapBuild(pcf, pci, pcp, values);
    return pcp;
}


PCachePixmap
PCacheFileGetPixmap(PCacheFile pcf, Screen *screen, PCacheAttributes values)
{
    PCacheImage pci;
    pci = pcf->images;
    
    if(values) {
	values->o_width = pcf->xpmimage->width;
	values->o_height = pcf->xpmimage->height;
    }
    while(pci) {
	if((screen == pci->screen)) {
	    if((values->valuemask & PCACHE_SYM_COLORS) &&
	       (pci->nsymcolors == values->nsymcolors) &&
	       (pci->symcolors == values->symcolors)) {
	    /* XImages are already built */
		return PCacheImageGetPixmap(pcf, pci, values);
	    }
	    else if(! (values->valuemask & PCACHE_SYM_COLORS))
		return PCacheImageGetPixmap(pcf, pci, values);
		
	}
	pci = pci->next;
    }
    /* the file is loaded, but there is no XImage */
    pci = XtNew(PCacheImageStruct);
    pci->image_pixmap = None;
    pci->image_mask = None;
    pci->pixmaps = NULL;
    pci->screen = screen;
    pci->symcolors = values->symcolors;
    pci->nsymcolors = values->nsymcolors;
    pci->next = pcf->images;
    pcf->images = pci;
    PCacheImageBuild(pcf, pci, values);
    return PCacheImageGetPixmap(pcf, pci, values);
}




/* will return the pixmap built from name, for the specified screen, and with
 * the specified size.
 * returns: 
 */
PCachePixmap
PCacheGetPixmap(String name, Screen *screen, PCacheAttributes call_values)
{
    PCacheFile pcf;
    PCacheAttributesStruct vals;
    PCacheAttributes values;

    
    values = call_values;
    if(NULL == call_values) {
	values = &vals;
	values->valuemask = 0;
    }
    pcf = PC;

    while(pcf) {
	if(! (strcmp(pcf->name, name))) {
	    /* name is found: the file is already loaded */
	    return PCacheFileGetPixmap(pcf, screen, values);
	}
	pcf = pcf->next;
    }
    /* the file has not been loaded, yet */
    /* create new entry */
    pcf = XtNew(PCacheFileStruct);
    pcf->name = XtNewString(name);
    pcf->xpmimage = NULL;
    pcf->images = NULL;
    pcf->next = PC;
    PC = pcf;
    /* load the file and search through cache */
    if(PCacheFileBuild(pcf, screen, values)) {
	return PCacheFileGetPixmap(pcf, screen, values);
    }
    else {
	Display *dpy = DisplayOfScreen(screen);
	static PCachePixmap defaulticon = NULL;
	/* remove the inserted record */
	PC = pcf->next;
	XtFree((char *)pcf);
	/* return a default pixmap */
	if(NULL == defaulticon) {
	    defaulticon = XtNew(PCachePixmapStruct);
	    defaulticon->pixmap = None;
	    defaulticon->mask = None;
	    defaulticon->next = NULL;
	    XpmCreatePixmapFromData(dpy,
				    DefaultRootWindow(dpy),
				    koala3,
				    &(defaulticon->pixmap),
				    &(defaulticon->mask),
				    NULL);
	}
	values->width = 35;
	values->height = 29;
	return defaulticon;
    }
}


Boolean
PCachePreload(String name, Screen *screen, PCacheAttributes call_values)
{
    PCacheFile pcf;
    Boolean found;
    PCacheAttributesStruct vals;
    PCacheAttributes values;

    
    values = call_values;
    if(NULL == call_values) {
	values = &vals;
	values->valuemask = 0;
    }
    /* NOW, values is not NULL */
    values->nextensions = 0;
    values->extensions = 0;

    pcf = PC;
    found = False;
    while(pcf && !found) {
	if (!(strcmp(pcf->name, name)))
	    found = True;
	else
	    pcf=pcf->next;
    }

    if(! found) {
	/* the file has not been loaded, yet */
	pcf = XtNew(PCacheFileStruct);
	pcf->name = XtNewString(name);
	pcf->xpmimage = NULL;
	pcf->images = NULL;
	pcf->next = PC;
	PC = pcf;
	found = PCacheFileBuild(pcf, screen, values);
    }
    if(found && values) {
	values->width = values->o_width = pcf->xpmimage->width;
	values->height = values->o_height = pcf->xpmimage->height;
	values->nextensions = pcf->xpminfos->nextensions;
	values->extensions = pcf->xpminfos->extensions;
    }
    if(! found) {
	/* destroy the previously added record */
	PC = pcf->next;
	XtFree((char *)pcf);
    }

    return found;
    
}

/*
 * $Id: pcache.c,v 1.8 95/09/25 14:55:30 leon Exp $
 * 
 * Copyright (c) 1991-1994  Lionel MALLET
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL Lionel MALLET	BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * 
 * Except as contained in this notice, the name of Lionel MALLET shall
 * not be used in advertising or otherwise to promote the sale, use or
 * other dealings in this Software without prior written authorization
 * from Lionel MALLET.
 *
 * Author:  Lionel Mallet, SIMULOG
 */

/*
 * $XConsortium: Pixmap.c,v 1.12 90/06/09 20:19:28 dmatic Exp $
 *
 * Copyright 1989 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of M.I.T. not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  M.I.T. makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * M.I.T. DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL M.I.T.
 * BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Author:  Davor Matic, MIT X Consortium
 */


typedef struct {
  KnIntPos *x, *y;
  KnIntDim *width, *height;
} Table;


XImage *
ScalePixmapImage(Screen *screen, XImage *src, double scale_x, double scale_y)
{
    XImage *dst;
    Table table;    
    KnIntPos x, y;
    KnIntDim width, height, w, h;
    Pixel pixel;
  
    width = Max(Roundint(scale_x * src->width), 1);
    height = Max(Roundint(scale_y * src->height), 1);
  
    dst = XCreateImage(DisplayOfScreen(screen),
		       DefaultVisualOfScreen(screen),
		       src->depth, src->format, 
		       0, NULL,
		       width, height,
		       src->bitmap_pad, 0);
    dst->data = XtCalloc(1, dst->bytes_per_line * height);
    /* no need to clear the image, because we clip it */
  
    table.x = (KnIntPos *) XtMalloc(sizeof(KnIntPos) * src->width);
    table.y = (KnIntPos *) XtMalloc(sizeof(KnIntPos) * src->height);
    table.width = (KnIntDim *) XtMalloc(sizeof(KnIntDim) * src->width);
    table.height = (KnIntDim *) XtMalloc(sizeof(KnIntDim) * src->height);
    
    for (x = 0; x < src->width; x++) {
	table.x[x] = Roundint(scale_x * x);
	table.width[x] = Roundint(scale_x * (x + 1)) - Roundint(scale_x * x);
    }
    for (y = 0; y < src->height; y++) {
	table.y[y] = Roundint(scale_y * y);
	table.height[y] = Roundint(scale_y * (y + 1)) - Roundint(scale_y * y);
    }
    
    for (x = 0; x < src->width; x++)
	for (y = 0; y < src->height; y++) {
	    pixel = XGetPixel(src, x, y);
	    for (w = 0; w < table.width[x]; w++)
		for (h = 0; h < table.height[y]; h++)
		    XPutPixel(dst, 
			      table.x[x] + w, 
			      table.y[y] + h,
			      pixel);
	}
    
    XtFree((char *)table.x);
    XtFree((char *)table.y);
    XtFree((char *)table.width);
    XtFree((char *)table.height);

    return (dst);
}

XImage *
ScaleMaskImage(Screen *screen, XImage *src, double scale_x, double scale_y)
{
    XImage *dst;
    Table table;     
    KnIntPos x, y;
    KnIntDim width, height, w, h;
    Pixel pixel;
  
    width = Max(Roundint(scale_x * src->width), 1);
    height = Max(Roundint(scale_y * src->height), 1);
  
    dst = XCreateImage(DisplayOfScreen(screen),
		       DefaultVisualOfScreen(screen),
		       src->depth, src->format, 
		       0, NULL,
		       width, height,
		       src->bitmap_pad, 0);
  
    dst->data = XtCalloc(1,
			 dst->bytes_per_line *
			 dst->height);
    XAddPixel(dst, 1);

    table.x = (KnIntPos *) XtMalloc(sizeof(KnIntPos) * src->width);
    table.y = (KnIntPos *) XtMalloc(sizeof(KnIntPos) * src->height);
    table.width = (KnIntDim *) XtMalloc(sizeof(KnIntDim) * src->width);
    table.height = (KnIntDim *) XtMalloc(sizeof(KnIntDim) * src->height);
    
    for (x = 0; x < src->width; x++) {
	table.x[x] = Roundint(scale_x * x);
	table.width[x] = Roundint(scale_x * (x + 1)) - Roundint(scale_x * x);
    }
    for (y = 0; y < src->height; y++) {
	table.y[y] = Roundint(scale_y * y);
	table.height[y] = Roundint(scale_y * (y + 1)) - Roundint(scale_y * y);
    }
    
    for (x = 0; x < src->width; x++)
	for (y = 0; y < src->height; y++) {
	    pixel = XGetPixel(src, x, y);
	    for (w = 0; w < table.width[x]; w++)
		for (h = 0; h < table.height[y]; h++)
		    if(pixel == 0) 
			XPutPixel(dst, table.x[x] + w, table.y[y] + h,
				  pixel);
	}
    
    XtFree((char *)table.x);
    XtFree((char *)table.y);
    XtFree((char *)table.width);
    XtFree((char *)table.height);
  
    return (dst);
}
