/*+****************************************************************************/
/*+                                                                           */
/*+ Copyright (c) 1995-1997 Bruce M. Corwin                                   */
/*+                                                                           */
/*+****************************************************************************/
/*+****************************************************************************/
/*+                                                                           */
/*+ Module Name: vtclib.c                                                     */
/*+                                                                           */
/*+ Program ID:  vtclib                                                       */
/*+                                                                           */
/*+ Functions:  normal_mode, manage_mode, clear,                              */
/*+             set_value, child_set_value,                                   */
/*+             create_widget, display_widget,                                */
/*+             destroy_all_widgets, register_button,                         */
/*+             noecho, echo, vtinput, vtset, find_size,                      */
/*+             terminate, wsize, resize_widget,                              */
/*+             append_text, append_list,                                     */
/*+             append_text_raw, dump_text,                                   */
/*+             append_list_raw, append_file,                                 */
/*+             append_list_file, question_dialog,                            */
/*+             warning_dialog, information_dialog,                           */
/*+             error_dialog, message_dialog,                                 */
/*+             working_dialog, dialog_reset,                                 */
/*+             file_dialog                                                   */
/*+                                                                           */
/*+****************************************************************************/

#include <stdio.h>
#include <signal.h>
#include <termio.h>
#include "vtclib.h"

struct termios save_state;
int rs_flag = 0;
int term_flag = 0;
int commands_count = 0;
char *commands[2000];
char command_id[2000];


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: normal_mode                                                */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine sends the Vt widget into the normal interactive */
/*+              mode where it has a border, a cursor, selection and a        */
/*+              scrollbar. This is called when an application is finished.   */
/*+                                                                           */
/*+ Parameters: None                                                          */
/*+                                                                           */
/*+ Input:      None                                                          */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void normal_mode ()
{
    set_value (VTuseShadow, "True");
    set_value (VTscrolled, "True");
    set_value (VTcursorMode, "True");
    set_value (VTcutPaste, "True");
    echo (&save_state);
    dialog_reset ();
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: manage_mode                                                */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine sends the Vt widget into a mode where it acts   */
/*+              like a specialized manager widget. It no longer has a        */
/*+              border, a cursor, selection or a scrollbar. This is used for */
/*+              most applications. This is done by calling set_value 4 times */
/*+              and disabling the appropriat flags.                          */
/*+                                                                           */
/*+ Parameters: None                                                          */
/*+                                                                           */
/*+ Input:      None                                                          */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void manage_mode ()
{
    set_value (VTuseShadow, "False");
    set_value (VTscrolled, "False");
    set_value (VTcursorMode, "False");
    set_value (VTcutPaste, "False");
    noecho (&save_state);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: clear                                                      */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine clears and homes the cursor.                    */
/*+                                                                           */
/*+ Parameters: None                                                          */
/*+                                                                           */
/*+ Input:      None                                                          */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void clear ()
{
    printf ("\033[H\033[2J");
    fflush (stdout);
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: set_value                                                  */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine is used to set a resource for the active Vt     */
/*+              widget. This routine can radicaly change the function of the */
/*+              Vt widget.                                                   */
/*+                                                                           */
/*+ Parameters: resource      - char *     - Resource to set the value of.    */
/*+             value         - char *     - Value to set the resource to.    */
/*+                                                                           */
/*+ Input:      resource, value                                               */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void set_value (char *resource, char *value)
{
    printf ("\033}8:%s:%s\007", resource, value);
    fflush (stdout);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: child_set_value                                            */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine is used to set a resource for the specified     */
/*+              child widget. This routine can radicaly change the function  */
/*+              of the child widget.                                         */
/*+                                                                           */
/*+ Parameters: identity      - char       - Identity character that          */
/*+                                          identifies the created widget.   */
/*+             resource      - char *     - Resource to set the value of.    */
/*+             value         - char *     - Value to set the resource to.    */
/*+                                                                           */
/*+ Input:      identity, resource, value                                     */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void child_set_value (char identity, char *resource, char *value)
{
    printf ("\033}9:%c:%s:%s\007", identity, resource, value);
    fflush (stdout);
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: create_widget                                              */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine creates a new child widget for the Vt widget.   */
/*+              It needs an identity character to be used to 'print' the new */
/*+              widget on the Vt widget. This identity is then used by the   */
/*+              display_widget routine.                                      */
/*+                                                                           */
/*+ Parameters: wtype         - int        - Type of widget to create.        */
/*+             identity      - char       - Identity character that          */
/*+                                          identifies the created widget.   */
/*+             name          - char *     - Name of the widget.              */
/*+             width         - int        - Number of columns the widget     */
/*+                                          will span.                       */
/*+             height        - int        - Number of rows the widget will   */
/*+                                          cover.                           */
/*+                                                                           */
/*+ Input:      wtype, identity, name, width, height                          */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void create_widget (int wtype, char identity, char *name, int width, int height)
{
    printf ("\033}5:%c:%s:%d:%d:%d\007", identity, name, wtype, width, height);
    fflush (stdout);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: display_widget                                             */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine is used to specify a row and column for a       */
/*+              specified created widget that is then printed to the screen. */
/*+                                                                           */
/*+ Parameters: identity      - char       - Identity character that          */
/*+                                          identifies the created widget.   */
/*+             row           - int        - Row of widget.                   */
/*+             column        - int        - Column of widget.                */
/*+                                                                           */
/*+ Input:      identity, row, column                                         */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void display_widget (char identity, int row, int column)
{
    printf ("\033[%d;%dH\033[55m%c\033[62m", row, column, identity);
    fflush (stdout);
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: destroy_all_widgets                                        */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine as its name implies destroys all child widgets  */
/*+              and clears them from the screen.                             */
/*+                                                                           */
/*+ Parameters: None                                                          */
/*+                                                                           */
/*+ Input:      None                                                          */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void destroy_all_widgets ()
{
    printf ("\033}7\007");
    fflush (stdout);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: register_button                                            */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine is used to define what text will be used to     */
/*+              output data back from a button when it is pressed.           */
/*+                                                                           */
/*+ Parameters: identity      - char       - Identity character that          */
/*+                                          identifies the created widget.   */
/*+                                                                           */
/*+ Input:      identity                                                      */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void register_button (char identity)
{
    char *temp = NULL;
    char text[BUFSIZ];
    sprintf (text, "BUTTON_%c", identity);
    commands[commands_count] = (char *) malloc (strlen (text) + 1);
    strcpy (commands[commands_count], text);
    command_id[commands_count] = identity;
    commands_count++;
    printf ("\033}10:%c:%s\007", identity, text);
    fflush (stdout);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: noecho                                                     */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: Turns off the echo stty state of standard input.             */
/*+                                                                           */
/*+ Parameters: tio           - struct termios *                              */
/*+                                        - Terminal io data structure.      */
/*+                                                                           */
/*+ Input:      tio                                                           */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 08/11/93    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void noecho (tio)
     struct termios *tio;
{
    struct termios tin;
    tcgetattr (0, tio);
    tin = *tio;
    tin.c_lflag &= ~ECHO;
    tin.c_cc[VMIN] = 1;
    tin.c_cc[VTIME] = 0;
    tcsetattr (0, TCSANOW, &tin);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: echo                                                       */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: Turns on the echo stty state of standard input.              */
/*+                                                                           */
/*+ Parameters: tio           - struct termios *                              */
/*+                                        - Terminal io data structure.      */
/*+                                                                           */
/*+ Input:      tio                                                           */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 08/11/93    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void echo (tio)
     struct termios *tio;
{
    tcsetattr (0, TCSANOW, tio);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: vtinput                                                    */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine is used to obtain input from the various child  */
/*+              widgets on the current string.                               */
/*+                                                                           */
/*+ Parameters: buffer        - char *     - Buffer of characters from the    */
/*+                                          input stream.                    */
/*+                                                                           */
/*+ Input:      buffer                                                        */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
int vtinput (char *buffer)
{
    int rcode = -1;
    int inx = 0;
    do
    {
	rs_flag = 0;
	term_flag = 0;
	while (gets (buffer) != NULL)
	{
	    rcode = -1;
	    for (inx = 0; inx < commands_count; inx++)
	    {
		if (strcmp (commands[inx], buffer) == 0)
		{
		    rcode = command_id[inx];
		    break;
		}
	    }
	    /* ************************************ */
	    /* If we have found a command we're     */
	    /* done.  If not look for dialog codes. */
	    /* ************************************ */
	    if (rcode != -1)
		break;

	    if (strcmp (buffer, "CANCEL") == 0)
	    {
		rcode = CANCEL;
		break;
	    }
	    if (strcmp (buffer, "OK") == 0)
	    {
		rcode = OK;
		break;
	    }
	    if (strncmp (buffer, "FILEOK:", 7) == 0)
	    {
		rcode = FILEOK;
		break;
	    }
	    if (strcmp (buffer, "HELP") == 0)
	    {
		rcode = HELP;
		break;
	    }
	}

	if (rs_flag)
	{
	    strcpy (buffer, "RESIZE");
	    rcode = RESIZE;
	    break;
	}
	if (term_flag)
	{
	    strcpy (buffer, "QUIT");
	    rcode = QUIT;
	    break;
	}
    }
    while (rs_flag || term_flag);
    rs_flag = 0;
    return (rcode);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: vtset                                                      */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine sets up the signal structure of the program to  */
/*+              utilize vtinput properly.                                    */
/*+                                                                           */
/*+ Parameters: None                                                          */
/*+                                                                           */
/*+ Input:      None                                                          */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void vtset ()
{
    struct sigaction sas;
    sigset_t myset;
    struct sigaction sas2;
    sigset_t myset2;
    sigfillset (&myset);
    sigfillset (&myset2);
    sas.sa_handler = find_size;
    sas.sa_mask = myset;
    sas.sa_flags = 0;
#ifdef _SIGWINCH
    sigaction (_SIGWINCH, &sas, NULL);
#else
    sigaction (SIGWINCH, &sas, NULL);
#endif
    sas2.sa_handler = terminate;
    sas2.sa_mask = myset2;
    sas2.sa_flags = 0;
    sigaction (SIGINT, &sas2, NULL);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: find_size                                                  */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine is called when a resize signal is caught. It    */
/*+              sets a global flag that tells the vtinput routine to output  */
/*+              a RESIZE string to the input parser.                         */
/*+                                                                           */
/*+ Parameters: None                                                          */
/*+                                                                           */
/*+ Input:      None                                                          */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void find_size ()
{
    rs_flag = -1;
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: terminate                                                  */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine is called when a control C is pressed while in  */
/*+              vtinput. It sets a global flag that tells the vtinput        */
/*+              routine to output a QUIT string to the input parser.         */
/*+                                                                           */
/*+ Parameters: None                                                          */
/*+                                                                           */
/*+ Input:      None                                                          */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void terminate ()
{
    term_flag = -1;
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: wsize                                                      */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine queries the current standard input file         */
/*+              descriptor to find the number of rows and columns of the     */
/*+              screen.                                                      */
/*+                                                                           */
/*+ Parameters: rows          - int *      - Number of rows for the screen.   */
/*+             columns       - int *      - Number of columns for the        */
/*+                                          screen.                          */
/*+                                                                           */
/*+ Input:      None                                                          */
/*+                                                                           */
/*+ Output:     rows, columns                                                 */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
int wsize (int *rows, int *columns)
{
    struct winsize arg;
    ioctl (0, TIOCGWINSZ, &arg);
    *rows = arg.ws_row;
    *columns = arg.ws_col;
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: resize_widget                                              */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine is called whenever a widget needs to be resized */
/*+              using normal row column type coordinates.                    */
/*+                                                                           */
/*+ Parameters: identity      - char       - Identity character that          */
/*+                                          identifies the created widget.   */
/*+             width         - int        - Number of columns the widget     */
/*+                                          will span.                       */
/*+             height        - int        - Number of rows the widget will   */
/*+                                          cover.                           */
/*+                                                                           */
/*+ Input:      identity, width, height                                       */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void resize_widget (char identity, int width, int height)
{
    printf ("\033}12:%c:%d:%d\007", identity, width, height);
    fflush (stdout);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: append_text                                                */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine can be used to append text to the data in a     */
/*+              text widget. This is useful when data needs to be pre-loaded */
/*+              into a text widget.                                          */
/*+                                                                           */
/*+ Parameters: identity      - char       - Identity character that          */
/*+                                          identifies the created widget.   */
/*+             ttext         - char *     - Data to be sent to the text      */
/*+                                          widget.                          */
/*+                                                                           */
/*+ Input:      identity, ttext                                               */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void append_text (char identity, char *ttext)
{
    int inx = 0;
    int jnx = 0;
    char obuffer[BUFSIZ];
    char escseq[BUFSIZ];
    for (inx = 0; inx < strlen (ttext); inx++)
    {
	if
	    (
		ttext[inx] == 13 ||
		ttext[inx] == 12 ||
		ttext[inx] == 7 ||
		ttext[inx] == 9 ||
		ttext[inx] == 10 ||
		ttext[inx] == '\\' ||
		ttext[inx] == 27
	    )
	{
	    obuffer[jnx] = '\\';
	    jnx++;
	    switch (ttext[inx])
	    {
		case 12:
		    obuffer[jnx] = 'l';
		    break;
		case 7:
		    obuffer[jnx] = 'g';
		    break;
		case 9:
		    obuffer[jnx] = 't';
		    break;
		case 13:
		    obuffer[jnx] = 'c';
		    break;
		case 10:
		    obuffer[jnx] = 'n';
		    break;
		case '\\':
		    obuffer[jnx] = '\\';
		    break;
		case 27:
		    obuffer[jnx] = 'e';
		    break;
	    }
	    jnx++;
	}
	else
	{
	    obuffer[jnx] = ttext[inx];
	    jnx++;
	}
    }
    obuffer[jnx] = 0;
    append_text_raw (identity, obuffer);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: append_list                                                */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine can be used to append text to the data in a     */
/*+              list widget. This is useful when data needs to be pre-loaded */
/*+              into a list widget.                                          */
/*+                                                                           */
/*+ Parameters: identity      - char       - Identity character that          */
/*+                                          identifies the created widget.   */
/*+             ttext         - char *     - Data to be sent to the text      */
/*+                                          widget.                          */
/*+                                                                           */
/*+ Input:      identity, ttext                                               */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void append_list (char identity, char *ttext)
{
    int inx = 0;
    int jnx = 0;
    char obuffer[BUFSIZ];
    char escseq[BUFSIZ];
    for (inx = 0; inx < strlen (ttext); inx++)
    {
	if
	    (
		ttext[inx] == 13 ||
		ttext[inx] == 12 ||
		ttext[inx] == 7 ||
		ttext[inx] == 9 ||
		ttext[inx] == 10 ||
		ttext[inx] == '\\' ||
		ttext[inx] == 27
	    )
	{
	    obuffer[jnx] = '\\';
	    jnx++;
	    switch (ttext[inx])
	    {
		case 12:
		    obuffer[jnx] = 'l';
		    break;
		case 7:
		    obuffer[jnx] = 'g';
		    break;
		case 9:
		    obuffer[jnx] = 't';
		    break;
		case 13:
		    obuffer[jnx] = 'c';
		    break;
		case 10:
		    obuffer[jnx] = 'n';
		    break;
		case '\\':
		    obuffer[jnx] = '\\';
		    break;
		case 27:
		    obuffer[jnx] = 'e';
		    break;
	    }
	    jnx++;
	}
	else
	{
	    obuffer[jnx] = ttext[inx];
	    jnx++;
	}
    }
    obuffer[jnx] = 0;
    append_list_raw (identity, obuffer);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: append_text_raw                                            */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine can be used to append text to the data in a     */
/*+              text widget. This is useful when data needs to be pre-loaded */
/*+              into a text widget. This routine as opposed to the           */
/*+              append_text routine does not try to escape control           */
/*+              characters.                                                  */
/*+                                                                           */
/*+ Parameters: identity      - char       - Identity character that          */
/*+                                          identifies the created widget.   */
/*+             ttext         - char *     - Data to be sent to the text      */
/*+                                          widget.                          */
/*+                                                                           */
/*+ Input:      identity, ttext                                               */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void append_text_raw (char identity, char *ttext)
{
    printf ("\033}4:%c:%s\007", identity, ttext);
    fflush (stdout);
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: dump_text                                                  */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine returns a dynamicaly allocated string that has  */
/*+              the text information from a text widget. This can be used to */
/*+              retrieve information from a text widget. Again this data     */
/*+              must be freed when its use is complete.                      */
/*+                                                                           */
/*+ Parameters: identity      - char       - Identity character that          */
/*+                                          identifies the created widget.   */
/*+             ttext         - char **    - Data to be sent to the text      */
/*+                                          widget.                          */
/*+                                                                           */
/*+ Input:      identity, ttext                                               */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void dump_text (char identity, char **ttext)
{
    int datasize = 0;
    int chardat = 0;
    char pre[BUFSIZ];
    char sizestr[BUFSIZ];
    char post[BUFSIZ];
    char buffer[BUFSIZ];
    int notdone = -1;
    int inx = 0;

    printf ("\033}6:%c:TOP:BOT\007", identity);
    gets (sizestr);
    sscanf (sizestr, "%d", &datasize);
    *ttext = (char *) malloc (datasize + 1);
    for (inx = 0; inx < datasize; inx++)
    {
	(*ttext)[inx] = getchar ();
    }
    (*ttext)[inx] = 0;
    getchar ();
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: append_list_raw                                            */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine can be used to append text to the data in a     */
/*+              list widget. This is useful when data needs to be pre-loaded */
/*+              into a list widget. This routine as opposed to the           */
/*+              append_list routine does not try to escape control           */
/*+              characters.                                                  */
/*+                                                                           */
/*+ Parameters: identity      - char       - Identity character that          */
/*+                                          identifies the created widget.   */
/*+             ttext         - char *     - Data to be sent to the text      */
/*+                                          widget.                          */
/*+                                                                           */
/*+ Input:      identity, ttext                                               */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void append_list_raw (char identity, char *ttext)
{
    printf ("\033}13:%c:%s\007", identity, ttext);
    fflush (stdout);
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: append_file                                                */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine loads an entire TEXT file into a text widget.   */
/*+              This will not deal very well with some binary files so this  */
/*+              kind of file should not be used with this routine.           */
/*+                                                                           */
/*+ Parameters: identity      - char       - Identity character that          */
/*+                                          identifies the created widget.   */
/*+             filename      - char *     - File name to load.               */
/*+                                                                           */
/*+ Input:      identity, filename                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
int append_file (char identity, char *filename)
{
    int rcode = 0;
    FILE *input = NULL;
    char buffer[BUFSIZ];

    input = fopen (filename, READMODE);
    if (input != NULL)
    {
	while (fgets (buffer, BUFSIZ, input) != NULL)
	{
	    if (buffer[strlen (buffer) - 1] == 10)
		buffer[strlen (buffer) - 1] = 0;
	    append_text (identity, buffer);
	}
	fclose (input);
    }
    else
    {
	rcode = -1;
    }
    return (rcode);
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: append_list_file                                           */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine load an entire TEXT file into a list widget.    */
/*+              This will not deal very well with binary files most likely.  */
/*+              Obviously this is for listing only. No editing can take      */
/*+              place in a text widget.                                      */
/*+                                                                           */
/*+ Parameters: identity      - char       - Identity character that          */
/*+                                          identifies the created widget.   */
/*+             filename      - char *     - File name to load.               */
/*+                                                                           */
/*+ Input:      identity, filename                                            */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 04/22/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
int append_list_file (char identity, char *filename)
{
    int rcode = 0;
    FILE *input = NULL;
    char buffer[BUFSIZ];

    input = fopen (filename, READMODE);
    if (input != NULL)
    {
	while (fgets (buffer, BUFSIZ, input) != NULL)
	{
	    if (buffer[strlen (buffer) - 1] == 10)
		buffer[strlen (buffer) - 1] = 0;
	    append_list (identity, buffer);
	}
	fclose (input);
    }
    else
    {
	rcode = -1;
    }
    return (rcode);
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: question_dialog                                            */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine displays a Question Dialog with the provided    */
/*+              title and text.                                              */
/*+                                                                           */
/*+ Parameters: title         - char *     - This is the title for the        */
/*+                                          created dialog.                  */
/*+             dtext         - char *     - This is the dialog text to       */
/*+                                          display in the dialog.           */
/*+                                                                           */
/*+ Input:      title, dtext                                                  */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 01/07/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void question_dialog (char *title, char *dtext)
{
    printf ("\033}18:%s:%s\007", title, dtext);
    fflush (stdout);
    return;
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: warning_dialog                                             */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine displays a Warning Dialog with the provided     */
/*+              title and text.                                              */
/*+                                                                           */
/*+ Parameters: title         - char *     - This is the title for the        */
/*+                                          created dialog.                  */
/*+             dtext         - char *     - This is the dialog text to       */
/*+                                          display in the dialog.           */
/*+                                                                           */
/*+ Input:      title, dtext                                                  */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 01/07/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void warning_dialog (char *title, char *dtext)
{
    printf ("\033}14:%s:%s\007", title, dtext);
    fflush (stdout);
    return;
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: information_dialog                                         */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine displays an Information Dialog with the         */
/*+              provided title and text.                                     */
/*+                                                                           */
/*+ Parameters: title         - char *     - This is the title for the        */
/*+                                          created dialog.                  */
/*+             dtext         - char *     - This is the dialog text to       */
/*+                                          display in the dialog.           */
/*+                                                                           */
/*+ Input:      title, dtext                                                  */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 01/07/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void information_dialog (char *title, char *dtext)
{
    printf ("\033}16:%s:%s\007", title, dtext);
    fflush (stdout);
    return;
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: error_dialog                                               */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine displays an Error Dialog with the provided      */
/*+              title and text.                                              */
/*+                                                                           */
/*+ Parameters: title         - char *     - This is the title for the        */
/*+                                          created dialog.                  */
/*+             dtext         - char *     - This is the dialog text to       */
/*+                                          display in the dialog.           */
/*+                                                                           */
/*+ Input:      title, dtext                                                  */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 01/07/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void error_dialog (char *title, char *dtext)
{
    printf ("\033}15:%s:%s\007", title, dtext);
    fflush (stdout);
    return;
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: message_dialog                                             */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine displays a Message Dialog with the provided     */
/*+              title and text.                                              */
/*+                                                                           */
/*+ Parameters: title         - char *     - This is the title for the        */
/*+                                          created dialog.                  */
/*+             dtext         - char *     - This is the dialog text to       */
/*+                                          display in the dialog.           */
/*+                                                                           */
/*+ Input:      title, dtext                                                  */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 01/07/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void message_dialog (char *title, char *dtext)
{
    printf ("\033}17:%s:%s\007", title, dtext);
    fflush (stdout);
    return;
}

/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: working_dialog                                             */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine displays a Working Dialog with the provided     */
/*+              title and text.                                              */
/*+                                                                           */
/*+ Parameters: title         - char *     - This is the title for the        */
/*+                                          created dialog.                  */
/*+             dtext         - char *     - This is the dialog text to       */
/*+                                          display in the dialog.           */
/*+                                                                           */
/*+ Input:      title, dtext                                                  */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 01/07/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void working_dialog (char *title, char *dtext)
{
    printf ("\033}19:%s:%s\007", title, dtext);
    fflush (stdout);
    return;
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: dialog_reset                                               */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine resets the current dialog setting of the Vt     */
/*+              widget.                                                      */
/*+                                                                           */
/*+ Parameters: None                                                          */
/*+                                                                           */
/*+ Input:      None                                                          */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 01/07/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void dialog_reset ()
{
    printf ("\033}20\007");
    fflush (stdout);
    return;
}


/*+****************************************************************************/
/*+                                                                           */
/*+ Function name: file_dialog                                                */
/*+                                                                           */
/*+ Program ID:    vtclib                                                     */
/*+                                                                           */
/*+ Description: This routine displays a File Dialog with the provided title. */
/*+                                                                           */
/*+ Parameters: title         - char *     - This is the title for the        */
/*+                                          created dialog.                  */
/*+                                                                           */
/*+ Input:      title                                                         */
/*+                                                                           */
/*+ Output:     None                                                          */
/*+                                                                           */
/*+ Special Logic Notes: None                                                 */
/*+                                                                           */
/*+****************************************************************************/
/*+                                                                           */
/*+                           MODIFICATION LOG                                */
/*+                                                                           */
/*+   DATE        AUTHOR         DESCRIPTION                                  */
/*+ --------    --------------   -------------------------------------------- */
/*+ 01/07/97    Bruce M. Corwin  Initial Release                              */
/*+                                                                           */
/*+****************************************************************************/
void file_dialog (char *title)
{
    printf ("\033}21:%s\007", title);
    fflush (stdout);
    return;
}
