/*
 * Author:      William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990-1994, William Cheng.
 *
 * Permission limited to the use, copy, modify, and distribute this software
 * and its documentation for any purpose is hereby granted by the Author without
 * fee, provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the Author not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.  All other
 * rights (including the right to sell "tgif" and the right to sell derivative
 * works of tgif) are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /amnt/maui/tangram/u/william/X11/TGIF2/RCS/eps.c,v 2.16 1994/03/21 04:54:18 william Exp $";
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <stdio.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include "const.h"
#include "types.h"

#include "cmd.e"
#include "color.e"
#include "cursor.e"
#include "dialog.e"
#include "drawing.e"
#include "dup.e"
#ifndef _NO_EXTERN
#include "eps.e"
#endif
#include "file.e"
#include "grid.e"
#include "mark.e"
#include "msg.e"
#include "obj.e"
#include "pattern.e"
#include "select.e"
#include "setup.e"
#include "xbitmap.e"

#ifdef __hpux
extern double	atof ARGS_DECL((const char *));
#else
extern double	atof ARGS_DECL((char *));
#endif

typedef struct LineRec {
   char			* s;
   struct LineRec	* next, * prev;
} * LineRecPtr;

float	defaultEPSScaling=1.0;
char	defaultEPSScalingStr[80];

static struct LineRec	* topLine=NULL, * botLine=NULL;
static int		numLines = 0;

void InitEPS ()
{
   char		* c_ptr;

   stripEPSComments = TRUE;
   if ((c_ptr = XGetDefault(mainDisplay,TOOL_NAME,"StripEPSComments")) != NULL)
      if (strcmp ("False", c_ptr) == 0 || strcmp ("false", c_ptr) == 0)
         stripEPSComments = FALSE;

   if ((c_ptr=XGetDefault(mainDisplay,TOOL_NAME,"ForceClearAfterEPS")) != NULL)
      fprintf (stderr, "%s*ForceClearAfterEPS is obsolete.\n", TOOL_NAME);

   defaultEPSScaling = 1.0;
   strcpy (defaultEPSScalingStr, "1");
   if ((c_ptr=XGetDefault(mainDisplay,TOOL_NAME,"DefaultEPSScaling")) != NULL)
   {
      strcpy (defaultEPSScalingStr, c_ptr);
      defaultEPSScaling = (float) atof (c_ptr);
      if (defaultEPSScaling <= 0.0)
      {
         fprintf (stderr, "Invalid %s*DefaultEPSScaling: '%s', %s.\n",
               TOOL_NAME, c_ptr, "1.0 is used");
         defaultEPSScaling = 1.0;
         strcpy (defaultEPSScalingStr, "1");
      }
      else if (strcmp(defaultEPSScalingStr,"1")==0 ||
            strcmp(defaultEPSScalingStr,"1.0")==0)
      {
         defaultEPSScaling = 1.0;
         strcpy (defaultEPSScalingStr, "1");
      }
   }
}

static
void CleanUpLines ()
{
   register struct LineRec	* line_ptr, * next_line;

   for (line_ptr=topLine; line_ptr != NULL; line_ptr = next_line)
   {
      next_line = line_ptr->next;
      if (line_ptr->s != NULL) cfree (line_ptr->s);
      cfree (line_ptr);
   }
   topLine = botLine = NULL;
   numLines = 0;
}

void CleanUpEPS ()
{
   CleanUpLines ();
   stripEPSComments = TRUE;
}

static
int ReadPreviewBitmap (fp, image_w, image_h, bps, bitmap, image)
   FILE		* fp;
   int		image_w, image_h, bps;
   Pixmap	* bitmap;
   XImage	* * image;
{
   register int	k, j;
   int		i, num_nibbles, bit_count;
   char		* line, * c_ptr, msg[MAXSTRING];

   switch (bps)
   {
      case 1:
         num_nibbles = ((image_w & 0x3)==0) ? (int)(image_w>>2) :
               (int)(image_w>>2)+1;
         if (num_nibbles & 0x1) num_nibbles++;
         break;
      default:
         sprintf (msg, "%1d bits per sample preview not supported.", bps);
         Msg (msg);
         return (FALSE);
   }
   line = (char *) calloc (num_nibbles+10, sizeof(char));
   *bitmap = XCreatePixmap (mainDisplay, dummyBitmap, image_w, image_h, 1);
   XFillRectangle (mainDisplay, *bitmap, xbmGC, 0, 0, image_w, image_h);
   *image = XGetImage (mainDisplay,*bitmap,0,0,image_w,image_h,1,ZPixmap);

   for (i = 0; i < image_h; i++)
   {
      for (j = 0, c_ptr = line; j < num_nibbles; j++, c_ptr++)
      {
         while (TRUE)
         {
            if ((*c_ptr = (char) getc (fp)) == EOF)
            {
               Msg ("Invalid preview bitmap in EPS file.");
               cfree (line);
               XFreePixmap (mainDisplay, *bitmap); *bitmap = None;
               XDestroyImage (*image); *image = NULL;
               return (FALSE);
            }
            if (*c_ptr >= '0' && *c_ptr <= '9' ||
                  *c_ptr >= 'a' && *c_ptr <= 'f' ||
                  *c_ptr >= 'A' && *c_ptr <= 'F')
               break;
         }
      }
      *c_ptr = '\0';
      bit_count = 0;
      c_ptr = line;
      for (j=0; j<num_nibbles && *c_ptr!='\0'; j++, c_ptr++)
      {
         int	data;

         if (*c_ptr >= '0' && *c_ptr <= '9')
            data = (int)(*c_ptr) - (int)('0');
         else if (*c_ptr >= 'a' && *c_ptr <= 'f')
            data = (int)(*c_ptr) - (int)('a') + 10;
         else if (*c_ptr >= 'A' && *c_ptr <= 'F')
            data = (int)(*c_ptr) - (int)('A') + 10;
         else
            break;

         for (k = 0; k < 4; k++)
         {
            if (bit_count++ == image_w) break;

            if (data & (1<<(3-k)))
               XPutPixel (*image, j*4+k, i, 1);
         }
      }
   }
   if (fgets (line, MAXSTRING, fp) == NULL)
   {
      Msg ("Invalid preview bitmap in EPS file.");
      cfree (line);
      XFreePixmap (mainDisplay, *bitmap); *bitmap = None;
      XDestroyImage (*image); *image = NULL;
      return (FALSE);
   }
   XPutImage (mainDisplay,*bitmap,xbmGC,*image,0,0,0,0,image_w,image_h);
   cfree (line);
   return (TRUE);
}

static
void AddLine (s)
   char	* s;
{
   struct LineRec	* line_ptr;

   line_ptr = (struct LineRec *) calloc (1, sizeof(struct LineRec));
   line_ptr->s = (char *) calloc (strlen(s)+2, sizeof(char));
   strcpy (line_ptr->s, s);

   line_ptr->next = NULL;
   line_ptr->prev = botLine;
   if (botLine == NULL)
      topLine = line_ptr;
   else
      botLine->next = line_ptr;
   botLine = line_ptr;

   numLines++;
}

int MyReadEPSFile (file_name, image_w, image_h, bitmap, image, num_lines,
      epsflines, epsf_level, llx, lly, urx, ury, write_date)
   char		* file_name;
   int		* image_w, * image_h, * num_lines, * epsf_level;
   Pixmap	* bitmap;
   XImage	* * image;
   char		* * * epsflines, * write_date;
   float	* llx, * lly, * urx, * ury;
{
   register struct LineRec	* line_ptr, * next_line;
   register int	i;
   char		inbuf[MAXSTRING+1], * line=NULL, * c_ptr, * loc_time;
   int		allocated=FALSE, len, cur_size, done, first_line=TRUE;
   int		boundingbox_found=FALSE, preview_found=FALSE, found;
   int		boundingbox_atend=FALSE;
   FILE		* fp;
   struct stat	stat_buf;

   *image_w = *image_h = *num_lines = *epsf_level = 0;
   *bitmap = None;
   *image = NULL;
   *epsflines = NULL;

   if ((fp = fopen (file_name, "r")) == NULL) return (BitmapOpenFailed);

   CleanUpLines ();

   while (fgets (inbuf, MAXSTRING, fp) != NULL)
   {
      len = strlen (inbuf);
      if (inbuf[len-1] != '\r' && inbuf[len-1] != '\n')
      {  /* line longer than MAXSTRING characters */
         /* inbuf[MAXSTRING-1] == '\0' and len == MAXSTRING-1 now */
         cur_size = 2*MAXSTRING-1;
         line = (char *) calloc (cur_size+1, sizeof(char));
         strcpy (line, inbuf);
         c_ptr = &(line[MAXSTRING-1]);
         allocated = TRUE;

         done = FALSE;
         while (!done && fgets (inbuf, MAXSTRING, fp) != NULL)
         {
            len = strlen(inbuf);
            if (inbuf[len-1] == '\r' || inbuf[len-1] == '\n')
            {
               done = TRUE;
               inbuf[len-1] = '\0';
               strcpy (c_ptr, inbuf);
            }
            else
            {
               int	n = c_ptr - line;	/* NMH */

               cur_size += MAXSTRING-1;
               line = (char *) realloc (line, cur_size+1);
               c_ptr = line + n;		/* NMH */
               strcpy (c_ptr, inbuf);
               c_ptr += MAXSTRING-1;
            }
         }
         if (!done)
         {
            Msg ("Unexpected EOF in reading EPS file.");
            cfree (line);
            fclose (fp);
            return (BitmapFileInvalid);
         }
      }
      else
      {
         line = inbuf;
         line[len-1] = '\0';
      }

      if (first_line)
      {
         first_line = FALSE;
         if (line[0] != '%' || line[1] != '!')
         {
            if (allocated) cfree (line);
            fclose (fp);
            return (BitmapFileInvalid);
         }
         AddLine (line);
      }
      else if ((!boundingbox_found || boundingbox_atend) &&
            strncmp (line, "%%BoundingBox:", 14) == 0)
      {
         if (sscanf (&(line[14]), "%f %f %f %f", llx, lly, urx, ury) == 4)
            boundingbox_found = TRUE;
         else if (!boundingbox_found)
         {
            c_ptr = FindChar ('(', &(line[14]));
            if (strncmp (c_ptr, "atend)", 6) == 0)
               boundingbox_atend = TRUE;
         }
      }
      else if (!preview_found && strncmp (line, "%%BeginPreview:", 15) == 0)
      {
         int	bps;

         if (sscanf (&(line[15]), "%d %d %d", image_w, image_h, &bps) != 3)
         {
            Msg ("Invalid preview box in EPS file.");
            if (allocated) cfree (line);
            fclose (fp);
            return (BitmapFileInvalid);
         }
         else if (bps != 1)
         {
            char	msg[MAXSTRING];

            sprintf (msg, "%1d bits per sample preview not supported.", bps);
            Msg (msg);
            if (allocated) cfree (line);
            fclose (fp);
            return (BitmapFileInvalid);
         }
         preview_found = TRUE;
         if (!ReadPreviewBitmap (fp, *image_w, *image_h, bps, bitmap, image))
         {
            Msg ("Invalid preview box in EPS file.");
            if (allocated) cfree (line);
            fclose (fp);
            return (BitmapFileInvalid);
         }
         found = FALSE;
         while (fgets (inbuf, MAXSTRING, fp) != NULL)
         {
            if (strncmp (inbuf, "%%EndPreview", 12) == 0)
            {
               found = TRUE;
               break;
            }
         }
         if (!found)
         {
            Msg ("Invalid preview box in EPS file.");
            if (allocated) cfree (line);
            if (*bitmap != None)
            {
               XFreePixmap (mainDisplay, *bitmap);
               *bitmap = None;
            }
            if (*image != NULL)
            {
               XDestroyImage (*image);
               *image = NULL;
            }
            fclose (fp);
            return (BitmapFileInvalid);
         }
      }
      else
      {
         if (line[0] == '%' && line[1] == '!') (*epsf_level)++;
         if (!stripEPSComments || line[0] != '%')
            AddLine (line);
      }
   }
   fclose (fp);
   if (!boundingbox_found)
   {
      Msg ("Can not find bounding box in EPS file.");
      if (allocated) cfree (line);
      if (*bitmap != None)
      {
         XFreePixmap (mainDisplay, *bitmap);
         *bitmap = None;
      }
      if (*image != NULL)
      {
         XDestroyImage (*image);
         *image = NULL;
      }
      return (BitmapFileInvalid);
   }
   stat (file_name, &stat_buf);
   loc_time = ctime (&(stat_buf.st_mtime));
   loc_time[24] = '\0';
   strcpy (write_date, loc_time);

   *epsflines = (char * *) calloc (numLines, sizeof(char *));
   *num_lines = numLines;

   for (i=0, line_ptr=topLine; line_ptr != NULL; line_ptr = next_line, i++)
   {
      next_line = line_ptr->next;
      (*epsflines)[i] = line_ptr->s;
      cfree (line_ptr);
   }
   topLine = botLine = NULL;
   numLines = 0;
   return (BitmapSuccess);
}

static
int JustReadEPSLines (xbm_ptr)
   struct XBmRec	* xbm_ptr;
{
   register struct LineRec	* line_ptr, * next_line;
   register int	i;
   char		inbuf[MAXSTRING+1], * c_ptr, * loc_time;
   int		allocated=FALSE, len, cur_size, done, first_line=TRUE;
   int		boundingbox_found=FALSE, preview_found=FALSE, found;
   int		boundingbox_atend=FALSE;
   float	llx, lly, urx, ury;
   char		* file_name=xbm_ptr->filename, msg[MAXSTRING];
   FILE		* fp;
   struct stat	stat_buf;

   if ((fp = fopen (file_name, "r")) == NULL)
   {
      sprintf (msg, "Can not open EPS file '%s'.", file_name);
      if (PRTGIF)
      {
         fprintf (stderr, "%s\n", msg);
         fprintf (stderr, "   EPS object skipped for printing.\n");
      }
      else
      {
         TwoLineMsg (msg, "   EPS object skipped for printing.");
      }
      return (FALSE);
   }

   CleanUpLines ();

   while (fgets (inbuf, MAXSTRING, fp) != NULL)
   {
      char	* line;

      len = strlen (inbuf);
      if (inbuf[len-1] != '\r' && inbuf[len-1] != '\n')
      {  /* line longer than MAXSTRING characters */
         /* inbuf[MAXSTRING-1] == '\0' and len == MAXSTRING-1 now */
         cur_size = 2*MAXSTRING-1;
         line = (char *) calloc (cur_size+1, sizeof(char));
         strcpy (line, inbuf);
         c_ptr = &(line[MAXSTRING-1]);
         allocated = TRUE;

         done = FALSE;
         while (!done && fgets (inbuf, MAXSTRING, fp) != NULL)
         {
            len = strlen(inbuf);
            if (inbuf[len-1] == '\r' || inbuf[len-1] == '\n')
            {
               done = TRUE;
               inbuf[len-1] = '\0';
               strcpy (c_ptr, inbuf);
            }
            else
            {
               int	n = c_ptr - line;	/* NMH */

               cur_size += MAXSTRING-1;
               line = (char *) realloc (line, cur_size+1);
               c_ptr = line + n;		/* NMH */
               strcpy (c_ptr, inbuf);
               c_ptr += MAXSTRING-1;
            }
         }
         if (!done)
         {
            Msg ("Unexpected EOF in reading EPS file");
            cfree (line);
            fclose (fp);
            return (FALSE);
         }
      }
      else
      {
         line = inbuf;
         line[len-1] = '\0';
      }

      if (first_line)
      {
         first_line = FALSE;
         if (line[0] != '%' || line[1] != '!')
         {
            if (allocated) cfree (line);
            fclose (fp);
            return (FALSE);
         }
         AddLine (line);
      }
      else if ((!boundingbox_found || boundingbox_atend) &&
            strncmp (line, "%%BoundingBox:", 14) == 0)
      {
         if (sscanf (&(line[14]), "%f %f %f %f", &llx, &lly, &urx, &ury) == 4)
            boundingbox_found = TRUE;
         else if (!boundingbox_found)
         {
            c_ptr = FindChar ('(', &(line[14]));
            if (strncmp (c_ptr, "atend)", 6) == 0)
               boundingbox_atend = TRUE;
         }
      }
      else if (!preview_found && strncmp (line, "%%BeginPreview:", 15) == 0)
      {
         preview_found = TRUE;
         found = FALSE;
         while (fgets (inbuf, MAXSTRING, fp) != NULL)
         {
            if (strncmp (inbuf, "%%EndPreview", 12) == 0)
            {
               found = TRUE;
               break;
            }
         }
         if (!found)
         {
            if (PRTGIF)
               fprintf (stderr, "Invalid preview box in EPS file.\n");
            else
               Msg ("Invalid preview box in EPS file.");
            if (allocated) cfree (line);
            fclose (fp);
            return (FALSE);
         }
      }
      else
      {
         if (line[0] == '%' && line[1] == '!') (xbm_ptr->epsf_level)++;
         if (!stripEPSComments || line[0] != '%')
            AddLine (line);
      }
      if (allocated) cfree (line);
   }
   fclose (fp);
   if (!boundingbox_found)
   {
      if (PRTGIF)
         fprintf (stderr, "Can not find bounding box in EPS file.\n");
      else
         Msg ("Can not find bounding box in EPS file.");
      return (FALSE);
   }
   stat (file_name, &stat_buf);
   loc_time = ctime (&(stat_buf.st_mtime));
   loc_time[24] = '\0';
   if (strcmp (xbm_ptr->write_date, loc_time) != 0)
   {
      sprintf (msg, "Warning:  EPS file '%s' is newer than the EPS object.",
            file_name);
      if (PRTGIF)
         fprintf (stderr, "%s\n", msg);
      else
         Msg (msg);
   }
   xbm_ptr->epsflines = (char * *) calloc (numLines, sizeof(char *));
   xbm_ptr->num_epsf_lines = numLines;

   for (i=0, line_ptr=topLine; line_ptr != NULL; line_ptr = next_line, i++)
   {
      next_line = line_ptr->next;
      (xbm_ptr->epsflines)[i] = line_ptr->s;
      cfree (line_ptr);
   }
   topLine = botLine = NULL;
   numLines = 0;

   return (TRUE);
}

void DumpEPSObj (FP, ObjPtr)
   FILE			* FP;
   struct ObjRec	* ObjPtr;
{
   register int		i;
   float		w, h, ltx, lty, scale;
   float		bbox_w, bbox_h, llx, lly, urx, ury, x=0, y=0, dx, dy;
   struct XBmRec	* xbm_ptr=ObjPtr->detail.xbm;
   struct MtrxRec	mtrx;

   scale = ((float)psDotsPerInch)/((float)PIX_PER_INCH*100)*((float)printMag);

   w = (float)((float)(ObjPtr->obbox.rbx-ObjPtr->obbox.ltx) * scale);
   h = (float)((float)(ObjPtr->obbox.rby-ObjPtr->obbox.lty) * scale);
   ltx = (float)((float)(ObjPtr->obbox.ltx) * scale);
   lty = (float)(((float)psDotsPerInch)*psYOff[pageStyle] -
         ((float)(ObjPtr->obbox.rby) * scale));

   llx = (float)(((float)xbm_ptr->llx) / 1000.0);
   lly = (float)(((float)xbm_ptr->lly) / 1000.0);
   urx = (float)(((float)xbm_ptr->urx) / 1000.0);
   ury = (float)(((float)xbm_ptr->ury) / 1000.0);

   bbox_w = urx-llx;
   bbox_h = ury-lly;

   mtrx.image_w = (float)bbox_w; mtrx.image_h = (float)bbox_h;
   mtrx.w = w; mtrx.h = h;
   mtrx.rotate = xbm_ptr->rotate; mtrx.flip = xbm_ptr->flip;

   CalcTransform (&mtrx);

   dx = llx*mtrx.m[0][0] + lly*mtrx.m[1][0];
   dy = llx*mtrx.m[0][1] + lly*mtrx.m[1][1];

   switch (xbm_ptr->rotate)
   {
      case ROTATE0:
         x = (mtrx.transformed_w >= 1.0e-6) ? (ltx)-dx : (ltx+w)-dx;
         y = (mtrx.transformed_h >= 1.0e-6) ? (lty)-dy : (lty+h)-dy;
         break;
      case ROTATE90:
         x = (mtrx.transformed_w >= 1.0e-6) ? (ltx+w)+dx : (ltx)+dx;
         y = (mtrx.transformed_h >= 1.0e-6) ? (lty+h)+dy : (lty)+dy;
         break;
      case ROTATE180:
         x = (mtrx.transformed_w >= 1.0e-6) ? (ltx)-dx : (ltx+w)-dx;
         y = (mtrx.transformed_h >= 1.0e-6) ? (lty)-dy : (lty+h)-dy;
         break;
      case ROTATE270:
         x = (mtrx.transformed_w >= 1.0e-6) ? (ltx+w)+dx : (ltx)+dx;
         y = (mtrx.transformed_h >= 1.0e-6) ? (lty+h)+dy : (lty)+dy;
         break;
   }

   if (!xbm_ptr->save_epsf)
   {
      if (xbm_ptr->bitmap == None || xbm_ptr->epsflines == NULL)
      {
         if (!JustReadEPSLines (xbm_ptr))
            return;
      }
      else
      {
         char		msg[MAXSTRING], * loc_time;
         struct stat	stat_buf;

         stat (xbm_ptr->filename, &stat_buf);
         loc_time = ctime (&(stat_buf.st_mtime));
         loc_time[24] = '\0';
         if (strcmp (xbm_ptr->write_date, loc_time) != 0)
         {
            sprintf (msg,
                  "Warning:  EPS file '%s' is newer than the EPS object.",
                  xbm_ptr->filename);
            if (PRTGIF)
               fprintf (stderr, "%s\n", msg);
            else
               Msg (msg);
         }
      }
   }

   fprintf (FP, "%% EPS\n");
   fprintf (FP, "end\n");
   fprintf (FP, "/tgiflevel%1d save def\n", xbm_ptr->epsf_level);
   fprintf (FP, "/tgifdictcount%1d countdictstack def\n", xbm_ptr->epsf_level);
   fprintf (FP, "/tgifopcount%1d count 1 sub def\n", xbm_ptr->epsf_level);
   fprintf (FP, "userdict begin\n");
   fprintf (FP, "/showpage {} def\n");
   fprintf (FP, "/letter {} def\n");
   fprintf (FP, "/legal {} def\n");
   fprintf (FP, "1 %1d %1d div %1d mul 100 div div dup neg scale\n",
         psDotsPerInch, PIX_PER_INCH, printMag);
   fprintf (FP, "%1d %s mul neg %1d %s mul neg translate\n", psDotsPerInch,
         psXOffStr[pageStyle], psDotsPerInch, psYOffStr[pageStyle]);
   fprintf (FP, "\n");
   fprintf (FP, "%.3f %.3f translate %.3f %.3f scale %1d rotate\n",
         x, y, mtrx.dump_h_scale, mtrx.dump_v_scale, (-mtrx.degree));
   fprintf (FP, "\n");
   fprintf (FP, "%%%%BeginDocument: %s\n", xbm_ptr->filename);

   for (i = 0; i < xbm_ptr->num_epsf_lines; i++)
      fprintf (FP, "%s\n", xbm_ptr->epsflines[i]);

   fprintf (FP, "%%%%EndDocument\n");
   fprintf (FP, "count tgifopcount%1d sub {pop} repeat\n", xbm_ptr->epsf_level);
   fprintf (FP, "countdictstack tgifdictcount%1d sub {end} repeat\n",
         xbm_ptr->epsf_level);
   fprintf (FP, "tgiflevel%1d restore\n", xbm_ptr->epsf_level);
   fprintf (FP, "tgifdict begin\n");
   fprintf (FP, "\n");
}

struct ObjRec * CreateEPSObj (FileName, ImageW, ImageH, bitmap, image,
      num_lines, lines, epsf_level, llx, lly, urx, ury, write_date)
   char		* FileName, * * lines, * write_date;
   int		ImageW, ImageH, num_lines, epsf_level;
   Pixmap	bitmap;
   XImage	* image;
   float	llx, lly, urx, ury;
{
   struct XBmRec	* xbm_ptr;
   struct ObjRec	* obj_ptr;
   int			len=strlen(FileName), w, h;
   char			* name=(char *) calloc (len+1, sizeof(char));

   strcpy (name, FileName);

   xbm_ptr = (struct XBmRec *) calloc (1, sizeof(struct XBmRec));

   xbm_ptr->image = image;
   xbm_ptr->image_w = ImageW;
   xbm_ptr->image_h = ImageH;
   xbm_ptr->bitmap = bitmap;
   xbm_ptr->data = NULL;

   if (bitmap == None)
   {
      xbm_ptr->eps_w = w = (urx >= llx) ? (int)(urx-llx) : (int)(llx-urx);
      xbm_ptr->eps_h = h = (ury >= lly) ? (int)(ury-lly) : (int)(lly-ury);
   }
   else
   {
      xbm_ptr->eps_w = w = ImageW;
      xbm_ptr->eps_h = h = ImageH;
   }

   xbm_ptr->fill = objFill;
   xbm_ptr->rotate = xbm_ptr->cached_rotate = ROTATE0;
   xbm_ptr->flip = xbm_ptr->cached_flip = 0;
   xbm_ptr->llx = (int)(llx*1000.0);
   xbm_ptr->lly = (int)(lly*1000.0);
   xbm_ptr->urx = (int)(urx*1000.0);
   xbm_ptr->ury = (int)(ury*1000.0);
   xbm_ptr->cached_zoom = 0;
   xbm_ptr->cached_bitmap = None;
   xbm_ptr->cached_w = xbm_ptr->cached_h = 0;

   xbm_ptr->real_type = XBM_EPS;
   xbm_ptr->filename = name;
   strcpy (xbm_ptr->write_date, write_date);
   xbm_ptr->epsflines = lines;
   xbm_ptr->num_epsf_lines = num_lines;
   xbm_ptr->epsf_level = epsf_level;
   xbm_ptr->save_epsf = saveEPSLines;

   obj_ptr = (struct ObjRec *) calloc (1, sizeof(struct ObjRec));

   obj_ptr->bbox.ltx = obj_ptr->obbox.ltx = obj_ptr->x = drawOrigX;
   obj_ptr->bbox.lty = obj_ptr->obbox.lty = obj_ptr->y = drawOrigY;
   obj_ptr->bbox.rbx = obj_ptr->obbox.rbx = w + drawOrigX;
   obj_ptr->bbox.rby = obj_ptr->obbox.rby = h + drawOrigY;
   obj_ptr->type = OBJ_XBM;
   obj_ptr->color = colorIndex;
   obj_ptr->id = objId++;
   obj_ptr->dirty = FALSE;
   obj_ptr->rotation = 0;
   obj_ptr->locked = FALSE;
   obj_ptr->detail.xbm = xbm_ptr;
   obj_ptr->fattr = obj_ptr->lattr = NULL;

   return (obj_ptr);
}

void UpdateEPS ()
{
   struct XBmRec	* xbm_ptr, * new_xbm_ptr;
   struct ObjRec	* obj_ptr, * new_obj_ptr;
   char			write_date[32], * * lines=NULL, msg[MAXSTRING];
   int			rc, num_lines, epsf_level, image_w, image_h;
   float		llx, lly, urx, ury;
   Pixmap		bitmap;
   XImage		* image=NULL;

   if (topSel==NULL || topSel!=botSel || topSel->obj->type!=OBJ_XBM ||
         topSel->obj->detail.xbm->real_type!=XBM_EPS)
   {
      Msg ("Please select an EPS object.");
      return;
   }
   obj_ptr = topSel->obj;
   xbm_ptr = obj_ptr->detail.xbm;

   importingFile = TRUE;
   SetWatchCursor (drawWindow);
   SetWatchCursor (mainWindow);
   rc = MyReadEPSFile (xbm_ptr->filename, &image_w, &image_h, &bitmap, &image,
         &num_lines, &lines, &epsf_level, &llx, &lly, &urx, &ury, write_date);
   SetDefaultCursor (mainWindow);
   SetDefaultCursor (drawWindow);

   if (rc != BitmapSuccess)
   {
      sprintf (msg, "Can not import EPS file '%s'.", xbm_ptr->filename);
      Msg (msg);
      importingFile = FALSE;
      return;
   }
   HighLightReverse ();
   PrepareToReplaceAnObj (obj_ptr);
   new_obj_ptr = CreateEPSObj (xbm_ptr->filename, image_w, image_h, bitmap,
         image, num_lines, lines, epsf_level, llx, lly, urx, ury,
         write_date);

   new_obj_ptr->x = obj_ptr->x;
   new_obj_ptr->y = obj_ptr->y;
   new_obj_ptr->bbox.ltx = obj_ptr->bbox.ltx;
   new_obj_ptr->bbox.lty = obj_ptr->bbox.lty;
   new_obj_ptr->bbox.rbx = obj_ptr->bbox.rbx;
   new_obj_ptr->bbox.rby = obj_ptr->bbox.rby;
   new_obj_ptr->obbox.ltx = obj_ptr->obbox.ltx;
   new_obj_ptr->obbox.lty = obj_ptr->obbox.lty;
   new_obj_ptr->obbox.rbx = obj_ptr->obbox.rbx;
   new_obj_ptr->obbox.rby = obj_ptr->obbox.rby;

   new_xbm_ptr = new_obj_ptr->detail.xbm;
   new_xbm_ptr->rotate = xbm_ptr->rotate;
   new_xbm_ptr->flip = xbm_ptr->flip;

   UnlinkObj (obj_ptr);
   topSel->obj = new_obj_ptr;
   AddObj (NULL, topObj, new_obj_ptr);
   RecordReplaceAnObj (new_obj_ptr);
   FreeObj (obj_ptr);

   UpdSelBBox ();
   RedrawAnArea (botObj, selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
         selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1));
   SetFileModified (TRUE);
   justDupped = FALSE;
   HighLightForward ();
   Msg ("EPS object updated.");
}
