/*
 * Author:      William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990-1994, William Cheng.
 *
 * Permission limited to the use, copy, modify, and distribute this software
 * and its documentation for any purpose is hereby granted by the Author without
 * fee, provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the Author not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.  All other
 * rights (including the right to sell "tgif" and the right to sell derivative
 * works of tgif) are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /amnt/maui/tangram/u/william/X11/TGIF2/RCS/xpixmap.c,v 2.140 1994/04/19 05:04:51 william Exp $";
#endif

#include <stdio.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include "const.h"
#include "types.h"

#include "attr.e"
#include "choice.e"
#include "cmd.e"
#include "color.e"
#include "cursor.e"
#include "dialog.e"
#include "drawing.e"
#include "dup.e"
#include "file.e"
#include "font.e"
#include "grid.e"
#include "mark.e"
#include "msg.e"
#include "names.e"
#include "obj.e"
#include "pattern.e"
#include "raster.e"
#include "rect.e"
#include "select.e"
#include "setup.e"
#include "xbitmap.e"
#ifndef _NO_EXTERN
#include "xpixmap.e"
#endif

struct BucketRec {
   int	pixel, index;
   char	s[10];
};

#define XPM_BUCKETS 67
#define XPM_BUCKET_INC 10

GC	xpmGC = NULL;
int	newColormapUsed = FALSE;

double	rotatedSine[4] = { 0.0, 1.0, 0.0, -1.0 };
double	rotatedCosine[4] = { 1.0, 0.0, -1.0, 0.0 };

static Pixmap	dummyPixmap;

static char	hexValue[] = "0123456789abcdef";

static int	numColorsToDump = 0;
static int	* pixelValue = NULL;
static int	* colorIndexToDumpIndex = NULL;
static char	* colorChar = NULL;
static char	* * colorStr = NULL;

static int	askForXPmSpec = FALSE;
static int	guessXPmBgColor = FALSE;

static struct BucketRec	* * xpmBucket = NULL;
static int	* xpmBucketSize = NULL;
static int	* xpmBucketMaxSize = NULL;

void InitXPm ()
{
   register int	i;
   XGCValues	values;
   char		* c_ptr;

   dummyPixmap = XCreatePixmap (mainDisplay, mainWindow, 1, 1, mainDepth);

   values.foreground = myFgPixel;
   values.background = myBgPixel;
   values.function = GXcopy;
   values.fill_style = FillSolid;
   xpmGC = XCreateGC (mainDisplay, dummyPixmap,
         GCForeground | GCBackground | GCFunction | GCFillStyle, &values);

   if ((c_ptr = XGetDefault(mainDisplay,TOOL_NAME,"AskForXPmSpec")) != NULL)
   {
      if (strcmp ("True", c_ptr) == 0 || strcmp ("true", c_ptr) == 0)
         askForXPmSpec = TRUE;
      else
         askForXPmSpec = FALSE;
   }

   guessXPmBgColor = FALSE;
   if ((c_ptr = XGetDefault(mainDisplay,TOOL_NAME,"GuessXPmBgColor"))!=NULL)
      if (strcmp ("True", c_ptr) == 0 || strcmp ("true", c_ptr) == 0)
         guessXPmBgColor = TRUE;

   newColormapUsed = FALSE;

   xpmBucket = (struct BucketRec * *) calloc (XPM_BUCKETS,
         sizeof (struct BucketRec *));
   xpmBucketSize = (int *) calloc (XPM_BUCKETS+1, sizeof (int));
   xpmBucketMaxSize = (int *) calloc (XPM_BUCKETS, sizeof (int));
   for (i = 0; i < XPM_BUCKETS; i++)
   {
      xpmBucket[i] = (struct BucketRec *) calloc (XPM_BUCKET_INC,
            sizeof (struct BucketRec));
      xpmBucketSize[i] = 0;
      xpmBucketMaxSize[i] = XPM_BUCKET_INC;
   }
   xpmBucketSize[XPM_BUCKETS] = INVALID;
}

void CleanUpXPm ()
{
   register int	i;

   if (colorChar != NULL)
   {
      for (i = 0; i < numColorsToDump+2; i++)
         if (colorStr[i] != NULL)
            cfree (colorStr[i]);
      cfree (colorStr);
      cfree (colorChar);
      cfree (pixelValue);
      cfree (colorIndexToDumpIndex);
   }

   if (xpmGC != NULL) XFreeGC (mainDisplay, xpmGC);
   XFreePixmap (mainDisplay, dummyPixmap);

   askForXPmSpec = FALSE;
   for (i = 0; i < XPM_BUCKETS; i++) cfree (xpmBucket[i]);
   cfree (xpmBucket);
   cfree (xpmBucketSize);
   cfree (xpmBucketMaxSize);
   xpmBucket = NULL;
   xpmBucketSize = xpmBucketMaxSize = NULL;
}

#define xpmpixelhash(X) ((X)%(XPM_BUCKETS))

static
int xpmcharhash (chars_per_pixel, color_char)
   int	chars_per_pixel;
   char	* color_char;
{
   register int	i, val=0;

   for (i = 0; i < chars_per_pixel; i++) val = (val<<1)+(int)(color_char[i]);
   return (xpmpixelhash(val));
}

void BuildXPmBuckets (ncolors, pixels, chars_per_pixel, color_char)
   int	ncolors, * pixels, chars_per_pixel;
   char	* color_char;
{
   register int	* ptr, i;
   int		bucket;

   if (xpmBucketSize == NULL)
   {
      xpmBucket = (struct BucketRec * *) calloc (XPM_BUCKETS,
            sizeof (struct BucketRec *));
      xpmBucketSize = (int *) calloc (XPM_BUCKETS+1, sizeof (int));
      xpmBucketMaxSize = (int *) calloc (XPM_BUCKETS, sizeof (int));
      for (i = 0; i < XPM_BUCKETS; i++)
      {
         xpmBucket[i] = (struct BucketRec *) calloc (XPM_BUCKET_INC,
               sizeof (struct BucketRec));
         xpmBucketSize[i] = 0;
         xpmBucketMaxSize[i] = XPM_BUCKET_INC;
      }
      xpmBucketSize[XPM_BUCKETS] = INVALID;
   }

   for (ptr = xpmBucketSize; *ptr != INVALID; ptr++) *ptr = 0;

   if (chars_per_pixel == INVALID)
   {  /* build the hash table according to the pixels */
      for (i = 0; i < ncolors; i++)
      {
         bucket = xpmpixelhash(pixels[i]);
         if (xpmBucketSize[bucket] == xpmBucketMaxSize[bucket])
         {
            xpmBucket[bucket] = (struct BucketRec *) realloc (xpmBucket[bucket],
                  xpmBucketMaxSize[bucket]+XPM_BUCKET_INC);
            xpmBucketMaxSize[bucket] += XPM_BUCKET_INC;
         }
         xpmBucket[bucket][xpmBucketSize[bucket]].index = i;
         xpmBucket[bucket][xpmBucketSize[bucket]].pixel = pixels[i];
         (xpmBucketSize[bucket])++;
      }
   }
   else
   {  /* build the hash table according to the color_char */
      if (chars_per_pixel >= 9)
      {
         char	msg[MAXSTRING];

         sprintf (msg, "Chars_per_pixel = %1d.", chars_per_pixel);
         Error ("BuildXPmBuckets()", msg);
      }
      for (i = 0; i < ncolors; i++)
      {
         bucket = xpmcharhash(chars_per_pixel, &color_char[i*chars_per_pixel]);
         if (xpmBucketSize[bucket] == xpmBucketMaxSize[bucket])
         {
            xpmBucket[bucket] = (struct BucketRec *) realloc (xpmBucket[bucket],
                  xpmBucketMaxSize[bucket]+XPM_BUCKET_INC);
            xpmBucketMaxSize[bucket] += XPM_BUCKET_INC;
         }
         xpmBucket[bucket][xpmBucketSize[bucket]].index = i;
         strncpy (xpmBucket[bucket][xpmBucketSize[bucket]].s,
               &color_char[i*chars_per_pixel], chars_per_pixel);
         (xpmBucketSize[bucket])++;
      }
   }
}

int XPmLookUp (pixel, chars_per_pixel, color_char)
   int	pixel, chars_per_pixel;
   char	* color_char;
{
   register int			i;
   register struct BucketRec	* ptr;
   int				size, bucket;

   if (chars_per_pixel == INVALID)
   {  /* hash according to the pixels */
      bucket = xpmpixelhash(pixel);
   
      size = xpmBucketSize[bucket];
      for (i = 0, ptr = xpmBucket[bucket]; i < size; i++, ptr++)
         if (ptr->pixel == pixel)
            return (ptr->index);
   }
   else
   {  /* hash according to the color_char */
      bucket = xpmcharhash(chars_per_pixel, color_char);
   
      size = xpmBucketSize[bucket];
      for (i = 0, ptr = xpmBucket[bucket]; i < size; i++, ptr++)
         if (strncmp (color_char, ptr->s, chars_per_pixel) == 0)
            return (ptr->index);
   }
   return (INVALID);
}

void MakeCachedPixmap (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   register int		c, r;
   int			w, h, rotate, flip;
   int			num_cols, num_rows, image_w, image_h, watch_cursor;
   int			start_col, start_row;
   struct XPmRec	* xpm_ptr = ObjPtr->detail.xpm;
   struct MtrxRec	mtrx;
   Pixmap		dest_pixmap;
   XImage		* image, * dest_image;

   w = ObjPtr->obbox.rbx - ObjPtr->obbox.ltx;
   h = ObjPtr->obbox.rby - ObjPtr->obbox.lty;
   num_cols = (zoomedIn) ? (w<<zoomScale) : (w>>zoomScale);
   num_rows = (zoomedIn) ? (h<<zoomScale) : (h>>zoomScale);

   if (xpm_ptr->cached_pixmap!=None &&
         xpm_ptr->cached_zoomed==zoomedIn && xpm_ptr->cached_zoom==zoomScale &&
         xpm_ptr->cached_w==num_cols && xpm_ptr->cached_h==num_rows &&
         xpm_ptr->cached_rotate==xpm_ptr->rotate &&
         xpm_ptr->cached_flip==xpm_ptr->flip)
      return;

   if ((w>>zoomScale)==0 || (h>>zoomScale)==0)
   {
      if (xpm_ptr->cached_pixmap != None)
         XFreePixmap (mainDisplay, xpm_ptr->cached_pixmap);
      xpm_ptr->cached_pixmap = None;
      return;
   }

   watch_cursor = watchCursorOnMainWindow;
   if (!watch_cursor)
   {
      SetWatchCursor (drawWindow);
      SetWatchCursor (mainWindow);
   }

   image = xpm_ptr->image;
   rotate = xpm_ptr->rotate;
   flip = xpm_ptr->flip;
   image_w = xpm_ptr->image_w;
   image_h = xpm_ptr->image_h;
   if (xpm_ptr->cached_pixmap != None)
      XFreePixmap (mainDisplay, xpm_ptr->cached_pixmap);
   xpm_ptr->cached_pixmap = None;

   if (image == NULL)
      image = xpm_ptr->image = XGetImage (mainDisplay, xpm_ptr->pixmap, 0, 0,
            image_w, image_h, AllPlanes, ZPixmap);

   mtrx.image_w = (float)image_w; mtrx.image_h = (float)image_h;
   mtrx.w = (float)num_cols; mtrx.h = (float)num_rows;
   mtrx.rotate = rotate; mtrx.flip = flip;

   CalcTransform (&mtrx);

   start_col = (mtrx.transformed_w >= 0.0) ? 0 : (-num_cols)+1;
   start_row = (mtrx.transformed_h >= 0.0) ? 0 : (-num_rows)+1;

   dest_pixmap = XCreatePixmap (mainDisplay, dummyPixmap, num_cols, num_rows,
         mainDepth);
   XFillRectangle (mainDisplay,dest_pixmap,xpmGC,0,0,num_cols,num_rows);
   dest_image = XGetImage (mainDisplay, dest_pixmap, 0, 0, num_cols, num_rows,
         AllPlanes, ZPixmap);

   for (r = 0; r < num_rows; r++)
   {
      float	part_x, part_y;

      part_x = (r+start_row)*mtrx.rev_m[1][0];
      part_y = (r+start_row)*mtrx.rev_m[1][1];
      for (c = 0; c < num_cols; c++)
      {
         int	x, y;

         x = (int)((c+start_col)*mtrx.rev_m[0][0]+part_x);
         y = (int)((c+start_col)*mtrx.rev_m[0][1]+part_y);
         if (x>=0 && x<image_w && y>=0 && y<image_h)
            XPutPixel (dest_image, c, r, XGetPixel(image,x,y));
      }
   }
   XPutImage (mainDisplay, dest_pixmap, xpmGC, dest_image, 0, 0, 0, 0,
         num_cols, num_rows);
   xpm_ptr->cached_pixmap = dest_pixmap;
   xpm_ptr->cached_zoomed = zoomedIn;
   xpm_ptr->cached_zoom = zoomScale;
   xpm_ptr->cached_rotate = xpm_ptr->rotate;
   xpm_ptr->cached_flip = xpm_ptr->flip;
   xpm_ptr->cached_w = num_cols;
   xpm_ptr->cached_h = num_rows;

   XDestroyImage (dest_image);

   if (!watch_cursor)
   {
      SetDefaultCursor (mainWindow);
      ShowCursor ();
   }
}

static
int ExtractPixmap (orig_pixmap, orig_image, x, y, w, h, pixmap, image)
   Pixmap	orig_pixmap, * pixmap;
   XImage	* orig_image, * * image;
   int		x, y, w, h;
{
   register int	j, i;
   char		msg[MAXSTRING];
   XImage	* src_image;

   SetWatchCursor (drawWindow);
   SetWatchCursor (mainWindow);

   if ((*pixmap = XCreatePixmap (mainDisplay, dummyPixmap, w, h,
         mainDepth)) == None)
   {
      sprintf (msg, "Can not allocate pixmap of size %1dx%1d.", w, h);
      Msg (msg);
      SetDefaultCursor (mainWindow);
      SetDefaultCursor (drawWindow);
      return (FALSE);
   }

   if ((*image = XGetImage (mainDisplay, *pixmap, 0, 0, w, h, AllPlanes,
         ZPixmap)) == NULL)
   {
      Msg ("XGetImage() failed!  May have run out of memory!");
      XFreePixmap (mainDisplay, *pixmap); *pixmap = None;
      SetDefaultCursor (mainWindow);
      SetDefaultCursor (drawWindow);
      return (FALSE);
   }

   if (orig_image!=NULL && x==0 && y==0)
      src_image = orig_image;
   else
   {
      if ((src_image = XGetImage (mainDisplay, orig_pixmap, x, y, w, h,
            AllPlanes, ZPixmap)) == NULL)
      {
         Msg ("XGetImage() failed!  May have run out of memory!");
         XFreePixmap (mainDisplay, *pixmap); *pixmap = None;
         XDestroyImage (*image); *image = NULL;
         SetDefaultCursor (mainWindow);
         SetDefaultCursor (drawWindow);
         return (FALSE);
      }
   }

   for (i = 0; i < h; i++)
      for (j = 0; j < w; j++)
         XPutPixel (*image, j, i, XGetPixel (src_image, j, i));

   XPutImage (mainDisplay, *pixmap, xpmGC, *image, 0, 0, 0, 0, w, h);
   SetDefaultCursor (mainWindow);
   SetDefaultCursor (drawWindow);

   return (TRUE);
}

void CutXPixmap ()
{
   register int		j, i;
   int			w, h, chars_per_pixel, ncolors, * pixels, len;
   int			ltx, lty, rbx, rby, new_w, new_h;
   int			src_x, src_y, src_w, src_h, image_w, image_h;
   char			mag_spec[MAXSTRING], msg[MAXSTRING];
   char			* color_char, * * color_str;
   float		h_scale=1.0, v_scale=1.0, mag;
   Pixmap		dest_pixmap;
   XImage		* dest_image;
   struct ObjRec	* obj_ptr = topSel->obj, * new_obj_ptr;
   struct XPmRec	* new_xpm_ptr;

   src_x = 0;
   src_y = 0;
   src_w = image_w = obj_ptr->detail.xpm->image_w;
   src_h = image_h = obj_ptr->detail.xpm->image_h;
   mag = 1.0;

   switch (obj_ptr->detail.xpm->rotate)
   {
      case ROTATE0:
      case ROTATE180:
         h_scale = ((float)((float)(obj_ptr->obbox.rbx-obj_ptr->obbox.ltx)) /
               ((float)image_w));
         v_scale = ((float)((float)(obj_ptr->obbox.rby-obj_ptr->obbox.lty)) /
               ((float)image_h));
         break;
      case ROTATE90:
      case ROTATE270:
         h_scale = ((float)((float)(obj_ptr->obbox.rby-obj_ptr->obbox.lty)) /
               ((float)image_w));
         v_scale = ((float)((float)(obj_ptr->obbox.rbx-obj_ptr->obbox.ltx)) /
               ((float)image_h));
         break;
   }

   sprintf (msg, "%s: [[MAG=]WxH+X+Y] (original size is %1dx%1d)",
         "Please enter geometry spec", image_w, image_h);
   Dialog (msg, "( <CR>: accept, <ESC>: cancel )", mag_spec);
   if (*mag_spec == '\0') return;

   ParseCutSpec (mag_spec,image_w,image_h,&mag,&src_x,&src_y,&src_w,&src_h);
   if (src_x==0 && src_y==0 && src_w==image_w && src_h==image_h && mag==1.0)
      return;

   if (src_w==0 || src_h==0)
   {
      Msg ("Pixmap can not have 0 width or height.");
      return;
   }

   PrepareToRecord (CMD_REPLACE, topSel, botSel, numObjSelected);
   if (!ExtractPixmap (obj_ptr->detail.xpm->pixmap,
         obj_ptr->detail.xpm->image, src_x, src_y, src_w, src_h,
         &dest_pixmap, &dest_image))
   {
      AbortPrepareCmd (CMD_REPLACE);
      return;
   }

   UnlinkObj (obj_ptr);

   sprintf (msg, "New pixmap size is %1dx%1d.", src_w, src_h);
   Msg (msg);

   ltx = selLtX; lty = selLtY; rbx = selRbX; rby = selRbY;
   HighLightReverse ();

   w = new_w = (int)(((float)src_w) * mag);
   h = new_h = (int)(((float)src_h) * mag);

   new_obj_ptr = (struct ObjRec *) calloc (1, sizeof(struct ObjRec));
   DupObjBasics (obj_ptr, new_obj_ptr);

   new_xpm_ptr = (struct XPmRec *) calloc (1, sizeof(struct XPmRec));
   new_obj_ptr->detail.xpm = new_xpm_ptr;

   new_xpm_ptr->image = dest_image;
   new_xpm_ptr->image_w = src_w;
   new_xpm_ptr->image_h = src_h;
   new_xpm_ptr->pixmap = dest_pixmap;
   new_xpm_ptr->data = NULL;
   new_xpm_ptr->fill = obj_ptr->detail.xpm->fill;
   new_xpm_ptr->rotate = obj_ptr->detail.xpm->rotate;
   new_xpm_ptr->flip = obj_ptr->detail.xpm->flip;
   new_xpm_ptr->cached_zoom = 0;
   new_xpm_ptr->cached_pixmap = None;
   new_xpm_ptr->cached_rotate = INVALID;
   new_xpm_ptr->cached_flip = 0;
   new_xpm_ptr->cached_w = 0;
   new_xpm_ptr->cached_h = 0;

   chars_per_pixel = new_xpm_ptr->chars_per_pixel =
         obj_ptr->detail.xpm->chars_per_pixel;
   new_xpm_ptr->first_pixel_is_bg = obj_ptr->detail.xpm->first_pixel_is_bg;

   ncolors = new_xpm_ptr->ncolors = obj_ptr->detail.xpm->ncolors;
   color_char = new_xpm_ptr->color_char =
         (char *) calloc (ncolors*chars_per_pixel, sizeof(char));
   color_str = new_xpm_ptr->color_str =
         (char * *) calloc (ncolors, sizeof(char *));
   pixels = new_xpm_ptr->pixels = (int *) calloc (ncolors, sizeof(int));
   for (i = 0; i < ncolors; i++)
   {
      pixels[i] = obj_ptr->detail.xpm->pixels[i];

      for (j = 0; j < chars_per_pixel; j++)
         color_char[i*chars_per_pixel+j] =
               obj_ptr->detail.xpm->color_char[i*chars_per_pixel+j];

      len = strlen (obj_ptr->detail.xpm->color_str[i]);
      color_str[i] = (char *) calloc (len+1, sizeof(char));
      strcpy (color_str[i], obj_ptr->detail.xpm->color_str[i]);
   }

   switch (obj_ptr->detail.xpm->rotate)
   {
      case ROTATE0:
      case ROTATE180:
         new_w = round(h_scale * ((float)w));
         new_h = round(v_scale * ((float)h));
         break;
      case ROTATE90:
      case ROTATE270:
         new_h = round(h_scale * ((float)w));
         new_w = round(v_scale * ((float)h));
         break;
   }

   new_obj_ptr->obbox.ltx = obj_ptr->obbox.ltx;
   new_obj_ptr->obbox.lty = obj_ptr->obbox.lty;
   new_obj_ptr->obbox.rbx = new_obj_ptr->bbox.rbx = obj_ptr->obbox.ltx+new_w;
   new_obj_ptr->obbox.rby = new_obj_ptr->bbox.rby = obj_ptr->obbox.lty+new_h;

   AdjObjBBox (new_obj_ptr);

   topSel->obj = botSel->obj = new_obj_ptr;
   AddObj (NULL, topObj, new_obj_ptr);
   RecordCmd (CMD_REPLACE, NULL, topSel, botSel, numObjSelected);
   FreeObj (obj_ptr);

   UpdSelBBox ();
   RedrawAreas (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
         rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1),
         selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
         selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1));
   HighLightForward ();
   SetFileModified (TRUE);
   justDupped = FALSE;
}

static
void SaveXPmColors (FP, ncolors, chars_per_pixel, color_char, color_str, pixels)
   FILE	* FP;
   int	ncolors, chars_per_pixel, * pixels;
   char	* color_char, * * color_str;
{
   register int	i, j;
   int		cur_pixel, found_index;

   for (i = 0; i < ncolors-1; i++)
   {
      found_index = 0;
      cur_pixel = pixels[i];
      for (j = 0; j < maxColors; j++)
         if (colorPixels[j] == cur_pixel)
         {
            found_index = j;
            break;
         }

      if (fprintf (FP, "   \"") == EOF) writeFileFailed = TRUE;
      for (j = 0; j < chars_per_pixel; j++)
         if (fprintf (FP, "%c", color_char[i*chars_per_pixel+j]) == EOF)
            writeFileFailed = TRUE;
      if (fprintf (FP, "\", \"%s\", %1d, %1d, %1d,\n", color_str[i],
            (int)(10000*tgifColors[found_index].red/maxRGB),
            (int)(10000*tgifColors[found_index].green/maxRGB),
            (int)(10000*tgifColors[found_index].blue/maxRGB)) == EOF)
         writeFileFailed = TRUE;
   }
   found_index = 0;
   cur_pixel = pixels[i];
   for (j = 0; j < maxColors; j++)
      if (colorPixels[j] == cur_pixel)
      {
         found_index = j;
         break;
      }

   if (fprintf (FP, "   \"") == EOF) writeFileFailed = TRUE;
   for (j = 0; j < chars_per_pixel; j++)
      if (fprintf (FP, "%c", color_char[i*chars_per_pixel+j]) == EOF)
         writeFileFailed = TRUE;
   if (fprintf (FP, "\", \"%s\", %1d, %1d, %1d],[\n", color_str[i],
         (int)(10000*tgifColors[found_index].red/maxRGB),
         (int)(10000*tgifColors[found_index].green/maxRGB),
         (int)(10000*tgifColors[found_index].blue/maxRGB)) == EOF)
      writeFileFailed = TRUE;
}

void DumpXPmObj (FP, ObjPtr)
   FILE			* FP;
   struct ObjRec	* ObjPtr;
{
   register int		row, i, index, col;
   int			ltx, lty, rbx, rby, w, h, image_w, image_h;
   int			ncolors, * pixels, rotate, flip, orig_x, orig_y;
   int			cur_pixel, chars_per_pixel;
   int			x, y, found_index=0, pixel_index;
   char			* xpm_data, * color_char;
   Pixmap		pixmap;
   XImage		* image=NULL;
   struct XPmRec	* xpm_ptr;
   struct MtrxRec	mtrx;

   ltx = ObjPtr->obbox.ltx;
   lty = ObjPtr->obbox.lty;
   rbx = ObjPtr->obbox.rbx;
   rby = ObjPtr->obbox.rby;

   xpm_ptr = ObjPtr->detail.xpm;

   pixmap = xpm_ptr->pixmap;
   pixels = xpm_ptr->pixels;
   ncolors = xpm_ptr->ncolors;
   rotate = xpm_ptr->rotate;
   flip = xpm_ptr->flip;
   image_w = xpm_ptr->image_w;
   image_h = xpm_ptr->image_h;

   w = rbx - ltx;
   h = rby - lty;

   if (!PRTGIF && (image=xpm_ptr->image) == NULL)
      if ((image = xpm_ptr->image = XGetImage (mainDisplay, pixmap, 0, 0,
            image_w, image_h, AllPlanes, ZPixmap)) == NULL)
      {
         Msg ("XGetImage() failed!  May have run out of memory!");
         Msg ("X pixmap object skipped for printing.");
         return;
      }

   mtrx.image_w = (float)image_w; mtrx.image_h = (float)image_h;
   mtrx.w = (float)w; mtrx.h = (float)h;
   mtrx.rotate = rotate; mtrx.flip = flip;

   CalcTransform (&mtrx);

   orig_x = (mtrx.transformed_w >= 0.0) ? ltx : ltx+w;
   orig_y = (mtrx.transformed_h >= 0.0) ? lty : lty+h;

   fprintf (FP, "%% XPM\n");
/* fprintf (FP, "newpath\n"); */
/* fprintf (FP, "   %1d %1d moveto ", ltx, lty); */
/* fprintf (FP, "%1d %1d lineto ", rbx, lty); */
/* fprintf (FP, "%1d %1d lineto ", rbx, rby); */
/* fprintf (FP, "%1d %1d lineto\n", ltx, rby); */
/* fprintf (FP, "closepath 1 setgray fill 0 setgray\n"); */

   xpm_data = xpm_ptr->data;
   chars_per_pixel = xpm_ptr->chars_per_pixel;
   color_char = xpm_ptr->color_char;

   if (PRTGIF && fileVersion < 25)
   {  /* can't print color with PRTGIF, treat it like a bitmap */
      int	j, num_nibbles, nibble_count;
      int	h_blocks, v_blocks, block_w, block_h, bit_count, data;
      char	bg_char[2];

      fprintf (stderr, "Warning (xpm object):  %s\n",
            "All non-background color will be treated as black.");
      switch (chars_per_pixel)
      {
         case 1:
            bg_char[0] = xpm_ptr->color_char[0];
            bg_char[1] = '\0';
            if (*bg_char != '`' && *bg_char != ' ')
               bg_char[0] = bg_char[1] = '\0';
            break;
         case 2:
            bg_char[0] = xpm_ptr->color_char[0];
            bg_char[1] = xpm_ptr->color_char[1];
            if (!(bg_char[0] == '`' && bg_char[1] == '`') &&
                  !(bg_char[0] == ' ' && bg_char[1] == ' '))
               bg_char[0] = '\0';
            break;
      }

      h_blocks = ((image_w&0xff) == 0) ? (image_w>>8) : ((image_w>>8)+1);
      v_blocks = ((image_h&0xff) == 0) ? (image_h>>8) : ((image_h>>8)+1);

      fprintf (FP, "gsave\n");
      fprintf (FP, "   %1d %1d translate %.3f %.3f scale %1d rotate\n\n",
            orig_x, orig_y, mtrx.dump_h_scale, mtrx.dump_v_scale, mtrx.degree);

      for (row = 0; row < v_blocks; row++)
      {
         y = row<<8;
         block_h = (row == v_blocks-1) ? image_h-y : 0x100;

         for (col = 0; col < h_blocks; col++)
         {
            x = col<<8;
            block_w = (col == h_blocks-1) ? image_w-x : 0x100;

            num_nibbles = ((block_w%4) == 0) ? (int)(block_w>>2) :
                  (int)(block_w>>2)+1;

            fprintf (FP, "   gsave\n");
            fprintf (FP, "   %1d %1d translate\n", x, y);
            fprintf (FP, "   %1d %1d true [1 0 0 1 0 0]\n   {<", block_w,
                  block_h);

            nibble_count = 0;
            for (i = 0; i < block_h; i++)
            {
               bit_count = 0;
               data = 0;
               for (j = 0; j < block_w; j++)
               {
                  switch (chars_per_pixel)
                  {
                     case 1:
                        data = (xpm_data[(i+y)*image_w+j+x] != *bg_char) ?
                              (data<<1) | 1 : (data<<1);
                        break;
                     case 2:
                        index = ((i+y)*image_w+j+x)*chars_per_pixel;
                        data = (xpm_data[index] == bg_char[0] &&
                              xpm_data[index+1] == bg_char[1]) ?
                              (data<<1) : (data<<1) | 1;
                        break;
                  }

                  if (++bit_count == 4)
                  {
                     if (nibble_count++ == 64)
                     {
                        nibble_count = 1;
                        fprintf (FP, "\n     ");
                     }
                     fprintf (FP, "%c", hexValue[data]);
                     bit_count = 0;
                     data = 0;
                  }
               }
               if ((block_w % 4) != 0)
               {
                  data <<= (4 - (block_w % 4));
                  if (nibble_count++ == 64)
                  {
                     nibble_count = 1;
                     fprintf (FP, "\n     ");
                  }
                  fprintf (FP, "%c", hexValue[data]);
               }
               if ((num_nibbles & 0x1) == 1)
               {
                  if (nibble_count++ == 64) {
                     nibble_count = 1;
                     fprintf (FP, "\n     ");
                  }
                  fprintf (FP, "0");
               }
            }
            fprintf (FP, ">}\n");
            fprintf (FP, "   imagemask\n");
            fprintf (FP, "   grestore\n");
            if (row!=v_blocks-1 || col!=h_blocks-1) fprintf (FP, "\n");
         }
      }
      fprintf (FP, "grestore\n");
      fprintf (FP, "\n");
      return;
   }

   if (ncolors > 0xff)
   {
      if (PRTGIF)
         fprintf (stderr, "Too many colors in an xpm objects -- %s.\n",
               "skipped for printing");
      else
         Msg ("Too many colors in an xpm objects -- skipped for printing.");
      return;
   }

   x = (mtrx.transformed_w >= 0.0) ? ltx : ltx+w;
   y = (mtrx.transformed_h >= 0.0) ? lty : lty+h;

   fprintf (FP, "gsave\n");
   if (colorDump)
   {  /* must not be PRTGIF */
      fprintf (FP, "   3 %1d mul array tgifsetpixels\n", ncolors);

      BuildXPmBuckets (ncolors, pixels, INVALID, NULL);

      for (index = 0; index < ncolors; index++)
      {
         cur_pixel = pixels[index];
         for (i = 0; i < maxColors; i++)
            if (colorPixels[i] == cur_pixel)
            {
               found_index = i;
               break;
            }
         fprintf (FP, "%s%3d [ %.3f %.3f %.3f ] tgifsetpix",
               ((index & 0x1) ? "" : "   "), index*3,
               ((float)tgifColors[found_index].red/maxRGB),
               ((float)tgifColors[found_index].green/maxRGB),
               ((float)tgifColors[found_index].blue/maxRGB));
         fprintf (FP, "%s", ((index & 0x1) ? "\n" : " "));
      }
      if (ncolors & 0x1) fprintf (FP, "\n");
   }
   else if (PRTGIF)
   {
      BuildXPmBuckets (ncolors, NULL, chars_per_pixel, color_char);
   }

   fprintf (FP, "   %1d %1d translate %.3f %.3f scale %1d rotate\n", x, y,
         mtrx.dump_h_scale, mtrx.dump_v_scale, mtrx.degree);
   fprintf (FP, "   %1d %1d 8\n   [1 0 0 1 0 0]\n", image_w, image_h);

   if (!colorDump)
   {
      fprintf (FP, "   {currentfile\n");
      fprintf (FP, "    tgifbwpicstr readhexstring pop}\n");
      fprintf (FP, "   image\n   ");
   }
   else
   {
      fprintf (FP, "   {currentfile\n");
      fprintf (FP, "    tgifbwpicstr readhexstring pop 0 get tgifcolorspot}\n");
      fprintf (FP, "   false 3 colorimage\n   ");
   }
   if (PRTGIF)
   {
      for (row = 0; row < image_h; row++)
      {
         for (col = 0; col < image_w; col++)
         {
            float	gray;
            int		value;

            pixel_index = XPmLookUp (INVALID, chars_per_pixel,
                  &(xpm_data[(row*image_w+col)*chars_per_pixel]));

            gray = 0.299*((float)xpm_ptr->red[pixel_index]/10000.0) +
                  0.587*((float)xpm_ptr->green[pixel_index]/10000.0) +
                  0.114*((float)xpm_ptr->blue[pixel_index]/10000.0);
            value = gray*256;
            pixel_index = ((value>255) ? 255 : ((value<0) ? 0 : value));

            fprintf (FP, "%c", hexValue[(pixel_index>>4)&0x0f]);
            fprintf (FP, "%c", hexValue[pixel_index&0x0f]);
            if (col%36 == 35) fprintf (FP, "\n   ");
         }
         if (col%36 != 0) fprintf (FP, "\n");
         if (row != image_h-1) fprintf (FP, "   ");
      }
   }
   else
   {
      for (row = 0; row < image_h; row++)
      {
         for (col = 0; col < image_w; col++)
         {
            if (colorDump)
               pixel_index = XPmLookUp (XGetPixel(image,col,row),INVALID,NULL);
            else
            {
               float	gray;
               int	value;

               found_index = 0;
               cur_pixel = XGetPixel (image, col, row);
               for (i = 0; i < maxColors; i++)
                  if (colorPixels[i] == cur_pixel)
                  {
                     found_index = i;
                     break;
                  }

               gray = 0.299*((float)tgifColors[found_index].red/maxRGB) +
                     0.587*((float)tgifColors[found_index].green/maxRGB) +
                     0.114*((float)tgifColors[found_index].blue/maxRGB);
               value = gray*256;
               pixel_index = ((value>255) ? 255 : ((value<0) ? 0 : value));
            }
            fprintf (FP, "%c", hexValue[(pixel_index>>4)&0x0f]);
            fprintf (FP, "%c", hexValue[pixel_index&0x0f]);
            if (col%36 == 35)
               fprintf (FP, "\n%s",
                     (col==image_w-1 && row==image_h-1) ? "" : "   ");
         }
         if (col%36 != 0) fprintf (FP, "\n");
         if (row != image_h-1 && (col%36) != 0) fprintf (FP, "   ");
      }
   }
   fprintf (FP, "grestore\n");
   fprintf (FP, "\n");
}

int NeedsToCacheXPmObj (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   register struct XPmRec	* xpm_ptr = ObjPtr->detail.xpm;
   int				w, h;

   w = ObjPtr->obbox.rbx - ObjPtr->obbox.ltx;
   h = ObjPtr->obbox.rby - ObjPtr->obbox.lty;

   return (zoomScale != 0 ||
      xpm_ptr->image_w != w || xpm_ptr->image_h != h ||
      xpm_ptr->rotate != ROTATE0 || xpm_ptr->flip != 0
   );
}

static
void DrawHiddenXPm (win, x, y, w, h, s)
   Window	win;
   int		x, y, w, h;
   char		* s;
{
   int		str_w, len, sx, sy;
   XGCValues	values;

   values.foreground = myBgPixel;
   values.function = GXcopy;;
   values.fill_style = FillSolid;
#ifdef NO_THIN_LINE
   values.line_width = 1;
#else
   values.line_width = 0;
#endif
   values.font = rulerFontPtr->fid;
   values.line_style = LineSolid;

   XChangeGC (mainDisplay, drawGC,
         GCForeground | GCFunction | GCFillStyle | GCLineWidth | GCFont |
         GCLineStyle, &values);
   XFillRectangle (mainDisplay, win, drawGC, x, y, w, h);

   XSetForeground (mainDisplay, drawGC, myFgPixel);
   XDrawRectangle (mainDisplay, win, drawGC, x, y, w, h);

   XSetForeground (mainDisplay, drawGC, colorPixels[colorIndex]);

   len = strlen (s);
   str_w = rulerFontWidth*len;
   if (str_w < w && rulerFontHeight < h)
   {
      sx = x + ((w-str_w)>>1);
      sy = y + ((h-rulerFontHeight)>>1);
      XDrawString (mainDisplay, win, drawGC, sx, sy+rulerFontAsc, s, len);
   }
   XSetFont (mainDisplay, drawGC, canvasFontPtr->fid);
}

void DrawXPmObj (win, XOff, YOff, ObjPtr)
   Window		win;
   int			XOff, YOff;
   struct ObjRec	* ObjPtr;
{
   int			ltx, lty, rbx, rby, w, h, scr_w, scr_h;
   int			real_x_off, real_y_off;
   char			s[80];
   struct XPmRec	* xpm_ptr = ObjPtr->detail.xpm;

   if (ObjPtr->prev != NULL && ObjPtr->prev->type == OBJ_XPM &&
         Inside(ObjPtr->obbox, ObjPtr->prev->obbox))
   {
      return;
   }

   w = ObjPtr->obbox.rbx - ObjPtr->obbox.ltx;
   h = ObjPtr->obbox.rby - ObjPtr->obbox.lty;

   real_x_off = (zoomedIn ? XOff : (XOff>>zoomScale)<<zoomScale);
   real_y_off = (zoomedIn ? YOff : (YOff>>zoomScale)<<zoomScale);
   ltx = ZOOMED_SIZE(ObjPtr->obbox.ltx - real_x_off);
   lty = ZOOMED_SIZE(ObjPtr->obbox.lty - real_y_off);
   rbx = ZOOMED_SIZE(ObjPtr->obbox.rbx - real_x_off);
   rby = ZOOMED_SIZE(ObjPtr->obbox.rby - real_y_off);
   scr_w = rbx - ltx;
   scr_h = rby - lty;

   if (!mapShown)
   {
      sprintf (s, "(%1dx%1d)", xpm_ptr->image_w, xpm_ptr->image_h);
      DrawHiddenXPm (win, ltx, lty, scr_w, scr_h, s);
      return;
   }

   if (NeedsToCacheXPmObj (ObjPtr)) MakeCachedPixmap (ObjPtr);

   XSetFunction (mainDisplay, drawGC, GXcopy);
   if (zoomScale==0 && xpm_ptr->rotate==ROTATE0 && xpm_ptr->flip==0 &&
         xpm_ptr->image_w==w && xpm_ptr->image_h==h)
      XCopyArea(mainDisplay, xpm_ptr->pixmap, win, drawGC, 0, 0, w, h,
            ltx, lty);
   else
   {
      if (xpm_ptr->cached_pixmap == None) return;
      XCopyArea(mainDisplay, xpm_ptr->cached_pixmap, win, drawGC, 0, 0,
            scr_w, scr_h, ltx, lty);
   }
}

struct ObjRec * CreateXPmObj (ImageW, ImageH, W, H, pixmap, image, ncolors,
      chars_per_pixel, first_pixel_is_bg, color_char, color_str, pixels)
   int		ImageW, ImageH, W, H;
   Pixmap	pixmap;
   XImage	* image;
   int		ncolors;
   int		chars_per_pixel, first_pixel_is_bg;
   char		* color_char;
   char		* * color_str;
   int		* pixels;
{
   struct XPmRec	* xpm_ptr;
   struct ObjRec	* obj_ptr;

   xpm_ptr = (struct XPmRec *) calloc (1, sizeof(struct XPmRec));

   xpm_ptr->image = image;
   xpm_ptr->image_w = ImageW;
   xpm_ptr->image_h = ImageH;
   xpm_ptr->pixmap = pixmap;
   xpm_ptr->data = NULL;

   xpm_ptr->fill = objFill;
   xpm_ptr->rotate = xpm_ptr->cached_rotate = ROTATE0;
   xpm_ptr->flip = xpm_ptr->cached_flip = 0;
   xpm_ptr->cached_zoom = 0;
   xpm_ptr->cached_pixmap = None;
   xpm_ptr->cached_w = xpm_ptr->cached_h = 0;

   xpm_ptr->ncolors = ncolors;
   xpm_ptr->chars_per_pixel = chars_per_pixel;
   xpm_ptr->first_pixel_is_bg = first_pixel_is_bg;
   xpm_ptr->color_char = color_char;
   xpm_ptr->color_str = color_str;
   xpm_ptr->pixels = pixels;
   xpm_ptr->red = xpm_ptr->green = xpm_ptr->blue = NULL;

   obj_ptr = (struct ObjRec *) calloc (1, sizeof(struct ObjRec));

   obj_ptr->bbox.ltx = obj_ptr->obbox.ltx = obj_ptr->x = drawOrigX;
   obj_ptr->bbox.lty = obj_ptr->obbox.lty = obj_ptr->y = drawOrigY;
   obj_ptr->bbox.rbx = obj_ptr->obbox.rbx = W + drawOrigX;
   obj_ptr->bbox.rby = obj_ptr->obbox.rby = H + drawOrigY;
   obj_ptr->type = OBJ_XPM;
   obj_ptr->color = colorIndex;
   obj_ptr->id = objId++;
   obj_ptr->dirty = FALSE;
   obj_ptr->rotation = 0;
   obj_ptr->locked = FALSE;
   obj_ptr->detail.xpm = xpm_ptr;
   obj_ptr->fattr = obj_ptr->lattr = NULL;
   return (obj_ptr);
}
 
static
void SaveXPmPixels (FP,pixmap,image,w,h,ncolors,chars_per_pixel,color_char,
      pixels)
   FILE		* FP;
   Pixmap	pixmap;
   XImage	* * image;
   int		w, h, ncolors, chars_per_pixel, * pixels;
   char		* color_char;
{
   register int	j, data, index, i, k;
   char		msg[MAXSTRING];

   if (*image == NULL)
   {
      *image = XGetImage (mainDisplay, pixmap, 0, 0, w, h, AllPlanes, ZPixmap);
      if (*image == NULL)
      {
         Msg ("XGetImage() failed!  May have run out of memory!");
         Msg ("Saved file may be corrupted.");
         return;
      }
   }
   for (i = 0; i < h; i++)
   {
      if (fprintf (FP, "   \"") == EOF) writeFileFailed = TRUE;
      for (j = 0; j < w; j++)
      {
         data = XGetPixel (*image,j,i);
         index = XPmLookUp (data, INVALID, NULL);
         if (index == INVALID)
         {
            sprintf (msg, "Unrecognized pixel value %1d!  Print aborted!",
                  data);
            Msg (msg);
            return;
         }
         for (k = 0; k < chars_per_pixel; k++)
            if (fprintf (FP, "%c", color_char[index*chars_per_pixel+k]) == EOF)
               writeFileFailed = TRUE;
      }
      if (i == h-1)
      {
         if (fprintf (FP, "\"],") == EOF) writeFileFailed = TRUE;
      }
      else
      {
         if (fprintf (FP, "\",\n") == EOF) writeFileFailed = TRUE;
      }
   }
}

void SaveXPmObj (FP, ObjPtr)
   FILE			* FP;
   struct ObjRec	* ObjPtr;
{
   int			ltx, lty, rbx, rby, fill, ncolors, * pixels;
   int			chars_per_pixel, first_pixel_is_bg, image_w, image_h;
   struct XPmRec	* xpm_ptr = ObjPtr->detail.xpm;
   char			* color_char, * * color_str;

   ltx = ObjPtr->obbox.ltx; lty = ObjPtr->obbox.lty;
   rbx = ObjPtr->obbox.rbx; rby = ObjPtr->obbox.rby;
   image_w = xpm_ptr->image_w;
   image_h = xpm_ptr->image_h;
   fill = xpm_ptr->fill;
   ncolors = xpm_ptr->ncolors;
   chars_per_pixel = xpm_ptr->chars_per_pixel;
   first_pixel_is_bg = xpm_ptr->first_pixel_is_bg;
   color_char = xpm_ptr->color_char;
   color_str = xpm_ptr->color_str;
   pixels = xpm_ptr->pixels;

   if (fprintf (FP, "xpm('%s',", colorMenuItems[ObjPtr->color]) == EOF)
      writeFileFailed = TRUE;
   if (fprintf (FP,
         "%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,%1d,[\n",
         ltx, lty, rbx, rby, fill, ncolors, chars_per_pixel, first_pixel_is_bg,
         ObjPtr->id, ObjPtr->rotation, image_w, image_h, xpm_ptr->rotate,
         xpm_ptr->flip, ObjPtr->locked) == EOF)
      writeFileFailed = TRUE;

   BuildXPmBuckets (ncolors, pixels, INVALID, NULL);
   SaveXPmColors (FP, ncolors, chars_per_pixel, color_char, color_str, pixels);
   SaveXPmPixels (FP, xpm_ptr->pixmap, &(xpm_ptr->image), image_w, image_h,
         ncolors, chars_per_pixel, color_char, pixels);

   SaveAttrs (FP, ObjPtr->lattr);
   if (fprintf (FP, ")") == EOF) writeFileFailed = TRUE;
}

int MyStrICmp (Str1, Str2)
   char	* Str1, * Str2;
{
   register char	c1, c2, * s1, * s2;

   for (s1=Str1, s2=Str2; *s1 != '\0'; s1++, s2++)
   {
      c1 = (*s1 >= 'a' && *s1 <= 'z' ? *s1-'a'+'A' : *s1);
      c2 = (*s2 >= 'a' && *s2 <= 'z' ? *s2-'a'+'A' : *s2);
      if (c1 != c2) return (*s1 - *s2);
   }
   return (*s2 == '\0' ? 0 : (-(*s2)));
}

int QuickFindColorIndex (s, new_alloc)
   char	* s;
   int	* new_alloc;
{
   register int i;
   XColor	exact_def, screen_def;
   Colormap	colormap;

   *new_alloc = FALSE;

   if (colorMenuItems == NULL) return (INVALID);

   if (*s == '#')
   {
      if (!XParseColor (mainDisplay, mainColormap, s, &exact_def))
         return (INVALID);
   }
   else
   {
      for (i = 0; i < maxColors; i++)
      {
         int	valid = FALSE;
         char	*s1 = s, *s2 = colorMenuItems[i];

         if (*s1 == *s2)
            valid = TRUE;
         else if (*s1 >= 'a' && *s1 <= 'z')
         {
            if (*s1+'A'-'a' == *s2)
               valid = TRUE;
         }
         else if (*s >= 'A' && *s <= 'Z')
         {
            if (*s1+'a'-'A' == *s2)
               valid = TRUE;
         }

         if (valid && (MyStrICmp(++s1,++s2) == 0))
            return (i);
      }

      if (!XParseColor (mainDisplay, mainColormap, s, &exact_def))
         return (INVALID);
   }

   for (i = 0; i < maxColors; i++)
      if (tgifRequestedColors[i].red == exact_def.red &&
            tgifRequestedColors[i].green == exact_def.green &&
            tgifRequestedColors[i].blue == exact_def.blue)
         return (i);

   screen_def = exact_def;

   if (!XAllocColor (mainDisplay, mainColormap, &screen_def))
   {
      if (newColormapUsed) return (INVALID);

      colormap = XCopyColormapAndFree (mainDisplay, mainColormap);
      mainColormap = colormap;
      newColormapUsed = TRUE;
      XSetWindowColormap (mainDisplay, mainWindow, mainColormap);
      if (!XAllocColor (mainDisplay, mainColormap, &screen_def))
         return (INVALID);
   }

   colorPixels = (int *) realloc (colorPixels, (maxColors+1)*sizeof(int));
   xorColorPixels = (int *) realloc (xorColorPixels,
         (maxColors+1)*sizeof(int));
   colorMenuItems = (char * *) realloc (colorMenuItems,
         (maxColors+1)*sizeof(char *));
   colorMenuItems[maxColors] = (char *) calloc (strlen(s)+1, sizeof(char *));
   tgifColors = (XColor *) realloc (tgifColors,
         (maxColors+1)*sizeof(XColor));
   tgifRequestedColors = (XColor *) realloc (tgifRequestedColors,
         (maxColors+1)*sizeof(XColor));

   colorPixels[maxColors] = screen_def.pixel;
   xorColorPixels[maxColors] = screen_def.pixel ^ myBgPixel;
   strcpy (colorMenuItems[maxColors], s);
   tgifRequestedColors[maxColors].red = exact_def.red;
   tgifRequestedColors[maxColors].green = exact_def.green;
   tgifRequestedColors[maxColors].blue = exact_def.blue;
   tgifColors[maxColors].red = screen_def.red;
   tgifColors[maxColors].green = screen_def.green;
   tgifColors[maxColors].blue = screen_def.blue;
   maxColors++;
   *new_alloc = TRUE;
   return (maxColors-1);
}

static
void FreeAuxData (fp, ncolors, color_char, color_str, pixels, pixmap, image)
   FILE		* fp;
   int		ncolors;
   char		* color_char, * * color_str;
   int		* pixels;
   Pixmap	pixmap;
   XImage	* image;
{
   register int	i;

   for (i = 0; i < ncolors; i++) cfree (color_str[i]);
   cfree (color_char);
   cfree (color_str);
   cfree (pixels);
   if (pixmap != None) XFreePixmap (mainDisplay, pixmap);
   if (image != NULL) XDestroyImage (image);
   fclose (fp);
}

int MyReadPixmapFile (file_name, image_w_return, image_h_return, w_return,
      h_return, pixmap_return, image_return, ncolors_return,
      chars_per_pixel_return, first_pixel_is_bg_return, color_char, color_str,
      pixels)
   char		* file_name;
   int		* image_w_return, * image_h_return, * w_return, * h_return;
   int		* ncolors_return;
   int		* chars_per_pixel_return, * first_pixel_is_bg_return;
   Pixmap	* pixmap_return;
   XImage	* * image_return;
   char		* * color_char, * * * color_str;
   int		* * pixels;
{
   register int		j, k;
   register char	* c_ptr;
   FILE			* fp;
   char			inbuf[MAXSTRING], * line, s[MAXSTRING], msg[MAXSTRING];
   char			mag_spec[MAXSTRING];
   int			i, len, format, found = FALSE, index;
   int			x, y, w, h, chars_per_pixel, new_alloc;
   float		mag;
   int			saved_max_colors = maxColors, xpm_version=1;

   if ((fp = fopen (file_name, "r")) == NULL) return (BitmapOpenFailed);

   while (fgets (inbuf, MAXSTRING, fp) != NULL)
      if (strncmp (inbuf, "#define ", 8) == 0)
      {
         xpm_version = 1;
         found = TRUE;
         break;
      }
      else if (strncmp (inbuf, "/* XPM */", 9) == 0)
      {
         xpm_version = 3;
         found = TRUE;
         break;
      }
   if (!found) return (BitmapFileInvalid);

   if (xpm_version == 1)
   {
      c_ptr = FindChar (' ', inbuf);
      c_ptr = ParseStr (c_ptr, ' ', s);
      len = strlen (s);
      if (len <= 7 || strcmp (&s[len-7], "_format") != 0)
         return (BitmapFileInvalid);
      sscanf (c_ptr, "%d", &format);
      if (format != 1)
      {
         sprintf (msg, "Can not process format %1d.", format);
         Msg (msg);
         return (BitmapFileInvalid);
      }

      if (fgets (inbuf, MAXSTRING, fp) == NULL) return (BitmapFileInvalid);
      c_ptr = FindChar (' ', inbuf);
      c_ptr = ParseStr (c_ptr, ' ', s);
      len = strlen (s);
      if (len <= 6 || strcmp (&s[len-6], "_width") != 0)
         return (BitmapFileInvalid);
      sscanf (c_ptr, "%d", image_w_return);

      if (fgets (inbuf, MAXSTRING, fp) == NULL) return (BitmapFileInvalid);
      c_ptr = FindChar (' ', inbuf);
      c_ptr = ParseStr (c_ptr, ' ', s);
      len = strlen (s);
      if (len <= 7 || strcmp (&s[len-7], "_height") != 0)
         return (BitmapFileInvalid);
      sscanf (c_ptr, "%d", image_h_return);

      if (fgets (inbuf, MAXSTRING, fp) == NULL) return (BitmapFileInvalid);
      c_ptr = FindChar (' ', inbuf);
      c_ptr = ParseStr (c_ptr, ' ', s);
      len = strlen (s);
      if (len <= 8 || strcmp (&s[len-8], "_ncolors") != 0)
         return (BitmapFileInvalid);
      sscanf (c_ptr, "%d", ncolors_return);

      if (fgets (inbuf, MAXSTRING, fp) == NULL) return (BitmapFileInvalid);
      c_ptr = FindChar (' ', inbuf);
      c_ptr = ParseStr (c_ptr, ' ', s);
      len = strlen (s);
      if (len <= 16 || strcmp (&s[len-16], "_chars_per_pixel") != 0)
         return (BitmapFileInvalid);
      sscanf (c_ptr, "%d", chars_per_pixel_return);
      chars_per_pixel = *chars_per_pixel_return;
      if (chars_per_pixel > 2)
      {
         sprintf (msg, "Can not handle chars_per_pixel=%1d.", chars_per_pixel);
         Msg (msg);
         return (BitmapFileInvalid);
      }

      if (fgets (inbuf, MAXSTRING, fp) == NULL) return (BitmapFileInvalid);
      len = strlen (inbuf);
      if (len <= 27 || strncmp (inbuf, "static char *", 13) != 0 ||
            strncmp (&inbuf[len-14], "_colors[] = {\n", 14) != 0)
         return (BitmapFileInvalid);
   }
   else
   {  /* xpm_version is 3 */
      found = FALSE;
      while (fgets (inbuf, MAXSTRING, fp) != NULL)
         if (*inbuf == '"')
         {
            found = TRUE;
            break;
         }
      if (!found) return (BitmapFileInvalid);
      c_ptr = &inbuf[1];
      if (sscanf (c_ptr, "%d %d %d %d", image_w_return, image_h_return,
            ncolors_return, chars_per_pixel_return) != 4)
         return (BitmapFileInvalid);
      chars_per_pixel = *chars_per_pixel_return;
   }

   *color_char = (char *) calloc ((*ncolors_return)*(chars_per_pixel),
         sizeof(char));
   *color_str = (char * *) calloc (*ncolors_return, sizeof(char *));
   *pixels = (int *) calloc (*ncolors_return, sizeof(int));
   *pixmap_return = None;
   *image_return = NULL;
   *first_pixel_is_bg_return = FALSE;

   if (xpm_version == 1)
   {
      for (i = 0; i < *ncolors_return; i++)
      {
         if (fgets (inbuf, MAXSTRING, fp) == NULL)
         {
            FreeAuxData (fp, i, *color_char, *color_str, *pixels,
                  *pixmap_return, *image_return);
            return (BitmapFileInvalid);
         }

         c_ptr = FindChar ('"', inbuf);
         for (j = 0; j < chars_per_pixel; j++)
            (*color_char)[i*(chars_per_pixel)+j] = c_ptr[j];
         if (guessXPmBgColor && i == 0 &&
               ((chars_per_pixel == 1 && (*c_ptr=='`' || *c_ptr==' ')) ||
               (chars_per_pixel == 2 && (c_ptr[0]=='`' && c_ptr[1]=='`' ||
               c_ptr[0]==' ' && c_ptr[1]==' '))))
         {
            strcpy (s, myBgColorStr);
            (*pixels)[0] = myBgPixel;
            *first_pixel_is_bg_return = TRUE;
         }
         else
         {
            c_ptr = FindChar ('"', c_ptr);
            c_ptr = FindChar ('"', c_ptr);
/*          c_ptr = ParseStr (c_ptr, '"', s); */
            ParseStr (c_ptr, '"', s);
            if ((index = QuickFindColorIndex (s,&new_alloc)) == INVALID)
            {
               sprintf (msg, "Can not allocate color '%s', use '%s' instead.",
                     s, colorMenuItems[colorIndex]);
               Msg (msg);
               strcpy (s, colorMenuItems[colorIndex]);
               (*pixels)[i] = colorPixels[colorIndex];
            }
            else
               (*pixels)[i] = colorPixels[index];
         }
         len = strlen (s);
         (*color_str)[i] = (char *) calloc (len+1, sizeof(char));
         strcpy ((*color_str)[i], s);
      }

      if (fgets (inbuf, MAXSTRING, fp) == NULL ||
            inbuf[0] != '}' ||
            fgets (inbuf, MAXSTRING, fp) == NULL)
      {
         FreeAuxData (fp, *ncolors_return, *color_char, *color_str, *pixels,
               *pixmap_return, *image_return);
         return (BitmapFileInvalid);
      }
      len = strlen (inbuf);
      if (len <= 27 || strncmp (inbuf, "static char *", 13) != 0 ||
            strncmp (&inbuf[len-14], "_pixels[] = {\n", 14) != 0)
         return (BitmapFileInvalid);
   }
   else
   {  /* xpm_version is 3 */
      for (i = 0; i < *ncolors_return; i++)
      {
         if (i == 0)
         {
            found = FALSE;
            while (fgets (inbuf, MAXSTRING, fp) != NULL)
               if (*inbuf == '"')
               {
                  found = TRUE;
                  break;
               }
            if (!found)
            {
               FreeAuxData (fp, i, *color_char, *color_str, *pixels,
                     *pixmap_return, *image_return);
               return (BitmapFileInvalid);
            }
         }
         else if (fgets (inbuf, MAXSTRING, fp) == NULL)
         {
            FreeAuxData (fp, i, *color_char, *color_str, *pixels,
                  *pixmap_return, *image_return);
            return (BitmapFileInvalid);
         }

         c_ptr = FindChar ('"', inbuf);
         for (j = 0; j < chars_per_pixel; j++)
            (*color_char)[i*(chars_per_pixel)+j] = c_ptr[j];
         if (guessXPmBgColor && i == 0 &&
               ((chars_per_pixel == 1 && (*c_ptr=='`' || *c_ptr==' ')) ||
               (chars_per_pixel == 2 && (c_ptr[0]=='`' && c_ptr[1]=='`' ||
               c_ptr[0]==' ' && c_ptr[1]==' '))))
         {
            strcpy (s, myBgColorStr);
            (*pixels)[0] = myBgPixel;
            *first_pixel_is_bg_return = TRUE;
         }
         else
         {
            char	* ptr;

            c_ptr += chars_per_pixel;
            while (*c_ptr != '\0')
            {
               while (*c_ptr == ' ' || *c_ptr == '\t') c_ptr++;
               if ((*c_ptr == 'c' || *c_ptr == 'm') &&
                     (c_ptr[1]==' ' || c_ptr[1]=='\t'))
                  break;
               while (*c_ptr!=' ' && *c_ptr!='\t' && *c_ptr!='\0') c_ptr++;
               while (*c_ptr == ' ' || *c_ptr == '\t') c_ptr++;
               while (*c_ptr!=' ' && *c_ptr!='\t' && *c_ptr!='\0') c_ptr++;
            }
            if (*c_ptr++ == '\0')
            {
               FreeAuxData (fp, i, *color_char, *color_str, *pixels,
                     *pixmap_return, *image_return);
               return (BitmapFileInvalid);
            }
            while (*c_ptr == ' ' || *c_ptr == '\t') c_ptr++;
            for (ptr = c_ptr; *ptr !=  '"' && *ptr != '\0'; ptr++)
               if ((*ptr == ' ' || *ptr == '\t') && (
                     strncmp(&ptr[1], "m ", 2) == 0 ||
                     strncmp(&ptr[1], "s ", 2) == 0 ||
                     strncmp(&ptr[1], "g ", 2) == 0 ||
                     strncmp(&ptr[1], "g4 ", 3) == 0))
                  break;
            if (*ptr == '\0')
            {
               FreeAuxData (fp, i, *color_char, *color_str, *pixels,
                     *pixmap_return, *image_return);
               return (BitmapFileInvalid);
            }
            while (ptr >= c_ptr)
            {
               if (*ptr == ' ' || *ptr == '\t' || *ptr == '"')
                  ptr--;
               else
                  break;
            }
            if (ptr < c_ptr)
            {
               FreeAuxData (fp, i, *color_char, *color_str, *pixels,
                     *pixmap_return, *image_return);
               return (BitmapFileInvalid);
            }
            *(++ptr) = '\0';
            strcpy (s, c_ptr);
            if ((index = QuickFindColorIndex (s,&new_alloc)) == INVALID)
            {
               sprintf (msg, "Can not allocate color '%s', use '%s' instead.",
                     s, colorMenuItems[colorIndex]);
               Msg (msg);
               strcpy (s, colorMenuItems[colorIndex]);
               (*pixels)[i] = colorPixels[colorIndex];
            }
            else
               (*pixels)[i] = colorPixels[index];
         }
         len = strlen (s);
         (*color_str)[i] = (char *) calloc (len+1, sizeof(char));
         strcpy ((*color_str)[i], s);
      }
   }

   x = 0;
   y = 0;
   w = *image_w_return;
   h = *image_h_return;
   mag = 1.0;
   if (askForXPmSpec)
   {
      sprintf (msg, "%s: [[MAG=]WxH+X+Y] (original size is %1dx%1d)",
         "Please enter geometry spec", *image_w_return, *image_h_return);
      Dialog (msg, "( <CR>: accept, <ESC>: continue )", mag_spec);
      if (*mag_spec != '\0')
         ParseCutSpec (mag_spec, *image_w_return, *image_h_return, &mag, &x, &y,
               &w, &h);
   }

   *pixmap_return = XCreatePixmap (mainDisplay, dummyPixmap, w, h, mainDepth);
   if (*pixmap_return == None)
   {
      FreeAuxData (fp, *ncolors_return, *color_char, *color_str, *pixels,
            *pixmap_return, *image_return);
      return (BitmapNoMemory);
   }
   XFillRectangle (mainDisplay, *pixmap_return, xpmGC, 0, 0, w, h);
   *image_return = XGetImage (mainDisplay, *pixmap_return, 0, 0, w, h,
         AllPlanes, ZPixmap);
   if (*image_return == NULL)
   {
      Msg ("XGetImage() failed!  May have run out of memory!");
      FreeAuxData (fp, *ncolors_return, *color_char, *color_str, *pixels,
            *pixmap_return, *image_return);
      return (BitmapNoMemory);
   }

   BuildXPmBuckets (*ncolors_return, NULL, chars_per_pixel, *color_char);

   line = (char *) calloc ((*image_w_return)*chars_per_pixel+10, sizeof(char));
   for (i = 0; i < y+h; i++)
   {
      if (xpm_version == 3 && i == 0)
      {
         found = FALSE;
         while (fgets (line, (*image_w_return)*chars_per_pixel+10, fp) != NULL)
            if (*line == '"')
            {
               found = TRUE;
               break;
            }
         if (!found)
         {
            FreeAuxData (fp, *ncolors_return, *color_char, *color_str, *pixels,
                  *pixmap_return, *image_return);
            return (BitmapFileInvalid);
         }
      }
      else if (fgets (line, (*image_w_return)*chars_per_pixel+10, fp) == NULL)
      {
         FreeAuxData (fp, *ncolors_return, *color_char, *color_str, *pixels,
               *pixmap_return, *image_return);
         cfree (line);
         return (BitmapFileInvalid);
      }
      if (i >= y)
      {
         c_ptr = FindChar ('"', line);
         for (j = 0; j < x+w; j++, c_ptr+=chars_per_pixel)
         {
            if (j >= x)
            {
               k = XPmLookUp (INVALID, chars_per_pixel, c_ptr);
               if (k == INVALID)
               {
                  FreeAuxData (fp, *ncolors_return, *color_char, *color_str,
                        *pixels, *pixmap_return, *image_return);
                  cfree (line);
                  return (BitmapFileInvalid);
               }
               XPutPixel (*image_return, j-x, i-y, (*pixels)[k]);
            }
         }
      }
   }
   cfree (line);
   fclose (fp);
   XPutImage (mainDisplay,*pixmap_return,xpmGC,*image_return,0,0,0,0,w,h);
   *image_w_return = w;
   *image_h_return = h;
   *w_return = (int)(((float)w) * mag);
   *h_return = (int)(((float)h) * mag);
   if (saved_max_colors != maxColors)
   {
      sprintf (msg, "%1d additional color allocated.",
            maxColors-saved_max_colors);
      Msg (msg);
   }
   return (BitmapSuccess);
}

#define GETVALUE(val,name) ScanValue("%d", &(val), name, "xpixmap")

void ReadXPmObj (FP, Inbuf, ObjPtr)
   FILE			* FP;
   char			* Inbuf;
   struct ObjRec	* * ObjPtr;
{
   struct XPmRec	* xpm_ptr;
   char			color_s[20], * s, inbuf[MAXSTRING], * c_ptr, * line;
   int			ltx, lty, rbx, rby, i, j, k, image_w, image_h;
   int			ncolors, * pixels, len, index, fill, color_index;
   int			* red=NULL, * green=NULL, * blue=NULL;
   int			unrecognized_color = FALSE, rotation, chars_per_pixel;
   int			first_pixel_is_bg, first_pixel_maybe_bg;
   int			new_alloc, id=0, rotate=ROTATE0, flip=0, locked=FALSE;
   char			* xpm_data, * color_char, * * color_str;
   Pixmap		pixmap;
   char			msg[MAXSTRING];
   XImage		* image;

   *ObjPtr = NULL;

   s = FindChar ('(', Inbuf);
   s = ParseStr (s, ',', color_s);

   InitScan (s, "\t\n, ");

   rotation = 0;
   chars_per_pixel = 1;
   first_pixel_maybe_bg = TRUE;
   first_pixel_is_bg = TRUE;
   if (fileVersion <= 9)
   {
      sprintf (msg, "Invalid xpixmap version (%1d).", fileVersion);
      if (PRTGIF)
         fprintf (stderr, "%s\n", msg);
      else
         Msg (msg);
      return;
   }
   else if (fileVersion <= 13)
   {
      if (GETVALUE (ltx,      "ltx") == INVALID ||
          GETVALUE (lty,      "lty") == INVALID ||
          GETVALUE (rbx,      "rbx") == INVALID ||
          GETVALUE (rby,      "rby") == INVALID ||
          GETVALUE (fill,     "fill") == INVALID ||
          GETVALUE (ncolors,  "ncolors") == INVALID ||
          GETVALUE (id,       "id") == INVALID)
      {
         return;
      }
      if (id >= objId) objId = id+1;
   }
   else if (fileVersion <= 14)
   {
      if (GETVALUE (ltx,      "ltx") == INVALID ||
          GETVALUE (lty,      "lty") == INVALID ||
          GETVALUE (rbx,      "rbx") == INVALID ||
          GETVALUE (rby,      "rby") == INVALID ||
          GETVALUE (fill,     "fill") == INVALID ||
          GETVALUE (ncolors,  "ncolors") == INVALID ||
          GETVALUE (id,       "id") == INVALID ||
          GETVALUE (rotation, "rotation") == INVALID)
      {
         return;
      }
      if (id >= objId) objId = id+1;
   }
   else if (fileVersion <= 22)
   {
      if (GETVALUE (ltx,               "ltx") == INVALID ||
          GETVALUE (lty,               "lty") == INVALID ||
          GETVALUE (rbx,               "rbx") == INVALID ||
          GETVALUE (rby,               "rby") == INVALID ||
          GETVALUE (fill,              "fill") == INVALID ||
          GETVALUE (ncolors,           "ncolors") == INVALID ||
          GETVALUE (chars_per_pixel,   "chars_per_pixel") == INVALID ||
          GETVALUE (first_pixel_is_bg, "first_pixel_is_bg") == INVALID ||
          GETVALUE (id,                "id") == INVALID ||
          GETVALUE (rotation,          "rotation") == INVALID)
      {
         return;
      }
      if (id >= objId) objId = id+1;
      first_pixel_maybe_bg = FALSE;
   }
   else if (fileVersion <= 25)
   {
      if (GETVALUE (ltx,               "ltx") == INVALID ||
          GETVALUE (lty,               "lty") == INVALID ||
          GETVALUE (rbx,               "rbx") == INVALID ||
          GETVALUE (rby,               "rby") == INVALID ||
          GETVALUE (fill,              "fill") == INVALID ||
          GETVALUE (ncolors,           "ncolors") == INVALID ||
          GETVALUE (chars_per_pixel,   "chars_per_pixel") == INVALID ||
          GETVALUE (first_pixel_is_bg, "first_pixel_is_bg") == INVALID ||
          GETVALUE (id,                "id") == INVALID ||
          GETVALUE (rotation,          "rotation") == INVALID ||
          GETVALUE (image_w,           "image_w") == INVALID ||
          GETVALUE (image_h,           "image_h") == INVALID ||
          GETVALUE (rotate,            "rotate") == INVALID ||
          GETVALUE (flip,              "flip") == INVALID)
      {
         return;
      }
      if (id >= objId) objId = id+1;
      first_pixel_maybe_bg = FALSE;
   }
   else
   {
      if (GETVALUE (ltx,               "ltx") == INVALID ||
          GETVALUE (lty,               "lty") == INVALID ||
          GETVALUE (rbx,               "rbx") == INVALID ||
          GETVALUE (rby,               "rby") == INVALID ||
          GETVALUE (fill,              "fill") == INVALID ||
          GETVALUE (ncolors,           "ncolors") == INVALID ||
          GETVALUE (chars_per_pixel,   "chars_per_pixel") == INVALID ||
          GETVALUE (first_pixel_is_bg, "first_pixel_is_bg") == INVALID ||
          GETVALUE (id,                "id") == INVALID ||
          GETVALUE (rotation,          "rotation") == INVALID ||
          GETVALUE (image_w,           "image_w") == INVALID ||
          GETVALUE (image_h,           "image_h") == INVALID ||
          GETVALUE (rotate,            "rotate") == INVALID ||
          GETVALUE (flip,              "flip") == INVALID ||
          GETVALUE (locked,            "locked") == INVALID)
      {
         return;
      }
      if (id >= objId) objId = id+1;
      first_pixel_maybe_bg = FALSE;
   }

   if (fileVersion <= 22)
   {
      image_w = rbx-ltx;
      image_h = rby-lty;
      rotate = ROTATE0;
      flip = 0;
   }
   fill = UpgradePenFill (fill);

   *ObjPtr = (struct ObjRec *) calloc (1, sizeof(struct ObjRec));
   xpm_ptr = (struct XPmRec *) calloc (1, sizeof(struct XPmRec));

   color_index = QuickFindColorIndex (color_s, &new_alloc);

   (*ObjPtr)->color = color_index;
   (*ObjPtr)->dirty = FALSE;
   (*ObjPtr)->id = id;
   (*ObjPtr)->rotation = rotation;
   (*ObjPtr)->locked = locked;
   (*ObjPtr)->type = OBJ_XPM;
   (*ObjPtr)->obbox.ltx = (*ObjPtr)->bbox.ltx = (*ObjPtr)->x = ltx;
   (*ObjPtr)->obbox.lty = (*ObjPtr)->bbox.lty = (*ObjPtr)->y = lty;
   (*ObjPtr)->obbox.rbx = (*ObjPtr)->bbox.rbx = rbx;
   (*ObjPtr)->obbox.rby = (*ObjPtr)->bbox.rby = rby;
   (*ObjPtr)->detail.xpm = xpm_ptr;

   color_char = (char *) calloc (ncolors*chars_per_pixel, sizeof(char));
   color_str = (char * *) calloc (ncolors, sizeof(char *));
   pixels = (int *) calloc (ncolors, sizeof(int));
   if (PRTGIF && fileVersion >= 25)
   {
      red = (int *) calloc (ncolors, sizeof(int));
      green = (int *) calloc (ncolors, sizeof(int));
      blue = (int *) calloc (ncolors, sizeof(int));
   }

   xpm_ptr->pixmap = None;
   xpm_ptr->image = NULL;
   xpm_ptr->cached_pixmap = None;
   xpm_ptr->cached_zoom = 0;
   xpm_ptr->data = NULL;
   xpm_ptr->fill = fill;
   xpm_ptr->rotate = rotate;
   xpm_ptr->flip = flip;
   xpm_ptr->cached_rotate = INVALID;
   xpm_ptr->cached_flip = 0;
   xpm_ptr->cached_w = xpm_ptr->cached_h = 0;
   xpm_ptr->image_w = image_w;
   xpm_ptr->image_h = image_h;

   xpm_ptr->ncolors = ncolors;
   xpm_ptr->chars_per_pixel = chars_per_pixel;
   xpm_ptr->first_pixel_is_bg = first_pixel_is_bg;
   xpm_ptr->color_char = color_char;
   xpm_ptr->color_str = color_str;
   xpm_ptr->pixels = pixels;
   xpm_ptr->red = red;
   xpm_ptr->green = green;
   xpm_ptr->blue = blue;

   for (i = 0; i < ncolors; i++)
   {
      fgets (inbuf, MAXSTRING, FP);
      scanLineNum++;
      c_ptr = FindChar ('"', inbuf);
      for (j = 0; j < chars_per_pixel; j++)
         color_char[i*chars_per_pixel+j] = c_ptr[j];
      c_ptr = FindChar ('"', c_ptr);
      c_ptr = FindChar ('"', c_ptr);
      c_ptr = ParseStr (c_ptr, '"', color_s);
      if (!PRTGIF)
      {
         if (i == 0 && (first_pixel_is_bg || first_pixel_maybe_bg &&
               (color_char[i]=='`' || color_char[i]==' ')))
         {
            strcpy (color_s, myBgColorStr);
            pixels[i] = myBgPixel;
            xpm_ptr->first_pixel_is_bg = first_pixel_is_bg = TRUE;
         }
         else if ((index = QuickFindColorIndex (color_s,&new_alloc)) == INVALID)
         {
            sprintf (msg, "Can not allocate color '%s', use '%s' instead.",
                  color_s, colorMenuItems[colorIndex]);
            if (PRTGIF)
               fprintf (stderr, "%s\n", msg);
            else
               Msg (msg);
            strcpy (color_s, colorMenuItems[colorIndex]);
            pixels[i] = colorPixels[colorIndex];
         }
         else
            pixels[i] = colorPixels[index];

         len = strlen (color_s);
         color_str[i] = (char *) calloc (len+1, sizeof(char));
         strcpy (color_str[i], color_s);
      }
      else if (fileVersion >= 25)
      {  /* has RGB information for PRTGIF */
         InitScan (c_ptr, "\t\n, ");

         if (GETVALUE (red[i],   "red") == INVALID ||
             GETVALUE (green[i], "green") == INVALID ||
             GETVALUE (blue[i],  "blue") == INVALID)
         {
            return;
         }
      }
   }

   if (PRTGIF)
   {
      xpm_data = (char *) calloc (image_w*image_h*chars_per_pixel,sizeof(char));
      xpm_ptr->data = xpm_data;

      line = (char *) calloc (image_w*chars_per_pixel+10, sizeof(char));
      for (i = 0; i < image_h; i++, xpm_data += image_w*chars_per_pixel)
      {
         fgets (line, image_w*chars_per_pixel+10, FP);
         scanLineNum++;
         c_ptr = &line[4];
         strncpy (xpm_data, c_ptr, image_w*chars_per_pixel);
      }
   }
   else
   {
      pixmap = XCreatePixmap (mainDisplay, dummyPixmap, image_w, image_h,
            mainDepth);
      XFillRectangle (mainDisplay, pixmap, xpmGC, 0, 0, image_w, image_h);
      image = XGetImage (mainDisplay, pixmap, 0, 0, image_w, image_h, AllPlanes,
            ZPixmap);
      if (image == NULL)
      {
         Msg ("XGetImage() failed!  May have run out of memory!");
         XFreePixmap (mainDisplay, pixmap);
         return;
      }

      BuildXPmBuckets (ncolors, NULL, chars_per_pixel, color_char);

      line = (char *) calloc (image_w*chars_per_pixel+10, sizeof(char));
      for (i = 0; i < image_h; i++)
      {
         fgets (line, image_w*chars_per_pixel+10, FP);
         scanLineNum++;
         c_ptr = &line[4];
         for (j = 0; j < image_w; j++, c_ptr+=chars_per_pixel)
         {
            k = XPmLookUp (INVALID, chars_per_pixel, c_ptr);
            if (k == INVALID)
            {
               XPutPixel (image, j, i, pixels[0]);
               unrecognized_color = TRUE;
            }
            else
               XPutPixel (image, j, i, pixels[k]);
         }
      }
      cfree (line);
      XPutImage (mainDisplay,pixmap,xpmGC,image,0,0,0,0,image_w,image_h);
      xpm_ptr->pixmap = pixmap;
      xpm_ptr->image = image;

      if (unrecognized_color)
      {
         sprintf (msg, "Unrecognized colors shown as '%s'", color_str[0]);
         Msg (msg);
      }
   }
}

void FreeXPmObj (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   register int		i, ncolors;
   struct XPmRec	* xpm_ptr = ObjPtr->detail.xpm;

   if (xpm_ptr->pixmap != None)
   {
      ncolors = xpm_ptr->ncolors;
      for (i = 0; i < ncolors; i++) cfree (xpm_ptr->color_str[i]);
      cfree (xpm_ptr->color_char);
      cfree (xpm_ptr->color_str);
      cfree (xpm_ptr->pixels);
      if (xpm_ptr->red != NULL) cfree (xpm_ptr->red);
      if (xpm_ptr->green != NULL) cfree (xpm_ptr->green);
      if (xpm_ptr->blue != NULL) cfree (xpm_ptr->blue);
      XFreePixmap (mainDisplay, xpm_ptr->pixmap);
   }
   if (xpm_ptr->image != NULL) XDestroyImage (xpm_ptr->image);
   if (xpm_ptr->cached_pixmap != None)
      XFreePixmap (mainDisplay, xpm_ptr->cached_pixmap);
   xpm_ptr->pixmap = None;
   xpm_ptr->cached_pixmap = None;
   xpm_ptr->cached_zoom = 0;
   if (xpm_ptr->data != NULL) cfree (xpm_ptr->data);
   cfree (xpm_ptr);
   cfree (ObjPtr);
}
