/*
 * Author:      William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990-1994, William Cheng.
 *
 * Permission limited to the use, copy, modify, and distribute this software
 * and its documentation for any purpose is hereby granted by the Author without
 * fee, provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the Author not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.  All other
 * rights (including the right to sell "tgif" and the right to sell derivative
 * works of tgif) are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /export/bourbon/multimedia/william/X11/TGIF2/RCS/choice.c,v 2.78 1995/01/05 06:29:20 william Exp $";
#endif

#include <stdio.h>
#include <X11/Xlib.h>
#include "const.h"
#include "types.h"

#include "align.e"
#include "arc.e"
#include "box.e"
#ifndef _NO_EXTERN
#include "choice.e"
#endif
#include "color.e"
#include "cursor.e"
#include "dialog.e"
#include "drawing.e"
#include "edit.e"
#include "file.e"
#include "font.e"
#include "grid.e"
#include "mainloop.e"
#include "mainmenu.e"
#include "mark.e"
#include "msg.e"
#include "navigate.e"
#include "oval.e"
#include "page.e"
#include "pattern.e"
#include "poly.e"
#include "polygon.e"
#include "raster.e"
#include "rcbox.e"
#include "select.e"
#include "setup.e"
#include "special.e"
#include "stretch.e"
#include "text.e"

#define MODE_ROW 0
#define EDIT_ROW 0
#define ZOOM_ROW 0
#define RADIUS_ROW 0
#define PAGE_ROW 0
#define HORI_ALIGN_ROW 0
#define FONT_ROW 0
#define VSPACE_ROW 0
#define DASH_ROW 0
#define LINE_TYPE_ROW 0
#define FILL_ROW 0
#define COLOR_ROW 0

#define FILE_ROW 1
#define PRINT_ROW 1
#define MOVE_MODE_ROW 1
#define ROTATE_ROW 1
#define PAGELAYOUT_ROW 1
#define VERT_ALIGN_ROW 1
#define TEXT_SIZE_ROW 1
#define JUST_ROW 1
#define LINE_STYLE_ROW 1
#define LINE_WIDTH_ROW 1
#define PEN_ROW 1
#define SPECIAL_ROW 1

#define MODE_COL 0
#define EDIT_COL 1
#define ZOOM_COL 2
#define RADIUS_COL 3
#define PAGE_COL 4
#define HORI_ALIGN_COL 5
#define FONT_COL 6
#define VSPACE_COL 7
#define DASH_COL 8
#define LINE_TYPE_COL 9
#define FILL_COL 10
#define COLOR_COL 11

#define FILE_COL 0
#define PRINT_COL 1
#define MOVE_MODE_COL 2
#define ROTATE_COL 3
#define PAGELAYOUT_COL 4
#define VERT_ALIGN_COL 5
#define TEXT_SIZE_COL 6
#define JUST_COL 7
#define LINE_STYLE_COL 8
#define LINE_WIDTH_COL 9
#define PEN_COL 10
#define SPECIAL_COL 11

#define CHOICE_SHIFT 4

#define CHOICE_MODE ((MODE_COL<<CHOICE_SHIFT)|MODE_ROW)
#define CHOICE_COLOR ((COLOR_COL<<CHOICE_SHIFT)|COLOR_ROW)
#define CHOICE_FILE ((FILE_COL<<CHOICE_SHIFT)|FILE_ROW)
#define CHOICE_EDIT ((EDIT_COL<<CHOICE_SHIFT)|EDIT_ROW)
#define CHOICE_PRINT ((PRINT_COL<<CHOICE_SHIFT)|PRINT_ROW)
#define CHOICE_SPECIAL ((SPECIAL_COL<<CHOICE_SHIFT)|SPECIAL_ROW)
#define CHOICE_HORI_ALIGN ((HORI_ALIGN_COL<<CHOICE_SHIFT)|HORI_ALIGN_ROW)
#define CHOICE_VERT_ALIGN ((VERT_ALIGN_COL<<CHOICE_SHIFT)|VERT_ALIGN_ROW)
#define CHOICE_ROTATE ((ROTATE_COL<<CHOICE_SHIFT)|ROTATE_ROW)
#define CHOICE_MOVE_MODE ((MOVE_MODE_COL<<CHOICE_SHIFT)|MOVE_MODE_ROW)
#define CHOICE_RADIUS ((RADIUS_COL<<CHOICE_SHIFT)|RADIUS_ROW)
#define CHOICE_ZOOM ((ZOOM_COL<<CHOICE_SHIFT)|ZOOM_ROW)
#define CHOICE_FONT ((FONT_COL<<CHOICE_SHIFT)|FONT_ROW)
#define CHOICE_JUST ((JUST_COL<<CHOICE_SHIFT)|JUST_ROW)
#define CHOICE_TEXT_SIZE ((TEXT_SIZE_COL<<CHOICE_SHIFT)|TEXT_SIZE_ROW)
#define CHOICE_VSPACE ((VSPACE_COL<<CHOICE_SHIFT)|VSPACE_ROW)
#define CHOICE_LINE_WIDTH ((LINE_WIDTH_COL<<CHOICE_SHIFT)|LINE_WIDTH_ROW)
#define CHOICE_LINE_STYLE ((LINE_STYLE_COL<<CHOICE_SHIFT)|LINE_STYLE_ROW)
#define CHOICE_LINE_TYPE ((LINE_TYPE_COL<<CHOICE_SHIFT)|LINE_TYPE_ROW)
#define CHOICE_DASH ((DASH_COL<<CHOICE_SHIFT)|DASH_ROW)
#define CHOICE_FILL ((FILL_COL<<CHOICE_SHIFT)|FILL_ROW)
#define CHOICE_PEN ((PEN_COL<<CHOICE_SHIFT)|PEN_ROW)
#define CHOICE_PAGE ((PAGE_COL<<CHOICE_SHIFT)|PAGE_ROW)
#define CHOICE_PAGELAYOUT ((PAGELAYOUT_COL<<CHOICE_SHIFT)|PAGELAYOUT_ROW)

int	curChoice = NOTHING;

static GC	choiceGC;
static Pixmap	choiceBackingPixmap;

void InitChoice ()
{
   XGCValues	values;

   values.foreground = xorOne;
   values.background = xorZero;
   values.fill_style = FillSolid;
   values.font = defaultFontPtr->fid;
   choiceGC = XCreateGC (mainDisplay, choiceWindow,
         GCForeground | GCBackground | GCFillStyle | GCFont, &values);

   choiceBackingPixmap = XCreatePixmap (mainDisplay, choiceWindow,
         choiceWindowW, choiceWindowH, mainDepth);
}

void CleanUpChoices ()
{
   XFreeGC (mainDisplay, choiceGC);
   XFreePixmap (mainDisplay, choiceBackingPixmap);
}

void ShowWhereToPrint ()
{
   if (colorDump)
      XSetForeground (mainDisplay, rasterGC, colorPixels[colorIndex]);
   XSetStipple (mainDisplay, rasterGC, whereToPrintPixmap[whereToPrint]);
   XFillRectangle (mainDisplay, choiceWindow, rasterGC, PRINT_COL*choiceImageW,
         PRINT_ROW*choiceImageH, choiceImageW, choiceImageH);
   if (colorDump) XSetForeground (mainDisplay, rasterGC, myFgPixel);
}

void ShowMode ()
{
   XSetStipple (mainDisplay, rasterGC, choicePixmap[curChoice]);
   XFillRectangle (mainDisplay, choiceWindow, rasterGC, MODE_COL*choiceImageW,
         MODE_ROW*choiceImageH, choiceImageW, choiceImageH);
}

void ShowColor (PropagateWhereToPrint)
   int	PropagateWhereToPrint;
{
   XGCValues	values;

   if (colorDisplay)
   {
      values.foreground = colorPixels[colorIndex];
      values.function = GXcopy;
      values.fill_style = FillOpaqueStippled;
      values.stipple = patPixmap[1];
      XChangeGC (mainDisplay, patGC,
            GCForeground | GCFunction | GCFillStyle | GCStipple, &values);

      XFillRectangle (mainDisplay, choiceWindow, patGC,
            COLOR_COL*choiceImageW, COLOR_ROW*choiceImageH,
            choiceImageW, choiceImageH);
   }
   if (PropagateWhereToPrint) ShowWhereToPrint ();
}

void ShowHoriAlign ()
{
   XSetStipple (mainDisplay, rasterGC, alignHoriPixmap[horiAlign]);
   XFillRectangle (mainDisplay, choiceWindow, rasterGC,
         HORI_ALIGN_COL*choiceImageW, HORI_ALIGN_ROW*choiceImageH,
         choiceImageW, choiceImageH);
}

void ShowVertAlign ()
{
   XSetStipple (mainDisplay, rasterGC, alignVertPixmap[vertAlign]);
   XFillRectangle (mainDisplay, choiceWindow, rasterGC,
         VERT_ALIGN_COL*choiceImageW, VERT_ALIGN_ROW*choiceImageH,
         choiceImageW, choiceImageH);
}

void ShowJust ()
{
   XSetStipple (mainDisplay, rasterGC, justPixmap[textJust]);
   XFillRectangle (mainDisplay, choiceWindow, rasterGC,
         JUST_COL*choiceImageW, JUST_ROW*choiceImageH,
         choiceImageW, choiceImageH);
}

void ShowCurFont ()
{
   int		x, y, w, h;
   XRectangle	recs[1];
   XGCValues	values;

   recs[0].x = FONT_COL*choiceImageW;
   recs[0].y = FONT_ROW*choiceImageH;
   recs[0].width = choiceImageW;
   recs[0].height = choiceImageH;

   w = XTextWidth (canvasFontPtr, "W", 1);
   h = canvasFontHeight;

   x = (w > choiceImageW) ?
         FONT_COL*choiceImageW - (w-choiceImageW)/2 :
         FONT_COL*choiceImageW + (choiceImageW-w)/2;
   y = (h > choiceImageH) ?
         FONT_ROW*choiceImageH + (canvasFontAsc-(h-choiceImageH)/2) :
         FONT_ROW*choiceImageH + (canvasFontAsc+(choiceImageH-h)/2);

   XClearArea (mainDisplay, choiceWindow, FONT_COL*choiceImageW,
         FONT_ROW*choiceImageH, choiceImageW, choiceImageH, FALSE);

   XSetFont (mainDisplay, choiceGC, canvasFontPtr->fid);
   XSetClipRectangles (mainDisplay, choiceGC, 0, 0, recs, 1, YXBanded);

   values.foreground = myFgPixel;
   values.background = myBgPixel;
   XChangeGC (mainDisplay, choiceGC, GCForeground | GCBackground, &values);

   XDrawString (mainDisplay, choiceWindow, choiceGC, x, y, "W", 1);

   values.foreground = xorOne;
   values.background = xorZero;
   XChangeGC (mainDisplay, choiceGC, GCForeground | GCBackground, &values);

   recs[0].x = 0;
   recs[0].y = 0;
   recs[0].width = choiceWindowW;
   recs[0].height = choiceWindowH;
   XSetClipRectangles (mainDisplay, choiceGC, 0, 0, recs, 1, YXBanded);
}

void ShowTextVSpace ()
{
   int		len, x, y;
   char		s[80];
   XGCValues	values;

   XSetStipple (mainDisplay, rasterGC, vspacePixmap);
   XFillRectangle (mainDisplay, choiceWindow, rasterGC, VSPACE_COL*choiceImageW,
         VSPACE_ROW*choiceImageH, choiceImageW, choiceImageH);
   sprintf (s, "%1d", textVSpace);
   len = strlen (s);
   x = (int)((VSPACE_COL+0.5)*choiceImageW-2);
   y = VSPACE_ROW*choiceImageH+((choiceImageH-rulerFontAsc)>>1)+rulerFontAsc;

   values.foreground = myFgPixel;
   values.background = myBgPixel;
   values.font = rulerFontPtr->fid;
   XChangeGC (mainDisplay, choiceGC, GCForeground | GCBackground | GCFont,
         &values);

   XDrawString (mainDisplay, choiceWindow, choiceGC, x, y, s, len);

   values.foreground = xorOne;
   values.background = xorZero;
   XChangeGC (mainDisplay, choiceGC, GCForeground | GCBackground, &values);
}

void ShowZoom ()
{
   int		len, x, y, w, x_w, one_w;
   char		s[80];
   XGCValues	values;

   XClearArea (mainDisplay, choiceWindow, ZOOM_COL*choiceImageW,
      ZOOM_ROW*choiceImageH, choiceImageW, choiceImageH, FALSE);

   sprintf (s, "x%1d", 1<<zoomScale);
   len = strlen (s);

   w = XTextWidth (rulerFontPtr, s, len)+1;
   x_w = XTextWidth (rulerFontPtr, "x", 1)+1;
   one_w = XTextWidth (rulerFontPtr, "1", 1);
   x = ZOOM_COL*choiceImageW+((choiceImageW-w)>>1);
   y = ZOOM_ROW*choiceImageH+((choiceImageH-rulerFontAsc)>>1)+rulerFontAsc;

   values.foreground = myFgPixel;
   values.background = myBgPixel;
   values.font = rulerFontPtr->fid;
   XChangeGC (mainDisplay, choiceGC, GCForeground | GCBackground | GCFont,
         &values);

   XDrawString (mainDisplay, choiceWindow, choiceGC, x, y, "x", 1);

   x += x_w;
   sprintf (s, "%1d", 1<<zoomScale);
   len--;
   if (zoomedIn || zoomScale==0)
      XDrawString (mainDisplay, choiceWindow, choiceGC, x, y, s, len);
   else
   {
      y = ZOOM_ROW*choiceImageH + (choiceImageH>>1);
      XDrawString (mainDisplay, choiceWindow, choiceGC, x+((w-x_w-one_w)>>1),
            y-2, "1", 1);
      XDrawLine (mainDisplay, choiceWindow, choiceGC, x, y, x+(w-x_w-1), y);
      XDrawString (mainDisplay, choiceWindow, choiceGC, x, y+rulerFontAsc,
            s, len);
   }

   values.foreground = xorOne;
   values.background = xorZero;
   XChangeGC (mainDisplay, choiceGC, GCForeground | GCBackground, &values);
}

void ShowTextSize ()
{
   int		len, x, y, w;
   char		s[80];
   XGCValues	values;

   XClearArea (mainDisplay, choiceWindow, TEXT_SIZE_COL*choiceImageW,
         TEXT_SIZE_ROW*choiceImageH, choiceImageW, choiceImageH, FALSE);
   sprintf (s, "%1d", curSize);
   len = strlen (s);
   w = XTextWidth (defaultFontPtr, s, len);
   x = TEXT_SIZE_COL*choiceImageW + ((choiceImageW-w)>>1);
   y = TEXT_SIZE_ROW*choiceImageH + ((choiceImageH-defaultFontAsc)>>1) + 
         defaultFontAsc;

   values.foreground = myFgPixel;
   values.background = myBgPixel;
   values.font = defaultFontPtr->fid;
   XChangeGC (mainDisplay, choiceGC, GCForeground | GCBackground | GCFont,
         &values);

   XDrawString (mainDisplay, choiceWindow, choiceGC, x, y, s, len);

   values.foreground = xorOne;
   values.background = xorZero;
   XChangeGC (mainDisplay, choiceGC, GCForeground | GCBackground, &values);
}

void ShowRotate ()
{
   XSetStipple (mainDisplay, rasterGC, rotatePixmap[curRotate]);
   XFillRectangle (mainDisplay, choiceWindow, rasterGC, ROTATE_COL*choiceImageW,
         ROTATE_ROW*choiceImageH, choiceImageW, choiceImageH);
}

void ShowSpecial ()
{
   XSetStipple (mainDisplay, rasterGC, specialPixmap);
   XFillRectangle (mainDisplay, choiceWindow, rasterGC,
         SPECIAL_COL*choiceImageW, SPECIAL_ROW*choiceImageH,
         choiceImageW, choiceImageH);
}

void ShowLineWidth ()
{
   XSetStipple (mainDisplay, rasterGC, shortLineWidthPixmap[lineWidth]);
   XFillRectangle (mainDisplay, choiceWindow, rasterGC,
         LINE_WIDTH_COL*choiceImageW, LINE_WIDTH_ROW*choiceImageH,
         choiceImageW, choiceImageH);
}

void ShowLineStyle ()
{
   XSetStipple (mainDisplay, rasterGC, shortLineStylePixmap[lineStyle]);
   XFillRectangle (mainDisplay, choiceWindow, rasterGC,
         LINE_STYLE_COL*choiceImageW, LINE_STYLE_ROW*choiceImageH,
         choiceImageW, choiceImageH);
}

void ShowLineType ()
{
   XSetStipple (mainDisplay, rasterGC, shortLineTypePixmap[curSpline]);
   XFillRectangle (mainDisplay, choiceWindow, rasterGC,
         LINE_TYPE_COL*choiceImageW, LINE_TYPE_ROW*choiceImageH,
         choiceImageW, choiceImageH);
}

void ShowDash ()
{
   XSetStipple (mainDisplay, rasterGC, shortDashPixmap[curDash]);
   XFillRectangle (mainDisplay, choiceWindow, rasterGC, DASH_COL*choiceImageW,
         DASH_ROW*choiceImageH, choiceImageW, choiceImageH);
}

void ShowFile ()
{
   switch (pageStyle)
   {
      case PORTRAIT: XSetStipple(mainDisplay,rasterGC,filePixmap); break;
      case LANDSCAPE: XSetStipple(mainDisplay,rasterGC,landscapePixmap); break;
   }
   XFillRectangle (mainDisplay, choiceWindow, rasterGC, FILE_COL*choiceImageW,
         FILE_ROW*choiceImageH, choiceImageW, choiceImageH);
}

void ShowEdit ()
{
   XSetStipple (mainDisplay, rasterGC, editPixmap);
   XFillRectangle (mainDisplay, choiceWindow, rasterGC, EDIT_COL*choiceImageW,
         EDIT_ROW*choiceImageH, choiceImageW, choiceImageH);
}

void ShowRCBRadius ()
{
   int		len, x, y;
   char		s[80];
   XGCValues	values;

   XSetStipple (mainDisplay, rasterGC, rcbRadiusPixmap);
   XFillRectangle (mainDisplay, choiceWindow, rasterGC, RADIUS_COL*choiceImageW,
         RADIUS_ROW*choiceImageH, choiceImageW, choiceImageH);
   sprintf (s, "%1d", rcbRadius);
   len = strlen (s);
   x = (int)((RADIUS_COL+0.5)*choiceImageW-2);
   y = (((RADIUS_ROW+1)*choiceImageH-rulerFontAsc)>>1)+rulerFontAsc;

   values.foreground = myFgPixel;
   values.background = myBgPixel;
   values.font = rulerFontPtr->fid;
   XChangeGC (mainDisplay, choiceGC, GCForeground | GCBackground | GCFont,
         &values);

   XDrawString (mainDisplay, choiceWindow, choiceGC, x, y, s, len);

   values.foreground = xorOne;
   values.background = xorZero;
   XChangeGC (mainDisplay, choiceGC, GCForeground | GCBackground, &values);
}

void ShowMoveMode ()
{
   XSetStipple (mainDisplay, rasterGC, moveModePixmap[moveMode]);
   XFillRectangle (mainDisplay, choiceWindow, rasterGC,
         MOVE_MODE_COL*choiceImageW, MOVE_MODE_ROW*choiceImageH,
         choiceImageW, choiceImageH);
}

void ShowFill ()
{
   XGCValues	values;

   values.foreground = myFgPixel;
   values.background = myBgPixel;
   values.function = GXcopy;
   values.fill_style = FillOpaqueStippled;
   values.stipple = patPixmap[objFill];
   XChangeGC (mainDisplay, patGC,
         GCForeground | GCBackground | GCFunction | GCFillStyle | GCStipple,
         &values);

   XFillRectangle (mainDisplay, choiceWindow, patGC, FILL_COL*choiceImageW,
         FILL_ROW*choiceImageH, choiceImageW, choiceImageH);
}

void ShowPen ()
{
   XGCValues	values;

   values.foreground = myFgPixel;
   values.background = myBgPixel;
   values.function = GXcopy;
   values.fill_style = FillOpaqueStippled;
   values.stipple = patPixmap[penPat];
   XChangeGC (mainDisplay, patGC,
         GCForeground | GCBackground | GCFunction | GCFillStyle | GCStipple,
         &values);

   XFillRectangle (mainDisplay, choiceWindow, patGC, PEN_COL*choiceImageW,
         PEN_ROW*choiceImageH, choiceImageW, choiceImageH);
   if (penPat != NONEPAT)
      XClearArea (mainDisplay, choiceWindow,
            PEN_COL*choiceImageW+(choiceImageW>>2),
            PEN_ROW*choiceImageH+(choiceImageH>>2),
            (choiceImageW>>1), (choiceImageH>>1), FALSE);
}

void ShowPage ()
{
   int                len, x, y, w;
   char               s[80];
   XGCValues  values;

   XClearArea (mainDisplay, choiceWindow, PAGE_COL*choiceImageW,
      PAGE_ROW*choiceImageH, choiceImageW, choiceImageH, FALSE);

   switch (pageLayoutMode)
   {
      case PAGE_STACK: sprintf (s, "%1d/%1d", curPageNum, lastPageNum); break;
      case PAGE_TILE: sprintf (s, "%1dx%1d", paperCol, paperRow); break;
   }
   len = strlen (s);
   w = XTextWidth (rulerFontPtr, s, len);
   x = PAGE_COL*choiceImageW+((choiceImageW-w)>>1);
   y = PAGE_ROW*choiceImageH+((choiceImageH-rulerFontAsc)>>1) +
         rulerFontAsc;

   values.foreground = myFgPixel;
   values.background = myBgPixel;
   values.font = rulerFontPtr->fid;
   XChangeGC (mainDisplay, choiceGC, GCForeground | GCBackground | GCFont,
         &values);

   XDrawString (mainDisplay, choiceWindow, choiceGC, x, y, s, len);

   values.foreground = xorOne;
   values.background = xorZero;
   XChangeGC (mainDisplay, choiceGC, GCForeground | GCBackground,
         &values);
}

void ShowPageLayout ()
{
   XSetStipple (mainDisplay, rasterGC, pageLayoutPixmap[pageLayoutMode]);
   XFillRectangle (mainDisplay, choiceWindow, rasterGC,
         PAGELAYOUT_COL*choiceImageW, PAGELAYOUT_ROW*choiceImageH,
         choiceImageW, choiceImageH);
}

static struct MouseStatusStrRec choiceMouseStatus[] = {
   { "select/move/resize objects", "Main Menu", "Mode Menu" },
   { "enter text", "Main Menu", "Mode Menu" },
   { "draw rectangles", "Main Menu", "Mode Menu" },
   { "draw ovals", "Main Menu", "Mode Menu" },
   { "draw poly/open splines", "Main Menu", "Mode Menu" },
   { "draw polygon/closed splines", "Main Menu", "Mode Menu" },
   { "draw arcs", "Main Menu", "Mode Menu" },
   { "draw rcboxes", "Main Menu", "Mode Menu" },
   { "freehand poly/open splines", "Main Menu", "Mode Menu" },
   { "select/move vertices", "Main Menu", "Mode Menu" }
};

void SetCurChoice (NewChoice)
   int	NewChoice;
{
   if (curChoice == NewChoice) return;

   switch (curChoice)
   {
      case NOTHING:
         if (topSel != NULL)
         {
            if (NewChoice == VERTEXMODE)
            {
               UnSelNonVertexObjs (TRUE); /* with highlight */
               UpdSelBBox ();
            }
            else
            {
               HighLightReverse ();
               RemoveAllSel ();
            }
         }
         break;
      case DRAWTEXT:
         CreateTextObj ();
         if (NewChoice == NOTHING && textDrawn) HighLightJustDrawnText ();
         textDrawn = FALSE;
         textCursorShown = FALSE;
         break;
      case DRAWBOX:
         if (NewChoice == NOTHING && boxDrawn) SelectTopObj ();
         boxDrawn = FALSE;
         break;
      case DRAWCIRCLE:
         if (NewChoice == NOTHING && ovalDrawn) SelectTopObj ();
         ovalDrawn = FALSE;
         break;
      case DRAWPOLY:
         if ((NewChoice==NOTHING || NewChoice==VERTEXMODE) && polyDrawn)
            SelectTopObj ();
         polyDrawn = FALSE;
         break;
      case DRAWPOLYGON:
         if ((NewChoice == NOTHING || NewChoice==VERTEXMODE) && polygonDrawn)
            SelectTopObj ();
         polygonDrawn = FALSE;
         break;
      case DRAWARC:
         if (NewChoice == NOTHING && arcDrawn) SelectTopObj ();
         arcDrawn = FALSE;
         break;
      case DRAWRCBOX:
         if (NewChoice == NOTHING && rcBoxDrawn) SelectTopObj ();
         rcBoxDrawn = FALSE;
         break;
      case FREEHAND:
         if ((NewChoice==NOTHING || NewChoice==VERTEXMODE) && polyDrawn)
            SelectTopObj ();
         polyDrawn = FALSE;
         break;
      case VERTEXMODE:
         if (NewChoice == NOTHING)
         {
            HighLightReverse ();
            JustRemoveAllVSel ();
            HighLightForward ();
         }
         else
         {
            HighLightReverse ();
            RemoveAllSel ();
         }
         break;
   }

   curChoice = NewChoice;

   ShowCursor ();
   textCursorShown = FALSE;

   ShowMode ();
   UpdateSubMenu (MENU_MODE);

   if (curChoice == NOTHING && inHyperSpace)
      SetMouseStatus ("(none)",
            choiceMouseStatus[curChoice].m, choiceMouseStatus[curChoice].r);
   else
      SetMouseStatus (choiceMouseStatus[curChoice].l,
            choiceMouseStatus[curChoice].m, choiceMouseStatus[curChoice].r);

   if (inHyperSpace && curChoice != NOTHING) ToggleHyperSpace ();
}

static int	prevChoice = NOTHING;

void PushCurChoice ()
   /* Kouichi Matsuda's modification */
{
   if (curChoice == NOTHING)
      SetCurChoice (prevChoice);
   else
   {
      prevChoice = curChoice;
      SetCurChoice (NOTHING);
   }
}

static
void TextVSpaceLoop (button_ev)
   XButtonEvent	* button_ev;
{
   int		saved_text_vspace = textVSpace, res=8, done=FALSE;
   int		orig_x=button_ev->x, orig_y=button_ev->y;
   int		saved_change=0;
   XEvent	ev;

   XGrabPointer (mainDisplay, choiceWindow, False,
         PointerMotionMask | ButtonReleaseMask, GrabModeAsync,
         GrabModeAsync, None, handCursor, CurrentTime);

   while (!done)
   {
      XNextEvent (mainDisplay, &ev);

      if (ev.type == Expose || ev.type == VisibilityNotify)
         ExposeEventHandler (&ev, TRUE);
      else if (ev.type == ButtonRelease)
      {
         XUngrabPointer (mainDisplay, CurrentTime);
         done = TRUE;
      }
      else if (ev.type == MotionNotify)
      {
         int	dx = ev.xmotion.x - orig_x;
         int	dy = ev.xmotion.y - orig_y;
         int	change = (abs(dx)>abs(dy)) ? (int)(dx/res) : (int)(dy/res);

         if (change != saved_change)
         {
            textVSpace = saved_text_vspace+change;
            ShowTextVSpace ();
            saved_change = change;
         }
         while (XCheckMaskEvent (mainDisplay, PointerMotionMask, &ev)) ;
      }
   }
   if (topSel == NULL)
   {
      if (textCursorH+textVSpace <= 0)
      {
         Msg ("Text vertical spacing too small.  No change.");
         textVSpace = saved_text_vspace;
         ShowTextVSpace ();
      }
      else
      {
         int	text_cursor_shown = textCursorShown;
         int	text_obj_created, new_text_vspace = textVSpace;

         textVSpace = saved_text_vspace;
         text_obj_created = TieLooseEnds ();
         if (!text_obj_created && curChoice == DRAWTEXT && text_cursor_shown)
         {
            NewCurText ();
            RedrawCurText ();
         }
         else
            textCursorShown = FALSE;
         textVSpace = new_text_vspace;
         ShowTextVSpace ();
      }
   }
   else
   {
      ShowTextVSpace ();
      ChangeVSpace (textVSpace);
   }
}

static
void RCBRadiusLoop (button_ev)
   XButtonEvent	* button_ev;
{
   int		saved_rcb_radius = rcbRadius, res=8, done=FALSE;
   int		orig_x=button_ev->x, orig_y=button_ev->y;
   int		saved_change=0;
   XEvent	ev;

   XGrabPointer (mainDisplay, choiceWindow, False,
         PointerMotionMask | ButtonReleaseMask, GrabModeAsync,
         GrabModeAsync, None, handCursor, CurrentTime);

   while (!done)
   {
      XNextEvent (mainDisplay, &ev);

      if (ev.type == Expose || ev.type == VisibilityNotify)
         ExposeEventHandler (&ev, TRUE);
      else if (ev.type == ButtonRelease)
      {
         XUngrabPointer (mainDisplay, CurrentTime);
         done = TRUE;
      }
      else if (ev.type == MotionNotify)
      {
         int	dx = ev.xmotion.x - orig_x;
         int	dy = ev.xmotion.y - orig_y;
         int	change = (abs(dx)>abs(dy)) ? (int)(dx/res) : (int)(dy/res);

         if (change != saved_change)
         {
            rcbRadius = max(MIN_RCB_RADIUS, saved_rcb_radius+change);
            ShowRCBRadius ();
            saved_change = change;
         }
         while (XCheckMaskEvent (mainDisplay, PointerMotionMask, &ev)) ;
      }
   }
   if (topSel != NULL) ChangeAllSelRCBRadius (rcbRadius);
}

static
void ZoomLoop (button_ev)
   XButtonEvent	* button_ev;
{
   int		saved_zoomed_in=zoomedIn, saved_zoom_scale=zoomScale, res=8;
   int		new_zoomed_in, new_zoom_scale;
   int		initial_zoom, zoom, max_zoom, win_w, win_h;
   int		orig_x=button_ev->x, orig_y=button_ev->y;
   int		done=FALSE, saved_change=0;
   XEvent	ev;

   initial_zoom = (zoomedIn) ? (MAX_ZOOMED_IN-zoomScale) :
         (MAX_ZOOMED_IN+zoomScale);

   win_w = drawWinW;
   win_h = drawWinH;
   new_zoomed_in = zoomedIn;
   new_zoom_scale = zoomScale;
   while ((win_w>>1) >= paperWidth && (win_h>>1) >= paperHeight)
   {
      if (new_zoomed_in)
         new_zoom_scale++;
      else if (new_zoom_scale == 0)
      {
         new_zoomed_in = TRUE;
         new_zoom_scale++;
      }
      else
         new_zoom_scale--;
      win_w >>= 1;
      win_h >>= 1;
   }
   while (win_w < paperWidth || win_h < paperHeight)
   {
      if (!new_zoomed_in)
         new_zoom_scale++;
      else if (new_zoom_scale == 1)
      {
         new_zoomed_in = FALSE;
         new_zoom_scale--;
      }
      else
         new_zoom_scale--;
      win_w <<= 1;
      win_h <<= 1;
   }
   max_zoom = (new_zoomed_in) ? (MAX_ZOOMED_IN-new_zoom_scale) :
         (MAX_ZOOMED_IN+new_zoom_scale);

   XGrabPointer (mainDisplay, choiceWindow, False,
         PointerMotionMask | ButtonReleaseMask, GrabModeAsync,
         GrabModeAsync, None, handCursor, CurrentTime);

   while (!done)
   {
      XNextEvent (mainDisplay, &ev);

      if (ev.type == Expose || ev.type == VisibilityNotify)
         ExposeEventHandler (&ev, TRUE);
      else if (ev.type == ButtonRelease)
      {
         XUngrabPointer (mainDisplay, CurrentTime);
         done = TRUE;
      }
      else if (ev.type == MotionNotify)
      {
         int	dx = ev.xmotion.x - orig_x;
         int	dy = ev.xmotion.y - orig_y;
         int	change = (abs(dx)>abs(dy)) ? (int)(dx/res) : (int)(dy/res);

         if (change != saved_change)
         {
            zoom = initial_zoom+change;
            if (zoom < 0) zoom = 0;
            if (zoom > max_zoom) zoom = max_zoom;
            zoomedIn = (zoom < MAX_ZOOMED_IN);
            zoomScale = (zoomedIn) ? (MAX_ZOOMED_IN-zoom) :
                  (zoom-MAX_ZOOMED_IN);
            ShowZoom ();
            saved_change = change;
         }
         while (XCheckMaskEvent (mainDisplay, PointerMotionMask, &ev)) ;
      }
   }
   if (saved_zoomed_in==zoomedIn && saved_zoom_scale==zoomScale) return;
   new_zoomed_in = zoomedIn;
   new_zoom_scale = zoomScale;
   zoomedIn = saved_zoomed_in;
   zoomScale = saved_zoom_scale;
   PreciseZoom (new_zoomed_in, new_zoom_scale, FALSE);
}

static struct MouseStatusStrRec modeMouseStatus[] = {
   { "text mode", "Mode Menu", "vertex mode" },
   { "rectangle mode", "Mode Menu", "select mode" },
   { "oval mode", "Mode Menu", "text mode" },
   { "poly mode", "Mode Menu", "rectangle mode" },
   { "polygon mode", "Mode Menu", "oval mode" },
   { "arc mode", "Mode Menu", "poly mode" },
   { "rcbox mode", "Mode Menu", "polygon mode" },
   { "freehand mode", "Mode Menu", "arc mode" },
   { "vertex mode", "Mode Menu", "rcbox mode" },
   { "select mode", "Mode Menu", "freehand mode" }
};
static struct MouseStatusStrRec colorMouseStatus[] = {
   { "next color", "Color Menu", "prev color" }
};
static struct MouseStatusStrRec hAlignMouseStatus[] = {
   { "next horizontal align", "HoriAlign Menu", "prev horizontal align" }
};
static struct MouseStatusStrRec vAlignMouseStatus[] = {
   { "next vertical align", "VertAlign Menu", "prev vertical align" }
};
static struct MouseStatusStrRec pageMouseStatus[] = {
   { "next page", "Page Menu", "prev page" },
   { "drawing size", "Page Menu", "drawing size" }
};
static struct MouseStatusStrRec pageLayoutMouseStatus[] = {
   { "tiled page mode", "PageLayout Menu", "tiled page mode" },
   { "stacked page mode", "PageLayout Menu", "stacked page mode" }
};
static struct MouseStatusStrRec justMouseStatus[] = {
   { "center justified", "TextStyle Menu", "right justified" },
   { "right justified", "TextStyle Menu", "left justified" },
   { "left justified", "TextStyle Menu", "center justified" }
};
static struct MouseStatusStrRec fontMouseStatus[] = {
   { "next font", "Font Menu", "prev font" }
};
static struct MouseStatusStrRec vspaceMouseStatus[] = {
   { "inc text vspace", "modify text vspace", "dec text vspace" }
};
static struct MouseStatusStrRec textSizeMouseStatus[] = {
   { "next text size", "TextSize Menu", "prev text size" }
};
static struct MouseStatusStrRec rotateMouseStatus[] = {
   { "text rotate = 90", "Arrange Menu", "text rotate = 270" },
   { "text rotate = 180", "Arrange Menu", "no text rotate" },
   { "text rotate = 270", "Arrange Menu", "text rotate = 90" },
   { "no text rotate", "Arrange Menu", "text rotate = 180" }
};
static struct MouseStatusStrRec editMouseStatus[] = {
   { "(none)", "Edit Menu", "(none)" }
};
static struct MouseStatusStrRec specialMouseStatus[] = {
   { "(none)", "Special Menu", "(none)" }
};
static struct MouseStatusStrRec lineWidthMouseStatus[] = {
   { "next line width", "LineWidth Menu", "prev line width" }
};
static struct MouseStatusStrRec lineStyleMouseStatus[] = {
   { "next line style", "LineStyle Menu", "prev line style" }
};
static struct MouseStatusStrRec lineTypeMouseStatus[] = {
   { "next line type", "LineType Menu", "prev line type" }
};
static struct MouseStatusStrRec lineDashMouseStatus[] = {
   { "next line dash", "LineDash Menu", "prev line dash" }
};
static struct MouseStatusStrRec rcbRadiusMouseStatus[] = {
   { "inc rcbox radius", "modify rcbox radius", "dec rcbox radius" }
};
static struct MouseStatusStrRec zoomMouseStatus[] = {
   { "zoom in", "modify zoom", "zoom out" }
};
static struct MouseStatusStrRec moveModeMouseStatus[] = {
   { "unconstrained move mode", "MoveMode Menu", "unconstrained move mode" },
   { "constrained move mode", "MoveMode Menu", "constrained move mode" }
};
static struct MouseStatusStrRec bwPrintMouseStatus[] = {
   { "EPS print mode", "Layout Menu", "EPSI print mode" },     /* printer */
   { "PS print mode", "Layout Menu", "printer print mode" },   /* EPS */
   { "XBM print mode", "Layout Menu", "EPS print mode" },      /* PS */
   { "ASCII print mode", "Layout Menu", "PS print mode" },     /* XBM */
   { "EPSI print mode", "Layout Menu", "XBM print mode" },     /* ASCII */
   { "printer print mode", "Layout Menu", "ASCII print mode" } /* EPSI */
};
static struct MouseStatusStrRec colorPrintMouseStatus[] = {
   { "EPS print mode", "Layout Menu", "EPSI print mode" },     /* printer */
   { "PS print mode", "Layout Menu", "printer print mode" },   /* EPS */
   { "XPM print mode", "Layout Menu", "EPS print mode" },      /* PS */
   { "ASCII print mode", "Layout Menu", "PS print mode" },     /* XPM */
   { "EPSI print mode", "Layout Menu", "XPM print mode" },     /* ASCII */
   { "printer print mode", "Layout Menu", "ASCII print mode" } /* EPSI */
};
static struct MouseStatusStrRec fileMouseStatus[] = {
   { "landscape", "File Menu", "landscape" },
   { "portrait", "File Menu", "portrait" }
};
static struct MouseStatusStrRec fillMouseStatus[] = {
   { "next fill pattern", "Fill Menu", "prev fill pattern" }
};
static struct MouseStatusStrRec penMouseStatus[] = {
   { "next pen pattern", "Pen Menu", "prev pen pattern" }
};

int ChoiceEventHandler (input)
   XEvent	* input;
{
   XEvent	ev;
   XButtonEvent	* button_ev;
   int		xindex, yindex, delta=0, root_x, root_y, x=0, y=0;
   int		choice;
   Window	root_win, child_win;
   unsigned int	status;

   if (input->type == Expose)
   {
      XSync (mainDisplay, False);
      while (XCheckWindowEvent (mainDisplay, choiceWindow, ExposureMask, &ev)) ;
      RedrawChoiceWindow ();
   }
   else if (input->type == EnterNotify)
   {
      SetMouseStatus ("", "", "");
   }
   else if (input->type == MotionNotify)
   {
      xindex = (int)(input->xmotion.x / choiceImageW);
      yindex = (int)(input->xmotion.y / choiceImageH);
      choice = (xindex<<CHOICE_SHIFT)|yindex;
      switch (choice)
      {
         case CHOICE_MODE:
            SetMouseStatus (modeMouseStatus[curChoice].l,
                  modeMouseStatus[curChoice].m, modeMouseStatus[curChoice].r);
            break;
         case CHOICE_COLOR:
            SetMouseStatus (colorMouseStatus[0].l,
                  colorMouseStatus[0].m, colorMouseStatus[0].r);
            break;
         case CHOICE_HORI_ALIGN:
            SetMouseStatus (hAlignMouseStatus[0].l,
                  hAlignMouseStatus[0].m, hAlignMouseStatus[0].r);
            break;
         case CHOICE_VERT_ALIGN:
            SetMouseStatus (vAlignMouseStatus[0].l,
                  vAlignMouseStatus[0].m, vAlignMouseStatus[0].r);
            break;
         case CHOICE_PAGE:
            SetMouseStatus (pageMouseStatus[pageLayoutMode].l,
                  pageMouseStatus[pageLayoutMode].m,
                  pageMouseStatus[pageLayoutMode].r);
            break;
         case CHOICE_PAGELAYOUT:
            SetMouseStatus (pageLayoutMouseStatus[pageLayoutMode].l,
                  pageLayoutMouseStatus[pageLayoutMode].m,
                  pageLayoutMouseStatus[pageLayoutMode].r);
            break;
         case CHOICE_JUST:
            SetMouseStatus (justMouseStatus[textJust].l,
                  justMouseStatus[textJust].m, justMouseStatus[textJust].r);
            break;
         case CHOICE_FONT:
            SetMouseStatus (fontMouseStatus[0].l,
                  fontMouseStatus[0].m, fontMouseStatus[0].r);
            break;
         case CHOICE_VSPACE:
            SetMouseStatus (vspaceMouseStatus[0].l,
                  vspaceMouseStatus[0].m, vspaceMouseStatus[0].r);
            break;
         case CHOICE_TEXT_SIZE:
            SetMouseStatus (textSizeMouseStatus[0].l,
                  textSizeMouseStatus[0].m, textSizeMouseStatus[0].r);
            break;
         case CHOICE_ROTATE:
            SetMouseStatus (rotateMouseStatus[curRotate].l,
                  rotateMouseStatus[curRotate].m,
                  rotateMouseStatus[curRotate].r);
            break;
         case CHOICE_EDIT:
            SetMouseStatus (editMouseStatus[0].l,
                  editMouseStatus[0].m, editMouseStatus[0].r);
            break;
         case CHOICE_SPECIAL:
            SetMouseStatus (specialMouseStatus[0].l,
                  specialMouseStatus[0].m, specialMouseStatus[0].r);
            break;
         case CHOICE_LINE_WIDTH:
            SetMouseStatus (lineWidthMouseStatus[0].l,
                  lineWidthMouseStatus[0].m, lineWidthMouseStatus[0].r);
            break;
         case CHOICE_LINE_STYLE:
            SetMouseStatus (lineStyleMouseStatus[0].l,
                  lineStyleMouseStatus[0].m, lineStyleMouseStatus[0].r);
            break;
         case CHOICE_LINE_TYPE:
            SetMouseStatus (lineTypeMouseStatus[0].l,
                  lineTypeMouseStatus[0].m, lineTypeMouseStatus[0].r);
            break;
         case CHOICE_DASH:
            SetMouseStatus (lineDashMouseStatus[0].l,
                  lineDashMouseStatus[0].m, lineDashMouseStatus[0].r);
            break;
         case CHOICE_RADIUS:
            SetMouseStatus (rcbRadiusMouseStatus[0].l,
                  rcbRadiusMouseStatus[0].m, rcbRadiusMouseStatus[0].r);
            break;
         case CHOICE_ZOOM:
            SetMouseStatus (zoomMouseStatus[0].l,
                  zoomMouseStatus[0].m, zoomMouseStatus[0].r);
            break;
         case CHOICE_MOVE_MODE:
            SetMouseStatus (moveModeMouseStatus[moveMode].l,
                  moveModeMouseStatus[moveMode].m,
                  moveModeMouseStatus[moveMode].r);
            break;
         case CHOICE_PRINT:
            if (colorDump)
               SetMouseStatus (colorPrintMouseStatus[whereToPrint].l,
                     colorPrintMouseStatus[whereToPrint].m,
                     colorPrintMouseStatus[whereToPrint].r);
            else
               SetMouseStatus (bwPrintMouseStatus[whereToPrint].l,
                     bwPrintMouseStatus[whereToPrint].m,
                     bwPrintMouseStatus[whereToPrint].r);
            break;
         case CHOICE_FILE:
            SetMouseStatus (fileMouseStatus[pageStyle].l,
                  fileMouseStatus[pageStyle].m, fileMouseStatus[pageStyle].r);
            break;
         case CHOICE_FILL:
            SetMouseStatus (fillMouseStatus[0].l,
                  fillMouseStatus[0].m, fillMouseStatus[0].r);
            break;
         case CHOICE_PEN:
            SetMouseStatus (penMouseStatus[0].l,
                  penMouseStatus[0].m, penMouseStatus[0].r);
            break;
      }
   }
   else if (input->type == ButtonPress)
   {
      Msg ("");
      button_ev = &(input->xbutton);
      switch (button_ev->button)
      {
         case Button1: delta = 1; break;
         case Button2:
            XQueryPointer (mainDisplay, rootWindow, &root_win, &child_win,
                  &root_x, &root_y, &x, &y, &status);
            break;
         case Button3: delta = -1; break;
         default: return (INVALID);
      }
      xindex = (int)(button_ev->x / choiceImageW);
      yindex = (int)(button_ev->y / choiceImageH);
      choice = (xindex<<CHOICE_SHIFT)|yindex;
      switch (choice)
      {
         case CHOICE_MODE:
            if (delta == 0)
               ModeMenu (x, y, FALSE);
            else
               SetCurChoice ((curChoice+MAXCHOICES+delta) % MAXCHOICES);
            SetMouseStatus (modeMouseStatus[curChoice].l,
                  modeMouseStatus[curChoice].m, modeMouseStatus[curChoice].r);
            break;
         case CHOICE_COLOR:
            if (colorDisplay)
            {
               if (delta == 0)
                  ColorMenu (x, y, FALSE);
               else
               {
                  if (curChoice != NOTHING) SetCurChoice (NOTHING);
                  colorIndex = (colorIndex+maxColors+delta) % maxColors;
                  ShowColor (TRUE);
                  if (topSel != NULL) ChangeAllSelColor (colorIndex, TRUE);
               }
            }
            SetMouseStatus (colorMouseStatus[0].l,
                  colorMouseStatus[0].m, colorMouseStatus[0].r);
            break;
         case CHOICE_HORI_ALIGN:
            if (delta == 0)
               HoriAlignMenu (x, y, FALSE);
            else
               HoriAlignSubMenu ((horiAlign+MAXALIGNS+delta) % MAXALIGNS);
            SetMouseStatus (hAlignMouseStatus[0].l,
                  hAlignMouseStatus[0].m, hAlignMouseStatus[0].r);
            break;
         case CHOICE_VERT_ALIGN:
            if (delta == 0)
               VertAlignMenu (x, y, FALSE);
            else
               VertAlignSubMenu ((vertAlign+MAXALIGNS+delta) % MAXALIGNS);
            SetMouseStatus (vAlignMouseStatus[0].l,
                  vAlignMouseStatus[0].m, vAlignMouseStatus[0].r);
            break;
         case CHOICE_PAGE:
            if (delta == 0)
               PageMenu (x, y, FALSE);
            else if (pageLayoutMode == PAGE_STACK)
            {
               if (delta == 1)
                  NextPage ();
               else
                  PrevPage ();
            }
            else
               SpecifyDrawingSize ();
            SetMouseStatus (pageMouseStatus[pageLayoutMode].l,
                  pageMouseStatus[pageLayoutMode].m,
                  pageMouseStatus[pageLayoutMode].r);
            break;
         case CHOICE_PAGELAYOUT:
            if (delta == 0)
               PageLayoutMenu (x, y, FALSE);
            else
               PageLayoutSubMenu (!pageLayoutMode);
            SetMouseStatus (pageLayoutMouseStatus[pageLayoutMode].l,
                  pageLayoutMouseStatus[pageLayoutMode].m,
                  pageLayoutMouseStatus[pageLayoutMode].r);
            break;
         case CHOICE_JUST:
            if (delta == 0)
               StyleMenu (x, y, FALSE);
            else
            {
               if (curChoice == DRAWTEXT)
               {
                  if (TieLooseEnds ())
                  {
                     SetCurChoice (NOTHING);
                     HighLightReverse ();
                  }
                  else
                     SetCurChoice (NOTHING);
                  RemoveAllSel ();
               }
               textJust = (textJust+MAXJUSTS+delta) % MAXJUSTS;
               ShowJust ();
               UpdateSubMenu (MENU_STYLE);
               if (topSel != NULL) ChangeFontJust (textJust);
            }
            SetMouseStatus (justMouseStatus[textJust].l,
                  justMouseStatus[textJust].m, justMouseStatus[textJust].r);
            break;
         case CHOICE_FONT:
            if (delta == 0)
               FontMenu (x, y, FALSE);
            else
            {
               int	saved_font=curFont;

               if (curChoice == DRAWTEXT)
               {
                  if (TieLooseEnds ())
                  {
                     SetCurChoice (NOTHING);
                     HighLightReverse ();
                  }
                  else
                     SetCurChoice (NOTHING);
                  RemoveAllSel ();
               }
               curFont = (curFont+numFonts+delta) % numFonts;
               attemptingToSetFontProperty = TRUE;
               SetCanvasFont ();
               attemptingToSetFontProperty = FALSE;
               if (canvasFontSize == INVALID)
               {
                  char  msg[80];

                  sprintf (msg, "%s-%1d not available.", fontMenuStr[curFont],
                        curSize);
                  Dialog (msg, NULL, NULL);
                  curFont = saved_font;
                  SetCanvasFont ();
               }
               ShowCurFont ();
               ShowTextSize ();
               UpdateSubMenu (MENU_FONT);
               if (topSel != NULL) ChangeFont (curFont);
            }
            SetMouseStatus (fontMouseStatus[0].l,
                  fontMouseStatus[0].m, fontMouseStatus[0].r);
            break;
         case CHOICE_VSPACE:
            if (delta == 0)
               TextVSpaceLoop (button_ev);
            else
            {
               if (curChoice == DRAWTEXT)
               {
                  if (TieLooseEnds ())
                  {
                     SetCurChoice (NOTHING);
                     HighLightReverse ();
                  }
                  else
                     SetCurChoice (NOTHING);
                  RemoveAllSel ();
               }
               if (topSel == NULL && delta < 0 &&
                     textCursorH+textVSpace+delta <= 0)
                  Msg ("Text vertical spacing too small.  No change.");
               else
               {
                  textVSpace += delta;
                  ShowTextVSpace ();
                  if (topSel != NULL) ChangeVSpace (textVSpace);
               }
            }
            SetMouseStatus (vspaceMouseStatus[0].l,
                  vspaceMouseStatus[0].m, vspaceMouseStatus[0].r);
            break;
         case CHOICE_TEXT_SIZE:
            if (delta == 0)
               SizeMenu (x, y, FALSE);
            else
            {
               int	cur_index, saved_index;

               if (curChoice == DRAWTEXT)
               {
                  if (TieLooseEnds ())
                  {
                     SetCurChoice (NOTHING);
                     HighLightReverse ();
                  }
                  else
                     SetCurChoice (NOTHING);
                  RemoveAllSel ();
               }
               cur_index = saved_index = GetSizeMenuIndex ();
               if (cur_index == INVALID)
                  cur_index = 0;
               else
                  cur_index = (cur_index+numFontSizes+delta) % numFontSizes;
               curSize = fontSizes[cur_index];
               attemptingToSetFontProperty = TRUE;
               SetCanvasFont ();
               attemptingToSetFontProperty = FALSE;
               if (canvasFontSize == INVALID)
               {
                  char	msg[80];

                  sprintf (msg, "%s-%1d not available.", fontMenuStr[curFont],
                        curSize);
                  Dialog (msg, NULL, NULL);
                  cur_index = saved_index;
                  curSize = fontSizes[cur_index];
                  SetCanvasFont ();
               }
               ShowCurFont ();
               ShowTextSize ();
               UpdateSubMenu (MENU_SIZE);
               if (topSel != NULL) ChangeFontSize (cur_index);
            }
            SetMouseStatus (textSizeMouseStatus[0].l,
                  textSizeMouseStatus[0].m, textSizeMouseStatus[0].r);
            break;
         case CHOICE_ROTATE:
            if (delta == 0)
               ArrangeMenu (x, y, FALSE);
            else
            {
               if (curChoice == DRAWTEXT)
               {
                  if (TieLooseEnds ())
                  {
                     SetCurChoice (NOTHING);
                     HighLightReverse ();
                  }
                  else
                     SetCurChoice (NOTHING);
                  RemoveAllSel ();
               }
               curRotate = (curRotate+4+delta) % 4;
               ShowRotate ();
               if (topSel != NULL)
               {
                  if (delta == 1)
                     RotateClockWise ();
                  else
                     RotateCounter ();
               }
            }
            SetMouseStatus (rotateMouseStatus[curRotate].l,
                  rotateMouseStatus[curRotate].m,
                  rotateMouseStatus[curRotate].r);
            break;
         case CHOICE_EDIT:
            if (delta == 0) EditMenu (x, y, FALSE);
            SetMouseStatus (editMouseStatus[0].l,
                  editMouseStatus[0].m, editMouseStatus[0].r);
            break;
         case CHOICE_SPECIAL:
            if (delta == 0) SpecialMenu (x, y, FALSE);
            SetMouseStatus (specialMouseStatus[0].l,
                  specialMouseStatus[0].m, specialMouseStatus[0].r);
            break;
         case CHOICE_LINE_WIDTH:
            if (delta == 0)
               LineWidthMenu (x, y, FALSE);
            else
            {
               lineWidth = (lineWidth+maxLineWidths+delta) % maxLineWidths;
               ShowLineWidth ();
               UpdateSubMenu (MENU_LINEWIDTH);
               if (topSel != NULL) ChangeAllSelLineWidth (lineWidth, TRUE);
            }
            SetMouseStatus (lineWidthMouseStatus[0].l,
                  lineWidthMouseStatus[0].m, lineWidthMouseStatus[0].r);
            break;
         case CHOICE_LINE_STYLE:
            if (delta == 0)
               LineStyleMenu (x, y, FALSE);
            else
            {
               lineStyle = (lineStyle+MAXLINESTYLES+delta) % MAXLINESTYLES;
               ShowLineStyle ();
               UpdateSubMenu (MENU_LINESTYLE);
               if (topSel != NULL) ChangeAllSelLineStyle (lineStyle, TRUE);
            }
            SetMouseStatus (lineStyleMouseStatus[0].l,
                  lineStyleMouseStatus[0].m, lineStyleMouseStatus[0].r);
            break;
         case CHOICE_LINE_TYPE:
            if (delta == 0)
               LineTypeMenu (x, y, FALSE);
            else
            {
               curSpline = (curSpline+MAXLINETYPES+delta) % MAXLINETYPES;
               ShowLineType ();
               UpdateSubMenu (MENU_LINETYPE);
               if (topSel != NULL) ChangeAllSelLineType (curSpline, TRUE);
            }
            SetMouseStatus (lineTypeMouseStatus[0].l,
                  lineTypeMouseStatus[0].m, lineTypeMouseStatus[0].r);
            break;
         case CHOICE_DASH:
            if (delta == 0)
               LineDashMenu (x, y, FALSE);
            else
            {
               curDash = (curDash+MAXDASHES+delta) % MAXDASHES;
               ShowDash ();
               UpdateSubMenu (MENU_LINEDASH);
               if (topSel != NULL) ChangeAllSelDashes (curDash, TRUE);
            }
            SetMouseStatus (lineDashMouseStatus[0].l,
                  lineDashMouseStatus[0].m, lineDashMouseStatus[0].r);
            break;
         case CHOICE_RADIUS:
            if (delta == 0)
               RCBRadiusLoop (button_ev);
            else
            {
               if (rcbRadius+delta >= MIN_RCB_RADIUS)
               {
                  rcbRadius += delta;
                  ShowRCBRadius ();
                  if (topSel != NULL) ChangeAllSelRCBRadius (rcbRadius);
               }
            }
            SetMouseStatus (rcbRadiusMouseStatus[0].l,
                  rcbRadiusMouseStatus[0].m, rcbRadiusMouseStatus[0].r);
            break;
         case CHOICE_ZOOM:
            switch (delta)
            {
               case -1: ZoomIn (); break;
               case 0: ZoomLoop (button_ev); break;
               case 1: ZoomOut (); break;
            }
            SetMouseStatus (zoomMouseStatus[0].l,
                  zoomMouseStatus[0].m, zoomMouseStatus[0].r);
            break;
         case CHOICE_MOVE_MODE:
            if (delta == 0)
               MoveModeMenu (x, y, FALSE);
            else
               ToggleMoveMode ();
            SetMouseStatus (moveModeMouseStatus[moveMode].l,
                  moveModeMouseStatus[moveMode].m,
                  moveModeMouseStatus[moveMode].r);
            break;
         case CHOICE_PRINT:
            if (delta == 0)
               LayoutMenu (x, y, FALSE);
            else
            {
               whereToPrint = (whereToPrint+MAXWHERETOPRINT+delta) %
                     MAXWHERETOPRINT-1;
               ToggleWhereToPrint ();
            }
            if (colorDump)
               SetMouseStatus (colorPrintMouseStatus[whereToPrint].l,
                     colorPrintMouseStatus[whereToPrint].m,
                     colorPrintMouseStatus[whereToPrint].r);
            else
               SetMouseStatus (bwPrintMouseStatus[whereToPrint].l,
                     bwPrintMouseStatus[whereToPrint].m,
                     bwPrintMouseStatus[whereToPrint].r);
            break;
         case CHOICE_FILE:
            if (delta == 0)
               return (FileMenu (x, y, FALSE));
            else
            {
               switch (pageStyle)
               {
                  case PORTRAIT:
                     ChangePageStyle (LANDSCAPE, "LandScape"); break;
                  case LANDSCAPE:
                     ChangePageStyle (PORTRAIT, "Portrait"); break;
               }
            }
            SetMouseStatus (fileMouseStatus[pageStyle].l,
                  fileMouseStatus[pageStyle].m, fileMouseStatus[pageStyle].r);
            break;
         case CHOICE_FILL:
            if (delta == 0)
               FillMenu (x, y, FALSE);
            else
            {
               if (curChoice == DRAWTEXT)
               {
                  if (TieLooseEnds ())
                  {
                     SetCurChoice (NOTHING);
                     HighLightReverse ();
                  }
                  else
                     SetCurChoice (NOTHING);
                  RemoveAllSel ();
               }
               objFill = (objFill+MAXPATTERNS+delta) % MAXPATTERNS;
               ShowFill ();
               if (topSel != NULL) ChangeAllSelFill (objFill, TRUE);
            }
            SetMouseStatus (fillMouseStatus[0].l,
                  fillMouseStatus[0].m, fillMouseStatus[0].r);
            break;
         case CHOICE_PEN:
            if (delta == 0)
               PenMenu (x, y, FALSE);
            else
            {
               if (curChoice == DRAWTEXT)
               {
                  if (TieLooseEnds ())
                  {
                     SetCurChoice (NOTHING);
                     HighLightReverse ();
                  }
                  else
                     SetCurChoice (NOTHING);
                  RemoveAllSel ();
               }
               penPat = (penPat+MAXPATTERNS+delta) % MAXPATTERNS;
               ShowPen ();
               if (topSel != NULL) ChangeAllSelPen (penPat, TRUE);
            }
            SetMouseStatus (penMouseStatus[0].l,
                  penMouseStatus[0].m, penMouseStatus[0].r);
            break;
      }
   }
   return (INVALID);
}
