/*
 * Author:      William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990-1994, William Cheng.
 *
 * Permission limited to the use, copy, modify, and distribute this software
 * and its documentation for any purpose is hereby granted by the Author without
 * fee, provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the Author not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.  All other
 * rights (including the right to sell "tgif" and the right to sell derivative
 * works of tgif) are reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /export/bourbon/multimedia/william/X11/TGIF2/RCS/stretch.c,v 2.95 1994/12/06 04:26:33 william Exp $";
#endif

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include "const.h"
#include "types.h"

#include "align.e"
#include "arc.e"
#include "auxtext.e"
#include "choice.e"
#include "cmd.e"
#include "color.e"
#include "cursor.e"
#include "dialog.e"
#include "drawing.e"
#include "dup.e"
#include "font.e"
#include "grid.e"
#include "mainloop.e"
#include "mark.e"
#include "move.e"
#include "msg.e"
#include "obj.e"
#include "poly.e"
#include "raster.e"
#include "ruler.e"
#include "select.e"
#include "setup.e"
#include "spline.e"
#ifndef _NO_EXTERN
#include "stretch.e"
#endif
#include "text.e"
#include "xbitmap.e"
#include "xpixmap.e"

#define CORNER_CC 0
#define CORNER_LT 1
#define CORNER_T 2
#define CORNER_RT 3
#define CORNER_R 4
#define CORNER_RB 5
#define CORNER_B 6
#define CORNER_LB 7
#define CORNER_L 8

static
int PtIn4Corners (XOff, YOff, BBox, Corner)
   int		XOff, YOff, * Corner;
   struct BBRec	BBox;
{
   if (PtInMark (XOff, YOff, OFFSET_X(BBox.ltx), OFFSET_Y(BBox.lty)))
   {
      *Corner = 1;
      return(TRUE);
   }
   if (PtInMark (XOff, YOff, OFFSET_X(BBox.ltx), OFFSET_Y(BBox.rby)))
   {
      *Corner = 7;
      return(TRUE);
   }
   if (PtInMark (XOff, YOff, OFFSET_X(BBox.rbx), OFFSET_Y(BBox.lty)))
   {
      *Corner = 3;
      return(TRUE);
   }
   if (PtInMark (XOff, YOff, OFFSET_X(BBox.rbx), OFFSET_Y(BBox.rby)))
   {
      *Corner = 5;
      return(TRUE);
   }
   return (FALSE);
}

static
int PtIn8Places (XOff, YOff, BBox, Corner)
   int		XOff, YOff, * Corner;
   struct BBRec	BBox;
{
   register int	xmid, ymid;

   if (PtIn4Corners (XOff, YOff, BBox, Corner)) return (TRUE);

   xmid = ((BBox.ltx+BBox.rbx)>>1);
   if (PtInMark (XOff, YOff, OFFSET_X(xmid), OFFSET_Y(BBox.lty)))
   {
      *Corner = 2;
      return(TRUE);
   }
   if (PtInMark (XOff, YOff, OFFSET_X(xmid), OFFSET_Y(BBox.rby)))
   {
      *Corner = 6;
      return(TRUE);
   }
   ymid = ((BBox.lty+BBox.rby)>>1);
   if (PtInMark (XOff, YOff, OFFSET_X(BBox.ltx), OFFSET_Y(ymid)))
   {
      *Corner = 8;
      return(TRUE);
   }
   if (PtInMark (XOff, YOff, OFFSET_X(BBox.rbx), OFFSET_Y(ymid)))
   {
      *Corner = 4;
      return(TRUE);
   }
   return (FALSE);
}

int PtInPolyMark (XOff, YOff, NumPts, V, Index)
   int		XOff, YOff, NumPts, * Index;
   XPoint	* V;
{
   register int	i;

   for (i = 0; i < NumPts; i++)
      if (PtInMark (XOff, YOff, OFFSET_X(V[i].x), OFFSET_Y(V[i].y)))
      {
         *Index = i;
         return (TRUE);
      }
   return (FALSE);
}

static
int ObjRigid (obj_ptr)
   struct ObjRec	* obj_ptr;
{
   register struct AttrRec	* attr_ptr;

   for (attr_ptr=obj_ptr->lattr; attr_ptr!=NULL; attr_ptr=attr_ptr->prev)
      if (attr_ptr->inherited && strcmp(attr_ptr->name,"")==0 &&
            strcmp(attr_ptr->s,"not_rigid")==0)
         return (FALSE);
   return (TRUE);
}

int AutoCenterAttr (obj_ptr)
   struct ObjRec	* obj_ptr;
{
   register struct AttrRec	* attr_ptr;

   for (attr_ptr=obj_ptr->lattr; attr_ptr!=NULL; attr_ptr=attr_ptr->prev)
      if (*attr_ptr->name=='\0' && strcmp (attr_ptr->s, "auto_center_attr")==0)
         return (TRUE);
   return (FALSE);
}

void CenterObjInOBBox (TextObjPtr, OBBox, BBoxReturn)
   struct ObjRec	* TextObjPtr;
   struct BBRec		OBBox, * BBoxReturn;
{
   int text_w, text_h, bbox_w, bbox_h, dx, dy;

   if (BBoxReturn != NULL)
   {
      BBoxReturn->ltx = TextObjPtr->bbox.ltx;
      BBoxReturn->lty = TextObjPtr->bbox.lty;
      BBoxReturn->rbx = TextObjPtr->bbox.rbx;
      BBoxReturn->rby = TextObjPtr->bbox.rby;
   }
   text_w = TextObjPtr->obbox.rbx-TextObjPtr->obbox.ltx;
   text_h = TextObjPtr->obbox.rby-TextObjPtr->obbox.lty;
   bbox_w = OBBox.rbx - OBBox.ltx;
   bbox_h = OBBox.rby - OBBox.lty;
   if (text_w > bbox_w)
      dx = OBBox.ltx-((text_w-bbox_w)>>1)-TextObjPtr->obbox.ltx;
   else
      dx = OBBox.ltx+((bbox_w-text_w)>>1)-TextObjPtr->obbox.ltx;
   if (text_h > bbox_h)
      dy = OBBox.lty-((text_h-bbox_h)>>1)-TextObjPtr->obbox.lty;
   else
      dy = OBBox.lty+((bbox_h-text_h)>>1)-TextObjPtr->obbox.lty;
   MoveObj (TextObjPtr, dx, dy);
   if (BBoxReturn != NULL)
   {
      if (TextObjPtr->bbox.ltx < BBoxReturn->ltx)
         BBoxReturn->ltx = TextObjPtr->bbox.ltx;
      if (TextObjPtr->bbox.lty < BBoxReturn->lty)
         BBoxReturn->lty = TextObjPtr->bbox.lty;
      if (TextObjPtr->bbox.rbx > BBoxReturn->rbx)
         BBoxReturn->rbx = TextObjPtr->bbox.rbx;
      if (TextObjPtr->bbox.rby > BBoxReturn->rby)
         BBoxReturn->rby = TextObjPtr->bbox.rby;
   }
}

struct SelRec * PtInSelMark (XOff, YOff, Corner)
   int	XOff, YOff, * Corner;
   /* XOff and YOff are screen offsets */
   /* 1 2 3 */
   /* 8   4 */
   /* 7 6 5 */
{
   register struct SelRec	* sel_ptr;
   register struct ObjRec	* obj_ptr;

   for (sel_ptr = topSel; sel_ptr != NULL; sel_ptr = sel_ptr->next)
   {
      obj_ptr = sel_ptr->obj;
      switch (obj_ptr->type)
      {
         case OBJ_POLY:
            if (PtInPolyMark (XOff, YOff, obj_ptr->detail.p->n,
                  obj_ptr->detail.p->vlist, Corner))
               return (sel_ptr);
            break;
         case OBJ_POLYGON:
            if (PtInPolyMark (XOff, YOff, obj_ptr->detail.g->n-1,
                  obj_ptr->detail.g->vlist, Corner))
               return (sel_ptr);
            break;
         case OBJ_BOX:
         case OBJ_GROUP:
         case OBJ_SYM:
         case OBJ_OVAL:
         case OBJ_ARC:
         case OBJ_RCBOX:
         case OBJ_XBM:
         case OBJ_XPM:
            if (PtIn8Places (XOff, YOff, obj_ptr->obbox, Corner))
               return (sel_ptr);
            break;
         case OBJ_ICON:
            if (PtIn8Places (XOff, YOff, obj_ptr->obbox, Corner) &&
                  !ObjRigid (obj_ptr))
               return (sel_ptr);
            break;
         case OBJ_TEXT:
            if (PtIn4Corners (XOff, YOff, obj_ptr->obbox, Corner))
               return (sel_ptr);
            break;
      }
   }
   return (NULL);
}

static XPoint v[5];

static
void StretchPoly (XGridOff, YGridOff, ObjPtr, NumPts, V, Index)
   int			XGridOff, YGridOff, NumPts, Index;
   XPoint		* V;
   struct ObjRec	* ObjPtr;
{
   register int	i;
   int		x, y, dx, dy, num=0, stretching=TRUE;
   int		ltx, lty, rbx, rby, curved=LT_STRAIGHT;
   int		grid_x=XGridOff, grid_y=YGridOff, sn, intn;
   int		auto_center_attr=AutoCenterAttr (ObjPtr);
   char		buf[80], * smooth=NULL;
   XEvent	input, ev;
   XPoint	* pv=NULL, * sv=NULL, * cntrlv=NULL;

   if (ObjPtr->locked)
   {
      Msg ("Locked object can not be stretched.");
      return;
   }

   switch (ObjPtr->type)
   {
      case OBJ_POLY:
         curved = ObjPtr->detail.p->curved;
         if (splineRubberband)
         {
            pv = (XPoint *) calloc (NumPts+1, sizeof (XPoint));
            if (pv == NULL) fprintf (stderr, "Can not calloc().\n");
            if (curved != LT_INTSPLINE && ObjPtr->detail.p->smooth != NULL)
            {
               smooth = (char *) calloc (NumPts+1, sizeof(char));
               if (smooth == NULL) fprintf (stderr, "Can not calloc().\n");
            }
            for (i = 0; i < NumPts; i++)
            {
               pv[i].x = V[i].x;
               pv[i].y = V[i].y;
               if (smooth != NULL) smooth[i] = ObjPtr->detail.p->smooth[i];
            }
            switch (curved)
            {
               case LT_STRAIGHT:
               case LT_SPLINE:
                  sv = MakeMultiSplinePolyVertex (&sn, smooth,
                        drawOrigX, drawOrigY, NumPts, pv);
                  break;
               case LT_INTSPLINE:
                  sv = MakeIntSplinePolyVertex (&sn, &intn, &cntrlv, drawOrigX,
                        drawOrigY, NumPts, pv);
                  break;
            }
         }
         else if (Index == 0 || Index == NumPts-1)
         {
            num = 2;
            if (Index == 0)
            {
               v[0].x = OFFSET_X(V[1].x); v[0].y = OFFSET_Y(V[1].y);
               v[1].x = OFFSET_X(V[0].x); v[1].y = OFFSET_Y(V[0].y);
            }
            else
            {
               v[0].x = OFFSET_X(V[NumPts-2].x);
               v[0].y = OFFSET_Y(V[NumPts-2].y);
               v[1].x = OFFSET_X(V[NumPts-1].x);
               v[1].y = OFFSET_Y(V[NumPts-1].y);
            }
         }
         else
         {
            num = 3;
            v[0].x = OFFSET_X(V[Index-1].x); v[0].y = OFFSET_Y(V[Index-1].y);
            v[1].x = OFFSET_X(V[Index].x);   v[1].y = OFFSET_Y(V[Index].y);
            v[2].x = OFFSET_X(V[Index+1].x); v[2].y = OFFSET_Y(V[Index+1].y);
         }
         break;
      case OBJ_POLYGON:
         curved = ObjPtr->detail.g->curved;
         if (splineRubberband)
         {
            pv = (XPoint *) calloc (NumPts+1, sizeof (XPoint));
            if (pv == NULL) fprintf (stderr, "Can not calloc().\n");
            if (curved != LT_INTSPLINE && ObjPtr->detail.g->smooth != NULL)
            {
               smooth = (char *) calloc (NumPts+1, sizeof(char));
               if (smooth == NULL) fprintf (stderr, "Can not calloc().\n");
            }
            for (i = 0; i < NumPts; i++)
            {
               pv[i].x = V[i].x;
               pv[i].y = V[i].y;
               if (smooth != NULL) smooth[i] = ObjPtr->detail.g->smooth[i];
            }
            switch (curved)
            {
               case LT_STRAIGHT:
               case LT_SPLINE:
                  sv = MakeMultiSplinePolygonVertex (&sn, smooth,
                        drawOrigX, drawOrigY, NumPts, pv);
                  break;
               case LT_INTSPLINE:
                  sv = MakeIntSplinePolygonVertex (&sn, &intn, &cntrlv,
                        drawOrigX, drawOrigY, NumPts, pv);
                  break;
            }
         }
         else if (Index == 0 || Index == NumPts-1)
         {
            num = 3;
            v[0].x = OFFSET_X(V[1].x); v[0].y = OFFSET_Y(V[1].y);
            v[1].x = OFFSET_X(V[0].x); v[1].y = OFFSET_Y(V[0].y);
            v[2].x = OFFSET_X(V[NumPts-2].x); v[2].y = OFFSET_Y(V[NumPts-2].y);
         }
         else
         {
            num = 3;
            v[0].x = OFFSET_X(V[Index-1].x); v[0].y = OFFSET_Y(V[Index-1].y);
            v[1].x = OFFSET_X(V[Index].x);   v[1].y = OFFSET_Y(V[Index].y);
            v[2].x = OFFSET_X(V[Index+1].x); v[2].y = OFFSET_Y(V[Index+1].y);
         }
         break;
   }

   ltx = ObjPtr->bbox.ltx;
   lty = ObjPtr->bbox.lty;
   rbx = ObjPtr->bbox.rbx;
   rby = ObjPtr->bbox.rby;

   XFlush (mainDisplay);
   XSync (mainDisplay, False);

   if (XCheckMaskEvent (mainDisplay, ExposureMask, &ev) ||
         XCheckMaskEvent (mainDisplay, VisibilityChangeMask, &ev))
      ExposeEventHandler (&ev, TRUE);

   sprintf (buf, "dx=+0,dy=+0");
   StartShowMeasureCursor (grid_x, grid_y, buf, TRUE);
   XGrabPointer (mainDisplay, drawWindow, False,
         PointerMotionMask | ButtonReleaseMask,
         GrabModeAsync, GrabModeAsync, None, handCursor, CurrentTime);

   dx = dy = 0;
   while (stretching)
   {
      XNextEvent (mainDisplay, &input);

      if (input.type == Expose || input.type == VisibilityNotify)
         ExposeEventHandler (&input, TRUE);
      else if (input.type == ButtonRelease)
      {
         XUngrabPointer (mainDisplay, CurrentTime);
         stretching = FALSE;
      }
      else if (input.type == MotionNotify)
      {
         sprintf (buf, "dx=%+1d,dy=%+1d", ABS_SIZE(grid_x-XGridOff),
               ABS_SIZE(grid_y-YGridOff));
         ShowMeasureCursor (grid_x, grid_y, buf, TRUE);
         x = input.xmotion.x;
         y = input.xmotion.y;
         GridXY (x, y, &grid_x, &grid_y);

         if (splineRubberband)
            XDrawLines (mainDisplay, drawWindow, revDefaultGC, sv, sn,
                  CoordModeOrigin);
         else
            MyDashedLine (drawWindow, revDefaultGC, v, num);

         dx = grid_x - XGridOff;
         dy = grid_y - YGridOff;
         v[1].x = OFFSET_X(V[Index].x) + dx;
         v[1].y = OFFSET_Y(V[Index].y) + dy;
         MarkRulers (v[1].x, v[1].y);

         if (splineRubberband)
         {
            cfree (sv);
            if (ObjPtr->type==OBJ_POLYGON && (Index==0 || Index==NumPts-1))
            {
               pv[0].x = pv[NumPts-1].x = V[Index].x + ABS_SIZE(dx);
               pv[0].y = pv[NumPts-1].y = V[Index].y + ABS_SIZE(dy);
            }
            else
            {
               pv[Index].x = V[Index].x + ABS_SIZE(dx);
               pv[Index].y = V[Index].y + ABS_SIZE(dy);
            }
            switch (ObjPtr->type)
            {
               case OBJ_POLY:
                  switch (curved)
                  {
                     case LT_STRAIGHT:
                     case LT_SPLINE:
                        sv = MakeMultiSplinePolyVertex (&sn, smooth,
                              drawOrigX, drawOrigY, NumPts, pv);
                        break;
                     case LT_INTSPLINE:
                        cfree (cntrlv);
                        sv = MakeIntSplinePolyVertex (&sn, &intn, &cntrlv,
                              drawOrigX, drawOrigY, NumPts, pv);
                        break;
                  }
                  break;
               case OBJ_POLYGON:
                  switch (curved)
                  {
                     case LT_STRAIGHT:
                     case LT_SPLINE:
                        sv = MakeMultiSplinePolygonVertex (&sn, smooth,
                              drawOrigX, drawOrigY, NumPts, pv);
                        break;
                     case LT_INTSPLINE:
                        cfree (cntrlv);
                        sv = MakeIntSplinePolygonVertex (&sn, &intn, &cntrlv,
                              drawOrigX, drawOrigY, NumPts, pv);
                        break;
                  }
                  break;
            }
            XDrawLines (mainDisplay, drawWindow, revDefaultGC, sv, sn,
                  CoordModeOrigin);
         }
         else
            MyDashedLine (drawWindow, revDefaultGC, v, num);
         sprintf (buf, "dx=%+1d,dy=%+1d", ABS_SIZE(grid_x-XGridOff),
               ABS_SIZE(grid_y-YGridOff));
         ShowMeasureCursor (grid_x, grid_y, buf, TRUE);
         while (XCheckMaskEvent (mainDisplay, PointerMotionMask, &ev)) ;
      }
   }
   sprintf (buf, "dx=%+1d,dy=%+1d", ABS_SIZE(grid_x-XGridOff),
         ABS_SIZE(grid_y-YGridOff));
   EndShowMeasureCursor (grid_x, grid_y, buf, TRUE);
   if (dx != 0 || dy != 0)
   {
      if (splineRubberband)
         XDrawLines (mainDisplay, drawWindow, revDefaultGC, sv, sn,
               CoordModeOrigin);
      else
         MyDashedLine (drawWindow, revDefaultGC, v, num);

      HighLightReverse ();

      PrepareToReplaceAnObj (ObjPtr);

      dx = ABS_SIZE(dx);
      dy = ABS_SIZE(dy);
      switch (ObjPtr->type)
      {
         case OBJ_POLY:
            V[Index].x += dx; V[Index].y += dy;
            AdjObjSplineVs (ObjPtr);
            if (ObjPtr->detail.p->curved != LT_INTSPLINE)
               UpdPolyBBox (ObjPtr, NumPts, V);
            else
               UpdPolyBBox (ObjPtr, ObjPtr->detail.p->intn,
                     ObjPtr->detail.p->intvlist);
            break;
         case OBJ_POLYGON:
            V[Index].x += dx; V[Index].y += dy;
            if (Index == 0)
            {
               V[NumPts-1].x += dx; V[NumPts-1].y += dy;
            }
            else if (Index == NumPts-1)
            {
               V[0].x += dx; V[0].y += dy;
            }
            AdjObjSplineVs (ObjPtr);
            if (ObjPtr->detail.g->curved != LT_INTSPLINE)
               UpdPolyBBox (ObjPtr, NumPts, V);
            else
               UpdPolyBBox (ObjPtr, ObjPtr->detail.g->intn,
                     ObjPtr->detail.g->intvlist);
            break;
      }
      if (auto_center_attr)
      {
         struct AttrRec	* attr_ptr=ObjPtr->fattr;
         int		modified=FALSE;

         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            if (attr_ptr->shown)
            {
               struct BBRec	bbox;

               CenterObjInOBBox (attr_ptr->obj, ObjPtr->obbox, &bbox);
               if (bbox.ltx < ltx) ltx = bbox.ltx;
               if (bbox.lty < lty) lty = bbox.lty;
               if (bbox.rbx > rbx) rbx = bbox.rbx;
               if (bbox.rby > rby) rby = bbox.rby;
               modified = TRUE;
            }
         if (modified) AdjObjBBox (ObjPtr);
      }
      RecordReplaceAnObj (ObjPtr);

      UpdSelBBox ();
      RedrawAreas (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
            rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1),
            ObjPtr->bbox.ltx-GRID_ABS_SIZE(1),
            ObjPtr->bbox.lty-GRID_ABS_SIZE(1),
            ObjPtr->bbox.rbx+GRID_ABS_SIZE(1),
            ObjPtr->bbox.rby+GRID_ABS_SIZE(1));
      SetFileModified (TRUE);
      justDupped = FALSE;

      HighLightForward ();
   }
   if (splineRubberband)
   {
      cfree (sv);
      cfree (pv);
      if (smooth != NULL) cfree (smooth);
      if (curved == LT_INTSPLINE && cntrlv != NULL) cfree (cntrlv);
   }
}

static double	multX, multY;
static int	pivotX, pivotY, changeX, changeY, moveX, moveY;
static int	absPivotX, absPivotY;

static
void StretchedXY (X, Y, NewX, NewY)
   int	X, Y, * NewX, * NewY; /* screen offsets */
{
   register int	dx, dy;

   dx = round((double)((double)(X - pivotX) * multX));
   dy = round((double)((double)(Y - pivotY) * multY));
   *NewX = pivotX + dx;
   *NewY = pivotY + dy;
}

static
void StretchedAbsXY (X, Y, NewX, NewY)
   int	X, Y, * NewX, * NewY; /* screen offsets */
{
   register int	dx, dy;

   dx = round((double)((double)(X - absPivotX) * multX));
   dy = round((double)((double)(Y - absPivotY) * multY));
   *NewX = absPivotX + dx;
   *NewY = absPivotY + dy;
}

static
void SetPivot (Corner, OBBox)
   int		Corner;
   struct BBRec OBBox;
   /* pivotX, pivotY, moveX, moveY will be set to screen offsets */
{
   switch (Corner)
   {
      case 1:
         pivotX = OBBox.rbx; pivotY = OBBox.rby;
         moveX = OBBox.ltx; moveY = OBBox.lty;
         changeX = changeY = TRUE;
         break;
      case 2:
         pivotX = moveX = ((OBBox.ltx+OBBox.rbx)>>1); pivotY = OBBox.rby;
         moveY = OBBox.lty;
         changeX = FALSE; changeY = TRUE;
         break;
      case 3:
         pivotX = OBBox.ltx; pivotY = OBBox.rby;
         moveX = OBBox.rbx; moveY = OBBox.lty;
         changeX = changeY = TRUE;
         break;
      case 4:
         pivotX = OBBox.ltx; pivotY = moveY = ((OBBox.lty+OBBox.rby)>>1);
         moveX = OBBox.rbx;
         changeX = TRUE; changeY = FALSE;
         break;
      case 5:
         pivotX = OBBox.ltx; pivotY = OBBox.lty;
         moveX = OBBox.rbx; moveY = OBBox.rby;
         changeX = changeY = TRUE;
         break;
      case 6:
         pivotX = moveX = ((OBBox.ltx+OBBox.rbx)>>1); pivotY = OBBox.lty;
         moveY = OBBox.rby;
         changeX = FALSE; changeY = TRUE;
         break;
      case 7:
         pivotX = OBBox.rbx; pivotY = OBBox.lty;
         moveX = OBBox.ltx; moveY = OBBox.rby;
         changeX = changeY = TRUE;
         break;
      case 8:
         pivotX = OBBox.rbx; pivotY = moveY = ((OBBox.lty+OBBox.rby)>>1);
         moveX = OBBox.ltx;
         changeX = TRUE; changeY = FALSE;
         break;
   }
   multX = 1.0;
   multY = 1.0;
   absPivotX = pivotX;
   absPivotY = pivotY;
   pivotX = OFFSET_X(absPivotX);
   pivotY = OFFSET_Y(absPivotY);
   moveX = OFFSET_X(moveX);
   moveY = OFFSET_Y(moveY);
}

static void StretchObj ();

static
void StretchAttr (ObjPtr, Corner, InsideIcon, AutoCenterAttr)
   struct ObjRec	* ObjPtr;
   int			Corner, InsideIcon, AutoCenterAttr;
{
   struct AttrRec	* attr_ptr=ObjPtr->fattr;

   if (AutoCenterAttr)
   {
      for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
         if (attr_ptr->shown)
            CenterObjInOBBox (attr_ptr->obj, ObjPtr->obbox, NULL);
         else
            StretchObj (attr_ptr->obj, Corner, InsideIcon);
   }
   else
      for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
         StretchObj (attr_ptr->obj, Corner, InsideIcon);
}

static
void StretchObj (ObjPtr, Corner, InsideIcon)
   struct ObjRec	* ObjPtr;
   int			Corner, InsideIcon;
{
   register int			i;
   register struct ObjRec	* ptr;
   struct ArcRec		* arc_ptr;
   int				x, y, dx, dy, new_x, new_y, h;
   int				ltx, lty, rbx, rby, xc, yc, icon_is_rigid=TRUE;
   int				real_ltx, real_lty, real_rbx, real_rby;
   int				* flip_ptr=NULL, * rotate_ptr=NULL;
   int				auto_center_attr = AutoCenterAttr (ObjPtr);

   StretchedAbsXY (ObjPtr->obbox.ltx, ObjPtr->obbox.lty, &ltx, &lty);
   StretchedAbsXY (ObjPtr->obbox.rbx, ObjPtr->obbox.rby, &rbx, &rby);
   CalcBBox (ltx, lty, rbx, rby, &real_ltx, &real_lty, &real_rbx, &real_rby);
   StretchedAbsXY (ObjPtr->x, ObjPtr->y, &x, &y);
   h = ABS_SIZE(ObjPtr->obbox.rby - ObjPtr->obbox.lty);

   switch (ObjPtr->type)
   {
      case OBJ_BOX:
      case OBJ_OVAL:
      case OBJ_ARC:
      case OBJ_RCBOX:
      case OBJ_SYM:
      case OBJ_GROUP:
      case OBJ_ICON:
      case OBJ_XBM:
      case OBJ_XPM:
         switch (ObjPtr->type)
         {
            case OBJ_XBM:
               rotate_ptr = &(ObjPtr->detail.xbm->rotate);
               flip_ptr = &(ObjPtr->detail.xbm->flip);
               break;
            case OBJ_XPM:
               rotate_ptr = &(ObjPtr->detail.xpm->rotate);
               flip_ptr = &(ObjPtr->detail.xpm->flip);
               break;
            case OBJ_ICON:
               rotate_ptr = &(ObjPtr->detail.r->rotate);
               flip_ptr = &(ObjPtr->detail.r->flip);
               break;
         }
         if (rotate_ptr != NULL)
         {
            switch (*rotate_ptr)
            {
               case ROTATE0:
               case ROTATE180:
                  if (multX < 0) (*flip_ptr) ^= HORI_EVEN;
                  if (multY < 0) (*flip_ptr) ^= VERT_EVEN;
                  break;
               case ROTATE90:
               case ROTATE270:
                  if (multX < 0) (*flip_ptr) ^= HORI_ODD;
                  if (multY < 0) (*flip_ptr) ^= VERT_ODD;
                  break;
            }
         }
         if (ObjPtr->type == OBJ_ICON &&
               (icon_is_rigid = (!InsideIcon && ObjRigid (ObjPtr))))
         {
            xc = HALF_W(ObjPtr->obbox.ltx + ObjPtr->obbox.rbx);
            yc = HALF_W(ObjPtr->obbox.lty + ObjPtr->obbox.rby);
            StretchedAbsXY (xc, yc, &new_x, &new_y);
            dx = new_x - xc;
            dy = new_y - yc;
            MoveObj (ObjPtr, dx, dy);
         }
         else
         {
            ObjPtr->obbox.ltx = ObjPtr->x = real_ltx;
            ObjPtr->obbox.lty = ObjPtr->y = real_lty;
            ObjPtr->obbox.rbx = real_rbx;
            ObjPtr->obbox.rby = real_rby;
         }
         break;
      case OBJ_TEXT:
         ObjPtr->x = x;
         ObjPtr->y = y;
         if (multX < 0)
            ObjPtr->detail.t->just = MAXJUSTS - 1 - ObjPtr->detail.t->just;
         if (multY < 0) ObjPtr->y -= h;
         UpdTextBBox (ObjPtr);
         break;
   }

   switch (ObjPtr->type)
   {
      case OBJ_POLY:
         for (i = 0; i < ObjPtr->detail.p->n; i++)
         {
            StretchedAbsXY (ObjPtr->detail.p->vlist[i].x,
                  ObjPtr->detail.p->vlist[i].y, &x, &y);
            ObjPtr->detail.p->vlist[i].x = x;
            ObjPtr->detail.p->vlist[i].y = y;
            if (i == 0)
            {
               ltx = rbx = x;
               lty = rby = y;
            }
            else
            {
               if (x < ltx) ltx = x; if (y < lty) lty = y;
               if (x > rbx) rbx = x; if (y > rby) rby = y;
            }
         }
         ObjPtr->obbox.ltx = ObjPtr->x = ltx;
         ObjPtr->obbox.lty = ObjPtr->y = lty;
         ObjPtr->obbox.rbx = rbx;
         ObjPtr->obbox.rby = rby;
         AdjObjSplineVs (ObjPtr);
         if (ObjPtr->detail.p->curved == LT_INTSPLINE)
            UpdPolyBBox (ObjPtr, ObjPtr->detail.p->intn,
                  ObjPtr->detail.p->intvlist);
         StretchAttr (ObjPtr, Corner, InsideIcon, auto_center_attr);
         break;
      case OBJ_ARC:
         arc_ptr =  ObjPtr->detail.a;

         StretchedAbsXY (arc_ptr->xc, arc_ptr->yc, &x, &y);
         arc_ptr->xc = ObjPtr->x = x;
         arc_ptr->yc = ObjPtr->y = y;
         StretchedAbsXY (arc_ptr->x1, arc_ptr->y1, &x, &y);
         arc_ptr->x1 = x;
         arc_ptr->y1 = y;
         StretchedAbsXY (arc_ptr->x2, arc_ptr->y2, &x, &y);
         arc_ptr->x2 = x;
         arc_ptr->y2 = y;
         StretchedAbsXY (arc_ptr->ltx, arc_ptr->lty, &x, &y);
         arc_ptr->ltx = arc_ptr->xc-abs(x-arc_ptr->xc);
         arc_ptr->lty = arc_ptr->yc-abs(y-arc_ptr->yc);
         arc_ptr->w = (arc_ptr->xc-arc_ptr->ltx)<<1;
         arc_ptr->h = (arc_ptr->yc-arc_ptr->lty)<<1;

         if (multX < 0)
         {
            arc_ptr->dir = !(arc_ptr->dir);
            arc_ptr->angle2 = -(arc_ptr->angle2);
            if (arc_ptr->angle1 > 0)
               arc_ptr->angle1 = (180*64) - arc_ptr->angle1;
            else
               arc_ptr->angle1 = (-180*64) - arc_ptr->angle1;
         }
         if (multY < 0)
         {
            arc_ptr->dir = !(arc_ptr->dir);
            arc_ptr->angle1 = -(arc_ptr->angle1);
            arc_ptr->angle2 = -(arc_ptr->angle2);
         }
         UpdArcBBox (ObjPtr);
         StretchAttr (ObjPtr, Corner, InsideIcon, auto_center_attr);
         break;
      case OBJ_RCBOX:
         StretchAttr (ObjPtr, Corner, InsideIcon, auto_center_attr);
         break;
      case OBJ_BOX:
         StretchAttr (ObjPtr, Corner, InsideIcon, auto_center_attr);
         break;
      case OBJ_OVAL:
         StretchAttr (ObjPtr, Corner, InsideIcon, auto_center_attr);
         break;
      case OBJ_TEXT: break;
      case OBJ_POLYGON:
         for (i = 0; i < ObjPtr->detail.g->n; i++)
         {
            StretchedAbsXY (ObjPtr->detail.g->vlist[i].x,
                  ObjPtr->detail.g->vlist[i].y, &x, &y);
            ObjPtr->detail.g->vlist[i].x = x;
            ObjPtr->detail.g->vlist[i].y = y;
            if (i == 0)
            {
               ltx = rbx = x;
               lty = rby = y;
            }
            else
            {
               if (x < ltx) ltx = x; if (y < lty) lty = y;
               if (x > rbx) rbx = x; if (y > rby) rby = y;
            }
         }
         ObjPtr->obbox.ltx = ObjPtr->x = ltx;
         ObjPtr->obbox.lty = ObjPtr->y = lty;
         ObjPtr->obbox.rbx = rbx;
         ObjPtr->obbox.rby = rby;
         AdjObjSplineVs (ObjPtr);
         if (ObjPtr->detail.g->curved == LT_INTSPLINE)
            UpdPolyBBox (ObjPtr, ObjPtr->detail.g->intn,
                  ObjPtr->detail.g->intvlist);
         StretchAttr (ObjPtr, Corner, InsideIcon, auto_center_attr);
         break;
      case OBJ_SYM:
      case OBJ_GROUP:
         for (ptr = ObjPtr->detail.r->first; ptr != NULL; ptr = ptr->next)
            StretchObj (ptr, Corner, InsideIcon);
         StretchAttr (ObjPtr, Corner, InsideIcon, auto_center_attr);
         break;
      case OBJ_ICON:
         if (!icon_is_rigid)
         {
            for (ptr = ObjPtr->detail.r->first; ptr != NULL; ptr = ptr->next)
               StretchObj (ptr, Corner, TRUE);
            StretchAttr (ObjPtr, Corner, InsideIcon, auto_center_attr);
         }
         break;
      case OBJ_XBM:
         if (ObjPtr->detail.xbm->cached_bitmap != None)
            XFreePixmap (mainDisplay, ObjPtr->detail.xbm->cached_bitmap);
         ObjPtr->detail.xbm->cached_bitmap = None;

         if (zoomScale != 0)
            ObjPtr->detail.xbm->cached_zoom = 0;
         else
            ObjPtr->detail.xbm->cached_rotate = INVALID;
         StretchAttr (ObjPtr, Corner, InsideIcon, auto_center_attr);
         break;
      case OBJ_XPM:
         if (ObjPtr->detail.xpm->cached_pixmap != None)
            XFreePixmap (mainDisplay, ObjPtr->detail.xpm->cached_pixmap);
         ObjPtr->detail.xpm->cached_pixmap = None;

         if (zoomScale != 0)
            ObjPtr->detail.xpm->cached_zoom = 0;
         else
            ObjPtr->detail.xpm->cached_rotate = INVALID;
         StretchAttr (ObjPtr, Corner, InsideIcon, auto_center_attr);
         break;
   }
   AdjObjBBox (ObjPtr);
}

static
void StretchAllSelObjects (Corner)
   int	Corner;
{
   struct SelRec	* sel_ptr;

   for (sel_ptr = topSel; sel_ptr != NULL; sel_ptr = sel_ptr->next)
      if (!sel_ptr->obj->locked)
         StretchObj (sel_ptr->obj, Corner, FALSE);
   if (numObjLocked != 0) Msg ("Locked objects are not stretched.");
}

static
void MarkObjectsForStretch ()
{
   register struct ObjRec	* obj_ptr;
   register struct SelRec	* sel_ptr;

   for (obj_ptr = botObj; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
      obj_ptr->marked = FALSE;

   for (sel_ptr = botSel; sel_ptr != NULL; sel_ptr = sel_ptr->prev)
      sel_ptr->obj->marked = TRUE;
}

static
int ConstrainedStretchAllSel (Corner, ltx, lty, rbx, rby)
   int	Corner;
   int	* ltx, * lty, * rbx, * rby;
{
   register struct ObjRec	* obj_ptr;
   int				something_stretched=FALSE, num_pts;
   int				x_off, y_off, move_first, move_last, x, y;
   XPoint			* v;

   for (obj_ptr = botObj; obj_ptr != NULL; obj_ptr = obj_ptr->prev)
   {
      if (!obj_ptr->marked && obj_ptr->type==OBJ_POLY && !obj_ptr->locked)
      {
         num_pts = obj_ptr->detail.p->n;
         v = obj_ptr->detail.p->vlist;

         x_off = OFFSET_X(v[0].x); y_off = OFFSET_Y(v[0].y);
         move_first = EndPtInSelected (x_off, y_off);
         x_off = OFFSET_X(v[num_pts-1].x); y_off = OFFSET_Y(v[num_pts-1].y);
         move_last = EndPtInSelected (x_off, y_off);

         if (move_first || move_last)
         {
            int	index=INVALID, seg_dx, seg_dy, dx, dy;
            int	cur_seg_dx, cur_seg_dy;

            PrepareToReplaceAnObj (obj_ptr);
            if (something_stretched)
            {
               if (obj_ptr->bbox.ltx < *ltx) *ltx = obj_ptr->bbox.ltx;
               if (obj_ptr->bbox.lty < *lty) *lty = obj_ptr->bbox.lty;
               if (obj_ptr->bbox.rbx > *rbx) *rbx = obj_ptr->bbox.rbx;
               if (obj_ptr->bbox.rby > *rby) *rby = obj_ptr->bbox.rby;
            }
            else
            {
               *ltx = obj_ptr->bbox.ltx; *lty = obj_ptr->bbox.lty;
               *rbx = obj_ptr->bbox.rbx; *rby = obj_ptr->bbox.rby;
            }
            something_stretched = TRUE;
            if (move_first && move_last && num_pts==3)
            {
               StretchedAbsXY (v[0].x, v[0].y, &x, &y);
               dx = x-v[0].x; dy = y-v[0].y;
               index = 1;
               cur_seg_dx = v[index-1].x - v[index].x;
               cur_seg_dy = v[index-1].y - v[index].y;
               seg_dx = v[index].x - v[index+1].x;
               seg_dy = v[index].y - v[index+1].y;

               if (cur_seg_dy==0 && seg_dx==0 &&
                     (seg_dy!=0 || seg_dy==0 && dx==0))
                  v[index].y += dy;
               else if (cur_seg_dx==0 && seg_dy==0 &&
                     (seg_dx!=0 || seg_dx==0 && dy==0))
                  v[index].x += dx;
            }
            else
            {
               if (move_first && num_pts>2)
               {
                  StretchedAbsXY (v[0].x, v[0].y, &x, &y);
                  dx = x-v[0].x; dy = y-v[0].y;
                  index = 1;
                  cur_seg_dx = v[index-1].x - v[index].x;
                  cur_seg_dy = v[index-1].y - v[index].y;
                  seg_dx = v[index].x - v[index+1].x;
                  seg_dy = v[index].y - v[index+1].y;

                  if (cur_seg_dy==0 && cur_seg_dx!=0 &&
                        (seg_dy!=0 || seg_dy==0 && dx==0))
                     v[index].y += dy;
                  else if (cur_seg_dx==0 && cur_seg_dy!=0 &&
                        (seg_dx!=0 || seg_dx==0 && dy==0))
                     v[index].x += dx;
               }
               if (move_last && num_pts>2)
               {
                  StretchedAbsXY (v[num_pts-1].x, v[num_pts-1].y, &x, &y);
                  dx = x-v[num_pts-1].x; dy = y-v[num_pts-1].y;
                  index = num_pts-2;
                  cur_seg_dx = v[index+1].x - v[index].x;
                  cur_seg_dy = v[index+1].y - v[index].y;
                  seg_dx = v[index].x - v[index-1].x;
                  seg_dy = v[index].y - v[index-1].y;

                  if (cur_seg_dy==0 && cur_seg_dx!=0 &&
                        (seg_dy!=0 || seg_dy==0 && dx==0))
                     v[index].y += dy;
                  else if (cur_seg_dx==0 && cur_seg_dy!=0 &&
                        (seg_dx!=0 || seg_dx==0 && dy==0))
                     v[index].x += dx;
               }
            }
            if (move_first)
            {
               StretchedAbsXY (v[0].x, v[0].y, &x, &y);
               v[0].x = x; v[0].y = y;
            }
            if (move_last)
            {
               StretchedAbsXY (v[num_pts-1].x, v[num_pts-1].y, &x, &y);
               v[num_pts-1].x = x; v[num_pts-1].y = y;
            }
            AdjObjSplineVs (obj_ptr);
            switch (obj_ptr->type)
            {
               case OBJ_POLY:
                  if (obj_ptr->detail.p->curved != LT_INTSPLINE)
                     UpdPolyBBox (obj_ptr, num_pts, v);
                  else
                     UpdPolyBBox (obj_ptr, obj_ptr->detail.p->intn,
                           obj_ptr->detail.p->intvlist);
                  break;
               case OBJ_POLYGON:
                  if (obj_ptr->detail.g->curved != LT_INTSPLINE)
                     UpdPolyBBox (obj_ptr, num_pts, v);
                  else
                     UpdPolyBBox (obj_ptr, obj_ptr->detail.g->intn,
                           obj_ptr->detail.g->intvlist);
                  break;
            }
            AdjObjBBox (obj_ptr);
            if (AutoCenterAttr (obj_ptr))
            {
               struct AttrRec	* attr_ptr=obj_ptr->fattr;
               int		modified=FALSE;

               for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
                  if (attr_ptr->shown)
                  {
                     struct BBRec	bbox;

                     CenterObjInOBBox (attr_ptr->obj, obj_ptr->obbox, &bbox);
                     if (bbox.ltx < *ltx) *ltx = bbox.ltx;
                     if (bbox.lty < *lty) *lty = bbox.lty;
                     if (bbox.rbx > *rbx) *rbx = bbox.rbx;
                     if (bbox.rby > *rby) *rby = bbox.rby;
                     modified = TRUE;
                  }
               if (modified) AdjObjBBox (obj_ptr);
            }
            if (obj_ptr->bbox.ltx < *ltx) *ltx = obj_ptr->bbox.ltx;
            if (obj_ptr->bbox.lty < *lty) *lty = obj_ptr->bbox.lty;
            if (obj_ptr->bbox.rbx > *rbx) *rbx = obj_ptr->bbox.rbx;
            if (obj_ptr->bbox.rby > *rby) *rby = obj_ptr->bbox.rby;
            RecordReplaceAnObj (obj_ptr);
         }
      }
   }
   return (something_stretched);
}

static
void StretchAllSel (Corner)
   int	Corner;
{
   int	ltx, lty, rbx, rby, saved_ltx, saved_lty, saved_rbx, saved_rby;
   int	poly_stretched;

   saved_ltx = selLtX; saved_lty = selLtY;
   saved_rbx = selRbX; saved_rby = selRbY;

   if (moveMode==CONST_MOVE)
   {
      MarkObjectsForStretch ();

      StartCompositeCmd ();
      PrepareToRecord (CMD_STRETCH, topSel, botSel, numObjSelected);
      RecordCmd (CMD_STRETCH, NULL, topSel, botSel, numObjSelected);

      poly_stretched = ConstrainedStretchAllSel (Corner,&ltx,&lty,&rbx,&rby);
      StretchAllSelObjects (Corner);
      if (poly_stretched)
      {
         UpdSelBBox ();
         ltx = min(ltx,min(selLtX,saved_ltx));
         lty = min(lty,min(selLtY,saved_lty));
         rbx = max(rbx,max(selRbX,saved_rbx));
         rby = max(rby,max(selRbY,saved_rby));
         RedrawAnArea (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
               rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1));
      }
      else
      {
         UpdSelBBox ();
         RedrawAreas (botObj, saved_ltx-GRID_ABS_SIZE(1),
               saved_lty-GRID_ABS_SIZE(1),
               saved_rbx+GRID_ABS_SIZE(1), saved_rby+GRID_ABS_SIZE(1),
               selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
               selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1));
      }
      EndCompositeCmd ();
   }
   else
   {
      PrepareToRecord (CMD_STRETCH, topSel, botSel, numObjSelected);
      StretchAllSelObjects (Corner);
      RecordCmd (CMD_STRETCH, NULL, topSel, botSel, numObjSelected);
      UpdSelBBox ();
      RedrawAreas (botObj, saved_ltx-GRID_ABS_SIZE(1),
            saved_lty-GRID_ABS_SIZE(1),
            saved_rbx+GRID_ABS_SIZE(1), saved_rby+GRID_ABS_SIZE(1),
            selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
            selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1));
   }
}

static
void GetMeasurement (ObjPtr, buf)
   struct ObjRec	* ObjPtr;
   char			* buf;
{
   int	ltx, lty, rbx, rby, real_ltx, real_lty, real_rbx, real_rby;

   StretchedAbsXY (ObjPtr->obbox.ltx, ObjPtr->obbox.lty, &ltx, &lty);
   StretchedAbsXY (ObjPtr->obbox.rbx, ObjPtr->obbox.rby, &rbx, &rby);
   CalcBBox (ltx, lty, rbx, rby, &real_ltx, &real_lty, &real_rbx, &real_rby);
   sprintf (buf, "%1dx%1d", abs(real_rbx-real_ltx), abs(real_rby-real_lty));
}

static
void StretchBox (XGridOff, YGridOff, ObjPtr, Corner)
   int			XGridOff, YGridOff, Corner;
   struct ObjRec	* ObjPtr;
{
   int		x, y, stretching=TRUE;
   int		ltx, lty, rbx, rby, sel_ltx, sel_lty, sel_rbx, sel_rby;
   int		stretched_ltx, stretched_lty, stretched_rbx, stretched_rby;
   int		stretched_sel_ltx, stretched_sel_lty, stretched_sel_rbx;
   int		stretched_sel_rby;
   int		ruler_ltx, ruler_lty, ruler_rbx, ruler_rby;
   int		sel_obj_ltx, sel_obj_lty, sel_obj_rbx, sel_obj_rby;
   int		grid_x = XGridOff, grid_y = YGridOff;
   char		buf[80];
   double	obj_w, obj_h;
   XEvent	input, ev;

   if (numObjSelected == numObjLocked)
   {
      Msg ("Locked object(s) can not be stretched.");
      return;
   }

   XFlush (mainDisplay);
   XSync (mainDisplay, False);

   if (XCheckMaskEvent (mainDisplay, ExposureMask, &ev) ||
         XCheckMaskEvent (mainDisplay, VisibilityChangeMask, &ev))
      ExposeEventHandler (&ev, TRUE);

   SetPivot (Corner, ObjPtr->obbox);

   stretched_sel_ltx = sel_ltx = OFFSET_X(selLtX);
   stretched_sel_lty = sel_lty = OFFSET_Y(selLtY);
   stretched_sel_rbx = sel_rbx = OFFSET_X(selRbX);
   stretched_sel_rby = sel_rby = OFFSET_Y(selRbY);
   SelBox (drawWindow, revDefaultGC, stretched_sel_ltx-2, stretched_sel_lty-2,
         stretched_sel_rbx+2, stretched_sel_rby+2);

   ruler_ltx = sel_obj_ltx = OFFSET_X(selObjLtX);
   ruler_lty = sel_obj_lty = OFFSET_Y(selObjLtY);
   ruler_rbx = sel_obj_rbx = OFFSET_X(selObjRbX);
   ruler_rby = sel_obj_rby = OFFSET_Y(selObjRbY);

   stretched_ltx = ltx = OFFSET_X(ObjPtr->obbox.ltx);
   stretched_lty = lty = OFFSET_Y(ObjPtr->obbox.lty);
   stretched_rbx = rbx = OFFSET_X(ObjPtr->obbox.rbx);
   stretched_rby = rby = OFFSET_Y(ObjPtr->obbox.rby);
   SelBox (drawWindow, revDefaultGC, stretched_ltx, stretched_lty,
         stretched_rbx, stretched_rby);

   if (ltx == rbx)
   {
      Msg ("Can not stretch!  Object has ZERO width!");
      SelBox (drawWindow, revDefaultGC, stretched_ltx, stretched_lty,
            stretched_rbx, stretched_rby);
      SelBox (drawWindow, revDefaultGC, stretched_sel_ltx-2,
            stretched_sel_lty-2, stretched_sel_rbx+2, stretched_sel_rby+2);
      return;
   }
   else if (lty == rby)
   {
      Msg ("Can not stretch!  Object has ZERO height!");
      SelBox (drawWindow, revDefaultGC, stretched_ltx, stretched_lty,
            stretched_rbx, stretched_rby);
      SelBox (drawWindow, revDefaultGC, stretched_sel_ltx-2,
            stretched_sel_lty-2, stretched_sel_rbx+2, stretched_sel_rby+2);
      return;
   }

   obj_w = (double)(moveX - pivotX);
   obj_h = (double)(moveY - pivotY);

   sprintf (buf, "%1dx%1d", ObjPtr->obbox.rbx-ObjPtr->obbox.ltx,
         ObjPtr->obbox.rby-ObjPtr->obbox.lty);
   StartShowMeasureCursor (grid_x, grid_y, buf, TRUE);
   BeginIntervalRulers (ruler_ltx, ruler_lty, ruler_rbx, ruler_rby);
   XGrabPointer (mainDisplay, drawWindow, False,
         PointerMotionMask | ButtonReleaseMask,
         GrabModeAsync, GrabModeAsync, None, handCursor, CurrentTime);

   while (stretching)
   {
      XNextEvent (mainDisplay, &input);

      if (input.type == Expose || input.type == VisibilityNotify)
         ExposeEventHandler (&input, TRUE);
      else if (input.type == ButtonRelease)
      {
         XUngrabPointer (mainDisplay, CurrentTime);
         stretching = FALSE;
      }
      else if (input.type == MotionNotify)
      {
         int	proportional = input.xmotion.state & (ShiftMask|ControlMask);

         GetMeasurement (ObjPtr, buf);
         EndShowMeasureCursor (grid_x, grid_y, buf, TRUE);

         x = input.xmotion.x;
         y = input.xmotion.y;
         GridXY (x, y, &grid_x, &grid_y);

         SelBox (drawWindow, revDefaultGC, stretched_ltx, stretched_lty,
               stretched_rbx, stretched_rby);
         SelBox (drawWindow, revDefaultGC, stretched_sel_ltx-2,
               stretched_sel_lty-2, stretched_sel_rbx+2, stretched_sel_rby+2);

         if (proportional)
         {
            int		new_w, new_h;
            double	w_ratio, h_ratio;

            new_w = moveX + grid_x - XGridOff - pivotX;
            new_h = moveY + grid_y - YGridOff - pivotY;
            w_ratio = (moveX!=pivotX) ? fabs(((double)new_w)/obj_w) : 0.0;
            h_ratio = (moveY!=pivotY) ? fabs(((double)new_h)/obj_h) : 0.0;
            if (changeX && changeY)
            {
               if (w_ratio >= h_ratio)
               {
                  multX = (moveX!=pivotX) ? ((double)new_w)/obj_w : 1.0;
                  multY = fabs(multX) * ((new_h*obj_h>=0) ? 1.0 : -1.0);
               }
               else
               {
                  multX = fabs(multY) * ((new_w*obj_w>=0) ? 1.0 : -1.0);
                  multY = (moveY!=pivotY) ? ((double)new_h)/obj_h : 1.0;
               }
            }
            else if (changeX)
            {
               multX = (moveX!=pivotX) ? ((double)new_w)/obj_w : 1.0;
               multY = fabs(multX);
            }
            else if (changeY)
            {
               multX = fabs(multY);
               multY = (moveY!=pivotY) ? ((double)new_h)/obj_h : 1.0;
            }
         }
         else
         {
            if (changeX)
               multX = (moveX!=pivotX) ?
                     (double)(moveX+grid_x-XGridOff-pivotX)/obj_w : 1.0;
            else
               multX = (double)1.0;
            if (changeY)
               multY = (moveY!=pivotY) ?
                     (double)(moveY+grid_y-YGridOff-pivotY)/obj_h : 1.0;
            else
               multY = (double)1.0;
         }

         StretchedXY (sel_ltx, sel_lty, &stretched_sel_ltx, &stretched_sel_lty);
         StretchedXY (sel_rbx, sel_rby, &stretched_sel_rbx, &stretched_sel_rby);
         StretchedXY (ltx, lty, &stretched_ltx, &stretched_lty);
         StretchedXY (rbx, rby, &stretched_rbx, &stretched_rby);
         StretchedXY (sel_obj_ltx, sel_obj_lty, &ruler_ltx, &ruler_lty);
         StretchedXY (sel_obj_rbx, sel_obj_rby, &ruler_rbx, &ruler_rby);

         DrawIntervalRulers (ruler_ltx, ruler_lty, ruler_rbx, ruler_rby);
         SelBox (drawWindow, revDefaultGC, stretched_sel_ltx-2,
               stretched_sel_lty-2, stretched_sel_rbx+2, stretched_sel_rby+2);
         SelBox (drawWindow, revDefaultGC, stretched_ltx, stretched_lty,
               stretched_rbx, stretched_rby);
         GetMeasurement (ObjPtr, buf);
         EndShowMeasureCursor (grid_x, grid_y, buf, TRUE);
         while (XCheckMaskEvent (mainDisplay, PointerMotionMask, &ev)) ;
      }
   }
   EndIntervalRulers (grid_x, grid_y);
   GetMeasurement (ObjPtr, buf);
   EndShowMeasureCursor (grid_x, grid_y, buf, TRUE);
   SelBox (drawWindow, revDefaultGC, stretched_ltx, stretched_lty,
         stretched_rbx, stretched_rby);
   SelBox (drawWindow, revDefaultGC, stretched_sel_ltx-2,
         stretched_sel_lty-2, stretched_sel_rbx+2, stretched_sel_rby+2);
   if (multX != (double)1.0 || multY != (double)1.0)
   {
      HighLightReverse ();
      StretchAllSel (Corner);
      HighLightForward ();
      SetFileModified (TRUE);
      justDupped = FALSE;
   }
}

void StretchSel (XGridOff, YGridOff, ObjPtr, Corner)
   int			XGridOff, YGridOff, Corner;
   struct ObjRec	* ObjPtr;
{
   switch (ObjPtr->type)
   {
      case OBJ_BOX:
      case OBJ_OVAL:
      case OBJ_GROUP:
      case OBJ_ICON:
      case OBJ_ARC:
      case OBJ_RCBOX:
      case OBJ_SYM:
      case OBJ_XBM:
      case OBJ_XPM:
         StretchBox (XGridOff, YGridOff, ObjPtr, Corner);
         break;
      case OBJ_POLY:
         StretchPoly (XGridOff, YGridOff, ObjPtr, ObjPtr->detail.p->n,
               ObjPtr->detail.p->vlist, Corner);
         break;
      case OBJ_POLYGON:
         StretchPoly (XGridOff, YGridOff, ObjPtr, ObjPtr->detail.g->n,
               ObjPtr->detail.g->vlist, Corner);
         break;
      case OBJ_TEXT: break;
   }
}

void ScaleAnEPSObj (ObjPtr, ScalingFactor)
   struct ObjRec	* ObjPtr;
   float		* ScalingFactor;
{
   struct BBRec	* obbox = &(ObjPtr->obbox);

   multX = multY = (double) (*ScalingFactor);
   changeX = changeY = (fabs(multX-1.0) > 1.0e-6);
   if (!changeX && !changeY) return;;

   absPivotX = obbox->ltx;
   absPivotY = obbox->lty;
   moveX = obbox->rbx;
   moveY = obbox->rby;
   StretchObj (ObjPtr, CORNER_RB, FALSE);
}

static
char * FindColon(s)
   register char	* s;
{
   while (*s!=':' && *s!='x' && *s!='X' && *s!=' ' && *s!='\0') s++;
   return ((*s==':' || *s=='x' || *s=='X' || *s==' ') ? (s) : (char *)NULL);
}

static
void ScaleAllSelObjects (Corner)
   int	Corner;
{
   register struct SelRec	* sel_ptr;
   register struct BBRec	* obbox;

   for (sel_ptr = topSel; sel_ptr != NULL; sel_ptr = sel_ptr->next)
      if (!sel_ptr->obj->locked)
      {
         obbox = &(sel_ptr->obj->obbox);
         switch (Corner)
         {
            case CORNER_CC:
               moveX = absPivotX = (obbox->ltx+obbox->rbx)>>1;
               moveY = absPivotY = (obbox->lty+obbox->rby)>>1;
               break;
            case CORNER_LT:
               absPivotX = obbox->rbx; absPivotY = obbox->rby;
               moveX = obbox->ltx; moveY = obbox->lty;
               break;
            case CORNER_T:
               moveX = absPivotX = (obbox->ltx+obbox->rbx)>>1;
               absPivotY = obbox->rby;
               moveY = obbox->lty;
               break;
            case CORNER_RT:
               absPivotX = obbox->ltx; absPivotY = obbox->rby;
               moveX = obbox->rbx; moveY = obbox->lty;
               break;
            case CORNER_R:
               absPivotX = obbox->ltx;
               moveX = obbox->rbx;
               moveY = absPivotY = (obbox->lty+obbox->rby)>>1;
               break;
            case CORNER_RB:
               absPivotX = obbox->ltx; absPivotY = obbox->lty;
               moveX = obbox->rbx; moveY = obbox->rby;
               break;
            case CORNER_B:
               moveX = absPivotX = (obbox->ltx+obbox->rbx)>>1;
               absPivotY = obbox->lty;
               moveY = obbox->rby;
               break;
            case CORNER_LB:
               absPivotX = obbox->rbx; absPivotY = obbox->lty;
               moveX = obbox->ltx; moveY = obbox->rby;
               break;
            case CORNER_L:
               absPivotX = obbox->rbx;
               moveX = obbox->ltx;
               moveY = absPivotY = (obbox->lty+obbox->rby)>>1;
               break;
         }
         StretchObj (sel_ptr->obj, Corner, FALSE);
      }
   if (numObjLocked != 0) Msg ("Locked objects are not scaled.");
}

static
void ScaleAllSel (Corner)
   int	Corner;
{
   int	ltx, lty, rbx, rby, saved_ltx, saved_lty, saved_rbx, saved_rby;
   int	poly_stretched;

   saved_ltx = selLtX; saved_lty = selLtY;
   saved_rbx = selRbX; saved_rby = selRbY;

   if (moveMode==CONST_MOVE)
   {
      MarkObjectsForStretch ();

      StartCompositeCmd ();
      PrepareToRecord (CMD_STRETCH, topSel, botSel, numObjSelected);
      RecordCmd (CMD_STRETCH, NULL, topSel, botSel, numObjSelected);

      poly_stretched = ConstrainedStretchAllSel (Corner,&ltx,&lty,&rbx,&rby);
      ScaleAllSelObjects (Corner);
      if (poly_stretched)
      {
         UpdSelBBox ();
         ltx = min(ltx,min(selLtX,saved_ltx));
         lty = min(lty,min(selLtY,saved_lty));
         rbx = max(rbx,max(selRbX,saved_rbx));
         rby = max(rby,max(selRbY,saved_rby));
         RedrawAnArea (botObj, ltx-GRID_ABS_SIZE(1), lty-GRID_ABS_SIZE(1),
               rbx+GRID_ABS_SIZE(1), rby+GRID_ABS_SIZE(1));
      }
      else
      {
         UpdSelBBox ();
         RedrawAreas (botObj, saved_ltx-GRID_ABS_SIZE(1),
               saved_lty-GRID_ABS_SIZE(1),
               saved_rbx+GRID_ABS_SIZE(1), saved_rby+GRID_ABS_SIZE(1),
               selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
               selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1));
      }
      EndCompositeCmd ();
   }
   else
   {
      PrepareToRecord (CMD_STRETCH, topSel, botSel, numObjSelected);
      ScaleAllSelObjects (Corner);
      RecordCmd (CMD_STRETCH, NULL, topSel, botSel, numObjSelected);
      UpdSelBBox ();
      RedrawAreas (botObj, saved_ltx-GRID_ABS_SIZE(1),
            saved_lty-GRID_ABS_SIZE(1),
            saved_rbx+GRID_ABS_SIZE(1), saved_rby+GRID_ABS_SIZE(1),
            selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
            selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1));
   }
}

void ScaleAllSelObj ()
{
   char	spec[MAXSTRING], * y_spec;
   int	corner=INVALID;

   if (topSel == NULL) return;
   if (numObjSelected == numObjLocked)
   {
      Msg ("Locked objects can not be scaled.");
      return;
   }

   Dialog ("Please enter scaling factor: [X:Y] or [Scale]",
         "( <CR>: accept, <ESC>: cancel )", spec);
   if (*spec == '\0') return;

   if ((y_spec = FindColon (spec)) == NULL)
   {
      sscanf (spec, "%lf", &multX);
      if (multX <= 0.0)
      {
         Msg ("Invalid scaling specification.");
         return;
      }
      multY = multX;
   }
   else
   {
      *y_spec++ = '\0';
      sscanf (spec, "%lf", &multX);
      sscanf (y_spec, "%lf", &multY);
      if (multX <= 0.0 || multY <= 0.0)
      {
         Msg ("Invalid scaling specification.");
         return;
      }
   }
   changeX = (fabs(multX-1.0) > 1.0e-6);
   changeY = (fabs(multY-1.0) > 1.0e-6);
   if (!changeX && !changeY) return;

   switch (horiAlign)
   {
      case ALIGN_N:
      case ALIGN_S:
         switch (vertAlign)
         {
            case ALIGN_N: corner = CORNER_CC; break;
            case ALIGN_S: corner = CORNER_CC; break;
            case ALIGN_T: corner = CORNER_B; break;
            case ALIGN_M: corner = CORNER_CC; break;
            case ALIGN_B: corner = CORNER_T; break;
         }
         break;
      case ALIGN_L:
         switch (vertAlign)
         {
            case ALIGN_N: corner = CORNER_R; break;
            case ALIGN_S: corner = CORNER_R; break;
            case ALIGN_T: corner = CORNER_RB; break;
            case ALIGN_M: corner = CORNER_R; break;
            case ALIGN_B: corner = CORNER_RT; break;
         }
         break;
      case ALIGN_C:
         switch (vertAlign)
         {
            case ALIGN_N: corner = CORNER_CC; break;
            case ALIGN_S: corner = CORNER_CC; break;
            case ALIGN_T: corner = CORNER_B; break;
            case ALIGN_M: corner = CORNER_CC; break;
            case ALIGN_B: corner = CORNER_T; break;
         }
         break;
      case ALIGN_R:
         switch (vertAlign)
         {
            case ALIGN_N: corner = CORNER_L; break;
            case ALIGN_S: corner = CORNER_L; break;
            case ALIGN_T: corner = CORNER_LB; break;
            case ALIGN_M: corner = CORNER_L; break;
            case ALIGN_B: corner = CORNER_LT; break;
         }
         break;
   }

   HighLightReverse ();
   ScaleAllSel (corner);
   HighLightForward ();
   SetFileModified (TRUE);
   justDupped = FALSE;
}

/*
 * The following code is INCORRECT because redo does not work!
 *	The main problem with redo is that for a composite cmd,
 *	if the first command inside the composite is a move or a
 *	stretch, it is treated as if the move comes from the side
 *	effect of a constrained move!
 */

void MoveSizeAllSelObj(AbsX, AbsY, AbsW, AbsH)
   int AbsX, AbsY, AbsW, AbsH;
   /* These are supposed to be points corresponding to obbox */
{
   int move=FALSE, stretch=FALSE;

   if (topSel == NULL) return;

   if (AbsX != selObjLtX || AbsY != selObjLtY) move = TRUE;
   if (AbsW != selObjRbX-selObjLtX || AbsH != selObjRbY-selObjLtY) {
      stretch = TRUE;
   }
   if (!move && !stretch) return;

   if (move && stretch) {
      StartCompositeCmd();
   }
   if (move) {
      MoveAllSel(AbsX-selObjLtX, AbsY-selObjLtY);
      UpdSelBBox();
   }
   if (stretch) {
      int saved_h_align=horiAlign, saved_v_align=vertAlign;

      horiAlign = ALIGN_L;
      vertAlign = ALIGN_T;
      multX = ((double)AbsW) / ((double)selObjRbX-selObjLtX);
      multY = ((double)AbsH) / ((double)selObjRbY-selObjLtY);
      changeX = (fabs(multX-1.0) > 1.0e-6);
      changeY = (fabs(multY-1.0) > 1.0e-6);
      ScaleAllSel(CORNER_RB);
      horiAlign = saved_h_align;
      vertAlign = saved_v_align;
   }
   if (move && stretch) {
      EndCompositeCmd();
   }
   UpdSelBBox();
   SetFileModified(TRUE);
   justDupped = FALSE;
}

void FlipObjHorizontal (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   register XPoint	* v;
   register int		i, two_x_pivot;
   int			new_obj_ltx, new_obj_rbx, num_pts;
   struct ObjRec	* obj_ptr;
   struct AttrRec	* attr_ptr;
   struct ArcRec	* arc_ptr;

   SetFileModified (TRUE);
   two_x_pivot = selObjLtX + selObjRbX;
   new_obj_ltx = two_x_pivot - ObjPtr->obbox.rbx;
   new_obj_rbx = two_x_pivot - ObjPtr->obbox.ltx;
   switch (ObjPtr->type)
   {
      case OBJ_BOX:
      case OBJ_OVAL:
      case OBJ_POLY:
      case OBJ_POLYGON:
      case OBJ_ARC:
      case OBJ_RCBOX:
      case OBJ_XBM:
      case OBJ_XPM:
      case OBJ_GROUP:
      case OBJ_SYM:
      case OBJ_ICON:
         switch (ObjPtr->type)
         {
            case OBJ_XBM:
               switch (ObjPtr->detail.xbm->rotate)
               {
                  case ROTATE0:
                  case ROTATE180:
                     ObjPtr->detail.xbm->flip ^= HORI_EVEN;
                     break;
                  case ROTATE90:
                  case ROTATE270:
                     ObjPtr->detail.xbm->flip ^= HORI_ODD;
                     break;
               }
               if (ObjPtr->detail.xbm->cached_bitmap != None)
                  XFreePixmap (mainDisplay, ObjPtr->detail.xbm->cached_bitmap);
               ObjPtr->detail.xbm->cached_bitmap = None;

               if (zoomScale != 0)
                  ObjPtr->detail.xbm->cached_zoom = 0;
               else
                  ObjPtr->detail.xbm->cached_rotate = INVALID;
               break;
            case OBJ_XPM:
               switch (ObjPtr->detail.xpm->rotate)
               {
                  case ROTATE0:
                  case ROTATE180:
                     ObjPtr->detail.xpm->flip ^= HORI_EVEN;
                     break;
                  case ROTATE90:
                  case ROTATE270:
                     ObjPtr->detail.xpm->flip ^= HORI_ODD;
                     break;
               }
               if (ObjPtr->detail.xpm->cached_pixmap != None)
                  XFreePixmap (mainDisplay, ObjPtr->detail.xpm->cached_pixmap);
               ObjPtr->detail.xpm->cached_pixmap = None;

               if (zoomScale != 0)
                  ObjPtr->detail.xpm->cached_zoom = 0;
               else
                  ObjPtr->detail.xpm->cached_rotate = INVALID;
               break;
            case OBJ_ICON:
               switch (ObjPtr->detail.r->rotate)
               {
                  case ROTATE0:
                  case ROTATE180: ObjPtr->detail.r->flip ^= HORI_EVEN; break;

                  case ROTATE90:
                  case ROTATE270: ObjPtr->detail.r->flip ^= HORI_ODD; break;
               }
               break;
         }
         ObjPtr->obbox.ltx = ObjPtr->x = new_obj_ltx;
         ObjPtr->obbox.rbx = new_obj_rbx;
         break;
      case OBJ_TEXT:
         switch (ObjPtr->detail.t->rotate)
         {
            case ROTATE0:
            case ROTATE180:
               ObjPtr->x = two_x_pivot - ObjPtr->x;
               if (ObjPtr->detail.t->just != JUST_C)
               {
                  ObjPtr->detail.t->just = MAXJUSTS-1-ObjPtr->detail.t->just;
                  if (ObjPtr->detail.t->cached_bitmap != None)
                     XFreePixmap (mainDisplay, ObjPtr->detail.t->cached_bitmap);
                  ObjPtr->detail.t->cached_bitmap = None;

                  if (zoomScale != 0)
                     ObjPtr->detail.t->cached_zoom = 0;
                  else
                     ObjPtr->detail.t->cached_rotate = INVALID;
               }
               break;
            case ROTATE90: ObjPtr->x = new_obj_rbx; break;
            case ROTATE270: ObjPtr->x = new_obj_ltx; break;
         }
         UpdTextBBox (ObjPtr);
         break;
   }
   switch (ObjPtr->type)
   {
      case OBJ_POLY:
         num_pts = ObjPtr->detail.p->n;
         v = ObjPtr->detail.p->vlist;
         for (i = 0; i < num_pts; i++, v++) (*v).x = two_x_pivot - (*v).x;
         AdjObjSplineVs (ObjPtr);
         attr_ptr = ObjPtr->fattr;
         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            FlipObjHorizontal (attr_ptr->obj);
         break;
      case OBJ_BOX:
      case OBJ_OVAL:
      case OBJ_RCBOX:
      case OBJ_XBM:
      case OBJ_XPM:
         attr_ptr = ObjPtr->fattr;
         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            FlipObjHorizontal (attr_ptr->obj);
         break;
      case OBJ_TEXT: break;
      case OBJ_POLYGON:
         num_pts = ObjPtr->detail.g->n;
         v = ObjPtr->detail.g->vlist;
         for (i = 0; i < num_pts; i++, v++) (*v).x = two_x_pivot - (*v).x;
         AdjObjSplineVs (ObjPtr);
         attr_ptr = ObjPtr->fattr;
         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            FlipObjHorizontal (attr_ptr->obj);
         break;
      case OBJ_ARC:
         arc_ptr = ObjPtr->detail.a;
         arc_ptr->xc = two_x_pivot - arc_ptr->xc;
         arc_ptr->x1 = two_x_pivot - arc_ptr->x1;
         arc_ptr->x2 = two_x_pivot - arc_ptr->x2;
         arc_ptr->dir = !(arc_ptr->dir);
         arc_ptr->ltx = two_x_pivot - arc_ptr->ltx - arc_ptr->w;
         if (arc_ptr->angle1 > 0)
            arc_ptr->angle1 = (180*64) - arc_ptr->angle1;
         else
            arc_ptr->angle1 = (-180)*64 - arc_ptr->angle1;
         arc_ptr->angle2 = -(arc_ptr->angle2);
         UpdArcBBox (ObjPtr);
         attr_ptr = ObjPtr->fattr;
         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            FlipObjHorizontal (attr_ptr->obj);
         break;
      case OBJ_GROUP:
      case OBJ_SYM:
      case OBJ_ICON:
         obj_ptr = ObjPtr->detail.r->first;
         for ( ; obj_ptr != NULL; obj_ptr = obj_ptr->next)
            FlipObjHorizontal (obj_ptr);
         attr_ptr = ObjPtr->fattr;
         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            FlipObjHorizontal (attr_ptr->obj);
         break;
   }
   AdjObjBBox (ObjPtr);
}

void FlipIconHorizontal (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   register int		two_x_pivot;
   int			new_obj_ltx, new_obj_rbx;
   struct ObjRec	* obj_ptr;
   struct AttrRec	* attr_ptr;

   two_x_pivot = selObjLtX + selObjRbX;
   new_obj_ltx = two_x_pivot - ObjPtr->obbox.rbx;
   new_obj_rbx = two_x_pivot - ObjPtr->obbox.ltx;

   switch (ObjPtr->detail.r->rotate)
   {
      case ROTATE0:
      case ROTATE180: ObjPtr->detail.r->flip ^= HORI_EVEN; break;

      case ROTATE90:
      case ROTATE270: ObjPtr->detail.r->flip ^= HORI_ODD; break;
   }

   ObjPtr->obbox.ltx = ObjPtr->x = new_obj_ltx;
   ObjPtr->obbox.rbx = new_obj_rbx;

   obj_ptr = ObjPtr->detail.r->first;
   for ( ; obj_ptr != NULL; obj_ptr = obj_ptr->next)
      FlipObjHorizontal (obj_ptr);
   attr_ptr = ObjPtr->fattr;
   for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
      FlipObjHorizontal (attr_ptr->obj);

   AdjObjBBox (ObjPtr);
}

void FlipObjVertical (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   register XPoint	* v;
   register int		i, two_x_pivot;
   int			new_obj_lty, new_obj_rby, num_pts;
   struct ObjRec	* obj_ptr;
   struct AttrRec	* attr_ptr;
   struct ArcRec	* arc_ptr;

   SetFileModified (TRUE);
   two_x_pivot = selObjLtY + selObjRbY;
   new_obj_lty = two_x_pivot - ObjPtr->obbox.rby;
   new_obj_rby = two_x_pivot - ObjPtr->obbox.lty;
   switch (ObjPtr->type)
   {
      case OBJ_BOX:
      case OBJ_OVAL:
      case OBJ_POLY:
      case OBJ_POLYGON:
      case OBJ_ARC:
      case OBJ_RCBOX:
      case OBJ_XBM:
      case OBJ_XPM:
      case OBJ_GROUP:
      case OBJ_SYM:
      case OBJ_ICON:
         switch (ObjPtr->type)
         {
            case OBJ_XBM:
               switch (ObjPtr->detail.xbm->rotate)
               {
                  case ROTATE0:
                  case ROTATE180:
                     ObjPtr->detail.xbm->flip ^= VERT_EVEN;
                     break;
                  case ROTATE90:
                  case ROTATE270:
                     ObjPtr->detail.xbm->flip ^= VERT_ODD;
                     break;
               }
               if (ObjPtr->detail.xbm->cached_bitmap != None)
                  XFreePixmap (mainDisplay, ObjPtr->detail.xbm->cached_bitmap);
               ObjPtr->detail.xbm->cached_bitmap = None;

               if (zoomScale != 0)
                  ObjPtr->detail.xbm->cached_zoom = 0;
               else
                  ObjPtr->detail.xbm->cached_rotate = INVALID;
               break;
            case OBJ_XPM:
               switch (ObjPtr->detail.xpm->rotate)
               {
                  case ROTATE0:
                  case ROTATE180:
                     ObjPtr->detail.xpm->flip ^= VERT_EVEN;
                     break;
                  case ROTATE90:
                  case ROTATE270:
                     ObjPtr->detail.xpm->flip ^= VERT_ODD;
                     break;
               }
               if (ObjPtr->detail.xpm->cached_pixmap != None)
                  XFreePixmap (mainDisplay, ObjPtr->detail.xpm->cached_pixmap);
               ObjPtr->detail.xpm->cached_pixmap = None;

               if (zoomScale != 0)
                  ObjPtr->detail.xpm->cached_zoom = 0;
               else
                  ObjPtr->detail.xpm->cached_rotate = INVALID;
               break;
            case OBJ_ICON:
               switch (ObjPtr->detail.r->rotate)
               {
                  case ROTATE0:
                  case ROTATE180: ObjPtr->detail.r->flip ^= VERT_EVEN; break;

                  case ROTATE90:
                  case ROTATE270: ObjPtr->detail.r->flip ^= VERT_ODD; break;
               }
               break;
         }
         ObjPtr->obbox.lty = ObjPtr->y = new_obj_lty;
         ObjPtr->obbox.rby = new_obj_rby;
         break;
      case OBJ_TEXT:
         switch (ObjPtr->detail.t->rotate)
         {
            case ROTATE0: ObjPtr->y = new_obj_lty; break;
            case ROTATE180: ObjPtr->y = new_obj_rby; break;
            case ROTATE90:
            case ROTATE270:
               ObjPtr->y = two_x_pivot - ObjPtr->y;
               if (ObjPtr->detail.t->just != JUST_C)
               {
                  ObjPtr->detail.t->just = MAXJUSTS-1-ObjPtr->detail.t->just;
                  if (ObjPtr->detail.t->cached_bitmap != None)
                     XFreePixmap (mainDisplay, ObjPtr->detail.t->cached_bitmap);
                  ObjPtr->detail.t->cached_bitmap = None;

                  if (zoomScale != 0)
                     ObjPtr->detail.t->cached_zoom = 0;
                  else
                     ObjPtr->detail.t->cached_rotate = INVALID;
               }
               break;
         }
         UpdTextBBox (ObjPtr);
         break;
   }
   switch (ObjPtr->type)
   {
      case OBJ_POLY:
         num_pts = ObjPtr->detail.p->n;
         v = ObjPtr->detail.p->vlist;
         for (i = 0; i < num_pts; i++, v++) (*v).y = two_x_pivot - (*v).y;
         AdjObjSplineVs (ObjPtr);
         attr_ptr = ObjPtr->fattr;
         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            FlipObjVertical (attr_ptr->obj);
         break;
      case OBJ_BOX:
      case OBJ_OVAL:
      case OBJ_RCBOX:
      case OBJ_XBM:
      case OBJ_XPM:
         attr_ptr = ObjPtr->fattr;
         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            FlipObjVertical (attr_ptr->obj);
         break;
      case OBJ_TEXT: break;
      case OBJ_POLYGON:
         num_pts = ObjPtr->detail.g->n;
         v = ObjPtr->detail.g->vlist;
         for (i = 0; i < num_pts; i++, v++) (*v).y = two_x_pivot - (*v).y;
         AdjObjSplineVs (ObjPtr);
         attr_ptr = ObjPtr->fattr;
         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            FlipObjVertical (attr_ptr->obj);
         break;
      case OBJ_ARC:
         arc_ptr = ObjPtr->detail.a;
         arc_ptr->yc = two_x_pivot - arc_ptr->yc;
         arc_ptr->y1 = two_x_pivot - arc_ptr->y1;
         arc_ptr->y2 = two_x_pivot - arc_ptr->y2;
         arc_ptr->dir = !(arc_ptr->dir);
         arc_ptr->lty = two_x_pivot - arc_ptr->lty - arc_ptr->h;
         arc_ptr->angle1 = -(arc_ptr->angle1);
         arc_ptr->angle2 = -(arc_ptr->angle2);
         UpdArcBBox (ObjPtr);
         attr_ptr = ObjPtr->fattr;
         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            FlipObjVertical (attr_ptr->obj);
         break;
      case OBJ_GROUP:
      case OBJ_SYM:
      case OBJ_ICON:
         obj_ptr = ObjPtr->detail.r->first;
         for ( ; obj_ptr != NULL; obj_ptr = obj_ptr->next)
            FlipObjVertical (obj_ptr);
         attr_ptr = ObjPtr->fattr;
         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            FlipObjVertical (attr_ptr->obj);
         break;
   }
   AdjObjBBox (ObjPtr);
}

void FlipIconVertical (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   register int		two_x_pivot;
   int			new_obj_lty, new_obj_rby;
   struct ObjRec	* obj_ptr;
   struct AttrRec	* attr_ptr;

   two_x_pivot = selObjLtY + selObjRbY;
   new_obj_lty = two_x_pivot - ObjPtr->obbox.rby;
   new_obj_rby = two_x_pivot - ObjPtr->obbox.lty;

   switch (ObjPtr->detail.r->rotate)
   {
      case ROTATE0:
      case ROTATE180: ObjPtr->detail.r->flip ^= VERT_EVEN; break;

      case ROTATE90:
      case ROTATE270: ObjPtr->detail.r->flip ^= VERT_ODD; break;
   }

   ObjPtr->obbox.lty = ObjPtr->y = new_obj_lty;
   ObjPtr->obbox.rby = new_obj_rby;

   obj_ptr = ObjPtr->detail.r->first;
   for ( ; obj_ptr != NULL; obj_ptr = obj_ptr->next)
      FlipObjVertical (obj_ptr);
   attr_ptr = ObjPtr->fattr;
   for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
      FlipObjVertical (attr_ptr->obj);

   AdjObjBBox (ObjPtr);
}

static int	rotatePivotX;
static int	rotatePivotY;

void SetRotatePivot ()
{
   if (topSel != NULL && topSel == botSel && topSel->obj->type == OBJ_TEXT)
   {
      rotatePivotX = topSel->obj->x;
      rotatePivotY = topSel->obj->y;
   }
   else
   {
      rotatePivotX = ((selObjLtX + selObjRbX)>>1);
      rotatePivotY = ((selObjLtY + selObjRbY)>>1);
   }
}

static
void RotatePtClockWise (X, Y, NewX, NewY)
   int	X, Y, * NewX, * NewY;
{
   *NewX = rotatePivotX + rotatePivotY - Y;
   *NewY = rotatePivotY - rotatePivotX + X;
}

void RotateObjClockWise (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   register XPoint	* v;
   register int		i;
   int			x, y, ltx, lty, rbx, rby, num_pts, new_h;
   struct ObjRec	* obj_ptr;
   struct AttrRec	* attr_ptr;
   struct ArcRec	* arc_ptr;

   SetFileModified (TRUE);
   switch (ObjPtr->type)
   {
      case OBJ_BOX:
      case OBJ_OVAL:
      case OBJ_POLY:
      case OBJ_POLYGON:
      case OBJ_ARC:
      case OBJ_RCBOX:
      case OBJ_XBM:
      case OBJ_XPM:
      case OBJ_GROUP:
      case OBJ_SYM:
      case OBJ_ICON:
         switch (ObjPtr->type)
         {
            case OBJ_XBM:
               ObjPtr->detail.xbm->rotate = (ObjPtr->detail.xbm->rotate+4+1)%4;
               if (ObjPtr->detail.xbm->cached_bitmap != None)
                  XFreePixmap (mainDisplay, ObjPtr->detail.xbm->cached_bitmap);
               ObjPtr->detail.xbm->cached_bitmap = None;

               if (zoomScale != 0)
                  ObjPtr->detail.xbm->cached_zoom = 0;
               else
                  ObjPtr->detail.xbm->cached_rotate = INVALID;
               break;
            case OBJ_XPM:
               ObjPtr->detail.xpm->rotate = (ObjPtr->detail.xpm->rotate+4+1)%4;
               if (ObjPtr->detail.xpm->cached_pixmap != None)
                  XFreePixmap (mainDisplay, ObjPtr->detail.xpm->cached_pixmap);
               ObjPtr->detail.xpm->cached_pixmap = None;

               if (zoomScale != 0)
                  ObjPtr->detail.xpm->cached_zoom = 0;
               else
                  ObjPtr->detail.xpm->cached_rotate = INVALID;
               break;
            case OBJ_ICON:
               ObjPtr->detail.r->rotate = (ObjPtr->detail.r->rotate+4+1) % 4;
               break;
         }
         RotatePtClockWise (ObjPtr->obbox.ltx, ObjPtr->obbox.rby, &ltx, &lty);
         RotatePtClockWise (ObjPtr->obbox.rbx, ObjPtr->obbox.lty, &rbx, &rby);
         ObjPtr->obbox.ltx = ObjPtr->x = ltx;
         ObjPtr->obbox.lty = ObjPtr->y = lty;
         ObjPtr->obbox.rbx = rbx;
         ObjPtr->obbox.rby = rby;
         break;
      case OBJ_TEXT:
         RotatePtClockWise (ObjPtr->x, ObjPtr->y, &x, &y);
         ObjPtr->x = x;
         ObjPtr->y = y;
         ObjPtr->detail.t->rotate = (ObjPtr->detail.t->rotate+1) & 0x3;

         if (ObjPtr->detail.t->cached_bitmap != None)
            XFreePixmap (mainDisplay, ObjPtr->detail.t->cached_bitmap);
         ObjPtr->detail.t->cached_bitmap = None;

         if (zoomScale != 0)
            ObjPtr->detail.t->cached_zoom = 0;
         else
            ObjPtr->detail.t->cached_rotate = INVALID;

         UpdTextBBox (ObjPtr);
         break;
   }
   switch (ObjPtr->type)
   {
      case OBJ_POLY:
         num_pts = ObjPtr->detail.p->n;
         v = ObjPtr->detail.p->vlist;
         for (i = 0; i < num_pts; i++, v++)
         {
            RotatePtClockWise ((*v).x, (*v).y, &x, &y);
            (*v).x = x;
            (*v).y = y;
         }
         AdjObjSplineVs (ObjPtr);
         attr_ptr = ObjPtr->fattr;
         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            RotateObjClockWise (attr_ptr->obj);
         break;
      case OBJ_BOX:
      case OBJ_OVAL:
      case OBJ_RCBOX:
      case OBJ_XBM:
      case OBJ_XPM:
         attr_ptr = ObjPtr->fattr;
         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            RotateObjClockWise (attr_ptr->obj);
         break;
      case OBJ_TEXT: break;
      case OBJ_POLYGON:
         num_pts = ObjPtr->detail.g->n;
         v = ObjPtr->detail.g->vlist;
         for (i = 0; i < num_pts; i++, v++)
         {
            RotatePtClockWise ((*v).x, (*v).y, &x, &y);
            (*v).x = x;
            (*v).y = y;
         }
         AdjObjSplineVs (ObjPtr);
         attr_ptr = ObjPtr->fattr;
         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            RotateObjClockWise (attr_ptr->obj);
         break;
      case OBJ_ARC:
         arc_ptr = ObjPtr->detail.a;

         RotatePtClockWise (arc_ptr->xc, arc_ptr->yc, &x, &y);
         arc_ptr->xc = ObjPtr->x = x; arc_ptr->yc = ObjPtr->y = y;
         RotatePtClockWise (arc_ptr->x1, arc_ptr->y1, &x, &y);
         arc_ptr->x1 = x; arc_ptr->y1 = y;
         RotatePtClockWise (arc_ptr->x2, arc_ptr->y2, &x, &y);
         arc_ptr->x2 = x; arc_ptr->y2 = y;
         RotatePtClockWise (arc_ptr->ltx, arc_ptr->lty+arc_ptr->h, &x, &y);
         arc_ptr->ltx = x; arc_ptr->lty = y;
         new_h = arc_ptr->w;
         arc_ptr->w = arc_ptr->h; arc_ptr->h = new_h;
         arc_ptr->angle1 -= (arc_ptr->angle1 < (-90*64)) ? (-270*64) : (90*64);
         UpdArcBBox (ObjPtr);

         attr_ptr = ObjPtr->fattr;
         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            RotateObjClockWise (attr_ptr->obj);
         break;
      case OBJ_GROUP:
      case OBJ_SYM:
      case OBJ_ICON:
         obj_ptr = ObjPtr->detail.r->first;
         for ( ; obj_ptr != NULL; obj_ptr = obj_ptr->next)
            RotateObjClockWise (obj_ptr);
         attr_ptr = ObjPtr->fattr;
         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            RotateObjClockWise (attr_ptr->obj);
         break;
   }
   AdjObjBBox (ObjPtr);
}

void RotateIconClockWise (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   int			ltx, lty, rbx, rby;
   struct ObjRec	* obj_ptr;
   struct AttrRec	* attr_ptr;

   SetRotatePivot ();
   ObjPtr->detail.r->rotate = (ObjPtr->detail.r->rotate+4+1) % 4;
   RotatePtClockWise (ObjPtr->obbox.ltx, ObjPtr->obbox.rby, &ltx, &lty);
   RotatePtClockWise (ObjPtr->obbox.rbx, ObjPtr->obbox.lty, &rbx, &rby);
   ObjPtr->obbox.ltx = ObjPtr->x = ltx;
   ObjPtr->obbox.lty = ObjPtr->y = lty;
   ObjPtr->obbox.rbx = rbx;
   ObjPtr->obbox.rby = rby;

   obj_ptr = ObjPtr->detail.r->first;
   for ( ; obj_ptr != NULL; obj_ptr = obj_ptr->next)
      RotateObjClockWise (obj_ptr);
   attr_ptr = ObjPtr->fattr;
   for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
      RotateObjClockWise (attr_ptr->obj);

   AdjObjBBox (ObjPtr);
}

static
void RotatePtCounter (X, Y, NewX, NewY)
   int	X, Y, * NewX, * NewY;
{
   *NewX = rotatePivotX - rotatePivotY + Y;
   *NewY = rotatePivotY + rotatePivotX - X;
}

void RotateObjCounter (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   register XPoint	* v;
   register int		i;
   int			x, y, ltx, lty, rbx, rby, num_pts, new_h;
   struct ObjRec	* obj_ptr;
   struct AttrRec	* attr_ptr;
   struct ArcRec	* arc_ptr;

   SetFileModified (TRUE);
   switch (ObjPtr->type)
   {
      case OBJ_BOX:
      case OBJ_OVAL:
      case OBJ_POLY:
      case OBJ_POLYGON:
      case OBJ_ARC:
      case OBJ_RCBOX:
      case OBJ_XBM:
      case OBJ_XPM:
      case OBJ_GROUP:
      case OBJ_SYM:
      case OBJ_ICON:
         switch (ObjPtr->type)
         {
            case OBJ_XBM:
               ObjPtr->detail.xbm->rotate = (ObjPtr->detail.xbm->rotate+4-1)%4;
               if (ObjPtr->detail.xbm->cached_bitmap != None)
                  XFreePixmap (mainDisplay, ObjPtr->detail.xbm->cached_bitmap);
               ObjPtr->detail.xbm->cached_bitmap = None;

               if (zoomScale != 0)
                  ObjPtr->detail.xbm->cached_zoom = 0;
               else
                  ObjPtr->detail.xbm->cached_rotate = INVALID;
               break;
            case OBJ_XPM:
               ObjPtr->detail.xpm->rotate = (ObjPtr->detail.xpm->rotate+4-1)%4;
               if (ObjPtr->detail.xpm->cached_pixmap != None)
                  XFreePixmap (mainDisplay, ObjPtr->detail.xpm->cached_pixmap);
               ObjPtr->detail.xpm->cached_pixmap = None;

               if (zoomScale != 0)
                  ObjPtr->detail.xpm->cached_zoom = 0;
               else
                  ObjPtr->detail.xpm->cached_rotate = INVALID;
               break;
            case OBJ_ICON:
               ObjPtr->detail.r->rotate = (ObjPtr->detail.r->rotate+4-1) % 4;
               break;
         }
         RotatePtCounter (ObjPtr->obbox.rbx, ObjPtr->obbox.lty, &ltx, &lty);
         RotatePtCounter (ObjPtr->obbox.ltx, ObjPtr->obbox.rby, &rbx, &rby);
         ObjPtr->obbox.ltx = ObjPtr->x = ltx;
         ObjPtr->obbox.lty = ObjPtr->y = lty;
         ObjPtr->obbox.rbx = rbx;
         ObjPtr->obbox.rby = rby;
         break;
      case OBJ_TEXT:
         RotatePtCounter (ObjPtr->x, ObjPtr->y, &x, &y);
         ObjPtr->x = x;
         ObjPtr->y = y;
         ObjPtr->detail.t->rotate = (ObjPtr->detail.t->rotate+3) & 0x3;

         if (ObjPtr->detail.t->cached_bitmap != None)
            XFreePixmap (mainDisplay, ObjPtr->detail.t->cached_bitmap);
         ObjPtr->detail.t->cached_bitmap = None;

         if (zoomScale != 0)
            ObjPtr->detail.t->cached_zoom = 0;
         else
            ObjPtr->detail.t->cached_rotate = INVALID;

         UpdTextBBox (ObjPtr);
         break;
   }
   switch (ObjPtr->type)
   {
      case OBJ_POLY:
         num_pts = ObjPtr->detail.p->n;
         v = ObjPtr->detail.p->vlist;
         for (i = 0; i < num_pts; i++, v++)
         {
            RotatePtCounter ((*v).x, (*v).y, &x, &y);
            (*v).x = x;
            (*v).y = y;
         }
         AdjObjSplineVs (ObjPtr);
         attr_ptr = ObjPtr->fattr;
         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            RotateObjCounter (attr_ptr->obj);
         break;
      case OBJ_BOX:
      case OBJ_OVAL:
      case OBJ_RCBOX:
      case OBJ_XBM:
      case OBJ_XPM:
         attr_ptr = ObjPtr->fattr;
         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            RotateObjCounter (attr_ptr->obj);
         break;
      case OBJ_TEXT: break;
      case OBJ_POLYGON:
         num_pts = ObjPtr->detail.g->n;
         v = ObjPtr->detail.g->vlist;
         for (i = 0; i < num_pts; i++, v++)
         {
            RotatePtCounter ((*v).x, (*v).y, &x, &y);
            (*v).x = x;
            (*v).y = y;
         }
         AdjObjSplineVs (ObjPtr);
         attr_ptr = ObjPtr->fattr;
         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            RotateObjCounter (attr_ptr->obj);
         break;
      case OBJ_ARC:
         arc_ptr = ObjPtr->detail.a;

         RotatePtCounter (arc_ptr->xc, arc_ptr->yc, &x, &y);
         arc_ptr->xc = ObjPtr->x = x; arc_ptr->yc = ObjPtr->y = y;
         RotatePtCounter (arc_ptr->x1, arc_ptr->y1, &x, &y);
         arc_ptr->x1 = x; arc_ptr->y1 = y;
         RotatePtCounter (arc_ptr->x2, arc_ptr->y2, &x, &y);
         arc_ptr->x2 = x; arc_ptr->y2 = y;
         RotatePtCounter (arc_ptr->ltx+arc_ptr->w, arc_ptr->lty, &x, &y);
         arc_ptr->ltx = x; arc_ptr->lty = y;
         new_h = arc_ptr->w;
         arc_ptr->w = arc_ptr->h; arc_ptr->h = new_h;
         arc_ptr->angle1 += (arc_ptr->angle1 > 90*64) ? (-270*64) : (90*64);
         UpdArcBBox (ObjPtr);

         attr_ptr = ObjPtr->fattr;
         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            RotateObjCounter (attr_ptr->obj);
         break;
      case OBJ_GROUP:
      case OBJ_SYM:
      case OBJ_ICON:
         obj_ptr = ObjPtr->detail.r->first;
         for ( ; obj_ptr != NULL; obj_ptr = obj_ptr->next)
            RotateObjCounter (obj_ptr);
         attr_ptr = ObjPtr->fattr;
         for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
            RotateObjCounter (attr_ptr->obj);
         break;
   }
   AdjObjBBox (ObjPtr);
}

void RotateIconCounter (ObjPtr)
   struct ObjRec	* ObjPtr;
{
   int			ltx, lty, rbx, rby;
   struct ObjRec	* obj_ptr;
   struct AttrRec	* attr_ptr;

   SetRotatePivot ();
   ObjPtr->detail.r->rotate = (ObjPtr->detail.r->rotate+4-1) % 4;
   RotatePtCounter (ObjPtr->obbox.rbx, ObjPtr->obbox.lty, &ltx, &lty);
   RotatePtCounter (ObjPtr->obbox.ltx, ObjPtr->obbox.rby, &rbx, &rby);
   ObjPtr->obbox.ltx = ObjPtr->x = ltx;
   ObjPtr->obbox.lty = ObjPtr->y = lty;
   ObjPtr->obbox.rbx = rbx;
   ObjPtr->obbox.rby = rby;

   obj_ptr = ObjPtr->detail.r->first;
   for ( ; obj_ptr != NULL; obj_ptr = obj_ptr->next)
      RotateObjCounter (obj_ptr);
   attr_ptr = ObjPtr->fattr;
   for ( ; attr_ptr != NULL; attr_ptr = attr_ptr->next)
      RotateObjCounter (attr_ptr->obj);

   AdjObjBBox (ObjPtr);
}

void FlipHorizontal ()
{
   register struct SelRec	* sel_ptr;
   int				saved_ltx, saved_lty, saved_rbx, saved_rby;

   if (topSel == NULL) return;
   if (numObjSelected == numObjLocked)
   {
      Msg ("Locked objects can not be flipped.");
      return;
   }

   saved_ltx = selLtX; saved_lty = selLtY;
   saved_rbx = selRbX; saved_rby = selRbY;
   HighLightReverse ();
   PrepareToRecord (CMD_REPLACE, topSel, botSel, numObjSelected);
   JustRemoveAllVSel ();
   for (sel_ptr = topSel; sel_ptr != NULL; sel_ptr = sel_ptr->next)
      if (!sel_ptr->obj->locked)
         FlipObjHorizontal (sel_ptr->obj);
   UpdSelBBox ();
   RecordCmd (CMD_REPLACE, NULL, topSel, botSel, numObjSelected);
   RedrawAreas (botObj, saved_ltx-GRID_ABS_SIZE(1), saved_lty-GRID_ABS_SIZE(1),
         saved_rbx+GRID_ABS_SIZE(1), saved_rby+GRID_ABS_SIZE(1),
         selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
         selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1));
   HighLightForward ();
   justDupped = FALSE;
   if (numObjLocked != 0)
      Msg ("Locked objects are not flipped.");
   else
      Msg ("Flipped horizontally.");
}

void FlipVertical ()
{
   register struct SelRec	* sel_ptr;
   int				saved_ltx, saved_lty, saved_rbx, saved_rby;

   if (topSel == NULL) return;
   if (numObjSelected == numObjLocked)
   {
      Msg ("Locked objects can not be flipped.");
      return;
   }

   saved_ltx = selLtX; saved_lty = selLtY;
   saved_rbx = selRbX; saved_rby = selRbY;
   HighLightReverse ();
   PrepareToRecord (CMD_REPLACE, topSel, botSel, numObjSelected);
   JustRemoveAllVSel ();
   for (sel_ptr = topSel; sel_ptr != NULL; sel_ptr = sel_ptr->next)
      if (!sel_ptr->obj->locked)
         FlipObjVertical (sel_ptr->obj);
   UpdSelBBox ();
   RecordCmd (CMD_REPLACE, NULL, topSel, botSel, numObjSelected);
   RedrawAreas (botObj, saved_ltx-GRID_ABS_SIZE(1), saved_lty-GRID_ABS_SIZE(1),
         saved_rbx+GRID_ABS_SIZE(1), saved_rby+GRID_ABS_SIZE(1),
         selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
         selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1));
   HighLightForward ();
   justDupped = FALSE;
   if (numObjLocked != 0)
      Msg ("Locked objects are not flipped.");
   else
      Msg ("Flipped vertically.");
}

void RotateClockWise ()
{
   register struct SelRec	* sel_ptr;
   int				saved_ltx, saved_lty, saved_rbx, saved_rby;
   int				text_obj_created, text_cursor_shown;

   if (topSel == NULL)
   {
      text_cursor_shown = textCursorShown;
      text_obj_created = TieLooseEnds ();
      curRotate = (curRotate+1) & 0x3;
      ShowRotate ();
      if (!text_obj_created && curChoice == DRAWTEXT && text_cursor_shown)
      {
         NewCurText ();
         RedrawCurText ();
      }
      else
         textCursorShown = FALSE;
      return;
   }
   if (numObjSelected == numObjLocked)
   {
      Msg ("Locked objects can not be rotated.");
      return;
   }
   saved_ltx = selLtX; saved_lty = selLtY;
   saved_rbx = selRbX; saved_rby = selRbY;
   HighLightReverse ();
   PrepareToRecord (CMD_REPLACE, topSel, botSel, numObjSelected);
   JustRemoveAllVSel ();
   SetRotatePivot ();
   for (sel_ptr = topSel; sel_ptr != NULL; sel_ptr = sel_ptr->next)
      if (!sel_ptr->obj->locked)
         RotateObjClockWise (sel_ptr->obj);
   UpdSelBBox ();
   RecordCmd (CMD_REPLACE, NULL, topSel, botSel, numObjSelected);
   RedrawAreas (botObj, saved_ltx-GRID_ABS_SIZE(1), saved_lty-GRID_ABS_SIZE(1),
         saved_rbx+GRID_ABS_SIZE(1), saved_rby+GRID_ABS_SIZE(1),
         selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
         selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1));
   HighLightForward ();
   justDupped = FALSE;
   if (numObjLocked != 0)
      Msg ("Locked objects are not rotated.");
   else
      Msg ("Rotated clockwise.");
}

void RotateCounter ()
{
   register struct SelRec	* sel_ptr;
   int				saved_ltx, saved_lty, saved_rbx, saved_rby;
   int				text_obj_created, text_cursor_shown;

   if (topSel == NULL)
   {
      text_cursor_shown = textCursorShown;
      text_obj_created = TieLooseEnds ();
      curRotate = (curRotate+3) & 0x3;
      ShowRotate ();
      if (!text_obj_created && curChoice == DRAWTEXT && text_cursor_shown)
      {
         NewCurText ();
         RedrawCurText ();
      }
      else
         textCursorShown = FALSE;
      return;
   }
   if (numObjSelected == numObjLocked)
   {
      Msg ("Locked objects can not be rotated.");
      return;
   }
   saved_ltx = selLtX; saved_lty = selLtY;
   saved_rbx = selRbX; saved_rby = selRbY;
   HighLightReverse ();
   PrepareToRecord (CMD_REPLACE, topSel, botSel, numObjSelected);
   JustRemoveAllVSel ();
   SetRotatePivot ();
   for (sel_ptr = topSel; sel_ptr != NULL; sel_ptr = sel_ptr->next)
      if (!sel_ptr->obj->locked)
         RotateObjCounter (sel_ptr->obj);
   UpdSelBBox ();
   RecordCmd (CMD_REPLACE, NULL, topSel, botSel, numObjSelected);
   RedrawAreas (botObj, saved_ltx-GRID_ABS_SIZE(1), saved_lty-GRID_ABS_SIZE(1),
         saved_rbx+GRID_ABS_SIZE(1), saved_rby+GRID_ABS_SIZE(1),
         selLtX-GRID_ABS_SIZE(1), selLtY-GRID_ABS_SIZE(1),
         selRbX+GRID_ABS_SIZE(1), selRbY+GRID_ABS_SIZE(1));
   HighLightForward ();
   justDupped = FALSE;
   if (numObjLocked != 0)
      Msg ("Locked objects are not rotated.");
   else
      Msg ("Rotated counter-clockwise.");
}
