=head1 NAME

Pod::Simple::FromTree - parse POD from tree form

=head1 SYNOPSIS

	use parent "Pod::Simple::FromTree";

	$parser->parse_tree($podtree);

=head1 DESCRIPTION

This is a subclass of L<Pod::Simple>, intended to be further subclassed.
Objects of this class, as with L<Pod::Simple>, can parse a POD document,
internally using an event-based protocol, and subclasses are expected
to add specific behaviour to do something with the events.  This class
provides a method to parse from an existing parse tree, of the type
that L<Pod::Simple::SimpleTree> generates.  (The basic L<Pod::Simple>
can only parse from POD source, either in a file or in a Perl string.)

Not only can this class be directly subclassed, but it can also be
mixed into an existing subclass of L<Pod::Simple>.  For example, a class
that inherits from this class and L<Pod::Man> (which is a subclass of
L<Pod::Simple>) can accept a POD parse tree and render it as man page
nroff source.  L<Class::Mix> provides a mechanism to generate such mixed
classes on demand.

Processing POD from the parse tree form is useful in almost any
non-trivial workflow.  Most obviously, it is useful if automatic or
structured editing of the POD is desired, which is most easily performed
on the parse tree form of the document.  It is also an efficiency
improvement if the same document is to be rendered in more than one
form: the POD source can be parsed once to generate a parse tree (using
L<Pod::Simple::SimpleTree>), and then the parse tree rendered multiple
times (using subclasses of this class).

=cut

package Pod::Simple::FromTree;

{ use 5.006; }
use warnings;
use strict;

our $VERSION = "0.001";

use parent "Pod::Simple";

=head1 METHODS

=over

=item $parser->parse_tree(PODTREE)

Takes a POD parse tree (or part thereof), of the type generated by
L<Pod::Simple::SimpleTree>, and generates the internal parser events
corresponding to that tree.

=cut

sub parse_tree {
	my($parser, $tree) = @_;
	if(ref($tree) eq "") {
		$parser->_handle_text($tree);
		return;
	}
	$parser->_handle_element_start($tree->[0], $tree->[1]);
	$parser->parse_tree($_) foreach @{$tree}[2..$#$tree];
	$parser->_handle_element_end($tree->[0]);
}

=back

=head1 SEE ALSO

L<Class::Mix>,
L<Pod::Simple>,
L<Pod::Simple::SimpleTree>

=head1 AUTHOR

Andrew Main (Zefram) <zefram@fysh.org>

=head1 COPYRIGHT

Copyright (C) 2010 PhotoBox Ltd

Copyright (C) 2011 Andrew Main (Zefram) <zefram@fysh.org>

=head1 LICENSE

This module is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.

=cut

1;
